"use strict";
/*
*  Copyright (C) 1998-2021 by Northwoods Software Corporation. All Rights Reserved.
*/

// A custom Tool for freehand drawing

/*
* This is an extension and not part of the main GoJS library.
* Note that the API for this class may change with any version, even point releases.
* If you intend to use an extension in production, you should copy the code to your own source directory.
* Extensions can be found in the GoJS kit under the extensions or extensionsTS folders.
* See the Extensions intro page (https://gojs.net/latest/intro/extensions.html) for more information.
*/

/**
* @constructor
* @extends Tool
* @class
* This tool allows the user to draw a shape using the mouse.
* It collects all of the points from a mouse-down, all mouse-moves, until a mouse-up,
* and puts all of those points in a {@link Geometry} used by a {@link Shape}.
* It then adds a node data object to the diagram's model.
* <p/>
* This tool may be installed as the first mouse down tool:
* <code>myDiagram.toolManager.mouseDownTools.insertAt(0, new FreehandDrawingTool());</code>
* <p/>
* The Shape used during the drawing operation can be customized by setting {@link #temporaryShape}.
* The node data added to the model can be customized by setting {@link #archetypePartData}.
*/
function FreehandDrawingTool() {
  go.Tool.call(this);
  this.name = "FreehandDrawing";
  this._archetypePartData = {}; // the data to copy for a new polyline Part
  this._isBackgroundOnly = true; // affects canStart()

  // this is the Shape that is shown during a drawing operation
  this._temporaryShape = go.GraphObject.make(go.Shape, { name: "SHAPE", fill: null, strokeWidth: 1.5 });
  // the Shape has to be inside a temporary Part that is used during the drawing operation
  go.GraphObject.make(go.Part, { layerName: "Tool" }, this._temporaryShape);

  this._points = [];
}
go.Diagram.inherit(FreehandDrawingTool, go.Tool);

/**
* Only start if the diagram is modifiable and allows insertions.
* OPTIONAL: if the user is starting in the diagram's background, not over an existing Part.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.canStart = function() {
  if (!this.isEnabled) return false;
  var diagram = this.diagram;
  if (diagram === null || diagram.isReadOnly || diagram.isModelReadOnly) return false;
  if (!diagram.allowInsert) return false;
  // don't include the following check when this tool is running modally
  if (diagram.currentTool !== this && this.isBackgroundOnly) {
    // only operates in the background, not on some Part
    var part = diagram.findPartAt(diagram.lastInput.documentPoint, true);
    if (part !== null) return false;
  }
  return true;
};

/**
* Capture the mouse and use a "crosshair" cursor.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.doActivate = function() {
  go.Tool.prototype.doActivate.call(this);
  this.diagram.isMouseCaptured = true;
  this.diagram.currentCursor = "";

  var shape = this.temporaryShape;
  if (shape === null) return;

  shape.stroke = this.archetypePartData.border;
  shape.strokeWidth = this.archetypePartData.thickness;

};

/**
* Cleanup.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.doDeactivate = function() {
  go.Tool.prototype.doDeactivate.call(this);
  if (this.temporaryShape !== null) {
    this.diagram.remove(this.temporaryShape.part);
  }
  this.diagram.currentCursor = "";
  this.diagram.isMouseCaptured = false;
  this._points = [];
};

/**
* This adds a Point to the {@link #temporaryShape}'s geometry.
* <p/>
* If the Shape is not yet in the Diagram, its geometry is initialized and
* its parent Part is added to the Diagram.
* <p/>
* If the point is less than half a pixel away from the previous point, it is ignored.
* @this {FreehandDrawingTool}
* @param {Point} p
*/

FreehandDrawingTool.prototype.addPoint = function(p) {
  var shape = this.temporaryShape;
  if (shape === null) return;

  // for the temporary Shape, normalize the geometry to be in the viewport
  var viewpt = this.diagram.viewportBounds.position;
  var q = new go.Point(p.x-viewpt.x, p.y-viewpt.y);

  var part = shape.part;
  if (part.diagram === null) {
    var fig = new go.PathFigure(q.x, q.y, true);  // possibly filled, depending on Shape.fill
    var geo = new go.Geometry().add(fig);  // the Shape.geometry consists of a single PathFigure
    this.temporaryShape.geometry = geo;
    // position the Shape's Part, accounting for the strokeWidth
    part.position = new go.Point(viewpt.x - shape.strokeWidth/2, viewpt.y - shape.strokeWidth/2);
    this.diagram.add(part);
  }

  // only add a point if it isn't too close to the last one
  // var segs = shape.geometry.figures.first().segments;
  var idx = this._points.length-1;
  if (idx >= 0) {
    var last = this._points[idx];
    if (Math.abs(q.x - last.x) < 15.0 && Math.abs(q.y - last.y) < 15.0) return;
  }

  this._points.push(q);

  var numpts = this._points.length;
  var geo = shape.geometry.copy();
  var fig = geo.figures.first();

  var segments = new go.List();
  if (numpts > 2) {

    var tension = 1.0;
    var t = (tension != null) ? tension : 1;
    for (var i = 0; i <  this._points.length - 1; i++) {
        
        var p0 = (i > 0) ?  this._points[i - 1] :  this._points[0];
        var p1 =  this._points[i];
        var p2 =  this._points[i + 1];
        var p3 = (i !=  this._points.length - 2) ?  this._points[i + 2] : p2;

        var cp1x = p1.x + (p2.x - p0.x) / 6 * t;
        var cp1y = p1.y + (p2.y - p0.y) / 6 * t;

        var cp2x = p2.x - (p3.x - p1.x) / 6 * t;
        var cp2y = p2.y - (p3.y - p1.y) / 6 * t;

        var seg = new go.PathSegment(go.PathSegment.QuadraticBezier, p2.x, p2.y, cp1x, cp1y, cp2x, cp2y);
        segments.add(seg);
    }
  }
  else {
    _.each(this._points, q => {
      var seg = new go.PathSegment(go.PathSegment.Line, q.x, q.y);
      segments.add(seg);
    });
  }

  fig.segments = segments;
  shape.geometry = geo;

};
/**
* Start drawing the line by starting to accumulate points in the {@link #temporaryShape}'s geometry.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.doMouseDown = function() {
  if (!this.isActive) {
    this.doActivate();
    // the first point
    this.addPoint(this.diagram.lastInput.documentPoint);
  }
};

/**
* Keep accumulating points in the {@link #temporaryShape}'s geometry.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.doMouseMove = function() {
  if (this.isActive) {
    this.addPoint(this.diagram.lastInput.documentPoint);
  }
};

/**
* Finish drawing the line by adding a node data object holding the
* geometry string and the node position that the node template can bind to.
* This copies the {@link #archetypePartData} and adds it to the model.
* @this {FreehandDrawingTool}
*/
FreehandDrawingTool.prototype.doMouseUp = function() {
  var started = false;
  if (this.isActive) {
    started = true;
    var diagram = this.diagram;
    // the last point
    this.addPoint(diagram.lastInput.documentPoint);
    // normalize geometry and node position
    var viewpt = diagram.viewportBounds.position;
    var geo = this.temporaryShape.geometry.copy();
    var pos = geo.normalize();
    pos.x = viewpt.x - pos.x;
    pos.y = viewpt.y - pos.y;

    diagram.startTransaction(this.name);
    // create the node data for the model
    var d = diagram.model.copyNodeData(this.archetypePartData);
    // adding data to model creates the actual Part
    diagram.model.addNodeData(d);
    var part = diagram.findPartForData(d);
    // assign the location
    part.location = new go.Point(pos.x + geo.bounds.width/2, pos.y + geo.bounds.height/2);
    // assign the Shape.geometry
    var shape = part.findObject("SHAPE");
    if (shape !== null) shape.geometry = geo;
    this._points = [];
  }
  this.stopTool();
  if (started) diagram.commitTransaction(this.name);
};

// Public properties

/**
* Gets or sets the Shape that is used to hold the line as it is being drawn.
* The default value is a simple Shape drawing an unfilled open thin black line.
* @name FreehandDrawingTool#temporaryShape

* @return {Shape}
*/
Object.defineProperty(FreehandDrawingTool.prototype, "temporaryShape", {
  get: function() { return this._temporaryShape; },
  set: function(val) {
    if (this._temporaryShape !== val && val !== null) {
      val.name = "SHAPE";
      var panel = this._temporaryShape.panel;
      panel.remove(this._temporaryShape);
      this._temporaryShape = val;
      panel.add(this._temporaryShape);
    }
  }
});

/**
* Gets or sets the node data object that is copied and added to the model
* when the freehand drawing operation completes.
* @name FreehandDrawingTool#archetypePartData

* @return {Object}
*/
Object.defineProperty(FreehandDrawingTool.prototype, "archetypePartData", {
  get: function() { return this._archetypePartData; },
  set: function(val) { this._archetypePartData = val; }
});

/**
* Gets or sets whether this tool can only run if the user starts in the diagram's background
* rather than on top of an existing Part.
* The default value is true.
* @name FreehandDrawingTool#isBackgroundOnly

* @return {Object}
*/
Object.defineProperty(FreehandDrawingTool.prototype, "isBackgroundOnly", {
  get: function() { return this._isBackgroundOnly; },
  set: function(val) { this._isBackgroundOnly = val; }
});







///........................................................................
// private EPointF[] computeControlPoints(int n, List<EPointF> knots) {
//     final EPointF[] result = new EPointF[2 * n];

//     final EPointF[] target = constructTargetVector(n, knots);
//     final Float[] lowerDiag = constructLowerDiagonalVector(n - 1);
//     final Float[] mainDiag = constructMainDiagonalVector(n);
//     final Float[] upperDiag = constructUpperDiagonalVector(n - 1);

//     final EPointF[] newTarget = new EPointF[n];
//     final Float[] newUpperDiag = new Float[n - 1];

//     // forward sweep for control points c_i,0:
//     newUpperDiag[0] = upperDiag[0] / mainDiag[0];
//     newTarget[0] = target[0].scaleBy(1 / mainDiag[0]);

//     for (int i = 1; i < n - 1; i++) {
//       newUpperDiag[i] = upperDiag[i] /
//           (mainDiag[i] - lowerDiag[i - 1] * newUpperDiag[i - 1]);
//     }

//     for (int i = 1; i < n; i++) {
//       final float targetScale = 1 /
//           (mainDiag[i] - lowerDiag[i - 1] * newUpperDiag[i - 1]);

//       newTarget[i] =
//           (target[i].minus(newTarget[i - 1].scaleBy(lowerDiag[i - 1]))).scaleBy(targetScale);
//     }

//     // backward sweep for control points c_i,0:
//     result[n - 1] = newTarget[n - 1];

//     for (int i = n - 2; i >= 0; i--) {
//       result[i] = newTarget[i].minus(newUpperDiag[i], result[i + 1]);
//     }

//     // calculate remaining control points c_i,1 directly:
//     for (int i = 0; i < n - 1; i++) {
//       result[n + i] = knots.get(i + 1).scaleBy(2).minus(result[i + 1]);
//     }

//     result[2 * n - 1] = knots.get(n).plus(result[n - 1]).scaleBy(0.5f);

//     return result;
//   }

// function plus(a,b) {
//   return {x: (a.x+b.x),y:(a.y+b.y)};
// }

// function minus(a,b) {
//   return {x: (a.x-b.x),y:(a.y-b.y)};
// }

// function scaleBy(a, s) {
//   return {x: a.x * s, y: a.y * s};
// }

// function constructTargetVector( n, knots) {

//   var result = [];
//   result.push(plus(knots[0],knots[1]));

//   for (var i = 1; i < n - 1; i++) {
//     result.push(plus( scaleBy([i], 2) , scaleBy([i + 1], 2) ) );
//   }
//   result.push() = plus(scaleBy(knots[n - 1], 8), knots[n]);
//   return result;
// }

// function constructLowerDiagonalVector(length) {
//   var result = [];
//   for (var i = 0; i < length - 1; i++) {
//     length.push(1);
//   }
//   result.push(2);
//   return result;
// }

// function constructMainDiagonalVector(n) {
//   var result = [];
//   result.push(2);
//   for (var i = 1; i < n - 1; i++) {
//     result.push(4);
//   }
//   result.push(7);
//   return result;
// }

// function constructUpperDiagonalVector(length) {
//   var result = [];
//   for (var i = 0; i < length; i++) {
//     result.push(1);
//   }
//   return result;
// }