/*
 *  Copyright (C) 1998-2024 by Northwoods Software Corporation. All Rights Reserved.
 */
/*
 * This is an extension and not part of the main GoJS library.
 * Note that the API for this class may change with any version, even point releases.
 * If you intend to use an extension in production, you should copy the code to your own source directory.
 * Extensions can be found in the GoJS kit under the extensions or extensionsJSM folders.
 * See the Extensions intro page (https://gojs.net/latest/intro/extensions.html) for more information.
 */

/**
 * The DragCreatingTool lets the user create a new node by dragging in the background
 * to indicate its size and position.
 *
 * The default drag selection box is a magenta rectangle.
 * You can modify the {@link box} to customize its appearance.
 *
 * This tool will not be able to start running unless you have set the
 * {@link archetypeNodeData} property to an object that can be copied and added to the diagram's model.
 *
 * You can use this tool in a modal manner by executing:
 * ```js
 *   diagram.currentTool = new DragCreatingTool();
 * ```
 *
 * Use this tool in a mode-less manner by executing:
 * ```js
 *   myDiagram.toolManager.mouseMoveTools.insertAt(2, new DragCreatingTool());
 * ```
 *
 * However when used mode-lessly as a mouse-move tool, in {@link go.ToolManager.mouseMoveTools},
 * this cannot start running unless there has been a motionless delay
 * after the mouse-down event of at least {@link delay} milliseconds.
 *
 * This tool does not utilize any {@link go.Adornment}s or tool handles,
 * but it does temporarily add the {@link box} Part to the diagram.
 * This tool does conduct a transaction when inserting the new node.
 *
 * If you want to experiment with this extension, try the <a href="../../samples/DragCreating.html">Drag Creating</a> sample.
 * @category Tool Extension
 */
class DragCreatingTool extends go.Tool {
	/**
	 * Constructs a DragCreatingTool, sets {@link box} to a magenta rectangle, and sets name of the tool.
	 */
	constructor(init) {
		super();

		this.name = "DragCreating";

		this._archetypeNodeData = null;
		this.createLink = false;
		this.archetypeLinkData = null;

		var b = new go.Part();
		b.layerName = "Tool";
		b.selectable = false;
		var r = new go.Shape();
		r.name = "SHAPE";
		r.figure = "Rectangle";
		r.fill = null;
		r.stroke = "dodgerblue";
		r.strokeWidth = 2;
		r.position = new go.Point(0, 0);

		r.bind(new go.Binding("strokeWidth", "scale", function(scale) { return 2 / scale }));

		b.add(r);
		/** @type {Part} */
		this._box = b;
		this._tempLink = null;

		/** @type {number} */
		this._delay = 175;

		this.fromNode = null;
		this.toNode = null;

		this.fromPort = null;
		this.toPort = null;

		this._widthToHeightRatio = 0;

		if (init)
			Object.assign(this, init);
	}
	/**
	 * Gets or sets the {@link go.Part} used as the "rubber-band box"
	 * that is stretched to follow the mouse, as feedback for what area will
	 * be passed to {@link insertPart} upon a mouse-up.
	 *
	 * Initially this is a {@link go.Part} containing only a simple magenta rectangular {@link go.Shape}.
	 * The object to be resized should be named "SHAPE".
	 * Setting this property does not raise any events.
	 *
	 * Modifying this property while this tool {@link go.Tool.isActive} might have no effect.
	 */
	get box() {
		return this._box;
	}
	set box(val) {
		this._box = val;
	}
	/**
	 * Gets or sets the time in milliseconds for which the mouse must be stationary
	 * before this tool can be started.
	 *
	 * The default value is 175 milliseconds.
	 * A value of zero will allow this tool to run without any wait after the mouse down.
	 * Setting this property does not raise any events.
	 */
	get delay() {
		return this._delay;
	}
	set delay(val) {
		this._delay = val;
	}
	/**
	 * Gets or sets a data object that will be copied and added to the diagram's model each time this tool executes.
	 *
	 * The default value is null.
	 * The value must be non-null for this tool to be able to run.
	 * Setting this property does not raise any events.
	 */
	get archetypeNodeData() {
		return this._archetypeNodeData;
	}
	set archetypeNodeData(val) {
		this._archetypeNodeData = val;
	}
	/**
	 * This tool can run when there has been a mouse-drag, far enough away not to be a click,
	 * and there has been delay of at least {@link delay} milliseconds
	 * after the mouse-down before a mouse-move.
	 */
	canStart() {
		if (!this.isEnabled) return false;

		// gotta have some node data that can be copied
		if (this.archetypeNodeData === null && this.archetypeLinkData === null) return false;

		var diagram = this.diagram;
		if (diagram === null) return false;
		// heed IsReadOnly & AllowInsert
		if (diagram.isReadOnly || diagram.isModelReadOnly) return false;
		if (!diagram.allowInsert) return false;

		var e = diagram.lastInput;
		// var parts = diagram.findPartsNear(e, 20, true);
		//   if (parts != null && parts.count > 0) {
		//     var part = parts.first;
		//   }

		// require left button & that it has moved far enough away from the mouse down point, so it isn't a click
		if (!e.left) return false;
		// don't include the following checks when this tool is running modally
		if (diagram.currentTool !== this) {
			if (!this.isBeyondDragSize()) return false;
			// must wait for "delay" milliseconds before that tool can run
			if (e.timestamp - diagram.firstInput.timestamp < this.delay) return false;
		}
		return true;
	}
	/**
	 * Capture the mouse and show the {@link box}.
	 */
	doActivate() {
		var diagram = this.diagram;
		if (diagram === null) return;
		this.isActive = true;
		diagram.isMouseCaptured = true;

		if (!this.createLink) {
			diagram.add(this.box);
		}
		else {

			var start = diagram.firstInput.documentPoint;
			var latest = diagram.lastInput.documentPoint;

			if (diagram.toolManager.linkingTool.highlightedNode != null) {
				diagram.isMouseCaptured = false;
				startLinkTool(diagram.toolManager.linkingTool.highlightedNode);
				return;
			}

			this.startTransaction(this.name);

			var linkTemplate = this.archetypeLinkData.category == "Arrow" ? getLinkTemplateArrow() : getLinkTemplateSimple();
			this._tempLink = linkTemplate.copy();

			diagram.commit(() => {
				diagram.add(this._tempLink);
			}, null);

			// set the TransactionResult before raising event, in case it changes the result or cancels the tool
			this.transactionResult = this.name;
			this.stopTransaction();
		}
		this.doMouseMove();
		showNodeEditingMenu();
	}
	/**
	 * Release the mouse and remove any {@link box}.
	 */
	doDeactivate() {
		var diagram = this.diagram;
		if (diagram === null) return;

		if (!this.createLink) {
			if (this.box != null) {
				diagram.commit(() => {
					diagram.remove(this.box);
				}, null);
			}
		}
		else if (this._tempLink != null) {

			var _this = this;
			diagram.commit(() => {
				if (_this._tempLink != null) {
					diagram.remove(_this._tempLink);
					_this._tempLink = null;
				}
			}, null);
			this.actionCompleted();
		}

		diagram.isMouseCaptured = false;
		this.isActive = false;

		showNodeEditingMenu();
	}
	/**
	 * Update the {@link box}'s position and size according to the value
	 * of {@link computeBoxBounds}.
	 */
	doMouseMove() {
		var diagram = this.diagram;
		if (diagram === null) return;

		// var start = diagram.lastInput.documentPoint;
		// console.log("DragCreatingTool: "  + start);

		if (this.isActive) {
			if (this.createLink) {

				diagram.startTransaction(null);
				if (this._tempLink != null) {

					var start = diagram.firstInput.documentPoint;
					var latest = diagram.lastInput.documentPoint;
					var points = new go.List(/*go.Point*/).addAll([start.copy(), latest.copy()]);
					this._tempLink.points = points;

					//...........................................
					var node = null;
					var port = null;

					var latest = diagram.lastInput.documentPoint;
					var parts = diagram.findPartsNear(latest, 20, true);
					if (parts != null && parts.count > 0) {
						if (this.toNode != null) {
							showPorts(this.toNode, false);
						}

						parts.each(function(p) {
							if (p.type && p.type.name != "Link") {
								node = p;
							}
						});
						port = null;
						if (node != null && node.ports != null && node.ports.count > 0) {
							node.ports.each(function(p) {
								var documentBound = p.getDocumentBounds();
								var newRect = new go.Rect(documentBound.x - 20, documentBound.y - 20, documentBound.width + 40, documentBound.height + 40);
								if (newRect.containsPoint(latest)) {
									port = p;
								}
							});
						} else {
							node = null;
						}
					}
					this.toNode = node;
					this.toPort = port;

					if (this.toNode != null) {
						showPorts(this.toNode, true);
					}

					this._tempLink.toNode = this.toNode;
					// this._tempLink.toPort = this.toPort;
					//....................................................

				}
				diagram.commitTransaction(null);

			} else if (this.box !== null) {
				var r = this.computeBoxBounds();
				var shape = this.box.findObject("SHAPE");
				if (shape === null) shape = this.box.findMainElement();
				shape.desiredSize = r.size;
				this.box.position = r.position;
			}
		} else {

		}
	}
	/**
	 * Call {@link insertPart} with the value of a call to {@link computeBoxBounds}.
	 */
	doMouseUp() {
		if (this.isActive) {
			var diagram = this.diagram;

			if (!this.createLink) {

				if (this.box != null) diagram.remove(this.box);

				try {
					diagram.currentCursor = "wait";
					this.insertPart(this.computeBoxBounds());
				} finally {
					diagram.currentCursor = "";
					this.actionCompleted();
				}
			}
			else {

				if (this.toNode != null) {
					showPorts(this.toNode, false);
					this.toNode = null;
				}

				if (this._tempLink != null) {
					try {
						diagram.currentCursor = "wait";
						this.insertLink();
						var _this = this;
						diagram.commit(() => {
							if (_this._tempLink != null) {
								diagram.remove(_this._tempLink);
							}
							_this._tempLink = null;
						}, null);

					} finally {
						diagram.currentCursor = "";
						this.actionCompleted();
					}
				}
			}
		}
		this.stopTool();
	}
	/**
	 * This just returns a {@link go.Rect} stretching from the mouse-down point to the current mouse point.
	 * @returns a {@link go.Rect} in document coordinates.
	 */
	computeBoxBounds() {
		var diagram = this.diagram;
		if (diagram === null) return new go.Rect(0, 0, 0, 0);
		var start = diagram.firstInput.documentPoint.copy();
		var latest = diagram.lastInput.documentPoint.copy();

		if (this._widthToHeightRatio == 0) {
			if (this.archetypeNodeData.category && this.archetypeNodeData.category === "TEXT") {
				latest.y = start.y + 40;
			}
			return new go.Rect(start, latest);
		}
		else {

			var rect = new go.Rect(start, latest);
			var newHeight = rect.width / this._widthToHeightRatio;

			var lastPoint = start.copy();
			lastPoint.x = start.x + rect.width;
			lastPoint.y = start.y + newHeight;

			return new go.Rect(start, lastPoint);
		}
	}
	/**
	 * Create a node by adding a copy of the {@link archetypeNodeData} object
	 * to the diagram's model, assign its {@link go.GraphObject.position} and {@link go.GraphObject.desiredSize}
	 * according to the given bounds, and select the new part.
	 *
	 * The actual part that is added to the diagram may be a {@link go.Part}, a {@link go.Node},
	 * or even a {@link go.Group}, depending on the properties of the {@link archetypeNodeData}
	 * and the type of the template that is copied to create the part.
	 * @param bounds - a Point in document coordinates.
	 * @returns the newly created Part, or null if it failed.
	 */
	insertPart(bounds) {
		var diagram = this.diagram;
		if (diagram === null) return null;
		var arch = this.archetypeNodeData;
		if (arch === null) return null;

		diagram.raiseDiagramEvent("ChangingSelection", diagram.selection);
		this.startTransaction(this.name);
		var part = null;
		if (arch !== null) {
			var data = diagram.model.copyNodeData(arch);
			if (data) {
				data.scale = 1;
				if (data.category === "TEXT") {
					data.size = go.Size.stringify(new go.Size(bounds.size.width, NaN));
				}
				else if (data.category === "NOTE") {

					var noteWidthHeight = 200;
					var scale = bounds.size.width / noteWidthHeight;
					data.size = go.Size.stringify(new go.Size(noteWidthHeight, noteWidthHeight));
					data.scale = scale;
				}
				else {
					data.size = go.Size.stringify(bounds.size);
				}
				diagram.model.addNodeData(data);
				part = diagram.findPartForData(data);
			}
		}
		if (part !== null) {
			part.position = bounds.position;
			if (part.data) {
				vueGoInstance.updateArcheTypeNodeData(part.data.category, "size", part.resizeObject.desiredSize);
			}
			if (diagram.allowSelect) {
				diagram.clearSelection();
				part.isSelected = true;
			}
		}

		// set the TransactionResult before raising event, in case it changes the result or cancels the tool
		this.transactionResult = this.name;
		this.stopTransaction();
		diagram.raiseDiagramEvent("ChangedSelection", diagram.selection);

		if (part !== null) {
			onNodeCreated(part);
		}

		return part;
	}

	insertLink(bounds) {
		var diagram = this.diagram;
		if (diagram === null) return null;
		var arch = this.archetypeLinkData;
		if (arch === null) return null;

		diagram.raiseDiagramEvent("ChangingSelection", diagram.selection);
		this.startTransaction(this.name);
		var link = null;

		var data = diagram.model.copyLinkData(arch);
		if (data) {
			data.points = this._tempLink.points;
			diagram.model.addLinkData(data);
			link = diagram.findLinkForData(data);
		}

		// set the TransactionResult before raising event, in case it changes the result or cancels the tool
		this.transactionResult = this.name;
		this.stopTransaction();
		diagram.raiseDiagramEvent("ChangedSelection", diagram.selection);
		return link;
	}

	mouseHover() {

	}

	selectFromNode() {

	}

	selectToNode() {

	}

	selectFromPort() {

	}

	selectToPort() {

	}

	actionCompleted() {
		this.toNode = null;
		this.toPort = null;
	}
}
