const EXPLORE_COLLABORATION_TEMPLATES = Vue.component('explore-collaboration-templates', {
    name: 'explore-collaboration-templates',
    template: `
        <div class="create-board-overlay">
            <div class="create-board-modal explore-templates-modal">
                <!-- Close Button -->
                <div class="head-modal DFCB">
                	<div class="lft DFA ftsb fs-18">Collaboration Boards</div>
                	<div class="rht DFA">
                        <div class="search-input-container mr-20">
                            <input v-model="searchText" type="text" placeholder="Search templates..." class="search-input rnd4" @input="onSearchInput" />
                            <span class="search-icon">
                            <svg width="26" height="26" viewBox="0 0 26 26" fill="none" xmlns="http://www.w3.org/2000/svg">
								<path d="M0.5 0.5L0.5 9L0.5 25.5" stroke="#97A7B8" stroke-linecap="round"/>
								<path fill-rule="evenodd" clip-rule="evenodd" d="M25.7555 19.5774L21.499 15.3199C22.229 14.3458 22.6665 13.1408 22.6665 11.8333C22.6665 8.61665 20.0491 6 16.8332 6C13.6174 6 11 8.6166 11 11.8333C11 15.05 13.6174 17.6666 16.8332 17.6666C18.1407 17.6666 19.3465 17.2283 20.3207 16.4983L24.5773 20.7558C24.7398 20.9183 24.9532 21 25.1665 21C25.3797 21 25.5931 20.9183 25.7556 20.7558C26.0815 20.43 26.0814 19.9032 25.7555 19.5774ZM16.8332 16.0001C14.5358 16.0001 12.6668 14.131 12.6668 11.8337C12.6668 9.53624 14.5359 7.66725 16.8332 7.66725C19.1306 7.66725 20.9996 9.53635 20.9996 11.8337C20.9996 14.1311 19.1305 16.0001 16.8332 16.0001Z" fill="#495961"/>
							</svg>
							</span>
                        </div>
                		<a href="javascript:void(0);" class="close-btn" @click="$emit('close')">
                			<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
							<path d="M13.9994 1.184C13.997 1.4865 13.8646 1.80554 13.6519 2.02059L13.633 2.0395C13.4817 2.19311 13.3305 2.342 13.1792 2.49325L8.6719 7L13.6519 11.9794C13.8765 12.2039 13.9852 12.5017 13.9994 12.816C14.0112 13.1327 13.8575 13.4257 13.6519 13.6526C13.4534 13.8724 13.1059 14 12.8152 14C12.5127 13.9976 12.1936 13.8653 11.9785 13.6526L11.9596 13.6337C11.806 13.4824 11.6571 13.3312 11.5058 13.1799L7.00086 8.67083L2.02084 13.6502C1.79394 13.8771 1.4985 13.9835 1.18414 13.9976C0.867426 14.0095 0.574345 13.8558 0.347443 13.6502C0.127632 13.4517 0 13.1043 0 12.8136C0.00236356 12.5111 0.134723 12.1921 0.347443 11.977L0.366352 11.9581C0.517619 11.8045 0.668887 11.6556 0.820155 11.5044L5.32983 7L0.349807 2.02059C0.122905 1.79372 0.0165447 1.49831 0.00236334 1.184C-0.00945445 0.867319 0.144177 0.574274 0.349807 0.3474C0.548346 0.127616 0.895789 0 1.18651 0C1.48904 0.00236327 1.80812 0.134706 2.02321 0.3474L2.04212 0.366307C2.19575 0.517556 2.34465 0.668805 2.49592 0.820054L7.00086 5.32917L11.9809 0.349764C12.2054 0.125253 12.5032 0.0165427 12.8176 0.00236305C13.1343 -0.00945329 13.4274 0.144159 13.6543 0.349764C13.8717 0.548278 13.9994 0.893315 13.9994 1.184Z" fill="#495961"/>
							</svg>
                		</a>
                	</div>
                </div>
                
                <div class="explore-templates-container">
                    <!-- Left Navigation Sidebar -->
                    <div class="explore-templates-sidebar DF FC">
                       
                        <ul class="template-types-list">
                            <!-- All Templates Tab - Always at the top -->
                            <li :class="['template-type-item', { 'selected': currentSelectedType && currentSelectedType.id === 'allTemplates' }]"
                                @click="selectAllTemplates">
                                <span class="type-icon">
                                    <svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M0.192212 3.6958C0.0138889 3.62079 0 3.45444 0 3.40527C0 3.3561 0.0138889 3.18975 0.192212 3.11488L7.23632 0.153996C7.4795 0.05179 7.73644 0 7.99997 0C8.26349 0 8.52044 0.05179 8.76362 0.153996L15.8077 3.11488C15.986 3.18988 15.9999 3.35611 15.9999 3.40527C15.9999 3.45444 15.986 3.62079 15.8077 3.69567L8.76362 6.65655C8.52044 6.75876 8.26349 6.81067 7.99997 6.81067C7.73644 6.81067 7.47949 6.75888 7.23632 6.65667L0.192212 3.6958ZM15.8077 8.3042C15.986 8.37921 15.9999 8.54556 15.9999 8.59473C15.9999 8.6439 15.986 8.81025 15.8077 8.88512L8.76362 11.846C8.52044 11.9482 8.26349 12 7.99997 12C7.73644 12 7.47949 11.9482 7.23632 11.846L0.192212 8.88512C0.0138889 8.81024 0 8.64389 0 8.59473C0 8.54556 0.0138889 8.37921 0.192212 8.30433L2.07367 7.51351L7.08349 9.61928C7.37677 9.74246 7.68829 9.80411 7.99991 9.80411C8.31152 9.80411 8.62305 9.74246 8.91619 9.61928L13.9263 7.51338L15.8077 8.3042ZM15.8077 6.2904L8.76362 9.25128C8.52044 9.35349 8.26349 9.40528 7.99997 9.40528C7.73644 9.40528 7.47949 9.35349 7.23632 9.25128L0.192212 6.2904C0.0138889 6.21552 0 6.04917 0 6.00001C0 5.95084 0.0138889 5.78449 0.192212 5.70961L2.07367 4.91879L7.08349 7.02456C7.37677 7.14774 7.68829 7.20939 7.99991 7.20939C8.31152 7.20939 8.62305 7.14774 8.91619 7.02456L13.9263 4.91879L15.8078 5.70961C15.9861 5.78449 16 5.95084 16 6.00001C16 6.04917 15.986 6.21553 15.8077 6.2904Z" fill="#7B889B"/>
                                    </svg>
                                </span>
                                <span class="type-name">All Templates</span>
                                <span class="type-count">{{ getAllTemplatesCount() }}</span>
                            </li>
                            
                            <template v-for="type in allTypes">
                                <li :key="type.id" :class="['template-type-item', { 'selected': currentSelectedType && currentSelectedType.id === type.id }]"
                                    @click="selectType(type)">
                                    <span class="type-icon">
                                        <svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M0.192212 3.6958C0.0138889 3.62079 0 3.45444 0 3.40527C0 3.3561 0.0138889 3.18975 0.192212 3.11488L7.23632 0.153996C7.4795 0.05179 7.73644 0 7.99997 0C8.26349 0 8.52044 0.05179 8.76362 0.153996L15.8077 3.11488C15.986 3.18988 15.9999 3.35611 15.9999 3.40527C15.9999 3.45444 15.986 3.62079 15.8077 3.69567L8.76362 6.65655C8.52044 6.75876 8.26349 6.81067 7.99997 6.81067C7.73644 6.81067 7.47949 6.75888 7.23632 6.65667L0.192212 3.6958ZM15.8077 8.3042C15.986 8.37921 15.9999 8.54556 15.9999 8.59473C15.9999 8.6439 15.986 8.81025 15.8077 8.88512L8.76362 11.846C8.52044 11.9482 8.26349 12 7.99997 12C7.73644 12 7.47949 11.9482 7.23632 11.846L0.192212 8.88512C0.0138889 8.81024 0 8.64389 0 8.59473C0 8.54556 0.0138889 8.37921 0.192212 8.30433L2.07367 7.51351L7.08349 9.61928C7.37677 9.74246 7.68829 9.80411 7.99991 9.80411C8.31152 9.80411 8.62305 9.74246 8.91619 9.61928L13.9263 7.51338L15.8077 8.3042ZM15.8077 6.2904L8.76362 9.25128C8.52044 9.35349 8.26349 9.40528 7.99997 9.40528C7.73644 9.40528 7.47949 9.35349 7.23632 9.25128L0.192212 6.2904C0.0138889 6.21552 0 6.04917 0 6.00001C0 5.95084 0.0138889 5.78449 0.192212 5.70961L2.07367 4.91879L7.08349 7.02456C7.37677 7.14774 7.68829 7.20939 7.99991 7.20939C8.31152 7.20939 8.62305 7.14774 8.91619 7.02456L13.9263 4.91879L15.8078 5.70961C15.9861 5.78449 16 5.95084 16 6.00001C16 6.04917 15.986 6.21553 15.8077 6.2904Z" fill="#7B889B"/>
                                        </svg>
                                    </span>
                                    <span class="type-name">
                                      {{
                                        type?.defaultOrigin
                                          ? type.title.charAt(0).toUpperCase() + type.title.slice(1)
                                          : type.title
                                      }}
                                    </span>
                                    <span class="type-count">{{ getTypeCount(type) }}</span>
                                    <!-- Dropdown indicator for selectTeams -->
                                    <span v-if="type.id === 'selectTeams'" class="dropdown-indicator" 
                                          :class="{ 'expanded': showTeamsDropdown }"
                                          @click.stop="toggleTeamsDropdown">
                                        <svg width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M3 4.5L6 7.5L9 4.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                        </svg>
                                    </span>
                                    <!-- Dropdown indicator for selectSessions -->
                                    <span v-if="type.id === 'selectSessions'" class="dropdown-indicator" 
                                          :class="{ 'expanded': showSessionsDropdown }"
                                          @click.stop="toggleSessionsDropdown">
                                        <svg width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M3 4.5L6 7.5L9 4.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                        </svg>
                                    </span>
                                    <!-- Dropdown indicator for selectSolutionBoards -->
                                    <span v-if="type.id === 'selectSolutionBoards'" class="dropdown-indicator" 
                                          :class="{ 'expanded': showSolutionDropdown }"
                                          @click.stop="toggleSolutionDropdown">
                                        <svg width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M3 4.5L6 7.5L9 4.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                        </svg>
                                    </span>
                                </li>
                                
                                <!-- Teams dropdown - positioned right after the Teams item -->
                                <li v-if="type.id === 'selectTeams' && showTeamsDropdown" class="teams-dropdown">
                                    <ul class="teams-sublist">
                                        <li v-if="sessionTeams && sessionTeams.length > 0" v-for="team in sessionTeams" :key="team.id" 
                                            :class="['team-item', { 'selected': currentSelectedType && currentSelectedType.id === team.id }]"
                                            @click="selectTeam(team)">
                                            <span class="team-name">{{ team.title }}</span>
                                            <span class="type-count">{{ getTeamCount(team.id) }}</span>
                                        </li>
                                        <li v-else class="no-teams-message">
                                            <span class="no-teams-text">No teams available</span>
                                        </li>
                                    </ul>
                                </li>
                                
                                <!-- Sessions dropdown - positioned right after the Program Boards item -->
                                <li v-if="type.id === 'selectSessions' && showSessionsDropdown" class="teams-dropdown">
                                    <ul class="teams-sublist">
                                        <li v-if="solutionSessions && solutionSessions.length > 0" v-for="session in solutionSessions" :key="session.id" 
                                            :class="['team-item', { 'selected': currentSelectedType && currentSelectedType.id === session.id }]"
                                            @click="selectSession(session)">
                                            <span class="team-icon">
                                                <svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="7" cy="7" r="6" fill="#E6F4F1"/>
                                                    <path d="M5 7L6.5 8.5L9 6" stroke="#1E9984" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                                </svg>
                                            </span>
                                            <span class="team-name">{{ session.title }}</span>
                                            <span class="type-count">{{ getSessionCount(session.id) }}</span>
                                        </li>
                                        <li v-else class="no-teams-message">
                                            <span class="no-teams-text">No sessions available</span>
                                        </li>
                                    </ul>
                                </li>
                                
                                <!-- Solution Boards dropdown - positioned right after the Solution Boards item -->
                                <li v-if="type.id === 'selectSolutionBoards' && showSolutionDropdown" class="teams-dropdown">
                                    <ul class="teams-sublist">
                                        <li v-if="workspaceSolutionBoards && workspaceSolutionBoards.length > 0" v-for="solution in workspaceSolutionBoards" :key="solution.id" 
                                            :class="['team-item', { 'selected': currentSelectedType && currentSelectedType.id === solution.id }]"
                                            @click="selectSolutionBoard(solution)">
                                            <span class="team-icon">
                                                <svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="7" cy="7" r="6" fill="#E6F4F1"/>
                                                    <path d="M5 7L6.5 8.5L9 6" stroke="#1E9984" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                                </svg>
                                            </span>
                                            <span class="team-name">{{ solution.title }}</span>
                                            <span class="type-count">{{ getSolutionBoardCount(solution.id) }}</span>
                                        </li>
                                        <li v-else class="no-teams-message">
                                            <span class="no-teams-text">No solution boards available</span>
                                        </li>
                                    </ul>
                                </li>
                            </template>
                        </ul>
                        
                      
                    </div>
                    
                    <!-- Main Content Area -->
                    <div class="explore-templates-main DF FC">
                        <div v-if="loading" class="loading-container">                            
                            <div class="loaderinteg"></div>
                            <span class="LOADER-TITLE">Loading templates...</span>                           
                        </div>
                        
                        <div v-else class="templates-content">
                            <!-- Templates List -->
                            <div class="templates-list">
                                <!-- Template Items -->
                                <div v-for="template in filteredTemplatesByType" :key="template.id" 
                                     :class="['template-card', { 'selected': selectedTemplate && selectedTemplate.id === template.id }]"
                                     @click="useTemplate(template)">
                                    <div class="template-preview">
                                        <img v-if="template.title == 'Planning Retro Template'" src="../assets/imgs/CollaborationBoard-Template.png" alt="Template Preview" />
                                        <img v-else :src="thumbnailPath(template)" alt="Template Preview" />
                                    </div>
                                    <div class="template-info ftsb">{{ template.title }}</div>
                                    <div class="template-actions trnsn">
                                        <div v-if="template.title != 'Planning Retro Template' && permissions && permissions.MANAGE_TEMPLATES" class="actions-dropdown p-abs">
                                            <a href="javascript:void(0);" class="actions-menu-btn DAJ ti-more-alt" @click="onClickMenuButton(template, $event)"></a>
                                            <vuepopup v-if="templateActionMenu==template.id" class="drop_box nwmu sml artp actions-menu" style="display: block;" @close="templateActionMenu='';">
                                                <ul>
                                                    <li @click="onClickCopySharableUrl(template, $event)">Copy Link</li>
                                                    <li @click="onClickEditTemplate(template, $event)">Edit</li>
                                                    <li @click="onClickDeleteTemplate(template, $event)">Delete</li>
                                                </ul>
                                            </vuepopup>
                                        </div>
                                        <div class="txt">Structure your team's sprint planning retro...</div>
                                        <div class="btn DFA gap-10px">
                                            <button v-if="permissions && permissions.CREATE_COLLAB_BOARD" class="button-w whtbtn h-auto rnd4" @click.stop="useTemplate(template)">Use</button>
                                            <button class="button-w clrbtn h-auto rnd4" @click.stop="$emit('preview-template', template)">Preview</button>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- No Templates Available Message (when count is 0 and no search) -->
                                <div v-if="!searchQuery.trim() && filteredTemplatesByType.length === 0" class="no-results-message">
                                    <div class="no-results-icon">
                                        <svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle opacity="0.1" cx="24" cy="24" r="24" fill="#6B7280"/>
                                            <path d="M24 16C24.5523 16 25 16.4477 25 17V23H31C31.5523 23 32 23.4477 32 24C32 24.5523 31.5523 25 31 25H25V31C25 31.5523 24.5523 32 24 32C23.4477 32 23 31.5523 23 31V25H17C16.4477 25 16 24.5523 16 24C16 23.4477 16.4477 23 17 23H23V17C23 16.4477 23.4477 16 24 16Z" fill="#6B7280"/>
                                        </svg>
                                    </div>
                                    <div class="no-results-text">
                                        <h3>No templates available</h3>
                                        <p>There are no templates available for this selection</p>
                                    </div>
                                </div>
                                
                                <!-- No Results Message (when search has no matches) -->
                                <div v-if="searchQuery.trim() && filteredTemplatesByType.length === 0" class="no-results-message">
                                    <div class="no-results-icon">
                                        <svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle opacity="0.1" cx="24" cy="24" r="24" fill="#6B7280"/>
                                            <path d="M24 16C24.5523 16 25 16.4477 25 17V23H31C31.5523 23 32 23.4477 32 24C32 24.5523 31.5523 25 31 25H25V31C25 31.5523 24.5523 32 24 32C23.4477 32 23 31.5523 23 31V25H17C16.4477 25 16 24.5523 16 24C16 23.4477 16.4477 23 17 23H23V17C23 16.4477 23.4477 16 24 16Z" fill="#6B7280"/>
                                        </svg>
                                    </div>
                                    <div class="no-results-text">
                                        <h3>No templates found</h3>
                                        <p>No templates match your search for "{{ searchQuery }}"</p>
                                        <button class="clear-search-btn DIB rnd4" @click="clearSearch">Clear search</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `,

    props: [
        'sessionId',
        'contextType', 
        'workspaceId',
        'solutionId',
        'batchId',
        'objectiveCustomViewId',
        'viewType',
        'templates',
        'templatesLoaded',
        'contextTypes',
        'selectedType',
        'collectionAssociatedProgramBoards',
        'collectionAssociatedSolBoards',
        'permissions',
        'isShowAllTemplates'
    ],

    data() {
        return {
            error: null,
            selectedTemplate: null,
            currentSelectedType: null,
            searchQuery: '',
            loading: false,
            localTemplatesList: [],
            searchText: "",
            searchDebounceTimeout: null,
            showTeams: false,
            showSessions: false,
            showWorkspaceSolutionBoards: false,
            allTypes: [],
            sessionTeams: [],
            solutionSessions: [],
            workspaceSolutionBoards: [],
            showTeamsDropdown: false,
            showSessionsDropdown: false,
            showSolutionDropdown: false,
            contextBoard: null,
            templateActionMenu: '',
            templatesCount: {},
            systemTemplates: []
        }
    },

    computed: {

        // Filter templates by search query
        filteredTemplatesByType() {
            let templates = this.localTemplatesList;

            // Filter by search query
            if (this.searchQuery.trim()) {
                const query = this.searchQuery.toLowerCase();
                templates = templates.filter(template => 
                    template.title.toLowerCase().includes(query) ||
                    (template.description && template.description.toLowerCase().includes(query))
                );
            }
            
            return templates;
        },

        // Thumbnail path helper
        thumbnailPath() {
            return function(template) {
                if (template.thumbnailPath) {
                    return "/upload/retrieve-image/" + template.thumbnailPath;
                }
                return "";
            };
        }
    },

    mounted() {
        // Reset component state when mounted
        this.resetComponentState();
        this.allTypes = this.contextTypes;
        if(this.contextType === "session"){
            this.showTeams = true;
            this.populateTeamsList(this.sessionId);
        }else if(this.contextType === "solution"){
            this.showSessions = true;
            this.populateSessionsList(this.solutionId);
        }else if(this.viewType){
            this.showSessions = true; // For PI boards
            this.showWorkspaceSolutionBoards = true; // For Solution boards
            this.populateWorkspaceBoardsFromProps();
        }else{
            this.reorderTypesForSession();
        }

        // Auto-select first type and fetch templates when component mounts
        // This should happen after allTypes is set up
        // this.$nextTick(() => {
        //     this.autoSelectFirstType();
        // });
    },

    created(){

    },

    beforeDestroy() {
        // Clean up any pending timeouts
        if (this.searchDebounceTimeout) {
            clearTimeout(this.searchDebounceTimeout);
        }
    },

    watch: {
        contextTypes: {
            handler(newVal) {
                // When contextTypes prop changes, auto-select first type if none is selected
                if (newVal && newVal.length > 0 && !this.currentSelectedType) {
                    this.autoSelectFirstType();
                }
            },
            immediate: true
        },
        selectedType: {
            handler(newVal) {
                // When selectedType prop changes, use it to set the selected type
                if (newVal && newVal.id && this.allTypes && this.allTypes.length > 0) {
                    // For dropdown types (team, piBoard, solutionBoard), set directly
                    if (newVal.type === "team" || newVal.type === "piBoard" || newVal.type === "solutionBoard") {
                        this.currentSelectedType = newVal;
                        this.fetchTemplates();
                    } else {
                        // For regular types (general, inspect, custom types), find matching type
                        const matchingType = this.allTypes.find(type => type.id === newVal.id);
                        if (matchingType) {
                            this.currentSelectedType = matchingType;
                            this.fetchTemplates();
                        }
                    }
                }
            },
            immediate: true
        },
        currentSelectedType: {
            handler(newVal) {
                // Automatically expand dropdowns based on the type of currentSelectedType
                if (newVal && newVal.type) {
                    if (newVal.type === "team") {
                        this.showTeamsDropdown = true;
                        this.showSessionsDropdown = false;
                        this.showSolutionDropdown = false;
                    } else if (newVal.type === "piBoard") {
                        this.showSessionsDropdown = true;
                        this.showTeamsDropdown = false;
                        this.showSolutionDropdown = false;
                    } else if (newVal.type === "solutionBoard") {
                        this.showSolutionDropdown = true;
                        this.showTeamsDropdown = false;
                        this.showSessionsDropdown = false;
                    } else {
                        // For regular types (general, inspect, custom types), close all dropdowns
                        this.showTeamsDropdown = false;
                        this.showSessionsDropdown = false;
                        this.showSolutionDropdown = false;
                    }
                }
            },
            deep: true
        }
    },

    methods: {

        resetComponentState() {
            // Reset all component state to prevent duplicate data
            this.selectedTemplate = null;
            this.currentSelectedType = null;
            this.searchQuery = '';
            this.searchText = "";
            this.showTeamsDropdown = false;
            this.showSessionsDropdown = false;
            this.showSolutionDropdown = false;
            this.allTypes = [];
            this.sessionTeams = [];
            this.solutionSessions = [];
            this.workspaceSolutionBoards = [];
            this.contextBoard = null;
            this.localTemplatesList = [];
            
            // Clear any existing debounce timeout
            if (this.searchDebounceTimeout) {
                clearTimeout(this.searchDebounceTimeout);
                this.searchDebounceTimeout = null;
            }
        },

        populateTeamsList(id){
            console.log('populateTeamsList called with id:', id);
            axios.get(`/plan/getBoard/${id}`)
                .then(response => {
                    console.log('populateTeamsList response:', response.data);
                    if (response.data.board) {
                        this.contextBoard = response.data.board
                        this.sessionTeams = this.contextBoard?.teams || [];
                        this.reorderTypesForSession();
                        this.autoSelectFirstType();
                    }
                })
                .catch(error => {
                    console.error('Error fetching board data:', error);
                })
        },

        populateSessionsList(id){
            axios.get(`/solution/sessions/${id}`)
                .then(response => {
                    if (response.data && response.data.success) {
                        this.solutionSessions = response.data.sessions || [];
                        this.reorderTypesForSession();
                        this.autoSelectFirstType();
                    }
                })
                .catch(error => {
                    console.error('Error fetching sessions data:', error);
                })
        },

        populateWorkspaceBoardsFromProps(){
            // Use data from parent props instead of making API call
            this.solutionSessions = this.collectionAssociatedProgramBoards || [];
            this.workspaceSolutionBoards = this.collectionAssociatedSolBoards || [];
            this.reorderTypesForWorkspace();
            this.autoSelectFirstType();
        },

        reorderTypesForSession() {
            const orderedTypes = [];

            const generalType = this.allTypes.find(type =>
                ["session", "solution", "workspace"].includes(type.defaultOrigin));
            if (generalType) {
                orderedTypes.push(generalType);
            }

            const inspectType = this.allTypes.find(type => type.defaultOrigin === "sessionInspect");
            if (inspectType) {
                orderedTypes.push(inspectType);
            }

            const selectTeamsExists = this.allTypes.some(type => type.id === "selectTeams");
            if (!selectTeamsExists && this.contextBoard?.teams && this.contextBoard.teams.length > 0) {
                orderedTypes.push({
                    id: "selectTeams",
                    title: this.contextBoard.session?.teamsLabel || "Teams",
                    rowStatus: 0
                });
            } else {
                const selectTeamsType = this.allTypes.find(type => type.id === "selectTeams");
                if (selectTeamsType) {
                    orderedTypes.push(selectTeamsType);
                }
            }

            // Handle selectSessions for solution context
            const selectSessionsExists = this.allTypes.some(type => type.id === "selectSessions");
            if (!selectSessionsExists && this.solutionSessions && this.solutionSessions.length > 0) {
                orderedTypes.push({
                    id: "selectSessions",
                    title: "Program Boards",
                    rowStatus: 0
                });
            } else {
                const selectSessionsType = this.allTypes.find(type => type.id === "selectSessions");
                if (selectSessionsType) {
                    orderedTypes.push(selectSessionsType);
                }
            }

            // 4. Custom types (all other types that are not general, inspect, or selectTeams)
            const customTypes = this.allTypes.filter(type =>
                !type.defaultOrigin);
            orderedTypes.push(...customTypes);
            this.allTypes = orderedTypes;
            // this.autoSelectFirstType();
        },

        reorderTypesForWorkspace() {
            const orderedTypes = [];

            const generalType = this.allTypes.find(type =>
                ["session", "solution", "workspace"].includes(type.defaultOrigin));
            if (generalType) {
                orderedTypes.push(generalType);
            }

            // Add selectSessions if we have PI boards (sessions)
            const selectSessionsExists = this.allTypes.some(type => type.id === "selectSessions");
            if (!selectSessionsExists && this.solutionSessions && this.solutionSessions.length > 0) {
                orderedTypes.push({
                    id: "selectSessions",
                    title: "Program Boards",
                    rowStatus: 0
                });
            } else {
                const selectSessionsType = this.allTypes.find(type => type.id === "selectSessions");
                if (selectSessionsType) {
                    orderedTypes.push(selectSessionsType);
                }
            }

            // Add selectSolutionBoards if we have solution boards
            const selectSolutionBoardsExists = this.allTypes.some(type => type.id === "selectSolutionBoards");
            if (!selectSolutionBoardsExists && this.workspaceSolutionBoards && this.workspaceSolutionBoards.length > 0) {
                orderedTypes.push({
                    id: "selectSolutionBoards",
                    title: "Solution Boards",
                    rowStatus: 0
                });
            } else {
                const selectSolutionBoardsType = this.allTypes.find(type => type.id === "selectSolutionBoards");
                if (selectSolutionBoardsType) {
                    orderedTypes.push(selectSolutionBoardsType);
                }
            }

            // Add custom types (all other types that are not general, inspect, or workspace types)
            const customTypes = this.allTypes.filter(type =>
                !type.defaultOrigin && type.id !== "selectSessions" && type.id !== "selectSolutionBoards");
            orderedTypes.push(...customTypes);
            
            // Update allTypes with the ordered sequence
            this.allTypes = orderedTypes;
        },

        autoSelectFirstType() {
            // Wait for contextTypes to be available, then select first type
            this.$nextTick(() => {
                if(this.isShowAllTemplates) {
                    this.currentSelectedType = this.selectedType;
                    this.fetchTemplates();
                }else{
                    this.selectAllTemplates();
                }
            });
        },

        onSearchInput() {
            if (this.searchDebounceTimeout) {
                clearTimeout(this.searchDebounceTimeout);
            }

            this.searchDebounceTimeout = setTimeout(() => {
                this.searchQuery = this.searchText;
            }, 300);
        },

        selectTemplate(template) {
            this.selectedTemplate = template;
        },
        
        selectAllTemplates() {
            // Set currentSelectedType to allTemplates
            this.currentSelectedType = {
                id: 'allTemplates',
                title: 'All Templates',
                type: 'allTemplates',
                defaultOrigin: 'allTemplates'
            };
            this.selectedTemplate = null;
            this.searchQuery = '';
            this.searchText = '';
            this.showTeamsDropdown = false;
            this.showSessionsDropdown = false;
            this.showSolutionDropdown = false;
            // Fetch all templates for the current context
            this.fetchAllTemplates();
        },

        selectType(type) {
            // If it's selectTeams, toggle the dropdown instead of selecting it
            if(type?.id === "selectTeams") {
                this.toggleTeamsDropdown();
                return;
            }
            
            // If it's selectSessions, toggle the dropdown instead of selecting it
            if(type?.id === "selectSessions") {
                this.toggleSessionsDropdown();
                return;
            }
            
            // If it's selectSolutionBoards, toggle the dropdown instead of selecting it
            if(type?.id === "selectSolutionBoards") {
                this.toggleSolutionDropdown();
                return;
            }

            this.currentSelectedType = type;
            this.selectedTemplate = null;
            this.searchQuery = '';
            this.searchText = '';
            this.showTeamsDropdown = false; // Close teams dropdown when selecting other types
            this.showSessionsDropdown = false; // Close sessions dropdown when selecting other types
            this.showSolutionDropdown = false; // Close solution dropdown when selecting other types
            // Fetch templates for the selected type
            this.fetchTemplates();
        },

        fetchAllTemplates() {
            this.loading = true;
            this.error = null;

            let contextId = this.contextType === "session" ? String(this.sessionId) : (this.contextType === "solution" ? String(this.solutionId) : (this.viewType ? String(this.workspaceId) : ""));
            let context = this.viewType ? "workspace" : this.contextType;

            let data = {
                context: context
            };

            axios.post(`graphx/getAllTemplate/${contextId}`, data)
                .then(response => {
                    this.loading = false;
                    if (response.data && response.data.success) {
                        this.localTemplatesList = response.data.templates || [];
                    } else {
                        this.localTemplatesList = [];
                        this.error = 'Failed to load all templates';
                    }
                    if(response.data.templateCount){
                        this.templatesCount = response.data.templateCount;
                        // console.log("response.data.count : ", response.data.templateCount);
                    }
                })
                .catch(error => {
                    this.loading = false;
                    this.localTemplatesList = [];
                    this.error = 'Error loading all templates';
                    console.error('Error fetching all templates:', error);
                });
        },

        fetchTemplates() {
            if (!this.currentSelectedType || !this.currentSelectedType.id) {
                this.localTemplatesList = [];
                return;
            }

            this.loading = true;
            this.error = null;

            let typeToUse = this.currentSelectedType;

            let data = {
                association: typeToUse?.type,
                contextId: this.contextType === "session" ? String(this.sessionId) : (this.contextType === "solution" ? String(this.solutionId) : (this.viewType ? String(this.workspaceId) : "")),
                context: this.viewType ? "workspace" : this.contextType
            };
            axios.post('/graphx/template/all/' + typeToUse.id, data)
                .then(response => {
                    this.loading = false;
                    this.localTemplatesList = [];
                    this.systemTemplates = [];
                    if (response.data && response.data.success) {
                        this.systemTemplates = response.data.systemTemplates || [];

                        if (response.data.templates) {
                            _.each(response.data.templates, templ => {
                                this.localTemplatesList.push(templ);
                            });
                        }
                        if(response.data.templatesCount){
                            this.templatesCount = response.data.templatesCount;
                            // console.log("response.data.count : ", response.data.templatesCount);
                        }
                    } else {
                        this.localTemplatesList = [];
                        this.error = 'Failed to load templates';
                    }
                })
                .catch(error => {
                    console.error('Error fetching templates:', error);
                    this.loading = false;
                    this.localTemplatesList = [];
                    this.error = 'Failed to load templates';
                });
        },

        useTemplate(template) {
            // Check permission before allowing template usage
            if (!this.permissions || !this.permissions.CREATE_COLLAB_BOARD) {
                // showTopMessage('You don\'t have permission to create collaboration boards', 'error');
                return;
            }

            // Ensure template has all required properties with proper defaults
            const templateData = {
                ...template, // Include all other properties
                // id: template.id || template.folderId || template.title,
                title: template?.title,
                description: template?.description || '', // Ensure description is never undefined
                type: template.type || 'template',
                isSystem: template.isSystem || false,
                backgroundColor: template.backgroundColor || '#ffffff',
                backgroundPattern: template.backgroundPattern || 'dotted',
                // Ensure arrays are initialized to prevent length errors
                associations: template.associations || [],
                piBoards: template.piBoards || [],
                solutionBoards: template.solutionBoards || [],
                collections: template.collections || [],
            };

            // Emit template-used event to match collaboration-templates behavior
            this.$emit('template-used', templateData);
            this.$emit('close');
        },

        
        clearSearch() {
            this.searchText = '';
            this.searchQuery = '';
        },
        
        toggleTeamsDropdown() {
            console.log('toggleTeamsDropdown called, current state:', this.showTeamsDropdown);
            this.showTeamsDropdown = !this.showTeamsDropdown;
            console.log('toggleTeamsDropdown new state:', this.showTeamsDropdown);
            console.log('sessionTeams available:', this.sessionTeams);
        },
        
        selectTeam(team) {
            // Set the team as currentSelectedType
            this.currentSelectedType = {
                id: team.id,
                title: team.title,
                type: 'team',
                defaultOrigin: 'sessionTeam'
            };
            this.selectedTemplate = null;
            this.searchQuery = '';
            this.searchText = '';
            // this.showTeamsDropdown = false;
            // Fetch templates for the selected team
            this.fetchTemplates();
        },

        toggleSessionsDropdown() {
            this.showSessionsDropdown = !this.showSessionsDropdown;
        },

        selectSession(session) {
            // Set the session as currentSelectedType
            this.currentSelectedType = {
                id: session.id,
                title: session.title,
                type: 'piBoard', // Use 'piBoard' to match backend expectation
                defaultOrigin: 'solutionSession'
            };
            this.selectedTemplate = null;
            this.searchQuery = '';
            this.searchText = '';
            // this.showSessionsDropdown = false;
            // Fetch templates for the selected session
            this.fetchTemplates();
        },

        toggleSolutionDropdown() {
            this.showSolutionDropdown = !this.showSolutionDropdown;
        },

        selectSolutionBoard(solution) {
            // Set the solution board as currentSelectedType
            this.currentSelectedType = {
                id: solution.id,
                title: solution.title,
                type: 'solutionBoard', // Use 'solutionBoard' to match backend expectation
                defaultOrigin: 'workspaceSolutionBoard'
            };
            this.selectedTemplate = null;
            this.searchQuery = '';
            this.searchText = '';
            // this.showSolutionDropdown = false;
            // Fetch templates for the selected solution board
            this.fetchTemplates();
        },

        // Template management methods
        onClickMenuButton(template, event) {
			event.stopPropagation();
            this.templateActionMenu = this.templateActionMenu === template.id ? '' : template.id;
        },

        onClickCopySharableUrl(template, event) {
			event.stopPropagation();
            this.$emit('copy-template-link', template);
            this.templateActionMenu = '';
        },

        onClickEditTemplate(template, event) {
			event.stopPropagation();
            // Emit edit event to parent component
            this.$emit('edit-template', template);
            this.templateActionMenu = '';
        },

        onClickDeleteTemplate(template, event) {
			event.stopPropagation();
            // Emit delete event to parent component (deletion is handled by parent)
            this.$emit('delete-template', template);
            this.templateActionMenu = '';
        },

        // Method to remove template from list after successful deletion
        removeTemplateFromList(templateId) {
            this.localTemplatesList = this.localTemplatesList.filter(t => t.id !== templateId);
            
            // Refresh templates and counts for current context after deletion
            this.refreshAfterDelete();
        },

        // Method to refresh data after deletion
        refreshAfterDelete() {
            // If we're on "All Templates" view, fetch all templates
            if (this.currentSelectedType && this.currentSelectedType.id === 'allTemplates') {
                this.fetchAllTemplates();
            } 
            // Otherwise, fetch templates for the current selected type
            else if (this.currentSelectedType) {
                this.fetchTemplates();
            }
        },

        // Method to get count for "All Templates"
        getAllTemplatesCount() {
            return this.templatesCount["allTemplates"] + 1 || 0;
        },

        // Method to get count for a specific type
        getTypeCount(type) {
            if (!type || !type.id) {
                return 0;
            }
            
            // Special handling for selectTeams - show sum of all team template counts
            if (type.id === "selectTeams") {
                if (!this.sessionTeams || !this.templatesCount) {
                    return 0;
                }
                return this.sessionTeams.reduce((total, team) => {
                    return total + (this.templatesCount[team.id] || 0);
                }, 0);
            }
            
            // Special handling for selectSessions - show sum of all session template counts
            if (type.id === "selectSessions") {
                if (!this.solutionSessions || !this.templatesCount) {
                    return 0;
                }
                return this.solutionSessions.reduce((total, session) => {
                    return total + (this.templatesCount[session.id] || 0);
                }, 0);
            }
            
            // Special handling for selectSolutionBoards - show sum of all solution board template counts
            if (type.id === "selectSolutionBoards") {
                if (!this.workspaceSolutionBoards || !this.templatesCount) {
                    return 0;
                }
                return this.workspaceSolutionBoards.reduce((total, solution) => {
                    return total + (this.templatesCount[solution.id] || 0);
                }, 0);
            }
            
            // For all other types, get count from templatesCount map
            if (!this.templatesCount) {
                return 0;
            }
            return this.templatesCount[type.id] || 0;
        },

        // Method to get count for a specific team
        getTeamCount(teamId) {
            if (!this.templatesCount || !teamId) {
                return 0;
            }
            return this.templatesCount[teamId] || 0;
        },

        // Method to get count for a specific session
        getSessionCount(sessionId) {
            if (!this.templatesCount || !sessionId) {
                return 0;
            }
            return this.templatesCount[sessionId] || 0;
        },

        // Method to get count for a specific solution board
        getSolutionBoardCount(solutionId) {
            if (!this.templatesCount || !solutionId) {
                return 0;
            }
            return this.templatesCount[solutionId] || 0;
        }
    }
});
