// Permission utility for OKRs roadmap component
const ComponentPermissionManager = {
    /**
     * Check if user has a specific permission
     * @param {Object} vueInstance - The Vue component instance
     * @param {string} permission - The permission to check
     * @returns {boolean} - True if user has permission
     */
    hasPermission(vueInstance, permission) {
        return vueInstance?.$options?.permissions?.[permission] || false;
    },

    /**
     * Show permission denied message
     * @param {Object} vueInstance - The Vue component instance
     * @param {string} action - Description of the action being blocked
     */
    showPermissionDenied(vueInstance, action = 'perform this action') {
        showTopMessageTargeted(
            `You don't have permission to ${action}`,
            'warning', 
            3000, 
            vueInstance?.isFullScreen ? '.okrsrdmp' : null
        );
    },

    /**
     * Check permission and show message if denied
     * @param {Object} vueInstance - The Vue component instance
     * @param {string} permission - The permission to check
     * @param {string} action - Description of the action
     * @returns {boolean} - True if permission granted
     */
    checkPermissionWithMessage(vueInstance, permission, action) {
        if (!this.hasPermission(vueInstance, permission)) {
            this.showPermissionDenied(vueInstance, action);
            return false;
        }
        return true;
    }
};

function findKeyByValue(map, searchValue) {
    // Iterate over the map entries
    for (const [key, value] of map.entries()) {
        // Check if the value (an array in your case) includes the searchValue
        if (value.includes(searchValue)) {
            return key; // Return the key if the value is found
        }
    }
    return null; // Return null if no matching value is found
}

function showTopMessageTargeted(message, type, time, targetElement) {
    if (message) {
        // Determine the parent container
        let parentContainer = targetElement || document.body;
        
        // If targetElement is a selector string, find the element
        if (typeof targetElement === 'string') {
            parentContainer = document.querySelector(targetElement) || document.body;
        }
        
        // Create unique ID for this container's message
        let messageId = 'top-message-targeted';
        if (targetElement) {
            messageId += '-' + Math.random().toString(36).substr(2, 9);
        }
        
        // Check if message already exists in this container
        let existingMsg = parentContainer.querySelector(`#${messageId}`);
        let topMsg;
        
        if (existingMsg) {
            topMsg = $(existingMsg);
        } else {
            // Create new message element
            topMsg = $(`<div id="${messageId}" class="kendis-targeted-message ${type}">
                <i class="ti-close rmv-ico" title="Close" onclick="$(this).closest('.kendis-targeted-message').hide();"></i>
                <div class="msg-content"></div>
            </div>`);
            
            // Append to target container
            $(parentContainer).append(topMsg);
        }
        
        // Update message content and styling
        topMsg.find('.msg-content').html(message);
        topMsg.show();
        topMsg.attr('class', `kendis-targeted-message ${type}`);
        
        // Set default time if not provided
        if (time === undefined) {
            time = 4000;
        }
        
        // Auto-hide after specified time
        if (time > 0) {
            setTimeout(function() {
                topMsg.hide();
            }, time);
        }
        
        // Add styles if not already added
        if (!document.querySelector('#kendis-targeted-message-styles')) {
            const styles = $(`<style id="kendis-targeted-message-styles">
                .kendis-targeted-message {
                    position: absolute;
                    top: 45px;
                    right: 10px;
                    z-index: 10000;
                    padding: 12px 16px;
                    border-radius: 6px;
                    font-size: 14px;
                    font-weight: 500;
                    color: white;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                    display: none;
                    min-width: 250px;
                    max-width: 400px;
                    animation: slideInRight 0.3s ease-out;
                }
                
                .kendis-targeted-message.success { background-color: #ECFDF5; border-left: 4px solid #07A972; }
                .kendis-targeted-message.success .msg-content { color: #07A972; }
                
                .kendis-targeted-message.warning { background-color: #FFFAEE; border-left: 4px solid #EDA11E; }
                .kendis-targeted-message.warning .msg-content { color: #EDA11E; }
                
                .kendis-targeted-message.error { background-color: #FEF2F2; border-left: 4px solid #D8403F; }
                .kendis-targeted-message.error .msg-content { color: #D8403F; }
                
                .kendis-targeted-message.info  { background-color: #E9EDFC; border-left: 4px solid #3859C0; }
                .kendis-targeted-message.info .msg-content { color: #D8403F; }
                
                .kendis-targeted-message .rmv-ico {
                    position: absolute;
                    top: 15px;
                    right: 11px;
                    cursor: pointer;
                    font-size: 11px;
                    opacity: 0.8;
                    transition: opacity 0.2s;
                }
                
                .kendis-targeted-message .rmv-ico:hover {
                    opacity: 1;
                }
                
                .kendis-targeted-message .msg-content {
                    padding-right: 20px;
                    line-height: 1.4;
                }
                
                @keyframes slideInRight {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
                
                /* Fullscreen mode adjustments */
                .okrsrdmp .kendis-targeted-message {
                    position: fixed;
                    top: 20px;
                    right: 20px;
                }
            </style>`);
            $('head').append(styles);
        }
    }
}

var scheduler;
var grid;
//create a timeperiod selection component
const TIME_PERIOD_SELECTION = Vue.component("time-period-selection", {
    name: 'time-period-selection',
    template: `
                <div class="slctdn DFA p-rel c-point rnd4" @click.stop="timePeriod.showPopup=!timePeriod.showPopup" >
									<div class="DAJ">{{fieldsValue ? (fieldsValue.title === 'Yearly' ? 'Year' : fieldsValue.title) + " - " + fieldsValue.selectedYear  : "Select Time Period"}}</div>
									<vuepopup v-if="timePeriod.showPopup " @close="timePeriod.showPopup=false;">
										<div class="drop_box nwmu lnk artp" style="display: block; width: auto;">
											<div class="DFA p-rel mb-10">
												<div @click.stop=" onChangeTimePeriodYear(-1)" class="add-year button-w mr-5 w-7pc">
													<em class="ti-angle-left"></em>
												</div>
												<div :class="{
													'slc-year-bg-color': fieldsValue && ['Yearly', 'Monthly'].includes(fieldsValue.title),
													'disabled': timePeriod.byMonth || isTimePeriodFieldDisabled('yearly')
													}"
													class="mr-5 slc-year-border button-w c-point w-55pc tooltip"
													@click="!timePeriod.byMonth && !isTimePeriodFieldDisabled('yearly') && onTimePeriodSelect('yearly')">
													<div class="">{{timePeriod.byMonth ? "Months " + timePeriod.selectedYear : "Year " + timePeriod.selectedYear}}</div>
													<div class="hovtooltipB">{{getOptionTooltip(timePeriod.byMonth ? 'monthly' : 'yearly')}}</div>
												</div>
												<div class="slc-year slc-year-border button-w c-point" @click.stop="timePeriod.showYearSelect=!timePeriod.showYearSelect;">
													<em class="ti-angle-down"></em>
												</div>	
												<vuepopup v-if="timePeriod.showYearSelect" @close="timePeriod.showYearSelect">
													<div class="drop_box nwmu lnk artp tmslcyear" style="display: block;">
														<ul>
															<li :class="timePeriod.byMonth ? '' : 'active'" @click.stop="timePeriod.byMonth=false;">
																<a href="javascript:void(0);" class="DFA">Yearly</a>
															</li>
															<li :class="timePeriod.byMonth ? 'active' : ''" @click.stop="timePeriod.byMonth=true;">
																<a href="javascript:void(0);" class="DFA">Monthly</a>
															</li>
														</ul>
													</div>
												</vuepopup>
												<div @click.stop=" onChangeTimePeriodYear(1)" class="add-year button-w ml-5">
													<em class="ti-angle-right"></em>
												</div>
											</div>
											<div class="DFA mb-10 ml-10 mr-10" v-if=timePeriod.byMonth>
												<div class="time-period-grid">
													<div class="slc-month button-w w-15pc slc-h1-border tooltip" v-for="month in timePeriod.months"
														:class="{
															'slc-h1-bg-color': fieldsValue && fieldsValue.title === month,
															'disabled': isTimePeriodFieldDisabled(month)
														}"
														@click="!isTimePeriodFieldDisabled(month) && onTimePeriodSelect(month)">
														{{month}}
														<div class="hovtooltipB">{{getOptionTooltip(month)}}</div>
													</div>
												</div>
											</div>
											<div v-else >
												<div class="DFA ml-10 mr-10 mb-10">
													<div :class="{
															'slc-h1-bg-color': fieldsValue && fieldsValue.title === 'H1',
															'disabled': isTimePeriodFieldDisabled('H1')
															}"
															@click="!isTimePeriodFieldDisabled('H1') && onTimePeriodSelect('h1')" class="button-w w-50pc mr-5 slc-h1-border tooltip">
														{{"H1-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('H1')}}</div>
													</div>
													<div :class="{
														'slc-h2-bg-color': fieldsValue && fieldsValue.title === 'H2',
														'disabled': isTimePeriodFieldDisabled('H2')
														}"
														@click="!isTimePeriodFieldDisabled('H2') && onTimePeriodSelect('h2')" class="button-w w-50pc slc-h2-border tooltip">
														{{"H2-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('H2')}}</div>
													</div>
												</div>
												<div class="DFA mb-10 mr-10 ml-10">
													<div 
														:class="{
															'slc-q1-bg-color': fieldsValue && fieldsValue.title === 'Q1',
															'disabled': isTimePeriodFieldDisabled('Q1')
														}"
														@click="!isTimePeriodFieldDisabled('Q1') && onTimePeriodSelect('q1')"  class="button-w w-25pc mr-5 slc-q1-border tooltip">
														{{"Q1-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('Q1')}}</div>
													</div>
													<div 
														:class="{
															'slc-q2-bg-color': fieldsValue && fieldsValue.title === 'Q2',
															'disabled': isTimePeriodFieldDisabled('Q2')
														}"
														@click="!isTimePeriodFieldDisabled('Q2') && onTimePeriodSelect('q2')" class="button-w w-25pc mr-5 slc-q2-border tooltip">
														{{"Q2-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('Q2')}}</div>
													</div>
													<div 
														:class="{
															'slc-q3-bg-color': fieldsValue && fieldsValue.title === 'Q3',
															'disabled': isTimePeriodFieldDisabled('Q3')
														}"
														@click="!isTimePeriodFieldDisabled('Q3') && onTimePeriodSelect('q3')" class="button-w w-25pc mr-5 slc-q3-border tooltip">
														{{"Q3-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('Q3')}}</div>
													</div>
													<div 
														:class="{
															'slc-q4-bg-color': fieldsValue && fieldsValue.title === 'Q4',
															'disabled': isTimePeriodFieldDisabled('Q4')
														}" 
														@click="!isTimePeriodFieldDisabled('Q4') && onTimePeriodSelect('q4')" class="button-w w-25pc slc-q4-border tooltip">
														{{"Q4-" + timePeriod.selectedYear % 100}}
														<div class="hovtooltipB">{{getOptionTooltip('Q4')}}</div>
													</div>
												</div>
											</div>
										</div>
									</vuepopup>
								</div>
    `,
    data() {
        return {
            timePeriod: {
                showPopup: false,
                selectedYear: new Date().getFullYear(),
                showYearSelect: false,
                byMonth: false,
                months: [
                    "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
                ],
                selectedMonth: '',
            },
        }
    },
    created() {
        // Initialize the selected year based on the already selected time period
        if (this.fieldsValue && this.fieldsValue.selectedYear) {
            this.timePeriod.selectedYear = this.fieldsValue.selectedYear;
        }
    },
    props: ["fieldsValue", "dateRestrictions"],
    methods: {

        onTimePeriodSelect(selectedValue) {
            if (this.timePeriod.byMonth) {
                const monthMap = {
                    Jan: { month: 0, days: 31 },
                    Feb: { month: 1, days: new Date(this.timePeriod.selectedYear, 2, 0).getDate() },
                    Mar: { month: 2, days: 31 },
                    Apr: { month: 3, days: 30 },
                    May: { month: 4, days: 31 },
                    Jun: { month: 5, days: 30 },
                    Jul: { month: 6, days: 31 },
                    Aug: { month: 7, days: 31 },
                    Sep: { month: 8, days: 30 },
                    Oct: { month: 9, days: 31 },
                    Nov: { month: 10, days: 30 },
                    Dec: { month: 11, days: 31 }
                };

                if (monthMap[selectedValue]) {
                    const { month, days } = monthMap[selectedValue];
                    this.setTimePeriod(selectedValue, month, 1, month, days);
                }
            }

            const periodMap = {
                monthly: { title: "Monthly", startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                yearly: { title: "Yearly", startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                h1: { title: "H1", startMonth: 0, startDay: 1, endMonth: 5, endDay: 31 },
                h2: { title: "H2", startMonth: 6, startDay: 1, endMonth: 11, endDay: 31 },
                q1: { title: "Q1", startMonth: 0, startDay: 1, endMonth: 2, endDay: 31 },
                q2: { title: "Q2", startMonth: 3, startDay: 1, endMonth: 5, endDay: 31 },
                q3: { title: "Q3", startMonth: 6, startDay: 1, endMonth: 8, endDay: 30 },
                q4: { title: "Q4", startMonth: 9, startDay: 1, endMonth: 11, endDay: 31 }
            };

            if (periodMap[selectedValue]) {
                const { title, startMonth, startDay, endMonth, endDay } = periodMap[selectedValue];
                this.setTimePeriod(title, startMonth, startDay, endMonth, endDay);
            }

        },
        resetTimePeriodField(id) {
            if (this.fieldModel[id]) {
                this.$delete(this.fieldModel, id);
                this.fieldModel[this.startDateFieldId] && this.$delete(this.fieldModel, this.startDateFieldId);
                this.fieldModel[this.endDateFieldId] && this.$delete(this.fieldModel, this.endDateFieldId);

                this.inlineEditingChangesHandle();
            }
        },
        onChangeTimePeriodYear(value) {
            this.timePeriod.selectedYear += value;
        },
        setTimePeriod(title, startMonth, startDay, endMonth, endDay) {
            const period = {
                title,
                selectedYear: this.timePeriod.selectedYear,
                monthly: this.timePeriod.byMonth
            };
            this.$emit("time-period-selected", period);
            // this.fieldModel[this.startDateFieldId] = new Date(this.timePeriod.selectedYear, startMonth, startDay);
            // this.fieldModel[this.endDateFieldId] = new Date(this.timePeriod.selectedYear, endMonth, endDay);
            // this.fieldModel[this.timePeriodFieldId] = period;
        },

        getOptionTooltip(option) {
            let selectedYear = this.timePeriod.selectedYear;

            let getDaysInMonth = function(month, year) {
                return new Date(year, month, 0).getDate();
            };

            if (option === 'yearly' || option === 'monthly' || option === 'Yearly' || option === 'Monthly') {
                return '1 Jan - 31 Dec ' + this.timePeriod.selectedYear;
            }
            if (this.timePeriod.months.indexOf(option) !== -1) {
                var monthIndex = this.timePeriod.months.indexOf(option) + 1;
                var daysInMonth = getDaysInMonth(monthIndex, selectedYear);
                return '1 ' + option + ' - ' + daysInMonth + ' ' + option;
            }
            if (option === 'H1') {
                return '1 Jan - 30 Jun ' + this.timePeriod.selectedYear;
            }
            if (option === 'H2') {
                return '1 Jul - 31 Dec ' + this.timePeriod.selectedYear;
            }
            if (option === 'Q1') {
                return '1 Jan - 31 Mar ' + this.timePeriod.selectedYear;
            }
            if (option === 'Q2') {
                return '1 Apr - 30 Jun ' + this.timePeriod.selectedYear;
            }
            if (option === 'Q3') {
                return '1 Jul - 30 Sep ' + this.timePeriod.selectedYear;
            }
            if (option === 'Q4') {
                return '1 Oct - 31 Dec ' + this.timePeriod.selectedYear;
            }
        },

        isTimePeriodFieldDisabled(timePeriodValue) {
            // If no date restrictions, allow all time periods
            if (!this.dateRestrictions) {
                return false;
            }

            // Convert time period value to date range
            let timePeriodStartDate, timePeriodEndDate;
            const year = this.timePeriod.selectedYear;

            if (timePeriodValue === 'yearly' || timePeriodValue === 'Yearly') {
                timePeriodStartDate = new Date(year, 0, 1);
                timePeriodEndDate = new Date(year, 11, 31);
            } else if (timePeriodValue === 'monthly' || timePeriodValue === 'Monthly') {
                timePeriodStartDate = new Date(year, 0, 1);
                timePeriodEndDate = new Date(year, 11, 31);
            } else if (timePeriodValue === 'H1' || timePeriodValue === 'h1') {
                timePeriodStartDate = new Date(year, 0, 1);
                timePeriodEndDate = new Date(year, 5, 30);
            } else if (timePeriodValue === 'H2' || timePeriodValue === 'h2') {
                timePeriodStartDate = new Date(year, 6, 1);
                timePeriodEndDate = new Date(year, 11, 31);
            } else if (timePeriodValue === 'Q1' || timePeriodValue === 'q1') {
                timePeriodStartDate = new Date(year, 0, 1);
                timePeriodEndDate = new Date(year, 2, 31);
            } else if (timePeriodValue === 'Q2' || timePeriodValue === 'q2') {
                timePeriodStartDate = new Date(year, 3, 1);
                timePeriodEndDate = new Date(year, 5, 30);
            } else if (timePeriodValue === 'Q3' || timePeriodValue === 'q3') {
                timePeriodStartDate = new Date(year, 6, 1);
                timePeriodEndDate = new Date(year, 8, 30);
            } else if (timePeriodValue === 'Q4' || timePeriodValue === 'q4') {
                timePeriodStartDate = new Date(year, 9, 1);
                timePeriodEndDate = new Date(year, 11, 31);
            } else if (this.timePeriod.months.indexOf(timePeriodValue) !== -1) {
                // Handle individual months
                const monthIndex = this.timePeriod.months.indexOf(timePeriodValue);
                const daysInMonth = new Date(year, monthIndex + 1, 0).getDate();
                timePeriodStartDate = new Date(year, monthIndex, 1);
                timePeriodEndDate = new Date(year, monthIndex, daysInMonth);
            } else {
                // Unknown time period, allow it
                return false;
            }

            // Check if the time period falls within the parent's date restrictions
            const parentStartDate = new Date(this.dateRestrictions.minDate);
            const parentEndDate = new Date(this.dateRestrictions.maxDate);

            // Time period is disabled if:
            // 1. It starts before the parent's start date, OR
            // 2. It ends after the parent's end date, OR
            // 3. It doesn't overlap with the parent's date range at all
            return timePeriodStartDate < parentStartDate || 
                   timePeriodEndDate > parentEndDate || 
                   timePeriodEndDate < parentStartDate || 
                   timePeriodStartDate > parentEndDate;
        },

    }
})

let KENDIS_ALERT = Vue.component("kendis-alert", {
    name: 'kendis-alert',
    template: `
        <div v-if="showAlert" class="kendis-alert-overlay" :style="overlayStyle">
            <div class="kendis-alert-container" :style="containerStyle">
                <div class="kendis-alert-header">
                    <h3 class="kendis-alert-title">{{ title }}</h3>
                </div>
                <div class="kendis-alert-body">
                    <div v-if="icon" class="kendis-alert-icon" :class="iconClass">
                        <i :class="iconName"></i>
                    </div>
                    <div class="kendis-alert-text" v-html="text"></div>
                </div>
                <div class="kendis-alert-footer">
                    <button v-if="showCancelButton" 
                            class="kendis-alert-btn kendis-alert-btn-cancel" 
                            @click="onCancel">
                        {{ cancelButtonText }}
                    </button>
                    <button class="kendis-alert-btn kendis-alert-btn-confirm" 
                            :class="confirmButtonClass"
                            @click="onConfirm">
                        {{ confirmButtonText }}
                    </button>
                </div>
            </div>
        </div>
    `,
    data() {
        return {
            showAlert: false,
            title: 'Are you sure?',
            text: '',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#6c757d',
            target: null,
            resolve: null,
            reject: null
        }
    },
    computed: {
        overlayStyle() {
            return {
                position: this.target ? 'absolute' : 'fixed',
                top: '0',
                left: '0',
                width: '100%',
                height: '100%',
                backgroundColor: 'rgba(0, 0, 0, 0.4)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                zIndex: this.target ? '1000' : '9999'
            }
        },
        containerStyle() {
            return {
                backgroundColor: '#fff',
                borderRadius: '8px',
                boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
                maxWidth: '400px',
                width: '90%',
                padding: '0',
                fontFamily: 'Arial, sans-serif'
            }
        },
        iconClass() {
            return {
                'kendis-alert-icon-warning': this.icon === 'warning',
                'kendis-alert-icon-error': this.icon === 'error',
                'kendis-alert-icon-success': this.icon === 'success',
                'kendis-alert-icon-info': this.icon === 'info'
            }
        },
        iconName() {
            const icons = {
                'warning': 'ti-alert-triangle',
                'error': 'ti-alert-circle',
                'success': 'ti-check-circle',
                'info': 'ti-info-circle'
            };
            return icons[this.icon] || 'ti-alert-triangle';
        },
        confirmButtonClass() {
            return {
                'kendis-alert-btn-danger': this.icon === 'warning' || this.icon === 'error'
            }
        }
    },
    methods: {
        fire(options) {
            // Set options
            this.title = options.title || 'Are you sure?';
            this.text = options.text || '';
            this.icon = options.icon || 'warning';
            this.showCancelButton = options.showCancelButton !== false;
            this.confirmButtonText = options.confirmButtonText || 'Yes';
            this.cancelButtonText = options.cancelButtonText || 'Cancel';
            this.confirmButtonColor = options.confirmButtonColor || '#3085d6';
            this.cancelButtonColor = options.cancelButtonColor || '#6c757d';
            this.target = options.target || null;
            
            // Show the alert
            this.showAlert = true;
            
            // Append to target if specified
            if (this.target) {
                this.$nextTick(() => {
                    const alertElement = this.$el;
                    if (alertElement && this.target) {
                        // Make target relative if it's not already positioned
                        const targetStyle = window.getComputedStyle(this.target);
                        if (targetStyle.position === 'static') {
                            this.target.style.position = 'relative';
                        }
                        this.target.appendChild(alertElement);
                    }
                });
            }
            
            // Return promise
            return new Promise((resolve, reject) => {
                this.resolve = resolve;
                this.reject = reject;
            });
        },
        
        onConfirm() {
            this.hideAlert();
            if (this.resolve) {
                this.resolve({ isConfirmed: true, value: true });
            }
        },
        
        onCancel() {
            this.hideAlert();
            if (this.resolve) {
                this.resolve({ isConfirmed: false, isDismissed: true, value: false });
            }
        },
        
        hideAlert() {
            this.showAlert = false;
            // Clean up DOM if appended to target
            this.$nextTick(() => {
                if (this.target && this.$el && this.$el.parentNode === this.target) {
                    this.target.removeChild(this.$el);
                }
            });
        }
    },
    
    // Add styles as a computed property or in mounted
    mounted() {
        // Inject CSS styles if not already present
        if (!document.getElementById('kendis-alert-styles')) {
            const style = document.createElement('style');
            style.id = 'kendis-alert-styles';
            style.textContent = `
                .kendis-alert-overlay {
                    font-family: Arial, sans-serif;
                }
                
                .kendis-alert-header {
                    padding: 20px 20px 10px 20px;
                    text-align: center;
                }
                
                .kendis-alert-title {
                    margin: 0;
                    font-size: 18px;
                    font-weight: 600;
                    color: #333;
                }
                
                .kendis-alert-body {
                    padding: 10px 20px 20px 20px;
                    text-align: center;
                }
                
                .kendis-alert-icon {
                    font-size: 48px;
                    margin-bottom: 15px;
                }
                
                .kendis-alert-icon-warning {
                    color: #f39c12;
                }
                
                .kendis-alert-icon-error {
                    color: #e74c3c;
                }
                
                .kendis-alert-icon-success {
                    color: #27ae60;
                }
                
                .kendis-alert-icon-info {
                    color: #3498db;
                }
                
                .kendis-alert-text {
                    font-size: 14px;
                    color: #666;
                    line-height: 1.4;
                }
                
                .kendis-alert-footer {
                    padding: 0 20px 20px 20px;
                    display: flex;
                    justify-content: center;
                    gap: 10px;
                }
                
                .kendis-alert-btn {
                    padding: 10px 20px;
                    border: none;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    transition: all 0.2s ease;
                    min-width: 80px;
                }
                
                .kendis-alert-btn:hover {
                    transform: translateY(-1px);
                    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
                }
                
                .kendis-alert-btn-confirm {
                    background-color: #3085d6;
                    color: white;
                }
                
                .kendis-alert-btn-confirm:hover {
                    background-color: #2773c3;
                }
                
                .kendis-alert-btn-danger {
                    background-color: #d33;
                    color: white;
                }
                
                .kendis-alert-btn-danger:hover {
                    background-color: #b82d2d;
                }
                
                .kendis-alert-btn-cancel {
                    background-color: #6c757d;
                    color: white;
                }
                
                .kendis-alert-btn-cancel:hover {
                    background-color: #5a6268;
                }
            `;
            document.head.appendChild(style);
        }
    }
});

let ASYNC_DATE_CONFIRMATION_POPUP = Vue.component("async-date-confirmation-popup", {
    name: 'async-date-confirmation-popup',
    template: `
<div v-if="showPopup">
        <div class="overlay" style="display: block; z-index: 501;"></div>
        <div class="add-pop ui-dialog roaddat-pop b" style="display: block;">
            <div v-if="loader.show" class="loader">
                <div>
                    <div class="loaderinteg"></div>
                    <div v-html="loader.text"></div>
                </div>
            </div>
            <div class="head">  
            	{{itemIdentifier}}
                <em href="javascript:void(0);" title="Close" class="rmv-ico ti-close" @click="onClickCloseButton"></em>
            </div>
            <div class="prow cntr">
                <!-- <div class="mnttl mb-20">
                    <span class="txt">{{itemIdentifier}}</span>
                </div> -->
                <!-- Text Input Section -->
                <div v-if="showTextField" class="prow mb-30">
                    <div class="prow fs-14 ftsb mb-5">{{ textInputLabel }}</div>
                    <div class="evdt DFA">
                        <input type="text" class="rnd4" v-model="editModel.textValue"
                            :placeholder="textInputPlaceholder"
                            :maxlength="maxLength"
                            :class="[errors.textValue ? 'required' : '']">
                    </div>
                </div>
                
                <div v-if = "showTimePeriod" class="prow tmprd mb-30">
                	<div class="prow fs-14 ftsb mb-5">Time Period</div>
                    <time-period-selection 
                    	:fields-value="editModel.timePeriod"
                    	:date-restrictions="dateRestrictions"
                         @time-period-selected="onSelectTimePeriod">
                        </time-period-selection>
                </div>
                
                <!-- Progress Criteria Dropdown for KR Creation -->
                <div v-if="isKrCreation && showProgressCriteria" class="prow mb-30">
                    <div class="prow fs-14 ftsb mb-5">Progress Criteria</div>
                    <div class="evdt DFA">
                        <select class="rnd4" v-model="editModel.progressCriteria" 
                                :class="[errors.progressCriteria ? 'required' : '']">
                            <option value="">Select Progress Criteria</option>
                            <option v-for="option in progressCriteriaOptions" 
                                    :key="option.value" 
                                    :value="option.value">
                                {{ option.label }}
                            </option>
                        </select>
                    </div>
                </div>
                
                <!-- Date restrictions warning for KR -->
                <div v-if="isKrCreation && dateRestrictions" class="prow mb-10">
                    <div class="alert alert-info" style="background: #e8f4f8; border: 1px solid #26a69a; border-radius: 4px; padding: 10px; font-size: 12px; color: #00695c;">
                        <i class="ti-info-alt" style="margin-right: 5px;"></i>
                        {{ dateRestrictions.tooltip }}
                    </div>
                </div>
                
                <!-- Date restrictions warning for Child Objective -->
                <div v-if="isChildObjectiveCreation && dateRestrictions" class="prow mb-10">
                    <div class="alert alert-info" style="background: #e8f4f8; border: 1px solid #26a69a; border-radius: 4px; padding: 10px; font-size: 12px; color: #00695c;">
                        <i class="ti-info-alt" style="margin-right: 5px;"></i>
                        {{ dateRestrictions.tooltip }}
                    </div>
                </div>
                
                <div v-if="showDateFields" class="prow DFA gap-2 mb-30">
                <div>    
                    <div class="prow hdin ftsb mb-5">
                        <span>{{ startDateLabel }}</span>
                        <span v-if="editModel.prevStartDate && !singleDateMode" class="fs-11 ml-10" style="color: #6c757d;">
                            (Previous: {{formatDate(new Date(editModel.prevStartDate), 'dd mmm, yy')}})
                        </span>
                    </div>
                    <div class="evdt">                        
                        <vuejs-datepicker class="icoDatepicker" 
                            :typeable="false" placeholder="Select date" 
                            v-model="editModel.startDate"
                            :class="[errors.startDate ? 'required' : '']"
                            :disabled-dates="getDisabledDatesConfig()"
                            @selected="onSelectDate($event, 'startDate')"></vuejs-datepicker>
                        
                    </div>
                </div>
                <div>
                    <template v-if = "!singleDateMode">
                        <div class="prow hdin ftsb mb-5">
                            <span>End Date</span>
                            <span v-if="editModel.prevEndDate" class="fs-11 ml-10" style="color: #6c757d;">
                                (Previous: {{formatDate(new Date(editModel.prevEndDate), 'dd mmm, yy')}})
                            </span>
                        </div>
                        <div class="evdt">
                            <vuejs-datepicker class="icoDatepicker" 
                                :typeable="false" placeholder="Select end date" 
                                v-model="editModel.endDate"
                                :class="[errors.endDate ? 'required' : '']"
                                :disabled-dates="getDisabledDatesConfig()"
                                @selected="onSelectDate($event, 'endDate')"></vuejs-datepicker>
                        </div>
                    </template>
                </div>
                </div>
                
                
                <!-- Date Picker Section -->
                <div v-if="showDateFields &&  editModel.prevStartDate  && !singleDateMode" class="prow bxmn">
                    <div class="prow hd ftsb">Update Dates</div>                    
                    <!-- Date Change Summary -->
                    <div v-if="editModel.prevStartDate || editModel.prevEndDate" class="prow date-change-summary mb-10">                        
                        <div v-if="editModel.prevStartDate" class="gap-1 DFA mb-10">
                            <span class="lbl">Start:</span>
                            <span class="" style="color: #dc3545;">{{formatDate(new Date(editModel.prevStartDate), 'dd mmm, yy')}}</span>
                            <span class="ti-arrow-right" style="color: #6c757d;"></span>
                            <span class="" style="color: #28a745;">{{editModel.startDate ? formatDate(editModel.startDate, 'dd mmm, yy') : 'Select new date'}}</span>
                        </div>
                        <div v-if="editModel.prevEndDate && !singleDateMode" class="gap-1 DFA mb-10">
                            <span class="lbl">End:</span>
                            <span class=" " style="color: #dc3545;">{{formatDate(new Date(editModel.prevEndDate), 'dd mmm, yy')}}</span>
                            <span class="ti-arrow-right" style="color: #6c757d;"></span>
                            <span class=" " style="color: #28a745;">{{editModel.endDate ? formatDate(editModel.endDate, 'dd mmm, yy') : 'Select new date'}}</span>
                        </div>
                    </div>
                </div>
                
                
                
            </div>

            <div class="rec btn">
                <button class="FR" @click="onClickSaveButton">Save</button>
                <a href="javascript:void(0);" class="cncl FR" @click="onClickCloseButton">Cancel</a>
            </div>
        </div>
    </div>
		`,

    data() {
        return {
            itemIdentifier: "",
            loader: {
                show: false,
                text: ""
            },
            editModel: {
                startDate: null,
                endDate: null,
                prevStartDate: null,
                prevEndDate: null,
                textValue: "",
                timePeriod :null,
                progressCriteria: ""
            },
            errors: {
                startDate: false,
                endDate: false,
                textValue: false,
                progressCriteria: false
            },
            showPopup: false,
            showDateFields: false,
            showTextField: false,
            textInputLabel: "Enter Text",
            textInputPlaceholder: "Type here...",
            showTimePeriod :false,
            singleDateMode :false,
            maxLength: null, // New parameter for character limit
            // Date restrictions for KR creation
            dateRestrictions: null,
            isKrCreation: false,
            isChildObjectiveCreation: false,
            parentObjective: null,
            // Progress criteria for KR creation
            showProgressCriteria: false,
            progressCriteriaOptions: [
                { value: "By Items", label: "By Items" },
                { value: "By Objectives", label: "By Objectives" },
                { value: "Target", label: "Target" }
            ]
        }
    },
    computed: {
        startDateLabel() {
            return this.singleDateMode ? 'Milestone Date' : 'Start Date';
        }
    },

    components: {
        vuejsDatepicker,
        'time-period-selection': TIME_PERIOD_SELECTION,
    },
    created() {
    },
    mounted() {
    },
    updated() {

    },
    watch: {

    },
    methods: {
        ////////////////////////////////////
        //////////  LOGIC
        onSelectTimePeriod(timePeriod) {
            if (timePeriod.monthly) {
                const monthMap = {
                    Jan: { month: 0, days: 31 },
                    Feb: { month: 1, days: new Date(timePeriod.selectedYear, 2, 0).getDate() },
                    Mar: { month: 2, days: 31 },
                    Apr: { month: 3, days: 30 },
                    May: { month: 4, days: 31 },
                    Jun: { month: 5, days: 30 },
                    Jul: { month: 6, days: 31 },
                    Aug: { month: 7, days: 31 },
                    Sep: { month: 8, days: 30 },
                    Oct: { month: 9, days: 31 },
                    Nov: { month: 10, days: 30 },
                    Dec: { month: 11, days: 31 }
                };

                if (monthMap[timePeriod.title]) {
                    const { month, days } = monthMap[timePeriod.title];
                    this.setTimePeriod(timePeriod.title, month, 1, month, days,timePeriod);
                }
            }

            const periodMap = {
                Monthly: { title: "Monthly", startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                Yearly: { title: "Yearly", startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                H1: { title: "H1", startMonth: 0, startDay: 1, endMonth: 5, endDay: 31 },
                H2: { title: "H2", startMonth: 6, startDay: 1, endMonth: 11, endDay: 31 },
                Q1: { title: "Q1", startMonth: 0, startDay: 1, endMonth: 2, endDay: 31 },
                Q2: { title: "Q2", startMonth: 3, startDay: 1, endMonth: 5, endDay: 31 },
                Q3: { title: "Q3", startMonth: 6, startDay: 1, endMonth: 8, endDay: 30 },
                Q4: { title: "Q4", startMonth: 9, startDay: 1, endMonth: 11, endDay: 31 }
            };

            if (periodMap[timePeriod.title]) {
                const { title, startMonth, startDay, endMonth, endDay } = periodMap[timePeriod.title];
                this.setTimePeriod(title, startMonth, startDay, endMonth, endDay,timePeriod);
            }
        },
        setTimePeriod(title, startMonth, startDay, endMonth, endDay,timePeriod) {
            this.editModel.startDate = new Date(timePeriod.selectedYear, startMonth, startDay);
            this.editModel.endDate = new Date(timePeriod.selectedYear, endMonth, endDay);
            this.editModel.timePeriod = timePeriod;
        },
        createPopup(input) {
            this.itemIdentifier = input.itemIdentifier || "";
            this.showDateFields = input.showDateFields || false;
            this.showTextField = input.showTextField || false;
            this.showTimePeriod = input.showTimePeriod || false;
            this.textInputLabel = input.textInputLabel || "Enter Text";
            this.textInputPlaceholder = input.textInputPlaceholder || "Type here...";
            this.singleDateMode = input.singleDateMode || false;
            this.maxLength = input.maxLength || null; // Set maxLength from input parameter

            // Store date restrictions for KR creation
            this.dateRestrictions = input.dateRestrictions || null;
            this.isKrCreation = input.isKrCreation || false;
            this.isChildObjectiveCreation = input.isChildObjectiveCreation || false;
            this.parentObjective = input.parentObjective || null;
            
            // Handle progress criteria for KR creation
            this.showProgressCriteria = input.showProgressCriteria || false;
            if (this.showProgressCriteria) {
                this.editModel.progressCriteria = input.progressCriteria || "";
            }
            
            

            if (this.showDateFields) {
                if(this.singleDateMode){
                    this.editModel.startDate = input.startDate || null;
                    this.editModel.endDate = null;
                }else {
                    this.editModel.startDate = input.startDate || null;
                    this.editModel.endDate = input.endDate || null;
                    // Set default dates from parent objective for KR and child objective creation
                        if ((this.isKrCreation || this.isChildObjectiveCreation) && input.parentObjective) {
                            const startDateFieldId = input.startDateFieldId || this.startDateFieldId;
                            const endDateFieldId = input.endDateFieldId || this.endDateFieldId;
                            
                            const parentStartDate = input.parentObjective.fields[startDateFieldId];
                            const parentEndDate = input.parentObjective.fields[endDateFieldId];
                            
                            if (parentStartDate && parentEndDate) {
                                this.editModel.startDate = new Date(parentStartDate);
                                this.editModel.endDate = new Date(parentEndDate);
                            }
                        }
                    this.editModel.prevStartDate = input.prevStartDate || null;
                    this.editModel.prevEndDate = input.prevEndDate || null;
                    if (this.showTimePeriod) {
                        this.editModel.timePeriod = input.timePeriod || null;
                        if (this.editModel.endDate) {
                            this.syncTimePeriod(this.editModel.endDate);
                        }
                    }
                }
            }
            if (this.showTextField) {
                this.editModel.textValue = input.textValue || "";
            }

            this.errors.startDate = false;
            this.errors.endDate = false;
            this.errors.textValue = false;
            this.showPopup = true;

            return new Promise((resolve, reject) => {
                this.resolve = resolve;
                this.reject = reject;
            });
        },

        showMe: function() {
            this.showPopup = true;
        },
        hideMe: function() {
            this.showPopup = false;
        },
        formatDate: function(date, format) {
            if (date) {
                return formatDate(date, format);
            }
            return "";
        },
        validateSave: function() {
            if(this.singleDateMode){
                return true;
            }
            let errorMessage = "";
            let valid = true;

            if(this.showTextField){
                //check if text is not empty
                this.errors.textValue = _.isEmpty(this.editModel.textValue)
                if(this.errors.textValue){
                    errorMessage = "field cannot be empty"
                    valid = false;
                }
            }
            
            // Validate progress criteria for KR creation
            if(this.showProgressCriteria){
                this.errors.progressCriteria = _.isEmpty(this.editModel.progressCriteria)
                if(this.errors.progressCriteria){
                    errorMessage = "Progress criteria is required"
                    valid = false;
                }
            }


            if(this.showDateFields) {
                this.errors.startDate = !_.isDate(this.editModel.startDate);
                this.errors.endDate = !_.isDate(this.editModel.endDate);

                if (this.errors.startDate || this.errors.endDate) {
                    valid = false;
                    errorMessage = "Start date/End date is missing";
                } else {

                    if (this.editModel.endDate <= this.editModel.startDate) {
                        this.errors.startDate = true;
                        this.errors.endDate = true;
                        valid = false;
                        errorMessage = "Start date must be smaller than End date";
                    }

                    // Additional validation for KR date restrictions
                    if (this.isKrCreation && this.dateRestrictions) {
                        const startDate = this.editModel.startDate;
                        const endDate = this.editModel.endDate;

                        if ((startDate < this.dateRestrictions.minDate || startDate > this.dateRestrictions.maxDate) ||
                            (endDate < this.dateRestrictions.minDate || endDate > this.dateRestrictions.maxDate)) {
                            this.errors.startDate = true;
                            this.errors.endDate = true;
                            valid = false;
                            errorMessage = "Dates are outside the allowed range of the parent's time period";
                        }
                    }
                }
            }
            if (!valid) {
                                        showTopMessageTargeted(errorMessage, "warning", 4000, this.isFullScreen ? '.okrsrdmp' : null);
            }
            return valid;
        },

        ////////////////////////////////////
        //////////  EVENTS
        onClickCloseButton: function() {
           this.reject('Popup was canceled');
           this.hideMe();
        },
        onClickSaveButton: function() {
            if (this.validateSave()) {
                this.resolve(this.editModel);
                this.hideMe();
            }
        },
        syncTimePeriod(endDate){
            const date = endDate
            const timePeriod = this.editModel.timePeriod || {};
            const month = date.getMonth();
            const year = date.getFullYear();

            const getQuarter = (month) => ['Q1', 'Q2', 'Q3', 'Q4'][Math.floor(month / 3)];

            if (_.isEmpty(timePeriod)) {
                this.editModel.timePeriod= {
                    title: getQuarter(month),
                    selectedYear: year,
                    monthly: false
                };
            } else {
                timePeriod.selectedYear = year;

                if (timePeriod.monthly) {
                    const selectedMonth = this.timePeriod.months[month];
                    if (selectedMonth !== timePeriod.title) {
                        timePeriod.title = selectedMonth;
                        this.timePeriod.selectedMonth = selectedMonth;
                    }
                } else {
                    const newTitle = timePeriod.title.startsWith('H') ? (month < 6 ? 'H1' : 'H2') : getQuarter(month);
                    if (timePeriod.title !== newTitle) timePeriod.title = newTitle;
                }

                this.editModel.timePeriod = timePeriod;
            }
        },
        onSelectDate: function(value, fieldName) {
            // Validate date restrictions for KR creation
            if (this.isKrCreation && this.dateRestrictions && value) {
                if (value < this.dateRestrictions.minDate || value > this.dateRestrictions.maxDate) {
                    showTopMessage("Date is outside the allowed range of the parent's time period.", "warning", 3000);
                    return;
                }
            }

            // Check start/end date logic
            if (fieldName === 'startDate' && this.editModel.endDate && value > this.editModel.endDate) {
                showTopMessage("Start date cannot be greater than End date.", "warning", 3000);
                return;
            }
            if (fieldName === 'endDate' && this.editModel.startDate && value < this.editModel.startDate) {
                showTopMessage("End date cannot be less than Start date.", "warning", 3000);
                return;
            }

            if (this.showTimePeriod && fieldName === 'endDate') {
                this.syncTimePeriod(value);
            }
        },

        // Get disabled dates configuration for date pickers
        getDisabledDatesConfig: function() {
            if (!this.isKrCreation || !this.dateRestrictions) {
                return {};
            }

            return {
                customPredictor: (date) => {
                    return date < this.dateRestrictions.minDate || date > this.dateRestrictions.maxDate;
                }
            };
        }
    }
});


const OKRS_ROADMAP_COMPONENT = Vue.component('okrs-roadmap-component', {
        name: 'okrs-roadmap-component',
        template: /*html*/`
                <div class="timelineview roadmap okrsrdmp" :class="{isFullScreen: isFullScreen}">
                
                  <objective-theme-progress v-if="objectiveThemeProgress.show" 
                  :objectiveGroupsId="objectiveThemeProgress.objectiveGroupsId" 
                  :isArt="objectiveThemeProgress.isArt" 
                  :isGlobalObjectiveView="objectiveThemeProgress.isGlobalObjectiveView" 
                  :radioTypeSelection="objectiveThemeProgress.radioTypeSelection" 
                  @closeThemeProgress="closeThemeProgress" 
                  :isOkrEnabled="objectiveThemeProgress.isOkrEnabled" 
                  :objectiveGroupList="objectiveThemeProgress.objectiveGroupList" 
                  :board="objectiveThemeProgress.board" 
                  :allObjectiveIds="objectiveThemeProgress.allObjectiveIds" 
                  :completionCriteria="objectiveThemeProgress.completionCriteria"
                  :releaseTrainTitle="isGlobalObj?roadmap.objectiveCustomView.title:releaseTrain.title"/>
                  
                  <vuepopup v-if="showAddMilestoneMenu" @close="hideMilestoneMenu">
					<div class="drop_boxa nwmu arr-top add w-150 piAddmilestoneDrop c"
					     :style="{
					         position: 'fixed',
					         top: milestoneMenuPopupPosition.top !== 0 ? (milestoneMenuPopupPosition.top + 10) + 'px' : '90px',
					         left: milestoneMenuPopupPosition.left !== 0 ? (milestoneMenuPopupPosition.left - 55) + 'px' : '34px'
					       }">
						<ul>
							<li class="c-point" @click="onCreateMilestonePopup">Create Milestone/Phase</li>
							<li class="c-point" @click="onAddExistingMilestone">Add Existing</li>
						</ul>
					</div>
				</vuepopup>
				
				<add-milestone-board v-if="addMilestoneToBoardPopup.show" 
		    		:position="addMilestoneToBoardPopup.position"
		    		:board-items="addMilestoneToBoardPopup.boardItems"
		    		:board-id="addMilestoneToBoardPopup.boardId"
		    		:release-train="addMilestoneToBoardPopup.releaseTrain"
		    		:show-item-edit-menu="false"
		    		:is-obj-custom-view = "isGlobalObj?true:false"
                    :is-okr-milestone="true"
                    :view-type="addMilestoneToBoardPopup.viewType"
		    		@close="onAddMilestoneToBoardPopupClose"
				    @item-selected="onAddToBoardPopupItemSelectedMilestone"
				    @archived-milestones="onArchivedMilestonesError"
		    	/>
		    	
		    	<milestone-popup
					v-if="milestoneData.milestonePopup.show"
					:board-id="milestoneData.milestonePopup.boardId"
					:is-update="milestoneData.milestonePopup.isUpdate"
					:item="milestoneData.milestonePopup.item"
					:item-type="milestoneData.milestonePopup.itemType"
					:key="milestoneData.milestonePopup.isUpdate"
					:read-only="milestoneData.milestonePopup.readOnly"
					:release-train="milestoneData.milestonePopup.releaseTrain"
					:hierarchy-level="milestoneData.milestonePopup.level"
					:statuses="milestoneData.milestonePopup.statuses"
					:selected-criteria="milestoneData.milestonePopup.selectedCalculationCriteria ? milestoneData.milestonePopup.selectedCalculationCriteria : 'c_st'"
					:edit-permission="editMilestonePermission()"
					:delete-permission="deleteMilestonePermission()"
		    	    :link-unlink-permission="linkUnlinkItemsPermission()"
		    	    :is-obj-custom-view-milestone="isGlobalObj?true:false"
		    	    :is-okr-milestone = "true"
					@archive="onClickArchiveMilestone"
					@close="onCloseMilestonePopup"
					@save="onSaveMilestone"
					
		    	></milestone-popup>
                  
                 <async-date-confirmation-popup  :ref="'timeline-item-edit'"/>
                 <kendis-alert :ref="'kendis-alert'"/>
                 
                 <!-- OBJECTIVE ACTIONS MENU -->
                 <vuepopup v-if="objectiveActionsMenu.show" @close="hideObjectiveActionsMenu">
                     <div class="drop_boxa nwmu arr-top add w-180 objectiveActionsMenu"
                          :style="{
                              position: 'fixed',
                              top: objectiveActionsMenu.position.top !== 0 ? (objectiveActionsMenu.position.top + 10) + 'px' : '90px',
                              left: objectiveActionsMenu.position.left !== 0 ? (objectiveActionsMenu.position.left - 90) + 'px' : '34px'
                            }">
                         <ul>
                             <li v-for="action in objectiveActionsMenu.actions" 
                                 :key="action.id"
                                 class="c-point" 
                                 @click="action.callback({objectiveId: objectiveActionsMenu.objectiveId})">
                                 <i :class="action.icon" style="margin-right: 8px;"></i>
                                 {{ action.label }}
                             </li>
                         </ul>
                     </div>
                 </vuepopup>
                 
                        <!-- OBJECTIVE GROUP POPUP -->
                        <dynamic-popup-container
                        v-if="itemTemplate.objGroupTemplate.showEditPopUp"
                        :is-mocking-mode = "true"
                        @close = "onCloseObjGrpPopup"
                        @save="onSaveObjGrpPopup"
                        @delete="onDeleteObjGrpPopup"
                        :template-to-render="itemTemplate.objGroupTemplate.templateToRenderer"
                        :template-statuses = "itemTemplate.objGroupTemplate.statuses"
                        :item-type="itemTemplate.objGroupTemplate.itemType"
                        :fields-value="itemTemplate.objGroupTemplate.model"
                        :template = "itemTemplate.objGroupTemplate"
                        :show-attachments="false"
                        :show-watchers="true"
                        :show-history="false"
                        :show-conversations="true"
                        :show-notify-watchers="false"
                        :show-external-entities="true"
                        :show-sub-phase-linking="false"
                        :teams-linking-title="'Link Team'"
                        :show-sprint-linking="false"
                        :entity-type="'BaseItem'"
                        :entity-id="itemTemplate.objGroupTemplate.id"
                        :is-inline-editing-enabled="false"
                        :show-linked-items="false"
                        :show-tagging-info="false"
                        :is-art="true" />
                         <!-- OBJECTIVE  POPUP -->
                        <dynamic-popup-container 
                            v-if="itemTemplate.objTemplate.showEditPopUp"
                            ref="active-objective"
                            :display-mode="displayMode"
                            :is-mocking-mode = "true"
                            :is-global-objective-view="false"
                            :is-art = "true"
                            :release-train-id="releaseTrainId"
                            :board = "itemTemplate.objTemplate.board"
                            :boards = "itemTemplate.objTemplate.boardSession"
                            :template-to-render="itemTemplate.objTemplate.templateToRenderer"
                            :template-statuses = "itemTemplate.objTemplate.statuses"
                            :item-type="itemTemplate.objTemplate.itemType"
                            :fields-value="itemTemplate.objTemplate.model"
                            :creation-and-update-data="itemTemplate.objTemplate.creationAndUpdateData"
                            :template = "itemTemplate.objTemplate"
                            :show-attachments="false"
                            :show-objective-linking="true"
                            :show-watchers="true"
                            :show-history="true"
                            :show-conversations="true"
                            :show-external-entities="true"
                            :show-sub-phase-linking="true"
                            :teams-linking-title="'Link Team'"
                            :show-sprint-linking="false"
                            :entity-type="'BaseItem'"
                            :entity-id="itemTemplate.objTemplate.id"
                            :is-inline-editing-enabled="true"
                            :data-loaded="objectiveModalDataLoaded"
                            :items-loaded="true"
                            :sessions-map="itemTemplate.objTemplate.sessionsMap"
                            :disable-teams-and-groups="false"
                            :show-board-linked-item="false"
                            :teams-and-groups-options  ="itemTemplate.objTemplate.objectiveGroupsCopy"
                            :kr-parent-time-period-field-id="itemTemplate.objTemplate.krParentTimePeriodFieldId"
                            :kr-parent-start-date-field-id="itemTemplate.objTemplate.krParentStartDateFieldId"
                            :kr-parent-end-date-field-id="itemTemplate.objTemplate.krParentEndDateFieldId"
                            :disable-date-based-on-parent-time-period="false"
                            :objective-group-id="itemTemplate.objTemplate.objectiveGroupId"
                            :kr-parent="itemTemplate.objTemplate.krParent"
                            :add-okr-permission="$options.permissions.EDIT_OKR"
                            :edit-permission="(  $options.permissions.EDIT_OBJECTIVE )"
                            :delete-permission="(   $options.permissions.DELETE_OBJECTIVE )"
                            :delete-okr-permission=" $options.permissions.DELETE_OKR"
                            :link-unlink-permission="(  $options.permissions.LINK_UNLINK_OBJECTIVE_ITEMS )"
                            :allow-open-sub-item = "false"
                            @open-key-result-container="openKeyResultContainer"
                            @close = "onCLoseEditPopUp"
                            @save = "onSaveEditObjectiveData"                             
                            @delete-objective-okr="onDeleteFromPopup"
                            @objectiveCompletionCriteria = "onObjectiveCompletionCriteria"
                            @remove-okr="removeOkr"
	                        @unlink-obj="unlinkObj"
                        />
                        <!-- KEY RESULT POPUP -->
                        <dynamic-popup-container 
                            v-if="itemTemplate.krTemplate.showEditPopUp"
                            ref="active-objective"
                            :display-mode="displayMode"
                            :is-mocking-mode = "true"
                            :is-global-objective-view="false"
                            :is-art = "true"
                            :release-train-id="releaseTrainId"
                            :board = "itemTemplate.krTemplate.board"
                            :boards = "itemTemplate.krTemplate.boardSession"
                            :template-to-render="itemTemplate.krTemplate.templateToRenderer"
                            :template-statuses = "itemTemplate.krTemplate.statuses"
                            :item-type="itemTemplate.krTemplate.itemType"
                            :fields-value="itemTemplate.krTemplate.model"
                            :creation-and-update-data="itemTemplate.krTemplate.creationAndUpdateData"
                            :template = "itemTemplate.krTemplate"
                            :show-attachments="false"
                            :show-objective-linking="true"
                            :show-watchers="true"
                            :show-history="true"
                            :show-conversations="true"
                            :show-external-entities="true"
                            :show-sub-phase-linking="true"
                            :teams-linking-title="'Link Team'"
                            :show-sprint-linking="false"
                            :entity-type="'BaseItem'"
                            :entity-id="itemTemplate.krTemplate.id"
                            :is-inline-editing-enabled="true"
                            :data-loaded="krModalDataLoaded"
                            :items-loaded="true"
                            :sessions-map="itemTemplate.krTemplate.sessionsMap"
                            :disable-teams-and-groups="false"
                            :show-board-linked-item="false"
                            :teams-and-groups-options  ="itemTemplate.krTemplate.objectiveGroupsCopy"
                            :kr-parent-time-period-field-id="itemTemplate.krTemplate.krParentTimePeriodFieldId"
                            :kr-parent-start-date-field-id="itemTemplate.krTemplate.krParentStartDateFieldId"
                            :kr-parent-end-date-field-id="itemTemplate.krTemplate.krParentEndDateFieldId"
                            :kr-parent="itemTemplate.krTemplate.krParent"
                            :objective-group-id="itemTemplate.krTemplate.objectiveGroupId"
                            :disable-date-based-on-parent-time-period="false"
                            :parent-key="itemTemplate.krTemplate.parentKey"
                            @keyResultCompletionCriteria="keyResultCompletionCriteria"
                            @open-key-result-container="openKeyResultContainer"
                            @close = "onCLoseEditKRPopUp"
                            @save = "onSaveEditObjectiveData"                             
                            @delete-objective-okr="onDeleteFromPopup"
                            @objectiveCompletionCriteria = "onObjectiveCompletionCriteria"
                        />
                        <!-- Objective Link Popup -->
                <objective-link v-if="showObjectiveLinkingPopup && selectedObjectiveGroup" 
                    :board="releaseTrain"
                    :item-key="selectedObjectiveGroup.key"
                    :okr-id="selectedObjectiveGroup.id"
                    :is-mocking-mode="true"
                    :linked-objectives="selectedObjectiveGroup.baseItemList ? selectedObjectiveGroup.baseItemList.filter(item => item.type === 'Objective') : []"
                    :associate-with="{
                        id: 'objective',
                        title: 'Child Objectives',
                        tabNum: 1
                    }"
                    :okr="selectedObjectiveGroup"
                    title="Link Child Objectives"
                    :okr-group="null"
                    container="link-item-container"
                    @cancel="showObjectiveLinkingPopup=false; selectedObjectiveGroup=null;"
                    @save="saveLinkedObjectives">
                </objective-link>
                        <objective-group-progress v-if="objectiveGroupProgressCompData.show"  
                        :objectiveGroupId="objectiveGroupProgressCompData.objectiveGroupId" 
                        :isArt="objectiveGroupProgressCompData.isArt" 
                        :isGlobalObjectiveView="false" 
                        :completionBy="objectiveGroupProgressCompData.completionBy" 
                        :radioTypeSelection="objectiveGroupProgressCompData.radioTypeSelection" 
                        :isOkrEnabled="objectiveGroupProgressCompData.isOkrEnabled" 
                        :objectiveList="objectiveGroupProgressCompData.objectiveList" 
                        :board="objectiveGroupProgressCompData.board" 
                        :objectiveGroup="objectiveGroupProgressCompData.objectiveGroup" 
                        :objectiveGroupList="objectiveGroupProgressCompData.objectiveGroupList"
                        :releaseTrainTitle="isGlobalObj?roadmap.objectiveCustomView.title:releaseTrain.title"
                         @closeGroupProgress="closeGroupProgress" />
                        </objective-group-progress>

                    <div class="DFCB roadmapTopBar bryntum" :style="{backgroundColor: getBackgroundColor()}">
                        <div class="DFA lft">
                            <!--
                            <div class="headerBackBtn" @click="onBackClick">
                                <span class="ti-angle-left backIcon"></span>
                                <span class="backText">Back</span>
                            </div> -->
                            <div class="back-arrowaaa mr-10 c-point" @click="onBackClick">
                            	<svg width="18" height="18" viewBox="0 0 18 18" xmlns="http://www.w3.org/2000/svg" :style="{fill: getTextColor()}">
								<path d="M9 18C13.9603 18 18 13.9603 18 9C18 4.03967 13.9603 0 9 0C4.03967 0 0 4.03967 0 9C0 13.9603 4.03967 18 9 18ZM9 0.939457C13.4342 0.939457 17.0605 4.54697 17.0605 9C17.0605 13.453 13.453 17.0605 9 17.0605C4.56576 17.0605 0.939457 13.4342 0.939457 9C0.939457 4.56576 4.56576 0.939457 9 0.939457ZM4.77244 9.3382C4.6785 9.24426 4.64092 9.13152 4.64092 9C4.64092 8.86848 4.69729 8.75574 4.77244 8.6618L7.29019 6.14405C7.47808 5.95616 7.77871 5.95616 7.94781 6.14405C8.1357 6.33194 8.1357 6.63257 7.94781 6.80167L6.21921 8.53027H12.8894C13.1524 8.53027 13.3591 8.73695 13.3591 9C13.3591 9.26305 13.1524 9.46973 12.8894 9.46973H6.238L7.9666 11.1983C8.15449 11.3862 8.15449 11.6868 7.9666 11.8559C7.87265 11.9499 7.75992 11.9875 7.62839 11.9875C7.49687 11.9875 7.38413 11.9499 7.29019 11.8559L4.77244 9.3382Z" fill=""/>
								</svg>
                            </div>
                            <!-- <div class="spr-vrt ml-10 mr-10" :style="{backgroundColor: getTextColor()}"></div> -->                                                     
                        </div><!-- left end --> 
                        <div class="roadmap-title elips" :style="{color: getTextColor()}">{{ roadmapDisplayName }}</div>
                        <div class="DFA rht">
	                       	<!-- Key Results Toggle Switch -->
	                        <div class="sw-btn wboxtp DAJ rnd4">
	                            <div class="switch-button switch-button-sm">
	                                <input id="key-result-toggle" type="checkbox" v-model="showKeyResults" @change="onKeyResultsToggle" />
	                                <span><label for="key-result-toggle"></label></span>
	                            </div>
	                            <span class="lbl wht-nwrp">Key Results</span>
	                            <span class="infotxt mht tooltip ml-5"><span class="hovtooltipB sml wid3" style="left: -36px; top: 27px;">Enable this option to display all supporting Key Results and nested objectives for detailed planning.</span></span>
	                        </div>
	                        <div v-if = "showZoomOptions" class="mnhd fst DAJ" style="gap: 10px;">
		                    	<div class="fit wboxtp c-point DAJ rnd4" @click="fitSchedulerToEvents();">FIT</div>
		                        <div class="ti-fullscreen fulscrn wboxtp c-point DAJ rnd4" :title="isFullScreen ? 'Exit Full Screen' : 'Full Screen'" @click = "toggleFullScreen"></div>
			                         <div class="navigator-controls DFA rnd4">
		                             	<a class="ftnv DFA zoom-btn out c-point ti-zoom-out" @click="zoomTimePeriod('out')"></a>
			                            <a class="ftnv DFA arrbtn lft c-point ti-angle-left" @click="navigateToAdjacentTimePeriod('prev')"></a>
			                            <div class="ftnv DFA navdown DFA p-rel c-point" @click.stop="timeNavigator.showNavigator = !timeNavigator.showNavigator">
			                                <div class="DAJ wht-nwrp">{{ getTimeNavigatorLabel() }}</div>
			                                <time-period-popup v-if="timeNavigator.showNavigator" :is-navigator="true" @navigateToTimePeriod="navigateToTimePeriod" />
			                            </div>
			                            <a class="ftnv DFA arrbtn rht c-point ti-angle-right" @click="navigateToAdjacentTimePeriod('next')"></a>	                            
			                         	<a class="ftnv DFA zoom-btn in c-point ti-zoom-in" @click="zoomTimePeriod('in')"></a>
		                            </div>
			                         <div class="reset-btn DFA c-point ti-reload rnd4 tooltip" @click="resetTimeNavigatorToCurrentYear">
			                         	<div class="hovtooltipB tprhtn sml">Reset to Current Year</div>
			                         </div>								
			                     </div>	                        
					    	</div>
        				</div> <!-- right end -->                     
                    
                    <div class="DFCB roadmapTopBar bryntum">
                        <div class="DFA lft">                            
                            <div class="srchnw DFA">
                                <em @click="showFilterPanel=true" class="ti-filter srico"></em>
                                <input type="text" placeholder="Search..." class="srch" v-model="objectiveFilters.search" @click ="showFilterPanel=true" @input="onInputSearch"/> 
                            </div>
                            <a class="c-point und fs-12 ml-10" v-if="isFilterApplied" href="javascript:void(0);" @click="resetFilters()">Reset Filter</a>
                        </div>
    
                        <div class="DFA rht">
                        	<a href="javascript:void(0);" class="o-add ti-plus addf nwabtn mr-10 rnd4 h-auto" id="add-c" @click="onAddCustomClick" v-if="permissions && permissions.EDIT_OBJECTIVE_GROUP">Add Custom Group</a>
                        	<a href="javascript:void(0);" class="o-add ti-plus addf nwabtn mr-10 rnd4 h-auto" id="add-a" @click="onSelectAddCollectionObjective" v-if="isGlobalObj && permissions && permissions.EDIT_OBJECTIVE_GROUP">Add Collection</a>
                            <div v-if="addMenu.loadingCollections" class="minldr mr-10"><div class="loaderinteg"></div></div>
                        	<div v-if="addMenu.showCollections" class="oplvlslc mr-10" style="display:block; width: 200px;">
							<treeselect
								:multiple="false"
								:options="workspaceList"
								:flat="true"
								placeholder="Search or Select Collection..."
								no-children-text="No Sub Level"
								:clearable="false"
	                            @select="onSelectCollection"
							/>
						    </div>
	                        <div class="c-point prgbtn" @click="initThemeProgressPopup" title="Progress"></div>
				    </div>
				    
                    </div>
                    <div class="roadmap-filters board_fitlers" v-if="showFilterPanel">
        			<div class="prow morefout cmp">
        				
                         <div class="rec">
                            <pi-user-picker class="h-auto" v-model="objectiveFilters.responsible" :multiple="true" :show-create-user="false" :show-button="false" :allow-unassigned="true" placeholder="Responsible" @select="onFilterResponsible"/>
                        </div>
                        <div class="rec">
	                        <div class="selectcont">
	                            <vz-select v-model="objectiveFilters.collections" :show-item-key="true" empty-selection-text="Collection Associations" selected-title="Collection Associations" :multiple="true" track-by="id" value-type="id" label="title" :options="collections" class="multi-filter" @select="onFilterAssociation" ></vz-select>
	                        </div>
                        </div>                        
                         <div class="rec">
	                         <div class="selectcont">
	                            <vz-select v-model="objectiveFilters.statuses" empty-selection-text="Statuses" selected-title="Statuses" :multiple="true" track-by="id" value-type="id" label="title" :options="okrStatuses" class="multi-filter" @select="onFilterStatus"></vz-select>
	                        </div>
                        </div>
<!--                        <div class="rec slctdn DFA p-rel c-point" @click.stop="timePeriodFilter.showFilter = !timePeriodFilter.showFilter">-->
<!--                            <div class="DAJ">Time Period</div>-->
<!--                            <time-period-filter-popup v-if="timePeriodFilter.showFilter" :filter="timePeriodFilter.filter" :view-type="'okrList'" @onTimePeriodFilter="onTimePeriodFilter"  />-->
<!--                        </div>-->
<!--                        <div class="rec slctdn DFA p-rel c-point" @click.stop="timeNavigator.showNavigator = !timeNavigator.showNavigator">-->
<!--                            <div class="DAJ">{{ timeNavigator.selectedTimePeriod ? timeNavigator.selectedTimePeriod.title + ' ' + timeNavigator.selectedTimePeriod.selectedYear : 'Navigate' }}</div>-->
<!--                            <time-period-popup v-if="timeNavigator.showNavigator" :is-navigator="true" @navigateToTimePeriod="navigateToTimePeriod" />-->
<!--                        </div>-->
                        <a @click="showFilterPanel=false" class="cncl p-abs" href="javascript:void(0);">Hide</a>
                    </div>
                    </div>
                    <div id="okrs-roadmap-bryntum" :class="[showFilterPanel?'size-1':'size-0',isMultilevel?'multilevel-view':'legacy-view']"></div>
                </div>

                
            `,
        props: ["releaseTrainId", "viewType", "roadmapId","isGlobalObj","viewId","viewKey"],
        data() {
            return {
                showObjectiveLinkingPopup: false,
                selectedObjectiveGroup: null, // Store the selected objective group for linking
                userPermissions:[],
                departmentsMap:{},
                batchesMap:{},
                sessionsMap:{},
                activeCustomView:{},
                objectiveData:{},

                krModalDataLoaded:false,
                showFilterPanel:false,
                collections:[],
                objectiveFilters: {
                    responsible: [],
                    collections: [],
                    statuses: [],
                    search: "",
                },
                startDateFieldId            : '',
                endDateFieldId              : '',
                responsibleFieldId          : '',
                responsibleKRFieldId        : '',
                startDateKRFieldId          : '',
                endDateKRFieldId            : '',
                timePeriodKRFieldId         : '',

                workspaceList: [],
                displayMode: "Objective",

                addMenu: {
                    show: false,
                    showCollections: false,
                    loadingCollections: false,
                    collections: []
                },
                linkedCustomViewsMap: new Map(),
                objectiveThemeProgress :{
                  show : false,
                    isArt:true,
                    isGlobalObjectiveView:false,
                    radioTypeSelection:undefined,
                    isOkrEnabled:true,
                    objectiveGroupList:undefined,
                    board:undefined,
                    completionCriteria:'OBJ_KR_WEIGHTS_FORMULA',
                    allObjectiveIds:undefined,
                    objectiveGroupsId:undefined

                },
                objectiveGroupProgressCompData:{
                    show:false,
                    objectiveGroupId:"",
                    isArt:true,
                    completionBy:undefined,
                    radioTypeSelection:undefined,
                    isOkrEnabled:true,
                    objectiveList:undefined,
                    board:undefined,
                    objectiveGroup:undefined,
                    objectiveGroupList:undefined
                },
                responsibleFilter:{
                    show:true,
                    selectedValues:[],
                },
                timePeriodFilter            : {
                    showFilter              : false,
                    filter                  : {},
                },
                timeNavigator               : {
                    showNavigator           : false,
                    selectedTimePeriod      : null,
                    customSpanYears         : 2,
                },
                timePeriodFieldId           : '',
                timePeriodKRFieldId         : '',
                objectiveModalDataLoaded: false,
                itemTemplate:{
                    objGroupTemplate:{},
                    objTemplate:{},
                    krTemplate:{}
                } ,
                showUnplannedGroupBy:false,
                isFullScreen:false,
                zoomState:{
                    isCustomZoom:false,
                },
                showZoomOptions: true,
                sectionPopup:{
                    show:false,
                    section:{}
                },
                roadmap: {},
                viewPreset: {
                    value: 'monthAndYear',
                    options: [
                        {id:'weekAndDay', value: 'weekAndDay', title: 'Weeks'},
                        {id:'dayAndWeek', value: 'dayAndWeek', title: 'Days'},
                        {id:'monthAndYear', value: 'monthAndYear', title: 'Months'},
                        {id:'year', value: 'year', title: 'Years'},
                        {id:'mileStoneWrap',value:'mileStoneWrap',title:'Milestones'},
                        {id:'custom',value:'c',title:'Custom'}
                    ]
                },

                showAddPopup:false,
                loadingItemRollups: false,
                roadmapColumnTitles: [],
                level: "",
                loader: {
                    show: false,
                    text: "Loading Roadmap"
                },
                itemEditPopup: {
                    show: false,
                    item: undefined,
                    changes: {}
                },

                itemPopup: {
                    parent: undefined,
                    item: undefined,
                    alm: undefined,
                    tfsProjects: undefined,
                    show: false,
                },
                addFromAlmPopup: {
                    show: false,
                    backlogLevel: 0,
                    alm: undefined,
                    parent: undefined,
                    item: []
                },
                sort: {
                    fields: []
                },
                defaultDurationUnit: "d",
                defaultDuration: 14,
                isdragRemoveFromGrid: false, //it controlles if the item from grid is deleted or dragged
                isChangeGroupBy: false,
                settings: {
                    show: false
                },
                showUnplannedGrid: true,
                selectedOptionsForYearWiseTimeCapsules:[],
                linkItemPopUp: {
                    showLinkItemPopup: false,
              
                },
                selectedReleaseTrain: {},
                releaseTrains: [],

                actionPopup: {
                    showPopup: false,
                    popupActions: [
                    ],
                    context:{}
                },

                //MILESTONES DATA
                milestoneMapById : {},

                milestoneData: {
                    milestones: [],
                    milestoneGroup: {},
                    loaded: false,
                    milestonePopup: {
                        show: false,
                        readOnly: false,
                        releaseTrain: {},
                        level: {},
                        statuses: [],
                        isUpdate: false,
                        item: {},
                        itemType: "Milestone",
                        selectedCalculationCriteria: undefined,
                        boardId: undefined,
                    },
                    linkItemConfirmationPopup: false,
                    confirmationPopup: {
                        show: false,
                        doNotAsk: {
                            message: "Don't ask me again",
                            isEnabled: true,
                        },
                        title: "",
                        description: "",
                        request: {},
                        onClickConfirm: this.onClickConfirmLinking,
                        onClickCancel: this.onClickCancelConfirmation,
                    }
                },
                showAddMilestoneMenu: false,
                milestoneMenuPopupPosition: { top: 0, left: 0 },
                addMilestoneToBoardPopup: {
                    show: false,
                    boardItems: [],
                    position:{left:0,top:0}
                },

                // Objective Actions Menu
                objectiveActionsMenu: {
                    show: false,
                    position: { top: 0, left: 0 },
                    objectiveId: '',
                    actions: []
                },

                // KR Creation Data (similar to objective-row-component.js)
                selectedObjectiveForKr: null,
                newOkr: {
                    title: '',
                    bvPlan: 0,
                    bvActual: 0,
                    fields: {}
                },
                timePeriod: {
                    period: null,
                    startDate: null,
                    endDate: null,
                    months: [
                        "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
                    ]
                },
                savingKeyResult: false,
                okrProgressCriteriaField: {},
                statusAndEstimateFilters: [],

                // Child Objective Creation Data
                selectedObjectiveForChild: null,
                newChildObjective: {
                    title: '',
                    bvPlan: 0,
                    bvActual: 0,
                    fields: {}
                },
                timePeriodChild: {
                    period: null,
                    startDate: null,
                    endDate: null,
                    months: [
                        "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
                    ]
                },
                savingChildObjective: false,

                isMultilevel: false, // Flag to control multilevel view vs legacy view
                viewMode: 'objectives', // 'objectives' for legacy, 'multilevel' for new
                showKeyResults: false, // Controls the Key Results toggle switch
                includeLinkedItems:false,
                showZoomOptions:true,
                objectiveModalDataLoaded:false,
                viewPreset: {
                    value: 'monthAndYear',
                    options: [
                        {id:'weekAndDay', value: 'weekAndDay', title: 'Weeks'},
                        {id:'dayAndWeek', value: 'dayAndWeek', title: 'Days'},
                        {id:'monthAndYear', value: 'monthAndYear', title: 'Months'},
                        {id:'year', value: 'year', title: 'Years'},
                        {id:'mileStoneWrap',value:'mileStoneWrap',title:'Milestones'},
                        {id:'custom',value:'c',title:'Custom'}
                    ]
                },
                itemTemplate: {
                    objTemplate:{
                      showEditPopUp : false,
                        model:{},
                        statuses:[],
                        templateToRenderer:{},
                        board:{},
                        objectiveGroupsCopy:[],
                        itemType:"",
                        krParentTimePeriodFieldId:"",
                        krParentStartDateFieldId:"",
                        krParentEndDateFieldId:"",
                        objectiveGroupId:"",
                        krParent:{}
                    },
                    krTemplate:{
                        showEditPopUp : false,
                        model:{},
                        statuses:[],
                        templateToRenderer:{},
                        board:{},
                        objectiveGroupsCopy:[],
                        itemType:"",
                        krParentTimePeriodFieldId:"",
                        krParentStartDateFieldId:"",
                        krParentEndDateFieldId:"",
                        objectiveGroupId:"",
                        krParent:{}
                    },
                    objGroupTemplate: {
                        showEditPopUp : false,
                        templateToRenderer:{},
                        statuses:[],
                        itemType:"",
                        model:{},
                        id:""
                    }
                },
                timePeriodFieldId: '',
                timePeriodKRFieldId:'',
                permissions:{},
                searchQuery: "",
                timePeriodFilter: {
                    showFilter: false,
                    filter: {}
                },
                groupBy: {
                    value: 'Section',
                    options: [
                        {id:'Section',value:'Section',title:'Section'},
                        {id:'NoGroup',value:'NoGroup',title:'No Group'}
                    ]
                },
                isFullScreen:false,
                cacheLoaded:false
            }
        },
        computed: {
            okrStatuses() {
                const statsues = kendisStore.getters.getObjectiveTemplate().statuses;
                if (_.isEmpty(statsues)) {
                    return [];
                }
                return statsues;
            },
            isFilterApplied() {
                // Check for actual filter values (excluding time navigator)
                const hasTimePeriodFilter = !_.isEmpty(this.timePeriodFilter.filter)
                    && !_.isEmpty(this.timePeriodFilter.filter.selectedValues);
                const hasObjectiveFilters = Object.values(this.objectiveFilters).some(filter => filter.length > 0);
                const hasSearchQuery = this.searchQuery && this.searchQuery.trim().length > 0;

                // Only check main filters: time period filter, objective filters, and search query
                // Time navigator is excluded since it's now independent of the main filter system
                return hasTimePeriodFilter || hasObjectiveFilters || hasSearchQuery;
            },
            store() {
                return this.$store;
            },
            getOkrCompletionMap: function() {
                return kendisStore.state.okrCompletionMap;
            },
            visibleTags() {
                return this.filters.searchTags.slice(0, 5);
            },
            hiddenTags() {
                return this.filters.searchTags.slice(5);
            },
            anyMessage () {
                return this.store.message.text == '' ? false : true;
            },
            roadmapYearsConfigsMap() {
                let map = {};
                if (this.roadmap.roadmapYearsConfigs) {
                    _.each(this.roadmap.roadmapYearsConfigs, config => {
                        map[config.year] = config.partitionViewType;
                    });
                }
                return map
            },
            releaseTrain() {
                if(!this.isGlobalObj) {
                    return this.store.releaseTrain;
                }else{
                    return {id:this.releaseTrainId}
                }
            },
            batchesLoaded() {
                return this.$root.flags.batches;
            },
            boardsLoaded() {
                return this.$root.flags.boards;
            },
            teamsLoaded() {
                return this.$root.flags.teams;
            },
            containersLoaded() {
                return this.$root.flags.containers;
            },
            solutionsLoaded() {
                return this.$root.flags.solutionBoards;
            },
            schedulerModuleLoaded() {
                return true;
            },
            metaLoaded() {
                return this.schedulerModuleLoaded && this.releaseTrain && this.releaseTrain.id;
            },
            backlogHierarchyLevels() {
                return this.store.backlogHierarchyLevels;
            },
            currentHierarchyLevel() {
                if (this.backlogHierarchyLevels && this.roadmap && this.roadmap.id) {
                    let index = this.roadmap.level;
                    if ( typeof index == "string") {
                        index = parseInt(index);
                    }
                    let level = _.find(this.backlogHierarchyLevels,{level: index});
                    return level;
                }
                return undefined;
            },


            almAccount() {
                if(this.isGlobalObj){
                    return {}//TODO
                }
                if (this.store.almAccounts && this.store.almAccounts[0]) {
                    return this.store.almAccounts[0];
                }
            },
            roadmapDisplayName() {
                // If global view and activeCustomView is set, use its title (Strategic Theme/Custom View)
                if (this.isGlobalObj && this.activeCustomView && this.activeCustomView.title) {
                    return this.activeCustomView.title;
                }
                // If global view and a collection is selected, use the collection's title
                if (!this.isGlobalObj && this.store.releaseTrain &&  this.store.releaseTrain.title) {
                   return this.store.releaseTrain.title
                }
                // Fallback: default roadmap name
                return  '';
            },
        },
        created() {
            console.log("-------------------------------------");
            console.log("Init OKR ROADMAP");
            console.log("-------------------------------------");
            roadmapVue = this;
            if (kendisStore.getters.getArtCollections().length === 0) {
                this.loadCollections();
            }else{
                this.collections = kendisStore.getters.getArtCollections();
            }
            this.setOptions()
            this.loadSchedulerConfigs();
        },
        mounted() {

            // this.loadReleaseTrains();
            //this.setBoardStartDate();
            this.initView();

            console.log("-------------------------------------");
            console.log("OKR ROADMAP mounted");
            console.log("-------------------------------------");
            // Initialize view mode from localStorage
            try {
                const savedMode = localStorage.getItem('okrs-roadmap-view-mode');
                if (savedMode && (savedMode === 'objectives' || savedMode === 'multilevel')) {
                    this.viewMode = savedMode;
                    this.isMultilevel = (savedMode === 'multilevel');
                    this.showKeyResults = (savedMode === 'multilevel');
                }
            } catch (error) {
                console.warn('Error loading view mode from localStorage:', error);
            }
        },
        beforeRouteLeave(to, from, next) {
            // Remove filters from URL before navigating away
            this.clearFiltersFromURL();
            next();
        },
        watch: {
            artCollections: {
                handler: function (_new, _old) {
                    if (!_.isEmpty(_new)) {
                        this.collections = _new;
                    }
                }
            },
            "releaseTrain":{
                handler(newValue, oldValue) {
                    if (newValue !== null) {
                        this.initView()
                    }
                }
            },
            getOkrCompletionMap: {
                handler: function(_new, _old) {
                    if(this.getOkrCompletionMap ) {
                        this.updateCompletionPercentages();

                    }
                },
                deep: true,
                immediate: true
            },
            viewMode: async function(newMode) {
                this.isMultilevel = (newMode === 'multilevel');
                // Recreate the scheduler with the new mode
                this.resetScheduler();
                await this.createObjectiveGroups();
                // Reload milestones after recreating the scheduler
                await this.loadMilestones();
                this.updateCompletionPercentages();
            }
        },
        beforeDestroy() {
            // Cleanup event listener if component is destroyed
            document.removeEventListener('click', this.handleClickOutside);
            // Ensure left menu bar is hidden when component is destroyed
            if(this.store) {
                this.store.setShowLeftMenuBar(true);
                this.store.setShowBreadCrumbs(true);
            }
        },
        methods: {
            keyResultCompletionCriteria(criteria,krId){
                //update the completion criteria of the key result
                let kr = this.$options.cache.keyResultsMap.get(krId);
                if(kr && criteria && criteria.code){
                    kr.completionCriteria = criteria.code;
                    this.$options.cache.keyResultsMap.set(krId,kr);
                    //save to db
                    this.onSaveEditOkr(kr);
                }

            },
            getBackgroundColor() {
                //if global view and activeCustomView is set, use the color of the activeCustomView
                if (this.isGlobalObj && this.activeCustomView && this.activeCustomView.color) {
                    return this.activeCustomView.color;
                }
                //if not global view and releaseTrain is set, use the color of the releaseTrain
                //decrease the opacity of the color by 50%
                if (!this.isGlobalObj && this.store.releaseTrain && this.store.releaseTrain.color) {
                    // Convert hex color to rgba with 50% opacity
                    const hexColor = this.store.releaseTrain.color;
                    if (hexColor.startsWith('#')) {
                        const r = parseInt(hexColor.slice(1, 3), 16);
                        const g = parseInt(hexColor.slice(3, 5), 16);
                        const b = parseInt(hexColor.slice(5, 7), 16);
                        return `rgba(${r}, ${g}, ${b}, 0.5)`;
                    }
                    return hexColor; // Return as-is if not a hex color
                }
                // Default background color if no conditions are met
                return '#ffffff';
            },
            getTextColor() {
                //if global view and activeCustomView is set, use the color of the activeCustomView
                if (this.isGlobalObj && this.activeCustomView && this.activeCustomView.color) {
                    return this.activeCustomView.textColor;
                }
                //if not global view and releaseTrain is set, use the color of the releaseTrain
                if (!this.isGlobalObj && this.store.releaseTrain && this.store.releaseTrain.color) {
                    return this.store.releaseTrain.color;
                }
                // Default text color if no conditions are met
                return '#000000';
            },
            // //newfilters:

            // doFilterObjectiveList: function (searchQuery, eventType) {
            //     if (!_.isEmpty(searchQuery) && eventType === "insertText" || eventType === "deleteContentBackward") {
            //         toggleDraggable(true);
            //     }
            //     if (this.isMockingMode) {
            //         this.objectiveFilters.searchQuery = searchQuery;
            //         this.loadObjectiveGroupsListByObjectId(true);
            //     }
            // },
            onInputSearch: function (e) {
                kendisStore.commit("setObjectivesSearchQuery", this.searchQuery);
                this.reloadObjectiveGroups();

                // Save filters and update URL
                this.saveFiltersToStorage();
                this.updateURLWithFilters();
            },
            resetFilters() {
                this.clearAllFilters();
            },



            onViewModeChange: function(mode) {
                this.viewMode = mode;
                this.isMultilevel = (mode === 'multilevel');
                this.showKeyResults = (mode === 'multilevel');
                // Save to localStorage
                try {
                    localStorage.setItem('okrs-roadmap-view-mode', mode);
                } catch (error) {
                    console.warn('Error saving view mode to localStorage:', error);
                }
            },

            // Objective Actions Menu Methods
            showObjectiveActionsMenu(event, objectiveId) {
                let _this = this;
                event.stopPropagation();

                // Set menu position
                this.objectiveActionsMenu.position = {
                    top: event.clientY || event.pageY || 0,
                    left: event.clientX || event.pageX || 0
                };

                this.objectiveActionsMenu.objectiveId = objectiveId;

                // Define available actions based on permissions
                let actions = [];

                // Edit action (always available with permission)
                if (ComponentPermissionManager.hasPermission(this, 'EDIT_OBJECTIVE')) {
                    actions.push({
                        id: 'edit',
                        label: 'Edit Objective',
                        icon: 'ti-edit',
                        callback: function(context) {
                            _this.handleObjectiveEdit(context.objectiveId);
                        }
                    });
                }

                // Create KR action
                if (ComponentPermissionManager.hasPermission(this, 'EDIT_OKR')) {
                    actions.push({
                        id: 'createKR',
                        label: 'Create Key Result',
                        icon: 'ti-plus',
                        callback: function(context) {
                            _this.handleCreateKR(context.objectiveId);
                        }
                    });
                }

                // Create Objective action
                if (ComponentPermissionManager.hasPermission(this, 'EDIT_OBJECTIVE')) {
                    actions.push({
                        id: 'createObjective',
                        label: 'Create Objective',
                        icon: 'ti-plus',
                        callback: function(context) {
                            _this.handleCreateObjective(context.objectiveId);
                        }
                    });
                }

                this.objectiveActionsMenu.actions = actions;
                this.objectiveActionsMenu.show = true;
            },

            hideObjectiveActionsMenu() {
                this.objectiveActionsMenu.show = false;
                this.objectiveActionsMenu.objectiveId = '';
                this.objectiveActionsMenu.actions = [];
            },

            handleObjectiveEdit(objectiveId) {
                let objective = this.$options.cache.objectivesMap.get(objectiveId);
                if (objective) {
                    this.openObjectiveEditPopup(objective);
                }
                this.hideObjectiveActionsMenu();
            },

            handleCreateKR(objectiveId) {
                let _this = this;
                let objective = this.$options.cache.objectivesMap.get(objectiveId);
                if (!objective) {
                    showTopMessageTargeted("Objective not found", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    this.hideObjectiveActionsMenu();
                    return;
                }

                // Initialize data structures similar to objective-row-component.js
                this.selectedObjectiveForKr = objective;
                this.newOkr = {
                    title: '',
                    bvPlan: 0,
                    bvActual: 0,
                    fields: {}
                };
                this.timePeriod = {
                    period: null,
                    startDate: null,
                    endDate: null
                };
                this.savingKeyResult = false;

                // Get field IDs from template
                let okrTemplate = kendisStore.getters.getOkrTemplate();

                // Find Progress Criteria field from the template structure
                this.okrProgressCriteriaField = {};
                if (okrTemplate.scheme && okrTemplate.scheme.fieldsTemplate && okrTemplate.scheme.fieldsTemplate.configuredFields) {
                    // Look for Progress Criteria group
                    let progressCriteriaGroup = okrTemplate.scheme.fieldsTemplate.configuredFields.find(field =>
                        field.title === 'Progress Criteria' && field.isGroup
                    );

                    if (progressCriteriaGroup && progressCriteriaGroup.groupFields) {
                        // Find the actual Progress Criteria field within the group
                        this.okrProgressCriteriaField = progressCriteriaGroup.groupFields.find(field =>
                            field.title === 'Progress Criteria'
                        ) || {};
                    }
                }

                // Don't set default progress criteria - let user select
                if (!_.isEmpty(this.okrProgressCriteriaField)) {
                    this.newOkr.fields[this.okrProgressCriteriaField.id] = "";
                }

                // Open async date confirmation popup for KR creation using proper method
                let input = {};
                input.itemIdentifier = "New Key Result";
                input.startDate = undefined;
                input.endDate = undefined;
                input.showTextField = true;
                input.showDateFields = true;
                input.textValue = "New Key Result";
                input.textInputLabel = "Enter Key Result title";
                input.showTimePeriod = true;
                input.maxLength = 750;

                // Add date restrictions for KR based on parent objective's time period
                input.parentObjective = objective;
                input.isKrCreation = true;
                input.dateRestrictions = this.getKrDateRestrictions(objective);
                
                // Add progress criteria options for KR creation
                input.showProgressCriteria = true;
                input.progressCriteria = ""; // No default value - user must select
                
                // Pass parent objective for default date setting
                input.parentObjective = objective;
                input.startDateFieldId = this.startDateFieldId;
                input.endDateFieldId = this.endDateFieldId;

                let promise = this.openAndHandlePopup(input);
                promise.then((response) => {
                    this.saveKrFromAsyncPopup(objective, response);
                }).catch((error) => {
                    console.log("KR creation cancelled", error);
                });

                this.hideObjectiveActionsMenu();
            },

            handleCreateChildObjective(objectiveId) {
                let _this = this;
                let objective = this.$options.cache.objectivesMap.get(objectiveId);
                if (!objective) {
                    showTopMessageTargeted("Objective not found", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    return;
                }

                // Initialize data structures for child objective creation
                this.selectedObjectiveForChild = objective;
                this.newChildObjective = {
                    title: '',
                    bvPlan: 0,
                    bvActual: 0,
                    fields: {}
                };
                this.timePeriodChild = {
                    period: null,
                    startDate: null,
                    endDate: null
                };
                this.savingChildObjective = false;

                // Open async date confirmation popup for child objective creation
                let input = {};
                input.itemIdentifier = "New Child Objective";
                input.startDate = undefined;
                input.endDate = undefined;
                input.showTextField = true;
                input.showDateFields = true;
                input.textValue = "New Child Objective";
                input.textInputLabel = "Enter Child Objective title";
                input.showTimePeriod = true;
                input.maxLength = 750;
                
                // Add date restrictions for child objective based on parent objective's dates
                input.parentObjective = objective;
                input.isChildObjectiveCreation = true;
                input.dateRestrictions = this.getChildObjectiveDateRestrictions(objective);
                input.startDateFieldId = this.startDateFieldId;
                input.endDateFieldId = this.endDateFieldId;

                let promise = this.openAndHandlePopup(input);
                promise.then((response) => {
                    this.saveChildObjectiveFromAsyncPopup(objective, response);
                }).catch((error) => {
                    console.log("Child Objective creation cancelled", error);
                });
            },

            handleLinkObjective(objectiveId) {
                // Check permissions
                if (!ComponentPermissionManager.hasPermission(this, 'LINK_UNLINK_OBJECTIVE_ITEMS')) {
                    ComponentPermissionManager.showPermissionDenied(this, 'link objectives');
                    return;
                }

                // Get the objective (parent objective)
                let objective = this.$options.cache.objectivesMap.get(objectiveId);
                if (!objective) {
                    showTopMessageTargeted("Objective not found", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    return;
                }

                // Store the selected objective and show the linking popup
                this.selectedObjectiveGroup = objective; // Reusing the same property but now it holds an objective
                this.showObjectiveLinkingPopup = true;
            },

            /**
             * Handles removal of a Key Result from the scheduler
             * @param {Object} okr - The Key Result object to remove
             */
            removeOkr(okr) {
                // Remove the KR event from the scheduler
                const eventRecord = scheduler.eventStore.getById(okr.id);
                if (eventRecord) {
                    scheduler.eventStore.remove(eventRecord);
                }
                
                // // Update the parent objective's baseItemList
                // const parentObjective = this.$options.cache.objectivesMap.get(okr.parentId || okr.objectiveId);
                // if (parentObjective && parentObjective.baseItemList) {
                //     _.remove(parentObjective.baseItemList, { id: okr.id });
                    
                //     // Update the cache
                //     let storePayload = {};
                //     storePayload.objective = parentObjective;
                //     kendisStore.commit("updateObjectiveAndOkrsInGroups", storePayload);
                // }
                
                // // Remove from keyResultsMap cache
                this.$options.cache.keyResultsMap.delete(okr.id);
                
                console.log(`Key Result "${okr.title}" removed from scheduler`);
            },

            /**
             * Handles unlinking of a child objective from the scheduler
             * @param {Object} obj - The child objective object to unlink
             */
            unlinkObj(obj) {
                // Remove the child objective event from the scheduler
                const eventRecord = scheduler.eventStore.getById(obj.id);
                if (eventRecord) {
                    scheduler.eventStore.remove(eventRecord);
                }
                
                // // Update the parent objective's baseItemLinks
                // const parentObjective = this.$options.cache.objectivesMap.get(obj.parentId || obj.objectiveId);
                // if (parentObjective && parentObjective.baseItemLinks) {
                //     _.remove(parentObjective.baseItemLinks, { baseItemId: obj.id });
                    
                //     // Update the cache
                //     let storePayload = {
                //         okr: {},
                //         objectiveId: ""
                //     };
                //     storePayload.okr.id = parentObjective.id;
                //     storePayload.objectiveId = parentObjective.id;
                //     storePayload.okr.baseItemList = parentObjective.baseItemList;
                //     storePayload.okr.baseItemLinks = parentObjective.baseItemLinks;
                //     storePayload.fields = ["baseItemList", "baseItemLinks"];
                //     kendisStore.commit("updateObjectiveAndOkrsInGroups", storePayload);
                // }
                
                // Remove from objectivesMap cache
                this.$options.cache.objectivesMap.delete(obj.id);
                
                console.log(`Child Objective "${obj.title}" unlinked from scheduler`);
            },

            saveLinkedObjectives(linkedObjectives) {
                // Convert map/object to array of objectives
                let objectivesArray = [];
                if (linkedObjectives && typeof linkedObjectives === 'object') {
                    // If it's a map/object with keys as IDs
                    objectivesArray = Object.values(linkedObjectives);
                } else if (Array.isArray(linkedObjectives)) {
                    // If it's already an array
                    objectivesArray = linkedObjectives;
                }

                if (!objectivesArray || objectivesArray.length === 0) {
                    this.showObjectiveLinkingPopup = false;
                    return;
                }

                const parentObjective = this.selectedObjectiveGroup; // This is now an objective, not an objective group

                // Prepare itemContainers for updateChildBaseItems (following objective-row-component pattern)
                let itemContainers = [];
                let linkedItemMap = {};
                
                // Build linkedItemMap from existing baseItemLinks
                if (parentObjective.baseItemLinks) {
                    parentObjective.baseItemLinks.forEach(link => {
                        if (link.baseItemId) {
                            linkedItemMap[link.baseItemId] = link;
                        }
                    });
                }

                // Process each linked objective
                objectivesArray.forEach(objective => {
                    // Add the objective to the cache if not already present
                    if (!this.$options.cache.objectivesMap.has(objective.id)) {
                        this.$options.cache.objectivesMap.set(objective.id, objective);
                    }

                    // Create itemContainer following the objective-row-component pattern
                    let itemContainer = {};
                    if (linkedItemMap[objective.id]) {
                        itemContainer.id = linkedItemMap[objective.id].id;
                    }
                    itemContainer.sessionId = this.isGlobalObj ? "-1" : this.releaseTrain.id;
                    itemContainer.sessionBoardId = this.isGlobalObj ? "-1" : this.releaseTrain.id;
                    itemContainer.baseItemId = objective.id;
                    itemContainer.linkType = "parent_okr"; // Use parent_okr for child objectives
                    itemContainers.push(itemContainer);
                });

                // Use the same updateChildBaseItems method as objective-row-component
                this.updateChildBaseItems(itemContainers, parentObjective, objectivesArray);
            },

            updateChildBaseItems(itemContainers, parentObjective, linkedObjectives) {
                let _this = this;
                let requestBody = {};
                requestBody.id = parentObjective.id;
                requestBody.baseItemLinks = itemContainers;

                let metaMap = {};
                metaMap.sessionId = this.isGlobalObj ? "-1" : this.releaseTrain.id;
                requestBody.metaMap = JSON.stringify(metaMap);

                axios.post('/sos/update-base-item-links/' + syncId + '/' + null, requestBody)
                    .then(response => {
                        if (response.data) {
                            showTopMessageTargeted("Child Objectives linked successfully", 'success', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                            
                            // Hide the popup
                            _this.showObjectiveLinkingPopup = false;
                            _this.selectedObjectiveGroup = null;

                            // Update parent objective's baseItemList with linked objectives
                            if (!parentObjective.baseItemList) {
                                parentObjective.baseItemList = [];
                            }

                            // Add linked objectives to parent's baseItemList
                            linkedObjectives.forEach(objective => {
                                let existingIndex = parentObjective.baseItemList.findIndex(item => item.id === objective.id);
                                if (existingIndex === -1) {
                                    // Create child objective entry for baseItemList
                                    let childObjectiveEntry = {
                                        id: objective.id,
                                        title: objective.title,
                                        type: 'Objective',
                                        percentDone: objective.percentDone || 0,
                                        fields: objective.fields || {},
                                        key: objective.key || '',
                                        status: objective.status || { category: "ToDo" }
                                    };
                                    parentObjective.baseItemList.push(childObjectiveEntry);
                                }

                                // Add child objective event to scheduler (same pattern as created child objectives)
                                _this.addLinkedObjectiveToScheduler(objective, parentObjective);
                            });

                            // Update cache with updated parent objective
                            _this.$options.cache.objectivesMap.set(parentObjective.id, parentObjective);

                            // Update store
                            let storePayload = {};
                            storePayload.objective = parentObjective;
                            kendisStore.commit("updateObjectiveAndOkrsInGroups", storePayload);

                            showTopMessageTargeted(`${linkedObjectives.length} objective(s) linked successfully`, 'success', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                        }
                    })
                    .catch(error => {
                        console.error("Child objective linking error", error);
                        showTopMessageTargeted("Error linking child objectives", 'error', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                        _this.showObjectiveLinkingPopup = false;
                        _this.selectedObjectiveGroup = null;
                    });
            },

            addLinkedObjectiveToScheduler(childObjective, parentObjective) {
                // Check if event already exists
                if (!scheduler.eventStore.getById(childObjective.id)) {
                    // Create the event only if it doesn't exist
                    let childObjectiveEvent = {
                        id: childObjective.id,
                        name: childObjective.title,
                        startDate: childObjective.fields && childObjective.fields[this.startDateFieldId] 
                            ? new Date(childObjective.fields[this.startDateFieldId])
                            : new Date(),
                        endDate: childObjective.fields && childObjective.fields[this.endDateFieldId] 
                            ? new Date(childObjective.fields[this.endDateFieldId])
                            : new Date(Date.now() + (30 * 24 * 60 * 60 * 1000)), // 30 days later
                        type: 'child-objective',
                        cls:'kendis-kr-roadmap-event',
                        percentDone: childObjective.percentDone || 0,
                        eventColor: childObjective.status && childObjective.status.category === "ToDo" ? "#E79362" : 
                                   childObjective.status && childObjective.status.category === "InProgress" ? "#3792C9" : 
                                   childObjective.status && childObjective.status.category === "Done" ? "#3FCA93" : "#7c50b1"
                    };

                    // Add event to scheduler
                    scheduler.eventStore.add(childObjectiveEvent);
                }
                
                // Always add assignment for all parent objective resources (original and clones)
                let parentResources = this.isMultilevel ? 
                    scheduler.resourceStore.query(resource => 
                        resource.get('originalObjectiveId') === parentObjective.id || resource.id === parentObjective.id
                    ) : [scheduler.resourceStore.getById(parentObjective.id)];
                
                parentResources.forEach((resource, index) => {
                    if (resource) {
                        // Generate unique assignment ID to avoid conflicts
                        let assignmentId = `a-${childObjective.id}-${resource.id}`;
                        // Check if assignment already exists to avoid duplicates
                        if (!scheduler.assignmentStore.getById(assignmentId)) {
                            scheduler.assignmentStore.add({
                                id: assignmentId,
                                resourceId: resource.id,
                                eventId: childObjective.id
                            });
                        }
                    }
                });
            },



            // Child Objective creation method (similar to KR creation but for objectives)
            saveChildObjectiveFromAsyncPopup(parentObjective, response) {
                let _this = this;

                if (!_this.savingChildObjective) {
                    _this.savingChildObjective = true;

                    // Set response data to newChildObjective
                    _this.newChildObjective.title = response.textValue || '';
                    _this.timePeriodChild.period = response.timePeriod;
                    _this.timePeriodChild.startDate = response.startDate;
                    _this.timePeriodChild.endDate = response.endDate;

                    // Validation
                    if (!_this.isTitleValid(_this.newChildObjective.title)) {
                        showTopMessage("Please enter a valid title.", "warning", 3000);
                        _this.savingChildObjective = false;
                        return;
                    }

                    // Create newObjective object (following same pattern as KR but for Objective)
                    let newObjective = {};
                    let completionCriteria = _.find(this.statusAndEstimateFilters, status => {
                        return status.code === "CHILDREN_LINKED_ITEM_STATUS";
                    });

                    // If no completion criteria found, create a default one
                    if (!completionCriteria) {
                        completionCriteria = { code: "CHILDREN_LINKED_ITEM_STATUS" };
                    }

                    // Set basic properties
                    newObjective.title = this.newChildObjective.title;
                    newObjective.bvPlan = Math.max(0, Math.floor(Number(this.newChildObjective.bvPlan)));
                    newObjective.bvActual = Math.max(0, Math.floor(Number(this.newChildObjective.bvActual)));

                    let metaMap = {};
                    metaMap.syncId = syncId;
                    metaMap.sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                    metaMap.teamId = 0;
                    metaMap.boardId =this.isGlobalObj? "-1":this.releaseTrain.id;
                    newObjective.metaMap = metaMap;

                    newObjective.metaMap.objectiveId = parentObjective.id;
                    newObjective.metaMap = JSON.stringify(newObjective.metaMap);
                    newObjective.type = "Objective-Committed";  // Child objective type
                    newObjective.itemType = kendisStore.getters.getObjectiveTemplate().scheme.itemType;
                    newObjective.completionCriteria = completionCriteria.code;

                    // Set time period data for objective
                    if (!_.isEmpty(this.timePeriodChild.period)) {
                        if (!newObjective.fields) {
                            newObjective.fields = {};
                        }
                        newObjective.fields[this.timePeriodFieldId] = this.timePeriodChild.period;
                        newObjective.fields[this.startDateFieldId] = this.timePeriodChild.startDate;
                        newObjective.fields[this.endDateFieldId] = this.timePeriodChild.endDate;
                    }

                    // Set default status for Objective
                    if (_.find(kendisStore.getters.getObjectiveTemplate().statuses, {"category": "ToDo"}) !== -1) {
                        newObjective.status = _.find(kendisStore.getters.getObjectiveTemplate().statuses, {"category": "ToDo"});
                    }

                    // Call onSaveObjectiveForChildObjective (same pattern as KR)
                    _this.onSaveObjectiveForChildObjective(newObjective, parentObjective);
                    _this.toggleGroupDragging(false);
                }
            },

            // Method to save Child Objective (adapted from onSaveObjectiveForKR)
            onSaveObjectiveForChildObjective(requestBody, parentObjective) {
                let _this = this;
                let baseItemLinks = [];

                let sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                baseItemLinks.push(
                    {sessionId: sessionId,  type: "Objective-Committed", linkType: "art_link_objective"},
                    {sessionId: sessionId,  baseItemId: parentObjective.id, linkType: "parent_objective"}
                );

                requestBody.baseItemLinks = baseItemLinks;
                requestBody.starred = false;
                requestBody.mentionUsers = [];

                // Save the Child Objective
                axios.post('external-entitites/save-base-item/popup-save', requestBody)
                    .then(response => {
                        if (response.status == "200" && response && response.data) {
                            let baseItem = _.clone(parentObjective);
                            delete baseItem.board;
                            delete baseItem.linkedItems;
                            delete baseItem.linkedItemsById;
                            delete baseItem.linkedItemsIds;
                            delete baseItem.editView;
                            delete baseItem.baseItemList;

                            let childObjective = response.data.impediment;
                            let childObjectiveId = childObjective.id;

                            // Initialize baseItemList if needed
                            if (parentObjective.baseItemList) {
                                let index = _.findIndex(parentObjective.baseItemList, {id: childObjective.id});
                                if (index === -1) {
                                    parentObjective.baseItemList.push(childObjective);
                                }
                            } else {
                                parentObjective.baseItemList = [];
                                parentObjective.baseItemList.push(childObjective);
                            }

                            // Create base item link for the parent objective (THIS IS THE KEY DIFFERENCE)
                            let baseItemLink = {};
                            if (requestBody.type.includes("Objective")) {
                                baseItemLink.baseItemId = childObjective.id;
                                baseItemLink.sessionId = _this.isGlobalObj ? "-1" : _this.releaseTrain.id;
                                baseItemLink.linkType = "parent_okr";  // For objectives, use parent_okr
                            } else {
                                baseItemLink.baseItemId = childObjective.id;
                                baseItemLink.sessionId = _this.isGlobalObj ? "-1" : _this.releaseTrain.id;
                                baseItemLink.linkType = "child_objective";
                            }

                            // Initialize baseItemLinks if needed
                            if (!baseItem.baseItemLinks) {
                                baseItem.baseItemLinks = [];
                            }
                            baseItem.baseItemLinks.push(baseItemLink);

                            // Set up metaMap for parent objective update
                            let metaMap = {};
                            metaMap.syncId = syncId;
                            metaMap = JSON.stringify(metaMap);
                            baseItem.metaMap = metaMap;

                            // Clean up baseItemLinks
                            _.each(baseItem.baseItemLinks, link => {
                                if (!_.isEmpty(link.linkedItemById)) {
                                    delete link.linkedItemById;
                                }
                            });

                            // Clean up highlighted fields
                            if (baseItem.highlightedKey) {
                                delete baseItem.highlightedKey;
                            }
                            if (baseItem.highlightedTitle) {
                                delete baseItem.highlightedTitle;
                            }

                            // Second API call to update parent objective with the new child objective link
                            axios.post('external-entitites/save-base-item/popup-save', baseItem)
                                .then(response => {
                                    if (response.status == "200") {
                                        // Reset form data
                                        _this.newChildObjective.title = "";

                                        // Update parent objective's baseItemList
                                        if (parentObjective.baseItemList == undefined) {
                                            parentObjective.baseItemList = [];
                                        }
                                        let index = _.findIndex(parentObjective.baseItemList, {id: childObjective.id});
                                        if (index === -1) {
                                            parentObjective.baseItemList.push(childObjective);
                                        }

                                        // Update parent objective data
                                        baseItem.baseItemList = parentObjective.baseItemList;
                                        baseItem.baseItemLinks = response.data.impediment.baseItemLinks;
                                        baseItem.linkedItems = parentObjective.linkedItems;
                                        baseItem.linkedItemsIds = parentObjective.linkedItemsIds;
                                        baseItem.linkedItemsById = parentObjective.linkedItemsById;

                                        // Update cache with updated parent objective and new child objective
                                        _this.$options.cache.objectivesMap.set(parentObjective.id, baseItem);
                                        _this.$options.cache.objectivesMap.set(childObjective.id, childObjective);

                                        // Update store
                                        let storePayload = {};
                                        storePayload.objective = baseItem;
                                        kendisStore.commit("updateObjectiveAndOkrsInGroups", storePayload);

                                        // Add Child Objective as event to scheduler (create event only if it doesn't exist)
                                        if (!scheduler.eventStore.getById(childObjective.id)) {
                                            let childObjectiveEvent = {
                                                id: childObjective.id,
                                                name: childObjective.title,
                                                startDate: childObjective.fields && childObjective.fields[_this.startDateFieldId],
                                                endDate: childObjective.fields && childObjective.fields[_this.endDateFieldId],
                                                type: 'child-objective',
                                                cls:'kendis-kr-roadmap-event',
                                                eventColor: childObjective.status.category === "ToDo" ? "#E79362" : childObjective.status.category === "InProgress" ? "#3792C9" : childObjective.status.category === "Done" ? "#3FCA93" : "#7c50b1"
                                            };
                                            scheduler.eventStore.add(childObjectiveEvent);
                                        }
                                        
                                        // Add assignments for all parent objective resources (original and clones)
                                        let parentResources = _this.isMultilevel ? 
                                            scheduler.resourceStore.query(resource => 
                                                resource.get('originalObjectiveId') === parentObjective.id || resource.id === parentObjective.id
                                            ) : [scheduler.resourceStore.getById(parentObjective.id)];
                                        
                                        parentResources.forEach((resource) => {
                                            if (resource) {
                                                let assignmentId = `a-${childObjective.id}-${resource.id}`;
                                                if (!scheduler.assignmentStore.getById(assignmentId)) {
                                                    scheduler.assignmentStore.add({
                                                        id: assignmentId,
                                                        resourceId: resource.id,
                                                        eventId: childObjective.id
                                                    });
                                                }
                                            }
                                        });

                                        showTopMessageTargeted("Child Objective created successfully", 'success', 3000, _this.isFullScreen ? '.okrsrdmp' : null);

                                        // Update completion percentages when child objective is saved
                                        _this.updateCompletionPercentages();

                                        // Don't reload - scheduler already updated above
                                        _this.savingChildObjective = false;
                                    }
                                })
                                .catch(error => {
                                    console.error("Error updating parent objective:", error);
                                    showTopMessageTargeted("Error linking Child Objective to objective", 'error', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                                    _this.savingChildObjective = false;
                                });
                        }
                    })
                    .catch(error => {
                        console.error("Error creating Child Objective:", error);
                        showTopMessageTargeted("Error creating Child Objective", 'error', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                        _this.savingChildObjective = false;
                    });
            },

            // KR creation method using saveOkr logic from objective-row-component.js
            saveKrFromAsyncPopup(objective, response) {
                let _this = this;

                if (!_this.savingKeyResult) {
                    _this.savingKeyResult = true;

                    // Set response data to newOkr (response format from openAndHandlePopup)
                    _this.newOkr.title = response.textValue || '';
                    _this.timePeriod.period = response.timePeriod;
                    _this.timePeriod.startDate = response.startDate;
                    _this.timePeriod.endDate = response.endDate;
                    
                    // Set progress criteria from response
                    if (response.progressCriteria) {
                        _this.newOkr.fields = _this.newOkr.fields || {};
                        _this.newOkr.fields[_this.okrProgressCriteriaField.id] = response.progressCriteria;
                    }

                    // Validation (similar to saveOkr method)
                    if (!_this.isTitleValid(_this.newOkr.title)) {
                        showTopMessage("Please enter a valid title.", "warning", 3000);
                        _this.savingKeyResult = false;
                        return;
                    } else if (_.isEmpty(_this.timePeriod.period) && this.isMockingMode) {
                        showTopMessage("Please select a time period.", "warning", 3000);
                        _this.savingKeyResult = false;
                        return;
                    } else if (!_.isEmpty(_this.okrProgressCriteriaField) && _.isEmpty(response.progressCriteria)) {
                        showTopMessage("Please select a progress criteria for KR.", "warning", 3000);
                        _this.savingKeyResult = false;
                        return;
                    }

                    // Create newOkr object (following saveOkr logic)
                    let newOkr = {};
                    let completionCriteria = _.find(this.statusAndEstimateFilters, status => {
                        return status.code === "CHILDREN_LINKED_ITEM_STATUS";
                    });

                    // If no completion criteria found, create a default one
                    if (!completionCriteria) {
                        completionCriteria = { code: "CHILDREN_LINKED_ITEM_STATUS" };
                    }

                    // Set progress criteria field
                    if (!_.isEmpty(_this.okrProgressCriteriaField)) {
                        if (_.isEmpty(newOkr.fields)) {
                            newOkr.fields = {};
                        }
                        // Use the selected progress criteria from the popup
                        newOkr.fields[_this.okrProgressCriteriaField.id] = response.progressCriteria || "By Items";
                    }

                    // Set basic properties
                    newOkr.title = this.newOkr.title;
                    newOkr.bvPlan = Math.max(0, Math.floor(Number(this.newOkr.bvPlan)));
                    newOkr.bvActual = Math.max(0, Math.floor(Number(this.newOkr.bvActual)));

                    let metaMap = {};
                    metaMap.syncId = syncId;
                    metaMap.sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                    metaMap.teamId = 0;
                    metaMap.boardId =this.isGlobalObj? "-1":this.releaseTrain.id;
                    newOkr.metaMap = metaMap;

                    newOkr.metaMap.objectiveId = objective.id;
                    newOkr.metaMap = JSON.stringify(newOkr.metaMap);
                    newOkr.type = "KR";
                    newOkr.itemType = kendisStore.getters.getOkrTemplate().scheme.itemType;
                    newOkr.completionCriteria = completionCriteria.code;

                    // Set time period data
                    if (!_.isEmpty(this.timePeriod.period)) {
                        if (!newOkr.fields) {
                            newOkr.fields = {};
                        }
                        newOkr.fields[this.timePeriodKRFieldId] = this.timePeriod.period;
                        newOkr.fields[this.startDateKRFieldId] = this.timePeriod.startDate;
                        newOkr.fields[this.endDateKRFieldId] = this.timePeriod.endDate;
                    }

                    // Call onSaveObjective (same as saveOkr method)
                    _this.onSaveObjectiveForKR(newOkr, objective);
                    _this.toggleGroupDragging(false);
                }
            },

            // Method to save KR (adapted from objective-row-component.js onSaveObjective)
            onSaveObjectiveForKR(requestBody, parentObjective) {
                let _this = this;
                let baseItemLinks = [];

                // if (this.isMockingMode) {
                //     let sessionId = !_.isEmpty(parentObjective.parentId) ? parentObjective.parentId : this.board.session.id;
                //     baseItemLinks.push(
                //         {sessionId: sessionId, sessionBoardId: this.board.id, type: "KR", linkType: "art_link_objective"},
                //         {sessionId: sessionId, sessionBoardId: this.board.id, baseItemId: parentObjective.id, linkType: "parent_objective"}
                //     );
                // } else {
                //     baseItemLinks.push(
                //         {sessionId: this.board.session.id, sessionBoardId: this.board.id, type: "KR"},
                //         {sessionId: this.board.session.id, sessionBoardId: this.board.id, baseItemId: parentObjective.id, linkType: "parent_objective"}
                //     );
                // }

                let sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                baseItemLinks.push(
                    {sessionId: sessionId,  type: "KR", linkType: "art_link_objective"},
                    {sessionId: sessionId,  baseItemId: parentObjective.id, linkType: "parent_objective"}
                );


                requestBody.baseItemLinks = baseItemLinks;
                requestBody.starred = false;
                requestBody.mentionUsers = [];

                // Set default status for KR
                if (requestBody.type === 'KR' && _.find(kendisStore.getters.getOkrTemplate().statuses, {"category": "ToDo"}) !== -1) {
                    requestBody.status = _.find(kendisStore.getters.getOkrTemplate().statuses, {"category": "ToDo"});
                }

                                // Save the KR
                axios.post('external-entitites/save-base-item/popup-save', requestBody)
                    .then(response => {
                        if (response.status == "200" && response && response.data) {
                            let baseItem = _.clone(parentObjective);
                            delete baseItem.board;
                            delete baseItem.linkedItems;
                            delete baseItem.linkedItemsById;
                            delete baseItem.linkedItemsIds;
                            delete baseItem.editView;
                            delete baseItem.baseItemList;

                            let okr = response.data.impediment;
                            let okrId = okr.id;

                            // Initialize baseItemList if needed
                            if (parentObjective.baseItemList) {
                                let index = _.findIndex(parentObjective.baseItemList, {id: okr.id});
                                if (index === -1) {
                                    parentObjective.baseItemList.push(okr);
                                }
                            } else {
                                parentObjective.baseItemList = [];
                                parentObjective.baseItemList.push(okr);
                            }

                            // Create base item link for the parent objective (following legacy pattern)
                            let baseItemLink = {};
                            if (requestBody.type.includes("Objective")) {
                                baseItemLink.baseItemId = okr.id;
                                baseItemLink.sessionId = _this.isGlobalObj ? "-1" : _this.releaseTrain.id;
                                baseItemLink.linkType = "parent_okr";
                            } else {
                                baseItemLink.baseItemId = okr.id;
                                baseItemLink.sessionId = _this.isGlobalObj ? "-1" : _this.releaseTrain.id;
                                baseItemLink.linkType = "child_objective";
                            }

                            // Initialize baseItemLinks if needed
                            if (!baseItem.baseItemLinks) {
                                baseItem.baseItemLinks = [];
                            }
                            baseItem.baseItemLinks.push(baseItemLink);

                            // Set up metaMap for parent objective update
                            let metaMap = {};
                            metaMap.syncId = syncId;
                            metaMap = JSON.stringify(metaMap);
                            baseItem.metaMap = metaMap;

                            // Clean up baseItemLinks
                            _.each(baseItem.baseItemLinks, link => {
                                if (!_.isEmpty(link.linkedItemById)) {
                                    delete link.linkedItemById;
                                }
                            });

                            // Clean up highlighted fields
                            if (baseItem.highlightedKey) {
                                delete baseItem.highlightedKey;
                            }
                            if (baseItem.highlightedTitle) {
                                delete baseItem.highlightedTitle;
                            }

                            // Second API call to update parent objective with the new KR link
                            axios.post('external-entitites/save-base-item/popup-save', baseItem)
                                .then(response => {
                                    if (response.status == "200") {
                                        // Reset form data
                                        _this.newOkr.title = "";

                                        // Update parent objective's baseItemList
                                        if (parentObjective.baseItemList == undefined) {
                                            parentObjective.baseItemList = [];
                                        }
                                        let index = _.findIndex(parentObjective.baseItemList, {id: okr.id});
                                        if (index === -1) {
                                            parentObjective.baseItemList.push(okr);
                                        }

                                        // Update parent objective data
                                        baseItem.baseItemList = parentObjective.baseItemList;
                                        baseItem.baseItemLinks = response.data.impediment.baseItemLinks;
                                        baseItem.linkedItems = parentObjective.linkedItems;
                                        baseItem.linkedItemsIds = parentObjective.linkedItemsIds;
                                        baseItem.linkedItemsById = parentObjective.linkedItemsById;

                                        // Update cache with updated parent objective
                                        _this.$options.cache.objectivesMap.set(parentObjective.id, baseItem);
                                        _this.$options.cache.keyResultsMap.set(okr.id, okr);

                                        // Update store
                                        let storePayload = {};
                                        storePayload.objective = baseItem;
                                        kendisStore.commit("updateObjectiveAndOkrsInGroups", storePayload);

                                        // Add KR as event to scheduler (create event only if it doesn't exist)
                                        if (!scheduler.eventStore.getById(okr.id)) {
                                            let krEvent = {
                                                id: okr.id,
                                                name: okr.title,
                                                startDate: okr.fields && okr.fields[_this.startDateKRFieldId],
                                                endDate: okr.fields && okr.fields[_this.endDateKRFieldId],
                                                type: 'keyResult',
                                                eventColor: okr.status.category === "ToDo" ? "#E79362" : okr.status.category === "InProgress" ? "#3792C9" : okr.status.category === "Done" ? "#3FCA93" : "#7c50b1",
                                                cls:'kendis-kr-roadmap-event'
                                            };
                                            scheduler.eventStore.add(krEvent);
                                        }
                                        
                                        // Add assignments for all parent objective resources (original and clones)
                                        let objectiveResources = _this.isMultilevel ? 
                                            scheduler.resourceStore.query(resource => 
                                                resource.get('originalObjectiveId') === parentObjective.id || resource.id === parentObjective.id
                                            ) : [scheduler.resourceStore.getById(parentObjective.id)];
                                        
                                        objectiveResources.forEach((resource) => {
                                            if (resource) {
                                                let assignmentId = `a-${okr.id}-${resource.id}`;
                                                if (!scheduler.assignmentStore.getById(assignmentId)) {
                                                    scheduler.assignmentStore.add({
                                                        id: assignmentId,
                                                        resourceId: resource.id,
                                                        eventId: okr.id
                                                    });
                                                }
                                            }
                                        });

                                        showTopMessageTargeted("Key Result created successfully", 'success', 3000, _this.isFullScreen ? '.okrsrdmp' : null);

                                        // Update completion percentages when KR is saved
                                        _this.updateCompletionPercentages();

                                        // Don't reload - scheduler already updated above
                                        _this.savingKeyResult = false;
                                    }
                                })
                                .catch(error => {
                                    console.error("Error updating parent objective:", error);
                                    showTopMessageTargeted("Error linking Key Result to objective", 'error', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                                    _this.savingKeyResult = false;
                                });
                        }
                    })
                    .catch(error => {
                        console.error("Error creating KR:", error);
                        showTopMessageTargeted("Error creating Key Result", 'error', 3000, _this.isFullScreen ? '.okrsrdmp' : null);
                        _this.savingKeyResult = false;
                    });
            },

            // Title validation method (similar to objective-row-component.js)
            isTitleValid(title) {
                return title && title.trim().length > 0 && title.trim().length <= 750;
            },

            // Toggle group dragging (similar to objective-row-component.js)
            toggleGroupDragging(option) {
                // In roadmap context, this is handled by Bryntum scheduler
                // We can safely ignore this or add specific roadmap logic if needed
                if (typeof toggleDraggable === 'function') {
                    toggleDraggable(option);
                }
            },

            // Get KR date restrictions based on parent objective's start and end dates
            getKrDateRestrictions(parentObjective) {
                if (!parentObjective || !parentObjective.fields) {
                    return null;
                }

                // Get parent objective's start and end dates
                let parentStartDate = parentObjective.fields[this.startDateFieldId];
                let parentEndDate = parentObjective.fields[this.endDateFieldId];
                
                if (!parentStartDate || !parentEndDate) {
                    return null;
                }

                // Convert to Date objects
                let minDate = new Date(parentStartDate);
                let maxDate = new Date(parentEndDate);

                // Validate dates
                if (isNaN(minDate.getTime()) || isNaN(maxDate.getTime())) {
                    return null;
                }

                // Format dates for tooltip
                const formatDate = (date) => {
                    return date.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                    });
                };

                return {
                    minDate: minDate,
                    maxDate: maxDate,
                    tooltip: `Please select key result dates within the main objective's timeframe. Key result dates must be within ${formatDate(minDate)} - ${formatDate(maxDate)}.`
                };
            },

            // Get child objective date restrictions based on parent objective's start and end dates
            getChildObjectiveDateRestrictions(parentObjective) {
                if (!parentObjective || !parentObjective.fields) {
                    return null;
                }

                // Get parent objective's start and end dates
                let parentStartDate = parentObjective.fields[this.startDateFieldId];
                let parentEndDate = parentObjective.fields[this.endDateFieldId];
                
                if (!parentStartDate || !parentEndDate) {
                    return null;
                }

                // Convert to Date objects
                let minDate = new Date(parentStartDate);
                let maxDate = new Date(parentEndDate);

                // Validate dates
                if (isNaN(minDate.getTime()) || isNaN(maxDate.getTime())) {
                    return null;
                }

                // Format dates for tooltip
                const formatDate = (date) => {
                    return date.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                    });
                };

                return {
                    minDate: minDate,
                    maxDate: maxDate,
                    tooltip: `Please select child objective dates within the parent objective's timeframe. Child objective dates must be within ${formatDate(minDate)} - ${formatDate(maxDate)}.`
                };
            },

            // Get date range from time period (following legacy getDateRange logic)
            getDateRangeFromTimePeriod(title, selectedYear) {
                if (!title || !selectedYear) return null;

                let minDate, maxDate;
                const year = parseInt(selectedYear);

                if (title.startsWith('Q')) {
                    // Handle quarters (Q1, Q2, Q3, Q4)
                    const quarter = parseInt(title.substring(1));
                    const startMonth = (quarter - 1) * 3;
                    const endMonth = startMonth + 2;

                    minDate = new Date(year, startMonth, 1);
                    maxDate = new Date(year, endMonth + 1, 0); // Last day of end month
                } else if (title.startsWith('H')) {
                    // Handle half-years (H1, H2)
                    const half = parseInt(title.substring(1));
                    const startMonth = (half - 1) * 6;
                    const endMonth = startMonth + 5;

                    minDate = new Date(year, startMonth, 1);
                    maxDate = new Date(year, endMonth + 1, 0); // Last day of end month
                } else {
                    // Handle specific months or custom periods
                    const monthNames = ['January', 'February', 'March', 'April', 'May', 'June',
                                       'July', 'August', 'September', 'October', 'November', 'December'];
                    const monthIndex = monthNames.indexOf(title);

                    if (monthIndex !== -1) {
                        minDate = new Date(year, monthIndex, 1);
                        maxDate = new Date(year, monthIndex + 1, 0); // Last day of month
                    } else {
                        // Default to full year if period not recognized
                        minDate = new Date(year, 0, 1);
                        maxDate = new Date(year, 11, 31);
                    }
                }

                return { minDate, maxDate };
            },

            // Get tooltip text for date range
            getDateRangeTooltip(title, selectedYear) {
                let dateRange = this.getDateRangeFromTimePeriod(title, selectedYear);
                if (!dateRange) return "";

                const options = { year: 'numeric', month: 'long', day: 'numeric' };
                return `${dateRange.minDate.toLocaleDateString(undefined, options)} - ${dateRange.maxDate.toLocaleDateString(undefined, options)}`;
            },

            handleCreateObjective(objectiveId) {
                let _this = this;
                let objective = this.$options.cache.objectivesMap.get(objectiveId);
                if (!objective) {
                    showTopMessageTargeted("Objective not found", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    this.hideObjectiveActionsMenu();
                    return;
                }

                // Find the objective group for this objective
                let objectiveGroupId = objective.parentId || this.findObjectiveGroupForObjective(objectiveId);
                if (!objectiveGroupId) {
                    showTopMessageTargeted("Cannot find objective group", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    this.hideObjectiveActionsMenu();
                    return;
                }

                // Use existing method to add objective to the same group
                this.addObjectiveToObjectiveGroup({dataset: {id: objectiveGroupId}});
                this.hideObjectiveActionsMenu();
            },

            findObjectiveGroupForObjective(objectiveId) {
                // Search through the relationship map to find the objective group
                for (let [groupId, objectiveIds] of this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.entries()) {
                    if (objectiveIds.includes(objectiveId)) {
                        return groupId;
                    }
                }
                return null;
            },

            createNewKR(krData, parentObjective) {
                let _this = this;
                return new Promise(function(resolve, reject) {
                    axios.post('objective/update-objective', krData).then(function(response) {
                        if (response && response.data && response.data.id) {
                            let newKR = response.data;
                            _this.$options.cache.keyResultsMap.set(newKR.id, newKR);

                            // Add KR as event to scheduler under the objective resource
                            let krEvent = {
                                id: newKR.id,
                                name: newKR.title,
                                resourceId: parentObjective.id,
                                startDate: newKR.fields && newKR.fields[_this.startDateKRFieldId],
                                endDate: newKR.fields && newKR.fields[_this.endDateKRFieldId],
                                type: 'keyResult',
                                cls:'kendis-kr-roadmap-event'
                            };

                            scheduler.eventStore.add(krEvent);
                            scheduler.assignmentStore.add({
                                id: newKR.id,
                                resourceId: parentObjective.id
                            });

                            resolve(newKR);
                        } else {
                            reject(new Error('Invalid response'));
                        }
                    }).catch(function(error) {
                        reject(error);
                    });
                });
            },

            /**
             * Returns count and display info for objectives in a group.
             * Shows actual count if expanded, preliminary if collapsed.
             * Handles both multilevel and legacy modes.
             * @param {string} resourceId
             * @returns {Object} {count, isPreliminary, hasValidCount, tooltip, cssClass, style, color}
             */
            getObjectiveGroupCount: function(resourceId) {
                var isExpanded = this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.has(resourceId);
                var isMultilevel = this.isMultilevel;
                var count = 0;
                var isPreliminary = false;
                var hasValidCount = false;

                /**
                 * Get count from scheduler assignment store (legacy) or resource children (multilevel)
                 * @returns {number}
                 */
                function getActualCount() {
                    if (isMultilevel) {
                        if (typeof scheduler !== 'undefined' && scheduler.resourceStore) {
                            var resource = scheduler.resourceStore.getById(resourceId);
                            if (resource && resource.children) {
                                return resource.children.length;
                            }
                        }
                    } else {
                    if (typeof scheduler !== 'undefined' && scheduler.assignmentStore) {
                            var n = scheduler.assignmentStore.getAssignmentsForResource(resourceId).length - 1;
                            return Math.max(0, n);
                        }
                    }
                    return 0;
                }

                /**
                 * Get preliminary count from backend cache
                 * @returns {number|undefined}
                 */
                function getPreliminaryCount(vm) {
                    if (vm.$options.preliminaryCounts && vm.$options.preliminaryCounts[resourceId] !== undefined) {
                        return vm.$options.preliminaryCounts[resourceId];
                    }
                    return undefined;
                }

                if (isExpanded) {
                    // Expanded: show actual count
                    count = getActualCount();
                        isPreliminary = false;
                        hasValidCount = true;
                } else {
                    // Collapsed: try preliminary, else fallback to actual
                    var prelim = getPreliminaryCount(this);
                    if (prelim !== undefined) {
                        count = prelim;
                    isPreliminary = true;
                    hasValidCount = true;
                    } else {
                        count = getActualCount();
                    isPreliminary = false;
                    hasValidCount = count > 0;
                    }
                }

                return {
                    count: count,
                    isPreliminary: isPreliminary,
                    hasValidCount: hasValidCount,
                    tooltip: isExpanded ? 'Number of objectives in this group' : 'Preliminary count - will update when expanded',
                    cssClass: isPreliminary ? 'preliminary-count' : 'actual-count',
                    style: isPreliminary ? 'border: 2px dashed rgba(255,255,255,0.5);' : '',
                    color: count > 0 ? '#4CAF50' : '#9E9E9E'
                };
            },

            onTimePeriodFilter(value) {
                if (_.isEmpty(value)) {
                    this.timePeriodFilter.filter = {};
                } else {
                    this.timePeriodFilter.filter = value;
                }
                this.reloadObjectiveGroups();

                // Save filters and update URL
                this.saveFiltersToStorage();
                this.updateURLWithFilters();
            },

            onFilterResponsible(responsibles) {
                this.objectiveFilters.responsible = responsibles;
                this.reloadObjectiveGroups();

                // Save filters and update URL
                this.saveFiltersToStorage();
                this.updateURLWithFilters();
            },

            onFilterAssociation(collections) {
                this.objectiveFilters.collections = collections;
                this.reloadObjectiveGroups();

                // Save filters and update URL
                this.saveFiltersToStorage();
                this.updateURLWithFilters();
            },

            onFilterStatus(statuses) {
                this.objectiveFilters.statuses = statuses;
                this.reloadObjectiveGroups();

                // Save filters and update URL
                this.saveFiltersToStorage();
                this.updateURLWithFilters();
            },

            openKeyResultContainer(keyResult){

                this.openKRModelForEditing(keyResult,true, null)

            },
            onAddCustomClick(){
                // Check EDIT_OBJECTIVE_GROUP permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'EDIT_OBJECTIVE_GROUP', 'create custom objective groups')) {
                    return;
                }
                
                let input = {};
                input.itemIdentifier = "New Objective Group";
                input.startDate = undefined
                input.endDate = undefined
                input.showTextField = true;
                input.showDateFields = false;
                input.textValue ="Objective Group";
                input.textInputLabel = "Enter Objective Group title"
                input.showTimePeriod = false;

                let promise = this.openAndHandlePopup(input);
                promise.then(async (response) => {
                    this.addObjectiveGroup(response.textValue);
                }).catch((error) => {
                    console.log(error);
                    showTopMessageTargeted("Action cancelled", 'warning', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            onSelectCollection: function (collection) {
                const selectedCollection = this.collections.find(item => item.id === collection.id);

                this.addMenu.loadingCollections = false;
                this.addMenu.showCollections = false;
                this.addMenu.show = false;

                this.onAddCollectionObjectiveGroup(selectedCollection);
            },
            loadCollections() {
                let _this = this;

                let data = {};
                data.fetchTypes = false;

                axios.post("/releasetrain/release-trains", data)
                    .then(response => {
                        if (response.data) {
                            let collections = response.data.releaseTrains;
                            this.collections = collections;

                            collections = collections.filter(col => col.id && col.id !== "-1" && col.type);
                            kendisStore.commit("setArtCollections", collections);
                        }
                    }).catch(error => {
                        console.log(error);
                        showTopMessageTargeted("Error loading collections", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    _this.addMenu.loadingCollections = false;
                });
            },
            onSelectAddCollectionObjective: function () {
                // Check EDIT_OBJECTIVE_GROUP permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'EDIT_OBJECTIVE_GROUP', 'add collection objective groups')) {
                    return;
                }
                
                if (this.addMenu.showCollections) {
                    this.addMenu.showCollections = false;
                    return;
                }
                if (kendisStore.getters.getArtCollections().length > 0) {
                    let collections = kendisStore.getters.getArtCollections();
                    collections = collections.filter(col => col.id && col.id !== "-1" && col.type);
                    this.addMenu.collections = collections;
                    this.collections = collections;
                }
                if (_.isEmpty(this.addMenu.collections)) {
                    //....   load collections
                    this.addMenu.loadingCollections = true;

                    let _this = this;

                    let data = {};
                    data.fetchTypes = false;

                    axios.post("/releasetrain/release-trains", data)
                        .then(response => {
                            _this.addMenu.loadingCollections = false;
                            if (response.data) {
                                let collections = response.data.releaseTrains;
                                collections = collections.filter(col => col.id && col.id !== "-1" && col.type);
                                _this.addMenu.collections = collections;
                                _this.collections = collections;
                                _this.prepareCollections();
                            }
                        }).catch(error => {
                            console.log(error);
                            showTopMessageTargeted("Error loading collections", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                        _this.addMenu.loadingCollections = false;
                    });
                }
                else {
                    this.addMenu.loadingCollections = false;
                    this.prepareCollections();
                }
            },
            prepareCollections() {
                let collections = this.addMenu.collections;

                let collectionsToDisabled = [];


                this.$options.cache.objectiveGroupsMap.values().forEach(group => {
                    group.baseItemLinks.forEach(link => {
                        if (link.linkType === 'art_link_objective' && link.sessionId !== "-1") {
                            collectionsToDisabled.push(link.sessionId);
                        }
                    })
                });

                this.workspaceList = [];
                let rootWorkSpaces = [];
                let workspaceMap = {};

                _.each(collections, workspace => {
                    if(_.isEmpty(workspaceMap[workspace.id])) {
                        workspaceMap[workspace.id] = {id: workspace.id, label: workspace.title, children: [], isDisabled: collectionsToDisabled.includes(workspace.id)};
                    }
                })

                _.each(collections, workspace => {
                    if(workspace.linkedWorkSpaces && workspace.linkedWorkSpaces.length > 0) {
                        _.each(workspace.linkedWorkSpaces, linkedWorkSpaceId => {
                            let parent = workspaceMap[linkedWorkSpaceId];
                            if(parent) {
                                parent.children.push(workspaceMap[workspace.id]);
                            }
                        });
                    } else {
                        rootWorkSpaces.push(workspaceMap[workspace.id]);
                    }
                });

                this.workspaceList = rootWorkSpaces;
                this.addMenu.showCollections = true;
            },
            getGroupCollection(objGroupId) {
                let objGroup = this.$options.cache.objectiveGroupsMap.get(objGroupId)
                if (!objGroup || _.isEmpty(objGroup.baseItemLinks)) {
                    return {};
                }
                return objGroup.baseItemLinks.find(link => link.linkType === 'art_link_objective' && link.sessionId !== "-1");
            },
            getCollection(objGroupId) {
                const collectionLink = this.getGroupCollection(objGroupId);
                if (collectionLink) {
                    const collectionId = collectionLink.sessionId;

                    return kendisStore.getters.getArtCollections().find(collection => collection.id === collectionId);
                }
                return {};
            },
            getGroupKey() {
                if (this.isArt) {
                    return this.linkedCustomView ? this.linkedCustomView.key : '';
                }
                return this.getCollection ? this.getCollection.key : '';
            },
            onZoomIncrease(operation) {

                let mod;
                if (this.$options.gantt) {
                    mod = this.$options.gantt;
                }
                else if (this.$options.roadmap) {
                    mod = this.$options.roadmap;
                }

                if (operation == 'increase') {
                    mod.zoomIn();
                } else {
                    mod.zoomOut();
                }
            },

            /////////////////////////
            //////MILESTONES/////////
            /////////////////////////
            onCloseMilestonePopup(){
                this.milestoneData.milestonePopup.show = false;
                this.milestoneData.milestonePopup.releaseTrain = {};
                this.milestoneData.milestonePopup.level = {};
                this.milestoneData.milestonePopup.statuses = [];
                this.milestoneData.milestonePopup.isUpdate = false;
                this.milestoneData.milestonePopup.item = {};
                this.milestoneData.milestonePopup.boardId = undefined;
                this.milestoneData.milestonePopup.selectedCalculationCriteria = "";
            },
            onAddMilestoneToBoardPopupClose: function() {
                //this.addMilestoneToBoardPopup.boardItems = undefined;
                this.addMilestoneToBoardPopup.show = false;
            },
            onAddToBoardPopupItemSelectedMilestone: function (itemData, shouldClose) {
                //.......   add item
                let item = itemData.item;
                if(item.fields.itemType === "Milestone" || item.fields.itemType === "Phase") {
                    //item.rollup = true;
                    showTopMessageTargeted(item.fields.itemType +" added successfully", 'success', 1000, this.isFullScreen ? '.okrsrdmp' : null);
                    this.milestoneMapById[item.id] = item;
                    this.milestoneData.milestones.push(item);
                    this.addMileStoneInRoadMap(item);
                } else {
                    //this.addNewItemsToTimeline([itemData.item]);
                }
                if(shouldClose) {
                    this.onAddMilestoneToBoardPopupClose();
                }
            },
            createItemForRoadmap: function(item,update) {

                item.orgId = item.id;

                //.........................
                if (update) {

                }else {
                    item.children = !_.isEmpty(item.childrenStatuses);
                    item.expanded = false;
                    item.manuallyScheduled = true;
                    item.isGroup = false;
                    item.count = item.children ? item.childrenStatuses.length : 0;
                    item.type = "";
                }

                //.........................
                //...........  status
                if (item.status) {

                    item.status = this.isGlobalObj?
                        item.status = this.milestoneData.milestoneStatuses.find(status => status.id === item.status.id):this.store.getStatusMap[item.status.id];
                    if (item.status) {
                        item.statusClass = item.status.category === "ToDo" ? "a" : item.status.category === "InProgress" ? "b" : item.status.category === "Done" ? "c" : "e";
                        item.statusColor = item.status.category === "ToDo" ? "#E79362" : item.status.category === "InProgress" ? "#3792C9" : item.status.category === "Done" ? "#3FCA93" : "#7c50b1";
                    }
                    item.barColor = item.statusColor;
                    item.style = "background-color:" + item.statusColor + ";";
                }
                else {
                    item.statusColor = "#7c50b1";
                    item.statusClass = "e";
                    item.barColor    = "#7c50b1";
                }
                if (item.fields && item.fields.Color) {
                    item.barColor    = item.fields.Color;
                }
                item.style = "background-color:" + item.barColor + ";";

                //......  set required fields
                this.setRequiredProppertiesOfItem(item);

                //......   set dates
                if (item._fields && item._fields.StartDate && item._fields.EndDate) {
                    if (item._fields.StartDate) {
                        item.startDate = new Date(item._fields.StartDate);
                    }
                    if (item._fields.EndDate) {
                        item.endDate = new Date(item._fields.EndDate);
                    }
                }

                return item;
            },
            setRequiredProppertiesOfItem: function(item) {
                const defaults = {
                    groupId: "",
                    subGroupApplied: false,
                    isSubGroup: false,
                    isRootGroup: false,
                    parentGroupId: "",
                    orgId: "",
                    unplanned: false,
                    planned: false,
                    _childLevel: 0,
                    _currentLevel: 0,
                    almKey: "",
                    almItemId: "",
                    url: "",
                    startDate: "",
                    endDate: "",
                    unStartDate: "",
                    unEndDate: "",
                    plStartDate: "",
                    plEndDate: "",
                    artRelationMap: {},
                    childBatches: [],
                    childContainers: [],
                    childBoards: [],
                    childRelations: [],
                    childSolutionBoards: [],
                    childStoryPoints: 0,
                    childrenStatuses: [],
                    linksMap: {},
                    parentRelations: [],
                    parents: [],
                    status: {},
                    storyPoints: 0,
                    percentDone: 0,
                    childStoryPointsByStatus: [],
                    leafStoryPoints: 0,
                    multiIds: [],
                    excludeGroupIds: [],
                    _fields: {
                        itemTypeIcon: "",
                        itemType: ""
                    }
                };

                // Apply defaults to the item if the property is undefined or null
                for (let key in defaults) {
                    if (key === "_fields") {
                        item[key] = item[key] || item.fields || {};
                        for (let subKey in defaults[key]) {
                            item[key][subKey] = item[key][subKey] != null ? item[key][subKey] : defaults[key][subKey];
                        }
                    } else {
                        item[key] = item[key] != null ? item[key] : defaults[key];
                    }
                }
            },
            addMileStoneInRoadMap(item){
                const resource     = this.$options.roadmap.resourceStore.getById(milestoneResourceId);
                let assignment = {
                    id: 'a-'+item.id,
                    resourceId: resource.id,
                    eventId: item.id
                }
                let newTask 	   = this.createItemForRoadmap(item);
                // newTask.resourceId = resource.id;
                newTask.rollup     = true;
                if(newTask.fields && newTask.fields.itemType === "Milestone"){
                    newTask.startDate  = (item.fields && item.fields.StartDate ) ?  new Date(item.fields.StartDate) : '';
                    newTask.duration   = 0;
                    newTask.cls = 'milstoneshape';
                    newTask.milestoneWidth = 50;
                    newTask.ttype = "milestone";
                }else if(newTask.fields && newTask.fields.itemType === "Phase"){
                    newTask.startDate  = (item.fields && item.fields.StartDate ) ?  new Date(item.fields.StartDate) : '';
                    newTask.duration   = 0;
                    newTask.cls = 'milstoneshape';
                    newTask.milestoneWidth = 50;
                    newTask.ttype = "phase";
                }else{
                    newTask.startDate.setHours(0, 0, 0, 0);
                    newTask.endDate.setHours(23, 59, 59, 999);
                }
                if(resource.groupItems == undefined){
                    resource.groupItems = [];
                }
                newTask.groupId = milestoneResourceId

                !resource.groupItems.some(i => i.id === item.id) && resource.groupItems.push(item);
                _.isEmpty(this.$options.roadmap.eventStore.getById(item.id)) && this.$options.roadmap.eventStore.add(newTask);
                _.isEmpty(this.$options.roadmap.assignmentStore.getById(assignment.id)) && this.$options.roadmap.assignmentStore.add(assignment);
            },
            onArchivedMilestonesError: function(archivedMilestones) {
                let _this = this;
                if (this.addMilestoneToBoardPopup.show) {
                    this.onAddMilestoneToBoardPopupClose();
                }

                let errorMsgHtml = `<span>`;

                archivedMilestones.forEach((milestone, index) => {
                    // Escape milestone.kendisKey and milestone.title to prevent XSS
                    let kendisKey = milestone.kendisKey.replace(/</g, "&lt;").replace(/>/g, "&gt;");
                    let title = milestone.title.replace(/</g, "&lt;").replace(/>/g, "&gt;");
                    errorMsgHtml += `<span class="key FN fs-13">${kendisKey}</span>${title}`;
                    if (index < archivedMilestones.length - 1) {
                        errorMsgHtml += ', ';
                    }
                });

                if (archivedMilestones.length > 1) {
                    errorMsgHtml += ` are archived and cannot be linked with this board.`;
                } else {
                    errorMsgHtml += ` is archived and cannot be linked with this board.`;
                }
                errorMsgHtml += `</span>`;

                setTimeout(function() {
                    _this.milestoneFieldUpdateMessage.text = errorMsgHtml;

                    // Set text to empty string after 3000ms
                    setTimeout(function() {
                        _this.milestoneFieldUpdateMessage.text = "";
                    }, 3000);
                }, 2000);
            },
            onAddExistingMilestone(){
                if (!_.isEmpty(this.milestoneData.milestones)) {
                    let allMilestones = this.milestoneData.milestones;
                    this.addMilestoneToBoardPopup.boardItems = _.map(allMilestones, milestone => milestone.id);
                }
                this.addMilestoneToBoardPopup.show = true;
                this.addMilestoneToBoardPopup.releaseTrain = this.isGlobalObj?{id:this.viewId}:this.store.releaseTrain;
                this.addMilestoneToBoardPopup.boardId = this.roadmap.id;
                this.addMilestoneToBoardPopup.viewType = this.roadmap.viewType;
                this.showAddMilestoneMenu = false;
            },
            hideMilestoneMenu() {
                this.showAddMilestoneMenu = false;
            },
            editMilestonePermission() {
                return this.isGlobalObj?undefined:this.store.isActionAllowed("edit-milestone");
            },
            deleteMilestonePermission() {
                return this.isGlobalObj?undefined:this.store.isActionAllowed("delete-milestone");
            },
            linkUnlinkItemsPermission() {
                return this.isGlobalObj?undefined:this.store.isActionAllowed("link-unlink-items-in-milestone");
            },
            onClickArchiveMilestone(item){
                let _this = this;
                _this.archiveMilestones([item.id], true);
            },
            archiveMilestones: function (itemIds, archive) {

                let _this = this;
                let data = {};
                // data.releaseTrainId = this.releaseTrain.id;
                data.itemIds = itemIds;
                data.archieve = archive;

                let requestId = getNewUUID();
                data.syncId = syncId + "#" + requestId;


                _this.loader.show = true;
                this.$options.archiveItemIds = itemIds;
                startPoll(requestId, "onMilestoneArchiveCompleted", _this);

                axios.post('/releasetrain/milestone/bulkArchive', data)
                    .then(res => {

                    }).catch(error => {
                    console.log(error);
                    _this.loader.show = false;
                });
            },
            onCreateMilestonePopup(event, item){
                if(item) {
                    this.milestoneData.milestonePopup.item = item;

                    if(item.fields && item.fields.selectedCalculationCriteria) {
                        this.milestoneData.milestonePopup.selectedCalculationCriteria = item.fields.selectedCalculationCriteria;
                    }
                    this.milestoneData.milestonePopup.isUpdate = true;
                }
                this.milestoneData.milestonePopup.releaseTrain = this.isGlobalObj?{id:this.viewId}:this.store.releaseTrain;
                // this.milestoneData.milestonePopup.level 	   = this.currentHierarchyLevel;
                this.milestoneData.milestonePopup.statuses     = this.isGlobalObj?this.milestoneData.milestoneStatuses:this.store.milestoneStatuses;
                this.milestoneData.milestonePopup.boardId      = this.roadmap.id;
                this.milestoneData.milestonePopup.show         = true;
                //this.addMilestonePopup.show                    = false;
            },
            onSaveMilestone(itemData){
                let item = itemData.item;
                if (item.status) { //bug fix
                    item.status = this.milestoneData.milestoneStatuses.find(status => status.id === item.status.id);
                }
                this.createMilestoneGroupIfNotExist();
                let groupResource = this.$options.roadmap.resourceStore.getById(milestoneResourceId);
                if(!groupResource.expanded){
                    scheduler.resourceStore.toggleCollapse(groupResource.id)
                }
                this.addMilestones(item);
                // scheduler.refresh();
                //this.addMileStoneInRoadMap(item);
                this.onCreateMilestonePopup(null, item);

            },
            async loadMilestoneMeta(){
                let endpoint ="/releasetrain/getMilestonesStatuses"
                const response = await fetch(endpoint, {
                    headers: getFetchAPIHeadersForKendis(),
                    credentials: 'include', // Include cookies for CSRF-TOKEN
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const data = await response.json();
                this.milestoneData.milestoneStatuses = data.milestoneStatuses;

            },
            async loadMilestones() {
                await this.loadMilestoneMeta();
                console.log("!!!!!!!!!! Loading Milestones !!!!!!")
                let _this = this;
                let data = {}
                data.hierachLevel = milestoneResourceId;
                data.requestId = getNewUUID();
                data.boardId = this.roadmap.id;
                data.sortBy = "date";
                data.sortOrder = "1";

                let contextId = this.isGlobalObj?this.viewId:this.store.releaseTrainId ;

                axios.post("/releasetrain/" + contextId + "/backlog", data).then(res => {
                    if (res.data.result.items) {
                        //let items = res.data.result.items;
                        _this.milestoneData.milestones     = res.data.result.items;
                        _this.milestoneData.milestoneGroup = {};
                        _this.createMilestoneGroupIfNotExist();
                        let groupResource = _this.$options.roadmap.resourceStore.getById(milestoneResourceId);
                        if (groupResource.expanded) {
                            _this.addMilestones();
                        }
                    }
                }).catch(error => {
                    console.log(error);
                    showTopMessageTargeted("Error loading milestones", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            showMilestoneMenu: function (event){
                if (event) {
                    // Capture the position of the button and round the values
                    const buttonRect = event.target.getBoundingClientRect();

                    this.milestoneMenuPopupPosition.top = Math.round(buttonRect.top + window.scrollY);
                    this.milestoneMenuPopupPosition.left = Math.round(buttonRect.left + window.scrollX);
                }else{
                    this.milestoneMenuPopupPosition.top = 0;
                    this.milestoneMenuPopupPosition.left = 0;
                }
                this.showAddMilestoneMenu = true;
            },
            addMilestones(item){
                if(item){
                    let _this = this;
                    let itemIndex = _.findIndex(_this.milestoneData.milestones, {id: item.id});
                    if(itemIndex > -1) {
                        // Item Exist so Update its existence in list
                        // Check if Milestone has relationship with timeline.
                        let isMilestoneRelatedToTimeline = _.find(item.relations, {baseItemId: _this.roadmap.id});
                        // If Relation Exist with current timeline
                        if(isMilestoneRelatedToTimeline) {
                            _this.milestoneMapById[item.id] = item;
                            _this.milestoneData.milestones.splice(itemIndex, 1, item);
                        } else {
                            // Relationship doesn't exist
                            _this.milestoneData.milestones.splice(itemIndex, 1);
                        }
                        const resource     = this.$options.roadmap.resourceStore.getAt(0);
                        if(resource.groupItems == undefined){
                            resource.groupItems = [];
                        }else {
                            let itemIndexOld = _.findIndex(resource.groupItems, {id: item.id});
                            if(itemIndexOld > -1){
                                let isMilestoneRelatedToTimeline = _.find(item.relations, {baseItemId: _this.roadmap.id});
                                if(isMilestoneRelatedToTimeline) {
                                    let oldTask = resource.getEventById(item.id);
                                    //let newTask 	   = this.createItemForRoadmap(item);
                                    //newTask.rollup     = true;
                                    oldTask.title    = item.title;
                                    oldTask.barColor = (item.fields && item.fields.Color ) ?  item.fields.Color : '#7c50b1';
                                    oldTask.status   = item.status  ?  item.status : oldTask.status;
                                    oldTask.style    = "background-color:" + oldTask.barColor + ";";
                                    if(item.fields && item.fields.itemType == "Milestone"){
                                        oldTask.startDate  = (item.fields && item.fields.StartDate ) ?  new Date(item.fields.StartDate) : '';
                                    }else {
                                        oldTask.startDate  = (item.fields && item.fields.StartDate ) ?  new Date(item.fields.StartDate) : '';
                                        oldTask.endDate    = (item.fields && item.fields.EndDate ) ?  new Date(item.fields.EndDate) : '';
                                    }

                                    //setting start and end time
                                    if(oldTask.startDate && oldTask.startDate != ''){
                                        oldTask.startDate.setHours(0, 0, 0, 0);
                                    }
                                    if(oldTask.endDate && oldTask.endDate != ''){
                                        oldTask.endDate.setHours(23, 59, 59, 999);
                                    }

                                    resource.groupItems.splice(itemIndex, 1, item);
                                } else {
                                    // Relationship doesn't exist
                                    this.$options.roadmap.eventStore.remove(item.id);
                                    resource.groupItems.splice(itemIndex, 1);
                                }
                            }
                        }

                    } else {
                        _this.milestoneMapById[item.id] = item;
                        _this.milestoneData.milestones.push(item);
                        _this.addMileStoneInRoadMap(item)
                        //_this.milestoneData.milestoneGroup.appendChild(data);
                    }

                }else{
                    for(let item of this.milestoneData.milestones){
                        this.milestoneMapById[item.id] = item;
                        this.addMileStoneInRoadMap(item)
                    }
                }

                // this.fillMilestoneInMenuList();
            },
            createMilestoneGroupIfNotExist(){
                if(this.$options.roadmap && !this.$options.roadmap.resourceStore.getById(milestoneResourceId)) {
                    let mileStoneTask 		= {}
                    mileStoneTask.name = "Milestones";
                    mileStoneTask.title 	= "Milestones";
                    mileStoneTask.id 		= milestoneResourceId;
                    mileStoneTask.ttype		= "milestone";
                    mileStoneTask.cls		= "kendis-ir-milestone-resource";
                    mileStoneTask.expanded		= false;
                    mileStoneTask.color = "#e8f4fd";
                    this.$options.roadmap.resourceStore.insert(0,mileStoneTask);
                    this.drawGroupHeaderEvents([this.$options.roadmap.resourceStore.getById(milestoneResourceId)])
                }
            },
            onObjectiveCompletionCriteria(completionCriteria){
                let updatedObj = _.cloneDeep(this.$options.cache.objectivesMap.get(this.itemTemplate.objTemplate.model.id));
                updatedObj.completionCriteria = completionCriteria.code
                this.onEditObjective(updatedObj);
            },
            closeThemeProgress() {
                this.objectiveThemeProgress.show = false;
            },
            initGroupProgressPopup(objectiveGroupId){
                this.objectiveGroupProgressCompData.objectiveGroupId = objectiveGroupId;
                this.objectiveGroupProgressCompData.isArt = true;
                this.objectiveGroupProgressCompData.isGlobalObjectiveView = false;
                this.objectiveGroupProgressCompData.completionBy = 'OBJ_KR_WEIGHTS_FORMULA';
                this.objectiveGroupProgressCompData.radioTypeSelection = 'objectiveGroup';
                this.objectiveGroupProgressCompData.isOkrEnabled = true;

                let relatedObjectivesIds = this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(objectiveGroupId);
                this.objectiveGroupProgressCompData.objectiveList = _.filter(this.objectiveList, objective => relatedObjectivesIds.includes(objective.id));

                this.objectiveGroupProgressCompData.board = {
                    session:{
                        uncommittedObjEnabled: false,
                        id: this.isGlobalObj? "-1":this.releaseTrain.id,
                    },
                    teams:[],
                }
                this.objectiveGroupProgressCompData.objectiveGroup = this.$options.cache.objectiveGroupsMap.get(objectiveGroupId)
                this.objectiveGroupProgressCompData.objectiveGroupList = [...this.$options.cache.objectiveGroupsMap.values()]
                this.objectiveGroupProgressCompData.show = true;
            },
            initThemeProgressPopup(){
                this.objectiveThemeProgress.isArt = true;
                this.objectiveThemeProgress.isGlobalObjectiveView = false;
                this.objectiveThemeProgress.completionBy = 'OBJ_KR_WEIGHTS_FORMULA';
                this.objectiveThemeProgress.radioTypeSelection = 'objectiveGroup';
                this.objectiveThemeProgress.isOkrEnabled = true;

                this.objectiveThemeProgress.board = {
                    session:{
                        uncommittedObjEnabled: false,
                        id: this.isGlobalObj? "-1":this.releaseTrain.id,
                    },
                    teams:[],
                }
                this.objectiveThemeProgress.objectiveGroupList = [...this.$options.cache.objectiveGroupsMap.values()];
                this.objectiveThemeProgress.allObjectiveIds = [...this.$options.cache.objectivesMap.keys()]
                this. objectiveThemeProgress.objectiveGroupsId = [...this.$options.cache.objectiveGroupsMap.keys()]
                this.objectiveThemeProgress.show = true;
            },
            closeGroupProgress(){
                this.objectiveGroupProgressCompData.show = false;


            },
            onSelectUser: function() {
               // console.log(this.responsibleFilter.selectedValues);
               //  localStorage.setItem(this.releaseTrain.id + "_ROADMAP_PRESET_RESPONSIBLE_FILTER_"
               //      +this.roadmap.id, JSON.stringify(this.responsibleFilter.selectedValues));


            },

            reloadObjectiveGroups(){
                this.createObjectiveGroups().then(() => {
                    this.createMilestoneGroupIfNotExist();
                    let groupResource = this.$options.roadmap.resourceStore.getById(milestoneResourceId);
                    if (groupResource.expanded) {
                        this.addMilestones();
                    }
                    this.loadCompletionPercentages(Array.from(this.$options.cache.objectiveGroupsMap.keys()),true);

                });
            },

            onTimePeriodFilterClick(value) {
                let year = this.timePeriodFilter.selectedYear;
                let values = this.timePeriodFilter.selectedValues;

                if (values[year] && values[year].includes(value)){
                    this.timePeriodFilter.selectedValues[year] = _.filter(values[year], _value => _value !== value);
                    if (_.isEmpty(this.timePeriodFilter.selectedValues[year])) {
                        value = "";
                    }
                } else {
                    if (value === "H1") {
                        if (values[year]){
                            if (values[year].includes("Q1") || values[year].includes("Q2")){
                                this.timePeriodFilter.selectedValues[year].push("H1");
                            } else {
                                this.timePeriodFilter.selectedValues[year].push("H1", "Q1", "Q2");
                            }
                        } else {
                            this.timePeriodFilter.selectedValues = { [year] : ["H1", "Q1", "Q2"]};
                        }
                    } else if (value === "H2") {
                        if (values[year]){
                            if (values[year].includes("Q3") || values[year].includes("Q4")){
                                this.timePeriodFilter.selectedValues[year].push("H2");
                            } else {
                                this.timePeriodFilter.selectedValues[year].push("H2", "Q3", "Q4");
                            }
                        } else {
                            this.timePeriodFilter.selectedValues = { [year] : ["H2", "Q3", "Q4"]};
                        }
                    } else if (value === "Yearly") {
                        this.timePeriodFilter.selectedValues = { [year] : ["Yearly", "H1", "H2", "Q1", "Q2", "Q3", "Q4"]};
                    } else if (value === "Monthly") {
                        this.timePeriodFilter.selectedValues = { [year] : _.cloneDeep(this.timePeriodFilter.months)};
                        this.timePeriodFilter.selectedValues[year].push("Monthly");
                    } else {
                        if (values[year]){
                            this.timePeriodFilter.selectedValues[year].push(value);
                        } else {
                            this.timePeriodFilter.selectedValues = { [year] : [value]};
                        }
                    }
                }

                if (_.isEmpty(value)){
                    this.timePeriodFilter.selectedValues = {};
                }

                this.timePeriodFilter.selectedValue = value;
                localStorage.setItem(this.releaseTrain.id + "_ROADMAP_PRESET_TIME_PERIOD_FILTER_"+this.roadmap.id, JSON.stringify(this.timePeriodFilter));
                this.createObjectiveGroups().then(() => {
                    this.createMilestoneGroupIfNotExist();
                    let groupResource = this.$options.roadmap.resourceStore.getById(milestoneResourceId);
                    if (groupResource.expanded) {
                        this.addMilestones();
                    }
                });
            },
            onChangeTimePeriodYear(value) {
                this.timePeriodFilter.selectedYear += value;
            },
            getOptionTooltip(option) {
                let selectedYear = this.timePeriodFilter.selectedYear;

                let getDaysInMonth = function(month, year) {
                    return new Date(year, month, 0).getDate();
                };

                if (option === 'yearly' || option === 'monthly') {
                    return '1 Jan - 31 Dec ' + this.timePeriodFilter.selectedYear;
                }
                if (this.timePeriodFilter.months.indexOf(option) !== -1) {
                    var monthIndex = this.timePeriodFilter.months.indexOf(option) + 1;
                    var daysInMonth = getDaysInMonth(monthIndex, selectedYear);
                    return '1 ' + option + ' - ' + daysInMonth + ' ' + option;
                }
                if (option === 'H1') {
                    return '1 Jan - 30 Jun ' + this.timePeriodFilter.selectedYear;
                }
                if (option === 'H2') {
                    return '1 Jul - 1 Jan ' + (this.timePeriodFilter.selectedYear + 1);
                }
                if (option === 'Q1') {
                    return '1 Jan - 31 Mar ' + this.timePeriodFilter.selectedYear;
                }
                if (option === 'Q2') {
                    return '1 Apr - 30 Jun ' + this.timePeriodFilter.selectedYear;
                }
                if (option === 'Q3') {
                    return '1 Jul - 30 Sep ' + this.timePeriodFilter.selectedYear;
                }
                if (option === 'Q4') {
                    return '1 Oct - 31 Dec ' + this.timePeriodFilter.selectedYear;
                }
            },
            addObjectiveGroup(title){
                // Check EDIT_OBJECTIVE_GROUP permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'EDIT_OBJECTIVE_GROUP', 'create objective groups')) {
                    return;
                }
                
                const defaultObjectiveGrouptTitle = title;
                let objective = {};
                objective.title = defaultObjectiveGrouptTitle;
                this.onCreateCustomObjectiveGroup(objective);
            },
            onCreateCustomObjectiveGroup: function (objective) {
                let teamLink = {};
                teamLink.type = "Objective-Group";
                teamLink.teamId = 0;
                teamLink.linkType = "art_link_objective";
                teamLink.sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;

                objective.type = "Objective-Group"
                objective.headerColor = "#f9bf3b";
                objective.baseItemLinks = [];
                objective.expand = false;
                objective.baseItemLinks.push(teamLink);

                if(_.find(kendisStore.getters.getObjectiveGroupTemplate().statuses, {"category":"ToDo"}) != -1) {
                    objective.status = _.find(kendisStore.getters.getObjectiveGroupTemplate().statuses, {"category": "ToDo"});
                }

                let metaMap = {};
                metaMap.syncId = syncId;
                metaMap.sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                metaMap.teamId = 0;
                metaMap.boardId =this.isGlobalObj? "-1":this.releaseTrain.id;

                if(this.isGlobalObj){
                    metaMap.viewId = this.viewId;
                    objective.linkedCustomView = this.viewId;
                }
                objective.metaMap = JSON.stringify(metaMap);

                this.onCreateObjectiveGroup(objective);
            },
            onAddCollectionObjectiveGroup: function (collection) {

                let objective = {};

                //............   team link
                let teamLink = {};
                teamLink.type = "Objective-Group";
                teamLink.teamId = 0;
                teamLink.linkType = "art_link_objective";
                teamLink.sessionId = "-1";


                //.......... collection link
                let collectionLink = {};
                collectionLink.type = "Objective-Group";
                collectionLink.teamId = 0;

                collectionLink.linkType = "art_link_objective";
                collectionLink.sessionId = collection.id;

                objective.baseItemLinks = [];
                objective.baseItemLinks.push(teamLink);
                objective.baseItemLinks.push(collectionLink);
                //.............................................
                objective.type = "Objective-Group"
                objective.headerColor = collection.color ? collection.color : "#f9bf3b";
                objective.title = collection.title;

                objective.expand = false;

                if(_.find(kendisStore.getters.getObjectiveGroupTemplate().statuses, {"category":"ToDo"}) != -1) {
                    objective.status = _.find(kendisStore.getters.getObjectiveGroupTemplate().statuses, {"category": "ToDo"});
                }

                let metaMap = {};
                    metaMap.syncId = syncId;
                    metaMap.sessionId = "-1"
                    metaMap.teamId = 0;
                    metaMap.boardId = "-1"
                    metaMap.viewId = this.viewId;
                    objective.linkedCustomView = this.viewId;

                if (collection && collection.id) {
                    metaMap.collectionId = collection.id;
                }

                objective.metaMap = JSON.stringify(metaMap);
                this.onCreateObjectiveGroup(objective);

            },

            onCreateObjectiveGroup: function (objective) {
                let _this = this;
                axios.post('objective/update-objective', objective).then(response => {
                    if(response && response.data &&  response.data.id) {
                        let newObjective = response.data;
                        _this.$options.cache.objectiveGroupsMap.set(newObjective.id, newObjective);

                        //add as resource
                        scheduler.resourceStore.add(_this.resourceMapper(newObjective));

                        //CREATE THE RESOURCE HEADER EEVENT
                        let resource = scheduler.resourceStore.getById(newObjective.id);
                        _this.drawGroupHeaderEvents([resource]);

                        //scroll to resource bryntum
                        scheduler.scrollResourceIntoView(newObjective.id,{animate:200})
                        _this.openObjectiveGroupEditPopup(newObjective);

                        scheduler.refresh()
                    }
                }).catch(error => {
                    showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    console.error("Objective save error", error);
                });
            },
            openAndHandlePopup(input) {
                const datePopup = this.$refs['timeline-item-edit'];
                return  datePopup.createPopup(input);
            },
            onCLoseEditPopUp: function () {
                let _this = this;
                _this.itemTemplate.objTemplate.showEditPopUp  = false;
                _this.objectiveModalDataLoaded = false;

            },
            onCLoseEditKRPopUp: function () {
                let _this = this;
                _this.itemTemplate.krTemplate.showEditPopUp  = false;
                _this.krModalDataLoaded = false;
                this.onCLoseEditPopUp();
            },
            isTitleValid(title) {
                /*This method checks the following things:
                1- Firstly it eliminates and leading and trailing space characters
                2- Secondly it checks that the title is not empty */
                if(title !== undefined) {
                    title = title.trim();
                    return title.length > 0;
                }
                return false;
            },
            onSaveEditObjectiveData: function (objective, updateObjFromContainer) {
                if (!this.isTitleValid(objective.title)) {
                    // this.isEditCustomObjectiveInputValid = false;
                    showTopMessageTargeted("Please enter a valid title.", "warning", 3000, this.isFullScreen ? '.okrsrdmp' : null);
                    return;
                }
                if(objective.itemType && objective.itemType.title === 'KR'){
                    this.onSaveEditOkr(objective);
                }else{
                    this.onSaveObjective(objective)
                }
            },

            onChangeDatesMilestone(startDate, endDate ,milestoneId){

                let _this 	= this;
                let metaMap = {};
                let item 	= {};
                item.id 	=milestoneId;
                metaMap.syncId 	= syncId;
                metaMap.event 	= "Update";
                //...
                var data = {};

                if (!item.fields) {
                    item.fields = {};
                }

                if (startDate && _.isDate(startDate)) {
                    item.fields.StartDate = startDate.format("yyyy-mm-dd'T'HH:MM:ss'Z'");
                }
                if (endDate && _.isDate(endDate)) {
                    item.fields.EndDate = endDate.format("yyyy-mm-dd'T'HH:MM:ss'Z'");
                }
                let alm = this.isGlobalObj?undefined:this.$store.almAccounts[0];
                if (alm) {
                    data.almAccountId = alm.id;
                }

                data.releaseTrain = this.isGlobalObj?{id:this.viewId}:{id:this.releaseTrain.id};
                data.item = item;
                data.metaMap = metaMap;
                axios.post('/releasetrain/item/save', data)
                    .then(response => {
                        if (response.data) {
                             let newItem = response.data.item;
                            let item = this.milestoneMapById[newItem.id];
                            if (!item.fields) {
                                item.fields = {};
                            }
                            if(item.type === "Milestone"){
                                item.fields.StartDate = newItem.fields.StartDate;
                                item.fields.EndDate   = newItem.fields.StartDate;
                            }else{
                                item.fields.StartDate = newItem.fields.StartDate;
                                item.fields.EndDate   = newItem.fields.EndDate;
                            }
                        }
                    })
                    .catch(error => {
                        console.error(error)
                        showTopMessageTargeted("Error saving milestone", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    });
            },
            onChangeDates: async function (newStartDate, newEndDate, objId) {
                let requestBody = {
                    objectiveId: objId,
                    isMockingMode: true,
                    fetchPIs: true,
                    requestId: getNewUUID()
                };
                try {
                    const response = await fetch('objective/get-objective-by-id', {
                        method: 'POST',
                        headers: getFetchAPIHeadersForKendis(),
                        credentials: 'include', // Include cookies for CSRF-TOKEN
                        body: JSON.stringify(requestBody)
                    });

                    if (response.ok) {
                        const data = await response.json();
                        if (data && data.objective) {
                            if(data.objective.type==="KR"){
                                let keyResult = data.objective;
                                let fieldsMapKR = kendisStore.getters.getOkrTemplate().fieldsMap;
                                let startDateFieldIdKr, endDateFieldIdKr,responsibleFieldIdKr,timePeriodFieldIdKr;

                                for (let fieldId in fieldsMapKR) {
                                    if(fieldsMapKR[fieldId]==="Start Date"){
                                        startDateFieldIdKr = fieldId;
                                    }else if(fieldsMapKR[fieldId]==="End Date"){
                                        endDateFieldIdKr = fieldId;
                                    }else if(fieldsMapKR[fieldId]==="Responsible"){
                                        responsibleFieldIdKr = fieldId;
                                    }else if(fieldsMapKR[fieldId]==="Time Period"){
                                        timePeriodFieldIdKr = fieldId;
                                    }
                                }
                                keyResult.fields = keyResult.fields || {};
                                if(newStartDate){
                                    keyResult.fields[startDateFieldIdKr] = newStartDate;
                                }
                                if(newEndDate){
                                    keyResult.fields[endDateFieldIdKr] = newEndDate;
                                }

                                    const date = newEndDate
                                    const timePeriod = keyResult.fields[timePeriodFieldIdKr] || {};
                                    const month = date.getMonth();
                                    const year = date.getFullYear();
                                    const getQuarter = (month) => ['Q1', 'Q2', 'Q3', 'Q4'][Math.floor(month / 3)];
                                    if (_.isEmpty(timePeriod)) {
                                        keyResult.fields[timePeriodFieldIdKr] = {
                                            title: getQuarter(month),
                                            selectedYear: year,
                                            monthly: false
                                        };
                                    }else{
                                        timePeriod.selectedYear = year;
                                        if (timePeriod.monthly) {
                                            const selectedMonth = this.timePeriod.months[month];
                                            if (selectedMonth !== timePeriod.title) {
                                                timePeriod.title = selectedMonth;
                                            }
                                        }else{
                                            const newTitle = timePeriod.title.startsWith('H') ? (month < 6 ? 'H1' : 'H2') : getQuarter(month);
                                            if (timePeriod.title !== newTitle) timePeriod.title = newTitle;
                                        }
                                        keyResult.fields[timePeriodFieldIdKr] = timePeriod;
                                    }
                                    this.onSaveEditOkr(keyResult);
                                }
                            else{
                            let objective = data.objective;
                            let fieldsMap = kendisStore.getters.getObjectiveTemplate().fieldsMap;
                            let startDateFieldId, endDateFieldId,timePeriodFieldId;

                            // Loop through the fieldsMap to find the IDs for 'Start Date' and 'End Date'
                            for (let fieldId in fieldsMap) {
                                if (fieldsMap[fieldId] === 'Start Date') {
                                    startDateFieldId = fieldId;
                                } else if (fieldsMap[fieldId] === 'End Date') {
                                    endDateFieldId = fieldId;
                                }else if( fieldsMap [fieldId] === "Time Period"){
                                    timePeriodFieldId = fieldId;

                                }
                            }

                            //modify dates

                            objective.fields = objective.fields || {};
                            if(newStartDate ) {
                                objective.fields[startDateFieldId] = newStartDate;
                            }
                            if(newEndDate ) {
                                objective.fields[endDateFieldId] = newEndDate;
                            }

                            //set the timeperiod according to the new enddate
                                const date = newEndDate
                                const timePeriod = objective.fields[timePeriodFieldId] || {};
                                const month = date.getMonth();
                                const year = date.getFullYear();
                                const getQuarter = (month) => ['Q1', 'Q2', 'Q3', 'Q4'][Math.floor(month / 3)];
                                if (_.isEmpty(timePeriod)) {
                                    objective.fields[timePeriodFieldId] = {
                                        title: getQuarter(month),
                                        selectedYear: year,
                                        monthly: false
                                    };
                                } else {
                                    timePeriod.selectedYear = year;

                                    if (timePeriod.monthly) {
                                        const selectedMonth = this.timePeriod.months[month];
                                        if (selectedMonth !== timePeriod.title) {
                                            timePeriod.title = selectedMonth;
                                        }
                                    } else {
                                        const newTitle = timePeriod.title.startsWith('H') ? (month < 6 ? 'H1' : 'H2') : getQuarter(month);
                                        if (timePeriod.title !== newTitle) timePeriod.title = newTitle;
                                    }

                                    objective.fields[timePeriodFieldId] = timePeriod;
                                }

                            this.onEditObjective(objective, true);
                                }
                        }
                    }
                } catch (error) {
                    console.error(error);
                    showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            onEditObjective: function (objective, updateSilent=false) {
                let _this = this;
                let baseItem = _.clone(_this.$options.cache.objectivesMap.get(objective.id));
                this.updateObjective(baseItem, objective, updateSilent);
            },
            updateObjective: function (baseItem, objective, updateSilent=false) {
                let _this = this;
                delete baseItem.board;
                delete baseItem.linkedItems;
                delete baseItem.linkedItemsById;
                delete baseItem.linkedItemsIds;
                delete baseItem.baseItemList;
                baseItem.title = objective.title;
                if (objective.itemType.title === "Objective") {
                    if(kendisStore.getters.getObjectiveMenuOptions()['isBVAchievementEnabled']
                        && (objective.bvPlan !== undefined && objective.bvPlan !== null)
                        && (objective.bvActual !== undefined && objective.bvActual !== null)) {
                        baseItem.bvPlan = objective.bvPlan;
                        baseItem.bvActual = objective.bvActual;
                    }
                }

                baseItem.notifyWatchers = objective.notifyWatchers;
                baseItem.starred = objective.starred;
                baseItem.metaMap = objective.metaMap;
                if (objective.fields){
                    baseItem.fields = objective.fields;
                }
                if (objective.description){
                    baseItem.description = objective.description;
                }
                if (objective.baseItemLinks){
                    baseItem.baseItemLinks = _.compact(objective.baseItemLinks);
                }
                if (objective.completionCriteria){
                    baseItem.completionCriteria = objective.completionCriteria;
                }
                if (objective.status){
                    baseItem.status = objective.status;
                }
                if (objective.type){
                    baseItem.type = objective.type;
                }
                let objectiveLink = _.filter(baseItem.baseItemLinks, {type: "Objective"})[0];
                if (objectiveLink) {
                    delete objectiveLink.deleted;
                }
                if (!baseItem.metaMap || baseItem.metaMap == "undefined") {
                    let metaMap = {};
                    metaMap.sosId = "0";
                    metaMap.syncId = syncId;
                    metaMap.sessionId = this.isGlobalObj? "-1":this.releaseTrain.id
                    metaMap.boardId = this.isGlobalObj? "-1":this.releaseTrain.id
                    baseItem.metaMap = JSON.stringify(metaMap);
                }else{
                    baseItem.metaMap = JSON.parse(baseItem.metaMap);
                    if (!baseItem.metaMap.groupId){
                        baseItem.metaMap.groupId = findKeyByValue(_this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap, objective.id);
                    }
                    baseItem.metaMap = JSON.stringify(baseItem.metaMap);
                }
                if (baseItem.eventType) {
                    delete baseItem.eventType;
                }

                if (baseItem.highlightedKey){
                    delete baseItem.highlightedKey;
                }
                if (baseItem.highlightedTitle){
                    delete baseItem.highlightedTitle;
                }
                _.each(baseItem.baseItemLinks, link=>{
                    if(!_.isEmpty(link.linkedItemById)) {
                        delete link.linkedItemById;
                    }
                });
                axios.post('external-entitites/save-base-item/popup-save', baseItem)
                    .then(response => {

                        //update cache
                        let updatedObjective = response.data.impediment;
                        _this.$options.cache.objectivesMap.set(updatedObjective.id, updatedObjective);
                        _this.updateRenderedObjective(updatedObjective);

                        if(!_.isEmpty(updatedObjective) && !_.isEmpty(updatedObjective.baseItemLinks)) {
                             this.loadCompletionPercentages(Array.from(this.$options.cache.objectiveGroupsMap.keys()),true);
                        }
                        showTopMessageTargeted("Objective Successfully Updated",'success',1500, this.isFullScreen ? '.okrsrdmp' : null)

                    })
                    .catch(error => {
                        console.error("Item save error", error);
                        showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    })
            },
            updateRenderedObjective(updatedObj){
                let event = scheduler.eventStore.getById(updatedObj.id);
                let fieldsMap = kendisStore.getters.getObjectiveTemplate().fieldsMap;
                let startDateFieldId, endDateFieldId,responsibleFieldId


                // Loop through the fieldsMap to find the IDs for 'Start Date' and 'End Date'
                for (let fieldId in fieldsMap) {
                    if (fieldsMap[fieldId] === 'Start Date') {
                        startDateFieldId = fieldId;
                    } else if (fieldsMap[fieldId] === 'End Date') {
                        endDateFieldId = fieldId;
                    }
                    else if (fieldsMap[fieldId] === 'Responsible') {
                        responsibleFieldId = fieldId;
                    }
                }
                    let startDateFieldValue = updatedObj.fields ? updatedObj.fields[startDateFieldId]:undefined;
                    let endDateFieldValue = updatedObj.fields? updatedObj.fields[endDateFieldId]:undefined;

                    let startDate = startDateFieldValue?new Date(startDateFieldValue):new Date()
                    let endDate = endDateFieldValue?new Date(endDateFieldValue):new Date(new Date().setMonth(new Date().getMonth() + 1));

                    if(event.get('type') === 'objective'){
                    startDate.setHours(0, 0, 0, 0); // for complete time range cover
                    endDate.setHours(23, 59, 59, 999);
                    }
                    else{
                        startDate.setHours(1, 0, 1, 0); // for complete time range cover
                        endDate.setHours(23, 59, 59, 999);
                    }

                    let responsibleFieldValue = updatedObj.fields ? updatedObj.fields[responsibleFieldId]:undefined
                if(event){
                    event.name = updatedObj.title;
                    event.startDate= startDate;
                    event.endDate = endDate;
                    event.responsible  =  responsibleFieldValue;
                    event.eventColor = updatedObj.status.category === "ToDo" ? "#E79362" : updatedObj.status.category === "InProgress" ? "#3792C9" : updatedObj.status.category === "Done" ? "#3FCA93" : "#7c50b1"
                }
                // Update the resource name as well (handle both original and prefixed IDs)
                let resource = scheduler.resourceStore.getById(updatedObj.id);
                if (!resource && this.isMultilevel) {
                    // Try to find resource with prefixed ID
                    let resources = scheduler.resourceStore.query(resource => 
                        resource.get('originalObjectiveId') === updatedObj.id
                    );
                    if (resources.length > 0) {
                        resource = resources[0];
                    }
                }
                if (resource) {
                    resource.name = updatedObj.title;
                }

                // Update completion percentages when objective is updated
                this.updateCompletionPercentages();

                // Refresh scheduler to show updated responsible person in event renderer
                scheduler.refresh();
            },
                        async addObjectiveToObjectiveGroup(event) {
                // Check EDIT_OBJECTIVE permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'EDIT_OBJECTIVE', 'add objectives')) {
                    return;
                }
                

                let groupId = event.dataset.id;

                //if the group is closed annd the data is not loaded
                // for that groupthen only fetch the group data
                let group = scheduler.resourceStore.getById(groupId);
                if (!group.isExpanded()) {
                    //clean the events and children resources related to that group
                    let childrenResources = scheduler.resourceStore.getChildren(groupId);
                    _.forEach(childrenResources, (resource) => {
                        scheduler.eventStore.remove(scheduler.eventStore.getByResourceId(resource.id));
                        scheduler.resourceStore.remove(resource.id);
                    });
                    //fetch the group data
                    this.loadObjectivesOfGroups(groupId, true);
                }


                let fieldsMap = kendisStore.getters.getObjectiveTemplate().fieldsMap;
                let startDateFieldId, endDateFieldId,timePeriodFieldId

                // Loop through the fieldsMap to find the IDs for 'Start Date' and 'End Date'
                for (let fieldId in fieldsMap) {
                    if (fieldsMap[fieldId] === 'Start Date') {
                        startDateFieldId = fieldId;
                    } else if (fieldsMap[fieldId] === 'End Date') {
                        endDateFieldId = fieldId;
                    } else if (fieldsMap[fieldId] === 'Time Period'){
                        timePeriodFieldId = fieldId
                    }
                }

                let input = {};
                input.itemIdentifier = "Create New Objective";
                input.startDate = undefined
                input.endDate = undefined
                input.showTextField = true;
                input.showDateFields = true;
                input.textValue ="Objective title";
                input.textInputLabel = "Enter Objective title"
                input.showTimePeriod = true;
                input.maxLength = 750; // Add character limit for objective creation

                let promise = this.openAndHandlePopup(input);

                promise.then(async (response) => {
                    let newObj = {
                        title: response.textValue,

                    }
                    newObj.fields = newObj.fields || {};
                    if (response.startDate) {
                        newObj.fields[startDateFieldId] = response.startDate;
                    }
                    if (response.endDate) {
                        newObj.fields[endDateFieldId] = response.endDate;
                    }
                    if(response.timePeriod){
                        newObj.fields[timePeriodFieldId] = response.timePeriod
                    }

                    //if the group is closed annd the data is not loaded
                    // for that groupthen only fetch the group data



                    let newObjResponse = await this.addNewObjective(newObj, groupId);
                    // let group = scheduler.resourceStore.getById(newObjResponse.objGroup.id);
                    // if (!group.expanded) {
                    //     scheduler.resourceStore.toggleCollapse(group.id)
                    // }
                    // if (group.expanded) {
                    //     let objGroup = this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(newObjResponse.objGroup.id);
                    //     if (objGroup) {
                    //         //update caches
                    //         this.$options.cache.objectivesMap.set(newObjResponse.objective.id, newObjResponse.objective);
                    //         this.$options.cache.objectiveGroupsMap.set(newObjResponse.objGroup.id, newObjResponse.objGroup);
                    //         this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(newObjResponse.objGroup.id).push(newObjResponse.objective.id);

                    //         let{events, assignments, dependencies}=
                    //             this.createEventsAssignmentsForObjectives([newObjResponse.objective.id],
                    //             newObjResponse.objGroup.id,true);

                    //         scheduler.eventStore.add(events);
                    //         scheduler.assignmentStore.add(assignments);

                    //         // Recalculate progress for the objective group after adding new objective
                    //         await this.loadCompletionPercentages([newObjResponse.objGroup.id], true);
                    //     } else {
                    //         this.loadObjectivesOfGroups(group.id, true);
                    //     }
                    // }
                    scheduler.refresh();
                    showTopMessageTargeted("Objective created successfully", 'success', 3000, this.isFullScreen ? '.okrsrdmp' : null);

                }).catch((error) => {
                    console.log(error);
                    showTopMessageTargeted("Action cancelled", 'warning', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            onDeleteFromPopup: function(parentKey,objId) {
                let _this = this;
                let message = "Do you want to delete selected entity?";

                // Determine target container for fullscreen mode
                let targetContainer = null;
                if (_this.isFullScreen) {
                    targetContainer = document.querySelector('.okrsrdmp');
                }

                this.$refs['kendis-alert'].fire({
                    title: 'Confirm Delete',
                    text: message,
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#d33',
                    cancelButtonColor: '#6c757d',
                    confirmButtonText: 'Yes, delete it!',
                    cancelButtonText: 'Cancel',
                    target: targetContainer
                }).then((result) => {
                    if (result.isConfirmed) {
                        _this.onCLoseEditKRPopUp();
                        _this.deleteObj(parentKey, objId);
                    }
                });
            },
            deleteObj: async function (parentKey,objId) {
                let _this = this;

                if(!parentKey){
                let requestBody = {};
                
                let metaMap = {};
                    metaMap = {
                        "syncId" : syncId,
                        "sessionId" :  this.isGlobalObj? "-1":this.releaseTrain.id
                    }
                requestBody.objId = objId;
                requestBody.metaMap = JSON.stringify(metaMap);

                    axios.post('/objective/delete-objective', requestBody).then(async response => {
                        if (response && response.status == "200") {
                            _this.onCLoseEditPopUp();
                            let deletedObj = response.data.deletedObj;
                            _this.$options.cache.objectivesMap.delete(objId);
                            //remove from cache.ObjectiveGroupsAndObjectivesRelationMap
                            let groupId = findKeyByValue(_this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap, objId);
                            if (groupId){
                                let objIds = _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(groupId);
                                let index = objIds.indexOf(objId);
                                if (index > -1) {
                                    objIds.splice(index, 1);
                                }
                                _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.set(groupId, objIds);
                                // Recalculate progress for the group
                                await _this.loadCompletionPercentages([groupId], true);
                            }
                            _this.deleteRenderedObjective(objId);
                            //remove the objective resource
                            let resource = scheduler.resourceStore.getById(objId);
                            if(resource){
                                scheduler.resourceStore.remove(resource.id);
                            }
                            scheduler.refresh();
                    }
                }).catch(error => {
                    console.error("Objective delete error", error);
                    showTopMessageTargeted("Error deleting objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            }else{
                let requestBody = {};
                requestBody.objId = objId;
                let metaMap = {};
                metaMap = {
                    "syncId" : syncId,
                    "sessionId" :  this.isGlobalObj? "-1":this.releaseTrain.id
                }

                //get the objective from the cache using parentKey not id as the map is keyed by id not key
                //loop on objectivesMap and find the objective with the parentKey
                let parent = null;
                this.$options.cache.objectivesMap.forEach((value, key) => {
                    if(value.key === parentKey){
                        parent = value;
                    }
                });
                if(!parent){
                    return;
                }
                requestBody.groupId = parent.id;
                requestBody.objId = objId;
                requestBody.metaMap = JSON.stringify(metaMap);

                axios.post('/objective/delete-objective', requestBody).then(async response => {
                    if (response && response.status == "200") {
                        _this.onCLoseEditPopUp();
                        //remove the child kr or objective from the cache
                        _this.$options.cache.objectivesMap.delete(objId);
                        _this.deleteRenderedObjective(objId);
                       
                    }
                }).catch(error => {
                    console.error("Objective delete error", error);
                    showTopMessageTargeted("Error deleting objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            }
            },
                        confirmDeleteObjective: function(objective) {
                // Check DELETE_OBJECTIVE permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'DELETE_OBJECTIVE', 'delete objectives')) {
                    return;
                }
                
                let _this = this;
                let message = `Are you sure you want to delete the objective "${objective.title}"?`;

                // Determine target container for fullscreen mode
                let targetContainer = null;
                if (_this.isFullScreen) {
                    targetContainer = document.querySelector('.okrsrdmp');
                }

                // Use KENDIS-ALERT component with target support
                this.$refs['kendis-alert'].fire({
                    title: 'Confirm Delete',
                    text: message,
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#d33',
                    cancelButtonColor: '#6c757d',
                    confirmButtonText: 'Yes, delete it!',
                    cancelButtonText: 'Cancel',
                    target: targetContainer
                }).then((result) => {
                    if (result.isConfirmed) {
                        _this.deleteObj(null, objective.id);
                    }
                });
            },
            deleteRenderedObjective(objId){
                let event = this.$options.roadmap.eventStore.getById(objId);
                if(event){
                    // 1. Remove all assignments for this event FIRST
                    let assignments = this.$options.roadmap.assignmentStore.getAssignmentsForEvent(objId);
                    if(assignments && assignments.length > 0) {
                        this.$options.roadmap.assignmentStore.remove(assignments);
                    }

                    // 2. Then remove the event
                    this.$options.roadmap.eventStore.remove(event);

                    // 3. Update resource counts for affected resources
                    if(assignments && assignments.length > 0) {
                        assignments.forEach(assignment => {
                            const resourceId = assignment.resourceId;
                            if(resourceId) {
                                // Update count display for the resource
                                const currentCount = this.$options.roadmap.assignmentStore.getAssignmentsForResource(resourceId).length - 1;
                                const resource = this.$options.roadmap.resourceStore.getById(resourceId);
                                if(resource) {
                                    resource.currentCount = Math.max(0, currentCount);
                                }
                            }
                        });
                    }

                    // 4. Show success message
                    showTopMessageTargeted("Objective deleted successfully", 'success', 2000, this.isFullScreen ? '.okrsrdmp' : null);
                }
            },

            openObjectiveEditPopup: function (objective) {
                this.openModalForEditing(objective, false);
            },
            createObjectivesGroupsAndTeams(){
                let teamOptions = {
                    id: "Teams",
                    items: [],
                    title: "Teams"
                };
                let customGroupOptions = {
                    id: "CustomObjective",
                    items: [],
                    title: "Custom Objective"
                };
                const customViews = kendisStore.getters.getCustomViews();
                let objectiveGroupsList = Array.from(this.$options.cache.objectiveGroupsMap.values()).map(group => ({ ...group }));
                _.forEach(objectiveGroupsList, group => {
                    if(group.baseItemLinks[0].teamId == 0){
                        // custom group
                        if (group.linkedCustomView) {
                            const view = customViews.find(view => view.id === group.linkedCustomView);
                            if (view) {
                                group.viewTitle = view.title;
                            }
                        }
                        group.label = !this.isGlobalObj ? (group.viewTitle ? group.viewTitle : group.title) : group.title;
                        customGroupOptions.items.push(group);
                    } else {
                        // team group
                        group.label = group.title;
                        teamOptions.items.push(group);
                    }

                });
                let tempArrayForTeamsAndGroupsOptions = [];
                tempArrayForTeamsAndGroupsOptions.push(teamOptions);
                tempArrayForTeamsAndGroupsOptions.push(customGroupOptions);
                return  tempArrayForTeamsAndGroupsOptions;
            },

            async getFreshOkr(id) {
                let requestBody = {};
                requestBody.objectiveId = id;
                requestBody.isMockingMode = true
                requestBody.fetchPIs = true
                requestBody.requestId = getNewUUID();

                let saveResponse = await axios.post('objective/get-objective-by-id', requestBody);
                return saveResponse.status === 200 ? saveResponse.data.objective : null
            },


            onSaveEditOkr: async function (updatedOkr) {
                let _this = this;
                let updatedIndex;
                let baseItem = await this.getFreshOkr(updatedOkr.id);
                if(baseItem !== undefined) {
                    if (baseItem.board && baseItem?.board !== undefined) {
                        delete baseItem.board;
                    }
                    if (baseItem.linkedItems !== undefined) {
                        delete baseItem.linkedItems;
                    }
                    if (baseItem.linkedItemsById !== undefined) {
                        delete baseItem.linkedItemsById;
                    }
                    if (baseItem.linkedItemsIds !== undefined) {
                        delete baseItem.linkedItemsIds;
                    }
                    if (baseItem.editView !== undefined) {
                        delete baseItem.editView;
                    }
                } else  {
                    return;
                }
                baseItem.title = updatedOkr.title;
                if (updatedOkr.bvPlan){
                    baseItem.bvPlan = updatedOkr.bvPlan;
                }
                if (updatedOkr.bvActual){
                    baseItem.bvActual = updatedOkr.bvActual;
                }
                if (updatedOkr.completionCriteria){
                    baseItem.completionCriteria = updatedOkr.completionCriteria;
                }
                baseItem.notifyWatchers = updatedOkr.notifyWatchers;
                baseItem.starred = updatedOkr.starred;
                baseItem.metaMap = updatedOkr.metaMap;
                baseItem.fields = updatedOkr.fields;
                baseItem.description = updatedOkr.description;
                baseItem.baseItemLinks = _.compact(updatedOkr.baseItemLinks);
                baseItem.status = updatedOkr.status;

                // Update scheduler event for KR
                if ((baseItem.type === 'KR' || baseItem.type === 'keyResult') ) {
                    let startDateKRFieldId = this.startDateKRFieldId;
                    let endDateKRFieldId = this.endDateKRFieldId;
                    let event = scheduler.eventStore.getById(baseItem.id);
                    if (event) {
                        if(baseItem.fields && baseItem.fields[startDateKRFieldId] && baseItem.fields[endDateKRFieldId]){
                            event.startDate = new Date(baseItem.fields[startDateKRFieldId]);
                            event.endDate = new Date(baseItem.fields[endDateKRFieldId]);
                        }
                        if(baseItem.status){
                        // Set color based on status
                        event.eventColor = baseItem.status.category === "ToDo"
                            ? "#E79362"
                            : baseItem.status.category === "InProgress"
                            ? "#3792C9"
                            : baseItem.status.category === "Done"
                            ? "#3FCA93"
                            : "#7c50b1";
                        }

                        event.name = baseItem.title;
                        scheduler.refresh();
                    }
                }
                if (updatedOkr.newIndex !== undefined){
                    updatedIndex = updatedOkr.newIndex
                    delete updatedOkr.newIndex;
                }
                // if (!baseItem.metaMap || baseItem.metaMap == "undefined") {
                //     let metaMap = {};
                //     metaMap.sosId = "0";
                //     metaMap.syncId = syncId;
                //     metaMap.sessionId = this.board.session.id;
                //     metaMap.boardId = this.board.id;
                //     metaMap.objectiveId= _this.objectiveData.id;
                //     baseItem.metaMap = JSON.stringify(metaMap);
                // }else{
                //     baseItem.metaMap = JSON.parse(baseItem.metaMap);
                //     if (!baseItem.metaMap.objectiveId && !baseItem.metaMap.groupId){
                //         baseItem.metaMap.objectiveId = _this.objectiveData.id;
                //     }
                //     baseItem.metaMap = JSON.stringify(baseItem.metaMap);
                // }

                if (baseItem.highlightedKey){
                    delete baseItem.highlightedKey;
                }
                if (baseItem.highlightedTitle){
                    delete baseItem.highlightedTitle;
                }

                _.each(baseItem.baseItemLinks, link=>{
                    if(!_.isEmpty(link.linkedItemById)) {
                        delete link.linkedItemById;
                    }
                });
                delete baseItem.parent;
                axios.post('external-entitites/save-base-item/popup-save', baseItem)
                    .then(response => {
                        if (response.status == "200" && response && response.data) {
                            if(baseItem.type==="KR"){
                                _this.$options.cache.keyResultsMap.set(baseItem.id, baseItem);
                            }else{
                                _this.$options.cache.objectivesMap.set(baseItem.id, baseItem);
                            }
                            _this.loadCompletionPercentages(Array.from(this.$options.cache.objectiveGroupsMap.keys()),true);
                            showTopMessageTargeted("Key Result Updated", 'success', 4000, _this.isFullScreen ? '.okrsrdmp' : null);
                        }
                    })
                    .catch(error => {
                        console.error("Item save error", error);
                        showTopMessageTargeted("Error saving objective", 'error', 4000, _this.isFullScreen ? '.okrsrdmp' : null)
                    })
            },
            findFirstObjectiveGroupIdofObjective(objectiveId){
                // ObjectiveGroupsAndObjectivesRelationMap is a map()
                let objectiveGroupId = null;
                this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.forEach((value, key) => {
                    if (value.includes(objectiveId)) {
                      objectiveGroupId = key;
                      return objectiveGroupId;
                    }
                });
                return objectiveGroupId;
            },

            openKRModelForEditing: function (objective, closeParent,parent) {
                let _this = this;
                _this.itemTemplate.krTemplate = {};
                _this.krModalDataLoaded = false;
                _this.itemTemplate.krTemplate.model = {};
                _this.itemTemplate.krTemplate.model.fields = {};
                _this.itemTemplate.krTemplate.statuses = [];
                _this.itemTemplate.krTemplate.id = objective.id;
                _this.itemTemplate.krTemplate.model.id = objective.id;
                _this.itemTemplate.krTemplate.model.title = objective.title;
                _this.itemTemplate.krTemplate.model.key = objective.title;
                _this.itemTemplate.krTemplate.objectiveGroupsCopy = _this.createObjectivesGroupsAndTeams();
                _this.itemTemplate.krTemplate.showEditPopUp = true;
                _this.itemTemplate.krTemplate.krParent = parent;
                _this.itemTemplate.krTemplate.parentKey = parent ? parent.key : null;

                _this.itemTemplate.krTemplate.objectiveGroupId = this.findFirstObjectiveGroupIdofObjective(objective.id);
                _this.itemTemplate.krTemplate.krParentTimePeriodFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["Time Period"];
                _this.itemTemplate.krTemplate.krParentStartDateFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["Start Date"];
                _this.itemTemplate.krTemplate.krParentEndDateFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["End Date"];
                
                let requestBody = {};
                requestBody.objectiveId = objective.id;
                requestBody.isMockingMode = true
                requestBody.fetchPIs = true
                requestBody.requestId = getNewUUID();
                axios.post('objective/get-objective-by-id', requestBody)
                  .then(response => {
                      if (response && response.data && response.data.objective) {
                          _this.fetchOkrCallback(objective, response.data.objective, response.data.sessionMapById,
                            closeParent,_this.itemTemplate.krTemplate, _this.krModalDataLoaded)
                      }
                  })
                  .catch(error => {
                      console.error(error);
                      showTopMessageTargeted("Error loading objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                  })
            },

            openModalForEditing: function (objective, closeParent) {
                let _this = this;
                _this.itemTemplate.objTemplate = {};
                _this.objectiveModalDataLoaded = false;
                _this.itemTemplate.objTemplate.model = {};
                _this.itemTemplate.objTemplate.model.fields = {};
                _this.itemTemplate.objTemplate.statuses = [];
                _this.itemTemplate.objTemplate.id = objective.id;
                _this.itemTemplate.objTemplate.model.id = objective.id;
                _this.itemTemplate.objTemplate.model.title = objective.title;
                _this.itemTemplate.objTemplate.model.key = objective.title;
                _this.itemTemplate.objTemplate.objectiveGroupsCopy = _this.createObjectivesGroupsAndTeams();
                _this.objectiveData = objective
                _this.itemTemplate.objTemplate.showEditPopUp = true;
                // _this.itemTemplate.objTemplate.krParent = objective;

                _this.itemTemplate.objTemplate.objectiveGroupId = this.findFirstObjectiveGroupIdofObjective(objective.id);
                _this.itemTemplate.objTemplate.krParentTimePeriodFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["Time Period"];
                _this.itemTemplate.objTemplate.krParentStartDateFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["Start Date"];
                _this.itemTemplate.objTemplate.krParentEndDateFieldId = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap["End Date"];
                let requestBody = {};
                requestBody.objectiveId = objective.id;
                requestBody.isMockingMode = true
                requestBody.fetchPIs = true
                requestBody.requestId = getNewUUID();
                axios.post('objective/get-objective-by-id', requestBody)
                    .then(response => {
                        if (response && response.data && response.data.objective) {
                            _this.fetchOkrCallback(objective, response.data.objective, response.data.sessionMapById, closeParent,
                              _this.itemTemplate.objTemplate, _this.objectiveModalDataLoaded)
                        }
                    })
                    .catch(error => {
                        console.error(error);
                        showTopMessageTargeted("Error loading objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    })
            },
            fetchOkrCallback: function(objective, baseItem, sessionMapById, closeParent, template, modelLoadedFlag) {
                let _this = this;
                if (objective && objective.itemType && objective.itemType.title === "Objective") {
                    template.templateToRenderer = _.cloneDeep(kendisStore.getters.getObjectiveTemplate().scheme.fieldsTemplate);
                    _.each(template.templateToRenderer.configuredFields, configField => {
                        _.each(configField.subTemplate.configuredFields, field => {
                            if (field.title === "Type") {
                                field.options.splice(1, 1);
                            }
                        })
                    })
                    template.statuses = kendisStore.getters.getObjectiveTemplate().statuses;
                    _this.displayMode = "Objective";

                }else if (objective && objective.itemType && objective.itemType.title == "KR") {
                    template.templateToRenderer = _.cloneDeep(kendisStore.getters.getOkrTemplate().scheme.fieldsTemplate);
                    let isKrOfPiObj = false;
                    for (let i = 0; i < baseItem.baseItemLinks.length; i++) {
                        if (baseItem.baseItemLinks[i].linkType === "session_link_objective") {
                            isKrOfPiObj = true;
                        }
                    }
                    if (isKrOfPiObj) {
                        _.each(template.templateToRenderer.configuredFields, configField => {
                            _.each(configField.subTemplate.configuredFields, field => {
                                if (field.title === "Progress Criteria") {
                                    field.options.splice(2, 1);
                                }
                            })
                        })
                    }
                    template.statuses = kendisStore.getters.getOkrTemplate().statuses;
                    if (sessionMapById) {
                        template.sessionsMap = sessionMapById;
                    }
                    _this.displayMode = "KR";
                    // if (_this.objectiveData.fields && _this.objectiveData.fields[_this.krProgressCriteriaFieldId] && _this.objectiveData.fields[_this.krProgressCriteriaFieldId] === "By Objectives") {
                    //     if (baseItem && baseItem.baseItemList) {
                    //         _this.baseItemList = [];
                    //         _this.baseItemList = baseItem.baseItemList
                    //         if (sessionMapById) {
                    //             _this.childSessionsMap = sessionMapById;
                    //         }
                    //     }
                    //     _this.childItemsLoaded = true;
                    // }
                }

                template.id = baseItem.id;
                template.model.id = baseItem.id;
                template.model.title = baseItem.title;
                template.model.description = baseItem.description;
                template.model.bvPlan = baseItem.bvPlan;
                template.model.bvActual = baseItem.bvActual;
                template.model.key = baseItem.key;
                let board = {
                    session:{
                                    uncommittedObjEnabled: false,
                                    id: this.isGlobalObj? "-1":this.releaseTrain.id
                                },
                    teams:[],
                }
                template.board = board;
                if (_this.displayMode == "Objective") {
                    template.model.type = baseItem.type.split("-")[1];
                }
                let boardSession = [];
                boardSession.push({id: this.releaseTrainId, name: ''})
                template.boardSession = boardSession;
                template.model.selectedSessions = []
                template.model.selectedSessions.push(boardSession);
                if (baseItem.fields) {
                    template.model.fields = baseItem.fields;
                }
                if (baseItem.status) {
                    template.model.status = baseItem.status;
                }
                if (baseItem.baseItemList) {
                    template.model.baseItemList = baseItem.baseItemList;
                }
                if (baseItem.fieldsTemplate) {
                    template.templateToRenderer = baseItem.fieldsTemplate;
                }
                if (baseItem.baseItemLinks) {
                    template.model.baseItemLinks = baseItem.baseItemLinks;
                }
                if (baseItem.itemTypeList) {
                    template.itemTypeList = baseItem.itemTypeList;
                }
                template.model.linkedItemObject = {};
                if (baseItem.id) {
                    template.model.linkedItemObject.id = baseItem.id;
                }
                if (baseItem.linkedItemsIds) {
                    template.model.linkedItemObject.linkedItemsIds = baseItem.linkedItemsIds;
                }
                if (baseItem.linkedItems) {
                    template.model.linkedItemObject.linkedItems = baseItem.linkedItems;
                }
                if (baseItem.linkedItemsById) {
                    template.model.linkedItemObject.linkedItemsById = baseItem.linkedItemsById;
                }
                template.itemType = baseItem.itemType;

                //Added for Backlog Items Mixin code checks
                //Consult nasir for details
                template.model.itemType = baseItem.itemType;

                template.creationAndUpdateData = {
                    "createdOn": baseItem.createdOn,
                    "createdBy": baseItem.createdBy,
                    "updatedOn": baseItem.updatedOn,
                    "updatedBy": baseItem.updatedBy
                };
                template.model.linkedCardWeights = baseItem.linkedCardWeights;
                template.model.completionCriteria = baseItem.completionCriteria;
                if (baseItem.selectedLinkedGroups) {
                    template.model.selectedLinkedGroups = baseItem.selectedLinkedGroups;
                } else {
                    template.model.selectedLinkedGroups = [];
                }

                // let groupId = _this.groupId;
                if (closeParent) {
                    _this.$emit("close-parent-container");
                }
                //
                // if (_this.objOkrPopupContainerMap[groupId] === undefined) {
                //     _this.objOkrPopupContainerMap[groupId] = {};
                // }
                // _this.objOkrPopupContainerMap[groupId].isPopupOpen = true;
                // _this.objOkrPopupContainerMap[groupId].displayMode = _this.displayMode;
                // _this.objOkrPopupContainerMap[groupId].itemTemplate = _this.itemTemplate;
                // kendisStore.commit("setObjOkrPopupContainerMap", _this.objOkrPopupContainerMap);

                if (template === _this.itemTemplate.objTemplate) {
                    _this.objectiveModalDataLoaded = true;
                } else if (template === _this.itemTemplate.krTemplate) {
                    _this.krModalDataLoaded = true;
                }

                _this.$forceUpdate();
            },
            openObjectiveGroupEditPopup(objGroup) {
                // Check EDIT_OBJECTIVE_GROUP permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'EDIT_OBJECTIVE_GROUP', 'edit objective groups')) {
                    return;
                }

                // if(!this.isGlobalObj && objGroup && objGroup.linkedCustomView){
                //     showTopMessageTargeted("You cannot edit a Custom View Referenced Group", 'warning', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                //     return
                // }
                // if(this.isGlobalObj && this.getCollection(objGroup.id) && this.getCollection(objGroup.id).type){
                //     showTopMessageTargeted("You cannot edit a collection", 'warning', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                //     return
                // }
                this.itemTemplate.objGroupTemplate = {
                    id: objGroup.id,
                    model: {
                        id: objGroup.id,
                        title: objGroup.title,
                        key: objGroup.title,
                        status: objGroup.status,
                        description: objGroup.description,
                        headerColor: objGroup.headerColor,
                        isDefaultGroup: objGroup.isDefaultGroup,
                        itemType: objGroup.itemType,
                        fields: objGroup.fields || {}
                    },
                    statuses: [],
                    showEditPopUp: true
                };

                if (objGroup.itemType?.title === "group_objective") {
                    this.itemTemplate.objGroupTemplate.templateToRenderer = kendisStore.getters.getObjectiveGroupTemplate().scheme.fieldsTemplate;
                    this.itemTemplate.objGroupTemplate.statuses = kendisStore.getters.getObjectiveGroupTemplate().statuses;
                }
            },
            onCloseObjGrpPopup(context) {
                this.itemTemplate.objGroupTemplate={};
            },
            onSaveObjGrpPopup(objGrp) {
                this.itemTemplate.objGroupTemplate={};
                let cachedObjGrp = this.$options.cache.objectiveGroupsMap.get(objGrp.id);
                cachedObjGrp.title = objGrp.title;
                cachedObjGrp.fields = objGrp.fields;
                cachedObjGrp.description = objGrp.description;
                cachedObjGrp.headerColor = objGrp.headerColor;
                cachedObjGrp.status = objGrp.status;

                let resource = scheduler.resourceStore.getById(objGrp.id);
                resource.name = objGrp.title;
                resource.color = objGrp.headerColor;

                let headerEvents = scheduler.eventStore.query(event => {
                    return event.resourceId === resource.id && event.originalData.isHeader;
                })

                headerEvents.forEach(event => {
                    event.eventColor = objGrp.headerColor+50;
                    event.name = objGrp.title;
                });

                this.onUpdateObjectiveGroup(cachedObjGrp);
            },
            onUpdateObjectiveGroup: async function(objectiveGroup) {
                if (!objectiveGroup) {
                    return;
                }
                let requestBody = {};
                let _this = this;

                if (!objectiveGroup.metaMap){
                    let metaMap = {};
                    metaMap.syncId = syncId;
                    if(typeof vueInstance != "undefined" && vueInstance.activeBoard && vueInstance.activeBoard.session) {
                        metaMap.sessionId = vueInstance.activeBoard.session.id;
                    }
                    requestBody.metaMap = JSON.stringify(metaMap);
                } else {
                    requestBody.metaMap = objectiveGroup.metaMap;
                }

                requestBody.id = objectiveGroup.id;
                requestBody.title = objectiveGroup.title;
                requestBody.fields = objectiveGroup.fields;
                requestBody.description = objectiveGroup.description;
                requestBody.headerColor = objectiveGroup.headerColor;
                requestBody.status = objectiveGroup.status;
                requestBody.type = objectiveGroup.type;
                requestBody.sequence = objectiveGroup.sequence;

                axios.post('external-entitites/save-base-item/custom-objective-save', requestBody).then(response => {
                    if(response.status == 200) {
                        if(response.data.impediment && response.data.impediment !== undefined) {

                        }
                        showTopMessageTargeted("Objective Group Updated", 'success', 4000, this.isFullScreen ? '.okrsrdmp' : null);
                    }
                }).catch(error => {
                    console.error("Objective Update error", error);
                    showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            onDeleteObjGrpPopup: function (objGroupId) {
                let objGroup = this.$options.cache.objectiveGroupsMap.get(objGroupId);
                if(objGroup.isDefaultGroup) {
                    alertFromSweetAlertWrapper("You cannot delete the default group", [takeNoAction])
                } else {
                    this.confirmDeleteObjectiveGroup(objGroup.id);
                }
            },
                        confirmDeleteObjectiveGroup: function (objectiveGroupId) {
                // Check DELETE_OBJECTIVE_GROUP permission
                if (!ComponentPermissionManager.checkPermissionWithMessage(this, 'DELETE_OBJECTIVE_GROUP', 'delete objective groups')) {
                    return;
                }
                
                let _this = this;

                // Determine target container for fullscreen mode
                let targetContainer = null;
                if (_this.isFullScreen) {
                    targetContainer = document.querySelector('.okrsrdmp');
                }

                this.$refs['kendis-alert'].fire({
                    title: 'Confirm Delete',
                    text: 'You want to delete objective group?',
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#d33',
                    cancelButtonColor: '#6c757d',
                    confirmButtonText: 'Yes, delete it!',
                    cancelButtonText: 'Cancel',
                    target: targetContainer
                }).then((result) => {
                    if (result.isConfirmed) {
                        _this.deleteObjectiveGroup([objectiveGroupId]);
                    }
                });
            },
            deleteObjectiveGroup: function (objectiveGroupId) {
                let _this = this;
                let requestBody = {};
                let metaMap = {};
                metaMap = {
                    "syncId" : syncId,
                    "sessionId" : this.isGlobalObj? "-1":this.releaseTrain.id,
                }

                requestBody.objId = objectiveGroupId[0];
                requestBody.metaMap = JSON.stringify(metaMap);

                axios.post('/objective/delete-objective', requestBody).then(response => {
                    if (response && response.status == "200") {

                        //remove resource
                        let resource = scheduler.resourceStore.getById(objectiveGroupId[0]);
                        scheduler.resourceStore.remove(resource);

                        showTopMessageTargeted("Objective group deleted.", 'success', 4000, this.isFullScreen ? '.okrsrdmp' : null);

                    }
                }).catch(error => {
                    console.error("Objective delete error", error);
                    showTopMessageTargeted("Error deleting objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            onViewPresetSelectValue: function () {
                // localStorage.setItem(this.releaseTrain.id + "_ROADMAP_PRESET_"+this.roadmap.id, this.viewPreset.value);
                // // Retrieve the corresponding preset based on selected value or default to week if undefined
                // scheduler.viewPreset = getPresetFactory(scheduler,this.viewPreset.value)
                // zoomToCompleteTimeRange(scheduler,this.viewPreset.value); // Zoom to the complete time range


                scheduler.viewPreset = getPresetFactory(scheduler,'y')
                // zoomToCompleteTimeRange(scheduler,'y'); // Zoom to the complete time range
            },

            /**
             * Toggle fullscreen mode (enter/exit)
             */
            toggleFullScreen(){
                if (this.isFullScreen) {
                    this.exitFullScreen();
                } else {
                    this.enterFullScreen();
                }
            },

            /**
             * Enter fullscreen mode
             */
            enterFullScreen(){
                try{
                    // this.showZoomOptions=false;
                    this.isFullScreen = true;
                    let scheudlerObj = {}
                    if(!vueInstance){
                        scheudlerObj = ROADMAP_MODULE;
                    }else{
                        scheudlerObj = vueInstance.$options.ROADMAP_MODULE
                    }
                    // Use the same fullscreen implementation as rt-roadmap-collection-items.js
                    scheudlerObj.Fullscreen.request(document.getElementsByClassName('mcontainer')[0]);
                    scheudlerObj.Fullscreen.onFullscreenChange((data)=>{
                        try {
                            if (!data.currentTarget.fullscreen) {
                                roadmapVue.isFullScreen = false;
                                roadmapVue.showZoomOptions = true;
                            }
                        }catch (e) {
                            console.log(e);
                            showTopMessageTargeted("Error entering fullscreen", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                        }
                    });
                }
                catch(e){
                    console.log(e);
                }
            },

            /**
             * Exit fullscreen mode
             */
            exitFullScreen(){
                try{
                    this.isFullScreen = false;
                    this.showZoomOptions = true;
                    let scheudlerObj = {}
                    if(!vueInstance){
                        scheudlerObj = ROADMAP_MODULE;
                    }else{
                        scheudlerObj = vueInstance.$options.ROADMAP_MODULE
                    }
                    // Exit fullscreen
                    scheudlerObj.Fullscreen.exit();
                }catch(e) {
                    console.error("Error exiting fullscreen:", e);
                    showTopMessageTargeted("Error exiting fullscreen", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },

            /**
             * Reset time navigator to current year
             */
            resetTimeNavigatorToCurrentYear(){
                const currentYear = new Date().getFullYear();
                const currentPeriod = {
                    title: 'YEARLY',
                    selectedYear: currentYear,
                    monthly: false
                };

                // Update time navigator
                this.timeNavigator.selectedTimePeriod = currentPeriod;
                this.timeNavigator.showNavigator = false;

                // Save to storage
                this.saveTimeNavigatorToStorage();

                // Navigate to current year
                const startDate = new Date(currentYear, 0, 1);
                const endDate = new Date(currentYear, 11, 31);
                this.navigateToTimePeriod({ startDate, endDate, timePeriod: currentPeriod });

                // Show success message
                showTopMessageTargeted(
                    `🔄 Time navigator reset to ${currentYear}`,
                    'success',
                    2000,
                    this.isFullScreen ? '.okrsrdmp' : null
                );
            },
            async loadObjectivesOfGroups(objGroupId) {
                var _this = this;

                var objGroup = _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(objGroupId);
                if (objGroup) {
                    return objGroup;
                }

                // Show loader for the resource
                var resource = scheduler.resourceStore.getById(objGroupId);
                if (resource) {
                    resource.isLoading = true;
                }

                try {
                    var data = await _this.fetchGroupsData([objGroupId]);
                    console.log("Response Returned", data);
                    var objectives = data.objectiveGroups[0] && data.objectiveGroups[0].baseItemList;

                    if (!objectives) {
                        if (!_this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.has(objGroupId)) {
                            _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.set(objGroupId, []);
                        }
                    } else {
                        var newObjectiveIds = [];
                        objectives.forEach(function(objective) {
                            _this.$options.cache.objectivesMap.set(objective.id, objective);
                            newObjectiveIds.push(objective.id);
                        });

                        if (_this.isMultilevel) {
                            // New multilevel mode: Add objectives as child resources
                            var newObjectiveResources = [];
                            objectives.forEach(function(objective) {
                                // Create objective resource with prefixed ID to avoid duplicates
                                let objectiveResource = {
                                    id: `${objGroupId}-${objective.id}`,
                                    name: objective.title,
                                    type: 'Objective',
                                    leaf: true,
                                    color: resource.color,
                                    // Store original objective ID for reference
                                    originalObjectiveId: objective.id
                                }
                                newObjectiveResources.push(objectiveResource);
                            });
                            _this.appendChildrenToResource(objGroupId, newObjectiveResources);

                            // Load linked items for objectives in multilevel mode
                            // _this.loadLinkedItemsForObjective(objectives);
                        }

                        // Update cache first (needed for both modes)
                            if (!_this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.has(objGroupId)) {
                            _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.set(objGroupId, newObjectiveIds);
                            } else {
                            var arr = _this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(objGroupId);
                            Array.prototype.push.apply(arr, newObjectiveIds);
                        }

                            // Legacy mode: Create events for objectives under the group resource

                    }

                    var departments = data.departments;
                    if (departments) {
                        _.each(departments, function(dpt) {
							_this.departmentsMap[dpt.id] = {};
							_this.departmentsMap[dpt.id] = dpt;
						});
					}
                    var batches = data.batches;
                    if (batches) {
                        _.each(batches, function(batch) {
							_this.batchesMap[batch.id] = {};
							_this.batchesMap[batch.id] = batch;
						});
					}
                    var sessions = data.sessions;
                    if (sessions) {
                        _.each(sessions, function(session) {
							_this.sessionsMap[session.id] = {};
							_this.sessionsMap[session.id] = session;
                        });
					}

                    // Only update group resource events (not background linked items yet)
                    _this.updateObjectiveGroupResourceEvents(objGroupId);
                } catch (e) {
                    console.error("Error occurred while fetching objectives of group", e);
                    showTopMessageTargeted("Error loading objectives", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                } finally {
                            // Hide loader after loading is complete or if an error occurs
                                if (resource) {
                                    resource.isLoading = false;
                                }
                }
            },


            updateObjectiveGroupResourceEvents(objGroupId){
                //remove resource events of the objGroupId
                let resourceEvents = scheduler.eventStore.query((record) => !record.id.includes("full-span-event") && record.resourceId === objGroupId);
                scheduler.eventStore.remove(resourceEvents);
                let objectivesIds = this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(objGroupId);

                // Safety check: ensure objectivesIds is an array
                if (!objectivesIds || !Array.isArray(objectivesIds)) {
                    console.warn('No objectives found for group:', objGroupId);
                    return;
                }

                let{events, assignments,dependencies}= this.createEventsAssignmentsForObjectives(objectivesIds,objGroupId,this.isMultilevel);
                scheduler.eventStore.add(events);
                scheduler.assignmentStore.add(assignments);
                scheduler.dependencyStore.add(dependencies);

                // this.createKREventsFilters();
            },
            // createKREventsFilters(){
            //     let filters = [];
            //     for(let [objectiveId, krId] of this.cache.objectivesKRRelationsMap){
            //         let kr = this.cache.keyResultsMap.get(krId);
            //         let objective = this.cache.objectivesMap.get(objectiveId);
            //         let filter = {
            //             id: `f-${objective.id}`,
            //             fn: (event) => {
            //                 return event.type === 'keyResult' && event.id === kr.id;
            //             },
            //         }
            //         filters.push(filter);
            //     }
            //
            //
            // },
            async addNewObjective(obj, groupId) {
                try {
                    let newObjective = {
                        title: obj.title,
                        bvPlan: 0,
                        bvActual: 0,
                        metaMap: JSON.stringify({
                            sessionId: this.isGlobalObj? "-1":this.releaseTrain.id,
                            boardId: this.isGlobalObj? "-1":this.releaseTrain.id,
                            syncId: syncId,
                            groupId: groupId
                        }),
                        type: "Objective-Committed",
                        completionCriteria: "CHILDREN_LINKED_ITEM_STATUS",
                        fields: obj.fields || {}
                    };
                    newObjective.bvPlan = Math.max(0, Math.floor(Number(newObjective.bvPlan)));
                    newObjective.bvActual = Math.max(0, Math.floor(Number(newObjective.bvActual)));
                    let todoStatus = _.find(kendisStore.getters.getObjectiveTemplate().statuses, { category: "ToDo" });
                    if (todoStatus) {
                        newObjective.status = todoStatus;
                    }
                    let parentResource = scheduler.resourceStore.getById(groupId);
                    // Save the objective to backend
                    let response = await this.onSaveObjective(newObjective, groupId);
                    let createdObjective = response.objective;
                    this.$options.cache.objectivesMap.set(createdObjective.id, createdObjective);

                    // Update cache for both modes
                    if (!this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.has(groupId)) {
                        this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.set(groupId, [createdObjective.id]);
                    } else {
                        var arr = this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.get(groupId);
                        Array.prototype.push.apply(arr, [createdObjective.id]);
                    }

                    if (this.isMultilevel) {
                        // Multilevel mode: Create objective resource with prefixed ID to avoid duplicates
                        let objectiveResource = {
                            id: `${groupId}-${createdObjective.id}`,
                            name: createdObjective.title,
                            type: 'Objective',
                            leaf: true,
                            color: parentResource.color,
                            // Store original objective ID for reference
                            originalObjectiveId: createdObjective.id
                        }
                        this.appendChildrenToResource(groupId, [objectiveResource]);

                        // Create events and assignments for the new objective resource
                        let{events, assignments, dependencies} = this.createEventsAssignmentsForObjectives([createdObjective.id], groupId, true);
                        scheduler.eventStore.add(events);
                        scheduler.assignmentStore.add(assignments);
                    } else {
                        // Legacy mode: Only create objective event under the group resource
                        let{events, assignments, dependencies} = this.createEventsAssignmentsForObjectives([createdObjective.id], groupId, false);
                        scheduler.eventStore.add(events);
                        scheduler.assignmentStore.add(assignments);
                    }
                    await this.loadCompletionPercentages([response.objGroup.id], true);
                    return response;
                } catch (error) {
                    console.error("Error in addNewObjective:", error);
                    showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    throw error;
                }
            },

            async  onSaveObjective(requestBody, groupId) {
                try {
                    if (requestBody.id) {
                        return await this.onEditObjective(requestBody);
                    }

                    let objectiveGroup = _.cloneDeep(this.$options.cache.objectiveGroupsMap.get(groupId));

                    let baseItemLinks = [
                        { sessionId: this.isGlobalObj? "-1":this.releaseTrain.id, sessionBoardId: undefined, type: "Objective", linkType: "art_link_objective" },
                        { sessionId: this.isGlobalObj? "-1":this.releaseTrain.id, sessionBoardId: undefined, baseItemId: groupId, linkType: "parent_group_objective" }
                    ];

                    requestBody.baseItemLinks = baseItemLinks;
                    requestBody.starred = false;
                    requestBody.mentionUsers = [];
                    requestBody.itemType = kendisStore.getters.getObjectiveTemplate().scheme.itemType;

                    let response = await axios.post('external-entitites/save-base-item/popup-save', requestBody);
                    if (response.status === 200 && response.data) {
                        let objective = response.data.impediment;

                        let baseItemLink = {
                            baseItemId: objective.id,
                            sessionId: this.isGlobalObj? "-1":this.releaseTrain.id,
                            linkType: "child_group_objective",
                            sessionBoardId: undefined
                        };
                        objectiveGroup.baseItemLinks.push(baseItemLink);

                        let metaMap = JSON.stringify({
                            syncId: syncId,
                            sessionId: this.isGlobalObj? "-1":this.releaseTrain.id

                        });

                        objectiveGroup.metaMap = metaMap;

                        let saveResponse = await axios.post('external-entitites/save-base-item/popup-save', objectiveGroup);
                        return saveResponse.status === 200 ? {objective: objective,objGroup: saveResponse.data.impediment} : null;
                    }
                } catch (error) {
                    console.error("Error in onSaveObjective:", error);
                    showTopMessageTargeted("Error saving objective", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    throw error;
                }
            },


            createEventsAssignmentsForObjectives(objectivesIds,objGroupId,createKrs){
                let events = [];
                let assignments = [];
                let dependencies = [];
                let fieldsMap = kendisStore.getters.getObjectiveTemplate().fieldsMap;
                let fieldsMapKR = kendisStore.getters.getOkrTemplate().fieldsMap;
                let startDateFieldId, endDateFieldId, responsibleFieldId;
                let startDateFieldIdKr, endDateFieldIdKr, responsibleFieldIdKr;

                // Loop through the fieldsMap to find the IDs for 'Start Date' and 'End Date'
                for (let fieldId in fieldsMap) {
                    if (fieldsMap[fieldId] === 'Start Date') {
                        startDateFieldId = fieldId;
                    } else if (fieldsMap[fieldId] === 'End Date') {
                        endDateFieldId = fieldId;
                    } else if (fieldsMap[fieldId] === 'Responsible') {
                        responsibleFieldId = fieldId;
                    }
                }

                for (let fieldId in fieldsMapKR) {
                    if (fieldsMapKR[fieldId] === 'Start Date') {
                        startDateFieldIdKr = fieldId;
                    } else if (fieldsMapKR[fieldId] === 'End Date') {
                        endDateFieldIdKr = fieldId;
                    } else if (fieldsMapKR[fieldId] === 'Responsible') {
                        responsibleFieldIdKr = fieldId;
                    }
                }

                for(let objectiveId of objectivesIds){
                    let objective = this.$options.cache.objectivesMap.get(objectiveId);
                    let startDateFieldValue = objective.fields ? objective.fields[startDateFieldId]:undefined;
                    let endDateFieldValue = objective.fields? objective.fields[endDateFieldId]:undefined;

                    let startDate = startDateFieldValue?new Date(startDateFieldValue):new Date()
                    let endDate = endDateFieldValue?new Date(endDateFieldValue):new Date(new Date().setMonth(new Date().getMonth() + 1));

                    startDate.setHours(0, 0, 0, 0); // for complete time range cover
                    endDate.setHours(23, 59, 59, 999);
                    let completion = kendisStore.getters.getOkrCompletionMap()?kendisStore.getters.getOkrCompletionMap()[objective.id]:0;


                    //get color og obj group
                    let objGroup = this.$options.cache.objectiveGroupsMap.get(objGroupId);

                    let associationsMap = {
                        sessions : this.getSessions(objective),
                        collections: this.getDepartments(objective),
                        batches: this.getBatches(objective)
                    };

                    // Check if objective event already exists in scheduler
                    if (!scheduler.eventStore.getById(objective.id)) {
                        let event = {
                            id: objective.id,
                            key : objective.key,
                            name: objective.title,
                            startDate: startDate,
                            endDate: endDate,
                            type: 'objective',
                            // eventStyle: 'line', // Parent objectives display as lines
                            manuallyScheduled: true,
                            percentDone: kendisStore.getters.getOkrCompletionMap()?kendisStore.getters.getOkrCompletionMap()[objective.id]:0,
                            eventColor: objective.status.category === "ToDo" ? "#E79362" : objective.status.category === "InProgress" ? "#3792C9" : objective.status.category === "Done" ? "#3FCA93" : "#7c50b1",
                            responsible: objective.fields ? objective.fields[responsibleFieldId] : undefined,
                            status: objective.status,
                            associations: associationsMap,
                            cls : this.showKeyResults?'parent-objective-dashed':'',
                        }
                        events.push(event);
                    }
                    // Always add assignment for this resource (even if event exists)
                    let resourceId = this.isMultilevel ? `${objGroupId}-${objective.id}` : objGroupId;
                    assignments.push({id: `a-${objective.id}-${objGroupId}`, resourceId: resourceId, eventId: objective.id});

                    if(createKrs){
                        if(objective.baseItemList && objective.baseItemList.length > 0){
                            for(let kr of objective.baseItemList) {
                                if(kr.type === 'KR'){
                                this.$options.cache.keyResultsMap.set(kr.id, kr);
                                if (!this.$options.cache.objectivesKRRelationsMap.has(objective.id))
                                    this.$options.cache.objectivesKRRelationsMap.set(objective.id, [kr.id]);
                                else
                                    this.$options.cache.objectivesKRRelationsMap.get(objective.id).push(kr.id);

                                let startDateFieldValueKr = kr.fields ? kr.fields[startDateFieldIdKr]:undefined;
                                let endDateFieldValueKr = kr.fields? kr.fields[endDateFieldIdKr]:undefined;

                                let startDateKr = startDateFieldValueKr?new Date(startDateFieldValueKr):new Date()
                                let endDateKr = endDateFieldValueKr?new Date(endDateFieldValueKr):new Date(new Date().setMonth(new Date().getMonth() + 1));

                                startDateKr.setHours(1, 0, 1, 0); // for complete time range cover
                                endDateKr.setHours(23, 59, 59, 999);

                                // Check if KR event already exists in scheduler
                                if (!scheduler.eventStore.getById(kr.id)) {
                                    let eventKr = {
                                        id: kr.id,
                                        key : kr.key,
                                        name: kr.title,
                                        type : 'keyResult',
                                        startDate: startDateKr,
                                        endDate: endDateKr,
                                        eventColor: kr.status.category === "ToDo" ? "#E79362" : kr.status.category === "InProgress" ? "#3792C9" : kr.status.category === "Done" ? "#3FCA93" : "#7c50b1",
                                        manuallyScheduled: true,
                                        percentDone:kendisStore.getters.getOkrCompletionMap()?kendisStore.getters.getOkrCompletionMap()[kr.id]:0,
                                        status: kr.status,
                                        cls:'kendis-kr-roadmap-event'
                                    }
                                    events.push(eventKr);
                                }
                                // Always add assignment for this resource (even if event exists)
                                let krResourceId = this.isMultilevel ? `${objGroupId}-${objectiveId}` : objectiveId;
                                assignments.push({id: `a-${kr.id}-${objectiveId}-${objGroupId}`, resourceId: krResourceId, eventId: kr.id});

                                //add dependencies
                                // dependencies.push({
                                //     id: objective.id+'-'+kr.id +'-'+objGroupId,
                                //     from: objective.id,
                                //     to: kr.id,
                                //     readOnly:true,
                                // });
                            } else if(kr.type === 'Objective-Committed'){
                                this.$options.cache.objectivesMap.set(kr.id, kr);
                                if (!this.$options.cache.objectivesObjectiveRelationsMap.has(objective.id))
                                    this.$options.cache.objectivesObjectiveRelationsMap.set(objective.id, [kr.id]);
                                else
                                    this.$options.cache.objectivesObjectiveRelationsMap.get(objective.id).push(kr.id);

                                    //create objective event and resource here
                               let linkedObjective = kr;
                                let startDateFieldValue = linkedObjective.fields ? linkedObjective.fields[startDateFieldId]:undefined;
                                let endDateFieldValue = linkedObjective.fields? linkedObjective.fields[endDateFieldId]:undefined;

                                let startDate = startDateFieldValue?new Date(startDateFieldValue):new Date()
                                let endDate = endDateFieldValue?new Date(endDateFieldValue):new Date(new Date().setMonth(new Date().getMonth() + 1));

                                startDate.setHours(1, 0, 1, 0); // for complete time range cover
                                endDate.setHours(23, 59, 59, 999);
                                let completion = kendisStore.getters.getOkrCompletionMap()?kendisStore.getters.getOkrCompletionMap()[linkedObjective.id]:0;
                                let associationsMap = {
                                    sessions : this.getSessions(linkedObjective),
                                    collections: this.getDepartments(linkedObjective),
                                    batches: this.getBatches(linkedObjective)
                                };

                                // Check if child objective event already exists in scheduler
                                if (!scheduler.eventStore.getById(linkedObjective.id)) {
                                    let event = {
                                        id: linkedObjective.id,
                                        key : linkedObjective.key,
                                        name: linkedObjective.title,
                                        startDate: startDate,
                                        endDate: endDate,
                                        type: 'child-objective',
                                        manuallyScheduled: true,
                                        percentDone: completion?completion:0,
                                        eventColor: linkedObjective.status.category === "ToDo" ? "#E79362" : linkedObjective.status.category === "InProgress" ? "#3792C9" : linkedObjective.status.category === "Done" ? "#3FCA93" : "#7c50b1",
                                        responsible: linkedObjective.fields ? linkedObjective.fields[responsibleFieldId] : undefined,
                                        status: linkedObjective.status,
                                        associations: associationsMap,
                                        cls:'kendis-kr-roadmap-event'
                                    }
                                    events.push(event);
                                }
                                // Always add assignment for this resource (even if event exists)
                                let childResourceId = this.isMultilevel ? `${objGroupId}-${objective.id}` : objective.id;
                                assignments.push({id: `a-${linkedObjective.id}-${objGroupId}-${objective.id}`, resourceId: childResourceId, eventId: linkedObjective.id});


                            }
                        }
                    }
                }
            }


                return {events, assignments,dependencies};
            },

            /**
             * Handles scroll button click to expand objective groups
             * @param {Object} resourceRecord - The resource record to expand
             * @param {boolean} isBefore - Whether it's a "before" scroll button
             * @param {number} nbrEvents - Number of events in the scroll button
             */
            handleScrollButtonExpansion(resourceRecord, isBefore, nbrEvents) {
                // Don't expand milestones resource
                if (resourceRecord.id === "-1001") {
                    return Promise.resolve();
                }

                // Check if the resource is not already expanded
                if (!resourceRecord.expanded) {
                    // Show loading feedback
                                            showTopMessageTargeted("Expanding objective group...", 'info', 1500, this.isFullScreen ? '.okrsrdmp' : null);
                    
                    // Toggle expansion and load objectives
                    toggleResourceExpandByAssignments(resourceRecord, 'X', scheduler);
                    if (resourceRecord.expanded) {
                        return this.loadObjectivesOfGroups(resourceRecord.id).then(() => {
                            // Refresh the scheduler to show the expanded content
                            scheduler.refresh();

                            // Show success message
                            showTopMessageTargeted("✅ Objective group expanded successfully", 'success', 2000, this.isFullScreen ? '.okrsrdmp' : null);
                        }).catch((error) => {
                            console.error('Error loading objectives:', error);
                            showTopMessageTargeted("Failed to expand objective group", 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                            throw error;
                        });
                    } else {
                        showTopMessageTargeted("Could not expand objective group", 'warning', 2000, this.isFullScreen ? '.okrsrdmp' : null);
                        return Promise.reject(new Error('Expansion failed'));
                    }
                } else {
                    // If already expanded, show a brief message
                    showTopMessageTargeted("Objective group is already expanded", 'info', 1500, this.isFullScreen ? '.okrsrdmp' : null);
                    return Promise.resolve();
                }
            },

            /**
             * Ensures the timeline has a full range for infinite scrolling
             */
            ensureFullTimelineRange() {
                if (scheduler) {
                    // Set a very wide date range to support infinite scrolling
                    const wideStartDate = new Date(1900, 0, 1);
                    const wideEndDate = new Date(2100, 11, 31);

                    // Only update if the current range is smaller
                    if (scheduler.startDate > wideStartDate || scheduler.endDate < wideEndDate) {
                        scheduler.setTimeSpan(wideStartDate, wideEndDate);
                        console.log('Timeline range expanded for infinite scrolling');
                    }
                }
            },

            /**
             * Maps time period titles to preset keys for getPresetFactory
             * @param {string} timePeriodTitle - The time period title (YEARLY, H1, Q1, Jan, etc.)
             * @returns {string} The preset key for getPresetFactory
             */
            getPresetKeyForTimePeriod(timePeriodTitle) {
                const upperTitle = (timePeriodTitle || '').toUpperCase();

                // Map time period titles to preset keys
                const presetMapping = {
                    // Year view
                    'YEARLY': 'y',
                    'YEAR': 'y',
                    'CUSTOM': 'y',

                    // Half-year view
                    'H1': 'h',
                    'H2': 'h',

                    // Quarter view
                    'Q1': 'q',
                    'Q2': 'q',
                    'Q3': 'q',
                    'Q4': 'q',

                    // Month view for individual months
                    'JAN': 'm',
                    'FEB': 'm',
                    'MAR': 'm',
                    'APR': 'm',
                    'MAY': 'm',
                    'JUN': 'm',
                    'JUL': 'm',
                    'AUG': 'm',
                    'SEP': 'm',
                    'OCT': 'm',
                    'NOV': 'm',
                    'DEC': 'm',
                    'JANUARY': 'm',
                    'FEBRUARY': 'm',
                    'MARCH': 'm',
                    'APRIL': 'm',
                    'JUNE': 'm',
                    'JULY': 'm',
                    'AUGUST': 'm',
                    'SEPTEMBER': 'm',
                    'OCTOBER': 'm',
                    'NOVEMBER': 'm',
                    'DECEMBER': 'm',

                    // Month view for monthly selection
                    'MONTHLY': 'm'
                };

                // Return the mapped preset key or default to month view
                return presetMapping[upperTitle] || 'm';
            },

            /**
             * Applies preset-specific CSS classes to the scheduler for enhanced visual styling
             * @param {string} presetKey - The preset key ('y', 'h', 'q', 'm', 'w')
             */
            applyPresetStyling(presetKey) {
                if (!scheduler || !scheduler.element) return;

                const schedulerElement = scheduler.element;

                // Remove existing preset classes
                schedulerElement.classList.remove(
                    'preset-year',
                    'preset-half',
                    'preset-quarter',
                    'preset-month',
                    'preset-week'
                );

                // Map preset keys to CSS classes
                const presetClassMap = {
                    'y': 'preset-year',
                    'h': 'preset-half',
                    'q': 'preset-quarter',
                    'm': 'preset-month',
                    'w': 'preset-week'
                };

                // Apply the appropriate preset class
                const presetClass = presetClassMap[presetKey];
                if (presetClass) {
                    schedulerElement.classList.add(presetClass);
                    console.log('Applied preset styling:', presetClass);
                }
            },

            /**
             * Save only the time navigator to localStorage (separate from main filters)
             */
            saveTimeNavigatorToStorage() {
                const storageKey = this.getRoadmapStorageKey();
                const navigatorStorageKey = `${storageKey}_navigator`;
                const navigatorData = {
                    timeNavigator: this.timeNavigator,
                    timestamp: new Date().toISOString()
                };

                try {
                    localStorage.setItem(navigatorStorageKey, JSON.stringify(navigatorData));
                    console.log('Time navigator saved to localStorage:', navigatorStorageKey, navigatorData);
                } catch (error) {
                    console.error('Error saving time navigator to localStorage:', error);
                }
            },

            /**
             * Clear only the time navigator (separate from main filters)
             */
            clearTimeNavigator() {
                this.timeNavigator.selectedTimePeriod = null;
                this.timeNavigator.showNavigator = false;

                // Clear navigator from localStorage
                const storageKey = this.getRoadmapStorageKey();
                const navigatorStorageKey = `${storageKey}_navigator`;
                localStorage.removeItem(navigatorStorageKey);

                // Update URL
                this.updateURLWithFilters();

                console.log('Time navigator cleared');
            },

            async fetchGroupsData(groupIds) {
                let requestBody = {};
                requestBody.sessionId = this.isGlobalObj?'-1':this.releaseTrain.id
                requestBody.groupIds = groupIds;
                // this.addResponsibleFilterRequest(requestBody,"Objective")
                // this.addTimePeriodFilterRequest(requestBody);
                this.appendFiltersToRequestBody(requestBody);

                let response = await fetch ('objective/get-objectives-of-group', {
                    method: "POST",
                    headers: getFetchAPIHeadersForKendis(),
                    credentials: 'include',
                    body: JSON.stringify(requestBody)
                })

                if(response.status !== 200){
                    console.error("Error occurred while fetching roadmap data");
                    throw new Error("Error occurred while fetching roadmap data");
                }
                return await response.json();
            },
            drawGroupHeaderEvents(allResources){
                for(let resourceRecord of allResources) {
                    if(resourceRecord.id === milestoneResourceId){
                        continue;
                    }
                    let groupId = resourceRecord.id;

                    let cachedData = roadmapVue.$options.cache.objectiveGroupsMap.get(groupId)
                    let objectiveGroupName = ''
                    let objectiveGroupColor = '#ffffff';
                    if(!this.isGlobalObj && cachedData && cachedData.linkedCustomView  && this.linkedCustomViewsMap.has(cachedData.linkedCustomView)){
                        objectiveGroupName =` ${this.linkedCustomViewsMap.get(cachedData.linkedCustomView).key} ${this.linkedCustomViewsMap.get(cachedData.linkedCustomView).title} `
                        objectiveGroupColor = this.linkedCustomViewsMap.get(cachedData.linkedCustomView).backgroundColor;
                    }
                    else{
                        objectiveGroupName = resourceRecord.name
                        objectiveGroupColor = resourceRecord.color ? resourceRecord.color : '#ffffff';
                    }

                    const fullSpanEvent = scheduler.eventStore.getById(`full-span-event-${groupId}`);
                    if (!fullSpanEvent) {
                        // Add the full-length immovable event
                        scheduler.eventStore.add({
                            id: `full-span-event-${groupId}`,
                            name: ` ${objectiveGroupName}`,
                            startDate: new Date(1000, 0, 1),
                            endDate: new Date(9999, 0, 1),
                            cls: 'kendis-full-length-event',
                            readOnly: true,// This event is marked as read-only
                            isHeader: true,
                            eventColor: resourceRecord.color?resourceRecord.color:'#ffffff',
                            percentDoneGroup:0,
                        });
                        scheduler.assignmentStore.add({
                            id: `a-full-span-event-${groupId}`,
                            resourceId: groupId,
                            eventId: `full-span-event-${groupId}`
                        })
                    }
                }
            },
            getLinkedEntities(objective, linkType, map) {
                const ids = new Set();
                const result = [];

                objective.baseItemLinks?.forEach(link => {
                    if (link.linkType === linkType) {
                        ids.add(link.sessionId);
                    }
                });

                ids.forEach(id => {
                    if (map[id]) {
                        result.push(map[id]);
                    }
                });

                return result;
            },

            getSessions(objective) {
                return this.getLinkedEntities(objective, "linked_session", this.sessionsMap);
            },

            getBatches(objective) {
                return this.getLinkedEntities(objective, "art_link_objective", this.batchesMap);
            },

            getDepartments(objective) {
                return this.getLinkedEntities(objective, "art_link_objective", this.departmentsMap);
            },
            fitSchedulerToEvents(){
                fitSchedulerToEvents(scheduler);
                
                // // Switch to custom mode after fitting
                // if (this.timeNavigator.selectedTimePeriod) {
                //     const currentYear = this.timeNavigator.selectedTimePeriod.selectedYear;
                //     // Calculate the span based on the visible date range
                //     const visibleRange = scheduler.visibleDateRange;
                //     const startYear = visibleRange.startDate.getFullYear();
                //     const endYear = visibleRange.endDate.getFullYear();
                //     const spanYears = endYear - startYear + 1;
                    
                //     // Apply custom zoom with calculated span
                //     this.applyCustomZoom(currentYear, Math.max(2, Math.min(5, spanYears)));
                // }
            },
            setOptions(){
                this.$options.loadingStats={
                    totalItems : 0 ,
                    loadedItems : 0
                }
                this.$options.firstOpenUnplannedGrid = false;
                this.$options.itemsParentsMap = new Map();
                // this.$options.paginator = new Paginator([], 10, this.updateGridData);

                this.$options.cache={
                        objectiveGroupsMap:new Map(),
                        objectivesMap:new Map(),
                        keyResultsMap:new Map(),
                        ObjectiveGroupsAndObjectivesRelationMap:new Map(), //MAP THAT HAS relations of objective Groups and Objectives
                        objectivesKRRelationsMap: new Map(),
                        objectivesObjectiveRelationsMap: new Map()
                }
            },
            closeActionPopup:function(context) {
                try {
                    this.actionPopup.showPopup = false;
                    context.eventContext.finalize(false)
                }catch (e) {
                    console.log(e)
                    showTopMessageTargeted("Error closing action popup", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            onBackClick: function(){
                try {
                    if(this.store) {
                        this.store.setShowLeftMenuBar(true);
                        this.store.setShowBreadCrumbs(true);

                    }

                    this.$router.back();
                }catch(e) {
                    console.error("Issue occurred while navigating back", e);
                    showTopMessageTargeted("Error navigating back", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            showMainLoader(){
                this.loader.show = true;
            },
            hideMainLoader(){
                this.loader.show = false;
            },
            loadSchedulerConfigs() {
                let _this = this;
                _this.$options.unplannedGrid = {};
                const project = new bryntum.schedulerpro.ProjectModel({
                    resources: [],
                    events: [],
                });
                // this.$options.unplannedGridConfig=getOkrsRoadmapUnplannedGridConfig(_this,project);
                this.$options.schedulerConfig = getOkrsRoadmapSchedulerConfig(_this,project);
            },
            onSearchFieldChange: function ({ value }) {
                this.$options.unplannedGrid.features.search.search(value);
            },
            onSearchFieldClear: function () {
                this.$options.unplannedGrid.features.search.clear();
            },
            onClickUnplannedButton() {
                const unplannedGrid = document.querySelector('[data-ref="unplanned"]');

                // Check if the element is currently visible
                if (unplannedGrid.style.display === 'none' || unplannedGrid.style.display === '') {
                    // If it's hidden (or display is not set), show it
                    unplannedGrid.style.display = 'flex';
                    this.showUnplannedGrid = true;
                } else {
                    // If it's currently shown, hide it
                    unplannedGrid.style.display = 'none';
                    this.showUnplannedGrid = false;

                }
            },
            initView: async function() {
                if (_.isEmpty(this.releaseTrain)) return;
                await this.initObjectiveModalTemplate();
                await this.loadRoadmap();
                this.fetchCustomViews();
                await this.loadCompletionPercentages(Array.from(this.$options.cache.objectiveGroupsMap.keys()));
                this.updateCompletionPercentages();
                this.loadMilestones();
                zoomToCurrentDate(scheduler);
                this.onViewPresetSelectValue();
                this.navigateToTimeperiodNow();
            },
            fetchCustomViews() {
                let _this = this;

                let data = {};

                data.releaseTrainId = !this.isGlobalObj ? this.releaseTrain.id : null;
                data.archive = false;

                axios.post("/objective/cv/all", data).then(res => {
                    if (res.data.success) {
                        let views = [];
                        if (this.tabIndex == 0) {
                            views.push({id:"-1", title: "Default View", key:""});
                        }
                        views.push(...res.data.views);
                        kendisStore.commit('setCustomViews', views);
                        if(_this.isGlobalObj){
                            _this.activeCustomView = views.find(view => view.key === _this.roadmap.objectiveCustomView.key);
                        }
                        _this.fetchWorkspace();
                    }
                }).catch(error=>{
                    _this.loader.show = false;
                    console.log(error);
                    showTopMessageTargeted("Error loading custom views", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },

            async loadResponsibles(){
                try {
                    let responsibleFilterselectedValues = localStorage.getItem(this.releaseTrain.id + "_ROADMAP_PRESET_RESPONSIBLE_FILTER_"+this.roadmap.id);
                    if(!_.isEmpty(responsibleFilterselectedValues)){
                        try{
                            responsibleFilterselectedValues = JSON.parse(responsibleFilterselectedValues);
                        }
                        catch (e) {
                            console.log(e);
                            this.responsibleFilter.selectedValues = response.data.users;
                            return;
                        }

                        this.responsibleFilter.selectedValues = responsibleFilterselectedValues;
                    }else{
                        this.responsibleFilter.selectedValues = [];
                    }
                    // const response = await axios.get("/admin/all-users");
                    // if(response.status === 200){
                    //     let responsibleFilterselectedValues = localStorage.getItem(this.releaseTrain.id + "_ROADMAP_PRESET_RESPONSIBLE_FILTER_"+this.roadmap.id);
                    //     if(!_.isEmpty(responsibleFilterselectedValues)){
                    //         try{
                    //             responsibleFilterselectedValues = JSON.parse(responsibleFilterselectedValues);
                    //         }
                    //         catch (e) {
                    //             this.responsibleFilter.selectedValues = response.data.users;
                    //             return;
                    //         }
                    //
                    //         this.responsibleFilter.selectedValues = responsibleFilterselectedValues;
                    //     }else{
                    //         this.responsibleFilter.selectedValues = response.data.users;
                    //     }
                    // }
                } catch (error) {
                    console.error("Error loading responsibles:", error);
                    showTopMessageTargeted("Error loading responsibles", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            updateCompletionPercentages : async function(){
                //update all events completion percentage
                if(!scheduler.eventStore){
                    return;
                }
                let resourceEvents = scheduler.eventStore.query((record) => record.id.includes("full-span-event"));
                for(let event of resourceEvents){
                    let groupId = event.id.split("-")[3];
                    event.percentDoneGroup = kendisStore.getters.getOkrCompletionMap() && kendisStore.getters.getOkrCompletionMap()[groupId]? kendisStore.getters.getOkrCompletionMap()[groupId] : 0;
                }

                let events = scheduler.eventStore.query((record) => !record.id.includes("full-span-event"));
                for(let event of events){
                    event.percentDone = kendisStore.getters.getOkrCompletionMap() ? kendisStore.getters.getOkrCompletionMap()[event.id] : 0;
                }

            },
            initObjectiveModalTemplate: async function() {
                try {
                    let _this = this;
                    let sessionId = this.isGlobalObj? "-1":this.releaseTrain.id;
                    const response = await fetch(`objective/get-objective-modal-template/${sessionId}`, {
                        headers: getFetchAPIHeadersForKendis(),
                        credentials: 'include', // Include cookies for CSRF-TOKEN
                    });

                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }

                    const data = await response.json();

                    let templates = {};
                    let progressCriteriaGroupFields = [];

                    // Objective Group Template
                    templates.objectiveGroupTemplate = {
                        scheme: data.groupTemplate.scheme,
                        statuses: data.groupTemplate.statuses,
                        fieldsMap: [],
                        invertedFieldsMap: {}
                    };
                    if (templates.objectiveGroupTemplate.scheme.fieldsTemplate) {
                        _.each(templates.objectiveGroupTemplate.scheme.fieldsTemplate.configuredFields, field => {
                            templates.objectiveGroupTemplate.fieldsMap[field.id] = field.title;
                            templates.objectiveGroupTemplate.invertedFieldsMap[field.title] = field.id;
                        });
                    }

                    // Objective Template
                    templates.objectiveTemplate = {
                        scheme: data.objectiveTemplate.scheme,
                        statuses: data.objectiveTemplate.statuses,
                        fieldsMap: [],
                        invertedFieldsMap: {}
                    };
                    if (templates.objectiveTemplate.scheme.fieldsTemplate) {
                        let objCounter = 0;
                        _.each(templates.objectiveTemplate.scheme.fieldsTemplate.configuredFields, field => {
                            if (field.isGroup) {
                                field.subTemplate = {};
                                field.subTemplate.configuredFields = field.groupFields;
                                _.each(field.groupFields, _field => {
                                    _field.tabIndex = objCounter++;
                                    templates.objectiveTemplate.fieldsMap[_field.id] = _field.title;
                                    templates.objectiveTemplate.invertedFieldsMap[_field.title] = _field.id;
                                    if (_field.title === 'Time Period') {
                                        _this.timePeriodFieldId = _field.id;
                                    }
                                    if (_field.title === 'Start Date') {
                                        _this.startDateFieldId = _field.id;
                                    }
                                    if (_field.title === 'End Date') {
                                        _this.endDateFieldId = _field.id;
                                    }
                                    if (_field.title === 'Responsible') {
                                        _this.responsibleFieldId = _field.id;
                                    }
                                   
                                });
                            }
                        });
                    }

                    // OKR Template
                    templates.okrTemplate = {
                        scheme: data.okrTemplate.scheme,
                        statuses: data.okrTemplate.statuses,
                        fieldsMap: [],
                        invertedFieldsMap: {}
                    };
                    if (templates.okrTemplate.scheme.fieldsTemplate) {
                        let okrCounter = 0;
                        _.each(templates.okrTemplate.scheme.fieldsTemplate.configuredFields, field => {
                            if (field.isGroup) {
                                field.subTemplate = {};
                                field.subTemplate.configuredFields = field.groupFields;
                                _.each(field.groupFields, _field => {
                                    _field.tabIndex = okrCounter++;
                                    templates.okrTemplate.fieldsMap[_field.id] = _field.title;
                                    templates.okrTemplate.invertedFieldsMap[_field.title] = _field.id;
                                    if (_field.title === 'Time Period') {
                                        _this.timePeriodKRFieldId = _field.id;
                                    }
                                    if (_field.title === 'Start Date') {
                                        _this.startDateKRFieldId = _field.id;
                                    }
                                    if (_field.title === 'End Date') {
                                        _this.endDateKRFieldId = _field.id;
                                    }
                                    if (_field.title === 'Responsible') {
                                        _this.responsibleKRFieldId = _field.id;
                                    }
                                    if (_field.title === 'Progress Criteria') {
                                        _this.okrProgressCriteriaField = _field;
                                    }
                                });
                            }
                        });
                    }

                    let configuredFieldObj = templates.okrTemplate.scheme.fieldsTemplate.configuredFields.find(field =>
                        field.title === "Progress Criteria"
                    );

                    _.each(configuredFieldObj.groupFields, groupField => {
                        progressCriteriaGroupFields.push({
                            title: groupField.title,
                            id: groupField.id
                        });
                    });

                    kendisStore.commit("setOkrProgressCriteriaGroupFields", progressCriteriaGroupFields);
                    kendisStore.commit("setObjectiveAndOkrTemplate", templates);

                    // Initialize statusAndEstimateFilters for KR creation
                    _this.statusAndEstimateFilters = [
                        { code: "CHILDREN_LINKED_ITEM_STATUS", name: "Children Linked Item Status" }
                    ];

                    if (data.objectiveMenuOptions) {
                        let temp = data.objectiveMenuOptions;
                        let objectiveMenuOptions = {
                            isBVAchievementEnabled: temp.bvachievementEnabled,
                            isUnCommittedEnabled: temp.unCommittedEnabled,
                            isStatusEnabled: temp.statusEnabled,
                            isProgressEnabled: temp.progressEnabled,
                            isOkrEnabled: temp.okrEnabled,
                            isOkrBVAchievementEnabled: temp.okrBVAchievementEnabled,
                            isOkrStatusEnabled: temp.okrStatusEnabled
                        };
                        kendisStore.commit("setObjectiveMenuOptions", objectiveMenuOptions);
                    }

                } catch (error) {
                    console.error("Error initializing objective modal template:", error);
                    showTopMessageTargeted("Error initializing objective modal template", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },

            revertFrontEndChanges: function (eventContext) {
                eventContext.finalize(false)
            },
            calculateNewSequenceOfItem: function (newIndex, oldIndex, list, key) {
                var sequence;
                if (_.isEmpty(list)) {
                    sequence = 0;
                } else {
                    if (newIndex == 0) {
                        var nextItem = list[0].item ? list[0].item : list[0];
                        sequence = nextItem[key] - 1000;
                    } else if (newIndex >= list.length - 1) {
                        var preItem = list[list.length - 1].item ? list[list.length - 1].item : list[list.length - 1];
                        sequence = preItem[key] + 1000;
                    } else {
                        var preItem;
                        var nextItem;

                        if (oldIndex == -1 || oldIndex > newIndex) {
                            preItem = list[newIndex - 1].item ? list[newIndex - 1].item : list[newIndex - 1];
                            nextItem = list[newIndex].item ? list[newIndex].item : list[newIndex];
                        } else {
                            preItem = list[newIndex].item ? list[newIndex].item : list[newIndex];
                            nextItem = list[newIndex + 1].item ? list[newIndex + 1].item : list[newIndex + 1];
                        }
                        sequence = (preItem[key] + nextItem[key]) / 2.0;
                    }
                }
                return {value: sequence, key: key};
            },
            doUpdateSequenceInDB: function (obj) {
                let _this = this;
                let result = _this.calculateNewSequenceOfItem(obj.evt.newIndex, obj.evt.oldIndex, obj.list, 'sequence');
                if (result) {
                    obj.objective.sequence = result.value;
                    _this.onEditObjective(obj.objective, true);
                    let requestBody = {
                        objectiveId: obj.objective.id,
                        fromId: obj.fromGroup.id,
                        toId: obj.toGroup.id
                    };
                    let metaMap = {
                        "syncId": syncId,
                        "sessionId": this.isGlobalObj? "-1":this.releaseTrain.id
                    };

                    requestBody.metaMap = JSON.stringify(metaMap);

                    axios.post("objective/update-objective-groups", requestBody).then(response=> {
                        if(response.status == "200") {
                            obj.eventContext.finalize(true);

                        }
                    }).catch(error => {
                        console.error("Error in doUpdateSequenceInDB() function ", error);
                        showTopMessageTargeted("Error updating objective sequence", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                    });
                }
            },
            initializeBryntumModules() {
                if (!bryntum.schedulerpro.Scheduler) throw new Error('Bryntum SchedulerPro is required for this component');
                const schedule = new bryntum.schedulerpro.SchedulerPro(this.$options.schedulerConfig);
                schedule.on('beforeCellEditStart', ({ editorContext }) => {
                    if (this.groupBy.value !== groupByValues.Section ||
                        editorContext.record.id === milestoneResourceId ||
                        editorContext.record.id === "null") {
                        return false;
                    }
                });
                schedule.eventStore.reapplyFilterOnAdd = true;
                schedule.eventStore.reapplyFilterOnUpdate = true;
                scheduler = schedule;
                this.$options.roadmap = scheduler;

                // Ensure full timeline range for infinite scrolling
                this.ensureFullTimelineRange();
                // new bryntum.schedulerpro.Splitter({ appendTo: 'okrs-roadmap-bryntum' });
                // const unplannedGrid = new UnplannedGrid(this.$options.unplannedGridConfig);
                // this.$options.unplannedGrid = unplannedGrid;
                // new DragHelperItemsRoadmap({
                //     grid: unplannedGrid,
                //     schedule,
                //     constrain: false,
                //     outerElement: unplannedGrid.element
                // });
                // schedule.features.drag = { enabled: true };
                // grid = unplannedGrid;
            },
            fetchWorkspace : function () {
                kendisStore.commit('setObjectiveWorkspace',{});
                var _this = this;
                _this.roleErrorMessage = "";
                axios.get("/releasetrain/get/workspace-module/" + this.type)
                  .then(response => {
                      if (response.data) {
                          if(response.data.workspace) {
                              kendisStore.commit('setObjectiveWorkspace',response.data.workspace);
                              if (!_this.isGlobalObj) {
                                  let workspace = {};
                                  workspace = {id: _this.releaseTrain.id}
                                  _this.setObjectivePermissions(_.isEmpty(workspace) ? {id: _this.releaseTrain.id} : workspace);
                              } else {
                                  _this.setObjectivePermissions(_this.activeCustomView);
                              }
                          }
                          _this.setupPermissions();
                      }
                  }).catch(error => {
                    console.log(error);
                    showTopMessageTargeted("Error loading workspace", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            setObjectivePermissions: function(wS) {
                var objectiveRoles = _.filter(loggedInUser.accessRoles, function(o) {
                    return (o.workSpace && o.workSpace.id && o.workSpace.id == wS.id) || (o.customView && o.customView.id && o.customView.id === wS.id);
                });
                var roles = [];
                var permissions = [];
                if(loggedInUser.isAdmin === true && _.isEmpty(objectiveRoles)) {
                    permissions = loggedInUser.permissions;
                }
                _.each(objectiveRoles, accessRole => {
                    if (loggedInUser.isGuest === true) {
                        permissions = [];
                    } else if (loggedInUser.isAdmin === true) {
                        permissions = loggedInUser.permissions;
                    } else {
                        _.each(accessRole.roles, role => {
                            roles.push(role);
                            permissions = _.union(permissions, _.difference(loggedInUser.permissions, role.restrictions))
                        });
                    }
                });

                if (!_.isEmpty(this.activeCustomView) && this.activeCustomView.id === "-1") {
                    this.userPermissions = loggedInUser.permissions;
                } else {
                    this.userPermissions = permissions;
                }
            },
            isActionAllowed: function(action) {
                var permissions = this.userPermissions;
                if ((permissions && permissions.length > 0 && permissions.indexOf(action) > -1)) {
                    return true;
                } else {
                    return false;
                }
            },
            setupPermissions: function () {
                let EDIT_OBJECTIVE;
                let EDIT_OBJECTIVE_GROUP;
                let DELETE_OBJECTIVE_GROUP;
                let LINK_UNLINK_OBJECTIVE_ITEMS;
                let DELETE_OBJECTIVE;
                let EDIT_OKR;
                let LINK_UNLINK_OKR_ITEMS;
                let DELETE_OKR;
                let VIEW_OBJECTIVE_SETTINGS;

                    VIEW_OBJECTIVE_SETTINGS 	= this.isActionAllowed("global-objective-settings");
                    EDIT_OBJECTIVE_GROUP		= this.isActionAllowed("global-edit-objective-group");
                    DELETE_OBJECTIVE_GROUP		= this.isActionAllowed("global-delete-objective-group");
                    EDIT_OBJECTIVE 				= this.isActionAllowed("global-edit-objective");
                    LINK_UNLINK_OBJECTIVE_ITEMS = this.isActionAllowed("global-link-unlink-objective");
                    DELETE_OBJECTIVE		    = this.isActionAllowed("global-delete-objective");
                    EDIT_OKR 					= this.isActionAllowed("global-edit-okr");
                    LINK_UNLINK_OKR_ITEMS		= this.isActionAllowed("global-link-unlink-okr");
                    DELETE_OKR 					= this.isActionAllowed("global-delete-okr");



                let permissions = {
                    VIEW_OBJECTIVE_SETTINGS 	: VIEW_OBJECTIVE_SETTINGS,
                    EDIT_OBJECTIVE_GROUP		: EDIT_OBJECTIVE_GROUP,
                    DELETE_OBJECTIVE_GROUP		: DELETE_OBJECTIVE_GROUP,
                    EDIT_OBJECTIVE 				: EDIT_OBJECTIVE,
                    LINK_UNLINK_OBJECTIVE_ITEMS : LINK_UNLINK_OBJECTIVE_ITEMS,
                    DELETE_OBJECTIVE 		    : DELETE_OBJECTIVE,
                    EDIT_OKR 					: EDIT_OKR,
                    LINK_UNLINK_OKR_ITEMS		: LINK_UNLINK_OKR_ITEMS,
                    DELETE_OKR					: DELETE_OKR
                };
                this.$options.permissions = permissions;
                this.permissions = permissions; // <-- Add this line for reactivity
                kendisStore.commit('setObjectiveAndOkrPermissions', permissions);
            },
            async loadRoadmap() {
                try {
                    if(this.store) {
                        this.store.setShowLeftMenuBar(false);
                        this.store.setShowBreadCrumbs(false);

                    }

                    let _this = this;
                    // this.store.setShowLeftMenuBar(false);

                    let roadmap = await fetch('/releasetrain/kanban/' + this.roadmapId, {
                        method: "GET",
                        headers: getFetchAPIHeadersForKendis(),
                        credentials: 'include', // Include cookies for CSRF-TOKEN
                    })

                    if (roadmap.status !== 200) {
                        console.error("Error occurred while fetching roadmap data");
                        return;
                    }
                    let roadmapData = await roadmap.json();
                    _this.roadmap = roadmapData.board;
                    _this.initializeBryntumModules();
                    console.log("Roadmap Data", roadmapData);
                    this.loadState();

                    // Load filters BEFORE loading data to ensure they're applied to the initial fetch
                    await this.loadFiltersOnStartup();

                    await this.loadResponsibles();
                    await this.createObjectiveGroups();

                    if(!this.isGlobalObj){
                        _this.createBreadCrums();
                    }
                }catch(e) {
                    console.error("Error occurred while fetching roadmap data", e);
                    showTopMessageTargeted("Error loading roadmap", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },

            loadState(){
                console.log("loading state")
                try {

                    let viewPreset = localStorage.getItem(this.releaseTrain.id + "_ROADMAP_PRESET_" + this.roadmap.id);
                    if (_.isEmpty(viewPreset)) {
                        viewPreset = "y";
                    }
                    this.viewPreset.value = viewPreset;

                    let timePeriodFilter = localStorage.getItem(this.releaseTrain.id + "_ROADMAP_PRESET_TIME_PERIOD_FILTER_" + this.roadmap.id);
                    if (!_.isEmpty(timePeriodFilter)) {
                        try {
                            timePeriodFilter = JSON.parse(timePeriodFilter);
                        } catch (error) {
                            console.log(error);
                        }
                        if (!_.isEmpty(timePeriodFilter)) {
                            timePeriodFilter.showFilter = false;
                            this.timePeriodFilter = timePeriodFilter;
                        }
                    }
                }catch (e) {
                    console.log(e,"Error occurred while loading state");
                    showTopMessageTargeted("Error loading state", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            async loadCompletionPercentages(groupIds,updateEvents=false) {
                let _this = this;
                let requestBody = {
                    sessionId: this.isGlobalObj? "-1":this.releaseTrain.id,
                    groupIds: groupIds
                };

                try {
                    let response = await axios.post('objective/calculate-progress-for-select-groups', requestBody);

                    if (response?.data?.completionById) {
                        let completionsMap = response.data.completionById;
                        kendisStore.commit("setOkrCompletionMap", completionsMap);
                        if(updateEvents)
                            await _this.updateCompletionPercentages();

                    }

                } catch (error) {
                    console.error("Error loading completion percentages:", error);
                    showTopMessageTargeted("Error loading completion percentages", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                }
            },
            getResponsibleFieldIdFromBaseItem(type){
                let invertedFieldsMap;
                if(type==="Objective"){
                    invertedFieldsMap = kendisStore.getters.getObjectiveTemplate().invertedFieldsMap;
                }else if(type==="ObjectiveGroup"){
                    invertedFieldsMap = kendisStore.getters.getObjectiveGroupTemplate().invertedFieldsMap;
                }else if(type==="KeyResult"){
                    let fieldsMapKR = kendisStore.getters.getOkrTemplate().fieldsMap;
                    for (let fieldId in fieldsMapKR) {
                        if (fieldsMapKR[fieldId] === 'Responsible') {
                            return fieldId;
                }
                    }
                }

                return invertedFieldsMap["Responsible"];
            },
            onClickCollectionObjectiveIcon(el) {
                const key = el.dataset.key;
                const type = el.dataset.type;
                const url = `/org#/${type}/${key}/objectives`;
                window.open(url, '_blank');
            },
            addResponsibleFilterRequest(requestBody,baseItemType){
                let fieldsMap
                if(baseItemType==="Objective"){
                    fieldsMap = kendisStore.getters.getObjectiveTemplate().fieldsMap;
                }else if(baseItemType==="ObjectiveGroup"){
                     fieldsMap = kendisStore.getters.getObjectiveGroupTemplate().fieldsMap;
                }
                for (let fieldId in fieldsMap) {
                    if (fieldsMap[fieldId] === 'Responsible') {
                        requestBody.responsibleFieldId = fieldId;
                    }
                }
                requestBody.responsibleIdsToFilter = this.responsibleFilter.selectedValues.map(responsible => responsible.id);
            },
            addTimePeriodFilterRequest(requestBody) {

                let filterCopy = _.cloneDeep(this.timePeriodFilter);
                delete filterCopy.showFilter;
                delete filterCopy.months;

                let filter = filterCopy;
                if (!_.isEmpty(filter.selectedValues)) {
                    requestBody.timeperiods = filter.selectedValues;
                    requestBody.byKrs = filter.byKrs;
                    requestBody.byObjectives = filter.byObjectives;
                }
                else {
                    requestBody.timeperiods = {};
                }

            },

            async getCustomViewIdByKey(key){
                const response = await fetch(`objective/get-custom-view-id-by-key/${key}`, {
                    headers: getFetchAPIHeadersForKendis(),
                    credentials: 'include', // Include cookies for CSRF-TOKEN
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.data
            },

            async loadResources(){
                let _this = this;
                let requestBody = {};
                let requestId = getNewUUID();
                if(this.isGlobalObj){
                    requestBody.sessionId = "-1"
                    if(this.viewId) {
                        requestBody.viewId = this.viewId
                    }else{
                        requestBody.viewId = "-1"
                    }
                }else{
                    requestBody.sessionId = this.releaseTrain.id
                }
                requestBody.isMockingMode = true;
                requestBody.requestId = requestId;
                // Enable preliminary counts for roadmap loading
                requestBody.includePreliminaryCounts = true;
                // this.addResponsibleFilterRequest(requestBody,"ObjectiveGroup");
                // this.addTimePeriodFilterRequest(requestBody);

                this.appendFiltersToRequestBody(requestBody);

                // Use the existing endpoint which now includes preliminary counts
                let response = await fetch(`objective/get-objective-groups-list-by-object-id`, {
                    method: "POST",
                    headers: getFetchAPIHeadersForKendis(),
                    credentials: 'include', // Include cookies for CSRF-TOKEN
                    body: JSON.stringify(requestBody)
                })
                if(response.status !== 200){
                    console.error("Error occurred while fetching roadmap data");
                    return;
                }
                let data = await response.json();
                console.log("Response Returned with preliminary counts", data.objectiveGroups);
                
                // Store preliminary counts for future reference
                this.$options.preliminaryCounts = data.preliminaryCounts || {};
                
                if(!this.isGlobalObj) {
                    await this.fetchObjGroupsMeta(data.objectiveGroups);
                }
                _this.cacheResources(data.objectiveGroups);
                _this.injectResources(data.objectiveGroups);
            },
            appendFiltersToRequestBody(requestBody){
                requestBody.timePeriodFieldId = this.timePeriodFieldId;
                requestBody.timePeriodKRFieldId = this.timePeriodKRFieldId;
                let filter = this.timePeriodFilter.filter;
                if (!_.isEmpty(filter.selectedValues)) {
                    requestBody.timeperiods = filter.selectedValues;
                    requestBody.byKrs = filter.byKrs;
                    requestBody.byObjectives = filter.byObjectives;
                }
                else {
                    requestBody.timeperiods = {};
                }

                if (!_.isEmpty(this.objectiveFilters.responsible)) {
                    requestBody.responsibles = this.objectiveFilters.responsible.map(res => res.id);
                    requestBody.responsibleFieldId = this.responsibleFieldId;
                    requestBody.krResponsibleFieldId = this.responsibleKRFieldId;
                }

                if (!_.isEmpty(this.objectiveFilters.statuses)) {
                    requestBody.statuses = this.objectiveFilters.statuses.map(status => status.id);
                }

                if (!_.isEmpty(this.objectiveFilters.collections)) {
                    requestBody.collections = this.objectiveFilters.collections.map(collection => collection.id);
                }

                if (!_.isEmpty(this.objectiveFilters.search)) {
                    requestBody.searchQuery = this.objectiveFilters.search;
                }

                // Also check the component-level searchQuery
                if (!_.isEmpty(this.searchQuery)) {
                    requestBody.searchQuery = this.searchQuery;
                }

            },
            async fetchObjGroupsMeta(ObjGroupsList){
                let linkedCustomViewsIds = ObjGroupsList.filter(obj => obj.linkedCustomView !== undefined).map(obj => obj.linkedCustomView);
                this.linkedCustomViewsMap = await this.fetchLinkedCustomViews(linkedCustomViewsIds);
            },
            async fetchLinkedCustomViews(linkedCustomViewsIds) {
                let response = await fetch(`objective/cv/multiple`, {
                    method: 'POST',
                    headers: getFetchAPIHeadersForKendis(),
                    credentials: 'include', // Include cookies for CSRF-TOKEN
                    body: JSON.stringify(linkedCustomViewsIds)
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const data = await response.json();
                let linkedCustomViewsList = data.views;
                if(!linkedCustomViewsList) return new Map();
                return new Map(linkedCustomViewsList.map(obj => [obj.id, obj]));
            },
            cacheResources(rawResources){
                let _this = this;
                for(let resource of rawResources){
                    // Cache resource data - preliminary counts will be used until group is expanded
                    _this.$options.cache.objectiveGroupsMap.set(resource.id,resource);
                }
                console.log("Caching Resources", _this.$options.cache.objectiveGroupsMap);
            },
            injectResources(rawResources){
                let _this = this;
                let trimmedResources = [];
                for(let resource of rawResources){
                    trimmedResources.push(_this.resourceMapper(resource));
                }
                //sort by sequence
                trimmedResources.sort((a,b) => a.sequence - b.sequence);
                console.log("Injecting Resources", trimmedResources);
                scheduler.resourceStore.data = trimmedResources;
            },
            resetScheduler(){
                scheduler.assignmentStore.removeAll();
                scheduler.eventStore.removeAll();
                scheduler.resourceStore.removeAll();
                //remove all caches
                this.$options.cache.objectiveGroupsMap.clear();
                this.$options.cache.objectivesMap.clear();
                this.$options.cache.keyResultsMap.clear();
                this.$options.cache.ObjectiveGroupsAndObjectivesRelationMap.clear();
                this.$options.cache.objectivesKRRelationsMap.clear();
                this.$options.cache.objectivesObjectiveRelationsMap.clear();
            },
           async createObjectiveGroups(){
                this.resetScheduler();
                await this.loadResources()
                this.drawGroupHeaderEvents(scheduler.resources);
            },
            resourceMapper(resource){
                let cachedData = roadmapVue.$options.cache.objectiveGroupsMap.get(resource.id);
                let objectiveGroupColor = '#ffffff';
                if(!this.isGlobalObj && cachedData && cachedData.linkedCustomView  && this.linkedCustomViewsMap.has(cachedData.linkedCustomView)){
                    objectiveGroupColor = this.linkedCustomViewsMap.get(cachedData.linkedCustomView).color ? this.linkedCustomViewsMap.get(cachedData.linkedCustomView).color : '#ffffff';
                }
                else{
                    objectiveGroupColor = resource.headerColor ? resource.headerColor : '#ffffff';
                }
                let resourceObj = {
                    id: resource.id,
                    name: resource.title,
                    sequence: resource.sequence,
                    color: objectiveGroupColor+50,
                    percentDone: 0,
                    isLoading :false,
                    children: [],
                    expanded: false,
                }
                return resourceObj;
            },
            createBreadCrums(){
                let _this = this;
                let displayName = _this.roadmapDisplayName;
                let breadCrumbItems =
                    [
                        {name: viewTypeToListName(_this.releaseTrain), url: "/" + _this.viewType},
                        {
                            name: _this.store.releaseTrain.title,
                            url: "/" + _this.viewType + "/" + _this.releaseTrainId + "/" + viewTypeModule(this.releaseTrain)
                        },
                        {name: 'Objectives', url: "/" + this.viewType + "/" + _this.releaseTrainId + "/objectives"},
                        {
                            name: displayName,
                            url: "/" + this.viewType + "/" + _this.releaseTrainId + "/roadmap/" + _this.roadmap.id
                        }
                    ];
                _this.store.setBreadCrumbItems(breadCrumbItems);
            },
            filterEventsByName(searchTerm) {
                if (!scheduler || !scheduler.eventStore) return;

                const searchString = searchTerm.toLowerCase().trim();

                if (!searchString) {
                    // If search is empty, clear filters
                    scheduler.eventStore.removeFilter('nameFilter');
                    return;
                }

                // Apply filter
                scheduler.eventStore.filter({
                    id: 'nameFilter',
                    filterBy: event => {
                        // Always show header events
                        if (event.get('isHeader') || event.resourceId ===milestoneResourceId) return true;


                        // Filter non-header events by name
                        const name = event.name || '';
                        return name.toLowerCase().includes(searchString);
                    }
                });
            },
            clearSearch() {
                this.searchQuery = "";
                this.filterEventsByName("");
            },

            /**
             * Get the localStorage key for this roadmap's filters
             * @returns {string} The storage key
             */
            getRoadmapStorageKey() {
                return `okr-roadmap-${this.roadmap.id}`;
            },

            /**
             * Save current filters to localStorage
             */
            saveFiltersToStorage() {
                const storageKey = this.getRoadmapStorageKey();
                const filtersData = {
                    objectiveFilters: this.objectiveFilters,
                    timePeriodFilter: this.timePeriodFilter,
                    searchQuery: this.searchQuery,
                    timestamp: new Date().toISOString()
                };

                // Save time navigator separately to preserve it when filters are reset
                const navigatorStorageKey = `${storageKey}_navigator`;
                const navigatorData = {
                    timeNavigator: this.timeNavigator,
                    timestamp: new Date().toISOString()
                };

                try {
                    localStorage.setItem(storageKey, JSON.stringify(filtersData));
                    localStorage.setItem(navigatorStorageKey, JSON.stringify(navigatorData));
                    console.log('Filters and navigator saved to localStorage:', storageKey, filtersData);
                } catch (error) {
                    console.error('Error saving filters to localStorage:', error);
                }
            },

            /**
             * Load filters from localStorage
             */
            loadFiltersFromStorage() {
                const storageKey = this.getRoadmapStorageKey();
                const navigatorStorageKey = `${storageKey}_navigator`;

                try {
                    const storedData = localStorage.getItem(storageKey);
                    const navigatorData = localStorage.getItem(navigatorStorageKey);

                    if (storedData) {
                        const filtersData = JSON.parse(storedData);

                        // Load filters with validation
                        if (filtersData.objectiveFilters) {
                            this.objectiveFilters = {
                                ...this.objectiveFilters,
                                ...filtersData.objectiveFilters
                            };
                        }

                        if (filtersData.timePeriodFilter) {
                            this.timePeriodFilter = {
                                ...this.timePeriodFilter,
                                ...filtersData.timePeriodFilter
                            };
                        }

                        if (filtersData.searchQuery) {
                            this.searchQuery = filtersData.searchQuery;
                        }

                        console.log('Filters loaded from localStorage:', storageKey, filtersData);
                    }

                    // Load time navigator separately
                    if (navigatorData) {
                        const navigatorStoredData = JSON.parse(navigatorData);
                        if (navigatorStoredData.timeNavigator) {
                            this.timeNavigator = {
                                ...this.timeNavigator,
                                ...navigatorStoredData.timeNavigator
                            };
                            this.timeNavigator.showNavigator = false;
                            console.log('Time navigator loaded from localStorage:', navigatorStorageKey, navigatorStoredData);
                        }
                    }

                    // Apply loaded filters
                    this.applyLoadedFilters();

                    return true;

                } catch (error) {
                    console.error('Error loading filters from localStorage:', error);
                }

                return false;
            },

            /**
             * Apply loaded filters to the UI and data
             */
            applyLoadedFilters() {
                // Apply search filter
                if (this.searchQuery) {
                    this.filterEventsByName(this.searchQuery);
                }

                // Apply time period filter
                if (this.timePeriodFilter.filter && Object.keys(this.timePeriodFilter.filter).length > 0) {
                    this.onTimePeriodFilter(this.timePeriodFilter.filter);
                }

                // Apply time navigator if available
                if (this.timeNavigator && this.timeNavigator.selectedTimePeriod) {
                    // Navigate to the selected time period
                    const period = this.timeNavigator.selectedTimePeriod;
                    if (period.title && period.selectedYear) {
                        // Calculate start and end dates based on the time period
                        let startDate, endDate;
                        const year = period.selectedYear;

                        if (period.title === 'YEARLY') {
                            startDate = new Date(year, 0, 1);
                            endDate = new Date(year, 11, 31);
                        } else if (period.title.startsWith('Q')) {
                            const quarter = parseInt(period.title.substring(1));
                            const startMonth = (quarter - 1) * 3;
                            startDate = new Date(year, startMonth, 1);
                            endDate = new Date(year, startMonth + 2, new Date(year, startMonth + 3, 0).getDate());
                        } else if (period.title === 'JAN' || period.title === 'FEB' || period.title === 'MAR' ||
                                   period.title === 'APR' || period.title === 'MAY' || period.title === 'JUN' ||
                                   period.title === 'JUL' || period.title === 'AUG' || period.title === 'SEP' ||
                                   period.title === 'OCT' || period.title === 'NOV' || period.title === 'DEC') {
                            const monthMap = {
                                'JAN': 0, 'FEB': 1, 'MAR': 2, 'APR': 3, 'MAY': 4, 'JUN': 5,
                                'JUL': 6, 'AUG': 7, 'SEP': 8, 'OCT': 9, 'NOV': 10, 'DEC': 11
                            };
                            const month = monthMap[period.title];
                            startDate = new Date(year, month, 1);
                            endDate = new Date(year, month, new Date(year, month + 1, 0).getDate());
                        }

                        if (startDate && endDate) {
                            this.navigateToTimePeriod({ startDate, endDate, timePeriod: period });
                        }
                    }
                }

                // Apply other filters
                if (this.objectiveFilters.responsible.length > 0) {
                    this.onFilterResponsible(this.objectiveFilters.responsible);
                }

                if (this.objectiveFilters.collections.length > 0) {
                    this.onFilterAssociation(this.objectiveFilters.collections);
                }

                if (this.objectiveFilters.statuses.length > 0) {
                    this.onFilterStatus(this.objectiveFilters.statuses);
                }
            },

            /**
             * Update URL parameters for sharing filters
             */
            updateURLWithFilters() {
                if (!this.roadmap || !this.roadmap.id) return;

                const url = new URL(window.location.href);
                const params = url.searchParams;

                // Clear existing filter params
                params.delete('filters');
                params.delete('search');
                params.delete('timePeriod');
                params.delete('navigator');

                // Create minimal filter data for URL sharing (only essential fields)
                const filtersData = {
                    objectiveFilters: {
                        // Only store IDs and essential fields for responsible users
                        responsible: this.objectiveFilters.responsible.map(user => ({
                            id: user.id,
                            fullName: user.fullName,
                            email: user.email
                        })),
                        // Only store IDs and titles for collections
                        collections: this.objectiveFilters.collections.map(collection => ({
                            id: collection.id,
                            title: collection.title
                        })),
                        // Only store IDs and titles for statuses
                        statuses: this.objectiveFilters.statuses.map(status => ({
                            id: status.id,
                            title: status.title,
                            category: status.category
                        })),
                        search: this.objectiveFilters.search
                    },
                    timePeriodFilter: this.timePeriodFilter,
                    timeNavigator: this.timeNavigator,
                    searchQuery: this.searchQuery
                };

                // Check if there are any filters or navigator to add to URL
                const hasFilters = this.isFilterApplied ||
                                 this.searchQuery ||
                                 (this.timePeriodFilter.filter && Object.keys(this.timePeriodFilter.filter).length > 0);

                const hasNavigator = this.timeNavigator && this.timeNavigator.selectedTimePeriod;

                if (hasFilters || hasNavigator) {
                    // Use URL-safe base64 encoding to prevent 404 errors with special characters
                    const base64 = btoa(JSON.stringify(filtersData))
                        .replace(/\+/g, '-')
                        .replace(/\//g, '_')
                        .replace(/=+$/, '');
                    params.set('filters', base64);

                    // Update URL without reloading the page
                    window.history.replaceState({}, '', url.toString());
                    console.log('URL updated with filters and navigator:', url.toString());
                } else {
                    // If no filters or navigator, remove the filters param entirely
                    params.delete('filters');
                    window.history.replaceState({}, '', url.toString());
                    console.log('URL cleared - no filters or navigator');
                }
            },

            /**
             * Load filters from URL parameters
             */
            loadFiltersFromURL() {
                const url = new URL(window.location.href);
                const filtersParam = url.searchParams.get('filters');
                if (filtersParam) {
                    try {
                        // Convert URL-safe base64 back to regular base64
                        const regularBase64 = filtersParam
                            .replace(/-/g, '+')
                            .replace(/_/g, '/')
                            .padEnd(filtersParam.length + (4 - filtersParam.length % 4) % 4, '=');

                        const decoded = JSON.parse(atob(regularBase64));

                        // Reconstruct filters from minimal URL data
                        if (decoded.objectiveFilters) {
                            // Responsible filter - minimal data is sufficient for the UI
                            if (decoded.objectiveFilters.responsible) {
                                this.objectiveFilters.responsible = decoded.objectiveFilters.responsible;
                            }

                            // Collections filter - minimal data is sufficient for the UI
                            if (decoded.objectiveFilters.collections) {
                                this.objectiveFilters.collections = decoded.objectiveFilters.collections;
                            }

                            // Statuses filter - minimal data is sufficient for the UI
                            if (decoded.objectiveFilters.statuses) {
                                this.objectiveFilters.statuses = decoded.objectiveFilters.statuses;
                            }

                            // Search filter
                            if (decoded.objectiveFilters.search) {
                                this.objectiveFilters.search = decoded.objectiveFilters.search;
                            }
                        }

                        if (decoded.timePeriodFilter) this.timePeriodFilter = decoded.timePeriodFilter;
                        if (decoded.timeNavigator) this.timeNavigator = decoded.timeNavigator;
                        if (decoded.searchQuery) this.searchQuery = decoded.searchQuery;

                        return true; // Indicate successful loading from URL
                    } catch (e) {
                        console.error('Failed to parse filters from URL', e);
                        return false;
                    }
                }
                return false; // No URL filters found
            },

            navigateToTimeperiodNow(){
                // If a time period is present in the navigator, navigate to it
                if (this.timeNavigator && this.timeNavigator.selectedTimePeriod) {
                    // Try to infer start/end dates for the selected period
                    const period = this.timeNavigator.selectedTimePeriod;
                    const type = (period.title || '').toUpperCase();
                    const year = period.selectedYear;
                    const periodMap = {
                        'YEARLY': { startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                        'H1': { startMonth: 0, startDay: 1, endMonth: 5, endDay: 31 },
                        'H2': { startMonth: 6, startDay: 1, endMonth: 11, endDay: 31 },
                        'Q1': { startMonth: 0, startDay: 1, endMonth: 2, endDay: 31 },
                        'Q2': { startMonth: 3, startDay: 1, endMonth: 5, endDay: 30 },
                        'Q3': { startMonth: 6, startDay: 1, endMonth: 8, endDay: 30 },
                        'Q4': { startMonth: 9, startDay: 1, endMonth: 11, endDay: 31 },
                        'JAN': { startMonth: 0, startDay: 1, endMonth: 0, endDay: 31 },
                        'FEB': { startMonth: 1, startDay: 1, endMonth: 1, endDay: 28 },
                        'MAR': { startMonth: 2, startDay: 1, endMonth: 2, endDay: 31 },
                        'APR': { startMonth: 3, startDay: 1, endMonth: 3, endDay: 30 },
                        'MAY': { startMonth: 4, startDay: 1, endMonth: 4, endDay: 31 },
                        'JUN': { startMonth: 5, startDay: 1, endMonth: 5, endDay: 31 },
                        'JUL': { startMonth: 6, startDay: 1, endMonth: 6, endDay: 31 },
                        'AUG': { startMonth: 7, startDay: 1, endMonth: 7, endDay: 31 },
                        'SEP': { startMonth: 8, startDay: 1, endMonth: 8, endDay: 30 },
                        'OCT': { startMonth: 9, startDay: 1, endMonth: 9, endDay: 31 },
                        'NOV': { startMonth: 10, startDay: 1, endMonth: 10, endDay: 30 },
                        'DEC': { startMonth: 11, startDay: 1, endMonth: 11, endDay: 31 }
                    };
                    const p = periodMap[type];
                    if (p) {
                        let endDay = p.endDay;
                        if (type === 'FEB') {
                            endDay = (new Date(year, 2, 0)).getDate();
                        }
                        const startDate = new Date(year, p.startMonth, p.startDay);
                        const endDate = new Date(year, p.endMonth, endDay);
                        this.navigateToTimePeriod({ startDate, endDate, timePeriod: period });
                    }
                }
            },

            /**
             * Clear all filters and remove from storage/URL
             */
            clearAllFilters() {
                // Clear main filters but preserve time navigator
                this.timePeriodFilter.showFilter = false;
                this.objectiveFilters.responsible = [];
                this.objectiveFilters.collections = [];
                this.objectiveFilters.statuses = [];
                this.objectiveFilters.search = "";
                this.searchQuery = "";
                this.timePeriodFilter.filter = {};

                // Clear only main filters from localStorage, preserve navigator
                const storageKey = this.getRoadmapStorageKey();
                const navigatorStorageKey = `${storageKey}_navigator`;
                localStorage.removeItem(storageKey);
                // Note: We don't remove navigatorStorageKey to preserve time navigator

                // Update URL to remove filters but keep navigator
                this.updateURLWithFilters();

                console.log('Main filters cleared, time navigator preserved');
                this.reloadObjectiveGroups();
            },

            /**
             * Load filters on component startup
             * Priority: URL parameters first, then localStorage
             */
            async loadFiltersOnStartup(retryCount = 0) {
                const MAX_RETRIES = 50; // Maximum 5 seconds (50 * 100ms)

                // Wait for roadmap data to be available
                if (!this.roadmap || !this.roadmap.id) {
                    // Check if we've exceeded the maximum retry attempts
                    if (retryCount >= MAX_RETRIES) {
                        console.error('Failed to load filters: roadmap data not available after maximum retries');
                        // Set default timeNavigator even if roadmap is not available
                        this.setDefaultTimeNavigator();
                        return;
                    }

                    // If roadmap not loaded yet, try again after a short delay
                    return new Promise((resolve) => {
                        setTimeout(async () => {
                            await this.loadFiltersOnStartup(retryCount + 1);
                            resolve();
                        }, 100);
                    });
                }

                console.log('Loading filters on startup for roadmap:', this.roadmap.id);

                // Try to load filters from URL first (for sharing)
                const urlLoaded = this.loadFiltersFromURL();
                let loadedFromStorage = false;
                if (!urlLoaded) {
                    // If no URL filters, try to load from localStorage
                    loadedFromStorage = this.loadFiltersFromStorage();
                    if (loadedFromStorage) {
                        console.log('Filters loaded from localStorage');
                    } else {
                        console.log('No saved filters found');
                    }
                } else {
                    console.log('Filters loaded from URL');
                }

                // If after loading, there is no selectedTimePeriod, set default to current year
                if (!this.timeNavigator || !this.timeNavigator.selectedTimePeriod) {
                    this.setDefaultTimeNavigator();
                }

                // Apply loaded filters to ensure they're active
                this.applyLoadedFilters();

                // Reload data with the applied filters to ensure consistency
                if (urlLoaded || loadedFromStorage) {
                    console.log('Reloading data with applied filters...');
                    await this.reloadObjectiveGroups();
                }

                console.log('Filters loaded and applied successfully');
            },

            /**
             * Set default time navigator configuration
             */
            setDefaultTimeNavigator() {
                const currentYear = new Date().getFullYear();
                this.timeNavigator = {
                    showNavigator: false,
                    selectedTimePeriod: {
                        title: 'YEARLY',
                        selectedYear: currentYear,
                        monthly: false
                    },
                    customSpanYears: 2
                };
                console.log('Defaulted timeNavigator to YEARLY', currentYear);
            },

            /**
             * Navigate to a specific time period in the scheduler with smooth animations
             * @param {Object} navigationData - Object containing startDate, endDate, and timePeriod
             */
            navigateToTimePeriod(navigationData) {
                if (!scheduler || !navigationData) {
                    console.warn('Scheduler not available or navigation data missing');
                    return;
                }

                const { startDate, endDate, timePeriod } = navigationData;

                if (!startDate || !endDate) {
                    console.warn('Start date or end date missing for navigation');
                    return;
                }

                try {
                    // Store the selected time period for display in dropdown title
                    this.timeNavigator.selectedTimePeriod = timePeriod;
                    // Preserve custom span years if provided
                    if (timePeriod && timePeriod.title && timePeriod.title.toUpperCase() === 'CUSTOM') {
                        this.timeNavigator.customSpanYears = timePeriod.spanYears || this.timeNavigator.customSpanYears;
                    }

                    // Apply appropriate preset based on time period using getPresetFactory
                    const presetKey = this.getPresetKeyForTimePeriod(timePeriod.title);
                    const preset = getPresetFactory(scheduler, presetKey);

                    // Apply the preset to the scheduler
                    scheduler.viewPreset = preset;

                    // Add preset-specific CSS class for enhanced styling
                    this.applyPresetStyling(presetKey);
                    
                    // Enable smooth animations for the navigation
                    this.enableSmoothNavigation();

                    // Focus on the selected time range with smooth animation
                    scheduler.zoomTo({
                        startDate: startDate,
                        endDate: endDate,
                        animate: {
                            duration: 800,
                            easing: 'easeInOutCubic'
                        }
                    });

                    // Add a subtle visual feedback for the navigation
                    this.showNavigationFeedback(timePeriod);

                    console.log('Navigated to time period with smooth animation:', {
                        period: timePeriod.title,
                        year: timePeriod.selectedYear,
                        presetKey: presetKey,
                        start: startDate,
                        end: endDate
                    });

                    // Save filters and update URL
                    this.saveFiltersToStorage();
                    this.saveTimeNavigatorToStorage();
                    this.updateURLWithFilters();

                } catch (error) {
                    console.error('Error navigating to time period:', error);
                    showTopMessageTargeted(
                        '❌ Failed to navigate to selected time period',
                        'error',
                        3000,
                        this.isFullScreen ? '.okrsrdmp' : null
                    );
                }
            },

            /**
             * Alternative navigation method that fits the time span to the viewport with smooth animations
             * @param {Object} navigationData - Object containing startDate, endDate, and timePeriod
             */
            navigateToTimePeriodFit(navigationData) {
                if (!scheduler || !navigationData) {
                    console.warn('Scheduler not available or navigation data missing');
                    return;
                }

                const { startDate, endDate, timePeriod } = navigationData;

                if (!startDate || !endDate) {
                    console.warn('Start date or end date missing for navigation');
                    return;
                }

                try {
                    // Store the selected time period for display
                    this.timeNavigator.selectedTimePeriod = timePeriod;
                    if (timePeriod && timePeriod.title && timePeriod.title.toUpperCase() === 'CUSTOM') {
                        this.timeNavigator.customSpanYears = timePeriod.spanYears || this.timeNavigator.customSpanYears;
                    }

                    // Apply appropriate preset based on time period using getPresetFactory
                    const presetKey = this.getPresetKeyForTimePeriod(timePeriod.title);
                    const preset = getPresetFactory(scheduler, presetKey);

                    // Apply the preset to the scheduler
                    scheduler.viewPreset = preset;

                    // Add preset-specific CSS class for enhanced styling
                    this.applyPresetStyling(presetKey);
                    
                    // Enable smooth animations for the navigation
                    this.enableSmoothNavigation();

                    // Scroll to the selected date with smooth animation
                    scheduler.scrollToDate(startDate, {
                        animate: {
                            duration: 600,
                            easing: 'easeInOutCubic'
                        }
                    });

                    // Add visual feedback for the navigation
                    this.showNavigationFeedback(timePeriod);

                    console.log('Navigated to time period with smooth scroll animation:', {
                        period: timePeriod.title,
                        year: timePeriod.selectedYear,
                        presetKey: presetKey,
                        start: startDate,
                        end: endDate
                    });

                    // Save filters and update URL
                    this.saveFiltersToStorage();
                    this.saveTimeNavigatorToStorage();
                    this.updateURLWithFilters();

                } catch (error) {
                    console.error('Error navigating to time period:', error);
                    showTopMessageTargeted(
                        'Failed to navigate to selected time period',
                        'error', 
                        3000, 
                        this.isFullScreen ? '.okrsrdmp' : null
                    );
                }
            },

            /**
             * Enable smooth navigation animations
             */
            enableSmoothNavigation() {
                // if (!scheduler) return;

                // // Enable smooth scrolling and zooming
                // scheduler.features.scrollTimePivot = true;
                // scheduler.features.zoom = {
                //     animate: true,
                //     duration: 600
                // };

                // // Enable smooth event animations
                // scheduler.features.eventDrag = {
                //     animate: true,
                //     duration: 300
                // };

                // // Enable smooth resize animations
                // scheduler.features.eventResize = {
                //     animate: true,
                //     duration: 300
                // };
            },

            /**
             * Show visual feedback for navigation
             * @param {Object} timePeriod - The time period being navigated to
             */
            showNavigationFeedback(timePeriod) {
                // Add a subtle highlight to the scheduler during navigation
                const schedulerElement = document.getElementById('okrs-roadmap-bryntum');
                if (schedulerElement) {
                    schedulerElement.style.transition = 'all 0.3s ease';
                    schedulerElement.style.boxShadow = '0 0 20px rgba(33, 150, 243, 0.3)';

                    setTimeout(() => {
                        schedulerElement.style.boxShadow = '';
                    }, 800);
                }

                // Show a brief success message
                // const periodText = `${timePeriod.title} ${timePeriod.selectedYear}`;
                // showTopMessageTargeted(
                //     `🎯 Navigated to ${periodText}`,
                //     'success',
                //     1500,
                //     this.isFullScreen ? '.okrsrdmp' : null
                // );
            },

            /**
             * Utility to clear the 'filters' parameter from the URL
             */
            clearFiltersFromURL() {
                const url = new URL(window.location.href);
                url.searchParams.delete('filters');
                window.history.replaceState({}, '', url.toString());
            },

            // ... existing code ...

            beforeRouteLeave(to, from, next) {
                // Remove filters from URL before navigating away
                this.clearFiltersFromURL();
                next();
            },
            /**
             * Navigate to previous or next time period (capsule) based on current selection with smooth animations
             * @param {'prev'|'next'} direction
             */
            navigateToAdjacentTimePeriod(direction) {
                const current = this.timeNavigator.selectedTimePeriod;
                if (!current) return;
                const year = current.selectedYear;
                const type = (current.title || '').toUpperCase();
                let newType = type, newYear = year;

                // Define period groups for type consistency
                const periodGroups = {
                    'HALF': ['H1', 'H2'],
                    'QUARTER': ['Q1', 'Q2', 'Q3', 'Q4'],
                    'MONTH': ['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC']
                };
                // Map type to group
                let group = null;
                if (periodGroups.HALF.includes(type)) group = periodGroups.HALF;
                else if (periodGroups.QUARTER.includes(type)) group = periodGroups.QUARTER;
                else if (periodGroups.MONTH.includes(type)) group = periodGroups.MONTH;

                if (group) {
                    let idx = group.indexOf(type);
                    if (idx === -1) return;
                    if (direction === 'prev') {
                        idx--;
                        if (idx < 0) {
                            idx = group.length - 1;
                            newYear = year - 1;
                        }
                    } else {
                        idx++;
                        if (idx >= group.length) {
                            idx = 0;
                            newYear = year + 1;
                        }
                    }
                    newType = group[idx];
                } else {
                    // For yearly/monthly, just increment/decrement year
                    if (type === 'YEARLY' || type === 'MONTHLY') {
                        if (direction === 'prev') newYear = year - 1;
                        else newYear = year + 1;
                    }
                }

                // Map type to navigation data
                const periodMap = {
                    'YEARLY': { title: 'Yearly', startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                    'H1': { title: 'H1', startMonth: 0, startDay: 1, endMonth: 5, endDay: 31 },
                    'H2': { title: 'H2', startMonth: 6, startDay: 1, endMonth: 11, endDay: 31 },
                    'Q1': { title: 'Q1', startMonth: 0, startDay: 1, endMonth: 2, endDay: 31 },
                    'Q2': { title: 'Q2', startMonth: 3, startDay: 1, endMonth: 5, endDay: 30 },
                    'Q3': { title: 'Q3', startMonth: 6, startDay: 1, endMonth: 8, endDay: 30 },
                    'Q4': { title: 'Q4', startMonth: 9, startDay: 1, endMonth: 11, endDay: 31 },
                    'MONTHLY': { title: 'Monthly', startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                    'JAN': { title: 'Jan', startMonth: 0, startDay: 1, endMonth: 0, endDay: 31 },
                    'FEB': { title: 'Feb', startMonth: 1, startDay: 1, endMonth: 1, endDay: 28 },
                    'MAR': { title: 'Mar', startMonth: 2, startDay: 1, endMonth: 2, endDay: 31 },
                    'APR': { title: 'Apr', startMonth: 3, startDay: 1, endMonth: 3, endDay: 30 },
                    'MAY': { title: 'May', startMonth: 4, startDay: 1, endMonth: 4, endDay: 31 },
                    'JUN': { title: 'Jun', startMonth: 5, startDay: 1, endMonth: 5, endDay: 30 },
                    'JUL': { title: 'Jul', startMonth: 6, startDay: 1, endMonth: 6, endDay: 31 },
                    'AUG': { title: 'Aug', startMonth: 7, startDay: 1, endMonth: 7, endDay: 31 },
                    'SEP': { title: 'Sep', startMonth: 8, startDay: 1, endMonth: 8, endDay: 30 },
                    'OCT': { title: 'Oct', startMonth: 9, startDay: 1, endMonth: 9, endDay: 31 },
                    'NOV': { title: 'Nov', startMonth: 10, startDay: 1, endMonth: 10, endDay: 30 },
                    'DEC': { title: 'Dec', startMonth: 11, startDay: 1, endMonth: 11, endDay: 31 }
                };
                const period = periodMap[newType];
                if (!period) return;
                // Handle leap year for Feb
                let endDay = period.endDay;
                if (newType === 'FEB') {
                    endDay = (new Date(newYear, 2, 0)).getDate();
                }
                const startDate = new Date(newYear, period.startMonth, period.startDay);
                let endDate = new Date(newYear, period.endMonth, endDay);

                // Add 1 day to end date for H2 to extend into next year
                if (newType === 'H2') {
                    endDate.setDate(endDate.getDate() + 1);
                }

                const timePeriod = { title: period.title, selectedYear: newYear };

                // Apply preset based on zoom level using getPresetFactory
                const presetKey = this.getPresetKeyForTimePeriod(period.title);
                const preset = getPresetFactory(scheduler, presetKey);
                scheduler.viewPreset = preset;

                // Add preset-specific CSS class for enhanced styling
                this.applyPresetStyling(presetKey);
                
                // Enable smooth animations for adjacent navigation
                this.enableSmoothNavigation();

                // Navigate to the time period with smooth animation
                this.navigateToTimePeriod({ startDate, endDate, timePeriod });

                // Save time navigator separately when zooming
                this.saveTimeNavigatorToStorage();
                
                // Show visual feedback for the navigation
                this.showNavigationFeedback(timePeriod);

                console.log('Adjacent navigation with smooth animation:', {
                    direction: direction,
                    newType: newType,
                    presetKey: presetKey,
                    timePeriod: timePeriod
                });
            },
            /**
             * Zoom in or out of the current time period (capsule)
             * @param {'in'|'out'} direction
             */
            zoomTimePeriod(direction) {
                const current = this.timeNavigator.selectedTimePeriod;
                if (!current) return;
                const year = current.selectedYear;
                const type = (current.title || '').toUpperCase();

                // Handle custom multi-year zoom levels when zooming out from YEARLY
                // and when already in CUSTOM span (2y -> 5y), and reverse on zoom in
                if (direction === 'out') {
                    if (type === 'YEARLY') {
                        // Switch to 2-year custom view
                        this.applyCustomZoom(year, 2);
                        return;
                    }
                    if (type === 'CUSTOM') {
                        let span = current.spanYears || this.timeNavigator.customSpanYears || 2;
                        if (span >= 5) {
                            // Stop at 5 years
                            return;
                        }
                        this.applyCustomZoom(year, span + 1);
                        return;
                    }
                } else if (direction === 'in') {
                    if (type === 'CUSTOM') {
                        let span = current.spanYears || this.timeNavigator.customSpanYears || 2;
                        if (span > 2) {
                            this.applyCustomZoom(year, span - 1);
                            return;
                        }
                        // Back to yearly when reducing below 2 years
                        const startDate = new Date(year, 0, 1);
                        const endDate = new Date(year, 11, 31);
                        const timePeriod = { title: 'Yearly', selectedYear: year };
                        this.navigateToTimePeriod({ startDate, endDate, timePeriod });
                        return;
                    }
                }

                // Map months to quarters
                const monthToQuarter = {
                    'JAN': 'Q1', 'FEB': 'Q1', 'MAR': 'Q1',
                    'APR': 'Q2', 'MAY': 'Q2', 'JUN': 'Q2',
                    'JUL': 'Q3', 'AUG': 'Q3', 'SEP': 'Q3',
                    'OCT': 'Q4', 'NOV': 'Q4', 'DEC': 'Q4'
                };
                // Map quarters to halves
                const quarterToHalf = {
                    'Q1': 'H1', 'Q2': 'H1',
                    'Q3': 'H2', 'Q4': 'H2'
                };
                // Map halves to year
                const halfToYear = {
                    'H1': 'YEARLY', 'H2': 'YEARLY'
                };
                // Map halves to quarters
                const halfToQuarters = {
                    'H1': ['Q1', 'Q2'],
                    'H2': ['Q3', 'Q4']
                };
                // Map quarters to months
                const quarterToMonths = {
                    'Q1': ['JAN', 'FEB', 'MAR'],
                    'Q2': ['APR', 'MAY', 'JUN'],
                    'Q3': ['JUL', 'AUG', 'SEP'],
                    'Q4': ['OCT', 'NOV', 'DEC']
                };

                let newType = type;
                let newYear = year;

                if (direction === 'in') {
                    // Zoom in: go to lower level
                    if (type === 'YEARLY') {
                        newType = 'H1';
                    } else if (type === 'H1' || type === 'H2') {
                        newType = halfToQuarters[type][0];
                    } else if (['Q1', 'Q2', 'Q3', 'Q4'].includes(type)) {
                        newType = quarterToMonths[type][0];
                    }
                } else {
                    // Zoom out: go to higher level
                    if (['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL', 'AUG', 'SEP', 'OCT', 'NOV', 'DEC'].includes(type)) {
                        newType = monthToQuarter[type];
                    } else if (['Q1', 'Q2', 'Q3', 'Q4'].includes(type)) {
                        newType = quarterToHalf[type];
                    } else if (['H1', 'H2'].includes(type)) {
                        newType = 'YEARLY';
                    }
                }

                // Map type to navigation data
                const periodMap = {
                    'YEARLY': { title: 'Yearly', startMonth: 0, startDay: 1, endMonth: 11, endDay: 31 },
                    'H1': { title: 'H1', startMonth: 0, startDay: 1, endMonth: 5, endDay: 31 },
                    'H2': { title: 'H2', startMonth: 6, startDay: 1, endMonth: 11, endDay: 31 },
                    'Q1': { title: 'Q1', startMonth: 0, startDay: 1, endMonth: 2, endDay: 31 },
                    'Q2': { title: 'Q2', startMonth: 3, startDay: 1, endMonth: 5, endDay: 30 },
                    'Q3': { title: 'Q3', startMonth: 6, startDay: 1, endMonth: 8, endDay: 30 },
                    'Q4': { title: 'Q4', startMonth: 9, startDay: 1, endMonth: 11, endDay: 31 },
                    'JAN': { title: 'Jan', startMonth: 0, startDay: 1, endMonth: 0, endDay: 31 },
                    'FEB': { title: 'Feb', startMonth: 1, startDay: 1, endMonth: 1, endDay: 28 },
                    'MAR': { title: 'Mar', startMonth: 2, startDay: 1, endMonth: 2, endDay: 31 },
                    'APR': { title: 'Apr', startMonth: 3, startDay: 1, endMonth: 3, endDay: 30 },
                    'MAY': { title: 'May', startMonth: 4, startDay: 1, endMonth: 4, endDay: 31 },
                    'JUN': { title: 'Jun', startMonth: 5, startDay: 1, endMonth: 5, endDay: 30 },
                    'JUL': { title: 'Jul', startMonth: 6, startDay: 1, endMonth: 6, endDay: 31 },
                    'AUG': { title: 'Aug', startMonth: 7, startDay: 1, endMonth: 7, endDay: 31 },
                    'SEP': { title: 'Sep', startMonth: 8, startDay: 1, endMonth: 8, endDay: 30 },
                    'OCT': { title: 'Oct', startMonth: 9, startDay: 1, endMonth: 9, endDay: 31 },
                    'NOV': { title: 'Nov', startMonth: 10, startDay: 1, endMonth: 10, endDay: 30 },
                    'DEC': { title: 'Dec', startMonth: 11, startDay: 1, endMonth: 11, endDay: 31 }
                };
                const period = periodMap[newType];
                if (!period) return;
                // Handle leap year for Feb
                let endDay = period.endDay;
                if (newType === 'FEB') {
                    endDay = (new Date(newYear, 2, 0)).getDate();
                }
                const startDate = new Date(newYear, period.startMonth, period.startDay);
                let endDate = new Date(newYear, period.endMonth, endDay);

                // Add 1 day to end date for H2 to extend into next year
                if (newType === 'H2') {
                    endDate.setDate(endDate.getDate() + 1);
                }

                const timePeriod = { title: period.title, selectedYear: newYear };

                // Enable smooth animations for zoom operations
                this.enableSmoothNavigation();

                // Navigate to the time period with smooth animation
                this.navigateToTimePeriod({ startDate, endDate, timePeriod });

                // Show visual feedback for the zoom operation
                this.showNavigationFeedback(timePeriod);

                console.log('Zoom operation with smooth animation:', {
                    direction: direction,
                    fromType: type,
                    toType: newType,
                    timePeriod: timePeriod
                });
            },
            loadLinkedItemsForObjective: function(objectives) {
                var _this = this;

                // Get field IDs for KR fields
                let fieldsMapKR = kendisStore.getters.getOkrTemplate().fieldsMap;
                let startDateFieldIdKr, endDateFieldIdKr;
                for (let fieldId in fieldsMapKR) {
                    if (fieldsMapKR[fieldId] === 'Start Date') {
                        startDateFieldIdKr = fieldId;
                    } else if (fieldsMapKR[fieldId] === 'End Date') {
                        endDateFieldIdKr = fieldId;
                    }
                }

                // Collect all objective IDs for batch request
                let objectiveIds = objectives.map(obj => obj.id);

                // Use the new batch endpoint for better performance
                axios.post('objective/get-objective-by-ids', {
                    fetchPIs: true,
                    objectiveIds: objectiveIds
                }).then(function(response) {
                    var objectivesData = response.data.objectives || [];

                    objectivesData.forEach(function(data) {
                        if (data && data.baseItemList && data.baseItemList.length) {
                            data.baseItemList.forEach(function(item) {
                                if (item.type === 'KR' || item.type === 'Objective-Committed') {
                                    // Add as event to scheduler, under objectiveId resource
                                    if (!scheduler.eventStore.getById(item.id)) {
                                        scheduler.eventStore.add({
                                            id: item.id,
                                            name: item.title,
                                            startDate: item.fields && item.fields[_this.startDateFieldId],
                                            endDate: item.fields && item.fields[_this.endDateFieldId],
                                            type: item.type,
                                            cls:'kendis-kr-roadmap-event'
                                            // ...other fields as needed
                                        });
                                    }
                                    //add in assignment store
                                    scheduler.assignmentStore.add({
                                        id: item.id,
                                        resourceId: data.id,
                                        eventId: item.id
                                    });
                                }
                            });
                        }
                    });
                }).catch(function(error) {
                    console.error('Error loading linked items for objectives', error);
                    showTopMessageTargeted("Error loading linked items for objectives", 'error', 4000, this.isFullScreen ? '.okrsrdmp' : null)
                });
            },
            
            // Handle Key Results toggle switch
            onKeyResultsToggle() {
                // Toggle between objectives and multilevel view
                if (this.showKeyResults) {
                    this.onViewModeChange('multilevel');
                } else {
                    this.onViewModeChange('objectives');
                }
            },
            
            // Get formatted time navigator label
            getTimeNavigatorLabel() {
                if (!this.timeNavigator.selectedTimePeriod) {
                    return 'Navigate';
                }
                
                const period = this.timeNavigator.selectedTimePeriod;
                if (period.title === 'Custom') {
                    return 'Custom';
                }
                
                return period.title + ' ' + period.selectedYear;
            },
            
            // Apply multi-year custom zoom (2y..5y) centered on a given year
            applyCustomZoom(centerYear, spanYears) {
                try {
                    const span = Math.max(2, Math.min(5, spanYears));
                    this.timeNavigator.customSpanYears = span;
                    let startYear, endYear;
                    if (span === 2) {
                        startYear = centerYear;
                        endYear = centerYear + 1;
                    } else if (span === 3) {
                        startYear = centerYear - 1;
                        endYear = centerYear + 1;
                    } else if (span === 4) {
                        startYear = centerYear - 1;
                        endYear = centerYear + 2;
                    } else { // 5
                        startYear = centerYear - 2;
                        endYear = centerYear + 2;
                    }

                    const startDate = new Date(startYear, 0, 1);
                    const endDate = new Date(endYear, 11, 31);
                    const timePeriod = { title: 'Custom', selectedYear: centerYear, spanYears: span };

                    const presetKey = this.getPresetKeyForTimePeriod('CUSTOM');
                    const preset = getPresetFactory(scheduler, presetKey);
                    scheduler.viewPreset = preset;
                    this.applyPresetStyling(presetKey);
                    this.enableSmoothNavigation();

                    this.navigateToTimePeriod({ startDate, endDate, timePeriod });
                } catch (e) {
                    console.error('Error applying custom zoom', e);
                    showTopMessageTargeted('Failed to apply custom zoom', 'error', 3000, this.isFullScreen ? '.okrsrdmp' : null);
                }
            },
            
            // Helper to append children to a resource in the resourceStore tree
            appendChildrenToResource: function(parentId, childrenArray) {
                var parentResource = scheduler.resourceStore.getById(parentId);
                if (!parentResource) return;
                childrenArray.forEach(function(child) {
                    // Only append if not already present
                    if (!scheduler.resourceStore.getById(child.id)) {
                        parentResource.appendChild(child);
                    }
                });
            },

            // Recursively update color of all child resources
            updateChildrenResourceColors: function(parentId, newColor) {
                var parentResource = scheduler.resourceStore.getById(parentId);
                if (!parentResource || !parentResource.children) return;
                parentResource.children.forEach(function(child) {
                    child.color = newColor;
                    // Recursively update for further descendants
                    if (child.children && child.children.length) {
                        roadmapVue.updateChildrenResourceColors(child.id, newColor);
                    }
                });
            },
        },
        components: {
            'roadmap-settings':ROADMAP_SETTINGS,
            'treeselect' : VueTreeselect.Treeselect,
            VueAdsPagination: window["vue-ads-pagination"].default,
            VueAdsPageButton: window["vue-ads-pagination"].VueAdsPageButton,
            'action_popup': ACTION_POPUP,
            'add-milestone-board' : ADD_MILESTONE_TO_BOARD,
            'search_suggestions': SEARCH_SUGGESTIONS,
            'async-date-confirmation-popup':ASYNC_DATE_CONFIRMATION_POPUP,
            'kendis-alert': KENDIS_ALERT,
            "pi-user-picker": PI_USER_PICKER,
            OBJECTIVE_GROUP_PROGRESS: 'objective-group-progress',
            'objective-theme-progress': OBJECTIVE_THEME_PROGRESS,
            'time-period-popup': TIME_PERIOD_POPUP,
            'objective-link': OBJECTIVE_LINK,
        }
    });


