/*!
 *
 * Bryntum Scheduler Pro 5.5.0
 *
 * Copyright(c) 2023 Bryntum AB
 * https://bryntum.com/contact
 * https://bryntum.com/license
 *
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory((global.bryntum = global.bryntum || {}, global.bryntum.schedulerpro = {})));
})(this, (function (exports) { 'use strict';

    const productName = 'schedulerpro'

    var _globalThis$matchMedi;
    /**
     * @module Core/helper/BrowserHelper
     */
    /**
     * Static helper class that does browser or platform detection and provides other helper functions.
     */
    class BrowserHelper {
      static supportsPointerEvents = Boolean(globalThis.PointerEvent || globalThis.MSPointerEvent);
      // Locker Service does not allow to instantiate PointerEvents. LWS apparently does, however.
      // https://github.com/bryntum/support/issues/5578
      static supportsPointerEventConstructor = typeof PointerEvent !== 'undefined';
      static PointerEventConstructor = globalThis.PointerEvent || globalThis.CustomEvent;
      //region Init
      /**
       * Yields `true` if the platform running is a phone (screen width or height <= 414 CSS pixels)
       * @property {Boolean}
       * @readonly
       * @static
       * @category Platform
       */
      static isPhone = (_globalThis$matchMedi = globalThis.matchMedia) === null || _globalThis$matchMedi === void 0 ? void 0 : _globalThis$matchMedi.call(globalThis, '(max-height:414px) or (max-width:414px)').matches;
      static cacheFlags(platform = navigator.platform, userAgent = navigator.userAgent) {
        const me = this;
        // os
        me._isLinux = Boolean(platform.match(/Linux/));
        me._isMac = Boolean(platform.match(/Mac/));
        me._isWindows = Boolean(platform.match(/Win32/));
        me._isMobile = Boolean(userAgent.match(/Mobile|Opera Mini|Opera Mobi|Puffin/) || typeof globalThis.orientation === 'number');
        // Edge user agent contains webkit too.
        // This is not a typo. Edge has "Safari/537.36 Edg/96.0.1054.34"
        me._isWebkit = Boolean(userAgent.match(/WebKit/) && !userAgent.match(/Edg/));
        me._firefoxVersion = me.getVersion(userAgent, /Firefox\/(\d+)\./);
        me._isFirefox = me._firefoxVersion > 0;
        me._chromeVersion = me.getVersion(userAgent, /Chrom(?:e|ium)\/(\d+)\./);
        me._isChrome = me._chromeVersion > 0;
        me._isSafari = Boolean(userAgent.match(/Safari/)) && !me._isChrome;
        me._isMobileSafari = Boolean(userAgent.match(/Mobile.*Safari/));
        me._safariVersion = me.getVersion(userAgent, /Version\/(.*).Safari/);
        me._isAndroid = Boolean(userAgent.match(/Android/g));
      }
      //endregion
      //region Device
      /**
       * Yields `true` if the current browser supports CSS style `overflow:clip`.
       * @property {Boolean}
       * @readonly
       * @internal
       */
      static get supportsOverflowClip() {
        if (this._supportsOverflowClip == null) {
          const div = document.createElement('div');
          div.style.overflow = 'clip';
          div.style.display = 'none';
          // If we're called before DOMContentLoaded, body won't be available.
          // HTML element works for style calcs.
          document.documentElement.appendChild(div);
          this._supportsOverflowClip = div.ownerDocument.defaultView.getComputedStyle(div).getPropertyValue('overflow') === 'clip';
          div.remove();
        }
        return this._supportsOverflowClip;
      }
      /**
       * Yields `true` if the current browser supports CSS style `position:sticky`.
       * @property {Boolean}
       * @readonly
       * @internal
       */
      static get supportsSticky() {
        return true;
      }
      /**
       * Returns matched version for userAgent.
       * @param {String} versionRe version match regular expression
       * @returns {Number} matched version
       * @readonly
       * @internal
       */
      static getVersion(userAgent, versionRe) {
        const match = userAgent.match(versionRe);
        return match ? parseFloat(match[1]) : 0;
      }
      /**
       * Determines if the user is using a touch device.
       * @property {Boolean}
       * @readonly
       * @internal
       */
      static get isTouchDevice() {
        // Allow tests or client code to set
        if (this._isTouchDevice === undefined) {
          this._isTouchDevice = globalThis.matchMedia('(pointer:coarse)').matches;
        }
        return this._isTouchDevice;
      }
      // Reports true by default for our tests
      static get isHoverableDevice() {
        if (this._isHoverableDevice === undefined) {
          this._isHoverableDevice = globalThis.matchMedia('(any-hover: hover)').matches;
        }
        return this._isHoverableDevice;
      }
      //endregion
      //region Platform
      static get isBrowserEnv() {
        // This window reference is left on purpose, globalThis is always defined
        // eslint-disable-next-line bryntum/no-window-in-lib
        return typeof window !== 'undefined';
      }
      /**
       * Checks if platform is Mac.
       * @property {Boolean}
       * @readonly
       * @category Platform
       */
      static get isMac() {
        return this._isMac;
      }
      /**
       * Checks if platform is Windows.
       * @property {Boolean}
       * @readonly
       * @category Platform
       */
      static get isWindows() {
        return this._isWindows;
      }
      /**
       * Checks if platform is Linux.
       * @property {Boolean}
       * @readonly
       * @category Platform
       */
      static get isLinux() {
        return this._isLinux;
      }
      /**
       * Checks if platform is Android.
       * @property {Boolean}
       * @readonly
       * @category Platform
       */
      static get isAndroid() {
        return this._isAndroid;
      }
      //endregion
      //region Browser
      /**
       * Checks if browser is Webkit.
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isWebkit() {
        return this._isWebkit;
      }
      /**
       * Checks if browser is Chrome or Chromium based browser.
       * Returns truthy value for Edge Chromium.
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isChrome() {
        return this._isChrome;
      }
      /**
       * Returns the major Chrome version or 0 for other browsers.
       * @property {Number}
       * @readonly
       * @category Browser
       */
      static get chromeVersion() {
        return this._chromeVersion;
      }
      /**
       * Checks if browser is Firefox.
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isFirefox() {
        return this._isFirefox;
      }
      /**
       * Returns the major Firefox version or 0 for other browsers.
       * @property {Number}
       * @readonly
       * @category Browser
       */
      static get firefoxVersion() {
        return this._firefoxVersion;
      }
      /**
       * Checks if browser is Safari.
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isSafari() {
        return this._isSafari;
      }
      static get safariVersion() {
        return this._safariVersion;
      }
      /**
       * Checks if browser is mobile Safari
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isMobileSafari() {
        return this._isMobileSafari;
      }
      /**
       * Checks if the active device is a mobile device
       * @property {Boolean}
       * @readonly
       * @category Browser
       */
      static get isMobile() {
        return this._isMobile;
      }
      static get platform() {
        const me = this;
        return me._isLinux ? 'linux' : me._isMac ? 'mac' : me._isWindows ? 'windows' : me._isAndroid ? 'android' : me._isMobileSafari ? 'ios' : null;
      }
      /**
       * Returns `true` if the browser supports passive event listeners.
       * @property {Boolean}
       * @internal
       * @deprecated Since 5.0. All modern browsers now support passive event listeners.
       * @category Browser
       */
      static get supportsPassive() {
        return true;
      }
      // Only works in secure contexts
      static get supportsRandomUUID() {
        if (this._supportsRandomUUID === undefined) {
          try {
            this._supportsRandomUUID = Boolean(globalThis.crypto.randomUUID().length > 0);
          } catch (e) {
            this._supportsRandomUUID = false;
          }
        }
        return this._supportsRandomUUID;
      }
      //endregion
      //region Storage
      // https://developer.mozilla.org/en-US/docs/Web/API/Web_Storage_API/Using_the_Web_Storage_API
      static get storageAvailable() {
        let storage, x;
        try {
          storage = localStorage;
          x = '__storage_test__';
          storage.setItem(x, x);
          storage.removeItem(x);
          return true;
        } catch (e) {
          return e instanceof DOMException && (
          // everything except Firefox
          e.code === 22 ||
          // Firefox
          e.code === 1014 ||
          // test name field too, because code might not be present
          // everything except Firefox
          e.name === 'QuotaExceededError' ||
          // Firefox
          e.name === 'NS_ERROR_DOM_QUOTA_REACHED') &&
          // acknowledge QuotaExceededError only if there's something already stored
          storage.length !== 0;
        }
      }
      static setLocalStorageItem(key, value) {
        this.storageAvailable && localStorage.setItem(key, value);
      }
      static getLocalStorageItem(key) {
        return this.storageAvailable && localStorage.getItem(key);
      }
      static removeLocalStorageItem(key) {
        this.storageAvailable && localStorage.removeItem(key);
      }
      //endregion
      //region Helpers
      /**
       * Returns parameter value from search string by parameter name.
       * @param {String} paramName search parameter name
       * @param {String} [defaultValue] default value if parameter not found
       * @param {String} [search] search string. Defaults to `document.location.search`
       * @returns {String} search parameter string value
       * @category Helper
       */
      static searchParam(paramName, defaultValue = null, search = document.location.search) {
        const re = new RegExp(`[?&]${paramName}=?([^&]*)`),
          match = search.match(re);
        return match && match[1] || defaultValue;
      }
      /**
       * Returns cookie by name.
       * @param {String} name cookie name
       * @returns {String} cookie string value
       * @category Helper
       */
      static getCookie(name) {
        const nameEq = encodeURIComponent(name) + '=',
          cookieItems = document.cookie.split(';');
        for (let i = 0; i < cookieItems.length; i++) {
          let c = cookieItems[i];
          while (c.charAt(0) === ' ') {
            c = c.substring(1, c.length);
          }
          if (c.indexOf(nameEq) === 0) {
            return decodeURIComponent(c.substring(nameEq.length, c.length));
          }
        }
        return '';
      }
      /**
       * Triggers a download of a file with the specified name / URL.
       * @param {String} filename The filename of the file to be downloaded
       * @param {String} [url] The URL where the file is to be downloaded from
       * @internal
       * @category Download
       */
      static download(filename, url) {
        const a = document.createElement('a');
        a.download = filename;
        a.href = url || filename;
        a.style.cssText = 'display:none';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
      }
      /**
       * Triggers a download of a Blob with the specified name.
       * @param {Blob} blob The Blob to be downloaded
       * @param {String} filename The filename of the file to be downloaded
       * @internal
       * @category Download
       */
      static downloadBlob(blob, filename) {
        const url = globalThis.URL.createObjectURL(blob);
        this.download(filename, url);
        globalThis.URL.revokeObjectURL(url);
      }
      static get queryString() {
        var _Object$fromEntries;
        // new URLSearchParams throws in salesforce
        // https://github.com/salesforce/lwc/issues/1812
        const params = new URL(globalThis.location.href).searchParams;
        // ?. to be nice to users with Chrome versions < 73
        return (_Object$fromEntries = Object.fromEntries) === null || _Object$fromEntries === void 0 ? void 0 : _Object$fromEntries.call(Object, params.entries());
      }
      // Used by docs fiddle
      static copyToClipboard(code) {
        let success = true;
        const textArea = document.createElement('textarea');
        textArea.value = code;
        textArea.style.height = textArea.style.width = 0;
        document.body.appendChild(textArea);
        textArea.select();
        try {
          document.execCommand('copy');
        } catch (e) {
          success = false;
        }
        textArea.remove();
        return success;
      }
      static isBryntumOnline(searchStrings) {
        var _searchStrings;
        searchStrings = Array.isArray(searchStrings) ? searchStrings : [searchStrings];
        return Boolean(/^(www\.)?bryntum\.com/.test(globalThis.location.host) || ((_searchStrings = searchStrings) === null || _searchStrings === void 0 ? void 0 : _searchStrings.some(str => this.queryString[str] != null)));
      }
      /**
       * Returns truthy value if page contains Content Security Policy meta tag or globalThis.bryntum.CSP is truthy value
       * @returns {Boolean}
       * @internal
       **/
      static get isCSP() {
        const {
          bryntum,
          document
        } = globalThis;
        if (bryntum.CSP == null) {
          bryntum.CSP = Boolean(document.querySelector('meta[http-equiv="Content-Security-Policy"]'));
        }
        return bryntum.CSP;
      }
      //endregion
    }

    if (BrowserHelper.isBrowserEnv) {
      BrowserHelper.cacheFlags();
    }
    BrowserHelper._$name = 'BrowserHelper';

    // IMPORTANT - adding imports here can create problems for Base class
    /**
     * @module Core/helper/StringHelper
     */
    let charsToEncode, entitiesToDecode, htmlEncodeRe, htmlDecodeRe;
    const camelLettersRe = /([a-z])([A-Z])/g,
      crlfRe = /[\n\r]/g,
      escapeRegExpRe = /[.*+?^${}()|[\]\\]/g,
      // same as NPM escape-string-regexp
      htmlRe$1 = /[&<]/,
      idRe = /(^[^a-z]+[^\w]+)/gi,
      whiteSpaceRe$1 = /\s+/,
      domIdRe = /^[^a-z]+|[^\w:.-]+/gi,
      htmlDecoder = (m, captured) => entitiesToDecode[captured.toLowerCase()] || String.fromCharCode(parseInt(captured.substr(2), 10)),
      htmlEncoder = (m, captured) => charsToEncode[captured],
      hyphenateCamelLetters = (all, g1, g2) => {
        return `${g1}-${g2.toLowerCase()}`;
      },
      separateCamelLetters = (all, g1, g2) => {
        return `${g1} ${g2.toLowerCase()}`;
      },
      replaceNonIdChar = c => {
        if (c) {
          return `_x${[...c].map(ch => ch.charCodeAt(0).toString(16)).join('')}`;
        }
        return '__blank__';
      },
      hyphenateCache = {},
      separatedCache = {};
    /**
     * Helper for string manipulation.
     */
    class StringHelper {
      //region Transform
      /**
       * Capitalizes the first letter of a string, "myString" -> "MyString".
       * @param {String} string The string to capitalize
       * @returns {String} The capitalized string or the value of `string` if falsy.
       * @category String formatting
       */
      static capitalize(string) {
        return string && string[0].toUpperCase() + string.substr(1);
      }
      /**
       * Makes the first letter of a string lowercase, "MyString" -> "myString".
       * @param {String} string The string to un-capitalize.
       * @returns {String} The un-capitalized string or the value of `string` if falsy.
       * @category String formatting
       */
      static uncapitalize(string) {
        return string && string[0].toLowerCase() + string.substr(1);
      }
      /**
       * Converts the passed camelCased string to a hyphen-separated string. eg "minWidth" -> "min-width"
       * @param {String} string The string to convert.
       * @returns {String} The string with adjoining lower and upper case letters
       * separated by hyphens and converted to lower case.
       * @category String formatting
       * @internal
       */
      static hyphenate(string) {
        // Cached since it is used heavily with DomHelper.sync()
        const cached = hyphenateCache[string];
        if (cached) {
          return cached;
        }
        return hyphenateCache[string] = string.replace(camelLettersRe, hyphenateCamelLetters);
      }
      /**
       * Converts the passed camelCased string to a capitalized, space-separated string. eg "startDate" -> "Start date".
       * @param {String} string The string to convert.
       * @returns {String} The string with spaces separating words.
       * @category String formatting
       * @internal
       */
      static separate(string) {
        // Cached since it may be used heavily
        const cached = separatedCache[string];
        if (cached) {
          return cached;
        }
        return separatedCache[string] = this.capitalize(string.replace(camelLettersRe, separateCamelLetters));
      }
      /**
       * Creates an alphanumeric identifier from any passed string. Encodes spaces and non-alpha characters.
       * @param {String} inString The string from which to strip non-identifier characters.
       * @returns {String}
       * @category Misc
       * @internal
       */
      static createId(inString) {
        return String(inString).replace(idRe, replaceNonIdChar);
      }
      static makeValidDomId(id, replaceValue = '') {
        if (id == null) {
          return null;
        }
        return String(id).replace(domIdRe, replaceValue);
      }
      //endregion
      //region Html
      static escapeRegExp(string, flags) {
        // $& means the whole matched string
        let ret = string.replace(escapeRegExpRe, '\\$&');
        if (flags !== undefined) {
          ret = new RegExp(ret, flags);
        }
        return ret;
      }
      /**
       * This method decodes HTML entities and returns the original HTML.
       *
       * See also {@link #function-encodeHtml-static}.
       * @param {String} str
       * @returns {String}
       * @category HTML
       */
      static decodeHtml(str) {
        return str && String(str).replace(htmlDecodeRe, htmlDecoder);
      }
      /**
       * This method encodes HTML entities and returns a string that can be placed in the document and produce the
       * original text rather than be interpreted as HTML. Using this method with user-entered values prevents those
       * values from executing as HTML (i.e., a cross-site scripting or "XSS" security issue).
       *
       * See also {@link #function-decodeHtml-static}.
       * @param {String} str
       * @returns {String}
       * @category HTML
       */
      static encodeHtml(str = '') {
        return str && String(str).replace(htmlEncodeRe, htmlEncoder);
      }
      /**
       * This method is similar to {@link #function-encodeHtml-static} except that `\n` and `\r` characters in the
       * given `str` are replaced by `<br>` tags _after_ first being encoded by {@link #function-encodeHtml-static}.
       * @param {String} str
       * @returns {String}
       * @category HTML
       * @internal
       */
      static encodeHtmlBR(str) {
        var _StringHelper$encodeH;
        return (_StringHelper$encodeH = StringHelper.encodeHtml(str)) === null || _StringHelper$encodeH === void 0 ? void 0 : _StringHelper$encodeH.replace(crlfRe, '<br>');
      }
      /**
       * Returns `true` if the provided `text` contains special HTML characters.
       * @param {String} text
       * @returns {Boolean}
       * @category HTML
       * @internal
       */
      static isHtml(text) {
        return typeof text === 'string' && htmlRe$1.test(text || '');
      }
      /**
       * Initializes HTML entities used by {@link #function-encodeHtml-static} and {@link #function-decodeHtml-static}.
       * @param {Object} [mappings] An object whose keys are characters that should be encoded and values are the HTML
       * entity for the character.
       * @private
       */
      static initHtmlEntities(mappings) {
        mappings = mappings || {
          '&': '&amp;',
          '>': '&gt;',
          '<': '&lt;',
          '"': '&quot;',
          "'": '&#39;'
        };
        const chars = Object.keys(mappings);
        // Maps '<' to '&lt;'
        charsToEncode = mappings;
        // Inverts the mapping so we can convert '&lt;' to '<'
        entitiesToDecode = chars.reduce((prev, val) => {
          prev[mappings[val]] = val;
          return prev;
        }, {});
        // Creates a regex char set like /([<&>])/g to match the characters that need to be encoded (escaping any of
        // the regex charset special chars '[', ']' and '-'):
        htmlEncodeRe = new RegExp(`([${chars.map(c => '[-]'.includes(c) ? '\\' + c : c).join('')}])`, 'g');
        // Creates a regex like /(&lt;|&amp;|&gt;)/ig to match encoded entities... good news is that (valid) HTML
        // entities do not contain any regex special characters:
        htmlDecodeRe = new RegExp(`(${Object.values(mappings).join('|')}|&#[0-9]+;)`, 'ig');
      }
      //endregion
      //region JSON
      /**
       * Parses JSON inside a try-catch block. Returns null if the string could not be parsed.
       *
       * @param {String} string String to parse
       * @returns {Object} Resulting object or `null` if parse failed
       * @category JSON
       */
      static safeJsonParse(string) {
        let parsed = null;
        try {
          parsed = JSON.parse(string);
        } catch (e) {}
        return parsed;
      }
      /**
       * Stringifies an object inside a try-catch block. Returns null if an exception is encountered.
       *
       * See [JSON.stringify on MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify)
       * for more information on the arguments.
       *
       * @param {Object} object The object to stringify
       * @param {Function|String[]|Number[]} [replacer] A function or array of string/number used to determine properties
       * to include in the JSON string
       * @param {String|Number} [space] Number of spaces to indent or string used as whitespace
       * @returns {String} Resulting object or null if stringify failed
       * @category JSON
       */
      static safeJsonStringify(object, replacer = null, space = null) {
        let result = null;
        try {
          result = JSON.stringify(object, replacer, space);
        } catch (e) {}
        return result;
      }
      //endregion
      //region Split & join
      /**
       * Joins all given paths together using the separator as a delimiter and normalizes the resulting path.
       * @param paths {Array} array of paths to join
       * @param pathSeparator [{String}] path separator. Default value is '/'
       * @returns {String}
       * @category Misc
       * @internal
       */
      static joinPaths(paths, pathSeparator = '/') {
        return paths.join(pathSeparator).replace(new RegExp('\\' + pathSeparator + '+', 'g'), pathSeparator);
      }
      /**
       * Returns the provided string split on whitespace. If the string is empty or consists of only whitespace, the
       * returned array will be empty. If `str` is not a string, it is simply returned. This allows `null` or already
       * split strings (arrays) to be passed through.
       *
       * For example:
       * ```
       *  console.log(StringHelper.split(' abc def xyz   '));
       *  > ['abc', 'def', 'xyz']
       *  console.log(StringHelper.split(''));
       *  > []
       * ```
       * Compare to the standard `split()` method:
       * ```
       *  console.log(' abc def xyz   '.split(/\s+/));
       *  > ['', 'abc', 'def', 'xyz', '']
       *  console.log(''.split(/\s+/));
       *  > ['']
       * ```
       * @param {String} str
       * @param {String|RegExp} delimiter
       * @returns {String[]}
       * @category Misc
       * @internal
       */
      static split(str, delimiter = whiteSpaceRe$1) {
        let ret = str;
        if (typeof ret === 'string') {
          ret = str.trim(); // w/o trim() whitespace on the ends will give us '' in the array
          ret = ret ? ret.split(delimiter) : []; // also ''.split() = ['']
        }

        return ret;
      }
      //endregion
      //region XSS
      /**
       * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
       * Scripting) attacks.
       *
       * For example:
       *
       * ```javascript
       *  eventRenderer(eventRecord) {
       *      return StringHelper.xss`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
       *  }
       * ```
       *
       * @param {TemplateStringsArray} strings The template string array
       * @param {...any} values The interpolated values in the template string
       * @returns {String} The encoded string
       * See {@link Core.helper.StringHelper#function-encodeHtml-static}.
       */
      static xss(strings, ...values) {
        const buf = [];
        let i = values.length;
        buf[i] = strings[i];
        while (i-- > 0) {
          buf[i] = strings[i] + StringHelper.encodeHtml(values[i]);
        }
        return buf.join('');
      }
      /**
       * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
       * Scripting) attacks. Unlike {@link Core.helper.StringHelper#function-xss-static}, this method converts `\n` and
       * `\r` characters into `<br>` tags.
       *
       * For example:
       *
       * ```javascript
       *  eventRenderer(eventRecord) {
       *      return StringHelper.xssBR`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
       *  }
       * ```
       *
       * See {@link Core.helper.StringHelper#function-encodeHtmlBR-static}.
       * @internal
       */
      static xssBR(strings, ...values) {
        const buf = [];
        let i = values.length;
        buf[i] = strings[i];
        while (i-- > 0) {
          buf[i] = strings[i] + StringHelper.encodeHtmlBR(values[i]);
        }
        return buf.join('');
      }
      //endregion
      //region JavaScript string
      /**
       * Converts a value to a JavaScript string (not JSON).
       *
       * For example a date to `"new Date(y, m, d)"`, an array to `"[...]"` etc.
       *
       * @param {*} value
       * @param {Object} [options]
       * @returns {String}
       * @internal
       */
      static toJavaScriptValue(value, options) {
        const type = Objects.typeOf(value);
        if (type === 'boolean' || type === 'string' || type === 'number' || value === null) {
          return StringHelper.safeJsonStringify(value);
        }
        if (value === globalThis) {
          return 'window';
        }
        if (type === 'date') {
          return `new Date(${value.getFullYear()}, ${value.getMonth()}, ${value.getDate()}, ${value.getHours()}, ${value.getMinutes()}, ${value.getSeconds()}, ${value.getMilliseconds()})`;
        }
        if (type === 'array') {
          return `[${value.map(v => StringHelper.toJavaScriptValue(v, options))}]`;
        }
        if (type === 'object' || type === 'instance') {
          return this.toJavaScriptString(value, options);
        }
        if (type === 'function') {
          let contents = value.toString();
          // async myFunction() {} => async function() {}
          if (contents.match(/^async (\w+?)\(/)) {
            contents = contents.replace(/^async (\w+?)\(/, 'async function(');
          }
          // Not an arrow fn? Replace name with function since we always add prop name prior to getting here
          // eventRenderer() {} -> function() {}
          else if (!contents.startsWith('async(') && contents.match(/^(\w+?)\(/)) {
            contents = contents.replace(/^(\w+?)\(/, 'function(');
          }
          return contents;
        }
        if (type === 'class') {
          if (value.toJavaScriptValue) {
            return value.toJavaScriptValue(options);
          }
          return Object.prototype.hasOwnProperty.call(value, '$name') ? value.$name : value.name;
        }
      }
      /**
       * Converts an object into a JavaScript string (not JSON).
       *
       * For example `{ a: 1, b: [2, 3] }` -> `"'{ a: 1, b: [2, 3] }'"`
       *
       * @param {Object} obj
       * @param {Object} [options]
       * @returns {String}
       * @internal
       */
      static toJavaScriptString(obj, options = {}) {
        const level = options.level ?? 0,
          intendSize = 2;
        // Not using template strings to have control over indentation
        return '{\n' + Object.keys(obj).map(key =>
        // All properties in an object are indented one step further than the object itself
        ' '.repeat((level + 1) * intendSize) + (key.match(/[- *]/) ? `"${key}"` : key) + `: ${StringHelper.toJavaScriptValue(obj[key], {
      ...options,
      level: level + 1
    })}`).join(',\n') +
        // Closing brace is indented to same level as the object
        '\n' + ' '.repeat(level * intendSize) + '}';
      }
      /**
       * Escapes " and \ in CSS attribute selectors, e.g. [data-id="somevalue"]
       *
       * Usage:
       * ```javascript
       * document.querySelector(StringHelper.cssAttributeQuery('data-id', 'id with & \\ chars'))
       * ```
       *
       * @param {String} attr
       * @param {String|Number} value
       * @returns {String}
       */
      static encodeAttributeSelector(attr, value) {
        return `[${attr}="${String(value).replace(/["\\]/g, '\\$&')}"]`;
      }
      //endregion
      //region UUID
      static fakeNodeUUIDIndex = 0;
      /**
       * Generates a UUID. Uses `Crypto.randomUUID()` if available, otherwise generates a random UUID using
       * `Crypto.getRandomValues()`.
       *
       * @returns {String}
       */
      static generateUUID() {
        var _globalThis$crypto;
        if (BrowserHelper.supportsRandomUUID) {
          return globalThis.crypto.randomUUID();
        }
        // Node does not have crypto built in
        if ((_globalThis$crypto = globalThis.crypto) !== null && _globalThis$crypto !== void 0 && _globalThis$crypto.getRandomValues) {
          // https://stackoverflow.com/questions/105034/how-do-i-create-a-guid-uuid
          return ([1e7] + -1e3 + -4e3 + -8e3 + -1e11).replace(/[018]/g, c => (c ^ globalThis.crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16));
        }
        // Node specific code, should never get here outside of node. Not actually a UUID, but should still be unique
        // enough for our purposes, as long as value is not serialized
        return `${Date.now()}-${++StringHelper.fakeNodeUUIDIndex}`;
      }
      //endregion
    }

    StringHelper.initHtmlEntities();
    StringHelper._$name = 'StringHelper';

    // IMPORTANT - adding imports here can create problems for Base class
    /**
     * @module Core/helper/util/Objects
     */
    const {
        hasOwnProperty: hasOwnProperty$5,
        toString: toString$3
      } = Object.prototype,
      {
        isFrozen
      } = Object,
      afterRe = /\s*<\s*/,
      beforeRe = /\s*>\s*/,
      blendOptions = {},
      typeCache = {},
      emptyObject$d = Object.freeze({});
    // Static methods are not displayed in derived class documentation. Therefore, since this is an internal class, the
    // workaround is to copy method documentation to ObjectHelper (the public interface). Also tried making ObjectHelper
    // a singleton.
    /**
     * Helper for low-level Object manipulation.
     *
     * While documented on {@link Core.helper.ObjectHelper}, the following static methods are implemented by this class:
     *
     * - `{@link Core.helper.ObjectHelper#function-assign-static}`
     * - `{@link Core.helper.ObjectHelper#function-assignIf-static}`
     * - `{@link Core.helper.ObjectHelper#function-clone-static}`
     * - `{@link Core.helper.ObjectHelper#function-createTruthyKeys-static}`
     * - `{@link Core.helper.ObjectHelper#function-getPath-static}`
     * - `{@link Core.helper.ObjectHelper#function-getTruthyKeys-static}`
     * - `{@link Core.helper.ObjectHelper#function-getTruthyValues-static}`
     * - `{@link Core.helper.ObjectHelper#function-isEmpty-static}`
     * - `{@link Core.helper.ObjectHelper#function-isObject-static}`
     * - `{@link Core.helper.ObjectHelper#function-merge-static}`
     * - `{@link Core.helper.ObjectHelper#function-setPath-static}`
     * - `{@link Core.helper.ObjectHelper#function-typeOf-static}`
     * @internal
     */
    class Objects {
      static assign(dest, ...sources) {
        for (let source, key, i = 0; i < sources.length; i++) {
          source = sources[i];
          if (source) {
            for (key in source) {
              dest[key] = source[key];
            }
          }
        }
        return dest;
      }
      static assignIf(dest, ...sources) {
        for (let source, key, i = 0; i < sources.length; i++) {
          source = sources[i];
          if (source) {
            for (key in source) {
              if (!(key in dest) || dest[key] === undefined) {
                dest[key] = source[key];
              }
            }
          }
        }
        return dest;
      }
      static blend(dest, source, options) {
        options = options || blendOptions;
        dest = dest || {};
        const {
          clone = Objects.clone,
          merge = Objects.blend
        } = options;
        if (Array.isArray(source)) {
          if (source.length > 1) {
            source.forEach(s => {
              dest = Objects.blend(dest, s, options);
            });
            return dest;
          }
          source = source[0];
        }
        if (source) {
          let destValue, key, value;
          for (key in source) {
            value = source[key];
            if (value && Objects.isObject(value)) {
              destValue = dest[key];
              options.key = key;
              if (destValue && Objects.isObject(destValue)) {
                if (isFrozen(destValue)) {
                  dest[key] = destValue = clone(destValue, options);
                }
                value = merge(destValue, value, options);
              } else {
                // We don't need to clone frozen objects, but we do clone mutable objects as they get
                // applied to the dest.
                value = isFrozen(value) ? value : clone(value, options);
              }
            }
            dest[key] = value;
          }
        }
        return dest;
      }
      static clone(value, handler) {
        let cloned = value,
          key;
        if (value && typeof value === 'object') {
          const options = handler && typeof handler === 'object' && handler;
          if (options) {
            // When using blend(), the 2nd argument is the options object, so ignore that case
            handler = null;
          }
          if (Objects.isObject(value)) {
            // When using DomSync, DomConfigs are usually recreated from scratch on each sync, we allow opting out
            // of cloning them (costly for many elements)
            if (value.skipClone) {
              cloned = value;
            } else {
              cloned = {};
              for (key in value) {
                cloned[key] = Objects.clone(value[key]);
              }
            }
          } else if (Array.isArray(value)) {
            cloned = [];
            // Loop backwards to:
            //  1. read source.length once
            //  2. get result array sized on first pass (avoid growing)
            for /* empty */
            (key = value.length; key-- > 0;) {
              cloned[key] = Objects.clone(value[key]);
            }
          } else if (Objects.isDate(value)) {
            cloned = new Date(value.getTime());
          } else if (handler) {
            // Allow other types to be handled (e.g., DOM nodes).
            cloned = handler(value);
          }
        }
        return cloned;
      }
      static createTruthyKeys(source) {
        const keys = StringHelper.split(source),
          result = keys && {};
        if (keys) {
          for (const key of keys) {
            // StringHelper.split won't return empty keys if passed a string, but we
            // could have been passed a String[]
            if (key) {
              result[key] = true;
            }
          }
        }
        return result;
      }
      /**
       * Returns value for a given path in the object
       * @param {Object} object Object to check path on
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @returns {*} Value associated with passed key
       */
      static getPath(object, path) {
        return path.split('.').reduce((result, key) => {
          return (result || emptyObject$d)[key];
        }, object);
      }
      /**
       * Returns value for a given path in the object, placing a passed default value in at the
       * leaf property and filling in undefined properties all the way down.
       * @param {Object} object Object to get path value for.
       * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
       * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
       * @param {*} [defaultValue] Optionally the value to put in as the `someKey` property.
       * @returns {*} Value at the leaf position of the path.
       */
      static getPathDefault(object, path, defaultValue) {
        const keys = Array.isArray(path) ? path : typeof path === 'string' ? path.split('.') : [path],
          length = keys.length - 1;
        return keys.reduce((result, key, index) => {
          if (defaultValue && !(key in result)) {
            // Can't use emptyObject here, we are creating a node in the object tree
            result[key] = index === length ? defaultValue : {};
          }
          return (result || emptyObject$d)[key];
        }, object);
      }
      /**
       * Determines if the specified path exists
       * @param {Object} object Object to check path on
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @returns {Boolean}
       */
      static hasPath(object, path) {
        return path.split('.').every(key => {
          if (object && key in object) {
            object = object[key];
            return true;
          }
          return false;
        });
      }
      static getTruthyKeys(source) {
        const keys = [];
        for (const key in source) {
          if (source[key]) {
            keys.push(key);
          }
        }
        return keys;
      }
      static getTruthyValues(source) {
        const values = [];
        for (const key in source) {
          if (source[key]) {
            values.push(source[key]);
          }
        }
        return values;
      }
      static isClass(object) {
        var _object$prototype;
        if (typeof object === 'function' && ((_object$prototype = object.prototype) === null || _object$prototype === void 0 ? void 0 : _object$prototype.constructor) === object) {
          return true;
        }
        return false;
      }
      static isDate(object) {
        // A couple quick rejections but only sure way is typeOf:
        return Boolean(object === null || object === void 0 ? void 0 : object.getUTCDate) && Objects.typeOf(object) === 'date';
      }
      /**
       * Check if passed object is a Promise or contains `then` method.
       * Used to fix problems with detecting promises in code with `instance of Promise` when
       * Promise class is replaced with any other implementation like `ZoneAwarePromise` in Angular.
       * Related to these issues:
       * https://github.com/bryntum/support/issues/791
       * https://github.com/bryntum/support/issues/2990
       *
       * @param {Object} object object to check
       * @returns {Boolean} truthy value if object is a Promise
       * @internal
       */
      static isPromise(object) {
        if (Promise && Promise.resolve) {
          return Promise.resolve(object) === object || typeof (object === null || object === void 0 ? void 0 : object.then) === 'function';
        }
        throw new Error('Promise not supported in your environment');
      }
      static isEmpty(object) {
        if (object && typeof object === 'object') {
          // noinspection LoopStatementThatDoesntLoopJS
          for (const p in object) {
            // eslint-disable-line no-unused-vars,no-unreachable-loop
            return false;
          }
        }
        return true;
      }
      static isObject(value) {
        const C = value === null || value === void 0 ? void 0 : value.constructor;
        return Boolean(C
        // Most things have a .constructor property
        ?
        // An in-frame instance of Object
        C === Object ||
        // Detect cross-frame objects, but exclude instance of custom classes named Object. typeOf(value) is
        // "object" even for instances of a class and typeOf(C) is "function" for all constructors. We'll have
        // to settle for relying on the fact that getPrototypeOf(Object.prototype) === null.
        // NOTE: this issue does come up in Scheduler unit tests at least.
        C.getPrototypeOf && C.prototype && !Object.getPrototypeOf(C.prototype)

        // Since all classes have a constructor property, an object w/o one is likely from Object.create(null). Of
        // course, primitive types do not have ".constructor"
        : value && typeof value === 'object');
      }
      static isInstantiated(object) {
        return object ? typeof object === 'object' && !Objects.isObject(object) : false;
      }
      static merge(dest, ...sources) {
        return Objects.blend(dest, sources);
      }
      /**
       * Merges two "items" objects. An items object is a simple object whose keys act as identifiers and whose values
       * are "item" objects. An item can be any object type. This method is used to merge such objects while maintaining
       * their property order. Special key syntax is used to allow a source object to insert a key before or after a key
       * in the `dest` object.
       *
       * For example:
       * ```javascript
       *  let dest = {
       *      foo : {},
       *      bar : {},
       *      fiz : {}
       *  }
       *
       *  console.log(Object.keys(dest));
       *  > ["foo", "bar", "fiz"]
       *
       *  dest = mergeItems(dest, {
       *      'zip > bar' : {}    // insert "zip" before "bar"
       *      'bar < zap' : {}    // insert "zap" after "bar"
       *  });
       *
       *  console.log(Object.keys(dest));
       *  > ["foo", "zip", "bar", "zap", "fiz"]
       * ```
       *
       * @param {Object} dest The destination object.
       * @param {Object|Object[]} src The source object or array of source objects to merge into `dest`.
       * @param {Object} [options] The function to use to merge items.
       * @param {Function} [options.merge] The function to use to merge items.
       * @returns {Object} The merged object. This will be the `dest` object.
       * @internal
       */
      static mergeItems(dest, src, options) {
        options = options || blendOptions;
        let anchor, delta, index, indexMap, key, shuffle, srcVal;
        const {
          merge = Objects.blend
        } = options;
        dest = dest || {};
        if (Array.isArray(src)) {
          src.forEach(s => {
            dest = Objects.mergeItems(dest, s, options);
          });
        } else if (src) {
          // https://2ality.com/2015/10/property-traversal-order-es6.html
          // Bottom line: Object keys are iterated in declared/insertion order... unless the key is an integer or
          // Symbol, but we don't care about those generally.
          for (key in src) {
            srcVal = src[key];
            anchor = null;
            // Allow a key to be added before or after another:
            //
            //  {
            //      'foo > bar' : {
            //          ...
            //      },
            //      'bar < derp' : {
            //          ...
            //      }
            //  }
            //
            // The goal above is to add a 'foo' key before the existing 'bar' key while adding a 'derp' key after
            // 'bar'.
            if (key.includes('>')) {
              [key, anchor] = key.split(beforeRe);
              delta = 0;
            } else if (key.includes('<')) {
              [anchor, key] = key.split(afterRe);
              delta = 1;
            }
            if (key in dest) {
              // Changing the value of a key does not change its iteration order. Since "key in dest" we can do
              // what we need directly.
              if (srcVal && dest[key] && merge) {
                options.key = key;
                srcVal = merge(dest[key], srcVal, options);
              }
              dest[key] = srcVal;
            } else if (!anchor) {
              var _indexMap;
              dest[key] = srcVal;
              (_indexMap = indexMap) === null || _indexMap === void 0 ? void 0 : _indexMap.set(key, indexMap.size);
            } else {
              // Lazily sprout the item index map. When we first merge an item into an items object, we create this
              // Map to control the ordering. This is because any keys we add would necessarily be iterated after
              // the original properties.
              if (!indexMap) {
                indexMap = new Map();
                index = 0;
                for (const k in dest) {
                  indexMap.set(k, index++);
                }
              }
              index = indexMap.get(anchor);
              dest[key] = srcVal;
              if (index == null && delta) {
                index = indexMap.size;
              } else {
                shuffle = shuffle || [];
                index = (index || 0) + delta;
                // Adjust all key indices >= index up by 1 to maintain integer indices (required by the above
                // use case).
                for (const item of indexMap) {
                  const [k, v] = item;
                  if (index <= v) {
                    /*
                    Consider object w/the following order:
                        {
                            foo : {}',
                            bar : {},
                            baz : {},
                            zip : {},
                            goo : {},
                            fiz : {}
                        }
                    The indexMap is:
                        foo : 0
                        bar : 1
                        baz : 2
                        zip : 3
                        goo : 4
                        fiz : 5
                    To insert before goo, we populate shuffle thusly (to set up for popping):
                        +-----+-----+
                        | fiz | goo |
                        +-----+-----+
                          0        1
                          =6-5-1   =6-4-1
                    */
                    shuffle && (shuffle[indexMap.size - v - 1] = k);
                    indexMap.set(k, v + 1);
                  }
                }
                // Delete and re-add the keys that should follow the new key to establish the iteration order
                // we need:
                if (shuffle) {
                  while (shuffle.length) {
                    const k = shuffle.pop(),
                      v = dest[k];
                    delete dest[k];
                    dest[k] = v;
                  }
                }
              }
              indexMap.set(key, index);
            }
          }
        }
        return dest;
      }
      /**
       * Sets value for a given path in the object
       * @param {Object} object Target object
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @param {*} value Value for a given path
       * @returns {Object} Returns passed object
       */
      static setPath(object, path, value) {
        path.split('.').reduce((result, key, index, array) => {
          const isLast = index === array.length - 1;
          if (isLast) {
            return result[key] = value;
          } else if (!(result[key] instanceof Object)) {
            result[key] = {};
          }
          return result[key];
        }, object);
        return object;
      }
      static typeOf(value) {
        let trueType, type;
        if (value === null) {
          type = 'null';
        }
        // NaN is the only value that is !== to itself
        else if (value !== value) {
          // eslint-disable-line no-self-compare
          type = 'nan';
        } else {
          type = typeof value;
          if (type === 'object') {
            if (value.isBase) {
              type = 'instance';
            } else if (Array.isArray(value)) {
              type = 'array';
            } else if (!(type = typeCache[trueType = toString$3.call(value)])) {
              typeCache[trueType] = type = trueType.slice(8, -1).toLowerCase(); // '[object Date]' => 'date'
            }
          } else if (type === 'function' && value.isBase) {
            type = 'class';
          }
        }
        return type;
      }
    }
    Object.defineProperty(Objects, 'hasOwn', {
      // When available, this avoids an extra layer of function call around it:
      value: Object.hasOwn || ((object, property) => hasOwnProperty$5.call(object, property))
    });
    Objects._$name = 'Objects';

    /**
     * @module Core/helper/VersionHelper
     */
    /**
     * Helper for version handling
     * @private
     * @example
     *
     * VersionHelper.setVersion('grid', '1.5');
     *
     * if (VersionHelper.getVersion('grid').isNewerThan('1.0')) {
     *   ...
     * }
     */
    class VersionHelper {
      /**
       * Set version for specified product
       * @private
       * @param {String} product
       * @param {String} version
       */
      static setVersion(product, version) {
        product = product.toLowerCase();
        VH[product] = {
          version,
          isNewerThan(otherVersion) {
            return VersionHelper.semanticCompareVersion(otherVersion, version, '<');
          },
          isOlderThan(otherVersion) {
            return VersionHelper.semanticCompareVersion(otherVersion, version, '>');
          }
        };
        let bundleFor = '';
        // Var productName is only defined in bundles, it is internal to bundle so not available on window. Used to
        // tell importing combinations of grid/scheduler/gantt bundles apart from loading same bundle twice
        if (typeof productName !== 'undefined') {
          // eslint-disable-next-line no-undef
          bundleFor = productName;
        }
        // Set "global" flag to detect bundle being loaded twice
        const globalKey = `${bundleFor}.${product}${version.replace(/\./g, '-')}`;
        if (BrowserHelper.isBrowserEnv && !globalThis.bryntum.silenceBundleException) {
          if (globalThis.bryntum[globalKey] === true) {
            if (this.isTestEnv) {
              globalThis.BUNDLE_EXCEPTION = true;
            } else {
              let errorProduct = bundleFor || product;
              if (errorProduct === 'core') {
                errorProduct = 'grid';
              }
              let capitalized = StringHelper.capitalize(errorProduct);
              if (errorProduct === 'schedulerpro') {
                capitalized = 'SchedulerPro';
              }
              throw new Error(`The Bryntum ${capitalized} bundle was loaded multiple times by the application.\n\n` + `Common reasons you are getting this error includes:\n\n` + `* Imports point to different types of the bundle (e.g. *.module.js and *.umd.js)\n` + `* Imports point to both sources and bundle\n` + `* Imports do not use the shortest relative path, JS treats them as different files\n` + `* Cache busters differ between imports, JS treats ${errorProduct}.module.js?1 and ${errorProduct}.module.js?2 as different files\n` + `* Imports missing file type, verify they all end in .js\n\n` + `See https://bryntum.com/products/${errorProduct}/docs/guide/${capitalized}/gettingstarted/es6bundle#troubleshooting for more information\n\n`);
            }
          } else {
            globalThis.bryntum[globalKey] = true;
          }
        }
      }
      /**
       * Get (previously set) version for specified product
       * @private
       * @param {String} product
       */
      static getVersion(product) {
        product = product.toLowerCase();
        if (!VH[product]) {
          throw new Error('No version specified! Please check that you import VersionHelper correctly into the class from where you call `deprecate` function.');
        }
        return VH[product].version;
      }
      /**
       * Checks the version1 against the passed version2 using the comparison operator.
       * Supports `rc`, `beta`, `alpha` release states. Eg. `1.2.3-alpha-1`.
       * State which is not listed above means some version below `alpha`.
       * @param {String} version1 The version to test against
       * @param {String} version2 The version to test against
       * @param {String} [comparison] The comparison operator, `<=`, `<`, `=`, `>` or `>=`.
       * @returns {Boolean} `true` if the test passes.
       * @internal
       */
      static semanticCompareVersion(version1, version2, comparison = '=') {
        version1 = version1 || '';
        version2 = version2 || '';
        const version1Arr = version1.split(/[-.]/),
          version2Arr = version2.split(/[-.]/),
          isLower = comparison.includes('<'),
          normalizeArr = (arr, maxLength) => {
            const states = ['rc', 'beta', 'alpha'],
              result = arr.map(v => {
                if (states.includes(v)) {
                  return -states.indexOf(v) - 2;
                }
                const res = Number.parseInt(v);
                return Number.isNaN(res) ? -states.length : res;
              });
            while (result.length < maxLength) {
              result.push(-1);
            }
            return result;
          },
          compareArr = () => {
            const maxLength = Math.max(version1Arr.length, version2Arr.length),
              arr1 = normalizeArr(version1Arr, maxLength),
              arr2 = normalizeArr(version2Arr, maxLength);
            for (let i = 0; i < maxLength; i++) {
              if (arr1[i] !== arr2[i]) {
                return isLower ? arr1[i] < arr2[i] : arr1[i] > arr2[i];
              }
            }
            return true;
          };
        switch (comparison) {
          case '=':
            return version1 === version2;
          case '<=':
          case '>=':
            return version1 === version2 || compareArr();
          case '<':
          case '>':
            return version1 !== version2 && compareArr();
        }
        return false;
      }
      /**
       * Checks the passed product against the passed version using the passed test.
       * @param {String} product The name of the product to test the version of
       * @param {String} version The version to test against
       * @param {String} operator The test operator, `<=`, `<`, `=`, `>` or `>=`.
       * @returns {Boolean} `true` if the test passes.
       * @internal
       */
      static checkVersion(product, version, operator) {
        return VersionHelper.semanticCompareVersion(VH.getVersion(product), version, operator);
      }
      /**
       * Based on a comparison of current product version and the passed version this method either outputs a console.warn
       * or throws an error.
       * @param {String} product The name of the product
       * @param {String} invalidAsOfVersion The version where the offending code is invalid (when any compatibility layer
       * is actually removed).
       * @param {String} message Required! A helpful warning message to show to the developer using a deprecated API.
       * @internal
       */
      static deprecate(product, invalidAsOfVersion, message) {
        const justWarn = VH.checkVersion(product, invalidAsOfVersion, '<');
        if (justWarn) {
          // During the grace period (until the next major release following the deprecated code), just show a console warning
          console.warn(`Deprecation warning: You are using a deprecated API which will change in v${invalidAsOfVersion}. ${message}`);
        } else {
          throw new Error(`Deprecated API use. ${message}`);
        }
      }
      /**
       * Returns truthy value if environment is in testing mode
       * @returns {Boolean}
       * @internal
       **/
      static get isTestEnv() {
        var _globalThis$bryntum;
        const isTestEnv = Boolean((_globalThis$bryntum = globalThis.bryntum) === null || _globalThis$bryntum === void 0 ? void 0 : _globalThis$bryntum.isTestEnv);
        try {
          var _globalThis$parent, _globalThis$parent$br;
          return isTestEnv || Boolean((_globalThis$parent = globalThis.parent) === null || _globalThis$parent === void 0 ? void 0 : (_globalThis$parent$br = _globalThis$parent.bryntum) === null || _globalThis$parent$br === void 0 ? void 0 : _globalThis$parent$br.isTestEnv);
        } catch (e) {
          // Accessing parent may cause CORS violation
          return isTestEnv;
        }
      }
      static get isDebug() {
        let result = false;
        return result;
      }
    }
    const VH = VersionHelper;
    if (BrowserHelper.isBrowserEnv) {
      if (VH.isTestEnv) {
        BrowserHelper._isHoverableDevice = true;
      }
      globalThis.bryntum = Object.assign(globalThis.bryntum || {}, {
        getVersion: VH.getVersion.bind(VH),
        checkVersion: VH.checkVersion.bind(VH),
        deprecate: VH.deprecate.bind(VH),
        license: '9c07f388-a9e9-11ec-9e5c-d094663d5c88'
      });
    }
    VersionHelper._$name = 'VersionHelper';

    // We cannot import ObjectHelper because of the import cycle:
    //  ObjectHelper -> DateHelper -> LocaleManager -> Base -> us
    /**
     * @module Core/Config
     */
    const {
        defineProperty: defineProperty$8,
        getOwnPropertyDescriptor: getOwnPropertyDescriptor$1
      } = Reflect,
      {
        hasOwnProperty: hasOwnProperty$4,
        toString: toString$2
      } = Object.prototype,
      instancePropertiesSymbol$1 = Symbol('instanceProperties'),
      configuringSymbol$1 = Symbol('configuring'),
      lazyConfigValues = Symbol('lazyConfigValues'),
      DATE_TYPE$1 = toString$2.call(new Date()),
      whitespace = /\s+/,
      createClsProps = (result, cls) => {
        result[cls] = 1;
        return result;
      };
    /**
     * This class holds the description of a config property. Only one instance of this class is needed for each config
     * name (e.g., "text"). If config options are supplied, however, they also contribute to the cached instance.
     *
     * Instances should always be retrieved by calling `Config.get()`.
     *
     * The **Configs** of this class correspond to `options` that can be supplied to the `get()` method. These affect the
     * behavior of the config property in some way, as descried by their respective documentation.
     *
     * This class is not used directly.
     *
     * ## The Setter
     * The primary functionality provided by `Config` is its standard setter. This setter function ensures consistent
     * behavior when modifying config properties.
     *
     * The standard setter algorithm is as follows (using the `'text'` config for illustration):
     *
     *  - If the class defines a `changeText()` method, call it passing the new value and the current value:
     *    `changeText(newText, oldText)`.<br>
     *    Then:
     *    * If `changeText()` exits without returning a value (i.e., `undefined`), exit and do nothing further. The
     *      assumption is that the changer method has done all that is required.
     *    * Otherwise, the return value of `changeText()` replaces the incoming value passed to the setter.
     *  - If the new value (or the value returned by `changeText()`) is `!==` to the current value:
     *    * Update the stored config value in `this._text`.
     *    * If the class defines an `updateText()` method, call it passing the new value and the previous value.
     *      `updateText(newText, oldText)`
     *    * If the class defines an `onConfigChange()` method, call it passing an object with the following properties:
     *        - `name` - The config's name
     *        - `value` - The new value
     *        - `was` - The previous value
     *        - `config` - The `Config` instance.
     *
     * NOTE: unlike `changeText()` and `updateText()`, the name of the `onConfigChange()` method is unaffected by the
     * config's name.
     *
     * @internal
     */
    class Config {
      /**
       * Returns the `Config` instance for the given `name` and `options`.
       * @param {String} name The name of the config (e.g., 'text' for the text config).
       * @param {Object} [options] Config behavior options.
       * @returns {Core.Config}
       * @internal
       */
      static get(name, options) {
        const {
            cache
          } = this,
          baseCfg = cache[name] || (cache[name] = new Config(name));
        let cfg = baseCfg,
          key;
        if (options) {
          key = Config.makeCacheKey(name, options);
          if (!(cfg = key && cache[key])) {
            cfg = baseCfg.extend(options);
            if (key) {
              cache[key] = cfg;
            }
          }
        }
        return cfg;
      }
      constructor(name) {
        const me = this,
          cap = name[0].toUpperCase() + name.substr(1);
        me.base = me; // so extend()ed configs have a link to the base definition
        me.name = name;
        me.field = '_' + name;
        me.capName = cap;
        me.changer = 'change' + cap;
        me.initializing = 'initializing' + cap;
        me.updater = 'update' + cap;
      }
      /**
       * The descriptor to use with `Reflect.defineProperty()` for defining this config's getter and setter.
       * @property {Object}
       * @private
       */
      get descriptor() {
        let descriptor = this._descriptor;
        if (!descriptor || !hasOwnProperty$4.call(this, '_descriptor')) {
          // lazily make the descriptor
          this._descriptor = descriptor = this.makeDescriptor();
        }
        return descriptor;
      }
      /**
       * The descriptor to use with `Reflect.defineProperty()` for defining this config's initter.
       * @property {Object}
       * @private
       */
      get initDescriptor() {
        let descriptor = this._initDescriptor;
        if (!descriptor || !hasOwnProperty$4.call(this, '_initDescriptor')) {
          // lazily make the descriptor
          this._initDescriptor = descriptor = this.makeInitter();
        }
        return descriptor;
      }
      /**
       * This method compares two values for semantic equality. By default, this is based on the `===` operator. This
       * is often overridden for configs that accept `Date` or array values.
       * @param {*} value1
       * @param {*} value2
       * @returns {Boolean}
       * @internal
       */
      equal(value1, value2) {
        return value1 === value2;
      }
      /**
       * Extends this config with a given additional set of options. These objects are just prototype extensions of this
       * instance.
       * @param {Object} options
       * @returns {Core.Config}
       * @internal
       */
      extend(options) {
        const cfg = Object.assign(Object.create(this), options),
          {
            equal,
            merge
          } = options,
          {
            equalityMethods
          } = Config;
        if (typeof equal === 'string') {
          if (equal.endsWith('[]')) {
            cfg.equal = Config.makeArrayEquals(equalityMethods[equal.substr(0, equal.length - 2)]);
          } else {
            cfg.equal = equalityMethods[equal];
          }
        }
        if (typeof merge === 'string') {
          // Base uses { merge : 'replace' } for defaultConfig properties
          cfg.merge = Config.mergeMethods[merge];
        }
        return cfg;
      }
      /**
       * Defines the property on a given target object via `Reflect.defineProperty()`. If the object has its own getter,
       * it will be preserved. It is invalid to define a setter.
       * @param {Object} target
       * @internal
       */
      define(target) {
        const existing = getOwnPropertyDescriptor$1(target, this.name);
        let descriptor = this.descriptor;
        if (existing && existing.get) {
          descriptor = Object.assign({}, descriptor);
          descriptor.get = existing.get;
        }
        defineProperty$8(target, this.name, descriptor);
      }
      /**
       * Defines the property initter on the `target`. This is a property getter/setter that propagates the configured
       * value when the property is read.
       * @param {Object} target
       * @param {*} value
       * @internal
       */
      defineInitter(target, value) {
        const {
            name
          } = this,
          properties = target[instancePropertiesSymbol$1];
        let lazyValues, prop;
        // If there is an existing property with a getter/setter, *not* a value
        // defined on the object for this config we must call it in our injected getter/setter.
        if (!properties[name] && ( /* assign */prop = getOwnPropertyDescriptor$1(target, name)) && !('value' in prop)) {
          properties[name] = prop;
        }
        // Set up a temporary instance property which will pull in the value from the initialConfig if the getter
        // is called first.
        defineProperty$8(target, name, this.initDescriptor);
        if (this.lazy) {
          lazyValues = target[lazyConfigValues] || (target[lazyConfigValues] = new Map());
          lazyValues.set(name, value);
        }
      }
      /**
       * Returns an equality function for arrays of a base type, for example `'date'`.
       * @param {Function} [fn] The function to use to compare array elements. By default, operator `===` is used.
       * @returns {Function}
       * @private
       */
      static makeArrayEquals(fn) {
        return (value1, value2) => {
          let i,
            equal = value1 && value2 && value1.length === (i = value2.length);
          if (equal && Array.isArray(value1) && Array.isArray(value2)) {
            if (fn) {
              while (equal && i-- > 0) {
                equal = fn(value1[i], value2[i]);
              }
            } else {
              while (equal && i-- > 0) {
                equal = value1[i] === value2[i];
              }
            }
          } else {
            equal = fn ? fn(value1, value2) : value1 === value2;
          }
          return equal;
        };
      }
      /**
       * Returns the key to use in the Config `cache`.
       * @param {String} name The name of the config property.
       * @param {Object} options The config property options.
       * @returns {String}
       * @private
       */
      static makeCacheKey(name, options) {
        const keys = Object.keys(options).sort();
        for /* empty */
        (let key, type, value, i = keys.length; i-- > 0;) {
          value = options[key = keys[i]];
          if (value == null && value === false) {
            keys.splice(i, 1);
          } else {
            type = typeof value;
            if (type === 'function') {
              return null;
            }
            if (type === 'string') {
              keys[i] = `${key}:"${value}"`;
            } else if (type === 'number') {
              keys[i] = `${key}:${value}`;
            }
            // that leaves bool and object, but there are no (valid) config options that are objects... so ignore
          }
        }

        return keys.length ? `${name}>${keys.join('|')}` : name; // eg: 'text>render|merge:v => v|bar'
      }
      /**
       * Creates and returns a property descriptor for this config suitable to be passed to `Reflect.defineProperty()`.
       * @returns {Object}
       * @private
       */
      makeDescriptor() {
        const config = this,
          {
            base,
            field,
            changer,
            updater,
            name
          } = config;
        if (base !== config && base.equal === config.equal) {
          // At present only the equal option affects the setter, so all configs can share the
          // descriptor of the base-most config definition unless their equality test fns differ.
          return base.descriptor;
        }
        return {
          get() {
            var _this$configObserver;
            // Allow folks like Widget.compose() to monitor getter calls
            (_this$configObserver = this.configObserver) === null || _this$configObserver === void 0 ? void 0 : _this$configObserver.get(name, this);
            return this[field];
          },
          set(value) {
            const me = this;
            let was = me[field],
              applied,
              newValue;
            // Resolve values starting with 'up.' by traversing owners to find it
            if (typeof value === 'string') {
              let resolvedValue = value;
              if (value.startsWith('up.')) {
                var _me$owner;
                resolvedValue = (_me$owner = me.owner) === null || _me$owner === void 0 ? void 0 : _me$owner.resolveProperty(value.substr(3));
              } else if (value.startsWith('this.')) {
                resolvedValue = me.resolveProperty(value.substr(5));
              }
              if (resolvedValue !== undefined && typeof resolvedValue !== 'function') {
                value = resolvedValue;
              }
            }
            // If the "changeTitle()" fellow falls off the end, it must have changed all the needful things.
            // Otherwise, it returned the final config value (it may have changed it instead, for example, making
            // an instance from a config object).
            if (me[changer]) {
              applied = (newValue = me[changer](value, was)) === undefined;
              if (!applied) {
                value = newValue;
                was = me[field]; // in case it was modified by the changer fn...
              }
            }
            // inline the default equal() for better perf:
            if (!applied && !(config.equal === equal ? was === value : config.equal(was, value))) {
              var _me$updater;
              me[field] = value;
              applied = true;
              // Check for a "updateTitle()" method and call it if present.
              (_me$updater = me[updater]) === null || _me$updater === void 0 ? void 0 : _me$updater.call(me, value, was);
            }
            // Trigger config change if the value changed, and updater did not lead to our destruction
            if (applied && !me.isDestroyed && !me.onConfigChange.$nullFn) {
              me.onConfigChange({
                name,
                value,
                was,
                config
              });
            }
          }
        };
      }
      /**
       * Creates and returns a property descriptor for this config's initter suitable to pass to
       * `Reflect.defineProperty()`.
       * @returns {Object}
       * @private
       */
      makeInitter() {
        const config = this;
        if (config !== config.base) {
          if (config.lazy) {
            return config.makeLazyInitter();
          }
          // At present no other options affect the setter, so all configs can share the descriptor of the base-most
          // config definition.
          return config.base.initDescriptor;
        }
        return config.makeBasicInitter();
      }
      makeBasicInitter() {
        const config = this,
          {
            initializing,
            name
          } = config;
        return {
          configurable: true,
          get() {
            const me = this;
            config.removeInitter(me);
            // Set the value from the configuration.
            me[initializing] = true;
            me[name] = me[configuringSymbol$1][name];
            me[initializing] = false;
            // The property has been *pulled* from the configuration.
            // Prevent the setting loop in configure from setting it again.
            me.configDone[name] = true;
            // Finally, allow the prototype getter to return the value.
            return me[name];
          },
          set(value) {
            config.removeInitter(this);
            // The config has been set (some internal code may have called the setter)
            // so prevent it from being called again and overwritten with data from initialConfig.
            this.configDone[name] = true;
            // Set the property normally (Any prototype setter will be invoked)
            this[name] = value;
          }
        };
      }
      makeLazyInitter() {
        const config = this,
          {
            initializing,
            name
          } = config;
        return {
          configurable: true,
          get() {
            const me = this,
              value = me[lazyConfigValues].get(name);
            config.removeInitter(me);
            if (!me.isDestroying) {
              // Set the value from the lazy config object.
              me[initializing] = true;
              me[name] = value;
              me[initializing] = false;
            }
            // Finally, allow the prototype getter to return the value.
            return me[name];
          },
          set(value) {
            config.removeInitter(this);
            // Set the property normally (Any prototype setter will be invoked)
            this[name] = value;
          }
        };
      }
      /**
       * Removes the property initter and restores the instance to its original form.
       * @param {Object} instance
       * @private
       */
      removeInitter(instance) {
        const {
            name
          } = this,
          instanceProperty = instance[instancePropertiesSymbol$1][name],
          lazyValues = instance[lazyConfigValues];
        // If we took over from an instance property, replace it
        if (instanceProperty) {
          defineProperty$8(instance, name, instanceProperty);
        }
        // Otherwise just delete the instance property who's getter we are in.
        else {
          delete instance[name];
        }
        if (lazyValues !== null && lazyValues !== void 0 && lazyValues.delete(name) && !lazyValues.size) {
          // we delete the keys so that we can tell if this particular lazy config has been initialized
          delete instance[lazyConfigValues];
        }
      }
      setDefault(cls, value) {
        defineProperty$8(cls.prototype, this.field, {
          configurable: true,
          writable: true,
          // or else "this._value = x" will fail
          value
        });
      }
      /**
       * This method combines (merges) two config values. This is called in two cases:
       *
       *  - When a derived class specifies the value of a config defined in a super class.
       *  - When a value is specified in the instance config object.
       *
       * @param {*} newValue In the case of derived classes, this is the config value of the derived class. In the case
       * of the instance config, this is the instance config value.
       * @param {*} currentValue In the case of derived classes, this is the config value of the super class. In the case
       * of the instance config, this is the class config value.
       * @param {Object} metaNew The class meta object from which the `newValue` is coming. This parameter is `null` if
       * the `newValue` is from an instance configuration.
       * @param {Object} metaCurrent The class meta object from which the `currentValue` is coming. This parameter is
       * `null` if the `currentValue` is not from a class configuration.
       * @returns {*}
       * @internal
       */
      merge(newValue, currentValue) {
        if (currentValue && newValue && Objects.isObject(newValue)) {
          // If existing value is a class instance, clone and merge won't work. Set the configs.
          if (currentValue.isBase) {
            return currentValue.setConfig(newValue);
          }
          if (Objects.isObject(currentValue)) {
            newValue = Objects.merge(Objects.clone(currentValue), newValue);
          }
        }
        return newValue;
      }
    }
    const {
        prototype
      } = Config,
      {
        equal
      } = prototype;
    Config.symbols = {
      configuring: configuringSymbol$1,
      instanceProperties: instancePropertiesSymbol$1,
      lazyConfigs: lazyConfigValues
    };
    /**
     * This object holds `Config` instances keyed by their name. For example:
     *
     * ```javascript
     *  Config.cache = {
     *      disabled : Config.get('disabled'),
     *      text     : Config.get('text'),
     *      title    : Config.get('title')
     *  };
     * ```
     *
     * @member {Object} cache
     * @static
     * @private
     */
    Config.cache = Object.create(null); // object w/no properties not even inherited ones
    /**
     * This object holds config value equality methods. By default, the `===` operator is used to compare config values for
     * semantic equality. When an `equal` option is specified as a string, that string is used as a key into this object.
     *
     * All equality methods in this object have the same signature as the {@link #function-equal equal()} method.
     *
     * This object has the following equality methods:
     *
     * - `array` : Compares arrays of values using `===` on each element.
     * - `date` : Compares values of `Date` type.
     * - `strict` : The default equal algorithm based on `===` operator.
     * @member {Object} equalityMethods
     * @static
     * @private
     */
    Config.equalityMethods = {
      array: Config.makeArrayEquals(),
      date(value1, value2) {
        if (value1 === value2) {
          return true;
        }
        // see DateHelper.isDate() but cannot import due to circularity
        if (value1 && value2 && toString$2.call(value1) === DATE_TYPE$1 && toString$2.call(value2) === DATE_TYPE$1) {
          // https://jsbench.me/ltkb3vk0ji/1 - getTime is >2x faster vs valueOf/Number/op+
          return value1.getTime() === value2.getTime();
        }
        return false;
      },
      strict: Config.equal = equal
    };
    /**
     * This object holds config value merge methods. By default, {@link Core.helper.ObjectHelper#function-merge-static} is
     * used to merge object's by their properties. Config merge methods are used to combine config values from derived
     * classes with config values from super classes, as well as instance config values with those of the class.
     *
     * All merge methods in this object have the same signature as the {@link #function-merge merge()} method.
     *
     * This object has the following merge methods:
     *
     * - `distinct`   : Combines arrays of values ensuring that no value is duplicated. When given an object, its truthy
     *   keys are included, while its falsy keys are removed from the result.
     * - `merge`      : The default merge algorithm for `configurable()` properties, based on
     *   {@link Core.helper.ObjectHelper#function-merge-static}.
     * - `items`      : Similar to `merge`, but allows reordering (see `Objects.mergeItems`).
     * - `objects`    : The same as to `merge` except this method promotes `true` to an empty object.
     * - 'classList'  : Incoming strings are converted to an object where the string is a property name with a truthy value.
     * - `replace`    : Always returns `newValue` to replace the super class value with the derived class value, or the
     *   class value with the instance value.
     * @member {Object} mergeMethods
     * @static
     * @internal
     */
    Config.mergeMethods = {
      distinct(newValue, oldValue) {
        let ret = oldValue ? oldValue.slice() : [];
        if (newValue != null) {
          if (Objects.isObject(newValue)) {
            if (oldValue === undefined) {
              ret = newValue;
            } else {
              let key, index;
              for (key in newValue) {
                index = ret.indexOf(key);
                if (newValue[key]) {
                  if (index < 0) {
                    ret.push(key);
                  }
                } else if (index > -1) {
                  ret.splice(index, 1);
                }
              }
            }
          } else if (Array.isArray(newValue)) {
            newValue.forEach(v => !ret.includes(v) && ret.push(v));
          } else if (!ret.includes(newValue)) {
            ret.push(newValue);
          }
        }
        return ret;
      },
      merge: Config.merge = prototype.merge,
      classList(newValue, oldValue) {
        // 'foo bar' -> { foo : 1, bar : 1 }
        if (typeof newValue === 'string') {
          if (!newValue.length) {
            return oldValue;
          }
          newValue = newValue.split(whitespace);
        }
        if (Array.isArray(newValue)) {
          newValue = newValue.reduce(createClsProps, {});
        }
        return Config.merge(newValue, oldValue);
      },
      objects(newValue, oldValue) {
        return newValue === true ? oldValue || {} : Config.merge(newValue, oldValue);
      },
      replace(newValue) {
        return newValue;
      },
      items(newValue, oldValue, metaNew, metaCurrent) {
        if (metaCurrent) {
          // When we have metaCurrent, we are merging with a class config object, so we apply the smart merge algo
          // only in that case. Merging instance configs would lose the 'clever > syntax' info needed when the
          // time comes to actually configure an instance.
          return Objects.mergeItems(oldValue, newValue, {
            merge: (oldValue, newValue) => prototype.merge(newValue, oldValue)
          });
        }
        return prototype.merge(newValue, oldValue);
      }
    };
    Object.assign(prototype, {
      _descriptor: null,
      _initDescriptor: null,
      /**
       * A function that compares values for equality. This test is used to determine if the `update` method should be
       * called when the setter is invoked.
       *
       * To handle `Date` values:
       * ```
       *  class Foo extends Base {
       *      static get configurable() {
       *          return {
       *              date : {
       *                  $config : {
       *                      equal : 'date'
       *                  },
       *
       *                  value : null
       *              }
       *          }
       *      }
       *
       *      updateDate(date) {
       *          // date has changed
       *      }
       *  }
       * ```
       *
       * Also useful for some configs:
       * ```
       *  class Foo extends Base {
       *      static get configurable() {
       *          return {
       *              bar : {
       *                  $config : {
       *                      equal : ObjectHelper.isEqual
       *                  },
       *
       *                  value : null
       *              }
       *          }
       *      }
       *
       *      updateBar(value) {
       *          // value has changed
       *      }
       *  }
       * ```
       * @config {Function} equal
       * @internal
       */
      /**
       * Indicates that this config property should not automatically initialize during construction. When this property
       * is set to `true`, initialization is triggered by the first use of the config property's getter.
       *
       * This property can alternatively be set to a string, in which case it can be initialized as a group using the
       * {@link Core.Base#function-triggerConfigs} method which will initialize all lazy configs with the same value for
       * this property. Note: the config will still initialize on first use if that occurs prior to the call to
       * `triggerConfigs`.
       * @config {Boolean|String}
       * @default
       * @internal
       */
      lazy: false,
      /**
       * Indicates that this config property should automatically be set to `null` on destroy.
       * @config {Boolean}
       * @default
       * @internal
       */
      nullify: false,
      /**
       * Indicates that this config participates in rendering. This has does not affect the behavior of the property
       * directly, but allows classes that perform rendering to detect which config changes will affect the rendered
       * result.
       * @config {Boolean}
       * @default
       * @internal
       */
      render: false
    });
    Config._$name = 'Config';

    /**
     * @module Core/Base
     */
    class MetaClass {
      constructor(options) {
        options && Object.assign(this, options);
      }
      getInherited(name, create = true) {
        let ret = this[name];
        // We use "in this" to allow the object to be set to null
        if (!(name in this)) {
          var _this$super;
          // If there is no object on this metaclass, but there may be one on a super class. If create=false, the
          // idea is that a super class object will be "properly" inherited but otherwise no object will be created.
          ret = (_this$super = this.super) === null || _this$super === void 0 ? void 0 : _this$super.getInherited(name, create);
          if (ret || create) {
            this[name] = ret = Object.create(ret || null);
          }
        }
        return ret;
      }
    }
    const // Using Object.getPrototypeOf instead of Reflect.getPrototypeOf because:
      // 1. They are almost the same, according to the MDN difference is handling getPrototypeOf('string')
      // 2. It allows us to pass security check in SalesForce environment
      {
        getPrototypeOf: getPrototypeOf$1
      } = Object,
      {
        hasOwn: hasOwn$4
      } = Objects,
      {
        defineProperty: defineProperty$7
      } = Reflect,
      metaSymbol = Symbol('classMetaData'),
      mixinTagSymbol = Symbol('mixinTag'),
      originalConfigSymbol = Symbol('originalConfig'),
      configuringSymbol = Config.symbols.configuring,
      instancePropertiesSymbol = Config.symbols.instanceProperties,
      lazyConfigsSymbol = Config.symbols.lazyConfigs,
      defaultConfigOptions = {
        merge: 'replace',
        simple: true
      },
      emptyFn$3 = () => {},
      newMeta = o => new MetaClass(o),
      setupNames = {/* foo : 'setupFoo' */},
      emptyObject$c = Object.freeze({}),
      emptyArray$a = Object.freeze([]);
    /**
     * Base class for all configurable classes.
     *
     * Subclasses do not have to implement a constructor with its restriction of having to call super()
     * before there is a `this` reference. Subclasses instead implement a `construct` method which is
     * called by the `Base` constructor. This may call its `super` implementation at any time.
     *
     * The `Base` constructor applies all configs to properties of the new instance. The instance
     * will have been configured after the `super.construct(config)` is called.
     *
     * See the Class System documentation in the guides for more information.
     *
     * @abstract
     */
    class Base$1 {
      static get isBase() {
        return true;
      }
      get isBase() {
        return true;
      }
      // defaultConfig & properties made private to not spam all other classes
      /**
       * A class property getter to add additional, special class properties.
       *
       * For example, a class adds a `declarable` class property like so:
       * ```
       *  class Something extends Base {
       *      static get declarable() {
       *          return ['extra'];
       *      }
       *
       *      static setupExtra(cls, meta) {
       *          // use cls.extra
       *      }
       *  }
       * ```
       * A derived class can then specify this property like so:
       * ```
       *  class Derived extends Something {
       *      static get extra() {
       *          // return extra information
       *      }
       *  }
       * ```
       * When the `Derived` class is initialized, the `setupExtra()` method is called and `Derived` is passed as the
       * argument. It is also the `this` pointer, but the parameter is minifiable. The second argument passed is the
       * `$meta` object for the class.
       *
       * Classes are initialized at the first occurrence of the following:
       *
       * - An instance is created
       * - The class `$meta` property is accessed
       *
       * @member {String[]} declarable
       * @static
       * @category Configuration
       * @internal
       */
      static get declarable() {
        return ['declarable',
        /**
         * A class property getter for the configuration properties of the class, which can be overridden by
         * configurations passed at construction time.
         *
         * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
         * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
         * individually and then combine them with those of derived classes.
         *
         * For example, a `Label` might declare a `text` config like so:
         * ```javascript
         *  class Label extends Base {
         *      static get configurable() {
         *          return {
         *              text : null
         *          };
         *      }
         *  }
         * ```
         * The `text` config is automatically inherited by classes derived from Label. By implementing
         * `get configurable()`, derived classes can change the default value of inherited configs, or define new
         * configs, or both.
         *
         * When a config property is declared in this way, the class author can also implement either of two
         * special methods that will be called when the config property is assigned a new value:
         *
         *  - `changeText()`
         *  - `updateText()`
         *
         * In the example above, the `Label` class could implement a `changeText()` method, an `updateText()`
         * method, or both. The generated property setter ensures these methods will be called when the `text`
         * property is assigned.
         *
         * The generated setter (for `text` in this example) performs the following steps:
         *
         *  - If the class defines a `changeText()` method, call it passing the new value and the current value:
         *    `changeText(newText, oldText)`.<br>
         *    Then:
         *    * If `changeText()` exits without returning a value (i.e., `undefined`), exit and do nothing
         *      further. The assumption is that the changer method has done all that is required.
         *    * Otherwise, the return value of `changeText()` replaces the incoming value passed to the setter.
         *  - If the new value (or the value returned by `changeText()`) is `!==` to the current value:
         *    * Update the stored config value in `this._text`.
         *    * If the class defines an `updateText()` method, call it passing the new value and the previous value.
         *      `updateText(newText, oldText)`
         *
         * #### Resolving a value from an owner
         * By specifying a value starting with `'up.'` for a config, the config system will resolve that value by
         * examining the ownership hierarchy. It will walk up the hierarchy looking for a property matching the name
         * (or dot separated path) after 'up.'. If one is found, the value will be read and used as the initial
         * value.
         *
         * ```javascript
         * class Parent extends Base {
         *     static get configurable() {
         *         return [
         *           'importantValue'
         *         ]
         *     }
         * }
         *
         * class Child extends Base {
         *     static get configurable() {
         *         return [
         *           'value'
         *         ]
         *     }
         * }
         *
         * const parent = new Parent({
         *     importantValue : 123
         * });
         *
         * const child = new Child({
         *     owner : parent,
         *     // Will be resolved from the owner
         *     value : 'up.importantValue'
         * });
         *
         * console.log(child.value); // logs 123
         * ```
         * Please note that this is for now a one way one time binding, the value will only be read initially and
         * not kept up to date on later changes.
         *
         * #### Value Merging
         * When a config property value is an object, the value declared by the base class is merged with values
         * declared by derived classes and the value passed to the constructor.
         * ```javascript
         *  class Example extends Base {
         *      static get configurable() {
         *          return {
         *              config : {
         *                  foo : 1,
         *                  bar : 2
         *              }
         *          };
         *      }
         *  }
         *
         *  class Example2 extends Example {
         *      static get configurable() {
         *          return {
         *              config : {
         *                  bar : 42,
         *                  zip : 'abc'
         *              }
         *          };
         *      }
         *  }
         *
         *  let ex = new Example2({
         *      config : {
         *          zip : 'xyz'
         *      }
         *  });
         * ```
         * The result of the merge would set `config` to:
         * ```javascript
         *  ex.foo = {
         *      foo : 1,    // from Example
         *      bar : 42,   // from Example2
         *      zip : 'xyz' // from constructor
         *  }
         * ```
         *
         * #### Config Options
         * Some config properties require additional options such as declarative information about the config that
         * may be useful to automate some operation. Consider a `Button`. It could declare that its `text` config
         * affects the rendered HTML by applying a `render` property to the config definition. Its base class could
         * then examine the config definition to find this property.
         *
         * To support this, config options ca be declared like so:
         * ```javascript
         *  class Button extends Widget {
         *      static get configurable() {
         *          return {
         *              text : {
         *                  value   : null,
         *                  $config : {
         *                      render : true
         *                  }
         *              }
         *          };
         *      }
         *  }
         * ```
         * The `$config` property can alternatively be just the names of the options that should be enabled (set
         * to `true`).
         *
         * For example, the following is equivalent to the above:
         * ```javascript
         *  class Button extends Widget {
         *      static get configurable() {
         *          return {
         *              text : {
         *                  value   : null,
         *                  $config : 'render'
         *              }
         *          };
         *  }
         * ```
         *
         * #### Default Value
         * It is common to set a config to a `null` value to take advantage of internal optimizations for `null`
         * values. In most cases the fact that this produces `undefined` as the actual initial value of the config
         * is acceptable. When this is not acceptable, a config can be declared like so:
         * ```javascript
         *  class Widget {
         *      static get configurable() {
         *          return {
         *              disabled : {
         *                  $config : null,
         *                  value   : null,
         *                  default : false
         *              }
         *          };
         *  }
         * ```
         * The `default` property above determines the value of the config while still gaining the benefits of
         * minimal processing due to the `null` value of the `value` property.
         * @member {Object} configurable
         * @static
         * @category Configuration
         * @internal
         */
        'configurable',
        /**
         * A class property getter for the default configuration of the class, which can be overridden by
         * configurations passed at construction time.
         *
         * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
         * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
         * individually and then combine them with those of derived classes.
         *
         * For example, a `Store` might declare its `url` config like so:
         * ```
         *  class Store extends Base {
         *      static get defaultConfig() {
         *          return {
         *              url : null
         *          };
         *      }
         *  }
         * ```
         * The `url` config is automatically inherited by classes derived from Store. By implementing
         * `get defaultConfig()`, derived classes can change the default value of inherited configs, or define new
         * configs, or both. When defining new configs, however, `configurable` is preferred.
         *
         * Config properties introduced to a class by this declaration do not participate in value merging and do
         * not get a generated setter. Config properties introduced by a base class using `configurable` can be
         * set to a different value using `defaultConfig` and in doing so, the values will be merged as appropriate
         * for `configurable`.
         *
         * @member {Object} defaultConfig
         * @static
         * @category Configuration
         * @internal
         */
        'defaultConfig',
        /**
         * A class property getter for the default values of internal properties for this class.
         * @member {Object} properties
         * @static
         * @category Configuration
         * @internal
         */
        'properties',
        /**
         * A class property getter for properties that will be applied to the class prototype.
         * @member {Object} prototypeProperties
         * @static
         * @category Configuration
         * @internal
         */
        'prototypeProperties'];
      }
      /**
       * Base constructor, passes arguments to {@link #function-construct}.
       * @param {...Object} [args] Usually called with a config object, but accepts any params
       * @function constructor
       * @category Lifecycle
       * @advanced
       */
      constructor(...args) {
        const me = this,
          C = me.constructor;
        if (me.$meta.class !== C) {
          // This will happen only once for each class. We need to call the C.$meta getter which puts $meta on our
          // prototype. Since that alone would be optimized away (and would generate IDE and lint warnings), we call
          // emptyFn and simply pass the value.
          emptyFn$3(C.$meta);
        }
        // Allow subclasses to have a pseudo constructor with "this" already set:
        me.construct(...args);
        me.afterConstruct();
        me.isConstructing = false;
      }
      /**
       * Factory version of the Base constructor. Merges all arguments to create a config object that is passed along to
       * the constructor.
       * @param {...Object} [configs] Allows passing multiple config objects
       * @returns {Core.Base} New instance
       * @private
       */
      static new(...configs) {
        configs = configs.filter(c => c);
        return new this(configs.length > 1 ? this.mergeConfigs(...configs) : configs[0]);
      }
      /**
       * Base implementation applies configuration.
       *
       * Subclasses need only implement this if they have to initialize instance specific
       * properties required by the class. Often a `construct` method is
       * unnecessary. All initialization of incoming configuration properties can be
       * done in a `set propName` implementation.
       * @param {...Object} [args] Usually called with a config object, but accepts any params
       * @category Lifecycle
       * @advanced
       */
      construct(...args) {
        // Passing null to base construct means bypass the config system and stack creation (to gain performance)
        if (args[0] !== null) {
          this.configure(...args);
        }
        this.afterConfigure();
      }
      /**
       * Destroys the provided objects by calling their {@link #function-destroy} method.
       * Skips empty values or objects that are already destroyed.
       *
       * ```javascript
       * Base.destroy(myButton, toolbar1, helloWorldMessageBox);
       * ```
       * @param {...Object} [args] Objects to be destroyed
       * @category Lifecycle
       * @advanced
       */
      static destroy(...args) {
        const shredder = object => {
          if (object !== null && object !== void 0 && object.destroy) {
            object.destroy();
          } else if (Array.isArray(object)) {
            object.forEach(shredder);
          }
        };
        shredder(args);
      }
      /**
       * Destroys this object.
       *
       * {@advanced}
       * This is primarily accomplished by calling {@link #function-doDestroy}, however, prior to
       * calling `doDestroy`, {@link #property-isDestroying} is set to `true`. After {@link #function-doDestroy} returns,
       * {@link #property-isDestroyed} is set to `true`.
       *
       * Do not override this method in subclasses. To provide class-specific cleanup, implement {@link #function-doDestroy}
       * instead.
       * {/@advanced}
       *
       * @category Lifecycle
       */
      destroy() {
        const me = this,
          {
            id
          } = me;
        // Let everyone know the object is going inert:
        me.isDestroying = true;
        // Make calling destroy() harmless:
        me.destroy = emptyFn$3;
        me.doDestroy();
        Object.setPrototypeOf(me, null);
        // Clear all remaining instance properties.
        for (const key in me) {
          if (key !== 'destroy' && key !== 'isDestroying') {
            delete me[key];
          }
        }
        delete me[originalConfigSymbol];
        // Let everyone know the object is inert:
        me.isDestroyed = true;
        me.id = id; // for diagnostic reasons
      }
      /**
       * This method is required to help `unused` getters to survive production build process. Some tools, like angular,
       * will remove `unused` code in production build, making our side-effected getters behind, breaking code heavily.
       * @internal
       * @param getter Getter to evaluate
       */
      _thisIsAUsedExpression(getter) {}
      static get $$name() {
        return hasOwn$4(this, '$name') && this.$name ||
        // _$name is filled by webpack for every class (cls._$name = '...')
        hasOwn$4(this, '_$name') && this._$name || this.name;
      }
      get $$name() {
        return this.constructor.$$name;
      }
      /**
       * Base implementation so that all subclasses and mixins may safely call super.startConfigure.
       *
       * This is called by the Base class before setting configuration properties, but after
       * the active initial getters have been set, so all configurations are available.
       *
       * This method allows all classes in the hierarchy to force some configs to be evaluated before others.
       * @internal
       * @category Lifecycle
       * @params {Object} config The configuration object use to set the initial state.
       */
      startConfigure(config) {}
      /**
       * Base implementation so that all subclasses and mixins may safely call super.finishConfigure.
       *
       * This is called by the Base class before exiting the {@link #function-configure} method.
       *
       * At this point, all configs have been applied, but the `isConfiguring` property is still set.
       *
       * This method allows all classes in the hierarchy to inject functionality
       * into the config phase.
       * @internal
       * @category Lifecycle
       * @params {Object} config The configuration object use to set the initial state.
       */
      finishConfigure(config) {}
      /**
       * Base implementation so that all subclasses and mixins may safely call `super.afterConfigure`. This is called by the Base class after the {@link #function-configure} method has been
       * called. At this point, all configs have been applied.
       *
       * This method allows all classes in the hierarchy to inject functionality
       * either before or after the super.afterConstruct();
       * @internal
       * @category Lifecycle
       */
      afterConfigure() {}
      /**
       * Base implementation so that all subclasses and mixins may safely call super.afterConstruct.
       *
       * This is called by the Base class after the {@link #function-construct} method has been
       * called.
       *
       * At this point, all configs have been applied.
       *
       * This method allows all classes in the hierarchy to inject functionality
       * either before or after the super.afterConstruct();
       * @internal
       * @function afterConstructor
       * @category Lifecycle
       */
      afterConstruct() {}
      /**
       * Provides a way of calling callbacks which may have been specified as the _name_ of a function
       * and optionally adds scope resolution.
       *
       * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
       * then the function is resolved in this object. This is useful when configuring listeners
       * at the class level.
       *
       * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
       * using the `owner` property until an object with the named function is present, then the
       * named function is called upon that object.
       *
       * If a named function is not found, an error is thrown. If the function should be only called when present,
       * and may not be present, add a `?` as a suffix.
       *
       * @param {String|Function} fn The function to call, or the name of the function to call.
       * @param {Object} thisObject The `this` object of the function.
       * @param {Object[]} args The argument list to pass.
       * @category Misc
       * @advanced
       */
      callback(fn, thisObject, args = emptyArray$a) {
        // Maintainer: do not make args ...args. This method may acquire more arguments
        const {
          handler,
          thisObj
        } = this.resolveCallback(fn, thisObject === 'this' ? this : thisObject) || emptyObject$c;
        return handler === null || handler === void 0 ? void 0 : handler.apply(thisObj, args);
      }
      resolveProperty(propertyPath) {
        let thisObj = this;
        while (thisObj) {
          if (Objects.hasPath(thisObj, propertyPath)) {
            return Objects.getPath(thisObj, propertyPath);
          }
          thisObj = thisObj.owner;
        }
        return undefined;
      }
      /**
       * Provides a way of locating callbacks which may have been specified as the _name_ of a function
       * and optionally adds scope resolution.
       *
       * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
       * then the function is resolved in this object. This is useful when configuring listeners
       * at the class level.
       *
       * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
       * using the `owner` property until an object with the named function is present, then the
       * named function is called upon that object.
       * @param {String|Function} handler The function to call, or the name of the function to call.
       * @param {Object} thisObj The `this` object of the function.
       * @param {Boolean} [enforceCallability = true] Pass `false` if the function may not exist, and a null return value is acceptable.
       * @returns {Object} `{ handler, thisObj }`
       * @category Misc
       * @advanced
       */
      resolveCallback(handler, thisObj = this, enforceCallability = true) {
        var _handler;
        // It's a string, we find it in its own thisObj
        if ((_handler = handler) !== null && _handler !== void 0 && _handler.substring) {
          if (handler.endsWith('?')) {
            enforceCallability = false;
            handler = handler.substring(0, handler.length - 1);
          }
          if (handler.startsWith('up.')) {
            handler = handler.substring(3);
            // Empty loop until we find the function owner
            for (thisObj = this.owner; thisObj && !thisObj[handler]; thisObj = thisObj.owner);
            if (!thisObj) {
              return;
            }
          } else if (handler.startsWith('this.')) {
            handler = handler.substring(5);
            thisObj = this;
          }
          if (!thisObj || !(thisObj instanceof Object)) {
            return;
          }
          handler = thisObj[handler];
        }
        // Any other type than string or function results in unresolved callback
        if (typeof handler === 'function') {
          return {
            handler,
            thisObj
          };
        }
        if (enforceCallability) {
          throw new Error(`No method named ${handler} on ${thisObj.$$name || 'thisObj object'}`);
        }
      }
      bindCallback(inHandler, inThisObj = this) {
        if (inHandler) {
          const {
            handler,
            thisObj
          } = this.resolveCallback(inHandler, inThisObj);
          if (handler) {
            return handler.bind(thisObj);
          }
        }
      }
      /**
       * Delays the execution of the passed function by the passed time quantum, or if the time is omitted
       * or not a number, delays until the next animation frame. Note that this will use
       * {@link Core.mixin.Delayable#function-setTimeout} || {@link Core.mixin.Delayable#function-requestAnimationFrame}
       * if this class mixes in `Delayable`, otherwise it uses the global methods. The function will
       * be called using `this` object as its execution scope.
       * @param {Function} fn The function to call on a delay.
       * @param {Number} [delay] The number of milliseconds to delay.
       * @param {String} [name] The name of delay
       * @returns {Number} The created timeout id.
       * @private
       */
      delay(fn, delay, name = fn.name || fn) {
        // Force scope on the fn if we are not a Delayable
        fn = this.setTimeout ? fn : fn.bind(this);
        const invoker = this.setTimeout ? this : globalThis;
        return invoker[typeof delay === 'number' ? 'setTimeout' : 'requestAnimationFrame'](fn, delay, name);
      }
      /**
       * Classes implement this method to provide custom cleanup logic before calling `super.doDestroy()`. The general
       * pattern is as follows:
       *
       * ```javascript
       *  class Foo extends Base {
       *      doDestroy() {
       *          // perform custom cleanup
       *
       *          super.doDestroy();
       *      }
       *  }
       * ```
       *
       * This method is called by {@link #function-destroy} which also prevents multiple calls from reaching `doDestroy`.
       * Prior to calling `doDestroy`, {@link #property-isDestroying} is set to `true`. Upon return, the object is fully
       * destructed and {@link #property-isDestroyed} is set to `true`.
       *
       * Do not call this method directly. Instead call {@link #function-destroy}.
       * @category Lifecycle
       * @advanced
       */
      doDestroy() {
        const me = this,
          {
            nullify
          } = me.$meta;
        if (nullify) {
          for (let i = 0; i < nullify.length; ++i) {
            if (me[nullify[i].field] != null) {
              // if backing property is null/undefined then skip
              me[nullify[i].name] = null; // else, call setter to run through change/update
            }
          }
        }
      }
      /**
       * Destroys the named properties if they have been initialized, and if they have a `destroy` method.
       * Deletes the property from this object. For example:
       *
       *      this.destroyProperties('store', 'resourceStore', 'eventStore', 'dependencyStore', 'assignmentStore');
       *
       * @param {String} properties The names of the properties to destroy.
       * @internal
       * @category Lifecycle
       */
      destroyProperties(...properties) {
        const me = this;
        let key;
        for (key of properties) {
          // If the value has *not* been pulled in from the configuration object yet
          // we must not try to access it, as that will cause the property to be initialized.
          if (key in me && (!me[configuringSymbol] || !me[configuringSymbol][key])) {
            var _me$key, _me$key$destroy, _me$key2;
            (_me$key = me[key]) === null || _me$key === void 0 ? void 0 : (_me$key$destroy = (_me$key2 = _me$key).destroy) === null || _me$key$destroy === void 0 ? void 0 : _me$key$destroy.call(_me$key2);
            delete me[key];
          }
        }
      }
      /**
       * Called by the Base constructor to apply configs to this instance. This must not be called.
       * @param {Object} config The configuration object from which instance properties are initialized.
       * @private
       * @category Lifecycle
       */
      configure(config = {}) {
        const me = this,
          meta = me.$meta,
          {
            beforeConfigure
          } = config,
          configs = meta.configs,
          fullConfig = me.getDefaultConfiguration();
        let cfg, key, value;
        me.initialConfig = config;
        // Important flag for setters to know whether they are being called during
        // configuration when this object is not fully alive, or whether it's being reconfigured.
        me.isConfiguring = true;
        // Assign any instance properties declared by the class.
        Object.assign(me, me.getProperties());
        // Apply configuration to default from class definition. This is safe because it's either chained from or a
        // fork of the class values.
        for (key in config) {
          value = config[key];
          cfg = configs[key];
          fullConfig[key] = cfg ? cfg.merge(value, fullConfig[key], null, meta) : value;
        }
        if (beforeConfigure) {
          delete fullConfig.beforeConfigure;
          // noinspection JSValidateTypes
          beforeConfigure(me, fullConfig);
        }
        // Cache me.config for use by get config.
        me.setConfig(me[originalConfigSymbol] = fullConfig, true);
        me.isConfiguring = false;
      }
      /**
       * Returns the value of the specified config property. This is a method to allow
       * property getters to be explicitly called in a way that does not get optimized out.
       *
       * The following triggers the getter call, but optimizers will remove it:
       *
       *      inst.foo;   // also raises "expression has no side-effects" warning
       *
       * Instead, do the following to trigger a getter:
       *
       *      inst.getConfig('foo');
       *
       * @param {String} name
       * @internal
       * @category Configuration
       */
      getConfig(name) {
        return this[name];
      }
      /**
       * Sets configuration options this object with all the properties passed in the parameter object.
       * Timing is taken care of. If the setter of one config is called first, and references
       * the value of another config which has not yet been set, that config will be set just
       * in time, and the *new* value will be used.
       * @param {Object} config An object containing configurations to change.
       * @category Lifecycle
       * @advanced
       */
      setConfig(config, isConstructing) {
        const me = this,
          wasConfiguring = me[configuringSymbol],
          configDone = wasConfiguring ? me.configDone : me.configDone = {},
          configs = me.$meta.configs;
        let cfg, key;
        me[instancePropertiesSymbol] = {};
        // Cache configuration for use by injected property initializers.
        me[configuringSymbol] = wasConfiguring ? Object.setPrototypeOf(Object.assign({}, config), wasConfiguring) : config;
        // For each incoming non-null configuration, create a temporary getter which will
        // pull the value in from the initialConfig so that it doesn't matter in
        // which order properties are set. You can access any property at any time.
        for (key in config) {
          // Don't default null configs in unless it's a direct property of the
          // the passed configuration. When used at construct time, defaultConfigs
          // will be prototype-chained onto the config.
          if (config[key] != null || hasOwn$4(config, key)) {
            cfg = configs[key] || Config.get(key);
            cfg.defineInitter(me, config[key]);
            if (!isConstructing) {
              configDone[key] = false;
            }
            // else if (cfg.lazy) {
            //     // This was done originally to prevent our for-loop below from poking the value on the instance
            //     // at this stage. It was removed since it confused triggerConfig, and it just isn't true that the
            //     // lazy config is done...
            //     configDone[key] = true;
            // }
          } else {
            configDone[key] = true;
          }
        }
        if (isConstructing) {
          me.startConfigure(config);
        }
        // Set all our properties from the config object.
        // If one of the properties needs to access a property that has not
        // yet been set, the above temporary property will pull it through.
        // Can't use Object.assign because that only uses own properties.
        // config value blocks are prototype chained subclass->superclass
        for (key in config) {
          var _configs$key;
          // Only push the value through if the property initializer is still present.
          // If it gets triggered to pull the configuration value in, it deleted itself.
          if (!configDone[key] && !((_configs$key = configs[key]) !== null && _configs$key !== void 0 && _configs$key.lazy)) {
            me[key] = config[key];
          }
        }
        if (wasConfiguring) {
          me[configuringSymbol] = wasConfiguring;
        } else {
          delete me[configuringSymbol];
        }
        if (isConstructing) {
          me.finishConfigure(config);
        }
        return me;
      }
      /**
       * Returns `true` if this instance has a non-null value for the specified config. This will not activate a lazy
       * config.
       *
       * @param {String} name The name of the config property.
       * @returns {Boolean}
       * @internal
       */
      hasConfig(name) {
        var _me$lazyConfigsSymbol;
        const me = this,
          config = me[configuringSymbol];
        return Boolean(me['_' + name] != null ||
        // value has been assigned to backing property
        ((_me$lazyConfigsSymbol = me[lazyConfigsSymbol]) === null || _me$lazyConfigsSymbol === void 0 ? void 0 : _me$lazyConfigsSymbol.get(name)) != null ||
        // a lazy value is pending

        // config value has not been assigned but will be
        !me.configDone[name] && config && (config[name] != null || hasOwn$4(config, name)));
      }
      /**
       * Returns the value of an uningested config *without* ingesting the config or transforming
       * it from its raw value using its `changeXxxxx` method.
       *
       * @param {String} name The name of the config property.
       * @returns {*} The raw incoming config value.
       * @internal
       */
      peekConfig(name) {
        const me = this,
          lazyConfig = me[lazyConfigsSymbol],
          config = me[configuringSymbol];
        // It's waiting in the lazy configs
        if (lazyConfig !== null && lazyConfig !== void 0 && lazyConfig.has(name)) {
          return lazyConfig.get(name);
        }
        if (config && name in config) {
          // It's been read in, so use the current value
          if (me.configDone[name]) {
            return me[name];
          }
          if (config[name] != null || hasOwn$4(config, name)) {
            return config[name];
          }
        }
      }
      /**
       * Ensures that the specified config is initialized if it is needed. If there is a config value specified, and it
       * was initialized by this call, this method returns `true`. If there was a config value specified, and it was
       * already initialized, this method returns `false`. If there was no value specified for the given config, this
       * method returns `null`.
       *
       * This is not the same as just reading the property, because some property getters exist that do not actually just
       * read the config value back, but instead produce some result. Reading such properties to incidentally trigger a
       * possible config initializer can lead to incorrect results. For example, the Combo items config.
       *
       * @param {String} name The name of the config property.
       * @returns {Boolean}
       * @internal
       */
      triggerConfig(name) {
        const me = this,
          {
            configDone
          } = me,
          lazyConfig = me[lazyConfigsSymbol],
          config = me[configuringSymbol],
          triggered = lazyConfig !== null && lazyConfig !== void 0 && lazyConfig.has(name) || config && (config[name] != null || hasOwn$4(config, name)) ? !configDone[name] : null;
        if (triggered) {
          me.getConfig(name);
        }
        return triggered;
      }
      /**
       * This call will activate any pending {@link Core.Config#config-lazy} configs that were assigned a string value
       * equal to the `group` parameter.
       *
       * @param {String} group The config property group as defined by a matching {@link Core.Config#config-lazy} value.
       * @returns {String[]} The names of any configs triggered by this call or `null` if no configs were triggered.
       * @internal
       */
      triggerConfigs(group) {
        const me = this,
          configs = me.$meta.configs,
          lazyConfigs = me[lazyConfigsSymbol],
          triggered = lazyConfigs ? [...lazyConfigs.keys()].filter(k => configs[k].lazy === group) : emptyArray$a;
        for (const key of triggered) {
          me.triggerConfig(key);
        }
        return triggered.length ? triggered : null;
      }
      onConfigChange() {} // declared above because lint/IDE get angry about not declaring the args...
      /**
       * This method is called when any config changes.
       * @param {Object} info Object containing information regarding the config change.
       * @param {String} info.name The name of the config that changed.
       * @param {*} info.value The new value of the config.
       * @param {*} info.was The previous value of the config.
       * @param {Core.Config} info.config The `Config` object for the changed config property.
       * @method onConfigChange
       * @internal
       * @category Configuration
       */
      /**
       * Returns a *copy* of the full configuration which was used to configure this object.
       * @property {Object}
       * @category Lifecycle
       * @readonly
       * @advanced
       */
      get config() {
        const result = {},
          myConfig = this[originalConfigSymbol];
        // The configuration was created as a prototype chain of the class hierarchy's
        // defaultConfig values hanging off a copy of the initialConfig object, so
        // we must loop and copy since Object.assign only copies own properties.
        for (const key in myConfig) {
          result[key] = myConfig[key];
        }
        return result;
      }
      // region Extract config
      static processConfigValue(currentValue, options) {
        if (currentValue === globalThis) {
          return globalThis;
        } else if (Array.isArray(currentValue)) {
          return currentValue.map(v => Base$1.processConfigValue(v, options));
        }
        // Not using isBase to avoid classes (modelClass for example)
        else if (currentValue instanceof Base$1) {
          if (options.visited.has(currentValue)) {
            return;
          }
          return currentValue.getCurrentConfig(options);
        }
        // appendTo, floatRoot etc
        else if (currentValue instanceof HTMLElement || currentValue instanceof DocumentFragment) {
          return null;
        }
        // Go deeply into objects, might have instances of our classes in them
        else if (Objects.isObject(currentValue)) {
          const result = {};
          for (const key in currentValue) {
            // Only step "down", not "up"
            if (key !== 'owner') {
              result[key] = Base$1.processConfigValue(currentValue[key], options);
            }
          }
          return result;
        }
        return currentValue;
      }
      // Recursively get the value of a config. Only intended to be called by getCurrentConfig()
      getConfigValue(name, options) {
        var _me$$meta$configs$nam;
        const me = this,
          lazyConfigs = me[lazyConfigsSymbol];
        // Do not trigger lazy configs
        if (!((_me$$meta$configs$nam = me.$meta.configs[name]) !== null && _me$$meta$configs$nam !== void 0 && _me$$meta$configs$nam.lazy)) {
          return Base$1.processConfigValue(me[name], options);
        }
        // Instead pull their initial config in
        if (lazyConfigs !== null && lazyConfigs !== void 0 && lazyConfigs.has(name)) {
          return Base$1.processConfigValue(lazyConfigs.get(name), options);
        }
      }
      // Allows removing / adding configs before values are extracted
      preProcessCurrentConfigs() {}
      // Extract the current values for all initially used configs, in a format that can be used to create a new instance.
      // Not intended to be called by any other code than getConfigString()
      getCurrentConfig(options = {}) {
        const me = this,
          configs = options.configs === 'all' ? me.config : Objects.clone(me.initialConfig),
          visited = options.visited || (options.visited = new Set()),
          depth = options.depth || (options.depth = 0),
          result = {};
        if (visited.has(me)) {
          return undefined;
        }
        visited.add(me);
        this.preProcessCurrentConfigs(configs);
        for (const name in configs) {
          const value = me.getConfigValue(name, {
            ...options,
            depth: depth + 1
          });
          if (value !== undefined) {
            result[name] = value;
          }
        }
        return result;
      }
      // Extract the current values for all initially used configs and convert them to a JavaScript string
      getConfigString(options = {}) {
        return StringHelper.toJavaScriptString(this.getCurrentConfig(options));
      }
      // Experimental helper function, extracts the currently used configs and wraps them as an app, returning code as a
      // string.
      //
      // This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
      //
      getTestCase(options = {}) {
        //<remove-on-lwc-release>
        const Product = this.isGantt ? 'Gantt' : this.isSchedulerPro ? 'SchedulerPro' : this.isCalendar ? 'Calendar' : this.isScheduler ? 'Scheduler' : this.isGrid ? 'Grid' : this.isTaskBoard ? 'TaskBoard' : null;
        if (Product) {
          const product = Product.toLowerCase(),
            // bundlePath = `https://bryntum.com/dist/${product}/build/${product}.module.js`,
            bundlePath = `../../build/${product}.module.js`;
          let preamble, postamble;
          if (options.import === 'static') {
            preamble = `import * as module from "${bundlePath}";` + 'Object.assign(window, module);'; // for (const c in module) window[c] = module[c];
            postamble = '';
          } else {
            preamble = `import("${bundlePath}").then(module => { Object.assign(window, module);\n`;
            postamble = '});';
          }
          const version = VersionHelper.getVersion(product);
          if (version) {
            preamble += `\nconsole.log('${Product} ${version}');\n`;
          }
          // De-indented on purpose
          return `${preamble}      \nconst ${product} = new ${Product}(${this.getConfigString(options)});\n${postamble}`;
        }
        //</remove-on-lwc-release>
      }
      /**
       * Experimental helper function, extracts the currently used configs and wraps them as an app, downloading the
       * resulting JS file.
       *
       * This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
       * @category Misc
       */
      downloadTestCase(options = {}) {
        options.output = 'return';
        const app = this.getTestCase(options);
        BrowserHelper.download(`app.js`, 'data:application/javascript;charset=utf-8,' + escape(app));
      }
      //endregion
      /**
       * Registers this class type with its Factory
       * @category Misc
       * @advanced
       */
      static initClass() {
        return this.$meta.class;
      }
      /**
       * The class's {@link #property-$meta-static meta} object.
       * @member {Object} $meta
       * @internal
       * @category Misc
       */
      /**
       * An object owned by this class that does not share properties with its super class.
       *
       * This object may contain other properties which are added as needed and are not documented here.
       *
       * @property {Object} $meta The class meta object.
       * @property {Function} $meta.class The class constructor that owns the meta object.
       * @property {Object} $meta.super The `$meta` object for the super class. This is `null` for `Base`.
       * @property {Object} $meta.config The object holding the default configuration values for this class.
       * @property {Object} $meta.configs An object keyed by config name that holds the defined configs for the class.
       * The value of each property is a {@link Core/Config} instance.
       * @property {Boolean} $meta.forkConfigs This will be `true` if the default configuration values for this class
       * (in the `config` property of the meta object) must be forked to avoid object sharing, or if the object can be
       * passed to `Object.create()` for efficiency.
       * @property {Function[]} $meta.hierarchy The array of classes in the ancestry of this class. This will start with
       * `Base` at index 0 and ends with this class.
       * @property {Function[]} $meta.properties The array of classes that define a "static get properties()" getter.
       * @internal
       * @static
       * @category Misc
       */
      static get $meta() {
        const me = this;
        let meta = me[metaSymbol];
        if (!hasOwn$4(me, metaSymbol)) {
          me[metaSymbol] = meta = newMeta();
          meta.class = me;
          me.setupClass(meta);
        }
        return meta;
      }
      /**
       * This optional class method is called when a class is mixed in using the {@link #function-mixin-static mixin()}
       * method.
       * @internal
       */
      static onClassMixedIn() {
        // empty
      }
      /**
       * Returns the merge of the `baseConfig` and `config` config objects based on the configs defined by this class.
       * @param {Object} baseConfig The base config or defaults.
       * @param {...Object} configs One or more config objects that takes priority over `baseConfig`.
       * @returns {Object}
       * @internal
       */
      static mergeConfigs(baseConfig, ...configs) {
        const classConfigs = this.$meta.configs,
          result = Objects.clone(baseConfig) || {};
        let config, i, key, value;
        for (i = 0; i < configs.length; ++i) {
          config = configs[i];
          if (config) {
            for (key in config) {
              value = config[key];
              if (classConfigs[key]) {
                value = classConfigs[key].merge(value, result[key]);
              } else if (result[key] && value) {
                value = Config.merge(value, result[key]);
              }
              result[key] = value;
            }
          }
        }
        return result;
      }
      /**
       * Applies one or more `mixins` to this class and returns the produced class constructor.
       *
       * For example, instead of writing this:
       * ```
       *  class A extends Delayable(Events(Localizable(Base))) {
       *      // ...
       *  }
       * ```
       *
       * Using this method, one would write this:
       * ```
       *  class A extends Base.mixin(Localizable, Events, Delayable) {
       *      // ...
       *  }
       * ```
       * If one of the mixins specified has already been mixed into the class, it will be ignored and not mixed in a
       * second time.
       * @param {...Function} mixins
       * @returns {Function}
       * @category Misc
       * @advanced
       */
      static mixin(...mixins) {
        // Starting w/the first class C = this
        let C = this,
          i;
        // wrap each class C using mixins[i] to produce the next class
        for (i = 0; i < mixins.length; ++i) {
          const mixin = mixins[i],
            // Grab or create a unique Symbol for this mixin so we can tell if we've already mixed it in
            tag = mixin[mixinTagSymbol] || (mixin[mixinTagSymbol] = Symbol('mixinTag'));
          if (C[tag]) {
            continue;
          }
          C = mixin(C);
          C[tag] = true; // properties on the constructor are inherited to subclass constructors...
          if (hasOwn$4(C, 'onClassMixedIn')) {
            C.onClassMixedIn();
          }
        }
        return C;
      }
      /**
       * This method is called only once for any class. This can occur when the first instance is created or when the
       * `$meta` object is first requested.
       * @param {Object} meta The `$meta` object for the class.
       * @internal
       * @category Misc
       */
      static setupClass(meta) {
        var _base$nullify;
        const cls = meta.class,
          // Trigger setupClass on the super class (if it has yet to happen):
          base = getPrototypeOf$1(cls).$meta,
          name = cls.$$name,
          names = base.names,
          proto = cls.prototype;
        defineProperty$7(proto, '$meta', {
          value: meta
        });
        Object.assign(meta, {
          super: base,
          config: Object.create(base.config),
          configs: Object.create(base.configs),
          declarables: base.declarables,
          forkConfigs: base.forkConfigs,
          hierarchy: Object.freeze([...base.hierarchy, cls]),
          names: names.includes(name) ? names : Object.freeze([...names, name]),
          properties: base.properties,
          nullify: (_base$nullify = base.nullify) === null || _base$nullify === void 0 ? void 0 : _base$nullify.slice()
        });
        if (names !== meta.names) {
          const isName = `is${name}`,
            defineIsProperty = obj => {
              if (!hasOwn$4(obj, isName)) {
                defineProperty$7(obj, isName, {
                  get() {
                    return true;
                  }
                });
              }
            };
          defineIsProperty(proto);
          defineIsProperty(cls);
        }
        // NOTE: we always use meta.declarables because setupDeclarable() can replace the array on the meta object
        // when new declarable properties are added...
        for (let decl, setupName, i = 0; i < meta.declarables.length; ++i) {
          decl = meta.declarables[i];
          if (hasOwn$4(cls, decl)) {
            setupName = setupNames[decl] || (setupNames[decl] = `setup${StringHelper.capitalize(decl)}`);
            cls[setupName](cls, meta);
          }
        }
        /*  Add slash to the front of this line to enable the diagnostic block:
        /**/
      }
      /**
       * This method is called as part of `setupClass()`. It will process the `configurable()` return object and the
       * `defaultConfig` return object.
       * @param {Object} meta The `meta` object for this class.
       * @param {Object} configs The config definition object.
       * @param {Boolean} simple `true` when processing `defaultConfig` and `false` when processing `configurable`.
       * @private
       * @category Configuration
       */
      static setupConfigs(meta, configs, simple) {
        const classConfigValues = meta.config,
          classConfigs = meta.configs,
          cls = meta.class,
          superMeta = meta.super;
        let {
            nullify
          } = meta,
          cfg,
          defaultValue,
          options,
          setDefault,
          value,
          wasNullify;
        for (const name in configs) {
          value = configs[name];
          if (simple) {
            // Using "defaultConfig"
            if (!(cfg = classConfigs[name])) {
              cfg = Config.get(name, defaultConfigOptions);
            } else {
              // The property may be declared in a base class using configurable(), so it may have special
              // merge processing:
              value = cfg.merge(value, classConfigValues[name], meta, superMeta);
            }
            /*  Add slash to the front of this line to enable the diagnostic block:
            /**/
          } else {
            // Using "configurable"
            defaultValue = options = setDefault = undefined;
            if (value && typeof value === 'object' && '$config' in value) {
              options = value.$config;
              if (options && !Objects.isObject(options)) {
                options = Objects.createTruthyKeys(options);
              }
              setDefault = 'default' in value;
              defaultValue = setDefault ? value.default : defaultValue;
              value = value.value;
            }
            if (!(cfg = classConfigs[name])) {
              cfg = Config.get(name, options);
              cfg.define(cls.prototype);
              setDefault = !(cfg.field in cls.prototype); // reduce object shape changes (helps JIT)
              wasNullify = false;
            } else {
              wasNullify = cfg.nullify;
              if (options) {
                // Defined by a base class, but maybe being adjusted by derived.
                cfg = cfg.extend(options);
                // In the future, we may need to redefine the property here if options affect the descriptor (such
                // as event firing)
              }

              value = cfg.merge(value, classConfigValues[name], meta, superMeta);
            }
            if (setDefault) {
              cfg.setDefault(cls, defaultValue);
            }
            if (cfg.nullify && !wasNullify) {
              (nullify || (nullify = meta.nullify || (meta.nullify = []))).push(cfg);
            }
          }
          // If any default properties are *mutable* Objects or Array we need to clone them.
          // so that instances do not share configured values.
          if (value && (Objects.isObject(value) || Array.isArray(value)) && !Object.isFrozen(value)) {
            meta.forkConfigs = true;
          }
          classConfigs[name] = cfg;
          classConfigValues[name] = value;
        }
      }
      static setupConfigurable(cls, meta) {
        cls.setupConfigs(meta, cls.configurable, false);
      }
      static setupDefaultConfig(cls, meta) {
        cls.setupConfigs(meta, cls.defaultConfig, true);
      }
      static setupDeclarable(cls, meta) {
        const declarable = cls.declarable;
        let all = meta.declarables,
          forked,
          i;
        for (i = 0; i < declarable.length; ++i) {
          if (!all.includes(declarable[i])) {
            if (!forked) {
              meta.declarables = forked = all = all.slice();
            }
            all.push(declarable[i]);
          }
        }
      }
      static setupProperties(cls, meta) {
        meta.properties = meta.super.properties.slice();
        meta.properties.push(cls);
        Object.freeze(meta.properties);
      }
      static setupPrototypeProperties(cls) {
        Object.assign(cls.prototype, cls.prototypeProperties);
      }
      /**
       * Gets the full {@link #property-defaultConfig-static} block for this object's entire inheritance chain
       * all the way up to but not including {@link Core.Base}
       * @returns {Object} All default config values for this class.
       * @private
       * @category Configuration
       */
      getDefaultConfiguration() {
        return this.constructor.getDefaultConfiguration();
      }
      /**
       * Gets the full {@link #property-defaultConfig-static} block for the entire inheritance chain for this class
       * all the way up to but not including {@link Core.Base}
       * @returns {Object} All default config values for this class.
       * @private
       * @category Configuration
       */
      static getDefaultConfiguration() {
        const meta = this.$meta,
          config = meta.forkConfigs ? Base$1.fork(meta.config) : Object.create(meta.config);
        if (VersionHelper.isTestEnv && BrowserHelper.isBrowserEnv && config.testConfig && globalThis.__applyTestConfigs) {
          for (const o in config.testConfig) {
            config[o] = config.testConfig[o];
          }
        }
        return config;
      }
      static fork(obj) {
        let ret = obj,
          key,
          value;
        if (obj && Objects.isObject(obj) && !Object.isFrozen(obj)) {
          ret = Object.create(obj);
          for (key in obj) {
            value = obj[key];
            if (value) {
              if (Objects.isObject(value)) {
                ret[key] = Base$1.fork(value);
              } else if (Array.isArray(value)) {
                ret[key] = value.slice();
              }
            }
          }
        }
        return ret;
      }
      /**
       * Gets the full {@link #property-properties-static} block for this class's entire inheritance chain
       * all the way up to but not including {@link Core.Base}
       * @returns {Object} All default config values for this class.
       * @private
       * @category Configuration
       */
      getProperties() {
        const
          // The meta.properties array is an array of classes that define "static get properties()"
          hierarchy = this.$meta.properties,
          result = {};
        for (let i = 0; i < hierarchy.length; i++) {
          // Gather the class result in top-down order so that subclass properties override superclass properties
          Object.assign(result, hierarchy[i].properties);
        }
        return result;
      }
      static get superclass() {
        return getPrototypeOf$1(this);
      }
      /**
       * Used by the Widget and GridFeatureManager class internally. Returns the class hierarchy of this object
       * starting from the `topClass` class (which defaults to `Base`).
       *
       * For example `classHierarchy(Widget)` on a Combo would yield `[Widget, Field, TextField, PickerField, Combo]`
       * @param {Function} [topClass] The topmost class constructor to start from.
       * @returns {Function[]} The class hierarchy of this instance.
       * @private
       * @category Configuration
       */
      classHierarchy(topClass) {
        const hierarchy = this.$meta.hierarchy,
          index = topClass ? hierarchy.indexOf(topClass) : 0;
        return index > 0 ? hierarchy.slice(index) : hierarchy;
      }
      /**
       * Checks if an obj is of type using object's $$name property and doing string comparison of the property with the
       * type parameter.
       *
       * @param {String} type
       * @returns {Boolean}
       * @category Misc
       * @advanced
       */
      static isOfTypeName(type) {
        return this.$meta.names.includes(type);
      }
      /**
       * Removes all event listeners that were registered with the given `name`.
       * @param {String|Symbol} name The name of the event listeners to be removed.
       * @category Events
       * @advanced
       */
      detachListeners(name) {
        var _detachers;
        let detachers = this.$detachers;
        detachers = (_detachers = detachers) === null || _detachers === void 0 ? void 0 : _detachers[name];
        if (detachers) {
          while (detachers.length) {
            detachers.pop()();
          }
        }
      }
      /**
       * Tracks a detacher function for the specified listener name.
       * @param {String} name The name assigned to the associated listeners.
       * @param {Function} detacher The detacher function.
       * @private
       */
      trackDetacher(name, detacher) {
        const detachers = this.$detachers || (this.$detachers = {}),
          bucket = detachers[name] || (detachers[name] = []);
        bucket.push(detacher);
      }
      /**
       * Removes all detacher functions for the specified `Events` object. This is called
       * by the `removeAllListeners` method on that object which is typically called by its
       * `destroy` invocation.
       * @param {Core.mixin.Events} eventer The `Events` instance to untrack.
       * @private
       */
      untrackDetachers(eventer) {
        const detachers = this.$detachers;
        if (detachers) {
          for (const name in detachers) {
            const bucket = detachers[name];
            for /* empty */
            (let i = bucket.length; i-- > 0;) {
              if (bucket[i].eventer === eventer) {
                bucket.splice(i, 1);
              }
            }
          }
        }
      }
    }
    const proto$2 = Base$1.prototype;
    // Informs the standard config setter there is no need to call this fn:
    proto$2.onConfigChange.$nullFn = emptyFn$3.$nullFn = true;
    Base$1[metaSymbol] = proto$2.$meta = newMeta({
      class: Base$1,
      config: Object.freeze({}),
      configs: Object.create(null),
      declarables: Base$1.declarable,
      forkConfigs: false,
      hierarchy: Object.freeze([Base$1]),
      names: Object.freeze(['Base']),
      nullify: null,
      properties: Object.freeze([]),
      super: null
    });
    // Avoid some object shape changes:
    Object.assign(proto$2, {
      $detachers: null,
      configObserver: null,
      /**
       * This property is set to `true` before the `constructor` returns.
       * @member {Boolean}
       * @readonly
       * @category Lifecycle
       * @advanced
       */
      isConstructing: true,
      /**
       * This property is set to `true` by {@link #function-destroy} after destruction.
       *
       * It is also one of the few properties that remains on the object after returning from `destroy()`. This property
       * is often checked in code paths that may encounter a destroyed object (like some event handlers) or in the
       * destruction path during cleanup.
       *
       * @member {Boolean}
       * @readonly
       * @category Lifecycle
       */
      isDestroyed: false,
      /**
       * This property is set to `true` on entry to the {@link #function-destroy} method. It remains on the objects after
       * returning from `destroy()`. If {@link #property-isDestroyed} is `true`, this property will also be `true`, so
       * there is no need to test for both (for example, `comp.isDestroying || comp.isDestroyed`).
       * @member {Boolean}
       * @readonly
       * @category Lifecycle
       * @advanced
       */
      isDestroying: false
    });
    Base$1.emptyFn = emptyFn$3;
    VersionHelper.setVersion('core', '5.5.0');
    Base$1._$name = 'Base';

    /**
     * @module Core/helper/ArrayHelper
     */
    /**
     * Helper with useful functions for handling Arrays
     * @internal
     */
    class ArrayHelper {
      static clean(array) {
        return array.reduce((res, item) => {
          if (item !== null && item !== undefined && !(Array.isArray(item) && item.length === 0) && item !== '') res.push(item);
          return res;
        }, []);
      }
      /**
       * Similar to [`Array.from()`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/from)
       * this method creates an array from an `iterable` object. Where `Array.from()` accepts a mapper function as the
       * second argument, this method accepts a `filter` function as its second argument. If a mapper function is also
       * needed, it can be passed as the third argument. Unlike `Array.from()`, if this method is passed `null`, it will
       * return an empty array.
       * @param {Array} iterable The iterable object to convert (must support `for-of` loop iteration).
       * @param {Function} [filter] A function to apply to each item of the `iterable` which must return a truthy value
       * to include that item in the resulting array.
       * @param {Function} [map] A function to apply to each item of the `iterable` that returns the actual value to put
       * into the returned array. If a `filter` is also supplied, this method is only called for those items that pass
       * the filter test.
       * @returns {Array}
       */
      static from(iterable, filter, map) {
        const array = [];
        if (iterable) {
          for (const it of iterable) {
            if (!filter || filter(it)) {
              array.push(map ? map(it) : it);
            }
          }
        }
        return array;
      }
      /**
       * Remove one or more items from an array
       * @param {Array} array Array to remove from
       * @param {Object[]|Set} items One or more items to remove, or one Set containing items to remove
       * @returns {Boolean} Returns true if any item was removed
       */
      static remove(array, ...items) {
        let index,
          item,
          removed = false;
        items = items[0] instanceof Set ? [...items[0]] : items;
        for (let i = 0; i < items.length; i++) {
          item = items[i];
          if ((index = array.indexOf(item)) !== -1) {
            array.splice(index, 1);
            removed = true;
          }
        }
        return removed;
      }
      /**
       * Calculates the insertion index of a passed object into the passed Array according
       * to the passed comparator function. Note that the passed Array *MUST* already be ordered.
       * @param {Object} item The item to calculate the insertion index for.
       * @param {Array} array The array into which the item is to be inserted.
       * @param {Function} comparatorFn The comparison function. Must return -1 or 0 or 1.
       * @param {Object} comparatorFn.lhs The left object to compare.
       * @param {Object} comparatorFn.rhs The right object to compare.
       * @param {Number} index The possible correct index to try first before a binary
       * search is instigated.
       * @internal
       */
      static findInsertionIndex(item, array, comparatorFn = this.lexicalCompare, index) {
        const len = array.length;
        let beforeCheck, afterCheck;
        if (index < len) {
          beforeCheck = index > 0 ? comparatorFn(array[index - 1], item) : 0;
          afterCheck = index < len - 1 ? comparatorFn(item, array[index]) : 0;
          if (beforeCheck < 1 && afterCheck < 1) {
            return index;
          }
        }
        return this.binarySearch(array, item, comparatorFn);
      }
      /**
       * Similar to the native `Array.find()` call, but this finds the *last* element in the array for which
       * the passed function returns a truthy value.
       * @param {Object[]} array The array to find in.
       * @param {Function} fn The testing function.
       * @param {Object} [thisObj] The scope (`this` reference) in which to call the function.
       */
      static findLast(array, fn, thisObj) {
        for (let {
            length
          } = array, i = length - 1; i >= 0; i--) {
          if (fn.call(thisObj, array[i], i, array)) {
            return array[i];
          }
        }
      }
      /**
       * This method returns the index that a given item would be inserted into the
       * given (sorted) `array`. Note that the given `item` may or may not be in the
       * array. This method will return the index of where the item *should* be.
       *
       * For example:
       *
       *      var array = [ 'A', 'D', 'G', 'K', 'O', 'R', 'X' ];
       *      var index = ArrayHelper.binarySearch(array, 'E');
       *
       *      console.log('index: ' + index);
       *      // logs "index: 2"
       *
       *      array.splice(index, 0, 'E');
       *
       *      console.log('array : ' + array.join(''));
       *      // logs "array: ADEGKORX"
       *
       * @param {Object[]} array The array to search.
       * @param {Object} item The item that you want to insert into the `array`.
       * @param {Number} [begin=0] The first index in the `array` to consider.
       * @param {Number} [end=array.length] The index that marks the end of the range
       * to consider. The item at this index is *not* considered.
       * @param {Function} [compareFn] The comparison function that matches the sort
       * order of the `array`. The default `compareFn` compares items using less-than
       * and greater-than operators.
       * @returns {Number} The index for the given item in the given array based on
       * the passed `compareFn`.
       */
      static binarySearch(array, item, begin = 0, end = array.length, compareFn = this.lexicalCompare) {
        const length = array.length;
        let middle, comparison;
        if (begin instanceof Function) {
          compareFn = begin;
          begin = 0;
        } else if (end instanceof Function) {
          compareFn = end;
          end = length;
        }
        --end;
        while (begin <= end) {
          middle = begin + end >> 1;
          comparison = compareFn(item, array[middle]);
          if (comparison >= 0) {
            begin = middle + 1;
          } else if (comparison < 0) {
            end = middle - 1;
          }
        }
        return begin;
      }
      magnitudeCompare(lhs, rhs) {
        return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
      }
      lexicalCompare(lhs, rhs) {
        lhs = String(lhs);
        rhs = String(rhs);
        return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
      }
      /**
       * Similar to Array.prototype.fill(), but constructs a new array with the specified item count and fills it with
       * clones of the supplied item.
       * @param {Number} count Number of entries to create
       * @param {Object|Array} itemOrArray Item or array of items to clone (uses object spread to create shallow clone)
       * @param {Function} [fn] An optional function that is called for each item added, to allow processing
       * @returns {Array} A new populated array
       */
      static fill(count, itemOrArray = {}, fn = null) {
        const result = [],
          items = Array.isArray(itemOrArray) ? itemOrArray : [itemOrArray];
        for (let i = 0; i < count; i++) {
          for (const item of items) {
            // Using object spread here forces us to use more babel plugins and will make
            // react_typescript demo very difficult to setup
            const processedItem = Object.assign({}, item);
            if (fn) {
              fn(processedItem, i);
            }
            result.push(processedItem);
          }
        }
        return result;
      }
      /**
       * Populates an array with the return value from `fn`.
       * @param {Number} count Number of entries to create
       * @param {Function} fn A function that is called `count` times, return value is added to array
       * @param {Number} fn.index Current index in the array
       * @privateparam {Boolean} [oneBased] Add 1 to the index before calling the fn (making it 1 based)
       * @returns {Array} A new populated array
       */
      static populate(count, fn, oneBased = false) {
        const items = [];
        for (let i = 0; i < count; i++) {
          items.push(fn(i + (oneBased ? 1 : 0)));
        }
        return items;
      }
      /**
       * Pushes `item` on to the `array` if not already included
       * @param {Array}  array Array to push to
       * @param {...Object} items Item(s) to push if not already included
       */
      static include(array, ...items) {
        for (const item of items) {
          if (!array.includes(item)) {
            array.push(item);
          }
        }
      }
      /**
       * Returns a new array with the unique items from the supplied array.
       * @param {Array} array Input array
       * @returns {Array} New array with unique items
       */
      static unique(array) {
        return [...new Set(array)];
      }
      // Kept for future reference : Wanted to create an indexer on Stores.
      static allowNegative(array) {
        // From https://github.com/sindresorhus/negative-array
        return new Proxy(array, {
          get(target, name, receiver) {
            if (typeof name !== 'string') {
              return Reflect.get(target, name, receiver);
            }
            const index = Number(name);
            if (Number.isNaN(index)) {
              return Reflect.get(target, name, receiver);
            }
            return target[index < 0 ? target.length + index : index];
          },
          set(target, name, value, receiver) {
            if (typeof name !== 'string') {
              return Reflect.set(target, name, value, receiver);
            }
            const index = Number(name);
            if (Number.isNaN(index)) {
              return Reflect.set(target, name, value, receiver);
            }
            target[index < 0 ? target.length + index : index] = value;
            return true;
          }
        });
      }
      static delta(a, b, useRelativeNaming = false) {
        // Nicer syntax but about 40% slower (an extra iteration)
        // const
        //     onlyInA = a.filter(item => !b.includes(item)),
        //     onlyInB = b.filter(item => !a.includes(item)),
        //     inBoth  = a.filter(item => b.includes(item));
        // Quick bailout for nonexisting target array
        if (!b) {
          return useRelativeNaming ? {
            toAdd: a,
            toRemove: [],
            toKeep: []
          } : {
            onlyInA: a,
            onlyInB: [],
            inBoth: []
          };
        }
        const onlyInA = [],
          onlyInB = [],
          inBoth = new Set(),
          bSet = new Set(b);
        for (let i = 0; i < a.length; i++) {
          const item = a[i];
          if (bSet.has(item)) {
            inBoth.add(item);
          } else {
            onlyInA.push(item);
          }
        }
        for (let i = 0; i < b.length; i++) {
          const item = b[i];
          if (!inBoth.has(item)) {
            onlyInB.push(item);
          }
        }
        if (useRelativeNaming) {
          return {
            toAdd: onlyInA,
            toRemove: onlyInB,
            toKeep: inBoth
          };
        }
        return {
          onlyInA,
          onlyInB,
          inBoth: [...inBoth]
        };
      }
      /**
       * Returns the passed object wrapped in an array. Special handling of the following cases:
       * * Passing an array returns it as is
       * * Passing a `Set` returns it converted to an Array
       * * Passing `null`/`undefined` returns the passed value
       *
       * ```javascript
       * const records = ArrayHelper.asArray(record);
       *
       * // { id : 1 } -> [{ id : 1 }]
       * // [{ id : 1 }] -> [{ id : 1 }]
       * ```
       *
       * @param {*} arrayOrObject
       * @returns {Array|null}
       * @internal
       */
      static asArray(arrayOrObject) {
        if (arrayOrObject == null) {
          return arrayOrObject;
        }
        if (arrayOrObject instanceof Set) {
          return Array.from(arrayOrObject);
        }
        return Array.isArray(arrayOrObject) ? arrayOrObject : [arrayOrObject];
      }
      /**
       * Identity function that returns its input.
       * @param {*} Any value
       * @returns {*} The input value
       * @internal
       */
      static identity(x) {
        return x;
      }
      /**
       * Transform an array into a key:value dictionary using the specified
       * key and value getters. Does not group values, so only one result will
       * appear in the output for a given key.
       *
       * ```javascript
       * const input = [{
       *     id: '1',
       *     other: 'one'
       * },{
       *     id: '2',
       *     other: 'two'
       * }];
       *
       * keyBy(input, rec => rec.id)
       *
       * // {
       * //    '1': { id: '1', other: 'one' },
       * //    '2': { id: '2', other: 'two' }
       * // }
       * ```
       *
       * @param {Array} array Array from which to build dictionary
       * @param {Function|String} getKey Function to produce the key for a given array entry, or a string property name to read
       * @param {Function} getValue Optional function to transform array elements. Defaults to returning the element itself
       * @returns {Object} A key->value dictionary with keys as returned by `getKey` and values returned by `getValue`
       * @internal
       */
      static keyBy(array, keyGetter, valueGetter = ArrayHelper.identity) {
        const getKey = typeof keyGetter === 'string' ? o => o === null || o === void 0 ? void 0 : o[keyGetter] : keyGetter,
          getValue = typeof valueGetter === 'string' ? o => o === null || o === void 0 ? void 0 : o[valueGetter] : valueGetter;
        return array.reduce((dict, next) => {
          dict[getKey(next)] = getValue(next);
          return dict;
        }, {});
      }
      /**
       * Combines provided arrays of by aggregating their element values.
       * For example the below code sums up numeric elements of the arrays:
       *
       * ```javascript
       * ArrayHelper.aggregate(
       *     [
       *         [0,   1,  2, 33]
       *         [10,  1, -1],
       *         [100, 1, -1]
       *     ],
       *     entry => entry || 0, // "|| 0" here to make it work for different array sizes
       *     (aggregated, entry) => aggregated + entry, // aggregate by summing up
       *     () => 0 //initial value is zero
       * );
       *
       * // returns [111, 3, 0, 33] array
       * ```
       *
       * @param {Array[]} arrays Array to combine
       * @param {Function} getEntryValueFn Function that extracts an array entry for aggregating.
       * @param {Function} aggregatorFn A function to execute for each element in the arrays. It's purpose is to
       * aggregate the element value to the corresponding entry of the resulting array.
       * The function's return value becomes the value of the `aggregated` parameter on the next invocation of
       * `aggregatorFn`.
       * The function is called with the following arguments:
       *
       * @param {Object} aggregatorFn.aggregated Resulting array entry value. On the first call
       * `getInitialValueFn` result.
       * @param {Object} aggregatorFn.entry Current entry to aggregate into `aggregated`.
       * @param {Number} aggregatorFn.arrayIndex Index of current array (in the provided `arrays`).
       * @param {Object[]} aggregatorFn.entryIndex Index of the current entry.
       * @param {Object} aggregationContext A shared object providing extra aggregation call context.
       * @param {Function} getInitialValueFn Function that returns an initial value for the combined array entries.
       * @param {Object} [aggregationContext] Optional object that is passed to all of the above functions that can
       * be used for keeping some additional parameters used when aggregating.
       * Out of the box the object will contain `arrays` and `targetArray` properties containing
       * input and resulting arrays respectively.
       */
      static aggregate(arrays, getEntryValueFn, aggregatorFn, getInitialValueFn, aggregationContext = {}) {
        const rowLength = arrays.length,
          columnLength = arrays[0].length,
          result = [];
        // provide some context by default
        // the resulting array reference and list of array given for aggregating
        aggregationContext.targetArray = result;
        aggregationContext.arrays = arrays;
        for (let colIndex = 0; colIndex < columnLength; colIndex++) {
          aggregationContext.entryIndex = colIndex;
          result.push(getInitialValueFn(colIndex, aggregationContext));
        }
        // iterate provided arrays
        for (let rowIndex = 0; rowIndex < rowLength; rowIndex++) {
          const row = arrays[rowIndex];
          aggregationContext.rowIndex = rowIndex;
          for (let colIndex = 0; colIndex < columnLength; colIndex++) {
            aggregationContext.entryIndex = colIndex;
            const entryValue = getEntryValueFn(row[colIndex], aggregationContext);
            // aggregate them
            result[colIndex] = aggregatorFn(result[colIndex], entryValue, rowIndex, colIndex, aggregationContext);
          }
        }
        return result;
      }
      /**
       * Group an array by keys (either the values in a specified property name, or the results of a string-generating function accepting
       * an array entry as input), returning an Object with those keys, whose values are arrays containing the array entries that
       * produced that key.
       *
       * ```javascript
       * const input = [{
       *     id: 1,
       *     color: 'red'
       * },{
       *     id: 2,
       *     color: 'green'
       * },{
       *     id: 3,
       *     color: 'green'
       * }];
       *
       * groupBy(input, 'color')
       *
       * // {
       * //    'red': [ { id: '1', color: 'red' } ],
       * //    'green': [ { id: '2', color: 'green' }, { id: '3', color: 'green' } ]
       * // }
       *
       * groupBy(input, rec => rec.color?.substr(0, 1))
       *
       * // {
       * //    'r': [ { id: '1', color: 'red' } ],
       * //    'g': [ { id: '2', color: 'green' }, { id: '3', color: 'green' } ]
       * // }
       * ```
       *
       * @param {Object} array Array from which to build dictionary
       * @param {Function|String} getGroupKey Function to produce the key for a given array entry, or a string property name to read
       * @returns {Object} A key->value[] dictionary with keys as returned by `getKey` and arrays of matching items in original order as values
       * @internal
       */
      static groupBy(array, groupKeyGetter) {
        const getKey = typeof groupKeyGetter === 'string' ? o => o === null || o === void 0 ? void 0 : o[groupKeyGetter] : groupKeyGetter;
        return array.reduce((grouped, value) => {
          const key = getKey(value);
          (grouped[key] = grouped[key] || []).push(value);
          return grouped;
        }, {});
      }
    }
    ArrayHelper._$name = 'ArrayHelper';

    /**
     * @module Core/helper/FunctionHelper
     */
    const commaSepRe = /,\s*/,
      decompiledSym = Symbol('decompiled'),
      // [async] p => ...
      fnRe1 = /^\s*(async\s+)?([a-z_]\w*)\s*=>([\s\S]+)$/i,
      // [async] (p1?[, px]*) => ...
      fnRe2 = /^\s*(async\s*)?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)\s+=>([\s\S]+)$/i,
      // [async] [function] [name] (p1?[, px]*) ...
      fnRe3 = /^(\s*async)?(?:\s*function)?(?:\s*([a-z_]\w*))?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)([\s\S]+)$/i,
      {
        hasOwnProperty: hasOwnProperty$3
      } = Object.prototype;
    /**
     * Provides functionality for working with functions
     * @internal
     */
    class FunctionHelper {
      /**
       * Inserts a function after the specified `method` is called on an `object`. To remove this hook, invoke the
       * function returned by this method.
       * ```
       *  class A {
       *      method() {
       *          console.log('method');
       *      }
       *  }
       *
       *  let instance = new A();
       *
       *  let detach = FunctionHelper.after(instance, 'method', () => { console.log('after') });
       *
       *  instance.method();
       *  > method
       *  > after
       *
       *  detach();
       *  instance.method();
       *  > method
       * ```
       * The value returned by the original method is passed as the first argument to `fn` followed by all the arguments
       * passed by the caller.
       *
       * If `fn` returns a value (not `undefined`), that value is returned from the method call instead of the value
       * returned by the original method.
       * ```
       *  class A {
       *      method(x) {
       *          console.log('method', x);
       *          return x * 2
       *      }
       *  }
       *
       *  let instance = new A();
       *
       *  let detach = FunctionHelper.after(instance, 'method', (ret, x) => {
       *      console.log('after', ret, x);
       *      return x / 2;
       *  });
       *
       *  console.log(instance.method(50));
       *  > method 50
       *  > after 100 50
       *  > 25
       *
       *  detach();
       *  console.log(instance.method(50));
       *  > method 50
       *  > 100
       * ```
       *
       * @param {Object} object The object to hook.
       * @param {String} method The name of the method on `object` to hook.
       * @param {Function|String} fn The function or method name (on `thisObj`) to call after `method`.
       * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
       * @param {Object} [options] Additional options
       * @param {Boolean} [options.return=true] Specify `false` to not include the return value of the hooked method as
       * the first argument to `fn`.
       * @returns {Function} The function to call to remove the hook.
       */
      static after(object, method, fn, thisObj, options) {
        const named = typeof fn === 'string',
          withReturn = (options === null || options === void 0 ? void 0 : options.return) !== false,
          hook = (...args) => {
            const
              // if object.destroy() occurs, our hook will be removed, so this fn won't be called in that case
              origResult = hook.$nextHook.call(object, ...args),
              hookResult = thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed ? undefined : withReturn ? named ? thisObj[fn](origResult, ...args) : fn.call(thisObj, origResult, ...args) : named ? thisObj[fn](...args) : fn.call(thisObj, ...args);
            return hookResult === undefined ? origResult : hookResult;
          };
        return FunctionHelper.hookMethod(object, method, hook);
      }
      /**
       * Inserts a function before the specified `method` is called on an `object`. To remove this hook, invoke the
       * function returned by this method.
       * ```
       *  class A {
       *      method() {
       *          console.log('method');
       *      }
       *  }
       *
       *  let instance = new A();
       *
       *  let detach = FunctionHelper.before(instance, 'method', () => { console.log('before') });
       *
       *  instance.method();
       *  > before
       *  > method
       *
       *  detach();
       *  instance.method();
       *  > method
       * ```
       * If `fn` returns `false`, the original method is not invoked and `false` is returned to the caller.
       * ```
       *  class A {
       *      method(x) {
       *          console.log('method', x);
       *          return x * 2;
       *      }
       *  }
       *
       *  let instance = new A();
       *
       *  let detach = FunctionHelper.before(instance, 'method', x => {
       *      console.log('before', x);
       *      return false;
       *  });
       *
       *  console.log(instance.method(50));
       *  > before 50
       *  > false
       *
       *  detach();
       *  console.log(instance.method(50));
       *  > method 50
       *  > 100
       * ```
       *
       * @param {Object} object The object to hook.
       * @param {String} method The name of the method on `object` to hook.
       * @param {Function|String} fn The function or method name (on `thisObj`) to call before `method`.
       * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
       * @returns {Function} The function to call to remove the hook.
       */
      static before(object, method, fn, thisObj) {
        const named = typeof fn === 'string',
          hook = (...args) => {
            const ret = thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed ? 0 : named ? thisObj[fn](...args) : fn.call(thisObj, ...args);
            return ret === false ? ret : hook.$nextHook.call(object, ...args);
          };
        return FunctionHelper.hookMethod(object, method, hook);
      }
      static curry(func) {
        return function curried(...args) {
          if (args.length >= func.length) {
            return func.apply(this, args);
          } else {
            return function (...args2) {
              return curried.apply(this, args.concat(args2));
            };
          }
        };
      }
      static bindAll(obj) {
        for (const key in obj) {
          if (typeof obj[key] === 'function') {
            obj[key] = obj[key].bind(obj);
          }
        }
      }
      /**
       * Returns a function which calls the passed `interceptor` function first, and the passed `original` after
       * as long as the `interceptor` does not return `false`.
       * @param {Function} original The function to call second.
       * @param {Function} interceptor The function to call first.
       * @param {Object} [thisObj] The `this` reference when the functions are called.
       * @returns {Function} A function which yields the return value from the `original` function **if it was called**, else `false`.
       */
      static createInterceptor(original, interceptor, thisObj) {
        return function (...args) {
          const theThis = thisObj || this;
          if (interceptor.call(theThis, ...args) !== false) {
            return original.call(theThis, ...args);
          }
          return false;
        };
      }
      /**
       * Returns a function which calls the passed `sequence` function after calling
       * the passed `original`.
       * @param {Function} original The function to call first.
       * @param {Function} sequence The function to call second.
       * @param {Object} [thisObj] The `this` reference when the functions are called.
       * @returns {Function} A function which yields the value returned from the sequence if it returned a value, else the return
       * value from the original function.
       */
      static createSequence(original, sequence, thisObj) {
        return (...args) => {
          const origResult = original.call(thisObj, ...args),
            sequenceResult = sequence.call(thisObj, ...args);
          return sequenceResult === undefined ? origResult : sequenceResult;
        };
      }
      /**
       * Create a "debounced" function which will call on the "leading edge" of a timer period.
       * When first invoked will call immediately, but invocations after that inside its buffer
       * period will be rejected, and *one* invocation will be made after the buffer period has expired.
       *
       * This is useful for responding immediately to a first mousemove, but from then on, only
       * calling the action function on a regular timer while the mouse continues to move.
       *
       * @param {Function} fn The function to call.
       * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
       * @param {Object} [thisObj] `this` reference for the function.
       * @param {Array} [extraArgs] The argument list to append to those passed to the function.
       * @param {Function} [alt] A function to call when the invocation is rejected due to buffer time not having expired.
       * @returns {Function} A function which calls the passed `fn` only if at least the passed `buffer`
       * milliseconds has elapsed since its last invocation.
       */
      static createThrottled(fn, buffer, thisObj, extraArgs, alt) {
        let lastCallTime = -Number.MAX_VALUE,
          callArgs,
          timerId;
        const invoke = () => {
            timerId = 0;
            lastCallTime = performance.now();
            callArgs.push.apply(callArgs, extraArgs);
            fn.apply(thisObj, callArgs);
          },
          result = function (...args) {
            const elapsed = performance.now() - lastCallTime;
            callArgs = args;
            // If it's been more then the buffer period since we invoked, we can call it now
            if (elapsed >= buffer) {
              clearTimeout(timerId);
              invoke();
            }
            // Otherwise, kick off a timer for the requested period.
            else {
              if (!timerId) {
                timerId = setTimeout(invoke, buffer - elapsed);
              }
              if (alt) {
                callArgs.push.apply(callArgs, extraArgs);
                alt.apply(thisObj, callArgs);
              }
            }
          };
        result.cancel = () => clearTimeout(timerId);
        return result;
      }
      /**
       * Create a "debounced" function which will call on the "trailing edge" of a timer period.
       * When first invoked will wait until the buffer period has expired to call the function, and
       * more calls within that time will restart the timer.
       *
       * This is useful for responding to keystrokes, but deferring action until the user pauses typing.
       *
       * @param {Function} fn The function to call.
       * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
       * @param {Object} [thisObj] `this` reference for the function.
       * @param {Array} [args] The argument list to append to those passed to the function.
       * @returns {Function} A function which calls the passed `fn` when at least the passed `buffer`
       * milliseconds has elapsed since its last invocation.
       */
      static createBuffered(fn, buffer, thisObj, args) {
        let callArgs, timerId;
        const invoke = () => {
            timerId = 0;
            result.isPending = false;
            callArgs.push.apply(callArgs, args);
            fn.apply(thisObj, callArgs);
          },
          result = function (...args) {
            callArgs = args;
            // Cancel any impending invocation. It's pushed out for <buffer> ms from each call
            if (timerId) {
              clearTimeout(timerId);
            }
            result.isPending = true;
            timerId = setTimeout(invoke, buffer);
          };
        result.cancel = () => {
          result.isPending = false;
          clearTimeout(timerId);
        };
        return result;
      }
      static decompile(fn) {
        if (!(decompiledSym in fn)) {
          var _body, _body2;
          const code = fn.toString();
          let m = fnRe1.exec(code),
            args,
            body,
            name,
            decompiled,
            t;
          if (m) {
            // [async] p => ...
            //   [1]   [2]  [3]
            args = [m[2]];
            body = m[3];
          } else if (m /* assignment */ = fnRe2.exec(code)) {
            // [async] (p1?[, px]*) => ...
            //   [1]   [2]             [3]
            t = m[2].trim();
            args = t ? t.split(commaSepRe) : [];
            body = m[3];
          } else if (m /* assignment */ = fnRe3.exec(code)) {
            // [async] [function] [name] (p1?[, px]*) ...
            //   [1]              [2]     [3]         [4]
            name = m[2];
            t = m[3].trim();
            args = t ? t.split(commaSepRe) : [];
            body = m[4];
          }
          body = (_body = body) === null || _body === void 0 ? void 0 : _body.trim();
          fn[decompiledSym] = decompiled = m && {
            args,
            async: Boolean(m[1]),
            body: (_body2 = body) !== null && _body2 !== void 0 && _body2.startsWith('{') ? body.substring(1, body.length - 1).trim() : body
          };
          if (name) {
            decompiled.name = name;
          }
        }
        return fn[decompiledSym];
      }
      static hookMethod(object, method, hook) {
        hook.$nextHook = object[method];
        object[method] = hook;
        return () => {
          // Object will have no hooks on the instance if it is destroyed (perhaps other reasons too)
          if (hasOwnProperty$3.call(object, method)) {
            let f = object[method],
              next;
            if (f === hook) {
              var _Object$getPrototypeO;
              // When this is the outermost hook, we may be the last hook. If $nextHook is found on the object's
              // prototype, simply delete the slot to expose it. Otherwise, there's another hook, so make it the
              // outermost.
              if (((_Object$getPrototypeO = Object.getPrototypeOf(object)) === null || _Object$getPrototypeO === void 0 ? void 0 : _Object$getPrototypeO[method]) === hook.$nextHook) {
                delete object[method];
              } else {
                object[method] = hook.$nextHook;
              }
            } else {
              // Not being the outermost hook means we have outer hooks that should chain to the one we want to
              // remove. Be cautious because the object could be destroyed.
              for (; next = (_f = f) === null || _f === void 0 ? void 0 : _f.$nextHook; f = next) {
                var _f;
                if (next === hook) {
                  f.$nextHook = hook.$nextHook;
                  break;
                }
              }
            }
          }
        };
      }
      /**
       * Protects the specified `method` on a given `object` such that calling it will not throw exceptions.
       * @param {Object} object The object whose method is to be protected.
       * @param {String} method The name of the method to protect.
       * @param {Function} [handler] An optional function to call for any thrown exceptions.
       * @internal
       */
      static noThrow(object, method, handler) {
        const fn = object[method];
        object[method] = (...args) => {
          try {
            return fn.apply(object, args);
          } catch (e) {
            return handler === null || handler === void 0 ? void 0 : handler(e);
          }
        };
      }
      static returnTrue() {
        return true;
      }
      static animate(duration, fn, thisObj, easing = 'linear') {
        let cancel = false;
        const result = new Promise(resolve => {
          const start = performance.now(),
            iterate = () => {
              const progress = Math.min((performance.now() - start) / duration, 1),
                delayable = thisObj && thisObj.setTimeout ? thisObj : globalThis;
              if (!cancel) {
                if (fn.call(thisObj, this.easingFunctions[easing](progress)) === false) {
                  resolve();
                }
              }
              if (cancel || progress === 1) {
                // Push resolution into the next animation frame so that
                // this frame completes before the resolution handler runs.
                delayable.requestAnimationFrame(() => resolve());
              } else {
                delayable.requestAnimationFrame(iterate);
              }
            };
          iterate();
        });
        result.cancel = () => {
          cancel = true;
          result.cancelled = true;
          return false;
        };
        return result;
      }
    }
    const half = 0.5,
      e1 = 1.70158,
      e2 = 7.5625,
      e3 = 1.525,
      e4 = 2 / 2.75,
      e5 = 2.25 / 2.75,
      e6 = 1 / 2.75,
      e7 = 1.5 / 2.75,
      e8 = 2.5 / 2.75,
      e9 = 2.625 / 2.75,
      e10 = 0.75,
      e11 = 0.9375,
      e12 = 0.984375,
      s1 = 1.70158,
      s2 = 1.70158;
    FunctionHelper.easingFunctions = {
      linear: t => t,
      easeInQuad: t => Math.pow(t, 2),
      easeOutQuad: t => -(Math.pow(t - 1, 2) - 1),
      easeInOutQuad: t => (t /= half) < 1 ? half * Math.pow(t, 2) : -half * ((t -= 2) * t - 2),
      easeInCubic: t => Math.pow(t, 3),
      easeOutCubic: t => Math.pow(t - 1, 3) + 1,
      easeInOutCubic: t => (t /= half) < 1 ? half * Math.pow(t, 3) : half * (Math.pow(t - 2, 3) + 2),
      easeInQuart: t => Math.pow(t, 4),
      easeOutQuart: t => -(Math.pow(t - 1, 4) - 1),
      easeInOutQuart: t => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
      easeInQuint: t => Math.pow(t, 5),
      easeOutQuint: t => Math.pow(t - 1, 5) + 1,
      easeInOutQuint: t => (t /= half) < 1 ? half * Math.pow(t, 5) : half * (Math.pow(t - 2, 5) + 2),
      easeInSine: t => -Math.cos(t * (Math.PI / 2)) + 1,
      easeOutSine: t => Math.sin(t * (Math.PI / 2)),
      easeInOutSine: t => -half * (Math.cos(Math.PI * t) - 1),
      easeInExpo: t => t === 0 ? 0 : Math.pow(2, 10 * (t - 1)),
      easeOutExpo: t => t === 1 ? 1 : -Math.pow(2, -10 * t) + 1,
      easeInOutExpo: t => t === 0 ? 0 : t === 1 ? 1 : (t /= half) < 1 ? half * Math.pow(2, 10 * (t - 1)) : half * (-Math.pow(2, -10 * --t) + 2),
      easeInCirc: t => -(Math.sqrt(1 - t * t) - 1),
      easeOutCirc: t => Math.sqrt(1 - Math.pow(t - 1, 2)),
      easeInOutCirc: t => (t /= half) < 1 ? -half * (Math.sqrt(1 - t * t) - 1) : half * (Math.sqrt(1 - (t -= 2) * t) + 1),
      easeOutBounce: t => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
      easeInBack: t => t * t * ((e1 + 1) * t - e1),
      easeOutBack: t => (t = t - 1) * t * ((e1 + 1) * t + e1) + 1,
      easeInOutBack: t => {
        let v1 = s1;
        return (t /= half) < 1 ? half * (t * t * (((v1 *= e3) + 1) * t - v1)) : half * ((t -= 2) * t * (((v1 *= e3) + 1) * t + v1) + 2);
      },
      elastic: t => -1 * Math.pow(4, -8 * t) * Math.sin((t * 6 - 1) * (2 * Math.PI) / 2) + 1,
      swingFromTo: t => {
        let v2 = s2;
        return (t /= half) < 1 ? half * (t * t * (((v2 *= e3) + 1) * t - v2)) : half * ((t -= 2) * t * (((v2 *= e3) + 1) * t + v2) + 2);
      },
      swingFrom: t => t * t * ((e1 + 1) * t - e1),
      swingTo: t => (t -= 1) * t * ((e1 + 1) * t + e1) + 1,
      bounce: t => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
      bouncePast: t => t < e6 ? e2 * t * t : t < e4 ? 2 - (e2 * (t -= e7) * t + e10) : t < e8 ? 2 - (e2 * (t -= e5) * t + e11) : 2 - (e2 * (t -= e9) * t + e12),
      easeFromTo: t => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
      easeFrom: t => Math.pow(t, 4),
      easeTo: t => Math.pow(t, 0.25)
    };
    FunctionHelper._$name = 'FunctionHelper';

    /* eslint-disable bryntum/no-listeners-in-lib,bryntum/no-on-in-lib */
    /**
     * @module Core/mixin/Events
     */
    /**
     * @typedef {Object.<String,Function|Boolean|Object|Object[]|Number|String>} BryntumListenerConfig
     * @property {Object} [thisObj] The `this` reference for all listeners. May be overridden if a handler is specified in object form.
     * @property {Boolean} [once] Specify as `true` to remove the listener as soon as it is invoked.
     * @property {Number|Object} [expires] The listener only waits for a specified time before
     * being removed. The value may be a number or an object containing an expiry handler.
     * @property {Number} [expires.delay] How long to wait for the event for.
     * @property {String|Function} [expires.alt] The function to call when the listener expires **without having been triggered**.
     * @property {Object[]} [args] An array of arguments to be passed to the handler before the event object.
     * @property {Number} [prio] The priority for all listeners; higher priority listeners are called before lower.
     * @property {Number} [buffer] A buffer time in milliseconds to wait after last event trigger to call the handler, to reduce the amount of handler calls for frequent events.
     * @property {Number} [throttle] A millisecond timeout value to throttle event triggering. With it specified a handler
     * will be called once immediately and then all following calls during the timeout period will be grouped together into one call once per throttle period.
     */
    const
      // Used by the config system to flatten configs from the class hierarchy.
      // In this case, a pure merge is not wanted. Listener definitions from
      // every class level are collected up into an array.
      // addListener iterates any passed array, adding each element.
      {
        isArray: isArray$1
      } = Array,
      {
        hasOwnProperty: hasOwnProperty$2
      } = Object.prototype,
      // Used to distinguish event names from listener options in addListener object config.
      specialProperties = {
        thisObj: 1,
        detachable: 1,
        once: 1,
        detacher: 1,
        prio: 1,
        args: 1,
        expires: 1,
        buffer: 1,
        throttle: 1,
        name: 1,
        $internal: 1
      },
      priorityComparator = (a, b) => b.prio - a.prio;
    /**
     * Mix this into another class to enable event handling.
     *
     * ## Basic usage
     * Listeners can be added either through config:
     *
     * ```javascript
     * let button = new Button({
     *   listeners: {
     *     click: () => {},
     *     press: () => {},
     *     ...
     *   }
     * });
     * ```
     *
     * *NOTE*: Do not reuse listeners config object, use new every time:
     * ```javascript
     * // wrong
     * let config = { click : () => {} }
     * new Button({
     *     listeners : config
     * })
     * new Button({
     *     listeners : config
     * })
     * // right
     * new Button({
     *     listeners : { click : () => {} }
     * })
     * new Button({
     *     listeners : { click : () => {} }
     * })
     * ```
     *
     * Or by calling on()/addListener():
     *
     * ```javascript
     * let button = new Button();
     *
     * button.addListener('press', () => {});
     * // on is an alias for addListener
     * button.on('click', () => {});
     * ```
     *
     * This style also accepts multiple listeners in same way as when using config:
     *
     * ```javascript
     * button.on({
     *   click: () => {},
     *   press: () => {},
     *   ...
     * });
     * ```
     *
     * ### Handlers as function name
     *
     * Event handlers may be specified as a function __name__. If a string is specified, it is the name
     * of the function in the `thisObj` object.
     *
     * If the string begins with `up.`, the owning object's ownership hierarchy
     * (if present) is scanned for an object which implements that function name:
     *
     * ```javascript
     * new Popup({
     *     tbar : {
     *         items : {
     *             myCombo : {
     *                 type      : 'combo',
     *                 editable  : false,
     *                 label     : 'Type',
     *                 listeners : {
     *                     // Look in owner chain for this function name
     *                     change : 'up.onFilterChange'
     *                 },
     *                 items     : [
     *                     'Event',
     *                     'Task',
     *                     'Appointment'
     *                 ]
     *             }
     *         }
     *     },
     *     items : {
     *         ...
     *     },
     *     onFilterChange({ value }) {
     *         // Handle event type selection here
     *     }
     * });
     *```
     *
     * ## Listener options
     * ### Once
     * Listeners can be configured to automatically deregister after first trigger by specifying config option `once`:
     *
     * ```javascript
     * button.on({
     *   click: () => {},
     *   once: true
     * });
     * ```
     *
     * ### Priority
     * Specifying priority affects the order in which listeners are called when triggering an event. Higher priorities will be
     * called before lower. Default value is 0.
     *
     * ```javascript
     * button.on({
     *   click: this.onClick,
     *   prio: 1
     * });
     * ```
     *
     * ### This reference
     * If desired, you can specify thisObj when configuring listeners. There is no need if you are using arrow functions as
     * listeners, but might be handy in other cases. Of course, you can also use bind to set `this` reference.
     *
     * ```javascript
     * button.on({
     *   click: this.onClick,
     *   thisObj: this
     * });
     *
     * // or
     *
     * button.on({
     *   click: this.onClick.bind(this)
     * });
     * ```
     *
     * ### Buffering
     * By specifying a `buffer` events that fire frequently can be grouped together and delayed. A handler for the event will be called once only, when no new event has been fired during the specified buffer time:
     *
     * ```javascript
     * button.on({
     *   click  : this.onClick,
     *   buffer : 200 // in milliseconds
     * });
     * ```
     *
     * In this example, if a user clicked the button 6 times very fast (<200ms between each click),
     * the `this.onClick` handler would be called only once 200 milliseconds after the last click.
     *
     * ### Throttling
     * Create a "debounced" function which will call on the "leading edge" of a timer period.
     * When first invoked will call immediately, but invocations after that inside its buffer
     * period will be rejected, and *one* invocation will be made after the buffer period has expired.
     *
     * This is useful for responding immediately to a first mousemove, but from then on, only
     * calling the action function on a regular timer while the mouse continues to move.
     * ```javascript
     * button.on({
     *   click    : this.onClick,
     *   throttle : 200 // in milliseconds
     * });
     * ```
     *
     * In this example, if a user clicked the button 6 times very fast, the `this.onClick` handler would be called once immediately on the first click and a second time 200 milliseconds after the **first** click.
     * So in reality the `click` event handler will be called every 200ms independent of amount of click in a middle, if the event was triggered at least once during the `throttle` timeout.
     *
     * ### Detacher
     * A convenient way of unregistering events is to use a detacher, a function returned when adding listeners that you
     * call later to deregister them. As of version 1.0, detachable defaults to true.
     *
     * ```javascript
     * let detacher = button.on({
     *   click: () => {},
     *   press: () => {},
     *   detachable: true
     * });
     *
     * // when you want to detach, for example in destroy()
     * detacher();
     * ```
     *
     * ### Auto detaching
     * When listeners are bound to a class instance using `thisObj`, the `thisObj`'s `doDestroy` method
     * is overridden to remove the listeners before calling the overridden doDestroy.
     *
     * ```javascript
     * class MyClass extends Base {
     *   construct() {
     *     let button = new Button({
     *       listeners: {
     *         click: () => {},
     *         thisObj: this
     *       }
     *     });
     *   }
     *
     *   doDestroy() {
     *     // clean up stuff
     *   }
     * }
     *
     * let myObj = new MyClass();
     * // clean up, also removes listeners
     * myObj.destroy();
     * ```
     *
     * ### On-functions
     * When mixing Events into another class it can be configured to call on-functions when events are triggered.
     * On-functions are functions named 'onEventName', for example 'onClick', 'onPress' declared on the class triggering
     * the event.
     *
     * ```javascript
     * // mix Events in with on-functions activated
     * let button = new Button({
     *   callOnFunctions: true,
     *
     *   onClick: () => {}
     * });
     *
     * // or add a getter in class declaration
     * ```
     *
     * Returning `false` from an on-function will prevent triggering listeners for the event.
     *
     * ### Catching all events
     * By specifying a listener for {@link #event-catchAll catchAll} a function can be notified when any event is triggered:
     *
     * ```javascript
     * const button = new Button({
     *    listeners : {
     *        catchAll(event) {
     *            // All events on the button will pass through here
     *        }
     *    }
     * });
     * ```
     *
     * ## Preventable events
     *
     * By returning `false` from a listener for an event documented as `preventable` the action that would otherwise be
     * executed after the event is prevented. These events are usually named `beforeXX`, for example `beforeRemove`,
     * `beforeDragStart` etc.
     *
     * <div class="note">Note that Angular does not support return values from listeners. Instead, assign to
     * <code>event.returnValue</code> as shown in the Angular snippet below</div>
     *
     * {@frameworktabs}
     * {@js}
     * ```javascript
     * taskBoard.on({
     *     beforeColumnDrag({ columnRecord }) {
     *         if (columnRecord.locked) {
     *             return false;
     *         }
     *     }
     * });
     * ```
     *
     * {@endjs}
     * {@react}
     *
     * ```jsx
     * const App = props => {
     *     function onBeforeColumnDrag({ columnRecord }) {
     *         if (columnRecord.locked) {
     *             return false;
     *         }
     *     }
     *
     *     return (
     *         <>
     *             <BryntumTaskBoard onBeforeColumnDrag={onBeforeColumnDrag} />
     *         </>
     *     )
     * }
     * ```
     *
     * {@endreact}
     * {@vue}
     *
     * ```html
     * <bryntum-task-board @beforeColumnDrag="onBeforeColumnDrag" />
     * ```
     *
     * ```javascript
     * export default {
     *     methods : {
     *         onBeforeColumnDrag({ columnRecord }) {
     *             if (columnRecord.locked) {
     *                 return false;
     *             }
     *         }
     *    }
     * }
     * ```
     *
     * {@endvue}
     * {@angular}
     *
     * ```html
     * <bryntum-task-board (onBeforeColumnDrag)="onBeforeColumnDrag({event : $event})"></bryntum-task-board>
     * ```
     *
     * ```typescript
     * export class AppComponent {
     *     onBeforeColumnDrag({ event }: { event: any }): void {
     *         event.returnValue = !event.columnRecord.locked;
     *     }
     *  }
     * ```
     *
     * {@endangular}
     * {@endframeworktabs}
     *
     * @mixin
     */
    var Events = (Target => class Events extends (Target || Base$1) {
      eventsSuspended = null;
      static get $name() {
        return 'Events';
      }
      //region Events
      /**
       * Fires before an object is destroyed.
       * @event beforeDestroy
       * @param {Core.Base} source The Object that is being destroyed.
       */
      /**
       * Fires when an object is destroyed.
       * @event destroy
       * @param {Core.Base} source The Object that is being destroyed.
       */
      /**
       * Fires when any other event is fired from the object.
       *
       * **Note**: `catchAll` is fired for both public and private events. Please rely on the public events only.
       * @event catchAll
       * @param {Object} event The Object that contains event details
       * @param {String} event.type The type of the event which is caught by the listener
       */
      //endregion
      static get declarable() {
        return [
        /**
         * The list of deprecated events as an object, where `key` is an event name which is deprecated and
         * `value` is an object which contains values for
         * {@link Core.helper.VersionHelper#function-deprecate-static VersionHelper}:
         * - product {String} The name of the product;
         * - invalidAsOfVersion {String} The version where the offending code is invalid (when any compatibility
         *   layer is actually removed);
         * - message {String} Warning message to show to the developer using a deprecated API;
         *
         * For example:
         *
         * ```javascript
         * return {
         *     click : {
         *         product            : 'Grid',
         *         invalidAsOfVersion : '1.0.0',
         *         message            : 'click is deprecated!'
         *     }
         * }
         * ```
         *
         * @name deprecatedEvents
         * @returns {Object}
         * @static
         * @internal
         */
        'deprecatedEvents'];
      }
      static setupDeprecatedEvents(cls, meta) {
        const all = meta.getInherited('deprecatedEvents'),
          add = cls.deprecatedEvents;
        for (const eventName in add) {
          // Event names are case-insensitive so build our map using toLowerCased names (but keep true case too):
          all[eventName.toLowerCase()] = all[eventName] = add[eventName];
        }
      }
      //region Config
      static get configurable() {
        return {
          /**
           * Set to true to call onXXX method names (e.g. `onShow`, `onClick`), as an easy way to listen for events.
           *
           * ```javascript
           * const container = new Container({
           *     callOnFunctions : true
           *
           *     onHide() {
           *          // Do something when the 'hide' event is fired
           *     }
           * });
           * ```
           *
           * @config {Boolean} callOnFunctions
           * @category Misc
           * @default false
           */
          /**
           * The listener set for this object.
           *
           * An object whose property names are the names of events to handle, or options which modifiy
           * __how__ the handlers are called.
           *
           * See {@link #function-addListener} for details about the options.
           *
           * Listeners can be specified in target class config and they will be merged with any listeners specified in
           * the instantiation config. Class listeners will be fired first:
           *
           * ```javascript
           * class MyStore extends Store({
           *     static get configurable() {
           *         return {
           *             listeners : {
           *                 myCustomEvent() {
           *                 },
           *                 load : {
           *                     prio : 10000,
           *                     fn() { // this load listener handles things first }
           *                 }
           *             }
           *         }
           *     }
           * });
           *
           * let store = new MyStore({
           *   listeners: {
           *     load: () => { // This load listener runs after the class's },
           *     ...
           *   }
           * });
           * ```
           *
           * ### Handlers as function name
           *
           * Object event handlers may be specified as a function __name__. If a string is specified, it is the name
           * of the function in the `thisObj` object.
           *
           * If the string begins with `up.`, this object's ownership hierarchy
           * (if present) is scanned for an object which implements that function name:
           *
           * ```javascript
           * new Popup({
           *     tbar : {
           *         items : {
           *             myCombo : {
           *                 type      : 'combo',
           *                 editable  : false,
           *                 label     : 'Type',
           *                 listeners : {
           *                     // Look in owner chain for this function name
           *                     change : 'up.onFilterChange'
           *                 },
           *                 items     : [
           *                     'Event',
           *                     'Task',
           *                     'Appointment'
           *                 ]
           *             }
           *         }
           *     },
           *     items : {
           *         ...
           *     },
           *     onFilterChange({ value }) {
           *         // Handle event type selection here
           *     }
           * });
           *```
           *
           * @config {Object}
           * @category Common
           */
          listeners: {
            value: null,
            $config: {
              merge(newValue, currentValue) {
                if (newValue !== null) {
                  if (!newValue) {
                    return currentValue;
                  }
                  if (currentValue) {
                    newValue = newValue ? [newValue] : [];
                    newValue.push[isArray$1(currentValue) ? 'apply' : 'call'](newValue, currentValue);
                  }
                }
                return newValue;
              }
            }
          },
          /**
           * Internal listeners, that cannot be removed by the user.
           * @config {Object}
           * @internal
           */
          internalListeners: null,
          /**
           * An object where property names with a truthy value indicate which events should bubble up the ownership
           * hierarchy when triggered.
           *
           * ```javascript
           * const container = new Container({
           *     items : [
           *        { type : 'text', bubbleEvents : { change : true }}
           *     ],
           *
           *     listeners : {
           *         change() {
           *             // Will catch change event from the text field
           *         }
           *     }
           * });
           * ```
           *
           * @config {Object}
           * @category Misc
           */
          bubbleEvents: null
        };
      }
      destroy() {
        this.trigger('beforeDestroy');
        super.destroy();
      }
      //endregion
      //region Init
      construct(config, ...args) {
        var _config;
        // Configured listeners use this as the thisObj
        if (this.configuredListeners /* assignment */ = (_config = config) === null || _config === void 0 ? void 0 : _config.listeners) {
          // We have to copy in case listeners have been forked
          config = Objects.assign({}, config);
          delete config.listeners;
        }
        super.construct(config, ...args);
        // Apply configured listeners after construction.
        // Note that some classes invoke this during parts of their construction.
        // Store invokes this prior to setting data so that observers are notified of data load.
        this.processConfiguredListeners();
      }
      processConfiguredListeners() {
        // This can only happen once
        if (this.configuredListeners) {
          const me = this,
            {
              isConfiguring
            } = me;
          // If called from config ingestion during configuration, listeners must be added
          // so temporarily clear the isConfiguring flag.
          me.isConfiguring = false;
          me.listeners = me.configuredListeners;
          me.configuredListeners = null;
          me.isConfiguring = isConfiguring;
        }
      }
      /**
       * Auto detaches listeners registered from start, if set as detachable
       * @internal
       */
      doDestroy() {
        this.trigger('destroy');
        this.removeAllListeners(false);
        super.doDestroy();
      }
      static setupClass(meta) {
        super.setupClass(meta);
        Events.prototype.onListen.$nullFn = true;
        Events.prototype.onUnlisten.$nullFn = true;
      }
      //endregion
      //region Listeners
      /**
       * Adds an event listener. This method accepts parameters in the following format:
       *
       * ```javascript
       *  myObject.addListener({
       *      thisObj    : this,          // The this reference for the handlers
       *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
       *      otherevent : {
       *          fn      : 'handlerFnName',
       *          once    : true          // Just this handler is auto-removed on fire
       *      },
       *      yetanother  : {
       *          fn      : 'yetAnotherHandler',
       *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
       *      },
       *      prio        : 100           // Higher prio listeners are called before lower
       *  });
       * ```
       *
       * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
       * When it is destroyed, those listeners are removed.
       *
       * The `config` parameter allows supplying options for the listener(s), for available options see {@link #typedef-BryntumListenerConfig}.
       *
       * A simpler signature may be used when only adding a listener for one event and no extra options
       * (such as `once` or `delay`) are required:
       *
       * ```javascript
       * myObject.addListener('click', myController.handleClicks, myController);
       * ```
       *
       * The args in this simple case are `eventName`, `handler` and `thisObj`
       *
       * @param {BryntumListenerConfig|String} config An object containing listener definitions, or the event name to listen for
       * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object, or the handler
       * function to call if providing a string as the first arg.
       * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
       * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
       */
      addListener(config, thisObj, oldThisObj) {
        if (isArray$1(config)) {
          for (let i = 0, {
              length
            } = config; i < length; i++) {
            this.addListener(config[i], thisObj);
          }
          return;
        }
        const me = this,
          deprecatedEvents = me.$meta.getInherited('deprecatedEvents');
        if (typeof config === 'string') {
          // arguments[2] is thisObj if (eventname, handler, thisObj) form called.
          // Note that the other side of the if compares to undefined, so this will work.
          return me.addListener({
            [config]: thisObj,
            detachable: thisObj.detachable !== false,
            thisObj: oldThisObj
          });
        } else {
          // Capture the default thisObj.
          thisObj = config.thisObj = config.thisObj !== undefined ? config.thisObj : thisObj;
          for (const key in config) {
            // Skip special properties or events without handlers (convenient syntax with optional handlers)
            if (!specialProperties[key] && config[key] != null) {
              var _me$afterAddListener;
              // comparing should be case insensitive
              const
                // comparing should be case insensitive
                eventName = key.toLowerCase(),
                deprecatedEvent = deprecatedEvents === null || deprecatedEvents === void 0 ? void 0 : deprecatedEvents[eventName],
                events = me.eventListeners || (me.eventListeners = {}),
                listenerSpec = config[key],
                expires = listenerSpec.expires || config.expires,
                listener = {
                  fn: typeof listenerSpec === 'object' ? listenerSpec.fn : listenerSpec,
                  thisObj: listenerSpec.thisObj !== undefined ? listenerSpec.thisObj : thisObj,
                  args: listenerSpec.args || config.args,
                  prio: listenerSpec.prio !== undefined ? listenerSpec.prio : config.prio !== undefined ? config.prio : 0,
                  once: listenerSpec.once !== undefined ? listenerSpec.once : config.once !== undefined ? config.once : false,
                  buffer: listenerSpec.buffer || config.buffer,
                  throttle: listenerSpec.throttle || config.throttle,
                  $internal: config.$internal,
                  catchAll: key === 'catchAll'
                };
              if (deprecatedEvent) {
                const {
                  product,
                  invalidAsOfVersion,
                  message
                } = deprecatedEvent;
                VersionHelper.deprecate(product, invalidAsOfVersion, message);
              }
              if (expires) {
                // Extract expires : { delay : 100, alt : 'onExpireFn' }
                const {
                    alt
                  } = expires,
                  delay = alt ? expires.delay : expires,
                  name = config.name || key,
                  fn = () => {
                    me.un(eventName, listener);
                    // If we make it here and the handler has not been called, invoke the alt handler
                    if (alt && !listener.called) {
                      me.callback(alt, thisObj);
                    }
                  };
                if (me.isDelayable) {
                  me.setTimeout({
                    fn,
                    name,
                    cancelOutstanding: true,
                    delay
                  });
                } else {
                  globalThis.setTimeout(fn, delay);
                }
              }
              let listeners = events[eventName] || (events[eventName] = []);
              if (listeners.$firing) {
                events[eventName] = listeners = listeners.slice();
              }
              // Insert listener directly in prio order
              listeners.splice(ArrayHelper.findInsertionIndex(listener, listeners, priorityComparator, listeners.length), 0, listener);
              if (!me.onListen.$nullFn && listeners.length < 2) {
                me.onListen(eventName);
              }
              // Hook to call when a listener is added
              (_me$afterAddListener = me.afterAddListener) === null || _me$afterAddListener === void 0 ? void 0 : _me$afterAddListener.call(me, eventName, listener);
            }
          }
          if (config.relayAll) {
            me.relayAll(config.relayAll);
          }
          // Hook into the thisObj's destruction sequence to remove these listeners.
          // Pass the default thisObj in for use when it comes to destruction time.
          if (thisObj && thisObj !== me) {
            me.attachAutoDetacher(config, thisObj);
          }
          const detachable = config.detachable !== false,
            name = config.name,
            destroy = config.expires || detachable || name ? () => {
              // drop listeners if not destroyed yet
              if (!me.isDestroyed) {
                me.removeListener(config, thisObj);
              }
            } : null;
          if (destroy) {
            var _thisObj;
            destroy.eventer = me;
            destroy.listenerName = name;
            if (name && (_thisObj = thisObj) !== null && _thisObj !== void 0 && _thisObj.trackDetacher) {
              thisObj.trackDetacher(name, destroy);
            }
            if (config.expires) {
              // handle expires : { alt : timeoutHandler, delay : 2000 }
              me.delay(destroy, isNaN(config.expires) ? config.expires.delay : config.expires, name);
            }
            if (detachable) {
              return destroy;
            }
          }
        }
      }
      /**
       * Alias for {@link #function-addListener}. Adds an event listener. This method accepts parameters in the following format:
       *
       * ```javascript
       *  myObject.on({
       *      thisObj    : this,          // The this reference for the handlers
       *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
       *      otherevent : {
       *          fn      : 'handlerFnName',
       *          once    : true          // Just this handler is auto-removed on fire
       *      },
       *      yetanother  : {
       *          fn      : 'yetAnotherHandler',
       *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
       *      },
       *      prio        : 100           // Higher prio listeners are called before lower
       *  });
       * ```
       *
       * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
       * When it is destroyed, those listeners are removed.
       *
       * The `config` parameter allows supplying options for the listener(s), for available options see {@link #typedef-BryntumListenerConfig}.
       *
       * A simpler signature may be used when only adding a listener for one event and no extra options
       * (such as `once` or `delay`) are required:
       *
       * ```javascript
       * myObject.on('click', myController.handleClicks, myController);
       * ```
       *
       * The args in this simple case are `eventName`, `handler` and `thisObj`
       *
       * @param {BryntumListenerConfig|String} config An object containing listener definitions, or the event name to listen for
       * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object, or the handler
       * function to call if providing a string as the first arg.
       * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
       * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
       */
      on(config, thisObj, oldThisObj) {
        return this.addListener(config, thisObj, oldThisObj);
      }
      /**
       * Internal convenience method for adding an internal listener, that cannot be removed by the user.
       *
       * Alias for `on({ $internal : true, ... })`. Only supports single argument form.
       *
       * @internal
       */
      ion(config) {
        config.$internal = true;
        return this.on(config);
      }
      /**
       * Shorthand for {@link #function-removeListener}
       * @param {Object|String} config A config object or the event name
       * @param {Object|Function} [thisObj] `this` reference for all listeners, or the listener function
       * @param {Object} [oldThisObj] `this` The `this` object for the legacy way of adding listeners
       */
      un(...args) {
        this.removeListener(...args);
      }
      updateInternalListeners(internalListeners, oldInternalListeners) {
        oldInternalListeners === null || oldInternalListeners === void 0 ? void 0 : oldInternalListeners.detach();
        if (internalListeners) {
          internalListeners.detach = this.ion(internalListeners);
        }
      }
      get listeners() {
        return this.eventListeners;
      }
      changeListeners(listeners) {
        // If we are receiving class listeners, add them early, and they do not become
        // the configured listeners, and are not removed by setting listeners during the lifecycle.
        if (this.isConfiguring) {
          // Pull in internal listeners first
          this.getConfig('internalListeners');
          if (listeners) {
            this.on(listeners, this);
          }
        }
        // Setting listeners after config time clears the old set and adds the new.
        // This will initially happen at the tail end of the constructor when config
        // listeners are set.
        else {
          // Configured listeners use this as the thisObj by default.
          // Flatten using Objects.assign because it may have been part of
          // a prototype chained default configuration of another object.
          // eg: the tooltip config of a Widget.
          // listener object blocks from multiple configuration levels are pushed
          // onto an array (see listeners merge function in configurable block above).
          // If this has happened, each entry must be processed like this.
          if (Array.isArray(listeners)) {
            for (let i = 0, l = listeners[0], {
                length
              } = listeners; i < length; l = listeners[++i]) {
              if (!('thisObj' in l)) {
                listeners[i] = Objects.assign({
                  thisObj: this
                }, l);
              }
            }
          } else if (listeners && !('thisObj' in listeners)) {
            listeners = Objects.assign({
              thisObj: this
            }, listeners);
          }
          return listeners;
        }
      }
      updateListeners(listeners, oldListeners) {
        // Only configured listeners get here. Class listeners are added by changeListeners.
        oldListeners && this.un(oldListeners);
        listeners && this.on(listeners);
      }
      /**
       * Removes an event listener. Same API signature as {@link #function-addListener}
       * @param {Object|String} config A config object or the event name
       * @param {Object|Function} thisObj `this` reference for all listeners, or the listener function
       * @param {Object} oldThisObj `this` The `this` object for the legacy way of adding listeners
       */
      removeListener(config, thisObj = config.thisObj, oldThisObj) {
        const me = this;
        if (typeof config === 'string') {
          return me.removeListener({
            [config]: thisObj
          }, oldThisObj);
        }
        Object.entries(config).forEach(([eventName, listenerToRemove]) => {
          if (!specialProperties[eventName] && listenerToRemove != null) {
            eventName = eventName.toLowerCase();
            const {
                eventListeners
              } = me,
              index = me.findListener(eventName, listenerToRemove, thisObj);
            if (index >= 0) {
              var _me$afterRemoveListen;
              let listeners = eventListeners[eventName];
              // Hook to call when a listener is removed (slightly before removing for now)
              (_me$afterRemoveListen = me.afterRemoveListener) === null || _me$afterRemoveListen === void 0 ? void 0 : _me$afterRemoveListen.call(me, eventName, listeners[index]);
              if (listeners.length > 1) {
                if (listeners.$firing) {
                  eventListeners[eventName] = listeners = listeners.slice();
                }
                // NOTE: we cannot untrack any detachers here because we may only be
                // removing some of its listeners
                listeners.splice(index, 1);
              } else {
                delete eventListeners[eventName];
                if (!me.onUnlisten.$nullFn) {
                  me.onUnlisten(eventName);
                }
              }
            }
          }
        });
        if (config.thisObj && !config.thisObj.isDestroyed) {
          me.detachAutoDetacher(config);
        }
      }
      /**
       * Finds the index of a particular listener to the named event. Returns `-1` if the passed
       * function/thisObj listener is not present.
       * @param {String} eventName The name of an event to find a listener for.
       * @param {String|Function} listenerToFind The handler function to find.
       * @param {Object} defaultThisObj The `thisObj` for the required listener.
       * @internal
       */
      findListener(eventName, listenerToFind, defaultThisObj) {
        var _this$eventListeners;
        const eventListeners = (_this$eventListeners = this.eventListeners) === null || _this$eventListeners === void 0 ? void 0 : _this$eventListeners[eventName],
          fn = listenerToFind.fn || listenerToFind,
          thisObj = listenerToFind.thisObj || defaultThisObj;
        if (eventListeners) {
          for (let listenerEntry, i = 0, {
              length
            } = eventListeners; i < length; i++) {
            listenerEntry = eventListeners[i];
            if (listenerEntry.fn === fn && listenerEntry.thisObj === thisObj) {
              return i;
            }
          }
        }
        return -1;
      }
      /**
       * Check if any listener is registered for the specified eventName
       * @param {String} eventName
       * @returns {Boolean} `true` if listener is registered, otherwise `false`
       * @advanced
       */
      hasListener(eventName) {
        var _this$eventListeners2;
        return Boolean((_this$eventListeners2 = this.eventListeners) === null || _this$eventListeners2 === void 0 ? void 0 : _this$eventListeners2[eventName === null || eventName === void 0 ? void 0 : eventName.toLowerCase()]);
      }
      /**
       * Relays all events through another object that also implements Events mixin. Adds a prefix to the event name
       * before relaying, for example add -> storeAdd
       * ```
       * // Relay all events from store through grid, will make it possible to listen for store events prefixed on grid:
       * 'storeLoad', 'storeChange', 'storeRemoveAll' etc.
       * store.relayAll(grid, 'store');
       *
       * grid.on('storeLoad', () => console.log('Store loaded');
       * ```
       * @param {Core.mixin.Events} through Object to relay the events through, needs to mix Events mixin in
       * @param {String} prefix Prefix to add to event name
       * @param {Boolean} [transformCase] Specify false to prevent making first letter of event name uppercase
       * @advanced
       */
      relayAll(through, prefix, transformCase = true) {
        if (!this.relayAllTargets) {
          this.relayAllTargets = [];
        }
        const {
          relayAllTargets
        } = this;
        through.ion({
          beforeDestroy: ({
            source
          }) => {
            if (source === through) {
              const configs = relayAllTargets.filter(r => r.through === through);
              configs.forEach(config => ArrayHelper.remove(relayAllTargets, config));
            }
          }
        });
        relayAllTargets.push({
          through,
          prefix,
          transformCase
        });
      }
      /**
       * Removes all listeners registered to this object by the application.
       */
      removeAllListeners(preserveInternal = true) {
        const listeners = this.eventListeners;
        let i, thisObj;
        for (const event in listeners) {
          const bucket = listeners[event];
          // We iterate backwards since we call removeListener which will splice out of
          // this array as we go...
          for /* empty */
          (i = bucket.length; i-- > 0;) {
            const cfg = bucket[i];
            if (!cfg.$internal || !preserveInternal) {
              var _thisObj2, _thisObj2$untrackDeta, _thisObj3;
              this.removeListener(event, cfg);
              thisObj = cfg.thisObj;
              (_thisObj2 = thisObj) === null || _thisObj2 === void 0 ? void 0 : (_thisObj2$untrackDeta = (_thisObj3 = _thisObj2).untrackDetachers) === null || _thisObj2$untrackDeta === void 0 ? void 0 : _thisObj2$untrackDeta.call(_thisObj3, this);
            }
          }
        }
      }
      relayEvents(source, eventNames, prefix = '') {
        const listenerConfig = {
          detachable: true,
          thisObj: this
        };
        eventNames.forEach(eventName => {
          listenerConfig[eventName] = (event, ...params) => {
            return this.trigger(prefix + eventName, event, ...params);
          };
        });
        return source.on(listenerConfig);
      }
      /**
       * This method is called when the first listener for an event is added.
       * @param {String} eventName
       * @internal
       */
      onListen() {}
      /**
       * This method is called when the last listener for an event is removed.
       * @param {String} eventName
       * @internal
       */
      onUnlisten() {}
      destructorInterceptor() {
        const {
          autoDetachers,
          target,
          oldDestructor
        } = this;
        // Remove listeners first, so that they do not fire during destruction.
        // The observable being listened to by the thisObj may already have
        // been destroyed in a clean up sequence
        for (let i = 0; i < autoDetachers.length; i++) {
          const {
            dispatcher,
            config
          } = autoDetachers[i];
          if (!dispatcher.isDestroyed) {
            dispatcher.removeListener(config, target);
          }
        }
        oldDestructor.call(target);
      }
      /**
       * Internal function used to hook destroy() calls when using thisObj
       * @private
       */
      attachAutoDetacher(config, thisObj) {
        const target = config.thisObj || thisObj,
          // If it's a Bryntum Base subclass, hook doDestroy, otherwise, destroy
          destructorName = 'doDestroy' in target ? 'doDestroy' : 'destroy';
        if (destructorName in target) {
          let {
            $autoDetachers
          } = target;
          if (!$autoDetachers) {
            target.$autoDetachers = $autoDetachers = [];
          }
          if (!target.$oldDestructor) {
            target.$oldDestructor = target[destructorName];
            // Binding instead of using closure (used to use FunctionHelper.createInterceptor) to not retain target
            // when detaching manually
            target[destructorName] = this.destructorInterceptor.bind({
              autoDetachers: $autoDetachers,
              oldDestructor: target.$oldDestructor,
              target
            });
          }
          $autoDetachers.push({
            config,
            dispatcher: this
          });
        } else {
          target[destructorName] = () => {
            this.removeListener(config);
          };
        }
      }
      /**
       * Internal function used restore hooked destroy() calls when using thisObj
       * @private
       */
      detachAutoDetacher(config) {
        const target = config.thisObj;
        // Restore old destructor and remove from auto detachers only if we are not called as part of destruction.
        // (Altering $autoDetachers affects destruction iterating over them, breaking it. It is pointless to clean up
        // during destruction anyway, since everything gets removed)
        if (target.$oldDestructor && !target.isDestroying) {
          ArrayHelper.remove(target.$autoDetachers, target.$autoDetachers.find(detacher => detacher.config === config && detacher.dispatcher === this));
          if (!target.$autoDetachers.length) {
            target['doDestroy' in target ? 'doDestroy' : 'destroy'] = target.$oldDestructor;
            target.$oldDestructor = null;
          }
        }
      }
      //endregion
      //region Promise based workflow
      // experimental, used in tests to support async/await workflow
      await(eventName, options = {
        checkLog: true,
        resetLog: true,
        args: null
      }) {
        const me = this;
        if (options === false) {
          options = {
            checkLog: false
          };
        }
        const {
          args
        } = options;
        return new Promise(resolve => {
          var _me$_triggered;
          // check if previously triggered?
          if (options.checkLog && (_me$_triggered = me._triggered) !== null && _me$_triggered !== void 0 && _me$_triggered[eventName]) {
            // resolve immediately, no params though...
            resolve();
            // reset log to be able to await again
            if (options.resetLog) {
              me.clearLog(eventName);
            }
          }
          // This branch will listen for events until catches one with specific arguments
          if (args) {
            const detacher = me.on({
              [eventName]: (...params) => {
                // if args is a function use it to match arguments
                const argsOk = typeof args === 'function' ? args(...params) : Object.keys(args).every(key => {
                  return key in params[0] && params[0][key] === args[key];
                });
                if (argsOk) {
                  // resolve when event is fired with required arguments
                  resolve(...params);
                  // reset log to be able to await again
                  if (options.resetLog) {
                    me.clearLog(eventName);
                  }
                  detacher();
                }
              },
              prio: -10000 // Let others do their stuff first
            });
          } else {
            me.on({
              [eventName]: (...params) => {
                // resolve when event is caught
                resolve(...params);
                // reset log to be able to await again
                if (options.resetLog) {
                  me.clearLog(eventName);
                }
              },
              prio: -10000,
              // Let others do their stuff first
              once: true // promises can only be resolved once anyway
            });
          }
        });
      }

      clearLog(eventName) {
        if (this._triggered) {
          if (eventName) {
            delete this._triggered[eventName];
          } else {
            this._triggered = {};
          }
        }
      }
      //endregion
      //region Trigger
      /**
       * Triggers an event, calling all registered listeners with the supplied arguments. Returning false from any listener
       * makes function return false.
       * @param {String} eventName Event name for which to trigger listeners
       * @param {Object} [param] Single parameter passed on to listeners, source property will be added to it (this)
       * @param {Boolean} [param.bubbles] Pass as `true` to indicate that the event will bubble up the widget
       * ownership hierarchy. For example up a `Menu`->`parent` Menu tree, or a `Field`->`Container` tree.
       * @typings param -> {{bubbles?: boolean, [key: string]: any}}
       * @returns {Boolean|Promise} Returns false if any listener returned `false`, or a `Promise` yielding
       * `true` / `false` based on what is returned from the async listener functions, otherwise `true`
       * @async
       * @advanced
       */
      trigger(eventName, param) {
        var _me$eventListeners, _me$eventListeners2, _me$bubbleEvents, _handlerPromises, _handlerPromises2;
        const me = this,
          name = eventName.toLowerCase(),
          {
            eventsSuspended,
            relayAllTargets,
            callOnFunctions
          } = me;
        let listeners = (_me$eventListeners = me.eventListeners) === null || _me$eventListeners === void 0 ? void 0 : _me$eventListeners[name],
          handlerPromises;
        // log trigger, used by experimental promise support to resolve immediately when needed
        if (!me._triggered) {
          me._triggered = {};
        }
        me._triggered[eventName] = true;
        if (eventsSuspended) {
          if (eventsSuspended.shouldQueue) {
            eventsSuspended.queue.push(arguments);
          }
          return true;
        }
        // Include catchall listener for all events.
        // Do not push the catchAll listeners onto the events own listener array.
        if ((_me$eventListeners2 = me.eventListeners) !== null && _me$eventListeners2 !== void 0 && _me$eventListeners2.catchall) {
          (listeners = listeners ? listeners.slice() : []).push(...me.eventListeners.catchall);
          // The catchAll listeners must honour their prio settings.
          listeners.sort(priorityComparator);
        }
        if (!listeners && !relayAllTargets && !callOnFunctions) {
          return true;
        }
        // default to include source : this in param
        if (param) {
          if (!('source' in param)) {
            if (Object.isExtensible(param)) {
              param.source = me;
            } else {
              param = Object.setPrototypeOf({
                source: me
              }, param);
            }
          }
        } else {
          param = {
            source: me
          };
        }
        // Lowercased event name should be the "type" property in keeping with DOM events.
        if (param.type !== name) {
          // Create instance property because "type" is read only
          if (param.constructor !== Object) {
            Reflect.defineProperty(param, 'type', {
              get: () => name
            });
          } else {
            param.type = name;
          }
        }
        param.eventName = eventName;
        // Bubble according to `bubbleEvents` config if `bubbles` is not explicitly set
        if (!('bubbles' in param) && (_me$bubbleEvents = me.bubbleEvents) !== null && _me$bubbleEvents !== void 0 && _me$bubbleEvents[eventName]) {
          param.bubbles = me.bubbleEvents[eventName];
        }
        if (callOnFunctions) {
          const fnName = 'on' + StringHelper.capitalize(eventName);
          if (fnName in me) {
            var _me$pluginFunctionCha;
            // Return true if the on[fnName] is not set to keep default behavior
            const result = me[fnName] ? me.callback(me[fnName], me, [param]) : true;
            let inhibit;
            if (Objects.isPromise(result)) {
              (handlerPromises || (handlerPromises = [])).push(result);
            } else {
              inhibit = result === false || inhibit;
            }
            // See if the called function was injected into the instance
            // masking an implementation in the prototype.
            // we must call the class's implementation after the injected one.
            // unless it's an injected chained function, in which case it will have been called above.
            // Note: The handler may have resulted in destruction.
            if (!me.isDestroyed && hasOwnProperty$2.call(me, fnName) && !((_me$pluginFunctionCha = me.pluginFunctionChain) !== null && _me$pluginFunctionCha !== void 0 && _me$pluginFunctionCha[fnName])) {
              const myProto = Object.getPrototypeOf(me);
              if (fnName in myProto) {
                const result = myProto[fnName].call(me, param);
                if (Objects.isPromise(result)) {
                  (handlerPromises || (handlerPromises = [])).push(result);
                } else {
                  inhibit = result === false || inhibit;
                }
                // A handler may have resulted in destruction.
                if (me.isDestroyed) {
                  return;
                }
              }
            }
            // Returning false from an on-function prevents further triggering
            if (inhibit) {
              return false;
            }
          }
        }
        let ret;
        if (listeners) {
          let i = 0,
            internalAbort = false;
          // Let add/removeListener know that we're using the array to protect against a situation where an event
          // listener changes the listeners when triggering the event.
          listeners.$firing = true;
          // If any listener resulted in our destruction, abort.
          for (i; i < listeners.length && !me.isDestroyed && !internalAbort; i++) {
            const listener = listeners[i];
            // Previously, returning false would abort all further listeners. But now internal listeners
            // are allowed to run anyway
            if (ret === false && !listener.$internal) {
              continue;
            }
            let handler,
              thisObj = listener.thisObj;
            // Listeners that have thisObj are auto removed when thisObj is destroyed. If thisObj is destroyed from
            // a listener we might still end up here, since listeners are sliced and not affected by the removal
            if (!thisObj || !thisObj.isDestroyed) {
              // Flag for the expiry timer
              listener.called = true;
              if (listener.once) {
                me.removeListener(name, listener);
              }
              // prepare handler function
              if (typeof listener.fn === 'string') {
                if (thisObj) {
                  handler = thisObj[listener.fn];
                }
                // keep looking for the callback in the hierarchy
                if (!handler) {
                  const result = me.resolveCallback(listener.fn);
                  handler = result.handler;
                  thisObj = result.thisObj;
                }
              } else {
                handler = listener.fn;
              }
              // if `buffer` option is provided, the handler will be wrapped into buffer function,
              // but only once on the first call
              if (listener.buffer) {
                if (!listener.bufferFn) {
                  const buffer = Number(listener.buffer);
                  if (typeof buffer !== 'number' || isNaN(buffer)) {
                    throw new Error(`Incorrect type for buffer, got "${buffer}" (expected a Number)`);
                  }
                  listener.bufferFn = FunctionHelper.createBuffered(handler, buffer, thisObj, listener.args);
                }
                handler = listener.bufferFn;
              }
              // if `throttle` option is provided, the handler will be called immediately, but all the rest calls
              // that happened during time specified in `throttle`, will be delayed and glued into 1 call
              if (listener.throttle) {
                const throttle = Number(listener.throttle);
                if (typeof throttle !== 'number' || isNaN(throttle)) {
                  throw new Error(`Incorrect type for throttle, got "${throttle}" (expected a Number)`);
                }
                if (!listener.throttledFn) {
                  listener.throttledFn = FunctionHelper.createThrottled(handler, throttle, thisObj, listener.args);
                }
                handler = listener.throttledFn;
              }
              const result = handler.call(thisObj || me, ...(listener.args || []), param);
              // Store result until we get a false return value, to mimic the old behavior from before we carried
              // on with calling internal listeners
              if (ret !== false) {
                ret = result;
              }
              if (listener.$internal && result === false) {
                internalAbort = true;
              }
              if (Objects.isPromise(result)) {
                result.$internal = listener.$internal;
                // If a handler is async (returns a Promise), then collect all Promises.
                // At the end we return a Promise which encapsulates all returned Promises
                // or, if only one handler was async, *the* Promise.
                // Don't allocate an Array until we have to.
                (handlerPromises || (handlerPromises = [])).push(result);
              }
            }
          }
          listeners.$firing = false;
          // An internal listener returned `false`, abort before relaying events etc.
          if (internalAbort) {
            return false;
          }
        }
        // relay all?
        relayAllTargets === null || relayAllTargets === void 0 ? void 0 : relayAllTargets.forEach(config => {
          let name = eventName;
          if (config.transformCase) {
            name = StringHelper.capitalize(name);
          }
          if (config.prefix) {
            name = config.prefix + name;
          }
          if (config.through.trigger(name, param) === false) {
            return false;
          }
        });
        // Use DOM standard event property name to indicate that the event
        // bubbles up the owner axis.
        // False from any handler cancels the bubble.
        // Must also avoid owner if any handlers destroyed the owner.
        if (param.bubbles && me.owner && !me.owner.isDestroyed) {
          return me.owner.trigger(eventName, param);
        }
        // Run internal promises even if external listener returned false
        handlerPromises = (_handlerPromises = handlerPromises) === null || _handlerPromises === void 0 ? void 0 : _handlerPromises.filter(p => ret !== false || p.$internal);
        // If any handlers were async functions (returned a Promise), then return a Promise
        // which resolves when they all resolve.
        if ((_handlerPromises2 = handlerPromises) !== null && _handlerPromises2 !== void 0 && _handlerPromises2.length) {
          return new Promise(resolve => {
            Promise.all(handlerPromises).then(promiseResults => {
              const finalResult = !promiseResults.some(result => result === false);
              resolve(finalResult);
            });
          });
        }
        return ret !== false;
      }
      /**
       * Prevents events from being triggered until {@link #function-resumeEvents()} is called. Optionally queues events that are triggered while
       * suspended. Multiple calls stack to require matching calls to `resumeEvents()` before actually resuming.
       * @param {Boolean} queue Specify true to queue events triggered while suspended
       * @advanced
       */
      suspendEvents(queue = false) {
        const eventsSuspended = this.eventsSuspended || (this.eventsSuspended = {
          shouldQueue: queue,
          queue: [],
          count: 0
        });
        eventsSuspended.count++;
      }
      /**
       * Resume event triggering after a call to {@link #function-suspendEvents()}. If any triggered events were queued they will be triggered.
       * @returns {Boolean} `true` if events have been resumed (multiple calls to suspend require an equal number of resume calls to resume).
       * @advanced
       */
      resumeEvents() {
        const suspended = this.eventsSuspended;
        if (suspended) {
          if (--suspended.count === 0) {
            this.eventsSuspended = null;
            if (suspended.shouldQueue) {
              for (const queued of suspended.queue) {
                this.trigger(...queued);
              }
            }
          }
        }
        return !Boolean(this.eventsSuspended);
      }
      //endregion
    });

    /**
     * @module Core/helper/AsyncHelper
     */
    /**
     * A helper class to make asynchronous tasks `await` friendly.
     */
    class AsyncHelper {
      /**
       * Returns a promise that resolves on next animation frame.
       * ```
       *  async method() {
       *      // do work
       *      await AsyncHelper.animationFrame();
       *      // do more work
       *  }
       * ```
       * @async
       */
      static animationFrame() {
        return new Promise(resolve => {
          requestAnimationFrame(resolve);
        });
      }
      /**
       * Returns a promise that resolves after a specified number of milliseconds.
       * ```
       *  async method() {
       *      await AsyncHelper.sleep(10);
       *      // ...
       *  }
       * ```
       * @param {Number} millis The number of milliseconds to sleep.
       * @async
       */
      static sleep(millis) {
        return new Promise(resolve => {
          setTimeout(() => {
            resolve();
          }, millis);
        });
      }
      /**
       * Returns a promise that resolves as soon as possible, allowing the browser to minimally process other messages.
       * This is the shortest possible delay the browser offers, so be aware that it does not necessarily allow the
       * browser to paint or reflow if used in a long loop. It does, however, allow other async methods to execute.
       * ```
       *  async method() {
       *      await AsyncHelper.yield();
       *      // ...
       *  }
       * ```
       * @async
       */
      static yield() {
        return Promise.resolve();
      }
    }
    AsyncHelper._$name = 'AsyncHelper';

    /**
     * @module Core/helper/AjaxHelper
     */
    /**
     * Options for the requests. Please see
     * [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details
     *
     * To set default values for the options please use {@link #property-DEFAULT_FETCH_OPTIONS-static} property:
     *
     * ```javascript
     * // enable passing parameters in request body by default
     * AjaxHelper.DEFAULT_FETCH_OPTIONS = { addQueryParamsToBody : true };
     * ```
     *
     * @typedef {Object} FetchOptions
     * @property {'GET'|'POST'|'PUT'|'PATCH'|'DELETE'} [method] The request method, e.g., `GET`, `POST`
     * @property {Object} [queryParams] A key-value pair Object containing the params to add to the query string
     * @property {Object} [headers] Any headers you want to add to your request, contained within a `Headers` object or an
     * object literal with ByteString values
     * @property {Object} [body] Any body that you want to add to your request: this can be a `Blob`, `BufferSource`,
     * `FormData`, `URLSearchParams`, or `USVString` object. Note that a request using the `GET` or `HEAD` method cannot have a body.
     * @property {Boolean} [addQueryParamsToBody=false] Indicates whether `queryParams` should be passed in the request
     * body. Adding them to the body applies for `application/x-www-form-urlencoded` and `multipart/form-data`
     * content types only, so make sure to pass corresponding `Content-Type` header to `headers`.
     *
     * When the argument is `true` and:
     * - if `application/x-www-form-urlencoded` content-type header is passed
     *   the method will make a `URLSearchParams` instance with `queryParams` and set it as the request body.
     *   And if `body` already has a `URLSearchParams` instance provided the parameters will be set there.
     * - if `multipart/form-data` content-type header is passed
     *   the method will make a `FormData` instance with `queryParams` and set it as the request body.
     *   And if `body` already has a `FormData` instance provided the parameters will be set there.
     *
     * Otherwise, `queryParams` are added to the query string.
     * @property {'cors'|'no-cors'|'same-origin'} [mode] The mode you want to use for the request, e.g., `'cors'`, `'no-cors'`, or `'same-origin'`.
     * @property {'omit'|'same-origin'|'include'} [credentials] The request credentials you want to use for the request: `'omit'`, `'same-origin'`, or
     * `'include'`. To automatically send cookies for the current domain, this option must be provided
     * @property {Boolean} [parseJson] Specify `true` to parses the response and attach the resulting object to the
     * `Response` object as `parsedJson`
     */
    const paramValueRegExp = /^(\w+)=(.*)$/,
      parseParams = function (paramString) {
        const result = {},
          params = paramString.split('&');
        // loop through each 'filter={"field":"name","operator":"=","value":"Sweden","caseSensitive":true}' string
        // So we cannot use .split('=')
        for (const nameValuePair of params) {
          const [match, name, value] = paramValueRegExp.exec(nameValuePair),
            decodedName = decodeURIComponent(name),
            decodedValue = decodeURIComponent(value);
          if (match) {
            let paramValue = result[decodedName];
            if (paramValue) {
              if (!Array.isArray(paramValue)) {
                paramValue = result[decodedName] = [paramValue];
              }
              paramValue.push(decodedValue);
            } else {
              result[decodedName] = decodedValue;
            }
          }
        }
        return result;
      };
    /**
     * Simplifies Ajax requests. Uses fetch & promises.
     *
     * ```javascript
     * AjaxHelper.get('some-url').then(response => {
     *     // process request response here
     * });
     * ```
     *
     * Uploading file to server via FormData interface.
     * Please visit [FormData](https://developer.mozilla.org/en-US/docs/Web/API/FormData) for details.
     *
     * ```javascript
     * const formData = new FormData();
     * formData.append('file', 'fileNameToUpload');
     * AjaxHelper.post('file-upload-url', formData).then(response => {
     *     // process request response here
     * });
     * ```
     *
     */
    class AjaxHelper {
      /**
       * Sets default options for {@link #function-fetch-static AjaxHelper#fetch()} calls. Please see
       * {@link #typedef-FetchOptions} and
       * [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details.
       *
       * ```javascript
       * // default content-type for all requests will be "application/json"
       * AjaxHelper.DEFAULT_FETCH_OPTIONS = {
       *     headers : {
       *         'content-type' : 'application/json'
       *     }
       * };
       * ```
       * @member {FetchOptions} DEFAULT_FETCH_OPTIONS
       * @static
       */
      static DEFAULT_FETCH_OPTIONS = {};
      /**
       * Make a request (using GET) to the specified url.
       * @param {String} url URL to `GET` from
       * @param {FetchOptions} [options] The options for the `fetch` API
       * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
       * @async
       */
      static get(url, options) {
        return this.fetch(url, options);
      }
      /**
       * POST data to the specified URL.
       * @param {String} url URL to `POST` to
       * @param {String|Object|FormData} payload The data to post. If an object is supplied, it will be stringified
       * @param {FetchOptions} [options] The options for the `fetch` API
       * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
       * @async
       */
      static post(url, payload, options = {}) {
        if (!(payload instanceof FormData) && !(typeof payload === 'string')) {
          payload = JSON.stringify(payload);
          options.headers = options.headers || {};
          options.headers['Content-Type'] = options.headers['Content-Type'] || 'application/json';
        }
        return this.fetch(url, Object.assign({
          method: 'POST',
          body: payload
        }, options));
      }
      /**
       * Fetch the specified resource using the `fetch` API.
       * @param {String} url URL to fetch from
       * @param {FetchOptions} [options] The options for the `fetch` API
       * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
       * @async
       */
      static fetch(url, options) {
        let controller;
        // inherit global options
        options = Objects.merge({}, AjaxHelper.DEFAULT_FETCH_OPTIONS, options);
        // AbortController is not supported by LockerService
        // https://github.com/bryntum/support/issues/3689
        if (typeof AbortController !== 'undefined') {
          controller = options.abortController = new AbortController();
          options.signal = controller.signal;
        }
        if (!('credentials' in options)) {
          options.credentials = 'include';
        }
        if (options.queryParams) {
          const params = Object.entries(options.queryParams);
          if (params.length) {
            let paramsAdded = false;
            // for some content types we are going to add parameters to body (if that's not disabled)
            if (options.headers && options.addQueryParamsToBody === true) {
              const contentType = new Headers(options.headers).get('Content-Type');
              let bodyClass;
              switch (contentType) {
                case 'application/x-www-form-urlencoded':
                  bodyClass = URLSearchParams;
                  break;
                case 'multipart/form-data':
                  bodyClass = FormData;
                  break;
              }
              // if that's one of supported content types
              if (bodyClass) {
                const body = options.body || (options.body = new bodyClass());
                // put parameters to body if it's of supported type
                if (body instanceof bodyClass) {
                  params.forEach(([key, value]) => body.set(key, value));
                  // remember parameters are already added
                  paramsAdded = true;
                }
              }
            }
            // if parameters are not added yet append them to the query string
            if (!paramsAdded) {
              url += (url.includes('?') ? '&' : '?') + params.map(([param, value]) => `${param}=${encodeURIComponent(value)}`).join('&');
            }
          }
        }
        // Promise that will be resolved either when network request is finished or when json is parsed
        const promise = new Promise((resolve, reject) => {
          fetch(url, options).then(response => {
            if (options.parseJson) {
              response.json().then(json => {
                response.parsedJson = json;
                resolve(response);
              }).catch(error => {
                response.parsedJson = null;
                response.error = error;
                reject(response);
              });
            } else {
              resolve(response);
            }
          }).catch(error => {
            error.stack = promise.stack;
            reject(error);
          });
        });
        promise.stack = new Error().stack;
        promise.abort = function () {
          var _controller;
          (_controller = controller) === null || _controller === void 0 ? void 0 : _controller.abort();
        };
        return promise;
      }
      /**
       * Registers the passed URL to return the passed mocked up Fetch Response object to the
       * AjaxHelper's promise resolve function.
       * @param {String} url The url to return mock data for
       * @param {Object|Function} response A mocked up Fetch Response object which must contain
       * at least a `responseText` property, or a function to which the `url` and a `params` object
       * and the `Fetch` `options` object is passed which returns that.
       * @param {String} response.responseText The data to return.
       * @param {Boolean} [response.synchronous] resolve the Promise immediately
       * @param {Number} [response.delay=100] resolve the Promise after this number of milliseconds.
       */
      static mockUrl(url, response) {
        const me = this;
        (me.mockAjaxMap || (me.mockAjaxMap = {}))[url] = response;
        // Inject the override into the AjaxHelper instance
        if (!AjaxHelper.originalFetch) {
          AjaxHelper.originalFetch = AjaxHelper.fetch;
          AjaxHelper.fetch = me.mockAjaxFetch.bind(me);
        }
      }
      static async mockAjaxFetch(url, options) {
        const urlAndParams = url.split('?');
        let result = this.mockAjaxMap[urlAndParams[0]],
          parsedJson = null;
        if (result) {
          if (typeof result === 'function') {
            result = await result(urlAndParams[0], urlAndParams[1] && parseParams(urlAndParams[1]), options);
          }
          try {
            parsedJson = (options === null || options === void 0 ? void 0 : options.parseJson) && JSON.parse(result.responseText);
          } catch (error) {
            parsedJson = null;
            result.error = error;
          }
          result = Object.assign({
            status: 200,
            ok: true,
            headers: new Headers(),
            statusText: 'OK',
            url,
            parsedJson,
            text: () => new Promise(resolve => {
              resolve(result.responseText);
            }),
            json: () => new Promise(resolve => {
              resolve(parsedJson);
            })
          }, result);
          return new Promise(function (resolve, reject) {
            if (result.synchronous) {
              resolve(result);
            } else {
              setTimeout(function () {
                resolve(result);
              }, 'delay' in result ? result.delay : 100);
            }
          });
        } else {
          return AjaxHelper.originalFetch(url, options);
        }
      }
    }
    AjaxHelper._$name = 'AjaxHelper';

    /**
     * @module Core/localization/LocaleHelper
     */
    /**
     * @typedef {Object} LocaleKeys
     * Object which contains `key: value` localization pairs.
     * Key value may have `String`, `Function`, `LocaleKeys` or `Object` type.
     *
     * Example:
     *
     * ```javascript
     * {
     *     title   : 'Title',
     *     count   : number => `Count is ${number}`,
     *     MyClass : {
     *        foo : 'bar'
     *     }
     * }
     * ```
     *
     * @property {String|Function|LocaleKeys|Object} key localization key
     * @typings {[key: string]}:{string|number|Function|LocaleKeys|object}
     */
    /**
     * @typedef {LocaleKeys} Locale
     * Locale configuration object which contains locale properties alongside with localization pairs.
     *
     * Example:
     *
     * ```javascript
     {
     *     localeName : 'En',
     *     localeDesc : 'English (US)',
     *     localeCode : 'en-US',
     *     ... (localization key:value pairs)
     * }
     * ```
     *
     * @property {String} localeName Locale name. For example: "En"
     * @property {String} localeDesc Locale description to be shown in locale picker list. For example: "English (US)"
     * @property {String} localeCode Locale code. Two letter locale code or two letter locale and two letter country code.
     * For example: "en" or 'en_US'
     * @property {String} [localePath] Locale path for asynchronous loading using
     * AjaxHelper {@link Core.helper.AjaxHelper#function-get-static} request
     */
    /**
     * @typedef {Object} Locales
     * Object which contains locales. Each object key represents published locale by its `localeName`.
     *
     * Example:
     *
     * ```javascript
     * // This returns English locale.
     * const englishLocale = LocaleHelper.locales.En;
     * ```
     *
     * @property {Locale} key localization object
     * @typings {[key: string]}:{Locale}
     */
    /**
     * Thin class which provides locale management methods.
     * Class doesn't import other API classes and can be used separately for publishing locales before importing product classes.
     *
     * Locale should be published with {@link ##function-publishLocale-static} method before it is available for localizing of Bryntum API classes and widgets.
     *
     * Example:
     *
     * ```javascript
     * LocaleHelper.publishLocale({
     *     localeName : 'En',
     *     localeDesc : 'English (US)',
     *     localeCode : 'en-US',
     *     ... (localization key:value pairs)
     * });
     * ```
     *
     * or for asynchronous loading from remote path on applying locale
     *
     * ```javascript
     *LocaleHelper.publishLocale({
     *     localeName : 'En',
     *     localeDesc : 'English (US)',
     *     localeCode : 'en-US',
     *     localePath : 'https://my-server/localization/en.js'
     * });
     * ```
     */
    class LocaleHelper {
      static skipLocaleIntegrityCheck = false;
      /**
       * Merges all properties of provided locale objects into new locale object.
       * Locales are merged in order they provided and locales which go later replace
       * same properties of previous locales.
       * @param {...Object} locales Locales to merge
       * @returns {Object} Merged locale
       */
      static mergeLocales(...locales) {
        const result = {};
        locales.forEach(locale => {
          Object.keys(locale).forEach(key => {
            if (typeof locale[key] === 'object') {
              result[key] = {
                ...result[key],
                ...locale[key]
              };
            } else {
              result[key] = locale[key];
            }
          });
        });
        return result;
      }
      /**
       * Removes all properties from `locale` that are present in the provided `toTrim`.
       * @param {Object} locale Locale to process
       * @param {Object} toTrim Object enumerating properties that should be removed.
       * When `false` throws exceptions in such cases.
       */
      static trimLocale(locale, toTrim) {
        const remove = (key, subKey) => {
          if (locale[key]) {
            if (subKey) {
              if (locale[key][subKey]) {
                delete locale[key][subKey];
              }
            } else {
              delete locale[key];
            }
          }
        };
        Object.keys(toTrim).forEach(key => {
          if (Object.keys(toTrim[key]).length > 0) {
            Object.keys(toTrim[key]).forEach(subKey => remove(key, subKey));
          } else {
            remove(key);
          }
        });
      }
      /**
       * Normalizes locale object to {@link Locale} type.
       *
       * Supported configs:
       *
       * ```javascript
       * LocaleHelper.normalizeLocale({
       *     localeName : 'En',
       *     localeDesc : 'English (US)',
       *     localeCode : 'en-US',
       *     ... (localization key:value pairs)
       * });
       * ```
       *
       * and for backward compatibility
       *
       * ```javascript
       * LocaleHelper.normalizeLocale('En', {
       *     name : 'En',
       *     desc : 'English (US)',
       *     code : 'en-US',
       *     locale : {
       *         ... (localization key:value pairs)
       *     }
       * });
       * ```
       * @param {String|Object} nameOrConfig String name of locale or locale object
       * @param {Object} [config] Locale object
       * @returns {Locale} Locale object
       * @internal
       */
      static normalizeLocale(nameOrConfig, config) {
        if (!nameOrConfig) {
          throw new Error(`"nameOrConfig" parameter can not be empty`);
        }
        if (typeof nameOrConfig === 'string') {
          if (!config) {
            throw new Error(`"config" parameter can not be empty`);
          }
          if (config.locale) {
            // Matches legacy locale type
            config.name = nameOrConfig || config.name;
          } else {
            config.localeName = nameOrConfig;
          }
        } else {
          config = nameOrConfig;
        }
        let locale = {};
        if (config.name || config.locale) {
          // Matches legacy locale type
          locale = Object.assign({
            localeName: config.name
          }, config.locale);
          config.desc && (locale.localeDesc = config.desc);
          config.code && (locale.localeCode = config.code);
          config.path && (locale.localePath = config.path);
        } else {
          if (!config.localeName) {
            throw new Error(`"config" parameter doesn't have "localeName" property`);
          }
          // Extract locale config from name object
          locale = Object.assign({}, config);
        }
        // Cleanup result
        for (const key of ['name', 'desc', 'code', 'path']) {
          if (locale[key]) {
            delete locale[key];
          }
        }
        if (!locale.localeName) {
          throw new Error(`Locale name can not be empty`);
        }
        return locale;
      }
      /**
       * Get/set currently published locales.
       * Returns an object with locales.
       *
       * Example:
       *
       * ```javascript
       * const englishLocale = LocaleHelper.locales.En;
       * ```
       *
       * `englishLocale` contains {@link Locale} object.
       *
       * @readonly
       * @member {Locales} locales
       * @static
       */
      static get locales() {
        return globalThis.bryntum.locales || {};
      }
      static set locales(locales) {
        globalThis.bryntum.locales = locales;
      }
      /**
       * Get/set current locale name. Defaults to "En"
       * @member {String} localeName
       * @static
       */
      static get localeName() {
        return globalThis.bryntum.locale || 'En';
      }
      static set localeName(localeName) {
        globalThis.bryntum.locale = localeName || LocaleHelper.localeName;
      }
      /**
       * Get current locale config specified by {@link ##property-localeName-static}.
       * If no current locale specified, returns default `En` locale or first published locale
       * or empty locale object if no published locales found.
       * @readonly
       * @member {Locales} locale
       * @static
       */
      static get locale() {
        return LocaleHelper.localeName && this.locales[LocaleHelper.localeName] || this.locales.En || Object.values(this.locales)[0] || {
          localeName: '',
          localeDesc: '',
          localeCoode: ''
        };
      }
      /**
       * Publishes a locale to make it available for applying.
       * Published locales are available in {@link ##property-locales-static}.
       *
       * Recommended usage:
       *
       * ```javascript
       * LocaleHelper.publishLocale({
       *     localeName : 'En',
       *     localeDesc : 'English (US)',
       *     localeCode : 'en-US',
       *     ... (localization key:value pairs)
       * });
       * ```
       *
       * for backward compatibility (prior to `5.3.0` version):
       *
       * ```javascript
       * LocaleHelper.publishLocale('En', {
       *     name : 'En',
       *     desc : 'English (US)',
       *     code : 'en-US',
       *     locale : {
       *         ... (localization key:value pairs)
       *     }
       * });
       * ```
       *
       * Publishing a locale will automatically merge it's localization keys with existing locale matching by locale name,
       * replacing existing one with new. To replace existing locale entirely pass `true` to optional `config` parameter.
       *
       * Example:
       *
       * ```javascript
       * LocaleHelper.publishLocale({
       *     localeName : 'En',
       *     localeDesc : 'English (US)',
       *     localeCode : 'en-US',
       *     ... (localization key:value pairs)
       * }, true);
       * ```
       *
       * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object
       * @param {Locale|Boolean} [config] Locale object.
       * Not used if locale object is passed as first method parameter.
       * Path `true` value and locale object as first method parameter to publish locale without merging with existing one.
       * @returns {Locale} Locale object
       */
      static publishLocale(nameOrConfig, config) {
        const {
            locales
          } = globalThis.bryntum,
          locale = LocaleHelper.normalizeLocale(nameOrConfig, config),
          {
            localeName
          } = locale;
        if (!locales[localeName] || config === true) {
          locales[localeName] = locale;
        } else {
          locales[localeName] = this.mergeLocales(locales[localeName] || {}, locale || {});
        }
        return locales[localeName];
      }
    }
    globalThis.bryntum = globalThis.bryntum || {};
    globalThis.bryntum.locales = globalThis.bryntum.locales || {};
    LocaleHelper._$name = 'LocaleHelper';

    /**
     * @module Core/localization/LocaleManager
     */
    // Documented at the export below, to work for singleton
    class LocaleManager extends Events(Base$1) {
      static get defaultConfig() {
        return {
          // Enable strict locale checking by default for tests
          throwOnMissingLocale: VersionHelper.isTestEnv
        };
      }
      construct(...args) {
        const me = this;
        super.construct(...args);
        if (BrowserHelper.isBrowserEnv) {
          var _me$locale;
          // Try get locale name from script's `default-locale` tag
          const scriptTag = document.querySelector('script[data-default-locale]');
          if (scriptTag) {
            me.applyLocale(scriptTag.dataset.defaultLocale);
          } else if ((_me$locale = me.locale) !== null && _me$locale !== void 0 && _me$locale.localeName) {
            me.applyLocale(me.locale.localeName);
          }
        }
      }
      /**
       * Get/set currently registered locales.
       * Alias for {@link Core.localization.LocaleHelper#property-locales-static LocaleHelper.locales}.
       * @readonly
       * @member {Locales} locales
       */
      get locales() {
        return LocaleHelper.locales;
      }
      set locales(locales) {
        LocaleHelper.locales = locales;
      }
      /**
       * Get/set currently used locale.
       * Setter calls {@link #function-applyLocale}.
       * @member {Locales} locale
       * @accepts {String|Locale}
       */
      set locale(nameOrConfig) {
        this.applyLocale(nameOrConfig);
      }
      get locale() {
        return LocaleHelper.locale;
      }
      /**
       * Publishes a locale to make it available for applying.
       * @deprecated Since 5.3.0. Use {@link Core.localization.LocaleHelper#function-publishLocale-static LocaleHelper.publishLocale} instead.
       *
       * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object.
       * @param {Locale} [config] Locale object. Not used if object is passed as first method parameter
       * @returns {Locale} published locale object is passed as first method parameter
       * @function registerLocale
       */
      registerLocale(nameOrConfig, config) {
        VersionHelper.deprecate('Core', '6.0.0', 'LocaleManager.registerLocale deprecated, use LocaleHelper.publishLocale instead');
        LocaleHelper.publishLocale(nameOrConfig, config);
      }
      /**
       * Extends locale specified by name to add additional translations and applies it.
       * @deprecated Since 5.3.0. Use {@link ##function-applyLocale} instead.
       *
       * @param {String} name Name of locale (for example `En` or `SvSE`).
       * @param {Locale} config Locale object
       * @returns {Locale|Promise} locale object or Promise which resolves with locale object after it was loaded
       * @function extendLocale
       */
      extendLocale(name, config) {
        VersionHelper.deprecate('Core', '6.0.0', 'LocaleManager.extendLocale deprecated, use LocaleManager.applyLocale instead');
        const locale = LocaleHelper.publishLocale(name, config);
        return this.applyLocale(locale, true);
      }
      /**
       * Applies a locale by string name or publishes new locale configuration with
       * {@link Core.localization.LocaleHelper#function-publishLocale-static} and applies it.
       * If locale is specified by string name, like 'En', it must be published before applying it.
       *
       * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object
       * @param {Locale|Boolean} [config] Locale object. Pass `true` to reapply locale which is passed as first method parameter.
       * @returns {Locale|Promise} locale object or Promise which resolves with locale object after it was loaded
       * @fires locale
       * @async
       * @function applyLocale
       */
      applyLocale(nameOrConfig, config, ignoreError = false) {
        const me = this;
        let localeConfig;
        if (typeof nameOrConfig === 'string') {
          if (typeof config !== 'object') {
            localeConfig = me.locales[nameOrConfig];
            if (!localeConfig) {
              if (ignoreError) {
                return true;
              }
              throw new Error(`Locale "${nameOrConfig}" is not published. Publish with LocaleHelper.publishLocale() before applying.`);
            }
          } else {
            localeConfig = LocaleHelper.publishLocale(nameOrConfig, config);
          }
        } else {
          localeConfig = LocaleHelper.publishLocale(nameOrConfig);
        }
        if (me.locale.localeName && me.locale.localeName === localeConfig.localeName && config !== true) {
          // no need to apply same locale again
          return me.locale;
        }
        // Set current locale name
        LocaleHelper.localeName = localeConfig.localeName;
        const triggerLocaleEvent = () => {
          /**
           * Fires when a locale is applied
           * @event locale
           * @param {Core.localization.LocaleManager} source The Locale manager instance.
           * @param {Locale} locale Locale configuration
           */
          me.trigger('locale', localeConfig);
        };
        if (localeConfig.localePath) {
          return new Promise((resolve, reject) => {
            me.loadLocale(localeConfig.localePath).then(response => {
              response.text().then(text => {
                const parseLocale = new Function(text);
                parseLocale();
                if (BrowserHelper.isBrowserEnv) {
                  localeConfig = me.locales[localeConfig.localeName];
                  // Avoid loading next time
                  if (localeConfig) {
                    delete localeConfig.localePath;
                  }
                }
                triggerLocaleEvent();
                resolve(localeConfig);
              });
            }).catch(response => reject(response));
          });
        }
        triggerLocaleEvent();
        return localeConfig;
      }
      /**
       * Loads a locale using AjaxHelper {@link Core.helper.AjaxHelper#function-get-static} request.
       * @private
       * @param {String} path Path to locale file
       * @async
       */
      loadLocale(path) {
        return AjaxHelper.get(path);
      }
      /**
       * Specifies if {@link Core.localization.Localizable#function-L-static Localizable.L()} function would throw error if no localization found at runtime.
       *
       * @member {Boolean} throwOnMissingLocale
       * @default false
       */
      set throwOnMissingLocale(value) {
        this._throwOnMissingLocale = value;
      }
      get throwOnMissingLocale() {
        return this._throwOnMissingLocale;
      }
    }
    const LocaleManagerSingleton = new LocaleManager();

    /**
     * @module Core/localization/Localizable
     */
    const ObjectProto = Object.getPrototypeOf(Object),
      localeRe = /L{.*?}/g,
      capturelocaleRe = /L{(.*?)}/g,
      classMatchRe = /((.*?)\.)?(.+)/g,
      escape$1 = txt => txt.replace(/{(\d+)}/gm, '[[$1]]'),
      unescape = txt => txt.replace(/\[\[(\d+)]]/gm, '{$1}'),
      emptyObject$b = Object.freeze(Object.create(null));
    /**
     * Mixin that provides localization functionality to a class.
     *
     * ```javascript
     * // Get localized string
     * grid.L('foo');
     * grid.L('L{foo}');
     * ```
     *
     * @mixin
     */
    var Localizable = (Target => class Localizable extends (Target || Base$1) {
      static get $name() {
        return 'Localizable';
      }
      static get configurable() {
        return {
          /**
           * A class translations of which are used for translating this entity.
           * This is often used when translations of an item are defined on its container class.
           * For example:
           *
           * ```js
           * // Toolbar class that has some predefined items
           * class MyToolbar extends Toolbar {
           *
           *     static get $name() {
           *         return 'MyToolbar';
           *     }
           *
           *     static get defaultConfig() {
           *         return {
           *             // this specifies default configs for the items
           *             defaults : {
           *                 // will tell items to use the toolbar locale
           *                 localeClass : this
           *             },
           *
           *             items : [
           *                 // The toolbar has 2 buttons and translation for their texts will be searched in
           *                 // the toolbar locales
           *                 { text : 'Agree' },
           *                 { text : 'Disagree' }
           *             ]
           *         };
           *     }
           *
           *    ...
           * }
           * ```
           * So if one makes a locale for the `MyToolbar` class that will include `Agree` and `Disagree` string translations:
           * ```js
           *     ...
           *     MyToolbar : {
           *         Agree    : 'Yes, I agree',
           *         Disagree : 'No, I do not agree'
           *     }
           * ```
           * They will be used for the toolbar buttons and the button captions will say `Yes, I agree` and `No, I do not agree`.
           *
           * @config {Core.Base}
           * @typings {typeof Base}
           * @category Misc
           * @advanced
           */
          localeClass: null,
          /**
           * Set to `false` to disable localization of this object.
           * @config {Boolean}
           * @default true
           * @category Misc
           * @advanced
           */
          localizable: null,
          /**
           * List of properties which values should be translated automatically upon a locale applying.
           * In case there is a need to localize not typical value (not a String value or a field with re-defined setter/getter),
           * you could use 'localeKey' meta configuration.
           * Example:
           * ```js
           *  static get configurable() {
           *     return {
           *          localizableProperties : ['width'],
           *
           *          width : {
           *              value   : '54em', // default value here
           *              $config : {
           *                  localeKey : 'L{editorWidth}' // name of the property that will be used in localization file
           *              }
           *          }
           *      };
           *  }
           * ```
           * @config {String[]}
           * @category Misc
           * @advanced
           */
          localizableProperties: {
            value: [],
            $config: {
              merge: 'distinct'
            }
          }
        };
      }
      static clsName(cls) {
        var _cls$prototype, _cls$prototype2;
        return typeof cls === 'string' ? cls : cls === ObjectProto ? 'Object' : cls.$$name || cls.name || ((_cls$prototype = cls.prototype) === null || _cls$prototype === void 0 ? void 0 : _cls$prototype.$$name) || ((_cls$prototype2 = cls.prototype) === null || _cls$prototype2 === void 0 ? void 0 : _cls$prototype2.name);
      }
      static parseLocaleString(text) {
        var _text;
        const matches = [];
        let m;
        // Parse locale text in case it's wrapped with L{foo}
        if ((_text = text) !== null && _text !== void 0 && _text.includes('L{')) {
          // Escape fix for {1}, {2} etc. in locale str
          text = escape$1(text);
          capturelocaleRe.lastIndex = 0;
          while ((m = capturelocaleRe.exec(text)) != null) {
            classMatchRe.lastIndex = 0;
            // Support for parsing class namespace L{Class.foo}
            const classMatch = classMatchRe.exec(m[1]);
            matches.push({
              match: unescape(m[0]),
              localeKey: unescape(classMatch[3]),
              localeClass: classMatch[2]
            });
          }
        }
        return matches.length > 0 ? matches : [{
          match: text,
          localeKey: text,
          localeClass: undefined
        }];
      }
      construct(config = {}, ...args) {
        // Base class applies configs.
        super.construct(config, ...args);
        LocaleManagerSingleton.ion({
          locale: 'updateLocalization',
          thisObj: this
        });
        this.updateLocalization();
      }
      get localeClass() {
        return this._localeClass || null;
      }
      localizeProperty(property) {
        var _me$$meta$configs$pro, _me$fieldMap, _me$fieldMap$property;
        const me = this,
          currentValue = Objects.getPath(me, property),
          // Grid.column.Column is Localizable too. It uses fields, not configs
          localeKey = ((_me$$meta$configs$pro = me.$meta.configs[property]) === null || _me$$meta$configs$pro === void 0 ? void 0 : _me$$meta$configs$pro.localeKey) || ((_me$fieldMap = me.fieldMap) === null || _me$fieldMap === void 0 ? void 0 : (_me$fieldMap$property = _me$fieldMap[property]) === null || _me$fieldMap$property === void 0 ? void 0 : _me$fieldMap$property.defaultValue);
        let localizedValue;
        // check if localeKey is defined and try to translate it
        if (localeKey) {
          localizedValue = Localizable.localize(localeKey, me, me.localeClass || me);
          // if a user set value directly in class definition, his value has a prio
          if (localizedValue && !(property in (me.initialConfig || emptyObject$b))) {
            Objects.setPath(me.isColumn ? me.data : me, property, localizedValue);
          }
        } else if (typeof currentValue === 'string') {
          me.originalLocales = me.originalLocales || {};
          localizedValue = Objects.getPath(me.originalLocales, property);
          // If we haven't saved original values yet let's do that
          if (localizedValue === undefined) {
            Objects.setPath(me.originalLocales, property, currentValue);
            localizedValue = currentValue;
          }
          // Doing localization from the original values
          if (localizedValue) {
            Objects.setPath(me, property, localizedValue = me.optionalL(localizedValue, me));
          }
        }
        return localizedValue || currentValue;
      }
      /**
       * Method that is triggered when applying a locale to the instance
       * (happens on the instance construction steps and when switching to another locale).
       *
       * The method can be overridden to dynamically translate the instance when locale is switched.
       * When overriding the method please make sure you call `super.updateLocalization()`.
       * @category Misc
       * @advanced
       */
      updateLocalization() {
        if (this.localizable !== false) {
          var _this$localizableProp, _this$trigger;
          (_this$localizableProp = this.localizableProperties) === null || _this$localizableProp === void 0 ? void 0 : _this$localizableProp.forEach(this.localizeProperty, this);
          (_this$trigger = this.trigger) === null || _this$trigger === void 0 ? void 0 : _this$trigger.call(this, 'localized');
        }
      }
      static getTranslation(text, templateData, localeCls) {
        const locale = LocaleManagerSingleton.locale;
        let result = null,
          clsName,
          cls;
        if (locale) {
          // Iterate over all found localization entries
          for (const {
            match,
            localeKey,
            localeClass
          } of this.parseLocaleString(text)) {
            const translate = clsName => {
              var _locale$clsName;
              const translation = (_locale$clsName = locale[clsName]) === null || _locale$clsName === void 0 ? void 0 : _locale$clsName[localeKey];
              if (translation) {
                if (typeof translation === 'function') {
                  result = templateData != null ? translation(templateData) : translation;
                } else if (typeof translation === 'object' || text === match) {
                  result = translation;
                }
                // Likely string
                else {
                  result = (result || text).replace(match, translation);
                }
                // Might have nested L{, recurse
                if (typeof translation === 'string' && translation.includes('L{')) {
                  result = this.getTranslation(translation, templateData, localeCls);
                }
              }
              return translation;
            };
            // Translate order
            // 1. Try to translate for current class
            // 2. Try to translate by Class hierarchy traversing prototypes
            // 3. Try to translate if Class is in {Class.foo} format
            let success = false;
            for (cls = localeCls; cls && (clsName = Localizable.clsName(cls)); cls = Object.getPrototypeOf(cls)) {
              if (success = translate(clsName)) {
                break;
              } else if (typeof cls === 'string') {
                break;
              }
            }
            if (!success && localeClass) {
              translate(localeClass);
            }
          }
        }
        return result;
      }
      /**
       * Get localized string, returns `null` if no localized string found.
       * @param {String} text String key
       * @param {Object} [templateData] Data to supply to template if localized string is a function
       * @returns {String}
       * @internal
       */
      static localize(text, templateData = undefined, ...localeClasses) {
        var _localeClasses;
        // In case this static method is called directly third argument is not provided
        // just fallback to searching locales for the class itself
        if (((_localeClasses = localeClasses) === null || _localeClasses === void 0 ? void 0 : _localeClasses.length) === 0) {
          localeClasses = [this];
        }
        let translation = null;
        localeClasses.some(cls => {
          translation = Localizable.getTranslation(text, templateData, cls);
          return translation != null;
        });
        return translation;
      }
      /**
       * Get localized string, returns value of `text` if no localized string found.
       *
       * If {@link Core.localization.LocaleManager#property-throwOnMissingLocale LocaleManager.throwOnMissingLocale}
       * is `true` then calls to `L()` will throw `Localization is not found for 'text' in 'ClassName'` exception when no
       * localization is found.
       *
       * @param {String} text String key
       * @param {Object} [templateData] Data to supply to template if localized string is a function
       * @static
       * @returns {String}
       * @advanced
       */
      static L(text, templateData = undefined, ...localeClasses) {
        var _localeClasses2;
        // In case this static method is called directly third argument is not provided
        // just fallback to searching locales for the class itself
        if (((_localeClasses2 = localeClasses) === null || _localeClasses2 === void 0 ? void 0 : _localeClasses2.length) === 0) {
          localeClasses = [this];
        }
        const translation = this.localize(text, templateData, ...localeClasses);
        // Throw error if not localized and text matches `L{foo}`
        if (translation == null && LocaleManagerSingleton.throwOnMissingLocale && text.includes('L{')) {
          throw new Error(`Localization is not found for '${text}' in '${localeClasses.map(cls => Localizable.clsName(cls)).join(', ')}'. ${LocaleManagerSingleton.locale.localeName ? `Locale : ${LocaleManagerSingleton.locale.localeName}` : ''}`);
        }
        return translation ?? text;
      }
      /**
       * Convenience function that can be called directly on the class that mixes Localizable in
       *
       * ```javascript
       * button.text = grid.L('L{group}');
       * ```
       *
       * @param {String} text String key
       * @param {Object} [templateData] Data to supply to template if localized string is a function
       * @returns {String}
       * @category Misc
       * @advanced
       */
      L(text, templateData) {
        const {
          localeClass,
          constructor
        } = this;
        // If we have a different class set as translations provider
        // pass it first and use the class being translated as a fallback provider
        if (localeClass && Localizable.clsName(localeClass) !== Localizable.clsName(constructor)) {
          return Localizable.L(text, templateData, localeClass, constructor);
        } else {
          return Localizable.L(text, templateData, constructor);
        }
      }
      /**
       * Convenience function to get an optional translation. The difference compared to `L()` is that it won't throw
       * an error when the translation is missing even if configured with `throwOnMissingLocale`
       *
       * ```javascript
       * button.text = grid.optionalL('L{group}');
       * ```
       *
       * @param {String} text String key
       * @param {Object} [templateData] Data to supply to template if localized string is a function
       * @returns {String}
       * @static
       * @category Misc
       * @advanced
       */
      static optionalL(text, templateData = undefined, ...localeClasses) {
        var _localeClasses3;
        const shouldThrow = LocaleManagerSingleton.throwOnMissingLocale;
        LocaleManagerSingleton.throwOnMissingLocale = shouldThrow && localeRe.test(text);
        // In case this static method is called directly third argument is not provided
        // just fallback to searching locales for the class itself
        if (((_localeClasses3 = localeClasses) === null || _localeClasses3 === void 0 ? void 0 : _localeClasses3.length) === 0) {
          localeClasses = [this];
        }
        const result = Localizable.L(text, templateData, ...localeClasses);
        LocaleManagerSingleton.throwOnMissingLocale = shouldThrow;
        return result;
      }
      /**
       * Convenience function to get an optional translation. The difference compared to `L()` is that it won't throw
       * an error when the translation is missing even if configured with `throwOnMissingLocale`
       *
       * ```javascript
       * button.text = grid.optionalL('L{group}');
       * ```
       *
       * @param {String} text String key
       * @param {Object} [templateData] Data to supply to template if localized string is a function
       * @param {Boolean} [preventThrow] Prevent throwing error even if localized text matches `L{foo}`
       * @returns {String}
       * @category Misc
       * @internal
       */
      optionalL(text, templateData = this, preventThrow = false) {
        const shouldThrow = LocaleManagerSingleton.throwOnMissingLocale;
        // Optional locale text should not include L{}
        LocaleManagerSingleton.throwOnMissingLocale = shouldThrow && localeRe.test(text) && !preventThrow;
        const result = this.L(text, templateData);
        LocaleManagerSingleton.throwOnMissingLocale = shouldThrow;
        return result;
      }
      /**
       * Get the global LocaleManager
       * @property {Core.localization.LocaleManager}
       * @typings {typeof LocaleManager}
       * @category Misc
       * @readonly
       * @advanced
       */
      get localeManager() {
        return LocaleManagerSingleton;
      }
      /**
       * Get the global LocaleHelper
       * @property {Core.localization.LocaleHelper}
       * @typings {typeof LocaleHelper}
       * @category Misc
       * @readonly
       * @advanced
       */
      get localeHelper() {
        return LocaleHelper;
      }
    });

    const locale$5 = {
      localeName: 'En',
      localeDesc: 'English (US)',
      localeCode: 'en-US',
      Object: {
        Yes: 'Yes',
        No: 'No',
        Cancel: 'Cancel',
        Ok: 'OK',
        Week: 'Week'
      },
      ColorPicker: {
        noColor: 'No color'
      },
      Combo: {
        noResults: 'No results',
        recordNotCommitted: 'Record could not be added',
        addNewValue: value => `Add ${value}`
      },
      FilePicker: {
        file: 'File'
      },
      Field: {
        badInput: 'Invalid field value',
        patternMismatch: 'Value should match a specific pattern',
        rangeOverflow: value => `Value must be less than or equal to ${value.max}`,
        rangeUnderflow: value => `Value must be greater than or equal to ${value.min}`,
        stepMismatch: 'Value should fit the step',
        tooLong: 'Value should be shorter',
        tooShort: 'Value should be longer',
        typeMismatch: 'Value is required to be in a special format',
        valueMissing: 'This field is required',
        invalidValue: 'Invalid field value',
        minimumValueViolation: 'Minimum value violation',
        maximumValueViolation: 'Maximum value violation',
        fieldRequired: 'This field is required',
        validateFilter: 'Value must be selected from the list'
      },
      DateField: {
        invalidDate: 'Invalid date input'
      },
      DatePicker: {
        gotoPrevYear: 'Go to previous year',
        gotoPrevMonth: 'Go to previous month',
        gotoNextMonth: 'Go to next month',
        gotoNextYear: 'Go to next year'
      },
      NumberFormat: {
        locale: 'en-US',
        currency: 'USD'
      },
      DurationField: {
        invalidUnit: 'Invalid unit'
      },
      TimeField: {
        invalidTime: 'Invalid time input'
      },
      TimePicker: {
        hour: 'Hour',
        minute: 'Minute',
        second: 'Second'
      },
      List: {
        loading: 'Loading...',
        selectAll: 'Select All'
      },
      GridBase: {
        loadMask: 'Loading...',
        syncMask: 'Saving changes, please wait...'
      },
      PagingToolbar: {
        firstPage: 'Go to first page',
        prevPage: 'Go to previous page',
        page: 'Page',
        nextPage: 'Go to next page',
        lastPage: 'Go to last page',
        reload: 'Reload current page',
        noRecords: 'No records to display',
        pageCountTemplate: data => `of ${data.lastPage}`,
        summaryTemplate: data => `Displaying records ${data.start} - ${data.end} of ${data.allCount}`
      },
      PanelCollapser: {
        Collapse: 'Collapse',
        Expand: 'Expand'
      },
      Popup: {
        close: 'Close'
      },
      UndoRedo: {
        Undo: 'Undo',
        Redo: 'Redo',
        UndoLastAction: 'Undo last action',
        RedoLastAction: 'Redo last undone action',
        NoActions: 'No items in the undo queue'
      },
      FieldFilterPicker: {
        equals: 'equals',
        doesNotEqual: 'does not equal',
        isEmpty: 'empty',
        isNotEmpty: 'not empty',
        contains: 'contains',
        doesNotContain: 'does not contain',
        startsWith: 'starts with',
        endsWith: 'ends with',
        isOneOf: 'one of',
        isNotOneOf: 'not one of',
        isGreaterThan: 'greater than',
        isLessThan: 'less than',
        isGreaterThanOrEqualTo: 'greater than or equal to',
        isLessThanOrEqualTo: 'less than or equal to',
        isBetween: 'between',
        isNotBetween: 'not between',
        isBefore: 'before',
        isAfter: 'after',
        isToday: 'today',
        isTomorrow: 'tomorrow',
        isYesterday: 'yesterday',
        isThisWeek: 'this week',
        isNextWeek: 'next week',
        isLastWeek: 'last week',
        isThisMonth: 'this month',
        isNextMonth: 'next month',
        isLastMonth: 'last month',
        isThisYear: 'this year',
        isNextYear: 'next year',
        isLastYear: 'last year',
        isYearToDate: 'year to date',
        isTrue: 'true',
        isFalse: 'false',
        selectAProperty: 'Select a property',
        selectAnOperator: 'Select an operator',
        caseSensitive: 'Case-sensitive',
        and: 'and',
        dateFormat: 'D/M/YY',
        selectOneOrMoreValues: 'Select one or more values',
        enterAValue: 'Enter a value',
        enterANumber: 'Enter a number',
        selectADate: 'Select a date'
      },
      FieldFilterPickerGroup: {
        addFilter: 'Add filter'
      },
      DateHelper: {
        locale: 'en-US',
        weekStartDay: 0,
        nonWorkingDays: {
          0: true,
          6: true
        },
        weekends: {
          0: true,
          6: true
        },
        unitNames: [{
          single: 'millisecond',
          plural: 'ms',
          abbrev: 'ms'
        }, {
          single: 'second',
          plural: 'seconds',
          abbrev: 's'
        }, {
          single: 'minute',
          plural: 'minutes',
          abbrev: 'min'
        }, {
          single: 'hour',
          plural: 'hours',
          abbrev: 'h'
        }, {
          single: 'day',
          plural: 'days',
          abbrev: 'd'
        }, {
          single: 'week',
          plural: 'weeks',
          abbrev: 'w'
        }, {
          single: 'month',
          plural: 'months',
          abbrev: 'mon'
        }, {
          single: 'quarter',
          plural: 'quarters',
          abbrev: 'q'
        }, {
          single: 'year',
          plural: 'years',
          abbrev: 'yr'
        }, {
          single: 'decade',
          plural: 'decades',
          abbrev: 'dec'
        }],
        unitAbbreviations: [['mil'], ['s', 'sec'], ['m', 'min'], ['h', 'hr'], ['d'], ['w', 'wk'], ['mo', 'mon', 'mnt'], ['q', 'quar', 'qrt'], ['y', 'yr'], ['dec']],
        parsers: {
          L: 'MM/DD/YYYY',
          LT: 'HH:mm A',
          LTS: 'HH:mm:ss A'
        },
        ordinalSuffix: number => {
          const hasSpecialCase = ['11', '12', '13'].find(n => number.endsWith(n));
          let suffix = 'th';
          if (!hasSpecialCase) {
            const lastDigit = number[number.length - 1];
            suffix = {
              1: 'st',
              2: 'nd',
              3: 'rd'
            }[lastDigit] || 'th';
          }
          return number + suffix;
        }
      }
    };
    LocaleHelper.publishLocale(locale$5);

    const {
        toString: toString$1
      } = Object.prototype,
      DATE_TYPE = toString$1.call(new Date()),
      tempDate$1 = new Date(),
      MS_PER_HOUR = 1000 * 60 * 60,
      defaultValue = (value, defValue) => isNaN(value) || value == null ? defValue : value,
      rangeFormatPartRe = /([ES]){([^}]+)}/g,
      enOrdinalSuffix = number => {
        const hasSpecialCase = ['11', '12', '13'].find(n => number.endsWith(n));
        let suffix = 'th';
        if (!hasSpecialCase) {
          const lastDigit = number[number.length - 1];
          suffix = {
            1: 'st',
            2: 'nd',
            3: 'rd'
          }[lastDigit] || 'th';
        }
        return number + suffix;
      },
      useIntlFormat = (name, options, date) => {
        const formatter = intlFormatterCache[name] || (intlFormatterCache[name] = new Intl.DateTimeFormat(locale$4, options));
        return formatter.format(date);
      },
      formatTime = (name, options, date, isShort = false) => {
        let strTime = useIntlFormat(name, options, date);
        // remove '0' from time when has AM/PM (from 01:00 PM to 1:00 PM): https://github.com/bryntum/support/issues/1483
        if (/am|pm/i.test(strTime)) {
          // remove first character only if is 0
          strTime = strTime.replace(/^0/, '');
          // if isShort is true, remove minutes if is :00
          if (isShort) {
            strTime = strTime.replace(/:00/, '');
          }
        }
        return strTime;
      },
      getDayDiff = (end, start) => Math.floor((end.getTime() - start.getTime() - (end.getTimezoneOffset() - start.getTimezoneOffset()) * validConversions.minute.millisecond) / validConversions.day.millisecond) + 1,
      normalizeDay = day => day >= 0 ? day : day + 7,
      msRegExp = /([^\w])(S+)/gm,
      msReplacer = (match, g1) => g1 + 'SSS',
      splitRegExp = /[:.\-/\s]/;
    // These vars are set when changing locale
    let locale$4 = 'en-US',
      ordinalSuffix = enOrdinalSuffix,
      // Used to cache used formats, to not have to parse format string each time
      formatCache = {},
      formatRedirects = {},
      intlFormatterCache = {},
      parserCache = {};
    const redirectFormat = format => {
      const intlConfig = intlFormatConfigs[format];
      if (!intlConfig) {
        throw new Error('Only international formats should be used here');
      }
      if (formatRedirects[format] !== undefined) {
        return formatRedirects[format];
      }
      const intl = new Intl.DateTimeFormat(locale$4, intlConfig),
        fmt = intl.formatToParts(new Date(2001, 1, 2, 3, 4, 5, 6)).map(part => {
          const type = part.type,
            intlCfg = intlConfig[type];
          if (type === 'literal') {
            // here we cheat again, because our parser can not skip unknown separators
            return part.value.replace(/,/g, '');
          } else if (type === 'day') {
            return intlCfg === 'numeric' ? 'D' : 'DD';
          } else if (type === 'month') {
            return intlCfg === 'short' ? 'MMM' : intlCfg === 'long' ? 'MMMM' : intlCfg === 'numeric' ? 'M' : 'MM';
          } else if (type === 'year') {
            // here we cheat a little, using `YYYY` for numeric year in `ll`
            // this is to simplify the fix for https://github.com/bryntum/support/issues/5179
            // to be fixed if anyone complains
            return intlCfg === 'numeric' ? 'YYYY' : 'YY';
          }
        }).join('');
      return formatRedirects[format] = fmt;
    };
    const DEFAULT_YEAR = 2020,
      // 2020 is the year that has no issues in Safari, see: https://github.com/bryntum/support/issues/554
      DEFAULT_MONTH = 0,
      DEFAULT_DAY = 1,
      intlFormatConfigs = {
        l: {
          year: 'numeric',
          month: 'numeric',
          day: 'numeric'
        },
        ll: {
          year: 'numeric',
          month: 'short',
          day: 'numeric'
        }
      },
      formats = {
        // 1, 2, ... 11, 12
        M: date => date.getMonth() + 1,
        //date.toLocaleDateString(locale, { month : 'numeric' }),
        // 1st, 2nd, 3rd, 4th, ... 11th, 12th
        Mo: date => ordinalSuffix(formats.M(date).toString()),
        // 01, 02, ...
        MM: date => (date.getMonth() + 1).toString().padStart(2, '0'),
        //date.toLocaleDateString(locale, { month : '2-digit' }),
        // Jan, Feb, ...
        MMM: date => useIntlFormat('MMM', {
          month: 'short'
        }, date),
        // January, February, ...
        MMMM: date => useIntlFormat('MMMM', {
          month: 'long'
        }, date),
        // 1, 2, ...
        Q: date => Math.ceil((date.getMonth() + 1) / 3),
        // 1st, 2nd, ...
        Qo: date => ordinalSuffix(formats.Q(date).toString()),
        // 1, 2, ...
        D: date => date.getDate(),
        //date.toLocaleDateString(locale, { day : 'numeric' }),
        // 1st, 2nd, ...
        Do: date => ordinalSuffix(formats.D(date).toString()),
        // 01, 02, ...
        DD: date => date.getDate().toString().padStart(2, '0'),
        //date.toLocaleDateString(locale, { day : '2-digit' }),
        // 1, 2, ..., 365, 365
        DDD: date => Math.ceil((new Date(date.getFullYear(), date.getMonth(), date.getDate(), 12, 0, 0) - new Date(date.getFullYear(), 0, 0, 12, 0, 0)) / validConversions.day.millisecond),
        // 1st, 2nd, ...
        DDDo: date => ordinalSuffix(formats.DDD(date).toString()),
        // 001, 002, ...
        DDDD: date => formats.DDD(date).toString().padStart(3, '0'),
        // 0, 1, ..., 6
        d: date => date.getDay(),
        // 0th, 1st, ...
        do: date => ordinalSuffix(date.getDay().toString()),
        // S, M, ...
        d1: date => useIntlFormat('d1', {
          weekday: 'narrow'
        }, date).substr(0, 1),
        // Su, Mo, ...
        dd: date => formats.ddd(date).substring(0, 2),
        // Sun, Mon, ...
        ddd: date => useIntlFormat('ddd', {
          weekday: 'short'
        }, date),
        // Sunday, Monday, ...
        dddd: date => useIntlFormat('dddd', {
          weekday: 'long'
        }, date),
        u: date => {
          const formatter = intlFormatterCache.u || (intlFormatterCache.u = new Intl.DateTimeFormat('en-GB', {
              timeZone: 'UTC',
              year: 'numeric',
              month: '2-digit',
              day: '2-digit'
            })),
            parts = formatter.formatToParts(date);
          return `${parts[4].value}${parts[2].value}${parts[0].value}Z`;
        },
        uu: date => {
          const formatter = intlFormatterCache.uu || (intlFormatterCache.uu = new Intl.DateTimeFormat('en-GB', {
              timeZone: 'UTC',
              hour12: false,
              year: 'numeric',
              month: '2-digit',
              day: '2-digit',
              hour: '2-digit',
              minute: '2-digit',
              second: '2-digit'
            })),
            parts = formatter.formatToParts(date);
          return `${parts[4].value}${parts[2].value}${parts[0].value}T${parts[6].value}${parts[8].value}${parts[10].value}Z`;
        },
        e: date => date.getDay(),
        E: date => date.getDay() + 1,
        // ISO week, 1, 2, ...
        W: date => DateHelper.getWeekNumber(date)[1],
        Wo: date => ordinalSuffix(formats.W(date).toString()),
        WW: date => formats.W(date).toString().padStart(2, '0'),
        // ISO week, 1, 2, ... with localized 'Week ' prefix
        Wp: date => `${DateHelper.localize('L{Week}')} ${formats.W(date)}`,
        WWp: date => `${DateHelper.localize('L{Week}')} ${formats.WW(date)}`,
        Wp0: date => `${DateHelper.localize('L{Week}')[0]}${formats.W(date)}`,
        WWp0: date => `${DateHelper.localize('L{Week}')[0]}${formats.WW(date)}`,
        // 1979, 2018
        Y: date => date.getFullYear(),
        //date.toLocaleDateString(locale, { year : 'numeric' }),
        // 79, 18
        YY: date => (date.getFullYear() % 100).toString().padStart(2, '0'),
        //date.toLocaleDateString(locale, { year : '2-digit' }),
        // 1979, 2018
        YYYY: date => date.getFullYear(),
        //date.toLocaleDateString(locale, { year : 'numeric' }),
        // AM, PM
        A: date => date.getHours() < 12 ? 'AM' : 'PM',
        a: date => date.getHours() < 12 ? 'am' : 'pm',
        // 0, 1, ... 23
        H: date => date.getHours(),
        // 00, 01, ...
        HH: date => date.getHours().toString().padStart(2, '0'),
        // 1, 2, ... 12
        h: date => date.getHours() % 12 || 12,
        // 01, 02, ...
        hh: date => formats.h(date).toString().padStart(2, '0'),
        // 1, 2, ... 24
        k: date => date.getHours() || 24,
        // 01, 02, ...
        kk: date => formats.k(date).toString().padStart(2, '0'),
        // Locale specific (0 -> 24 or 1 AM -> 12 PM)
        K: date => formatTime('K', {
          hour: 'numeric'
        }, date),
        // Locale specific (00 -> 24 or 1 AM -> 12 PM)
        KK: date => formatTime('KK', {
          hour: '2-digit'
        }, date),
        // 0, 1, ... 59
        m: date => date.getMinutes(),
        //date.toLocaleTimeString(locale, { minute : 'numeric' }),
        // 00, 01, ...
        mm: date => formats.m(date).toString().padStart(2, '0'),
        // 0, 1, ... 59
        s: date => date.getSeconds(),
        //date.toLocaleTimeString(locale, { second : 'numeric' }),
        // 00, 01, ...
        ss: date => formats.s(date).toString().padStart(2, '0'),
        // 0, 1, ... 9 which are 000, 100, 200 ... 900 in milliseconds
        S: date => Math.floor(date.getMilliseconds() / 100).toString(),
        // 00, 01, ... 99 which are 000, 010, 020 ... 990 in milliseconds
        SS: date => Math.floor(date.getMilliseconds() / 10).toString().padStart(2, '0'),
        // 000, 001, ... 999 in milliseconds
        SSS: date => date.getMilliseconds().toString().padStart(3, '0'),
        z: date => useIntlFormat('z', {
          timeZoneName: 'short'
        }, date),
        zz: date => useIntlFormat('zz', {
          timeZoneName: 'long'
        }, date),
        Z: date => DH$2.getGMTOffset(date),
        LT: date => formatTime('LT', {
          hour: '2-digit',
          minute: '2-digit'
        }, date),
        // if minutes is 0, doesn't show it
        LST: date => formatTime('LST', {
          hour: 'numeric',
          minute: '2-digit'
        }, date, true),
        LTS: date => formatTime('LTS', {
          hour: '2-digit',
          minute: '2-digit',
          second: '2-digit'
        }, date),
        L: date => useIntlFormat('L', {
          year: 'numeric',
          month: '2-digit',
          day: '2-digit'
        }, date),
        l: date => useIntlFormat('l', intlFormatConfigs.l, date),
        LL: date => useIntlFormat('LL', {
          year: 'numeric',
          month: 'long',
          day: 'numeric'
        }, date),
        ll: date => useIntlFormat('ll', intlFormatConfigs.ll, date),
        LLL: date => useIntlFormat('LLL', {
          year: 'numeric',
          month: 'long',
          day: 'numeric',
          hour: 'numeric',
          minute: '2-digit'
        }, date),
        lll: date => useIntlFormat('lll', {
          year: 'numeric',
          month: 'short',
          day: 'numeric',
          hour: 'numeric',
          minute: '2-digit'
        }, date),
        LLLL: date => useIntlFormat('LLLL', {
          year: 'numeric',
          month: 'long',
          day: 'numeric',
          hour: 'numeric',
          minute: '2-digit',
          weekday: 'long'
        }, date),
        llll: date => useIntlFormat('llll', {
          year: 'numeric',
          month: 'short',
          day: 'numeric',
          hour: 'numeric',
          minute: '2-digit',
          weekday: 'short'
        }, date)
      },
      // Want longest keys first, to not stop match at L of LTS etc.
      formatKeys = Object.keys(formats).sort((a, b) => b.length - a.length),
      formatRegexp = `^(?:${formatKeys.join('|')})`,
      // return empty object, meaning value cannot be processed to a valuable date part
      emptyFn$2 = () => ({}),
      isNumber = str => numberRegex.test(str),
      parseMilliseconds = str => isNumber(str) && {
        milliseconds: parseInt(str.padEnd(3, '0').substring(0, 3))
      },
      parsers = {
        YYYY: str => {
          const year = parseInt(str);
          return {
            year: year >= 1000 && year <= 9999 ? year : NaN
          };
        },
        Y: str => ({
          year: parseInt(str)
        }),
        YY: str => {
          const year = parseInt(str);
          return {
            year: year + (year > 1968 ? 1900 : 2000)
          };
        },
        M: str => ({
          month: parseInt(str) - 1
        }),
        MM: str => ({
          month: parseInt(str) - 1
        }),
        Mo: str => ({
          month: parseInt(str) - 1
        }),
        MMM: str => {
          const month = (str || '').toLowerCase();
          for (const [name, entry] of Object.entries(DateHelper._monthShortNamesIndex)) {
            if (month.startsWith(name)) {
              return {
                month: entry.value
              };
            }
          }
        },
        MMMM: str => {
          const month = (str || '').toLowerCase();
          for (const [name, entry] of Object.entries(DateHelper._monthNamesIndex)) {
            if (month.startsWith(name)) {
              return {
                month: entry.value
              };
            }
          }
        },
        DD: str => ({
          date: parseInt(str)
        }),
        D: str => ({
          date: parseInt(str)
        }),
        Do: str => ({
          date: parseInt(str)
        }),
        DDD: emptyFn$2,
        DDDo: emptyFn$2,
        DDDD: emptyFn$2,
        d: emptyFn$2,
        do: emptyFn$2,
        d1: emptyFn$2,
        dd: emptyFn$2,
        ddd: emptyFn$2,
        dddd: emptyFn$2,
        Q: emptyFn$2,
        Qo: emptyFn$2,
        W: emptyFn$2,
        Wo: emptyFn$2,
        WW: emptyFn$2,
        e: emptyFn$2,
        E: emptyFn$2,
        HH: str => ({
          hours: parseInt(str)
        }),
        hh: str => ({
          hours: parseInt(str)
        }),
        mm: str => ({
          minutes: parseInt(str)
        }),
        H: str => ({
          hours: parseInt(str)
        }),
        m: str => ({
          minutes: parseInt(str)
        }),
        ss: str => ({
          seconds: parseInt(str)
        }),
        s: str => ({
          seconds: parseInt(str)
        }),
        S: parseMilliseconds,
        SS: parseMilliseconds,
        SSS: parseMilliseconds,
        A: str => ({
          amPm: str.toLowerCase()
        }),
        a: str => ({
          amPm: str.toLowerCase()
        }),
        L: 'MM/DD/YYYY',
        LT: 'HH:mm A',
        LTS: 'HH:mm:ss A',
        l: {
          type: 'dynamic',
          parser: () => redirectFormat('l')
        },
        ll: {
          type: 'dynamic',
          parser: () => redirectFormat('ll')
        },
        // Can either be Z (=UTC, 0) or +-HH:MM
        Z: str => {
          if (!str || !timeZoneRegEx.test(str) && str !== 'Z') {
            return null;
          }
          let timeZone = 0;
          // If string being parsed is more detailed than the format specified we can have more chars left,
          // thus check the last (for example HH:mmZ with input HH:mm:ssZ -> ssZ)
          if (str !== 'Z') {
            const matches = timeZoneRegEx.exec(str);
            // If timezone regexp matches, sting has time zone offset like '+02:00'
            if (matches) {
              const sign = matches[1] === '+' ? 1 : -1,
                hours = parseInt(matches[2]) || 0,
                minutes = parseInt(matches[3]) || 0;
              timeZone = sign * (hours * 60 + minutes);
            }
            // otherwise we just return current time zone, because there's a Z key in the input
            else {
              timeZone = -1 * new Date().getTimezoneOffset();
            }
          }
          return {
            timeZone
          };
        }
      },
      parserKeys = Object.keys(parsers).sort((a, b) => b.length - a.length),
      parserRegexp = new RegExp(`(${parserKeys.join('|')})`),
      // Following regexp includes all formats that should be handled by Date class
      // !!! except `l|ll`, plus made all-string capturing, otherwise the left-most `l` pattern
      // matches all `l*` formats
      // localeStrRegExp            = new RegExp('(l|LL|ll|LLL|lll|LLLL|llll)'),
      localeStrRegExp = new RegExp('^(LL|LLL|lll|LLLL|llll)$'),
      //    ISODateRegExp             = new RegExp('YYYY-MM-DD[T ]HH:mm:ss(.s+)?Z'),
      // Some validConversions are negative to show that it's not an exact conversion, just an estimate.
      validConversions = {
        // The units below assume:
        // 30 days in a month, 91 days for a quarter and 365 for a year
        // 52 weeks per year, 4 per month, 13 per quarter
        // 3652 days per decade (assuming two of the years will be leap with 366 days)
        decade: {
          decade: 1,
          year: 10,
          quarter: 40,
          month: 120,
          week: 520,
          day: 3652,
          hour: 24 * 3652,
          minute: 1440 * 3652,
          second: 86400 * 3652,
          millisecond: 86400000 * 3652
        },
        year: {
          decade: 0.1,
          year: 1,
          quarter: 4,
          month: 12,
          week: 52,
          day: 365,
          hour: 24 * 365,
          minute: 1440 * 365,
          second: 86400 * 365,
          millisecond: 86400000 * 365
        },
        quarter: {
          decade: 1 / 40,
          year: 1 / 4,
          quarter: 1,
          month: 3,
          week: 4,
          day: 91,
          hour: 24 * 91,
          minute: 1440 * 91,
          second: 86400 * 91,
          millisecond: 86400000 * 91
        },
        month: {
          decade: 1 / 120,
          year: 1 / 12,
          quarter: 1 / 3,
          month: 1,
          week: 4,
          day: -30,
          hour: -24 * 30,
          minute: -1440 * 30,
          second: -86400 * 30,
          millisecond: -86400000 * 30
        },
        week: {
          decade: -1 / 520,
          year: -1 / 52,
          quarter: -1 / 13,
          month: -1 / 4,
          day: 7,
          hour: 168,
          minute: 10080,
          second: 604800,
          millisecond: 604800000
        },
        day: {
          decade: -1 / 3652,
          year: -1 / 365,
          quarter: -1 / 91,
          month: -1 / 30,
          week: 1 / 7,
          hour: 24,
          minute: 1440,
          second: 86400,
          millisecond: 86400000
        },
        hour: {
          decade: -1 / (3652 * 24),
          year: -1 / (365 * 24),
          quarter: -1 / (91 * 24),
          month: -1 / (30 * 24),
          week: 1 / 168,
          day: 1 / 24,
          minute: 60,
          second: 3600,
          millisecond: 3600000
        },
        minute: {
          decade: -1 / (3652 * 1440),
          year: -1 / (365 * 1440),
          quarter: -1 / (91 * 1440),
          month: -1 / (30 * 1440),
          week: 1 / 10080,
          day: 1 / 1440,
          hour: 1 / 60,
          second: 60,
          millisecond: 60000
        },
        second: {
          decade: -1 / (3652 * 86400),
          year: -1 / (365 * 86400),
          quarter: -1 / (91 * 86400),
          month: -1 / (30 * 86400),
          week: 1 / 604800,
          day: 1 / 86400,
          hour: 1 / 3600,
          minute: 1 / 60,
          millisecond: 1000
        },
        millisecond: {
          decade: -1 / (3652 * 86400000),
          year: -1 / (365 * 86400000),
          quarter: -1 / (91 * 86400000),
          month: -1 / (30 * 86400000),
          week: 1 / 604800000,
          day: 1 / 86400000,
          hour: 1 / 3600000,
          minute: 1 / 60000,
          second: 1 / 1000
        }
      },
      normalizedUnits = {
        ms: 'millisecond',
        milliseconds: 'millisecond',
        s: 'second',
        seconds: 'second',
        m: 'minute',
        mi: 'minute',
        min: 'minute',
        minutes: 'minute',
        h: 'hour',
        hours: 'hour',
        d: 'day',
        days: 'day',
        w: 'week',
        weeks: 'week',
        M: 'month',
        mo: 'month',
        mon: 'month',
        months: 'month',
        q: 'quarter',
        quarters: 'quarter',
        y: 'year',
        years: 'year',
        dec: 'decade',
        decades: 'decade'
      },
      withDecimalsDurationRegex = /^\s*([-+]?\d+(?:[.,]\d*)?|[-+]?(?:[.,]\d+))\s*([^\s]+)?/i,
      noDecimalsDurationRegex = /^\s*([-+]?\d+)(?![.,])\s*([^\s]+)?/i,
      canonicalUnitNames = ['millisecond', 'second', 'minute', 'hour', 'day', 'week', 'month', 'quarter', 'year', 'decade'],
      canonicalUnitAbbreviations = [['mil'], ['s', 'sec'], ['m', 'min'], ['h', 'hr'], ['d'], ['w', 'wk'], ['mo', 'mon', 'mnt'], ['q', 'quar', 'qrt'], ['y', 'yr'], ['dec']],
      deltaUnits = ['decade', 'year', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond'],
      // Used when creating a date from an object, to fill in any blanks
      dateProperties = ['milliseconds', 'seconds', 'minutes', 'hours', 'date', 'month', 'year'],
      parseNumber = n => {
        const result = parseFloat(n);
        return isNaN(result) ? null : result;
      },
      numberRegex = /^[0-9]+$/,
      timeZoneRegEx = /([+-])(\d\d):*(\d\d)*$/,
      unitMagnitudes = {
        millisecond: 0,
        second: 1,
        minute: 2,
        hour: 3,
        day: 4,
        week: 5,
        month: 6,
        quarter: 7,
        year: 8,
        decade: 9
      },
      snapFns = {
        round(number, step = 1) {
          return Math.round(number / step) * step;
        },
        floor(number, step = 1) {
          return Math.floor(number / step) * step;
        },
        ceil(number, step = 1) {
          return Math.ceil(number / step) * step;
        }
      },
      keyCache = {};
    /**
     * @module Core/helper/DateHelper
     */
    /**
     * A static class offering date manipulation, comparison, parsing and formatting helper methods.
     *
     * ## Parsing strings
     * Use `DateHelper.parse()` to parse strings into dates. It accepts a date string and a format specifier.
     * The format specifier is string built up using the following tokens:
     *
     * | Unit        | Token | Description                       |
     * |-------------|-------|-----------------------------------|
     * | Year        | YYYY  | 4-digits year, like: 2018         |
     * |             | Y     | numeric, any number of digits     |
     * |             | YY    | < 68 -> 2000, > 68 -> 1900        |
     * | Month       | MM    | 01 - 12                           |
     * | Month       | MMM   | Short name of the month           |
     * | Date        | DD    | 01 - 31                           |
     * | Hour        | HH    | 00 - 23 or 1 - 12                 |
     * | Minute      | mm    | 00 - 59                           |
     * | Second      | ss    | 00 - 59                           |
     * | Millisecond | S     | 0 - 9 [000, 100, 200 .. 900 ]     |
     * |             | SS    | 00 - 99 [000, 010, 020 .. 990 ]   |
     * |             | SSS   | 000 - 999 [000, 001, 002 .. 999 ] |
     * | AM/PM       | A     | AM or PM                          |
     * |             | a     | am or pm                          |
     * | TimeZone    | Z     | Z for UTC or +-HH:mm              |
     * | Predefined  | L     | Long date, MM/DD/YYYY             |
     * |             | LT    | Long time, HH:mm A                |
     *
     * Default parse format is: `'YYYY-MM-DDTHH:mm:ss.SSSZ'` see {@link #property-defaultParseFormat-static}
     *
     * For example:
     *
     * ```javascript
     * DateHelper.parse('2018-11-06', 'YYYY-MM-DD');
     * DateHelper.parse('13:14', 'HH:mm');
     * DateHelper.parse('6/11/18', 'DD/MM/YY');
     * ```
     *
     * ## Formatting dates
     * Use `DateHelper.format()` to create a string from a date using a format specifier. The format specifier is similar to
     * that used when parsing strings. It can use the following tokens (input used for output below is
     * `new Date(2018,8,9,18,7,8,145)`):
     *
     * | Unit                  | Token | Description & output                  |
     * |-----------------------|-------|---------------------------------------|
     * | Year                  | YYYY  | 2018                                  |
     * |                       | YY    | 18                                    |
     * |                       | Y     | 2018                                  |
     * | Quarter               | Q     | 3                                     |
     * |                       | Qo    | 3rd                                   |
     * | Month                 | MMMM  | September                             |
     * |                       | MMM   | Sep                                   |
     * |                       | MM    | 09                                    |
     * |                       | Mo    | 9th                                   |
     * |                       | M     | 9                                     |
     * | Week (iso)            | WW    | 37 (2 digit, zero padded)             |
     * |                       | Wo    | 37th                                  |
     * |                       | W     | 37                                    |
     * |                       | WWp   | Week 37 (localized prefix, zero pad)  |
     * |                       | Wp    | Week 37 (localized prefix)            |
     * |                       | WWp0  | W37 (localized prefix)                |
     * |                       | Wp0   | W37 (localized prefix)                |
     * | Date                  | DDDD  | Day of year, 3 digits                 |
     * |                       | DDDo  | Day of year, ordinal                  |
     * |                       | DDD   | Day of year                           |
     * |                       | DD    | 09                                    |
     * |                       | Do    | 9th                                   |
     * |                       | D     | 9                                     |
     * | Weekday               | dddd  | Sunday                                |
     * |                       | ddd   | Sun                                   |
     * |                       | dd    | Su                                    |
     * |                       | d1    | S                                     |
     * |                       | do    | 0th                                   |
     * |                       | d     | 0                                     |
     * | Hour                  | HH    | 18 (00 - 23)                          |
     * |                       | H     | 18 (0 - 23)                           |
     * |                       | hh    | 06 (00 - 12)                          |
     * |                       | h     | 6 (0 - 12)                            |
     * |                       | KK    | 19 (01 - 24)                          |
     * |                       | K     | 19 (1 - 24)                           |
     * |                       | kk    | 06 or 18, locale determines           |
     * |                       | k     | 6 or 18, locale determines            |
     * | Minute                | mm    | 07                                    |
     * |                       | m     | 7                                     |
     * | Second                | ss    | 08                                    |
     * |                       | s     | 8                                     |
     * | Millisecond           | S     | 1 (100ms)                             |
     * |                       | SS    | 14 (140ms)                            |
     * |                       | SSS   | 145 (145ms)                           |
     * | AM/PM                 | A     | AM or PM                              |
     * |                       | a     | am or pm                              |
     * | Predefined            | LT    | H: 2-digit (2d), m: 2d                |
     * | (uses browser locale) | LTS   | H: 2d, m: 2d, s : 2d                  |
     * |                       | LST   | Depends on 12 or 24 hour clock        |
     * |                       |       | 12h, H : 1d, m : 0 or 2d              |
     * |                       |       | 24h, H : 2d, m : 2d                   |
     * |                       | L     | Y: numeric (n), M : 2d, D : 2d        |
     * |                       | l     | Y: n, M : n, D : n                    |
     * |                       | LL    | Y: n, M : long (l), D : n             |
     * |                       | ll    | Y: n, M : short (s), D : n            |
     * |                       | LLL   | Y: n, M : l, D : n, H: n, m: 2d       |
     * |                       | lll   | Y: n, M : s, D : n, H: n, m: 2d       |
     * |                       | LLLL  | Y: n, M : l, D : n, H: n, m: 2d, d: l |
     * |                       | llll  | Y: n, M : s, D : n, H: n, m: 2d, d: s |
     * |                       | u     | YYYYMMDDZ in UTC zone                 |
     * |                       | uu    | YYYYMMDDTHHMMSSZ in UTC zone          |
     *
     * Default format is: `'YYYY-MM-DDTHH:mm:ssZ'` see {@link #property-defaultFormat-static}
     *
     * For example:
     *
     * ```javascript
     * DateHelper.format(new Date(2018,10,6), 'YYYY-MM-DD'); // 2018-11-06
     * DateHelper.format(new Date(2018,10,6), 'M/D/YY'); // 11/6/18
     * ```
     *
     * Arbitrary text can be embedded in the format string by wrapping it with {}:
     *
     * ```javascript
     * DateHelper.format(new Date(2019, 7, 16), '{It is }dddd{, yay!}') -> It is Friday, yay!
     * ```
     *
     * ## Unit names
     * Many DateHelper functions (for example add, as, set) accepts a unit among their params. The following units are
     * available:
     *
     * | Unit        | Aliases                       |
     * |-------------|-------------------------------|
     * | millisecond | millisecond, milliseconds, ms |
     * | second      | second, seconds, s            |
     * | minute      | minute, minutes, m            |
     * | hour        | hour, hours, h                |
     * | day         | day, days, d                  |
     * | week        | week, weeks, w                |
     * | month       | month, months, mon, mo, M     |
     * | quarter     | quarter, quarters, q          |
     * | year        | year, years, y                |
     * | decade      | decade, decades, dec          |
     *
     * For example:
     * ```javascript
     * DateHelper.add(date, 2, 'days');
     * DateHelper.as('hour', 7200, 'seconds');
     * ```
     * @static
     */
    class DateHelper extends Localizable() {
      static MS_PER_DAY = MS_PER_HOUR * 24;
      static get $name() {
        return 'DateHelper';
      }
      //region Parse & format
      /**
       * Get/set the default format used by `format()` and `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ssZ'`
       * (~ISO 8601 Date and time, `'1962-06-17T09:21:34Z'`).
       * @member {String}
       */
      static set defaultFormat(format) {
        DH$2._defaultFormat = format;
      }
      static get defaultFormat() {
        return DH$2._defaultFormat || 'YYYY-MM-DDTHH:mm:ssZ';
      }
      /**
       * Get/set the default format used by `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ss.SSSZ'` or {@link #property-defaultFormat-static}
       * (~ISO 8601 Date and time, `'1962-06-17T09:21:34.123Z'`).
       * @member {String}
       */
      static set defaultParseFormat(parseFormat) {
        this._defaultParseFormat = parseFormat;
      }
      static get defaultParseFormat() {
        return this._defaultParseFormat || this._defaultFormat || 'YYYY-MM-DDTHH:mm:ss.SSSZ';
      }
      static buildParser(format) {
        // Split input format by regexp, which includes predefined patterns. Normally format would have some
        // splitters, like 'YYYY-MM-DD' or 'D/M YYYY' so output will contain matched patterns as well as splitters
        // which would serve as anchors. E.g. provided format is 'D/M!YYYY' and input is `11/6!2019` algorithm would work like:
        // 1. split format by regexp                // ['', 'D', '/', 'M', '!', 'YYYY', '']
        // 2. find splitters                        // ['/', '!']
        // 3. split input by seps, step by step     // ['11', ['6', ['2019']]]
        // Inputs like 'YYYYY' (5*Y) means 'YYYY' + 'Y', because it matches patterns from longer to shorter,
        // but if few patterns describe same unit the last one is applied, for example
        // DH.parse('20182015', 'YYYYY') equals to new Date(2015, 0, 0)
        const parts = format.split(parserRegexp),
          parser = [];
        // if length of the parts array is 1 - there are no regexps in the input string. thus - no parsers
        // do same if there are patterns matching locale strings (l, ll, LLLL etc.)
        // returning empty array to use new Date() as parser
        if (parts.length === 1 || localeStrRegExp.test(format)) {
          return [];
        } else {
          parts.reduce((prev, curr, index, array) => {
            // ignore first and last empty string
            if (index !== 0 || curr !== '') {
              // if current element matches parser regexp store it as a parser
              if (parserRegexp.test(curr)) {
                const localeParsers = this.localize('L{parsers}') || {},
                  fn = localeParsers[curr] || parsers[curr];
                // Z should be last element in the string that matches regexp. Last array element is always either
                // an empty string (if format ends with Z) or splitter (everything that doesn't match regexp after Z)
                // If there is a pattern after Z, then Z index will be lower than length - 2
                if (curr === 'Z' && index < array.length - 2) {
                  throw new Error(`Invalid format ${format} TimeZone (Z) must be last token`);
                }
                const parserObj = typeof fn === 'function' || typeof fn === 'string' ? fn : fn.parser();
                // If fn is a string, we found an alias (L, LLL, l etc.).
                // Need to build parsers from mapped format and merge with existing
                if (typeof parserObj === 'string') {
                  // we are going to merge nested parsers with current, some cleanup required:
                  // 1. last element is no longer last
                  // 2. need to pass last parser to the next step
                  const nestedParsers = DH$2.buildParser(parserObj),
                    lastItem = nestedParsers.pop();
                  delete lastItem.last;
                  // elevate nested parsers
                  parser.push(...nestedParsers);
                  prev = lastItem;
                } else {
                  prev.pattern = curr;
                  prev.fn = parserObj;
                }
              }
              // if it doesn't match - we've found a splitter
              else {
                prev.splitter = curr;
                parser.push(prev);
                prev = {};
              }
            } else if (Object.prototype.hasOwnProperty.call(prev, 'pattern')) {
              parser.push(prev);
            }
            return prev;
          }, {});
        }
        parser[parser.length - 1].last = true;
        return parser;
      }
      /**
       * A utility function to create a sortable string key for the passed date or ms timestamp using the `'YYYY-MM-DD'`
       * format.
       * @param {Number|Date} ms The Date instance or ms timestamp to generate a key for
       * @returns {String} Date/timestamp as a string with `'YYYY-M-D'` format
       * @internal
       */
      static makeKey(ms) {
        // If an ten character string passed, assume it's already a key
        if (ms.length === 10) {
          return ms;
        }
        // Convert Date to ms timestamp
        if (ms.getTime) {
          ms = ms.getTime();
        }
        // Cache holds ms -> YYYY-MM-DD
        const cached = keyCache[Math.trunc(ms / MS_PER_HOUR)];
        if (cached) {
          return cached;
        }
        tempDate$1.setTime(ms);
        const month = tempDate$1.getMonth() + 1,
          date = tempDate$1.getDate();
        // Not using DateHelper.format to save some cycles, hit a lot
        return keyCache[Math.trunc(ms / MS_PER_HOUR)] = `${tempDate$1.getFullYear()}-${month < 10 ? '0' + month : month}-${date < 10 ? '0' + date : date}`;
      }
      /**
       * A utility function to parse a sortable string to a date using the `'YYYY-MM-DD'` format.
       * @param {String} key The string to return a date for
       * @returns {Date} new Date instance
       * @internal
       */
      static parseKey(key) {
        return DH$2.parse(key, 'YYYY-MM-DD');
      }
      /**
       * Returns a date created from the supplied string using the specified format. Will try to create even if format
       * is left out, by first using the default format (see {@link #property-defaultFormat-static}, by default
       * `YYYY-MM-DDTHH:mm:ssZ`) and then using `new Date(dateString)`.
       * Supported tokens:
       *
       * | Unit        | Token | Description                       |
       * |-------------|-------|-----------------------------------|
       * | Year        | YYYY  | 2018                              |
       * |             | YY    | < 68 -> 2000, > 68 -> 1900        |
       * | Month       | MM    | 01 - 12                           |
       * | Date        | DD    | 01 - 31                           |
       * | Hour        | HH    | 00 - 23 or 1 - 12                 |
       * | Minute      | mm    | 00 - 59                           |
       * | Second      | ss    | 00 - 59                           |
       * | Millisecond | S     | 0 - 9 [000, 100, 200 .. 900 ]     |
       * |             | SS    | 00 - 99 [000, 010, 020 .. 990 ]   |
       * |             | SSS   | 000 - 999 [000, 001, 002 .. 999 ] |
       * | AM/PM       | A     | AM or PM                          |
       * |             | a     | am or pm                          |
       * | TimeZone    | Z     | Z for UTC or +-HH:mm              |
       * | Predefined  | L     | Long date, MM/DD/YYYY             |
       * |             | LT    | Long time, HH:mm A                |
       *
       * Predefined formats and functions used to parse tokens can be localized, see for example the swedish locale SvSE.js
       *
       * NOTE: If no date parameters are provided then `Jan 01 2020` is used as a default date
       *
       * @param {String} dateString Date string
       * @param {String} [format] Date format (or {@link #property-defaultParseFormat-static} if left out)
       * @returns {Date} new Date instance parsed from the string
       * @category Parse & format
       */
      static parse(dateString, format = DH$2.defaultParseFormat, strict = false) {
        if (dateString instanceof Date) {
          return dateString;
        }
        if (typeof dateString !== 'string' || !dateString) {
          return null;
        }
        // // For ISO 8601 native is faster, but not very forgiving
        // if (format === defaultFormat) {
        //     const dt = new Date(dateString);
        //     if (!isNaN(dt)) {
        //         return dt;
        //     }
        // }
        const config = {
          year: null,
          month: null,
          date: null,
          hours: null,
          minutes: null,
          seconds: null,
          milliseconds: null
        };
        // Milliseconds parser is the same for S, SS, SSS
        // We search for a string of 'S' characters *not* preceded by an alpha character.
        // So that the formats such as 'LTS' do not get corrupted
        format = format.replace(msRegExp, msReplacer);
        let parser = parserCache[format],
          result;
        if (!parser) {
          parser = parserCache[format] = DH$2.buildParser(format);
        }
        // Since Unicode 15 standard arrived to browsers (Chrome 110+ and FF 109+) they add unicode "thin" space before AM/PM
        // https://icu.unicode.org/download/72
        // Convert unicode spaces to regular for parser
        if (dateString.includes('\u202f')) {
          dateString = dateString.replace(/\s/g, ' ');
        }
        // Each parser knows its pattern and splitter. It looks for splitter in the
        // input string, takes first substring and tries to process it. Remaining string
        // is passed to the next parser.
        parser.reduce((dateString, parser) => {
          if (parser.last) {
            Object.assign(config, parser.fn(dateString));
          } else {
            let splitAt;
            // ISO 8601 says that T symbol can be replaced with a space
            if (parser.splitter === 'T' && dateString.indexOf('T') === -1) {
              splitAt = dateString.indexOf(' ');
            } else {
              var _parser$pattern;
              const timeZoneIndex = dateString.indexOf('+');
              let {
                splitter
              } = parser;
              // Use more forgiving regexp for parsing if strict mode is off
              if (!strict && splitRegExp.test(splitter)) {
                splitter = splitRegExp;
              }
              // If splitter specified find its position, otherwise try to determine pattern length
              splitAt = parser.splitter !== '' ? dateString.search(typeof splitter === 'string' ? StringHelper.escapeRegExp(splitter) : splitter) : ((_parser$pattern = parser.pattern) === null || _parser$pattern === void 0 ? void 0 : _parser$pattern.length) || -1;
              // Don't split in the time zone part
              if (timeZoneIndex > -1 && splitAt > timeZoneIndex) {
                splitAt = -1;
              }
            }
            let part, rest;
            // If splitter is not found in the current string we may be dealing with
            // 1. partial input - in that case we just feed all string to current parser and move on
            // 2. time zone (ssZ - splitter is empty string) and pattern is not specified, see comment below
            // 3. parse milliseconds before Z
            if (splitAt === -1 || parser.pattern === 'SSS' && dateString.match(/^\d+Z$/)) {
              // NOTE: parentheses are required here as + and - signs hold valuable information
              // with parentheses we get array like ['00','+','01:00'], omitting them we won't get
              // regexp match in result, loosing information
              const chunks = dateString.split(/([Z\-+])/);
              // If splitter is not found in the string, we may be dealing with string that contains info about TZ.
              // For instance, if format contains Z as last arg which is not separated (normally it is not indeed),
              // like 'YYYY-MM-DD HH:mm:ssZ', then second to last parser will have string that it cannot just parse, like
              // '2010-01-01 10:00:00'        -> '00'
              // '2010-01-01 10:00:00Z'       -> '00Z'
              // '2010-01-01 10:00:00-01'     -> '00-01'
              // '2010-01-01 10:00:00+01:30'  -> '00+01:30'
              // this cannot be processed by date parsers, so we need to process that additionally. So we
              // split string by symbols that can be found around timezone info: Z,-,+
              if (chunks.length === 1) {
                part = dateString;
                rest = '';
              } else {
                part = chunks[0];
                rest = `${chunks[1]}${chunks[2]}`;
              }
            } else {
              part = dateString.substring(0, splitAt) || dateString;
              rest = dateString.substring(splitAt + parser.splitter.length);
            }
            if (parser.fn) {
              // Run parser and add result to config on successful parse otherwise continue parsing
              const res = parser.fn(part);
              if (res) {
                Object.assign(config, res);
              } else {
                rest = part + rest;
              }
            }
            return rest;
          }
        }, dateString);
        // If year is specified date has to be greater than 0
        if (config.year && !config.date) {
          config.date = 1;
        }
        if (config.date > 31 || config.month > 12) {
          return null;
        }
        const date = DH$2.create(config, strict);
        if (date) {
          result = date;
        } else if (!strict) {
          // Last resort, try if native passing can do it
          result = new Date(dateString);
        }
        return result;
      }
      /**
       * Creates a date from a date definition object. The object can have the following properties:
       * - year
       * - month
       * - date (day in month)
       * - hours
       * - minutes
       * - seconds
       * - milliseconds
       * - amPm : 'am' or 'pm', implies 12-hour clock
       * - timeZone : offset from UTC in minutes
       * @param {Object} definition
       * @param {Number} definition.year
       * @param {Number} [definition.month]
       * @param {Number} [definition.date]
       * @param {Number} [definition.hours]
       * @param {Number} [definition.minutes]
       * @param {Number} [definition.seconds]
       * @param {Number} [definition.milliseconds]
       * @param {Number} [definition.amPm]
       * @param {Number} [definition.timeZone]
       * @returns {Date} new Date instance
       * @category Parse & format
       */
      static create(definition, strict = false) {
        // Shallow clone to not alter input
        const def = {
          ...definition
        };
        let invalid = isNaN(def.year) || strict && (isNaN(def.month) || isNaN(def.date)),
          useUTC = false;
        // Not much validation yet, only considered invalid if all properties are null
        if (!invalid) {
          let allNull = true;
          dateProperties.forEach(property => {
            if (!(property in def) || isNaN(def[property])) {
              def[property] = 0;
            }
            allNull = allNull && def[property] === null;
          });
          invalid = allNull;
        }
        if (invalid) {
          return null;
        }
        if (def.amPm === 'am') {
          def.hours = def.hours % 12;
        } else if (def.amPm === 'pm') {
          def.hours = def.hours % 12 + 12;
        }
        if ('timeZone' in def) {
          useUTC = true;
          def.minutes -= def.timeZone;
        }
        if (strict && (def.year == null || def.month == null || def.date == null)) {
          return null;
        }
        const args = [defaultValue(def.year, DEFAULT_YEAR), defaultValue(def.month, DEFAULT_MONTH), defaultValue(def.date, DEFAULT_DAY), def.hours, def.minutes, def.seconds, def.milliseconds];
        return useUTC ? new Date(Date.UTC(...args)) : new Date(...args);
      }
      static toUTC(date) {
        return new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate(), date.getUTCHours(), date.getUTCMinutes(), date.getUTCSeconds(), date.getUTCMilliseconds()));
      }
      /**
       * Converts a date to string with the specified format. Formats heavily inspired by https://momentjs.com.
       * Available formats (input used for output below is `new Date(2018,8,9,18,7,8,145)`):
       *
       * | Unit                  | Token | Description & output                  |
       * |-----------------------|-------|---------------------------------------|
       * | Year                  | YYYY  | 2018                                  |
       * |                       | YY    | 18                                    |
       * |                       | Y     | 2018                                  |
       * | Quarter               | Q     | 3                                     |
       * |                       | Qo    | 3rd                                   |
       * | Month                 | MMMM  | September                             |
       * |                       | MMM   | Sep                                   |
       * |                       | MM    | 09                                    |
       * |                       | Mo    | 9th                                   |
       * |                       | M     | 9                                     |
       * | Week (iso)            | WW    | 37 (2 digit, zero padded)             |
       * |                       | Wo    | 37th                                  |
       * |                       | W     | 37                                    |
       * |                       | WWp   | Week 37 (localized prefix, zero pad)  |
       * |                       | Wp    | Week 37 (localized prefix)            |
       * |                       | WWp0  | W37 (localized prefix)                |
       * |                       | Wp0   | W37 (localized prefix)                |
       * | Date                  | DDDD  | Day of year, 3 digits                 |
       * |                       | DDDo  | Day of year, ordinal                  |
       * |                       | DDD   | Day of year                           |
       * |                       | DD    | 09                                    |
       * |                       | Do    | 9th                                   |
       * |                       | D     | 9                                     |
       * | Weekday               | dddd  | Sunday                                |
       * |                       | ddd   | Sun                                   |
       * |                       | dd    | Su                                    |
       * |                       | d1    | S                                     |
       * |                       | do    | 0th                                   |
       * |                       | d     | 0                                     |
       * | Hour                  | HH    | 18 (00 - 23)                          |
       * |                       | H     | 18 (0 - 23)                           |
       * |                       | hh    | 06 (00 - 12)                          |
       * |                       | h     | 6 (0 - 12)                            |
       * |                       | KK    | 19 (01 - 24)                          |
       * |                       | K     | 19 (1 - 24)                           |
       * |                       | kk    | 06 or 18, locale determines           |
       * |                       | k     | 6 or 18, locale determines            |
       * | Minute                | mm    | 07                                    |
       * |                       | m     | 7                                     |
       * | Second                | ss    | 08                                    |
       * |                       | s     | 8                                     |
       * | Millisecond           | S     | 1 (100ms)                             |
       * |                       | SS    | 14 (140ms)                            |
       * |                       | SSS   | 145 (145ms)                           |
       * | AM/PM                 | A     | AM or PM                              |
       * |                       | a     | am or pm                              |
       * | Predefined            | LT    | H: 2-digit (2d), m: 2d                |
       * | (uses browser locale) | LTS   | H: 2d, m: 2d, s : 2d                  |
       * |                       | LST   | Depends on 12 or 24 hour clock        |
       * |                       |       | 12h, H : 1d, m : 0 or 2d              |
       * |                       |       | 24h, H : 2d, m : 2d                   |
       * |                       | L     | Y: numeric (n), M : 2d, D : 2d        |
       * |                       | l     | Y: n, M : n, D : n                    |
       * |                       | LL    | Y: n, M : long (l), D : n             |
       * |                       | ll    | Y: n, M : short (s), D : n            |
       * |                       | LLL   | Y: n, M : l, D : n, H: n, m: 2d       |
       * |                       | lll   | Y: n, M : s, D : n, H: n, m: 2d       |
       * |                       | LLLL  | Y: n, M : l, D : n, H: n, m: 2d, d: l |
       * |                       | llll  | Y: n, M : s, D : n, H: n, m: 2d, d: s |
       *
       * Some examples:
       *
       * ```javascript
       * DateHelper.format(new Date(2019, 7, 16), 'dddd') -> Friday
       * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'HH:mm') --> 14:27
       * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'L HH') --> 2019-07-16 14
       * ```
       *
       * Arbitrary text can be embedded in the format string by wrapping it with {}:
       *
       * ```javascript
       * DateHelper.format(new Date(2019, 7, 16), '{It is }dddd{, yay!}') -> It is Friday, yay!
       * ```
       *
       * @param {Date} date Date
       * @param {String} [format] Desired format (uses `defaultFormat` if left out)
       * @returns {String} Formatted string
       * @category Parse & format
       */
      static format(date, format = DH$2.defaultFormat) {
        // Bail out if no date or invalid date
        if (!date || isNaN(date)) {
          return null;
        }
        let formatter = formatCache[format],
          output = '';
        if (!formatter) {
          formatter = formatCache[format] = [];
          // Build formatter array with the steps needed to format the date
          for (let i = 0; i < format.length; i++) {
            // Matches a predefined format?
            const formatMatch = format.slice(i).match(formatRegexp),
              predefined = formatMatch === null || formatMatch === void 0 ? void 0 : formatMatch[0];
            if (predefined) {
              const localeFormats = this.localize('L{formats}') || {},
                fn = localeFormats[predefined] || formats[predefined];
              formatter.push(fn);
              i += predefined.length - 1;
            }
            // Start of text block? Append it
            else if (format[i] === '{') {
              // Find closing brace
              const index = format.indexOf('}', i + 1);
              // No closing brace, grab rest of string
              if (index === -1) {
                formatter.push(format.substr(i + 1));
                i = format.length;
              }
              // Closing brace found
              else {
                formatter.push(format.substring(i + 1, index));
                // Carry on after closing brace
                i = index;
              }
            }
            // Otherwise append to output (for example - / : etc)
            else {
              formatter.push(format[i]);
            }
          }
        }
        formatter.forEach(step => {
          if (typeof step === 'string') {
            output += step;
          } else {
            output += step(date);
          }
        });
        return output;
      }
      /**
       * Formats a range of `dates` using the specified `format`. Because two dates are involved, the `format` specifier
       * uses the tokens `S{}` and `E{}`. The text contained between the `{}` is the {@link #function-format-static format}
       * for the start date or end date, respectively. Text not inside these tokens is retained verbatim.
       *
       * For example:
       *
       * ```javascript
       *  DateHelper.formatRange(dates, 'S{DD MMM YYYY} - E{DD MMM YYYY}');
       * ```
       *
       * The above will format `dates[0]` based on the `S{DD MMM YYYY}` segment and `dates[1] using `E{DD MMM YYYY}`. The
       * `' - '` between these will remain between the two formatted dates.
       *
       * @param {Date[]} dates An array of start date and end date (`[startDate, endDate]`)
       * @param {String} format The format specifier
       * @returns {String}
       */
      static formatRange(dates, format) {
        return format.replace(rangeFormatPartRe, (s, which, fmt) => DateHelper.format(dates[which === 'S' ? 0 : 1], fmt));
      }
      /**
       * Converts the specified amount of desired unit into milliseconds. Can be called by only specifying a unit as the
       * first argument, it then uses `amount = 1`.
       *
       * For example:
       *
       * ```javascript
       * asMilliseconds('hour') == asMilliseconds(1, 'hour')
       * ```
       *
       * @param {Number|String} amount Amount, what of is decided by specifying unit (also takes a unit which implies an amount of 1)
       * @param {String} [unit] Time unit (s, hour, months etc.)
       * @returns {Number}
       * @category Parse & format
       */
      static asMilliseconds(amount, unit = null) {
        if (typeof amount === 'string') {
          unit = amount;
          amount = 1;
        }
        return DH$2.as('millisecond', amount, unit);
      }
      /**
       * Converts the passed Date to an accurate number of months passed since the epoch start.
       * @param {Date} time The Date to find the month value of
       * @returns {Number} The number of months since the system time epoch start. May be a fractional value
       */
      static asMonths(time) {
        const monthLength = DH$2.as('ms', DH$2.daysInMonth(time), 'day'),
          fraction = (time.valueOf() - DH$2.startOf(time, 'month').valueOf()) / monthLength;
        return time.getYear() * 12 + time.getMonth() + fraction;
      }
      static monthsToDate(months) {
        const intMonths = Math.floor(months),
          fraction = months - intMonths,
          result = new Date(0, intMonths),
          msInMonth = DH$2.as('ms', DH$2.daysInMonth(result), 'days');
        result.setTime(result.getTime() + fraction * msInMonth);
        return result;
      }
      /**
       * Converts a millisecond time delta to a human-readable form. For example `1000 * 60 * 60 * 50`
       * milliseconds would be rendered as `'2 days, 2 hours'`.
       * @param {Number} delta The millisecond delta value
       * @param {Object} [options] Formatting options
       * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg `'2d, 2h'` for the above example
       * @param {String} [options.precision] The minimum precision unit
       * @param {String} [options.separator] The separator to use
       * @param {Boolean} [options.asString] Pass `false` to return the result as an array, eg ['2d', '2h'] for the above example
       * @returns {String} Formatted string
       * @category Parse & format
       */
      static formatDelta(delta, options) {
        let abbrev, unitName;
        if (typeof options === 'boolean') {
          abbrev = options;
        } else if (options) {
          abbrev = options.abbrev;
        }
        const deltaObj = this.getDelta(delta, options),
          result = [],
          sep = (options === null || options === void 0 ? void 0 : options.separator) || (abbrev ? '' : ' ');
        for (unitName in deltaObj) {
          result.push(`${deltaObj[unitName]}${sep}${unitName}`);
        }
        return (options === null || options === void 0 ? void 0 : options.asString) === false ? result : result.join(', ');
      }
      /**
       * Converts a millisecond time delta to an object structure. For example `1000 * 60 * 60 * 50`
       * milliseconds the result would be as:
       *
       * ```javascript
       * {
       *     day  : 2,
       *     hour : 2
       * }
       *```
       *
       * @param {Number} delta The millisecond delta value
       * @param {Object} [options] Formatting options
       * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg `{ d: 2, h: 2 }` for the above example
       * @param {String} [options.precision] The minimum precision unit
       * @param {Boolean} [options.ignoreLocale] Pass true to return unlocalized unit name. Requires `abbrev` to be false
       * @param {String} [options.maxUnit] Name of the maximum unit in the output. e.g. if you pass `day` then you'll get
       * `{ h: 25 }` instead of `{ d: 1, h: 1 }`
       * @returns {Object} The object with the values for each unit
       */
      static getDelta(delta, options) {
        let abbrev, d, done, precision, unitName, maxUnit, ignoreLocale;
        if (typeof options === 'boolean') {
          abbrev = options;
        } else if (options) {
          abbrev = options.abbrev;
          precision = DH$2.normalizeUnit(options.precision);
          maxUnit = options.maxUnit;
          ignoreLocale = !abbrev && options.ignoreLocale;
        }
        const result = {},
          getUnit = abbrev ? DH$2.getShortNameOfUnit : DH$2.getLocalizedNameOfUnit;
        const units = maxUnit ? deltaUnits.slice(deltaUnits.indexOf(maxUnit)) : deltaUnits;
        // Loop downwards through the magnitude of units from year -> ms
        for (unitName of units) {
          d = DH$2.as(unitName, delta);
          done = precision === unitName;
          d = Math[done ? 'round' : 'floor'](d);
          // If there's a non-zero integer quantity of this unit, add it to result
          // and subtract from the delta, then go round to next unit down.
          if (d || done && !result.length) {
            result[ignoreLocale ? unitName : getUnit.call(DH$2, unitName, d !== 1)] = d;
            delta -= DH$2.as('ms', d, unitName);
          }
          if (done || !delta) {
            break;
          }
        }
        return result;
      }
      /**
       * Converts the specified amount of one unit (`fromUnit`) into an amount of another unit (`toUnit`).
       * @param {String} toUnit The name of units to convert to, eg: `'ms'`
       * @param {Number|String} amount The time to convert. Either the magnitude number form or a duration string such as '2d'
       * @param {String} [fromUnit='ms'] If the amount was passed as a number, the units to use to convert from
       * @returns {Number}
       * @category Parse & format
       */
      static as(toUnit, amount, fromUnit = 'ms') {
        // Allow DH.as('ms', '2d')
        if (typeof amount === 'string') {
          amount = DH$2.parseDuration(amount);
        }
        // Allow DH.as('ms', myDurationObject)
        if (typeof amount === 'object') {
          fromUnit = amount.unit;
          amount = amount.magnitude;
        }
        if (toUnit === fromUnit) {
          return amount;
        }
        toUnit = DH$2.normalizeUnit(toUnit);
        fromUnit = DH$2.normalizeUnit(fromUnit);
        if (toUnit === fromUnit) {
          return amount;
        }
        // validConversions[][] can be negative to signal that conversion is not exact, ignore sign here
        else if (unitMagnitudes[fromUnit] > unitMagnitudes[toUnit]) {
          return amount * Math.abs(validConversions[fromUnit][toUnit]);
        } else {
          return amount / Math.abs(validConversions[toUnit][fromUnit]);
        }
      }
      static formatContainsHourInfo(format) {
        const stripEscapeRe = /(\\.)/g,
          hourInfoRe = /([HhKkmSsAa]|LT|L{3,}|l{3,})/;
        return hourInfoRe.test(format.replace(stripEscapeRe, ''));
      }
      /**
       * Returns `true` for 24-hour format.
       * @param {String} format Date format
       * @returns {Boolean} `true` for 24-hour format
       * @category Parse & format
       */
      static is24HourFormat(format) {
        return DH$2.format(DH$2.getTime(13, 0, 0), format).includes('13');
      }
      //endregion
      //region Manipulate
      /**
       * Add days, hours etc. to a date. Always clones the date, original will be left unaffected.
       * @param {Date|String} date Original date
       * @param {Number|String|Core.data.Duration|DurationConfig} amount Amount of days, hours etc. or a string representation of a duration
       * as accepted by {@link #function-parseDuration-static} or an object with `{ magnitude, unit }` properties
       * @param {String} [unit='ms'] Unit for amount
       * @privateparam {Boolean} [clone=true] Pass `false` to affect the original
       * @returns {Date} New calculated date
       * @category Manipulate
       */
      static add(date, amount, unit = 'ms', clone = true) {
        let d;
        if (typeof date === 'string') {
          d = DH$2.parse(date);
        } else if (clone) {
          d = new Date(date.getTime());
        } else {
          d = date;
        }
        if (typeof amount === 'string') {
          const duration = DateHelper.parseDuration(amount);
          amount = duration.magnitude;
          unit = duration.unit;
        } else if (amount && typeof amount === 'object') {
          unit = amount.unit;
          amount = amount.magnitude;
        }
        if (!unit || amount === 0) {
          return d;
        }
        unit = DH$2.normalizeUnit(unit);
        switch (unit) {
          case 'millisecond':
            d.setTime(d.getTime() + amount);
            break;
          case 'second':
            d.setTime(d.getTime() + amount * 1000);
            break;
          case 'minute':
            d.setTime(d.getTime() + amount * 60000);
            break;
          case 'hour':
            d.setTime(d.getTime() + amount * 3600000);
            break;
          case 'day':
            // Integer value added, do calendar calculation to correctly handle DST etc.
            if (amount % 1 === 0) {
              d.setDate(d.getDate() + amount);
              // When crossing DST in Brazil, we expect hours to end up the same
              if (d.getHours() === 23 && date.getHours() === 0) {
                d.setHours(d.getHours() + 1);
              }
            }
            // No browsers support fractional values for dates any longer, do time based calculation
            else {
              d.setTime(d.getTime() + amount * 86400000);
            }
            break;
          case 'week':
            d.setDate(d.getDate() + amount * 7);
            break;
          case 'month':
            {
              let day = d.getDate();
              if (day > 28) {
                day = Math.min(day, DH$2.getLastDateOfMonth(DH$2.add(DH$2.getFirstDateOfMonth(d), amount, 'month')).getDate());
              }
              d.setDate(day);
              d.setMonth(d.getMonth() + amount);
              break;
            }
          case 'quarter':
            DH$2.add(d, amount * 3, 'month', false);
            break;
          case 'year':
            d.setFullYear(d.getFullYear() + amount);
            break;
          case 'decade':
            d.setFullYear(d.getFullYear() + amount * 10);
            break;
        }
        return d;
      }
      /**
       * Calculates the difference between two dates, in the specified unit.
       * @param {Date} start First date
       * @param {Date} end Second date
       * @param {String} [unit='ms'] Unit to calculate difference in
       * @param {Boolean} [fractional=true] Specify false to round result
       * @returns {Number} Difference in the specified unit
       * @category Manipulate
       */
      static diff(start, end, unit = 'ms', fractional = true) {
        unit = DH$2.normalizeUnit(unit);
        if (!start || !end) return 0;
        let amount;
        switch (unit) {
          case 'year':
            amount = DH$2.diff(start, end, 'month') / 12;
            break;
          case 'quarter':
            amount = DH$2.diff(start, end, 'month') / 3;
            break;
          case 'month':
            amount = (end.getFullYear() - start.getFullYear()) * 12 + (end.getMonth() - start.getMonth());
            if (amount === 0 && fractional) {
              amount = DH$2.diff(start, end, 'day', fractional) / DH$2.daysInMonth(start);
            }
            break;
          case 'week':
            amount = DH$2.diff(start, end, 'day') / 7;
            break;
          case 'day':
            {
              const dstDiff = start.getTimezoneOffset() - end.getTimezoneOffset();
              amount = (end - start + dstDiff * 60 * 1000) / 86400000;
              break;
            }
          case 'hour':
            amount = (end - start) / 3600000;
            break;
          case 'minute':
            amount = (end - start) / 60000;
            break;
          case 'second':
            amount = (end - start) / 1000;
            break;
          case 'millisecond':
            amount = end - start;
            break;
        }
        return fractional ? amount : Math.round(amount);
      }
      /**
       * Sets the date to the start of the specified unit, by default returning a clone of the date instead of changing it
       * in place.
       * @param {Date} date Original date
       * @param {String} [unit='day'] Start of this unit, `'day'`, `'month'` etc
       * @param {Boolean} [clone=true] Manipulate a copy of the date
       * @param {Number} [weekStartDay] The first day of week, `0-6` (Sunday-Saturday). Defaults to the {@link #property-weekStartDay-static}
       * @returns {Date} Manipulated date
       * @category Manipulate
       */
      static startOf(date, unit = 'day', clone = true, weekStartDay = DH$2.weekStartDay) {
        if (!date) {
          return null;
        }
        unit = DH$2.normalizeUnit(unit);
        if (clone) {
          date = DH$2.clone(date);
        }
        switch (unit) {
          case 'year':
            date.setMonth(0, 1);
            date.setHours(0, 0, 0, 0);
            return date;
          case 'quarter':
            date.setMonth((DH$2.get(date, 'quarter') - 1) * 3, 1);
            date.setHours(0, 0, 0, 0);
            return date;
          case 'month':
            date.setDate(1);
            date.setHours(0, 0, 0, 0);
            return date;
          case 'week':
            {
              const delta = date.getDay() - weekStartDay;
              date.setDate(date.getDate() - delta);
              date.setHours(0, 0, 0, 0);
              return date;
            }
          case 'day':
            date.setHours(0, 0, 0, 0);
            return date;
          // Cant use setMinutes(0, 0, 0) etc. for DST transitions
          case 'hour':
            date.getMinutes() > 0 && date.setMinutes(0);
          // eslint-disable-next-line no-fallthrough
          case 'minute':
            date.getSeconds() > 0 && date.setSeconds(0);
          // eslint-disable-next-line no-fallthrough
          case 'second':
            date.getMilliseconds() > 0 && date.setMilliseconds(0);
          // eslint-disable-next-line no-fallthrough
          case 'millisecond':
            return date;
        }
      }
      /**
       * Returns the end point of the passed date, that is 00:00:00 of the day after the passed date.
       * @param {Date} date The date to return the end point of
       * @returns {Date} Manipulated date
       */
      static endOf(date) {
        return new Date(date.getFullYear(), date.getMonth(), date.getDate() + 1);
      }
      /**
       * Creates a clone of the specified date
       * @param {Date} date Original date
       * @returns {Date} Cloned date
       * @category Manipulate
       */
      static clone(date) {
        return new Date(date.getTime());
      }
      /**
       * Removes time from a date (same as calling {@link #function-startOf-static startOf(date)}).
       * @param {Date} date Date to remove time from
       * @param {Boolean} [clone=true] Manipulate a copy of the date
       * @returns {Date} Manipulated date
       * @category Manipulate
       */
      static clearTime(date, clone = true) {
        if (!date) {
          return null;
        }
        if (clone) {
          date = new Date(date.getTime());
        }
        date.setHours(0, 0, 0, 0);
        return date;
      }
      static midnight(date, inclusive) {
        let ret = DH$2.clearTime(date);
        if (inclusive && ret < date) {
          ret = DH$2.add(ret, 1, 'd');
        }
        return ret;
      }
      /**
       * Returns the elapsed milliseconds from the start of the specified date.
       * @param {Date} date Date to remove date from
       * @param {String} [unit='ms'] The time unit to return
       * @returns {Number} The elapsed milliseconds from the start of the specified date
       * @category Manipulate
       */
      static getTimeOfDay(date, unit = 'ms') {
        const t = date.getHours() * validConversions.hour.millisecond + date.getMinutes() * validConversions.minute.millisecond + date.getSeconds() * validConversions.second.millisecond + date.getMilliseconds();
        return unit === 'ms' ? t : DH$2.as(unit, t, 'ms');
      }
      /**
       * Sets a part of a date (in place).
       * @param {Date} date Date to manipulate
       * @param {String|Object} unit Part of date to set, for example `'minute'`. Or an object like `{ second: 1, minute: 1 }`
       * @param {Number} amount Value to set
       * @returns {Date} Passed date instance modified according to the arguments
       * @category Manipulate
       */
      static set(date, unit, amount) {
        if (!unit) {
          return date;
        }
        if (typeof unit === 'string') {
          switch (DH$2.normalizeUnit(unit)) {
            case 'millisecond':
              // Setting value to 0 when it is 0 at DST crossing messes it up
              if (amount !== 0 || date.getMilliseconds() > 0) {
                date.setMilliseconds(amount);
              }
              break;
            case 'second':
              // Setting value to 0 when it is 0 at DST crossing messes it up
              if (amount !== 0 || date.getSeconds() > 0) {
                date.setSeconds(amount);
              }
              break;
            case 'minute':
              // Setting value to 0 when it is 0 at DST crossing messes it up
              if (amount !== 0 || date.getMinutes() > 0) {
                date.setMinutes(amount);
              }
              break;
            case 'hour':
              date.setHours(amount);
              break;
            case 'day':
            case 'date':
              date.setDate(amount);
              break;
            case 'week':
              throw new Error('week not implemented');
            case 'month':
              date.setMonth(amount);
              break;
            case 'quarter':
              // Setting quarter = first day of first month of that quarter
              date.setDate(1);
              date.setMonth((amount - 1) * 3);
              break;
            case 'year':
              date.setFullYear(amount);
              break;
          }
        } else {
          Object.entries(unit)
          // Make sure smallest unit goes first, to not change month before changing day
          .sort((a, b) => unitMagnitudes[a[0]] - unitMagnitudes[b[0]]).forEach(([unit, amount]) => {
            DH$2.set(date, unit, amount);
          });
        }
        return date;
      }
      static setDateToMidday(date, clone = true) {
        return DH$2.set(DH$2.clearTime(date, clone), 'hour', 12);
      }
      /**
       * Constrains the date within a min and a max date.
       * @param {Date} date The date to constrain
       * @param {Date} [min] Min date
       * @param {Date} [max] Max date
       * @returns {Date} The constrained date
       * @category Manipulate
       */
      static constrain(date, min, max) {
        if (min != null) {
          date = DH$2.max(date, min);
        }
        return max == null ? date : DH$2.min(date, max);
      }
      /**
       * Returns time with default year, month, and day (Jan 1, 2020).
       * @param {Number|Date} hours Hours value or the full date to extract the time of
       * @param {Number} [minutes=0] Minutes value
       * @param {Number} [seconds=0] Seconds value
       * @param {Number} [ms=0] Milliseconds value
       * @returns {Date} A new default date with the time extracted from the given date or from the time values provided individually
       * @category Manipulate
       */
      static getTime(hours, minutes = 0, seconds = 0, ms = 0) {
        if (hours instanceof Date) {
          ms = hours.getMilliseconds();
          seconds = hours.getSeconds();
          minutes = hours.getMinutes();
          hours = hours.getHours();
        }
        return new Date(DEFAULT_YEAR, DEFAULT_MONTH, DEFAULT_DAY, hours, minutes, seconds, ms);
      }
      /**
       * Copies hours, minutes, seconds, milliseconds from one date to another.
       *
       * @param {Date} targetDate The target date
       * @param {Date} sourceDate The source date
       * @returns {Date} The adjusted target date
       * @category Manipulate
       * @static
       */
      static copyTimeValues(targetDate, sourceDate) {
        targetDate.setHours(sourceDate.getHours());
        targetDate.setMinutes(sourceDate.getMinutes());
        targetDate.setSeconds(sourceDate.getSeconds());
        targetDate.setMilliseconds(sourceDate.getMilliseconds());
        return targetDate;
      }
      //endregion
      //region Comparison
      static get isDSTEnabled() {
        const year = new Date().getFullYear(),
          jan = new Date(year, 0, 1),
          jul = new Date(year, 6, 1);
        return jan.getTimezoneOffset() !== jul.getTimezoneOffset();
      }
      static isDST(date) {
        const year = date.getFullYear(),
          jan = new Date(year, 0, 1),
          jul = new Date(year, 6, 1);
        return date.getTimezoneOffset() < Math.max(jan.getTimezoneOffset(), jul.getTimezoneOffset());
      }
      /**
       * Determines if a date precedes another.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @returns {Boolean} `true` if first precedes second, otherwise false
       * @category Comparison
       */
      static isBefore(first, second) {
        return first < second;
      }
      /**
       * Determines if a date succeeds another.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @returns {Boolean} `true` if first succeeds second, otherwise false
       * @category Comparison
       */
      static isAfter(first, second) {
        return first > second;
      }
      /**
       * Checks if two dates are equal.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @param {String} [unit] Unit to calculate difference in. If not given, the comparison will be done up to a millisecond
       * @returns {Boolean} `true` if the dates are equal
       * @category Comparison
       */
      static isEqual(first, second, unit = null) {
        if (unit === null) {
          // https://jsbench.me/3jk2bom2r3/1
          // https://jsbench.me/ltkb3vk0ji/1 (more flavors) - getTime is >2x faster vs valueOf/Number/op+
          return first && second && first.getTime() === second.getTime();
        }
        return DH$2.startOf(first, unit) - DH$2.startOf(second, unit) === 0;
      }
      /**
       * Compares two dates using the specified precision.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @param {String} [unit] Unit to calculate difference in. If not given, the comparison will be done up to a millisecond
       * @returns {Number} `0` = equal, `-1` = first before second, `1` = first after second
       * @category Comparison
       */
      static compare(first, second, unit = null) {
        // Unit specified, cut the rest out
        if (unit) {
          first = DH$2.startOf(first, unit);
          second = DH$2.startOf(second, unit);
        }
        // Comparison on ms level
        if (first < second) return -1;
        if (first > second) return 1;
        return 0;
      }
      /**
       * Coerces the passed Date between the passed minimum and maximum values.
       * @param {Date} date The date to clamp between the `min` and `max`
       * @param {Date} min The minimum Date
       * @param {Date} max The maximum Date
       * @returns {Date} If the passed `date` is valid, a *new* Date object which is clamped between the `min` and `max`
       */
      static clamp(date, min, max) {
        if (!isNaN(date)) {
          if (min != null) {
            date = Math.max(date, min);
          }
          if (max != null) {
            date = Math.min(date, max);
          }
          return new Date(date);
        }
      }
      static isSameDate(first, second) {
        return DH$2.compare(first, second, 'd') === 0;
      }
      static isSameTime(first, second) {
        return first.getHours() === second.getHours() && first.getMinutes() === second.getMinutes() && first.getSeconds() === second.getSeconds() && first.getMilliseconds() === second.getMilliseconds();
      }
      /**
       * Checks if date is the start of specified unit.
       * @param {Date} date Date
       * @param {String} unit Time unit
       * @returns {Boolean} `true` if date is the start of specified unit
       * @category Comparison
       */
      static isStartOf(date, unit) {
        return DH$2.isEqual(date, DH$2.startOf(date, unit));
      }
      /**
       * Checks if this date is `>= start` and `< end`.
       * @param {Date} date The source date
       * @param {Date} start Start date
       * @param {Date} end End date
       * @returns {Boolean} `true` if this date falls on or between the given start and end dates
       * @category Comparison
       */
      static betweenLesser(date, start, end) {
        //return start <= date && date < end;
        return start.getTime() <= date.getTime() && date.getTime() < end.getTime();
      }
      /**
       * Checks if this date is `>= start` and `<= end`.
       * @param {Date} date The source date
       * @param {Date} start Start date
       * @param {Date} end End date
       * @returns {Boolean} `true` if this date falls on or between the given start and end dates
       * @category Comparison
       */
      static betweenLesserEqual(date, start, end) {
        return start.getTime() <= date.getTime() && date.getTime() <= end.getTime();
      }
      /**
       * Returns `true` if dates intersect.
       * @param {Date} date1Start Start date of first span
       * @param {Date} date1End End date of first span
       * @param {Date} date2Start Start date of second span
       * @param {Date} date2End End date of second span
       * @returns {Boolean} Returns `true` if dates intersect
       * @category Comparison
       */
      static intersectSpans(date1Start, date1End, date2Start, date2End) {
        return DH$2.betweenLesser(date1Start, date2Start, date2End) || DH$2.betweenLesser(date2Start, date1Start, date1End);
      }
      /**
       * Compare two units. Returns `1` if first param is a greater unit than second param, `-1` if the opposite is true or `0` if they're equal.
       * @param {String} unit1 The 1st unit
       * @param {String} unit2 The 2nd unit
       * @returns {Number} Returns `1` if first param is a greater unit than second param, `-1` if the opposite is true or `0` if they're equal
       * @category Comparison
       */
      static compareUnits(unit1, unit2) {
        return Math.sign(unitMagnitudes[DH$2.normalizeUnit(unit1)] - unitMagnitudes[DH$2.normalizeUnit(unit2)]);
      }
      /**
       * Returns `true` if the first time span completely 'covers' the second time span.
       *
       * @example
       * DateHelper.timeSpanContains(
       *     new Date(2010, 1, 2),
       *     new Date(2010, 1, 5),
       *     new Date(2010, 1, 3),
       *     new Date(2010, 1, 4)
       * ) ==> true
       * DateHelper.timeSpanContains(
       *   new Date(2010, 1, 2),
       *   new Date(2010, 1, 5),
       *   new Date(2010, 1, 3),
       *   new Date(2010, 1, 6)
       * ) ==> false
       *
       * @param {Date} spanStart The start date for initial time span
       * @param {Date} spanEnd The end date for initial time span
       * @param {Date} otherSpanStart The start date for the 2nd time span
       * @param {Date} otherSpanEnd The end date for the 2nd time span
       * @returns {Boolean} `true` if the first time span completely 'covers' the second time span
       * @category Comparison
       */
      static timeSpanContains(spanStart, spanEnd, otherSpanStart, otherSpanEnd) {
        return otherSpanStart - spanStart >= 0 && spanEnd - otherSpanEnd >= 0;
      }
      //endregion
      //region Query
      /**
       * Get the first day of week, 0-6 (Sunday-Saturday).
       * This is determined by the current locale's `DateHelper.weekStartDay` parameter.
       * @property {Number}
       * @readonly
       */
      static get weekStartDay() {
        // Cache is reset in applyLocale
        if (DH$2._weekStartDay == null) {
          // Defaults to 0, should not need to happen in real world scenarios when a locale is always loaded
          DH$2._weekStartDay = this.localize('L{weekStartDay}') || 0;
        }
        return DH$2._weekStartDay;
      }
      /**
       * Get non-working days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
       * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
       *
       * For example:
       * ```javascript
       * {
       *     0 : true, // Sunday
       *     6 : true  // Saturday
       * }
       * ```
       *
       * @property {Object<Number,Boolean>}
       * @readonly
       */
      static get nonWorkingDays() {
        return {
          ...this.localize('L{nonWorkingDays}')
        };
      }
      /**
       * Get non-working days as an array of day indices, 0-6 (Sunday-Saturday).
       * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
       *
       * For example:
       *
       * ```javascript
       * [ 0, 6 ] // Sunday & Saturday
       * ```
       *
       * @property {Number[]}
       * @readonly
       * @internal
       */
      static get nonWorkingDaysAsArray() {
        // transform string keys to integers
        return Object.keys(this.nonWorkingDays).map(Number);
      }
      /**
       * Get weekend days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
       * Weekends are days which are declared as weekend days by the selected country and defined by the current locale's
       * `DateHelper.weekends` parameter.
       * To get non-working days see {@link #property-nonWorkingDays-static}.
       *
       * For example:
       * ```javascript
       * {
       *     0 : true, // Sunday
       *     6 : true  // Saturday
       * }
       * ```
       * @property {Object<Number,Boolean>}
       * @readonly
       * @internal
       */
      static get weekends() {
        return {
          ...this.localize('L{weekends}')
        };
      }
      /**
       * Get the specified part of a date.
       * @param {Date} date
       * @param {String} unit Part of date, hour, minute etc.
       * @returns {Number} The requested part of the specified date
       * @category Query
       */
      static get(date, unit) {
        switch (DH$2.normalizeUnit(unit)) {
          case 'millisecond':
            return date.getMilliseconds();
          case 'second':
            return date.getSeconds();
          case 'minute':
            return date.getMinutes();
          case 'hour':
            return date.getHours();
          case 'date':
          case 'day':
            // Scheduler has a lot of calculations expecting this to work
            return date.getDate();
          case 'week':
            return formats.W(date);
          case 'month':
            return date.getMonth();
          case 'quarter':
            return Math.floor(date.getMonth() / 3) + 1;
          case 'year':
            return date.getFullYear();
        }
        return null;
      }
      /**
       * Get number of days in the current year for the supplied date.
       * @param {Date} date Date to check
       * @returns {Number} Days in year
       * @category Query
       * @internal
       */
      static daysInYear(date) {
        const fullYear = date.getFullYear(),
          duration = new Date(fullYear + 1, 0, 1) - new Date(fullYear, 0, 1);
        return this.as('day', duration);
      }
      /**
       * Get number of days in the current month for the supplied date.
       * @param {Date} date Date which month should be checked
       * @returns {Number} Days in month
       * @category Query
       */
      static daysInMonth(date) {
        return 32 - new Date(date.getFullYear(), date.getMonth(), 32).getDate();
      }
      /**
       * Get number of hours in the current day for the supplied date.
       * @param {Date} date Date to check
       * @returns {Number} Hours in day
       * @category Query
       * @internal
       */
      static hoursInDay(date) {
        const fullYear = date.getFullYear(),
          month = date.getMonth(),
          day = date.getDate(),
          duration = new Date(fullYear, month, day + 1) - new Date(fullYear, month, day);
        return this.as('hour', duration);
      }
      /**
       * Converts unit related to the date to actual amount of milliseconds in it. Takes into account leap years and
       * different duration of months.
       * @param {Date} date Date
       * @param {String} unit Time unit
       * @returns {Number} Returns amount of milliseconds
       * @internal
       */
      static getNormalizedUnitDuration(date, unit) {
        let result;
        switch (unit) {
          case 'month':
            result = DH$2.asMilliseconds(DH$2.daysInMonth(date), 'day');
            break;
          case 'year':
            result = DH$2.asMilliseconds(DH$2.daysInYear(date), 'day');
            break;
          case 'day':
            result = DH$2.asMilliseconds(DH$2.hoursInDay(date), 'hour');
            break;
          default:
            result = DH$2.asMilliseconds(unit);
        }
        return result;
      }
      /**
       * Get the first date of the month for the supplied date.
       * @param {Date} date Date
       * @returns {Date} New Date instance
       * @category Query
       */
      static getFirstDateOfMonth(date) {
        return new Date(date.getFullYear(), date.getMonth(), 1);
      }
      /**
       * Get the last date of the month for the supplied date.
       * @param {Date} date Date
       * @returns {Date} New Date instance
       * @category Query
       */
      static getLastDateOfMonth(date) {
        return new Date(date.getFullYear(), date.getMonth() + 1, 0);
      }
      /**
       * Get the earliest of two dates.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @returns {Date} Earliest date
       * @category Query
       */
      static min(first, second) {
        return first.getTime() < second.getTime() ? first : second;
      }
      /**
       * Get the latest of two dates.
       * @param {Date} first First date
       * @param {Date} second Second date
       * @returns {Date} Latest date
       * @category Query
       */
      static max(first, second) {
        return first.getTime() > second.getTime() ? first : second;
      }
      /**
       * Get an incremented date. Incrementation based on specified unit and optional amount.
       * @param {Date} date Date
       * @param {String} unit Time unit
       * @param {Number} [increment=1] Increment amount
       * @param {Number} [weekStartDay] Will default to what is set in locale
       * @returns {Date} New Date instance
       * @category Query
       */
      static getNext(date, unit, increment = 1, weekStartDay = DH$2.weekStartDay) {
        if (unit === 'week') {
          const dt = DH$2.clone(date),
            day = dt.getDay();
          DH$2.startOf(dt, 'day', false);
          DH$2.add(dt, weekStartDay - day + 7 * (increment - (weekStartDay <= day ? 0 : 1)), 'day', false);
          // For south american timezones, midnight does not exist on DST transitions, adjust...
          if (dt.getDay() !== weekStartDay) {
            DH$2.add(dt, 1, 'hour');
          }
          return dt;
        }
        return DH$2.startOf(DH$2.add(date, increment, unit), unit, false);
      }
      /**
       * Checks if date object is valid.
       *
       * For example:
       *
       * ```javascript
       * date = new Date('foo')
       * date instanceof Date // true
       * date.toString() // Invalid Date
       * isNaN(date) // true
       * DateHelper.isValidDate(date) // false
       *
       * date = new Date()
       * date instanceof Date // true
       * date.toString() // Mon Jan 13 2020 18:27:38 GMT+0300 (GMT+03:00)
       * isNaN(date) // false
       * DateHelper.isValidDate(date) // true
       * ```
       *
       * @param {Date} date Date
       * @returns {Boolean} `true` if date object is valid
       */
      static isValidDate(date) {
        return DH$2.isDate(date) && !isNaN(date);
      }
      /**
       * Checks if value is a date object. Allows to recognize date object even from another context,
       * like the top frame when used in an iframe.
       *
       * @param {*} value Value to check
       * @returns {Boolean} `true` if value is a date object
       */
      static isDate(value) {
        // see https://jsbench.me/s7kb49w83j/1 (cannot use instanceof cross-frame):
        return value && toString$1.call(value) === DATE_TYPE;
      }
      /**
       * Get the start of the next day.
       * @param {Date} date Date
       * @param {Boolean} [clone=false] Clone date
       * @param {Boolean} [noNeedToClearTime=false] Flag to not clear time from the result
       * @returns {Date} Passed Date or new Date instance, depending on the `clone` flag
       * @category Query
       */
      static getStartOfNextDay(date, clone = false, noNeedToClearTime = false) {
        let nextDay = DH$2.add(noNeedToClearTime ? date : DH$2.clearTime(date, clone), 1, 'day');
        // DST case
        if (nextDay.getDate() === date.getDate()) {
          const offsetNextDay = DH$2.add(DH$2.clearTime(date, clone), 2, 'day').getTimezoneOffset(),
            offsetDate = date.getTimezoneOffset();
          nextDay = DH$2.add(nextDay, offsetDate - offsetNextDay, 'minute');
        }
        return nextDay;
      }
      /**
       * Get the end of previous day.
       * @param {Date} date Date
       * @param {Boolean} [noNeedToClearTime=false] Flag to not clear time from the result
       * @returns {Date} New Date instance
       * @category Query
       */
      static getEndOfPreviousDay(date, noNeedToClearTime = false) {
        const dateOnly = noNeedToClearTime ? date : DH$2.clearTime(date, true);
        // dates are different
        if (dateOnly - date) {
          return dateOnly;
        } else {
          return DH$2.add(dateOnly, -1, 'day');
        }
      }
      /**
       * Returns a string describing the specified week. For example, `'39, September 2020'` or `'40, Sep - Oct 2020'`.
       * @param {Date} startDate Start date
       * @param {Date} [endDate] End date
       * @returns {String} String describing the specified week
       * @internal
       */
      static getWeekDescription(startDate, endDate = startDate) {
        const monthDesc = startDate.getMonth() === endDate.getMonth() ? DateHelper.format(startDate, 'MMMM') : `${DateHelper.format(startDate, 'MMM')} - ${DateHelper.format(endDate, 'MMM')}`,
          week = DateHelper.getWeekNumber(startDate);
        return `${week[1]}, ${monthDesc} ${week[0]}`;
      }
      /**
       * Get week number for the date.
       * @param {Date} date The date
       * @param {Number} [weekStartDay] The first day of week, 0-6 (Sunday-Saturday). Defaults to the {@link #property-weekStartDay-static}
       * @returns {Number[]} year and week number
       * @category Query
       */
      static getWeekNumber(date, weekStartDay = DateHelper.weekStartDay) {
        const jan01 = new Date(date.getFullYear(), 0, 1),
          dec31 = new Date(date.getFullYear(), 11, 31),
          firstDay = normalizeDay(jan01.getDay() - weekStartDay),
          lastDay = normalizeDay(dec31.getDay() - weekStartDay),
          dayNumber = getDayDiff(date, jan01);
        let weekNumber;
        // Check if the year starts before the middle of a week
        if (firstDay < 4) {
          weekNumber = Math.floor((dayNumber + firstDay - 1) / 7) + 1;
        } else {
          weekNumber = Math.floor((dayNumber + firstDay - 1) / 7);
        }
        if (weekNumber) {
          let year = date.getFullYear();
          // Might be week 1 of next year if the year ends before day 3 (0 based)
          if (weekNumber === 53 && lastDay < 3) {
            year++;
            weekNumber = 1;
          }
          return [year, weekNumber];
        }
        // We're in week zero which is the last week of the previous year, so ask what
        // week encapsulates 31 Dec in the previous year.
        const lastWeekOfLastYear = DateHelper.getWeekNumber(new Date(date.getFullYear() - 1, 11, 31))[1];
        return [date.getFullYear() - 1, lastWeekOfLastYear];
      }
      //endregion
      //region Unit helpers
      /**
       * Turns `(10, 'day')` into `'10 days'` etc.
       * @param {Number} count Amount of unit
       * @param {String} unit Unit, will be normalized (days, d -> day etc.)
       * @returns {String} Amount formatted to string
       * @category Unit helpers
       */
      static formatCount(count, unit) {
        unit = DH$2.normalizeUnit(unit);
        if (count !== 1) unit += 's';
        return count + ' ' + unit;
      }
      /**
       * Get the ratio between two units ( year, month -> 1/12 ).
       * @param {String} baseUnit Base time unit
       * @param {String} unit Time unit
       * @param {Boolean} [acceptEstimate=false] If `true`, process negative values of validConversions
       * @returns {Number} Ratio
       * @category Unit helpers
       */
      static getUnitToBaseUnitRatio(baseUnit, unit, acceptEstimate = false) {
        baseUnit = DH$2.normalizeUnit(baseUnit);
        unit = DH$2.normalizeUnit(unit);
        if (baseUnit === unit) return 1;
        // Some validConversions have negative sign to signal that it is not an exact conversion.
        // Ignore those here unless acceptEstimate is provided.
        if (validConversions[baseUnit] && validConversions[baseUnit][unit] && (acceptEstimate || validConversions[baseUnit][unit] > 0)) {
          return 1 / DH$2.as(unit, 1, baseUnit);
        }
        if (validConversions[unit] && validConversions[unit][baseUnit] && (acceptEstimate || validConversions[unit][baseUnit] > 0)) {
          return DH$2.as(baseUnit, 1, unit);
        }
        return -1;
      }
      /**
       * Returns a localized abbreviated form of the name of the duration unit.
       * For example in the `EN` locale, for `'qrt'` it will return `'q'`.
       * @param {String} unit Duration unit
       * @returns {String} Localized abbreviated form of the name of the duration unit
       * @category Unit helpers
       */
      static getShortNameOfUnit(unit) {
        // Convert abbreviations to the canonical name.
        // See locale file and the applyLocale method below.
        unit = DH$2.parseTimeUnit(unit);
        // unitLookup is keyed by eg 'DAY', 'day', 'MILLISECOND', 'millisecond' etc
        return DH$2.unitLookup[unit].abbrev;
      }
      /**
       * Returns a localized full name of the duration unit.
       *
       * For example in the `EN` locale, for `'d'` it will return either
       * `'day'` or `'days'`, depending on the `plural` argument
       *
       * Preserves casing of first letter.
       *
       * @static
       * @param {String} unit Time unit
       * @param {Boolean} [plural=false] Whether to return a plural name or singular
       * @returns {String} Localized full name of the duration unit
       * @category Unit helpers
       */
      static getLocalizedNameOfUnit(unit, plural = false) {
        const capitalize = unit.charAt(0) === unit.charAt(0).toUpperCase();
        // Normalize to not have to have translations for each variation used in code
        unit = DH$2.normalizeUnit(unit);
        // Convert abbreviations to the canonical name.
        // See locale file and the applyLocale method below.
        unit = DH$2.parseTimeUnit(unit);
        // Translate
        // unitLookup is keyed by eg 'DAY', 'day', 'MILLISECOND', 'millisecond' etc
        unit = DH$2.unitLookup[unit][plural ? 'plural' : 'single'];
        // Preserve casing of first letter
        if (capitalize) {
          unit = StringHelper.capitalize(unit);
        }
        return unit;
      }
      /**
       * Normalizes a unit for easier usage in conditionals. For example `'year'`, `'years'`, `'y'` -> `'year'`.
       * @param {String} unit Time unit
       * @returns {String} Normalized unit name
       * @category Unit helpers
       */
      static normalizeUnit(unit) {
        if (!unit) {
          return null;
        }
        const unitLower = unit.toLowerCase();
        if (unitLower === 'date') {
          return unitLower;
        }
        return canonicalUnitNames.includes(unitLower)
        // Already valid
        ? unitLower
        // Trying specified case first, since we have both 'M' for month and 'm' for minute
        : normalizedUnits[unit] || normalizedUnits[unitLower];
      }
      static getUnitByName(name) {
        // Allow either a canonical name to be passed, or, if that fails, parse it as a localized name or abbreviation.
        return DH$2.normalizeUnit(name) || DH$2.normalizeUnit(DH$2.parseTimeUnit(name));
      }
      /**
       * Returns a duration of the timeframe in the given unit.
       * @param {Date} start The start date of the timeframe
       * @param {Date} end The end date of the timeframe
       * @param {String} unit Duration unit
       * @privateparam {Boolean} [doNotRound]
       * @returns {Number} The duration in the units
       * @category Unit helpers
       * @ignore
       */
      static getDurationInUnit(start, end, unit, doNotRound) {
        return DH$2.diff(start, end, unit, doNotRound);
      }
      /**
       * Checks if two date units align.
       * @private
       * @param {String} majorUnit Major time unit
       * @param {String} minorUnit Minor time unit
       * @returns {Boolean} `true` if two date units align
       * @category Unit helpers
       */
      static doesUnitsAlign(majorUnit, minorUnit) {
        return !(majorUnit !== minorUnit && minorUnit === 'week');
      }
      static getSmallerUnit(unit) {
        return canonicalUnitNames[unitMagnitudes[DH$2.normalizeUnit(unit)] - 1] || null;
      }
      static getLargerUnit(unit) {
        return canonicalUnitNames[unitMagnitudes[DH$2.normalizeUnit(unit)] + 1] || null;
      }
      /**
       *
       * Rounds the passed Date value to the nearest `increment` value.
       *
       * Optionally may round relative to a certain base time point.
       *
       * For example `DH.round(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
       * would round to 9:45 because that's the nearest integer number of 30 minute increments
       * from the base.
       *
       * Note that `base` is ignored when rounding to weeks. The configured {@link #property-weekStartDay-static}
       * dictates what the base of a week is.
       *
       * @param {Date} time The time to round
       * @param {String|Number} increment A millisecond value by which to round the time
       * May be specified in string form eg: `'15 minutes'`
       * @param {Date} [base] The start from which to apply the rounding
       * @param {Number} [weekStartDay] Will default to what is set in locale
       * @returns {Date} New Date instance
       */
      static round(time, increment, base, weekStartDay) {
        return DH$2.snap('round', time, increment, base, weekStartDay);
      }
      /**
       *
       * Floor the passed Date value to the nearest `increment` value.
       *
       * Optionally may floor relative to a certain base time point.
       *
       * For example `DH.floor(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
       * would floor to 9:15 because that's the closest lower integer number of 30 minute increments
       * from the base.
       *
       * Note that `base` is ignored when flooring to weeks. The configured {@link #property-weekStartDay-static}
       * dictates what the base of a week is.
       *
       * @param {Date} time The time to floor
       * @param {String|Number|DurationConfig|Object} increment A numeric millisecond value by which to floor the time.
       * or a duration in string form eg `'30 min'` or object form : `{unit: 'minute', magnitude: 30}`
       * or `{unit: 'minute', increment: 30}`
       * @param {Date} [base] The start from which to apply the flooring
       * @param {Number} [weekStartDay] Will default to what is set in locale
       * @returns {Date} New Date instance
       */
      static floor(time, increment, base, weekStartDay) {
        return DH$2.snap('floor', time, increment, base, weekStartDay);
      }
      /**
       *
       * Ceils the passed Date value to the nearest `increment` value.
       *
       * Optionally may ceil relative to a certain base time point.
       *
       * For example `DH.ceil(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
       * would ceil to 9:45 because that's the closest higher integer number of 30 minute increments
       * from the base.
       *
       * Note that `base` is ignored when ceiling to weeks. Use weekStartDay argument which default to the configured
       * {@link #property-weekStartDay-static} dictates what the base of a week is
       *
       * @param {Date} time The time to ceil
       * @param {String|Number|DurationConfig|Object} increment A numeric millisecond value by which to ceil the time
       * or a duration in string form eg `'30 min'` or object form : `{unit: 'minute', magnitude: 30}`
       * or `{unit: 'minute', increment: 30}`
       * @param {Date} [base] The start from which to apply the ceiling
       * @param {Number} [weekStartDay] Will default to what is set in locale
       * @returns {Date} New Date instance
       */
      static ceil(time, increment, base, weekStartDay) {
        return DH$2.snap('ceil', time, increment, base, weekStartDay);
      }
      /**
       * Implementation for round, floor and ceil.
       * @internal
       */
      static snap(operation, time, increment, base, weekStartDay = DH$2.weekStartDay) {
        const snapFn = snapFns[operation];
        if (typeof increment === 'string') {
          increment = DH$2.parseDuration(increment);
        }
        if (Objects.isObject(increment)) {
          // Allow {unit: 'minute', increment: 30} or {unit: 'minute', magnitude: 30}
          // parseDuration produces 'magnitude'. The Scheduler's TimeAxis uses 'increment'
          // in its resolution object, so we allow that too.
          const magnitude = increment.magnitude || increment.increment;
          // increment is in weeks, months, quarters or years, then it can't be handled
          // using millisecond arithmetic.
          switch (increment.unit) {
            case 'week':
              {
                const weekDay = time.getDay();
                // weekStartDay gives our base
                // Our base is the start of the week
                base = DH$2.add(DH$2.clearTime(time), weekDay >= weekStartDay ? weekStartDay - weekDay : -(weekDay - weekStartDay + 7), 'day');
                return DH$2[operation](time, `${magnitude * 7} days`, base);
              }
            case 'month':
              {
                // Express the time as a number of months from epoch start.
                // May be a fraction, eg the 15th will be 0.5 through a month.
                time = DH$2.asMonths(time);
                let resultMonths;
                // Snap the month count in the way requested
                if (base) {
                  base = DH$2.asMonths(base);
                  resultMonths = time + snapFn(time - base, magnitude);
                } else {
                  resultMonths = snapFn(time, magnitude);
                }
                // Convert resulting month value back to a date
                return DH$2.monthsToDate(resultMonths);
              }
            case 'quarter':
              return DH$2[operation](time, `${magnitude * 3} months`, base);
            case 'year':
              return DH$2[operation](time, `${magnitude * 12} months`, base);
            case 'decade':
              // We assume that decades begin with a year divisible by 10
              return DH$2[operation](time, `${magnitude * 10} years`, base);
          }
          // Convert to a millisecond value
          increment = DH$2.as('ms', magnitude, increment.unit);
        }
        // It's a simple round to milliseconds
        if (base) {
          const tzChange = DH$2.as('ms', base.getTimezoneOffset() - time.getTimezoneOffset(), 'ms');
          return new Date(base.valueOf() + snapFn(DH$2.diff(base, time, 'ms') + tzChange, increment));
        } else {
          const offset = time.getTimezoneOffset() * 60 * 1000;
          // Assuming current TZ is GMT+3
          // new Date(2000, 0, 1) / 86400000      -> 10956.875
          // new Date(2000, 0, 1, 3) / 86400000   -> 10957
          // Before calculation we need to align time value of the current timezone to GMT+0
          // And after calculate we need to adjust time back
          return new Date(snapFn(time.valueOf() - offset, increment) + offset);
        }
      }
      //endregion
      //region Date picker format
      /**
       * Parses a typed duration value according to locale rules.
       *
       * The value is taken to be a string consisting of the numeric magnitude and the units:
       * - The numeric magnitude can be either an integer or a float value. Both `','` and `'.'` are valid decimal separators.
       * - The units may be a recognised unit abbreviation of this locale or the full local unit name.
       *
       * For example:
       * `'2d'`, `'2 d'`, `'2 day'`, `'2 days'` will be turned into `{ magnitude : 2, unit : 'day' }`
       * `'2.5d'`, `'2,5 d'`, `'2.5 day'`, `'2,5 days'` will be turned into `{ magnitude : 2.5, unit : 'day' }`
       *
       * **NOTE:** Doesn't work with complex values like `'2 days, 2 hours'`
       *
       * @param {String} value The value to parse
       * @param {Boolean} [allowDecimals=true] Decimals are allowed in the magnitude
       * @param {String} [defaultUnit] Default unit to use if only magnitude passed
       * @returns {DurationConfig} If successfully parsed, the result contains two properties, `magnitude` being a number, and
       * `unit` being the canonical unit name, *NOT* a localized name. If parsing was unsuccessful, `null` is returned
       * @category Parse & format
       */
      static parseDuration(value, allowDecimals = true, defaultUnit) {
        var _match$;
        const durationRegEx = allowDecimals ? withDecimalsDurationRegex : noDecimalsDurationRegex,
          match = durationRegEx.exec(value);
        if (value == null || !match) {
          return null;
        }
        const magnitude = parseNumber((_match$ = match[1]) === null || _match$ === void 0 ? void 0 : _match$.replace(',', '.')),
          unit = DH$2.parseTimeUnit(match[2]) || defaultUnit;
        if (!unit) {
          return null;
        }
        return {
          magnitude,
          unit
        };
      }
      /**
       * Parses a typed unit name, for example `'ms'` or `'hr'` or `'yr'` into the
       * canonical form of the unit name which may be passed to {@link #function-add-static}
       * or {@link #function-diff-static}.
       * @param {*} unitName Time unit name
       * @category Parse & format
       */
      static parseTimeUnit(unitName) {
        // NOTE: In case you get a crash here when running tests, it is caused by missing locale. Build locales
        // using `scripts/build.js locales` to resolve.
        const unitMatch = unitName == null ? null : DH$2.durationRegEx.exec(unitName.toLowerCase());
        if (!unitMatch) {
          return null;
        }
        // See which group in the unitAbbrRegEx matched match[2]
        for (let unitOrdinal = 0; unitOrdinal < canonicalUnitNames.length; unitOrdinal++) {
          if (unitMatch[unitOrdinal + 1]) {
            return canonicalUnitNames[unitOrdinal];
          }
        }
      }
      //endregion
      //region Internal
      static getGMTOffset(date = new Date()) {
        if (!date) {
          return;
        }
        const offsetInMinutes = date.getTimezoneOffset();
        // return 'Z' for UTC
        if (!offsetInMinutes) return 'Z';
        return (offsetInMinutes > 0 ? '-' : '+') + Math.abs(Math.trunc(offsetInMinutes / 60)).toString().padStart(2, '0') + ':' + Math.abs(offsetInMinutes % 60).toString().padStart(2, '0');
      }
      static fillDayNames() {
        const tempDate = new Date('2000-01-01T12:00:00'),
          dayNames = DH$2._dayNames || [],
          dayShortNames = DH$2._dayShortNames || [];
        dayNames.length = 0;
        dayShortNames.length = 0;
        for (let day = 2; day < 9; day++) {
          tempDate.setDate(day);
          dayNames.push(DH$2.format(tempDate, 'dddd'));
          dayShortNames.push(DH$2.format(tempDate, 'ddd'));
        }
        DH$2._dayNames = dayNames;
        DH$2._dayShortNames = dayShortNames;
      }
      static getDayNames() {
        return DH$2._dayNames;
      }
      static getDayName(day) {
        return DH$2._dayNames[day];
      }
      static getDayShortNames() {
        return DH$2._dayShortNames;
      }
      static getDayShortName(day) {
        return DH$2._dayShortNames[day];
      }
      static fillMonthNames() {
        const tempDate = new Date('2000-01-15T12:00:00'),
          monthNames = DH$2._monthNames || [],
          monthShortNames = DH$2._monthShortNames || [],
          monthNamesIndex = {},
          monthShortNamesIndex = {};
        monthNames.length = 0;
        monthShortNames.length = 0;
        for (let month = 0; month < 12; month++) {
          tempDate.setMonth(month);
          const monthName = DH$2.format(tempDate, 'MMMM');
          monthNames.push(monthName);
          const monthShortName = DH$2.format(tempDate, 'MMM');
          monthShortNames.push(monthShortName);
          monthNamesIndex[monthName.toLowerCase()] = {
            name: monthName,
            value: month
          };
          monthShortNamesIndex[monthShortName.toLowerCase()] = {
            name: monthShortName,
            value: month
          };
        }
        DH$2._monthNames = monthNames;
        DH$2._monthShortNames = monthShortNames;
        DH$2._monthNamesIndex = monthNamesIndex;
        DH$2._monthShortNamesIndex = monthShortNamesIndex;
      }
      static getMonthShortNames() {
        return DH$2._monthShortNames;
      }
      static getMonthShortName(month) {
        return DH$2._monthShortNames[month];
      }
      static getMonthNames() {
        return DH$2._monthNames;
      }
      static getMonthName(month) {
        return DH$2._monthNames[month];
      }
      static set locale(name) {
        locale$4 = name;
        intlFormatterCache = {};
        formatCache = {};
        formatRedirects = {};
      }
      static get locale() {
        return locale$4;
      }
      static setupDurationRegEx(unitNames = [], unitAbbreviations = []) {
        const me = this,
          unitLookup = {};
        let unitAbbrRegEx = '';
        for (let i = 0; i < unitAbbreviations.length; i++) {
          const
            // for example ['s', 'sec']
            abbreviations = unitAbbreviations[i],
            // for example { single : 'second', plural : 'seconds', abbrev : 's' }
            unitNamesCfg = unitNames[i];
          unitNamesCfg.canonicalUnitName = canonicalUnitNames[i];
          // Create a unitLookup object keyed by unit full names
          // both lower and upper case to be able to look up plurals or abbreviations
          // also always include english names, since those are used in sources
          unitLookup[unitNamesCfg.single] = unitLookup[unitNamesCfg.single.toUpperCase()] = unitLookup[unitNamesCfg.canonicalUnitName] = unitLookup[unitNamesCfg.canonicalUnitName.toUpperCase()] = unitNamesCfg;
          unitAbbrRegEx += `${i ? '|' : ''}(`;
          for (let j = 0; j < abbreviations.length; j++) {
            unitAbbrRegEx += `${abbreviations[j]}|`;
          }
          locale$4 = me.localize('L{locale}') || 'en-US';
          if (locale$4 !== 'en-US') {
            // Add canonical values to be able to parse durations specified in configs
            const canonicalAbbreviations = canonicalUnitAbbreviations[i];
            for (let j = 0; j < canonicalAbbreviations.length; j++) {
              unitAbbrRegEx += `${canonicalAbbreviations[j]}|`;
            }
          }
          unitAbbrRegEx += `${unitNamesCfg.single}|${unitNamesCfg.plural}|${unitNamesCfg.canonicalUnitName}|${unitNamesCfg.canonicalUnitName}s)`;
        }
        me.unitLookup = unitLookup;
        me.durationRegEx = new RegExp(`^(?:${unitAbbrRegEx})$`);
      }
      static applyLocale() {
        const me = this,
          unitAbbreviations = me.localize('L{unitAbbreviations}') || [],
          unitNames = me.unitNames = me.localize('L{unitNames}');
        // This happens when applying an incomplete locale, as done in Localizable.t.js.
        // Invalid usecase, but return to prevent a crash in that test.
        if (unitNames === 'unitNames') {
          return;
        }
        locale$4 = me.localize('L{locale}') || 'en-US';
        if (locale$4 === 'en-US') {
          ordinalSuffix = enOrdinalSuffix;
        } else {
          ordinalSuffix = me.localize('L{ordinalSuffix}') || ordinalSuffix;
        }
        formatCache = {};
        formatRedirects = {};
        parserCache = {};
        intlFormatterCache = {};
        DH$2._weekStartDay = null;
        DH$2.setupDurationRegEx(unitNames, unitAbbreviations);
        // rebuild day/month names cache
        DH$2.fillDayNames();
        DH$2.fillMonthNames();
      }
      //endregion
    }

    const DH$2 = DateHelper;
    DH$2.useIntlFormat = useIntlFormat; // to use on tests
    // Update when changing locale
    LocaleManagerSingleton.ion({
      locale: 'applyLocale',
      prio: 1000,
      thisObj: DH$2
    });
    // Apply default locale
    if (LocaleManagerSingleton.locale) {
      DH$2.applyLocale();
    }
    DateHelper._$name = 'DateHelper';

    /**
     * @module Core/helper/ObjectHelper
     */
    // Detect if browser has bad implementation of toFixed()
    const {
        hasOwn: hasOwn$3
      } = Objects,
      toFixedFix = 1.005.toFixed(2) === '1.01' ? null : function (number, fractionDigits) {
        const split = number.toString().split('.'),
          newNumber = +(!split[1] ? split[0] : split.join('.') + '1');
        return number.toFixed.call(newNumber, fractionDigits);
      };
    /**
     * Helper for Object manipulation.
     */
    class ObjectHelper extends Objects {
      // These methods are inherited from Objects (an internal class) but need to be documented here for public use.
      // This is primarily because static methods, while inherited by JavaScript classes, are not displayed in derived
      // classes in the docs.
      /**
       * Copies all enumerable properties from the supplied source objects to `dest`. Unlike `Object.assign`, this copy
       * also includes inherited properties.
       * @param {Object} dest The destination object.
       * @param {...Object} sources The source objects.
       * @returns {Object} The `dest` object.
       * @method assign
       * @static
       */
      /**
       * Copies all enumerable properties from the supplied source objects to `dest`, only including properties that does
       * not already exist on `dest`. Unlike `Object.assign`, this copy also includes inherited properties.
       * @param {Object} dest The destination object.
       * @param {...Object} sources The source objects.
       * @returns {Object} The `dest` object.
       * @method assignIf
       * @static
       */
      /**
       * Creates a deep copy of the `value`. Simple objects ({@link #function-isObject-static}, arrays and `Date` objects
       * are cloned. The enumerable properties of simple objects and the elements of arrays are cloned recursively.
       * @param {*} value The value to clone.
       * @param {Function} [handler] An optional function to call for values of types other than simple object, array or
       * `Date`. This function should return the clone of the `value` passed to it. It is only called for truthy values
       * whose `typeof` equals `'object'`.
       * @param {*} handler.value The value to clone.
       * @returns {*} The cloned value.
       * @method clone
       * @static
       */
      /**
       * Converts a list of names (either a space separated string or an array), into an object with those properties
       * assigned truthy values. The converse of {@link #function-getTruthyKeys-static}.
       * @param {String|String[]} source The list of names to convert to object form.
       * @method createTruthyKeys
       * @static
       */
      /**
       * Gathers the names of properties which have truthy values into an array.
       *
       * This is useful when gathering CSS class names for complex element production.
       * Instead of appending to an array or string which may already contain the
       * name, and instead of contending with space separation and concatenation
       * and conditional execution, just set the properties of an object:
       *
       *     cls = {
       *         [this.selectedCls] : this.isSelected(thing),
       *         [this.dirtyCls] : this.isDirty(thing)
       *     };
       *
       * @param {Object} source Source of keys to gather into an array.
       * @returns {String[]} The keys which had a truthy value.
       * @method getTruthyKeys
       * @static
       */
      /**
       * Gathers the values of properties which are truthy into an array.
       * @param {Object} source Source of values to gather into an array.
       * @returns {String[]} The truthy values from the passed object.
       * @method getTruthyValues
       * @static
       */
      /**
       * Tests whether a passed object has any enumerable properties.
       * @param {Object} object
       * @returns {Boolean} `true` if the passed object has no enumerable properties.
       * @method isEmpty
       * @static
       */
      /**
       * Returns `true` if the `value` is a simple `Object`.
       * @param {Object} value
       * @returns {Boolean} `true` if the `value` is a simple `Object`.
       * @method isObject
       * @static
       */
      /**
       * Copies all enumerable properties from the supplied source objects to `dest`, recursing when the properties of
       * both the source and `dest` are objects.
       * ```
       *  const o = {
       *      a : 1,
       *      b : {
       *          c : 2
       *      }
       *  };
       *  const o2 = {
       *      b : {
       *          d : 3
       *      }
       *  }
       *
       *  console.log(merge(o, o2));
       *
       *  > { a : 1, b : { c : 2, d : 3 } }
       * ```
       * @param {Object} dest The destination object.
       * @param {...Object} sources The source objects.
       * @returns {Object} The `dest` object.
       * @method merge
       * @static
       */
      /**
       * Returns the specific type of the given `value`. Unlike the `typeof` operator, this function returns the text
       * from the `Object.prototype.toString` result allowing `Date`, `Array`, `RegExp`, and others to be differentiated.
       * ```
       *  console.log(typeOf(null));
       *  > null
       *
       *  console.log(typeOf({}));
       *  > object
       *
       *  console.log(typeOf([]));
       *  > array
       *
       *  console.log(typeOf(new Date()));
       *  > date
       *
       *  console.log(typeOf(NaN));
       *  > nan
       *
       *  console.log(typeOf(/a/));
       *  > regexp
       * ```
       * @param {*} value
       * @returns {String}
       * @method typeOf
       * @static
       */
      /**
       * Returns value for a given path in the object
       * @param {Object} object Object to check path on
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @returns {*} Value associated with passed key
       * @method getPath
       * @static
       */
      /**
       * Sets value for a given path in the object
       * @param {Object} object Target object
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @param {*} value Value for a given path
       * @returns {Object} Returns passed object
       * @method setPath
       * @static
       */
      /**
       * Creates a new object where key is a property in array item (`ref` by default) or index in the array and value is array item.
       *
       * From:
       * ```
       * [
       *     {
       *          text : 'foo',
       *          ref : 'fooItem'
       *     },
       *     {
       *          text : 'bar'
       *     }
       * ]
       * ```
       *
       * To:
       * ```
       * {
       *     fooItem : {
       *         text : 'foo',
       *         ref  : 'fooItem'
       *     },
       *     1 : {
       *         text : 'bar'
       *     }
       * }
       * ```
       *
       * @param {Object[]} arrayOfItems Array to transform.
       * @param {String} [prop] Property to read the key from. `ref` by default.
       * @returns {Object} namedItems
       */
      static transformArrayToNamedObject(arrayOfItems, prop = 'ref') {
        const namedItems = {};
        arrayOfItems.forEach((item, index) => {
          const
          // 0 is valid value, but empty string in not valid
          key = item[prop] != null && item[prop].toString().length ? item[prop] : index;
          namedItems[key] = item;
        });
        return namedItems;
      }
      /**
       * Creates a new array from object values and saves key in a property (`ref` by default) of each item.
       *
       * From:
       * ```
       * {
       *     fooItem : {
       *         text : 'foo'
       *     },
       *     1 : {
       *         text : 'bar'
       *     },
       *     barItem : false // will be ignored
       * }
       * ```
       *
       * To:
       * ```
       * [
       *     {
       *          text : 'foo',
       *          ref : 'fooItem'
       *     },
       *     {
       *          text : 'bar',
       *          ref : 1
       *     }
       * ]
       * ```
       *
       * @param {Object} namedItems Object to transform.
       * @param {String} [prop] Property to save the key to. `ref` by default.
       * @returns {Object[]} arrayOfItems
       */
      static transformNamedObjectToArray(namedItems, prop = 'ref') {
        return Object.keys(namedItems).filter(key => namedItems[key]).map(key => {
          const item = namedItems[key];
          item[prop] = key;
          return item;
        });
      }
      /**
       * Checks if two values are equal. Basically === but special handling of dates.
       * @param {*} a First value
       * @param {*} b Second value
       * @returns {*} true if values are equal, otherwise false
       */
      static isEqual(a, b, useIsDeeply = false) {
        // Eliminate null vs undefined mismatch
        if (a === null && b !== null || a === undefined && b !== undefined || b === null && a !== null || b === undefined && a !== undefined) {
          return false;
        }
        // Covers undefined === undefined and null === null, since mismatches are eliminated above
        if (a == null && b == null) {
          return true;
        }
        // The same instance should equal itself.
        if (a === b) {
          return true;
        }
        const typeA = typeof a,
          typeB = typeof b;
        if (typeA === typeB) {
          switch (typeA) {
            case 'number':
            case 'string':
            case 'boolean':
              return a === b;
          }
          switch (true) {
            case a instanceof Date && b instanceof Date:
              // faster than calling DateHelper.isEqual
              // https://jsbench.me/3jk2bom2r3/1
              return a.getTime() === b.getTime();
            case Array.isArray(a) && Array.isArray(b):
              return a.length === b.length ? a.every((v, idx) => OH.isEqual(v, b[idx], useIsDeeply)) : false;
            case typeA === 'object' && a.constructor.prototype === b.constructor.prototype:
              return useIsDeeply ? OH.isDeeplyEqual(a, b, useIsDeeply) : StringHelper.safeJsonStringify(a) === StringHelper.safeJsonStringify(b);
          }
        }
        return String(a) === String(b);
      }
      /**
       * Checks if two objects are deeply equal
       * @param {Object} a
       * @param {Object} b
       * @param {Object} [options] Additional comparison options
       * @param {Object} [options.ignore] Map of property names to ignore when comparing
       * @param {Function} [options.shouldEvaluate] Function used to evaluate if a property should be compared or not.
       * Return false to prevent comparison
       * @param {Function} [options.evaluate] Function used to evaluate equality. Return `true`/`false` as evaluation
       * result or anything else to let `isEqual` handle the comparison
       * @returns {Boolean}
       */
      static isDeeplyEqual(a, b, options = {}) {
        // Same object, equal :)
        if (a === b) {
          return true;
        }
        // Nothing to compare, not equal
        if (!a || !b) {
          return false;
        }
        // Property names excluding ignored
        const aKeys = OH.keys(a, options.ignore),
          bKeys = OH.keys(b, options.ignore);
        // Property count differs, not equal
        if (aKeys.length !== bKeys.length) {
          return false;
        }
        for (let i = 0; i < aKeys.length; i++) {
          const aKey = aKeys[i],
            bKey = bKeys[i];
          // Property name differs, not equal
          if (aKey !== bKey) {
            return false;
          }
          const aVal = a[aKey],
            bVal = b[bKey];
          // Allow caller to determine if property values should be evaluated or not
          if (options.shouldEvaluate) {
            if (options.shouldEvaluate(aKey, {
              value: aVal,
              object: a
            }, {
              value: bVal,
              object: b
            }) === false) {
              continue;
            }
          }
          // Allow caller to determine equality of properties
          if (options.evaluate) {
            const result = options.evaluate(aKey, {
              value: aVal,
              object: a
            }, {
              value: bVal,
              object: b
            });
            // Not equal
            if (result === false) {
              return false;
            }
            // Equal, skip isEqual call below
            if (result === true) {
              continue;
            }
          }
          // Values differ, not equal (also digs deeper)
          if (!OH.isEqual(aVal, bVal, options)) {
            return false;
          }
        }
        // Found to be equal
        return true;
      }
      /**
       * Checks if value B is partially equal to value A.
       * @param {*} a First value
       * @param {*} b Second value
       * @returns {Boolean} true if values are partially equal, false otherwise
       */
      static isPartial(a, b) {
        a = String(a).toLowerCase();
        b = String(b).toLowerCase();
        return a.indexOf(b) !== -1;
      }
      /**
       * Checks if value a is smaller than value b.
       * @param {*} a First value
       * @param {*} b Second value
       * @returns {Boolean} true if a < b
       */
      static isLessThan(a, b) {
        if (a instanceof Date && b instanceof Date) {
          return DateHelper.isBefore(a, b);
        }
        return a < b;
      }
      /**
       * Checks if value a is bigger than value b.
       * @param {*} a First value
       * @param {*} b Second value
       * @returns {Boolean} true if a > b
       */
      static isMoreThan(a, b) {
        if (a instanceof Date && b instanceof Date) {
          return DateHelper.isAfter(a, b);
        }
        return a > b;
      }
      /**
       * Used by the Base class to make deep copies of defaultConfig blocks
       * @private
       */
      static fork(obj) {
        let ret, key, value;
        if (obj && obj.constructor === Object) {
          ret = Object.setPrototypeOf({}, obj);
          for (key in obj) {
            value = obj[key];
            if (value) {
              if (value.constructor === Object) {
                ret[key] = OH.fork(value);
              } else if (value instanceof Array) {
                ret[key] = value.slice();
              }
            }
          }
        } else {
          ret = obj;
        }
        return ret;
      }
      /**
       * Copies the named properties from the `source` parameter into the `dest` parameter.
       * @param {Object} dest The destination into which properties are copied.
       * @param {Object} source The source from which properties are copied.
       * @param {String[]} props The list of property names.
       * @returns {Object} The `dest` object.
       */
      static copyProperties(dest, source, props) {
        let prop, i;
        for (i = 0; i < props.length; i++) {
          prop = props[i];
          if (prop in source) {
            dest[prop] = source[prop];
          }
        }
        return dest;
      }
      /**
       * Copies the named properties from the `source` parameter into the `dest` parameter
       * unless the property already exists in the `dest`.
       * @param {Object} dest The destination into which properties are copied.
       * @param {Object} source The source from which properties are copied.
       * @param {String[]} props The list of property names.
       * @returns {Object} The `dest` object.
       */
      static copyPropertiesIf(dest, source, props) {
        if (source) {
          for (const prop of props) {
            if (!(prop in dest) && prop in source) {
              dest[prop] = source[prop];
            }
          }
        }
        return dest;
      }
      /**
       * Returns an array containing the keys and values of all enumerable properties from every prototype level for the
       * object. If `object` is `null`, this method returns an empty array.
       * @param {Object} object Object from which to retrieve entries.
       * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
       * which returns `true` to ignore the item.
       * @returns {Array}
       * @internal
       */
      static entries(object, ignore) {
        const result = [],
          call = typeof ignore === 'function';
        if (object) {
          for (const p in object) {
            if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
              result.push([p, object[p]]);
            }
          }
        }
        return result;
      }
      /**
       * Populates an `object` with the provided `entries`.
       * @param {Array} entries The key/value pairs (2-element arrays).
       * @param {Object} [object={}] The object onto which to add `entries`.
       * @returns {Object} The passed `object` (by default, a newly created object).
       * @internal
       */
      static fromEntries(entries, object) {
        object = object || {};
        if (entries) {
          for (let i = 0; i < entries.length; ++i) {
            object[entries[i][0]] = entries[i][1];
          }
        }
        return object;
      }
      /**
       * Returns an array containing all enumerable property names from every prototype level for the object. If `object`
       * is `null`, this method returns an empty array.
       * @param {Object} object Object from which to retrieve property names.
       * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
       * which returns `true` to ignore the item.
       * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
       * function is stored in the returned array. It is called with the array element as the first parameter, and the
       * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
       * on).
       * @returns {String[]}
       */
      static keys(object, ignore, mapper) {
        const result = [],
          call = typeof ignore === 'function';
        if (object) {
          let index = 0;
          for (const p in object) {
            if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
              result.push(mapper ? mapper(p, index) : p);
              ++index;
            }
          }
        }
        return result;
      }
      /**
       * Returns an array containing the values of all enumerable properties from every prototype level for the object.
       * If `object` is `null`, this method returns an empty array.
       * @param {Object} object Object from which to retrieve values.
       * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
       * which returns `true` to ignore the item.
       * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
       * function is stored in the returned array. It is called with the array element as the first parameter, and the
       * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
       * on).
       * @returns {Array}
       * @internal
       */
      static values(object, ignore, mapper) {
        const result = [],
          call = typeof ignore === 'function';
        if (object) {
          let index = 0;
          for (const p in object) {
            if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
              result.push(mapper ? mapper(object[p], index) : object[p]);
              ++index;
            }
          }
        }
        return result;
      }
      //region Path
      /**
       * Checks if a given path exists in an object
       * @param {Object} object Object to check path on
       * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
       * @returns {Boolean} Returns `true` if path exists or `false` if it does not
       */
      static pathExists(object, path) {
        const properties = path.split('.');
        return properties.every(property => {
          if (!object || !(property in object)) {
            return false;
          }
          object = object[property];
          return true;
        });
      }
      /**
       * Creates a simple single level key-value object from complex deep object.
       * @param {Object} object Object to extract path and values from
       * @returns {Object} Key-value object where key is a path to the corresponding value
       * @internal
       *
       * ```javascript
       * // converts deep object
       * {
       *     foo : {
       *         bar : {
       *             test : 1
       *         }
       *     }
       * }
       * // into a single level object
       * {
       *     'foo.bar.test' : 1
       * }
       * ```
       */
      static pathifyKeys(object, fieldDataSourceMap) {
        const result = {};
        for (const key in object) {
          if (hasOwn$3(object, key)) {
            const field = fieldDataSourceMap === null || fieldDataSourceMap === void 0 ? void 0 : fieldDataSourceMap[key];
            // do not use path keys if `fieldDataSourceMap` is provided (for top level keys)
            const usesPathKeys = (field === null || field === void 0 ? void 0 : field.type) === 'object' || (field === null || field === void 0 ? void 0 : field.complexMapping) || !Boolean(fieldDataSourceMap);
            if (usesPathKeys && Array.isArray(object[key])) {
              result[key] = object[key].slice();
            } else if (usesPathKeys && object[key] instanceof Object) {
              const paths = this.pathifyKeys(object[key]);
              for (const path in paths) {
                result[`${key}.${path}`] = paths[path];
              }
            } else {
              result[key] = object[key];
            }
          }
        }
        return result;
      }
      /**
       * Removes value for a given path in the object. Doesn't cleanup empty objects.
       * @param {Object} object
       * @param {String} path Dot-separated path, e.g. `obj.child.someKey`
       * @internal
       */
      static deletePath(object, path) {
        path.split('.').reduce((result, key, index, array) => {
          if (result == null) {
            return null;
          }
          if (hasOwn$3(result, key)) {
            if (index === array.length - 1) {
              delete result[key];
            } else {
              return result[key];
            }
          }
        }, object);
      }
      //endregion
      static coerce(from, to) {
        const fromType = Objects.typeOf(from),
          toType = Objects.typeOf(to),
          isString = typeof from === 'string';
        if (fromType !== toType) {
          switch (toType) {
            case 'string':
              return String(from);
            case 'number':
              return Number(from);
            case 'boolean':
              // See http://ecma262-5.com/ELS5_HTML.htm#Section_11.9.3 as to why '0'.
              // TL;DR => ('0' == 0), so if given string '0', we must return boolean false.
              return isString && (!from || from === 'false' || from === '0') ? false : Boolean(from);
            case 'null':
              return isString && (!from || from === 'null') ? null : false;
            case 'undefined':
              return isString && (!from || from === 'undefined') ? undefined : false;
            case 'date':
              return isString && isNaN(from) ? DateHelper.parse(from) : Date(Number(from));
          }
        }
        return from;
      }
      static wrapProperty(object, propertyName, newGetter, newSetter, deep = true) {
        const newProperty = {};
        let proto = Object.getPrototypeOf(object),
          existingProperty = Object.getOwnPropertyDescriptor(proto, propertyName);
        while (!existingProperty && proto && deep) {
          proto = Object.getPrototypeOf(proto);
          if (proto) {
            existingProperty = Object.getOwnPropertyDescriptor(proto, propertyName);
          }
        }
        if (existingProperty) {
          if (existingProperty.set) {
            newProperty.set = v => {
              existingProperty.set.call(object, v);
              // Must invoke the getter in case "v" has been transformed.
              newSetter && newSetter.call(object, existingProperty.get.call(object));
            };
          } else {
            newProperty.set = newSetter;
          }
          if (existingProperty.get) {
            newProperty.get = () => {
              let result = existingProperty.get.call(object);
              if (newGetter) {
                result = newGetter.call(object, result);
              }
              return result;
            };
          } else {
            newProperty.get = newGetter;
          }
        } else {
          newProperty.set = v => {
            object[`_${propertyName}`] = v;
            newSetter && newSetter.call(object, v);
          };
          newProperty.get = () => {
            let result = object[`_${propertyName}`];
            if (newGetter) {
              result = newGetter.call(object, result);
            }
            return result;
          };
        }
        Object.defineProperty(object, propertyName, newProperty);
      }
      /**
       * Intercepts access to a `property` of a given `object`.
       *
       * ```javascript
       *      ObjectHelper.hookProperty(object, 'prop', class {
       *          get value() {
       *              return super.value;
       *          }
       *          set value(v) {
       *              super.value = v;
       *          }
       *      });
       * ```
       * The use of `super` allows the hook's getter and setter to invoke the object's existing get/set.
       *
       * @param {Object} object
       * @param {String} property
       * @param {Function} hook A `class` defining a `value` property getter and/or setter.
       * @returns {Function} A function that removes the hook when called.
       * @internal
       */
      static hookProperty(object, property, hook) {
        const desc = ObjectHelper.getPropertyDescriptor(hook.prototype, 'value'),
          existingDesc = ObjectHelper.getPropertyDescriptor(object, property),
          fieldName = `_${property}`,
          base = class {
            get value() {
              return existingDesc ? existingDesc.get.call(this) : this[fieldName];
            }
            set value(v) {
              if (existingDesc) {
                existingDesc.set.call(this, v);
              } else {
                this[fieldName] = v;
              }
            }
          },
          baseDesc = ObjectHelper.getPropertyDescriptor(base.prototype, 'value');
        Object.setPrototypeOf(hook.prototype, base.prototype); // direct super calls to our "base" implementation
        Object.defineProperty(object, property, {
          configurable: true,
          get: desc.get || baseDesc.get,
          set: desc.set || baseDesc.set
        });
        return () => delete object[property];
      }
      /**
       * Finds a property descriptor for the passed object from all inheritance levels.
       * @param {Object} object The Object whose property to find.
       * @param {String} propertyName The name of the property to find.
       * @returns {Object} An ECMA property descriptor is the property was found, otherwise `null`
       */
      static getPropertyDescriptor(object, propertyName) {
        let result = null;
        for (let o = object; o && !result && !hasOwn$3(o, 'isBase'); o = Object.getPrototypeOf(o)) {
          result = Object.getOwnPropertyDescriptor(o, propertyName);
        }
        return result;
      }
      /**
       * Changes the passed object and removes all null and undefined properties from it
       * @param {Object} object Target object
       * @param {Boolean} [keepNull] Pass true to only remove undefined properties
       * @returns {Object} Passed object
       */
      static cleanupProperties(object, keepNull = false) {
        Object.entries(object).forEach(([key, value]) => {
          if (keepNull) {
            value === undefined && delete object[key];
          } else {
            value == null && delete object[key];
          }
        });
        return object;
      }
      /**
       * Changes the passed object and removes all properties from it.
       * Used while mutating when need to keep reference to the object but replace its properties.
       * @param {Object} object Target object
       * @returns {Object} Passed object
       */
      static removeAllProperties(obj) {
        Object.keys(obj).forEach(key => delete obj[key]);
        return obj;
      }
      //region Assert type
      /**
       * Checks that the supplied value is of the specified type.Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} type Expected type
       * @param {String} name Name of the value, used in error message
       * @param {Boolean} [allowNull] Accept `null` without throwing
       */
      static assertType(value, type, name) {
        const valueType = Objects.typeOf(value);
        if (value != null && valueType !== type) {
          throw new Error(`Incorrect type "${valueType}" for ${name}, expected "${type}"`);
        }
      }
      /**
       * Checks that the supplied value is a plain object. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertObject(value, name) {
        OH.assertType(value, 'object', name);
      }
      /**
       * Checks that the supplied value is an instance of a Bryntum class. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertInstance(value, name) {
        OH.assertType(value, 'instance', name);
      }
      /**
       * Checks that the supplied value is a Bryntum class. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertClass(value, name) {
        OH.assertType(value, 'class', name);
      }
      /**
       * Checks that the supplied value is a function. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertFunction(value, name) {
        if (typeof value !== 'function' || value.isBase || value.$$name) {
          throw new Error(`Incorrect type for ${name}, got "${value}" (expected a function)`);
        }
      }
      /**
       * Checks that the supplied value is a number. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertNumber(value, name) {
        const asNumber = Number(value);
        if (typeof value !== 'number' || isNaN(asNumber)) {
          throw new Error(`Incorrect type for ${name}, got "${value}" (expected a Number)`);
        }
      }
      /**
       * Checks that the supplied value is a boolean. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertBoolean(value, name) {
        OH.assertType(value, 'boolean', name);
      }
      /**
       * Checks that the supplied value is a string. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertString(value, name) {
        OH.assertType(value, 'string', name);
      }
      /**
       * Checks that the supplied value is an array. Throws if it is not
       * @param {Object} value Value to check type of
       * @param {String} name Name of the value, used in error message
       */
      static assertArray(value, name) {
        OH.assertType(value, 'array', name);
      }
      //endregion
      /**
       * Number.toFixed(), with polyfill for browsers that needs it
       * @param {Number} number
       * @param {Number} digits
       * @returns {String} A fixed point string representation of the passed number.
       */
      static toFixed(number, digits) {
        if (toFixedFix) {
          return toFixedFix(number, digits);
        }
        return number.toFixed(digits);
      }
      /**
       * Round the passed number to closest passed step value.
       * @param {Number} number The number to round.
       * @param {Number} [step] The step value to round to.
       * @returns {Number} The number rounded to the closest step.
       */
      static roundTo(number, step = 1) {
        return Math.round(number / step) * step;
      }
      /**
       * Round the passed number to the passed number of decimals.
       * @param {Number} number The number to round.
       * @param {Number} digits The number of decimal places to round to.
       * @returns {Number} The number rounded to the passed number of decimal places.
       */
      static round(number, digits) {
        // Undefined or null means do not round. NOT round to no decimals.
        if (digits == null) {
          return number;
        }
        const factor = 10 ** digits;
        return Math.round(number * factor) / factor;
      }
      /**
       * Returns a non-null entry from a Map for a given key path. This enables a specified defaultValue to be added "just
       * in time" which is returned if the key is not already present.
       * @param {Map} map The Map to find the key in (and potentially add to).
       * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
       * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
       * @param {Object} [defaultValue] Optionally the value to insert if the key is not found.
       */
      static getMapPath(map, path, defaultValue) {
        const keyPath = Array.isArray(path) ? path : typeof path === 'string' ? path.split('.') : [path],
          simpleKey = keyPath.length === 1,
          topKey = keyPath[0],
          topValue = map.has(topKey) ? map.get(topKey) : map.set(topKey, simpleKey ? defaultValue : {}).get(topKey);
        // If it was a simple key, we are done.
        if (simpleKey) {
          return topValue;
        }
        // Go down the property path on the top Object, filling entries in until the leaf.
        return OH.getPathDefault(topValue, keyPath.slice(1), defaultValue);
      }
    }
    const OH = ObjectHelper;
    ObjectHelper._$name = 'ObjectHelper';

    /**
     * @module Core/helper/util/Rectangle
     */
    const allBorders = ['border-top-width', 'border-right-width', 'border-bottom-width', 'border-left-width'],
      allMargins = ['margin-top', 'margin-right', 'margin-bottom', 'margin-left'],
      allPaddings = ['padding-top', 'padding-right', 'padding-bottom', 'padding-left'],
      borderNames = {
        t: 'border-top-width',
        r: 'border-right-width',
        b: 'border-bottom-width',
        l: 'border-left-width'
      },
      paddingNames = {
        t: 'padding-top',
        r: 'padding-right',
        b: 'padding-bottom',
        l: 'padding-left'
      },
      zeroBased = Object.freeze({
        x: 0,
        y: 0
      }),
      alignSpecRe$1 = /^([trblc])(\d*)-([trblc])(\d*)$/i,
      alignPointRe = /^([trblc])(\d*)$/i,
      edgeNames = ['top', 'right', 'bottom', 'left'],
      edgeIndices = {
        t: 0,
        r: 1,
        b: 2,
        l: 3
      },
      defaultAlignments = ['b-t', 'l-r', 't-b', 'r-l'],
      edgeAligments = {
        bt: 1,
        tb: 1,
        lr: 2,
        rl: 2
      },
      zeroOffsets = Object.freeze([0, 0]),
      matchDimensions = ['width', 'height'],
      parseEdges = (top, right = top, bottom = top, left = right) => {
        return Array.isArray(top) ? parseEdges.apply(null, top) : [top, right, bottom, left];
      };
    // Parse a l0-r0 (That's how Menus align to their owning MenuItem) align spec.
    // If we are in an RTL env, then reverse the percentage values if we are
    // aligning horizontal edges.
    function parseAlign(alignSpec, rtl) {
      const parts = alignSpecRe$1.exec(alignSpec),
        myEdge = parts[1],
        targetEdge = parts[3],
        mO = parseInt(parts[2] || 50),
        tO = parseInt(parts[4] || 50),
        myOffset = rtl && !(edgeIndices[myEdge] & 1) ? 100 - mO : mO,
        targetOffset = rtl && !(edgeIndices[targetEdge] & 1) ? 100 - tO : tO,
        edgeAligned = edgeAligments[myEdge + targetEdge];
      // Comments assume the Menu's alignSpec of l0-r0 is used.
      return {
        myAlignmentPoint: myEdge + myOffset,
        // l0
        myEdge,
        // l
        myOffset,
        // 0
        targetAlignmentPoint: targetEdge + targetOffset,
        // r0
        targetEdge,
        // r
        targetOffset,
        // 0
        startZone: edgeIndices[targetEdge],
        // 1 - start trying zone 1 in TRBL order
        edgeAligned // Edge-to-edge align requested
      };
    }
    // Takes a result from the above function and flips edges for the axisLock config
    function flipAlign(align) {
      return `${edgeNames[(edgeIndices[align.myEdge] + 2) % 4][0]}${align.myOffset}-${edgeNames[(edgeIndices[align.targetEdge] + 2) % 4][0]}${align.targetOffset}`;
    }
    function createOffsets(offset) {
      if (offset == null) {
        return zeroOffsets;
      } else if (typeof offset === 'number') {
        return [offset, offset];
      }
      return offset;
    }
    /**
     * Encapsulates rectangular areas for comparison, intersection etc.
     *
     * Note that the `right` and `bottom` properties are *exclusive*.
     *
     */
    class Rectangle {
      // Class does not extend Base, so we need to define this
      get isRectangle() {
        return true;
      }
      /**
       * Returns the Rectangle in document based coordinates of the passed element.
       *
       * *Note:* If the element passed is the `document` or `window` the `window`'s
       * rectangle is returned which is always at `[0, 0]` and encompasses the
       * browser's entire document viewport.
       * @param {HTMLElement|Core.widget.Widget|Core.widget.Mask} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @param {Boolean} [ignorePageScroll=false] Use browser viewport based coordinates.
       * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
       */
      static from(element, relativeTo, ignorePageScroll) {
        var _element, _relativeTo;
        // Convenient in tests
        if (typeof element === 'string') {
          element = document.querySelector(element);
        }
        // If a shadowRoot or other type of document fragment passed, get closest Element
        else if (((_element = element) === null || _element === void 0 ? void 0 : _element.nodeType) === Node.DOCUMENT_FRAGMENT_NODE) {
          element = element.host || element.ownerDocument;
        }
        if (typeof relativeTo === 'string') {
          relativeTo = document.querySelector(relativeTo);
        }
        if (element == null || element.isRectangle) {
          return element;
        }
        element = element.element || element; // works for Widget and Mask
        if (ignorePageScroll === undefined && typeof relativeTo === 'boolean') {
          ignorePageScroll = relativeTo;
          relativeTo = null;
        }
        if (!((_relativeTo = relativeTo) !== null && _relativeTo !== void 0 && _relativeTo.isRectangle)) {
          if (relativeTo) {
            let {
              scrollLeft,
              scrollTop
            } = relativeTo;
            if (BrowserHelper.isSafari && relativeTo === document.body) {
              scrollLeft = scrollTop = 0;
            }
            relativeTo = Rectangle.from(relativeTo).translate(-scrollLeft, -scrollTop);
          } else {
            relativeTo = zeroBased;
          }
        }
        // Viewport is denoted by requesting window or document.
        // document.body may overflow the viewport, so this must not be evaluated as the viewport.
        const
          // If body is 0 height we should treat this case as a viewport
          isViewport = element === document || element === globalThis,
          isSFViewport = element === document.body && document.body.offsetHeight === 0,
          sfElRect = isSFViewport && element.getBoundingClientRect(),
          viewRect = isSFViewport
          // In Salesforce body and html have 0 height so to get correct viewport vertical size we have to use
          // scrollHeight on html element.
          ? new Rectangle(sfElRect.left, sfElRect.top, sfElRect.width, document.body.parentElement.scrollHeight) : isViewport ? new Rectangle(0, 0, globalThis.innerWidth, globalThis.innerHeight) : element.getBoundingClientRect(),
          scrollOffset = ignorePageScroll || isViewport ? [0, 0] : [globalThis.pageXOffset, globalThis.pageYOffset];
        return new Rectangle(viewRect.left + scrollOffset[0] - relativeTo.x, viewRect.top + scrollOffset[1] - relativeTo.y, viewRect.width, viewRect.height);
      }
      /**
       * Returns the Rectangle in viewport coordinates of the passed element.
       *
       * *Note:* If the element passed is the `document` or `window` the `window`'s rectangle is returned which is always
       * at `[0, 0]` and encompasses the browser's entire document viewport.
       * @param {HTMLElement} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @returns {Core.helper.util.Rectangle} The Rectangle in viewport based coordinates. Relative to the _relativeTo_
       * parameter if provided.
       */
      static fromScreen(element, relativeTo) {
        return Rectangle.from(element, relativeTo, /* ignorePageScroll = */true);
      }
      /**
       * Returns the inner Rectangle (within border) in document based coordinates of the passed element.
       * @param {HTMLElement} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
       * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
       */
      static inner(element, relativeTo, ignorePageScroll = false) {
        const result = this.from(element, relativeTo, ignorePageScroll);
        // Can only ask for the following styles if element is in the document.
        if (document.body.contains(element)) {
          const borders = DomHelper.getStyleValue(element, allBorders);
          result.x += parseFloat(borders[borderNames.l]);
          result.y += parseFloat(borders[borderNames.t]);
          result.right -= parseFloat(borders[borderNames.r]);
          result.bottom -= parseFloat(borders[borderNames.b]);
        }
        return result;
      }
      /**
       * Returns the content Rectangle (within border and padding) in document based coordinates of the passed element.
       * @param {HTMLElement} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
       * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
       */
      static content(element, relativeTo, ignorePageScroll = false) {
        const result = this.from(element, relativeTo, ignorePageScroll);
        // Can only ask for the following styles if element is in the document.
        if (document.body.contains(element)) {
          const borders = DomHelper.getStyleValue(element, allBorders),
            padding = DomHelper.getStyleValue(element, allPaddings);
          result.x += parseFloat(borders[borderNames.l]) + parseFloat(padding[paddingNames.l]);
          result.y += parseFloat(borders[borderNames.t]) + parseFloat(padding[paddingNames.t]);
          result.right -= parseFloat(borders[borderNames.r]) + parseFloat(padding[paddingNames.r]);
          result.bottom -= parseFloat(borders[borderNames.b]) + parseFloat(padding[paddingNames.b]);
        }
        return result;
      }
      /**
       * Returns the client Rectangle (within border and padding and scrollbars) in document based coordinates of the
       * passed element.
       * @param {HTMLElement} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
       * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
       */
      static client(element, relativeTo, ignorePageScroll = false) {
        const result = this.content(element, relativeTo, ignorePageScroll),
          scrollbarWidth = DomHelper.scrollBarWidth;
        let padding;
        if (scrollbarWidth) {
          // Capture width taken by any vertical scrollbar.
          // If there is a vertical scrollbar, shrink the box.
          if (element.scrollHeight > element.clientHeight && DomHelper.getStyleValue(element, 'overflow-y') !== 'hidden') {
            padding = parseFloat(DomHelper.getStyleValue(element, 'padding-right'));
            result.right += padding - Math.max(padding, scrollbarWidth);
          }
          // Capture height taken by any horizontal scrollbar.
          // If there is a horizontal scrollbar, shrink the box.
          if (element.scrollWidth > element.clientWidth && DomHelper.getStyleValue(element, 'overflow-x') !== 'hidden') {
            padding = parseFloat(DomHelper.getStyleValue(element, 'padding-bottom'));
            result.bottom += padding - Math.max(padding, scrollbarWidth);
          }
        }
        // The client region excluding any scrollbars.
        return result;
      }
      /**
       * Returns the outer Rectangle (including margin) in document based coordinates of the passed element.
       * @param {HTMLElement} element The element to calculate the Rectangle for.
       * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
       * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
       * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates.
       * Relative to the _relativeTo_ parameter if passed.
       * @internal
       */
      static outer(element, relativeTo, ignorePageScroll = false) {
        const result = this.from(element, relativeTo, ignorePageScroll);
        // Can only ask for the following styles if element is in the document.
        if (document.body.contains(element)) {
          const margin = DomHelper.getStyleValue(element, allMargins);
          result.x -= parseFloat(margin['margin-left']);
          result.y -= parseFloat(margin['margin-top']);
          result.right += parseFloat(margin['margin-right']);
          result.bottom += parseFloat(margin['margin-bottom']);
        }
        return result;
      }
      /**
       * Returns a new rectangle created as the union of all supplied rectangles.
       * @param {Core.helper.util.Rectangle[]} rectangles
       * @returns {Core.helper.util.Rectangle}
       */
      static union(...rectangles) {
        let {
            x,
            y,
            right,
            bottom
          } = rectangles[0],
          current;
        if (rectangles.length > 1) {
          for (let i = 1; i < rectangles.length; i++) {
            current = rectangles[i];
            if (current.x < x) {
              x = current.x;
            }
            if (current.y < y) {
              y = current.y;
            }
            if (current.right > right) {
              right = current.right;
            }
            if (current.bottom > bottom) {
              bottom = current.bottom;
            }
          }
        }
        return new Rectangle(x, y, right - x, bottom - y);
      }
      /**
       * Rounds this Rectangle to the pixel resolution of the current display or to the nearest
       * passed unit which defaults to the current display's [`devicePixelRatio`](https://developer.mozilla.org/en-US/docs/Web/API/Window/devicePixelRatio).
       * @param {Number} [devicePixelRatio] device pixel ratio which defaults to `window.devicePixelRatio`
       */
      roundPx(devicePixelRatio = globalThis.devicePixelRatio || 1) {
        const me = this;
        me._x = DomHelper.roundPx(me._x, devicePixelRatio);
        me._y = DomHelper.roundPx(me._y, devicePixelRatio);
        me._width = DomHelper.roundPx(me._width, devicePixelRatio);
        me._height = DomHelper.roundPx(me._height, devicePixelRatio);
        return me;
      }
      // This class doesn't extend Base and extending doesn't seem to be
      // the way to go. Instead we duplicate smallest piece of logic here
      static get $$name() {
        return hasOwnProperty.call(this, '$name') && this.$name ||
        // _$name is filled by webpack for every class (cls._$name = '...')
        hasOwnProperty.call(this, '_$name') && this._$name || this.name;
      }
      get $$name() {
        return this.constructor.$$name;
      }
      /**
       * Constructs a Rectangle
       * @param {Number} x The X coordinate
       * @param {Number} y The Y coordinate
       * @param {Number} width The width
       * @param {Number} height The height
       */
      constructor(x, y, width, height) {
        ObjectHelper.assertNumber(x, 'Rectangle.x');
        ObjectHelper.assertNumber(y, 'Rectangle.y');
        ObjectHelper.assertNumber(width, 'Rectangle.width');
        ObjectHelper.assertNumber(height, 'Rectangle.height');
        const me = this;
        // Normalize rectangle definitions with -ve offsets from their origin
        if (width < 0) {
          x += width;
          width = -width;
        }
        if (height < 0) {
          y += height;
          height = -height;
        }
        me._x = x;
        me._y = y;
        me._width = width;
        me._height = height;
      }
      /**
       * Creates a copy of this Rectangle.
       */
      clone() {
        const me = this,
          result = new Rectangle(me.x, me.y, me.width, me.height);
        result.isAlignRectangle = me.isAlignRectangle;
        result.minHeight = me.minHeight;
        result.minWidth = me.minWidth;
        return result;
      }
      /**
       * Returns `true` if this Rectangle wholly contains the passed rectangle.
       *
       * Note that a {@link Core.helper.util.Point} may be passed.
       * @param {Core.helper.util.Rectangle} other The Rectangle to test for containment within this Rectangle
       * @returns {Boolean} `true` if the other Rectangle is wholly contained within this Rectangle
       */
      contains(other) {
        const me = this;
        if (other.isRectangle) {
          return other._x >= me._x && other._y >= me._y && other.right <= me.right && other.bottom <= me.bottom;
        } else {
          return false;
        }
      }
      /**
       * Checks if this Rectangle intersects the passed Rectangle
       * @param {Core.helper.util.Rectangle} other The Rectangle to intersect with this.
       * @param {Boolean} [useBoolean] Specify `true` to return a boolean value instead of constructing a new Rectangle
       * @param {Boolean} [allowZeroDimensions] `true` to consider zero-width or zero-hight rectangles as intersecting if coordinates indicate the intersection
       * @returns {Core.helper.util.Rectangle|Boolean} Returns the intersection Rectangle or `false` if there is no intersection.
       */
      intersect(other, useBoolean = false, allowZeroDimensions = false) {
        const me = this,
          y = Math.max(me.y, other.y),
          r = Math.min(me.right, other.right),
          b = Math.min(me.bottom, other.bottom),
          x = Math.max(me.x, other.x),
          intersect = allowZeroDimensions ? b >= y && r >= x : b > y && r > x;
        if (intersect) {
          return useBoolean ? true : new Rectangle(x, y, r - x, b - y);
        } else {
          return false;
        }
      }
      equals(other, round = false) {
        const processor = round ? x => Math.round(x) : x => x;
        return other.isRectangle && processor(other.x) === processor(this.x) && processor(other.y) === processor(this.y) && processor(other.width) === processor(this.width) && processor(other.height) === processor(this.height);
      }
      /**
       * Translates this Rectangle by the passed vector. Size is maintained.
       * @param {Number} x The X translation vector.
       * @param {Number} y The Y translation vector.
       * @returns {Core.helper.util.Rectangle} This Rectangle;
       */
      translate(x, y) {
        this._x += x || 0;
        this._y += y || 0;
        return this;
      }
      /**
       * Moves this Rectangle to the passed `x`, `y` position. Size is maintained.
       * @param {Number} x The new X position.
       * @param {Number} y The new Y position.
       * @returns {Core.helper.util.Rectangle}  This Rectangle;
       */
      moveTo(x, y) {
        if (x != null) {
          this._x = x;
        }
        if (y != null) {
          this._y = y;
        }
        return this;
      }
      /**
       * Returns the vector which would translate this Rectangle (or Point) to the same position as the other Rectangle (or point)
       * @param {Core.helper.util.Rectangle|Core.helper.util.Point} other The Rectangle or Point to calculate the delta to.
       * @returns {Array} Returns a vector using format `[deltaX, deltaY]`
       * @internal
       */
      getDelta(other) {
        return [other.x - this.x, other.y - this.y];
      }
      /**
       * The center point of this rectangle.
       * @property {Core.helper.util.Point}
       */
      get center() {
        return new Rectangle.Point(this.x + this.width / 2, this.y + this.height / 2);
      }
      /**
       * Get/sets the X coordinate of the Rectangle. Note that this does *not* translate the
       * Rectangle. The requested {@link #property-width} will change.
       * @property {Number}
       */
      set x(x) {
        const xDelta = x - this._x;
        this._x = x;
        this._width -= xDelta;
      }
      get x() {
        return this._x;
      }
      get start() {
        return this.left;
      }
      /**
       * Alias for x. To match DOMRect.
       * @property {Number}
       */
      set left(x) {
        this.x = x;
      }
      get left() {
        return this.x;
      }
      /**
       * Alias for y. To match DOMRect.
       * @property {Number}
       */
      set top(y) {
        this.y = y;
      }
      get top() {
        return this.y;
      }
      /**
       * Get/sets the Y coordinate of the Rectangle. Note that this does *not* translate the
       * Rectangle. The requested {@link #property-height} will change.
       * @property {Number}
       */
      set y(y) {
        const yDelta = y - this._y;
        this._y = y;
        this._height -= yDelta;
      }
      get y() {
        return this._y;
      }
      /**
       * Get/sets the width of the Rectangle. Note that the requested {@link #property-right} will change.
       * @property {Number}
       */
      set width(width) {
        this._width = width;
      }
      get width() {
        return this._width;
      }
      /**
       * Get/sets the height of the Rectangle. Note that the requested {@link #property-bottom} will change.
       * @property {Number}
       */
      set height(height) {
        this._height = height;
      }
      get height() {
        return this._height;
      }
      /**
       * Get/sets the right edge of the Rectangle. Note that the requested {@link #property-width} will change.
       *
       * The right edge value is exclusive of the calculated rectangle width. So x=0 and right=10
       * means a width of 10.
       * @property {Number}
       */
      set right(right) {
        this._width = right - this._x;
      }
      get right() {
        return this._x + this._width;
      }
      get end() {
        return this.right;
      }
      /**
       * Get/sets the bottom edge of the Rectangle. Note that the requested {@link #property-height} will change.
       *
       * The bottom edge value is exclusive of the calculated rectangle height. So y=0 and bottom=10
       * means a height of 10.
       * @property {Number}
       */
      set bottom(bottom) {
        this._height = bottom - this._y;
      }
      get bottom() {
        return this._y + this._height;
      }
      getStart(rtl, horizontal = true) {
        if (horizontal) {
          return rtl ? this.right : this.left;
        }
        return this.top;
      }
      getEnd(rtl, horizontal = true) {
        if (horizontal) {
          return rtl ? this.left : this.right;
        }
        return this.bottom;
      }
      get area() {
        return this.width * this.height;
      }
      set minWidth(minWidth) {
        const me = this;
        if (isNaN(minWidth)) {
          me._minWidth = null;
        } else {
          me._minWidth = Number(minWidth);
          // If this is being used as an alignment calculation rectangle, minWidth has a different meaning.
          // It does not mean that the width has to be at least this value. It means that under constraint,
          // it is willing to shrink down to that value before falling back to another align position.
          if (!me.isAlignRectangle) {
            me.width = Math.max(me.width, me._minWidth);
          }
        }
      }
      get minWidth() {
        return this._minWidth;
      }
      set minHeight(minHeight) {
        const me = this;
        if (isNaN(minHeight)) {
          me._minHeight = null;
        } else {
          me._minHeight = Number(minHeight);
          // If this is being used as an alignment calculation rectangle, minHeight has a different meaning.
          // It does not mean that the height has to be at least this value. It means that under constraint,
          // it is willing to shrink down to that value before falling back to another align position.
          if (!me.isAlignRectangle) {
            me.height = Math.max(me.height, me._minHeight);
          }
        }
      }
      get minHeight() {
        return this._minHeight;
      }
      /**
       * Modifies the bounds of this Rectangle by the specified deltas.
       * @param {Number} x How much to *add* to the x position.
       * @param {Number} y  How much to *add* to the y position.
       * @param {Number} width  How much to add to the width.
       * @param {Number} height  How much to add to the height.
       * @returns {Core.helper.util.Rectangle} This Rectangle
       */
      adjust(x, y, width, height) {
        const me = this;
        me.x += x;
        me.y += y;
        me.width += width;
        me.height += height;
        return me;
      }
      /**
       * Modifies the bounds of this rectangle by expanding them by the specified amount in all directions.
       * The parameters are read the same way as CSS margin values.
       *
       * - If one value is passed, all edges are inflated by that value.
       * - If two values are passed they are top/bottom inflation and left/right inflation.
       * - If four values are passed, that's top, right, bottom, left.
       * @param {Number} amount How much to inflate, or the top value if more than one values passed.
       * @param {Number} right How much to inflate the right side, or both left and right is only two values passed.
       * @param {Number} bottom How much to inflate the bottom side.
       * @param {Number} left How much to inflate the left side.
       * @returns {Core.helper.util.Rectangle} This Rectangle
       * @internal
       */
      inflate(top, right = top, bottom = top, left = right) {
        if (Array.isArray(top)) {
          [top, right, bottom, left] = parseEdges(top);
        }
        return this.adjust(-left, -top, right, bottom);
      }
      /**
       * Modifies the bounds of this rectangle by reducing them by the specified amount in all directions.
       * The parameters are read the same way as CSS margin values.
       *
       * - If one value is passed, all edges are deflated by that value.
       * - If two values are passed they are top/bottom deflation and left/right deflation.
       * - If four values are passed, that's top, right, bottom, left.
       * @param {Number} amount How much to deflate, or the top value if more than one values passed.
       * @param {Number} right How much to deflate the right side, or both left and right is only two values passed.
       * @param {Number} bottom How much to deflate the bottom side.
       * @param {Number} left How much to deflate the left side.
       * @returns {Core.helper.util.Rectangle} This Rectangle
       * @internal
       */
      deflate(top, right = top, bottom = top, left = right) {
        if (Array.isArray(top)) {
          [top, right, bottom, left] = parseEdges(top);
        }
        return this.adjust(left, top, -right, -bottom);
      }
      /**
       * Attempts constrain this Rectangle into the passed Rectangle. If the `strict` parameter is `true`
       * then this method will return `false` if constraint could not be achieved.
       *
       * If this Rectangle has a `minHeight` or `minWidth` property, size will be adjusted while attempting to constrain.
       *
       * Right and bottom are adjusted first leaving the top and bottom sides to "win" in the case that this Rectangle overflows
       * the constrainTo Rectangle.
       * @param {Core.helper.util.Rectangle} constrainTo The Rectangle to constrain this Rectangle into if possible.
       * @param {Boolean} strict Pass `true` to return false, and leave this Rectangle unchanged if constraint
       * could not be achieved.
       * @returns {Core.helper.util.Rectangle|Boolean} This Rectangle. If `strict` is true, and constraining was not successful, `false`.
       */
      constrainTo(constrainTo, strict) {
        const me = this,
          originalHeight = me.height,
          originalY = me.y,
          minWidth = me.minWidth || me.width,
          minHeight = me.minHeight || me.height;
        if (me.height >= constrainTo.height) {
          // If we're strict, fail if we could *never* fit into available height.
          if (strict && minHeight > constrainTo.height) {
            return false;
          }
          // If we are >= constrain height, we will have to be at top edge of constrainTo
          me._y = constrainTo.y;
          me.height = constrainTo.height;
        }
        if (me.width >= constrainTo.width) {
          // If we're strict, fail if we could *never* fit into available width.
          if (strict && minWidth > constrainTo.width) {
            // Could not be constrained; undo any previous attempt with height
            me.y = originalY;
            me.height = originalHeight;
            return false;
          }
          // If we are >= constrain width, we will have to be at left edge of constrainTo
          me._x = constrainTo.x;
          me.width = constrainTo.width;
        }
        // Overflowing the bottom or right sides, translate upwards or leftwards.
        me.translate.apply(me, me.constrainVector = [Math.min(constrainTo.right - me.right, 0), Math.min(constrainTo.bottom - me.bottom, 0)]);
        // Now, after possible translation upwards or left,
        // if we overflow the top or left, translate downwards or right.
        me.translate(Math.max(constrainTo.x - me.x, 0), Math.max(constrainTo.y - me.y, 0));
        return me;
      }
      /**
       * Returns a cloned version of this Rectangle aligned to a target Rectangle, or element or {@link Core.widget.Widget}.
       * @param {Object} spec Alignment specification.
       * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} spec.target The Widget or element or Rectangle to align to.
       * @param {Number[]} [spec.anchorSize] The `[width, height]` of the anchor pointer when in `top` position. The
       * width is the baseline length, and the height is the height of the arrow. If passed, the anchor position
       * will be calculated to be at the centre of the overlap of the two aligned edges and returned in the `anchor`
       * property of the resulting Rectangle:
       *
       *     {
       *         edge: 'top',         // or 'right' or 'bottom' or 'left'
       *         x/y: offset          // dimension to translate and value to translate by.
       *     }
       *
       * @param {Object} [spec.anchorPosition] an `{ x: n, y: n }` anchor translation to be used *if the requested alignment
       * succeeds without violating constraints*. If a fallback alignment is used, the anchor will be centered in the
       * overlap of the aligned edges as usual.
       * @param {Boolean} [spec.overlap] True to allow this to overlap the target.
       * @param {String} spec.align The edge alignment specification string, specifying two points to bring together.
       *
       * Each point is described by an edge initial (`t` for top edge, `b` for bottom edge etc) followed
       * by a percentage along that edge.
       *
       * So the form would be `[trblc][n]-[trblc][n].` The `n` is the percentage offset along that edge
       * which defines the alignment point. This is not valid for alignment point `c` which means the center point.
       *
       * For example `t0-b0' would align this Rectangle's top left corner with the bottom left corner of the `target`.
       * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} [spec.constrainTo] The Widget or Element or Rectangle to constrain to.
       * If the requested alignment cannot be constrained (it will first shrink the resulting Rectangle according
       * to the `minWidth` and `minHeight` properties of this rectangle), then it will try aligning at other edges
       * (honouring the `axisLock` option), and pick the fallback alignment which results in the shortest translation.
       * @param {Boolean} [spec.axisLock] Specify as a truthy value to fall back to aligning against the opposite
       * edge first if the requested alignment cannot be constrained into the `constrainTo` option. If specified
       * as `'flexible'`, then fallback will continue searching for solutions on the remaining two sides.
       * @param {Boolean} [spec.matchSize] When aligning edge-to-edge, match the length of the aligned-to
       * edge of the target. This is only honored when `axisLock` is enabled and alignment succeeds on the requested axis.
       * If __not__ aligning edge-to-edge, `matchSize` matches both dimensions of the target.
       * @param {Number|Number[]} [spec.offset] The 'x' and 'y' offset values to create an extra margin round the target
       * to offset the aligned widget further from the target. May be configured as -ve to move the aligned widget
       * towards the target - for example producing the effect of the anchor pointer piercing the target.
       * @param {Number|Number[]} [spec.constrainPadding] The amount of pixels to pad from the `constrainTo` target,
       * either a single value, or an array of values in CSS edge order.
       * @param {Boolean} [spec.rtl] Pass as true if this is being used in an RTL environment, and aligning 0% to
       * 100% along a horizontal edge must proceed from right to left.
       * @returns {Core.helper.util.Rectangle} A new Rectangle aligned as requested if possible, but if the requested position violates
       * the `constrainTo` Rectangle, the shortest translation from the requested position which obeys constraints will be used.
       */
      alignTo(spec) {
        // The target and constrainTo may be passed as HtmlElements or Widgets.
        // If so, extract the Rectangles without mutating the incoming spec.
        let result = this.clone(),
          {
            target,
            constrainTo,
            constrainPadding
          } = spec,
          calculatedAnchorPosition,
          zone,
          resultZone,
          constrainingToViewport;
        if (target && !target.isRectangle) {
          target = Rectangle.from(target.element ? target.element : target);
        }
        if (constrainTo) {
          if (!constrainTo.isRectangle) {
            // Viewport is denoted by requesting window or document.
            // document.body may overflow the viewport, so this must not be evaluated as the viewport.
            constrainingToViewport = constrainTo === globalThis || constrainTo === document;
            // When rectangle is constrained to some element on the page other than window/document, page scroll
            // should not be taken into account
            const ignorePageScroll = 'ignorePageScroll' in spec ? spec.ignorePageScroll : !constrainingToViewport;
            constrainTo = Rectangle.from(constrainTo.element ? constrainTo.element : constrainTo, null, ignorePageScroll);
          }
          // Shrink the constrainTo Rectangle to account for the constrainPadding
          if (constrainPadding) {
            // An array may be used to specify sides in the CSS standard order.
            // One value means all sides reduce by the same amount.
            constrainPadding = parseEdges(constrainPadding);
            // If we are aligning to an element which is closer to an edge than the
            // constrainPadding value for that edge, override the constrainPadding so that
            // the visual alignment is maintained.
            constrainPadding[0] = Math.min(constrainPadding[0], target.top);
            constrainPadding[1] = Math.min(constrainPadding[1], constrainTo.right - target.right);
            constrainPadding[2] = Math.min(constrainPadding[0], constrainTo.bottom - target.bottom);
            constrainPadding[3] = Math.min(constrainPadding[0], target.left);
            // Must clone a passed Rectangle so as not to mutate objects passed in.
            constrainTo = constrainTo.deflate.apply(constrainTo.clone(), constrainPadding);
          }
        }
        const me = this,
          targetOffsets = createOffsets(spec.offset),
          {
            align,
            axisLock,
            anchorSize,
            anchorPosition,
            matchSize,
            position,
            rtl
          } = spec,
          alignSpec = parseAlign(align, rtl),
          targetConstrainRect = constrainTo && constrainTo.clone(),
          constraintZones = [],
          zoneOrder = [{
            zone: zone = alignSpec.startZone,
            align
          }],
          matchDimension = matchSize && matchDimensions[alignSpec.startZone & 1],
          originalSize = me[matchDimension];
        // Match the size of the edge we are aligning against
        if (matchDimension && axisLock) {
          result[matchDimension] = target[matchDimension];
        }
        // If we are not aligning to an edge, match both diensions.
        else if (!alignSpec.edgeAligned && matchSize) {
          result.width = target.width;
          result.height = target.height;
        }
        // Ensure we will fit before trying
        if (constrainTo) {
          result.constrainTo(constrainTo);
        }
        // If we are aligning edge-to-edge, then plan our fallback strategy when we are constrained.
        if (constrainTo && alignSpec.startZone != null) {
          // Create the list of zone numbers and alignments to try in the preferred order.
          //
          // In the case of axisLock, go through the zones by each axis.
          // So if they asked for t-b, which is zone 2,
          // the array will be [2, 0, 3, 1] (t-b, b-t, r-l, l-r)
          if (axisLock) {
            // First axis flip has to maintain the offset along that axis.
            // so align: l0-r0 has to flip to align: r0-l0. See submenu flipping when
            // constrained to the edge. It flips sides but maintains vertical position.
            zoneOrder.push({
              zone: zone = (zone + 2) % 4,
              align: flipAlign(alignSpec)
            });
            // Only try the other axis is axisLock is 'flexible'
            if (axisLock === 'flexible') {
              zoneOrder.push({
                zone: zone = (alignSpec.startZone + 1) % 4,
                align: defaultAlignments[zone]
              });
              zoneOrder.push({
                zone: zone = (zone + 2) % 4,
                align: defaultAlignments[zone]
              });
            }
          }
          // Go through the zones in order from the requested start.
          // So if they asked for t-b, which is zone 2,
          // the array will be [2, 3, 0, 1] (t-b, r-l, b-t, l-r)
          else {
            for (let i = 1; i < 4; i++) {
              zoneOrder.push({
                zone: zone = (zone + 1) % 4,
                align: defaultAlignments[zone]
              });
            }
          }
        }
        // Allow them to pass anchorPosition: {x: 10} to indicate that after a fully successful,
        // unconstrained align, the anchor should be 10px from the start.
        if (anchorPosition) {
          const pos = alignSpec.startZone & 1 ? 'y' : 'x';
          calculatedAnchorPosition = {
            [pos]: anchorPosition[pos],
            edge: edgeNames[(alignSpec.startZone + 2) % 4]
          };
        }
        // Keep the target within reach. If it's way outside, pull it back so that it's only just outside);
        if (targetConstrainRect && target) {
          targetConstrainRect.adjust(-target.width, -target.height, target.width, target.height);
          target.constrainTo(targetConstrainRect);
        }
        // As part of fallback process when fitting within constraints, result may shrink to our minima
        result.minWidth = me.minWidth;
        result.minHeight = me.minHeight;
        // We're being commanded to try to align at a position
        if (position) {
          result.moveTo(position.x, position.y);
          if (constrainTo) {
            result.constrainTo(constrainTo);
          }
        }
        // We're aligning to a Target Rectangle within a ConstrainTo Rectangle, taking into account
        // a possible anchor pointer, or x/y offsets. Here's the situation:
        //
        //                             <-- ConstrainTo Rectangle -->
        //  +-----------------------------------+--------------------+-------------------------+
        //  |                                   |                    |                         |
        //  |                                   |                    |                         |
        //  |                                   |                    |                         |
        //  |-----------------------------------+--------------------+-------------------------+
        //  |                                   |          ▼         |                         |
        //  |                                   | +----------------+ |                         |
        //  |                                   | |                | |                         |
        //  |                                   | |                | |                         |
        //  |                                   |▶|     Target     |◀|                         |
        //  |                                   | |                | |                         |
        //  |                                   | |                | |                         |
        //  |                                   | +----------------+ |                         |
        //  |                                   |          ▲         |                         |
        //  +-----------------------------------+--------------------+-------------------------|
        //  |                                   |                    |                         |
        //  |                                   |                    |                         |
        //  |                                   |                    |                         |
        //  +-----------------------------------+--------------------+-------------------------+
        //
        // Which results in the four possible constraint zones above, which we index in standard CSS order.
        //
        // Top    = 0
        // Right  = 1
        // Bottom = 2
        // Left   = 3
        //
        // If the initially requested alignment is not within the constrainTo rectangle
        // then, calculate these four, and then loop through them, beginning at the requested one,
        // quitting when we find a position which does not violate constraints. This includes
        // shrinking the aligning Rectangle towards its minima to attempt a fit.
        //
        // The final fallback, if there is no position which does not violate constraints
        // is to position in whichever of the four rectangles has the largest area shrinking overflowing
        // dimensions down to minima if specified.
        //
        else {
          // Offsets: If we are using an anchor to move away from the target, use anchor height in both dimensions.
          // It's rotated so that "height" always has the same meaning. It's the height of the arrow.
          const centerAligned = alignSpec.myEdge === 'c' || alignSpec.targetEdge === 'c',
            offsets = anchorSize && !centerAligned ? [anchorSize[1] + targetOffsets[0], anchorSize[1] + targetOffsets[1]] : targetOffsets,
            targetPoint = target.getAlignmentPoint(alignSpec.targetAlignmentPoint, offsets),
            myPoint = result.getAlignmentPoint(alignSpec.myAlignmentPoint);
          result.translate(targetPoint[0] - myPoint[0], targetPoint[1] - myPoint[1]);
          // If an overlapping position was requested, then we are *not* trying out those four zones.
          // We just respect constraint, and that's it.
          let overlap = result.intersect(target, true);
          // If we are aligned over our target, we just obey that within any constraint.
          // No complex edge alignment attempts to fall back to.
          if (overlap) {
            if (constrainTo) {
              result.constrainTo(constrainTo);
            }
            resultZone = alignSpec.startZone;
            result.translate(...offsets);
          }
          // Aligned to outside of our target, and we need to be constrained
          else if (constrainTo && !constrainTo.contains(result)) {
            const requestedResult = result.clone(),
              solutions = [];
            let zone, largestZone;
            // Any configured anchorPosition becomes invalid now that we're having to move the resulting zone
            // to some unpredictable new place where it fits. It will have to be calculated based upon where
            // we end up aligning.
            calculatedAnchorPosition = null;
            // Calculate the four constraint zones illustrated above.
            // Top
            constraintZones[0] = zone = constrainTo.clone();
            zone.bottom = target.y - offsets[1];
            // Right
            constraintZones[1] = zone = constrainTo.clone();
            zone.x = target.right + offsets[0];
            // Bottom
            constraintZones[2] = zone = constrainTo.clone();
            zone.y = target.bottom + offsets[1];
            // Left
            constraintZones[3] = zone = constrainTo.clone();
            zone.right = target.x - offsets[0];
            // Start from the preferred edge and see if we are able to constrain to within each rectangle
            for (let i = 0; i < zoneOrder.length; i++) {
              // Revert to incoming dimension for fallback out of axisLock
              if (matchDimension && i == 2) {
                result[matchDimension] = originalSize;
              }
              zone = constraintZones[resultZone = zoneOrder[i].zone];
              // Perform unconstrained alignment at the calculated alignment for the zone
              result = result.alignTo({
                target,
                offsets,
                align: zoneOrder[i].align
              });
              // If we are able to strictly constrain into this area, then it's one of the possible solutions.
              // We choose the solution which result in the shortest translation from the initial position.
              if (result.constrainTo(zone, true)) {
                solutions.push({
                  result,
                  zone: resultZone
                });
                // If this successful constraint is at the requested alignment, or at a fallback
                // alignment which has used min size constraints, then that's the correct solution.
                // If there's no size compromising, we have to pick the shortest translation.
                if (!largestZone || result.width < me.width || result.height < me.height) {
                  result.align = zoneOrder[i].align;
                  break;
                }
              }
              // Cache the largest zone we find in case we need the final fallback.
              if (!largestZone || zone.area > largestZone.area) {
                const r = result.clone();
                // And just move the result clone into the edge zone
                switch (resultZone) {
                  // Top
                  case 0:
                    r.moveTo(null, zone.bottom - r.height);
                    break;
                  // Right
                  case 1:
                    r.moveTo(zone.left);
                    break;
                  // Bottom
                  case 2:
                    r.moveTo(null, zone.top);
                    break;
                  // Left
                  case 3:
                    r.moveTo(zone.right - r.width);
                    break;
                }
                largestZone = {
                  area: zone.area,
                  result: r,
                  zone: resultZone
                };
              }
            }
            // The loop found at least one solution
            if (solutions.length) {
              // Multiple fallbacks with no axisLock.
              // Use the solution which resulted in the shortest translation distance from the requested alignment.
              if (solutions.length > 1 && !axisLock) {
                solutions.sort((s1, s2) => {
                  const s1TranslationDistance = Math.sqrt((requestedResult.x - s1.result.x) ** 2 + (requestedResult.y - s1.result.y) ** 2),
                    s2TranslationDistance = Math.sqrt((requestedResult.x - s2.result.x) ** 2 + (requestedResult.y - s2.result.y) ** 2);
                  return s1TranslationDistance - s2TranslationDistance;
                });
              }
              // Initial success, or axisLock. Use first successful solution.
              result = solutions[0].result;
              resultZone = solutions[0].zone;
            }
            // No solutions found - use the largest rectangle.
            else {
              result = largestZone.result;
              resultZone = largestZone.zone;
              // When we are constraining to the viewport, we must still must be constrained,
              // even after we've given up making it align *and* constrain.
              if (constrainingToViewport) {
                result.constrainTo(constrainTo);
              }
            }
          } else {
            resultZone = alignSpec.startZone;
          }
          result.zone = resultZone;
          result.overlap = overlap = result.intersect(target, true);
          // If they included an anchor, calculate its position along its edge.
          if (anchorSize && !overlap) {
            // If we were passed an anchorPosition, and it has remained valid (meaning the requested
            // alignment succeeded with no constraint), then anchorPosition will be set. If not,
            // we have to calculate it based upon the aligned edge.
            if (!calculatedAnchorPosition) {
              const isLeftOrRight = resultZone & 1,
                start = isLeftOrRight ? 'y' : 'x',
                end = isLeftOrRight ? 'bottom' : 'right',
                startValue = Math.max(target[start], result[start]),
                endValue = Math.min(target[end], result[end]);
              let anchorStart = startValue + (endValue - startValue) / 2 - anchorSize[0] / 2;
              const anchorEnd = anchorStart + anchorSize[0];
              if (anchorEnd > result[end]) {
                anchorStart -= anchorEnd - result[end];
              }
              if (anchorStart < result[start]) {
                anchorStart += result[start] - anchorStart;
              }
              // Return an anchor property which will have an x or y property and an edge name onto which the
              // arrow should be aligned.
              calculatedAnchorPosition = {
                [start]: anchorStart - result[start],
                edge: edgeNames[(resultZone + 2) % 4]
              };
            }
            result.anchor = calculatedAnchorPosition;
          }
        }
        return result;
      }
      /**
       * Returns the `[x, y]` position of the specified anchor point of this Rectangle in <edge><offset> format.
       * for example passing "t50" will return the centre point of the top edge, passing "r0" will return the start
       * position of the right edge (the top right corner).
       *
       * Note that the offset defaults to 50, so "t" means the centre of the top edge.
       * @param {String} alignmentPoint The alignment point to calculate. Must match the RegExp `[trbl]\d*`
       * @param {Number[]} margins The `[x, y]` margins to add from the left/right, top/bottom edge.
       * @internal
       */
      getAlignmentPoint(alignmentPoint, margins = zeroOffsets) {
        const me = this,
          parts = alignPointRe.exec(alignmentPoint),
          edge = parts[1].toLowerCase(),
          edgeOffset = Math.min(Math.max(parseInt(parts[2] || 50), 0), 100) / 100;
        switch (edge) {
          case 't':
            return [me.x + me.width * edgeOffset, me.y - margins[1]];
          case 'r':
            return [me.right + margins[0], me.y + me.height * edgeOffset];
          case 'b':
            return [me.x + me.width * edgeOffset, me.bottom + margins[1]];
          case 'l':
            return [me.x - margins[0], me.y + me.height * edgeOffset];
          case 'c':
            {
              return [me.x + me.width / 2, me.y + me.height / 2];
            }
        }
      }
      /**
       * Highlights this Rectangle using the highlighting effect of {@link Core.helper.DomHelper}
       * on a transient element which encapsulates the region's area.
       */
      highlight() {
        const me = this,
          highlightElement = DomHelper.createElement({
            parent: document.body,
            style: `position:absolute;z-index:9999999;pointer-events:none;
                            left:${me.x}px;top:${me.y}px;width:${me.width}px;height:${me.height}px`
          });
        return DomHelper.highlight(highlightElement).then(() => highlightElement.remove());
      }
      /**
       * Visualizes this Rectangle by adding a DOM element which encapsulates the region's area into the provided parent element.
       * @param {DomConfig} config Element config object
       * @returns {Element} The highlight element
       * @internal
       */
      visualize(config, asDomConfig) {
        const me = this,
          domConfig = ObjectHelper.merge({
            style: {
              left: `${me.x}px`,
              top: `${me.y}px`,
              width: `${me.width}px`,
              height: `${me.height}px`,
              pointerEvents: 'none',
              // If this visualization is provided a CSS class, let outside handle position + z-index
              ...(config.class ? {} : {
                position: 'absolute',
                'z-index': 9999999
              })
            }
          }, config);
        return asDomConfig ? domConfig : DomHelper.createElement(domConfig);
      }
      toString(delimiter = ',') {
        return [`${this.top}px`, `${this.right}px`, `${this.bottom}px`, `${this.left}px`].join(delimiter);
      }
    }
    Rectangle._$name = 'Rectangle';

    /**
     * @module Core/helper/util/DomClassList
     */
    const valueSymbol = Symbol('value'),
      lengthSymbol = Symbol('length');
    /**
     * This class encapsulates a list of CSS classes which can be set as the `className`
     * on an `HTMLElement`.
     *
     * Properties names set on this class equate to *adding* a class if the property's value
     * is _truthy_, or removing a class if the value is _falsy_.
     *
     * ```javascript
     * const myClassList = new DomClassList('b-test-button');
     *
     * myClassList.add('test-class');
     * myClassList.important = 1;
     *
     * myHtmlElement.className = myClassList; // Sets it to "b-test-button test-class important"
     * ```
     */
    class DomClassList {
      static change(cls, add, remove, as = 'string') {
        remove = DomClassList.normalize(remove, 'object');
        const after = DomClassList.normalize(cls, 'array').filter(c => !remove[c]);
        if (add) {
          add = DomClassList.normalize(add, 'array');
          for (let i = 0; i < add.length; ++i) {
            if (!after.includes(add[i])) {
              after.push(add[i]);
            }
          }
        }
        return DomClassList.normalize(after, as);
      }
      static from(classes, returnEmpty) {
        if (classes) {
          if (classes.isDomClassList) {
            returnEmpty = returnEmpty ?? true;
          } else {
            returnEmpty = returnEmpty ?? (Objects.isObject(classes) && !Objects.isEmpty(classes));
            classes = new DomClassList(classes);
          }
          if (!classes.value && !returnEmpty) {
            classes = null;
          }
        }
        return classes || (returnEmpty ? new DomClassList() : null);
      }
      /**
       * Converts a class name of any understood type to a desired form.
       * @param {String|String[]|Object|Set|Map|HTMLElement} cls
       * @param {String} as Pass `'object'` to return an object with the class names as its keys (all keys will have a
       * value of `true`), or pass `'array'` to return an array of class names, or pass `'string'` (the default) to
       * return a space-separated string of class names.
       * @returns {String|String[]|Object}
       * @internal
       */
      static normalize(cls, as = 'string') {
        cls = cls || ''; // promote null to '' to avoid typeof snag
        const type = typeof cls,
          asArray = as === 'array',
          asObject = as === 'object',
          asString = !asArray && !asObject;
        let isString = type === 'string',
          c,
          i,
          ret;
        if (type === 'object') {
          var _cls;
          if (cls.nodeType === Element.ELEMENT_NODE && typeof cls.getAttribute === 'function') {
            cls = cls.getAttribute('class') || ''; // cannot use className for SVG el's
            isString = true;
          } else if ((_cls = cls) !== null && _cls !== void 0 && _cls.isDomClassList) {
            cls = cls.values;
          } else if (cls instanceof DOMTokenList) {
            cls = Array.from(cls);
          } else if (cls instanceof Map) {
            cls = Array.from(cls.keys()).filter(k => cls.get(k));
          } else if (cls instanceof Set) {
            cls = Array.from(cls);
          } else if (!Array.isArray(cls)) {
            cls = Objects.getTruthyKeys(cls);
          }
        }
        if (isString) {
          // Pass through Set to ensure only unique class names
          cls = [...new Set(StringHelper.split(cls))];
        }
        // cls is now an array
        for /* empty */
        (i = cls.length; i-- > 0;) {
          c = cls[i];
          if (!c.length) {
            cls.splice(i, 1);
          } else if (c.includes(' ')) {
            cls.splice(i, 1, ...StringHelper.split(c));
          }
        }
        if (asArray) {
          ret = cls;
        } else if (asString) {
          ret = cls.join(' ');
        } else {
          ret = Object.create(null);
          for (i = 0; i < cls.length; ++i) {
            ret[cls[i]] = true;
          }
        }
        return ret;
      }
      /**
       * Initializes a new DomClassList.
       * @param {...String|Object} classes The CSS classes as strings or objects.
       * @function constructor
       */
      constructor(...classes) {
        this.process(1, classes);
      }
      /**
       * Clears all class names from this DomClassList instance.
       * @returns {Core.helper.util.DomClassList} this DomClassList.
       */
      clear() {
        // Keep internal class same shape. Do not delete members.
        for (const key in this) {
          this[key] = false;
        }
        return this;
      }
      /**
       * Sets this DomClassList instance to represent the classes passed as either strings or objects.
       * @returns {Core.helper.util.DomClassList} this DomClassList.
       */
      set(...classes) {
        return this.clear().process(1, classes);
      }
      // To gain some speed in DomHelper.sync(), faster than instanceof etc
      get isDomClassList() {
        return true;
      }
      /**
       * Returns a clone of this DomClassList with all the same keys set.
       * @returns {Core.helper.util.DomClassList} A clone of this DomClassList.
       */
      clone() {
        return new DomClassList(this);
      }
      /**
       * Returns a Boolean value, indicating whether this ClassList has the specified CSS class name.
       * @param {String} className CSS class name to check
       * @returns {Boolean} true if this ClassList contains the passed CSS class name, false otherwise
       */
      contains(className) {
        if (typeof className === 'string' && className) {
          return Boolean(this[className]);
        }
        return false;
      }
      // An instance of this class may be assigned directly to an element's className
      // it will be coerced to a string value using this method.
      toString() {
        // Adding space at the end if there is content to make concatenation code simpler in renderers.
        return this.length ? `${this.value} ` : '';
      }
      toJSON() {
        return this.toString();
      }
      /**
       * Analogous to string.trim, returns the string value of this `DomClassList` with no trailing space.
       * @returns {String} A concatenated string value of all the class names in this `DomClassList`
       * separated by spaces.
       */
      trim() {
        return this.value;
      }
      /**
       * Compares this DomClassList to another DomClassList (or class name string of space separated classes).
       * If the same class names (regardless of order) are present, the two are considered equal.
       *
       * So `new DomClassList('foo bar bletch').isEqual('bletch bar foo')` would return `true`
       * @param {Core.helper.util.DomClassList|String} other The `DomClassList` or string of classes to compare to.
       * @returns {Boolean} `true` if the two contain the same class names.
       */
      isEqual(other) {
        const otherClasses = DomClassList.normalize(other, 'array'),
          len = otherClasses.length;
        if (this.length === len) {
          for (let i = 0; i < len; i++) {
            if (!this[otherClasses[i]]) {
              return false;
            }
          }
          return true;
        }
        return false;
      }
      /**
       * Get/set string value.
       * Class names separated with space.
       * @property {String}
       */
      get value() {
        let value = this[valueSymbol],
          keys;
        if (value == null) {
          keys = Objects.getTruthyKeys(this);
          this[lengthSymbol] = keys.length;
          this[valueSymbol] = value = keys.join(' ');
        }
        return value;
      }
      set value(value) {
        const me = this,
          keys = Object.keys(me),
          len = keys.length;
        for (let i = 0; i < len; i++) {
          delete me[keys[i]];
        }
        if (value) {
          me.process(1, [value]);
        } else {
          // String value needs recalculating
          delete me[valueSymbol];
        }
      }
      /**
       * Returns string values as an array.
       * @readonly
       * @property {String[]}
       */
      get values() {
        return Objects.getTruthyKeys(this);
      }
      get length() {
        // Maintainer: We MUST access the value getter to force
        // the value to be calculated if it's currently dirty.
        return this.value ? this[lengthSymbol] : 0;
      }
      process(value, classes) {
        for (let cls, k, i = 0; i < classes.length; i++) {
          if (classes[i]) {
            cls = classes[i];
            if (cls.isDomClassList || Objects.isObject(cls)) {
              // preserve all keys, even falsy ones
              for (k in cls) {
                this[k] = value ? cls[k] : !cls[k];
              }
            } else {
              cls = DomClassList.normalize(classes[i], 'array');
              for (k = 0; k < cls.length; ++k) {
                this[cls[k]] = value;
              }
            }
          }
        }
        // String value needs recalculating
        delete this[valueSymbol];
        return this;
      }
      /**
       * Adds/removes class names according to the passed object's properties.
       *
       * Properties with truthy values are added.
       * Properties with falsy values are removed.
       * @param {Object} classList Object containing properties to set/clear
       */
      assign(classList) {
        for (const cls in classList) {
          if (!this[cls] !== !classList[cls]) {
            this[cls] = classList[cls];
            // String value needs recalculating
            delete this[valueSymbol];
          }
        }
        return this;
      }
      /**
       * Adds/removes this objects classes to the passed `classList` or element.
       *
       * Properties with truthy values are added.
       * Properties with falsy values are removed.
       * @param {HTMLElement|DOMTokenList} element The element or the element's `classList` to be updated.
       */
      assignTo(element) {
        const classList = element.nodeType === Element.ELEMENT_NODE ? element.classList : element,
          temp = DomClassList.from(classList);
        temp.add(this);
        classList.value = temp.value;
      }
      /**
       * Add CSS class(es)
       * ```
       * myClassList.add('bold', 'small');
       * ```javascript
       * @param {String|String[]|Object} classes CSS classes to add
       */
      add(...classes) {
        return this.process(1, classes);
      }
      /**
       * Remove CSS class(es)
       * ```javascript
       * myClassList.remove('bold', 'small');
       * ```
       * @param {String} classes CSS classes to remove
       */
      remove(...classes) {
        return this.process(0, classes);
      }
      /**
       * Toggles the passed CSS class name.
       *
       * If the `force` parameter is passed, `true` means add the class name, `false` means remove it.
       *
       * ```javascript
       * myClassList.toggle('bold', isImportant);
       * ```
       * @param {String} className CSS class to toggle
       * @param {Boolean} [force] `true` to add the class, `false` to remove it.
       * @returns {Boolean} `true` if the operation changed the value.
       */
      toggle(className, flag = Boolean(!this[className])) {
        flag = Boolean(flag);
        // Only disturb the set classwes if we need to.
        if (Boolean(this[className]) !== flag) {
          this[className] = flag;
          // String value needs recalculating
          delete this[valueSymbol];
          return true;
        }
      }
      /**
       * Analogous to the `String#split` method, but with no delimiter
       * parameter. This method returns an array containing the individual
       * CSS class names set.
       * @returns {String[]} The individual class names in this `DomClassList`
       */
      split() {
        return Objects.getTruthyKeys(this);
      }
      forEach(fn) {
        return Objects.getTruthyKeys(this).forEach(fn);
      }
    }
    // We put this on the prototype and delete it from instances to mark dirty state. This maintains the same shape for
    // the instance to help the JIT
    DomClassList.prototype[valueSymbol] = null;
    DomClassList._$name = 'DomClassList';

    /**
     * @module Core/helper/util/Point
     */
    /**
     * Encapsulates an X,Y coordinate point.
     * @extends Core/helper/util/Rectangle
     */
    class Point extends Rectangle {
      /**
       * Creates a new Point encapsulating the event's page position.
       * @param {Event} event
       * @returns {Core.helper.util.Point}
       * @typings ignore
       */
      static from(event) {
        const touchPoints = event.changedTouches;
        return new Point(touchPoints ? touchPoints[0].screenX : event.screenX, touchPoints ? touchPoints[0].screenY : event.pageY);
      }
      /**
       * Constructs a Point
       * @param x The X coordinate
       * @param y The Y coordinate
       */
      constructor(x, y) {
        super(x, y, 0, 0);
      }
      /**
       * Coerces this Point to be within the passed Rectangle. Translates it into the bounds.
       * @param {Core.helper.util.Rectangle} into The Rectangle into which to coerce this Point.
       */
      constrain(into) {
        this.x = Math.min(Math.max(this.x, into.x), into.right - 1);
        this.y = Math.min(Math.max(this.y, into.y), into.bottom - 1);
        return this;
      }
      toArray() {
        return [this.x, this.y];
      }
    }
    // The Rectangle class uses the Point class, but cannot import it.
    Object.getPrototypeOf(Point).Point = Point;
    Point._$name = 'Point';

    /* eslint-disable bryntum/no-on-in-lib */
    /**
     * @module Core/helper/EventHelper
     */
    /**
     * @typedef {Object.<String,Function|Boolean|Object|Object[]|Number|String>} ElementListenerConfig
     * @property {HTMLElement} options.element The element to add the listener to.
     * @property {Object} options.thisObj The default `this` reference for all handlers added in this call.
     * @property {Boolean} [options.autoDetach=true] The listeners are automatically removed when the `thisObj` is destroyed.
     * @property {String} [options.delegate] A CSS selector string which only fires the handler when the event takes place in a matching element.
     * @property {Boolean} [options.once] Specify as `true` to have the listener(s) removed upon first invocation.
     * @property {Number} [options.delay] The number of milliseconds to delay the handler call after the event fires:
     * @property {Number|Object} [options.expires] The listener only waits for a specified time before
     * being removed. The value may be a number or an object containing an expiry handler.
     * @property {Number} [options.expires.delay] How long to wait for the event for.
     * @property {String|Function} [options.expires.alt] The function to call when the listener expires
     * **without having been triggered**.
     */
    const touchProperties = ['clientX', 'clientY', 'pageX', 'pageY', 'screenX', 'screenY'],
      isOption = {
        element: 1,
        thisObj: 1,
        once: 1,
        delegate: 1,
        delay: 1,
        capture: 1,
        passive: 1,
        throttled: 1,
        autoDetach: 1,
        expires: 1,
        block: 1
      },
      configurable = true,
      returnTrueProp = {
        configurable,
        value: true
      },
      normalizedKeyNames = {
        Spacebar: 'Space',
        Del: 'Delete',
        Esc: 'Escape',
        Left: 'ArrowLeft',
        Up: 'ArrowUp',
        Right: 'ArrowRight',
        Down: 'ArrowDown'
      },
      // Required to identify a keyup event for special key
      specialKeys = {
        Control: 'ctrl',
        Alt: 'alt',
        Shift: 'shift'
      },
      specialKeyRe = /^(ctrl|shift|alt|meta)$/,
      eventProps$1 = ['altKey', 'bubbles', 'button', 'buttons', 'cancelBubble', 'cancelable', 'clientX', 'clientY', 'ctrlKey', 'layerX', 'layerY', 'metaKey', 'pageX', 'pageY', 'returnValue', 'screenX', 'screenY', 'shiftKey'];
    /**
     * Utility methods for dealing with Events, normalizing Touch/Pointer/Mouse events.
     */
    class EventHelper {
      /**
       * DOM event to trigger name mapping.
       * @internal
       */
      static eventNameMap = {
        mousedown: 'MouseDown',
        mouseup: 'MouseUp',
        click: 'Click',
        dblclick: 'DblClick',
        contextmenu: 'ContextMenu',
        mouseenter: 'MouseEnter',
        mouseleave: 'MouseLeave',
        mousemove: 'MouseMove',
        mouseover: 'MouseOver',
        mouseout: 'MouseOut',
        keyup: 'KeyUp',
        keydown: 'KeyDown',
        keypress: 'KeyPress'
      };
      static normalizeEvent(event) {
        return ObjectHelper.copyPropertiesIf(event, event.changedTouches[0] || event.touches[0], touchProperties);
      }
      /**
       * For use when synthesizing events from native DOM events. Copies valid properties from the passed
       * event into the destination object;
       * @param {Object} dest Destination object
       * @param {Event} event The event whose properties to copy
       * @returns {Object} An event construction object.
       * @internal
       */
      static copyEvent(dest, event) {
        return ObjectHelper.copyProperties(dest, event, eventProps$1);
      }
      /**
       * Returns the `[x, y]` coordinates of the event in the viewport coordinate system.
       * @param {Event} event The event
       * @returns {Number[]} The coordinate.
       */
      static getXY(event) {
        if (event.touches) {
          event = event.touches[0];
        }
        return [event.clientX, event.clientY];
      }
      /**
       * Returns the pixel distance between two mouse/touch/pointer events.
       * @param {Event} event1 The first event.
       * @param {Event} event2 The second event.
       * @returns {Number} The distance in pixels between the two events.
       */
      static getDistanceBetween(event1, event2) {
        const xy1 = EH.getXY(event1),
          xy2 = EH.getXY(event2);
        // No point in moving this to Point. We are dealing only with number values here.
        return Math.sqrt(Math.pow(xy1[0] - xy2[0], 2) + Math.pow(xy1[1] - xy2[1], 2));
      }
      /**
       * Returns a {@link Core.helper.util.Point} which encapsulates the `pageX/Y` position of the event.
       * May be used in {@link Core.helper.util.Rectangle} events.
       * @param {Event} event A browser mouse/touch/pointer event.
       * @returns {Core.helper.util.Point} The page point.
       */
      static getPagePoint(event) {
        return new Rectangle.Point(event.pageX, event.pageY);
      }
      /**
       * Returns a {@link Core.helper.util.Point} which encapsulates the `clientX/Y` position of the event.
       * May be used in {@link Core.helper.util.Rectangle} events.
       * @param {Event} event A browser mouse/touch/pointer event.
       * @returns {Core.helper.util.Point} The page point.
       */
      static getClientPoint(event) {
        return new Rectangle.Point(event.clientX, event.clientY);
      }
      /**
       * Add a listener or listeners to an element
       * The `options` parameter allows supplying options for the listener(s), for available options see {@link #typedef-ElementListenerConfig}.
       *
       * @param {HTMLElement} element The element to add a listener/listeners to.
       * @param {String|Object} eventName Either a string, being the name of the event to listen for,
       * or an options object containing event names and options as keys. See the options parameter
       * for details, or the {@link #function-on-static} method for details.
       * @param {Function} [handler] If the second parameter is a string event name, this is the handler function.
       * @param {ElementListenerConfig} [options] If the second parameter is a string event name, this is the options.
       * @returns {Function} A detacher function which removes all the listeners when called.
       */
      static addListener(element, eventName, handler, options) {
        if (element.nodeType) {
          // All separate params, element, eventName and handler
          if (typeof eventName === 'string') {
            options = Object.assign({
              element,
              [eventName]: handler
            }, options);
          }
          // element, options
          else {
            options = Object.assign({
              element
            }, eventName);
          }
        }
        // Just an options object passed
        else {
          options = element;
        }
        return EH.on(options);
      }
      /**
       * Adds a listener or listeners to an element.
       * all property names other than the options listed below are taken to be event names,
       * and the values as handler specs.
       *
       * A handler spec is usually a function reference or the name of a function in the `thisObj`
       * option.
       *
       * But a handler spec may also be an options object containing a `handler` property which is
       * the function or function name, and local options, including `element` and `thisObj`
       * which override the top level options.
       *
       * The `options` parameter allows supplying options for the listener(s), for available options see {@link #typedef-ElementListenerConfig}.
       *
       *  Usage example
       *
       * ```javascript
       * construct(config) {
       *     super.construct(config);
       *
       *     // Add auto detaching event handlers to this Widget's reference elements
       *     EventHelper.on({
       *         element : this.iconElement,
       *         click   : '_handleIconClick',
       *         thisObj : this,
       *         contextmenu : {
       *             element : document,
       *             handler : '_handleDocumentContextMenu'
       *         }
       *     });
       * }
       *```
       *
       * The `click` handler on the `iconElement` calls `this._handleIconClick`.
       *
       * The `contextmenu` handler is added to the `document` element, but the `thisObj`
       * is defaulted in from the top `options` and calls `this._handleDocumentContextMenu`.
       *
       * Note that on touch devices, `dblclick` and `contextmenu` events are synthesized.
       * Synthesized events contain a `browserEvent` property containing the final triggering
       * event of the gesture. For example a synthesized `dblclick` event would contain a
       * `browserEvent` property which is the last `touchend` event. A synthetic `contextmenu`
       * event will contain a `browserEvent` property which the longstanding `touchstart` event.
       *
       * @param {ElementListenerConfig} options The full listener specification.
       * @returns {Function} A detacher function which removes all the listeners when called.
       */
      static on(options) {
        const element = options.element,
          thisObj = options.thisObj,
          handlerDetails = [];
        for (const eventName in options) {
          // Only treat it as an event name if it's not a supported option
          if (!isOption[eventName]) {
            let handlerSpec = options[eventName];
            if (typeof handlerSpec !== 'object') {
              handlerSpec = {
                handler: handlerSpec
              };
            }
            const targetElement = handlerSpec.element || element;
            // Keep track of the real handlers added.
            // addElementLister returns [ element, eventName, addedFunction, capture ]
            handlerDetails.push(EH.addElementListener(targetElement, eventName, handlerSpec, options));
          }
        }
        const detacher = () => {
          for (let handlerSpec, i = 0; i < handlerDetails.length; i++) {
            handlerSpec = handlerDetails[i];
            EH.removeEventListener(handlerSpec[0], handlerSpec[1], handlerSpec[2]);
          }
          handlerDetails.length = 0;
        };
        // { autoDetach : true, thisObj : scheduler } means remove all listeners when the scheduler dies.
        if (thisObj && options.autoDetach !== false) {
          thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, detacher, thisObj);
        }
        return detacher;
      }
      /**
       * Used internally to add a single event handler to an element.
       * @param {HTMLElement} element The element to add the handler to.
       * @param {String} eventName The name of the event to add a handler for.
       * @param {Function|String|Object} handlerSpec Either a function to call, or
       * the name of a function to call in the `thisObj`, or an object containing
       * the handler local options.
       * @param {Function|String} [handlerSpec.handler] Either a function to call, or
       * the name of a function to call in the `thisObj`.
       * @param {HTMLElement} [handlerSpec.element] Optionally a local element for the listener.
       * @param {Object} [handlerSpec.thisObj] A local `this` specification for the handler.
       * @param {Object} defaults The `options` parameter from the {@link #function-addListener-static} call.
       * @private
       */
      static addElementListener(element, eventName, handlerSpec, defaults) {
        const handler = EH.createHandler(element, eventName, handlerSpec, defaults),
          {
            spec
          } = handler,
          expires = handlerSpec.expires || defaults.expires,
          options = spec.capture != null || spec.passive != null ? {
            capture: spec.capture,
            passive: spec.passive
          } : undefined;
        element.addEventListener(eventName, handler, options);
        if (expires) {
          // Extract expires : { delay : 100, alt : 'onExpireFn' }
          const thisObj = handlerSpec.thisObj || defaults.thisObj,
            delayable = thisObj !== null && thisObj !== void 0 && thisObj.isDelayable ? thisObj : globalThis,
            {
              alt
            } = expires,
            delay = alt ? expires.delay : expires,
            {
              spec
            } = handler;
          // expires is not applied with other options in createHandler(), store it here
          spec.expires = expires;
          spec.timerId = delayable[typeof delay === 'number' ? 'setTimeout' : 'requestAnimationFrame'](() => {
            spec.timerId = null;
            EH.removeEventListener(element, eventName, handler);
            // If we make it here and the handler has not been called, invoke the alt handler
            if (alt && !handler.called) {
              (typeof alt === 'string' ? thisObj[alt] : alt).call(thisObj);
            }
          }, delay, `listener-timer-${performance.now()}`);
        }
        return [element, eventName, handler, options];
      }
      // composedPath throws in salesforce
      // https://github.com/bryntum/support/issues/4432
      static getComposedPathTarget(event) {
        return event.composedPath()[0] || event.path[0];
      }
      static fixEvent(event) {
        var _target$getRootNode, _target$getRootNode$c, _event$relatedTarget;
        if (event.fixed) {
          return event;
        }
        const {
          type,
          target
        } = event;
        // When we listen to event on document and get event which bubbled from shadow dom, reading its target would
        // return shadow root element, or null if accessed in an async timeframe.
        // We need actual element which started the event
        if ((target !== null && target !== void 0 && target.shadowRoot || target !== null && target !== void 0 && (_target$getRootNode = target.getRootNode) !== null && _target$getRootNode !== void 0 && (_target$getRootNode$c = _target$getRootNode.call(target)) !== null && _target$getRootNode$c !== void 0 && _target$getRootNode$c.host) && event.composedPath) {
          const targetElement = this.getComposedPathTarget(event),
            originalTarget = target;
          // Can there be an event which actually originated from custom element, not its shadow dom?
          Object.defineProperty(event, 'target', {
            value: targetElement,
            configurable
          });
          // Save original target just in case
          Object.defineProperty(event, 'originalTarget', {
            value: originalTarget,
            configurable
          });
        }
        // Flag that we have fixed this event
        Object.defineProperty(event, 'fixed', returnTrueProp);
        // Normalize key names
        if (type.startsWith('key')) {
          const normalizedKeyName = normalizedKeyNames[event.key];
          if (normalizedKeyName) {
            Object.defineProperty(event, 'key', {
              value: normalizedKeyName,
              configurable
            });
          }
          // Polyfill the code property for SPACE because it is not set for synthetic events.
          if (event.key === ' ' && !event.code) {
            Object.defineProperty(event, 'code', {
              value: 'Space',
              configurable
            });
          }
        }
        // Sync OSX's meta key with the ctrl key. This will only happen on Mac platform.
        // It's read-only, so define a local property to return true for ctrlKey.
        if (event.metaKey && !event.ctrlKey) {
          Object.defineProperty(event, 'ctrlKey', returnTrueProp);
        }
        // if (isRTL && (type.startsWith('mouse') || type.startsWith('pointer') || type === 'click')) {
        //     event.nativePageX = event.pageX;
        //
        //     if (!Object.getOwnPropertyDescriptor(event, 'pageX')) {
        //         Object.defineProperties(event, {
        //             pageX : {
        //                 get : () => {
        //                     return document.body.offsetWidth - event.nativePageX;
        //                 }
        //             }
        //         });
        //     }
        // }
        // offsetX/Y are within padding box. Border is outside padding box, so -ve values are possible
        // which are not useful for calculating intra-element positions.
        // We add borderOffsetX and borderOffsetY properties which are offsets within the border box.
        // Tested in EventHelper.js
        if (target && 'offsetX' in event) {
          // Wrap calculating `borderOffsetX/Y` until this property is actually accessed in the code to avoid forced reflow.
          if (!Object.getOwnPropertyDescriptor(event, 'borderOffsetX')) {
            Object.defineProperty(event, 'borderOffsetX', {
              get: () => {
                return event.offsetX + (BrowserHelper.isSafari ? 0 : parseInt(target.ownerDocument.defaultView.getComputedStyle(target).getPropertyValue('border-left-width')));
              }
            });
          }
          if (!Object.getOwnPropertyDescriptor(event, 'borderOffsetY')) {
            Object.defineProperty(event, 'borderOffsetY', {
              get: () => {
                return event.offsetY + (BrowserHelper.isSafari ? 0 : parseInt(target.ownerDocument.defaultView.getComputedStyle(target).getPropertyValue('border-top-width')));
              }
            });
          }
        }
        // Firefox has a bug where it can report that the target is the #document when mouse is over a pseudo element
        if ((target === null || target === void 0 ? void 0 : target.nodeType) === Element.DOCUMENT_NODE && 'clientX' in event) {
          const targetElement = DomHelper.elementFromPoint(event.clientX, event.clientY);
          Object.defineProperty(event, 'target', {
            value: targetElement,
            configurable
          });
        }
        // Firefox has a bug where it can report a textNode as an event target/relatedTarget.
        // We standardize this to report the parentElement.
        if ((target === null || target === void 0 ? void 0 : target.nodeType) === Element.TEXT_NODE) {
          const targetElement = event.target.parentElement;
          Object.defineProperty(event, 'target', {
            value: targetElement,
            configurable
          });
        }
        if (((_event$relatedTarget = event.relatedTarget) === null || _event$relatedTarget === void 0 ? void 0 : _event$relatedTarget.nodeType) === Element.TEXT_NODE) {
          const relatedTargetElement = event.target.parentElement;
          Object.defineProperty(event, 'relatedTarget', {
            value: relatedTargetElement,
            configurable
          });
        }
        // If it's a touch event, move the positional details
        // of touches[0] up to the event.
        if (type.startsWith('touch') && event.touches.length) {
          this.normalizeEvent(event);
        }
        return event;
      }
      static createHandler(element, eventName, handlerSpec, defaults) {
        const delay = handlerSpec.delay || defaults.delay,
          throttled = handlerSpec.throttled || defaults.throttled,
          block = handlerSpec.block || defaults.block,
          once = 'once' in handlerSpec ? handlerSpec.once : defaults.once,
          capture = 'capture' in handlerSpec ? handlerSpec.capture : defaults.capture,
          passive = 'passive' in handlerSpec ? handlerSpec.passive : defaults.passive,
          delegate = 'delegate' in handlerSpec ? handlerSpec.delegate : defaults.delegate,
          wrappedFn = handlerSpec.handler,
          expires = handlerSpec.expires,
          thisObj = handlerSpec.thisObj || defaults.thisObj,
          {
            rtlSource = thisObj
          } = thisObj || {};
        //Capture initial conditions in case of destruction of thisObj.
        // Destruction completely wipes the object.
        // Innermost level of wrapping which calls the user's handler.
        // Normalize the event cross-browser, and attempt to normalize touch events.
        let handler = (event, ...args) => {
          // When playing a demo using DemoBot, only handle synthetic events
          if (EH.playingDemo && event.isTrusted) {
            return;
          }
          // If the thisObj is already destroyed, we cannot call the function.
          // If in dev mode, warn the developer with a JS error.
          if (thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed) {
            return;
          }
          // Fix up events to handle various browser inconsistencies
          event = EH.fixEvent(event, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl);
          // Flag for the expiry timer
          handler.called = true;
          (typeof wrappedFn === 'string' ? thisObj[wrappedFn] : wrappedFn).call(thisObj, event, ...args);
          // Remove properties that our fixEvent method added.
          // Other applications to which this may bubble need the pure browser event.
          delete event.target;
          delete event.relatedTarget;
          delete event.originalarget;
          delete event.key;
          delete event.code;
          delete event.ctrlKey;
          delete event.fixed;
        };
        // Allow events to be blocked for a certain time
        if (block) {
          const wrappedFn = handler;
          let lastCallTime, lastTarget;
          handler = (e, ...args) => {
            const now = performance.now();
            if (!lastCallTime || e.target !== lastTarget || now - lastCallTime > block) {
              lastTarget = e.target;
              lastCallTime = now;
              wrappedFn(e, ...args);
            }
          };
        }
        // Go through options, each creates a new handler by wrapping the previous handler to implement the options.
        // Right now, we have delay. Note that it may be zero, so test != null
        if (delay != null) {
          const wrappedFn = handler,
            delayable = thisObj !== null && thisObj !== void 0 && thisObj.setTimeout ? thisObj : globalThis;
          handler = (...args) => {
            delayable.setTimeout(() => {
              wrappedFn(...args);
            }, delay);
          };
        }
        // If they specified the throttled option, wrap the handler in a createdThrottled
        // version. Allow the called to specify an alt function to call when the event
        // fires before the buffer time has expired.
        if (throttled != null) {
          let alt,
            buffer = throttled;
          if (throttled.buffer) {
            alt = e => {
              return throttled.alt.call(EH, EH.fixEvent(e, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl));
            };
            buffer = throttled.buffer;
          }
          if (thisObj !== null && thisObj !== void 0 && thisObj.isDelayable) {
            handler = thisObj.throttle(handler, {
              delay: buffer,
              throttled: alt
            });
          } else {
            handler = FunctionHelper.createThrottled(handler, buffer, thisObj, null, alt);
          }
        }
        // This must always be added late to be processed before delay so that the handler is removed immediately.
        // Note that we cant use native once because of our support for `delegate`, it would remove the listener even
        // when delegate does not match
        if (once) {
          const wrappedFn = handler;
          handler = (...args) => {
            EH.removeEventListener(element, eventName, handler);
            wrappedFn(...args);
          };
        }
        // This must be added last to be called first, once and delay should not act on wrong targets when configured
        // with a delegate
        if (delegate) {
          const wrappedFn = handler;
          handler = (event, ...args) => {
            var _event$target$closest;
            event = EH.fixEvent(event, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl);
            // delegate: '.b-field-trigger' only fires when click is in a matching el.
            // currentTarget becomes the delegate.
            // Maintainer: In Edge event.target can be an empty object for transitionend events
            const delegatedTarget = ((_event$target$closest = event.target.closest) === null || _event$target$closest === void 0 ? void 0 : _event$target$closest.call) && event.target.closest(delegate);
            if (!delegatedTarget) {
              return;
            }
            // Allow this to be redefined as it bubbles through listeners up the parentNode axis
            // which might have their own delegate settings.
            Object.defineProperty(event, 'currentTarget', {
              get: () => delegatedTarget,
              configurable: true
            });
            wrappedFn(event, ...args);
          };
        }
        // Only autoDetach here if there's a local thisObj is in the handlerSpec for this one listener.
        // If it's in the defaults, then the "on" method will handle it.
        if (handlerSpec.thisObj && handlerSpec.autoDetach !== false) {
          thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, () => EH.removeEventListener(element, eventName, handler), thisObj);
        }
        handler.spec = {
          delay,
          throttled,
          block,
          once,
          thisObj,
          capture,
          expires,
          passive,
          delegate
        };
        return handler;
      }
      static removeEventListener(element, eventName, handler) {
        const {
          expires,
          timerId,
          thisObj,
          capture
        } = handler.spec;
        // Cancel outstanding expires.alt() call when removing the listener
        if (expires !== null && expires !== void 0 && expires.alt && timerId) {
          const delayable = thisObj !== null && thisObj !== void 0 && thisObj.isDelayable ? thisObj : globalThis;
          delayable[typeof expires.delay === 'number' ? 'clearTimeout' : 'cancelAnimationFrame'](timerId);
        }
        element.removeEventListener(eventName, handler, capture);
      }
      /**
       * Calls a callback when the described animation completes.
       *
       * @param {Object} detail
       * @param {HTMLElement} detail.element The element which is being animated.
       * @param {String|RegExp} [detail.animationName] The name of the animation to wait for.
       * @param {String} [detail.property] If no `animationName` specified, the CSS property
       * which is being animated.
       * @param {Function} detail.handler The function to call on animation end.
       * @param {Number} [detail.duration] Optional fallback time to wait until calling the callback.
       * @param {Object} [detail.thisObj] The `this` reference to call the callback with.
       * @param {Array} [detail.args] Optional arguments to call the callback with.
       * @param {Core.mixin.Delayable} [detail.timerSource] A Delayable to provide the fallback timeout.
       * @param {Boolean} [detail.runOnDestroy] If `timerSource` is a {@link Core.mixin.Delayable},
       * `true` to invoke the callback if it is destroyed during the animation.
       * @returns {Function} a function which detaches the animation end listener.
       */
      static onTransitionEnd({
        element,
        animationName,
        property,
        handler,
        mode = animationName ? 'animation' : 'transition',
        duration = DomHelper[`get${mode === 'transition' ? 'Property' : ''}${StringHelper.capitalize(mode)}Duration`](element, property),
        thisObj = globalThis,
        args = [],
        timerSource,
        runOnDestroy
      }) {
        let timerId;
        timerSource = timerSource || (thisObj.isDelayable ? thisObj : globalThis);
        const callbackArgs = [element, property, ...args],
          doCallback = () => {
            detacher();
            if (!thisObj.isDestroyed) {
              if (thisObj.callback) {
                thisObj.callback(handler, thisObj, callbackArgs);
              } else {
                handler.apply(thisObj, callbackArgs);
              }
            }
          },
          detacher = EH.on({
            element,
            [`${mode}end`]({
              animationName: endedAnimation,
              propertyName,
              target
            }) {
              if (target === element) {
                if (propertyName === property || endedAnimation !== null && endedAnimation !== void 0 && endedAnimation.match(animationName)) {
                  if (timerId) {
                    var _timerSource$clearTim, _timerSource;
                    (_timerSource$clearTim = (_timerSource = timerSource).clearTimeout) === null || _timerSource$clearTim === void 0 ? void 0 : _timerSource$clearTim.call(_timerSource, timerId);
                    timerId = null;
                  }
                  doCallback();
                }
              }
            }
          });
        // If the transition has not signalled its end within duration + 50 milliseconds
        // then give up on it. Remove the listener and call the handler.
        if (duration != null) {
          timerId = timerSource.setTimeout(doCallback, duration + 50, 'onTransitionEnd', runOnDestroy);
        }
        return detacher;
      }
      /**
       * Waits for the described animation completes.
       *
       * @param {Object} config
       * @param {HTMLElement} config.element The element which is being animated.
       * @param {String|RegExp} [config.animationName] The name of the animation to wait for.
       * @param {String} [config.property] If no `animationName` specified, the CSS property
       * which is being animated.
       * @param {Number} [config.duration] Optional fallback time to wait until calling the callback.
       * @param {Core.mixin.Delayable} [config.timerSource] A Delayable to provide the fallback timeout.
       * @param {Boolean} [config.runOnDestroy] If `timerSource` is a {@link Core.mixin.Delayable},
       * `true` to invoke the callback if it is destroyed during the animation.
       * @async
       */
      static async waitForTransitionEnd(config) {
        return new Promise(resolve => {
          config.handler = resolve;
          EventHelper.onTransitionEnd(config);
        });
      }
      /**
       * Private function to wrap the passed function. The returned wrapper function to be used as
       * a `touchend` handler which will call the passed function passing a fabricated `dblclick`
       * event if there is a `click` within 300ms.
       * @param {Element} element element
       * @param {String|Function} handler The handler to call.
       * @param {Object} thisObj The owner of the function.
       * @private
       */
      static createDblClickWrapper(element, handler, thisObj) {
        let startId, secondListenerDetacher, tapholdTimer;
        return () => {
          if (!secondListenerDetacher) {
            secondListenerDetacher = EH.on({
              element,
              // We only get here if a touchstart arrives within 300ms of a click
              touchstart: secondStart => {
                startId = secondStart.changedTouches[0].identifier;
                // Prevent zoom
                secondStart.preventDefault();
              },
              touchend: secondClick => {
                if (secondClick.changedTouches[0].identifier === startId) {
                  secondClick.preventDefault();
                  clearTimeout(tapholdTimer);
                  startId = secondListenerDetacher = null;
                  const targetRect = Rectangle.from(secondClick.changedTouches[0].target, null, true),
                    offsetX = secondClick.changedTouches[0].pageX - targetRect.x,
                    offsetY = secondClick.changedTouches[0].pageY - targetRect.y,
                    dblclickEventConfig = Object.assign({
                      browserEvent: secondClick
                    }, secondClick),
                    dblclickEvent = new MouseEvent('dblclick', dblclickEventConfig);
                  Object.defineProperties(dblclickEvent, {
                    target: {
                      value: secondClick.target
                    },
                    offsetX: {
                      value: offsetX
                    },
                    offsetY: {
                      value: offsetY
                    }
                  });
                  if (typeof handler === 'string') {
                    handler = thisObj[handler];
                  }
                  // Call the wrapped handler passing the fabricated dblclick event
                  handler.call(thisObj, dblclickEvent);
                }
              },
              once: true
            });
            // Cancel the second listener is there's no second click within <dblClickTime> milliseconds.
            tapholdTimer = setTimeout(() => {
              secondListenerDetacher();
              startId = secondListenerDetacher = null;
            }, EH.dblClickTime);
          }
        };
      }
      /**
       * Handles various inputs to figure out the name of the special key of the event.
       *
       * ```javascript
       * EventHelper.toSpecialKey('ctrl') // 'ctrlKey'
       * EventHelper.toSpecialKey(true)   // 'ctrlKey', default for PC (Cmd for Mac)
       * EventHelper.toSpecialKey(false)  // false
       * EventHelper.toSpecialKey('foo')  // false
       * ```
       *
       * @param {*} value User input value to process.
       * @param {String} defaultValue Default value to fall back to if `true` value is passed.
       * @returns {Boolean|String} Returns `false` if provided value cannot be converted to special key and special key
       * name otherwise.
       * @internal
       */
      static toSpecialKey(value, defaultValue = BrowserHelper.isMac ? 'metaKey' : 'ctrlKey') {
        let result = false;
        if (value === true) {
          result = defaultValue;
        } else if (typeof value === 'string') {
          value = value.toLowerCase();
          if (value.match(specialKeyRe)) {
            result = `${value}Key`;
          }
        }
        return result;
      }
      /**
       * If keyup event is triggered when special key is pressed, we don't get special key value from properties like
       * `ctrlKey`. Instead we need to read `event.key`. That property uses full name and we use abbreviations, so we
       * need to convert the key.
       * @param {String} code
       * @returns {String}
       * @internal
       */
      static specialKeyFromEventKey(code) {
        return specialKeys[code] || 'no-special-key';
      }
    }
    const EH = EventHelper;
    /**
     * The time in milliseconds for a `taphold` gesture to trigger a `contextmenu` event.
     * @member {Number} [longPressTime=700]
     * @readonly
     * @static
     */
    EH.longPressTime = 700;
    /**
     * The time in milliseconds within which a second touch tap event triggers a `dblclick` event.
     * @member {Number} [dblClickTime=300]
     * @readonly
     * @static
     */
    EH.dblClickTime = 300;
    // When dragging on a touch device, we need to prevent scrolling from happening.
    // Dragging only starts on a touchmove event, by which time it's too late to preventDefault
    // on the touchstart event which started it.
    // To do this we need a capturing, non-passive touchmove listener at the document level so we can preventDefault.
    // This is in lieu of a functioning touch-action style on iOS Safari. When that's fixed, this will not be needed.
    if (BrowserHelper.isTouchDevice) {
      EH.on({
        element: document,
        touchmove: event => {
          // If we're touching a b-dragging event, then stop any panning by preventing default.
          if (event.target.closest('.b-dragging')) {
            event.preventDefault();
          }
        },
        passive: false,
        capture: true
      });
    }
    EventHelper._$name = 'EventHelper';

    const DEFAULT_FONT_SIZE = 14,
      t0t0 = {
        align: 't0-t0'
      },
      ELEMENT_NODE = Node.ELEMENT_NODE,
      TEXT_NODE = Node.TEXT_NODE,
      {
        isObject: isObject$1
      } = ObjectHelper,
      // Transform matrix parse Regex. CSS transform computed style looks like this:
      // matrix(scaleX(), skewY(), skewX(), scaleY(), translateX(), translateY())
      // or
      // matrix3d(scaleX(), skewY(), 0, 0, skewX(), scaleY(), 0, 0, 0, 0, 1, 0, translateX(), translateY())
      // This is more reliable than using the style literal which may include
      // relative styles such as "translateX(-20em)", or not include the translation at all if it's from a CSS rule.
      // Use a const so as to only compile RexExp once
      // Extract repeating number regexp to simplify next expressions. Available values are: https://developer.mozilla.org/en-US/docs/Web/CSS/number
      numberRe = /[+-]?\d*\.?\d+[eE]?-?\d*/g,
      // -2.4492935982947064e-16 should be supported
      numberReSrc = numberRe.source,
      translateMatrix2dRe = new RegExp(`matrix\\((?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(${numberReSrc}),\\s?(${numberReSrc})`),
      translateMatrix3dRe = new RegExp(`matrix3d\\((?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(-?\\d*),\\s?(-?\\d*)`),
      translateMatrixRe = new RegExp(`(?:${translateMatrix2dRe.source})|(?:${translateMatrix3dRe.source})`),
      pxTtranslateXRe = new RegExp(`translate(3d|X)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`),
      pxTtranslateYRe = new RegExp(`translate(3d|Y)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`),
      whiteSpaceRe = /\s+/,
      semicolonRe = /\s*;\s*/,
      colonRe = /\s*:\s*/,
      digitsRe$1 = /^-?((\d+(\.\d*)?)|(\.?\d+))$/,
      elementPropKey = '$bryntum',
      // A blank value means the expando name is the same as the key in this object, otherwise the key in this object is
      // the name of the domConfig property and the value is the name of the DOM element expando property.
      elementCreateExpandos = {
        elementData: '',
        for: 'htmlFor',
        retainElement: ''
      },
      // DomHelper#createElement properties which require special processing.
      // All other configs such as id and type are applied directly to the element.
      elementCreateProperties = {
        // these two are handled by being in elementCreateExpands:
        // elementData  : 1,
        // for          : 1,
        tag: 1,
        html: 1,
        text: 1,
        children: 1,
        tooltip: 1,
        style: 1,
        dataset: 1,
        parent: 1,
        nextSibling: 1,
        ns: 1,
        reference: 1,
        class: 1,
        className: 1,
        unmatched: 1,
        // Used by syncId approach
        onlyChildren: 1,
        // Used by sync to not touch the target element itself,
        listeners: 1,
        // eslint-disable-line bryntum/no-listeners-in-lib
        compareHtml: 1,
        // Sync
        syncOptions: 1,
        // Sync
        keepChildren: 1 // Sync
      },
      styleIgnoreProperties = {
        length: 1,
        parentRule: 1,
        style: 1
      },
      nativeEditableTags = {
        INPUT: 1,
        TEXTAREA: 1
      },
      nativeFocusableTags = {
        BUTTON: 1,
        IFRAME: 1,
        EMBED: 1,
        INPUT: 1,
        OBJECT: 1,
        SELECT: 1,
        TEXTAREA: 1,
        BODY: 1
      },
      win = globalThis,
      doc = document,
      emptyObject$a = Object.freeze({}),
      arraySlice$1 = Array.prototype.slice,
      immediatePromise$6 = Promise.resolve(),
      fontProps = ['font-size', 'font-size-adjust', 'font-style', 'font-weight', 'font-family', 'font-kerning', 'font-stretch', 'line-height', 'text-transform', 'text-decoration', 'letter-spacing', 'word-break'],
      isHiddenWidget = e => e._hidden,
      parentNode = el => el.parentNode || el.host,
      mergeChildren = (dest, src, options) => {
        if (options.key === 'children') {
          // Normally "children" is an array (for which we won't be here, due to isObject check in caller). To
          // maintain declarative order of children as an object, we need some special sauce:
          return ObjectHelper.mergeItems(dest, src, options);
        }
        return ObjectHelper.blend(dest, src, options);
      },
      isVisible = e => {
        const style = e.ownerDocument.defaultView.getComputedStyle(e);
        return style.getPropertyValue('display') !== 'none' && style.getPropertyValue('visibility') !== 'hidden';
      },
      // Nodes such as SVG which do not expose such a property must have an ancestor which has an offsetParent.
      // If position:fixed, there's no offsetParent, so continue to interrogate parentNode.
      // If the el has appeared through a timer from a destroyed frame, the defaultView will be null.
      hasLayout = el => el && (el === doc.body || Boolean(el.offsetParent) || (el.ownerDocument.defaultView && 'offsetParent' in el && DomHelper.getStyleValue(el, 'position') !== 'fixed' ? el.offsetParent : hasLayout(el.parentNode))),
      elementOrConfigToElement = elementOrConfig => {
        if (elementOrConfig instanceof Node) {
          return elementOrConfig;
        }
        if (typeof elementOrConfig === 'string') {
          return DH$1.createElementFromTemplate(elementOrConfig);
        }
        return DH$1.createElement(elementOrConfig);
      },
      canonicalStyles = Object.create(null),
      canonicalizeStyle = (name, hasUnit) => {
        const entry = canonicalStyles[name] || [StringHelper.hyphenate(name), hasUnit];
        if (!canonicalStyles[name]) {
          canonicalStyles[entry[0]] = canonicalStyles[name] = entry;
        }
        return entry;
      },
      getOffsetParent = node => node.ownerSVGElement ? node.ownerSVGElement.parentNode : node.offsetParent,
      slideInAnimationName = /b-slide-in-from-\w+/;
    // Push the styles that have units into the map:
    ['top', 'right', 'bottom', 'left', 'width', 'height', 'maxWidth', 'maxHeight', 'minWidth', 'minHeight', 'borderSpacing', 'borderWidth', 'borderTopWidth', 'borderRightWidth', 'borderBottomWidth', 'borderLeftWidth', 'marginTop', 'marginRight', 'marginBottom', 'marginLeft', 'paddingTop', 'paddingRight', 'paddingBottom', 'paddingLeft', 'fontSize', 'letterSpacing', 'lineHeight', 'outlineWidth', 'textIndent', 'wordSpacing'].forEach(name => canonicalizeStyle(name, true));
    // We only do the measurement once, if the value is null
    let scrollBarWidth = null,
      idCounter = 0,
      themeInfo = null,
      templateElement,
      htmlParser,
      scrollBarMeasureElement;
    /**
     * @module Core/helper/DomHelper
     */
    /**
     * An object that describes a DOM element. Used for example by {@link #function-createElement-static createElement()}
     * and by {@link Core.helper.DomSync#function-sync-static DomSync.sync()}.
     *
     * ```javascript
     * DomHelper.createElement({
     *    class : {
     *        big   : true,
     *        small : false
     *    },
     *    children : [
     *        { tag : 'img', src : 'img.png' },
     *        { html : '<b style="color: red">Red text</b>' }
     *    ]
     * });
     * ```
     *
     * @typedef {Object} DomConfig
     * @property {String} [tag='div'] Tag name, for example 'span'
     * @property {HTMLElement} [parent] Parent element
     * @property {HTMLElement} [nextSibling] Element's next sibling in the parent element
     * @property {String|Object} [class] CSS classes, as a string or an object (truthy keys will be applied)
     * @property {String|Object} [className] Alias for `class`
     * @property {String|Object} [style] Style, as a string or an object (keys will be hyphenated)
     * @property {Object} [elementData] Data object stored as an expando on the resulting element
     * @property {Object} [dataset] Dataset applied to the resulting element
     * @property {DomConfig[]|Object<String,DomConfig>|String[]|HTMLElement[]} [children] Child elements, as an array that can include
     * DomConfigs that will be turned into elements, plain strings that will be used as text nodes or existing elements that
     * will be moved. Also accepts an object map of DomConfigs, but please note that it cannot be used with
     * `DomHelper.createElement()`
     * @property {String} [html] Html string, used as the resulting elements `innerHTML`. Mutually exclusive with the `children` property
     * @property {TooltipConfig|String} [tooltip] Tooltip config applied to the resulting element
     * @property {String} [text] Text content, XSS safe when you want to display text in the element. Mutually exclusive with the `children` property
     * @property {String} [id] Element's `id`
     * @property {String} [href] Element's `href`
     * @property {String} [ns] Element's namespace
     * @property {String} [src] Element's `src`
     */
    /**
     * Helps with dom querying and manipulation.
     *
     * ```javascript
     * DomHelper.createElement({
     *   tag: 'div',
     *   className: 'parent',
     *   style: 'background: red',
     *   children: [
     *      { tag: 'div', className: 'child' },
     *      { tag: 'div', className: 'child' }
     *   ]
     * });
     * ```
     */
    class DomHelper {
      /**
       * Animates the specified element to slide it into view within the visible viewport
       * of its parentElement from the direction of movement.
       *
       * So in a left-to-right Widget, `direction` 1 means it slides in from the right
       * and `direction` -1 means it slides in from the left. RTL reverses the movement.
       *
       * See the forward/backward navigations in {@link Core.widget.DatePicker} for an example
       * of this in action.
       *
       * If "next" should arrive from below and "previous" should arrive from above, add the
       * class `b-slide-vertical` to the element.
       * @param {HTMLElement} element The element to slide in.
       * @param {Number} direction
       * * `1` to slide in from the "next" direction.
       * * `-1` to slide in from the "previous" direction.
       *
       * If the element is inside an RTL widget the directions are reversed.
       * @async
       */
      static async slideIn(element, direction = 1) {
        const cls = `b-slide-in-${direction > 0 ? 'next' : 'previous'}`,
          {
            classList
          } = element,
          {
            style
          } = element.parentNode,
          {
            overflow,
            overflowX,
            overflowY
          } = style;
        style.overflow = 'hidden';
        classList.add(cls);
        await EventHelper.waitForTransitionEnd({
          element,
          animationName: slideInAnimationName
        });
        style.overflow = overflow;
        style.overflowX = overflowX;
        style.overflowY = overflowY;
        classList.remove(cls);
      }
      /**
       * Returns `true` if the passed element is focusable either programmatically or through pointer gestures.
       * @param {HTMLElement} element The element to test.
       * @returns {Boolean} Returns `true` if the passed element is focusable
       */
      static isFocusable(element, skipAccessibilityCheck = false) {
        if (!skipAccessibilityCheck) {
          // If element is hidden or in a hidden Widget, it's not focusable.
          if (!DH$1.isVisible(element) || DH$1.Widget.fromElement(element, isHiddenWidget)) {
            return false;
          }
        }
        const nodeName = element.nodeName;
        /*
         * An element is focusable if:
         *   - It is natively focusable, or
         *   - It is an anchor or link with href attribute, or
         *   - It has a tabIndex, or
         *   - It is an editing host (contenteditable="true")
         */
        return nativeFocusableTags[nodeName] || (nodeName === 'A' || nodeName === 'LINK') && !!element.href || element.getAttribute('tabIndex') != null || element.contentEditable === 'true';
      }
      /**
       * Returns `true` if the passed element accepts keystrokes to edit its contents.
       * @returns {Boolean} Returns `true` if the passed element is editable.
       */
      static isEditable(element) {
        return element.isContentEditable || nativeEditableTags[element.nodeName];
      }
      /**
       * Returns the rectangle of the element which is currently visible in the browser viewport, i.e. user can find it on
       * screen, or `false` if it is scrolled out of view.
       * @param {HTMLElement} target The element to test.
       * @param {Boolean} [whole=false] Whether to check that whole element is visible, not just part of it.
       * If this is passed as true, the result will be a boolean, `true` or `false`.
       * @privateparam {Core.widget.Widget} [caller] the Widget aligning to the target.
       * @returns {Core.helper.util.Rectangle|Boolean} Returns the rectangle of the element which is currently visible in
       * the browser viewport, or `false` if it is out of view.
       */
      static isInView(target, whole = false, caller) {
        // If the target cannot yield a Rectangle, shortcut all processing.
        if (!hasLayout(target)) {
          return false;
        }
        const positioned = (caller === null || caller === void 0 ? void 0 : caller.positioned) && DomHelper.getStyleValue(caller.element, 'position') !== 'fixed',
          docRect = Rectangle.from(globalThis),
          method = whole ? 'contains' : 'intersect',
          cOp = positioned && caller.element.offsetParent,
          cOpR = positioned && Rectangle.from(cOp);
        // If we get to the top, the visible rectangle is the entire document.
        docRect.height = doc.scrollingElement.scrollHeight;
        // If they asked to test the body, it's always in view
        if (target === doc.body) {
          return docRect;
        }
        const result = this.getViewportIntersection(target, docRect, method);
        // We must use the *viewport* coordinate system to ascertain viewability
        if (result && positioned) {
          result.translate(doc.scrollingElement.scrollLeft, doc.scrollingElement.scrollTop);
        }
        // Return any rectangle to its positioned coordinate system
        return positioned && result ? result.translate(-cOpR.x + cOp.scrollLeft, -cOpR.y + cOp.scrollTop) : result;
      }
      /**
       * This method goes up the DOM tree checking that all ancestors are visible in the viewport
       * @param {HTMLElement} target Starting html element
       * @param {Core.helper.util.Rectangle} docRect Window rectangle
       * @param {String} method 'contains' or 'intersect'
       * @returns {Core.helper.util.Rectangle}
       */
      static getViewportIntersection(target, docRect, method) {
        const {
            parentNode
          } = target,
          {
            parentElement
          } = parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ? target.getRootNode().host : target,
          peStyle = parentElement.ownerDocument.defaultView.getComputedStyle(parentElement),
          parentScroll = peStyle.overflowX !== 'visible' || peStyle.overflowY !== 'visible',
          offsetParent = getOffsetParent(target);
        let result = Rectangle.from(target, null, true);
        for (let viewport = parentScroll ? target.parentNode : offsetParent; result && viewport !== doc.documentElement; viewport = viewport.parentNode) {
          // Skip shadow root node.
          if (viewport.nodeType === Node.DOCUMENT_FRAGMENT_NODE && viewport.host) {
            viewport = viewport.host.parentNode;
          }
          const isTop = viewport === doc.body,
            style = viewport.ownerDocument.defaultView.getComputedStyle(viewport),
            viewportRect = isTop ? docRect : Rectangle.inner(viewport, null, true);
          // If this level allows overflow to show, don't clip. Obv, <body> can't show overflowing els.
          if (isTop || style.overflow !== 'visible') {
            result = viewportRect[method](result, false, true);
          }
        }
        return result;
      }
      /**
       * Returns `true` if the passed element is deeply visible. Meaning it is not hidden using `display`
       * or `visibility` and no ancestor node is hidden.
       * @param {HTMLElement} element The element to test.
       * @returns {Boolean} `true` if deeply visible.
       */
      static isVisible(element) {
        const document = element.ownerDocument;
        // Use the parentNode function so that we can traverse upwards through shadow DOM
        // to correctly ascertain visibility of nodes in web components.
        for (; element; element = parentNode(element)) {
          // Visible if we've reached top of the owning document without finding a hidden Element.
          if (element === document) {
            return true;
          }
          // Must not evaluate a shadow DOM's root fragment.
          if (element.nodeType === element.ELEMENT_NODE && !isVisible(element)) {
            return false;
          }
        }
        // We get here if the node is detached.
        return false;
      }
      /**
       * Returns true if DOM Event instance is passed. It is handy to override to support Locker Service.
       * @param event
       * @internal
       * @returns {Boolean}
       */
      static isDOMEvent(event) {
        return event instanceof Event;
      }
      /**
       * Merges specified source DOM config objects into a `dest` object.
       * @param {DomConfig} dest The destination DOM config object.
       * @param {...DomConfig} sources The DOM config objects to merge into `dest`.
       * @returns {DomConfig} The `dest` object.
       * @internal
       */
      static merge(dest, ...sources) {
        return ObjectHelper.blend(dest, sources, {
          merge: mergeChildren
        });
      }
      /**
       * Updates in-place a DOM config object whose `children` property may be an object instead of the typical array.
       * The keys of such objects become the `reference` property upon conversion.
       *
       * @param {DomConfig} domConfig
       * @param {Function} [namedChildren] A function to call for each named child element.
       * @privateparam {Boolean} [ignoreRefs] Not meant to be manually set, used when recursing.
       * @returns {DomConfig} Returns the altered DOM config
       * @internal
       */
      static normalizeChildren(domConfig, namedChildren, ignoreRefs) {
        var _domConfig$syncOption;
        let children = domConfig === null || domConfig === void 0 ? void 0 : domConfig.children,
          child,
          i,
          name,
          kids,
          ref;
        // Allow redirecting/opting out of ref ownership in a hierarchy
        if (domConfig !== null && domConfig !== void 0 && (_domConfig$syncOption = domConfig.syncOptions) !== null && _domConfig$syncOption !== void 0 && _domConfig$syncOption.ignoreRefs) {
          ignoreRefs = true;
        }
        if (children && !(domConfig instanceof Node)) {
          if (Array.isArray(children)) {
            for (i = 0; i < children.length; ++i) {
              DH$1.normalizeChildren(children[i], namedChildren, ignoreRefs);
            }
          } else {
            kids = children;
            domConfig.children = children = [];
            for (name in kids) {
              var _child;
              child = kids[name];
              if ((_child = child) !== null && _child !== void 0 && _child.isWidget) {
                child = child.element;
              }
              // $ prefix indicates element is not a reference:
              ref = !name.startsWith('$') && !DH$1.isElement(child);
              ref && (namedChildren === null || namedChildren === void 0 ? void 0 : namedChildren(name, /* hoist = */!ignoreRefs));
              if (child) {
                if (!(child instanceof Node)) {
                  if (child.reference === false) {
                    delete child.reference;
                  } else if (ref && typeof child !== 'string') {
                    child.reference = name;
                  }
                  DH$1.normalizeChildren(child, namedChildren, ignoreRefs);
                }
                children.push(child);
              }
            }
          }
        }
        return domConfig;
      }
      static roundPx(px, devicePixelRatio = globalThis.devicePixelRatio || 1) {
        const multiplier = 1 / devicePixelRatio;
        return Math.round(px / multiplier) * multiplier;
      }
      // For use when we are dividing a DOM element into even parts. The resulting value
      // must be floored to prevent overflow. But only floored to the device's resolution,
      // so raw Math.floor will not work - it would leave empty space in hi resolution screens.
      static floorPx(px, devicePixelRatio = globalThis.devicePixelRatio || 1) {
        const multiplier = 1 / devicePixelRatio;
        return Math.floor(px * multiplier) / multiplier;
      }
      /**
       * Returns true if element has opened shadow root
       * @param {HTMLElement} element Element to check
       * @returns {Boolean}
       */
      static isCustomElement(element) {
        return Boolean(element === null || element === void 0 ? void 0 : element.shadowRoot);
      }
      /**
       * Resolves element from point, checking shadow DOM if required
       * @param {Number} x
       * @param {Number} y
       * @returns {HTMLElement}
       */
      static elementFromPoint(x, y) {
        let el = document.elementFromPoint(x, y);
        // Try to check shadow dom if it exists
        if (DH$1.isCustomElement(el)) {
          el = el.shadowRoot.elementFromPoint(x, y) || el;
        }
        return el;
      }
      /**
       * Resolves child element from point __in the passed element's coordinate space__.
       * @param {HTMLElement} parent The element to find the occupying element in.
       * @param {Number|Core.helper.util.Point} x Either the `X` part of a point, or the point to find.
       * @param {Number} [y] The `Y` part of the point.
       * @returns {HTMLElement}
       * @internal
       */
      static childFromPoint(el, x, y, /* internal */parent = el) {
        const p = y == null ? x : new Rectangle(x, y, 0, 0);
        let result = null;
        Array.from(el.children).reverse().some(el => {
          if (Rectangle.from(el, parent).contains(p)) {
            // All rectangles must be relative to the topmost el, so that must be
            // passed down as the "parent" of all Rectangles.
            result = el.children.length && DH$1.childFromPoint(el, p, null, parent) || el;
            return true;
          }
        });
        return result;
      }
      /**
       * Converts a name/value pair of a style name and its value into the canonical (hyphenated) name of the style
       * property and a value with the `defaultUnit` suffix appended if no unit is already present in the `value`.
       *
       * For example:
       * ```javascript
       *  const [property, value] = DomHelper.unitize('marginLeft', 50);
       *  console.log(property, value);
       * ```
       *
       * ```
       *  > margin-left 50px
       * ```
       * @param {String} name
       * @param {String|Number} value
       * @param {String} [defaultUnit]
       * @returns {String[]}
       * @internal
       */
      static unitize(name, value, defaultUnit = 'px') {
        const [trueName, hasUnits] = canonicalizeStyle(name);
        if (value != null) {
          value = String(value);
          value = hasUnits && digitsRe$1.test(value) ? value + defaultUnit : value;
        }
        return [trueName, value];
      }
      /**
       * Returns active element checking shadow dom too
       * @readonly
       * @property {HTMLElement}
       */
      static get activeElement() {
        let el = document.activeElement;
        while (el.shadowRoot) {
          el = el.shadowRoot.activeElement;
        }
        return el;
      }
      // returns active element for DOM tree / shadow DOM tree to which element belongs
      static getActiveElement(element) {
        var _element, _element2;
        if ((_element = element) !== null && _element !== void 0 && _element.isWidget) {
          element = element.element;
        }
        // If no element passed, fallback to document
        let el = (((_element2 = element) === null || _element2 === void 0 ? void 0 : _element2.getRootNode()) || document).activeElement;
        while ((_el = el) !== null && _el !== void 0 && _el.shadowRoot) {
          var _el;
          el = el.shadowRoot.activeElement;
        }
        return el;
      }
      // Returns the visible root (either document.body or a web component shadow root)
      static getRootElement(element) {
        var _element$getRootNode;
        const root = (_element$getRootNode = element.getRootNode) === null || _element$getRootNode === void 0 ? void 0 : _element$getRootNode.call(element),
          {
            nodeType
          } = root;
        // If the root is a document, return its body.
        // If it is a document fragment, then it us a shadow root, so return that.
        // fall back to using the passed element's owning document body.
        return nodeType === Node.DOCUMENT_NODE ? root.body : nodeType === Node.DOCUMENT_FRAGMENT_NODE ? root : element.ownerDocument.contains(element) ? element.ownerDocument.body : null;
      }
      // Returns the topmost HTMLElement inside the current context (either document.body or a direct child of a web component shadow root)
      static getOutermostElement(element) {
        var _element$getRootNode2, _element3;
        const root = (_element$getRootNode2 = (_element3 = element).getRootNode) === null || _element$getRootNode2 === void 0 ? void 0 : _element$getRootNode2.call(_element3);
        if (root !== null && root !== void 0 && root.body) {
          return root === null || root === void 0 ? void 0 : root.body;
        }
        // we are in a shadow root
        // parentNode might be null in salesforce
        while (element.parentNode !== root && element.parentNode) {
          element = element.parentNode;
        }
        return element;
      }
      static isValidFloatRootParent(target) {
        return target === document.body || target.constructor.name === 'ShadowRoot';
      }
      /**
       * Returns the `id` of the passed element. Generates a unique `id` if the element does not have one.
       * @param {HTMLElement} element The element to return the `id` of.
       */
      static getId(element) {
        return element.id || (element.id = 'b-element-' + ++idCounter);
      }
      /**
       * Returns common widget/node ancestor for from/to arguments
       * @param {Core.widget.Widget|HTMLElement} from
       * @param {Core.widget.Widget|HTMLElement} to
       * @returns {Core.widget.Widget|HTMLElement}
       * @internal
       */
      static getCommonAncestor(from, to) {
        if (from === to) {
          return from;
        }
        while (from && !((_from = (_from2 = from)[from.isWidget ? 'owns' : 'contains']) !== null && _from !== void 0 && _from.call(_from2, to) || from === to)) {
          var _from, _from2;
          from = from.owner || from.parentNode;
        }
        return from;
      }
      //region Internal
      /**
       * Internal convenience fn to allow specifying either an element or a CSS selector to retrieve one
       * @private
       * @param {String|HTMLElement} elementOrSelector element or selector to lookup in DOM
       * @returns {HTMLElement}
       */
      static getElement(elementOrSelector) {
        // also used for SVG elements, so need to use more basic class, that is also returned by querySelector
        if (elementOrSelector instanceof Element) {
          return elementOrSelector;
        }
        return doc.querySelector(elementOrSelector);
      }
      /**
       * Sets attributes passed as object to given element
       * @param {String|Element} elementOrSelector
       * @param {Object} attributes
       * @internal
       */
      static setAttributes(elementOrSelector, attributes) {
        const element = DH$1.getElement(elementOrSelector);
        if (element && attributes) {
          for (const key in attributes) {
            if (attributes[key] == null) {
              element.removeAttribute(key);
            } else {
              element.setAttribute(key, attributes[key]);
            }
          }
        }
      }
      /**
       * Sets a CSS [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length) style value.
       * @param {String|HTMLElement} element The element to set the style in, or, if just the result is required,
       * the style magnitude to return with units added. If a nullish value is passed, an empty string
       * is returned.
       * @param {String} [style] The name of a style property which specifies a [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length)
       * @param {Number|String} [value] The magnitude. If a number is used, the value will be set in `px` units.
       * @returns {String} The style value string.
       */
      static setLength(element, style, value) {
        if (arguments.length === 1) {
          value = typeof element === 'number' ? `${element}px` : element ?? '';
        } else {
          element = DH$1.getElement(element);
          value = element.style[style] = typeof value === 'number' ? `${value}px` : value ?? '';
        }
        return value;
      }
      /**
       * Returns string percentified and rounded value for setting element's height, width etc.
       * @param {String|Number} value percent value
       * @param {Number} digits number of decimal digits for rounding
       * @returns {string} percentified value or empty string if value can not be parsed
       * @internal
       */
      static percentify(value, digits = 2) {
        const mult = Math.pow(10, digits);
        return value == null || value === '' || isNaN(value) ? '' : `${Math.round(value * mult) / mult}%`;
      }
      //endregion
      //region Children, going down...
      /**
       * Gets the first direct child of `element` that matches `selector`.
       * @param {HTMLElement} element Parent element
       * @param {String} selector CSS selector
       * @returns {HTMLElement}
       * @category Query children
       */
      static getChild(element, selector) {
        return element.querySelector(':scope>' + selector);
      }
      /**
       * Checks if `element` has any child that matches `selector`.
       * @param {HTMLElement} element Parent element
       * @param {String} selector CSS selector
       * @returns {Boolean} true if any child matches selector
       * @category Query children
       */
      static hasChild(element, selector) {
        return DH$1.getChild(element, selector) != null;
      }
      /**
       * Returns all child elements (not necessarily direct children) that matches `selector`.
       *
       * If `selector` starts with `'>'` or `'# '`, then all components of the `selector` must match inside of `element`.
       * The scope selector, `:scope` is prepended to the selector (and if `#` was used, it is removed).
       *
       * These are equivalent:
       *
       *      DomHelper.children(el, '# .foo .bar');
       *
       *      el.querySelectorAll(':scope .foo .bar');
       *
       * These are also equivalent:
       *
       *      DomHelper.children(el, '> .foo .bar');
       *
       *      el.querySelectorAll(':scope > .foo .bar');
       *
       * @param {HTMLElement} element The parent element
       * @param {String} selector The CSS selector
       * @returns {HTMLElement[]} Matched elements, somewhere below `element`
       * @category Query children
       */
      static children(element, selector) {
        // a '#' could be '#id' but '# ' (hash and space) is not a valid selector...
        if (selector[0] === '>' || selector.startsWith('# ')) {
          if (selector[0] === '#') {
            selector = selector.substr(2);
          }
          selector = ':scope ' + selector;
        }
        return Array.from(element.querySelectorAll(selector));
      }
      // Salesforce doesn't yet support childElementCount. So we relace all native usages with this wrapper and
      // override it for salesforce environment.
      // https://github.com/bryntum/support/issues/3008
      static getChildElementCount(element) {
        return element.childElementCount;
      }
      /**
       * Looks at the specified `element` and all of its children for the one that first matches `selector.
       * @param {HTMLElement} element Parent element
       * @param {String} selector CSS selector
       * @returns {HTMLElement} Matched element, either element or an element below it
       * @category Query children
       */
      static down(element, selector) {
        if (!element) {
          return null;
        }
        if (element.matches && element.matches(selector)) {
          return element;
        }
        selector = ':scope ' + selector;
        return element.querySelector(selector);
      }
      /**
       * Checks if childElement is a descendant of parentElement (contained in it or a sub element)
       * @param {HTMLElement} parentElement Parent element
       * @param {HTMLElement} childElement Child element, at any level below parent (includes nested shadow roots)
       * @returns {Boolean}
       * @category Query children
       */
      static isDescendant(parentElement, childElement) {
        const parentRoot = DH$1.getRootElement(parentElement),
          childRoot = DH$1.getRootElement(childElement);
        if (childRoot && parentRoot !== childRoot && childRoot.host) {
          return DH$1.isDescendant(parentRoot, childRoot.host);
        }
        return parentElement.contains(childElement);
      }
      /**
       * Returns the specified element of the given `event`. If the `event` is an `Element`, it is returned. Otherwise,
       * the `eventName` argument is used to retrieve the desired element property from `event` (this defaults to the
       * `'target'` property).
       * @param {Event|Element} event
       * @param {String} [elementName]
       * @returns {Element}
       */
      static getEventElement(event, elementName = 'target') {
        return !event || DH$1.isElement(event) ? event : event[elementName];
      }
      /**
       * Returns `true` if the provided value is _likely_ a DOM element. If the element can be assured to be from the
       * same document, `instanceof Element` is more reliable.
       * @param {*} value
       * @returns {Boolean}
       */
      static isElement(value) {
        return (value === null || value === void 0 ? void 0 : value.nodeType) === document.ELEMENT_NODE && DH$1.isNode(value);
      }
      /**
       * Returns `true` if the provided element is an instance of React Element.
       * All React elements require an additional $$typeof: Symbol.for('react.element') field declared on the object for security reasons.
       * The object which React.createElement() return has $$typeof property equals to Symbol.for('react.element')
       *
       * Sources:
       * https://reactjs.org/blog/2015/12/18/react-components-elements-and-instances.html
       * https://github.com/facebook/react/pull/4832
       *
       * @param {*} element
       * @returns {Boolean}
       * @internal
       */
      static isReactElement(element) {
        return (element === null || element === void 0 ? void 0 : element.$$typeof) === Symbol.for('react.element');
      }
      /**
       * Returns `true` if the provided value is _likely_ a DOM node. If the node can be assured to be from the same
       * document, `instanceof Node` is more reliable.
       * @param {*} value
       * @returns {Boolean}
       */
      static isNode(value) {
        // cannot use instanceof across frames. Using it here won't help since we'd need the same logic if it were
        // false... meaning we'd have the same chances of a false-positive.
        return Boolean(value) && typeof value.nodeType === 'number' && !isObject$1(value);
      }
      /**
       * Iterates over each result returned from `element.querySelectorAll(selector)`. First turns it into an array to
       * work in IE. Can also be called with only two arguments, in which case the first argument is used as selector and
       * document is used as the element.
       * @param {HTMLElement} element Parent element
       * @param {String} selector CSS selector
       * @param {Function} fn Function called for each found element
       * @category Query children
       */
      static forEachSelector(element, selector, fn) {
        if (typeof element === 'string') {
          // Legacy internal API, no longer valid
          throw new Error('DomHelper.forEachSelector must provide a root element context (for shadow root scenario)');
        }
        DH$1.children(element, selector).forEach(fn);
      }
      /**
       * Iterates over the direct child elements of the specified element. First turns it into an array to
       * work in IE.
       * @param {HTMLElement} element Parent element
       * @param {Function} fn Function called for each child element
       * @category Query children
       */
      static forEachChild(element, fn) {
        Array.from(element.children).forEach(fn);
      }
      /**
       * Removes each element returned from `element.querySelectorAll(selector)`.
       * @param {HTMLElement} element
       * @param {String} selector
       * @category Query children
       */
      static removeEachSelector(element, selector) {
        DH$1.forEachSelector(element, selector, child => child.remove());
      }
      static removeClsGlobally(element, ...classes) {
        classes.forEach(cls => DH$1.forEachSelector(element, '.' + cls, child => child.classList.remove(cls)));
      }
      //endregion
      //region Parents, going up...
      /**
       * Looks at the specified element and all of its parents for the one that first matches selector.
       * @deprecated Since 5.3.9, use native `element.closest()` instead
       * @param {HTMLElement} element Element
       * @param {String} selector CSS selector
       * @returns {HTMLElement} Matched element, either the passed in element or an element above it
       * @category Query parents
       */
      static up(element, selector) {
        VersionHelper.deprecate('Core', '6.0.0', 'DomHelper.up() deprecated, use native `element.closest()` instead');
        return element.closest(selector);
      }
      static getAncestor(element, possibleAncestorParents, outerElement = null) {
        let found = false,
          ancestor,
          parent = element;
        possibleAncestorParents = ArrayHelper.asArray(possibleAncestorParents);
        while (parent = parent.parentElement) {
          if (possibleAncestorParents.includes(parent)) {
            found = true;
            break;
          }
          if (outerElement && parent === outerElement) break;
          ancestor = parent;
        }
        if (!found) return null;
        return ancestor || element;
      }
      /**
       * Retrieves all parents to the specified element.
       * @param {HTMLElement} element Element
       * @returns {HTMLElement[]} All parent elements, bottom up
       * @category Query parents
       */
      static getParents(element) {
        const parents = [];
        while (element.parentElement) {
          parents.push(element.parentElement);
          element = element.parentElement;
        }
        return parents;
      }
      //endregion
      //region Creation
      /**
       * Converts the passed id to an id valid for usage as id on a DOM element.
       * @param {String} id
       * @returns {String}
       */
      static makeValidId(id, replaceValue = '') {
        return StringHelper.makeValidDomId(id, replaceValue);
      }
      /**
       * Creates an Element, accepts a {@link #typedef-DomConfig} object. Example usage:
       *
       * ```javascript
       * DomHelper.createElement({
       *   tag         : 'table', // defaults to 'div'
       *   className   : 'nacho',
       *   html        : 'I am a nacho',
       *   children    : [ { tag: 'tr', ... }, myDomElement ],
       *   parent      : myExistingElement // Or its id
       *   style       : 'font-weight: bold;color: red',
       *   dataset     : { index: 0, size: 10 },
       *   tooltip     : 'Yay!',
       *   ns          : 'http://www.w3.org/1999/xhtml'
       * });
       * ```
       *
       * @param {DomConfig} config Element config object
       * @param {Object} [options] An object specifying creation options. If this is a boolean value, it is
       * understood to be the `returnAll` option.
       * @param {Boolean} [options.ignoreRefs] Pass `true` to ignore element references.
       * @param {Boolean} [options.returnAll] Specify true to return all elements & child elements
       * created as an array.
       * @returns {HTMLElement|HTMLElement[]|Object<String,HTMLElement>} Single element or array of elements `returnAll` was set to true.
       * If any elements had a `reference` property, this will be an object containing a reference to
       * all those elements, keyed by the reference name.
       * @category Creation
       */
      static createElement(config = {}, options) {
        var _options, _options$callback, _config$dataset;
        let returnAll = options,
          element,
          i,
          ignoreChildRefs,
          ignoreRefOption,
          ignoreRefs,
          key,
          name,
          value,
          refOwner,
          refs,
          syncIdField;
        if (typeof returnAll === 'boolean') {
          throw new Error('Clean up');
        } else if (options) {
          ignoreRefs = options.ignoreRefs;
          refOwner = options.refOwner;
          refs = options.refs;
          returnAll = options.returnAll;
          syncIdField = options.syncIdField;
          if (ignoreRefs) {
            ignoreChildRefs = true;
            ignoreRefs = ignoreRefs !== 'children';
          }
        }
        if (typeof config.parent === 'string') {
          config.parent = document.getElementById(config.parent);
        }
        // nextSibling implies a parent
        const parent = config.parent || config.nextSibling && config.nextSibling.parentNode,
          {
            dataset,
            html,
            reference,
            syncOptions,
            text
          } = config;
        if (syncOptions) {
          syncIdField = syncOptions.syncIdField || syncIdField;
          ignoreRefOption = syncOptions.ignoreRefs;
          if (ignoreRefOption) {
            ignoreChildRefs = true;
            ignoreRefs = ignoreRefOption !== 'children';
            options = {
              ...options,
              ignoreRefs: true
            };
          }
        }
        if (ignoreRefs) {
          refOwner = null;
        }
        if (config.ns) {
          element = doc.createElementNS(config.ns, config.tag || 'svg');
        } else {
          element = doc.createElement(config.tag || 'div');
        }
        if (text != null) {
          DH$1.setInnerText(element, text);
        } else if (html != null) {
          if (html instanceof DocumentFragment) {
            element.appendChild(html);
          } else {
            element.innerHTML = html;
          }
        }
        if (config.tooltip) {
          DH$1.Widget.attachTooltip(element, config.tooltip);
        }
        if (config.style) {
          DH$1.applyStyle(element, config.style);
        }
        if (dataset) {
          for (name in dataset) {
            value = dataset[name];
            if (value != null) {
              element.dataset[name] = value;
            }
          }
        }
        if (parent) {
          this.addChild(parent, element, config.nextSibling);
        }
        if (refOwner) {
          // Tag each element created by the refOwner's id to enable DomSync
          element.$refOwnerId = refOwner.id;
        }
        if (reference && !ignoreRefs) {
          // SalesForce platform does not allow custom attributes, but existing code
          // uses querySelector('[reference]'), so bypass it when we can:
          if (refOwner) {
            element.$reference = reference;
            refOwner.attachRef(reference, element, config);
          } else {
            if (!refs) {
              options = Object.assign({}, options);
              options.refs = refs = {};
            }
            refs[reference] = element;
            element.setAttribute('data-reference', reference);
          }
        }
        const className = config.className || config.class,
          // matches DomSync
          keys = Object.keys(config);
        if (className) {
          element.setAttribute('class', DomClassList.normalize(className));
        }
        for (i = 0; i < keys.length; ++i) {
          name = keys[i];
          value = config[name];
          // We have to use setAttribute() for custom attributes to work and this is inline with how DomSync
          // handles attributes. For "expando" properties, however, we have to simply assign them.
          if ((key = elementCreateExpandos[name]) != null) {
            element[key || name] = value;
          } else if (!elementCreateProperties[name] && name && value != null) {
            // if (config.ns) {
            //     element.setAttributeNS(config.ns, name, value);
            // }
            // else {
            //     element.setAttribute(name, value);
            // }
            element.setAttribute(name, value);
          }
        }
        // ARIA. In the absence of a defined role or the element being hidden from ARIA,
        // omit unfocusable elements from the accessibility tree.
        if (!config['aria-hidden'] && !config.role && !config.tabIndex && !DomHelper.isFocusable(element, true) && !element.htmlFor) {
          element.setAttribute('role', 'presentation');
        }
        // Mimic the way DomSync issues callbacks as elements are created (needed by TaskBoard to trigger custom
        // taskRenderer calls as elements get produced).
        (_options = options) === null || _options === void 0 ? void 0 : (_options$callback = _options.callback) === null || _options$callback === void 0 ? void 0 : _options$callback.call(_options, {
          action: 'newElement',
          domConfig: config,
          targetElement: element,
          syncId: refOwner ? reference : options.syncIdField && ((_config$dataset = config.dataset) === null || _config$dataset === void 0 ? void 0 : _config$dataset[options.syncIdField])
        });
        // if returnAll is true, use array
        if (returnAll === true) {
          options.returnAll = returnAll = [element];
        }
        // if it already is an array, add to it (we are probably a child)
        else if (Array.isArray(returnAll)) {
          returnAll.push(element);
        }
        if (config.children) {
          if (syncIdField) {
            // Map syncId -> child element to avoid querying dom later on
            element.syncIdMap = {};
          }
          config.children.forEach(child => {
            // Skip null children, convenient to allow those for usage with Array.map()
            if (child) {
              // Append string children as text nodes
              if (typeof child === 'string') {
                const textNode = document.createTextNode(child);
                if (refOwner) {
                  textNode.$refOwnerId = refOwner.id;
                }
                element.appendChild(textNode);
              }
              // Just append Elements directly.
              else if (isNaN(child.nodeType)) {
                var _config$syncOptions, _child$dataset;
                child.parent = element;
                if (!child.ns && config.ns) {
                  child.ns = config.ns;
                }
                const childElement = DH$1.createElement(child, {
                    ...options,
                    ignoreRefs: ((_config$syncOptions = config.syncOptions) === null || _config$syncOptions === void 0 ? void 0 : _config$syncOptions.ignoreRef) ?? ignoreChildRefs
                  }),
                  syncId = (_child$dataset = child.dataset) === null || _child$dataset === void 0 ? void 0 : _child$dataset[syncIdField];
                // syncId is used with DomHelper.sync to match elements. Populate a map here to make finding them faster
                if (syncId != null) {
                  element.syncIdMap[syncId] = childElement;
                }
                // Do not want to alter the initial config
                delete child.parent;
              } else {
                element.appendChild(child);
              }
            }
          });
        }
        // Store used config, to be able to compare on sync to determine if changed without hitting dom
        element.lastDomConfig = config;
        // If references were used, return them in an object
        // If returnAll was specified, return the array
        // By default, return the root element
        return refs || returnAll || element;
      }
      /**
       * Create element(s) from a template (html string). Note that
       * `textNode`s are discarded unless the `raw` option is passed
       * as `true`.
       *
       * If the template has a single root element, then the single element will be returned
       * unless the `array` option is passed as `true`.
       *
       * If there are multiple elements, then an Array will be returned.
       *
       * @param {String} template The HTML string from which to create DOM content
       * @param {Object} [options] An object containing properties to modify how the DOM is created and returned.
       * @param {Boolean} [options.array] `true` to return an array even if there's only one resulting element.
       * @param {Boolean} [options.raw] Return all child nodes, including text nodes.
       * @param {Boolean} [options.fragment] Return a DocumentFragment.
       * @private
       */
      static createElementFromTemplate(template, options = emptyObject$a) {
        const {
          array,
          raw,
          fragment
        } = options;
        let result;
        // Use template by preference if it exists. It's faster on most supported platforms
        // https://jsperf.com/domparser-vs-template/
        if (DH$1.supportsTemplate) {
          (templateElement || (templateElement = doc.createElement('template'))).innerHTML = template;
          result = templateElement.content;
          if (fragment) {
            // The template is reused, so therefore is its fragment.
            // If we release the fragment to a caller, it must be a clone.
            return result.cloneNode(true);
          }
        } else {
          result = (htmlParser || (htmlParser = new DOMParser())).parseFromString(template, 'text/html').body;
          // We must return a DocumentFragment.
          // myElement.append(fragment) inserts the contents of the fragment, not the fragment itself.
          if (fragment) {
            const nodes = result.childNodes;
            result = document.createDocumentFragment();
            while (nodes.length) {
              result.appendChild(nodes[0]);
            }
            return result;
          }
        }
        // Raw means all child nodes are returned
        if (raw) {
          result = result.childNodes;
        }
        // Otherwise, only element nodes
        else {
          result = result.children;
        }
        return result.length === 1 && !array ? result[0] : arraySlice$1.call(result);
      }
      /**
       * Dispatches a MouseEvent of the passed type to the element at the visible centre of the passed element.
       * @param {HTMLElement} targetElement The element whose center receives the mouse event.
       * @param {String} [type=contextmenu] The mouse event type to dispatch.
       * @internal
       */
      static triggerMouseEvent(targetElement, type = 'contextmenu') {
        const isInView = this.isInView(targetElement),
          targetRect = isInView || Rectangle.from(targetElement),
          targetPoint = targetRect.center,
          contextmenuEvent = new MouseEvent(type, {
            clientX: targetPoint.x,
            clientY: targetPoint.y,
            bubbles: true
          });
        targetElement.dispatchEvent(contextmenuEvent);
      }
      /**
       * Inserts an `element` at first position in `into`.
       * @param {HTMLElement} into Parent element
       * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
       * @returns {HTMLElement}
       * @category Creation
       */
      static insertFirst(into, element) {
        if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
          element = DH$1.createElement(element);
        }
        return into.insertBefore(element, into.firstElementChild);
      }
      /**
       * Inserts a `element` before `beforeElement` in `into`.
       * @param {HTMLElement} into Parent element
       * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
       * @param {HTMLElement} beforeElement Element before which passed element should be inserted
       * @returns {HTMLElement}
       * @category Creation
       */
      static insertBefore(into, element, beforeElement) {
        if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
          element = DH$1.createElement(element);
        }
        return beforeElement ? into.insertBefore(element, beforeElement) : DH$1.insertFirst(into, element);
      }
      static insertAt(parentElement, newElement, index) {
        const siblings = Array.from(parentElement.children);
        if (index >= siblings.length) {
          return DH$1.append(parentElement, newElement);
        }
        const beforeElement = siblings[index];
        return DH$1.insertBefore(parentElement, newElement, beforeElement);
      }
      /**
       * Appends element to parentElement.
       * @param {HTMLElement} parentElement Parent element
       * @param {HTMLElement|DomConfig|String} elementOrConfig Element to insert, or an element config passed on to
       * `createElement()`, or an html string passed to `createElementFromTemplate()`
       * @returns {HTMLElement}
       * @category Creation
       */
      static append(parentElement, elementOrConfig) {
        if (elementOrConfig.forEach) {
          // Ensure all elements of an Array are HTMLElements.
          // The other implementor of forEach is a NodeList which needs no conversion.
          if (Array.isArray(elementOrConfig)) {
            elementOrConfig = elementOrConfig.map(elementOrConfig => elementOrConfigToElement(elementOrConfig));
          }
          if (parentElement.append) {
            parentElement.append(...elementOrConfig);
          } else {
            const docFrag = document.createDocumentFragment();
            elementOrConfig.forEach(function (child) {
              docFrag.appendChild(child);
            });
            parentElement.appendChild(docFrag);
          }
          return elementOrConfig;
        } else {
          return parentElement.appendChild(elementOrConfigToElement(elementOrConfig));
        }
      }
      //endregion
      //region Get position
      /**
       * Returns the element's `transform translateX` value in pixels.
       * @param {HTMLElement} element
       * @returns {Number} X transform
       * @category Position, get
       */
      static getTranslateX(element) {
        const transformStyle = element.style.transform;
        let matches = pxTtranslateXRe.exec(transformStyle);
        // Use inline transform style if it contains "translate(npx, npx" or "translate3d(npx, npx" or "translateX(npx"
        if (matches) {
          return parseFloat(matches[2]);
        } else {
          // If the inline style is the matrix() form, then use that, otherwise, use computedStyle
          matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH$1.getStyleValue(element, 'transform'));
          return matches ? parseFloat(matches[1] || matches[3]) : 0;
        }
      }
      /**
       * Returns the element's `transform translateY` value in pixels.
       * @param {HTMLElement} element
       * @returns {Number} Y coordinate
       * @category Position, get
       */
      static getTranslateY(element) {
        const transformStyle = element.style.transform;
        let matches = pxTtranslateYRe.exec(transformStyle);
        // Use inline transform style if it contains "translate(npx, npx" or "translate3d(npx, npx" or "translateY(npx"
        if (matches) {
          // If it was translateY(npx), use first item in the parens.
          const y = parseFloat(matches[matches[1] === 'Y' ? 2 : 3]);
          // FF will strip `translate(x, 0)` -> `translate(x)`, so need to check for isNaN also
          return isNaN(y) ? 0 : y;
        } else {
          // If the inline style is the matrix() form, then use that, otherwise, use computedStyle
          matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH$1.getStyleValue(element, 'transform'));
          return matches ? parseFloat(matches[2] || matches[4]) : 0;
        }
      }
      /**
       * Gets both X and Y coordinates as an array [x, y]
       * @param {HTMLElement} element
       * @returns {Number[]} [x, y]
       * @category Position, get
       */
      static getTranslateXY(element) {
        return [DH$1.getTranslateX(element), DH$1.getTranslateY(element)];
      }
      /**
       * Get elements X offset within a containing element
       * @param {HTMLElement} element
       * @param {HTMLElement} container
       * @returns {Number} X offset
       * @category Position, get
       */
      static getOffsetX(element, container = null) {
        return container ? element.getBoundingClientRect().left - container.getBoundingClientRect().left : element.offsetLeft;
      }
      /**
       * Get elements Y offset within a containing element
       * @param {HTMLElement} element
       * @param {HTMLElement} container
       * @returns {Number} Y offset
       * @category Position, get
       */
      static getOffsetY(element, container = null) {
        return container ? element.getBoundingClientRect().top - container.getBoundingClientRect().top : element.offsetTop;
      }
      /**
       * Gets elements X and Y offset within containing element as an array [x, y]
       * @param {HTMLElement} element
       * @param {HTMLElement} container
       * @returns {Number[]} [x, y]
       * @category Position, get
       */
      static getOffsetXY(element, container = null) {
        return [DH$1.getOffsetX(element, container), DH$1.getOffsetY(element, container)];
      }
      /**
       * Focus element without scrolling the element into view.
       * @param {HTMLElement} element
       */
      static focusWithoutScrolling(element) {
        function resetScroll(scrollHierarchy) {
          scrollHierarchy.forEach(({
            element,
            scrollLeft,
            scrollTop
          }) => {
            // Check first to avoid triggering unnecessary `scroll` events
            if (element.scrollLeft !== scrollLeft) {
              element.scrollLeft = scrollLeft;
            }
            if (element.scrollTop !== scrollTop) {
              element.scrollTop = scrollTop;
            }
          });
        }
        // Check browsers which do support focusOptions. Currently only Safari lags.
        // https://caniuse.com/mdn-api_htmlelement_focus_preventscroll_option
        const preventScrollSupported = !BrowserHelper.isSafari;
        if (preventScrollSupported) {
          element.focus({
            preventScroll: true
          });
        } else {
          // Examine every parentNode of the target and cache the scrollLeft and scrollTop,
          // and restore all values after the focus has taken place
          const parents = DH$1.getParents(element),
            scrollHierarchy = parents.map(parent => ({
              element: parent,
              scrollLeft: parent.scrollLeft,
              scrollTop: parent.scrollTop
            }));
          element.focus();
          // Reset in async.
          setTimeout(() => resetScroll(scrollHierarchy), 0);
        }
      }
      /**
       * Get elements X position on page
       * @param {HTMLElement} element
       * @returns {Number}
       * @category Position, get
       */
      static getPageX(element) {
        return element.getBoundingClientRect().left + win.pageXOffset;
      }
      /**
       * Get elements Y position on page
       * @param {HTMLElement} element
       * @returns {Number}
       * @category Position, get
       */
      static getPageY(element) {
        return element.getBoundingClientRect().top + win.pageYOffset;
      }
      /**
       * Returns extremal (min/max) size (height/width) of the element in pixels
       * @param {HTMLElement} element
       * @param {String} style minWidth/minHeight/maxWidth/maxHeight
       * @returns {Number}
       * @internal
       */
      static getExtremalSizePX(element, style) {
        const prop = StringHelper.hyphenate(style),
          measure = prop.split('-')[1];
        let value = DH$1.getStyleValue(element, prop);
        if (/%/.test(value)) {
          // Element might be detached from DOM
          if (element.parentElement) {
            value = parseInt(DH$1.getStyleValue(element.parentElement, measure), 10);
          } else {
            value = NaN;
          }
        } else {
          value = parseInt(value, 10);
        }
        return value;
      }
      //endregion
      //region Set position
      /**
       * Set element's `scale`.
       * @param {HTMLElement} element
       * @param {Number} scaleX The value by which the element should be scaled in the X axis (0 to 1)
       * @param {Number} [scaleY] The value by which the element should be scaled in the Y axis (0 to 1).
       * Defaults to `scaleX`
       * @category Position, set
       * @internal
       */
      static setScale(element, scaleX, scaleY = scaleX) {
        const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/);
        if (t.length > 1) {
          if (t[0].startsWith('matrix3d')) {
            t[0] = `matrix3d(${scaleX}`;
            t[5] = scaleY;
          } else {
            t[0] = `matrix(${scaleX}`;
            t[3] = scaleY;
          }
          element.style.transform = t.join(',');
        } else {
          element.style.transform = `scale(${scaleX}, ${scaleY})`;
        }
      }
      /**
       * Set element's `X` translation in pixels.
       * @param {HTMLElement} element
       * @param {Number} x The value by which the element should be translated from its default position.
       * @category Position, set
       */
      static setTranslateX(element, x) {
        const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/);
        // Avoid blurry text on non-retina displays
        x = DH$1.roundPx(x);
        if (t.length > 1) {
          t[t[0].startsWith('matrix3d') ? 12 : 4] = x;
          element.style.transform = t.join(',');
        } else {
          element.style.transform = `translateX(${x}px)`;
        }
      }
      /**
       * Set element's `Y` translation in pixels.
       * @param {HTMLElement} element
       * @param {Number} y  The value by which the element should be translated from its default position.
       * @category Position, set
       */
      static setTranslateY(element, y) {
        const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/);
        // Avoid blurry text on non-retina displays
        y = DH$1.roundPx(y);
        if (t.length > 1) {
          t[t[0].startsWith('matrix3d') ? 13 : 5] = y;
          element.style.transform = t.join(',') + ')';
        } else {
          element.style.transform = `translateY(${y}px)`;
        }
      }
      /**
       * Set element's style `top`.
       * @param {HTMLElement} element
       * @param {Number|String} y The top position. If numeric, `'px'` is used as the unit.
       * @category Position, set
       */
      static setTop(element, y) {
        DH$1.setLength(element, 'top', y);
      }
      /**
       * Set element's style `left`.
       * @param {HTMLElement} element
       * @param {Number|String} x The top position. If numeric, `'px'` is used as the unit.
       * @category Position, set
       */
      static setLeft(element, x) {
        DH$1.setLength(element, 'left', x);
      }
      static setTopLeft(element, y, x) {
        DH$1.setLength(element, 'top', y);
        DH$1.setLength(element, 'left', x);
      }
      static setRect(element, {
        x,
        y,
        width,
        height
      }) {
        DH$1.setTopLeft(element, y, x);
        DH$1.setLength(element, 'width', width);
        DH$1.setLength(element, 'height', height);
      }
      /**
       * Set elements `X` and `Y` translation in pixels.
       * @param {HTMLElement} element
       * @param {Number} [x] The `X translation.
       * @param {Number} [y] The `Y translation.
       * @category Position, set
       */
      static setTranslateXY(element, x, y) {
        if (x == null) {
          return DH$1.setTranslateY(element, y);
        }
        if (y == null) {
          return DH$1.setTranslateX(element, x);
        }
        // Avoid blurry text on non-retina displays
        x = DH$1.roundPx(x);
        y = DH$1.roundPx(y);
        const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/),
          is3d = t[0].startsWith('matrix3d');
        if (t.length > 1) {
          t[is3d ? 12 : 4] = x;
          t[is3d ? 13 : 5] = y;
          element.style.transform = t.join(',') + ')';
        } else {
          element.style.transform = `translate(${x}px, ${y}px)`;
        }
      }
      /**
       * Increase `X` translation
       * @param {HTMLElement} element
       * @param {Number} x The number of pixels by which to increase the element's `X` translation.
       * @category Position, set
       */
      static addTranslateX(element, x) {
        DH$1.setTranslateX(element, DH$1.getTranslateX(element) + x);
      }
      /**
       * Increase `Y` position
       * @param {HTMLElement} element
       * @param {Number} y The number of pixels by which to increase the element's `Y` translation.
       * @category Position, set
       */
      static addTranslateY(element, y) {
        DH$1.setTranslateY(element, DH$1.getTranslateY(element) + y);
      }
      /**
       * Increase X position
       * @param {HTMLElement} element
       * @param {Number} x
       * @category Position, set
       */
      static addLeft(element, x) {
        DH$1.setLeft(element, DH$1.getOffsetX(element) + x);
      }
      /**
       * Increase Y position
       * @param {HTMLElement} element
       * @param {Number} y
       * @category Position, set
       */
      static addTop(element, y) {
        DH$1.setTop(element, DH$1.getOffsetY(element) + y);
      }
      /**
       * Align the passed element with the passed target according to the align spec.
       * @param {HTMLElement} element The element to align.
       * @param {HTMLElement|Core.helper.util.Rectangle} target The target element or rectangle to align to
       * @param {Object} [alignSpec] See {@link Core.helper.util.Rectangle#function-alignTo} Defaults to `{ align : 't0-t0' }`
       * @param {Boolean} [round] Round the calculated Rectangles (for example if dealing with scrolling which
       * is integer based).
       */
      static alignTo(element, target, alignSpec = t0t0, round) {
        target = target instanceof Rectangle ? target : Rectangle.from(target, true);
        const elXY = DH$1.getTranslateXY(element),
          elRect = Rectangle.from(element, true);
        if (round) {
          elRect.roundPx();
          target.roundPx();
        }
        const targetRect = elRect.alignTo(Object.assign(alignSpec, {
          target
        }));
        DH$1.setTranslateXY(element, elXY[0] + targetRect.x - elRect.x, elXY[1] + targetRect.y - elRect.y);
      }
      //endregion
      //region Styles & CSS
      /**
       * Returns a style value or values for the passed element.
       * @param {HTMLElement} element The element to read styles from
       * @param {String|String[]} propName The property or properties to read
       * @param {Boolean} [inline=false] Pass as `true` to read the element's inline style.
       * Note that this could return inaccurate results if CSS rules apply to this element.
       * @returns {String|Object} The value or an object containing the values keyed by the requested property name.
       * @category CSS
       */
      static getStyleValue(element, propName, inline, pseudo) {
        const styles = inline ? element.style : element.ownerDocument.defaultView.getComputedStyle(element, pseudo);
        if (Array.isArray(propName)) {
          const result = {};
          for (const prop of propName) {
            result[prop] = styles.getPropertyValue(StringHelper.hyphenate(prop));
          }
          return result;
        }
        // Use the elements owning view to get the computed style.
        // Ensure the property name asked for is hyphenated.
        // getPropertyValue doesn't work with camelCase
        return styles.getPropertyValue(StringHelper.hyphenate(propName));
      }
      /**
       * Returns an object with the parse style values for the top, right, bottom, and left
       * components of the given edge style.
       *
       * The return value is an object with `top`, `right`, `bottom`, and `left` properties
       * for the respective components of the edge style, as well as `width` (the sum of
       * `left` and `right`) and `height` (the sum of `top` and `bottom`).
       *
       * @param {HTMLElement} element
       * @param {String} edgeStyle The element's desired edge style such as 'padding', 'margin',
       * or 'border'.
       * @param {String} [edges='trbl'] A string with one character codes for each edge. Only
       * those edges will be populated in the returned object. By default, all edges will be
       * populated.
       * @returns {Object}
       */
      static getEdgeSize(element, edgeStyle, edges) {
        const suffix = edgeStyle === 'border' ? '-width' : '',
          ret = {
            raw: {}
          };
        for (const edge of ['top', 'right', 'bottom', 'left']) {
          if (!edges || edges.includes(edge[0])) {
            // This produces px units even if the provided style is em or other (i.e.,
            // getComputedStyle normalizes this):
            ret[edge] = parseFloat(ret.raw[edge] = DH$1.getStyleValue(element, `${edgeStyle}-${edge}${suffix}`));
          }
        }
        // These may not even be requested (based on "edges") but conditional code here
        // would be wasted since the caller would still need to know not to use them...
        // Replace NaN with 0 to keep calculations correct if they only asked for one side.
        ret.width = (ret.left || 0) + (ret.right || 0);
        ret.height = (ret.top || 0) + (ret.bottom || 0);
        return ret;
      }
      /**
       * Splits a style string up into object form. For example `'font-weight:bold;font-size:150%'`
       * would convert to
       *
       * ```javascript
       * {
       *     font-weight : 'bold',
       *     font-size : '150%'
       * }
       * ```
       * @param {String} style A DOM style string
       * @returns {Object} the style declaration in object form.
       */
      static parseStyle(style) {
        if (typeof style === 'string') {
          const styles = style.split(semicolonRe);
          style = {};
          for (let i = 0, {
              length
            } = styles; i < length; i++) {
            const propVal = styles[i].split(colonRe);
            style[propVal[0]] = propVal[1];
          }
        }
        return style || {};
      }
      /**
       * Applies specified style to the passed element. Style can be an object or a string.
       * @param {HTMLElement} element Target element
       * @param {String|Object} style Style to apply, 'border: 1px solid black' or { border: '1px solid black' }
       * @param {Boolean} [overwrite] Specify `true` to replace style instead of applying changes
       * @category CSS
       */
      static applyStyle(element, style, overwrite = false) {
        if (typeof style === 'string') {
          if (overwrite) {
            // Only assign if either end has any styles, do not want to add empty `style` tag on element
            if (style.length || element.style.cssText.length) {
              element.style.cssText = style;
            }
          } else {
            // Add style so as not to delete configs in style such as width, height, flex etc.
            // If a style is already there, the newest, appended one will take precedence.
            element.style.cssText += style;
          }
        } else if (style) {
          if (overwrite) {
            element.style.cssText = '';
            //element.removeAttribute('style');
          }
          // Has a sub-style block in object form? Use it to override style block
          if (style.style && typeof style.style !== 'string') {
            style = ObjectHelper.assign({}, style, style.style);
          }
          let key, value;
          // Prototype chained objects may be passed, so use direct loop.
          for (key in style) {
            // Ignore readonly properties of the CSSStyleDeclaration object:
            // https://developer.mozilla.org/en-US/docs/Web/API/CSSStyleDeclaration
            // Also ignores sub-style blocks, which are applied above
            if (!styleIgnoreProperties[key]) {
              [key, value] = DH$1.unitize(key, style[key]);
              // Cannot use element.style[key], won't work with CSS vars
              if (value == null) {
                element.style.removeProperty(key);
              } else {
                element.style.setProperty(key, value);
              }
            }
          }
          // Has sub-styles as string? Add to cssText after applying style block, to override it
          if (typeof style.style === 'string') {
            element.style.cssText += style.style;
          }
        }
      }
      static getCSSText(style) {
        if (typeof style === 'string') {
          return style;
        }
        let cssText = '';
        for (const key in style) {
          // Ignore readonly properties of the CSSStyleDeclaration object:
          // https://developer.mozilla.org/en-US/docs/Web/API/CSSStyleDeclaration
          if (!styleIgnoreProperties[key]) {
            cssText += `${StringHelper.hyphenate(key)}:${style[key]};`;
          }
        }
        return cssText;
      }
      /**
       * Add multiple classes to elements classList.
       * @param {HTMLElement} element
       * @param {String[]} classes
       * @deprecated Since 5.0. Use {@link https://developer.mozilla.org/en-US/docs/Web/API/DOMTokenList/add add} method
       * for {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/classList Element.classlist}
       * @category CSS
       */
      static addClasses(element, classes) {
        VersionHelper.deprecate('Core', '6.0.0', 'DomHelper.addClasses should be replaced by native classList.add');
        element.classList.add(...classes);
      }
      /**
       * Remove multiple classes from elements classList.
       * @param {HTMLElement} element
       * @param {String[]} classes
       * @deprecated Since 5.0. Use {@link https://developer.mozilla.org/en-US/docs/Web/API/DOMTokenList/remove remove} method
       * for {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/classList Element.classlist}
       * @category CSS
       */
      static removeClasses(element, classes) {
        VersionHelper.deprecate('Core', '6.0.0', 'DomHelper.removeClasses should be replaced by native classList.remove');
        element.classList.remove(...classes);
      }
      /**
       * Toggle multiple classes in elements classList. Helper for toggling multiple classes at once.
       * @param {HTMLElement} element
       * @param {String[]} classes
       * @param {Boolean} [force] Specify true to add classes, false to remove. Leave blank to toggle
       * @category CSS
       */
      static toggleClasses(element, classes, force = null) {
        classes = ArrayHelper.asArray(classes);
        if (force === true) {
          element.classList.add(...classes);
        } else if (force === false) {
          element.classList.remove(...classes);
        } else {
          classes.forEach(cls => element.classList.toggle(cls));
        }
      }
      /**
       * Adds a CSS class to an element during the specified duration
       * @param {HTMLElement} element Target element
       * @param {String} cls CSS class to add temporarily
       * @param {Number} duration Duration in ms, 0 means cls will not be applied
       * @param {Core.mixin.Delayable} delayable The delayable to tie the setTimeout call to
       * @typings delayable -> {typeof Delayable}
       * @category CSS
       */
      static addTemporaryClass(element, cls, duration, delayable = globalThis) {
        if (duration > 0) {
          element.classList.add(cls);
          delayable.setTimeout({
            fn: cls => element.classList.remove(cls),
            delay: duration,
            name: cls,
            args: [cls],
            cancelOutstanding: true
          });
        }
      }
      /**
       * Reads computed style from the element and returns transition duration for a given property in milliseconds
       * @param {HTMLElement} element Target DOM element
       * @param {String} property Animated property name
       * @returns {Number} Duration in ms
       * @internal
       */
      static getPropertyTransitionDuration(element, property) {
        const style = globalThis.getComputedStyle(element),
          properties = style.transitionProperty.split(', '),
          durations = style.transitionDuration.split(', '),
          index = properties.indexOf(StringHelper.hyphenate(property));
        let result;
        if (index !== -1) {
          // get floating value of transition duration in seconds and convert into milliseconds
          result = parseFloat(durations[index]) * 1000;
        }
        return result;
      }
      /**
       * Reads computed style from the element and returns the animation duration for any
       * attached animation in milliseconds
       * @param {HTMLElement} element Target DOM element
       * @returns {Number} Duration in ms
       * @internal
       */
      static getAnimationDuration(element) {
        return parseFloat(DH$1.getStyleValue(element, 'animation-duration')) * 1000;
      }
      //endregion
      //region Effects
      /**
       * Highlights the passed element or Rectangle according to the theme's highlighting rules.
       * Usually an animated framing effect.
       *
       * The framing effect is achieved by adding the CSS class `b-fx-highlight` which references
       * a `keyframes` animation named `b-fx-highlight-animation`. You may override the animation
       * name referenced, or the animation itself in your own CSS.
       *
       * @param {HTMLElement|Core.helper.util.Rectangle} element The element or Rectangle to highlight.
       */
      static highlight(element, delayable = globalThis) {
        if (element instanceof Rectangle) {
          return element.highlight();
        }
        return new Promise(resolve => {
          delayable.setTimeout(() => {
            element.classList.add('b-fx-highlight');
            delayable.setTimeout(() => {
              element.classList.remove('b-fx-highlight');
              resolve();
            }, 1000);
          }, 0);
        });
      }
      //endregion
      //region Measuring / Scrollbar
      /**
       * Measures the scrollbar width using a hidden div. Caches result
       * @property {Number}
       * @readonly
       */
      static get scrollBarWidth() {
        // Ensure the measurement is only done once, when the value is null.
        // Leave measure element in place. It needs to be remeasured when the zoom level is changed
        // which is detected using a window resize listener, so *may* be called frequently.
        if (scrollBarWidth === null) {
          const element = scrollBarMeasureElement || (scrollBarMeasureElement = DH$1.createElement({
            parent: doc.documentElement,
            style: 'position:absolute;top:-9999em;height:100px;overflow-y:scroll'
          }));
          if (element.parentNode !== doc.documentElement) {
            doc.documentElement.appendChild(element);
          }
          scrollBarWidth = element.offsetWidth;
        }
        return scrollBarWidth;
      }
      static get scrollBarPadElement() {
        return {
          className: 'b-yscroll-pad',
          children: [{
            className: 'b-yscroll-pad-sizer'
          }]
        };
      }
      /**
       * Resets DomHelper.scrollBarWidth cache, triggering a new measurement next time it is read
       */
      static resetScrollBarWidth() {
        scrollBarWidth = null;
      }
      /**
       * Measures the text width using a hidden div
       * @param {String} text
       * @param {HTMLElement} sourceElement
       * @returns {Number} width
       * @category Measure
       */
      static measureText(text, sourceElement, useHTML = false, parentElement = undefined) {
        const offScreenDiv = DH$1.getMeasureElement(sourceElement, parentElement);
        offScreenDiv[useHTML ? 'innerHTML' : 'innerText'] = text;
        const result = offScreenDiv.clientWidth;
        offScreenDiv.className = '';
        return result;
      }
      /**
       * Measures a relative size, such as a size specified in `em` units for the passed element.
       * @param {String} size The CSS size value to measure.
       * @param {HTMLElement} sourceElement
       * @param {Boolean} [round] Pass true to return exact width, not rounded value
       * @returns {Number} size The size in pixels of the passed relative measurement.
       * @category Measure
       */
      static measureSize(size, sourceElement, round = true) {
        if (!size) {
          return 0;
        }
        if (typeof size === 'number') {
          return size;
        }
        if (!size.length) {
          return 0;
        }
        if (/^\d+(px)?$/.test(size)) {
          return parseInt(size);
        }
        if (sourceElement) {
          const offScreenDiv = DH$1.getMeasureElement(sourceElement);
          offScreenDiv.innerHTML = '';
          offScreenDiv.style.width = DH$1.setLength(size);
          const result = round ? offScreenDiv.offsetWidth : offScreenDiv.getBoundingClientRect().width;
          offScreenDiv.style.width = offScreenDiv.className = '';
          return result;
        }
        if (/^\d+em$/.test(size)) {
          return parseInt(size) * DEFAULT_FONT_SIZE;
        }
        return isNaN(size) ? 0 : parseInt(size);
      }
      // parentElement allows measurement to happen inside a specific element, allowing scoped css rules to match
      static getMeasureElement(sourceElement, parentElement = doc.body) {
        const sourceElementStyle = win.getComputedStyle(sourceElement),
          offScreenDiv = parentElement.offScreenDiv = parentElement.offScreenDiv || DH$1.createElement({
            parent: parentElement,
            style: 'position:fixed;top:-10000px;left:-10000px;visibility:hidden;contain:strict',
            className: 'b-measure-element',
            children: [{
              style: 'white-space:nowrap;display:inline-block;will-change:contents;width:auto;contain:none'
            }]
          }, {
            returnAll: true
          })[1];
        fontProps.forEach(prop => {
          if (offScreenDiv.style[prop] !== sourceElementStyle[prop]) {
            offScreenDiv.style[prop] = sourceElementStyle[prop];
          }
        });
        offScreenDiv.className = sourceElement.className;
        // In case the measure element was moved/removed, re-add it
        if (offScreenDiv.parentElement.parentElement !== parentElement) {
          parentElement.appendChild(offScreenDiv.parentElement);
        }
        return offScreenDiv;
      }
      /**
       * Strips the tags from a html string, returning text content.
       *
       * ```javascript
       * DomHelper.stripTags('<div class="custom"><b>Bold</b><i>Italic</i></div>'); // -> BoldItalic
       * ```
       *
       * @internal
       * @param {String} htmlString HTML string
       * @returns {String} Text content
       */
      static stripTags(htmlString) {
        const
          // we need to avoid any kind of evaluation of embedded XSS scripts or "web bugs" (img tags that issue
          // GET requests)
          parser = DH$1.$domParser || (DH$1.$domParser = new DOMParser()),
          doc = parser.parseFromString(htmlString, 'text/html');
        return doc.body.textContent;
      }
      //endregion
      //region Sync
      /**
       * Sync one source element attributes, children etc. to a target element. Source element can be specified as a html
       * string or an actual HTMLElement.
       *
       * NOTE: This function is superseded by {@link Core/helper/DomSync#function-sync-static DomSync.sync()}, which works
       * with DOM configs. For most usecases, use it instead.
       *
       * @param {String|HTMLElement} sourceElement Source "element" to copy from
       * @param {HTMLElement} targetElement Target element to apply to, can also be specified as part of the config object
       * @returns {HTMLElement} Returns the updated targetElement (which is also updated in place)
       */
      static sync(sourceElement, targetElement) {
        if (typeof sourceElement === 'string') {
          if (sourceElement === '') {
            targetElement.innerHTML = '';
            return;
          } else {
            sourceElement = DH$1.createElementFromTemplate(sourceElement);
          }
        }
        DH$1.performSync(sourceElement, targetElement);
        return targetElement;
      }
      // Internal helper used for recursive syncing
      static performSync(sourceElement, targetElement) {
        // Syncing identical elements is a no-op
        if (sourceElement.outerHTML !== targetElement.outerHTML) {
          DH$1.syncAttributes(sourceElement, targetElement);
          DH$1.syncContent(sourceElement, targetElement);
          DH$1.syncChildren(sourceElement, targetElement);
          return true;
        }
        return false;
      }
      // Attributes as map { attr : value, ... }, either from an html element or from a config
      static getSyncAttributes(element) {
        const attributes = {},
          // Attribute names, simplifies comparisons and calls to set/removeAttribute
          names = [];
        // Extract from element
        for (let i = 0; i < element.attributes.length; i++) {
          const attr = element.attributes[i];
          if (attr.specified) {
            const name = attr.name.toLowerCase();
            attributes[name] = attr.value;
            names.push(name);
          }
        }
        return {
          attributes,
          names
        };
      }
      /**
       * Syncs attributes from sourceElement to targetElement.
       * @private
       * @param {HTMLElement} sourceElement
       * @param {HTMLElement} targetElement
       */
      static syncAttributes(sourceElement, targetElement) {
        const
          // Extract attributes from elements (sourceElement might be a config)
          {
            attributes: sourceAttributes,
            names: sourceNames
          } = DH$1.getSyncAttributes(sourceElement),
          {
            attributes: targetAttributes,
            names: targetNames
          } = DH$1.getSyncAttributes(targetElement),
          // Used to ignore data-xx attributes when we will be setting entire dataset
          hasDataset = sourceNames.includes('dataset'),
          // Intersect arrays to determine what needs adding, removing and syncing
          toAdd = sourceNames.filter(attr => !targetNames.includes(attr)),
          toRemove = targetNames.filter(attr => !sourceNames.includes(attr) && (!hasDataset || !attr.startsWith('data-'))),
          toSync = sourceNames.filter(attr => targetNames.includes(attr));
        if (toAdd.length > 0) {
          for (let i = 0; i < toAdd.length; i++) {
            const attr = toAdd[i];
            // Style requires special handling
            if (attr === 'style') {
              DH$1.applyStyle(targetElement, sourceAttributes.style, true);
            }
            // So does dataset
            else if (attr === 'dataset') {
              Object.assign(targetElement.dataset, sourceAttributes.dataset);
            }
            // Other attributes are set using setAttribute (since it calls toString() DomClassList works fine)
            else {
              targetElement.setAttribute(attr, sourceAttributes[attr]);
            }
          }
        }
        if (toRemove.length > 0) {
          for (let i = 0; i < toRemove.length; i++) {
            targetElement.removeAttribute(toRemove[i]);
          }
        }
        if (toSync.length > 0) {
          for (let i = 0; i < toSync.length; i++) {
            const attr = toSync[i];
            // Set all attributes that has changed, with special handling for style
            if (attr === 'style') {
              DH$1.applyStyle(targetElement, sourceAttributes.style, true);
            }
            // And dataset
            else if (attr === 'dataset') {
              Object.assign(targetElement.dataset, sourceAttributes.dataset);
            }
            // And class, which might be a DomClassList or an config for a DomClassList
            else if (attr === 'class' && (sourceAttributes.class.isDomClassList || typeof sourceAttributes.class === 'object')) {
              let classList;
              if (sourceAttributes.class.isDomClassList) {
                classList = sourceAttributes.class;
              } else {
                classList = new DomClassList(sourceAttributes.class);
              }
              if (!classList.isEqual(targetAttributes.class)) {
                targetElement.setAttribute('class', classList);
              }
            } else if (targetAttributes[attr] !== sourceAttributes[attr]) {
              targetElement.setAttribute(attr, sourceAttributes[attr]);
            }
          }
        }
      }
      /**
       * Sync content (innerText) from sourceElement to targetElement
       * @private
       * @param {HTMLElement} sourceElement
       * @param {HTMLElement} targetElement
       */
      static syncContent(sourceElement, targetElement) {
        if (DH$1.getChildElementCount(sourceElement) === 0) {
          targetElement.innerText = sourceElement.innerText;
        }
      }
      static setInnerText(targetElement, text) {
        // setting firstChild.data is faster than innerText (and innerHTML),
        // but in some cases the inner node is lost and needs to be recreated
        const {
          firstChild
        } = targetElement;
        if ((firstChild === null || firstChild === void 0 ? void 0 : firstChild.nodeType) === Element.TEXT_NODE) {
          firstChild.data = text;
        } else {
          // textContent is supposed to be faster than innerText, since it does not trigger layout
          targetElement.textContent = text;
        }
      }
      /**
       * Sync traversing children
       * @private
       * @param {HTMLElement} sourceElement Source element
       * @param {HTMLElement} targetElement Target element
       */
      static syncChildren(sourceElement, targetElement) {
        const me = this,
          sourceNodes = arraySlice$1.call(sourceElement.childNodes),
          targetNodes = arraySlice$1.call(targetElement.childNodes);
        while (sourceNodes.length) {
          const sourceNode = sourceNodes.shift(),
            targetNode = targetNodes.shift();
          // only textNodes and elements allowed (no comments)
          if (sourceNode && sourceNode.nodeType !== TEXT_NODE && sourceNode.nodeType !== ELEMENT_NODE) {
            throw new Error(`Source node type ${sourceNode.nodeType} not supported by DomHelper.sync()`);
          }
          if (targetNode && targetNode.nodeType !== TEXT_NODE && targetNode.nodeType !== ELEMENT_NODE) {
            throw new Error(`Target node type ${targetNode.nodeType} not supported by DomHelper.sync()`);
          }
          if (!targetNode) {
            // out of target nodes, add to target
            targetElement.appendChild(sourceNode);
          } else {
            // match node
            if (sourceNode.nodeType === targetNode.nodeType) {
              // same type of node, take action depending on which type
              if (sourceNode.nodeType === TEXT_NODE) {
                // text
                targetNode.data = sourceNode.data;
              } else {
                if (sourceNode.tagName === targetNode.tagName) {
                  me.performSync(sourceNode, targetNode);
                } else {
                  // new tag, remove targetNode and insert new element
                  targetElement.insertBefore(sourceNode, targetNode);
                  targetNode.remove();
                }
              }
            }
            // Trying to set text node as element, use it as innerText
            // (we get this in FF with store mutations and List)
            else if (sourceNode.nodeType === TEXT_NODE && targetNode.nodeType === ELEMENT_NODE) {
              targetElement.innerText = sourceNode.data.trim();
            } else {
              const logElement = sourceNode.parentElement || sourceNode;
              throw new Error(`Currently no support for transforming nodeType.\n${logElement.outerHTML}`);
            }
          }
        }
        // Out of source nodes, remove remaining target nodes
        targetNodes.forEach(targetNode => {
          targetNode.remove();
        });
      }
      /**
       * Replaces the passed element's `className` with the class names
       * passed in either Array or String format or Object.
       *
       * This method compares the existing class set with the incoming class set and
       * avoids mutating the element's class name set if possible.
       *
       * This can avoid browser style invalidations.
       * @param {HTMLElement} element The element whose class list to synchronize.
       * @param {String[]|String|Object} newClasses The incoming class names to set on the element.
       * @returns {Boolean} `true` if the DOM class list was changed.
       * @category CSS
       */
      static syncClassList(element, newClasses) {
        const {
            classList
          } = element,
          isString = typeof newClasses === 'string',
          newClsArray = isString ? newClasses.split(whiteSpaceRe) : DomClassList.normalize(newClasses, 'array'),
          classCount = newClsArray.length;
        let changed = classList.length !== classCount,
          i;
        // If the incoming and existing class lists are the same length
        // then check that each contains the same names. As soon as
        // we find a non-matching name, we know we have to update the
        // className.
        for (i = 0; !changed && i < classCount; i++) {
          changed = !classList.contains(newClsArray[i]);
        }
        if (changed) {
          element.className = isString ? newClasses : newClsArray.join(' ');
        }
        return changed;
      }
      /**
       * Applies the key state of the passed object or DomClassList to the passed element.
       *
       * Properties with a falsy value mean that property name is *removed* as a class name.
       *
       * Properties with a truthy value mean that property name is *added* as a class name.
       *
       * This is different from {@link #function-syncClassList-static}. That sets the `className` of the element to the
       * sum of all its truthy keys, regardless of what the pre-existing value of the `className` was, and ignoring falsy
       * keys.
       *
       * This _selectively_ updates the classes in the `className`. If there is a truthy key, the name is added. If there
       * is a falsy key, the name is removed.
       * @param {HTMLElement} element The element to apply the class list to .
       * @param {Object|Core.helper.util.DomClassList} classes The classes to add or remove.
       * @returns {Boolean} `true` if the DOM class list was changed.
       * @category CSS
       */
      static updateClassList(element, classes) {
        const {
          classList
        } = element;
        let cls,
          add,
          changed = false;
        for (cls in classes) {
          add = Boolean(classes[cls]);
          if (classList.contains(cls) !== add) {
            classList[add ? 'add' : 'remove'](cls);
            changed = true;
          }
        }
        return changed;
      }
      /**
       * Changes the theme to the passed theme name if possible.
       *
       * Theme names are case insensitive. The `href` used is all lower case.
       *
       * To use this method, the `<link rel="stylesheet">` _must_ use the default,
       * Bryntum-supplied CSS files where the `href` end with `<themeName>.css`, so that
       * it can be found in the document, and switched out for a new link with
       * the a modified `href`. The new `href` will use the same path, just
       * with the `themeName` portion substituted for the new name.
       *
       * If no `<link>` with that name pattern can be found, an error will be thrown.
       *
       * If you use this method, you  must ensure that the theme files are
       * all accessible on your server.
       *
       * Because this is an asynchronous operation, a `Promise` is returned.
       * The theme change event is passed to the success function. If the
       * theme was not changed, because the theme name passed is the current theme,
       * nothing is passed to the success function.
       *
       * The theme change event contains two properties:
       *
       *  - `prev` The previous Theme name.
       *  - `theme` The new Theme name.
       *
       * @param {String} newThemeName the name of the theme that should be applied
       * @privateparam {String} [defaultTheme] Optional, the name of the theme that should be used in case of fail
       * @returns {Promise} A promise who's success callback receives the theme change
       * event if the theme in fact changed. If the theme `href` could not be loaded,
       * the failure callback is called, passing the error event caught.
       * @async
       */
      static setTheme(newThemeName, defaultTheme) {
        newThemeName = newThemeName.toLowerCase();
        const {
            head
          } = document,
          oldThemeName = DH$1.getThemeInfo(defaultTheme).name.toLowerCase();
        let oldThemeLinks = head.querySelectorAll('[data-bryntum-theme]:not([data-loading])'),
          loaded = 0;
        if (oldThemeName === newThemeName) {
          return immediatePromise$6;
        }
        // Remove any links currently loading
        DH$1.removeEachSelector(head, '#bryntum-theme[data-loading],link[data-bryntum-theme][data-loading]');
        const themeEvent = {
          theme: newThemeName,
          prev: oldThemeName
        };
        function replaceTheme(oldThemeLink, resolve, reject) {
          const newThemeLink = DomHelper.createElement({
            tag: 'link',
            rel: 'stylesheet',
            dataset: {
              loading: true,
              bryntumTheme: true
            },
            href: oldThemeLink.href.replace(oldThemeName, newThemeName),
            nextSibling: oldThemeLink
          });
          newThemeLink.addEventListener('load', () => {
            delete newThemeLink.dataset.loading;
            themeInfo = null;
            // Flip all products to the new theme at the same time
            if (++loaded === oldThemeLinks.length) {
              oldThemeLinks.forEach(link => link.remove());
              GlobalEvents.trigger('theme', themeEvent);
              resolve(themeEvent);
            }
          });
          newThemeLink.addEventListener('error', e => {
            delete newThemeLink.dataset.loading;
            reject(e);
          });
        }
        if (oldThemeLinks.length) {
          return new Promise((resolve, reject) => {
            oldThemeLinks.forEach((oldThemeLink, i) => {
              replaceTheme(oldThemeLink, resolve, reject, i === oldThemeLinks.length - 1);
            });
          });
        } else {
          const oldThemeLink = head.querySelector('#bryntum-theme:not([data-loading])') || head.querySelector(`[href*="${oldThemeName}.css"]:not([data-loading])`);
          // Theme link href ends with <themeName>.css also there could be a query - css?11111...
          if (!(oldThemeLink !== null && oldThemeLink !== void 0 && oldThemeLink.href.includes(`${oldThemeName}.css`))) {
            throw new Error(`Theme link for ${oldThemeName} not found`);
          }
          oldThemeLinks = [oldThemeLink];
          return new Promise((resolve, reject) => replaceTheme(oldThemeLink, resolve, reject));
        }
      }
      /**
       * A theme information object about the current theme.
       *
       * Currently, this has only one property:
       *
       *   - `name` The current theme name.
       * @property {Object}
       * @readonly
       */
      static get themeInfo() {
        return DomHelper.getThemeInfo();
      }
      /**
       * A theme information object about the current theme.
       *
       * Currently this has only one property:
       *
       *   - `name` The current theme name.
       * @param {String} defaultTheme the name of the theme used as backup value in case of fail
       * @param {HTMLElement} contextElement The element for which to find the theme. If using a
       * web component, the theme will be encapsulated in the web component's encapsulated style
       * so a context element is required. If no web components are in use, this may be omitted and
       * `document.body` will be used.
       * @returns {Object} info, currently it contains only one property - 'name'.
       * @private
       */
      static getThemeInfo(defaultTheme) {
        if (!themeInfo) {
          const
            // The content it creates for 'b-theme-info' is described in corresponding theme in Core/resources/sass/themes
            // for example in Core/resources/sass/themes/material.scss
            // ```
            // .b-theme-info:before {
            //     content : '{"name":"Material"}';
            // }
            // ```
            testDiv = DH$1.createElement({
              parent: document.body,
              className: 'b-theme-info'
            }),
            // Theme desc object is in the :before pseudo element.
            themeData = DH$1.getStyleValue(testDiv, 'content', false, ':before');
          if (themeData) {
            // themeData could be invalid JSON string in case there is no content rule
            try {
              themeInfo = JSON.parse(themeData.replace(/^["']|["']$|\\/g, ''));
            } catch (e) {
              themeInfo = null;
            }
          }
          // CSS file has to be loaded to make the themeInfo available, so fallback to the default theme name
          themeInfo = themeInfo || (defaultTheme ? {
            name: defaultTheme
          } : null);
          testDiv.remove();
        }
        return themeInfo;
      }
      //endregion
      //region Transition
      static async transition({
        element: outerElement,
        selector = '[data-dom-transition]',
        duration,
        action,
        thisObj = this,
        addTransition = {},
        removeTransition = {}
      }) {
        const scrollers = new Set(),
          beforeElements = Array.from(outerElement.querySelectorAll(selector)),
          beforeMap = new Map(beforeElements.map(element => {
            let depth = 0,
              parent = element.parentElement;
            while (parent && parent !== outerElement) {
              depth++;
              parent = parent.parentElement;
            }
            element.$depth = depth;
            // Store scrolling elements and their current scroll pos, for restoring later
            if (element.scrollHeight > element.offsetHeight && getComputedStyle(element).overflow === 'auto') {
              element.$scrollTop = element.scrollTop;
              scrollers.add(element);
            }
            // Intersect our bounds with parents, to trim away overflow
            const {
                parentElement
              } = element,
              globalBounds = Rectangle.from(element, outerElement),
              localBounds = Rectangle.from(element, parentElement),
              style = getComputedStyle(parentElement),
              borderLeftWidth = parseFloat(style.borderLeftWidth);
            if (borderLeftWidth) {
              globalBounds.left -= borderLeftWidth;
              localBounds.left -= borderLeftWidth;
            }
            return [element.id, {
              element,
              globalBounds,
              localBounds,
              depth,
              parentElement
            }];
          }));
        action.call(thisObj);
        const afterElements = Array.from(outerElement.querySelectorAll(selector)),
          afterMap = new Map(afterElements.map(element => {
            const globalBounds = Rectangle.from(element, outerElement),
              localBounds = Rectangle.from(element, element.parentElement),
              style = globalThis.getComputedStyle(element.parentElement),
              borderLeftWidth = parseFloat(style.borderLeftWidth);
            if (borderLeftWidth) {
              globalBounds.left -= borderLeftWidth;
              localBounds.left -= borderLeftWidth;
            }
            return [element.id, {
              element,
              globalBounds,
              localBounds
            }];
          })),
          styleProps = ['position', 'top', 'left', 'width', 'height', 'padding', 'margin', 'zIndex', 'minWidth', 'minHeight', 'opacity', 'overflow'];
        // Convert to absolute layout, iterating elements remaining after action
        for (const [id, before] of beforeMap) {
          // We match before vs after on id and not actual element, allowing adding a new element with the same id to
          // transition from the old (which was removed or released). To match what will happen when DomSyncing with
          // multiple containing elements (columns in TaskBoard)
          const after = afterMap.get(id);
          if (after) {
            const {
                element
              } = after,
              {
                style,
                parentElement
              } = element,
              // Need to keep explicit zIndex to keep above other stuff
              zIndex = parseInt(DH$1.getStyleValue(element, 'zIndex')),
              {
                globalBounds,
                localBounds,
                depth,
                parentElement: beforeParent
              } = before,
              parentChanged = beforeParent !== parentElement;
            // Store initial state, in case element has a style prop we need to restore later
            ObjectHelper.copyProperties(element.$initial = {
              parentElement
            }, style, styleProps);
            // Prevent transition during the process, forced further down instead
            // element.remove();
            let bounds;
            // Action moved element to another parent, move it to the outer element to allow transitioning to the
            // new parent. Also use coordinates relative to that element
            if (parentChanged) {
              after.bounds = after.globalBounds;
              bounds = globalBounds;
              outerElement.appendChild(element);
            }
            // Keep element in current parent if it was not moved during the action call above.
            // Need to use coords relative to the parent
            else {
              after.bounds = after.localBounds;
              bounds = localBounds;
              beforeParent.appendChild(element);
            }
            let overflow = 'hidden'; // Looks weird with content sticking out if height is transitioned
            if (scrollers.has(element)) {
              element.$scrollPlaceholder = DH$1.createElement({
                parent: element,
                style: {
                  height: element.scrollHeight
                }
              });
              overflow = 'auto';
            }
            const targetStyle = {
              position: 'absolute',
              top: `${bounds.top}px`,
              left: `${bounds.left}px`,
              width: `${bounds.width}px`,
              height: `${bounds.height}px`,
              minWidth: 0,
              minHeight: 0,
              margin: 0,
              zIndex: depth + (zIndex || 0),
              overflow
            };
            if (element.dataset.domTransition !== 'preserve-padding') {
              targetStyle.padding = 0;
            }
            // Move element back to where it started
            Object.assign(style, targetStyle);
            after.processed = true;
          }
          // Existed before but not after = removed
          else {
            const {
              element,
              localBounds: bounds,
              depth,
              parentElement
            } = before;
            element.$initial = {
              removed: true
            };
            Object.assign(element.style, {
              position: 'absolute',
              top: `${bounds.top}px`,
              left: `${bounds.left}px`,
              width: `${bounds.width}px`,
              height: `${bounds.height}px`,
              minWidth: 0,
              minHeight: 0,
              padding: 0,
              margin: 0,
              zIndex: depth,
              overflow: 'hidden' // Looks weird with content sticking out if height is transitioned
            });

            parentElement.appendChild(element);
            // Inject among non-removed elements to have it transition away
            afterMap.set(id, {
              element,
              bounds,
              removed: true,
              processed: true
            });
            afterElements.push(element);
          }
        }
        // Handle new elements
        for (const [, after] of afterMap) {
          if (!after.processed) {
            const {
                element
              } = after,
              {
                style,
                parentElement
              } = element,
              bounds = after.bounds = after.localBounds;
            element.classList.add('b-dom-transition-adding');
            ObjectHelper.copyProperties(element.$initial = {
              parentElement
            }, style, styleProps);
            // Props in `addTransition` will be transitioned
            Object.assign(style, {
              position: 'absolute',
              top: addTransition.top ? 0 : `${bounds.top}px`,
              left: addTransition.left ? 0 : `${bounds.left}px`,
              width: addTransition.width ? 0 : `${bounds.width}px`,
              height: addTransition.height ? 0 : `${bounds.height}px`,
              opacity: addTransition.opacity ? 0 : null,
              zIndex: parentElement.$depth + 1,
              overflow: 'hidden' // Looks weird with content sticking out if height is transitioned
            });
          }
        }
        // Restore scroll after modifying layout
        for (const element of scrollers) {
          element.scrollTop = element.$scrollTop;
        }
        // Enable transitions
        outerElement.classList.add('b-dom-transition');
        // Trigger layout, to be able to transition below
        outerElement.firstElementChild.offsetWidth;
        // Transition to new layout
        for (const [, {
          element,
          bounds: afterBounds,
          removed
        }] of afterMap) {
          if (removed) {
            Object.assign(element.style, {
              top: removeTransition.top ? 0 : `${afterBounds.top}px`,
              left: removeTransition.left ? 0 : `${afterBounds.left}px`,
              width: removeTransition.width ? 0 : `${afterBounds.width}px`,
              height: removeTransition.height ? 0 : `${afterBounds.height}px`,
              opacity: removeTransition.opacity ? 0 : element.$initial.opacity
            });
          } else {
            Object.assign(element.style, {
              top: `${afterBounds.top}px`,
              left: `${afterBounds.left}px`,
              width: `${afterBounds.width}px`,
              height: `${afterBounds.height}px`,
              opacity: element.$initial.opacity
            });
          }
        }
        // Wait for transition to finish
        await AsyncHelper.sleep(duration);
        outerElement.classList.remove('b-dom-transition');
        // Restore layout after transition
        for (const element of afterElements) {
          if (element.$initial) {
            if (element.$initial.removed) {
              element.remove();
            } else {
              ObjectHelper.copyProperties(element.style, element.$initial, styleProps);
              if (element.$scrollPlaceholder) {
                element.$scrollPlaceholder.remove();
                delete element.$scrollPlaceholder;
              }
              element.classList.remove('b-dom-transition-adding');
              element.$initial.parentElement.appendChild(element);
            }
          }
        }
        // Restore scroll positions last when all layout is restored
        for (const element of scrollers) {
          element.scrollTop = element.$scrollTop;
          delete element.$scrollTop;
        }
      }
      //endregion
      static async loadScript(url) {
        return new Promise((resolve, reject) => {
          const script = document.createElement('script');
          script.src = url;
          script.onload = resolve;
          script.onerror = reject;
          document.head.appendChild(script);
        });
      }
      static isNamedColor(color) {
        return color && !/^(#|hsl|rgb|hwb)/.test(color);
      }
      //#region Salesforce hooks
      // Wrap NodeFilter to support salesforce
      static get NodeFilter() {
        return NodeFilter;
      }
      static addChild(parent, child, sibling) {
        parent.insertBefore(child, sibling);
      }
      static cloneStylesIntoShadowRoot(shadowRoot, removeExisting) {
        return new Promise((resolve, reject) => {
          if (removeExisting) {
            // Removes all style-tags or stylesheet link tags from shadowRoot
            shadowRoot.querySelectorAll('style, link[rel="stylesheet"]').forEach(el => el.remove());
          }
          // Clones all stylesheet link tags from document into shadowRoot and waits for them to load
          const links = document.querySelectorAll('link[rel="stylesheet"]');
          let loadCount = 0;
          links.forEach(node => {
            const clone = node.cloneNode();
            clone.addEventListener('load', () => {
              loadCount += 1;
              if (loadCount === links.length) {
                resolve();
              }
            });
            clone.addEventListener('error', e => {
              reject(clone.href);
            });
            shadowRoot.appendChild(clone);
          });
          // Clones all style tags from document into shadowRoot
          document.querySelectorAll('style').forEach(node => {
            shadowRoot.appendChild(node.cloneNode());
          });
        });
      }
      //#endregion
    }

    const DH$1 = DomHelper;
    let clearTouchTimer;
    const clearTouchEvent = () => DH$1.isTouchEvent = false,
      setTouchEvent = () => {
        DH$1.isTouchEvent = true;
        // Jump round the click delay
        clearTimeout(clearTouchTimer);
        clearTouchTimer = setTimeout(clearTouchEvent, 400);
      };
    // Set event type flags so that mousedown and click handlers can know whether a touch gesture was used.
    // This is used. This must stay until we have a unified DOM event system which handles both touch and mouse events.
    doc.addEventListener('touchstart', setTouchEvent, true);
    doc.addEventListener('touchend', setTouchEvent, true);
    DH$1.canonicalStyles = canonicalStyles;
    DH$1.supportsTemplate = 'content' in doc.createElement('template');
    DH$1.elementPropKey = elementPropKey;
    DH$1.numberRe = numberRe;
    //region Polyfills
    if (!('children' in Node.prototype)) {
      const elementFilter = node => node.nodeType === node.ELEMENT_NODE;
      Object.defineProperty(Node.prototype, 'children', {
        get: function () {
          return Array.prototype.filter.call(this.childNodes, elementFilter);
        }
      });
    }
    if (!Element.prototype.matches) {
      Element.prototype.matches = Element.prototype.matchesSelector || Element.prototype.mozMatchesSelector || Element.prototype.msMatchesSelector || Element.prototype.oMatchesSelector || Element.prototype.webkitMatchesSelector || function (s) {
        const matches = (this.document || this.ownerDocument).querySelectorAll(s);
        let i = matches.length;
        while (--i >= 0 && matches.item(i) !== this) {/* empty */}
        return i > -1;
      };
    }
    if (win.Element && !Element.prototype.closest) {
      Node.prototype.closest = Element.prototype.closest = function (s) {
        let el = this;
        if (!doc.documentElement.contains(el)) return null;
        do {
          if (el.matches(s)) return el;
          el = el.parentElement || el.parentNode;
        } while (el !== null && el.nodeType === el.ELEMENT_NODE);
        return null;
      };
    } else {
      // It's crazy that closest is not already on the Node interface!
      // Note that some Node types (eg DocumentFragment) do not have a parentNode.
      Node.prototype.closest = function (selector) {
        var _this$parentNode;
        return (_this$parentNode = this.parentNode) === null || _this$parentNode === void 0 ? void 0 : _this$parentNode.closest(selector);
      };
    }
    // from MDN (public domain): https://developer.mozilla.org/en-US/docs/Web/API/ChildNode/remove
    (function (arr) {
      arr.forEach(function (item) {
        if (Object.prototype.hasOwnProperty.call(item, 'remove')) {
          return;
        }
        Object.defineProperty(item, 'remove', {
          configurable: true,
          enumerable: true,
          writable: true,
          value: function remove() {
            this.parentNode && this.parentNode.removeChild(this);
          }
        });
      });
    })([Element.prototype, CharacterData.prototype, DocumentType.prototype]);
    //endregion
    // CTRL/+ and  CTRL/- zoom gestures must invalidate the scrollbar width.
    // Window resize is triggered by this operation on Blink (Chrome & Edge), Firefox and Safari.
    globalThis.addEventListener('resize', () => scrollBarWidth = null);
    DomHelper._$name = 'DomHelper';

    /**
     * @module Core/GlobalEvents
     */
    const // Allow an unsteady finger to jiggle by this much
      longpressMoveThreshold = 5,
      isFloatingWidget = w => w.floating,
      longPressCancelEvents = {
        touchend: 1,
        pointerup: 1
      },
      ignoreModifierKeys = {
        Meta: 1,
        Control: 1,
        Alt: 1
      },
      GlobalEvents = new class GlobalEventsHandler extends Base$1.mixin(Events) {
        suspendFocusEvents() {
          focusEventsSuspended = true;
        }
        resumeFocusEvents() {
          focusEventsSuspended = false;
        }
        setupFocusListenersOnce(rootElement, EventHelper) {
          if (rootElement && !GlobalEvents.observedElements.has(rootElement)) {
            GlobalEvents.setupFocusListeners(rootElement, EventHelper);
            GlobalEvents.observedElements.add(rootElement);
          }
        }
        // This is imported by EventHelper and that makes the call to set up the listeners
        // `detach` argument is required to not setup more listeners than we need to. In case of salesforce we include floatroot
        // inside the webcomponent element and thus don't need default listeners on document. In regular webcomponents demo we
        // don't need to do it, because with multiple components on one page that would force us to make more complex lookups.
        setupFocusListeners(element = document, EventHelper, detach = false) {
          var _detacher;
          const listeners = {
            element,
            touchstart(touchstart) {
              if (!globaltouchStart && touchstart.changedTouches.length === 1) {
                globaltouchStart = touchstart.changedTouches[0];
                if (!BrowserHelper.isAndroid) {
                  // On single touch start, set up a timer so that a longpress results in a
                  // synthesized contextmenu event being injected into the target element.
                  const
                    // This is what gets called if the user moves their touchpoint,
                    // or releases the touch before <longPressTime>ms is up
                    onMoveOrPointerUp = ({
                      clientX,
                      clientY,
                      type
                    }) => {
                      if (longPressCancelEvents[type] || Math.max(Math.abs(clientX - globaltouchStart.clientX), Math.abs(clientY - globaltouchStart.clientY)) > longpressMoveThreshold) {
                        contextMenuTouchId = null;
                        touchMoveRemover();
                        clearTimeout(tapholdTimer);
                      }
                    },
                    // Touchmove or touchend before that timer fires cancels the timer and removes these listeners.
                    touchMoveRemover = EventHelper.on({
                      element: document,
                      touchmove: onMoveOrPointerUp,
                      touchend: onMoveOrPointerUp,
                      pointermove: onMoveOrPointerUp,
                      pointerup: onMoveOrPointerUp,
                      capture: true
                    }),
                    tapholdTimer = setTimeout(() => {
                      // The global touchend listener prevents a touchend from proceeding
                      // to a click if it was used to trigger contextmenu
                      contextMenuTouchId = globaltouchStart.identifier;
                      // Remove the gesture cancelling listeners
                      touchMoveRemover();
                      touchstart.target.dispatchEvent(new MouseEvent('contextmenu', EventHelper.copyEvent({}, touchstart)));
                    }, EventHelper.longPressTime);
                }
              } else {
                globaltouchStart = null;
              }
            },
            // Just this one has to be passive: false so that we are allowed to preventDefault
            // if we are part of a contextmenu longpress emulation. Otherwise the gesture will
            // proceed to cause a mousedown event.
            touchend: {
              handler: event => {
                if (globaltouchStart) {
                  // If the touchstart was used to synthesize a contextmenu event
                  // stop the touch gesture processing right now.
                  // Also prevent the conversion of the touch into click.
                  if (event.changedTouches[0].identifier === contextMenuTouchId) {
                    event.stopImmediatePropagation();
                    if (event.cancelable !== false) {
                      event.preventDefault();
                    }
                  } else if (event.changedTouches.length === 1 && event.changedTouches[0].identifier === globaltouchStart.identifier) {
                    GlobalEvents.trigger('globaltap', {
                      event
                    });
                  }
                  globaltouchStart = null;
                }
              },
              passive: false
            },
            mousedown: {
              handler: event => {
                lastInteractionType = 'mouse';
                if (!globaltouchStart) {
                  GlobalEvents.trigger('globaltap', {
                    event
                  });
                }
                currentMouseDown = event;
                // If no keydown is catched, and the mouse down has modifier key. Add a fake key event.
                const hasModifierKey = event.ctrlKey || event.altKey || event.shiftKey || event.metaKey;
                if (!currentKeyDown && hasModifierKey) {
                  currentKeyDown = new KeyboardEvent('keydown', {
                    key: event.ctrlKey ? 'Control' : event.shiftKey ? 'Shift' : event.altKey ? 'Alt' : 'Meta',
                    ctrlKey: event.ctrlKey,
                    altKey: event.altKey,
                    shiftKey: event.shiftKey,
                    metaKey: event.metaKey
                  });
                } else if (currentKeyDown && !hasModifierKey) {
                  currentKeyDown = null;
                }
              },
              passive: false
            },
            mouseup() {
              currentMouseDown = null;
            },
            pointerdown: {
              passive: false,
              handler: event => {
                var _element$classList;
                currentPointerDown = event;
                // Remove keyboard flag if last user action used pointer
                // Moved this from EventHelper, need to be once per root to support nested widgets
                DomHelper.usingKeyboard = false;
                (_element$classList = element.classList) === null || _element$classList === void 0 ? void 0 : _element$classList.remove('b-using-keyboard');
                // if shadow root, remove from children (shadow root itself lacks a classList :( )
                if (element.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
                  DomHelper.removeClsGlobally(element, 'b-using-keyboard');
                }
              }
            },
            pointerup: {
              passive: false,
              handler: event => {
                var _currentPointerDown;
                if (((_currentPointerDown = currentPointerDown) === null || _currentPointerDown === void 0 ? void 0 : _currentPointerDown.pointerId) === event.pointerId) {
                  currentPointerDown = null;
                }
              }
            },
            keydown(ev) {
              lastInteractionType = 'key';
              currentKeyDown = ev;
              // Flag root if last user action used keyboard, used for focus styling etc.
              // Moved this from EventHelper, need to be once per root to support nested widgets
              if (!ignoreModifierKeys[ev.key]) {
                DomHelper.usingKeyboard = true;
                // if shadow root, add to outer children (shadow root itself lacks a classList :( )
                if (element.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
                  for (const node of element.children) {
                    if (node.matches('.b-outer')) {
                      node.classList.add('b-using-keyboard');
                    }
                  }
                } else {
                  element.classList.add('b-using-keyboard');
                }
              }
            },
            keypress() {
              lastInteractionType = 'key';
            },
            keyup() {
              currentKeyDown = null;
            },
            focusin(focusin) {
              var _focusin$target, _focusin$target2;
              const {
                Widget
              } = GlobalEvents;
              // Ignore if focus is going into a shadow root
              if ((_focusin$target = focusin.target) !== null && _focusin$target !== void 0 && _focusin$target.shadowRoot || (_focusin$target2 = focusin.target) !== null && _focusin$target2 !== void 0 && _focusin$target2._shadowRoot) {
                return;
              }
              Widget.resetFloatRootScroll();
              if (focusEventsSuspended) {
                return;
              }
              const fromElement = !focusin.relatedTarget ? null : focusin.relatedTarget instanceof HTMLElement ? focusin.relatedTarget : document.body,
                toElement = focusin.target || document.body,
                fromWidget = Widget.fromElement(fromElement),
                toWidget = Widget.fromElement(toElement),
                commonAncestor = DomHelper.getCommonAncestor(fromWidget, toWidget),
                // Flag if the fromElement is DOCUMENT_POSITION_FOLLOWING toElement
                backwards = !!(fromElement && toElement.compareDocumentPosition(fromElement) & 4),
                topVisibleModal = Widget.query(isTopVisibleModal);
              let currentFocus = null;
              if (toElement && toElement !== document.body) {
                currentFocus = DomHelper.getActiveElement(toElement);
              } else {
                currentFocus = DomHelper.getActiveElement(document);
              }
              // If there is a topmost modal that is not actively reverting focus, and the focus is moving to
              // somewhere *not* a descendant of that modal, and that somewhere is not in a floater that us
              // *above* that modal (the compareDocumentPosition call), then we enforce modality and sweep focus
              // back into the modal.
              // By default, the Container class will yield the first focusable descendant widget's focusEl as its
              // focusEl, so that will be out of the box behaviour for Popups.
              if (topVisibleModal && !topVisibleModal._isRevertingFocus) {
                if (!toWidget || !topVisibleModal.owns(toWidget) && !(topVisibleModal.element.compareDocumentPosition(toWidget.element) & 4 && toWidget.up(isFloatingWidget))) {
                  return topVisibleModal.focus();
                }
              }
              let event = createWidgetEvent('focusout', fromElement, focusin.target, fromWidget, toWidget, backwards);
              // Bubble focusout event up the "from" side of the tree
              for (let target = fromWidget, owner; target && target !== commonAncestor; target = owner) {
                // Capture before any focus out handling is done. It may be manipulated.
                owner = target.owner;
                if (!target.isDestroying && target.onFocusOut) {
                  target.onFocusOut(event);
                  // It is possible for focusout handlers to refocus themselves (editor's invalidAction='block'), so
                  // check if the focus is still where it was when we started unless we are in a document
                  // loss of focus situation (no target)
                  if (focusin.target && currentFocus !== DomHelper.getActiveElement(focusin.target)) {
                    // If the focus has moved, that movement would have kicked off a nested sequence of focusin/out
                    // notifications, so everyone has already been notified... no more to do here.
                    return;
                  }
                }
              }
              // Focus is moving upwards to the ancestor widget.
              // Its focus method might delegate focus to a focusable descendant.
              if (commonAncestor && focusin.target === commonAncestor.element) {
                // If one of the handlers above has not moved focus onwards
                // and the common ancestor is a container which delegates
                // focus inwards to a descendant, then give it the opportunity to do that.
                if (!commonAncestor.isDestroying && DomHelper.getActiveElement(commonAncestor) === toElement && commonAncestor.focusElement && commonAncestor.focusElement !== commonAncestor.element) {
                  // If focus is not inside, move focus inside
                  if (!commonAncestor.element.contains(currentFocus) || commonAncestor.focusDescendant) {
                    // Wait until out of the focusin handler to move focus on.
                    commonAncestor.setTimeout(() => {
                      var _commonAncestor$focus;
                      return (_commonAncestor$focus = commonAncestor.focus) === null || _commonAncestor$focus === void 0 ? void 0 : _commonAncestor$focus.call(commonAncestor);
                    }, 0);
                  }
                }
              }
              // Focus is moving between two branches of a subtree.
              // Bubble focusin event up the "to" side of the tree
              else {
                event = createWidgetEvent('focusin', toElement, fromElement, fromWidget, toWidget, backwards);
                for (let target = toWidget; target && target !== commonAncestor; target = target.owner) {
                  if (!target.isDestroying) {
                    var _target$onFocusIn, _target;
                    (_target$onFocusIn = (_target = target).onFocusIn) === null || _target$onFocusIn === void 0 ? void 0 : _target$onFocusIn.call(_target, event);
                  }
                }
              }
              // Fire element focusmove event. Grid navigation will use  this when cells are focusable.
              const commonAncestorEl = DomHelper.getCommonAncestor((fromElement === null || fromElement === void 0 ? void 0 : fromElement.nodeType) === Element.ELEMENT_NODE ? fromElement : null, toElement) || toElement.parentNode;
              // Common ancestor may be null is salesforce
              // https://github.com/bryntum/support/issues/4865
              if (commonAncestorEl) {
                event = createWidgetEvent('focusmove', toElement, fromElement, fromWidget, toWidget, backwards, {
                  bubbles: true
                });
                commonAncestorEl.dispatchEvent(event);
              }
            },
            focusout(focusout) {
              if (focusEventsSuspended) {
                return;
              }
              if (!focusout.relatedTarget || !GlobalEvents.Widget.fromElement(focusout.relatedTarget)) {
                // When switching between tabs in Salesforce app `relatedTarget` of the focusout event might be not an instance of
                // HTMLElement.
                const target = focusout.relatedTarget && focusout.relatedTarget instanceof HTMLElement ? focusout.relatedTarget : null;
                listeners.focusin({
                  target,
                  relatedTarget: focusout.target
                });
                currentKeyDown = currentMouseDown = null;
              }
            },
            // This will clear keydown and mousedown status on window blur
            blur: {
              element: globalThis,
              handler(event) {
                if (event.target === globalThis) {
                  currentKeyDown = null;
                  currentMouseDown = null;
                }
              }
            },
            capture: true,
            passive: true
          };
          // detach previous listeners
          detach && ((_detacher = detacher) === null || _detacher === void 0 ? void 0 : _detacher());
          detacher = this.detachEvents = EventHelper.on(listeners);
        }
        get lastInteractionType() {
          return lastInteractionType;
        }
        get shiftKeyDown() {
          var _currentKeyDown;
          return (_currentKeyDown = currentKeyDown) === null || _currentKeyDown === void 0 ? void 0 : _currentKeyDown.shiftKey;
        }
        get ctrlKeyDown() {
          var _currentKeyDown2, _currentKeyDown3;
          return ((_currentKeyDown2 = currentKeyDown) === null || _currentKeyDown2 === void 0 ? void 0 : _currentKeyDown2.ctrlKey) || ((_currentKeyDown3 = currentKeyDown) === null || _currentKeyDown3 === void 0 ? void 0 : _currentKeyDown3.metaKey);
        }
        get altKeyDown() {
          var _currentKeyDown4;
          return (_currentKeyDown4 = currentKeyDown) === null || _currentKeyDown4 === void 0 ? void 0 : _currentKeyDown4.altKey;
        }
        isKeyDown(key) {
          var _currentKeyDown5;
          return !key ? Boolean(currentKeyDown) : ((_currentKeyDown5 = currentKeyDown) === null || _currentKeyDown5 === void 0 ? void 0 : _currentKeyDown5.key) === key || currentKeyDown[(key === null || key === void 0 ? void 0 : key.toLowerCase()) + 'Key'] === true;
        }
        isMouseDown(button = 0) {
          var _currentMouseDown;
          return ((_currentMouseDown = currentMouseDown) === null || _currentMouseDown === void 0 ? void 0 : _currentMouseDown.button) === button;
        }
        get currentMouseDown() {
          return currentMouseDown;
        }
        get currentPointerDown() {
          return currentPointerDown;
        }
        get currentTouch() {
          return globaltouchStart;
        }
        get currentKeyDown() {
          return currentKeyDown;
        }
      }(),
      isTopVisibleModal = w => w.isVisible && w.isTopModal;
    GlobalEvents.observedElements = new Set();
    /**
     * Fired after the theme is changed
     * @event theme
     * @param {Core.GlobalEvents} source
     * @param {String} theme The new theme name
     */
    let globaltouchStart,
      contextMenuTouchId,
      focusEventsSuspended = false,
      lastInteractionType,
      currentKeyDown,
      currentMouseDown,
      currentPointerDown,
      detacher;
    function createWidgetEvent(eventName, target, relatedTarget, fromWidget, toWidget, backwards, options) {
      const result = new CustomEvent(eventName, options);
      // Workaround for Salesforce. They use strict mode and define non-configurable property `target`. We use this
      // CustomEvent as a synthetic one, feels fine to use non-standard handle for target.
      Object.defineProperty(result, '_target', {
        get() {
          return target;
        }
      });
      Object.defineProperty(result, 'relatedTarget', {
        get() {
          return relatedTarget;
        }
      });
      result.fromWidget = fromWidget;
      result.toWidget = toWidget;
      result.backwards = backwards;
      return result;
    }

    /**
     * @module Core/mixin/InstancePlugin
     */
    function getDescriptor(me, fnName) {
      const property = ObjectHelper.getPropertyDescriptor(me, fnName); //Object.getOwnPropertyDescriptor(Object.getPrototypeOf(me), fnName);
      return property && (property.get || property.set) ? property : null;
    }
    /**
     * Base class for plugins. Published functions will be available from the other class. `this` in published functions is
     * referenced to the plugin, access the other class using `this.client`.
     *
     * Observe that plugin doesn't apply itself on class level but instead on instance level. Plugin is its own instance
     * that can have own functions and data that is not exposed to target class.
     *
     * Functions can be published in four ways:
     *
     * * `assign` (when function is not already available on target)
     * * `before` (when function is already available on target, will be called before original function)
     * * `after` (when function is already available on target, will be called after original function)
     * * `override` (replaces function on target, but old function can be reached)
     *
     * To configure which functions get published and in what way, specify `pluginConfig` getter on plugin:
     *
     * ```javascript
     * class Sort extends InstancePlugin {
     *   static get pluginConfig {
     *      return {
     *          before   : ['init'],
     *          after    : ['destroy', 'onElementClick'],
     *          override : ['render']
     *      };
     *   }
     * }
     * ```
     *
     * @extends Core/Base
     * @mixes Core/localization/Localizable
     * @mixes Core/mixin/Events
     * @plugin
     */
    class InstancePlugin extends Base$1.mixin(Events, Localizable) {
      static $name = 'InstancePlugin';
      //region Config
      static get configurable() {
        return {
          clientListeners: null,
          /**
           * The plugin/feature `disabled` state.
           *
           * For a feature that is **off** by default that you want to enable later during runtime,
           * configure it with `disabled : true`.
           * ```javascript
           * const grid = new Grid({
           *      features : {
           *          featureName : {
           *              disabled : true // on and disabled, can be enabled later
           *          }
           *      }
           * });
           *
           * // enable the feature
           * grid.features.featureName.disabled = false;
           * ```
           *
           * If the feature is **off** by default, and you want to include and enable the feature, configure it as `true`:
           * ```javascript
           * const grid = new Grid({
           *      features : {
           *          featureName : true // on and enabled, can be disabled later
           *      }
           * });
           *
           * // disable the feature
           * grid.features.featureName.disabled = true;
           * ```
           *
           * If the feature is **on** by default, but you want to turn it **off**, configure it as `false`:
           * ```javascript
           * const grid = new Grid({
           *      features : {
           *          featureName : false // turned off, not included at all
           *      }
           * });
           * ```
           *
           * If the feature is **enabled** by default and you have no need of reconfiguring it,
           * you can omit the feature configuration.
           *
           * @prp {Boolean}
           * @default
           * @category Common
           */
          disabled: false,
          /**
           * The Widget which was passed into the constructor,
           * which is the Widget we are providing extra services for.
           * @member {Core.widget.Widget} client
           * @readonly
           * @category Misc
           * @advanced
           */
          /**
           * The widget which this plugin is to attach to.
           * @config {Core.widget.Widget}
           * @category Misc
           * @advanced
           */
          client: null,
          /**
           * @hideconfigs bubbleEvents, callOnFunctions
           */
          /**
           * @hidefunctions downloadTestCase, destroy
           */
          /**
           * @hideproperties isDestroyed
           */
          /**
           * @hideevents destroy, beforeDestroy
           */
          // The plugins can define their own keyMap which will then be merged with their client's keyMap.
          keyMap: null
        };
      }
      //endregion
      updateClient(client) {
        // So that this.callback can reach the owning Widget when resolving function names.
        if (!this.owner) {
          this.owner = client;
        }
      }
      /**
       * This will merge a feature's (subclass of InstancePlugin) keyMap with it's client's keyMap.
       * @private
       */
      updateKeyMap(keyMap) {
        const {
          client
        } = this;
        client.keyMap = client.mergeKeyMaps(client.keyMap, keyMap, StringHelper.uncapitalize(this.constructor.$name));
      }
      //region Init
      /**
       * Call from another instance to add plugins to it.
       * @example
       * InstancePlugin.initPlugins(this, Search, Stripe);
       * @param plugInto Instance to mix into (usually this)
       * @param plugins Classes to plug in
       * @internal
       */
      static initPlugins(plugInto, ...plugins) {
        const property = plugInto.plugins || (plugInto.plugins = {});
        for (const PluginClass of plugins) {
          property[PluginClass.$$name] = new PluginClass(plugInto);
        }
      }
      /**
       * Simple wrapper for {@link #property-disabled} to make optional chaining simple:
       *
       * ```javascript
       * grid.features.myFeature?.enabled // returns true when feature exists and is enabled
       * ```
       * @returns {Boolean}
       * @internal
       */
      get enabled() {
        return !this.disabled;
      }
      // We can act as an owner of a widget, so must be able to participate in focus reversion
      getFocusRevertTarget() {
        var _this$client;
        return (_this$client = this.client) === null || _this$client === void 0 ? void 0 : _this$client.getFocusRevertTarget();
      }
      construct(...args) {
        const me = this;
        let [plugInto, config] = args,
          listeners;
        // When called with one argument (a config object), grab the "client" from the config object.
        if (args.length === 1) {
          if (ObjectHelper.isObject(plugInto)) {
            config = plugInto;
            plugInto = config.client;
          }
        }
        // Two args, so client is the first. Ensure the config doesn't contain a client property.
        else {
          config = ObjectHelper.assign({}, config);
          delete config.client;
        }
        me.client = plugInto;
        super.construct(config);
        me.applyPluginConfig(plugInto);
        listeners = me.clientListeners;
        if (listeners) {
          listeners = ObjectHelper.assign({}, listeners);
          listeners.thisObj = me;
          // NOTE: If clientListeners are ever made public, we need to separate internal clientListeners from app ones
          plugInto.ion(listeners);
        }
      }
      /**
       * Applies config as found in plugInto.pluginConfig, or published all if no config found.
       * @private
       * @param plugInto Target instance to plug into
       */
      applyPluginConfig(plugInto) {
        const me = this,
          config = me.pluginConfig || me.constructor.pluginConfig;
        if (config) {
          const {
            assign,
            chain,
            after,
            before,
            override
          } = config;
          assign && me.applyAssign(plugInto, assign);
          (chain || after) && me.applyChain(plugInto, chain || after);
          before && me.applyChain(plugInto, before, false);
          override && me.applyOverride(plugInto, override);
        }
      }
      /**
       * Applies assigning for specified functions.
       * @private
       * @param plugInto
       * @param fnNames
       */
      applyAssign(plugInto, fnNames) {
        fnNames.forEach(fnName => this.assign(plugInto, fnName));
      }
      /**
       * Applies chaining for specified functions.
       * @private
       * @param plugInto
       * @param functions
       * @param after
       */
      applyChain(plugInto, functions, after = true) {
        if (Array.isArray(functions)) {
          for (const fnName of functions) {
            this.chain(plugInto, fnName, fnName, after);
          }
        } else {
          for (const intoName in functions) {
            this.chain(plugInto, intoName, functions[intoName], after);
          }
        }
      }
      /**
       * Applies override for specified functions.
       * @private
       * @param plugInto
       * @param fnNames
       */
      applyOverride(plugInto, fnNames) {
        const me = this;
        if (!me.overridden) {
          me.overridden = {};
        }
        fnNames.forEach(fnName => {
          if (!me[fnName]) {
            throw new Error(`Trying to chain fn ${plugInto.$$name}#${fnName}, but plugin fn ${me.$$name}#${fnName} does not exist`);
          }
          // override
          if (typeof plugInto[fnName] === 'function') {
            me.overridden[fnName] = plugInto[fnName].bind(plugInto);
          }
          plugInto[fnName] = me[fnName].bind(me);
        });
      }
      /**
       * Assigns specified functions.
       * @private
       * @param plugInto
       * @param fnName
       */
      assign(plugInto, fnName) {
        const me = this,
          property = getDescriptor(me, fnName);
        if (property) {
          // getter/setter, define corresponding property on target
          Object.defineProperty(plugInto, fnName, {
            configurable: true,
            enumerable: true,
            get: property.get && property.get.bind(me),
            set: property.set && property.set.bind(me)
          });
        } else {
          plugInto[fnName] = me[fnName].bind(me);
        }
      }
      //endregion
      //region Chaining
      /**
       * Chains functions. When the function is called on the target class all functions in the chain will be called in
       * the order they where added.
       * @private
       * @param plugInto
       * @param intoName
       * @param hookName
       * @param after
       */
      chain(plugInto, intoName, hookName, after = true) {
        // default hook prio
        let prio = 0;
        if (typeof intoName === 'object') {
          intoName = intoName.fn;
        }
        // if hook is provided as an object
        if (typeof hookName === 'object') {
          // hook prio to order runs
          prio = hookName.prio || 0;
          hookName = hookName.fn;
        }
        const me = this,
          chains = plugInto.pluginFunctionChain || (plugInto.pluginFunctionChain = {}),
          hookFn = me[hookName] && me[hookName].bind(me),
          // Grab the fn so that we won't need our this pointer to call it. This is due
          // to this instance possibly being destroyed by the time a chain call is made.
          functionChainRunner = me.functionChainRunner;
        if (!hookFn) {
          throw new Error(`Trying to chain fn ${plugInto.$$name}#${hookName}, but plugin fn ${me.$$name}#${hookName} does not exist`);
        }
        if (!chains[intoName]) {
          let intoFn = plugInto[intoName];
          if (intoFn) {
            intoFn = intoFn.bind(plugInto);
            intoFn.$this = plugInto;
            // use default prio
            intoFn.$prio = 0;
          }
          chains[intoName] = intoFn ? [intoFn] : [];
          plugInto[intoName] = (...params) => functionChainRunner(chains[intoName], params);
        }
        hookFn.$this = me;
        hookFn.$prio = prio;
        chains[intoName][after ? 'push' : 'unshift'](hookFn);
        chains[intoName].$sorted = false;
      }
      /**
       * Used to run multiple plugged in functions with the same name, see chain above. Returning false from a
       * function will abort chain.
       * @private
       * @param {Array} chain
       * @param {Array} params
       * @returns {*} value returned from last function in chain (or false if any returns false)
       */
      functionChainRunner(chain, params) {
        // NOTE: even though we are an instance method, we must not use our "this" pointer
        // since our instance may be destroyed. We cope with that by receiving parameters
        // for everything we need (so we're just a pure function).
        let fn, i, returnValue;
        // sort hooks by prio before running them
        if (!chain.$sorted) {
          chain.sort((a, b) => b.$prio - a.$prio);
          chain.$sorted = true;
        }
        for (i = 0; i < chain.length; i++) {
          fn = chain[i];
          // Feature hooks remain in place even after GridBase loops and destroys its
          // features, so skip over any destroyed features on the chain. In particular,
          // bindStore hooks will be called when GridBase sets store to null.
          if (!fn.$this.isDestroyed) {
            returnValue = fn(...params);
            if (returnValue === false) {
              break;
            }
          }
        }
        return returnValue;
      }
      //endregion
      /**
       * Called when disabling/enabling the plugin/feature, not intended to be called directly. To enable or disable a
       * plugin/feature, see {@link #property-disabled}.
       *
       * By default removes the cls of the plugin from its client. Override in subclasses to take any other actions necessary.
       * @category Misc
       * @advanced
       */
      doDisable(disable) {
        const me = this,
          {
            constructor
          } = me,
          cls = 'featureClass' in constructor ? constructor.featureClass : `b-${constructor.$$name.toLowerCase()}`,
          key = StringHelper.uncapitalize(constructor.$$name);
        // Some features do not use a cls
        if (cls) {
          var _me$client, _me$client$_element;
          // _element to not flush composable
          (_me$client = me.client) === null || _me$client === void 0 ? void 0 : (_me$client$_element = _me$client._element) === null || _me$client$_element === void 0 ? void 0 : _me$client$_element.classList[disable ? 'remove' : 'add'](cls);
        }
        if (!me.isConfiguring) {
          var _me$client$syncSplits, _me$client2;
          if (disable) {
            /**
             * Fired when the plugin/feature is disabled.
             * @event disable
             * @param {Core.mixin.InstancePlugin} source
             */
            me.trigger('disable');
          } else {
            /**
             * Fired when the plugin/feature is enabled.
             * @event enable
             * @param {Core.mixin.InstancePlugin} source
             */
            me.trigger('enable');
          }
          // Enable/disable in other splits, if any
          (_me$client$syncSplits = (_me$client2 = me.client).syncSplits) === null || _me$client$syncSplits === void 0 ? void 0 : _me$client$syncSplits.call(_me$client2, other => {
            // Might be a plugin that is not a feature (there is no Feature baseclass in Grid)
            const otherFeature = other.features[key];
            if (otherFeature) {
              otherFeature.disabled = disable;
            }
          });
        }
      }
      updateDisabled(disabled) {
        this.doDisable(disabled);
      }
      throwOverrideIsMissing(data) {
        throw new Error(`Trying to override fn ${data.plugIntoName}#${data.fnName}, but plugin fn ${data.pluginName}#${data.fnName} does not exist`);
      }
      // Convenience method to read the rootElement from the owner widget
      get rootElement() {
        return this.client.rootElement;
      }
    }
    InstancePlugin._$name = 'InstancePlugin';

    /**
     * @module Core/mixin/Pluggable
     */
    /**
     * Enables using plugins for a class by specifying property plugins as an array of plugin classes. If only a single plugin
     * is used, just give the plugin class instead of an array. This class isn't required for using plugins, just makes it
     * easier. Without mixin you can otherwise use `InstancePlugin.initPlugins(this, PluginClass)`.
     *
     * @example
     * new Store({
     *   plugins: [PluginClass, ...]
     * });
     *
     * @mixin
     */
    var Pluggable = (Target => class Pluggable extends (Target || Base$1) {
      static get $name() {
        return 'Pluggable';
      }
      /**
       * Specify plugins (an array of classes) in config
       * @config {Function[]} plugins
       * @category Misc
       * @advanced
       */
      /**
       * Map of applied plugins
       * @property {Object<String,Core.mixin.InstancePlugin>}
       * @readonly
       * @category Misc
       * @advanced
       */
      get plugins() {
        if (!this._plugins) {
          this._plugins = {};
        }
        return this._plugins;
      }
      set plugins(plugins) {
        if (plugins) {
          if (!Array.isArray(plugins)) plugins = [plugins];
          InstancePlugin.initPlugins(this, ...plugins);
        }
        this.initPlugins();
      }
      /**
       * Template method which may be implemented in subclasses to initialize any plugins.
       * This method is empty in the `Pluggable` base class.
       * @internal
       */
      initPlugins() {}
      /**
       * Adds plugins to an instance.
       * @param {Function[]} plugins The plugins to add
       * @category Misc
       * @advanced
       */
      addPlugins(...plugins) {
        InstancePlugin.initPlugins(this, ...plugins);
      }
      /**
       * Checks if instance has plugin.
       * @param {String|Function} pluginClassOrName Plugin or name to check for
       * @returns {Boolean}
       * @category Misc
       * @advanced
       */
      hasPlugin(pluginClassOrName) {
        return this.getPlugin(pluginClassOrName) != null;
      }
      /**
       * Get a plugin instance.
       * @param {String|Function} pluginClassOrName
       * @returns {Core.mixin.InstancePlugin}
       * @category Misc
       * @advanced
       */
      getPlugin(pluginClassOrName) {
        var _this$plugins;
        if (typeof pluginClassOrName === 'function') {
          pluginClassOrName = pluginClassOrName.$$name;
        }
        return (_this$plugins = this.plugins) === null || _this$plugins === void 0 ? void 0 : _this$plugins[pluginClassOrName];
      }
      // This does not need a className on Widgets.
      // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
      // to the Widget it's mixed in to should implement thus.
      get widgetClass() {}
    });

    const {
      defineProperty: defineProperty$6
    } = Reflect;
    let performance$1;
    if (BrowserHelper.isBrowserEnv) {
      performance$1 = globalThis.performance;
    } else {
      performance$1 = {
        now() {
          return new Date().getTime();
        }
      };
    }
    /**
     * @module Core/mixin/Delayable
     */
    // Global timeout collections for tests
    let globalDelays = null;
    if (VersionHelper.isTestEnv) {
      const bryntum = globalThis.bryntum || (globalThis.bryntum = {});
      globalDelays = bryntum.globalDelays = {
        timeouts: new Map(),
        intervals: new Map(),
        animationFrames: new Map(),
        idleCallbacks: new Map(),
        isEmpty(includeIntervals = false) {
          return globalDelays.timeouts.size + globalDelays.animationFrames.size + globalDelays.idleCallbacks.size + (includeIntervals ? globalDelays.intervals.size : 0) === 0;
        },
        /**
         * Returns filtered delays array
         * @param {Object} options
         * @param {String[]} [options.ignoreTimeouts] array of delays names to ignore
         * @param {Number} [options.maxDelay] maximum delay in milliseconds. Timeouts with bigger delay will be filtered out
         * @param {Boolean} [options.includeIntervals] include intervals
         * @returns {Object[]} array of filtered delays
         * @internal
         */
        getFiltered({
          ignoreTimeouts = [],
          maxDelay = 5000,
          includeIntervals = false
        }) {
          const result = [],
            scopes = ['timeouts', 'animationFrames', 'idleCallbacks'];
          if (includeIntervals) {
            scopes.push('intervals');
          }
          // Filter delays
          for (const scope of scopes) {
            const map = globalDelays[scope];
            for (const [, entry] of map.entries()) {
              if (!ignoreTimeouts.includes(entry.name) && (!Number.isInteger(entry.delay) || entry.delay < maxDelay)) {
                result.push(entry);
              }
            }
          }
          return result;
        }
      };
    }
    const
      /**
       * Creates and returns a function that will call the user-supplied `fn`.
       *
       * @param {Core.mixin.Delayable} me
       * @param {Function} fn The user function to call when the timer fires.
       * @param {Function} wrapFn The function the user will call to start the timer.
       * @param {Object} options The invoke options.
       * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
       * @param {Object} [options.thisObj] The `this` reference for `fn`.
       * @returns {Function}
       * @private
       */
      makeInvoker = (me, fn, wrapFn, options) => {
        const named = typeof fn === 'string',
          appendArgs = (options === null || options === void 0 ? void 0 : options.appendArgs) || [],
          // The invoker fn is intended to be wired directly to native setTimeout/requestAnimationFrame/etc. and so
          // it does not receive any arguments worth passing on to the user's fn. Those come from the original call
          // to the wrapFn.
          invoker = () => {
            wrapFn.timerId = null;
            wrapFn.lastCallTime = performance$1.now();
            // Grab args now and null the stored args out (to avoid leaks):
            const args = wrapFn.args;
            wrapFn.args = null;
            if (named) {
              me[fn](...args, ...appendArgs);
            } else {
              fn.call(me, ...args, ...appendArgs);
            }
            wrapFn.called = true;
            ++wrapFn.calls;
          };
        if (options) {
          me = options.thisObj || me;
        }
        // We put most everything as properties on the wrapFn so that it can all be inspected in the debugger (unlike
        // closure variables) and expected in tests.
        wrapFn.lastCallTime = -9e9; // performance.now() = 0 at start...
        wrapFn.calls = 0;
        wrapFn.invoker = invoker;
        invoker.wrapFn = wrapFn;
        return invoker;
      },
      /**
       * Decorates the supported `wrapFn` with additional methods and an `isPending` readonly
       * property. These decorations are available to user code to help manage the scheduling
       * behavior of the buffered function.
       *
       * @param {Core.mixin.Delayable} me
       * @param {Function} wrapFn The function the user will call to start the timer.
       * @param {String} cancelFn The name of the function that will cancel a timer.
       * @returns {Function} The `wrapFn` is returned.
       * @private
       */
      decorateWrapFn = (me, wrapFn, cancelFn = 'clearTimeout') => {
        wrapFn.cancel = () => {
          if (wrapFn.isPending) {
            me[cancelFn](wrapFn.timerId);
            // avoid leaks and cleanup:
            wrapFn.args = wrapFn.timerId = null;
          }
        };
        wrapFn.flush = () => {
          if (wrapFn.isPending) {
            me[cancelFn](wrapFn.timerId);
            wrapFn.timerId = null;
            // we don't call cancel() since it also sets args=null
            wrapFn.invoker();
          }
        };
        wrapFn.now = (...args) => {
          wrapFn.cancel();
          wrapFn.args = args;
          wrapFn.invoker();
        };
        wrapFn.resume = all => {
          const n = wrapFn.suspended;
          wrapFn.suspended = all || n < 1 ? 0 : n - 1;
        };
        wrapFn.suspend = () => {
          ++wrapFn.suspended;
        };
        wrapFn.immediate = false;
        wrapFn.suspended = 0;
        wrapFn.timerId = null;
        defineProperty$6(wrapFn, 'isPending', {
          get() {
            return wrapFn.timerId !== null;
          }
        });
        return wrapFn;
      };
    /**
     * Configuration options available when defining a delayable function.
     *
     * @typedef {Object} DelayableConfig
     * @property {'buffer'|'raf'|'idle'|'throttle'} type Type of delay to use. `raf` is short for `requestAnimationFrame`,
     * 'idle' for `requestIdleCallback` (not supported in Safari)
     * @property {Number} [delay] Number of milliseconds to wait before (buffer) or after (throttle) calling the underlying
     * method. A value of 0 is equivalent to setting `immediate: true`.
     * @property {Boolean} [immediate] Set to `true` to call immediately (effectively disabling the buffer/throttle)
     * @property {Boolean} [cancelOutstanding] Set to `true` to cancel any pending animation frame requests and
     * schedule a new one on each call.
     */
    /**
     * Tracks setTimeout, setInterval and requestAnimationFrame calls and clears them on destroy.
     *
     * @example
     * someClass.setTimeout(() => console.log('hi'), 200);
     * someClass.setInterval(() => console.log('annoy'), 100);
     * // can also use named timeouts for easier tracking
     * someClass.setTimeout(() => console.log('named'), 300, 'named');
     * someClass.clearTimeout('named');
     *
     * @mixin
     */
    var Delayable = (Target => class Delayable extends (Target || Base$1) {
      static get $name() {
        return 'Delayable';
      }
      static get declarable() {
        return [
        /**
         * This class property returns an object that specifies methods to wrap with configurable timer behaviors.
         *
         * It is used like so:
         * ```javascript
         *  class Foo extends Base.mixin(Delayable) {
         *      static get delayable() {
         *          return {
         *              expensiveMethod : 500
         *          };
         *      }
         *
         *      expensiveMethod() {
         *          this.things();
         *          this.moreThings();
         *          this.evenMoreThings();
         *      }
         *  }
         * ```
         * With the above in place, consider:
         * ```javascript
         *  let instance = new Foo();
         *
         *  instance.expensiveMethod();
         * ```
         * Instead of the above code immediately calling the `expensiveMethod()`, it will start a timer that will
         * invoke the method 500ms later. Because `expensiveMethod()` is an instance method, each instance of `Foo`
         * will have its own timer.
         *
         * NOTE: Only instance methods are currently supported (i.e., only non-`static` methods).
         *
         * #### Options
         * The value of each key configures how the method will be scheduled. If the value is a number, it is
         * promoted to a config object of `type='buffer'` as in the following:
         * ```javascript
         *  class Foo extends Base.mixin(Delayable) {
         *      static get delayable() {
         *          return {
         *              expensiveMethod : {
         *                  type  : 'buffer',
         *                  delay : 500
         *              }
         *          };
         *      }
         *  }
         * ```
         * The `type` property of the config object must be one of three values. Other options can be provided
         * depending on the `type`:
         *
         *  - `buffer`<br>
         *    Other options:
         *     - `delay` (Number) : The number of milliseconds to wait before calling the underlying method. A
         *       value of 0 is equivalent to setting `immediate: true`.
         *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the buffer).
         *  - `raf` (short for "request animation frame")<br>
         *  - `idle` (short for "request idle callback") __Not available on Safari__ <br>
         *    Other options:
         *     - `cancelOutstanding` (Boolean) : Set to `true` to cancel any pending animation frame requests and
         *       schedule a new one on each call.
         *     - `immediate` (Boolean) : Set to `true` to call immediately.
         *  - `throttle`<br>
         *    Other options:
         *     - `delay` (Number) : The number of milliseconds to wait after each execution before another
         *       execution takes place. A value of 0 is equivalent to setting `immediate: true`.
         *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the throttle).
         *
         * While `immediate: true` can be specified at the class level, it is more typical to set it on the
         * instance's method as described below.
         *
         * #### Delayable Method API
         * Delayable methods have a consistent API to manage their scheduling. This API is added to the methods
         * themselves.
         *
         * For example:
         * ```javascript
         *  let instance = new Foo();
         *
         *  instance.expensiveMethod();         // schedule a call in 500ms
         *  instance.expensiveMethod.isPending; // true
         *  instance.expensiveMethod.cancel();
         *  instance.expensiveMethod.flush();
         *  instance.expensiveMethod.now();
         *
         *  instance.expensiveMethod.delay = 10;
         *  instance.expensiveMethod();         // schedule a call in 10ms
         * ```
         *
         * ##### `isPending` (Boolean, readonly)
         * This boolean property will be `true` if a call has been scheduled, and false otherwise.
         *
         * ##### `cancel()`
         * Cancels a pending call if one has been scheduled. Otherwise this method does nothing.
         *
         * ##### `flush()`
         * Cancels the timer and causes the pending call to execute immediately. If there is no pending call, this
         * method does nothing.
         *
         * ##### `now()`
         * Cancels the timer (if one is pending) and executes the method immediately. If there is no pending call,
         * this method will still call the underlying method.
         *
         * @static
         * @member {Object<String,'raf'|Number|DelayableConfig>} delayable
         * @internal
         */
        'delayable'];
      }
      doDestroy() {
        const me = this;
        // Normally one would expect this call at the end of this method... but in this case we need to run cleanup
        // of this stuff after config nullification since those can trigger delayable method calls.
        super.doDestroy();
        if (me.timeoutIds) {
          me.timeoutIds.forEach((fn, id) => {
            var _globalDelays;
            if (typeof fn === 'function') {
              fn();
            }
            clearTimeout(id);
            (_globalDelays = globalDelays) === null || _globalDelays === void 0 ? void 0 : _globalDelays.timeouts.delete(id);
          });
          me.timeoutIds = null;
        }
        if (me.timeoutMap) {
          me.timeoutMap.forEach((name, id) => clearTimeout(id));
          me.timeoutMap = null;
        }
        if (me.intervalIds) {
          me.intervalIds.forEach(id => {
            var _globalDelays2;
            clearInterval(id);
            (_globalDelays2 = globalDelays) === null || _globalDelays2 === void 0 ? void 0 : _globalDelays2.intervals.delete(id);
          });
          me.intervalIds = null;
        }
        if (me.animationFrameIds) {
          me.animationFrameIds.forEach(id => {
            var _globalDelays3;
            cancelAnimationFrame(id);
            (_globalDelays3 = globalDelays) === null || _globalDelays3 === void 0 ? void 0 : _globalDelays3.animationFrames.delete(id);
          });
          me.animationFrameIds = null;
        }
        if (me.idleCallbackIds) {
          me.idleCallbackIds.forEach(id => {
            var _globalDelays4;
            cancelIdleCallback(id);
            (_globalDelays4 = globalDelays) === null || _globalDelays4 === void 0 ? void 0 : _globalDelays4.idleCallbacks.delete(id);
          });
          me.idleCallbackIds = null;
        }
      }
      /**
       * Check if a named timeout is active
       * @param name
       * @internal
       */
      hasTimeout(name) {
        var _this$timeoutMap;
        return Boolean((_this$timeoutMap = this.timeoutMap) === null || _this$timeoutMap === void 0 ? void 0 : _this$timeoutMap.has(name));
      }
      /**
       * Same as native setTimeout, but will be cleared automatically on destroy. If a propertyName is supplied it will
       * be used to store the timeout id.
       * @param {Object} timeoutSpec An object containing the details about that function, and the time delay.
       * @param {Function|String} timeoutSpec.fn The function to call, or name of function in this object to call. Used as the `name` parameter if a string.
       * @param {Number} timeoutSpec.delay The milliseconds to delay the call by.
       * @param {Object[]} timeoutSpec.args The arguments to pass.
       * @param {String} [timeoutSpec.name] The name under which to register the timer. Defaults to `fn.name`.
       * @param {Boolean} [timeoutSpec.runOnDestroy] Pass `true` if this function should be executed if the Delayable instance is destroyed while function is scheduled.
       * @param {Boolean} [timeoutSpec.cancelOutstanding] Pass `true` to cancel any outstanding invocation of the passed function.
       * @returns {Number}
       * @internal
       */
      setTimeout({
        fn,
        delay,
        name,
        runOnDestroy,
        cancelOutstanding,
        args
      }) {
        var _globalDelays6;
        if (arguments.length > 1 || typeof arguments[0] === 'function') {
          [fn, delay, name, runOnDestroy] = arguments;
        }
        if (typeof fn === 'string') {
          name = fn;
        } else if (!name) {
          name = fn.name || fn;
        }
        if (cancelOutstanding) {
          this.clearTimeout(name);
        }
        const me = this,
          timeoutIds = me.timeoutIds || (me.timeoutIds = new Map()),
          timeoutMap = me.timeoutMap || (me.timeoutMap = new Map()),
          timeoutId = setTimeout(() => {
            var _globalDelays5;
            if (typeof fn === 'string') {
              fn = me[name];
            }
            // Cleanup before invocation in case fn throws
            timeoutIds === null || timeoutIds === void 0 ? void 0 : timeoutIds.delete(timeoutId);
            timeoutMap === null || timeoutMap === void 0 ? void 0 : timeoutMap.delete(name);
            (_globalDelays5 = globalDelays) === null || _globalDelays5 === void 0 ? void 0 : _globalDelays5.timeouts.delete(timeoutId);
            fn.apply(me, args);
          }, delay);
        timeoutIds.set(timeoutId, runOnDestroy ? fn : true);
        // Commented out code is helpful when debugging timeouts in tests
        (_globalDelays6 = globalDelays) === null || _globalDelays6 === void 0 ? void 0 : _globalDelays6.timeouts.set(timeoutId, {
          fn,
          delay,
          name /*, stack : new Error().stack*/
        });
        if (name) {
          timeoutMap.set(name, timeoutId);
        }
        return timeoutId;
      }
      /**
       * clearTimeout wrapper, either call with timeout id as normal clearTimeout or with timeout name (if you specified
       * a name to setTimeout())
       * property to null.
       * @param {Number|String} idOrName timeout id or name
       * @internal
       */
      clearTimeout(idOrName) {
        var _this$timeoutIds, _globalDelays7;
        let id = idOrName;
        if (typeof id === 'string') {
          if (this.timeoutMap) {
            id = this.timeoutMap.get(idOrName);
            this.timeoutMap.delete(idOrName);
          } else {
            return;
          }
        }
        clearTimeout(id);
        (_this$timeoutIds = this.timeoutIds) === null || _this$timeoutIds === void 0 ? void 0 : _this$timeoutIds.delete(id);
        (_globalDelays7 = globalDelays) === null || _globalDelays7 === void 0 ? void 0 : _globalDelays7.timeouts.delete(id);
      }
      /**
       * clearInterval wrapper
       * @param {Number} id
       * @internal
       */
      clearInterval(id) {
        var _this$intervalIds, _globalDelays8;
        clearInterval(id);
        (_this$intervalIds = this.intervalIds) === null || _this$intervalIds === void 0 ? void 0 : _this$intervalIds.delete(id);
        (_globalDelays8 = globalDelays) === null || _globalDelays8 === void 0 ? void 0 : _globalDelays8.intervals.delete(id);
      }
      /**
       * Same as native setInterval, but will be cleared automatically on destroy
       * @param {Function} fn callback method
       * @param {Number} delay delay in milliseconds
       * @param {String} name delay name for debugging
       * @returns {Number}
       * @internal
       */
      setInterval(fn, delay, name) {
        var _globalDelays9;
        const intervalId = setInterval(fn, delay);
        (this.intervalIds || (this.intervalIds = new Set())).add(intervalId);
        (_globalDelays9 = globalDelays) === null || _globalDelays9 === void 0 ? void 0 : _globalDelays9.intervals.set(intervalId, {
          fn,
          delay,
          name
        });
        return intervalId;
      }
      /**
       * Relays to native requestAnimationFrame and adds to tracking to have call automatically canceled on destroy.
       * @param {Function} fn
       * @param {Object[]} [extraArgs] The argument list to append to those passed to the function.
       * @param {Object} [thisObj] `this` reference for the function.
       * @returns {Number}
       * @internal
       */
      requestAnimationFrame(fn, extraArgs = [], thisObj = this) {
        var _globalDelays11;
        const animationFrameIds = this.animationFrameIds || (this.animationFrameIds = new Set()),
          frameId = requestAnimationFrame(() => {
            var _globalDelays10;
            (_globalDelays10 = globalDelays) === null || _globalDelays10 === void 0 ? void 0 : _globalDelays10.animationFrames.delete(frameId);
            // [dongriffin 2022-01-19] It was observed that we can still be called even though we issued the
            // cancelAnimationFrame call. Since delete() returns true if our frameId was present and is now
            // removed, we can tell that we haven't been cancelled before we call our fn:
            animationFrameIds.delete(frameId) && fn.apply(thisObj, extraArgs);
          });
        animationFrameIds.add(frameId);
        (_globalDelays11 = globalDelays) === null || _globalDelays11 === void 0 ? void 0 : _globalDelays11.animationFrames.set(frameId, {
          fn,
          extraArgs,
          thisObj
        });
        return frameId;
      }
      /**
       * Relays to native requestIdleCallback and adds to tracking to have call automatically canceled on destroy.
       * @param {Function} fn
       * @param {Object[]} [extraArgs] The argument list to append to those passed to the function.
       * @param {Object} [thisObj] `this` reference for the function.
       * @returns {Number}
       * @internal
       */
      requestIdleCallback(fn, extraArgs = [], thisObj = this) {
        var _globalDelays13;
        const idleCallbackIds = this.idleCallbackIds || (this.idleCallbackIds = new Set()),
          frameId = requestIdleCallback(() => {
            var _globalDelays12;
            (_globalDelays12 = globalDelays) === null || _globalDelays12 === void 0 ? void 0 : _globalDelays12.idleCallbacks.delete(frameId);
            // Since delete() returns true if our frameId was present and is now
            // removed, we can tell that we haven't been cancelled before we call our fn:
            idleCallbackIds.delete(frameId) && fn.apply(thisObj, extraArgs);
          });
        idleCallbackIds.add(frameId);
        (_globalDelays13 = globalDelays) === null || _globalDelays13 === void 0 ? void 0 : _globalDelays13.idleCallbacks.set(frameId, {
          fn,
          extraArgs,
          thisObj
        });
        return frameId;
      }
      /**
       * Creates a function which will execute once, on the next animation frame. However many time it is
       * called in one event run, it will only be scheduled to run once.
       * @param {Function|String} fn The function to call, or name of function in this object to call.
       * @param {Object[]} [args] The argument list to append to those passed to the function.
       * @param {Object} [thisObj] `this` reference for the function.
       * @param {Boolean} [cancelOutstanding] Cancel any outstanding queued invocation upon call.
       * @internal
       */
      createOnFrame(fn, args = [], thisObj = this, cancelOutstanding) {
        let rafId;
        const result = (...callArgs) => {
          // Cancel if outstanding if requested
          if (rafId != null && cancelOutstanding) {
            this.cancelAnimationFrame(rafId);
            rafId = null;
          }
          if (rafId == null) {
            rafId = this.requestAnimationFrame(() => {
              if (typeof fn === 'string') {
                fn = thisObj[fn];
              }
              rafId = null;
              callArgs.push(...args);
              fn.apply(thisObj, callArgs);
            });
          }
        };
        result.cancel = () => this.cancelAnimationFrame(rafId);
        return result;
      }
      /**
       * Relays to native cancelAnimationFrame and removes from tracking.
       * @param {Number} handle
       * @internal
       */
      cancelAnimationFrame(handle) {
        var _this$animationFrameI, _globalDelays14;
        cancelAnimationFrame(handle);
        (_this$animationFrameI = this.animationFrameIds) === null || _this$animationFrameI === void 0 ? void 0 : _this$animationFrameI.delete(handle);
        (_globalDelays14 = globalDelays) === null || _globalDelays14 === void 0 ? void 0 : _globalDelays14.animationFrames.delete(handle);
      }
      /**
       * Relays to native cancelIdleCallback and removes from tracking.
       * @param {Number} handle
       * @internal
       */
      cancelIdleCallback(handle) {
        var _this$idleCallbackIds, _globalDelays15;
        cancelIdleCallback(handle);
        (_this$idleCallbackIds = this.idleCallbackIds) === null || _this$idleCallbackIds === void 0 ? void 0 : _this$idleCallbackIds.delete(handle);
        (_globalDelays15 = globalDelays) === null || _globalDelays15 === void 0 ? void 0 : _globalDelays15.idleCallbacks.delete(handle);
      }
      async nextAnimationFrame() {
        return new Promise(resolve => this.requestAnimationFrame(resolve));
      }
      /**
       * Wraps a function with another function that delays it specified amount of time, repeated calls to the wrapper
       * resets delay.
       * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
       * instance (or `options.thisObj` instead, if provided).
       * @param {Object|Number} options The delay in milliseconds or an options object.
       * @param {Number} options.delay The delay in milliseconds.
       * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
       * @param {Object} [options.thisObj] The `this` reference for the function.
       * @returns {Function} Wrapped function to call.
       * @internal
       */
      buffer(fn, options) {
        let delay = options;
        if (options && typeof options !== 'number') {
          // if (config object)
          delay = options.delay;
        } else {
          options = null;
        }
        const bufferWrapFn = (...params) => {
            if (bufferWrapFn.suspended) {
              return;
            }
            const {
              delay
            } = bufferWrapFn;
            bufferWrapFn.cancel();
            bufferWrapFn.called = false;
            bufferWrapFn.args = params;
            // If delay=0, the buffer has been disabled so always call immediately.
            if (bufferWrapFn.immediate || !delay) {
              invoker();
            } else {
              bufferWrapFn.timerId = this.setTimeout(invoker, delay);
            }
          },
          invoker = makeInvoker(this, fn, bufferWrapFn, options);
        bufferWrapFn.delay = delay;
        return decorateWrapFn(this, bufferWrapFn);
      }
      /**
       * Returns a function that when called will schedule a call to `fn` on the next animation frame.
       *
       * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
       * instance (or `options.thisObj` instead, if provided).
       * @param {Boolean|Object} [options] An options object or the `cancelOutstanding` boolean property of it.
       * @param {Boolean} [options.cancelOutstanding] Pass `true` to cancel any pending animation frame requests and
       * schedule a new one on each call to the returned function.
       * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
       * @param {Object} [options.thisObj] The `this` reference for the function.
       * @returns {Function}
       * @internal
       */
      raf(fn, options) {
        // NOTE: This method is only intended for use with `delayable`. It has a signature that is compatible
        // with `buffer()` and `throttle()`. The name is 'raf' to make the following aesthetically pleasing:
        //
        //  class Foo extends Delayable() {
        //      static get delayable() {
        //          return {
        //              bar : 'raf'
        //          };
        //      }
        //  }
        let cancelOutstanding = options;
        if (options && typeof options !== 'boolean') {
          // if (config object)
          cancelOutstanding = options.cancelOutstanding;
        } else {
          options = null;
        }
        const rafWrapFn = (...params) => {
            if (rafWrapFn.suspended) {
              return;
            }
            // Reschedule the frame on each call if requested
            if (rafWrapFn.cancelOutstanding) {
              rafWrapFn.cancel();
            }
            rafWrapFn.called = false;
            rafWrapFn.args = params;
            if (rafWrapFn.immediate) {
              invoker();
            } else if (!rafWrapFn.isPending) {
              rafWrapFn.timerId = this.requestAnimationFrame(invoker);
            }
          },
          invoker = makeInvoker(this, fn, rafWrapFn, options);
        rafWrapFn.cancelOutstanding = cancelOutstanding;
        return decorateWrapFn(this, rafWrapFn, 'cancelAnimationFrame');
      }
      idle(fn, options) {
        let cancelOutstanding = options;
        if (options && typeof options !== 'boolean') {
          // if (config object)
          cancelOutstanding = options.cancelOutstanding;
        } else {
          options = null;
        }
        const idleWrapFn = (...params) => {
            if (idleWrapFn.suspended) {
              return;
            }
            // Reschedule the frame on each call if requested
            if (idleWrapFn.cancelOutstanding) {
              idleWrapFn.cancel();
            }
            idleWrapFn.called = false;
            idleWrapFn.args = params;
            if (idleWrapFn.immediate) {
              invoker();
            } else if (!idleWrapFn.isPending) {
              idleWrapFn.timerId = this.requestIdleCallback(invoker);
            }
          },
          invoker = makeInvoker(this, fn, idleWrapFn, options);
        idleWrapFn.cancelOutstanding = cancelOutstanding;
        // If the timer is still there in 100ms, then invoke it.
        this.setTimeout(() => this.idleCallbackIds.delete(idleWrapFn.timerId) && idleWrapFn.now(), 100);
        return decorateWrapFn(this, idleWrapFn, 'cancelIdleCallback');
      }
      /**
       * Create a "debounced" function which will call on the "leading edge" of a timer period.
       * When first invoked will call immediately, but invocations after that inside its buffer
       * period will be rejected, and *one* invocation will be made after the buffer period has expired.
       *
       * This is useful for responding immediately to a first mousemove, but from then on, only
       * calling the action function on a regular timer while the mouse continues to move.
       *
       * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
       * instance (or `options.thisObj` instead, if provided).
       * @param {Number|Object} options The milliseconds to wait after each execution before another execution takes place
       * or a object containing options.
       * @param {Object} [options.thisObj] `this` reference for the function.
       * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
       * @param {Function|String} [options.throttled] A function to call when the invocation is delayed due to buffer
       * time not having expired. If this is a string, it is looked up as a method on `this` instance (or `options.thisObj`
       * instead, if provided). When called, the same arguments are passed as would have been passed to `fn`, including
       * any `options.appendArgs`.
       * @internal
       */
      throttle(fn, options) {
        let delay = options,
          throttled;
        if (options && typeof options !== 'number') {
          // if (config object)
          delay = options.delay;
          throttled = options.throttled;
        } else {
          options = null;
        }
        const me = this,
          throttleWrapFn = (...args) => {
            if (throttleWrapFn.suspended) {
              return;
            }
            const {
                delay
              } = throttleWrapFn,
              elapsed = performance$1.now() - throttleWrapFn.lastCallTime;
            throttleWrapFn.args = args;
            // If it's been more then the delay period since we invoked, we can call it now.
            // Setting delay=0 effectively disables the throttle (which is the goal)
            if (throttleWrapFn.immediate || elapsed >= delay) {
              me.clearTimeout(throttleWrapFn.timerId);
              invoker();
            } else {
              // Kick off a timer for the requested period.
              if (!throttleWrapFn.isPending) {
                throttleWrapFn.timerId = me.setTimeout(invoker, delay - elapsed);
                throttleWrapFn.called = false;
              }
              if (throttled) {
                // Args have to be stored on the wrapFn for the invoker to pick them up:
                throttled.wrapFn.args = args;
                throttled();
              }
            }
          },
          invoker = makeInvoker(me, fn, throttleWrapFn, options);
        throttleWrapFn.delay = delay;
        if (throttled) {
          // Make an invoker for this callback to handle thisObj and typeof=string etc (pass a dud wrapFn):
          throttled = makeInvoker(me, throttled, () => {}, options);
        }
        return decorateWrapFn(me, throttleWrapFn);
      }
      static setupDelayable(cls) {
        cls.setupDelayableMethods(cls.delayable);
      }
      /**
       * This method initializes the `delayable` methods on this class.
       * @param {Object} delayable The `delayable` property.
       * @param {Function} [cls] This parameter will be used internally to process static methods.
       * @private
       */
      static setupDelayableMethods(delayable, cls = null) {
        const me = this,
          statics = delayable.static,
          target = cls || me.prototype;
        if (statics) {
          delete delayable.static;
        }
        for (const name in delayable) {
          let options = delayable[name];
          const implName = name + 'Now',
            type = typeof options;
          if (!target[implName]) {
            // Only move foo() -> fooNow() if a base class hasn't done so already
            target[implName] = target[name];
          }
          if (type === 'number') {
            options = {
              type: 'buffer',
              delay: options
            };
          } else if (type === 'string') {
            options = {
              type: options
            };
          }
          // For instance methods, we place a getter on the prototype. When the method is first accessed from the
          // prototype, we create an instance-specific version by calling this.buffer()/throttle() (based on the type
          // desired) and set that as the instance-level property.
          defineProperty$6(target, name, {
            get() {
              const value = this[options.type]((...params) => {
                this[implName](...params);
              }, options);
              defineProperty$6(this, name, {
                value
              });
              return value;
            }
          });
        }
      }
      // This does not need a className on Widgets.
      // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
      // to the Widget it's mixed in to should implement thus.
      get widgetClass() {}
    });

    /**
     * @module Core/state/StateStorage
     */
    /**
     * Base class representing interface used by the {@link Core.state.StateProvider} to actually store the state data.
     * This class is not intended to be used directly.
     *
     * This class has an interface similar to the [Storage API](https://developer.mozilla.org/en-US/docs/Web/API/Storage).
     */
    class StateStorage {
      /**
       * Returns an object with all stored keys and their values as its properties
       * @member {Object}
       */
      get data() {
        return Object.create(null);
      }
      /**
       * Returns the stored keys as set by {@link #function-setItem}
       * @member {String[]}
       */
      get keys() {
        return [];
      }
      /**
       * Remove all stored keys
       */
      clear() {}
      /**
       * Returns key value as set by {@link #function-setItem}
       * @param {String} key
       * @returns {*}
       */
      getItem(key) {
        return null;
      }
      /**
       * Removes the specified key
       * @param {String} key
       */
      removeItem(key) {}
      /**
       * Sets the specified key to the given value
       * @param {String} key
       * @param {*} value The item value
       */
      setItem(key, value) {}
    }
    StateStorage._$name = 'StateStorage';

    /**
     * @module Core/state/StateProvider
     */
    class Local extends StateStorage {
      constructor(stateProvider) {
        super();
        this.prefix = stateProvider.prefix || '';
      }
      get isLocal() {
        return true;
      }
      get data() {
        const data = empty(),
          keys = this.keys;
        for (const key of keys) {
          data[key] = this.getItem(key);
        }
        return data;
      }
      get keys() {
        return getKeys(this.prefix, this.prefix.length);
      }
      clear() {
        // It's important that we only clear our own StateProvider's keys, not all of localStorage. We get the
        // full keys not the suffixes since we're just going to call removeItem() with them...
        const keys = getKeys(this.prefix);
        for (const key of keys) {
          localStorage.removeItem(key);
        }
      }
      getItem(key) {
        const value = localStorage.getItem(this.prefix + key);
        // We handle the JSON translation at this layer because the Memory storage does not do any such pickling
        // of data but localStorage only handles strings
        return value === null ? value : JSON.parse(value);
      }
      removeItem(key) {
        return localStorage.removeItem(this.prefix + key);
      }
      setItem(key, value) {
        return localStorage.setItem(this.prefix + key, JSON.stringify(value));
      }
    }
    class Memory extends StateStorage {
      constructor() {
        super();
        this.clear();
      }
      get isMemory() {
        return true;
      }
      get data() {
        return ObjectHelper.clone(this._data);
      }
      get keys() {
        return Object.keys(this._data);
      }
      clear() {
        this._data = empty();
      }
      getItem(key) {
        return key in this._data ? this._data[key] : null;
      }
      removeItem(key) {
        delete this._data[key];
      }
      setItem(key, value) {
        this._data[key] = value;
      }
    }
    const empty = () => Object.create(null),
      getKeys = (prefix, pos = 0) => {
        const keys = [],
          count = localStorage.length;
        for (let key, i = 0; i < count; ++i) {
          key = localStorage.key(i);
          key.startsWith(prefix) && keys.push(key.slice(pos));
        }
        return keys;
      },
      nullStorage = new StateStorage(),
      storageTypes = {
        local: Local,
        memory: Memory
      };
    /**
     * Instances of this class are used to manage data storage for objects that use the {@link Core.mixin.State} mixin, i.e.
     * stateful components. When such components change their {@link Core.mixin.State#config-stateful} properties, they
     * notify the associated {@link Core.mixin.State#config-stateProvider}, which will save the changes after a short
     * delay (to allow multiple changes to coalesce into a single save operation).
     *
     * There are two (2) built-in types of storage supported by `StateProvider`:
     *
     *  - `local` : Stores data in the browser's `localStorage`. Because of this, all `StateProvider` instances share their
     *    state data if they have the same {@link #config-prefix}.
     *  - `memory` : Stores data in the provider's memory. Each instance has its own storage. This is typically used when
     *    the state data is saved to a backend server.
     *
     * ## Using `local` Storage
     *
     * The global `StateProvider` is typically to use `localStorage` for the page or application like so:
     *
     * ```javascript
     *  StateProvider.setup('local');
     * ```
     *
     * With this provider in place, all {@link Core.mixin.State stateful components} will save their
     * {@link Core.mixin.State#property-state} to this provider by default.
     *
     * This is the most typical, and recommended, strategy for proving data to stateful components. This approach allows
     * various widgets on the page to simply declare their {@link Core.mixin.State#config-stateId} to participate in the
     * saving and restoring of application state.
     *
     * Because this storage type uses `localStorage`, the `StateProvider` applies a string prefix to isolate its data from
     * other users of `localStorage`. The default prefix is `'bryntum-state:'`, but this can be configured to a different
     * value. This could be desired, for example, to isolate state data from multiple pages or for version changes.
     *
     * ```javascript
     *  StateProvider.setup({
     *      storage : 'local',
     *      prefix  : 'myApp-v1:'
     *  });
     * ```
     *
     * ## Using `memory` Storage
     *
     * In some applications it may be desirable to save state to a server and restore it on other devices for the user.
     * Because state data is consumed synchronously, and server I/O is asynchronous, the `StateProvider` can be configured
     * to use `'memory'` storage and the actual state data can be loaded/saved by the application.
     *
     * Two factors are important to consider before deciding to save application state on the server (beyond the async
     * adaptation):
     *
     * - State properties are often more of a reflection of the user's device than they are application preferences
     *   and, therefore, may not apply well on other devices.
     * - Potentially undesired application state will not be cleared by clearing local browser user data (a common
     *   troubleshooting strategy) and will follow the user to other browsers (another common troubleshooting technique).
     *
     * The use this type of storage, the global `StateProvider` is configured like so:
     *
     * ```javascript
     * StateProvider.setup('memory');
     * ```
     *
     * In this scenario, application code would download the user's state and use {@link #property-data} to populate
     * the {@link #property-instance-static StateProvider.instance}. In this case, the {@link #event-save} event is used
     * to save the data back to the server when it changes.
     *
     * See [state](https://bryntum.com/products/grid/examples/state/) demo for a usage example.
     * @mixes Core/mixin/Events
     */
    class StateProvider extends Base$1.mixin(Delayable, Events) {
      static get $name() {
        return 'StateProvider';
      }
      static get configurable() {
        return {
          /**
           * The key prefix applied when using the `'local'` {@link #config-storage} type.
           * @config {String}
           * @default
           */
          prefix: 'bryntum-state:',
          /**
           * Storage instance
           * @member {Core.state.StateStorage} storage
           */
          /**
           * One of the following storage types:
           *  - `local` : Stores data in the browser's `localStorage` using the {@link #config-prefix}.
           *  - `memory` : Stores data in the provider's memory.
           *
           * @config {'local'|'memory'|Core.state.StateStorage}
           * @default
           */
          storage: 'local'
        };
      }
      static get delayable() {
        /*
            The StateProvider uses a delayed write to save stateful components in batches. To illustrate, consider the
            "collapsed" config for a Panel that has been marked as "stateful":
                App                         Stateful                      State
                Code                        Component                    Provider
                  :                             :                           :
                  | .collapsed=true             :                           :
                  |---------------------------->|                           :
                  |         onConfigChange() +--|                           :
                  |                          |  |                           :
                  |                          +->|                           :
                  |              saveState() +--|                           :
                  |                          |  |                           :
                  |                          +->| saveStateful()            :
                  |                             |-------------------------->|
                  |                             |                           | pendingSaves.push()
                  |                             |                           |----+ writeStatefuls()
                  | .collapsed=true             |<..........................:    :
                  |<............................:                           :    :  (maybe other changes)
                  :                             :                           :    :
                  :                             :                           |<---+ (50 ms later)
                  :                             :                           | writeStatefuls()
                  :                             :       saveState({         |
                  :                             :         immediate:true})  | <---------------+
                  :                             |<--------------------------|                  \
                  :                             | setValue()                |                   \
                  :                             |-------------------------->|                    \
                  :                             |                           | .trigger('set')     ) one or more of these
                  :                             |<..........................|                    /
                  :                             |              saveState()  |                   /
                  :                             :..........................>|                  /
                  :                             :                           | <---------------+
                  :                             :                           |
                  :                             :                           | .trigger('save')
                  :                             :                           |
                  :                             :                           :....> writeStatefuls()
                  :                             :                           :
        */
        return {
          writeStatefuls: 50
        };
      }
      /**
       * The default {@link Core.mixin.State#config-stateProvider} for stateful objects.
       * @property {Core.state.StateProvider}
       */
      static get instance() {
        return this._instance;
      }
      static set instance(inst) {
        if (inst == null) {
          inst = nullProvider;
        } else {
          if (typeof inst === 'string' || ObjectHelper.isClass(inst) || inst instanceof StateStorage) {
            inst = {
              storage: inst
            };
          }
          if (ObjectHelper.isObject(inst)) {
            inst = new StateProvider(inst);
          }
        }
        this._instance = inst;
      }
      /**
       * Initializes the default `StateProvider` instance for the page. This method can be passed an instance or one of
       * the following type aliases:
       *
       *  - `'local'` : use `localStorage` to store application state (most common)
       *  - `'memory'` : holds application state in the `StateProvider` instance (used when state is saved to a server)
       *
       * Once the `StateProvider` is initialized, components that use {@link Core.mixin.State} and assign components a
       * {@link Core.mixin.State#config-stateId} will use this default provider to automatically save and restore their
       * state.
       *
       * @param {'local'|'memory'|Core.state.StateProvider} inst The state provider storage type ('local' or 'memory') or
       * the `StateProvider` instance.
       * @returns {Core.state.StateProvider}
       */
      static setup(inst) {
        this.instance = inst; // use smart setter
        return this.instance;
      }
      doDestroy() {
        self.writeStatefuls.flush();
        super.doDestroy();
      }
      /**
       * On read, this property returns all state data stored in the provider. On write, this property _adds_ all the
       * given values to the state provider's data. To replace the data, call {@link #function-clear} before assigning
       * this property. This is used to bulk populate this `StateProvider` with data for stateful components.
       * @member {Object}
       */
      get data() {
        return this.storage.data;
      }
      set data(data) {
        if (!data) {
          this.clear();
        } else {
          for (const key in data) {
            this.setValue(key, data[key]);
          }
        }
      }
      /**
       * Clears all state date
       * @returns {Core.state.StateProvider} this instance
       */
      clear() {
        this.storage.clear();
        return this;
      }
      changeStorage(storage) {
        if (storage == null) {
          storage = nullStorage;
        } else {
          if (typeof storage === 'string') {
            if (!storageTypes[storage]) {
              throw new Error(`Invalid storage type "${storage}" (expected one of: "${Object.keys(storageTypes).join('", "')}")`);
            }
            storage = storageTypes[storage];
          }
          if (ObjectHelper.isClass(storage)) {
            storage = new storage(this);
          }
        }
        return storage;
      }
      /**
       * This method is called to schedule saving the given `stateful` object.
       * @param {Core.mixin.State} stateful The stateful object to save.
       * @param {Object} [options] An object of options that affect the state saving process.
       * @param {String} [options.id] The key for the saved state.
       * @param {Boolean} [options.immediate] Pass `true` to save the data synchronously instead of on a delay.
       * @internal
       */
      saveStateful(stateful, options) {
        (this.pendingSaves || (this.pendingSaves = [])).push([stateful, options]);
        this.writeStatefuls();
      }
      /**
       * A delayable method that flushes pending stateful objects.
       * @private
       */
      writeStatefuls() {
        const me = this,
          {
            pendingSaves
          } = me,
          n = pendingSaves === null || pendingSaves === void 0 ? void 0 : pendingSaves.length,
          stateIds = [],
          saved = [];
        me.pendingSaves = null;
        if (n) {
          for (let options, stateful, stateId, i = 0; i < n; ++i) {
            [stateful, options] = pendingSaves[i];
            if (!stateful.isDestroying && stateful.isSaveStatePending) {
              stateId = stateful.saveState({
                ...options,
                immediate: true
              });
              if (stateId) {
                stateIds.push(stateId);
                saved.push(stateful);
              }
            }
          }
          if (stateIds.length) {
            /**
             * Triggered after one or more stateful objects save their state to the state provider. This event can
             * be used to save state to a backend server.
             *
             * For example, to save the page's state object as a single object on the server:
             *
             * ```javascript
             *  StateProvider.instance.on({
             *      save() {
             *          const data = StateProvider.instance.data;
             *          // Save "data" to server
             *      }
             *  });
             * ```
             *
             * Or, to save individual stateful components to the server:
             *
             * ```javascript
             *  StateProvider.instance.on({
             *      save({ stateIds }) {
             *          for (const stateId of stateIds) {
             *              const data = StateProvider.instance.getValue(stateId);
             *
             *              if (data == null) {
             *                  // Remove "stateId" from the server
             *              }
             *              else {
             *                  // Save new "data" for "stateId" to the server
             *              }
             *          }
             *      }
             *  });
             * ```
             *
             * Multi-page applications should probably include a page identifier in addition to the `stateId` to
             * prevent state from one page affecting other pages. If there are common components across all (or
             * many) pages, the `stateId` values would need to be assigned with all pages in mind.
             *
             * @event save
             * @param {Core.state.StateProvider} source The source of the event
             * @param {String[]} stateIds An array of `stateId` values that were saved to the state provider.
             * @param {Core.mixin.State[]} saved An array of stateful objects saved just saved to state provider
             * storage, in the same order as the `stateIds` array.
             */
            me.trigger('save', {
              stateIds,
              saved
            });
          }
        }
      }
      /**
       * Returns the stored state given its `key`.
       * @param {String} key The identifier of the state to return.
       * @returns {Object}
       */
      getValue(key) {
        this.writeStatefuls.flush();
        return this.storage.getItem(key);
      }
      /**
       * Stores the given state `value` under the specified `key`.
       * @param {String} key The identifier of the state value.
       * @param {Object} value The state value to set.
       * @returns {Core.state.StateProvider} this instance
       */
      setValue(key, value) {
        const me = this,
          {
            storage
          } = me,
          was = me.getValue(key);
        if (value != null) {
          storage.setItem(key, value);
          /**
           * Triggered after an item is stored to the state provider.
           * @event set
           * @param {Core.state.StateProvider} source The source of the event
           * @param {String} key The name of the stored item.
           * @param {*} value The value of the stored item.
           * @param {*} was The previous value of the stored item.
           */
          me.trigger('set', {
            key,
            value,
            was
          });
        } else if (was !== null) {
          storage.removeItem(key);
          /**
           * Triggered after an item is removed from the state provider.
           * @event remove
           * @param {Core.state.StateProvider} source The source of the event
           * @param {String} key The name of the removed item.
           * @param {*} was The value of the removed item.
           */
          me.trigger('remove', {
            key,
            was
          });
        }
        return me;
      }
    }
    const nullProvider = new StateProvider({
      storage: nullStorage
    });
    StateProvider._instance = nullProvider;
    StateProvider._$name = 'StateProvider';

    //import Config from '../Config.js';
    /**
     * @module Core/mixin/State
     */
    const primitiveRe = /boolean|number|string/;
    /**
     * A mixin that handles accessing, saving, and restoring an object's persistent state.
     *
     * ## Using Stateful Components
     *
     * Instances of classes that use this mixin (i.e., "stateful components") have a {@link #property-state} property that
     * provides read/write access to their persistable state in the form of a simple object. These state objects can be
     * saved and restored under application control, e.g., using `localStorage`.
     *
     * This approach can be streamlined using a {@link Core.state.StateProvider} either by setting the
     * {@link Core.state.StateProvider#property-instance-static default state provider} or by using an instance-level
     * {@link #config-stateProvider} config.
     *
     * When using a state provider, stateful components with a {@link #config-stateId} or an
     * {@link Core.widget.Widget#config-id} will automatically save (see {@link #function-saveState}) and restore
     * (see {@link #function-loadState}) their `state`. This use of the `id` as a `stateId` can be disabled by assigning
     * the {@link #config-stateful} config to `false`. When using a `stateId` and a state provider, it is not necessary to
     * call the {@link #function-loadState} and {@link #function-saveState} methods directly.
     *
     * ### Simple vs Complex State
     *
     * Some stateful components (e.g., {@link Core.widget.Panel panels}) have state that can be described purely by their
     * config properties. For these components, the {@link #config-stateful} config can be used to control which config
     * properties to include in their persistent state. For example:
     *
     * ```javascript
     *  const mainPanel = new Panel({
     *      collapsible : true,
     *      stateId     : 'mainPanel',
     *      stateful    : ['collapsed']
     *  });
     * ```
     *
     * Other components have a complex state (e.g., `GridState`), and do not use the `stateful` config in this way. In all
     * other ways, however, these components behave the same as their simple state counterparts.
     *
     * ## Implementing Stateful Components
     *
     * Implementors of stateful components have two main design points to consider:
     *
     *  - Getting and setting their persistent {@link #property-state} object.
     *  - Initiating calls to {@link #function-saveState} when the object's persistent state changes.
     *
     * ### Persistent State
     *
     * For simple cases, the {@link #config-stateful} config can be set to the list of config property names that should be
     * saved:
     *
     * ```javascript
     *  class MyStatefulComponent extends Base.mixin(State) {
     *      static get configurable() {
     *          return {
     *              stateful : ['text', 'size']
     *          };
     *      }
     *  }
     * ```
     *
     * While the `stateful` config supports an object form (where keys with truthy values are the config names), this form
     * is typically reserved for configuring instances.
     *
     * Classes can choose to implement the {@link #function-getState} and {@link #function-applyState} methods to enhance
     * the `state` object with data not easily mapped to config properties. These method can call their `super` methods or
     * fully replace them.
     *
     * ```javascript
     *  class MyStatefulComponent extends Base.mixin(State) {
     *      getState() {
     *          return {
     *              text : this.text,
     *              size : this.size
     *          };
     *      }
     *
     *      applyState(state) {
     *          this.text = state.text;
     *          this.size = state.size;
     *      }
     *  }
     * ```
     *
     * ### Saving Dates
     *
     * A stateful property may be a `Date` property if the `changeDate` method of the class accepts an
     * ISO 8601 formatted date. Dates are saved in state using ISO 8601 format: `YYYY-MM-DDTHH:mm:ssZ`
     *
     * ### Saving State
     *
     * When the persistent state of a stateful component changes, it must call {@link #function-saveState}. This method
     * schedules an update of the component's persistence {@link #property-state} with the appropriate
     * {@link #config-stateProvider}. When a config property named in the {@link #config-stateful} config changes, this
     * call will be made automatically. This means that even if a component replaces {@link #function-getState} and
     * {@link #function-applyState}, it can still be helpful to specify a value for the `stateful` config.
     *
     * ```javascript
     *  class MyStatefulComponent extends Base.mixin(State) {
     *      static get configurable() {
     *          return {
     *              stateful : ['text', 'size']
     *          };
     *      }
     *
     *      getState() { ... }
     *      applyState(state) { ... }
     *  }
     * ```
     *
     * Another way to ensure {@link #function-saveState} is called when necessary is to use {@link #config-statefulEvents}.
     *
     * ```javascript
     *  class MyStatefulComponent extends Base.mixin(State) {
     *      static get configurable() {
     *          return {
     *              statefulEvents : ['change', 'resize']
     *          };
     *      }
     *  }
     * ```
     *
     * @mixin
     */
    var State = (Target => class State extends (Target || Base$1) {
      static $name = 'State';
      static configurable = {
        /**
         * This value can be one of the following:
         *
         * - `false` to not use an explicitly assigned {@link Core.widget.Widget#config-id} as the component's
         * {@link #config-stateId} (this is only necessary when there is a {@link #config-stateProvider}).
         * - An array of strings naming the config properties to save in the component's {@link #property-state}
         * object.
         * - An object whose truthy keys are the config properties to save in the component's {@link #property-state}
         * object.
         *
         * These last two uses of the `stateful` config property do not apply to components that have a complex
         * state, as described in the {@link Core.mixin.State State mixin documentation}.
         *
         * This config property is typically set by derived classes to a value including any config property that
         * the user can affect via the user interface. For example, the {@link Core.widget.Panel#config-collapsed}
         * config property is listed for a {@link Core.widget.Panel} since the user can toggle this config property
         * using the {@link Core.widget.panel.PanelCollapser#config-tool collapse tool}.
         *
         * @config {Boolean|Object|String[]}
         * @category State
         */
        stateful: {
          value: null,
          $config: {
            merge: 'classList'
          }
        },
        /**
         * The events that, when fired by this component, should trigger it to save its state by calling
         * {@link #function-saveState}.
         *
         * ```javascript
         *  class MyStatefulComponent extends Base.mixin(State) {
         *      static get configurable() {
         *          return {
         *              statefulEvents : [ 'change', 'resize' ]
         *          };
         *      }
         *  }
         * ```
         * In the above example, {@link #function-saveState} will be called any time an instance of this class
         * fires the `change` or `resize` event.
         *
         * This config is typically set by derived classes as a way to ensure {@link #function-saveState} is called
         * whenever their persistent state changes.
         *
         * @config {Object|String[]}
         * @category State
         * @default
         */
        statefulEvents: {
          $config: {
            merge: 'classList'
          },
          value: ['stateChange']
        },
        /**
         * The key to use when saving this object's state in the {@link #config-stateProvider}. If this config is
         * not assigned, and {@link #config-stateful} is not set to `false`, the {@link Core.widget.Widget#config-id}
         * (if explicitly specified) will be used as the `stateId`.
         *
         * If neither of these is given, the {@link #function-loadState} and {@link #function-saveState} methods
         * will need to be called directly to make use of the `stateProvider`.
         *
         * For single page applications (SPA's), or multi-page applications (MPA's) that have common, stateful
         * components on multiple pages, the `stateId` should be unique across all stateful components (similar to DOM
         * element id's). MPA's that want each page to be isolated can more easily achieve that isolation using the
         * {@link Core.state.StateProvider#config-prefix}.
         *
         * @config {String}
         * @category State
         */
        stateId: null,
        /**
         * The `StateProvider` to use to save and restore this object's {@link #property-state}. By default, `state`
         * will be saved using the {@link Core.state.StateProvider#property-instance-static default state provider}.
         *
         * This config is useful for multi-page applications that have a set of common components that want to share
         * state across pages, as well as other components that want their state to be isolated. One of these groups
         * of stateful components could be assigned an explicit `stateProvider` while the other group could use the
         * default state provider.
         *
         * @config {Core.state.StateProvider}
         * @category State
         */
        stateProvider: null
      };
      static prototypeProperties = {
        statefulLoaded: false,
        statefulSuspended: 0
      };
      afterConstruct() {
        super.afterConstruct();
        this.loadState();
      }
      finalizeInit() {
        // For widgets, this should happen before rendering which happens in Widget.finalizeInit():
        this.loadState();
        super.finalizeInit();
      }
      /**
       * Returns `true` if this instance implements the {@link Core.mixin.State} interface.
       * @property {Boolean}
       * @readonly
       * @advanced
       */
      get isStateful() {
        return true;
      }
      /**
       * Returns `true` if this instance is ready to participate in state activities.
       * @property {Boolean}
       * @readonly
       * @internal
       */
      get isStatefulActive() {
        // If a widget is rendered via appendTo (for example), this happens inside construct(), before we are called
        // in afterConstruct(). When the Widget uses Responsive mixin, that will trigger its initial responsive update.
        // In short, when isResponsivePending, the Widget is Responsive _and_ has not yet determined its responsiveState.
        // In this case we do NOT want to activate statefulness.
        // Further, if we are updating configs from a responsiveUpdate, we do not want to save to state.
        return !this.statefulSuspended && !this.isResponsivePending && !this.isResponsiveUpdating;
      }
      // state
      /**
       * Gets or sets a component's state
       * @property {Object}
       * @category State
       */
      get state() {
        return this._state = this.getState();
      }
      set state(state) {
        this._state = state;
        if (state) {
          this.applyState(state);
        }
      }
      // statefulEvents
      updateStatefulEvents(events) {
        const me = this,
          listeners = {
            name: 'statefulEvents',
            thisObj: me
          };
        me.detachListeners(listeners.name);
        if (events) {
          if (typeof events === 'string') {
            events = StringHelper.split(events);
          } else if (!Array.isArray(events)) {
            events = ObjectHelper.getTruthyKeys(events);
          }
          if (events.length) {
            var _me$ion;
            for (const event of events) {
              listeners[event] = 'onStatefulEvent';
            }
            (_me$ion = me.ion) === null || _me$ion === void 0 ? void 0 : _me$ion.call(me, listeners);
          }
        }
      }
      // statefulId
      /**
       * Returns the state key to use for this instance. This will be either the {@link #config-stateId} or the
       * {@link Core.widget.Widget#config-id} (if explicitly specified and {@link #config-stateful} is not `false`).
       * @property {String}
       * @category State
       * @internal
       */
      get statefulId() {
        const me = this,
          {
            responsiveState
          } = me;
        let statefulId = me.stateId;
        if (statefulId == null && me.hasGeneratedId === false && me.stateful !== false) {
          statefulId = me.id;
        }
        if (statefulId && responsiveState) {
          statefulId = `${statefulId}[${responsiveState}]`; // ex = 'foo[small]'
        }

        return statefulId;
      }
      // statefulness
      /**
       * Returns an object whose truthy keys are the config properties to include in this object's {@link #property-state}.
       * @property {Object}
       * @category State
       * @readonly
       * @private
       */
      get statefulness() {
        const {
          stateful
        } = this;
        return Array.isArray(stateful) ? ObjectHelper.createTruthyKeys(stateful) : stateful;
      }
      // stateProvider
      get stateProvider() {
        return this._stateProvider || StateProvider.instance;
      }
      //---------------------------------------------------------------------------------------------------------------
      // Methods
      /**
       * Applies the given `state` to this instance.
       *
       * This method is not called directly, but is called when the {@link #property-state} property is assigned a value.
       *
       * This method is implemented by derived classes that have complex state which exceeds the simple list of config
       * properties provided by {@link #config-stateful}. In these cases, the `super` method can be called to handle any
       * config properties that are part of the complex state. The default implementation of this method will only assign
       * those config properties listed in {@link #config-stateful} from the provided `state` object.
       *
       * @param {Object} state The state object to apply to this instance.
       * @category State
       * @advanced
       */
      applyState(state) {
        state = this.pruneState(state);
        if (state) {
          this.setConfig(state);
        }
      }
      /**
       * Returns this object's state information.
       *
       * This method is not called directly, but is called to return the value of the {@link #property-state} property.
       *
       * This method is implemented by derived classes that have complex state which exceeds the simple list of config
       * properties provided by {@link #config-stateful}. In these cases, the `super` method can be called to gather the
       * config properties that are part of the complex state. The default implementation of this method will only copy
       * those config properties listed in {@link #config-stateful} to the returned `state` object.
       *
       * @returns {Object}
       * @category State
       * @advanced
       */
      getState() {
        const me = this,
          {
            initialConfig,
            statefulness,
            isConstructing: defaultState
          } = me,
          {
            configs
          } = me.$meta,
          // If we are reading state at construction time, we are collecting the defaultState, so
          // we should read from the initial config and the defaults.
          source = defaultState ? Object.setPrototypeOf(initialConfig, me.$meta.config) : me;
        let state = null,
          key,
          value;
        if (statefulness) {
          state = {};
          for (key in statefulness) {
            if (statefulness[key]) {
              var _value;
              value = source[key];
              if ((_value = value) !== null && _value !== void 0 && _value.isStateful) {
                value = value.state; // e.g.: stateful : { store : true }
              } else if (!defaultState) {
                // Dates can be saved in state as ISO 8601 Date and time.
                // This class's changer must be able to ingest this format.
                if (ObjectHelper.isDate(value)) {
                  value = DateHelper.format(value, 'YYYY-MM-DDTHH:mm:ssZ');
                }
                // If we are reading state to save, ignore configs that have their initial value or aren't primitives
                if (configs[key].equal(value, initialConfig === null || initialConfig === void 0 ? void 0 : initialConfig[key]) || !primitiveRe.test(typeof value)) {
                  continue;
                }
              }
              state[key] = value;
            }
          }
        }
        return state;
      }
      /**
       * Loads this object's state from its {@link #config-stateProvider} and applies it to its {@link #property-state}.
       *
       * This method only acts upon its first invocation for a given instance (unless `true` is passed for the `reload`
       * parameter). This allows for flexibility in the timing of that call during the early stages of the instances'
       * lifecycle. To reload the state after this time, manually assign the desired value to the {@link #property-state}
       * property or call this method and pass `reload` as `true`.
       *
       * This method is called automatically during construction when a {@link #config-stateId} or (in some cases) an
       * explicit {@link Core.widget.Widget#config-id} is provided.
       *
       * @param {String} [stateId] An overriding key to use instead of this object's {@link #config-stateId}.
       * @param {Boolean} [reload=false] Pass `true` to load the state even if previously loaded.
       * @category State
       */
      loadState(stateId, reload) {
        if (typeof stateId === 'boolean') {
          reload = stateId;
          stateId = null;
        }
        const me = this,
          {
            statefulLoaded
          } = me;
        if (me.isStatefulActive && (reload || !statefulLoaded)) {
          const state = me.loadStatefulData(stateId || (stateId = me.statefulId));
          if (!statefulLoaded && stateId) {
            // Whether we have state data or not, we attempted to load it, so track the defaults and load attempt.
            // The state as gathered when statefulLoaded not set is gathered from the configuration, *not*
            // the running state.
            me.defaultState = me.state;
            me.statefulLoaded = true;
          }
          if (state) {
            me.state = state;
          }
        }
      }
      loadStatefulData(stateId) {
        var _this$stateProvider;
        stateId = this.isStatefulActive ? stateId || this.statefulId : null;
        return stateId && ((_this$stateProvider = this.stateProvider) === null || _this$stateProvider === void 0 ? void 0 : _this$stateProvider.getValue(stateId));
      }
      resetDefaultState() {
        if (this.defaultState) {
          this.state = this.defaultState;
        }
      }
      resumeStateful(full = false) {
        this.statefulSuspended = full ? 0 : Math.max(this.statefulSuspended - 1, 0);
      }
      /**
       * Saves this object's state to its {@link #config-stateProvider}.
       *
       * When a {@link #config-stateId} or (in some cases) an explicit {@link Core.widget.Widget#config-id} is provided,
       * this method will be called automatically any time a config property listed in {@link #config-stateful} changes or
       * when a {@link #config-statefulEvents stateful event} is fired.
       *
       * Derived classes are responsible for calling this method whenever the persistent {@link #property-state} of the
       * object changes.
       *
       * @param {Object|String} [options] Options that affect the state saving process or, if a string, the state `id`.
       * @param {String} [options.id] The state id for the saved state (overrides {@link #config-stateId}).
       * @param {Boolean} [options.immediate] Pass `true` to save the data synchronously instead of on a delay.
       * @category State
       */
      saveState(options) {
        if (typeof options === 'string') {
          options = {
            id: options
          };
        } else {
          options = options || {};
        }
        const me = this,
          {
            stateProvider
          } = me,
          statefulId = options.id || me.isStatefulActive && me.statefulId;
        if (statefulId && stateProvider) {
          if (options.immediate) {
            me.isSaveStatePending = false;
            stateProvider.setValue(statefulId, me.state);
          } else if (!me.isSaveStatePending) {
            me.isSaveStatePending = true;
            stateProvider.saveStateful(me, options);
          }
          return statefulId;
        }
      }
      suspendStateful() {
        ++this.statefulSuspended;
      }
      //---------------------------------------------------------------------------------------------------------------
      // Private / Internal
      onConfigChange({
        name,
        value,
        was,
        config
      }) {
        super.onConfigChange({
          name,
          value,
          was,
          config
        });
        if (!this.isConstructing && this.isStatefulActive && this.statefulId) {
          const {
            stateful
          } = this;
          if (Array.isArray(stateful) ? stateful.includes(name) : stateful === null || stateful === void 0 ? void 0 : stateful[name]) {
            this.saveState();
          }
        }
      }
      onStatefulEvent() {
        if (!this.isConstructing) {
          this.saveState();
        }
      }
      /**
       * Returns an object that copies the {@link #config-stateful} config properties from the provided `state` object.
       *
       * @param {Object} state A state object from which to copy stateful configs.
       * @returns {Object}
       * @category State
       * @private
       */
      pruneState(state) {
        const {
          statefulness
        } = this;
        if (statefulness) {
          const pruned = {};
          for (const key in state) {
            if (statefulness[key]) {
              pruned[key] = state[key];
            }
          }
          state = pruned;
        }
        return state;
      }
      //---------------------------------------------------------------------------------------------------------------
      // This does not need a className on Widgets.
      // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
      // to the Widget it's mixed in to should implement thus.
      get widgetClass() {}
    });

    /**
     * @module Core/mixin/Identifiable
     */
    const
      // Id generation should be on a per page basis, not per module
      idCounts$1 = ObjectHelper.getPathDefault(globalThis, 'bryntum.idCounts', Object.create(null)),
      idTypes = {
        string: 1,
        number: 1
      };
    /**
     * A mixin which provides identifier services such as auto-creation of `id`s and registration and
     * lookup of instances by `id`.
     *
     * @mixin
     * @internal
     */
    var Identifiable = (Target => class Identifiable extends (Target || Base$1) {
      static get $name() {
        return 'Identifiable';
      }
      static get declarable() {
        return ['identifiable'];
      }
      static get configurable() {
        return {
          /**
           * The id of this object.  If not specified one will be generated. Also used for lookups through the
           * static `getById` of the class which mixes this in. An example being {@link Core.widget.Widget}.
           *
           * For a {@link Core.widget.Widget Widget}, this is assigned as the `id` of the DOM
           * {@link Core.widget.Widget#config-element element} and must be unique across all elements
           * in the page's `document`.
           * @config {String}
           * @category Common
           */
          id: ''
        };
      }
      static setupIdentifiable(cls, meta) {
        const {
          identifiable
        } = cls;
        identifiable.idMap = Object.create(null);
        Reflect.defineProperty(cls, 'identifiable', {
          get() {
            return identifiable;
          }
        });
      }
      doDestroy() {
        this.constructor.unregisterInstance(this);
        super.doDestroy();
      }
      changeId(id) {
        return (this.hasGeneratedId /* assignment */ = !id) ? this.generateAutoId() : id;
      }
      updateId(id, oldId) {
        const me = this,
          C = me.constructor;
        oldId && C.unregisterInstance(me, oldId);
        if (!me.hasGeneratedId || C.identifiable.registerGeneratedId !== false) {
          C.registerInstance(me, id);
        }
      }
      /**
       * This method generates an id for this instance.
       * @returns {String}
       * @internal
       */
      generateAutoId() {
        return this.constructor.generateId(`b-${this.$$name.toLowerCase()}-`);
      }
      static get all() {
        // not documented here since type of array is not knowable... documented at mixin target class
        return Object.values(this.identifiable.idMap);
      }
      /**
       * Generate a new id, using an internal counter and a prefix.
       * @param {String} prefix Id prefix
       * @returns {String} Generated id
       */
      static generateId(prefix = 'generatedId') {
        // This produces "b-foo-1, b-foo-2, ..." for each prefix independently of the others. In other words, it makes
        // id's more stable since the counter is on a per-class basis.
        return prefix + (idCounts$1[prefix] = (idCounts$1[prefix] || 0) + 1);
      }
      static registerInstance(instance, instanceId = instance.id) {
        const {
          idMap
        } = this.identifiable;
        // Code editor sets `disableThrow` to not get conflicts when loading the same module again
        if (instanceId in idMap && !this.disableThrow) {
          throw new Error('Id ' + instanceId + ' already in use');
        }
        idMap[instanceId] = instance;
      }
      /**
       * Unregister Identifiable instance, normally done on destruction
       * @param {Object} instance Object to unregister
       * @param {String} id The id of the instance to unregister.
       */
      static unregisterInstance(instance, id = instance.id) {
        const {
          idMap
        } = this.identifiable;
        // ID may be passed, for example if the instance is destroyed and can no longer yield an id.
        if (idTypes[typeof instance]) {
          delete idMap[instance];
        }
        // Have to check for identity in case another instance by the same id has been created.
        // Allow that to be overridden. Stores have always just evicted the previous owner of their IDs
        else if (idMap[id] === instance) {
          delete idMap[id];
        }
      }
      static getById(id) {
        const idMap = this.identifiable.idMap;
        if (idMap) {
          return idMap[id];
        }
      }
      static get registeredInstances() {
        const idMap = this.identifiable.idMap;
        return idMap ? Object.values(idMap) : [];
      }
    });

    /**
     * @module Core/data/stm/mixin/ModelStm
     */
    const STM_PROP$1 = Symbol('STM_PROP'),
      unrecordedFields = {
        // This field's value is a by product of node insertion and must not be recorded here.
        // It's the node insertion operation which is recorded by STM.
        parentIndex: 1
      };
    /**
     * Mixin making a model compatible with {@link Core/data/stm/StateTrackingManager}
     *
     * @mixin
     */
    var ModelStm = (Target => class ModelStm extends (Target || Base$1) {
      static get $name() {
        return 'ModelStm';
      }
      static get defaultConfig() {
        return {
          stm: null
        };
      }
      joinStore(store) {
        // No super on purpose, micro optimization of critical perf path
        // super.joinStore && super.joinStore(store);
        if (!this.stm) {
          this.stm = store.stm;
        }
      }
      unjoinStore(store, isReplacing = false) {
        var _super$unjoinStore;
        if (this.stm === store.stm) {
          this.stm = null;
        }
        (_super$unjoinStore = super.unjoinStore) === null || _super$unjoinStore === void 0 ? void 0 : _super$unjoinStore.call(this, store, isReplacing);
      }
      /**
       * Reference to STM manager, if used
       * @member {Core.data.stm.StateTrackingManager}
       * @category Misc
       */
      get stm() {
        return this[STM_PROP$1];
      }
      set stm(stm) {
        this[STM_PROP$1] = stm;
      }
      // Hook for chronograph entity field accessors, for example; task.duration = 123.
      // Triggers before setting the value.
      beforeChronoFieldSet(fieldName, value) {
        var _me$stm;
        const me = this;
        if (!me.inSetting && (_me$stm = me.stm) !== null && _me$stm !== void 0 && _me$stm.enabled && !unrecordedFields[fieldName] && !me.constructor.nonPersistableFields[fieldName]) {
          // Do not record changes of identifiers that are not fields
          if (me.getFieldDefinition(fieldName)) {
            return {
              [fieldName]: {
                value,
                oldValue: me[fieldName]
              }
            };
          }
        }
        return null;
      }
      // Hook for chronograph entity field accessors, for example; task.duration = 123
      // Triggers after setting the value.
      afterChronoFieldSet(fieldName, value, wasSet) {
        wasSet && this.afterSet(fieldName, value, false, false, wasSet, true);
      }
      shouldRecordFieldChange(fieldName, oldValue, newValue) {
        const store = this.firstStore;
        // By default, we do not record:
        // - not persistable field changes
        // - null vs undefined changes
        // - same value changes, compared by reference (by value for dates)
        // - "id" changes
        // - "parentId" changes caused by parent record idChange
        return !(this.constructor.nonPersistableFields[fieldName] || oldValue == null && newValue == null || oldValue === newValue || oldValue instanceof Date && newValue instanceof Date && oldValue.getTime() === newValue.getTime() || fieldName === 'id' || fieldName === '$PhantomId' || fieldName === 'parentId' && store && store.oldIdMap[oldValue] === store.getById(newValue));
      }
      /**
       * Overridden to store initial data of the changed fields and to notify STM
       * manager about the change action if anything has been changed in result.
       *
       * The method is called from within {@link Core/data/Model#function-set} method.
       *
       * @private
       */
      afterSet(field, value, silent, fromRelationUpdate, wasSet, isChronoFieldSet) {
        const {
            stm
          } = this,
          nonPersistableFields = this.constructor.nonPersistableFields;
        if (stm !== null && stm !== void 0 && stm.isBase && stm.enabled && !unrecordedFields[field] && !nonPersistableFields[field]) {
          if (wasSet) {
            let shouldRecord;
            const [newData, oldData] = Object.keys(wasSet).reduce((data, fieldName) => {
              const {
                value,
                oldValue
              } = wasSet[fieldName];
              if (this.shouldRecordFieldChange(fieldName, oldValue, value)) {
                shouldRecord = true;
                data[0][fieldName] = value;
                data[1][fieldName] = oldValue;
              }
              return data;
            }, [{}, {}]);
            if (shouldRecord) {
              stm.onModelUpdate(this, newData, oldData, isChronoFieldSet);
            }
          }
        }
        // No super on purpose, micro optimization of critical perf path
        // super.afterSet?.afterSet(field, value, silent, fromRelationUpdate, wasSet, isChronoFieldSet);
      }
      /**
       * Called from {@link Core/data/mixin/TreeNode#function-insertChild} to obtain inserted
       * records initial parents and parent index, to be able to restore the state back upon undo.
       *
       * @param {Core.data.Model[]} childRecords
       * @returns {Array} Array of results from this call and any of super calls if any.
       *               This result is consumed by {@link #function-afterInsertChild} which pops
       *               from the result array to take only results of this method call and leave
       *               results from super calls untouched.
       *
       * @private
       */
      beforeInsertChild(childRecords) {
        var _super$beforeInsertCh;
        const preResult = ((_super$beforeInsertCh = super.beforeInsertChild) === null || _super$beforeInsertCh === void 0 ? void 0 : _super$beforeInsertCh.call(this, childRecords)) || [],
          {
            stm
          } = this;
        if (stm !== null && stm !== void 0 && stm.enabled) {
          preResult.push(childRecords.reduce((result, childRecord) => {
            // We are interested only in records from the same root node.
            // Removing (which is done before insertion) of the records
            // from another root (and store) should
            // be handled by that store STM instance.
            if (childRecord.root === this.root) {
              result.set(childRecord, {
                parent: childRecord.parent,
                index: childRecord.parent ? childRecord.parentIndex : undefined
              });
            }
            return result;
          }, new Map()));
        }
        return preResult;
      }
      /**
       * Called from {@link Core/data/mixin/TreeNode#function-insertChild} to notify {@link Core/data/stm/StateTrackingManager}
       * about children insertion. Provides it with all necessary context information collected
       * in {@link #function-beforeInsertChild} required to undo/redo the action.
       *
       * @private
       */
      afterInsertChild(index, childRecords, beforeResult, inserted) {
        var _super$afterInsertChi;
        const {
          stm
        } = this;
        if (stm !== null && stm !== void 0 && stm.enabled) {
          const context = beforeResult.pop();
          if (inserted) {
            stm.onModelInsertChild(this, index, inserted, context);
          }
        }
        (_super$afterInsertChi = super.afterInsertChild) === null || _super$afterInsertChi === void 0 ? void 0 : _super$afterInsertChi.call(this, index, childRecords, beforeResult, inserted);
      }
      /**
       * Called from {@link Core/data/mixin/TreeNode#function-removeChild} to obtain removed
       * records initial parent index, to be able to restore the state back upon undo.
       *
       * @param {Core.data.Model[]} childRecords
       * @param {Boolean} isMove
       * @returns {Array} Array of results from this call and any of super calls if any.
       *               This result is consumed by {@link #function-afterRemoveChild} which pops
       *               from the result array to take only results of this method call and leave
       *               results from super calls untouched.
       *
       * @private
       */
      beforeRemoveChild(childRecords, isMove) {
        const preResult = super.beforeRemoveChild ? super.beforeRemoveChild(childRecords, isMove) : [],
          {
            stm
          } = this;
        // If it's move then InsertChildAction will handle this case
        if (stm !== null && stm !== void 0 && stm.enabled && !isMove) {
          // Child records we receive are guaranteed to be direct children
          // of this node, see Core/data/mixin/TreeNode#removeChild method
          // Here we interested in the original index for each child removed,
          // we collect it and store for future use in RemoveChildAction
          preResult.push(childRecords.reduce((result, childRecord) => {
            result.set(childRecord, {
              parentIndex: childRecord.parentIndex,
              orderedParentIndex: childRecord.orderedParentIndex
            });
            return result;
          }, new Map()));
        }
        return preResult;
      }
      /**
       * Called from {@link Core/data/mixin/TreeNode#function-removeChild} to notify {@link Core/data/stm/StateTrackingManager}
       * about children removing. Provides it with all necessary context information collected
       * in {@link #function-beforeRemoveChild} required to undo/redo the action.
       *
       * @private
       */
      afterRemoveChild(childRecords, beforeResult, isMove) {
        var _super$afterRemoveChi;
        const {
          stm
        } = this;
        // If it's move then InsertChildAction will handle this case
        if (stm !== null && stm !== void 0 && stm.enabled && !isMove) {
          const context = beforeResult.pop();
          if (childRecords && childRecords.length) {
            stm.onModelRemoveChild(this, childRecords, context);
          }
        }
        (_super$afterRemoveChi = super.afterRemoveChild) === null || _super$afterRemoveChi === void 0 ? void 0 : _super$afterRemoveChi.call(this, childRecords, beforeResult, isMove);
      }
    });

    /**
     * @module Core/data/mixin/TreeNode
     */
    const defaultTraverseOptions$1 = {
        includeFilteredOutRecords: false
      },
      fixTraverseOptions$1 = options => {
        options = options || false;
        if (typeof options === 'boolean') {
          options = {
            includeFilteredOutRecords: options
          };
        }
        return options || defaultTraverseOptions$1;
      };
    /**
     * Mixin for Model with tree node related functionality. This class is mixed into the {@link Core/data/Model} class.
     *
     * ## Adding and removing child nodes
     *
     * ```javascript
     * const parent = store.getById(1),
     *
     * firstBorn = parent.insertChild({
     *     name : 'Child node'
     * }, parent.children[0]); // Insert a child at a specific place in the children array
     *
     * parent.removeChild(parent.children[0]); // Removes a child node
     * parent.appendChild({ name : 'New child node' }); // Appends a child node
     * ```
     *
     * @mixin
     */
    var TreeNode = (Target => class TreeNode extends (Target || Base$1) {
      static get $name() {
        return 'TreeNode';
      }
      /**
       * This static configuration option allows you to control whether an empty parent task should be converted into a
       * leaf. Enable/disable it for a whole class:
       *
       * ```javascript
       * Model.convertEmptyParentToLeaf = false;
       * ```
       *
       * By specifying `true`, all empty parents will be considered leafs. Can also be assigned a configuration object
       * with the following Boolean properties to customize the behaviour:
       *
       * ```javascript
       * Model.convertEmptyParentToLeaf = {
       *     onLoad   : false,
       *     onRemove : true
       * }
       * ```
       *
       * @member {Boolean|{ onLoad : Boolean, onRemove : Boolean }} convertEmptyParentToLeaf
       * @property {Boolean} onLoad Apply the transformation on load to any parents without children (`children : []`)
       * @property {Boolean} onRemove Apply the transformation when all children have been removed from a parent
       * @default false
       * @static
       * @category Parent & children
       * */
      static set convertEmptyParentToLeaf(value) {
        if (value === true) {
          value = {
            onLoad: true,
            onRemove: true
          };
        } else if (value === false) {
          value = {
            onLoad: false,
            onRemove: false
          };
        }
        this._convertEmptyParentToLeaf = value;
      }
      constructor(...args) {
        super(...args);
        if (this.children) {
          this.orderedChildren = this.orderedChildren || [];
        }
      }
      static get convertEmptyParentToLeaf() {
        return this._convertEmptyParentToLeaf || {
          onLoad: false,
          onRemove: false
        };
      }
      /**
       * This is a read-only property providing access to the parent node.
       * @member {Core.data.Model} parent
       * @readonly
       * @category Parent & children
       */
      /**
       * This is a read-only field provided in server synchronization packets to specify
       * which record id is the parent of the record.
       * @readonly
       * @field {String|Number|null} parentId
       * @category Tree
       */
      /**
       * This is a read-only field provided in server synchronization packets to specify
       * which position the node takes in the parent's children array.
       * This index is set on load and gets updated automatically after row reordering, sorting, etc.
       * To save the order, need to persist the field on the server and when data is fetched to be loaded,
       * need to sort by this field.
       * @readonly
       * @field {Number} parentIndex
       * @category Tree
       */
      /**
       * This is a read-only field provided in server synchronization packets to specify
       * which position the node takes in the parent's ordered children array.
       * This index is set on load and gets updated on reordering nodes in tree. Sorting and filtering
       * have no effect on it.
       * @readonly
       * @field {Number} orderedParentIndex
       * @category Tree
       */
      ingestChildren(childRecord, stores = this.stores) {
        const {
            inProcessChildren,
            constructor: MyClass
          } = this,
          store = stores[0];
        if (childRecord === true) {
          if (inProcessChildren) {
            return true;
          }
          return [];
        }
        if (childRecord) {
          childRecord = ArrayHelper.asArray(childRecord);
          const len = childRecord.length,
            result = [];
          for (let i = 0, child; i < len; i++) {
            child = childRecord[i];
            child = child.isModel ? child : store ? store.createRecord(child, false, true) : new MyClass(child, null, null, true);
            child = store ? store.processRecord(child) : child;
            result.push(child);
          }
          if (this.children === true && store) {
            const sorter = store.createSorterFn(store.sorters);
            result.sort(sorter);
          }
          return result;
        }
      }
      /**
       * Child nodes. To allow loading children on demand, specify `children : true` in your data. Omit the field for leaf
       * tasks.
       *
       * Note, if the tree store loads data from a remote origin, make sure {@link Core/data/AjaxStore#config-readUrl}
       * is specified, and optionally {@link Core/data/AjaxStore#config-parentIdParamName} is set, otherwise
       * {@link Core/data/Store#function-loadChildren} has to be implemented.
       *
       * @field {Boolean|Object[]|Core.data.Model[]} children
       * @category Parent & children
       */
      /**
       * Array of sorted tree nodes but without a filter applied
       * @member {Core.data.Model[]|null} unfilteredChildren
       * @category Parent & children
       * @private
       */
      /**
       * Array of children unaffected by sorting and filtering, keeps original tree structure
       * @member {Core.data.Model[]|null} orderedChildren
       * @category Parent & children
       * @private
       */
      /**
       * Called during creation to also turn any children into Models joined to the same stores as this model
       * @internal
       * @category Parent & children
       */
      processChildren(stores = this.stores) {
        const me = this,
          {
            meta
          } = me;
        me.inProcessChildren = true;
        const children = me.ingestChildren(me.data[me.constructor.childrenField], stores);
        if (children) {
          const {
              convertEmptyParentToLeaf
            } = me.constructor,
            shouldConvert = convertEmptyParentToLeaf === true || convertEmptyParentToLeaf.onLoad;
          if (shouldConvert ? children.length : Array.isArray(children)) {
            meta.isLeaf = false;
            // We are processing a remote load
            if (me.children === true) {
              me.children = [];
            } else if (children.length === 0) {
              me.children = children;
              return;
            }
            me.insertChild(children);
          }
          // Flagged for load on demand
          else if (children === true) {
            meta.isLeaf = false;
            me.children = true;
          }
          // Empty child array, flag is leaf if configured to do so
          else if (!me.isRoot) {
            meta.isLeaf = me.constructor.convertEmptyParentToLeaf.onLoad;
          }
        }
        me.inProcessChildren = false;
      }
      /**
       * This method returns `true` if this record has all expanded ancestors and is therefore
       * eligible for inclusion in a UI.
       * @param {Core.data.Store} [store] Optional store, defaults to nodes first store
       * @returns {Boolean}
       * @readonly
       * @category Parent & children
       * @returns {Boolean}
       */
      ancestorsExpanded(store = this.firstStore) {
        const {
          parent
        } = this;
        return !parent || parent.isExpanded(store) && parent.ancestorsExpanded(store);
      }
      /**
       * Used by stores to assess the record's collapsed/expanded state in that store.
       * @param {Core.data.Store} store
       * @category Parent & children
       * @returns {Boolean}
       */
      isExpanded(store = this.firstStore) {
        const mapMeta = this.instanceMeta(store.id);
        // Default initial expanded/collapsed state when in the store
        // to the record's original expanded property.
        if (!Object.prototype.hasOwnProperty.call(mapMeta, 'collapsed')) {
          mapMeta.collapsed = !this.expanded;
        }
        return !mapMeta.collapsed;
      }
      // A read-only property. It provides the initial state upon load
      // The UI's expanded/collapsed state is in the store's meta map.
      get expanded() {
        return this.data.expanded;
      }
      /**
       * Depth in the tree at which this node exists. First visual level of nodes are at level 0, their direct children at
       * level 1 and so on.
       * @property {Number}
       * @readonly
       * @category Parent & children
       */
      get childLevel() {
        let node = this,
          ret = -1;
        while (node && !node.isRoot) {
          ++ret;
          node = node.parent;
        }
        return ret;
      }
      /**
       * Is a leaf node in a tree structure?
       * @property {Boolean}
       * @readonly
       * @category Parent & children
       */
      get isLeaf() {
        return this.meta.isLeaf !== false && !this.isRoot;
      }
      /**
       * Returns `true` if this node is the root of the tree
       * @member {Boolean} isRoot
       * @readonly
       * @category Parent & children
       */
      /**
       * Is a parent node in a tree structure?
       * @property {Boolean}
       * @readonly
       * @category Parent & children
       */
      get isParent() {
        return !this.isLeaf;
      }
      /**
       * Returns true for parent nodes with children loaded (there might still be no children)
       * @property {Boolean}
       * @readonly
       * @category Parent & children
       */
      get isLoaded() {
        return this.isParent && Array.isArray(this.children);
      }
      /**
       * Count all children (including sub-children) for a node (in its `firstStore´)
       * @member {Number}
       * @category Parent & children
       */
      get descendantCount() {
        return this.getDescendantCount();
      }
      /**
       * Count visible (expanded) children (including sub-children) for a node (in its `firstStore`)
       * @member {Number}
       * @category Parent & children
       */
      get visibleDescendantCount() {
        return this.getDescendantCount(true);
      }
      /**
       * Count visible (expanded)/all children for this node, optionally specifying for which store.
       * @param {Boolean} [onlyVisible] Specify `true` to only count visible (expanded) children.
       * @param {Core.data.Store} [store] A Store to which this node belongs
       * @returns {Number}
       * @category Parent & children
       */
      getDescendantCount(onlyVisible = false, store = this.firstStore) {
        const {
          children
        } = this;
        if (!children || !Array.isArray(children) || onlyVisible && !this.isExpanded(store)) {
          return 0;
        }
        return children.reduce((count, child) => count + child.getDescendantCount(onlyVisible), children.length);
      }
      /**
       * Retrieve all children, not including filtered out nodes (by traversing sub nodes)
       * @property {Core.data.Model[]}
       * @category Parent & children
       */
      get allChildren() {
        return this.getAllChildren(false);
      }
      /**
       * Retrieve all children, including filtered out nodes (by traversing sub nodes)
       * @property {Core.data.Model[]}
       * @private
       * @category Parent & children
       */
      get allUnfilteredChildren() {
        return this.getAllChildren(true);
      }
      getAllChildren(unfiltered = false) {
        const {
          [unfiltered ? 'unfilteredChildren' : 'children']: children
        } = this;
        if (!children || children === true) {
          return [];
        }
        return children.reduce((all, child) => {
          all.push(child);
          // push.apply is faster than push with array spread:
          // https://jsperf.com/push-apply-vs-push-with-array-spread/1
          all.push.apply(all, unfiltered ? child.allUnfilteredChildren : child.allChildren);
          return all;
        }, []);
      }
      /**
       * Get the first child of this node
       * @property {Core.data.Model}
       * @readonly
       * @category Parent & children
       */
      get firstChild() {
        const {
          children
        } = this;
        return (children === null || children === void 0 ? void 0 : children.length) && children[0] || null;
      }
      /**
       * Get the last child of this node
       * @property {Core.data.Model}
       * @readonly
       * @category Parent & children
       */
      get lastChild() {
        const {
          children
        } = this;
        return (children === null || children === void 0 ? void 0 : children.length) && children[children.length - 1] || null;
      }
      /**
       * Get the previous sibling of this node
       * @member {Core.data.Model} previousSibling
       * @readonly
       * @category Parent & children
       */
      /**
       * Get the next sibling of this node
       * @member {Core.data.Model} nextSibling
       * @readonly
       * @category Parent & children
       */
      /**
       * Returns count of all preceding sibling nodes (including their children).
       * @property {Number}
       * @category Parent & children
       */
      get previousSiblingsTotalCount() {
        let task = this.previousSibling,
          count = this.parentIndex;
        while (task) {
          count += task.descendantCount;
          task = task.previousSibling;
        }
        return count;
      }
      get previousOrderedSibling() {
        var _this$parent;
        return (_this$parent = this.parent) === null || _this$parent === void 0 ? void 0 : _this$parent.orderedChildren[this.orderedParentIndex - 1];
      }
      get nextOrderedSibling() {
        var _this$parent2;
        return (_this$parent2 = this.parent) === null || _this$parent2 === void 0 ? void 0 : _this$parent2.orderedChildren[this.orderedParentIndex + 1];
      }
      get root() {
        var _this$parent3;
        return ((_this$parent3 = this.parent) === null || _this$parent3 === void 0 ? void 0 : _this$parent3.root) || this;
      }
      /**
       * Reading this property returns the id of the parent node, if this record is a child of a node.
       *
       * Setting this property appends this record to the record with the passed id **in the same store that this record
       * is already in**.
       *
       * Note that setting this property is **only valid if this record is already part of a tree store**.
       *
       * This is not intended for general use. This is for when a server responds to a record mutation and the server
       * decides to move a record to a new parent. If a `parentId` property is passed in the response data for a record,
       * that record will be moved.
       *
       * @property {Number|String|null}
       * @category Parent & children
       */
      get parentId() {
        return this.parent && !this.parent.isAutoRoot ? this.parent.id : null;
      }
      set parentId(parentId) {
        const me = this,
          {
            parent
          } = me,
          newParent = parentId === null ? me.firstStore.rootNode : me.firstStore.getById(parentId);
        // Handle exact equality of parent.
        // Also handle one being null and the other being undefined meaning no change.
        if (!(newParent === parent || !parent && !newParent)) {
          // If we are batching, we do not trigger a change immediately.
          // endBatch will set the field which will set the property again.
          if (me.isBatchUpdating) {
            me.meta.batchChanges.parentId = parentId;
          } else {
            if (newParent) {
              newParent.appendChild(me);
            } else {
              me.parent.removeChild(me);
            }
          }
        }
      }
      static set parentIdField(parentIdField) {
        // Maintainer: the "this" references in here reference two different contexts.
        // Outside of the property definition, it's the Model Class.
        // In the getter and setter, it's the record instance.
        this._parentIdField = parentIdField;
        Object.defineProperty(this.prototype, parentIdField, {
          set: function (parentId) {
            // no arrow functions here, need `this` to change to instance
            // noinspection JSPotentiallyInvalidUsageOfClassThis
            this.parentId = parentId;
          },
          get: function () {
            // no arrow functions here, need `this` to change to instance
            // noinspection JSPotentiallyInvalidUsageOfClassThis
            return this.parentId;
          }
        });
      }
      static get parentIdField() {
        return this._parentIdField || 'parentId';
      }
      getChildren(options) {
        let result;
        if (options.includeFilteredOutRecords) {
          result = this.unfilteredChildren || this.children;
        } else if (options.useOrderedTree) {
          result = this.orderedChildren;
        } else {
          result = this.children;
        }
        return result;
      }
      /**
       * Traverses all child nodes recursively calling the passed function
       * on a target node **before** iterating the child nodes.
       * @param {Function} fn The function to call
       * @param {Boolean} [skipSelf=false] True to ignore self
       * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
       * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
       * @param {Boolean} [options.useOrderedTree] True to traverse unsorted/unfiltered tree
       * @category Parent & children
       */
      traverse(fn, skipSelf, options) {
        options = fixTraverseOptions$1(options);
        const me = this,
          children = me.getChildren(options);
        if (!skipSelf) {
          fn.call(me, me);
        }
        // Simply testing whether there is non-zero children length
        // is 10x faster than using this.isLoaded
        for (let i = 0, l = children === null || children === void 0 ? void 0 : children.length; i < l; i++) {
          children[i].traverse(fn, false, options);
        }
      }
      /**
       * Traverses all child nodes recursively calling the passed function
       * on child nodes of a target **before** calling it on the node.
       * @param {Function} fn The function to call
       * @param {Boolean} [skipSelf=false] True to skip this node in the traversal
       * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
       * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
       * @category Parent & children
       */
      traverseBefore(fn, skipSelf, options) {
        options = fixTraverseOptions$1(options);
        const me = this,
          children = me.getChildren(options);
        // Simply testing whether there is non-zero children length
        // is 10x faster than using me.isLoaded
        for (let i = 0, l = children === null || children === void 0 ? void 0 : children.length; i < l; i++) {
          children[i].traverse(fn, false, options);
        }
        if (!skipSelf) {
          fn.call(me, me);
        }
      }
      /**
       * Traverses child nodes recursively while fn returns true
       * @param {Function} fn
       * @param {Boolean} [skipSelf=false] True to skip this node in the traversal
       * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
       * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
       * @category Parent & children
       * @returns {Boolean}
       */
      traverseWhile(fn, skipSelf, options) {
        options = fixTraverseOptions$1(options);
        const me = this;
        let goOn = skipSelf || fn.call(me, me) !== false;
        if (goOn) {
          const children = me.getChildren(options);
          // Simply testing whether there is non-zero children length
          // is 10x faster than using me.isLoaded
          if (children !== null && children !== void 0 && children.length) {
            goOn = children.every(child => child.traverseWhile(fn, false, options));
          }
        }
        return goOn;
      }
      /**
       * Bubbles up from this node, calling the specified function with each node.
       *
       * @param {Function} fn The function to call for each node
       * @param {Boolean} [skipSelf] True to skip this node in the traversal
       * @category Parent & children
       */
      bubble(fn, skipSelf = false) {
        let me = this;
        if (!skipSelf) {
          fn.call(me, me);
        }
        while (me.parent) {
          me = me.parent;
          fn.call(me, me);
        }
      }
      /**
       * Bubbles up from this node, calling the specified function with each node,
       * while the function returns true.
       *
       * @param {Function} fn The function to call for each node
       * @param {Boolean} [skipSelf] True to skip this node in the traversal
       * @category Parent & children
       * @returns {Boolean}
       */
      bubbleWhile(fn, skipSelf = false) {
        let me = this,
          goOn = true;
        if (!skipSelf) {
          goOn = fn.call(me, me);
        }
        while (goOn && me.parent) {
          me = me.parent;
          goOn = fn.call(me, me);
        }
        return goOn;
      }
      /**
       * Checks if this model contains another model as one of it's descendants
       *
       * @param {Core.data.Model|String|Number} childOrId child node or id
       * @param {Boolean} [skipSelf=false] True to ignore self in the traversal
       * @category Parent & children
       * @returns {Boolean}
       */
      contains(childOrId, skipSelf = false) {
        if (childOrId && typeof childOrId === 'object') {
          childOrId = childOrId.id;
        }
        return !this.traverseWhile(node => node.id != childOrId, skipSelf);
      }
      getTopParent(all) {
        let result;
        if (all) {
          result = [];
          this.bubbleWhile(t => {
            result.push(t);
            return t.parent && !t.parent.isRoot;
          });
        } else {
          result = null;
          this.bubbleWhile(t => {
            result = t;
            return t.parent && !t.parent.isRoot;
          });
        }
        return result;
      }
      /**
       * Append a child record(s) to any current children.
       * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} childRecord Array of records/data or a single
       * record/data to append
       * @param {Boolean} [silent] Pass `true` to not trigger events during append
       * @returns {Core.data.Model|Core.data.Model[]|null}
       * @category Parent & children
       */
      appendChild(childRecord, silent = false) {
        return this.insertChild(childRecord, null, silent);
      }
      /**
       * Insert a child record(s) before an existing child record.
       * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} childRecord Array of records/data or a single
       * record/data to insert
       * @param {Core.data.Model} [before] Optional record to insert before, leave out to append to the end
       * @param {Boolean} [silent] Pass `true` to not trigger events during append
       * @returns {Core.data.Model|Core.data.Model[]|null}
       * @category Parent & children
       */
      insertChild(childRecord, before = null, silent = false, options = {}) {
        var _before, _me$children2, _me$beforeInsertChild, _me$afterInsertChild;
        const me = this,
          returnArray = Array.isArray(childRecord);
        childRecord = ArrayHelper.asArray(childRecord);
        if (typeof before === 'number') {
          var _me$children;
          before = ((_me$children = me.children) === null || _me$children === void 0 ? void 0 : _me$children[before]) ?? null;
        }
        if (!silent && !me.stores.every(s => s.trigger('beforeAdd', {
          records: childRecord,
          parent: me
        }) !== false)) {
          return null;
        }
        // This call makes child record an array containing Models
        childRecord = me.ingestChildren(childRecord);
        // NOTE: see comment in Model::set() about before/in/after calls approach.
        const index = ((_before = before) === null || _before === void 0 ? void 0 : _before.parentIndex) ?? ((_me$children2 = me.children) === null || _me$children2 === void 0 ? void 0 : _me$children2.length) ?? 0,
          preResult = (_me$beforeInsertChild = me.beforeInsertChild) === null || _me$beforeInsertChild === void 0 ? void 0 : _me$beforeInsertChild.call(me, childRecord),
          inserted = me.internalAppendInsert(childRecord, before, silent, options);
        // Turn into a parent if not already one
        if (inserted.length) {
          me.convertToParent(silent);
        }
        (_me$afterInsertChild = me.afterInsertChild) === null || _me$afterInsertChild === void 0 ? void 0 : _me$afterInsertChild.call(me, index, childRecord, preResult, inserted);
        return returnArray || !inserted ? inserted : inserted[0];
      }
      /**
       * Converts a leaf node to a parent node, assigning an empty array as its children
       * @param {Boolean} [silent] Pass `true` to not trigger any event
       * @category Parent & children
       */
      convertToParent(silent = false) {
        const me = this,
          wasLeaf = me.isLeaf;
        me.meta.isLeaf = false;
        if (!me.children) {
          me.children = [];
        }
        // Signal a change event so that the UI updates, unless it is during load in which case StoreTree#onNodeAddChild
        // will handle it
        if (wasLeaf && !me.root.isLoading && !silent) {
          me.signalNodeChanged({
            isLeaf: {
              value: false,
              oldValue: true
            }
          });
        }
      }
      signalNodeChanged(changes, stores = this.stores) {
        stores.forEach(s => {
          s.trigger('update', {
            record: this,
            records: [this],
            changes
          });
          s.trigger('change', {
            action: 'update',
            record: this,
            records: [this],
            changes
          });
        });
      }
      tryInsertChild() {
        return this.insertChild(...arguments);
      }
      internalAppendInsert(recordsToInsert, beforeRecord, silent, options) {
        const me = this,
          {
            stores,
            root,
            children
          } = me,
          {
            firstStore: rootStore
          } = root,
          {
            parentIdField
          } = me.constructor,
          parentId = me.id;
        let isNoop, start, i, newRecordsCloned, oldParentIndices, isMove;
        if (!root.isLoading && rootStore) {
          // Only collect this info if not loading, to not produce garbage
          isMove = {};
          oldParentIndices = [];
          for (i = 0; i < recordsToInsert.length; i++) {
            const newRecord = recordsToInsert[i];
            // Store added should not be modified for adds
            // caused by moving.
            isMove[newRecord.id] = newRecord.root === root;
            oldParentIndices[i] = newRecord.parentIndex;
          }
        }
        // The reference node must be one of our children. If not, fall back to an append.
        if (beforeRecord && beforeRecord.parent !== me) {
          beforeRecord = null;
        }
        // If the records starting at insertAt or (insertAt - 1), are the same sequence
        // that we are being asked to add, this is a no-op.
        if (children) {
          const insertAt = beforeRecord ? beforeRecord.parentIndex : children.length;
          if (children[start = insertAt] === recordsToInsert[0] || children[start = insertAt - 1] === recordsToInsert[0]) {
            for (isNoop = true, i = 0; isNoop && i < recordsToInsert.length; i++) {
              if (recordsToInsert[i] !== children[start + i]) {
                isNoop = false;
              }
            }
          }
        }
        // Fulfill the contract of appendChild/insertChild even if we did not have to do anything.
        // Callers must be able to correctly postprocess the returned value as an array.
        if (isNoop) {
          return recordsToInsert;
        }
        // Remove incoming child nodes from any current parent.
        for (i = 0; i < recordsToInsert.length; i++) {
          var _isMove;
          const newRecord = recordsToInsert[i],
            oldParent = newRecord.parent;
          // Check if any descendants of the added node are moves.
          if (rootStore && !root.isLoading) {
            newRecord.traverse(r => {
              if (r.root === root) {
                isMove[r.id] = true;
              }
            });
          }
          // If the new record has a parent, remove from that parent.
          // This operation may be vetoed by listeners.
          // If it is vetoed, then remove from the newRecords and do not
          // set the parent property
          if ((oldParent === null || oldParent === void 0 ? void 0 : oldParent.removeChild(newRecord, (_isMove = isMove) === null || _isMove === void 0 ? void 0 : _isMove[newRecord.id], silent, {
            isInserting: true,
            ...options
          })) === false) {
            if (!newRecordsCloned) {
              recordsToInsert = recordsToInsert.slice();
              newRecordsCloned = true;
            }
            recordsToInsert.splice(i--, 1);
          } else {
            newRecord.parent = me;
            // Set parentId directly to data, record.parentId uses a getter to return record.parent.id
            newRecord.data[parentIdField] = parentId;
            const {
              meta
            } = newRecord;
            // it seems we rely on setting the `oldParentId` flag on the "meta" object for managing `parentId` field
            // presence in the `meta.modified` in the code below
            // however, `oldParent` might be missing in case of STM removal restoring for example, but
            // `oldParentId` should still be set
            // covered with `Core/tests/data/stm/TreeRemoval.t.js//Should not include `parentId` to modifications after undo-ing the child removal`
            if (meta.modified[parentIdField] === parentId && !oldParent) {
              meta.oldParentId = parentId;
            }
            if (oldParent) {
              meta.oldParentId = oldParent.id;
            }
          }
        }
        // Still records to insert after beforeRemove listeners may have vetoed some
        if (recordsToInsert.length) {
          if (!Array.isArray(children)) {
            me.children = [];
          }
          if (!Array.isArray(me.orderedChildren)) {
            me.orderedChildren = [];
          }
          // Add to the children
          const insertAt = me.addToChildren(beforeRecord, recordsToInsert, options);
          stores.forEach(store => {
            if (!store.isChained) {
              recordsToInsert.forEach(record => {
                // Initialize context for newly added records
                record.joinStore(store);
              });
              // Add to store (will also add any child records and trigger events)
              store.onNodeAddChild(me, recordsToInsert, insertAt, isMove, silent);
              recordsToInsert.forEach((record, i) => {
                // If we are in the recursive inclusion of children at construction
                // time, or in a store load, that must not be a data modification.
                // Otherwise, we have to signal a change
                if (record.meta.oldParentId != null && !(me.inProcessChildren || me.isLoading)) {
                  const toSet = {
                      [parentIdField]: parentId,
                      [me.getDataSource('parentIndex')]: record.parentIndex
                    },
                    wasSet = {},
                    {
                      modified,
                      oldParentId
                    } = record.meta,
                    oldParentIndex = oldParentIndices[i];
                  delete record.meta.oldParentId;
                  if (me.id !== oldParentId) {
                    wasSet[parentIdField] = {
                      value: parentId,
                      oldValue: oldParentId
                    };
                  }
                  if (record.parentIndex !== oldParentIndex) {
                    wasSet.parentIndex = {
                      value: record.parentIndex,
                      oldValue: oldParentIndex
                    };
                  }
                  // Changing back to its original value
                  if (modified[parentIdField] === me.id) {
                    Reflect.deleteProperty(modified, parentIdField);
                  }
                  // Cache its original value
                  else if (!(parentIdField in modified)) {
                    modified[parentIdField] = oldParentId;
                  }
                  if (isMove[record.id]) {
                    const oldParent = store.getById(oldParentId);
                    // If old parent transitioned to being a leaf node, signal a change event so that the UI
                    // updates. Handled here and not on remove to get the correct order of events on move
                    if (oldParent.isLeaf && !silent) {
                      oldParent.signalNodeChanged({
                        isLeaf: {
                          value: true,
                          oldValue: false
                        }
                      }, [store]);
                    }
                  }
                  record.afterChange(toSet, wasSet);
                }
                // should only perform this after all changes to `node.modified` are completed
                record.traverse(node => {
                  if (!node.ignoreBag && !node.isLinked) {
                    store.updateModifiedBagForRecord(node);
                  }
                });
              });
            }
          });
        }
        return recordsToInsert;
      }
      /**
       * Remove a child record. Only direct children of this node can be removed, others are ignored.
       * @param {Core.data.Model|Core.data.Model[]} childRecords The record(s) to remove.
       * @param {Boolean} [isMove] Pass `true` if the record is being moved within the same store.
       * @param {Boolean} [silent] Pass `true` to not trigger events during remove.
       * @privateparam {Object} [options]
       * @privateparam {Object} [options.isInserting] `true` is passed when removal is part of record inserting (acted on by
       * ModelLink)
       * @returns {Core.data.Model[]} All records (including nested children) removed
       * @category Parent & children
       */
      removeChild(childRecords, isMove = false, silent = false, options = {}) {
        var _me$beforeRemoveChild, _me$afterRemoveChild;
        const me = this,
          allRemovedRecords = [],
          wasLeaf = me.isLeaf,
          {
            children,
            stores
          } = me;
        childRecords = ArrayHelper.asArray(childRecords);
        childRecords = childRecords.filter(r => r.parent === me);
        if (!silent) {
          // Allow store listeners to veto the beforeRemove event
          for (const store of stores) {
            if (!store.isChained && store.trigger('beforeRemove', {
              parent: me,
              records: childRecords,
              isMove
            }) === false) {
              return false;
            }
          }
        }
        const preResult = (_me$beforeRemoveChild = me.beforeRemoveChild) === null || _me$beforeRemoveChild === void 0 ? void 0 : _me$beforeRemoveChild.call(me, childRecords, isMove);
        for (const childRecord of childRecords) {
          const {
              parentIdField
            } = childRecord.constructor,
            {
              modified
            } = childRecord.meta,
            oldParentId = childRecord.parent ? childRecord.parent.id : null;
          // Cache its original value (not if it is a link, that would pollute original)
          if (!(parentIdField in modified) && !childRecord.isLinked) {
            modified[parentIdField] = oldParentId;
          }
          const index = me.removeFromChildren(childRecord, options);
          stores.forEach(store => {
            if (!store.isChained) {
              const {
                isRemoving
              } = store;
              // Raise the store isRemoving flag (it's set in Store#remove() but not when we call record#removeChild() directly)
              store.isRemoving = true;
              allRemovedRecords.push(...store.onNodeRemoveChild(me, [childRecord], index, {
                isMove,
                silent
              }));
              // restore the flag initial state
              store.isRemoving = isRemoving;
            }
          });
          // No need to clean up the node parent info and other meta data in case it is "move" operation. The info will be updated after "insert" operation.
          if (!isMove) {
            childRecord.parent = childRecord.parentIndex = childRecord.unfilteredIndex = childRecord.nextSibling = childRecord.previousSibling = null;
            // Reset parentId in data, record.parentId uses a getter to return record.parent.id
            childRecord.data[parentIdField] = null;
          }
        }
        // Convert emptied parent into leaf if configured to do so
        if ((me.unfilteredChildren || children).length === 0 && me.constructor.convertEmptyParentToLeaf.onRemove && !me.isRoot) {
          me.meta.isLeaf = true;
        }
        // If we've transitioned to being a leaf node, signal a change event so that the UI updates
        // (but not if part of move, will be signaled by insert)
        if (me.isLeaf !== wasLeaf && !silent && !isMove) {
          me.signalNodeChanged({
            isLeaf: {
              value: true,
              oldValue: false
            }
          });
        }
        (_me$afterRemoveChild = me.afterRemoveChild) === null || _me$afterRemoveChild === void 0 ? void 0 : _me$afterRemoveChild.call(me, childRecords, preResult, isMove);
        return allRemovedRecords;
      }
      clearParentId() {
        const me = this;
        Reflect.deleteProperty(me.data, me.parentIdField);
        Reflect.deleteProperty(me.originalData, me.parentIdField);
        if (me.meta.modified) {
          Reflect.deleteProperty(me.meta.modified, me.parentIdField);
        }
      }
      /**
       * Replaces all child nodes with the new node set.
       * @param {Core.data.Model|Core.data.Model[]} childRecords The new child record set.
       * @returns {Core.data.Model[]}
       * @category Parent & children
       */
      replaceChildren(newChildren) {
        this.clearChildren();
        this.data[this.constructor.childrenField] = newChildren;
        this.processChildren();
        return this.children;
      }
      /**
       * Removes all child nodes from this node.
       * @param {Boolean} [silent=false] Pass `true` to not fire Store events during the remove.
       * @returns {Core.data.Model[]}
       * @category Parent & children
       */
      clearChildren(silent = false) {
        const me = this,
          {
            stores
          } = me,
          children = me.unfilteredChildren || me.children;
        me.children = [];
        me.orderedChildren = [];
        if (children && children !== true) {
          stores.forEach(store => {
            if (!store.isChained) {
              // unfiltered:true to unregister children on filtered stores
              store.onNodeRemoveChild(me, children, 0, {
                unfiltered: true,
                silent
              });
            }
          });
          // clear unfilteredChildren (must be after the above loop)
          if (me.unfilteredChildren) {
            me.unfilteredChildren = [];
          }
        }
      }
      /**
       * Removes all records from the rootNode
       * @private
       */
      clear() {
        var _me$children3;
        const me = this,
          {
            stores
          } = me,
          children = (_me$children3 = me.children) === null || _me$children3 === void 0 ? void 0 : _me$children3.slice();
        // Only allow for root node and if data is present
        if (!me.isRoot || !children) {
          return;
        }
        for (const store of stores) {
          if (!store.isChained) {
            if (store.trigger('beforeRemove', {
              parent: me,
              records: children,
              isMove: false,
              removingAll: true
            }) === false) {
              return false;
            }
          }
        }
        me.children.length = 0;
        if (me.unfilteredChildren) {
          me.unfilteredChildren.length = 0;
        }
        stores.forEach(store => {
          children.forEach(child => {
            if (child.stores.includes(store)) {
              // this will drill down the child, unregistering whole branch
              child.unjoinStore(store);
            }
            child.parent = child.parentIndex = child.nextSibling = child.previousSibling = null;
          });
          store.storage.suspendEvents();
          store.storage.clear();
          store.storage.resumeEvents();
          store.added.clear();
          store.modified.clear();
          store.trigger('removeAll');
          store.trigger('change', {
            action: 'removeall'
          });
        });
      }
      updateChildrenIndices(children, indexName, silent = false) {
        let previousSibling = null;
        for (let i = 0; i < children.length; i++) {
          const child = children[i],
            oldValue = child[indexName];
          if (indexName === 'parentIndex' || indexName === 'orderedParentIndex') {
            // Record should not be considered modified by initial assignment of parentIndex
            if (oldValue === undefined || silent) {
              child.setData(indexName, i);
            }
            // Check to avoid pointless beforeUpdates from inSet
            else if (oldValue !== i) {
              // Silent set, do not want to trigger events from updated indices
              child.set(indexName, i, true);
            }
          } else {
            child[indexName] = i;
          }
          if (indexName === 'parentIndex') {
            child.previousSibling = previousSibling;
            if (previousSibling) {
              previousSibling.nextSibling = child;
            }
            // Last child never has a nextSibling
            if (i === children.length - 1) {
              child.nextSibling = null;
            }
            previousSibling = child;
          }
        }
      }
      addToChildren(beforeRecord, newRecords, options = {}) {
        // children can be sorted and filtered
        // unfilteredChildren can not be filtered
        // orderedChildren can not be nor filtered nor sorted. it holds true tree hierarchy
        const me = this,
          configs = [[me.children, 'parentIndex', beforeRecord], [me.unfilteredChildren, 'unfilteredIndex', beforeRecord], [me.orderedChildren, 'orderedParentIndex', (options === null || options === void 0 ? void 0 : options.orderedBeforeNode) ?? ((options === null || options === void 0 ? void 0 : options.orderedParentIndex) !== undefined ? me.orderedChildren[options === null || options === void 0 ? void 0 : options.orderedParentIndex] : beforeRecord)]];
        for (const config of configs) {
          const [children, indexName, beforeRecord] = config;
          if (children) {
            var _options$indexName;
            // On undo/redo it might happen that we redo orderedParentIndex change first breaking indexing. Safe
            // way is to always check array for record index. We also cannot filter out orderedParentIndex change
            // because by itself it is valid.
            const index = beforeRecord ? indexName === 'orderedParentIndex' ? children.indexOf(beforeRecord) : beforeRecord[indexName] : children.length;
            config.push(index);
            children.splice(index, 0, ...newRecords);
            if (!(options !== null && options !== void 0 && (_options$indexName = options[indexName]) !== null && _options$indexName !== void 0 && _options$indexName.skip)) {
              me.updateChildrenIndices(children, indexName);
            }
          }
        }
        // always return index of the record in the children array
        return configs[0][3];
      }
      removeFromChildren(childRecord, options) {
        const configs = [[this.children, 'parentIndex'], [this.unfilteredChildren, 'unfilteredIndex'], [this.orderedChildren, 'orderedParentIndex']];
        for (const config of configs) {
          const [children, indexName] = config;
          if (children) {
            // parentIndex/orderedParentIndex might be changed when applying a remote changeset leading to
            // record getting removed from the wrong position in the children array. Therefore, we should
            // not rely on the index value, instead we query array itself
            const index = children.indexOf(childRecord);
            config.push(index);
            if (index > -1) {
              var _options$indexName2;
              children.splice(index, 1);
              if (!(options !== null && options !== void 0 && (_options$indexName2 = options[indexName]) !== null && _options$indexName2 !== void 0 && _options$indexName2.skip)) {
                this.updateChildrenIndices(children, indexName);
              }
            }
          }
        }
        // always return index of the record in the children array
        return configs[0][2];
      }
      /**
       * Iterates orderedChildren array to apply sorting order according to `orderedParentIndex`.
       * Normally sorting is not required because order is maintained on append/insert. But is useful
       * when pasting number of records to restore their original order.
       * @param {Boolean} [deep=true] True to dive into children. False to sort own children.
       * @param {Boolean} [usePreviousOrder=false] Enable to use previous value of `orderedParentIndex`.
       * @returns {Set} Returns Set of moved nodes which require WBS update
       * @private
       */
      sortOrderedChildren(deep = true, usePreviousOrder = false) {
        // Collect moved nodes, we need to recalculate WBS on them.
        const movedNodes = [];
        if (!this.isLeaf) {
          this.orderedChildren.sort((a, b) => {
            if (usePreviousOrder) {
              const aPrevIndex = a.meta.modified.orderedParentIndex ?? a.orderedParentIndex,
                bPrevIndex = b.meta.modified.orderedParentIndex ?? b.orderedParentIndex,
                result = aPrevIndex - bPrevIndex;
              if (result !== 0) {
                movedNodes.push(a);
                movedNodes.push(b);
              }
              return result;
            } else {
              return a.orderedParentIndex - b.orderedParentIndex;
            }
          });
          if (deep) {
            this.orderedChildren.forEach(child => {
              movedNodes.push(...child.sortOrderedChildren(deep, usePreviousOrder));
            });
          }
          this.updateChildrenIndices(this.orderedChildren, 'orderedParentIndex', true);
        }
        return new Set(movedNodes);
      }
      unjoinStore(store, isReplacing = false) {
        var _super$unjoinStore;
        const me = this;
        // clear the filtering when unjoining the record from the store
        if (me.unfilteredChildren) {
          me.children = me.unfilteredChildren.slice();
          me.unfilteredChildren = null;
        }
        (_super$unjoinStore = super.unjoinStore) === null || _super$unjoinStore === void 0 ? void 0 : _super$unjoinStore.call(this, store, isReplacing);
      }
    });

    /**
     * @module Core/data/mixin/ModelLink
     */
    const
      // Properties set on the proxy instead of on the original
      propertyOverrides = {
        id: 1,
        stores: 1,
        parentIndex: 1,
        parent: 1,
        previousSibling: 1,
        nextSibling: 1,
        unfilteredIndex: 1
      },
      proxyConfig = {
        get(target, prop) {
          // Proxy record has some additional meta
          if (prop === 'proxyMeta') {
            return this.proxyMeta;
          }
          // Accessing constructor in functions should lead to original records constructor
          // (for static fns etc.)
          if (prop === 'constructor') {
            return target.constructor;
          }
          // Override setData / set to reroute parentIndex updates
          if (prop === 'setData') {
            return this.setDataOverride;
          }
          if (prop === 'set') {
            return this.setOverride;
          }
          // Special properties not shared with the original record
          if (propertyOverrides[prop]) {
            return this.proxyMeta.data[prop];
          }
          // Everything else is scoped to the proxy record
          return Reflect.get(target, prop, this.proxyRecord);
        },
        set(target, prop, value) {
          // Special properties not shared with the original record
          if (propertyOverrides[prop]) {
            this.proxyMeta.data[prop] = value;
          }
          // Everything else is relayed to the original record
          else {
            target[prop] = value;
          }
          return true;
        },
        // Override setData & set to reroute parentIndex updates
        setDataOverride(toSet, value) {
          if (toSet === 'parentIndex') {
            this.proxyMeta.data.parentIndex = value;
          } else {
            this.proxyMeta.originalRecord.setData(toSet, value);
          }
        },
        setOverride(field, value, ...args) {
          if (field === 'parentIndex') {
            this.proxyMeta.data.parentIndex = value;
          } else {
            this.proxyMeta.originalRecord.set(field, value, ...args);
          }
        }
      };
    /**
     * Mixin that allows creating proxy records linked to an original record. See {@link #function-link} for more
     * information.
     *
     * <div class="note">Note that not all UI features support linked records</div>
     *
     * @mixin
     */
    var ModelLink = (Target => class ModelLink extends (Target || Base$1) {
      static $name = 'ModelLink';
      /**
       * Creates a proxy record (using native Proxy) linked to this record (the original). The proxy records shares most
       * data with the original, except for its `id` (which is always generated), and ordering fields such as
       * `parentIndex` and `parentId` etc.
       *
       * Any change to the proxy record will be reflected on the original, and vice versa. A proxy record is not meant to
       * be persisted, only the original record should be persisted. Thus, proxy records are not added to stores change
       * tracking (added, modified and removed records).
       *
       * Removing the original record removes all proxies.
       *
       * Creating a proxy record allows a Store to seemingly contain the record multiple times, something that is
       * otherwise not possible. It also allows a record to be used in both a tree store and in a flat store.
       *
       * <div class="note">Note that not all UI features support linked records</div>
       *
       * @returns {Proxy} Proxy record linked to the original record
       * @category Misc
       */
      link() {
        // Calling link on a link creates another link of the original record
        if (this.isLinked) {
          return this.$original.link();
        }
        const me = this,
          useConfig = {
            ...proxyConfig,
            // Data not shared with the original record
            proxyMeta: {
              originalRecord: me,
              data: {
                id: `${me.id}_link_${StringHelper.generateUUID()}`,
                stores: []
              }
            }
          },
          proxyRecord = new Proxy(me, useConfig);
        useConfig.proxyRecord = proxyRecord;
        // Original record keeps tracks of all proxies
        (me.meta.linkedRecords || (me.meta.linkedRecords = [])).push(proxyRecord);
        return proxyRecord;
      }
      /**
       * Is this record linked to another record?
       * @member {Boolean}
       * @readonly
       * @category Misc
       */
      get isLinked() {
        var _this$proxyMeta;
        return Boolean((_this$proxyMeta = this.proxyMeta) === null || _this$proxyMeta === void 0 ? void 0 : _this$proxyMeta.originalRecord);
      }
      /**
       * Are other records linked to this record?
       * @member {Boolean}
       * @readonly
       * @category Misc
       */
      get hasLinks() {
        return Boolean(!this.proxyMeta && this.$links.length);
      }
      // Logic to remove a link shared between removing in a flat store and a tree store
      removeLink(link, records = null, silent = false) {
        // Removing original, also remove linked records
        if (link.hasLinks) {
          for (const linked of link.$links.slice()) {
            // Flat
            if (records) {
              ArrayHelper.include(records, linked);
            }
            // Tree
            else {
              linked.remove(silent);
            }
          }
        }
        // Removing linked record, remove from originals link tracking
        else if (link.isLinked) {
          ArrayHelper.remove(link.$original.$links, link);
        }
      }
      // Overrides beforeRemove in Model, to remove all linked records when original record is removed.
      beforeRemove(records) {
        this.removeLink(this, records);
      }
      // Overrides removeChild in TreeNode, to remove the original node and all linked nodes when either a linked or
      // original node is removed.
      removeChild(childRecords, isMove, silent, options) {
        if (!(options !== null && options !== void 0 && options.isInserting)) {
          childRecords = ArrayHelper.asArray(childRecords);
          for (const child of childRecords) {
            this.removeLink(child, null, silent);
          }
        }
        return super.removeChild(childRecords, isMove, silent, options);
      }
      // Convenience getter for code keying by id that needs to work with both link and original
      get $originalId() {
        return this.$original.id;
      }
      // Convenience getter to retrieve linked records
      get $links() {
        return this.meta.linkedRecords ?? [];
      }
    });

    /**
     * @module Core/mixin/Factoryable
     */
    const {
        defineProperty: defineProperty$5
      } = Reflect,
      ownerSymbol = Symbol('owner'),
      typeSplitRe = /[\s,]+/;
    /**
     * This mixin is applied to base classes of a type that will be dynamically created by type name aliases.
     *
     * ```javascript
     *  class Layout extends Base.mixin(Factoryable) {
     *      static get factoryable() {
     *          return {
     *              defaultType : 'default'
     *          };
     *      }
     *
     *      static get type() {
     *          return 'default';
     *      }
     *  }
     *
     *  class Fit extends Layout {
     *      static get type() {
     *          return 'fit';
     *      }
     *  }
     * ```
     *
     * Once a family of classes has been defined, instances are created using the `create()` method:
     *
     * ```javascript
     *  const layout = Layout.create(config);
     * ```
     *
     * In the above example, `config` can be a type name (such as "fit") or a config object with a `type` property that
     * holds the type name.
     *
     * Factories can also extend other factories. For example, one factory creates objects that are useful across a wide
     * range of consumers, and a second factory creates objects for a more specialized consumer. If that specialized
     * consumer can also consume objects from the first factory, then the second factory can specify this relationship:
     *
     * ```javascript
     *  class General extends Base.mixin(Factoryable) {
     *      ...
     *  }
     *
     *  class Specialized extends Base.mixin(Factoryable) {
     *      static get factoryable() {
     *          return {
     *              extends : General,
     *              ...
     *          };
     *      }
     *  }
     * ```
     *
     * The `extends` factoryable option can be either a class that mixes in `Factoryable` or an array of such classes.
     * @mixin
     * @internal
     */
    var Factoryable = (Target => class Factoryable extends (Target || Base$1) {
      static get $name() {
        return 'Factoryable';
      }
      static get declarable() {
        return [
        /**
         * This property getter returns options that control the factory process. This property getter must be
         * defined by the class that mixes in `Factoryable` in order to initialize the factory properly.
         * ```
         *  static get factoryable() {
         *      return {
         *          defaultType : 'default'
         *      };
         *  }
         * ```
         * If there are no special options to provide, this method can return nothing (`undefined`):
         * ```
         *  static get factoryable() {
         *      // initialize the factory with all default options
         *  }
         * ```
         * @static
         * @member {Object} factoryable
         * @property {Boolean} [factoryable.caseless=true] Specify `false` to use case-sensitive type names. The
         * default is to ignore case.
         * @property {String} [factoryable.defaultType=null] The default type to create when a config object has
         * no `typeKey` property.
         * @property {Function|Function[]} [factoryable.extends] One or more classes that mix in `Factoryable` to
         * use for resolving type names when a type name is not found in this factory.
         * @property {String} [factoryable.typeKey='type'] The name of the property in a config object that holds
         * the type name.
         * @internal
         */
        'factoryable',
        /**
         * One or more additional type name aliases for this class. This can be useful for renaming and maintaining
         * a previous type name.
         * ```
         *  class Fit extends Layout {
         *      static get type() {
         *          return 'fit';
         *      }
         *
         *      static get alias() {
         *          return 'fill';  // deprecated type name (now known as 'fit')
         *      }
         *  }
         * ```
         * @static
         * @member {String|String[]} alias
         * @internal
         */
        'alias',
        /**
         * The (canonical) type name for this class by which instances can be created using the static
         * {@link #function-create-static create()} method.
         * @static
         * @member {String} type
         */
        'type'];
      }
      /**
       * Registers a class (`cls`) associated with the given `type`.
       * @param {String|String[]} type A string, array of strings or a comma-separated string containing the type names
       * for the specified `cls` class.
       * @param {Function} cls The class (constructor function)
       * @param {Boolean} [replace] Pass `true` to overwrite existing registered types. Otherwise, this method will throw
       * an exception if the `type` is already registered with this factory.
       * @internal
       */
      static register(type, cls, replace = globalThis.__BRYNTUM_EXAMPLE) {
        // `replace` default value is set to true while we are run inside a bryntum demo where code editor changes might
        // lead to same widget class being registered over and over.
        const {
            factoryable
          } = this.initClass(),
          {
            caseless,
            registry
          } = factoryable,
          types = StringHelper.split(type, typeSplitRe); // if type is a string[] it will just be returned
        for (let lower, name, i = 0; i < types.length; ++i) {
          name = types[i];
          lower = caseless ? name.toLowerCase() : name;
          if (!replace && lower in registry) {
            throw new Error(`Type "${name}" already registered with ${factoryable.class.name} factory`);
          }
          // Ensure class being registered is initialized. (initClass returns the class)
          registry[name] = registry[lower] = cls.initClass();
        }
      }
      /**
       * Returns `true` if the passed instance is of the passed type or of a derived class.
       * @param {Object} instance The object to test.
       * @param {String} type The type to test against
       */
      static isA(instance, type) {
        return this.isType(instance, type, true);
      }
      /**
       * Returns `true` if the passed instance is of the passed type.
       * @param {Object} instance The object to test.
       * @param {String} type The type to test against
       * @param {Boolean} [deep] Pass `true` to return `true` if the class is a subclass of the passed type.
       */
      static isType(instance, type, deep) {
        const {
            factoryable
          } = this,
          {
            caseless,
            registry
          } = factoryable,
          typeCls = registry[caseless ? type.toLowerCase() : type];
        // If the type to be tested against maps to a class, see if the instance is an instanceof that
        if (typeCls) {
          if (deep) {
            return instance instanceof typeCls;
          }
          return instance.constructor === typeCls;
        }
        return false;
      }
      static setupAlias(cls) {
        cls.register(cls.alias, cls);
      }
      static setupFactoryable(cls, meta) {
        const superClass = meta.super.class;
        let {
          factoryable
        } = cls;
        factoryable = {
          caseless: true,
          defaultType: null,
          extends: superClass.factoryable ? [superClass] : null,
          typeKey: 'type',
          ...factoryable
        };
        factoryable.class = cls;
        factoryable.registry = Object.create(null);
        if (factoryable.extends && !Array.isArray(factoryable.extends)) {
          factoryable.extends = [factoryable.extends];
        }
        // Replace the class/static getter with a new one that returns the complete factoryable object:
        defineProperty$5(cls, 'factoryable', {
          get() {
            return factoryable;
          }
        });
      }
      static setupType(cls, meta) {
        const {
          type
        } = cls;
        cls.register(type, cls, meta.replaceType);
        // Copy the static type property onto the prototype as a readonly property:
        defineProperty$5(cls.prototype, 'type', {
          value: type
        });
      }
      /**
       * Creates an instance from this factory, given the type name or a config object.
       * @param {String|Object} config The type name string or config object.
       * @param {String|Function|Object} [options] Creation options (for details see {@link #function-reconfigure-static}).
       * @returns {Object}
       */
      static create(config, options) {
        return this.reconfigure(null, config, options);
      }
      /**
       * Reconfigures an optional existing instance based on the provided config and returns the correctly configured
       * instance. This will be the `existingInstance` if the `config` does not specify a different type.
       *
       * If `config` is `null` (or simply falsy), this method will destroy the `existingInstance` (if any) and return
       * `null`.
       *
       * If there is no `existingInstance`, the config must specify a type. That is, it must be a string (the type name)
       * or an object containing a `type` property, the `defaultType` must be provided or the factory itself must have
       * a `defaultType` specified (see {@link #property-factoryable-static}).
       *
       * When an `existingInstance` is provided and a type is specified, the instance will be reconfigured via `setConfig`
       * if it is of that type. Otherwise, the `existingInstance` is destroyed (if it is owned by the `options.owner`)
       * and a new instance of the correct type is created.
       *
       * @param {Object} existingInstance The instance to reconfigure. This can be `null`.
       *
       * @param {String|Object} config The type name string or config object.
       *
       * @param {String|Function|Object} [options] Additional options to control the reconfiguration process. If this
       * value is a string or a class constructor, it treated as `options.type`. If this value is a class instance, it
       * is used as the `options.owner`. If this is a function, it is treated as `options.setup`. NOTE: functions declared
       * using the `function` keyword are equivalent to class constructors. Use an arrow function or a class method to
       * avoid this when a `setup` function is intended.
       *
       * @param {String|Function} [options.type] The default type to use if the `config` object does not specify a type.
       *
       * @param {Object} [options.owner] The owner of any created instances. If the `existingInstance` is being replaced,
       * this value determines if the instance will be destroyed.
       *
       * @param {Object} [options.defaults] A config object of default values to use when creating a new instance.
       *
       * @param {Function|String} [options.setup] A function or the name of a method (on the `options.owner`) to call
       * prior to creating a new instance. It is passed the config object that will be used to create the instance. If a
       * truthy value is returned, that value is passed to the constructor instead of the provided config object.
       *
       * @param {Function|String} [options.transform] A function or the name of a method (on the `options.owner`) to call
       * with the raw config object prior to processing and the value it returns replaces the raw value. This function is
       * used to transform strings or arrays (for example) into proper config objects.
       *
       * @param {Function|String} [options.cleanup] A function or the name of a method (on the `options.owner`) to call
       * prior to destroying the `existingInstance`. The `existingInstance` is passed as the sole argument.
       *
       * @returns {Object} The reconfigured instance (either `existingInstance` or a new instance of the desired type)
       */
      static reconfigure(existingInstance, config, options) {
        const me = this,
          {
            factoryable
          } = me,
          {
            typeKey
          } = factoryable;
        let defaultType = options,
          cleanup,
          defaults,
          mergeType,
          owner,
          prepared,
          setup,
          t,
          transform,
          type;
        // Pick apart the options and set the vars accordingly
        if (options && !ObjectHelper.isClass(options)) {
          // if (options is not the defaultType)
          defaultType = null;
          t = typeof options;
          if (t === 'function') {
            setup = options;
          } else if (t === 'string') {
            defaultType = options;
          } else if (ObjectHelper.isObject(options)) {
            cleanup = options.cleanup;
            defaults = options.defaults;
            owner = options.owner;
            setup = options.setup;
            defaultType = options.type;
            transform = options.transform;
          } else {
            owner = options;
          }
        }
        if (transform) {
          config = typeof transform === 'string' ? owner[transform](config) : transform(config);
        }
        type = config;
        // Figure out config... it's either a type (string), a config object or the actual instance.
        if (typeof type === 'string') {
          config = {};
        } else if (config) {
          if (config === true) {
            config = {};
          }
          if (!ObjectHelper.isObject(config)) {
            // If we are being given an instance (not a config object), discard or destroy the existingInstance
            if (owner && config !== existingInstance && (existingInstance === null || existingInstance === void 0 ? void 0 : existingInstance[ownerSymbol]) === owner) {
              var _cleanup;
              typeof cleanup === 'string' ? owner[cleanup](existingInstance) : (_cleanup = cleanup) === null || _cleanup === void 0 ? void 0 : _cleanup(existingInstance);
              existingInstance.destroy();
            }
            return config;
          }
          type = config[typeKey];
        }
        type = type && me.resolveType(type);
        // We've got our orders... make it so...
        if (existingInstance) {
          // We can have a type-less config object when reconfiguring an existing instance, but if we have a type,
          // the existingInstance must be of that type. If !config that means we are nulling out.
          if (config && (!type || existingInstance.constructor === type)) {
            if (typeKey in config) {
              config = ObjectHelper.assign({}, config);
              delete config[typeKey]; // so "type" won't be processed as a config property
            }

            existingInstance.setConfig(config);
            return existingInstance;
          }
          if (owner && existingInstance[ownerSymbol] === owner) {
            var _cleanup2;
            typeof cleanup === 'string' ? owner[cleanup](existingInstance) : (_cleanup2 = cleanup) === null || _cleanup2 === void 0 ? void 0 : _cleanup2(existingInstance);
            existingInstance.destroy();
          }
        }
        if (config) {
          if (defaults) {
            if (!(mergeType = type)) {
              if (!(mergeType = defaults[typeKey] || defaultType || factoryable.defaultType)) {
                throw new Error(`No default mergeType defined for ${factoryable.class.name} factory`);
              }
              mergeType = me.resolveType(mergeType);
            }
            if (mergeType) {
              // Allow the merge fn of each config to perform the task:
              config = mergeType.mergeConfigs(defaults, config);
            }
          }
          if (setup) {
            prepared = typeof setup === 'string' ? owner[setup](config, type, defaults) : setup(config, type, defaults);
            if (prepared === null) {
              return prepared;
            }
            config = prepared || config;
          }
          if (!type) {
            // One more check on config[typeKey] since the setup() function may have added it...
            if (!(type = config[typeKey] || defaultType || factoryable.defaultType)) {
              throw new Error(`No default type defined for ${factoryable.class.name} factory`);
            }
            type = me.resolveType(type);
          }
          if (defaults && !mergeType) {
            config = type.mergeConfigs(defaults, config);
          }
          if (typeKey in config) {
            config = ObjectHelper.assign({}, config);
            delete config[typeKey]; // so "type" won't be processed as a config property
          }

          config = new type(config);
          if (owner) {
            config[ownerSymbol] = owner;
          }
        }
        return config || null;
      }
      /**
       * This method returns the constructor of the class registered for the given type name.
       * @param {String} type The type name to look up.
       * @param {Boolean} [optional] Pass `true` to return `null` if `type` is not found instead of throwing an exception.
       * @returns {Function}
       */
      static resolveType(type, optional) {
        if (typeof type !== 'string') {
          return type;
        }
        const {
            factoryable
          } = this,
          bases = factoryable.extends;
        let result = factoryable.registry[factoryable.caseless ? type.toLowerCase() : type],
          i;
        for (i = 0; !result && bases && i < bases.length; ++i) {
          // Pass optional=true to base factory so the error is our own should the lookup fail:
          result = bases[i].resolveType(type, /* optional = */true);
        }
        if (!result && !optional) {
          throw new Error(`Invalid type name "${type}" passed to ${factoryable.class.name} factory`);
        }
        return result;
      }
    });

    /**
     * @module Core/data/field/DataField
     */
    const {
      getOwnPropertyDescriptor
    } = Reflect;
    /**
     * This is the base class for Model field classes. A field class defines how to handle the data for a particular type
     * of field. Many of these behaviors can be configured on individual field instances.
     *
     * @extends Core/Base
     * @datafield
     */
    class DataField extends Base$1.mixin(Factoryable) {
      static get $name() {
        return 'DataField';
      }
      static get type() {
        return 'auto';
      }
      static get factoryable() {
        return {
          defaultType: 'auto'
        };
      }
      static get prototypeProperties() {
        return {
          /**
           * The name of the field.
           * @config {String} name
           */
          /**
           * The label text for a form item generated for this field. This is also used to create
           * a column header for a {@link #config-column} for this field.
           * @config {String} label
           */
          /**
           * A column config object for a column to display this field in a grid. For simple, atomic
           * data types, such as `date`, `string`, `boolean`, `number` and `integer`, this is optional
           * and the appropriate column type can be inferred.
           *
           * This also provides default values for column configuration if a configured column definition
           * for a grid lacks a property.
           *
           * For complex fields, such as identifiers which link to other records, a more capable
           * column type may be specified, for example a `type : `number'` field may be configured
           * with
           *
           * ```javascript
           * column : 'percent'
           * ```
           * or
           * ```javascript
           * column : {
           *     type : 'percent',
           *     width : 100
           * }
           * ```
           * if it represents a percentage vaue and needs appropriate rendering and editing.
           * @config {String|Object} column
           */
          /**
           * A config object for a widget to edit this field in a form. For simple, atomic
           * data types, such as `date`, `string`, `boolean`, `number` and `integer`, this is optional
           * and the appropriate input widget type can be inferred.
           *
           * For complex fields, such as identifiers which link to other records, a more capable
           * widget may be specified.
           * @config {String|InputFieldConfig} editor
           * @private
           */
          /**
           * A function that compares two values and returns a value < 0 if the first is less than the second, or 0
           * if the values are equal, or a value > 0 if the first is greater than the second.
           * @config {Function}
           * @default
           */
          compare: null,
          /**
           * A function that compares two objects or records using the `compare` function on the properties of each
           * objects based on the `name` of this field.
           * @config {Function}
           * @default
           * @internal
           */
          compareItems: null,
          /**
           * The property in a record's data object that contains the field's value.
           * Defaults to the field's `name`.
           * @config {String}
           */
          dataSource: null,
          /**
           * The default value to assign to this field in a record if no value is provided.
           * @config {*} defaultValue
           */
          /**
           * Setting to `true` will ensure this field is included in any update/insert request payload
           * when a Store / Project / CrudManager performs a request.
           * @config {Boolean}
           * @default
           */
          alwaysWrite: false,
          /**
           * Setting to `false` indicates that `null` is not a valid value.
           * @config {Boolean}
           * @default
           */
          nullable: true,
          /**
           * The value to return from {@link #function-print} for a `null` or `undefined` value.
           * @config {String}
           * @default
           */
          nullText: null,
          /**
           * The value to replace `null` when the field is not `nullable`.
           * @config {*}
           * @default
           */
          nullValue: undefined,
          /**
           * Set to `false` to exclude this field when saving records to a server.
           * @config {Boolean}
           * @default
           */
          persist: true,
          /**
           * Set to `true` for the field's set accessor to ignore attempts to set this field.
           * @config {Boolean}
           * @default
           */
          readOnly: false,
          /**
           * By default, defined {@link Core.data.Model} fields may be used to create a grid column
           * suitable for diplaying that field in a grid cell. Some fields may not be suitable for
           * features which automatically generate columns for view. These fields are created using
           * `internal : true`. Some examples are the `expanded` and `rowHeight` fields which are used
           * internally.
           * @config {Boolean}
           * @default
           */
          internal: false,
          useProp: null
        };
      }
      /**
       * The class that first defined this field. Derived classes that override a field do not change this property.
       * @member {Core.data.Model} definedBy
       * @private
       * @readonly
       */
      /**
       * The class that most specifically defined this field. Derived classes that override a field set this property to
       * themselves.
       * @member {Core.data.Model} owner
       * @private
       * @readonly
       */
      // NOTE: Since we create lots of instances, they have no life cycle (they are not destroyed) and are readonly after
      // creation, this class does not use configurable.
      construct(config) {
        const me = this;
        if (config) {
          me.name = config.name; // assign name first for diagnostic reasons
          Object.assign(me, config);
        }
        if (me.compare) {
          // We wrap in this way to allow compareItems() to be used as an array sorter fn (which gets no "this"):
          me.compareItems = (itemA, itemB) => me.compare(itemA === null || itemA === void 0 ? void 0 : itemA[me.name], itemB === null || itemB === void 0 ? void 0 : itemB[me.name]);
        }
      }
      /**
       * This method transforms a data value into the desired form for storage in the record's data object.
       *
       * ```javascript
       * export default class Task extends TaskModel {
       *    static get fields() {
       *        return [
       *            {
       *                name    : 'status',
       *                convert : (value, data) => {
       *                    if (value >= 100) {
       *                        return 'done';
       *                    }
       *                    else if (value > 0) {
       *                        return 'started';
       *                    }
       *                }
       *            }
       *        ];
       *    }
       * }
       * ```
       *
       * @method convert
       * @param {*} value The value to convert for storage in a record.
       * @param {Object} data The raw record data object
       * @returns {*} The converted value.
       */
      /**
       * This method transforms a data value into the desired form for transmitting to a server.
       * @method serialize
       * @param {*} value The value to serialize
       * @param {Core.data.Model} record The record that contains the value being serialized.
       * @returns {*} The serialized value.
       */
      /**
       * This optional method is called when setting a data value on a record.
       * @method set
       * @param {*} value The value to set
       * @param {Object} data The records future or current data object to set value to
       * @param {Core.data.Model} record The record that owns or will own the data object
       * @internal
       */
      /**
       * This optional method is called when a record using this field is created.
       * @method init
       * @param {Core.data.Model} record The record being created
       * @internal
       */
      /**
       * Create getter and setter functions for the specified field name under the specified key.
       * @internal
       */
      defineAccessor(target, force) {
        const {
          name,
          dataSource
        } = this;
        // Bail out if trying to override an explicitly defined accessor
        if (!force && name in target && target.$meta.hierarchy.some(current => {
          var _getOwnPropertyDescri;
          return ((_getOwnPropertyDescri = getOwnPropertyDescriptor(current.prototype, name)) === null || _getOwnPropertyDescri === void 0 ? void 0 : _getOwnPropertyDescri.enumerable) === false;
        })) {
          return;
        }
        Reflect.defineProperty(target, name, {
          configurable: true,
          // To allow removing it later
          enumerable: true,
          // no arrow functions here, need `this` to change to instance
          get: this.complexMapping ? function () {
            return this.complexGet(name, dataSource);
          } : function () {
            // Inlined copy of Model#flatGet, to save a fn call since this is hit very often
            // When changes are batched, they get stored by field name, not dataSource
            if (this.batching && name in this.meta.batchChanges) {
              return this.meta.batchChanges[name];
            }
            return dataSource in this.data ? this.data[dataSource] : this.data[name];
          },
          // no arrow functions here, need `this` to change to instance
          set(value) {
            // Since the accessor is defined on a base class, we dip into the fields map for the actual
            // calling class to get the correct field definition
            const field = this.$meta.fields.map[name];
            // Only set if field is read/write. Privately, we use setData to set its value
            if (!(field && field.readOnly)) {
              this.set(name, value);
            }
          }
        });
      }
      /**
       * Compares two values for this field and returns `true` if they are equal, and `false` if not.
       * @param {*} first The first value to compare for equality.
       * @param {*} second The second value to compare for equality.
       * @returns {Boolean} `true` if `first` and `second` are equal.
       */
      isEqual(first, second) {
        return ObjectHelper.isEqual(first, second);
      }
      /**
       * Returns the given field value as a `String`. If `value` is `null` or `undefined`, the value specified by
       * {@link #config-nullText} is returned.
       * @param {*} value The value to convert to a string.
       * @returns {String}
       */
      print(value) {
        return value == null ? this.nullText : this.printValue(value);
      }
      /**
       * Returns the given, non-null field value as a `String`.
       * @param {*} value The value to convert to a string (will not be `null` or `undefined`).
       * @returns {String}
       * @protected
       */
      printValue(value) {
        return String(value);
      }
    }
    DataField._$name = 'DataField';

    /**
     * @module Core/data/field/ArrayDataField
     */
    /**
     * This field class handles fields that hold an array.
     *
     * ```javascript
     * class Task extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             // Array field
     *             { name : 'todo', type : 'array' }
     *         ];
     *     }
     * }
     * ```
     *
     * A record can be constructed like this:
     *
     * ```javascript
     * const task = new Task({
     *     name : 'Task 1',
     *     todo : [
     *         { text : 'Something', done : false },
     *         { text : 'Some other thing', done : true }
     *     ]
     * };
     * ```
     *
     * Or by populating a store:
     *
     * ```javascript
     * const store = new Store({
     *     modelClass : Task,
     *     data : [
     *         {
     *             name : 'Task 1',
     *             todo : [
     *                 { text : 'Something', done : false },
     *                 { text : 'Some other thing', done : true }
     *             ]
     *         },
     *         ...
     *     ]
     * });
     * ```
     *
     * For the field to count as modified, the whole array has to be replaced:
     *
     * ```javascript
     * // This won't be detected as a modification
     * task.todo[0].done = true;
     * // task.isModified === false
     *
     * // But this will
     * const todo = task.todo.slice(); // Create a new array with same contents
     * todo[0].done = true;
     * task.todo = todo;
     * // task.isModified === true
     * ```
     *
     * @extends Core/data/field/DataField
     * @classtype array
     * @datafield
     */
    class ArrayDataField extends DataField {
      static $name = 'ArrayDataField';
      static type = 'array';
      isEqual(a, b) {
        return a === b;
      }
      getAt(record, index) {
        return record.get(this.name)[index];
      }
    }
    ArrayDataField.initClass();
    ArrayDataField._$name = 'ArrayDataField';

    /**
     * @module Core/data/field/BooleanDataField
     */
    /**
     * This field class handles field of type `Boolean`.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'active', type : 'boolean' }
     *         ];
     *     }
     * }
     * ```
     *
     * When a field is declared as a `'boolean'`, non-null values are promoted to `Boolean` type. This is seldom required,
     * but can be useful if a field value is received as a number but should be treated as a boolean.
     *
     * @extends Core/data/field/DataField
     * @classtype boolean
     * @datafield
     */
    class BooleanDataField extends DataField {
      static get $name() {
        return 'BooleanDataField';
      }
      static get type() {
        return 'boolean';
      }
      static get alias() {
        return 'bool';
      }
      static get prototypeProperties() {
        return {
          /**
           * The value to replace `null` when the field is not `nullable`.
           * @config {Boolean}
           * @default
           */
          nullValue: false
        };
      }
      isEqual(first, second) {
        if (first == null && second == null) {
          return true;
        }
        return super.isEqual(first, second);
      }
      convert(value) {
        var _value$toLowerCase;
        if (value == null) {
          return this.nullable ? value : this.nullValue;
        }
        // string 'false' will convert to false, other strings to true
        if (((_value$toLowerCase = value.toLowerCase) === null || _value$toLowerCase === void 0 ? void 0 : _value$toLowerCase.call(value)) === 'false') {
          return false;
        }
        return Boolean(value);
      }
    }
    BooleanDataField.initClass();
    BooleanDataField._$name = 'BooleanDataField';

    /**
     * @module Core/data/field/DateDataField
     */
    /**
     * This field class handles field of type `Date`.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'birthday', type : 'date', format : 'YYYY-MM-DD' },
     *             { name : 'age', readOnly : true }
     *         ];
     *     }
     * }
     * ```
     *
     * When a field is declared as a `'date'`, non-null values are promoted to `Date` type. This is frequently needed due
     * to how date types are serialized to JSON strings.
     *
     * Date fields can have a special `defaultValue` of `'now'` which will convert to the current date/time.
     *
     * @extends Core/data/field/DataField
     * @classtype date
     * @datafield
     */
    class DateDataField extends DataField {
      static get $name() {
        return 'DateDataField';
      }
      static get type() {
        return 'date';
      }
      static get prototypeProperties() {
        return {
          /**
           * The format of the date field.
           *
           * See {@link Core.helper.DateHelper DateHelper} for details.
           * @config {String} format
           * @default DateHelper.defaultFormat
           */
          format: null
        };
      }
      convert(value) {
        if (value == null) {
          if (!this.nullable) {
            value = this.nullValue;
          }
        } else if (value === 'now') {
          value = new Date();
        } else if (!(value instanceof Date)) {
          // Use configured format, if null/undefined use DateHelpers default format
          value = DateHelper.parse(value, this.format || DateHelper.defaultParseFormat);
          // if parsing has failed, we would like to return `undefined` to indicate the "absence" of data
          // instead of `null` (presence of "empty" data)
          if (!value || isNaN(value)) {
            value = undefined;
          }
        }
        return value;
      }
      serialize(value) {
        if (value instanceof Date) {
          // Use configured format or DateHelpers default one
          value = DateHelper.format(value, this.format || DateHelper.defaultFormat);
        }
        return value;
      }
      printValue(value) {
        return DateHelper.format(value, this.format || DateHelper.defaultFormat);
      }
    }
    DateDataField.initClass();
    DateDataField._$name = 'DateDataField';

    /**
     * @module Core/data/field/StringDataField
     */
    /**
     * This field class handles field of type `String`.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             { name : 'name', type : 'string' }
     *         ];
     *     }
     * }
     * ```
     *
     * When a field is declared as a `'string'`, non-null values are promoted to `String` type. This is seldom required, but
     * can be useful if a field value is received as a number but should be treated as a string.
     *
     * @extends Core/data/field/DataField
     * @classtype string
     * @datafield
     */
    class StringDataField extends DataField {
      static get $name() {
        return 'StringDataField';
      }
      static get type() {
        return 'string';
      }
      static get prototypeProperties() {
        return {
          /**
           * The value to replace `null` when the field is not `nullable`.
           * @config {String}
           * @default
           */
          nullValue: ''
        };
      }
      convert(value) {
        return value == null ? this.nullable ? value : this.nullValue : String(value);
      }
    }
    StringDataField.initClass();
    StringDataField._$name = 'StringDataField';

    /**
     * @module Core/data/field/DurationUnitDataField
     */
    /**
     * This field class handles field of type `durationunit` (string type). See {@link Core.data.Duration} for more information.
     *
     * ```javascript
     * class Event extends Model {
     *     static get fields() {
     *         return [
     *             { name : 'durationUnit', type : 'durationunit' }
     *         ];
     *     }
     * }
     * ```
     *
     * @extends Core/data/field/StringDataField
     * @classtype durationunit
     * @datafield
     */
    class DurationUnitDataField extends StringDataField {
      static get $name() {
        return 'DurationUnitDataField';
      }
      static get type() {
        return 'durationunit';
      }
      isEqual(first, second) {
        return DateHelper.compareUnits(first, second) === 0;
      }
    }
    DurationUnitDataField.initClass();
    DurationUnitDataField._$name = 'DurationUnitDataField';

    /**
     * @module Core/data/field/IntegerDataField
     */
    /**
     * This field class handles field of type `Number` with no decimal digits.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'age', type : 'int' }
     *         ];
     *     }
     * }
     * ```
     *
     * When a field is declared as a `'int'`, non-null values are promoted to `Number` type and decimals are removed using
     * a specified `rounding`. This field type can be useful if a field value is received as a string but should be stored
     * as a number or has a fractional component that must be rounded or truncated.
     *
     * @extends Core/data/field/DataField
     * @classtype integer
     * @datafield
     */
    class IntegerDataField extends DataField {
      static get $name() {
        return 'IntegerDataField';
      }
      static get type() {
        return 'integer';
      }
      static get alias() {
        return 'int';
      }
      static get prototypeProperties() {
        return {
          /**
           * The value to replace `null` when the field is not `nullable`.
           * @config {Number}
           * @default
           */
          nullValue: 0,
          /**
           * The `Math` method to use to ensure fractional component is removed.
           * @config {'round'|'floor'|'ceil'}
           * @default
           */
          rounding: 'round'
        };
      }
      convert(value) {
        return value == null ? this.nullable ? value : this.nullValue : Math[this.rounding](Number(value));
      }
    }
    IntegerDataField.initClass();
    IntegerDataField._$name = 'IntegerDataField';

    /**
     * @module Core/data/field/ModelDataField
     */
    /**
     * This field class handles fields that hold other records.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'address', type : 'model' }
     *         ];
     *     }
     * }
     * ```
     *
     * @internal
     * @extends Core/data/field/DataField
     * @classtype model
     * @datafield
     */
    class ModelDataField extends DataField {
      static get $name() {
        return 'ModelDataField';
      }
      static get type() {
        return 'model';
      }
      static get prototypeProperties() {
        return {
          complexMapping: true
        };
      }
      isEqual(first, second) {
        // Check for semantic equality. An instance of the same Model class of the same ID is equal.
        return first && second && second instanceof first.constructor && second.id == first.id;
      }
    }
    ModelDataField.initClass();
    ModelDataField._$name = 'ModelDataField';

    /**
     * @module Core/data/field/NumberDataField
     */
    /**
     * This field class handles field of type `Number`.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'age', type : 'number' }
     *         ];
     *     }
     * }
     * ```
     *
     * When a field is declared as a `'number'`, non-null values are promoted to `Number` type. This is seldom required, but
     * can be useful if a field value is received as a string but should be treated as a number.
     *
     * @extends Core/data/field/DataField
     * @classtype number
     * @datafield
     */
    class NumberDataField extends DataField {
      static get $name() {
        return 'NumberDataField';
      }
      static get type() {
        return 'number';
      }
      static get alias() {
        return 'float';
      }
      static get prototypeProperties() {
        return {
          /**
           * The value to replace `null` when the field is not `nullable`.
           * @config {Number}
           * @default
           */
          nullValue: 0,
          /**
           * The numeric precision of this field. Values are rounded to the specified number of digits. If `null`,
           * the default, no rounding is performed.
           * @config {Number}
           * @default
           */
          precision: null
        };
      }
      isEqual(first, second) {
        // NaN !== NaN in JS which results having a number field w/ such value always dirty
        // Not sure having two NaN-s not equal each other makes any sense here to us ..so handle it
        return isNaN(Number(first)) && isNaN(Number(second)) || super.isEqual(first, second);
      }
      convert(value) {
        if (value == null) {
          return this.nullable ? value : this.nullValue;
        }
        value = Number(value);
        // Returning undefined to let set know that this is a invalid value
        if (isNaN(value)) {
          return;
        }
        let scale = this.precision;
        if (scale) {
          scale = 10 ** scale;
          value = Math.round(value * scale) / scale;
        } else if (scale === 0) {
          value = Math.round(value);
        }
        return value;
      }
    }
    NumberDataField.initClass();
    NumberDataField._$name = 'NumberDataField';

    /**
     * @module Core/data/field/ObjectDataField
     */
    /**
     * This field class handles fields that hold an object.
     *
     * ```javascript
     * class Person extends Model {
     *     static fields = [
     *         'name',
     *         { name : 'address', type : 'object' }
     *     ];
     * }
     * ```
     *
     * For the field to count as modified, the whole object has to be replaced:
     *
     * ```javascript
     * person.address = { ...address };
     * ```
     *
     * Or, sub properties of the object has to be modified using calls to `set()`:
     *
     * ```javascript
     * person.set('address.street', 'Main Street');
     * ```
     *
     * Note that if any property of the nested object requires conversion after load, you have to define that property as
     * a field:
     *
     * ```javascript
     * class Order extends Model {
     *     static fields = [
     *         'title',
     *         { name : 'details', type : 'object' },
     *         { name : 'details.date', type : 'date' }
     *     ];
     * }
     *
     * const order = new Order({
     *    title   : 'Order 1',
     *    details : {
     *      customer : 'Bill',
     *      // Definition above required for this to be converted to a date
     *      date     : '2020-01-01'
     *    }
     * });
     * ```
     *
     * @extends Core/data/field/DataField
     * @classtype object
     * @datafield
     */
    class ObjectDataField extends DataField {
      static get $name() {
        return 'ObjectDataField';
      }
      static get type() {
        return 'object';
      }
      static get prototypeProperties() {
        return {
          complexMapping: true
        };
      }
    }
    ObjectDataField.initClass();
    ObjectDataField._$name = 'ObjectDataField';

    /**
     * @module Core/data/field/StoreDataField
     */
    /**
     * This field class handles fields that accepts an array that is then converted to a store.
     *
     * ```javascript
     * class Task extends Model {
     *     static fields = [
     *         'name',
     *         // Store field
     *         { name : 'subTasks', type : 'store', storeClass : Store }
     *     ];
     * }
     * ```
     *
     * A record can be constructed like this:
     *
     * ```javascript
     * const task = new Task({
     *     name : 'Task 1',
     *     subTasks : [
     *         { text : 'Something', done : false },
     *         { text : 'Some other thing', done : true }
     *     ]
     * };
     * ```
     *
     * Or by populating a store:
     *
     * ```javascript
     * const store = new Store({
     *     modelClass : Task,
     *     data : [
     *         {
     *             name : 'Task 1',
     *             subTasks : [
     *                 { text : 'Something', done : false },
     *                 { text : 'Some other thing', done : true }
     *             ]
     *         },
     *         ...
     *     ]
     * });
     * ```
     *
     * Whenever the store or its records are manipulated, the field will be marked as modified:
     *
     * ```javascript
     * // These will all be detected as modifications
     * task.subTasks.first.done = true;
     * task.subTasks.last.remove();
     * task.subTasks.add({ text : 'New task', done : false });
     * ```
     *
     * <div class="note">Note that the underlying store by default will be configured with <code>syncDataOnLoad</code> set
     * to <code>true</code></div>
     *
     * @extends Core/data/field/DataField
     * @classtype store
     * @datafield
     */
    class StoreDataField extends DataField {
      static $name = 'StoreDataField';
      static type = 'store';
      /**
       * Store class to use when creating the store.
       *
       * ```javascript
       * class TodoStore extends Store {
       *     ...
       * }
       *
       * const task = new Store({
       *     static fields = [
       *         { type : 'store', name: 'todoItems', storeClass : TodoStore }
       *     ]
       * });
       * ```
       *
       * @config {Class} storeClass
       * @typings {typeof Store}
       */
      /**
       * Model class to use for the store (can also be configured as usual on the store class, this config is for
       * convenience).
       *
       * ```javascript
       * class TodoItem extends Model {
       *   ...
       * }
       *
       * const task = new Store({
       *     static fields = [
       *         { type : 'store', name: 'todoItems', storeClass : Store, modelClass : TodoItem }
       *     ]
       * });
       * ```
       *
       * @config {Class} modelClass
       * @typings {typeof Model}
       */
      /**
       * Optional store configuration object to apply when creating the store.
       *
       * ```javascript
       * const task = new Store({
       *     static fields = [
       *         {
       *             type       : 'store',
       *             name       : 'todoItems',
       *             storeClass : Store
       *             store      : {
       *                  syncDataOnLoad : false
       *             }
       *         }
       *     ]
       * });
       * ```
       *
       * @config {StoreConfig} store
       */
      // Initializer, called when creating a record. Sets up the store and populates it with any initial data
      init(data, record) {
        var _record;
        const me = this,
          storeName = `${me.name}Store`,
          config = {
            skipStack: true,
            syncDataOnLoad: true
          }; // Optimization when used from sources, don't create a stack in Base
        if (me.store) {
          ObjectHelper.assign(config, me.store);
        }
        // Optionally apply modelClass, for convenient configuration
        if (me.modelClass) {
          config.modelClass = me.modelClass;
        }
        // Call optional initializer (initSubTasksStore for subTasks field) on the record, letting it manipulate the
        // config before creating a store
        (_record = record[`init${StringHelper.capitalize(storeName)}`]) === null || _record === void 0 ? void 0 : _record.call(record, config);
        if (!config.storeClass && !me.storeClass) {
          throw new Error(`Field '${me.name}' with type 'store' must have a storeClass configured`);
        }
        // Store has to be assigned on the record, field is shared
        const store = record.meta[storeName] = new (config.storeClass || me.storeClass)(config);
        if (me.complexMapping) {
          ObjectHelper.setPath(data, me.dataSource, store);
        } else {
          data[me.dataSource] = store;
        }
        // Don't warn about generated ids, responsibility lies elsewhere
        store.verifyNoGeneratedIds = false;
        // Keep track of if id should be included when serializing or not
        store.usesId = !store.count || !store.every(record => record.hasGeneratedId);
        // Cache value
        store.$currentValue = me.getValue(store);
        // Track changes to the store, applying them to the record and caching current value to be used when
        // serializing and in comparisons (required, otherwise we would be comparing to already updated store
        store.ion({
          change: ({
            action
          }) => {
            const value = me.getValue(store);
            if (!store.$isSettingStoreFieldData) {
              const oldPreserveCurrentDataset = store.$preserveCurrentDataset;
              store.$preserveCurrentDataset = me.subStore && (action === 'update' || action === 'remove' || action === 'add');
              me.$isUpdatingRecord = true;
              record.set(me.name, value);
              me.$isUpdatingRecord = false;
              store.$preserveCurrentDataset = oldPreserveCurrentDataset;
            }
            // cache the field current value
            store.$currentValue = value;
          }
        });
      }
      // Called when setting a new value to the field on a record
      set(value, data, record) {
        var _record2;
        const me = this,
          storeName = `${me.name}Store`,
          {
            [storeName]: store
          } = record.meta;
        // Lazy store might not be created yet, gets created on first access. Returning false keeps the value for later
        // if called during init
        if (!store) {
          // Missing store suggests value was not yet initialized and future value resides
          // in a special meta property. In which case we need to update it there
          record.meta.initableValues.set(me, value);
          return false;
        }
        // Prevent changes from leading to recursive calls
        if (store.$isSettingStoreFieldData) {
          return;
        }
        store.$isSettingStoreFieldData = true;
        // Call optional processing fn (processSubTasksStoreData for subTasks field) on the record, letting it
        // manipulate the data before creating records
        value = ((_record2 = record[`process${StringHelper.capitalize(storeName)}Data`]) === null || _record2 === void 0 ? void 0 : _record2.call(record, value, record)) ?? value;
        // Apply incoming array to store
        if (!store.$preserveCurrentDataset) {
          store.data = value;
        }
        store.$isSettingStoreFieldData = false;
        // Keep track of if id should be included when serializing or not
        store.usesId = !store.count || !store.every(record => record.hasGeneratedId);
      }
      serialize(value, record) {
        const store = record.meta[`${this.name}Store`];
        return this.$isUpdatingRecord ? this.getValue(store) : store.$currentValue;
      }
      // Extract persistable values, optionally including id depending on if ids are used
      getValue(store) {
        return store.allRecords.map(r => {
          const data = r.persistableData;
          if (!store.usesId) {
            delete data.id;
          }
          return data;
        });
      }
      isEqual(a, b) {
        var _a, _b;
        if ((_a = a) !== null && _a !== void 0 && _a.isStore) {
          a = a.$currentValue;
        }
        if ((_b = b) !== null && _b !== void 0 && _b.isStore) {
          b = b.$currentValue;
        }
        return ObjectHelper.isDeeplyEqual(a, b);
      }
      // Cloned value to be able to restore it later using STM
      getOldValue(record) {
        const store = record.meta[`${this.name}Store`];
        return store ? ObjectHelper.clone(store.$currentValue) : null;
      }
      getAt(record, index) {
        const store = record.meta[`${this.name}Store`];
        return store === null || store === void 0 ? void 0 : store.getAt(index);
      }
    }
    StoreDataField.initClass();
    StoreDataField._$name = 'StoreDataField';

    /**
     * @module Core/data/Model
     */
    const nestedRe = new RegExp(/^(.*?)\.(.*)/),
      arrayRe = /(.*)\[(.*)]\.?(.*)/;
    /**
     * Defines the properties of a relation between two stores.
     *
     * Used as the values of a Model's {@link Core.data.Model#property-relations-static} definition.
     *
     * This snippet will define a relation called `team`, allowing access to the foreign record via `player.team`. It will
     * point to a record in the `teamStore` (must be available as `record.firstStore.teamStore)` with an id matching the
     * players `teamId` field. The team record in turn, will have a field called `players` which is a collection of all
     * players in the team.
     *
     * ```javascript
     * class Player extends Model {
     *     static relations = {
     *         team : {
     *             foreignKey            : 'teamId',
     *             foreignStore          : 'teamStore',
     *             relatedCollectionName : 'players'
     *         }
     *     }
     * }
     * ```
     *
     * See {@link Core.data.Model#property-relations-static} for a more extensive example.
     *
     * @typedef {Object} RelationConfig
     * @property {String} foreignKey Name of a field on this model which holds the foreign key value.
     * @property {String|Core.data.Store} foreignStore Name of a property on the model's first store, which holds the
     * foreign store. Or the actual store instance
     * @property {String} [relatedCollectionName] Optionally, name of a property that will be added to the records of the
     * foreign store, which will hold all records from the model's store related to it.
     */
    const {
        defineProperty: defineProperty$4
      } = Reflect,
      {
        hasOwn: hasOwn$2
      } = ObjectHelper,
      _undefined = undefined,
      internalProps = {
        children: 1,
        data: 1,
        meta: 1
      },
      abbreviationFields = ['name', 'title', 'text', 'label', 'description'],
      fieldDataTypes = {
        boolean: 1,
        number: 1,
        date: 1,
        object: 1
      },
      fieldsOrder = {
        parentId: 1,
        $PhantomId: 2,
        id: 3
      };
    /**
     * A Model is the definition of a record which can be added to (or loaded into) a {@link Core.data.Store}. It defines
     * which fields the data contains and exposes an interface to access and manipulate that data. The Model data is
     * populated through simple a JSON object.
     *
     * By default, a Model stores a shallow copy of its raw json, but for records in stores configured with
     * `useRawData: true` it stores the supplied json object as is.
     *
     * ## Defining fields
     *
     * A Model can either define its fields explicitly (see {@link #property-fields-static}) or have them created from its
     * data (see {@link #property-autoExposeFields-static}). This snippet shows a model with 4 fields defined explicitly:
     *
     * ```javascript
     * class Person extends Model {
     *     static fields = [
     *         'name',
     *         { name : 'birthday', type : 'date', format : 'YYYY-MM-DD' },
     *         { name : 'shoeSize', type : 'number', defaultValue : 11 },
     *         { name : 'age', readOnly : true }
     *     ]
     * }
     * ```
     *
     * The first field (name) has an unspecified type, which means the field's value is held as received with no conversion
     * applied. The second field (birthday) is defined to be a date, which will make the model parse any supplied value into
     * an actual date. The parsing is handled by {@link Core/helper/DateHelper#function-parse-static DateHelper.parse()}
     * using the specified `format`, or if no format is specified using
     * {@link Core/helper/DateHelper#property-defaultFormat-static DateHelper.defaultFormat}.
     *
     * The set of standard field types is as follows:
     *
     *  - {@link Core.data.field.ArrayDataField `array`}
     *  - {@link Core.data.field.BooleanDataField `boolean`}
     *  - {@link Core.data.field.DateDataField `date`}
     *  - {@link Core.data.field.IntegerDataField `integer`}
     *  - {@link Core.data.field.ObjectDataField `object`}
     *  - {@link Core.data.field.NumberDataField `number`}
     *  - {@link Core.data.field.StoreDataField `store`}
     *  - {@link Core.data.field.StringDataField `string`}
     *
     * You can also set a `defaultValue` that will be used if the data does not contain a value for the field:
     *
     * ```javascript
     * { name : 'shoeSize', type : 'number', defaultValue : 11 }
     * ```
     *
     * To create a record from a Model, supply data to its constructor:
     *
     * ```javascript
     * let guy = new Person({
     *     id       : 1,
     *     name     : 'Dude',
     *     birthday : '2014-09-01'
     * });
     * ```
     *
     * If no id is specified, a temporary id based on a UUID will be generated. This id is not meant to be serialized, it
     * should instead be replaced by the backend with a proper id from the underlying database (or similar).
     *
     * Please avoid using reserved names for your fields (such as `parent`, `children` and others that are used as Model
     * properties) to avoid possible data collisions and bugs.
     *
     * ## Nested fields
     *
     * Model supports mapping fields to nested data structures using dot `.` separated paths as the `dataSource`. For
     * example given this JSON object:
     *
     * ```json
     * {
     *     name : 'Borje Salming',
     *     team : {
     *         name   : 'Toronto Maple Leafs',
     *         league : 'NHL'
     *     }
     * }
     * ```
     *
     * A field can be mapped to the nested team name by using `dataSource : 'team.name'`:
     *
     * ```javascript
     * class Player extends Model {
     *     static fields = [
     *         'name',
     *         // Field mapped to a property on a nested object
     *         { name : 'teamName', dataSource : 'team.name' }
     *     ];
     * }
     * ```
     *
     * Usage:
     *
     * ```javascript
     * const player = new Player(json);
     *
     * console.log(player.teamName); // > Toronto Maple Leafs
     * player.teamName = 'Minnesota Wild'; // Updates name property of the team object
     * ```
     *
     * Alternatively, you can define the top level of the nested object as a field of type `object`:
     *
     * ```javascript
     * class Person extends Model {
     *     static fields = [
     *         'name',
     *         // Nested object
     *         { name : 'address', type : 'object' }
     *     ];
     * }
     * ```
     *
     * You can then access properties of the nested object using dot notation with the {@link #function-get} function:
     *
     * ```javascript
     * const person = new Person({
     *    name    : 'Borje Salming',
     *    address : {
     *        city : 'Toronto'
     *    }
     * });
     *
     * person.get('address.city'); // > Toronto
     * ```
     *
     * ### Updating a nested object
     *
     * Note that directly altering a property of the nested object won't register as an update of the record, record does
     * not track changes deeply. If nested fields (as described above) is not enough for your usecase you can map a field
     * directly to the nested object and then assign a shallow copy of it to the record on changes:
     *
     * ```javascript
     * class Player extends Model {
     *     static get fields() {
     *         return [
     *             ...,
     *             // Field mapped directly to the nested object
     *             { name : 'team', type : 'object' }
     *         ]
     *     }
     * }
     *
     * // "External object" to nest
     * const team = {
     *     name   : 'Brynas',
     *     league : 'SHL'
     * }
     *
     * const player = new Player({
     *     name : 'Borje Salming',
     *     team
     * });
     *
     * // This will not flag player as dirty
     * team.league = 'CHL';
     *
     * // Instead you have to reassign the mapped field
     * player.team = { ...player.team };
     * ```
     *
     * You can also use the {@link #function-set} function to update a property of the nested object:
     *
     * ```javascript
     * // This will flag player as dirty
     * player.set('team.name', 'BIF');
     * ```
     *
     * ## Arrays of atomic types
     *
     * When a field holds an array of atomic types (strings, numbers etc.) we recommend using the
     * {@link Core/data/field/ArrayDataField `array`} type for the field:
     *
     * ```javascript
     * class GroceryList extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'items', type : 'array' }
     *         ];
     *     }
     * }
     *
     * const list = new GroceryList({
     *    name  : 'My list',
     *    items : ['Milk', 'Bread', 'Eggs']
     * });
     * ```
     *
     * {@note}Modifying items in the array will not flag the field as updated, since the array itself does not
     * change. For it to register a change, you must assign it a new array (could be a copy of the old one). For more info,
     * see {@link Core/data/field/ArrayDataField}{/@note}
     *
     * ## Arrays of objects
     *
     * When a field holds an array of objects, we recommend using the {@link Core/data/field/StoreDataField `store`} type
     * for the field:
     *
     * ```javascript
     * class GroceryList extends Model {
     *     static fields = [
     *         'name',
     *         { name : 'items', type : 'store', storeClass : Store }
     *     ]
     * }
     *
     * const list = new GroceryList({
     *    name  : 'My list',
     *    items : [
     *        { name : 'Milk', quantity : 1 },
     *        { name : 'Bread', quantity : 2 },
     *        { name : 'Eggs', quantity : 12 }
     *    ]
     * });
     * ```
     *
     * The `items` field on the `list` above will be a {@link Core.data.Store} instance (because we passed that as
     * `storeClass`), which can be used to manipulate the items in the list. Doing so will flag the `list` as modified. For
     * more info, see {@link Core.data.field.StoreDataField}.
     *
     * ## Persisting fields
     *
     * By default, all fields are persisted. If you don't want particular field to get saved to the server, configure it
     * with `persist: false`. In this case field will not be among changes which are sent by
     * {@link Core/data/AjaxStore#function-commit store.commit()}, otherwise its behavior doesn't change.
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             'name',
     *             { name : 'age', persist : false }
     *         ];
     *     }
     * }
     * ```
     *
     * ## The `id` field
     * By default Model expects its id field to be stored in a data source named "id". The data source for the id field can
     * be customized by setting {@link Core/data/field/DataField#config-dataSource} on the id field object configuration.
     *
     * ```javascript
     * class Person extends Model {
     *     static fields = [
     *         { name : 'id', dataSource: 'personId'},
     *         'name',
     *         { name : 'age', persist : false },
     *         { name : 'birthday', type : 'date' }
     *      ];
     * }
     *
     * let girl = new Person({
     *     personId : 2,
     *     name     : 'Lady',
     *     birthday : '2011-11-05'
     * });
     * ```
     *
     * Also, it is possible to change the id field data source by setting {@link #property-idField-static}:
     *
     * ```javascript
     * class Person extends Model {
     *     // Id drawn from 'id' property by default; use custom field here
     *     static idField = 'personId';
     *
     *     static fields = [
     *         'name',
     *         { name : 'age', persist : false },
     *         { name : 'birthday', type : 'date' }
     *     ];
     * }
     * ```
     *
     * ## Getting and setting values
     *
     * Fields are used to generate getters and setters on the records. Use them to access or modify values (they are
     * reactive):
     *
     * ```javascript
     * console.log(guy.name);
     * girl.birthday = new Date(2011,10,6);
     * ```
     *
     * NOTE: In an application with multiple different models you should subclass Model, since the prototype is decorated
     * with getters and setters. Otherwise, you might get unforeseen collisions.
     *
     * ## Field data mapping
     *
     * By default, fields are mapped to data using their name. If you for example have a "name" field it expects data to be
     * `{ name: 'Some name' }`. If you need to map it to some other property, specify `dataSource` in your field definition:
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields {
     *         return [
     *             { name : 'name', dataSource : 'TheName' }
     *         ];
     *     }
     * }
     *
     * // This is now OK:
     * let dude = new Person({ TheName : 'Manfred' });
     * console.log(dude.name); // --> Manfred
     * ```
     *
     * ## Field inheritance
     *
     * Fields declared in a derived model class are added to those from its superclass. If a field declared by a derived
     * class has also been declared by its super class, the field properties of the super class are merged with those of
     * the derived class.
     *
     * For example:
     * ```javascript
     *  class Person extends Model {
     *      static get fields() {
     *          return [
     *              'name',
     *              { name : 'birthday', type : 'date', format : 'YYYY-MM-DD' }
     *          ];
     *      }
     *  }
     *
     *  class User extends Person {
     *      static get fields() {
     *          return [
     *              { name : 'birthday', dataSource : 'dob' },
     *              { name : 'lastLogin', type : 'date' }
     *          ];
     *      }
     *  }
     * ```
     *
     * In the above, the `Person` model declares the `birthday` field as a `date` with a specified `format`. The `User`
     * model extends `Person` and also declares the `birthday` field. This redeclared field only specifies `dataSource`, so
     * all the other fields are preserved from `Person`. The `User` model also adds a `lastLogin` field.
     *
     * The `User` from above could have been declared like so to achieve the same `fields`:
     *
     * ```javascript
     *  class User extends Model {
     *      static get fields() {
     *          return [
     *              'name',
     *              { name : 'birthday', type : 'date', format : 'YYYY-MM-DD', dataSource : 'dob' },
     *              { name : 'lastLogin', type : 'date' }
     *          ];
     *      }
     *  }
     * ```
     *
     * ## Override default values
     *
     * In case you need to define default value for a specific field, or override an existing default value, you can
     * define a new or re-define an existing field definition in {@link #property-fields-static} static getter:
     *
     * ```javascript
     * class Person extends Model {
     *     static get fields() {
     *         return [
     *             { name : 'username', defaultValue : 'New person' },
     *             { name : 'birthdate', type : 'date' }
     *         ];
     *     }
     * }
     *
     * class Bot extends Person {
     *     static get fields() {
     *         return [
     *             { name : 'username', defaultValue : 'Bot' } // default value of 'username' field is overridden
     *         ];
     *     }
     * }
     * ```
     *
     * ## Read-only records
     *
     * Model has a default field called {@link #field-readOnly}, which is used to make the record read-only in the UI while
     * still allowing programmatic changes to it. Setting it to `true` will prevent it from being edited by the built-in
     * editing features (cell editing in Grid, event draging in Scheduler, task editor in Gantt etc.). Please note that it
     * is not made read-only on the data level, the record can still be manipulated by application code.
     *
     * ```javascript
     * // Prevent record from being manipulated by the user
     * record.readOnly = true;
     *
     * // Programmatic manipulation is still allowed
     * record.remove();
     * ```
     *
     * ## Tree API
     *
     * This class mixes in the {@link Core/data/mixin/TreeNode TreeNode} mixin which provides an API for tree related
     * functionality (only relevant if your store is configured to be a {@link Core/data/Store#config-tree tree}).
     *
     * @mixes Core/data/mixin/ModelLink
     * @mixes Core/data/mixin/TreeNode
     * @mixes Core/data/stm/mixin/ModelStm
     */
    class Model extends Base$1.mixin(ModelStm, TreeNode, ModelLink) {
      static get $name() {
        return 'Model';
      }
      static get declarable() {
        return [
        /**
         * Array of defined fields for this model class. Subclasses add new fields by implementing this static
         * getter:
         *
         * ```javascript
         * // Model defining two fields
         * class Person extends Model {
         *     static get fields() {
         *         return [
         *             { name : 'username', defaultValue : 'New person' },
         *             { name : 'birthdate', type : 'date' }
         *         ];
         *     }
         * }
         *
         * // Subclass overriding one of the fields
         * class Bot extends Person {
         *     static get fields() {
         *         return [
         *             // Default value of 'username' field is overridden, any other setting from the parents
         *             // definition is preserved
         *             { name : 'username', defaultValue : 'Bot' }
         *         ];
         *     }
         * }
         * ```
         *
         * Fields in a subclass are merged with those from the parent class, making it easy to override mappings,
         * formats etc.
         *
         * @member {Array<String|ModelFieldConfig|Core.data.field.DataField>} fields
         * @readonly
         * @static
         * @category Fields
         */
        'fields'];
      }
      static get fields() {
        return [
        // The index of this item in its parent (respects filtering)
        {
          name: 'parentIndex',
          type: 'number',
          persist: false,
          internal: true
        },
        // The index of this item in its parent ghost (non-sortable) children array
        {
          name: 'orderedParentIndex',
          type: 'number',
          persist: false,
          internal: true
        },
        /**
         * Flag the record as read-only on the UI level, preventing the end user from manipulating it using editing
         * features such as cell editing and event dragging.
         *
         * Does not prevent altering the record programmatically, it can still be manipulated by application code.
         *
         * For more info, see the "Read-only records" section above.
         *
         * @field {Boolean} readOnly
         * @category Common
         */
        {
          name: 'readOnly',
          type: 'boolean'
        },
        /**
         * Start expanded or not (only valid for tree data)
         * @readonly
         * @field {Boolean} expanded
         * @category Tree
         */
        {
          name: 'expanded',
          internal: true
        }];
      }
      /**
       * Override in a subclass of Model to define relations to records in other stores.
       *
       * Always defined on the "one" side, not the "many" side.
       *
       * Expects an object where keys are relation names and values are {@link #typedef-RelationConfig relation configs}.
       *
       * This snippet will define a relation called `team`, allowing access to the foreign record via `player.team`. It
       * will point to a record in the `teamStore` (must be available as `record.firstStore.teamStore)` with an id
       * matching the players `teamId` field. The team record in turn, will have a field called `players` which is a
       * collection of all players in the team.
       *
       * ```javascript
       * class Player extends Model {
       *     static relations = {
       *         // Define a relation between a player and a team
       *         team : {
       *             foreignKey            : 'teamId',
       *             foreignStore          : 'teamStore',
       *             relatedCollectionName : 'players'
       *         }
       *     }
       * }
       *
       * const teamStore = new Store({
       *     data : [
       *         { id : 1, name : 'Brynas' },
       *         { id : 2, name : 'Leksand' }
       *     ]
       * });
       *
       * const playerStore = new Store({
       *     modelClass : Player,
       *     // Matches foreignStore, allowing records of playerStore to find the related store
       *     teamStore,
       *     data       : [
       *         // teamId is specified as foreignKey, will be used to match the team
       *         { id : 1, name : 'Nicklas Backstrom', teamId : 1  },
       *         { id : 2, name : 'Elias Lindholm',   teamId : 1  },
       *         { id : 3, name : 'Filip Forsberg',  teamId : 2  }
       *     ],
       * }
       *
       * playerStore.first.team.name // > Brynas
       * playerStore.last.team.name // > Leksand
       * teamStore.first.players // > [nick, elias]
       * teamStore.last.players // > [filip]
       * ```
       *
       * To access the related record from the many side, use dot notation for the field name. For example in a Grid
       * column:
       *
       * ```javascript
       * const grid = new Grid({
       *    store : playerStore,
       *    columns : [
       *        { field : 'name', text : 'Name' },
       *        { field : 'team.name', text : 'Team' }
       *    ]
       * });
       * ```
       *
       * @member {Object<String,RelationConfig>} relations
       * @static
       */
      static relations = null;
      /**
       * Template static getter which is supposed to be overridden to define default field values for the Model class.
       * Overrides `defaultValue` config specified by the {@link #property-fields-static} getter.
       * Returns a named object where key is a field name and value is a default value for the field.
       *
       * NOTE: This is a legacy way of defining default values, we recommend using {@link #property-fields-static} moving
       * forward.
       *
       * ```javascript
       * class Person extends Model {
       *     static get fields() {
       *         return [
       *             { name : 'username', defaultValue : 'New person' }
       *         ];
       *     }
       * }
       *
       * class Bot extends Person {
       *     static get defaults() {
       *         return {
       *             username : 'Bot' // default value of 'username' field is overridden
       *         };
       *     }
       * }
       * ```
       *
       * @member {Object} defaults
       * @static
       * @category Fields
       */
      /**
       * The data source for the id field which provides the ID of instances of this Model.
       * @property {String}
       * @category Fields
       */
      static set idField(idField) {
        this._assignedIdField = true;
        this._idField = idField;
      }
      static get idField() {
        return this._idField;
      }
      /**
       * The name of the data field which holds children of this Model when used in a tree structure
       * ```javascript
       * MyModel.childrenField = 'kids';
       * const parent = new MyModel({
       *     name : 'Dad',
       *     kids : [
       *         { name : 'Daughter' },
       *         { name : 'Son' }
       *     ]
       * });
       * ```
       * @property {String}
       * @category Fields
       */
      static set childrenField(childrenField) {
        this._childrenField = childrenField;
      }
      static get childrenField() {
        if (!this._childrenField) {
          const dataField = this.fieldMap.children;
          this._childrenField = (dataField === null || dataField === void 0 ? void 0 : dataField.dataSource) || 'children';
        }
        return this._childrenField;
      }
      /**
       * Returns index path to this node. This is the index of each node in the node path
       * starting from the topmost parent. (only relevant when its part of a tree store).
       * @returns {Number[]} The index of each node in the path from the topmost parent to this node.
       * @category Parent & children
       * @private
       */
      get indexPath() {
        const indices = [];
        let node = this,
          depth = node.childLevel;
        for (node = this; node && !node.isRoot; node = node.parent) {
          indices[depth--] = node.parentIndex + 1;
        }
        return indices;
      }
      /**
       * Unique identifier for the record. Might be mapped to another dataSource using idField, but always exposed as
       * record.id. Will get a generated value if none is specified in records data.
       *
       * {@note}Note that generated ids are meant to be temporary (phantom ids), they should not be serialized
       * but instead replaced by the backend on commit{/@note}
       *
       * @field {String|Number} id
       * @category Common
       */
      //region Init
      /**
       * Constructs a new record from the supplied data config.
       * @param {Object} [config] Raw model config
       * @param {Core.data.Store} [store] Data store
       * @param {Object} [meta] Meta data
       * @privateparam {Boolean} [skipExpose] Skip exposing properties from data
       * @privateparam {Boolean} [forceUseRaw] Force using raw data, used by copy to not clone data twice
       * @function constructor
       * @category Lifecycle
       */
      construct(config = {}, store = null, meta = null, skipExpose = false, forceUseRaw = false, rawData = false) {
        var _me$meta$initableValu;
        const me = this,
          stores = ArrayHelper.asArray(store) ?? [],
          {
            constructor,
            fieldMap
          } = me;
        // null passed to Base construct inhibits config processing.
        let configs = null;
        store = stores[0];
        me.meta = {
          modified: {},
          ...constructor.metaConfig,
          ...meta
        };
        // Should apply configs?
        if (constructor.applyConfigs) {
          // Extract from data and combine with defaultConfigs
          for (const key in me.getDefaultConfiguration()) {
            if (!configs) {
              // if (first config)
              configs = {};
              if (!me.useRawData || !me.useRawData.enabled) {
                // Shallow copy of data to not mutate incoming object
                config = {
                  ...config
                };
              }
            }
            // Loop through configs excluding fields
            if (key in config) {
              // Let defaults override any config supplied with an `undefined` value
              if (config[key] !== undefined) {
                // Use as config
                configs[key] = config[key];
              }
              // Always remove config from data
              delete config[key];
            }
          }
        }
        super.construct(configs);
        // make getters/setters for fields, needs to be done before processing data to make sure defaults are available
        if (!skipExpose) {
          constructor.exposeProperties(config, rawData);
        }
        // It's only valid to do this once, on construction of the first instance
        if (!hasOwn$2(constructor, 'idFieldProcessed')) {
          // idField can be overridden from meta, or from the store if we have not had an idField set programmatically
          // and if we have not had an id field defined above the base Model class level.
          let overriddenIdField = me.meta.idField;
          if (!overriddenIdField) {
            // Might have been set to Model after construction but before load
            if (constructor._assignedIdField) {
              overriddenIdField = constructor.idField;
            } else if (store) {
              overriddenIdField = store.idField;
            }
          }
          // If it's overridden to something different than we already have, replace the 'id' field in the fieldMap
          if (overriddenIdField && overriddenIdField !== fieldMap.id.dataSource) {
            constructor.addField({
              name: 'id',
              dataSource: overriddenIdField,
              internal: true
            });
          }
          // Model.idField should always reflect the idField mapping
          constructor._idField = fieldMap.id.dataSource;
          constructor.idFieldProcessed = true;
        }
        // assign internalId, unique among all records
        me._internalId = Model._internalIdCounter++;
        // relation code expects store to be available for relation lookup, but actual join done below
        me.stores = [];
        me.unjoinedStores = [];
        // Superclass constructors may set this in their own way before this is called.
        if (!me.originalData) {
          me.originalData = config;
        }
        me.data = constructor.processData(config, false, store, me, forceUseRaw);
        // Assign any intiable value last, so that it can reference this record if needed
        ((_me$meta$initableValu = me.meta.initableValues) === null || _me$meta$initableValu === void 0 ? void 0 : _me$meta$initableValu.size) && me.assignInitables();
        // Consider undefined and null as missing id and generate one
        if (me.id == null) {
          // Assign a generated id silently, record should not be considered modified
          me.setData('id', me.generateId(store));
        }
        if (me.data[constructor.childrenField]) {
          me.processChildren(stores);
        }
        me.generation = 0;
      }
      /**
       * Set this property to `true` when adding a record on a conditional basis, that is, it is yet
       * to be confirmed as an addition.
       *
       * When this is set, the {@link #property-isPersistable} value of the record is **false**, and upon being
       * added to a Store it will *not* be eligible to be synced with the server as an added record.
       *
       * Subsequently, *clearing* this property means this record will become persistable and eligible
       * for syncing as an added record.
       * @property {Boolean}
       * @category Editing
       */
      set isCreating(isCreating) {
        const me = this;
        // A no-change must not have any effect.
        if (Boolean(me.meta.isCreating) !== isCreating) {
          // This flag contributes to the evaluation of isPersistable.
          // A record is not persistable if it isCreating.
          me.meta.isCreating = isCreating;
          // Owning Stores may have things to do at this lifecycle point
          me.stores.forEach(s => {
            s.onIsCreatingToggle(me, isCreating);
          });
        }
      }
      get isCreating() {
        return Boolean(this.meta.isCreating);
      }
      /**
       * Compares this Model instance to the passed instance. If they are of the same type, and all fields
       * (except, obviously, `id`) are equal, this returns `true`.
       * @param {Core.data.Model} other The record to compare this record with.
       * @returns {Boolean} `true` if the other is of the same class and has all fields equal.
       * @category Misc
       */
      equals(other) {
        if (other instanceof this.constructor) {
          for (let fields = this.$meta.fields.defs, i = 0, {
              length
            } = fields; i < length; i++) {
            const field = fields[i],
              {
                name
              } = field;
            if (name !== 'id' && !field.isEqual(this.getValue(name), other.getValue(name))) {
              return false;
            }
          }
          return true;
        }
        return false;
      }
      get subclass() {
        return new this.constructor(Object.setPrototypeOf({
          id: _undefined
        }, this.data), this.stores[0], null, true);
      }
      /**
       * Processes raw data, converting values and setting defaults.
       * @private
       * @param {Object} data Raw data
       * @param {Boolean} [ignoreDefaults] Ignore setting default values, used when updating
       * @returns {Object} Processed data
       * @category Fields
       */
      static processData(data, ignoreDefaults = false, store, record, forceUseRaw) {
        const {
            fieldMap,
            defaultValues
          } = this,
          {
            useRawData = {
              enabled: false
            }
          } = store || {},
          // Store configured with useRawData uses the supplied data object, polluting it. When not configured with
          // useRawData it instead makes a copy (intentionally deep, in case data has a prototype chain or contains
          // arrays or objects)
          processed = forceUseRaw || useRawData.enabled ? data : ObjectHelper.clone(data);
        let fieldName;
        ignoreDefaults = ignoreDefaults || useRawData.disableDefaultValue || forceUseRaw;
        if (!ignoreDefaults) {
          for (fieldName in defaultValues) {
            if (processed[fieldName] === _undefined) {
              let defaultValue = defaultValues[fieldName];
              // Avoid sharing arrays across instances
              if (Array.isArray(defaultValue)) {
                defaultValue = defaultValue.slice();
              }
              processed[fieldName] = defaultValue;
            }
          }
        }
        if (!useRawData.disableTypeConversion && !forceUseRaw) {
          // Convert field types which need converting
          for (fieldName in fieldMap) {
            const field = fieldMap[fieldName],
              {
                name,
                dataSource
              } = field,
              // Value might have been supplied either using mapped dataSource (when loading JSON etc. for example
              // event.myStartDate) or as field name (from internal code, for example event.startDate). If [name]
              // exists but not [dataSource], use it.
              hasSource = dataSource !== name,
              complex = field.complexMapping,
              sourceExists = hasSource && (complex ? ObjectHelper.pathExists(data, dataSource) : dataSource in data),
              useNameForValue = name in data && (!hasSource || !sourceExists),
              convert = !useRawData.disableTypeConversion && field.convert;
            // Only action field definitions which have a convert function or remap data
            if (useNameForValue || convert) {
              // When ignoringDefaults, do not convert unspecified values
              if (!ignoreDefaults || useNameForValue || sourceExists) {
                const value = useNameForValue ? processed[name] : complex ? ObjectHelper.getPath(processed, dataSource) : processed[dataSource],
                  converted = convert ? field.convert(value, data, record) : value;
                if (complex) {
                  ObjectHelper.setPath(processed, dataSource, converted);
                } else {
                  processed[dataSource] = converted;
                }
                // Remove [startDate] from internal data holder, only keeping [myStartDate]
                if (hasSource) {
                  delete processed[name];
                }
              }
            }
          }
        }
        // Fields that needs initializing
        this.$meta.fields.initable.length && this.initInitables(record, processed);
        return processed;
      }
      static setupClass(meta) {
        super.setupClass(meta);
        if (!meta.fields) {
          // Normally setupFields will only run when a Model defines a fields getter, but we want to always run it:
          this.setupFields(this, meta);
        }
      }
      static setupFields(cls, meta) {
        const classFields = hasOwn$2(cls, 'fields') && cls.fields,
          base = meta.super.fields,
          fieldsInfo = meta.fields = {
            defs: (base === null || base === void 0 ? void 0 : base.defs.slice()) ?? [],
            // Set to true when an instance's data object is run through exposeProperties
            exposedData: false,
            // These objects are all keyed by field name:
            defaults: base ? {
              ...base.defaults
            } : {},
            // value=field.defaultValue
            exposed: Object.create((base === null || base === void 0 ? void 0 : base.exposed) ?? null),
            // value=true if we've done defineProperty
            ordinals: Object.create((base === null || base === void 0 ? void 0 : base.ordinals) ?? null),
            // value=index in the defs array
            map: Object.create((base === null || base === void 0 ? void 0 : base.map) ?? null),
            // value=definition object
            sources: Object.create((base === null || base === void 0 ? void 0 : base.sources) ?? null) // value=source definition object
          };
        // We use Object.create(null) as the base for these maps because some models declare "constructor" as a field
        // NOTE: instead of chaining the defaults, we copy them so the defaults object can be used with Object.assign
        // in other contexts (since it does not copy inherited properties from the prototype chain)
        // Clone the superclass's defaults, and override that with our own defaults.
        // As we find fields with a defaultValue, more defaults may be added
        if (hasOwn$2(cls, 'defaults')) {
          Object.assign(fieldsInfo.defaults, cls.defaults);
        }
        // Hook up our field maps with the class hierarchy's fieldMaps.
        // We need to be able to look up field definitions by the name, or by the dataSource property name
        // If the idField is overridden at this level, create a new field
        if (hasOwn$2(cls, 'idField')) {
          cls.addField({
            name: 'id',
            dataSource: cls.idField,
            internal: true
          });
          fieldsInfo.exposed[cls.idField] = true;
        }
        // Process fields defined in the class definition
        if (classFields !== null && classFields !== void 0 && classFields.length) {
          classFields.map(cls.addField, cls);
        }
        // Collect fields that need to be initialized (StoreDataField for example)
        fieldsInfo.initable = fieldsInfo.defs.filter(field => field.init);
        cls.exposeRelations();
      }
      static get defaultValues() {
        return this.$meta.fields.defaults;
      }
      /**
       * An array containing all the _defined_ fields for this Model class. This will include all superclass's
       * defined fields.
       * @property {Core.data.field.DataField[]}
       * @static
       * @readonly
       * @category Fields
       */
      static get allFields() {
        return this.$meta.fields.defs;
      }
      /**
       * Same as {@link #property-allFields-static}.
       * @property {Core.data.field.DataField[]}
       * @readonly
       * @category Fields
       */
      get allFields() {
        return this.$meta.fields.defs;
      }
      /**
       * An object containing all the _defined_ fields for this Model class. This will include all superclass's
       * defined fields through its prototype chain. So be aware that `Object.keys` and `Object.entries` will only
       * access this class's defined fields.
       * @property {Object<String,Core.data.field.DataField>}
       * @static
       * @readonly
       * @category Fields
       */
      static get fieldMap() {
        return this.$meta.fields.map;
      }
      /**
       * Same as {@link #property-fieldMap-static}.
       * @property {Object<String,Core.data.field.DataField>}
       * @readonly
       * @category Fields
       */
      get fieldMap() {
        return this.$meta.fields.map;
      }
      static get fieldDataSourceMap() {
        return this.$meta.fields.sources;
      }
      /**
       * Makes getters and setters for fields (from definitions and data). Called once when class is defined and once when
       * data is loaded first time.
       * @internal
       * @param {Object} [data] Raw data
       * @param {Boolean} [raw=true] True if data is raw (contains data sources), False if data contains field names
       * @category Fields
       */
      static exposeProperties(data, raw = true) {
        const me = this,
          fieldsInfo = me.$meta.fields,
          // exposeProperties method is called from two different places: from the model constructor which receives
          // field names, and from store loadData method, which handles raw data. When loading data to store we need
          // to use names as specified in the dataSource. And when calling a model constructor we need to use field
          // names
          fieldMapProperty = raw ? 'exposed' : 'map';
        // Process the raw data properties and expose them as fields unless the property name
        // has already been used by the "dataSource" of a defined field.
        if (data && me.autoExposeFields && !fieldsInfo.exposedData) {
          let dataProperty, fieldDef, type;
          for (dataProperty in data) {
            // We need to skip children field because it can be `true` and that would create boolean field.
            // See https://github.com/bryntum/support/issues/2705
            if (!fieldsInfo[fieldMapProperty][dataProperty] && dataProperty !== me.childrenField) {
              type = ObjectHelper.typeOf(data[dataProperty]);
              // Create a field definition in our fieldMap with the flag that it's from data
              fieldDef = {
                name: dataProperty,
                dataSource: dataProperty,
                fromData: true
              };
              if (fieldDataTypes[type]) {
                fieldDef.type = type;
              }
              me.addField(fieldDef);
            }
          }
          fieldsInfo.exposedData = true;
        }
        me.exposeRelations();
      }
      /**
       * Add a field definition in addition to those predefined in `fields`.
       * @param {String|ModelFieldConfig} fieldDef A field name or definition
       * @category Fields
       */
      static addField(fieldDef) {
        if (fieldDef == null) {
          return;
        }
        if (typeof fieldDef === 'string') {
          fieldDef = {
            name: fieldDef
          };
        }
        const me = this.initClass(),
          fieldsInfo = me.$meta.fields,
          {
            ordinals
          } = fieldsInfo,
          propertiesExposed = fieldsInfo.exposed,
          {
            name
          } = fieldDef,
          existing = fieldsInfo.map[name],
          dataSource = fieldDef.dataSource || (fieldDef.dataSource = name);
        let field, key;
        if (!existing || fieldDef.type && fieldDef.type !== existing.type) {
          field = DataField.create(fieldDef);
          field.definedBy = existing ? existing.definedBy : me;
          field.ordinal = existing ? existing.ordinal : ordinals[name] = fieldsInfo.defs.length;
        } else {
          field = Object.create(existing);
          for (key in fieldDef) {
            if (key !== 'type') {
              field[key] = fieldDef[key];
            }
          }
        }
        field.owner = me;
        fieldsInfo.defs[field.ordinal] = field;
        fieldsInfo.map[name] = field;
        if (!fieldsInfo.sources[dataSource]) {
          fieldsInfo.sources[dataSource] = field;
        }
        // With complex mapping avoid exposing object as model field
        if (dataSource.includes('.')) {
          field.complexMapping = true;
        }
        if (field.complexMapping) {
          // model fields have this set on their prototype...
          propertiesExposed[dataSource.split('.')[0]] = true;
        } else {
          // When iterating through the raw data, if autoExposeFields is set
          // We do not need to create properties for raw property names we've processed here
          propertiesExposed[dataSource] = true;
        }
        // Maintain an object of defaultValues for fields.
        if ('defaultValue' in field) {
          fieldsInfo.defaults[dataSource] = field.defaultValue;
        }
        // Create a property on this Model's prototype, named for the defined field name
        // which reads the correct property out of the raw data object.
        if (!internalProps[name]) {
          // Either creates a new accessor or redefines an existing
          field.defineAccessor(me.prototype);
        }
        me._nonPersistableFields = null;
        me._alwaysWriteFields = null;
        return field;
      }
      /**
       * Remove a field definition by name.
       * @param {String} fieldName Field name
       * @category Fields
       */
      static removeField(fieldName) {
        const me = this.initClass(),
          fieldsInfo = me.$meta.fields,
          definition = fieldsInfo.map[fieldName],
          {
            ordinals
          } = fieldsInfo,
          index = ordinals[fieldName];
        if (definition) {
          fieldsInfo.defs.splice(index, 1);
          delete ordinals[fieldName];
          delete fieldsInfo.defaults[fieldName];
          delete fieldsInfo.exposed[fieldName];
          delete fieldsInfo.map[fieldName];
          delete fieldsInfo.sources[definition.dataSource];
          for (const name in ordinals) {
            if (ordinals[name] > index) {
              --ordinals[name];
            }
          }
          // Note: if field was exposed by superclass, this won't do anything...
          delete me.prototype[fieldName];
        }
      }
      /**
       * Makes getters and setters for related records. Populates a Model#relation array with the relations, to allow it
       * to be modified later when assigning stores.
       * @internal
       * @category Relations
       */
      static exposeRelations() {
        const me = this;
        if (hasOwn$2(me, 'exposedRelations')) {
          return;
        }
        if (me.relations) {
          me.exposedRelations = [];
          for (const relationName in me.relations) {
            const relation = me.relations[relationName];
            relation.relationName = relationName;
            me.exposedRelations.push(relation);
            // getter and setter for related object
            if (!Reflect.ownKeys(me.prototype).includes(relationName)) {
              defineProperty$4(me.prototype, relationName, {
                enumerable: true,
                get: function () {
                  // noinspection JSPotentiallyInvalidUsageOfClassThis
                  return this.getForeign(relationName);
                },
                set: function (value) {
                  // noinspection JSPotentiallyInvalidUsageOfClassThis
                  this.setForeign(relationName, value, relation);
                }
              });
            }
          }
        }
      }
      //endregion
      //region Initable fields
      // Initializes any fields using a data type that has an init method, and caches the value to assign to that field to
      // be able to assign it after all others. That allows the initter to reference the records other data if needed
      // (baselines use that to reference the task to get default values)
      static initInitables(record, processedData) {
        const laterValues = record.meta.initableValues = new Map();
        // Initialize any initializable fields (only StoreDataField currently)
        for (const field of this.$meta.fields.initable) {
          // Set data (if any) later, in case it needs to reference this record
          const value = ObjectHelper.getPath(processedData, field.dataSource);
          value !== undefined && laterValues.set(field, value);
          // Init field if not lazy, if lazy app is responsible for doing it
          !field.lazy && field.init(processedData, record);
        }
      }
      // Assigns values to the fields that were initialized earlier (see initInitables above)
      assignInitables() {
        const {
          initableValues
        } = this.meta;
        for (const [field, value] of initableValues) {
          if (field.set(value, this.data, this) !== false) {
            initableValues.delete(field);
          }
        }
      }
      //endregion
      //region Fields
      /**
       * Flag checked from Store when loading data that determines if fields found in first records should be exposed in
       * same way as predefined fields.
       *
       * {@note}Note that we for all but the most basic use cases recommend explicitly defining the fields.
       * Having them auto exposed can lead to unexpected behavior, if the first record is not complete (fields missing,
       * null etc).
       * {/@note}
       *
       * @property {Boolean}
       * @category Fields
       */
      static get autoExposeFields() {
        return true;
      }
      /**
       * This function forces correct field order. Correct order is parentId before id. If we process id field before
       * parentId, idMap won't be updated and changing parent node will lead to duplicated records in storage
       * @param {String} a
       * @param {String} b
       * @returns {number}
       * @private
       */
      static fieldSorter(a, b) {
        return (fieldsOrder[a] || 100) - (fieldsOrder[b] || 100);
      }
      /**
       * Convenience getter to get field definitions from class.
       * @property {Core.data.field.DataField[]}
       * @readonly
       * @category Fields
       */
      get fields() {
        return this.$meta.fields.defs;
      }
      /**
       * Convenience function to get the definition for a field from class.
       * @param {String} fieldName Field name
       * @returns {Core.data.field.DataField}
       * @category Fields
       */
      getFieldDefinition(fieldName) {
        return this.$meta.fields.map[fieldName];
      }
      getFieldDefinitionFromDataSource(dataSource) {
        return this.$meta.fields.sources[dataSource];
      }
      /**
       * Get the names of all fields in data.
       * @property {String[]}
       * @readonly
       * @category Fields
       */
      get fieldNames() {
        return Object.keys(this.data);
      }
      /**
       * Get the definition for a field by name. Caches results.
       * @param {String} fieldName Field name
       * @returns {Core.data.field.DataField} Field definition or null if none found
       * @category Fields
       */
      static getFieldDefinition(fieldName) {
        return this.$meta.fields.map[fieldName];
      }
      /**
       * Returns dataSource configuration for a given field name
       * @param {String} fieldName
       * @returns {String} Field `dataSource` mapping
       * @internal
       */
      static getFieldDataSource(fieldName) {
        var _this$getFieldDefinit;
        return ((_this$getFieldDefinit = this.getFieldDefinition(fieldName)) === null || _this$getFieldDefinit === void 0 ? void 0 : _this$getFieldDefinit.dataSource) || fieldName;
      }
      /**
       * Get the data source used by specified field. Returns the fieldName if no data source specified.
       * @param {String} fieldName Field name
       * @returns {String}
       * @category Fields
       */
      getDataSource(fieldName) {
        const def = this.constructor.getFieldDefinition(fieldName);
        return (def === null || def === void 0 ? void 0 : def.dataSource) || (def === null || def === void 0 ? void 0 : def.name);
      }
      /**
       * Processes input to a field, converting to expected type.
       * @param {String} fieldName Field name
       * @param {*} value Value to process
       * @returns {*} Converted value
       * @category Fields
       */
      static processField(fieldName, value, record) {
        const field = this.fieldMap[fieldName];
        return field !== null && field !== void 0 && field.convert ? field.convert(value, this.data, record) : value;
      }
      //endregion
      //region Relations
      /**
       * Initializes model relations. Called from store when adding a record.
       * @private
       * @category Relations
       */
      initRelations() {
        const me = this,
          relations = me.constructor.exposedRelations;
        if (!relations) {
          return;
        }
        me.stores.forEach(store => {
          var _store$modelRelations;
          if (!store.modelRelations) {
            store.initRelations();
          }
          const relatedRecords = [];
          (_store$modelRelations = store.modelRelations) === null || _store$modelRelations === void 0 ? void 0 : _store$modelRelations.forEach(config => {
            relatedRecords.push({
              related: me.initRelation(config),
              config
            });
          });
          store.updateRecordRelationCache(me, relatedRecords);
        });
      }
      /**
       * Initializes/updates a single relation.
       * @param config Relation config
       * @returns {Core.data.Model} Related record
       * @private
       * @category Relations
       */
      initRelation(config) {
        const me = this,
          foreignId = me.get(config.foreignKey),
          foreign = foreignId !== _undefined && typeof config.foreignStore !== 'string' && config.foreignStore.getById(foreignId),
          relationCache = me.meta.relationCache || (me.meta.relationCache = {});
        // apparently scheduler tests expect cache to work without matched related record, thus the placeholder
        relationCache[config.relationName] = foreign || (foreignId != null ? {
          id: foreignId,
          placeHolder: true
        } : null);
        return foreign;
      }
      removeRelation(config) {
        const {
          relationName,
          foreignKey,
          nullFieldOnRemove
        } = config;
        // (have to check for existence before deleting to work in Safari)
        if (this.meta.relationCache[relationName]) {
          delete this.meta.relationCache[relationName];
          if (nullFieldOnRemove) {
            // Setting to null silently, to not trigger additional relation behaviour
            this.setData(foreignKey, null);
          }
        }
      }
      getForeign(name) {
        var _this$meta$relationCa;
        return (_this$meta$relationCa = this.meta.relationCache) === null || _this$meta$relationCa === void 0 ? void 0 : _this$meta$relationCa[name];
      }
      setForeign(name, value, config) {
        const id = Model.asId(value);
        return this.set(config.foreignKey, id);
      }
      /**
       * Get a relation config by name, from the first store.
       * @param {String} name
       * @returns {Object}
       * @private
       * @category Relations
       */
      getRelationConfig(name) {
        var _this$firstStore, _this$firstStore$mode;
        //?
        // Using first store for relations, might have to revise later...
        return (_this$firstStore = this.firstStore) === null || _this$firstStore === void 0 ? void 0 : (_this$firstStore$mode = _this$firstStore.modelRelations) === null || _this$firstStore$mode === void 0 ? void 0 : _this$firstStore$mode.find(r => r.foreignKey === name);
      }
      //endregion
      //region Get/set values, data handling
      flatGet(fieldName, dataSource) {
        // NOTE: There is an inlined copy of this fn in DataField, when changing here make sure it is updated too
        // When changes are batched, they get stored by field name, not dataSource
        if (this.batching && fieldName in this.meta.batchChanges) {
          return this.meta.batchChanges[fieldName];
        }
        return dataSource in this.data ? this.data[dataSource] : this.data[fieldName];
      }
      complexGet(fieldName, dataSource) {
        // When changes are batched, they get stored by field name, not dataSource
        if (this.batching && fieldName in this.meta.batchChanges) {
          return this.meta.batchChanges[fieldName];
        }
        return ObjectHelper.getPath(this.data, dataSource);
      }
      /**
       * Get value for specified field name. You can also use the generated getters if loading through a Store.
       * If model is currently in batch operation this will return updated batch values which are not applied to Model
       * until endBatch() is called.
       * @param {String} fieldName Field name to get value from
       * @returns {*} Fields value
       * @category Fields
       */
      get(fieldName) {
        if (!fieldName) {
          return;
        }
        const me = this,
          field = me.fieldMap[fieldName];
        if (!field) {
          // team.players[0], team.players[0].name
          if (fieldName.includes('[')) {
            const [, arrayFieldName, index, path] = fieldName.match(arrayRe);
            // team
            const arrayField = me.fieldMap[arrayFieldName];
            if (arrayField !== null && arrayField !== void 0 && arrayField.getAt) {
              // Access the field to make sure it is initialized if lazy
              me._thisIsAUsedExpression(me[arrayFieldName]);
              // 0
              const subRecord = arrayField.getAt(me, index);
              // name
              if (subRecord && path) {
                if (subRecord.isModel) {
                  return subRecord.getValue(path);
                }
                return subRecord[path];
              }
              return subRecord;
            }
            return null;
          }
          if (fieldName.includes('.')) {
            // Related record?
            if (!ObjectHelper.hasPath(me.data, fieldName)) {
              return ObjectHelper.getPath(me, fieldName);
            }
            // Nested data object
            return me.complexGet(fieldName, fieldName);
          }
        }
        if (field !== null && field !== void 0 && field.complexMapping) {
          return me.complexGet(fieldName, field.dataSource);
        }
        return me.flatGet(fieldName, (field === null || field === void 0 ? void 0 : field.dataSource) || fieldName);
      }
      // Used to get field values, replaces `record[fieldName]` in internal code to allow relations etc.
      getValue(fieldName) {
        if (!fieldName) {
          return;
        }
        // Try matching "user defined" getters
        if (fieldName in this) {
          return this[fieldName];
        }
        // Fall back to normal get
        return this.get(fieldName);
      }
      // Used to set field values, replacing `record[fieldName] = value` to handle dot notation
      setValue(fieldName, value) {
        if (fieldName in this) {
          this[fieldName] = value;
        } else {
          this.set(fieldName, value);
        }
      }
      /**
       * Internal function used to update a records underlying data block (record.data) while still respecting field
       * mappings. Needed in cases where a field needs setting without triggering any associated behaviour and it has a
       * dataSource with a different name.
       *
       * For example:
       * ```javascript
       * // startDate mapped to data.beginDate
       * { name : 'startDate', dataSource : 'beginDate' }
       *
       * // Some parts of our code needs to update the data block without triggering any of the behaviour associated with
       * // calling set. This would then not update "beginDate":
       * record.data.startDate = xx;
       *
       * // But this would
       * record.setData('startDate', xx);
       * ```
       * @internal
       * @category Editing
       */
      setData(toSet, value) {
        const {
          data,
          fieldMap
        } = this;
        // Two separate paths for performance reasons
        // setData('name', 'Quicksilver');
        if (typeof toSet === 'string') {
          const field = fieldMap[toSet],
            dataSource = (field === null || field === void 0 ? void 0 : field.dataSource) ?? toSet;
          if (field !== null && field !== void 0 && field.set) {
            field.set(value, this.data, this);
          } else if (field !== null && field !== void 0 && field.complexMapping) {
            ObjectHelper.setPath(data, dataSource, value);
          } else {
            data[dataSource] = value;
          }
        }
        // setData({ name : 'Magneto', power : 'Magnetism' });
        else {
          const keys = Object.keys(toSet);
          for (let i = 0; i < keys.length; i++) {
            const fieldName = keys[i],
              field = fieldMap[fieldName],
              dataSource = (field === null || field === void 0 ? void 0 : field.dataSource) ?? fieldName;
            if (field !== null && field !== void 0 && field.set) {
              field.set(value, this.data, this);
            } else if (dataSource) {
              if (field !== null && field !== void 0 && field.complexMapping) {
                ObjectHelper.setPath(data, dataSource, toSet[fieldName]);
              } else {
                data[dataSource] = toSet[fieldName];
              }
            }
          }
        }
      }
      /**
       * Returns raw data from the encapsulated data object for the passed field name
       * @param {String} fieldName The field to get data for.
       * @returns {*} The raw data value for the field.
       * @category Editing
       */
      getData(fieldName) {
        const field = this.fieldMap[fieldName],
          dataSource = (field === null || field === void 0 ? void 0 : field.dataSource) ?? fieldName;
        if (dataSource) {
          if (field !== null && field !== void 0 && field.complexMapping) {
            return ObjectHelper.getPath(this.data, dataSource);
          }
          return this.data[dataSource];
        }
      }
      /**
       * Silently updates record's id with no flagging the property as modified.
       * Triggers onModelChange event for changed id.
       * @param {String|Number} value id value
       * @private
       */
      syncId(value) {
        const oldValue = this.id;
        if (oldValue !== value) {
          this.setData('id', value);
          const data = {
            id: {
              value,
              oldValue
            }
          };
          this.afterChange(data, data);
        }
      }
      /**
       * Set value for the specified field. You can also use the generated setters if loading through a Store.
       *
       * Setting a single field, supplying name and value:
       *
       * ```javascript
       * record.set('name', 'Clark');
       * ```
       *
       * Setting multiple fields, supplying an object:
       *
       * ```javascript
       * record.set({
       *     name : 'Clark',
       *     city : 'Metropolis'
       * });
       * ```
       *
       * @param {String|Object} field The field to set value for, or an object with multiple values to set in one call
       * @param {*} [value] Value to set
       * @param {Boolean} [silent] Set to true to not trigger events. If event is recurring, occurrences won't be updated automatically.
       * @privateparam {Boolean} [validOnly] If set to `true` it will ignore setting a `undefined` value, allowing conversion functions to invalidate a data input
       * @privateparam {Boolean} [triggerBeforeUpdate]
       * @fires Store#idChange
       * @fires Store#update
       * @fires Store#change
       * @category Editing
       */
      set(field, value, silent = false, fromRelationUpdate = false, skipAccessors = false, validOnly = false, triggerBeforeUpdate = true) {
        const me = this;
        if (me.isBatchUpdating) {
          me.inBatchSet(field, value, silent || me.$silenceBatch);
          return null;
        } else {
          var _me$afterSet;
          // We use inSet/afterSet approach here because mixin interested in overriding set() method like STM, for
          // example, might be mixed before Model class or after. In general, we have no control over this.
          // STM mixed before, so the only option to wrap set() method body is actually to call afterSet().
          const wasSet = me.inSet(field, value, silent, fromRelationUpdate, skipAccessors, validOnly, triggerBeforeUpdate);
          (_me$afterSet = me.afterSet) === null || _me$afterSet === void 0 ? void 0 : _me$afterSet.call(me, field, value, silent, fromRelationUpdate, wasSet);
          return wasSet;
        }
      }
      fieldToKeys(field, value) {
        if (typeof field !== 'string') {
          // will get in trouble when setting same field on multiple models without this
          return ObjectHelper.assign({}, field);
        }
        return {
          [field]: value
        };
      }
      inBatchSet(field, value, silent) {
        const me = this,
          {
            meta,
            constructor,
            fieldMap
          } = me,
          wasSet = {};
        let cmp,
          changed = false;
        if (typeof field !== 'string') {
          Object.keys(me.fieldToKeys(field, value)).forEach(key => {
            cmp = fieldMap[key] || ObjectHelper;
            value = constructor.processField(key, field[key], me);
            // Store batch changes
            if (!cmp.isEqual(meta.batchChanges[key], value)) {
              wasSet[key] = {
                value,
                oldValue: me.get(key)
              };
              meta.batchChanges[key] = value;
              changed = true;
            }
          });
        } else {
          cmp = fieldMap[field] || ObjectHelper;
          // Minor optimization for engine writing back a lot of changes
          if (!cmp.isEqual(meta.batchChanges[field], value)) {
            wasSet[field] = {
              value,
              oldValue: me.get(field)
            };
            meta.batchChanges[field] = value;
            changed = true;
          }
        }
        // Callers need to be able to detect changes
        if (changed) {
          me.generation++;
          if (!silent) {
            // Fire batched events for UIs which need to update themselves during batched updates.
            // An example is evenResize feature which batches the changes to the endDate
            // or startDate, but the UI must update during the drag.
            const event = {
              action: 'update',
              record: me,
              records: [me],
              changes: wasSet
            };
            me.stores.forEach(store => {
              store.trigger('batchedUpdate', {
                ...event
              });
            });
            // Propagate to linked records
            me.forEachLinked((store, record) => store.trigger('batchedUpdate', {
              ...event,
              record,
              records: [record]
            }));
          }
        }
      }
      inSet(fieldNameOrObject, value, silent, fromRelationUpdate, skipAccessors = false, validOnly = false, triggerBeforeUpdate = true) {
        const me = this,
          {
            data,
            meta,
            fieldMap,
            constructor
          } = me,
          {
            prototype: myProto,
            childrenField,
            relations
          } = constructor,
          wasSet = {},
          toSet = me.fieldToKeys(fieldNameOrObject, value),
          // Sort fields to make sure parentId is processed before id
          // https://github.com/bryntum/support/issues/6851
          keys = Object.keys(toSet).sort(constructor.fieldSorter);
        let changed = false;
        // Give a chance to cancel action before records are updated.
        if (!silent && !me.triggerBeforeUpdate(toSet)) {
          return null;
        }
        me.inSetting = true;
        for (let i = 0; i < keys.length; i++) {
          const key = keys[i];
          // Currently not allowed to set children in a TreeNode this way, will be ignored
          if (key === childrenField) {
            continue;
          }
          // Setting a field on a record in a store data field (players[0].name = 'foo')
          if (key.includes('[')) {
            const [, arrayFieldName, index, path] = key.match(arrayRe),
              field = me.fieldMap[arrayFieldName];
            if (field !== null && field !== void 0 && field.getAt) {
              const subRecord = field.getAt(me, index);
              if (subRecord.isModel) {
                subRecord.set(path, toSet[key]);
              } else {
                ObjectHelper.setPath(subRecord, path, toSet[key]);
              }
              continue;
            }
          }
          const complexKey = key.includes('.');
          // Handle updating related record
          if (relations && complexKey) {
            const [, relationName, prop] = key.match(nestedRe);
            if (relations[relationName]) {
              me[relationName].set(prop, toSet[key]);
              continue;
            }
          }
          const field = fieldMap[key],
            cmp = field || ObjectHelper,
            readOnly = field === null || field === void 0 ? void 0 : field.readOnly,
            mapping = (field === null || field === void 0 ? void 0 : field.dataSource) ?? key,
            useProp = !skipAccessors && !field && key in myProto || (field === null || field === void 0 ? void 0 : field.useProp),
            oldValue = useProp ? me[mapping] : field !== null && field !== void 0 && field.complexMapping || complexKey ? ObjectHelper.getPath(data, mapping) : data[mapping],
            value = constructor.processField(key, toSet[key], me),
            val = toSet[key] = {
              value
            },
            relation = me.getRelationConfig(key);
          if (!readOnly && !cmp.isEqual(oldValue, value) && (!validOnly || value !== undefined)) {
            var _field$getOldValue;
            // Indicate to observers that data has changed.
            me.generation++;
            // Give fields a shot at affecting the old value (used by StoreDataField)
            val.oldValue = (field === null || field === void 0 ? void 0 : (_field$getOldValue = field.getOldValue) === null || _field$getOldValue === void 0 ? void 0 : _field$getOldValue.call(field, me)) ?? oldValue;
            changed = true;
            // Update `modified` state which is used in sync request
            if (key in meta.modified && cmp.isEqual(meta.modified[key], value)) {
              // Remove changes if values are the same
              Reflect.deleteProperty(meta.modified, key);
              // Hack for when reverting record changes when using engine. If data is not made up to date here,
              // the write-back from engine coming later will detect the revert as a change. And round we go
              if (me.isReverting) {
                me.data[mapping] = value;
              }
            } else if (!me.ignoreBag) {
              // Private flag in engine, speeds initial commit up by not recording changes
              // Cache its original value
              if (!(key in meta.modified)) {
                me.storeFieldChange(key, oldValue);
              }
              if (val.oldValue === _undefined) {
                Reflect.deleteProperty(val, 'oldValue');
              }
            }
            // The wasSet object keys must be the field *name*, not its dataSource.
            wasSet[key] = val;
            me.applyValue(useProp, mapping, value, skipAccessors, field);
            // changing foreign key
            if (relation && !fromRelationUpdate) {
              me.initRelation(relation);
              me.stores.forEach(store => store.cacheRelatedRecord(me, value, relation.relationName, val.oldValue));
            }
          }
        }
        if (changed) {
          me.afterChange(toSet, wasSet, silent, fromRelationUpdate, skipAccessors);
        }
        me.inSetting = false;
        return changed ? wasSet : null;
      }
      // Provided as a hook for Engine to do what needs to be done which ever way a field value is changed
      applyValue(useProp, key, value, skipAccessors, field) {
        var _field, _field3;
        const me = this;
        // Setting parentId moves the node, always route through setter
        if ((((_field = field) === null 