/*!
 *
 * Bryntum Scheduler Pro 5.5.0
 *
 * Copyright(c) 2023 Bryntum AB
 * https://bryntum.com/contact
 * https://bryntum.com/license
 *
 */
var __defProp = Object.defineProperty;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __publicField = (obj, key, value) => {
  __defNormalProp(obj, typeof key !== "symbol" ? key + "" : key, value);
  return value;
};
var __accessCheck = (obj, member, msg) => {
  if (!member.has(obj))
    throw TypeError("Cannot " + msg);
};
var __privateGet = (obj, member, getter) => {
  __accessCheck(obj, member, "read from private field");
  return getter ? getter.call(obj) : member.get(obj);
};
var __privateAdd = (obj, member, value) => {
  if (member.has(obj))
    throw TypeError("Cannot add the same private member more than once");
  member instanceof WeakSet ? member.add(obj) : member.set(obj, value);
};
var __privateSet = (obj, member, value, setter) => {
  __accessCheck(obj, member, "write to private field");
  setter ? setter.call(obj, value) : member.set(obj, value);
  return value;
};

// ../Core/lib/Core/helper/BrowserHelper.js
var _a;
var BrowserHelper = class {
  static cacheFlags(platform = navigator.platform, userAgent = navigator.userAgent) {
    const me = this;
    me._isLinux = Boolean(platform.match(/Linux/));
    me._isMac = Boolean(platform.match(/Mac/));
    me._isWindows = Boolean(platform.match(/Win32/));
    me._isMobile = Boolean(userAgent.match(/Mobile|Opera Mini|Opera Mobi|Puffin/) || typeof globalThis.orientation === "number");
    me._isWebkit = Boolean(userAgent.match(/WebKit/) && !userAgent.match(/Edg/));
    me._firefoxVersion = me.getVersion(userAgent, /Firefox\/(\d+)\./);
    me._isFirefox = me._firefoxVersion > 0;
    me._chromeVersion = me.getVersion(userAgent, /Chrom(?:e|ium)\/(\d+)\./);
    me._isChrome = me._chromeVersion > 0;
    me._isSafari = Boolean(userAgent.match(/Safari/)) && !me._isChrome;
    me._isMobileSafari = Boolean(userAgent.match(/Mobile.*Safari/));
    me._safariVersion = me.getVersion(userAgent, /Version\/(.*).Safari/);
    me._isAndroid = Boolean(userAgent.match(/Android/g));
  }
  //endregion
  //region Device
  /**
   * Yields `true` if the current browser supports CSS style `overflow:clip`.
   * @property {Boolean}
   * @readonly
   * @internal
   */
  static get supportsOverflowClip() {
    if (this._supportsOverflowClip == null) {
      const div = document.createElement("div");
      div.style.overflow = "clip";
      div.style.display = "none";
      document.documentElement.appendChild(div);
      this._supportsOverflowClip = div.ownerDocument.defaultView.getComputedStyle(div).getPropertyValue("overflow") === "clip";
      div.remove();
    }
    return this._supportsOverflowClip;
  }
  /**
   * Yields `true` if the current browser supports CSS style `position:sticky`.
   * @property {Boolean}
   * @readonly
   * @internal
   */
  static get supportsSticky() {
    return true;
  }
  /**
   * Returns matched version for userAgent.
   * @param {String} versionRe version match regular expression
   * @returns {Number} matched version
   * @readonly
   * @internal
   */
  static getVersion(userAgent, versionRe) {
    const match = userAgent.match(versionRe);
    return match ? parseFloat(match[1]) : 0;
  }
  /**
   * Determines if the user is using a touch device.
   * @property {Boolean}
   * @readonly
   * @internal
   */
  static get isTouchDevice() {
    if (this._isTouchDevice === void 0) {
      this._isTouchDevice = globalThis.matchMedia("(pointer:coarse)").matches;
    }
    return this._isTouchDevice;
  }
  // Reports true by default for our tests
  static get isHoverableDevice() {
    if (this._isHoverableDevice === void 0) {
      this._isHoverableDevice = globalThis.matchMedia("(any-hover: hover)").matches;
    }
    return this._isHoverableDevice;
  }
  //endregion
  //region Platform
  static get isBrowserEnv() {
    return typeof window !== "undefined";
  }
  /**
   * Checks if platform is Mac.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */
  static get isMac() {
    return this._isMac;
  }
  /**
   * Checks if platform is Windows.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */
  static get isWindows() {
    return this._isWindows;
  }
  /**
   * Checks if platform is Linux.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */
  static get isLinux() {
    return this._isLinux;
  }
  /**
   * Checks if platform is Android.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */
  static get isAndroid() {
    return this._isAndroid;
  }
  //endregion
  //region Browser
  /**
   * Checks if browser is Webkit.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isWebkit() {
    return this._isWebkit;
  }
  /**
   * Checks if browser is Chrome or Chromium based browser.
   * Returns truthy value for Edge Chromium.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isChrome() {
    return this._isChrome;
  }
  /**
   * Returns the major Chrome version or 0 for other browsers.
   * @property {Number}
   * @readonly
   * @category Browser
   */
  static get chromeVersion() {
    return this._chromeVersion;
  }
  /**
   * Checks if browser is Firefox.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isFirefox() {
    return this._isFirefox;
  }
  /**
   * Returns the major Firefox version or 0 for other browsers.
   * @property {Number}
   * @readonly
   * @category Browser
   */
  static get firefoxVersion() {
    return this._firefoxVersion;
  }
  /**
   * Checks if browser is Safari.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isSafari() {
    return this._isSafari;
  }
  static get safariVersion() {
    return this._safariVersion;
  }
  /**
   * Checks if browser is mobile Safari
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isMobileSafari() {
    return this._isMobileSafari;
  }
  /**
   * Checks if the active device is a mobile device
   * @property {Boolean}
   * @readonly
   * @category Browser
   */
  static get isMobile() {
    return this._isMobile;
  }
  static get platform() {
    const me = this;
    return me._isLinux ? "linux" : me._isMac ? "mac" : me._isWindows ? "windows" : me._isAndroid ? "android" : me._isMobileSafari ? "ios" : null;
  }
  /**
   * Returns `true` if the browser supports passive event listeners.
   * @property {Boolean}
   * @internal
   * @deprecated Since 5.0. All modern browsers now support passive event listeners.
   * @category Browser
   */
  static get supportsPassive() {
    return true;
  }
  // Only works in secure contexts
  static get supportsRandomUUID() {
    if (this._supportsRandomUUID === void 0) {
      try {
        this._supportsRandomUUID = Boolean(globalThis.crypto.randomUUID().length > 0);
      } catch (e) {
        this._supportsRandomUUID = false;
      }
    }
    return this._supportsRandomUUID;
  }
  //endregion
  //region Storage
  // https://developer.mozilla.org/en-US/docs/Web/API/Web_Storage_API/Using_the_Web_Storage_API
  static get storageAvailable() {
    let storage, x;
    try {
      storage = localStorage;
      x = "__storage_test__";
      storage.setItem(x, x);
      storage.removeItem(x);
      return true;
    } catch (e) {
      return e instanceof DOMException && // everything except Firefox
      (e.code === 22 || // Firefox
      e.code === 1014 || // test name field too, because code might not be present
      // everything except Firefox
      e.name === "QuotaExceededError" || // Firefox
      e.name === "NS_ERROR_DOM_QUOTA_REACHED") && // acknowledge QuotaExceededError only if there's something already stored
      storage.length !== 0;
    }
  }
  static setLocalStorageItem(key, value) {
    this.storageAvailable && localStorage.setItem(key, value);
  }
  static getLocalStorageItem(key) {
    return this.storageAvailable && localStorage.getItem(key);
  }
  static removeLocalStorageItem(key) {
    this.storageAvailable && localStorage.removeItem(key);
  }
  //endregion
  //region Helpers
  /**
   * Returns parameter value from search string by parameter name.
   * @param {String} paramName search parameter name
   * @param {String} [defaultValue] default value if parameter not found
   * @param {String} [search] search string. Defaults to `document.location.search`
   * @returns {String} search parameter string value
   * @category Helper
   */
  static searchParam(paramName, defaultValue2 = null, search = document.location.search) {
    const re = new RegExp(`[?&]${paramName}=?([^&]*)`), match = search.match(re);
    return match && match[1] || defaultValue2;
  }
  /**
   * Returns cookie by name.
   * @param {String} name cookie name
   * @returns {String} cookie string value
   * @category Helper
   */
  static getCookie(name) {
    const nameEq = encodeURIComponent(name) + "=", cookieItems = document.cookie.split(";");
    for (let i = 0; i < cookieItems.length; i++) {
      let c = cookieItems[i];
      while (c.charAt(0) === " ") {
        c = c.substring(1, c.length);
      }
      if (c.indexOf(nameEq) === 0) {
        return decodeURIComponent(c.substring(nameEq.length, c.length));
      }
    }
    return "";
  }
  /**
   * Triggers a download of a file with the specified name / URL.
   * @param {String} filename The filename of the file to be downloaded
   * @param {String} [url] The URL where the file is to be downloaded from
   * @internal
   * @category Download
   */
  static download(filename, url) {
    const a = document.createElement("a");
    a.download = filename;
    a.href = url || filename;
    a.style.cssText = "display:none";
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
  }
  /**
   * Triggers a download of a Blob with the specified name.
   * @param {Blob} blob The Blob to be downloaded
   * @param {String} filename The filename of the file to be downloaded
   * @internal
   * @category Download
   */
  static downloadBlob(blob, filename) {
    const url = globalThis.URL.createObjectURL(blob);
    this.download(filename, url);
    globalThis.URL.revokeObjectURL(url);
  }
  static get queryString() {
    var _a2;
    const params = new URL(globalThis.location.href).searchParams;
    return (_a2 = Object.fromEntries) == null ? void 0 : _a2.call(Object, params.entries());
  }
  // Used by docs fiddle
  static copyToClipboard(code) {
    let success2 = true;
    const textArea = document.createElement("textarea");
    textArea.value = code;
    textArea.style.height = textArea.style.width = 0;
    document.body.appendChild(textArea);
    textArea.select();
    try {
      document.execCommand("copy");
    } catch (e) {
      success2 = false;
    }
    textArea.remove();
    return success2;
  }
  static isBryntumOnline(searchStrings) {
    searchStrings = Array.isArray(searchStrings) ? searchStrings : [searchStrings];
    return Boolean(/^(www\.)?bryntum\.com/.test(globalThis.location.host) || (searchStrings == null ? void 0 : searchStrings.some((str) => this.queryString[str] != null)));
  }
  /**
   * Returns truthy value if page contains Content Security Policy meta tag or globalThis.bryntum.CSP is truthy value
   * @returns {Boolean}
   * @internal
   **/
  static get isCSP() {
    const { bryntum, document: document2 } = globalThis;
    if (bryntum.CSP == null) {
      bryntum.CSP = Boolean(document2.querySelector('meta[http-equiv="Content-Security-Policy"]'));
    }
    return bryntum.CSP;
  }
  //endregion
};
__publicField(BrowserHelper, "supportsPointerEvents", Boolean(globalThis.PointerEvent || globalThis.MSPointerEvent));
// Locker Service does not allow to instantiate PointerEvents. LWS apparently does, however.
// https://github.com/bryntum/support/issues/5578
__publicField(BrowserHelper, "supportsPointerEventConstructor", typeof PointerEvent !== "undefined");
__publicField(BrowserHelper, "PointerEventConstructor", globalThis.PointerEvent || globalThis.CustomEvent);
//region Init
/**
 * Yields `true` if the platform running is a phone (screen width or height <= 414 CSS pixels)
 * @property {Boolean}
 * @readonly
 * @static
 * @category Platform
 */
__publicField(BrowserHelper, "isPhone", (_a = globalThis.matchMedia) == null ? void 0 : _a.call(globalThis, "(max-height:414px) or (max-width:414px)").matches);
if (BrowserHelper.isBrowserEnv) {
  BrowserHelper.cacheFlags();
}
BrowserHelper._$name = "BrowserHelper";

// ../Core/lib/Core/helper/StringHelper.js
var charsToEncode;
var entitiesToDecode;
var htmlEncodeRe;
var htmlDecodeRe;
var camelLettersRe = /([a-z])([A-Z])/g;
var crlfRe = /[\n\r]/g;
var escapeRegExpRe = /[.*+?^${}()|[\]\\]/g;
var htmlRe = /[&<]/;
var idRe = /(^[^a-z]+[^\w]+)/gi;
var whiteSpaceRe = /\s+/;
var domIdRe = /^[^a-z]+|[^\w:.-]+/gi;
var htmlDecoder = (m, captured) => entitiesToDecode[captured.toLowerCase()] || String.fromCharCode(parseInt(captured.substr(2), 10));
var htmlEncoder = (m, captured) => charsToEncode[captured];
var hyphenateCamelLetters = (all, g1, g2) => {
  return `${g1}-${g2.toLowerCase()}`;
};
var separateCamelLetters = (all, g1, g2) => {
  return `${g1} ${g2.toLowerCase()}`;
};
var replaceNonIdChar = (c) => {
  if (c) {
    return `_x${[...c].map((ch) => ch.charCodeAt(0).toString(16)).join("")}`;
  }
  return "__blank__";
};
var hyphenateCache = {};
var separatedCache = {};
var _StringHelper = class {
  //region Transform
  /**
   * Capitalizes the first letter of a string, "myString" -> "MyString".
   * @param {String} string The string to capitalize
   * @returns {String} The capitalized string or the value of `string` if falsy.
   * @category String formatting
   */
  static capitalize(string2) {
    return string2 && string2[0].toUpperCase() + string2.substr(1);
  }
  /**
   * Makes the first letter of a string lowercase, "MyString" -> "myString".
   * @param {String} string The string to un-capitalize.
   * @returns {String} The un-capitalized string or the value of `string` if falsy.
   * @category String formatting
   */
  static uncapitalize(string2) {
    return string2 && string2[0].toLowerCase() + string2.substr(1);
  }
  /**
   * Converts the passed camelCased string to a hyphen-separated string. eg "minWidth" -> "min-width"
   * @param {String} string The string to convert.
   * @returns {String} The string with adjoining lower and upper case letters
   * separated by hyphens and converted to lower case.
   * @category String formatting
   * @internal
   */
  static hyphenate(string2) {
    const cached = hyphenateCache[string2];
    if (cached) {
      return cached;
    }
    return hyphenateCache[string2] = string2.replace(camelLettersRe, hyphenateCamelLetters);
  }
  /**
   * Converts the passed camelCased string to a capitalized, space-separated string. eg "startDate" -> "Start date".
   * @param {String} string The string to convert.
   * @returns {String} The string with spaces separating words.
   * @category String formatting
   * @internal
   */
  static separate(string2) {
    const cached = separatedCache[string2];
    if (cached) {
      return cached;
    }
    return separatedCache[string2] = this.capitalize(string2.replace(camelLettersRe, separateCamelLetters));
  }
  /**
   * Creates an alphanumeric identifier from any passed string. Encodes spaces and non-alpha characters.
   * @param {String} inString The string from which to strip non-identifier characters.
   * @returns {String}
   * @category Misc
   * @internal
   */
  static createId(inString) {
    return String(inString).replace(idRe, replaceNonIdChar);
  }
  static makeValidDomId(id, replaceValue = "") {
    if (id == null) {
      return null;
    }
    return String(id).replace(domIdRe, replaceValue);
  }
  //endregion
  //region Html
  static escapeRegExp(string2, flags) {
    let ret = string2.replace(escapeRegExpRe, "\\$&");
    if (flags !== void 0) {
      ret = new RegExp(ret, flags);
    }
    return ret;
  }
  /**
   * This method decodes HTML entities and returns the original HTML.
   *
   * See also {@link #function-encodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   */
  static decodeHtml(str) {
    return str && String(str).replace(htmlDecodeRe, htmlDecoder);
  }
  /**
   * This method encodes HTML entities and returns a string that can be placed in the document and produce the
   * original text rather than be interpreted as HTML. Using this method with user-entered values prevents those
   * values from executing as HTML (i.e., a cross-site scripting or "XSS" security issue).
   *
   * See also {@link #function-decodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   */
  static encodeHtml(str = "") {
    return str && String(str).replace(htmlEncodeRe, htmlEncoder);
  }
  /**
   * This method is similar to {@link #function-encodeHtml-static} except that `\n` and `\r` characters in the
   * given `str` are replaced by `<br>` tags _after_ first being encoded by {@link #function-encodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   * @internal
   */
  static encodeHtmlBR(str) {
    var _a2;
    return (_a2 = _StringHelper.encodeHtml(str)) == null ? void 0 : _a2.replace(crlfRe, "<br>");
  }
  /**
   * Returns `true` if the provided `text` contains special HTML characters.
   * @param {String} text
   * @returns {Boolean}
   * @category HTML
   * @internal
   */
  static isHtml(text) {
    return typeof text === "string" && htmlRe.test(text || "");
  }
  /**
   * Initializes HTML entities used by {@link #function-encodeHtml-static} and {@link #function-decodeHtml-static}.
   * @param {Object} [mappings] An object whose keys are characters that should be encoded and values are the HTML
   * entity for the character.
   * @private
   */
  static initHtmlEntities(mappings) {
    mappings = mappings || {
      "&": "&amp;",
      ">": "&gt;",
      "<": "&lt;",
      '"': "&quot;",
      "'": "&#39;"
    };
    const chars = Object.keys(mappings);
    charsToEncode = mappings;
    entitiesToDecode = chars.reduce((prev, val) => {
      prev[mappings[val]] = val;
      return prev;
    }, {});
    htmlEncodeRe = new RegExp(`([${chars.map((c) => "[-]".includes(c) ? "\\" + c : c).join("")}])`, "g");
    htmlDecodeRe = new RegExp(`(${Object.values(mappings).join("|")}|&#[0-9]+;)`, "ig");
  }
  //endregion
  //region JSON
  /**
   * Parses JSON inside a try-catch block. Returns null if the string could not be parsed.
   *
   * @param {String} string String to parse
   * @returns {Object} Resulting object or `null` if parse failed
   * @category JSON
   */
  static safeJsonParse(string2) {
    let parsed = null;
    try {
      parsed = JSON.parse(string2);
    } catch (e) {
    }
    return parsed;
  }
  /**
   * Stringifies an object inside a try-catch block. Returns null if an exception is encountered.
   *
   * See [JSON.stringify on MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify)
   * for more information on the arguments.
   *
   * @param {Object} object The object to stringify
   * @param {Function|String[]|Number[]} [replacer] A function or array of string/number used to determine properties
   * to include in the JSON string
   * @param {String|Number} [space] Number of spaces to indent or string used as whitespace
   * @returns {String} Resulting object or null if stringify failed
   * @category JSON
   */
  static safeJsonStringify(object, replacer = null, space = null) {
    let result = null;
    try {
      result = JSON.stringify(object, replacer, space);
    } catch (e) {
    }
    return result;
  }
  //endregion
  //region Split & join
  /**
   * Joins all given paths together using the separator as a delimiter and normalizes the resulting path.
   * @param paths {Array} array of paths to join
   * @param pathSeparator [{String}] path separator. Default value is '/'
   * @returns {String}
   * @category Misc
   * @internal
   */
  static joinPaths(paths, pathSeparator = "/") {
    return paths.join(pathSeparator).replace(new RegExp("\\" + pathSeparator + "+", "g"), pathSeparator);
  }
  /**
   * Returns the provided string split on whitespace. If the string is empty or consists of only whitespace, the
   * returned array will be empty. If `str` is not a string, it is simply returned. This allows `null` or already
   * split strings (arrays) to be passed through.
   *
   * For example:
   * ```
   *  console.log(StringHelper.split(' abc def xyz   '));
   *  > ['abc', 'def', 'xyz']
   *  console.log(StringHelper.split(''));
   *  > []
   * ```
   * Compare to the standard `split()` method:
   * ```
   *  console.log(' abc def xyz   '.split(/\s+/));
   *  > ['', 'abc', 'def', 'xyz', '']
   *  console.log(''.split(/\s+/));
   *  > ['']
   * ```
   * @param {String} str
   * @param {String|RegExp} delimiter
   * @returns {String[]}
   * @category Misc
   * @internal
   */
  static split(str, delimiter = whiteSpaceRe) {
    let ret = str;
    if (typeof ret === "string") {
      ret = str.trim();
      ret = ret ? ret.split(delimiter) : [];
    }
    return ret;
  }
  //endregion
  //region XSS
  /**
   * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
   * Scripting) attacks.
   *
   * For example:
   *
   * ```javascript
   *  eventRenderer(eventRecord) {
   *      return StringHelper.xss`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
   *  }
   * ```
   *
   * @param {TemplateStringsArray} strings The template string array
   * @param {...any} values The interpolated values in the template string
   * @returns {String} The encoded string
   * See {@link Core.helper.StringHelper#function-encodeHtml-static}.
   */
  static xss(strings, ...values) {
    const buf = [];
    let i = values.length;
    buf[i] = strings[i];
    while (i-- > 0) {
      buf[i] = strings[i] + _StringHelper.encodeHtml(values[i]);
    }
    return buf.join("");
  }
  /**
   * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
   * Scripting) attacks. Unlike {@link Core.helper.StringHelper#function-xss-static}, this method converts `\n` and
   * `\r` characters into `<br>` tags.
   *
   * For example:
   *
   * ```javascript
   *  eventRenderer(eventRecord) {
   *      return StringHelper.xssBR`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
   *  }
   * ```
   *
   * See {@link Core.helper.StringHelper#function-encodeHtmlBR-static}.
   * @internal
   */
  static xssBR(strings, ...values) {
    const buf = [];
    let i = values.length;
    buf[i] = strings[i];
    while (i-- > 0) {
      buf[i] = strings[i] + _StringHelper.encodeHtmlBR(values[i]);
    }
    return buf.join("");
  }
  //endregion
  //region JavaScript string
  /**
   * Converts a value to a JavaScript string (not JSON).
   *
   * For example a date to `"new Date(y, m, d)"`, an array to `"[...]"` etc.
   *
   * @param {*} value
   * @param {Object} [options]
   * @returns {String}
   * @internal
   */
  static toJavaScriptValue(value, options) {
    const type = Objects.typeOf(value);
    if (type === "boolean" || type === "string" || type === "number" || value === null) {
      return _StringHelper.safeJsonStringify(value);
    }
    if (value === globalThis) {
      return "window";
    }
    if (type === "date") {
      return `new Date(${value.getFullYear()}, ${value.getMonth()}, ${value.getDate()}, ${value.getHours()}, ${value.getMinutes()}, ${value.getSeconds()}, ${value.getMilliseconds()})`;
    }
    if (type === "array") {
      return `[${value.map((v) => _StringHelper.toJavaScriptValue(v, options))}]`;
    }
    if (type === "object" || type === "instance") {
      return this.toJavaScriptString(value, options);
    }
    if (type === "function") {
      let contents = value.toString();
      if (contents.match(/^async (\w+?)\(/)) {
        contents = contents.replace(/^async (\w+?)\(/, "async function(");
      } else if (!contents.startsWith("async(") && contents.match(/^(\w+?)\(/)) {
        contents = contents.replace(/^(\w+?)\(/, "function(");
      }
      return contents;
    }
    if (type === "class") {
      if (value.toJavaScriptValue) {
        return value.toJavaScriptValue(options);
      }
      return Object.prototype.hasOwnProperty.call(value, "$name") ? value.$name : value.name;
    }
  }
  /**
   * Converts an object into a JavaScript string (not JSON).
   *
   * For example `{ a: 1, b: [2, 3] }` -> `"'{ a: 1, b: [2, 3] }'"`
   *
   * @param {Object} obj
   * @param {Object} [options]
   * @returns {String}
   * @internal
   */
  static toJavaScriptString(obj, options = {}) {
    var _a2;
    const level = (_a2 = options.level) != null ? _a2 : 0, intendSize = 2;
    return "{\n" + Object.keys(obj).map(
      (key) => (
        // All properties in an object are indented one step further than the object itself
        " ".repeat((level + 1) * intendSize) + (key.match(/[- *]/) ? `"${key}"` : key) + `: ${_StringHelper.toJavaScriptValue(obj[key], { ...options, level: level + 1 })}`
      )
    ).join(",\n") + // Closing brace is indented to same level as the object
    "\n" + " ".repeat(level * intendSize) + "}";
  }
  /**
   * Escapes " and \ in CSS attribute selectors, e.g. [data-id="somevalue"]
   *
   * Usage:
   * ```javascript
   * document.querySelector(StringHelper.cssAttributeQuery('data-id', 'id with & \\ chars'))
   * ```
   *
   * @param {String} attr
   * @param {String|Number} value
   * @returns {String}
   */
  static encodeAttributeSelector(attr, value) {
    return `[${attr}="${String(value).replace(/["\\]/g, "\\$&")}"]`;
  }
  /**
   * Generates a UUID. Uses `Crypto.randomUUID()` if available, otherwise generates a random UUID using
   * `Crypto.getRandomValues()`.
   *
   * @returns {String}
   */
  static generateUUID() {
    var _a2;
    if (BrowserHelper.supportsRandomUUID) {
      return globalThis.crypto.randomUUID();
    }
    if ((_a2 = globalThis.crypto) == null ? void 0 : _a2.getRandomValues) {
      return ([1e7] + -1e3 + -4e3 + -8e3 + -1e11).replace(
        /[018]/g,
        (c) => (c ^ globalThis.crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16)
      );
    }
    return `${Date.now()}-${++_StringHelper.fakeNodeUUIDIndex}`;
  }
  //endregion
};
var StringHelper = _StringHelper;
//endregion
//region UUID
__publicField(StringHelper, "fakeNodeUUIDIndex", 0);
StringHelper.initHtmlEntities();
StringHelper._$name = "StringHelper";

// ../Core/lib/Core/helper/util/Objects.js
var { hasOwnProperty: hasOwnProperty2, toString } = Object.prototype;
var { isFrozen } = Object;
var afterRe = /\s*<\s*/;
var beforeRe = /\s*>\s*/;
var blendOptions = {};
var typeCache = {};
var emptyObject = Object.freeze({});
var Objects = class {
  static assign(dest, ...sources) {
    for (let source, key, i = 0; i < sources.length; i++) {
      source = sources[i];
      if (source) {
        for (key in source) {
          dest[key] = source[key];
        }
      }
    }
    return dest;
  }
  static assignIf(dest, ...sources) {
    for (let source, key, i = 0; i < sources.length; i++) {
      source = sources[i];
      if (source) {
        for (key in source) {
          if (!(key in dest) || dest[key] === void 0) {
            dest[key] = source[key];
          }
        }
      }
    }
    return dest;
  }
  static blend(dest, source, options) {
    options = options || blendOptions;
    dest = dest || {};
    const { clone = Objects.clone, merge = Objects.blend } = options;
    if (Array.isArray(source)) {
      if (source.length > 1) {
        source.forEach((s) => {
          dest = Objects.blend(dest, s, options);
        });
        return dest;
      }
      source = source[0];
    }
    if (source) {
      let destValue, key, value;
      for (key in source) {
        value = source[key];
        if (value && Objects.isObject(value)) {
          destValue = dest[key];
          options.key = key;
          if (destValue && Objects.isObject(destValue)) {
            if (isFrozen(destValue)) {
              dest[key] = destValue = clone(destValue, options);
            }
            value = merge(destValue, value, options);
          } else {
            value = isFrozen(value) ? value : clone(value, options);
          }
        }
        dest[key] = value;
      }
    }
    return dest;
  }
  static clone(value, handler) {
    let cloned = value, key;
    if (value && typeof value === "object") {
      const options = handler && typeof handler === "object" && handler;
      if (options) {
        handler = null;
      }
      if (Objects.isObject(value)) {
        if (value.skipClone) {
          cloned = value;
        } else {
          cloned = {};
          for (key in value) {
            cloned[key] = Objects.clone(value[key]);
          }
        }
      } else if (Array.isArray(value)) {
        cloned = [];
        for (key = value.length; key-- > 0; ) {
          cloned[key] = Objects.clone(value[key]);
        }
      } else if (Objects.isDate(value)) {
        cloned = new Date(value.getTime());
      } else if (handler) {
        cloned = handler(value);
      }
    }
    return cloned;
  }
  static createTruthyKeys(source) {
    const keys = StringHelper.split(source), result = keys && {};
    if (keys) {
      for (const key of keys) {
        if (key) {
          result[key] = true;
        }
      }
    }
    return result;
  }
  /**
   * Returns value for a given path in the object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {*} Value associated with passed key
   */
  static getPath(object, path) {
    return path.split(".").reduce((result, key) => {
      return (result || emptyObject)[key];
    }, object);
  }
  /**
   * Returns value for a given path in the object, placing a passed default value in at the
   * leaf property and filling in undefined properties all the way down.
   * @param {Object} object Object to get path value for.
   * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
   * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
   * @param {*} [defaultValue] Optionally the value to put in as the `someKey` property.
   * @returns {*} Value at the leaf position of the path.
   */
  static getPathDefault(object, path, defaultValue2) {
    const keys = Array.isArray(path) ? path : typeof path === "string" ? path.split(".") : [path], length = keys.length - 1;
    return keys.reduce((result, key, index) => {
      if (defaultValue2 && !(key in result)) {
        result[key] = index === length ? defaultValue2 : {};
      }
      return (result || emptyObject)[key];
    }, object);
  }
  /**
   * Determines if the specified path exists
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {Boolean}
   */
  static hasPath(object, path) {
    return path.split(".").every((key) => {
      if (object && key in object) {
        object = object[key];
        return true;
      }
      return false;
    });
  }
  static getTruthyKeys(source) {
    const keys = [];
    for (const key in source) {
      if (source[key]) {
        keys.push(key);
      }
    }
    return keys;
  }
  static getTruthyValues(source) {
    const values = [];
    for (const key in source) {
      if (source[key]) {
        values.push(source[key]);
      }
    }
    return values;
  }
  static isClass(object) {
    var _a2;
    if (typeof object === "function" && ((_a2 = object.prototype) == null ? void 0 : _a2.constructor) === object) {
      return true;
    }
    return false;
  }
  static isDate(object) {
    return Boolean(object == null ? void 0 : object.getUTCDate) && Objects.typeOf(object) === "date";
  }
  /**
   * Check if passed object is a Promise or contains `then` method.
   * Used to fix problems with detecting promises in code with `instance of Promise` when
   * Promise class is replaced with any other implementation like `ZoneAwarePromise` in Angular.
   * Related to these issues:
   * https://github.com/bryntum/support/issues/791
   * https://github.com/bryntum/support/issues/2990
   *
   * @param {Object} object object to check
   * @returns {Boolean} truthy value if object is a Promise
   * @internal
   */
  static isPromise(object) {
    if (Promise && Promise.resolve) {
      return Promise.resolve(object) === object || typeof (object == null ? void 0 : object.then) === "function";
    }
    throw new Error("Promise not supported in your environment");
  }
  static isEmpty(object) {
    if (object && typeof object === "object") {
      for (const p in object) {
        return false;
      }
    }
    return true;
  }
  static isObject(value) {
    const C = value == null ? void 0 : value.constructor;
    return Boolean(
      C ? (
        // An in-frame instance of Object
        C === Object || // Detect cross-frame objects, but exclude instance of custom classes named Object. typeOf(value) is
        // "object" even for instances of a class and typeOf(C) is "function" for all constructors. We'll have
        // to settle for relying on the fact that getPrototypeOf(Object.prototype) === null.
        // NOTE: this issue does come up in Scheduler unit tests at least.
        C.getPrototypeOf && C.prototype && !Object.getPrototypeOf(C.prototype)
      ) : value && typeof value === "object"
    );
  }
  static isInstantiated(object) {
    return object ? typeof object === "object" && !Objects.isObject(object) : false;
  }
  static merge(dest, ...sources) {
    return Objects.blend(dest, sources);
  }
  /**
   * Merges two "items" objects. An items object is a simple object whose keys act as identifiers and whose values
   * are "item" objects. An item can be any object type. This method is used to merge such objects while maintaining
   * their property order. Special key syntax is used to allow a source object to insert a key before or after a key
   * in the `dest` object.
   *
   * For example:
   * ```javascript
   *  let dest = {
   *      foo : {},
   *      bar : {},
   *      fiz : {}
   *  }
   *
   *  console.log(Object.keys(dest));
   *  > ["foo", "bar", "fiz"]
   *
   *  dest = mergeItems(dest, {
   *      'zip > bar' : {}    // insert "zip" before "bar"
   *      'bar < zap' : {}    // insert "zap" after "bar"
   *  });
   *
   *  console.log(Object.keys(dest));
   *  > ["foo", "zip", "bar", "zap", "fiz"]
   * ```
   *
   * @param {Object} dest The destination object.
   * @param {Object|Object[]} src The source object or array of source objects to merge into `dest`.
   * @param {Object} [options] The function to use to merge items.
   * @param {Function} [options.merge] The function to use to merge items.
   * @returns {Object} The merged object. This will be the `dest` object.
   * @internal
   */
  static mergeItems(dest, src, options) {
    options = options || blendOptions;
    let anchor, delta, index, indexMap, key, shuffle, srcVal;
    const { merge = Objects.blend } = options;
    dest = dest || {};
    if (Array.isArray(src)) {
      src.forEach((s) => {
        dest = Objects.mergeItems(dest, s, options);
      });
    } else if (src) {
      for (key in src) {
        srcVal = src[key];
        anchor = null;
        if (key.includes(">")) {
          [key, anchor] = key.split(beforeRe);
          delta = 0;
        } else if (key.includes("<")) {
          [anchor, key] = key.split(afterRe);
          delta = 1;
        }
        if (key in dest) {
          if (srcVal && dest[key] && merge) {
            options.key = key;
            srcVal = merge(dest[key], srcVal, options);
          }
          dest[key] = srcVal;
        } else if (!anchor) {
          dest[key] = srcVal;
          indexMap == null ? void 0 : indexMap.set(key, indexMap.size);
        } else {
          if (!indexMap) {
            indexMap = /* @__PURE__ */ new Map();
            index = 0;
            for (const k in dest) {
              indexMap.set(k, index++);
            }
          }
          index = indexMap.get(anchor);
          dest[key] = srcVal;
          if (index == null && delta) {
            index = indexMap.size;
          } else {
            shuffle = shuffle || [];
            index = (index || 0) + delta;
            for (const item of indexMap) {
              const [k, v] = item;
              if (index <= v) {
                shuffle && (shuffle[indexMap.size - v - 1] = k);
                indexMap.set(k, v + 1);
              }
            }
            if (shuffle) {
              while (shuffle.length) {
                const k = shuffle.pop(), v = dest[k];
                delete dest[k];
                dest[k] = v;
              }
            }
          }
          indexMap.set(key, index);
        }
      }
    }
    return dest;
  }
  /**
   * Sets value for a given path in the object
   * @param {Object} object Target object
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @param {*} value Value for a given path
   * @returns {Object} Returns passed object
   */
  static setPath(object, path, value) {
    path.split(".").reduce((result, key, index, array) => {
      const isLast = index === array.length - 1;
      if (isLast) {
        return result[key] = value;
      } else if (!(result[key] instanceof Object)) {
        result[key] = {};
      }
      return result[key];
    }, object);
    return object;
  }
  static typeOf(value) {
    let trueType, type;
    if (value === null) {
      type = "null";
    } else if (value !== value) {
      type = "nan";
    } else {
      type = typeof value;
      if (type === "object") {
        if (value.isBase) {
          type = "instance";
        } else if (Array.isArray(value)) {
          type = "array";
        } else if (!(type = typeCache[trueType = toString.call(value)])) {
          typeCache[trueType] = type = trueType.slice(8, -1).toLowerCase();
        }
      } else if (type === "function" && value.isBase) {
        type = "class";
      }
    }
    return type;
  }
};
Object.defineProperty(Objects, "hasOwn", {
  // When available, this avoids an extra layer of function call around it:
  value: Object.hasOwn || ((object, property) => hasOwnProperty2.call(object, property))
});
Objects._$name = "Objects";

// ../Core/lib/Core/helper/VersionHelper.js
var VersionHelper = class {
  /**
   * Set version for specified product
   * @private
   * @param {String} product
   * @param {String} version
   */
  static setVersion(product, version) {
    product = product.toLowerCase();
    VH[product] = {
      version,
      isNewerThan(otherVersion) {
        return VersionHelper.semanticCompareVersion(otherVersion, version, "<");
      },
      isOlderThan(otherVersion) {
        return VersionHelper.semanticCompareVersion(otherVersion, version, ">");
      }
    };
    let bundleFor = "";
    if (typeof productName !== "undefined") {
      bundleFor = productName;
    }
    const globalKey = `${bundleFor}.${product}${version.replace(/\./g, "-")}`;
    if (BrowserHelper.isBrowserEnv && !globalThis.bryntum.silenceBundleException) {
      if (globalThis.bryntum[globalKey] === true) {
        if (this.isTestEnv) {
          globalThis.BUNDLE_EXCEPTION = true;
        } else {
          let errorProduct = bundleFor || product;
          if (errorProduct === "core") {
            errorProduct = "grid";
          }
          let capitalized = StringHelper.capitalize(errorProduct);
          if (errorProduct === "schedulerpro") {
            capitalized = "SchedulerPro";
          }
          throw new Error(
            `The Bryntum ${capitalized} bundle was loaded multiple times by the application.

Common reasons you are getting this error includes:

* Imports point to different types of the bundle (e.g. *.module.js and *.umd.js)
* Imports point to both sources and bundle
* Imports do not use the shortest relative path, JS treats them as different files
* Cache busters differ between imports, JS treats ${errorProduct}.module.js?1 and ${errorProduct}.module.js?2 as different files
* Imports missing file type, verify they all end in .js

See https://bryntum.com/products/${errorProduct}/docs/guide/${capitalized}/gettingstarted/es6bundle#troubleshooting for more information

`
          );
        }
      } else {
        globalThis.bryntum[globalKey] = true;
      }
    }
  }
  /**
   * Get (previously set) version for specified product
   * @private
   * @param {String} product
   */
  static getVersion(product) {
    product = product.toLowerCase();
    if (!VH[product]) {
      throw new Error("No version specified! Please check that you import VersionHelper correctly into the class from where you call `deprecate` function.");
    }
    return VH[product].version;
  }
  /**
   * Checks the version1 against the passed version2 using the comparison operator.
   * Supports `rc`, `beta`, `alpha` release states. Eg. `1.2.3-alpha-1`.
   * State which is not listed above means some version below `alpha`.
   * @param {String} version1 The version to test against
   * @param {String} version2 The version to test against
   * @param {String} [comparison] The comparison operator, `<=`, `<`, `=`, `>` or `>=`.
   * @returns {Boolean} `true` if the test passes.
   * @internal
   */
  static semanticCompareVersion(version1, version2, comparison = "=") {
    version1 = version1 || "";
    version2 = version2 || "";
    const version1Arr = version1.split(/[-.]/), version2Arr = version2.split(/[-.]/), isLower = comparison.includes("<"), normalizeArr = (arr, maxLength) => {
      const states = ["rc", "beta", "alpha"], result = arr.map((v) => {
        if (states.includes(v)) {
          return -states.indexOf(v) - 2;
        }
        const res = Number.parseInt(v);
        return Number.isNaN(res) ? -states.length : res;
      });
      while (result.length < maxLength) {
        result.push(-1);
      }
      return result;
    }, compareArr = () => {
      const maxLength = Math.max(version1Arr.length, version2Arr.length), arr1 = normalizeArr(version1Arr, maxLength), arr2 = normalizeArr(version2Arr, maxLength);
      for (let i = 0; i < maxLength; i++) {
        if (arr1[i] !== arr2[i]) {
          return isLower ? arr1[i] < arr2[i] : arr1[i] > arr2[i];
        }
      }
      return true;
    };
    switch (comparison) {
      case "=":
        return version1 === version2;
      case "<=":
      case ">=":
        return version1 === version2 || compareArr();
      case "<":
      case ">":
        return version1 !== version2 && compareArr();
    }
    return false;
  }
  /**
   * Checks the passed product against the passed version using the passed test.
   * @param {String} product The name of the product to test the version of
   * @param {String} version The version to test against
   * @param {String} operator The test operator, `<=`, `<`, `=`, `>` or `>=`.
   * @returns {Boolean} `true` if the test passes.
   * @internal
   */
  static checkVersion(product, version, operator) {
    return VersionHelper.semanticCompareVersion(VH.getVersion(product), version, operator);
  }
  /**
   * Based on a comparison of current product version and the passed version this method either outputs a console.warn
   * or throws an error.
   * @param {String} product The name of the product
   * @param {String} invalidAsOfVersion The version where the offending code is invalid (when any compatibility layer
   * is actually removed).
   * @param {String} message Required! A helpful warning message to show to the developer using a deprecated API.
   * @internal
   */
  static deprecate(product, invalidAsOfVersion, message) {
    const justWarn = VH.checkVersion(product, invalidAsOfVersion, "<");
    if (justWarn) {
      console.warn(`Deprecation warning: You are using a deprecated API which will change in v${invalidAsOfVersion}. ${message}`);
    } else {
      throw new Error(`Deprecated API use. ${message}`);
    }
  }
  /**
   * Returns truthy value if environment is in testing mode
   * @returns {Boolean}
   * @internal
   **/
  static get isTestEnv() {
    var _a2, _b, _c;
    const isTestEnv = Boolean((_a2 = globalThis.bryntum) == null ? void 0 : _a2.isTestEnv);
    try {
      return isTestEnv || Boolean((_c = (_b = globalThis.parent) == null ? void 0 : _b.bryntum) == null ? void 0 : _c.isTestEnv);
    } catch (e) {
      return isTestEnv;
    }
  }
  static get isDebug() {
    let result = false;
    return result;
  }
};
var VH = VersionHelper;
if (BrowserHelper.isBrowserEnv) {
  if (VH.isTestEnv) {
    BrowserHelper._isHoverableDevice = true;
  }
  globalThis.bryntum = Object.assign(globalThis.bryntum || {}, {
    getVersion: VH.getVersion.bind(VH),
    checkVersion: VH.checkVersion.bind(VH),
    deprecate: VH.deprecate.bind(VH),
    license: "9c07f388-a9e9-11ec-9e5c-d094663d5c88"
  });
}
VersionHelper._$name = "VersionHelper";

// ../Core/lib/Core/Config.js
var { defineProperty, getOwnPropertyDescriptor } = Reflect;
var { hasOwnProperty: hasOwnProperty3, toString: toString2 } = Object.prototype;
var instancePropertiesSymbol = Symbol("instanceProperties");
var configuringSymbol = Symbol("configuring");
var lazyConfigValues = Symbol("lazyConfigValues");
var DATE_TYPE = toString2.call(/* @__PURE__ */ new Date());
var whitespace = /\s+/;
var createClsProps = (result, cls2) => {
  result[cls2] = 1;
  return result;
};
var Config = class {
  /**
   * Returns the `Config` instance for the given `name` and `options`.
   * @param {String} name The name of the config (e.g., 'text' for the text config).
   * @param {Object} [options] Config behavior options.
   * @returns {Core.Config}
   * @internal
   */
  static get(name, options) {
    const { cache } = this, baseCfg = cache[name] || (cache[name] = new Config(name));
    let cfg = baseCfg, key;
    if (options) {
      key = Config.makeCacheKey(name, options);
      if (!(cfg = key && cache[key])) {
        cfg = baseCfg.extend(options);
        if (key) {
          cache[key] = cfg;
        }
      }
    }
    return cfg;
  }
  constructor(name) {
    const me = this, cap = name[0].toUpperCase() + name.substr(1);
    me.base = me;
    me.name = name;
    me.field = "_" + name;
    me.capName = cap;
    me.changer = "change" + cap;
    me.initializing = "initializing" + cap;
    me.updater = "update" + cap;
  }
  /**
   * The descriptor to use with `Reflect.defineProperty()` for defining this config's getter and setter.
   * @property {Object}
   * @private
   */
  get descriptor() {
    let descriptor = this._descriptor;
    if (!descriptor || !hasOwnProperty3.call(this, "_descriptor")) {
      this._descriptor = descriptor = this.makeDescriptor();
    }
    return descriptor;
  }
  /**
   * The descriptor to use with `Reflect.defineProperty()` for defining this config's initter.
   * @property {Object}
   * @private
   */
  get initDescriptor() {
    let descriptor = this._initDescriptor;
    if (!descriptor || !hasOwnProperty3.call(this, "_initDescriptor")) {
      this._initDescriptor = descriptor = this.makeInitter();
    }
    return descriptor;
  }
  /**
   * This method compares two values for semantic equality. By default, this is based on the `===` operator. This
   * is often overridden for configs that accept `Date` or array values.
   * @param {*} value1
   * @param {*} value2
   * @returns {Boolean}
   * @internal
   */
  equal(value1, value2) {
    return value1 === value2;
  }
  /**
   * Extends this config with a given additional set of options. These objects are just prototype extensions of this
   * instance.
   * @param {Object} options
   * @returns {Core.Config}
   * @internal
   */
  extend(options) {
    const cfg = Object.assign(Object.create(this), options), { equal: equal2, merge } = options, { equalityMethods } = Config;
    if (typeof equal2 === "string") {
      if (equal2.endsWith("[]")) {
        cfg.equal = Config.makeArrayEquals(equalityMethods[equal2.substr(0, equal2.length - 2)]);
      } else {
        cfg.equal = equalityMethods[equal2];
      }
    }
    if (typeof merge === "string") {
      cfg.merge = Config.mergeMethods[merge];
    }
    return cfg;
  }
  /**
   * Defines the property on a given target object via `Reflect.defineProperty()`. If the object has its own getter,
   * it will be preserved. It is invalid to define a setter.
   * @param {Object} target
   * @internal
   */
  define(target) {
    const existing = getOwnPropertyDescriptor(target, this.name);
    let descriptor = this.descriptor;
    if (existing && existing.get) {
      descriptor = Object.assign({}, descriptor);
      descriptor.get = existing.get;
    }
    defineProperty(target, this.name, descriptor);
  }
  /**
   * Defines the property initter on the `target`. This is a property getter/setter that propagates the configured
   * value when the property is read.
   * @param {Object} target
   * @param {*} value
   * @internal
   */
  defineInitter(target, value) {
    const { name } = this, properties = target[instancePropertiesSymbol];
    let lazyValues, prop;
    if (!properties[name] && /* assign */
    (prop = getOwnPropertyDescriptor(target, name)) && !("value" in prop)) {
      properties[name] = prop;
    }
    defineProperty(target, name, this.initDescriptor);
    if (this.lazy) {
      lazyValues = target[lazyConfigValues] || (target[lazyConfigValues] = /* @__PURE__ */ new Map());
      lazyValues.set(name, value);
    }
  }
  /**
   * Returns an equality function for arrays of a base type, for example `'date'`.
   * @param {Function} [fn] The function to use to compare array elements. By default, operator `===` is used.
   * @returns {Function}
   * @private
   */
  static makeArrayEquals(fn2) {
    return (value1, value2) => {
      let i, equal2 = value1 && value2 && value1.length === (i = value2.length);
      if (equal2 && Array.isArray(value1) && Array.isArray(value2)) {
        if (fn2) {
          while (equal2 && i-- > 0) {
            equal2 = fn2(value1[i], value2[i]);
          }
        } else {
          while (equal2 && i-- > 0) {
            equal2 = value1[i] === value2[i];
          }
        }
      } else {
        equal2 = fn2 ? fn2(value1, value2) : value1 === value2;
      }
      return equal2;
    };
  }
  /**
   * Returns the key to use in the Config `cache`.
   * @param {String} name The name of the config property.
   * @param {Object} options The config property options.
   * @returns {String}
   * @private
   */
  static makeCacheKey(name, options) {
    const keys = Object.keys(options).sort();
    for (let key, type, value, i = keys.length; i-- > 0; ) {
      value = options[key = keys[i]];
      if (value == null && value === false) {
        keys.splice(i, 1);
      } else {
        type = typeof value;
        if (type === "function") {
          return null;
        }
        if (type === "string") {
          keys[i] = `${key}:"${value}"`;
        } else if (type === "number") {
          keys[i] = `${key}:${value}`;
        }
      }
    }
    return keys.length ? `${name}>${keys.join("|")}` : name;
  }
  /**
   * Creates and returns a property descriptor for this config suitable to be passed to `Reflect.defineProperty()`.
   * @returns {Object}
   * @private
   */
  makeDescriptor() {
    const config = this, { base, field: field2, changer, updater, name } = config;
    if (base !== config && base.equal === config.equal) {
      return base.descriptor;
    }
    return {
      get() {
        var _a2;
        (_a2 = this.configObserver) == null ? void 0 : _a2.get(name, this);
        return this[field2];
      },
      set(value) {
        var _a2, _b;
        const me = this;
        let was = me[field2], applied, newValue;
        if (typeof value === "string") {
          let resolvedValue = value;
          if (value.startsWith("up.")) {
            resolvedValue = (_a2 = me.owner) == null ? void 0 : _a2.resolveProperty(value.substr(3));
          } else if (value.startsWith("this.")) {
            resolvedValue = me.resolveProperty(value.substr(5));
          }
          if (resolvedValue !== void 0 && typeof resolvedValue !== "function") {
            value = resolvedValue;
          }
        }
        if (me[changer]) {
          applied = (newValue = me[changer](value, was)) === void 0;
          if (!applied) {
            value = newValue;
            was = me[field2];
          }
        }
        if (!applied && !(config.equal === equal ? was === value : config.equal(was, value))) {
          me[field2] = value;
          applied = true;
          (_b = me[updater]) == null ? void 0 : _b.call(me, value, was);
        }
        if (applied && !me.isDestroyed && !me.onConfigChange.$nullFn) {
          me.onConfigChange({ name, value, was, config });
        }
      }
    };
  }
  /**
   * Creates and returns a property descriptor for this config's initter suitable to pass to
   * `Reflect.defineProperty()`.
   * @returns {Object}
   * @private
   */
  makeInitter() {
    const config = this;
    if (config !== config.base) {
      if (config.lazy) {
        return config.makeLazyInitter();
      }
      return config.base.initDescriptor;
    }
    return config.makeBasicInitter();
  }
  makeBasicInitter() {
    const config = this, { initializing, name } = config;
    return {
      configurable: true,
      get() {
        const me = this;
        config.removeInitter(me);
        me[initializing] = true;
        me[name] = me[configuringSymbol][name];
        me[initializing] = false;
        me.configDone[name] = true;
        return me[name];
      },
      set(value) {
        config.removeInitter(this);
        this.configDone[name] = true;
        this[name] = value;
      }
    };
  }
  makeLazyInitter() {
    const config = this, { initializing, name } = config;
    return {
      configurable: true,
      get() {
        const me = this, value = me[lazyConfigValues].get(name);
        config.removeInitter(me);
        if (!me.isDestroying) {
          me[initializing] = true;
          me[name] = value;
          me[initializing] = false;
        }
        return me[name];
      },
      set(value) {
        config.removeInitter(this);
        this[name] = value;
      }
    };
  }
  /**
   * Removes the property initter and restores the instance to its original form.
   * @param {Object} instance
   * @private
   */
  removeInitter(instance) {
    const { name } = this, instanceProperty = instance[instancePropertiesSymbol][name], lazyValues = instance[lazyConfigValues];
    if (instanceProperty) {
      defineProperty(instance, name, instanceProperty);
    } else {
      delete instance[name];
    }
    if ((lazyValues == null ? void 0 : lazyValues.delete(name)) && !lazyValues.size) {
      delete instance[lazyConfigValues];
    }
  }
  setDefault(cls2, value) {
    defineProperty(cls2.prototype, this.field, {
      configurable: true,
      writable: true,
      // or else "this._value = x" will fail
      value
    });
  }
  /**
   * This method combines (merges) two config values. This is called in two cases:
   *
   *  - When a derived class specifies the value of a config defined in a super class.
   *  - When a value is specified in the instance config object.
   *
   * @param {*} newValue In the case of derived classes, this is the config value of the derived class. In the case
   * of the instance config, this is the instance config value.
   * @param {*} currentValue In the case of derived classes, this is the config value of the super class. In the case
   * of the instance config, this is the class config value.
   * @param {Object} metaNew The class meta object from which the `newValue` is coming. This parameter is `null` if
   * the `newValue` is from an instance configuration.
   * @param {Object} metaCurrent The class meta object from which the `currentValue` is coming. This parameter is
   * `null` if the `currentValue` is not from a class configuration.
   * @returns {*}
   * @internal
   */
  merge(newValue, currentValue) {
    if (currentValue && newValue && Objects.isObject(newValue)) {
      if (currentValue.isBase) {
        return currentValue.setConfig(newValue);
      }
      if (Objects.isObject(currentValue)) {
        newValue = Objects.merge(Objects.clone(currentValue), newValue);
      }
    }
    return newValue;
  }
};
var { prototype } = Config;
var { equal } = prototype;
Config.symbols = {
  configuring: configuringSymbol,
  instanceProperties: instancePropertiesSymbol,
  lazyConfigs: lazyConfigValues
};
Config.cache = /* @__PURE__ */ Object.create(null);
Config.equalityMethods = {
  array: Config.makeArrayEquals(),
  date(value1, value2) {
    if (value1 === value2) {
      return true;
    }
    if (value1 && value2 && toString2.call(value1) === DATE_TYPE && toString2.call(value2) === DATE_TYPE) {
      return value1.getTime() === value2.getTime();
    }
    return false;
  },
  strict: Config.equal = equal
};
Config.mergeMethods = {
  distinct(newValue, oldValue) {
    let ret = oldValue ? oldValue.slice() : [];
    if (newValue != null) {
      if (Objects.isObject(newValue)) {
        if (oldValue === void 0) {
          ret = newValue;
        } else {
          let key, index;
          for (key in newValue) {
            index = ret.indexOf(key);
            if (newValue[key]) {
              if (index < 0) {
                ret.push(key);
              }
            } else if (index > -1) {
              ret.splice(index, 1);
            }
          }
        }
      } else if (Array.isArray(newValue)) {
        newValue.forEach((v) => !ret.includes(v) && ret.push(v));
      } else if (!ret.includes(newValue)) {
        ret.push(newValue);
      }
    }
    return ret;
  },
  merge: Config.merge = prototype.merge,
  classList(newValue, oldValue) {
    if (typeof newValue === "string") {
      if (!newValue.length) {
        return oldValue;
      }
      newValue = newValue.split(whitespace);
    }
    if (Array.isArray(newValue)) {
      newValue = newValue.reduce(createClsProps, {});
    }
    return Config.merge(newValue, oldValue);
  },
  objects(newValue, oldValue) {
    return newValue === true ? oldValue || {} : Config.merge(newValue, oldValue);
  },
  replace(newValue) {
    return newValue;
  },
  items(newValue, oldValue, metaNew, metaCurrent) {
    if (metaCurrent) {
      return Objects.mergeItems(oldValue, newValue, {
        merge: (oldValue2, newValue2) => prototype.merge(newValue2, oldValue2)
      });
    }
    return prototype.merge(newValue, oldValue);
  }
};
Object.assign(prototype, {
  _descriptor: null,
  _initDescriptor: null,
  /**
   * A function that compares values for equality. This test is used to determine if the `update` method should be
   * called when the setter is invoked.
   *
   * To handle `Date` values:
   * ```
   *  class Foo extends Base {
   *      static get configurable() {
   *          return {
   *              date : {
   *                  $config : {
   *                      equal : 'date'
   *                  },
   *
   *                  value : null
   *              }
   *          }
   *      }
   *
   *      updateDate(date) {
   *          // date has changed
   *      }
   *  }
   * ```
   *
   * Also useful for some configs:
   * ```
   *  class Foo extends Base {
   *      static get configurable() {
   *          return {
   *              bar : {
   *                  $config : {
   *                      equal : ObjectHelper.isEqual
   *                  },
   *
   *                  value : null
   *              }
   *          }
   *      }
   *
   *      updateBar(value) {
   *          // value has changed
   *      }
   *  }
   * ```
   * @config {Function} equal
   * @internal
   */
  /**
   * Indicates that this config property should not automatically initialize during construction. When this property
   * is set to `true`, initialization is triggered by the first use of the config property's getter.
   *
   * This property can alternatively be set to a string, in which case it can be initialized as a group using the
   * {@link Core.Base#function-triggerConfigs} method which will initialize all lazy configs with the same value for
   * this property. Note: the config will still initialize on first use if that occurs prior to the call to
   * `triggerConfigs`.
   * @config {Boolean|String}
   * @default
   * @internal
   */
  lazy: false,
  /**
   * Indicates that this config property should automatically be set to `null` on destroy.
   * @config {Boolean}
   * @default
   * @internal
   */
  nullify: false,
  /**
   * Indicates that this config participates in rendering. This has does not affect the behavior of the property
   * directly, but allows classes that perform rendering to detect which config changes will affect the rendered
   * result.
   * @config {Boolean}
   * @default
   * @internal
   */
  render: false
});
Config._$name = "Config";

// ../Core/lib/Core/Base.js
var MetaClass = class {
  constructor(options) {
    options && Object.assign(this, options);
  }
  getInherited(name, create = true) {
    var _a2;
    let ret = this[name];
    if (!(name in this)) {
      ret = (_a2 = this.super) == null ? void 0 : _a2.getInherited(name, create);
      if (ret || create) {
        this[name] = ret = Object.create(ret || null);
      }
    }
    return ret;
  }
};
var { getPrototypeOf } = Object;
var { hasOwn } = Objects;
var { defineProperty: defineProperty2 } = Reflect;
var metaSymbol = Symbol("classMetaData");
var mixinTagSymbol = Symbol("mixinTag");
var originalConfigSymbol = Symbol("originalConfig");
var createdAtSymbol = Symbol("createdAt");
var configuringSymbol2 = Config.symbols.configuring;
var instancePropertiesSymbol2 = Config.symbols.instanceProperties;
var lazyConfigsSymbol = Config.symbols.lazyConfigs;
var defaultConfigOptions = { merge: "replace", simple: true };
var emptyFn = () => {
};
var newMeta = (o) => new MetaClass(o);
var setupNames = {
  /* foo : 'setupFoo' */
};
var emptyObject2 = Object.freeze({});
var emptyArray = Object.freeze([]);
var Base = class {
  static get isBase() {
    return true;
  }
  get isBase() {
    return true;
  }
  // defaultConfig & properties made private to not spam all other classes
  /**
   * A class property getter to add additional, special class properties.
   *
   * For example, a class adds a `declarable` class property like so:
   * ```
   *  class Something extends Base {
   *      static get declarable() {
   *          return ['extra'];
   *      }
   *
   *      static setupExtra(cls, meta) {
   *          // use cls.extra
   *      }
   *  }
   * ```
   * A derived class can then specify this property like so:
   * ```
   *  class Derived extends Something {
   *      static get extra() {
   *          // return extra information
   *      }
   *  }
   * ```
   * When the `Derived` class is initialized, the `setupExtra()` method is called and `Derived` is passed as the
   * argument. It is also the `this` pointer, but the parameter is minifiable. The second argument passed is the
   * `$meta` object for the class.
   *
   * Classes are initialized at the first occurrence of the following:
   *
   * - An instance is created
   * - The class `$meta` property is accessed
   *
   * @member {String[]} declarable
   * @static
   * @category Configuration
   * @internal
   */
  static get declarable() {
    return [
      "declarable",
      /**
       * A class property getter for the configuration properties of the class, which can be overridden by
       * configurations passed at construction time.
       *
       * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
       * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
       * individually and then combine them with those of derived classes.
       *
       * For example, a `Label` might declare a `text` config like so:
       * ```javascript
       *  class Label extends Base {
       *      static get configurable() {
       *          return {
       *              text : null
       *          };
       *      }
       *  }
       * ```
       * The `text` config is automatically inherited by classes derived from Label. By implementing
       * `get configurable()`, derived classes can change the default value of inherited configs, or define new
       * configs, or both.
       *
       * When a config property is declared in this way, the class author can also implement either of two
       * special methods that will be called when the config property is assigned a new value:
       *
       *  - `changeText()`
       *  - `updateText()`
       *
       * In the example above, the `Label` class could implement a `changeText()` method, an `updateText()`
       * method, or both. The generated property setter ensures these methods will be called when the `text`
       * property is assigned.
       *
       * The generated setter (for `text` in this example) performs the following steps:
       *
       *  - If the class defines a `changeText()` method, call it passing the new value and the current value:
       *    `changeText(newText, oldText)`.<br>
       *    Then:
       *    * If `changeText()` exits without returning a value (i.e., `undefined`), exit and do nothing
       *      further. The assumption is that the changer method has done all that is required.
       *    * Otherwise, the return value of `changeText()` replaces the incoming value passed to the setter.
       *  - If the new value (or the value returned by `changeText()`) is `!==` to the current value:
       *    * Update the stored config value in `this._text`.
       *    * If the class defines an `updateText()` method, call it passing the new value and the previous value.
       *      `updateText(newText, oldText)`
       *
       * #### Resolving a value from an owner
       * By specifying a value starting with `'up.'` for a config, the config system will resolve that value by
       * examining the ownership hierarchy. It will walk up the hierarchy looking for a property matching the name
       * (or dot separated path) after 'up.'. If one is found, the value will be read and used as the initial
       * value.
       *
       * ```javascript
       * class Parent extends Base {
       *     static get configurable() {
       *         return [
       *           'importantValue'
       *         ]
       *     }
       * }
       *
       * class Child extends Base {
       *     static get configurable() {
       *         return [
       *           'value'
       *         ]
       *     }
       * }
       *
       * const parent = new Parent({
       *     importantValue : 123
       * });
       *
       * const child = new Child({
       *     owner : parent,
       *     // Will be resolved from the owner
       *     value : 'up.importantValue'
       * });
       *
       * console.log(child.value); // logs 123
       * ```
       * Please note that this is for now a one way one time binding, the value will only be read initially and
       * not kept up to date on later changes.
       *
       * #### Value Merging
       * When a config property value is an object, the value declared by the base class is merged with values
       * declared by derived classes and the value passed to the constructor.
       * ```javascript
       *  class Example extends Base {
       *      static get configurable() {
       *          return {
       *              config : {
       *                  foo : 1,
       *                  bar : 2
       *              }
       *          };
       *      }
       *  }
       *
       *  class Example2 extends Example {
       *      static get configurable() {
       *          return {
       *              config : {
       *                  bar : 42,
       *                  zip : 'abc'
       *              }
       *          };
       *      }
       *  }
       *
       *  let ex = new Example2({
       *      config : {
       *          zip : 'xyz'
       *      }
       *  });
       * ```
       * The result of the merge would set `config` to:
       * ```javascript
       *  ex.foo = {
       *      foo : 1,    // from Example
       *      bar : 42,   // from Example2
       *      zip : 'xyz' // from constructor
       *  }
       * ```
       *
       * #### Config Options
       * Some config properties require additional options such as declarative information about the config that
       * may be useful to automate some operation. Consider a `Button`. It could declare that its `text` config
       * affects the rendered HTML by applying a `render` property to the config definition. Its base class could
       * then examine the config definition to find this property.
       *
       * To support this, config options ca be declared like so:
       * ```javascript
       *  class Button extends Widget {
       *      static get configurable() {
       *          return {
       *              text : {
       *                  value   : null,
       *                  $config : {
       *                      render : true
       *                  }
       *              }
       *          };
       *      }
       *  }
       * ```
       * The `$config` property can alternatively be just the names of the options that should be enabled (set
       * to `true`).
       *
       * For example, the following is equivalent to the above:
       * ```javascript
       *  class Button extends Widget {
       *      static get configurable() {
       *          return {
       *              text : {
       *                  value   : null,
       *                  $config : 'render'
       *              }
       *          };
       *  }
       * ```
       *
       * #### Default Value
       * It is common to set a config to a `null` value to take advantage of internal optimizations for `null`
       * values. In most cases the fact that this produces `undefined` as the actual initial value of the config
       * is acceptable. When this is not acceptable, a config can be declared like so:
       * ```javascript
       *  class Widget {
       *      static get configurable() {
       *          return {
       *              disabled : {
       *                  $config : null,
       *                  value   : null,
       *                  default : false
       *              }
       *          };
       *  }
       * ```
       * The `default` property above determines the value of the config while still gaining the benefits of
       * minimal processing due to the `null` value of the `value` property.
       * @member {Object} configurable
       * @static
       * @category Configuration
       * @internal
       */
      "configurable",
      /**
       * A class property getter for the default configuration of the class, which can be overridden by
       * configurations passed at construction time.
       *
       * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
       * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
       * individually and then combine them with those of derived classes.
       *
       * For example, a `Store` might declare its `url` config like so:
       * ```
       *  class Store extends Base {
       *      static get defaultConfig() {
       *          return {
       *              url : null
       *          };
       *      }
       *  }
       * ```
       * The `url` config is automatically inherited by classes derived from Store. By implementing
       * `get defaultConfig()`, derived classes can change the default value of inherited configs, or define new
       * configs, or both. When defining new configs, however, `configurable` is preferred.
       *
       * Config properties introduced to a class by this declaration do not participate in value merging and do
       * not get a generated setter. Config properties introduced by a base class using `configurable` can be
       * set to a different value using `defaultConfig` and in doing so, the values will be merged as appropriate
       * for `configurable`.
       *
       * @member {Object} defaultConfig
       * @static
       * @category Configuration
       * @internal
       */
      "defaultConfig",
      /**
       * A class property getter for the default values of internal properties for this class.
       * @member {Object} properties
       * @static
       * @category Configuration
       * @internal
       */
      "properties",
      /**
       * A class property getter for properties that will be applied to the class prototype.
       * @member {Object} prototypeProperties
       * @static
       * @category Configuration
       * @internal
       */
      "prototypeProperties"
    ];
  }
  /**
   * Base constructor, passes arguments to {@link #function-construct}.
   * @param {...Object} [args] Usually called with a config object, but accepts any params
   * @function constructor
   * @category Lifecycle
   * @advanced
   */
  constructor(...args) {
    const me = this, C = me.constructor;
    if (me.$meta.class !== C) {
      emptyFn(C.$meta);
    }
    me.construct(...args);
    me.afterConstruct();
    me.isConstructing = false;
  }
  /**
   * Factory version of the Base constructor. Merges all arguments to create a config object that is passed along to
   * the constructor.
   * @param {...Object} [configs] Allows passing multiple config objects
   * @returns {Core.Base} New instance
   * @private
   */
  static new(...configs) {
    configs = configs.filter((c) => c);
    return new this(configs.length > 1 ? this.mergeConfigs(...configs) : configs[0]);
  }
  /**
   * Base implementation applies configuration.
   *
   * Subclasses need only implement this if they have to initialize instance specific
   * properties required by the class. Often a `construct` method is
   * unnecessary. All initialization of incoming configuration properties can be
   * done in a `set propName` implementation.
   * @param {...Object} [args] Usually called with a config object, but accepts any params
   * @category Lifecycle
   * @advanced
   */
  construct(...args) {
    if (args[0] !== null) {
      this.configure(...args);
    }
    this.afterConfigure();
  }
  /**
   * Destroys the provided objects by calling their {@link #function-destroy} method.
   * Skips empty values or objects that are already destroyed.
   *
   * ```javascript
   * Base.destroy(myButton, toolbar1, helloWorldMessageBox);
   * ```
   * @param {...Object} [args] Objects to be destroyed
   * @category Lifecycle
   * @advanced
   */
  static destroy(...args) {
    const shredder = (object) => {
      if (object == null ? void 0 : object.destroy) {
        object.destroy();
      } else if (Array.isArray(object)) {
        object.forEach(shredder);
      }
    };
    shredder(args);
  }
  /**
   * Destroys this object.
   *
   * {@advanced}
   * This is primarily accomplished by calling {@link #function-doDestroy}, however, prior to
   * calling `doDestroy`, {@link #property-isDestroying} is set to `true`. After {@link #function-doDestroy} returns,
   * {@link #property-isDestroyed} is set to `true`.
   *
   * Do not override this method in subclasses. To provide class-specific cleanup, implement {@link #function-doDestroy}
   * instead.
   * {/@advanced}
   *
   * @category Lifecycle
   */
  destroy() {
    const me = this, { id } = me;
    me.isDestroying = true;
    me.destroy = emptyFn;
    me.doDestroy();
    Object.setPrototypeOf(me, null);
    for (const key in me) {
      if (key !== "destroy" && key !== "isDestroying") {
        delete me[key];
      }
    }
    delete me[originalConfigSymbol];
    me.isDestroyed = true;
    me.id = id;
  }
  /**
   * This method is required to help `unused` getters to survive production build process. Some tools, like angular,
   * will remove `unused` code in production build, making our side-effected getters behind, breaking code heavily.
   * @internal
   * @param getter Getter to evaluate
   */
  _thisIsAUsedExpression(getter) {
  }
  static get $$name() {
    return hasOwn(this, "$name") && this.$name || // _$name is filled by webpack for every class (cls._$name = '...')
    hasOwn(this, "_$name") && this._$name || this.name;
  }
  get $$name() {
    return this.constructor.$$name;
  }
  /**
   * Base implementation so that all subclasses and mixins may safely call super.startConfigure.
   *
   * This is called by the Base class before setting configuration properties, but after
   * the active initial getters have been set, so all configurations are available.
   *
   * This method allows all classes in the hierarchy to force some configs to be evaluated before others.
   * @internal
   * @category Lifecycle
   * @params {Object} config The configuration object use to set the initial state.
   */
  startConfigure(config) {
  }
  /**
   * Base implementation so that all subclasses and mixins may safely call super.finishConfigure.
   *
   * This is called by the Base class before exiting the {@link #function-configure} method.
   *
   * At this point, all configs have been applied, but the `isConfiguring` property is still set.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * into the config phase.
   * @internal
   * @category Lifecycle
   * @params {Object} config The configuration object use to set the initial state.
   */
  finishConfigure(config) {
  }
  /**
   * Base implementation so that all subclasses and mixins may safely call `super.afterConfigure`. This is called by the Base class after the {@link #function-configure} method has been
   * called. At this point, all configs have been applied.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * either before or after the super.afterConstruct();
   * @internal
   * @category Lifecycle
   */
  afterConfigure() {
  }
  /**
   * Base implementation so that all subclasses and mixins may safely call super.afterConstruct.
   *
   * This is called by the Base class after the {@link #function-construct} method has been
   * called.
   *
   * At this point, all configs have been applied.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * either before or after the super.afterConstruct();
   * @internal
   * @function afterConstructor
   * @category Lifecycle
   */
  afterConstruct() {
  }
  /**
   * Provides a way of calling callbacks which may have been specified as the _name_ of a function
   * and optionally adds scope resolution.
   *
   * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
   * then the function is resolved in this object. This is useful when configuring listeners
   * at the class level.
   *
   * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
   * using the `owner` property until an object with the named function is present, then the
   * named function is called upon that object.
   *
   * If a named function is not found, an error is thrown. If the function should be only called when present,
   * and may not be present, add a `?` as a suffix.
   *
   * @param {String|Function} fn The function to call, or the name of the function to call.
   * @param {Object} thisObject The `this` object of the function.
   * @param {Object[]} args The argument list to pass.
   * @category Misc
   * @advanced
   */
  callback(fn2, thisObject, args = emptyArray) {
    const { handler, thisObj } = this.resolveCallback(fn2, thisObject === "this" ? this : thisObject) || emptyObject2;
    return handler == null ? void 0 : handler.apply(thisObj, args);
  }
  resolveProperty(propertyPath) {
    let thisObj = this;
    while (thisObj) {
      if (Objects.hasPath(thisObj, propertyPath)) {
        return Objects.getPath(thisObj, propertyPath);
      }
      thisObj = thisObj.owner;
    }
    return void 0;
  }
  /**
   * Provides a way of locating callbacks which may have been specified as the _name_ of a function
   * and optionally adds scope resolution.
   *
   * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
   * then the function is resolved in this object. This is useful when configuring listeners
   * at the class level.
   *
   * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
   * using the `owner` property until an object with the named function is present, then the
   * named function is called upon that object.
   * @param {String|Function} handler The function to call, or the name of the function to call.
   * @param {Object} thisObj The `this` object of the function.
   * @param {Boolean} [enforceCallability = true] Pass `false` if the function may not exist, and a null return value is acceptable.
   * @returns {Object} `{ handler, thisObj }`
   * @category Misc
   * @advanced
   */
  resolveCallback(handler, thisObj = this, enforceCallability = true) {
    if (handler == null ? void 0 : handler.substring) {
      if (handler.endsWith("?")) {
        enforceCallability = false;
        handler = handler.substring(0, handler.length - 1);
      }
      if (handler.startsWith("up.")) {
        handler = handler.substring(3);
        for (thisObj = this.owner; thisObj && !thisObj[handler]; thisObj = thisObj.owner)
          ;
        if (!thisObj) {
          return;
        }
      } else if (handler.startsWith("this.")) {
        handler = handler.substring(5);
        thisObj = this;
      }
      if (!thisObj || !(thisObj instanceof Object)) {
        return;
      }
      handler = thisObj[handler];
    }
    if (typeof handler === "function") {
      return { handler, thisObj };
    }
    if (enforceCallability) {
      throw new Error(`No method named ${handler} on ${thisObj.$$name || "thisObj object"}`);
    }
  }
  bindCallback(inHandler, inThisObj = this) {
    if (inHandler) {
      const { handler, thisObj } = this.resolveCallback(inHandler, inThisObj);
      if (handler) {
        return handler.bind(thisObj);
      }
    }
  }
  /**
   * Delays the execution of the passed function by the passed time quantum, or if the time is omitted
   * or not a number, delays until the next animation frame. Note that this will use
   * {@link Core.mixin.Delayable#function-setTimeout} || {@link Core.mixin.Delayable#function-requestAnimationFrame}
   * if this class mixes in `Delayable`, otherwise it uses the global methods. The function will
   * be called using `this` object as its execution scope.
   * @param {Function} fn The function to call on a delay.
   * @param {Number} [delay] The number of milliseconds to delay.
   * @param {String} [name] The name of delay
   * @returns {Number} The created timeout id.
   * @private
   */
  delay(fn2, delay3, name = fn2.name || fn2) {
    fn2 = this.setTimeout ? fn2 : fn2.bind(this);
    const invoker = this.setTimeout ? this : globalThis;
    return invoker[typeof delay3 === "number" ? "setTimeout" : "requestAnimationFrame"](fn2, delay3, name);
  }
  /**
   * Classes implement this method to provide custom cleanup logic before calling `super.doDestroy()`. The general
   * pattern is as follows:
   *
   * ```javascript
   *  class Foo extends Base {
   *      doDestroy() {
   *          // perform custom cleanup
   *
   *          super.doDestroy();
   *      }
   *  }
   * ```
   *
   * This method is called by {@link #function-destroy} which also prevents multiple calls from reaching `doDestroy`.
   * Prior to calling `doDestroy`, {@link #property-isDestroying} is set to `true`. Upon return, the object is fully
   * destructed and {@link #property-isDestroyed} is set to `true`.
   *
   * Do not call this method directly. Instead call {@link #function-destroy}.
   * @category Lifecycle
   * @advanced
   */
  doDestroy() {
    const me = this, { nullify } = me.$meta;
    if (nullify) {
      for (let i = 0; i < nullify.length; ++i) {
        if (me[nullify[i].field] != null) {
          me[nullify[i].name] = null;
        }
      }
    }
  }
  /**
   * Destroys the named properties if they have been initialized, and if they have a `destroy` method.
   * Deletes the property from this object. For example:
   *
   *      this.destroyProperties('store', 'resourceStore', 'eventStore', 'dependencyStore', 'assignmentStore');
   *
   * @param {String} properties The names of the properties to destroy.
   * @internal
   * @category Lifecycle
   */
  destroyProperties(...properties) {
    var _a2, _b;
    const me = this;
    let key;
    for (key of properties) {
      if (key in me && (!me[configuringSymbol2] || !me[configuringSymbol2][key])) {
        (_b = (_a2 = me[key]) == null ? void 0 : _a2.destroy) == null ? void 0 : _b.call(_a2);
        delete me[key];
      }
    }
  }
  /**
   * Called by the Base constructor to apply configs to this instance. This must not be called.
   * @param {Object} config The configuration object from which instance properties are initialized.
   * @private
   * @category Lifecycle
   */
  configure(config = {}) {
    const me = this, meta = me.$meta, { beforeConfigure } = config, configs = meta.configs, fullConfig = me.getDefaultConfiguration();
    let cfg, key, value;
    me.initialConfig = config;
    me.isConfiguring = true;
    Object.assign(me, me.getProperties());
    for (key in config) {
      value = config[key];
      cfg = configs[key];
      fullConfig[key] = cfg ? cfg.merge(value, fullConfig[key], null, meta) : value;
    }
    if (beforeConfigure) {
      delete fullConfig.beforeConfigure;
      beforeConfigure(me, fullConfig);
    }
    me.setConfig(me[originalConfigSymbol] = fullConfig, true);
    me.isConfiguring = false;
  }
  /**
   * Returns the value of the specified config property. This is a method to allow
   * property getters to be explicitly called in a way that does not get optimized out.
   *
   * The following triggers the getter call, but optimizers will remove it:
   *
   *      inst.foo;   // also raises "expression has no side-effects" warning
   *
   * Instead, do the following to trigger a getter:
   *
   *      inst.getConfig('foo');
   *
   * @param {String} name
   * @internal
   * @category Configuration
   */
  getConfig(name) {
    return this[name];
  }
  /**
   * Sets configuration options this object with all the properties passed in the parameter object.
   * Timing is taken care of. If the setter of one config is called first, and references
   * the value of another config which has not yet been set, that config will be set just
   * in time, and the *new* value will be used.
   * @param {Object} config An object containing configurations to change.
   * @category Lifecycle
   * @advanced
   */
  setConfig(config, isConstructing) {
    var _a2;
    const me = this, wasConfiguring = me[configuringSymbol2], configDone = wasConfiguring ? me.configDone : me.configDone = {}, configs = me.$meta.configs;
    let cfg, key;
    me[instancePropertiesSymbol2] = {};
    me[configuringSymbol2] = wasConfiguring ? Object.setPrototypeOf(Object.assign({}, config), wasConfiguring) : config;
    for (key in config) {
      if (config[key] != null || hasOwn(config, key)) {
        cfg = configs[key] || Config.get(key);
        cfg.defineInitter(me, config[key]);
        if (!isConstructing) {
          configDone[key] = false;
        }
      } else {
        configDone[key] = true;
      }
    }
    if (isConstructing) {
      me.startConfigure(config);
    }
    for (key in config) {
      if (!configDone[key] && !((_a2 = configs[key]) == null ? void 0 : _a2.lazy)) {
        me[key] = config[key];
      }
    }
    if (wasConfiguring) {
      me[configuringSymbol2] = wasConfiguring;
    } else {
      delete me[configuringSymbol2];
    }
    if (isConstructing) {
      me.finishConfigure(config);
    }
    return me;
  }
  /**
   * Returns `true` if this instance has a non-null value for the specified config. This will not activate a lazy
   * config.
   *
   * @param {String} name The name of the config property.
   * @returns {Boolean}
   * @internal
   */
  hasConfig(name) {
    var _a2;
    const me = this, config = me[configuringSymbol2];
    return Boolean(
      me["_" + name] != null || // value has been assigned to backing property
      ((_a2 = me[lazyConfigsSymbol]) == null ? void 0 : _a2.get(name)) != null || // a lazy value is pending
      // config value has not been assigned but will be
      !me.configDone[name] && config && (config[name] != null || hasOwn(config, name))
    );
  }
  /**
   * Returns the value of an uningested config *without* ingesting the config or transforming
   * it from its raw value using its `changeXxxxx` method.
   *
   * @param {String} name The name of the config property.
   * @returns {*} The raw incoming config value.
   * @internal
   */
  peekConfig(name) {
    const me = this, lazyConfig = me[lazyConfigsSymbol], config = me[configuringSymbol2];
    if (lazyConfig == null ? void 0 : lazyConfig.has(name)) {
      return lazyConfig.get(name);
    }
    if (config && name in config) {
      if (me.configDone[name]) {
        return me[name];
      }
      if (config[name] != null || hasOwn(config, name)) {
        return config[name];
      }
    }
  }
  /**
   * Ensures that the specified config is initialized if it is needed. If there is a config value specified, and it
   * was initialized by this call, this method returns `true`. If there was a config value specified, and it was
   * already initialized, this method returns `false`. If there was no value specified for the given config, this
   * method returns `null`.
   *
   * This is not the same as just reading the property, because some property getters exist that do not actually just
   * read the config value back, but instead produce some result. Reading such properties to incidentally trigger a
   * possible config initializer can lead to incorrect results. For example, the Combo items config.
   *
   * @param {String} name The name of the config property.
   * @returns {Boolean}
   * @internal
   */
  triggerConfig(name) {
    const me = this, { configDone } = me, lazyConfig = me[lazyConfigsSymbol], config = me[configuringSymbol2], triggered = (lazyConfig == null ? void 0 : lazyConfig.has(name)) || config && (config[name] != null || hasOwn(config, name)) ? !configDone[name] : null;
    if (triggered) {
      me.getConfig(name);
    }
    return triggered;
  }
  /**
   * This call will activate any pending {@link Core.Config#config-lazy} configs that were assigned a string value
   * equal to the `group` parameter.
   *
   * @param {String} group The config property group as defined by a matching {@link Core.Config#config-lazy} value.
   * @returns {String[]} The names of any configs triggered by this call or `null` if no configs were triggered.
   * @internal
   */
  triggerConfigs(group) {
    const me = this, configs = me.$meta.configs, lazyConfigs = me[lazyConfigsSymbol], triggered = lazyConfigs ? [...lazyConfigs.keys()].filter((k) => configs[k].lazy === group) : emptyArray;
    for (const key of triggered) {
      me.triggerConfig(key);
    }
    return triggered.length ? triggered : null;
  }
  onConfigChange() {
  }
  // declared above because lint/IDE get angry about not declaring the args...
  /**
   * This method is called when any config changes.
   * @param {Object} info Object containing information regarding the config change.
   * @param {String} info.name The name of the config that changed.
   * @param {*} info.value The new value of the config.
   * @param {*} info.was The previous value of the config.
   * @param {Core.Config} info.config The `Config` object for the changed config property.
   * @method onConfigChange
   * @internal
   * @category Configuration
   */
  /**
   * Returns a *copy* of the full configuration which was used to configure this object.
   * @property {Object}
   * @category Lifecycle
   * @readonly
   * @advanced
   */
  get config() {
    const result = {}, myConfig = this[originalConfigSymbol];
    for (const key in myConfig) {
      result[key] = myConfig[key];
    }
    return result;
  }
  // region Extract config
  static processConfigValue(currentValue, options) {
    if (currentValue === globalThis) {
      return globalThis;
    } else if (Array.isArray(currentValue)) {
      return currentValue.map((v) => Base.processConfigValue(v, options));
    } else if (currentValue instanceof Base) {
      if (options.visited.has(currentValue)) {
        return;
      }
      return currentValue.getCurrentConfig(options);
    } else if (currentValue instanceof HTMLElement || currentValue instanceof DocumentFragment) {
      return null;
    } else if (Objects.isObject(currentValue)) {
      const result = {};
      for (const key in currentValue) {
        if (key !== "owner") {
          result[key] = Base.processConfigValue(currentValue[key], options);
        }
      }
      return result;
    }
    return currentValue;
  }
  // Recursively get the value of a config. Only intended to be called by getCurrentConfig()
  getConfigValue(name, options) {
    var _a2;
    const me = this, lazyConfigs = me[lazyConfigsSymbol];
    if (!((_a2 = me.$meta.configs[name]) == null ? void 0 : _a2.lazy)) {
      return Base.processConfigValue(me[name], options);
    }
    if (lazyConfigs == null ? void 0 : lazyConfigs.has(name)) {
      return Base.processConfigValue(lazyConfigs.get(name), options);
    }
  }
  // Allows removing / adding configs before values are extracted
  preProcessCurrentConfigs() {
  }
  // Extract the current values for all initially used configs, in a format that can be used to create a new instance.
  // Not intended to be called by any other code than getConfigString()
  getCurrentConfig(options = {}) {
    const me = this, configs = options.configs === "all" ? me.config : Objects.clone(me.initialConfig), visited = options.visited || (options.visited = /* @__PURE__ */ new Set()), depth = options.depth || (options.depth = 0), result = {};
    if (visited.has(me)) {
      return void 0;
    }
    visited.add(me);
    this.preProcessCurrentConfigs(configs);
    for (const name in configs) {
      const value = me.getConfigValue(name, { ...options, depth: depth + 1 });
      if (value !== void 0) {
        result[name] = value;
      }
    }
    return result;
  }
  // Extract the current values for all initially used configs and convert them to a JavaScript string
  getConfigString(options = {}) {
    return StringHelper.toJavaScriptString(this.getCurrentConfig(options));
  }
  // Experimental helper function, extracts the currently used configs and wraps them as an app, returning code as a
  // string.
  //
  // This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
  //
  getTestCase(options = {}) {
    const Product = this.isGantt ? "Gantt" : this.isSchedulerPro ? "SchedulerPro" : this.isCalendar ? "Calendar" : this.isScheduler ? "Scheduler" : this.isGrid ? "Grid" : this.isTaskBoard ? "TaskBoard" : null;
    if (Product) {
      const product = Product.toLowerCase(), bundlePath = `../../build/${product}.module.js`;
      let preamble, postamble;
      if (options.import === "static") {
        preamble = `import * as module from "${bundlePath}";Object.assign(window, module);`;
        postamble = "";
      } else {
        preamble = `import("${bundlePath}").then(module => { Object.assign(window, module);
`;
        postamble = "});";
      }
      const version = VersionHelper.getVersion(product);
      if (version) {
        preamble += `
console.log('${Product} ${version}');
`;
      }
      return `${preamble}      
const ${product} = new ${Product}(${this.getConfigString(options)});
${postamble}`;
    }
  }
  /**
   * Experimental helper function, extracts the currently used configs and wraps them as an app, downloading the
   * resulting JS file.
   *
   * This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
   * @category Misc
   */
  downloadTestCase(options = {}) {
    options.output = "return";
    const app = this.getTestCase(options);
    BrowserHelper.download(`app.js`, "data:application/javascript;charset=utf-8," + escape(app));
  }
  //endregion
  /**
   * Registers this class type with its Factory
   * @category Misc
   * @advanced
   */
  static initClass() {
    return this.$meta.class;
  }
  /**
   * The class's {@link #property-$meta-static meta} object.
   * @member {Object} $meta
   * @internal
   * @category Misc
   */
  /**
   * An object owned by this class that does not share properties with its super class.
   *
   * This object may contain other properties which are added as needed and are not documented here.
   *
   * @property {Object} $meta The class meta object.
   * @property {Function} $meta.class The class constructor that owns the meta object.
   * @property {Object} $meta.super The `$meta` object for the super class. This is `null` for `Base`.
   * @property {Object} $meta.config The object holding the default configuration values for this class.
   * @property {Object} $meta.configs An object keyed by config name that holds the defined configs for the class.
   * The value of each property is a {@link Core/Config} instance.
   * @property {Boolean} $meta.forkConfigs This will be `true` if the default configuration values for this class
   * (in the `config` property of the meta object) must be forked to avoid object sharing, or if the object can be
   * passed to `Object.create()` for efficiency.
   * @property {Function[]} $meta.hierarchy The array of classes in the ancestry of this class. This will start with
   * `Base` at index 0 and ends with this class.
   * @property {Function[]} $meta.properties The array of classes that define a "static get properties()" getter.
   * @internal
   * @static
   * @category Misc
   */
  static get $meta() {
    const me = this;
    let meta = me[metaSymbol];
    if (!hasOwn(me, metaSymbol)) {
      me[metaSymbol] = meta = newMeta();
      meta.class = me;
      me.setupClass(meta);
    }
    return meta;
  }
  /**
   * This optional class method is called when a class is mixed in using the {@link #function-mixin-static mixin()}
   * method.
   * @internal
   */
  static onClassMixedIn() {
  }
  /**
   * Returns the merge of the `baseConfig` and `config` config objects based on the configs defined by this class.
   * @param {Object} baseConfig The base config or defaults.
   * @param {...Object} configs One or more config objects that takes priority over `baseConfig`.
   * @returns {Object}
   * @internal
   */
  static mergeConfigs(baseConfig, ...configs) {
    const classConfigs = this.$meta.configs, result = Objects.clone(baseConfig) || {};
    let config, i, key, value;
    for (i = 0; i < configs.length; ++i) {
      config = configs[i];
      if (config) {
        for (key in config) {
          value = config[key];
          if (classConfigs[key]) {
            value = classConfigs[key].merge(value, result[key]);
          } else if (result[key] && value) {
            value = Config.merge(value, result[key]);
          }
          result[key] = value;
        }
      }
    }
    return result;
  }
  /**
   * Applies one or more `mixins` to this class and returns the produced class constructor.
   *
   * For example, instead of writing this:
   * ```
   *  class A extends Delayable(Events(Localizable(Base))) {
   *      // ...
   *  }
   * ```
   *
   * Using this method, one would write this:
   * ```
   *  class A extends Base.mixin(Localizable, Events, Delayable) {
   *      // ...
   *  }
   * ```
   * If one of the mixins specified has already been mixed into the class, it will be ignored and not mixed in a
   * second time.
   * @param {...Function} mixins
   * @returns {Function}
   * @category Misc
   * @advanced
   */
  static mixin(...mixins) {
    let C = this, i;
    for (i = 0; i < mixins.length; ++i) {
      const mixin2 = mixins[i], tag = mixin2[mixinTagSymbol] || (mixin2[mixinTagSymbol] = Symbol("mixinTag"));
      if (C[tag]) {
        continue;
      }
      C = mixin2(C);
      C[tag] = true;
      if (hasOwn(C, "onClassMixedIn")) {
        C.onClassMixedIn();
      }
    }
    return C;
  }
  /**
   * This method is called only once for any class. This can occur when the first instance is created or when the
   * `$meta` object is first requested.
   * @param {Object} meta The `$meta` object for the class.
   * @internal
   * @category Misc
   */
  static setupClass(meta) {
    var _a2;
    const cls2 = meta.class, base = getPrototypeOf(cls2).$meta, name = cls2.$$name, names = base.names, proto4 = cls2.prototype;
    defineProperty2(proto4, "$meta", {
      value: meta
    });
    Object.assign(meta, {
      super: base,
      config: Object.create(base.config),
      configs: Object.create(base.configs),
      declarables: base.declarables,
      forkConfigs: base.forkConfigs,
      hierarchy: Object.freeze([...base.hierarchy, cls2]),
      names: names.includes(name) ? names : Object.freeze([...names, name]),
      properties: base.properties,
      nullify: (_a2 = base.nullify) == null ? void 0 : _a2.slice()
    });
    if (names !== meta.names) {
      const isName = `is${name}`, defineIsProperty = (obj) => {
        if (!hasOwn(obj, isName)) {
          defineProperty2(obj, isName, {
            get() {
              return true;
            }
          });
        }
      };
      defineIsProperty(proto4);
      defineIsProperty(cls2);
    }
    for (let decl, setupName, i = 0; i < meta.declarables.length; ++i) {
      decl = meta.declarables[i];
      if (hasOwn(cls2, decl)) {
        setupName = setupNames[decl] || (setupNames[decl] = `setup${StringHelper.capitalize(decl)}`);
        cls2[setupName](cls2, meta);
      }
    }
  }
  /**
   * This method is called as part of `setupClass()`. It will process the `configurable()` return object and the
   * `defaultConfig` return object.
   * @param {Object} meta The `meta` object for this class.
   * @param {Object} configs The config definition object.
   * @param {Boolean} simple `true` when processing `defaultConfig` and `false` when processing `configurable`.
   * @private
   * @category Configuration
   */
  static setupConfigs(meta, configs, simple) {
    const classConfigValues = meta.config, classConfigs = meta.configs, cls2 = meta.class, superMeta = meta.super;
    let { nullify } = meta, cfg, defaultValue2, options, setDefault, value, wasNullify;
    for (const name in configs) {
      value = configs[name];
      if (simple) {
        if (!(cfg = classConfigs[name])) {
          cfg = Config.get(name, defaultConfigOptions);
        } else {
          value = cfg.merge(value, classConfigValues[name], meta, superMeta);
        }
      } else {
        defaultValue2 = options = setDefault = void 0;
        if (value && typeof value === "object" && "$config" in value) {
          options = value.$config;
          if (options && !Objects.isObject(options)) {
            options = Objects.createTruthyKeys(options);
          }
          setDefault = "default" in value;
          defaultValue2 = setDefault ? value.default : defaultValue2;
          value = value.value;
        }
        if (!(cfg = classConfigs[name])) {
          cfg = Config.get(name, options);
          cfg.define(cls2.prototype);
          setDefault = !(cfg.field in cls2.prototype);
          wasNullify = false;
        } else {
          wasNullify = cfg.nullify;
          if (options) {
            cfg = cfg.extend(options);
          }
          value = cfg.merge(value, classConfigValues[name], meta, superMeta);
        }
        if (setDefault) {
          cfg.setDefault(cls2, defaultValue2);
        }
        if (cfg.nullify && !wasNullify) {
          (nullify || (nullify = meta.nullify || (meta.nullify = []))).push(cfg);
        }
      }
      if (value && (Objects.isObject(value) || Array.isArray(value)) && !Object.isFrozen(value)) {
        meta.forkConfigs = true;
      }
      classConfigs[name] = cfg;
      classConfigValues[name] = value;
    }
  }
  static setupConfigurable(cls2, meta) {
    cls2.setupConfigs(meta, cls2.configurable, false);
  }
  static setupDefaultConfig(cls2, meta) {
    cls2.setupConfigs(meta, cls2.defaultConfig, true);
  }
  static setupDeclarable(cls2, meta) {
    const declarable = cls2.declarable;
    let all = meta.declarables, forked, i;
    for (i = 0; i < declarable.length; ++i) {
      if (!all.includes(declarable[i])) {
        if (!forked) {
          meta.declarables = forked = all = all.slice();
        }
        all.push(declarable[i]);
      }
    }
  }
  static setupProperties(cls2, meta) {
    meta.properties = meta.super.properties.slice();
    meta.properties.push(cls2);
    Object.freeze(meta.properties);
  }
  static setupPrototypeProperties(cls2) {
    Object.assign(cls2.prototype, cls2.prototypeProperties);
  }
  /**
   * Gets the full {@link #property-defaultConfig-static} block for this object's entire inheritance chain
   * all the way up to but not including {@link Core.Base}
   * @returns {Object} All default config values for this class.
   * @private
   * @category Configuration
   */
  getDefaultConfiguration() {
    return this.constructor.getDefaultConfiguration();
  }
  /**
   * Gets the full {@link #property-defaultConfig-static} block for the entire inheritance chain for this class
   * all the way up to but not including {@link Core.Base}
   * @returns {Object} All default config values for this class.
   * @private
   * @category Configuration
   */
  static getDefaultConfiguration() {
    const meta = this.$meta, config = meta.forkConfigs ? Base.fork(meta.config) : Object.create(meta.config);
    if (VersionHelper.isTestEnv && BrowserHelper.isBrowserEnv && config.testConfig && globalThis.__applyTestConfigs) {
      for (const o in config.testConfig) {
        config[o] = config.testConfig[o];
      }
    }
    return config;
  }
  static fork(obj) {
    let ret = obj, key, value;
    if (obj && Objects.isObject(obj) && !Object.isFrozen(obj)) {
      ret = Object.create(obj);
      for (key in obj) {
        value = obj[key];
        if (value) {
          if (Objects.isObject(value)) {
            ret[key] = Base.fork(value);
          } else if (Array.isArray(value)) {
            ret[key] = value.slice();
          }
        }
      }
    }
    return ret;
  }
  /**
   * Gets the full {@link #property-properties-static} block for this class's entire inheritance chain
   * all the way up to but not including {@link Core.Base}
   * @returns {Object} All default config values for this class.
   * @private
   * @category Configuration
   */
  getProperties() {
    const hierarchy = this.$meta.properties, result = {};
    for (let i = 0; i < hierarchy.length; i++) {
      Object.assign(result, hierarchy[i].properties);
    }
    return result;
  }
  static get superclass() {
    return getPrototypeOf(this);
  }
  /**
   * Used by the Widget and GridFeatureManager class internally. Returns the class hierarchy of this object
   * starting from the `topClass` class (which defaults to `Base`).
   *
   * For example `classHierarchy(Widget)` on a Combo would yield `[Widget, Field, TextField, PickerField, Combo]`
   * @param {Function} [topClass] The topmost class constructor to start from.
   * @returns {Function[]} The class hierarchy of this instance.
   * @private
   * @category Configuration
   */
  classHierarchy(topClass) {
    const hierarchy = this.$meta.hierarchy, index = topClass ? hierarchy.indexOf(topClass) : 0;
    return index > 0 ? hierarchy.slice(index) : hierarchy;
  }
  /**
   * Checks if an obj is of type using object's $$name property and doing string comparison of the property with the
   * type parameter.
   *
   * @param {String} type
   * @returns {Boolean}
   * @category Misc
   * @advanced
   */
  static isOfTypeName(type) {
    return this.$meta.names.includes(type);
  }
  /**
   * Removes all event listeners that were registered with the given `name`.
   * @param {String|Symbol} name The name of the event listeners to be removed.
   * @category Events
   * @advanced
   */
  detachListeners(name) {
    let detachers = this.$detachers;
    detachers = detachers == null ? void 0 : detachers[name];
    if (detachers) {
      while (detachers.length) {
        detachers.pop()();
      }
    }
  }
  /**
   * Tracks a detacher function for the specified listener name.
   * @param {String} name The name assigned to the associated listeners.
   * @param {Function} detacher The detacher function.
   * @private
   */
  trackDetacher(name, detacher2) {
    const detachers = this.$detachers || (this.$detachers = {}), bucket2 = detachers[name] || (detachers[name] = []);
    bucket2.push(detacher2);
  }
  /**
   * Removes all detacher functions for the specified `Events` object. This is called
   * by the `removeAllListeners` method on that object which is typically called by its
   * `destroy` invocation.
   * @param {Core.mixin.Events} eventer The `Events` instance to untrack.
   * @private
   */
  untrackDetachers(eventer) {
    const detachers = this.$detachers;
    if (detachers) {
      for (const name in detachers) {
        const bucket2 = detachers[name];
        for (let i = bucket2.length; i-- > 0; ) {
          if (bucket2[i].eventer === eventer) {
            bucket2.splice(i, 1);
          }
        }
      }
    }
  }
};
var proto = Base.prototype;
proto.onConfigChange.$nullFn = emptyFn.$nullFn = true;
Base[metaSymbol] = proto.$meta = newMeta({
  class: Base,
  config: Object.freeze({}),
  configs: /* @__PURE__ */ Object.create(null),
  declarables: Base.declarable,
  forkConfigs: false,
  hierarchy: Object.freeze([Base]),
  names: Object.freeze(["Base"]),
  nullify: null,
  properties: Object.freeze([]),
  super: null
});
Object.assign(proto, {
  $detachers: null,
  configObserver: null,
  /**
   * This property is set to `true` before the `constructor` returns.
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   * @advanced
   */
  isConstructing: true,
  /**
   * This property is set to `true` by {@link #function-destroy} after destruction.
   *
   * It is also one of the few properties that remains on the object after returning from `destroy()`. This property
   * is often checked in code paths that may encounter a destroyed object (like some event handlers) or in the
   * destruction path during cleanup.
   *
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   */
  isDestroyed: false,
  /**
   * This property is set to `true` on entry to the {@link #function-destroy} method. It remains on the objects after
   * returning from `destroy()`. If {@link #property-isDestroyed} is `true`, this property will also be `true`, so
   * there is no need to test for both (for example, `comp.isDestroying || comp.isDestroyed`).
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   * @advanced
   */
  isDestroying: false
});
Base.emptyFn = emptyFn;
VersionHelper.setVersion("core", "5.5.0");
Base._$name = "Base";

// ../Core/lib/Core/helper/ArrayHelper.js
var ArrayHelper = class {
  static clean(array) {
    return array.reduce((res, item) => {
      if (item !== null && item !== void 0 && !(Array.isArray(item) && item.length === 0) && item !== "")
        res.push(item);
      return res;
    }, []);
  }
  /**
   * Similar to [`Array.from()`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/from)
   * this method creates an array from an `iterable` object. Where `Array.from()` accepts a mapper function as the
   * second argument, this method accepts a `filter` function as its second argument. If a mapper function is also
   * needed, it can be passed as the third argument. Unlike `Array.from()`, if this method is passed `null`, it will
   * return an empty array.
   * @param {Array} iterable The iterable object to convert (must support `for-of` loop iteration).
   * @param {Function} [filter] A function to apply to each item of the `iterable` which must return a truthy value
   * to include that item in the resulting array.
   * @param {Function} [map] A function to apply to each item of the `iterable` that returns the actual value to put
   * into the returned array. If a `filter` is also supplied, this method is only called for those items that pass
   * the filter test.
   * @returns {Array}
   */
  static from(iterable, filter2, map2) {
    const array = [];
    if (iterable) {
      for (const it of iterable) {
        if (!filter2 || filter2(it)) {
          array.push(map2 ? map2(it) : it);
        }
      }
    }
    return array;
  }
  /**
   * Remove one or more items from an array
   * @param {Array} array Array to remove from
   * @param {Object[]|Set} items One or more items to remove, or one Set containing items to remove
   * @returns {Boolean} Returns true if any item was removed
   */
  static remove(array, ...items2) {
    let index, item, removed = false;
    items2 = items2[0] instanceof Set ? [...items2[0]] : items2;
    for (let i = 0; i < items2.length; i++) {
      item = items2[i];
      if ((index = array.indexOf(item)) !== -1) {
        array.splice(index, 1);
        removed = true;
      }
    }
    return removed;
  }
  /**
   * Calculates the insertion index of a passed object into the passed Array according
   * to the passed comparator function. Note that the passed Array *MUST* already be ordered.
   * @param {Object} item The item to calculate the insertion index for.
   * @param {Array} array The array into which the item is to be inserted.
   * @param {Function} comparatorFn The comparison function. Must return -1 or 0 or 1.
   * @param {Object} comparatorFn.lhs The left object to compare.
   * @param {Object} comparatorFn.rhs The right object to compare.
   * @param {Number} index The possible correct index to try first before a binary
   * search is instigated.
   * @internal
   */
  static findInsertionIndex(item, array, comparatorFn = this.lexicalCompare, index) {
    const len = array.length;
    let beforeCheck, afterCheck;
    if (index < len) {
      beforeCheck = index > 0 ? comparatorFn(array[index - 1], item) : 0;
      afterCheck = index < len - 1 ? comparatorFn(item, array[index]) : 0;
      if (beforeCheck < 1 && afterCheck < 1) {
        return index;
      }
    }
    return this.binarySearch(array, item, comparatorFn);
  }
  /**
   * Similar to the native `Array.find()` call, but this finds the *last* element in the array for which
   * the passed function returns a truthy value.
   * @param {Object[]} array The array to find in.
   * @param {Function} fn The testing function.
   * @param {Object} [thisObj] The scope (`this` reference) in which to call the function.
   */
  static findLast(array, fn2, thisObj) {
    for (let { length } = array, i = length - 1; i >= 0; i--) {
      if (fn2.call(thisObj, array[i], i, array)) {
        return array[i];
      }
    }
  }
  /**
   * This method returns the index that a given item would be inserted into the
   * given (sorted) `array`. Note that the given `item` may or may not be in the
   * array. This method will return the index of where the item *should* be.
   *
   * For example:
   *
   *      var array = [ 'A', 'D', 'G', 'K', 'O', 'R', 'X' ];
   *      var index = ArrayHelper.binarySearch(array, 'E');
   *
   *      console.log('index: ' + index);
   *      // logs "index: 2"
   *
   *      array.splice(index, 0, 'E');
   *
   *      console.log('array : ' + array.join(''));
   *      // logs "array: ADEGKORX"
   *
   * @param {Object[]} array The array to search.
   * @param {Object} item The item that you want to insert into the `array`.
   * @param {Number} [begin=0] The first index in the `array` to consider.
   * @param {Number} [end=array.length] The index that marks the end of the range
   * to consider. The item at this index is *not* considered.
   * @param {Function} [compareFn] The comparison function that matches the sort
   * order of the `array`. The default `compareFn` compares items using less-than
   * and greater-than operators.
   * @returns {Number} The index for the given item in the given array based on
   * the passed `compareFn`.
   */
  static binarySearch(array, item, begin = 0, end = array.length, compareFn = this.lexicalCompare) {
    const length = array.length;
    let middle, comparison;
    if (begin instanceof Function) {
      compareFn = begin;
      begin = 0;
    } else if (end instanceof Function) {
      compareFn = end;
      end = length;
    }
    --end;
    while (begin <= end) {
      middle = begin + end >> 1;
      comparison = compareFn(item, array[middle]);
      if (comparison >= 0) {
        begin = middle + 1;
      } else if (comparison < 0) {
        end = middle - 1;
      }
    }
    return begin;
  }
  magnitudeCompare(lhs, rhs) {
    return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
  }
  lexicalCompare(lhs, rhs) {
    lhs = String(lhs);
    rhs = String(rhs);
    return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
  }
  /**
   * Similar to Array.prototype.fill(), but constructs a new array with the specified item count and fills it with
   * clones of the supplied item.
   * @param {Number} count Number of entries to create
   * @param {Object|Array} itemOrArray Item or array of items to clone (uses object spread to create shallow clone)
   * @param {Function} [fn] An optional function that is called for each item added, to allow processing
   * @returns {Array} A new populated array
   */
  static fill(count, itemOrArray = {}, fn2 = null) {
    const result = [], items2 = Array.isArray(itemOrArray) ? itemOrArray : [itemOrArray];
    for (let i = 0; i < count; i++) {
      for (const item of items2) {
        const processedItem = Object.assign({}, item);
        if (fn2) {
          fn2(processedItem, i);
        }
        result.push(processedItem);
      }
    }
    return result;
  }
  /**
   * Populates an array with the return value from `fn`.
   * @param {Number} count Number of entries to create
   * @param {Function} fn A function that is called `count` times, return value is added to array
   * @param {Number} fn.index Current index in the array
   * @privateparam {Boolean} [oneBased] Add 1 to the index before calling the fn (making it 1 based)
   * @returns {Array} A new populated array
   */
  static populate(count, fn2, oneBased = false) {
    const items2 = [];
    for (let i = 0; i < count; i++) {
      items2.push(fn2(i + (oneBased ? 1 : 0)));
    }
    return items2;
  }
  /**
   * Pushes `item` on to the `array` if not already included
   * @param {Array}  array Array to push to
   * @param {...Object} items Item(s) to push if not already included
   */
  static include(array, ...items2) {
    for (const item of items2) {
      if (!array.includes(item)) {
        array.push(item);
      }
    }
  }
  /**
   * Returns a new array with the unique items from the supplied array.
   * @param {Array} array Input array
   * @returns {Array} New array with unique items
   */
  static unique(array) {
    return [...new Set(array)];
  }
  // Kept for future reference : Wanted to create an indexer on Stores.
  static allowNegative(array) {
    return new Proxy(array, {
      get(target, name, receiver) {
        if (typeof name !== "string") {
          return Reflect.get(target, name, receiver);
        }
        const index = Number(name);
        if (Number.isNaN(index)) {
          return Reflect.get(target, name, receiver);
        }
        return target[index < 0 ? target.length + index : index];
      },
      set(target, name, value, receiver) {
        if (typeof name !== "string") {
          return Reflect.set(target, name, value, receiver);
        }
        const index = Number(name);
        if (Number.isNaN(index)) {
          return Reflect.set(target, name, value, receiver);
        }
        target[index < 0 ? target.length + index : index] = value;
        return true;
      }
    });
  }
  static delta(a, b, useRelativeNaming = false) {
    if (!b) {
      return useRelativeNaming ? { toAdd: a, toRemove: [], toKeep: [] } : { onlyInA: a, onlyInB: [], inBoth: [] };
    }
    const onlyInA = [], onlyInB = [], inBoth = /* @__PURE__ */ new Set(), bSet = new Set(b);
    for (let i = 0; i < a.length; i++) {
      const item = a[i];
      if (bSet.has(item)) {
        inBoth.add(item);
      } else {
        onlyInA.push(item);
      }
    }
    for (let i = 0; i < b.length; i++) {
      const item = b[i];
      if (!inBoth.has(item)) {
        onlyInB.push(item);
      }
    }
    if (useRelativeNaming) {
      return { toAdd: onlyInA, toRemove: onlyInB, toKeep: inBoth };
    }
    return { onlyInA, onlyInB, inBoth: [...inBoth] };
  }
  /**
   * Returns the passed object wrapped in an array. Special handling of the following cases:
   * * Passing an array returns it as is
   * * Passing a `Set` returns it converted to an Array
   * * Passing `null`/`undefined` returns the passed value
   *
   * ```javascript
   * const records = ArrayHelper.asArray(record);
   *
   * // { id : 1 } -> [{ id : 1 }]
   * // [{ id : 1 }] -> [{ id : 1 }]
   * ```
   *
   * @param {*} arrayOrObject
   * @returns {Array|null}
   * @internal
   */
  static asArray(arrayOrObject) {
    if (arrayOrObject == null) {
      return arrayOrObject;
    }
    if (arrayOrObject instanceof Set) {
      return Array.from(arrayOrObject);
    }
    return Array.isArray(arrayOrObject) ? arrayOrObject : [arrayOrObject];
  }
  /**
   * Identity function that returns its input.
   * @param {*} Any value
   * @returns {*} The input value
   * @internal
   */
  static identity(x) {
    return x;
  }
  /**
   * Transform an array into a key:value dictionary using the specified
   * key and value getters. Does not group values, so only one result will
   * appear in the output for a given key.
   *
   * ```javascript
   * const input = [{
   *     id: '1',
   *     other: 'one'
   * },{
   *     id: '2',
   *     other: 'two'
   * }];
   *
   * keyBy(input, rec => rec.id)
   *
   * // {
   * //    '1': { id: '1', other: 'one' },
   * //    '2': { id: '2', other: 'two' }
   * // }
   * ```
   *
   * @param {Array} array Array from which to build dictionary
   * @param {Function|String} getKey Function to produce the key for a given array entry, or a string property name to read
   * @param {Function} getValue Optional function to transform array elements. Defaults to returning the element itself
   * @returns {Object} A key->value dictionary with keys as returned by `getKey` and values returned by `getValue`
   * @internal
   */
  static keyBy(array, keyGetter, valueGetter = ArrayHelper.identity) {
    const getKey = typeof keyGetter === "string" ? (o) => o == null ? void 0 : o[keyGetter] : keyGetter, getValue = typeof valueGetter === "string" ? (o) => o == null ? void 0 : o[valueGetter] : valueGetter;
    return array.reduce((dict, next) => {
      dict[getKey(next)] = getValue(next);
      return dict;
    }, {});
  }
  /**
   * Combines provided arrays of by aggregating their element values.
   * For example the below code sums up numeric elements of the arrays:
   *
   * ```javascript
   * ArrayHelper.aggregate(
   *     [
   *         [0,   1,  2, 33]
   *         [10,  1, -1],
   *         [100, 1, -1]
   *     ],
   *     entry => entry || 0, // "|| 0" here to make it work for different array sizes
   *     (aggregated, entry) => aggregated + entry, // aggregate by summing up
   *     () => 0 //initial value is zero
   * );
   *
   * // returns [111, 3, 0, 33] array
   * ```
   *
   * @param {Array[]} arrays Array to combine
   * @param {Function} getEntryValueFn Function that extracts an array entry for aggregating.
   * @param {Function} aggregatorFn A function to execute for each element in the arrays. It's purpose is to
   * aggregate the element value to the corresponding entry of the resulting array.
   * The function's return value becomes the value of the `aggregated` parameter on the next invocation of
   * `aggregatorFn`.
   * The function is called with the following arguments:
   *
   * @param {Object} aggregatorFn.aggregated Resulting array entry value. On the first call
   * `getInitialValueFn` result.
   * @param {Object} aggregatorFn.entry Current entry to aggregate into `aggregated`.
   * @param {Number} aggregatorFn.arrayIndex Index of current array (in the provided `arrays`).
   * @param {Object[]} aggregatorFn.entryIndex Index of the current entry.
   * @param {Object} aggregationContext A shared object providing extra aggregation call context.
   * @param {Function} getInitialValueFn Function that returns an initial value for the combined array entries.
   * @param {Object} [aggregationContext] Optional object that is passed to all of the above functions that can
   * be used for keeping some additional parameters used when aggregating.
   * Out of the box the object will contain `arrays` and `targetArray` properties containing
   * input and resulting arrays respectively.
   */
  static aggregate(arrays, getEntryValueFn, aggregatorFn, getInitialValueFn, aggregationContext = {}) {
    const rowLength = arrays.length, columnLength = arrays[0].length, result = [];
    aggregationContext.targetArray = result;
    aggregationContext.arrays = arrays;
    for (let colIndex = 0; colIndex < columnLength; colIndex++) {
      aggregationContext.entryIndex = colIndex;
      result.push(getInitialValueFn(colIndex, aggregationContext));
    }
    for (let rowIndex = 0; rowIndex < rowLength; rowIndex++) {
      const row = arrays[rowIndex];
      aggregationContext.rowIndex = rowIndex;
      for (let colIndex = 0; colIndex < columnLength; colIndex++) {
        aggregationContext.entryIndex = colIndex;
        const entryValue = getEntryValueFn(row[colIndex], aggregationContext);
        result[colIndex] = aggregatorFn(result[colIndex], entryValue, rowIndex, colIndex, aggregationContext);
      }
    }
    return result;
  }
  /**
   * Group an array by keys (either the values in a specified property name, or the results of a string-generating function accepting
   * an array entry as input), returning an Object with those keys, whose values are arrays containing the array entries that
   * produced that key.
   *
   * ```javascript
   * const input = [{
   *     id: 1,
   *     color: 'red'
   * },{
   *     id: 2,
   *     color: 'green'
   * },{
   *     id: 3,
   *     color: 'green'
   * }];
   *
   * groupBy(input, 'color')
   *
   * // {
   * //    'red': [ { id: '1', color: 'red' } ],
   * //    'green': [ { id: '2', color: 'green' }, { id: '3', color: 'green' } ]
   * // }
   *
   * groupBy(input, rec => rec.color?.substr(0, 1))
   *
   * // {
   * //    'r': [ { id: '1', color: 'red' } ],
   * //    'g': [ { id: '2', color: 'green' }, { id: '3', color: 'green' } ]
   * // }
   * ```
   *
   * @param {Object} array Array from which to build dictionary
   * @param {Function|String} getGroupKey Function to produce the key for a given array entry, or a string property name to read
   * @returns {Object} A key->value[] dictionary with keys as returned by `getKey` and arrays of matching items in original order as values
   * @internal
   */
  static groupBy(array, groupKeyGetter) {
    const getKey = typeof groupKeyGetter === "string" ? (o) => o == null ? void 0 : o[groupKeyGetter] : groupKeyGetter;
    return array.reduce((grouped, value) => {
      const key = getKey(value);
      (grouped[key] = grouped[key] || []).push(value);
      return grouped;
    }, {});
  }
};
ArrayHelper._$name = "ArrayHelper";

// ../Core/lib/Core/helper/FunctionHelper.js
var commaSepRe = /,\s*/;
var decompiledSym = Symbol("decompiled");
var fnRe1 = /^\s*(async\s+)?([a-z_]\w*)\s*=>([\s\S]+)$/i;
var fnRe2 = /^\s*(async\s*)?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)\s+=>([\s\S]+)$/i;
var fnRe3 = /^(\s*async)?(?:\s*function)?(?:\s*([a-z_]\w*))?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)([\s\S]+)$/i;
var { hasOwnProperty: hasOwnProperty4 } = Object.prototype;
var FunctionHelper = class {
  /**
   * Inserts a function after the specified `method` is called on an `object`. To remove this hook, invoke the
   * function returned by this method.
   * ```
   *  class A {
   *      method() {
   *          console.log('method');
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.after(instance, 'method', () => { console.log('after') });
   *
   *  instance.method();
   *  > method
   *  > after
   *
   *  detach();
   *  instance.method();
   *  > method
   * ```
   * The value returned by the original method is passed as the first argument to `fn` followed by all the arguments
   * passed by the caller.
   *
   * If `fn` returns a value (not `undefined`), that value is returned from the method call instead of the value
   * returned by the original method.
   * ```
   *  class A {
   *      method(x) {
   *          console.log('method', x);
   *          return x * 2
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.after(instance, 'method', (ret, x) => {
   *      console.log('after', ret, x);
   *      return x / 2;
   *  });
   *
   *  console.log(instance.method(50));
   *  > method 50
   *  > after 100 50
   *  > 25
   *
   *  detach();
   *  console.log(instance.method(50));
   *  > method 50
   *  > 100
   * ```
   *
   * @param {Object} object The object to hook.
   * @param {String} method The name of the method on `object` to hook.
   * @param {Function|String} fn The function or method name (on `thisObj`) to call after `method`.
   * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
   * @param {Object} [options] Additional options
   * @param {Boolean} [options.return=true] Specify `false` to not include the return value of the hooked method as
   * the first argument to `fn`.
   * @returns {Function} The function to call to remove the hook.
   */
  static after(object, method, fn2, thisObj, options) {
    const named = typeof fn2 === "string", withReturn = (options == null ? void 0 : options.return) !== false, hook = (...args) => {
      const origResult = hook.$nextHook.call(object, ...args), hookResult = (thisObj == null ? void 0 : thisObj.isDestroyed) ? void 0 : withReturn ? named ? thisObj[fn2](origResult, ...args) : fn2.call(thisObj, origResult, ...args) : named ? thisObj[fn2](...args) : fn2.call(thisObj, ...args);
      return hookResult === void 0 ? origResult : hookResult;
    };
    return FunctionHelper.hookMethod(object, method, hook);
  }
  /**
   * Inserts a function before the specified `method` is called on an `object`. To remove this hook, invoke the
   * function returned by this method.
   * ```
   *  class A {
   *      method() {
   *          console.log('method');
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.before(instance, 'method', () => { console.log('before') });
   *
   *  instance.method();
   *  > before
   *  > method
   *
   *  detach();
   *  instance.method();
   *  > method
   * ```
   * If `fn` returns `false`, the original method is not invoked and `false` is returned to the caller.
   * ```
   *  class A {
   *      method(x) {
   *          console.log('method', x);
   *          return x * 2;
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.before(instance, 'method', x => {
   *      console.log('before', x);
   *      return false;
   *  });
   *
   *  console.log(instance.method(50));
   *  > before 50
   *  > false
   *
   *  detach();
   *  console.log(instance.method(50));
   *  > method 50
   *  > 100
   * ```
   *
   * @param {Object} object The object to hook.
   * @param {String} method The name of the method on `object` to hook.
   * @param {Function|String} fn The function or method name (on `thisObj`) to call before `method`.
   * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
   * @returns {Function} The function to call to remove the hook.
   */
  static before(object, method, fn2, thisObj) {
    const named = typeof fn2 === "string", hook = (...args) => {
      const ret = (thisObj == null ? void 0 : thisObj.isDestroyed) ? 0 : named ? thisObj[fn2](...args) : fn2.call(thisObj, ...args);
      return ret === false ? ret : hook.$nextHook.call(object, ...args);
    };
    return FunctionHelper.hookMethod(object, method, hook);
  }
  static curry(func) {
    return function curried(...args) {
      if (args.length >= func.length) {
        return func.apply(this, args);
      } else {
        return function(...args2) {
          return curried.apply(this, args.concat(args2));
        };
      }
    };
  }
  static bindAll(obj) {
    for (const key in obj) {
      if (typeof obj[key] === "function") {
        obj[key] = obj[key].bind(obj);
      }
    }
  }
  /**
   * Returns a function which calls the passed `interceptor` function first, and the passed `original` after
   * as long as the `interceptor` does not return `false`.
   * @param {Function} original The function to call second.
   * @param {Function} interceptor The function to call first.
   * @param {Object} [thisObj] The `this` reference when the functions are called.
   * @returns {Function} A function which yields the return value from the `original` function **if it was called**, else `false`.
   */
  static createInterceptor(original, interceptor, thisObj) {
    return function(...args) {
      const theThis = thisObj || this;
      if (interceptor.call(theThis, ...args) !== false) {
        return original.call(theThis, ...args);
      }
      return false;
    };
  }
  /**
   * Returns a function which calls the passed `sequence` function after calling
   * the passed `original`.
   * @param {Function} original The function to call first.
   * @param {Function} sequence The function to call second.
   * @param {Object} [thisObj] The `this` reference when the functions are called.
   * @returns {Function} A function which yields the value returned from the sequence if it returned a value, else the return
   * value from the original function.
   */
  static createSequence(original, sequence, thisObj) {
    return (...args) => {
      const origResult = original.call(thisObj, ...args), sequenceResult = sequence.call(thisObj, ...args);
      return sequenceResult === void 0 ? origResult : sequenceResult;
    };
  }
  /**
   * Create a "debounced" function which will call on the "leading edge" of a timer period.
   * When first invoked will call immediately, but invocations after that inside its buffer
   * period will be rejected, and *one* invocation will be made after the buffer period has expired.
   *
   * This is useful for responding immediately to a first mousemove, but from then on, only
   * calling the action function on a regular timer while the mouse continues to move.
   *
   * @param {Function} fn The function to call.
   * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Array} [extraArgs] The argument list to append to those passed to the function.
   * @param {Function} [alt] A function to call when the invocation is rejected due to buffer time not having expired.
   * @returns {Function} A function which calls the passed `fn` only if at least the passed `buffer`
   * milliseconds has elapsed since its last invocation.
   */
  static createThrottled(fn2, buffer, thisObj, extraArgs, alt2) {
    let lastCallTime = -Number.MAX_VALUE, callArgs, timerId;
    const invoke = () => {
      timerId = 0;
      lastCallTime = performance.now();
      callArgs.push.apply(callArgs, extraArgs);
      fn2.apply(thisObj, callArgs);
    }, result = function(...args) {
      const elapsed = performance.now() - lastCallTime;
      callArgs = args;
      if (elapsed >= buffer) {
        clearTimeout(timerId);
        invoke();
      } else {
        if (!timerId) {
          timerId = setTimeout(invoke, buffer - elapsed);
        }
        if (alt2) {
          callArgs.push.apply(callArgs, extraArgs);
          alt2.apply(thisObj, callArgs);
        }
      }
    };
    result.cancel = () => clearTimeout(timerId);
    return result;
  }
  /**
   * Create a "debounced" function which will call on the "trailing edge" of a timer period.
   * When first invoked will wait until the buffer period has expired to call the function, and
   * more calls within that time will restart the timer.
   *
   * This is useful for responding to keystrokes, but deferring action until the user pauses typing.
   *
   * @param {Function} fn The function to call.
   * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Array} [args] The argument list to append to those passed to the function.
   * @returns {Function} A function which calls the passed `fn` when at least the passed `buffer`
   * milliseconds has elapsed since its last invocation.
   */
  static createBuffered(fn2, buffer, thisObj, args) {
    let callArgs, timerId;
    const invoke = () => {
      timerId = 0;
      result.isPending = false;
      callArgs.push.apply(callArgs, args);
      fn2.apply(thisObj, callArgs);
    }, result = function(...args2) {
      callArgs = args2;
      if (timerId) {
        clearTimeout(timerId);
      }
      result.isPending = true;
      timerId = setTimeout(invoke, buffer);
    };
    result.cancel = () => {
      result.isPending = false;
      clearTimeout(timerId);
    };
    return result;
  }
  static decompile(fn2) {
    if (!(decompiledSym in fn2)) {
      const code = fn2.toString();
      let m = fnRe1.exec(code), args, body, name, decompiled, t;
      if (m) {
        args = [m[2]];
        body = m[3];
      } else if (m = fnRe2.exec(code)) {
        t = m[2].trim();
        args = t ? t.split(commaSepRe) : [];
        body = m[3];
      } else if (m = fnRe3.exec(code)) {
        name = m[2];
        t = m[3].trim();
        args = t ? t.split(commaSepRe) : [];
        body = m[4];
      }
      body = body == null ? void 0 : body.trim();
      fn2[decompiledSym] = decompiled = m && {
        args,
        async: Boolean(m[1]),
        body: (body == null ? void 0 : body.startsWith("{")) ? body.substring(1, body.length - 1).trim() : body
      };
      if (name) {
        decompiled.name = name;
      }
    }
    return fn2[decompiledSym];
  }
  static hookMethod(object, method, hook) {
    hook.$nextHook = object[method];
    object[method] = hook;
    return () => {
      var _a2;
      if (hasOwnProperty4.call(object, method)) {
        let f = object[method], next;
        if (f === hook) {
          if (((_a2 = Object.getPrototypeOf(object)) == null ? void 0 : _a2[method]) === hook.$nextHook) {
            delete object[method];
          } else {
            object[method] = hook.$nextHook;
          }
        } else {
          for (; next = f == null ? void 0 : f.$nextHook; f = next) {
            if (next === hook) {
              f.$nextHook = hook.$nextHook;
              break;
            }
          }
        }
      }
    };
  }
  /**
   * Protects the specified `method` on a given `object` such that calling it will not throw exceptions.
   * @param {Object} object The object whose method is to be protected.
   * @param {String} method The name of the method to protect.
   * @param {Function} [handler] An optional function to call for any thrown exceptions.
   * @internal
   */
  static noThrow(object, method, handler) {
    const fn2 = object[method];
    object[method] = (...args) => {
      try {
        return fn2.apply(object, args);
      } catch (e) {
        return handler == null ? void 0 : handler(e);
      }
    };
  }
  static returnTrue() {
    return true;
  }
  static animate(duration, fn2, thisObj, easing = "linear") {
    let cancel = false;
    const result = new Promise((resolve) => {
      const start = performance.now(), iterate = () => {
        const progress = Math.min((performance.now() - start) / duration, 1), delayable = thisObj && thisObj.setTimeout ? thisObj : globalThis;
        if (!cancel) {
          if (fn2.call(thisObj, this.easingFunctions[easing](progress)) === false) {
            resolve();
          }
        }
        if (cancel || progress === 1) {
          delayable.requestAnimationFrame(() => resolve());
        } else {
          delayable.requestAnimationFrame(iterate);
        }
      };
      iterate();
    });
    result.cancel = () => {
      cancel = true;
      result.cancelled = true;
      return false;
    };
    return result;
  }
};
var half = 0.5;
var e1 = 1.70158;
var e2 = 7.5625;
var e3 = 1.525;
var e4 = 2 / 2.75;
var e5 = 2.25 / 2.75;
var e6 = 1 / 2.75;
var e7 = 1.5 / 2.75;
var e8 = 2.5 / 2.75;
var e9 = 2.625 / 2.75;
var e10 = 0.75;
var e11 = 0.9375;
var e12 = 0.984375;
var s1 = 1.70158;
var s2 = 1.70158;
FunctionHelper.easingFunctions = {
  linear: (t) => t,
  easeInQuad: (t) => Math.pow(t, 2),
  easeOutQuad: (t) => -(Math.pow(t - 1, 2) - 1),
  easeInOutQuad: (t) => (t /= half) < 1 ? half * Math.pow(t, 2) : -half * ((t -= 2) * t - 2),
  easeInCubic: (t) => Math.pow(t, 3),
  easeOutCubic: (t) => Math.pow(t - 1, 3) + 1,
  easeInOutCubic: (t) => (t /= half) < 1 ? half * Math.pow(t, 3) : half * (Math.pow(t - 2, 3) + 2),
  easeInQuart: (t) => Math.pow(t, 4),
  easeOutQuart: (t) => -(Math.pow(t - 1, 4) - 1),
  easeInOutQuart: (t) => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
  easeInQuint: (t) => Math.pow(t, 5),
  easeOutQuint: (t) => Math.pow(t - 1, 5) + 1,
  easeInOutQuint: (t) => (t /= half) < 1 ? half * Math.pow(t, 5) : half * (Math.pow(t - 2, 5) + 2),
  easeInSine: (t) => -Math.cos(t * (Math.PI / 2)) + 1,
  easeOutSine: (t) => Math.sin(t * (Math.PI / 2)),
  easeInOutSine: (t) => -half * (Math.cos(Math.PI * t) - 1),
  easeInExpo: (t) => t === 0 ? 0 : Math.pow(2, 10 * (t - 1)),
  easeOutExpo: (t) => t === 1 ? 1 : -Math.pow(2, -10 * t) + 1,
  easeInOutExpo: (t) => t === 0 ? 0 : t === 1 ? 1 : (t /= half) < 1 ? half * Math.pow(2, 10 * (t - 1)) : half * (-Math.pow(2, -10 * --t) + 2),
  easeInCirc: (t) => -(Math.sqrt(1 - t * t) - 1),
  easeOutCirc: (t) => Math.sqrt(1 - Math.pow(t - 1, 2)),
  easeInOutCirc: (t) => (t /= half) < 1 ? -half * (Math.sqrt(1 - t * t) - 1) : half * (Math.sqrt(1 - (t -= 2) * t) + 1),
  easeOutBounce: (t) => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
  easeInBack: (t) => t * t * ((e1 + 1) * t - e1),
  easeOutBack: (t) => (t = t - 1) * t * ((e1 + 1) * t + e1) + 1,
  easeInOutBack: (t) => {
    let v1 = s1;
    return (t /= half) < 1 ? half * (t * t * (((v1 *= e3) + 1) * t - v1)) : half * ((t -= 2) * t * (((v1 *= e3) + 1) * t + v1) + 2);
  },
  elastic: (t) => -1 * Math.pow(4, -8 * t) * Math.sin((t * 6 - 1) * (2 * Math.PI) / 2) + 1,
  swingFromTo: (t) => {
    let v2 = s2;
    return (t /= half) < 1 ? half * (t * t * (((v2 *= e3) + 1) * t - v2)) : half * ((t -= 2) * t * (((v2 *= e3) + 1) * t + v2) + 2);
  },
  swingFrom: (t) => t * t * ((e1 + 1) * t - e1),
  swingTo: (t) => (t -= 1) * t * ((e1 + 1) * t + e1) + 1,
  bounce: (t) => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
  bouncePast: (t) => t < e6 ? e2 * t * t : t < e4 ? 2 - (e2 * (t -= e7) * t + e10) : t < e8 ? 2 - (e2 * (t -= e5) * t + e11) : 2 - (e2 * (t -= e9) * t + e12),
  easeFromTo: (t) => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
  easeFrom: (t) => Math.pow(t, 4),
  easeTo: (t) => Math.pow(t, 0.25)
};
FunctionHelper._$name = "FunctionHelper";

// ../Core/lib/Core/mixin/Events.js
var { isArray } = Array;
var { hasOwnProperty: hasOwnProperty5 } = Object.prototype;
var specialProperties = {
  thisObj: 1,
  detachable: 1,
  once: 1,
  detacher: 1,
  prio: 1,
  args: 1,
  expires: 1,
  buffer: 1,
  throttle: 1,
  name: 1,
  $internal: 1
};
var priorityComparator = (a, b) => b.prio - a.prio;
var Events_default = (Target) => class Events extends (Target || Base) {
  constructor() {
    super(...arguments);
    __publicField(this, "eventsSuspended", null);
  }
  static get $name() {
    return "Events";
  }
  //region Events
  /**
   * Fires before an object is destroyed.
   * @event beforeDestroy
   * @param {Core.Base} source The Object that is being destroyed.
   */
  /**
   * Fires when an object is destroyed.
   * @event destroy
   * @param {Core.Base} source The Object that is being destroyed.
   */
  /**
   * Fires when any other event is fired from the object.
   *
   * **Note**: `catchAll` is fired for both public and private events. Please rely on the public events only.
   * @event catchAll
   * @param {Object} event The Object that contains event details
   * @param {String} event.type The type of the event which is caught by the listener
   */
  //endregion
  static get declarable() {
    return [
      /**
       * The list of deprecated events as an object, where `key` is an event name which is deprecated and
       * `value` is an object which contains values for
       * {@link Core.helper.VersionHelper#function-deprecate-static VersionHelper}:
       * - product {String} The name of the product;
       * - invalidAsOfVersion {String} The version where the offending code is invalid (when any compatibility
       *   layer is actually removed);
       * - message {String} Warning message to show to the developer using a deprecated API;
       *
       * For example:
       *
       * ```javascript
       * return {
       *     click : {
       *         product            : 'Grid',
       *         invalidAsOfVersion : '1.0.0',
       *         message            : 'click is deprecated!'
       *     }
       * }
       * ```
       *
       * @name deprecatedEvents
       * @returns {Object}
       * @static
       * @internal
       */
      "deprecatedEvents"
    ];
  }
  static setupDeprecatedEvents(cls2, meta) {
    const all = meta.getInherited("deprecatedEvents"), add = cls2.deprecatedEvents;
    for (const eventName in add) {
      all[eventName.toLowerCase()] = all[eventName] = add[eventName];
    }
  }
  //region Config
  static get configurable() {
    return {
      /**
       * Set to true to call onXXX method names (e.g. `onShow`, `onClick`), as an easy way to listen for events.
       *
       * ```javascript
       * const container = new Container({
       *     callOnFunctions : true
       *
       *     onHide() {
       *          // Do something when the 'hide' event is fired
       *     }
       * });
       * ```
       *
       * @config {Boolean} callOnFunctions
       * @category Misc
       * @default false
       */
      /**
       * The listener set for this object.
       *
       * An object whose property names are the names of events to handle, or options which modifiy
       * __how__ the handlers are called.
       *
       * See {@link #function-addListener} for details about the options.
       *
       * Listeners can be specified in target class config and they will be merged with any listeners specified in
       * the instantiation config. Class listeners will be fired first:
       *
       * ```javascript
       * class MyStore extends Store({
       *     static get configurable() {
       *         return {
       *             listeners : {
       *                 myCustomEvent() {
       *                 },
       *                 load : {
       *                     prio : 10000,
       *                     fn() { // this load listener handles things first }
       *                 }
       *             }
       *         }
       *     }
       * });
       *
       * let store = new MyStore({
       *   listeners: {
       *     load: () => { // This load listener runs after the class's },
       *     ...
       *   }
       * });
       * ```
       *
       * ### Handlers as function name
       *
       * Object event handlers may be specified as a function __name__. If a string is specified, it is the name
       * of the function in the `thisObj` object.
       *
       * If the string begins with `up.`, this object's ownership hierarchy
       * (if present) is scanned for an object which implements that function name:
       *
       * ```javascript
       * new Popup({
       *     tbar : {
       *         items : {
       *             myCombo : {
       *                 type      : 'combo',
       *                 editable  : false,
       *                 label     : 'Type',
       *                 listeners : {
       *                     // Look in owner chain for this function name
       *                     change : 'up.onFilterChange'
       *                 },
       *                 items     : [
       *                     'Event',
       *                     'Task',
       *                     'Appointment'
       *                 ]
       *             }
       *         }
       *     },
       *     items : {
       *         ...
       *     },
       *     onFilterChange({ value }) {
       *         // Handle event type selection here
       *     }
       * });
       *```
       *
       * @config {Object}
       * @category Common
       */
      listeners: {
        value: null,
        $config: {
          merge(newValue, currentValue) {
            if (newValue !== null) {
              if (!newValue) {
                return currentValue;
              }
              if (currentValue) {
                newValue = newValue ? [newValue] : [];
                newValue.push[isArray(currentValue) ? "apply" : "call"](newValue, currentValue);
              }
            }
            return newValue;
          }
        }
      },
      /**
       * Internal listeners, that cannot be removed by the user.
       * @config {Object}
       * @internal
       */
      internalListeners: null,
      /**
       * An object where property names with a truthy value indicate which events should bubble up the ownership
       * hierarchy when triggered.
       *
       * ```javascript
       * const container = new Container({
       *     items : [
       *        { type : 'text', bubbleEvents : { change : true }}
       *     ],
       *
       *     listeners : {
       *         change() {
       *             // Will catch change event from the text field
       *         }
       *     }
       * });
       * ```
       *
       * @config {Object}
       * @category Misc
       */
      bubbleEvents: null
    };
  }
  destroy() {
    this.trigger("beforeDestroy");
    super.destroy();
  }
  //endregion
  //region Init
  construct(config, ...args) {
    if (this.configuredListeners = config == null ? void 0 : config.listeners) {
      config = Objects.assign({}, config);
      delete config.listeners;
    }
    super.construct(config, ...args);
    this.processConfiguredListeners();
  }
  processConfiguredListeners() {
    if (this.configuredListeners) {
      const me = this, { isConfiguring } = me;
      me.isConfiguring = false;
      me.listeners = me.configuredListeners;
      me.configuredListeners = null;
      me.isConfiguring = isConfiguring;
    }
  }
  /**
   * Auto detaches listeners registered from start, if set as detachable
   * @internal
   */
  doDestroy() {
    this.trigger("destroy");
    this.removeAllListeners(false);
    super.doDestroy();
  }
  static setupClass(meta) {
    super.setupClass(meta);
    Events.prototype.onListen.$nullFn = true;
    Events.prototype.onUnlisten.$nullFn = true;
  }
  //endregion
  //region Listeners
  /**
   * Adds an event listener. This method accepts parameters in the following format:
   *
   * ```javascript
   *  myObject.addListener({
   *      thisObj    : this,          // The this reference for the handlers
   *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
   *      otherevent : {
   *          fn      : 'handlerFnName',
   *          once    : true          // Just this handler is auto-removed on fire
   *      },
   *      yetanother  : {
   *          fn      : 'yetAnotherHandler',
   *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
   *      },
   *      prio        : 100           // Higher prio listeners are called before lower
   *  });
   * ```
   *
   * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
   * When it is destroyed, those listeners are removed.
   *
   * The `config` parameter allows supplying options for the listener(s), for available options see {@link #typedef-BryntumListenerConfig}.
   *
   * A simpler signature may be used when only adding a listener for one event and no extra options
   * (such as `once` or `delay`) are required:
   *
   * ```javascript
   * myObject.addListener('click', myController.handleClicks, myController);
   * ```
   *
   * The args in this simple case are `eventName`, `handler` and `thisObj`
   *
   * @param {BryntumListenerConfig|String} config An object containing listener definitions, or the event name to listen for
   * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object, or the handler
   * function to call if providing a string as the first arg.
   * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
   * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
   */
  addListener(config, thisObj, oldThisObj) {
    var _a2;
    if (isArray(config)) {
      for (let i = 0, { length } = config; i < length; i++) {
        this.addListener(config[i], thisObj);
      }
      return;
    }
    const me = this, deprecatedEvents = me.$meta.getInherited("deprecatedEvents");
    if (typeof config === "string") {
      return me.addListener({
        [config]: thisObj,
        detachable: thisObj.detachable !== false,
        thisObj: oldThisObj
      });
    } else {
      thisObj = config.thisObj = config.thisObj !== void 0 ? config.thisObj : thisObj;
      for (const key in config) {
        if (!specialProperties[key] && config[key] != null) {
          const eventName = key.toLowerCase(), deprecatedEvent = deprecatedEvents == null ? void 0 : deprecatedEvents[eventName], events = me.eventListeners || (me.eventListeners = {}), listenerSpec = config[key], expires = listenerSpec.expires || config.expires, listener = {
            fn: typeof listenerSpec === "object" ? listenerSpec.fn : listenerSpec,
            thisObj: listenerSpec.thisObj !== void 0 ? listenerSpec.thisObj : thisObj,
            args: listenerSpec.args || config.args,
            prio: listenerSpec.prio !== void 0 ? listenerSpec.prio : config.prio !== void 0 ? config.prio : 0,
            once: listenerSpec.once !== void 0 ? listenerSpec.once : config.once !== void 0 ? config.once : false,
            buffer: listenerSpec.buffer || config.buffer,
            throttle: listenerSpec.throttle || config.throttle,
            $internal: config.$internal,
            catchAll: key === "catchAll"
          };
          if (deprecatedEvent) {
            const { product, invalidAsOfVersion, message } = deprecatedEvent;
            VersionHelper.deprecate(product, invalidAsOfVersion, message);
          }
          if (expires) {
            const { alt: alt2 } = expires, delay3 = alt2 ? expires.delay : expires, name2 = config.name || key, fn2 = () => {
              me.un(eventName, listener);
              if (alt2 && !listener.called) {
                me.callback(alt2, thisObj);
              }
            };
            if (me.isDelayable) {
              me.setTimeout({ fn: fn2, name: name2, cancelOutstanding: true, delay: delay3 });
            } else {
              globalThis.setTimeout(fn2, delay3);
            }
          }
          let listeners = events[eventName] || (events[eventName] = []);
          if (listeners.$firing) {
            events[eventName] = listeners = listeners.slice();
          }
          listeners.splice(
            ArrayHelper.findInsertionIndex(listener, listeners, priorityComparator, listeners.length),
            0,
            listener
          );
          if (!me.onListen.$nullFn && listeners.length < 2) {
            me.onListen(eventName);
          }
          (_a2 = me.afterAddListener) == null ? void 0 : _a2.call(me, eventName, listener);
        }
      }
      if (config.relayAll) {
        me.relayAll(config.relayAll);
      }
      if (thisObj && thisObj !== me) {
        me.attachAutoDetacher(config, thisObj);
      }
      const detachable = config.detachable !== false, name = config.name, destroy = config.expires || detachable || name ? () => {
        if (!me.isDestroyed) {
          me.removeListener(config, thisObj);
        }
      } : null;
      if (destroy) {
        destroy.eventer = me;
        destroy.listenerName = name;
        if (name && (thisObj == null ? void 0 : thisObj.trackDetacher)) {
          thisObj.trackDetacher(name, destroy);
        }
        if (config.expires) {
          me.delay(destroy, isNaN(config.expires) ? config.expires.delay : config.expires, name);
        }
        if (detachable) {
          return destroy;
        }
      }
    }
  }
  /**
   * Alias for {@link #function-addListener}. Adds an event listener. This method accepts parameters in the following format:
   *
   * ```javascript
   *  myObject.on({
   *      thisObj    : this,          // The this reference for the handlers
   *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
   *      otherevent : {
   *          fn      : 'handlerFnName',
   *          once    : true          // Just this handler is auto-removed on fire
   *      },
   *      yetanother  : {
   *          fn      : 'yetAnotherHandler',
   *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
   *      },
   *      prio        : 100           // Higher prio listeners are called before lower
   *  });
   * ```
   *
   * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
   * When it is destroyed, those listeners are removed.
   *
   * The `config` parameter allows supplying options for the listener(s), for available options see {@link #typedef-BryntumListenerConfig}.
   *
   * A simpler signature may be used when only adding a listener for one event and no extra options
   * (such as `once` or `delay`) are required:
   *
   * ```javascript
   * myObject.on('click', myController.handleClicks, myController);
   * ```
   *
   * The args in this simple case are `eventName`, `handler` and `thisObj`
   *
   * @param {BryntumListenerConfig|String} config An object containing listener definitions, or the event name to listen for
   * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object, or the handler
   * function to call if providing a string as the first arg.
   * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
   * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
   */
  on(config, thisObj, oldThisObj) {
    return this.addListener(config, thisObj, oldThisObj);
  }
  /**
   * Internal convenience method for adding an internal listener, that cannot be removed by the user.
   *
   * Alias for `on({ $internal : true, ... })`. Only supports single argument form.
   *
   * @internal
   */
  ion(config) {
    config.$internal = true;
    return this.on(config);
  }
  /**
   * Shorthand for {@link #function-removeListener}
   * @param {Object|String} config A config object or the event name
   * @param {Object|Function} [thisObj] `this` reference for all listeners, or the listener function
   * @param {Object} [oldThisObj] `this` The `this` object for the legacy way of adding listeners
   */
  un(...args) {
    this.removeListener(...args);
  }
  updateInternalListeners(internalListeners, oldInternalListeners) {
    oldInternalListeners == null ? void 0 : oldInternalListeners.detach();
    if (internalListeners) {
      internalListeners.detach = this.ion(internalListeners);
    }
  }
  get listeners() {
    return this.eventListeners;
  }
  changeListeners(listeners) {
    if (this.isConfiguring) {
      this.getConfig("internalListeners");
      if (listeners) {
        this.on(listeners, this);
      }
    } else {
      if (Array.isArray(listeners)) {
        for (let i = 0, l = listeners[0], { length } = listeners; i < length; l = listeners[++i]) {
          if (!("thisObj" in l)) {
            listeners[i] = Objects.assign({ thisObj: this }, l);
          }
        }
      } else if (listeners && !("thisObj" in listeners)) {
        listeners = Objects.assign({ thisObj: this }, listeners);
      }
      return listeners;
    }
  }
  updateListeners(listeners, oldListeners) {
    oldListeners && this.un(oldListeners);
    listeners && this.on(listeners);
  }
  /**
   * Removes an event listener. Same API signature as {@link #function-addListener}
   * @param {Object|String} config A config object or the event name
   * @param {Object|Function} thisObj `this` reference for all listeners, or the listener function
   * @param {Object} oldThisObj `this` The `this` object for the legacy way of adding listeners
   */
  removeListener(config, thisObj = config.thisObj, oldThisObj) {
    const me = this;
    if (typeof config === "string") {
      return me.removeListener({ [config]: thisObj }, oldThisObj);
    }
    Object.entries(config).forEach(([eventName, listenerToRemove]) => {
      var _a2;
      if (!specialProperties[eventName] && listenerToRemove != null) {
        eventName = eventName.toLowerCase();
        const { eventListeners } = me, index = me.findListener(eventName, listenerToRemove, thisObj);
        if (index >= 0) {
          let listeners = eventListeners[eventName];
          (_a2 = me.afterRemoveListener) == null ? void 0 : _a2.call(me, eventName, listeners[index]);
          if (listeners.length > 1) {
            if (listeners.$firing) {
              eventListeners[eventName] = listeners = listeners.slice();
            }
            listeners.splice(index, 1);
          } else {
            delete eventListeners[eventName];
            if (!me.onUnlisten.$nullFn) {
              me.onUnlisten(eventName);
            }
          }
        }
      }
    });
    if (config.thisObj && !config.thisObj.isDestroyed) {
      me.detachAutoDetacher(config);
    }
  }
  /**
   * Finds the index of a particular listener to the named event. Returns `-1` if the passed
   * function/thisObj listener is not present.
   * @param {String} eventName The name of an event to find a listener for.
   * @param {String|Function} listenerToFind The handler function to find.
   * @param {Object} defaultThisObj The `thisObj` for the required listener.
   * @internal
   */
  findListener(eventName, listenerToFind, defaultThisObj) {
    var _a2;
    const eventListeners = (_a2 = this.eventListeners) == null ? void 0 : _a2[eventName], fn2 = listenerToFind.fn || listenerToFind, thisObj = listenerToFind.thisObj || defaultThisObj;
    if (eventListeners) {
      for (let listenerEntry, i = 0, { length } = eventListeners; i < length; i++) {
        listenerEntry = eventListeners[i];
        if (listenerEntry.fn === fn2 && listenerEntry.thisObj === thisObj) {
          return i;
        }
      }
    }
    return -1;
  }
  /**
   * Check if any listener is registered for the specified eventName
   * @param {String} eventName
   * @returns {Boolean} `true` if listener is registered, otherwise `false`
   * @advanced
   */
  hasListener(eventName) {
    var _a2;
    return Boolean((_a2 = this.eventListeners) == null ? void 0 : _a2[eventName == null ? void 0 : eventName.toLowerCase()]);
  }
  /**
   * Relays all events through another object that also implements Events mixin. Adds a prefix to the event name
   * before relaying, for example add -> storeAdd
   * ```
   * // Relay all events from store through grid, will make it possible to listen for store events prefixed on grid:
   * 'storeLoad', 'storeChange', 'storeRemoveAll' etc.
   * store.relayAll(grid, 'store');
   *
   * grid.on('storeLoad', () => console.log('Store loaded');
   * ```
   * @param {Core.mixin.Events} through Object to relay the events through, needs to mix Events mixin in
   * @param {String} prefix Prefix to add to event name
   * @param {Boolean} [transformCase] Specify false to prevent making first letter of event name uppercase
   * @advanced
   */
  relayAll(through, prefix, transformCase = true) {
    if (!this.relayAllTargets) {
      this.relayAllTargets = [];
    }
    const { relayAllTargets } = this;
    through.ion({
      beforeDestroy: ({ source }) => {
        if (source === through) {
          const configs = relayAllTargets.filter((r) => r.through === through);
          configs.forEach((config) => ArrayHelper.remove(relayAllTargets, config));
        }
      }
    });
    relayAllTargets.push({ through, prefix, transformCase });
  }
  /**
   * Removes all listeners registered to this object by the application.
   */
  removeAllListeners(preserveInternal = true) {
    var _a2;
    const listeners = this.eventListeners;
    let i, thisObj;
    for (const event in listeners) {
      const bucket2 = listeners[event];
      for (i = bucket2.length; i-- > 0; ) {
        const cfg = bucket2[i];
        if (!cfg.$internal || !preserveInternal) {
          this.removeListener(event, cfg);
          thisObj = cfg.thisObj;
          (_a2 = thisObj == null ? void 0 : thisObj.untrackDetachers) == null ? void 0 : _a2.call(thisObj, this);
        }
      }
    }
  }
  relayEvents(source, eventNames, prefix = "") {
    const listenerConfig = { detachable: true, thisObj: this };
    eventNames.forEach((eventName) => {
      listenerConfig[eventName] = (event, ...params) => {
        return this.trigger(prefix + eventName, event, ...params);
      };
    });
    return source.on(listenerConfig);
  }
  /**
   * This method is called when the first listener for an event is added.
   * @param {String} eventName
   * @internal
   */
  onListen() {
  }
  /**
   * This method is called when the last listener for an event is removed.
   * @param {String} eventName
   * @internal
   */
  onUnlisten() {
  }
  destructorInterceptor() {
    const { autoDetachers, target, oldDestructor } = this;
    for (let i = 0; i < autoDetachers.length; i++) {
      const { dispatcher, config } = autoDetachers[i];
      if (!dispatcher.isDestroyed) {
        dispatcher.removeListener(config, target);
      }
    }
    oldDestructor.call(target);
  }
  /**
   * Internal function used to hook destroy() calls when using thisObj
   * @private
   */
  attachAutoDetacher(config, thisObj) {
    const target = config.thisObj || thisObj, destructorName = "doDestroy" in target ? "doDestroy" : "destroy";
    if (destructorName in target) {
      let { $autoDetachers } = target;
      if (!$autoDetachers) {
        target.$autoDetachers = $autoDetachers = [];
      }
      if (!target.$oldDestructor) {
        target.$oldDestructor = target[destructorName];
        target[destructorName] = this.destructorInterceptor.bind({
          autoDetachers: $autoDetachers,
          oldDestructor: target.$oldDestructor,
          target
        });
      }
      $autoDetachers.push({ config, dispatcher: this });
    } else {
      target[destructorName] = () => {
        this.removeListener(config);
      };
    }
  }
  /**
   * Internal function used restore hooked destroy() calls when using thisObj
   * @private
   */
  detachAutoDetacher(config) {
    const target = config.thisObj;
    if (target.$oldDestructor && !target.isDestroying) {
      ArrayHelper.remove(
        target.$autoDetachers,
        target.$autoDetachers.find((detacher2) => detacher2.config === config && detacher2.dispatcher === this)
      );
      if (!target.$autoDetachers.length) {
        target["doDestroy" in target ? "doDestroy" : "destroy"] = target.$oldDestructor;
        target.$oldDestructor = null;
      }
    }
  }
  //endregion
  //region Promise based workflow
  // experimental, used in tests to support async/await workflow
  await(eventName, options = { checkLog: true, resetLog: true, args: null }) {
    const me = this;
    if (options === false) {
      options = { checkLog: false };
    }
    const { args } = options;
    return new Promise((resolve) => {
      var _a2;
      if (options.checkLog && ((_a2 = me._triggered) == null ? void 0 : _a2[eventName])) {
        resolve();
        if (options.resetLog) {
          me.clearLog(eventName);
        }
      }
      if (args) {
        const detacher2 = me.on({
          [eventName]: (...params) => {
            const argsOk = typeof args === "function" ? args(...params) : Object.keys(args).every((key) => {
              return key in params[0] && params[0][key] === args[key];
            });
            if (argsOk) {
              resolve(...params);
              if (options.resetLog) {
                me.clearLog(eventName);
              }
              detacher2();
            }
          },
          prio: -1e4
          // Let others do their stuff first
        });
      } else {
        me.on({
          [eventName]: (...params) => {
            resolve(...params);
            if (options.resetLog) {
              me.clearLog(eventName);
            }
          },
          prio: -1e4,
          // Let others do their stuff first
          once: true
          // promises can only be resolved once anyway
        });
      }
    });
  }
  clearLog(eventName) {
    if (this._triggered) {
      if (eventName) {
        delete this._triggered[eventName];
      } else {
        this._triggered = {};
      }
    }
  }
  //endregion
  //region Trigger
  /**
   * Triggers an event, calling all registered listeners with the supplied arguments. Returning false from any listener
   * makes function return false.
   * @param {String} eventName Event name for which to trigger listeners
   * @param {Object} [param] Single parameter passed on to listeners, source property will be added to it (this)
   * @param {Boolean} [param.bubbles] Pass as `true` to indicate that the event will bubble up the widget
   * ownership hierarchy. For example up a `Menu`->`parent` Menu tree, or a `Field`->`Container` tree.
   * @typings param -> {{bubbles?: boolean, [key: string]: any}}
   * @returns {Boolean|Promise} Returns false if any listener returned `false`, or a `Promise` yielding
   * `true` / `false` based on what is returned from the async listener functions, otherwise `true`
   * @async
   * @advanced
   */
  trigger(eventName, param) {
    var _a2, _b, _c, _d;
    const me = this, name = eventName.toLowerCase(), {
      eventsSuspended,
      relayAllTargets,
      callOnFunctions
    } = me;
    let listeners = (_a2 = me.eventListeners) == null ? void 0 : _a2[name], handlerPromises;
    if (!me._triggered) {
      me._triggered = {};
    }
    me._triggered[eventName] = true;
    if (eventsSuspended) {
      if (eventsSuspended.shouldQueue) {
        eventsSuspended.queue.push(arguments);
      }
      return true;
    }
    if ((_b = me.eventListeners) == null ? void 0 : _b.catchall) {
      (listeners = listeners ? listeners.slice() : []).push(...me.eventListeners.catchall);
      listeners.sort(priorityComparator);
    }
    if (!listeners && !relayAllTargets && !callOnFunctions) {
      return true;
    }
    if (param) {
      if (!("source" in param)) {
        if (Object.isExtensible(param)) {
          param.source = me;
        } else {
          param = Object.setPrototypeOf({
            source: me
          }, param);
        }
      }
    } else {
      param = {
        source: me
      };
    }
    if (param.type !== name) {
      if (param.constructor !== Object) {
        Reflect.defineProperty(param, "type", { get: () => name });
      } else {
        param.type = name;
      }
    }
    param.eventName = eventName;
    if (!("bubbles" in param) && ((_c = me.bubbleEvents) == null ? void 0 : _c[eventName])) {
      param.bubbles = me.bubbleEvents[eventName];
    }
    if (callOnFunctions) {
      const fnName = "on" + StringHelper.capitalize(eventName);
      if (fnName in me) {
        const result = me[fnName] ? me.callback(me[fnName], me, [param]) : true;
        let inhibit;
        if (Objects.isPromise(result)) {
          (handlerPromises || (handlerPromises = [])).push(result);
        } else {
          inhibit = result === false || inhibit;
        }
        if (!me.isDestroyed && hasOwnProperty5.call(me, fnName) && !((_d = me.pluginFunctionChain) == null ? void 0 : _d[fnName])) {
          const myProto = Object.getPrototypeOf(me);
          if (fnName in myProto) {
            const result2 = myProto[fnName].call(me, param);
            if (Objects.isPromise(result2)) {
              (handlerPromises || (handlerPromises = [])).push(result2);
            } else {
              inhibit = result2 === false || inhibit;
            }
            if (me.isDestroyed) {
              return;
            }
          }
        }
        if (inhibit) {
          return false;
        }
      }
    }
    let ret;
    if (listeners) {
      let i = 0, internalAbort = false;
      listeners.$firing = true;
      for (i; i < listeners.length && !me.isDestroyed && !internalAbort; i++) {
        const listener = listeners[i];
        if (ret === false && !listener.$internal) {
          continue;
        }
        let handler, thisObj = listener.thisObj;
        if (!thisObj || !thisObj.isDestroyed) {
          listener.called = true;
          if (listener.once) {
            me.removeListener(name, listener);
          }
          if (typeof listener.fn === "string") {
            if (thisObj) {
              handler = thisObj[listener.fn];
            }
            if (!handler) {
              const result2 = me.resolveCallback(listener.fn);
              handler = result2.handler;
              thisObj = result2.thisObj;
            }
          } else {
            handler = listener.fn;
          }
          if (listener.buffer) {
            if (!listener.bufferFn) {
              const buffer = Number(listener.buffer);
              if (typeof buffer !== "number" || isNaN(buffer)) {
                throw new Error(`Incorrect type for buffer, got "${buffer}" (expected a Number)`);
              }
              listener.bufferFn = FunctionHelper.createBuffered(handler, buffer, thisObj, listener.args);
            }
            handler = listener.bufferFn;
          }
          if (listener.throttle) {
            const throttle = Number(listener.throttle);
            if (typeof throttle !== "number" || isNaN(throttle)) {
              throw new Error(`Incorrect type for throttle, got "${throttle}" (expected a Number)`);
            }
            if (!listener.throttledFn) {
              listener.throttledFn = FunctionHelper.createThrottled(handler, throttle, thisObj, listener.args);
            }
            handler = listener.throttledFn;
          }
          const result = handler.call(thisObj || me, ...listener.args || [], param);
          if (ret !== false) {
            ret = result;
          }
          if (listener.$internal && result === false) {
            internalAbort = true;
          }
          if (Objects.isPromise(result)) {
            result.$internal = listener.$internal;
            (handlerPromises || (handlerPromises = [])).push(result);
          }
        }
      }
      listeners.$firing = false;
      if (internalAbort) {
        return false;
      }
    }
    relayAllTargets == null ? void 0 : relayAllTargets.forEach((config) => {
      let name2 = eventName;
      if (config.transformCase) {
        name2 = StringHelper.capitalize(name2);
      }
      if (config.prefix) {
        name2 = config.prefix + name2;
      }
      if (config.through.trigger(name2, param) === false) {
        return false;
      }
    });
    if (param.bubbles && me.owner && !me.owner.isDestroyed) {
      return me.owner.trigger(eventName, param);
    }
    handlerPromises = handlerPromises == null ? void 0 : handlerPromises.filter((p) => ret !== false || p.$internal);
    if (handlerPromises == null ? void 0 : handlerPromises.length) {
      return new Promise((resolve) => {
        Promise.all(handlerPromises).then((promiseResults) => {
          const finalResult = !promiseResults.some((result) => result === false);
          resolve(finalResult);
        });
      });
    }
    return ret !== false;
  }
  /**
   * Prevents events from being triggered until {@link #function-resumeEvents()} is called. Optionally queues events that are triggered while
   * suspended. Multiple calls stack to require matching calls to `resumeEvents()` before actually resuming.
   * @param {Boolean} queue Specify true to queue events triggered while suspended
   * @advanced
   */
  suspendEvents(queue = false) {
    const eventsSuspended = this.eventsSuspended || (this.eventsSuspended = { shouldQueue: queue, queue: [], count: 0 });
    eventsSuspended.count++;
  }
  /**
   * Resume event triggering after a call to {@link #function-suspendEvents()}. If any triggered events were queued they will be triggered.
   * @returns {Boolean} `true` if events have been resumed (multiple calls to suspend require an equal number of resume calls to resume).
   * @advanced
   */
  resumeEvents() {
    const suspended = this.eventsSuspended;
    if (suspended) {
      if (--suspended.count === 0) {
        this.eventsSuspended = null;
        if (suspended.shouldQueue) {
          for (const queued of suspended.queue) {
            this.trigger(...queued);
          }
        }
      }
    }
    return !Boolean(this.eventsSuspended);
  }
  //endregion
};

// ../Core/lib/Core/helper/AsyncHelper.js
var AsyncHelper = class {
  /**
   * Returns a promise that resolves on next animation frame.
   * ```
   *  async method() {
   *      // do work
   *      await AsyncHelper.animationFrame();
   *      // do more work
   *  }
   * ```
   * @async
   */
  static animationFrame() {
    return new Promise((resolve) => {
      requestAnimationFrame(resolve);
    });
  }
  /**
   * Returns a promise that resolves after a specified number of milliseconds.
   * ```
   *  async method() {
   *      await AsyncHelper.sleep(10);
   *      // ...
   *  }
   * ```
   * @param {Number} millis The number of milliseconds to sleep.
   * @async
   */
  static sleep(millis) {
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve();
      }, millis);
    });
  }
  /**
   * Returns a promise that resolves as soon as possible, allowing the browser to minimally process other messages.
   * This is the shortest possible delay the browser offers, so be aware that it does not necessarily allow the
   * browser to paint or reflow if used in a long loop. It does, however, allow other async methods to execute.
   * ```
   *  async method() {
   *      await AsyncHelper.yield();
   *      // ...
   *  }
   * ```
   * @async
   */
  static yield() {
    return Promise.resolve();
  }
};
AsyncHelper._$name = "AsyncHelper";

// ../Core/lib/Core/helper/AjaxHelper.js
var paramValueRegExp = /^(\w+)=(.*)$/;
var parseParams = function(paramString) {
  const result = {}, params = paramString.split("&");
  for (const nameValuePair of params) {
    const [match, name, value] = paramValueRegExp.exec(nameValuePair), decodedName = decodeURIComponent(name), decodedValue = decodeURIComponent(value);
    if (match) {
      let paramValue = result[decodedName];
      if (paramValue) {
        if (!Array.isArray(paramValue)) {
          paramValue = result[decodedName] = [paramValue];
        }
        paramValue.push(decodedValue);
      } else {
        result[decodedName] = decodedValue;
      }
    }
  }
  return result;
};
var _AjaxHelper = class {
  /**
   * Make a request (using GET) to the specified url.
   * @param {String} url URL to `GET` from
   * @param {FetchOptions} [options] The options for the `fetch` API
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   * @async
   */
  static get(url, options) {
    return this.fetch(url, options);
  }
  /**
   * POST data to the specified URL.
   * @param {String} url URL to `POST` to
   * @param {String|Object|FormData} payload The data to post. If an object is supplied, it will be stringified
   * @param {FetchOptions} [options] The options for the `fetch` API
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   * @async
   */
  static post(url, payload, options = {}) {
    if (!(payload instanceof FormData) && !(typeof payload === "string")) {
      payload = JSON.stringify(payload);
      options.headers = options.headers || {};
      options.headers["Content-Type"] = options.headers["Content-Type"] || "application/json";
    }
    return this.fetch(url, Object.assign({
      method: "POST",
      body: payload
    }, options));
  }
  /**
   * Fetch the specified resource using the `fetch` API.
   * @param {String} url URL to fetch from
   * @param {FetchOptions} [options] The options for the `fetch` API
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   * @async
   */
  static fetch(url, options) {
    let controller;
    options = Objects.merge({}, _AjaxHelper.DEFAULT_FETCH_OPTIONS, options);
    if (typeof AbortController !== "undefined") {
      controller = options.abortController = new AbortController();
      options.signal = controller.signal;
    }
    if (!("credentials" in options)) {
      options.credentials = "include";
    }
    if (options.queryParams) {
      const params = Object.entries(options.queryParams);
      if (params.length) {
        let paramsAdded = false;
        if (options.headers && options.addQueryParamsToBody === true) {
          const contentType = new Headers(options.headers).get("Content-Type");
          let bodyClass;
          switch (contentType) {
            case "application/x-www-form-urlencoded":
              bodyClass = URLSearchParams;
              break;
            case "multipart/form-data":
              bodyClass = FormData;
              break;
          }
          if (bodyClass) {
            const body = options.body || (options.body = new bodyClass());
            if (body instanceof bodyClass) {
              params.forEach(([key, value]) => body.set(key, value));
              paramsAdded = true;
            }
          }
        }
        if (!paramsAdded) {
          url += (url.includes("?") ? "&" : "?") + params.map(
            ([param, value]) => `${param}=${encodeURIComponent(value)}`
          ).join("&");
        }
      }
    }
    const promise = new Promise((resolve, reject) => {
      fetch(url, options).then(
        (response) => {
          if (options.parseJson) {
            response.json().then((json) => {
              response.parsedJson = json;
              resolve(response);
            }).catch((error) => {
              response.parsedJson = null;
              response.error = error;
              reject(response);
            });
          } else {
            resolve(response);
          }
        }
      ).catch((error) => {
        error.stack = promise.stack;
        reject(error);
      });
    });
    promise.stack = new Error().stack;
    promise.abort = function() {
      controller == null ? void 0 : controller.abort();
    };
    return promise;
  }
  /**
   * Registers the passed URL to return the passed mocked up Fetch Response object to the
   * AjaxHelper's promise resolve function.
   * @param {String} url The url to return mock data for
   * @param {Object|Function} response A mocked up Fetch Response object which must contain
   * at least a `responseText` property, or a function to which the `url` and a `params` object
   * and the `Fetch` `options` object is passed which returns that.
   * @param {String} response.responseText The data to return.
   * @param {Boolean} [response.synchronous] resolve the Promise immediately
   * @param {Number} [response.delay=100] resolve the Promise after this number of milliseconds.
   */
  static mockUrl(url, response) {
    const me = this;
    (me.mockAjaxMap || (me.mockAjaxMap = {}))[url] = response;
    if (!_AjaxHelper.originalFetch) {
      _AjaxHelper.originalFetch = _AjaxHelper.fetch;
      _AjaxHelper.fetch = me.mockAjaxFetch.bind(me);
    }
  }
  static async mockAjaxFetch(url, options) {
    const urlAndParams = url.split("?");
    let result = this.mockAjaxMap[urlAndParams[0]], parsedJson = null;
    if (result) {
      if (typeof result === "function") {
        result = await result(urlAndParams[0], urlAndParams[1] && parseParams(urlAndParams[1]), options);
      }
      try {
        parsedJson = (options == null ? void 0 : options.parseJson) && JSON.parse(result.responseText);
      } catch (error) {
        parsedJson = null;
        result.error = error;
      }
      result = Object.assign({
        status: 200,
        ok: true,
        headers: new Headers(),
        statusText: "OK",
        url,
        parsedJson,
        text: () => new Promise((resolve) => {
          resolve(result.responseText);
        }),
        json: () => new Promise((resolve) => {
          resolve(parsedJson);
        })
      }, result);
      return new Promise(function(resolve, reject) {
        if (result.synchronous) {
          resolve(result);
        } else {
          setTimeout(function() {
            resolve(result);
          }, "delay" in result ? result.delay : 100);
        }
      });
    } else {
      return _AjaxHelper.originalFetch(url, options);
    }
  }
};
var AjaxHelper = _AjaxHelper;
/**
 * Sets default options for {@link #function-fetch-static AjaxHelper#fetch()} calls. Please see
 * {@link #typedef-FetchOptions} and
 * [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details.
 *
 * ```javascript
 * // default content-type for all requests will be "application/json"
 * AjaxHelper.DEFAULT_FETCH_OPTIONS = {
 *     headers : {
 *         'content-type' : 'application/json'
 *     }
 * };
 * ```
 * @member {FetchOptions} DEFAULT_FETCH_OPTIONS
 * @static
 */
__publicField(AjaxHelper, "DEFAULT_FETCH_OPTIONS", {});
AjaxHelper._$name = "AjaxHelper";

// ../Core/lib/Core/localization/LocaleHelper.js
var _LocaleHelper = class {
  /**
   * Merges all properties of provided locale objects into new locale object.
   * Locales are merged in order they provided and locales which go later replace
   * same properties of previous locales.
   * @param {...Object} locales Locales to merge
   * @returns {Object} Merged locale
   */
  static mergeLocales(...locales) {
    const result = {};
    locales.forEach((locale7) => {
      Object.keys(locale7).forEach((key) => {
        if (typeof locale7[key] === "object") {
          result[key] = { ...result[key], ...locale7[key] };
        } else {
          result[key] = locale7[key];
        }
      });
    });
    return result;
  }
  /**
   * Removes all properties from `locale` that are present in the provided `toTrim`.
   * @param {Object} locale Locale to process
   * @param {Object} toTrim Object enumerating properties that should be removed.
   * When `false` throws exceptions in such cases.
   */
  static trimLocale(locale7, toTrim) {
    const remove = (key, subKey) => {
      if (locale7[key]) {
        if (subKey) {
          if (locale7[key][subKey]) {
            delete locale7[key][subKey];
          }
        } else {
          delete locale7[key];
        }
      }
    };
    Object.keys(toTrim).forEach((key) => {
      if (Object.keys(toTrim[key]).length > 0) {
        Object.keys(toTrim[key]).forEach((subKey) => remove(key, subKey));
      } else {
        remove(key);
      }
    });
  }
  /**
   * Normalizes locale object to {@link Locale} type.
   *
   * Supported configs:
   *
   * ```javascript
   * LocaleHelper.normalizeLocale({
   *     localeName : 'En',
   *     localeDesc : 'English (US)',
   *     localeCode : 'en-US',
   *     ... (localization key:value pairs)
   * });
   * ```
   *
   * and for backward compatibility
   *
   * ```javascript
   * LocaleHelper.normalizeLocale('En', {
   *     name : 'En',
   *     desc : 'English (US)',
   *     code : 'en-US',
   *     locale : {
   *         ... (localization key:value pairs)
   *     }
   * });
   * ```
   * @param {String|Object} nameOrConfig String name of locale or locale object
   * @param {Object} [config] Locale object
   * @returns {Locale} Locale object
   * @internal
   */
  static normalizeLocale(nameOrConfig, config) {
    if (!nameOrConfig) {
      throw new Error(`"nameOrConfig" parameter can not be empty`);
    }
    if (typeof nameOrConfig === "string") {
      if (!config) {
        throw new Error(`"config" parameter can not be empty`);
      }
      if (config.locale) {
        config.name = nameOrConfig || config.name;
      } else {
        config.localeName = nameOrConfig;
      }
    } else {
      config = nameOrConfig;
    }
    let locale7 = {};
    if (config.name || config.locale) {
      locale7 = Object.assign({
        localeName: config.name
      }, config.locale);
      config.desc && (locale7.localeDesc = config.desc);
      config.code && (locale7.localeCode = config.code);
      config.path && (locale7.localePath = config.path);
    } else {
      if (!config.localeName) {
        throw new Error(`"config" parameter doesn't have "localeName" property`);
      }
      locale7 = Object.assign({}, config);
    }
    for (const key of ["name", "desc", "code", "path"]) {
      if (locale7[key]) {
        delete locale7[key];
      }
    }
    if (!locale7.localeName) {
      throw new Error(`Locale name can not be empty`);
    }
    return locale7;
  }
  /**
   * Get/set currently published locales.
   * Returns an object with locales.
   *
   * Example:
   *
   * ```javascript
   * const englishLocale = LocaleHelper.locales.En;
   * ```
   *
   * `englishLocale` contains {@link Locale} object.
   *
   * @readonly
   * @member {Locales} locales
   * @static
   */
  static get locales() {
    return globalThis.bryntum.locales || {};
  }
  static set locales(locales) {
    globalThis.bryntum.locales = locales;
  }
  /**
   * Get/set current locale name. Defaults to "En"
   * @member {String} localeName
   * @static
   */
  static get localeName() {
    return globalThis.bryntum.locale || "En";
  }
  static set localeName(localeName) {
    globalThis.bryntum.locale = localeName || _LocaleHelper.localeName;
  }
  /**
   * Get current locale config specified by {@link ##property-localeName-static}.
   * If no current locale specified, returns default `En` locale or first published locale
   * or empty locale object if no published locales found.
   * @readonly
   * @member {Locales} locale
   * @static
   */
  static get locale() {
    return _LocaleHelper.localeName && this.locales[_LocaleHelper.localeName] || this.locales.En || Object.values(this.locales)[0] || { localeName: "", localeDesc: "", localeCoode: "" };
  }
  /**
   * Publishes a locale to make it available for applying.
   * Published locales are available in {@link ##property-locales-static}.
   *
   * Recommended usage:
   *
   * ```javascript
   * LocaleHelper.publishLocale({
   *     localeName : 'En',
   *     localeDesc : 'English (US)',
   *     localeCode : 'en-US',
   *     ... (localization key:value pairs)
   * });
   * ```
   *
   * for backward compatibility (prior to `5.3.0` version):
   *
   * ```javascript
   * LocaleHelper.publishLocale('En', {
   *     name : 'En',
   *     desc : 'English (US)',
   *     code : 'en-US',
   *     locale : {
   *         ... (localization key:value pairs)
   *     }
   * });
   * ```
   *
   * Publishing a locale will automatically merge it's localization keys with existing locale matching by locale name,
   * replacing existing one with new. To replace existing locale entirely pass `true` to optional `config` parameter.
   *
   * Example:
   *
   * ```javascript
   * LocaleHelper.publishLocale({
   *     localeName : 'En',
   *     localeDesc : 'English (US)',
   *     localeCode : 'en-US',
   *     ... (localization key:value pairs)
   * }, true);
   * ```
   *
   * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object
   * @param {Locale|Boolean} [config] Locale object.
   * Not used if locale object is passed as first method parameter.
   * Path `true` value and locale object as first method parameter to publish locale without merging with existing one.
   * @returns {Locale} Locale object
   */
  static publishLocale(nameOrConfig, config) {
    const { locales } = globalThis.bryntum, locale7 = _LocaleHelper.normalizeLocale(nameOrConfig, config), { localeName } = locale7;
    if (!locales[localeName] || config === true) {
      locales[localeName] = locale7;
    } else {
      locales[localeName] = this.mergeLocales(locales[localeName] || {}, locale7 || {});
    }
    return locales[localeName];
  }
};
var LocaleHelper = _LocaleHelper;
__publicField(LocaleHelper, "skipLocaleIntegrityCheck", false);
globalThis.bryntum = globalThis.bryntum || {};
globalThis.bryntum.locales = globalThis.bryntum.locales || {};
LocaleHelper._$name = "LocaleHelper";

// ../Core/lib/Core/localization/LocaleManager.js
var LocaleManager = class extends Events_default(Base) {
  static get defaultConfig() {
    return {
      // Enable strict locale checking by default for tests
      throwOnMissingLocale: VersionHelper.isTestEnv
    };
  }
  construct(...args) {
    var _a2;
    const me = this;
    super.construct(...args);
    if (BrowserHelper.isBrowserEnv) {
      const scriptTag = document.querySelector("script[data-default-locale]");
      if (scriptTag) {
        me.applyLocale(scriptTag.dataset.defaultLocale);
      } else if ((_a2 = me.locale) == null ? void 0 : _a2.localeName) {
        me.applyLocale(me.locale.localeName);
      }
    }
  }
  /**
   * Get/set currently registered locales.
   * Alias for {@link Core.localization.LocaleHelper#property-locales-static LocaleHelper.locales}.
   * @readonly
   * @member {Locales} locales
   */
  get locales() {
    return LocaleHelper.locales;
  }
  set locales(locales) {
    LocaleHelper.locales = locales;
  }
  /**
   * Get/set currently used locale.
   * Setter calls {@link #function-applyLocale}.
   * @member {Locales} locale
   * @accepts {String|Locale}
   */
  set locale(nameOrConfig) {
    this.applyLocale(nameOrConfig);
  }
  get locale() {
    return LocaleHelper.locale;
  }
  /**
   * Publishes a locale to make it available for applying.
   * @deprecated Since 5.3.0. Use {@link Core.localization.LocaleHelper#function-publishLocale-static LocaleHelper.publishLocale} instead.
   *
   * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object.
   * @param {Locale} [config] Locale object. Not used if object is passed as first method parameter
   * @returns {Locale} published locale object is passed as first method parameter
   * @function registerLocale
   */
  registerLocale(nameOrConfig, config) {
    VersionHelper.deprecate("Core", "6.0.0", "LocaleManager.registerLocale deprecated, use LocaleHelper.publishLocale instead");
    LocaleHelper.publishLocale(nameOrConfig, config);
  }
  /**
   * Extends locale specified by name to add additional translations and applies it.
   * @deprecated Since 5.3.0. Use {@link ##function-applyLocale} instead.
   *
   * @param {String} name Name of locale (for example `En` or `SvSE`).
   * @param {Locale} config Locale object
   * @returns {Locale|Promise} locale object or Promise which resolves with locale object after it was loaded
   * @function extendLocale
   */
  extendLocale(name, config) {
    VersionHelper.deprecate("Core", "6.0.0", "LocaleManager.extendLocale deprecated, use LocaleManager.applyLocale instead");
    const locale7 = LocaleHelper.publishLocale(name, config);
    return this.applyLocale(locale7, true);
  }
  /**
   * Applies a locale by string name or publishes new locale configuration with
   * {@link Core.localization.LocaleHelper#function-publishLocale-static} and applies it.
   * If locale is specified by string name, like 'En', it must be published before applying it.
   *
   * @param {String|Locale} nameOrConfig String name of locale (for example `En` or `SvSE`) or locale object
   * @param {Locale|Boolean} [config] Locale object. Pass `true` to reapply locale which is passed as first method parameter.
   * @returns {Locale|Promise} locale object or Promise which resolves with locale object after it was loaded
   * @fires locale
   * @async
   * @function applyLocale
   */
  applyLocale(nameOrConfig, config, ignoreError = false) {
    const me = this;
    let localeConfig;
    if (typeof nameOrConfig === "string") {
      if (typeof config !== "object") {
        localeConfig = me.locales[nameOrConfig];
        if (!localeConfig) {
          if (ignoreError) {
            return true;
          }
          throw new Error(`Locale "${nameOrConfig}" is not published. Publish with LocaleHelper.publishLocale() before applying.`);
        }
      } else {
        localeConfig = LocaleHelper.publishLocale(nameOrConfig, config);
      }
    } else {
      localeConfig = LocaleHelper.publishLocale(nameOrConfig);
    }
    if (me.locale.localeName && me.locale.localeName === localeConfig.localeName && config !== true) {
      return me.locale;
    }
    LocaleHelper.localeName = localeConfig.localeName;
    const triggerLocaleEvent = () => {
      me.trigger("locale", localeConfig);
    };
    if (localeConfig.localePath) {
      return new Promise((resolve, reject) => {
        me.loadLocale(localeConfig.localePath).then((response) => {
          response.text().then((text) => {
            const parseLocale = new Function(text);
            parseLocale();
            if (BrowserHelper.isBrowserEnv) {
              localeConfig = me.locales[localeConfig.localeName];
              if (localeConfig) {
                delete localeConfig.localePath;
              }
            }
            triggerLocaleEvent();
            resolve(localeConfig);
          });
        }).catch((response) => reject(response));
      });
    }
    triggerLocaleEvent();
    return localeConfig;
  }
  /**
   * Loads a locale using AjaxHelper {@link Core.helper.AjaxHelper#function-get-static} request.
   * @private
   * @param {String} path Path to locale file
   * @async
   */
  loadLocale(path) {
    return AjaxHelper.get(path);
  }
  /**
   * Specifies if {@link Core.localization.Localizable#function-L-static Localizable.L()} function would throw error if no localization found at runtime.
   *
   * @member {Boolean} throwOnMissingLocale
   * @default false
   */
  set throwOnMissingLocale(value) {
    this._throwOnMissingLocale = value;
  }
  get throwOnMissingLocale() {
    return this._throwOnMissingLocale;
  }
};
var LocaleManagerSingleton = new LocaleManager();
var LocaleManager_default = LocaleManagerSingleton;

// ../Core/lib/Core/localization/Localizable.js
var ObjectProto = Object.getPrototypeOf(Object);
var localeRe = /L{.*?}/g;
var capturelocaleRe = /L{(.*?)}/g;
var classMatchRe = /((.*?)\.)?(.+)/g;
var escape2 = (txt) => txt.replace(/{(\d+)}/gm, "[[$1]]");
var unescape = (txt) => txt.replace(/\[\[(\d+)]]/gm, "{$1}");
var emptyObject3 = Object.freeze(/* @__PURE__ */ Object.create(null));
var Localizable_default = (Target) => class Localizable extends (Target || Base) {
  static get $name() {
    return "Localizable";
  }
  static get configurable() {
    return {
      /**
       * A class translations of which are used for translating this entity.
       * This is often used when translations of an item are defined on its container class.
       * For example:
       *
       * ```js
       * // Toolbar class that has some predefined items
       * class MyToolbar extends Toolbar {
       *
       *     static get $name() {
       *         return 'MyToolbar';
       *     }
       *
       *     static get defaultConfig() {
       *         return {
       *             // this specifies default configs for the items
       *             defaults : {
       *                 // will tell items to use the toolbar locale
       *                 localeClass : this
       *             },
       *
       *             items : [
       *                 // The toolbar has 2 buttons and translation for their texts will be searched in
       *                 // the toolbar locales
       *                 { text : 'Agree' },
       *                 { text : 'Disagree' }
       *             ]
       *         };
       *     }
       *
       *    ...
       * }
       * ```
       * So if one makes a locale for the `MyToolbar` class that will include `Agree` and `Disagree` string translations:
       * ```js
       *     ...
       *     MyToolbar : {
       *         Agree    : 'Yes, I agree',
       *         Disagree : 'No, I do not agree'
       *     }
       * ```
       * They will be used for the toolbar buttons and the button captions will say `Yes, I agree` and `No, I do not agree`.
       *
       * @config {Core.Base}
       * @typings {typeof Base}
       * @category Misc
       * @advanced
       */
      localeClass: null,
      /**
       * Set to `false` to disable localization of this object.
       * @config {Boolean}
       * @default true
       * @category Misc
       * @advanced
       */
      localizable: null,
      /**
       * List of properties which values should be translated automatically upon a locale applying.
       * In case there is a need to localize not typical value (not a String value or a field with re-defined setter/getter),
       * you could use 'localeKey' meta configuration.
       * Example:
       * ```js
       *  static get configurable() {
       *     return {
       *          localizableProperties : ['width'],
       *
       *          width : {
       *              value   : '54em', // default value here
       *              $config : {
       *                  localeKey : 'L{editorWidth}' // name of the property that will be used in localization file
       *              }
       *          }
       *      };
       *  }
       * ```
       * @config {String[]}
       * @category Misc
       * @advanced
       */
      localizableProperties: {
        value: [],
        $config: {
          merge: "distinct"
        }
      }
    };
  }
  static clsName(cls2) {
    var _a2, _b;
    return typeof cls2 === "string" ? cls2 : cls2 === ObjectProto ? "Object" : cls2.$$name || cls2.name || ((_a2 = cls2.prototype) == null ? void 0 : _a2.$$name) || ((_b = cls2.prototype) == null ? void 0 : _b.name);
  }
  static parseLocaleString(text) {
    const matches = [];
    let m;
    if (text == null ? void 0 : text.includes("L{")) {
      text = escape2(text);
      capturelocaleRe.lastIndex = 0;
      while ((m = capturelocaleRe.exec(text)) != null) {
        classMatchRe.lastIndex = 0;
        const classMatch = classMatchRe.exec(m[1]);
        matches.push({
          match: unescape(m[0]),
          localeKey: unescape(classMatch[3]),
          localeClass: classMatch[2]
        });
      }
    }
    return matches.length > 0 ? matches : [{
      match: text,
      localeKey: text,
      localeClass: void 0
    }];
  }
  construct(config = {}, ...args) {
    super.construct(config, ...args);
    LocaleManager_default.ion({ locale: "updateLocalization", thisObj: this });
    this.updateLocalization();
  }
  get localeClass() {
    return this._localeClass || null;
  }
  localizeProperty(property) {
    var _a2, _b, _c;
    const me = this, currentValue = Objects.getPath(me, property), localeKey = ((_a2 = me.$meta.configs[property]) == null ? void 0 : _a2.localeKey) || ((_c = (_b = me.fieldMap) == null ? void 0 : _b[property]) == null ? void 0 : _c.defaultValue);
    let localizedValue;
    if (localeKey) {
      localizedValue = Localizable.localize(localeKey, me, me.localeClass || me);
      if (localizedValue && !(property in (me.initialConfig || emptyObject3))) {
        Objects.setPath(me.isColumn ? me.data : me, property, localizedValue);
      }
    } else if (typeof currentValue === "string") {
      me.originalLocales = me.originalLocales || {};
      localizedValue = Objects.getPath(me.originalLocales, property);
      if (localizedValue === void 0) {
        Objects.setPath(me.originalLocales, property, currentValue);
        localizedValue = currentValue;
      }
      if (localizedValue) {
        Objects.setPath(me, property, localizedValue = me.optionalL(localizedValue, me));
      }
    }
    return localizedValue || currentValue;
  }
  /**
   * Method that is triggered when applying a locale to the instance
   * (happens on the instance construction steps and when switching to another locale).
   *
   * The method can be overridden to dynamically translate the instance when locale is switched.
   * When overriding the method please make sure you call `super.updateLocalization()`.
   * @category Misc
   * @advanced
   */
  updateLocalization() {
    var _a2, _b;
    if (this.localizable !== false) {
      (_a2 = this.localizableProperties) == null ? void 0 : _a2.forEach(this.localizeProperty, this);
      (_b = this.trigger) == null ? void 0 : _b.call(this, "localized");
    }
  }
  static getTranslation(text, templateData, localeCls) {
    const locale7 = LocaleManager_default.locale;
    let result = null, clsName, cls2;
    if (locale7) {
      for (const { match, localeKey, localeClass } of this.parseLocaleString(text)) {
        const translate = (clsName2) => {
          var _a2;
          const translation = (_a2 = locale7[clsName2]) == null ? void 0 : _a2[localeKey];
          if (translation) {
            if (typeof translation === "function") {
              result = templateData != null ? translation(templateData) : translation;
            } else if (typeof translation === "object" || text === match) {
              result = translation;
            } else {
              result = (result || text).replace(match, translation);
            }
            if (typeof translation === "string" && translation.includes("L{")) {
              result = this.getTranslation(translation, templateData, localeCls);
            }
          }
          return translation;
        };
        let success2 = false;
        for (cls2 = localeCls; cls2 && (clsName = Localizable.clsName(cls2)); cls2 = Object.getPrototypeOf(cls2)) {
          if (success2 = translate(clsName)) {
            break;
          } else if (typeof cls2 === "string") {
            break;
          }
        }
        if (!success2 && localeClass) {
          translate(localeClass);
        }
      }
    }
    return result;
  }
  /**
   * Get localized string, returns `null` if no localized string found.
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @internal
   */
  static localize(text, templateData = void 0, ...localeClasses) {
    if ((localeClasses == null ? void 0 : localeClasses.length) === 0) {
      localeClasses = [this];
    }
    let translation = null;
    localeClasses.some((cls2) => {
      translation = Localizable.getTranslation(text, templateData, cls2);
      return translation != null;
    });
    return translation;
  }
  /**
   * Get localized string, returns value of `text` if no localized string found.
   *
   * If {@link Core.localization.LocaleManager#property-throwOnMissingLocale LocaleManager.throwOnMissingLocale}
   * is `true` then calls to `L()` will throw `Localization is not found for 'text' in 'ClassName'` exception when no
   * localization is found.
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @static
   * @returns {String}
   * @advanced
   */
  static L(text, templateData = void 0, ...localeClasses) {
    if ((localeClasses == null ? void 0 : localeClasses.length) === 0) {
      localeClasses = [this];
    }
    const translation = this.localize(text, templateData, ...localeClasses);
    if (translation == null && LocaleManager_default.throwOnMissingLocale && text.includes("L{")) {
      throw new Error(`Localization is not found for '${text}' in '${localeClasses.map((cls2) => Localizable.clsName(cls2)).join(", ")}'. ${LocaleManager_default.locale.localeName ? `Locale : ${LocaleManager_default.locale.localeName}` : ""}`);
    }
    return translation != null ? translation : text;
  }
  /**
   * Convenience function that can be called directly on the class that mixes Localizable in
   *
   * ```javascript
   * button.text = grid.L('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @category Misc
   * @advanced
   */
  L(text, templateData) {
    const { localeClass, constructor } = this;
    if (localeClass && Localizable.clsName(localeClass) !== Localizable.clsName(constructor)) {
      return Localizable.L(text, templateData, localeClass, constructor);
    } else {
      return Localizable.L(text, templateData, constructor);
    }
  }
  /**
   * Convenience function to get an optional translation. The difference compared to `L()` is that it won't throw
   * an error when the translation is missing even if configured with `throwOnMissingLocale`
   *
   * ```javascript
   * button.text = grid.optionalL('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @static
   * @category Misc
   * @advanced
   */
  static optionalL(text, templateData = void 0, ...localeClasses) {
    const shouldThrow = LocaleManager_default.throwOnMissingLocale;
    LocaleManager_default.throwOnMissingLocale = shouldThrow && localeRe.test(text);
    if ((localeClasses == null ? void 0 : localeClasses.length) === 0) {
      localeClasses = [this];
    }
    const result = Localizable.L(text, templateData, ...localeClasses);
    LocaleManager_default.throwOnMissingLocale = shouldThrow;
    return result;
  }
  /**
   * Convenience function to get an optional translation. The difference compared to `L()` is that it won't throw
   * an error when the translation is missing even if configured with `throwOnMissingLocale`
   *
   * ```javascript
   * button.text = grid.optionalL('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @param {Boolean} [preventThrow] Prevent throwing error even if localized text matches `L{foo}`
   * @returns {String}
   * @category Misc
   * @internal
   */
  optionalL(text, templateData = this, preventThrow = false) {
    const shouldThrow = LocaleManager_default.throwOnMissingLocale;
    LocaleManager_default.throwOnMissingLocale = shouldThrow && localeRe.test(text) && !preventThrow;
    const result = this.L(text, templateData);
    LocaleManager_default.throwOnMissingLocale = shouldThrow;
    return result;
  }
  /**
   * Get the global LocaleManager
   * @property {Core.localization.LocaleManager}
   * @typings {typeof LocaleManager}
   * @category Misc
   * @readonly
   * @advanced
   */
  get localeManager() {
    return LocaleManager_default;
  }
  /**
   * Get the global LocaleHelper
   * @property {Core.localization.LocaleHelper}
   * @typings {typeof LocaleHelper}
   * @category Misc
   * @readonly
   * @advanced
   */
  get localeHelper() {
    return LocaleHelper;
  }
};

// ../Core/lib/Core/localization/En.js
var locale = {
  localeName: "En",
  localeDesc: "English (US)",
  localeCode: "en-US",
  Object: {
    Yes: "Yes",
    No: "No",
    Cancel: "Cancel",
    Ok: "OK",
    Week: "Week"
  },
  ColorPicker: {
    noColor: "No color"
  },
  Combo: {
    noResults: "No results",
    recordNotCommitted: "Record could not be added",
    addNewValue: (value) => `Add ${value}`
  },
  FilePicker: {
    file: "File"
  },
  Field: {
    badInput: "Invalid field value",
    patternMismatch: "Value should match a specific pattern",
    rangeOverflow: (value) => `Value must be less than or equal to ${value.max}`,
    rangeUnderflow: (value) => `Value must be greater than or equal to ${value.min}`,
    stepMismatch: "Value should fit the step",
    tooLong: "Value should be shorter",
    tooShort: "Value should be longer",
    typeMismatch: "Value is required to be in a special format",
    valueMissing: "This field is required",
    invalidValue: "Invalid field value",
    minimumValueViolation: "Minimum value violation",
    maximumValueViolation: "Maximum value violation",
    fieldRequired: "This field is required",
    validateFilter: "Value must be selected from the list"
  },
  DateField: {
    invalidDate: "Invalid date input"
  },
  DatePicker: {
    gotoPrevYear: "Go to previous year",
    gotoPrevMonth: "Go to previous month",
    gotoNextMonth: "Go to next month",
    gotoNextYear: "Go to next year"
  },
  NumberFormat: {
    locale: "en-US",
    currency: "USD"
  },
  DurationField: {
    invalidUnit: "Invalid unit"
  },
  TimeField: {
    invalidTime: "Invalid time input"
  },
  TimePicker: {
    hour: "Hour",
    minute: "Minute",
    second: "Second"
  },
  List: {
    loading: "Loading...",
    selectAll: "Select All"
  },
  GridBase: {
    loadMask: "Loading...",
    syncMask: "Saving changes, please wait..."
  },
  PagingToolbar: {
    firstPage: "Go to first page",
    prevPage: "Go to previous page",
    page: "Page",
    nextPage: "Go to next page",
    lastPage: "Go to last page",
    reload: "Reload current page",
    noRecords: "No records to display",
    pageCountTemplate: (data) => `of ${data.lastPage}`,
    summaryTemplate: (data) => `Displaying records ${data.start} - ${data.end} of ${data.allCount}`
  },
  PanelCollapser: {
    Collapse: "Collapse",
    Expand: "Expand"
  },
  Popup: {
    close: "Close"
  },
  UndoRedo: {
    Undo: "Undo",
    Redo: "Redo",
    UndoLastAction: "Undo last action",
    RedoLastAction: "Redo last undone action",
    NoActions: "No items in the undo queue"
  },
  FieldFilterPicker: {
    equals: "equals",
    doesNotEqual: "does not equal",
    isEmpty: "empty",
    isNotEmpty: "not empty",
    contains: "contains",
    doesNotContain: "does not contain",
    startsWith: "starts with",
    endsWith: "ends with",
    isOneOf: "one of",
    isNotOneOf: "not one of",
    isGreaterThan: "greater than",
    isLessThan: "less than",
    isGreaterThanOrEqualTo: "greater than or equal to",
    isLessThanOrEqualTo: "less than or equal to",
    isBetween: "between",
    isNotBetween: "not between",
    isBefore: "before",
    isAfter: "after",
    isToday: "today",
    isTomorrow: "tomorrow",
    isYesterday: "yesterday",
    isThisWeek: "this week",
    isNextWeek: "next week",
    isLastWeek: "last week",
    isThisMonth: "this month",
    isNextMonth: "next month",
    isLastMonth: "last month",
    isThisYear: "this year",
    isNextYear: "next year",
    isLastYear: "last year",
    isYearToDate: "year to date",
    isTrue: "true",
    isFalse: "false",
    selectAProperty: "Select a property",
    selectAnOperator: "Select an operator",
    caseSensitive: "Case-sensitive",
    and: "and",
    dateFormat: "D/M/YY",
    selectOneOrMoreValues: "Select one or more values",
    enterAValue: "Enter a value",
    enterANumber: "Enter a number",
    selectADate: "Select a date"
  },
  FieldFilterPickerGroup: {
    addFilter: "Add filter"
  },
  DateHelper: {
    locale: "en-US",
    weekStartDay: 0,
    nonWorkingDays: {
      0: true,
      6: true
    },
    weekends: {
      0: true,
      6: true
    },
    unitNames: [
      { single: "millisecond", plural: "ms", abbrev: "ms" },
      { single: "second", plural: "seconds", abbrev: "s" },
      { single: "minute", plural: "minutes", abbrev: "min" },
      { single: "hour", plural: "hours", abbrev: "h" },
      { single: "day", plural: "days", abbrev: "d" },
      { single: "week", plural: "weeks", abbrev: "w" },
      { single: "month", plural: "months", abbrev: "mon" },
      { single: "quarter", plural: "quarters", abbrev: "q" },
      { single: "year", plural: "years", abbrev: "yr" },
      { single: "decade", plural: "decades", abbrev: "dec" }
    ],
    unitAbbreviations: [
      ["mil"],
      ["s", "sec"],
      ["m", "min"],
      ["h", "hr"],
      ["d"],
      ["w", "wk"],
      ["mo", "mon", "mnt"],
      ["q", "quar", "qrt"],
      ["y", "yr"],
      ["dec"]
    ],
    parsers: {
      L: "MM/DD/YYYY",
      LT: "HH:mm A",
      LTS: "HH:mm:ss A"
    },
    ordinalSuffix: (number) => {
      const hasSpecialCase = ["11", "12", "13"].find((n) => number.endsWith(n));
      let suffix = "th";
      if (!hasSpecialCase) {
        const lastDigit = number[number.length - 1];
        suffix = { 1: "st", 2: "nd", 3: "rd" }[lastDigit] || "th";
      }
      return number + suffix;
    }
  }
};
var En_default = LocaleHelper.publishLocale(locale);

// ../Core/lib/Core/helper/DateHelper.js
var { toString: toString3 } = Object.prototype;
var DATE_TYPE2 = toString3.call(/* @__PURE__ */ new Date());
var tempDate = /* @__PURE__ */ new Date();
var MS_PER_HOUR = 1e3 * 60 * 60;
var defaultValue = (value, defValue) => isNaN(value) || value == null ? defValue : value;
var rangeFormatPartRe = /([ES]){([^}]+)}/g;
var enOrdinalSuffix = (number) => {
  const hasSpecialCase = ["11", "12", "13"].find((n) => number.endsWith(n));
  let suffix = "th";
  if (!hasSpecialCase) {
    const lastDigit = number[number.length - 1];
    suffix = { 1: "st", 2: "nd", 3: "rd" }[lastDigit] || "th";
  }
  return number + suffix;
};
var useIntlFormat = (name, options, date) => {
  const formatter = intlFormatterCache[name] || (intlFormatterCache[name] = new Intl.DateTimeFormat(locale2, options));
  return formatter.format(date);
};
var formatTime = (name, options, date, isShort = false) => {
  let strTime = useIntlFormat(name, options, date);
  if (/am|pm/i.test(strTime)) {
    strTime = strTime.replace(/^0/, "");
    if (isShort) {
      strTime = strTime.replace(/:00/, "");
    }
  }
  return strTime;
};
var getDayDiff = (end, start) => Math.floor((end.getTime() - start.getTime() - (end.getTimezoneOffset() - start.getTimezoneOffset()) * validConversions.minute.millisecond) / validConversions.day.millisecond) + 1;
var normalizeDay = (day2) => day2 >= 0 ? day2 : day2 + 7;
var msRegExp = /([^\w])(S+)/gm;
var msReplacer = (match, g1) => g1 + "SSS";
var splitRegExp = /[:.\-/\s]/;
var locale2 = "en-US";
var ordinalSuffix = enOrdinalSuffix;
var formatCache = {};
var formatRedirects = {};
var intlFormatterCache = {};
var parserCache = {};
var redirectFormat = (format2) => {
  const intlConfig = intlFormatConfigs[format2];
  if (!intlConfig) {
    throw new Error("Only international formats should be used here");
  }
  if (formatRedirects[format2] !== void 0) {
    return formatRedirects[format2];
  }
  const intl = new Intl.DateTimeFormat(locale2, intlConfig), fmt = intl.formatToParts(new Date(2001, 1, 2, 3, 4, 5, 6)).map((part) => {
    const type = part.type, intlCfg = intlConfig[type];
    if (type === "literal") {
      return part.value.replace(/,/g, "");
    } else if (type === "day") {
      return intlCfg === "numeric" ? "D" : "DD";
    } else if (type === "month") {
      return intlCfg === "short" ? "MMM" : intlCfg === "long" ? "MMMM" : intlCfg === "numeric" ? "M" : "MM";
    } else if (type === "year") {
      return intlCfg === "numeric" ? "YYYY" : "YY";
    }
  }).join("");
  return formatRedirects[format2] = fmt;
};
var DEFAULT_YEAR = 2020;
var DEFAULT_MONTH = 0;
var DEFAULT_DAY = 1;
var intlFormatConfigs = {
  l: { year: "numeric", month: "numeric", day: "numeric" },
  ll: { year: "numeric", month: "short", day: "numeric" }
};
var formats = {
  // 1, 2, ... 11, 12
  M: (date) => date.getMonth() + 1,
  //date.toLocaleDateString(locale, { month : 'numeric' }),
  // 1st, 2nd, 3rd, 4th, ... 11th, 12th
  Mo: (date) => ordinalSuffix(formats.M(date).toString()),
  // 01, 02, ...
  MM: (date) => (date.getMonth() + 1).toString().padStart(2, "0"),
  //date.toLocaleDateString(locale, { month : '2-digit' }),
  // Jan, Feb, ...
  MMM: (date) => useIntlFormat("MMM", { month: "short" }, date),
  // January, February, ...
  MMMM: (date) => useIntlFormat("MMMM", { month: "long" }, date),
  // 1, 2, ...
  Q: (date) => Math.ceil((date.getMonth() + 1) / 3),
  // 1st, 2nd, ...
  Qo: (date) => ordinalSuffix(formats.Q(date).toString()),
  // 1, 2, ...
  D: (date) => date.getDate(),
  //date.toLocaleDateString(locale, { day : 'numeric' }),
  // 1st, 2nd, ...
  Do: (date) => ordinalSuffix(formats.D(date).toString()),
  // 01, 02, ...
  DD: (date) => date.getDate().toString().padStart(2, "0"),
  //date.toLocaleDateString(locale, { day : '2-digit' }),
  // 1, 2, ..., 365, 365
  DDD: (date) => Math.ceil(
    (new Date(date.getFullYear(), date.getMonth(), date.getDate(), 12, 0, 0) - new Date(date.getFullYear(), 0, 0, 12, 0, 0)) / validConversions.day.millisecond
  ),
  // 1st, 2nd, ...
  DDDo: (date) => ordinalSuffix(formats.DDD(date).toString()),
  // 001, 002, ...
  DDDD: (date) => formats.DDD(date).toString().padStart(3, "0"),
  // 0, 1, ..., 6
  d: (date) => date.getDay(),
  // 0th, 1st, ...
  do: (date) => ordinalSuffix(date.getDay().toString()),
  // S, M, ...
  d1: (date) => useIntlFormat("d1", { weekday: "narrow" }, date).substr(0, 1),
  // Su, Mo, ...
  dd: (date) => formats.ddd(date).substring(0, 2),
  // Sun, Mon, ...
  ddd: (date) => useIntlFormat("ddd", { weekday: "short" }, date),
  // Sunday, Monday, ...
  dddd: (date) => useIntlFormat("dddd", { weekday: "long" }, date),
  u: (date) => {
    const formatter = intlFormatterCache.u || (intlFormatterCache.u = new Intl.DateTimeFormat("en-GB", {
      timeZone: "UTC",
      year: "numeric",
      month: "2-digit",
      day: "2-digit"
    })), parts = formatter.formatToParts(date);
    return `${parts[4].value}${parts[2].value}${parts[0].value}Z`;
  },
  uu: (date) => {
    const formatter = intlFormatterCache.uu || (intlFormatterCache.uu = new Intl.DateTimeFormat("en-GB", {
      timeZone: "UTC",
      hour12: false,
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit"
    })), parts = formatter.formatToParts(date);
    return `${parts[4].value}${parts[2].value}${parts[0].value}T${parts[6].value}${parts[8].value}${parts[10].value}Z`;
  },
  e: (date) => date.getDay(),
  E: (date) => date.getDay() + 1,
  // ISO week, 1, 2, ...
  W: (date) => DateHelper.getWeekNumber(date)[1],
  Wo: (date) => ordinalSuffix(formats.W(date).toString()),
  WW: (date) => formats.W(date).toString().padStart(2, "0"),
  // ISO week, 1, 2, ... with localized 'Week ' prefix
  Wp: (date) => `${DateHelper.localize("L{Week}")} ${formats.W(date)}`,
  WWp: (date) => `${DateHelper.localize("L{Week}")} ${formats.WW(date)}`,
  Wp0: (date) => `${DateHelper.localize("L{Week}")[0]}${formats.W(date)}`,
  WWp0: (date) => `${DateHelper.localize("L{Week}")[0]}${formats.WW(date)}`,
  // 1979, 2018
  Y: (date) => date.getFullYear(),
  //date.toLocaleDateString(locale, { year : 'numeric' }),
  // 79, 18
  YY: (date) => (date.getFullYear() % 100).toString().padStart(2, "0"),
  //date.toLocaleDateString(locale, { year : '2-digit' }),
  // 1979, 2018
  YYYY: (date) => date.getFullYear(),
  //date.toLocaleDateString(locale, { year : 'numeric' }),
  // AM, PM
  A: (date) => date.getHours() < 12 ? "AM" : "PM",
  a: (date) => date.getHours() < 12 ? "am" : "pm",
  // 0, 1, ... 23
  H: (date) => date.getHours(),
  // 00, 01, ...
  HH: (date) => date.getHours().toString().padStart(2, "0"),
  // 1, 2, ... 12
  h: (date) => date.getHours() % 12 || 12,
  // 01, 02, ...
  hh: (date) => formats.h(date).toString().padStart(2, "0"),
  // 1, 2, ... 24
  k: (date) => date.getHours() || 24,
  // 01, 02, ...
  kk: (date) => formats.k(date).toString().padStart(2, "0"),
  // Locale specific (0 -> 24 or 1 AM -> 12 PM)
  K: (date) => formatTime("K", { hour: "numeric" }, date),
  // Locale specific (00 -> 24 or 1 AM -> 12 PM)
  KK: (date) => formatTime("KK", { hour: "2-digit" }, date),
  // 0, 1, ... 59
  m: (date) => date.getMinutes(),
  //date.toLocaleTimeString(locale, { minute : 'numeric' }),
  // 00, 01, ...
  mm: (date) => formats.m(date).toString().padStart(2, "0"),
  // 0, 1, ... 59
  s: (date) => date.getSeconds(),
  //date.toLocaleTimeString(locale, { second : 'numeric' }),
  // 00, 01, ...
  ss: (date) => formats.s(date).toString().padStart(2, "0"),
  // 0, 1, ... 9 which are 000, 100, 200 ... 900 in milliseconds
  S: (date) => Math.floor(date.getMilliseconds() / 100).toString(),
  // 00, 01, ... 99 which are 000, 010, 020 ... 990 in milliseconds
  SS: (date) => Math.floor(date.getMilliseconds() / 10).toString().padStart(2, "0"),
  // 000, 001, ... 999 in milliseconds
  SSS: (date) => date.getMilliseconds().toString().padStart(3, "0"),
  z: (date) => useIntlFormat("z", { timeZoneName: "short" }, date),
  zz: (date) => useIntlFormat("zz", { timeZoneName: "long" }, date),
  Z: (date) => DH.getGMTOffset(date),
  LT: (date) => formatTime("LT", { hour: "2-digit", minute: "2-digit" }, date),
  // if minutes is 0, doesn't show it
  LST: (date) => formatTime("LST", { hour: "numeric", minute: "2-digit" }, date, true),
  LTS: (date) => formatTime("LTS", { hour: "2-digit", minute: "2-digit", second: "2-digit" }, date),
  L: (date) => useIntlFormat("L", { year: "numeric", month: "2-digit", day: "2-digit" }, date),
  l: (date) => useIntlFormat("l", intlFormatConfigs.l, date),
  LL: (date) => useIntlFormat("LL", { year: "numeric", month: "long", day: "numeric" }, date),
  ll: (date) => useIntlFormat("ll", intlFormatConfigs.ll, date),
  LLL: (date) => useIntlFormat("LLL", {
    year: "numeric",
    month: "long",
    day: "numeric",
    hour: "numeric",
    minute: "2-digit"
  }, date),
  lll: (date) => useIntlFormat("lll", {
    year: "numeric",
    month: "short",
    day: "numeric",
    hour: "numeric",
    minute: "2-digit"
  }, date),
  LLLL: (date) => useIntlFormat("LLLL", {
    year: "numeric",
    month: "long",
    day: "numeric",
    hour: "numeric",
    minute: "2-digit",
    weekday: "long"
  }, date),
  llll: (date) => useIntlFormat("llll", {
    year: "numeric",
    month: "short",
    day: "numeric",
    hour: "numeric",
    minute: "2-digit",
    weekday: "short"
  }, date)
};
var formatKeys = Object.keys(formats).sort((a, b) => b.length - a.length);
var formatRegexp = `^(?:${formatKeys.join("|")})`;
var emptyFn2 = () => ({});
var isNumber = (str) => numberRegex.test(str);
var parseMilliseconds = (str) => isNumber(str) && { milliseconds: parseInt(str.padEnd(3, "0").substring(0, 3)) };
var parsers = {
  YYYY: (str) => {
    const year = parseInt(str);
    return { year: year >= 1e3 && year <= 9999 ? year : NaN };
  },
  Y: (str) => ({ year: parseInt(str) }),
  YY: (str) => {
    const year = parseInt(str);
    return { year: year + (year > 1968 ? 1900 : 2e3) };
  },
  M: (str) => ({ month: parseInt(str) - 1 }),
  MM: (str) => ({ month: parseInt(str) - 1 }),
  Mo: (str) => ({ month: parseInt(str) - 1 }),
  MMM: (str) => {
    const month2 = (str || "").toLowerCase();
    for (const [name, entry] of Object.entries(DateHelper._monthShortNamesIndex)) {
      if (month2.startsWith(name)) {
        return { month: entry.value };
      }
    }
  },
  MMMM: (str) => {
    const month2 = (str || "").toLowerCase();
    for (const [name, entry] of Object.entries(DateHelper._monthNamesIndex)) {
      if (month2.startsWith(name)) {
        return { month: entry.value };
      }
    }
  },
  DD: (str) => ({ date: parseInt(str) }),
  D: (str) => ({ date: parseInt(str) }),
  Do: (str) => ({ date: parseInt(str) }),
  DDD: emptyFn2,
  DDDo: emptyFn2,
  DDDD: emptyFn2,
  d: emptyFn2,
  do: emptyFn2,
  d1: emptyFn2,
  dd: emptyFn2,
  ddd: emptyFn2,
  dddd: emptyFn2,
  Q: emptyFn2,
  Qo: emptyFn2,
  W: emptyFn2,
  Wo: emptyFn2,
  WW: emptyFn2,
  e: emptyFn2,
  E: emptyFn2,
  HH: (str) => ({ hours: parseInt(str) }),
  hh: (str) => ({ hours: parseInt(str) }),
  mm: (str) => ({ minutes: parseInt(str) }),
  H: (str) => ({ hours: parseInt(str) }),
  m: (str) => ({ minutes: parseInt(str) }),
  ss: (str) => ({ seconds: parseInt(str) }),
  s: (str) => ({ seconds: parseInt(str) }),
  S: parseMilliseconds,
  SS: parseMilliseconds,
  SSS: parseMilliseconds,
  A: (str) => ({ amPm: str.toLowerCase() }),
  a: (str) => ({ amPm: str.toLowerCase() }),
  L: "MM/DD/YYYY",
  LT: "HH:mm A",
  LTS: "HH:mm:ss A",
  l: { type: "dynamic", parser: () => redirectFormat("l") },
  ll: { type: "dynamic", parser: () => redirectFormat("ll") },
  // Can either be Z (=UTC, 0) or +-HH:MM
  Z: (str) => {
    if (!str || !timeZoneRegEx.test(str) && str !== "Z") {
      return null;
    }
    let timeZone = 0;
    if (str !== "Z") {
      const matches = timeZoneRegEx.exec(str);
      if (matches) {
        const sign = matches[1] === "+" ? 1 : -1, hours = parseInt(matches[2]) || 0, minutes = parseInt(matches[3]) || 0;
        timeZone = sign * (hours * 60 + minutes);
      } else {
        timeZone = -1 * (/* @__PURE__ */ new Date()).getTimezoneOffset();
      }
    }
    return { timeZone };
  }
};
var parserKeys = Object.keys(parsers).sort((a, b) => b.length - a.length);
var parserRegexp = new RegExp(`(${parserKeys.join("|")})`);
var localeStrRegExp = new RegExp("^(LL|LLL|lll|LLLL|llll)$");
var validConversions = {
  // The units below assume:
  // 30 days in a month, 91 days for a quarter and 365 for a year
  // 52 weeks per year, 4 per month, 13 per quarter
  // 3652 days per decade (assuming two of the years will be leap with 366 days)
  decade: {
    decade: 1,
    year: 10,
    quarter: 40,
    month: 120,
    week: 520,
    day: 3652,
    hour: 24 * 3652,
    minute: 1440 * 3652,
    second: 86400 * 3652,
    millisecond: 864e5 * 3652
  },
  year: {
    decade: 0.1,
    year: 1,
    quarter: 4,
    month: 12,
    week: 52,
    day: 365,
    hour: 24 * 365,
    minute: 1440 * 365,
    second: 86400 * 365,
    millisecond: 864e5 * 365
  },
  quarter: {
    decade: 1 / 40,
    year: 1 / 4,
    quarter: 1,
    month: 3,
    week: 4,
    day: 91,
    hour: 24 * 91,
    minute: 1440 * 91,
    second: 86400 * 91,
    millisecond: 864e5 * 91
  },
  month: {
    decade: 1 / 120,
    year: 1 / 12,
    quarter: 1 / 3,
    month: 1,
    week: 4,
    day: -30,
    hour: -24 * 30,
    minute: -1440 * 30,
    second: -86400 * 30,
    millisecond: -864e5 * 30
  },
  week: {
    decade: -1 / 520,
    year: -1 / 52,
    quarter: -1 / 13,
    month: -1 / 4,
    day: 7,
    hour: 168,
    minute: 10080,
    second: 604800,
    millisecond: 6048e5
  },
  day: {
    decade: -1 / 3652,
    year: -1 / 365,
    quarter: -1 / 91,
    month: -1 / 30,
    week: 1 / 7,
    hour: 24,
    minute: 1440,
    second: 86400,
    millisecond: 864e5
  },
  hour: {
    decade: -1 / (3652 * 24),
    year: -1 / (365 * 24),
    quarter: -1 / (91 * 24),
    month: -1 / (30 * 24),
    week: 1 / 168,
    day: 1 / 24,
    minute: 60,
    second: 3600,
    millisecond: 36e5
  },
  minute: {
    decade: -1 / (3652 * 1440),
    year: -1 / (365 * 1440),
    quarter: -1 / (91 * 1440),
    month: -1 / (30 * 1440),
    week: 1 / 10080,
    day: 1 / 1440,
    hour: 1 / 60,
    second: 60,
    millisecond: 6e4
  },
  second: {
    decade: -1 / (3652 * 86400),
    year: -1 / (365 * 86400),
    quarter: -1 / (91 * 86400),
    month: -1 / (30 * 86400),
    week: 1 / 604800,
    day: 1 / 86400,
    hour: 1 / 3600,
    minute: 1 / 60,
    millisecond: 1e3
  },
  millisecond: {
    decade: -1 / (3652 * 864e5),
    year: -1 / (365 * 864e5),
    quarter: -1 / (91 * 864e5),
    month: -1 / (30 * 864e5),
    week: 1 / 6048e5,
    day: 1 / 864e5,
    hour: 1 / 36e5,
    minute: 1 / 6e4,
    second: 1 / 1e3
  }
};
var normalizedUnits = {
  ms: "millisecond",
  milliseconds: "millisecond",
  s: "second",
  seconds: "second",
  m: "minute",
  mi: "minute",
  min: "minute",
  minutes: "minute",
  h: "hour",
  hours: "hour",
  d: "day",
  days: "day",
  w: "week",
  weeks: "week",
  M: "month",
  mo: "month",
  mon: "month",
  months: "month",
  q: "quarter",
  quarters: "quarter",
  y: "year",
  years: "year",
  dec: "decade",
  decades: "decade"
};
var withDecimalsDurationRegex = /^\s*([-+]?\d+(?:[.,]\d*)?|[-+]?(?:[.,]\d+))\s*([^\s]+)?/i;
var noDecimalsDurationRegex = /^\s*([-+]?\d+)(?![.,])\s*([^\s]+)?/i;
var canonicalUnitNames = [
  "millisecond",
  "second",
  "minute",
  "hour",
  "day",
  "week",
  "month",
  "quarter",
  "year",
  "decade"
];
var canonicalUnitAbbreviations = [
  ["mil"],
  ["s", "sec"],
  ["m", "min"],
  ["h", "hr"],
  ["d"],
  ["w", "wk"],
  ["mo", "mon", "mnt"],
  ["q", "quar", "qrt"],
  ["y", "yr"],
  ["dec"]
];
var deltaUnits = [
  "decade",
  "year",
  "month",
  "week",
  "day",
  "hour",
  "minute",
  "second",
  "millisecond"
];
var dateProperties = [
  "milliseconds",
  "seconds",
  "minutes",
  "hours",
  "date",
  "month",
  "year"
];
var parseNumber = (n) => {
  const result = parseFloat(n);
  return isNaN(result) ? null : result;
};
var numberRegex = /^[0-9]+$/;
var timeZoneRegEx = /([+-])(\d\d):*(\d\d)*$/;
var unitMagnitudes = {
  millisecond: 0,
  second: 1,
  minute: 2,
  hour: 3,
  day: 4,
  week: 5,
  month: 6,
  quarter: 7,
  year: 8,
  decade: 9
};
var snapFns = {
  round(number, step = 1) {
    return Math.round(number / step) * step;
  },
  floor(number, step = 1) {
    return Math.floor(number / step) * step;
  },
  ceil(number, step = 1) {
    return Math.ceil(number / step) * step;
  }
};
var keyCache = {};
var _DateHelper = class extends Localizable_default() {
  static get $name() {
    return "DateHelper";
  }
  //region Parse & format
  /**
   * Get/set the default format used by `format()` and `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ssZ'`
   * (~ISO 8601 Date and time, `'1962-06-17T09:21:34Z'`).
   * @member {String}
   */
  static set defaultFormat(format2) {
    DH._defaultFormat = format2;
  }
  static get defaultFormat() {
    return DH._defaultFormat || "YYYY-MM-DDTHH:mm:ssZ";
  }
  /**
   * Get/set the default format used by `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ss.SSSZ'` or {@link #property-defaultFormat-static}
   * (~ISO 8601 Date and time, `'1962-06-17T09:21:34.123Z'`).
   * @member {String}
   */
  static set defaultParseFormat(parseFormat) {
    this._defaultParseFormat = parseFormat;
  }
  static get defaultParseFormat() {
    return this._defaultParseFormat || this._defaultFormat || "YYYY-MM-DDTHH:mm:ss.SSSZ";
  }
  static buildParser(format2) {
    const parts = format2.split(parserRegexp), parser = [];
    if (parts.length === 1 || localeStrRegExp.test(format2)) {
      return [];
    } else {
      parts.reduce((prev, curr, index, array) => {
        if (index !== 0 || curr !== "") {
          if (parserRegexp.test(curr)) {
            const localeParsers = this.localize("L{parsers}") || {}, fn2 = localeParsers[curr] || parsers[curr];
            if (curr === "Z" && index < array.length - 2) {
              throw new Error(`Invalid format ${format2} TimeZone (Z) must be last token`);
            }
            const parserObj = typeof fn2 === "function" || typeof fn2 === "string" ? fn2 : fn2.parser();
            if (typeof parserObj === "string") {
              const nestedParsers = DH.buildParser(parserObj), lastItem = nestedParsers.pop();
              delete lastItem.last;
              parser.push(...nestedParsers);
              prev = lastItem;
            } else {
              prev.pattern = curr;
              prev.fn = parserObj;
            }
          } else {
            prev.splitter = curr;
            parser.push(prev);
            prev = {};
          }
        } else if (Object.prototype.hasOwnProperty.call(prev, "pattern")) {
          parser.push(prev);
        }
        return prev;
      }, {});
    }
    parser[parser.length - 1].last = true;
    return parser;
  }
  /**
   * A utility function to create a sortable string key for the passed date or ms timestamp using the `'YYYY-MM-DD'`
   * format.
   * @param {Number|Date} ms The Date instance or ms timestamp to generate a key for
   * @returns {String} Date/timestamp as a string with `'YYYY-M-D'` format
   * @internal
   */
  static makeKey(ms) {
    if (ms.length === 10) {
      return ms;
    }
    if (ms.getTime) {
      ms = ms.getTime();
    }
    const cached = keyCache[Math.trunc(ms / MS_PER_HOUR)];
    if (cached) {
      return cached;
    }
    tempDate.setTime(ms);
    const month2 = tempDate.getMonth() + 1, date = tempDate.getDate();
    return keyCache[Math.trunc(ms / MS_PER_HOUR)] = `${tempDate.getFullYear()}-${month2 < 10 ? "0" + month2 : month2}-${date < 10 ? "0" + date : date}`;
  }
  /**
   * A utility function to parse a sortable string to a date using the `'YYYY-MM-DD'` format.
   * @param {String} key The string to return a date for
   * @returns {Date} new Date instance
   * @internal
   */
  static parseKey(key) {
    return DH.parse(key, "YYYY-MM-DD");
  }
  /**
   * Returns a date created from the supplied string using the specified format. Will try to create even if format
   * is left out, by first using the default format (see {@link #property-defaultFormat-static}, by default
   * `YYYY-MM-DDTHH:mm:ssZ`) and then using `new Date(dateString)`.
   * Supported tokens:
   *
   * | Unit        | Token | Description                       |
   * |-------------|-------|-----------------------------------|
   * | Year        | YYYY  | 2018                              |
   * |             | YY    | < 68 -> 2000, > 68 -> 1900        |
   * | Month       | MM    | 01 - 12                           |
   * | Date        | DD    | 01 - 31                           |
   * | Hour        | HH    | 00 - 23 or 1 - 12                 |
   * | Minute      | mm    | 00 - 59                           |
   * | Second      | ss    | 00 - 59                           |
   * | Millisecond | S     | 0 - 9 [000, 100, 200 .. 900 ]     |
   * |             | SS    | 00 - 99 [000, 010, 020 .. 990 ]   |
   * |             | SSS   | 000 - 999 [000, 001, 002 .. 999 ] |
   * | AM/PM       | A     | AM or PM                          |
   * |             | a     | am or pm                          |
   * | TimeZone    | Z     | Z for UTC or +-HH:mm              |
   * | Predefined  | L     | Long date, MM/DD/YYYY             |
   * |             | LT    | Long time, HH:mm A                |
   *
   * Predefined formats and functions used to parse tokens can be localized, see for example the swedish locale SvSE.js
   *
   * NOTE: If no date parameters are provided then `Jan 01 2020` is used as a default date
   *
   * @param {String} dateString Date string
   * @param {String} [format] Date format (or {@link #property-defaultParseFormat-static} if left out)
   * @returns {Date} new Date instance parsed from the string
   * @category Parse & format
   */
  static parse(dateString, format2 = DH.defaultParseFormat, strict = false) {
    if (dateString instanceof Date) {
      return dateString;
    }
    if (typeof dateString !== "string" || !dateString) {
      return null;
    }
    const config = {
      year: null,
      month: null,
      date: null,
      hours: null,
      minutes: null,
      seconds: null,
      milliseconds: null
    };
    format2 = format2.replace(msRegExp, msReplacer);
    let parser = parserCache[format2], result;
    if (!parser) {
      parser = parserCache[format2] = DH.buildParser(format2);
    }
    if (dateString.includes("\u202F")) {
      dateString = dateString.replace(/\s/g, " ");
    }
    parser.reduce((dateString2, parser2) => {
      var _a2;
      if (parser2.last) {
        Object.assign(config, parser2.fn(dateString2));
      } else {
        let splitAt;
        if (parser2.splitter === "T" && dateString2.indexOf("T") === -1) {
          splitAt = dateString2.indexOf(" ");
        } else {
          const timeZoneIndex = dateString2.indexOf("+");
          let { splitter } = parser2;
          if (!strict && splitRegExp.test(splitter)) {
            splitter = splitRegExp;
          }
          splitAt = parser2.splitter !== "" ? dateString2.search(typeof splitter === "string" ? StringHelper.escapeRegExp(splitter) : splitter) : ((_a2 = parser2.pattern) == null ? void 0 : _a2.length) || -1;
          if (timeZoneIndex > -1 && splitAt > timeZoneIndex) {
            splitAt = -1;
          }
        }
        let part, rest;
        if (splitAt === -1 || parser2.pattern === "SSS" && dateString2.match(/^\d+Z$/)) {
          const chunks = dateString2.split(/([Z\-+])/);
          if (chunks.length === 1) {
            part = dateString2;
            rest = "";
          } else {
            part = chunks[0];
            rest = `${chunks[1]}${chunks[2]}`;
          }
        } else {
          part = dateString2.substring(0, splitAt) || dateString2;
          rest = dateString2.substring(splitAt + parser2.splitter.length);
        }
        if (parser2.fn) {
          const res = parser2.fn(part);
          if (res) {
            Object.assign(config, res);
          } else {
            rest = part + rest;
          }
        }
        return rest;
      }
    }, dateString);
    if (config.year && !config.date) {
      config.date = 1;
    }
    if (config.date > 31 || config.month > 12) {
      return null;
    }
    const date = DH.create(config, strict);
    if (date) {
      result = date;
    } else if (!strict) {
      result = new Date(dateString);
    }
    return result;
  }
  /**
   * Creates a date from a date definition object. The object can have the following properties:
   * - year
   * - month
   * - date (day in month)
   * - hours
   * - minutes
   * - seconds
   * - milliseconds
   * - amPm : 'am' or 'pm', implies 12-hour clock
   * - timeZone : offset from UTC in minutes
   * @param {Object} definition
   * @param {Number} definition.year
   * @param {Number} [definition.month]
   * @param {Number} [definition.date]
   * @param {Number} [definition.hours]
   * @param {Number} [definition.minutes]
   * @param {Number} [definition.seconds]
   * @param {Number} [definition.milliseconds]
   * @param {Number} [definition.amPm]
   * @param {Number} [definition.timeZone]
   * @returns {Date} new Date instance
   * @category Parse & format
   */
  static create(definition, strict = false) {
    const def = { ...definition };
    let invalid = isNaN(def.year) || strict && (isNaN(def.month) || isNaN(def.date)), useUTC = false;
    if (!invalid) {
      let allNull = true;
      dateProperties.forEach((property) => {
        if (!(property in def) || isNaN(def[property])) {
          def[property] = 0;
        }
        allNull = allNull && def[property] === null;
      });
      invalid = allNull;
    }
    if (invalid) {
      return null;
    }
    if (def.amPm === "am") {
      def.hours = def.hours % 12;
    } else if (def.amPm === "pm") {
      def.hours = def.hours % 12 + 12;
    }
    if ("timeZone" in def) {
      useUTC = true;
      def.minutes -= def.timeZone;
    }
    if (strict && (def.year == null || def.month == null || def.date == null)) {
      return null;
    }
    const args = [
      defaultValue(def.year, DEFAULT_YEAR),
      defaultValue(def.month, DEFAULT_MONTH),
      defaultValue(def.date, DEFAULT_DAY),
      def.hours,
      def.minutes,
      def.seconds,
      def.milliseconds
    ];
    return useUTC ? new Date(Date.UTC(...args)) : new Date(...args);
  }
  static toUTC(date) {
    return new Date(Date.UTC(
      date.getUTCFullYear(),
      date.getUTCMonth(),
      date.getUTCDate(),
      date.getUTCHours(),
      date.getUTCMinutes(),
      date.getUTCSeconds(),
      date.getUTCMilliseconds()
    ));
  }
  /**
   * Converts a date to string with the specified format. Formats heavily inspired by https://momentjs.com.
   * Available formats (input used for output below is `new Date(2018,8,9,18,7,8,145)`):
   *
   * | Unit                  | Token | Description & output                  |
   * |-----------------------|-------|---------------------------------------|
   * | Year                  | YYYY  | 2018                                  |
   * |                       | YY    | 18                                    |
   * |                       | Y     | 2018                                  |
   * | Quarter               | Q     | 3                                     |
   * |                       | Qo    | 3rd                                   |
   * | Month                 | MMMM  | September                             |
   * |                       | MMM   | Sep                                   |
   * |                       | MM    | 09                                    |
   * |                       | Mo    | 9th                                   |
   * |                       | M     | 9                                     |
   * | Week (iso)            | WW    | 37 (2 digit, zero padded)             |
   * |                       | Wo    | 37th                                  |
   * |                       | W     | 37                                    |
   * |                       | WWp   | Week 37 (localized prefix, zero pad)  |
   * |                       | Wp    | Week 37 (localized prefix)            |
   * |                       | WWp0  | W37 (localized prefix)                |
   * |                       | Wp0   | W37 (localized prefix)                |
   * | Date                  | DDDD  | Day of year, 3 digits                 |
   * |                       | DDDo  | Day of year, ordinal                  |
   * |                       | DDD   | Day of year                           |
   * |                       | DD    | 09                                    |
   * |                       | Do    | 9th                                   |
   * |                       | D     | 9                                     |
   * | Weekday               | dddd  | Sunday                                |
   * |                       | ddd   | Sun                                   |
   * |                       | dd    | Su                                    |
   * |                       | d1    | S                                     |
   * |                       | do    | 0th                                   |
   * |                       | d     | 0                                     |
   * | Hour                  | HH    | 18 (00 - 23)                          |
   * |                       | H     | 18 (0 - 23)                           |
   * |                       | hh    | 06 (00 - 12)                          |
   * |                       | h     | 6 (0 - 12)                            |
   * |                       | KK    | 19 (01 - 24)                          |
   * |                       | K     | 19 (1 - 24)                           |
   * |                       | kk    | 06 or 18, locale determines           |
   * |                       | k     | 6 or 18, locale determines            |
   * | Minute                | mm    | 07                                    |
   * |                       | m     | 7                                     |
   * | Second                | ss    | 08                                    |
   * |                       | s     | 8                                     |
   * | Millisecond           | S     | 1 (100ms)                             |
   * |                       | SS    | 14 (140ms)                            |
   * |                       | SSS   | 145 (145ms)                           |
   * | AM/PM                 | A     | AM or PM                              |
   * |                       | a     | am or pm                              |
   * | Predefined            | LT    | H: 2-digit (2d), m: 2d                |
   * | (uses browser locale) | LTS   | H: 2d, m: 2d, s : 2d                  |
   * |                       | LST   | Depends on 12 or 24 hour clock        |
   * |                       |       | 12h, H : 1d, m : 0 or 2d              |
   * |                       |       | 24h, H : 2d, m : 2d                   |
   * |                       | L     | Y: numeric (n), M : 2d, D : 2d        |
   * |                       | l     | Y: n, M : n, D : n                    |
   * |                       | LL    | Y: n, M : long (l), D : n             |
   * |                       | ll    | Y: n, M : short (s), D : n            |
   * |                       | LLL   | Y: n, M : l, D : n, H: n, m: 2d       |
   * |                       | lll   | Y: n, M : s, D : n, H: n, m: 2d       |
   * |                       | LLLL  | Y: n, M : l, D : n, H: n, m: 2d, d: l |
   * |                       | llll  | Y: n, M : s, D : n, H: n, m: 2d, d: s |
   *
   * Some examples:
   *
   * ```javascript
   * DateHelper.format(new Date(2019, 7, 16), 'dddd') -> Friday
   * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'HH:mm') --> 14:27
   * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'L HH') --> 2019-07-16 14
   * ```
   *
   * Arbitrary text can be embedded in the format string by wrapping it with {}:
   *
   * ```javascript
   * DateHelper.format(new Date(2019, 7, 16), '{It is }dddd{, yay!}') -> It is Friday, yay!
   * ```
   *
   * @param {Date} date Date
   * @param {String} [format] Desired format (uses `defaultFormat` if left out)
   * @returns {String} Formatted string
   * @category Parse & format
   */
  static format(date, format2 = DH.defaultFormat) {
    if (!date || isNaN(date)) {
      return null;
    }
    let formatter = formatCache[format2], output = "";
    if (!formatter) {
      formatter = formatCache[format2] = [];
      for (let i = 0; i < format2.length; i++) {
        const formatMatch = format2.slice(i).match(formatRegexp), predefined = formatMatch == null ? void 0 : formatMatch[0];
        if (predefined) {
          const localeFormats = this.localize("L{formats}") || {}, fn2 = localeFormats[predefined] || formats[predefined];
          formatter.push(fn2);
          i += predefined.length - 1;
        } else if (format2[i] === "{") {
          const index = format2.indexOf("}", i + 1);
          if (index === -1) {
            formatter.push(format2.substr(i + 1));
            i = format2.length;
          } else {
            formatter.push(format2.substring(i + 1, index));
            i = index;
          }
        } else {
          formatter.push(format2[i]);
        }
      }
    }
    formatter.forEach((step) => {
      if (typeof step === "string") {
        output += step;
      } else {
        output += step(date);
      }
    });
    return output;
  }
  /**
   * Formats a range of `dates` using the specified `format`. Because two dates are involved, the `format` specifier
   * uses the tokens `S{}` and `E{}`. The text contained between the `{}` is the {@link #function-format-static format}
   * for the start date or end date, respectively. Text not inside these tokens is retained verbatim.
   *
   * For example:
   *
   * ```javascript
   *  DateHelper.formatRange(dates, 'S{DD MMM YYYY} - E{DD MMM YYYY}');
   * ```
   *
   * The above will format `dates[0]` based on the `S{DD MMM YYYY}` segment and `dates[1] using `E{DD MMM YYYY}`. The
   * `' - '` between these will remain between the two formatted dates.
   *
   * @param {Date[]} dates An array of start date and end date (`[startDate, endDate]`)
   * @param {String} format The format specifier
   * @returns {String}
   */
  static formatRange(dates, format2) {
    return format2.replace(
      rangeFormatPartRe,
      (s, which, fmt) => _DateHelper.format(dates[which === "S" ? 0 : 1], fmt)
    );
  }
  /**
   * Converts the specified amount of desired unit into milliseconds. Can be called by only specifying a unit as the
   * first argument, it then uses `amount = 1`.
   *
   * For example:
   *
   * ```javascript
   * asMilliseconds('hour') == asMilliseconds(1, 'hour')
   * ```
   *
   * @param {Number|String} amount Amount, what of is decided by specifying unit (also takes a unit which implies an amount of 1)
   * @param {String} [unit] Time unit (s, hour, months etc.)
   * @returns {Number}
   * @category Parse & format
   */
  static asMilliseconds(amount, unit = null) {
    if (typeof amount === "string") {
      unit = amount;
      amount = 1;
    }
    return DH.as("millisecond", amount, unit);
  }
  /**
   * Converts the passed Date to an accurate number of months passed since the epoch start.
   * @param {Date} time The Date to find the month value of
   * @returns {Number} The number of months since the system time epoch start. May be a fractional value
   */
  static asMonths(time) {
    const monthLength = DH.as("ms", DH.daysInMonth(time), "day"), fraction = (time.valueOf() - DH.startOf(time, "month").valueOf()) / monthLength;
    return time.getYear() * 12 + time.getMonth() + fraction;
  }
  static monthsToDate(months) {
    const intMonths = Math.floor(months), fraction = months - intMonths, result = new Date(0, intMonths), msInMonth = DH.as("ms", DH.daysInMonth(result), "days");
    result.setTime(result.getTime() + fraction * msInMonth);
    return result;
  }
  /**
   * Converts a millisecond time delta to a human-readable form. For example `1000 * 60 * 60 * 50`
   * milliseconds would be rendered as `'2 days, 2 hours'`.
   * @param {Number} delta The millisecond delta value
   * @param {Object} [options] Formatting options
   * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg `'2d, 2h'` for the above example
   * @param {String} [options.precision] The minimum precision unit
   * @param {String} [options.separator] The separator to use
   * @param {Boolean} [options.asString] Pass `false` to return the result as an array, eg ['2d', '2h'] for the above example
   * @returns {String} Formatted string
   * @category Parse & format
   */
  static formatDelta(delta, options) {
    let abbrev, unitName;
    if (typeof options === "boolean") {
      abbrev = options;
    } else if (options) {
      abbrev = options.abbrev;
    }
    const deltaObj = this.getDelta(delta, options), result = [], sep = (options == null ? void 0 : options.separator) || (abbrev ? "" : " ");
    for (unitName in deltaObj) {
      result.push(`${deltaObj[unitName]}${sep}${unitName}`);
    }
    return (options == null ? void 0 : options.asString) === false ? result : result.join(", ");
  }
  /**
   * Converts a millisecond time delta to an object structure. For example `1000 * 60 * 60 * 50`
   * milliseconds the result would be as:
   *
   * ```javascript
   * {
   *     day  : 2,
   *     hour : 2
   * }
   *```
   *
   * @param {Number} delta The millisecond delta value
   * @param {Object} [options] Formatting options
   * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg `{ d: 2, h: 2 }` for the above example
   * @param {String} [options.precision] The minimum precision unit
   * @param {Boolean} [options.ignoreLocale] Pass true to return unlocalized unit name. Requires `abbrev` to be false
   * @param {String} [options.maxUnit] Name of the maximum unit in the output. e.g. if you pass `day` then you'll get
   * `{ h: 25 }` instead of `{ d: 1, h: 1 }`
   * @returns {Object} The object with the values for each unit
   */
  static getDelta(delta, options) {
    let abbrev, d, done, precision, unitName, maxUnit, ignoreLocale;
    if (typeof options === "boolean") {
      abbrev = options;
    } else if (options) {
      abbrev = options.abbrev;
      precision = DH.normalizeUnit(options.precision);
      maxUnit = options.maxUnit;
      ignoreLocale = !abbrev && options.ignoreLocale;
    }
    const result = {}, getUnit = abbrev ? DH.getShortNameOfUnit : DH.getLocalizedNameOfUnit;
    const units = maxUnit ? deltaUnits.slice(deltaUnits.indexOf(maxUnit)) : deltaUnits;
    for (unitName of units) {
      d = DH.as(unitName, delta);
      done = precision === unitName;
      d = Math[done ? "round" : "floor"](d);
      if (d || done && !result.length) {
        result[ignoreLocale ? unitName : getUnit.call(DH, unitName, d !== 1)] = d;
        delta -= DH.as("ms", d, unitName);
      }
      if (done || !delta) {
        break;
      }
    }
    return result;
  }
  /**
   * Converts the specified amount of one unit (`fromUnit`) into an amount of another unit (`toUnit`).
   * @param {String} toUnit The name of units to convert to, eg: `'ms'`
   * @param {Number|String} amount The time to convert. Either the magnitude number form or a duration string such as '2d'
   * @param {String} [fromUnit='ms'] If the amount was passed as a number, the units to use to convert from
   * @returns {Number}
   * @category Parse & format
   */
  static as(toUnit, amount, fromUnit = "ms") {
    if (typeof amount === "string") {
      amount = DH.parseDuration(amount);
    }
    if (typeof amount === "object") {
      fromUnit = amount.unit;
      amount = amount.magnitude;
    }
    if (toUnit === fromUnit) {
      return amount;
    }
    toUnit = DH.normalizeUnit(toUnit);
    fromUnit = DH.normalizeUnit(fromUnit);
    if (toUnit === fromUnit) {
      return amount;
    } else if (unitMagnitudes[fromUnit] > unitMagnitudes[toUnit]) {
      return amount * Math.abs(validConversions[fromUnit][toUnit]);
    } else {
      return amount / Math.abs(validConversions[toUnit][fromUnit]);
    }
  }
  static formatContainsHourInfo(format2) {
    const stripEscapeRe = /(\\.)/g, hourInfoRe = /([HhKkmSsAa]|LT|L{3,}|l{3,})/;
    return hourInfoRe.test(format2.replace(stripEscapeRe, ""));
  }
  /**
   * Returns `true` for 24-hour format.
   * @param {String} format Date format
   * @returns {Boolean} `true` for 24-hour format
   * @category Parse & format
   */
  static is24HourFormat(format2) {
    return DH.format(DH.getTime(13, 0, 0), format2).includes("13");
  }
  //endregion
  //region Manipulate
  /**
   * Add days, hours etc. to a date. Always clones the date, original will be left unaffected.
   * @param {Date|String} date Original date
   * @param {Number|String|Core.data.Duration|DurationConfig} amount Amount of days, hours etc. or a string representation of a duration
   * as accepted by {@link #function-parseDuration-static} or an object with `{ magnitude, unit }` properties
   * @param {String} [unit='ms'] Unit for amount
   * @privateparam {Boolean} [clone=true] Pass `false` to affect the original
   * @returns {Date} New calculated date
   * @category Manipulate
   */
  static add(date, amount, unit = "ms", clone = true) {
    let d;
    if (typeof date === "string") {
      d = DH.parse(date);
    } else if (clone) {
      d = new Date(date.getTime());
    } else {
      d = date;
    }
    if (typeof amount === "string") {
      const duration = _DateHelper.parseDuration(amount);
      amount = duration.magnitude;
      unit = duration.unit;
    } else if (amount && typeof amount === "object") {
      unit = amount.unit;
      amount = amount.magnitude;
    }
    if (!unit || amount === 0) {
      return d;
    }
    unit = DH.normalizeUnit(unit);
    switch (unit) {
      case "millisecond":
        d.setTime(d.getTime() + amount);
        break;
      case "second":
        d.setTime(d.getTime() + amount * 1e3);
        break;
      case "minute":
        d.setTime(d.getTime() + amount * 6e4);
        break;
      case "hour":
        d.setTime(d.getTime() + amount * 36e5);
        break;
      case "day":
        if (amount % 1 === 0) {
          d.setDate(d.getDate() + amount);
          if (d.getHours() === 23 && date.getHours() === 0) {
            d.setHours(d.getHours() + 1);
          }
        } else {
          d.setTime(d.getTime() + amount * 864e5);
        }
        break;
      case "week":
        d.setDate(d.getDate() + amount * 7);
        break;
      case "month": {
        let day2 = d.getDate();
        if (day2 > 28) {
          day2 = Math.min(day2, DH.getLastDateOfMonth(DH.add(DH.getFirstDateOfMonth(d), amount, "month")).getDate());
        }
        d.setDate(day2);
        d.setMonth(d.getMonth() + amount);
        break;
      }
      case "quarter":
        DH.add(d, amount * 3, "month", false);
        break;
      case "year":
        d.setFullYear(d.getFullYear() + amount);
        break;
      case "decade":
        d.setFullYear(d.getFullYear() + amount * 10);
        break;
    }
    return d;
  }
  /**
   * Calculates the difference between two dates, in the specified unit.
   * @param {Date} start First date
   * @param {Date} end Second date
   * @param {String} [unit='ms'] Unit to calculate difference in
   * @param {Boolean} [fractional=true] Specify false to round result
   * @returns {Number} Difference in the specified unit
   * @category Manipulate
   */
  static diff(start, end, unit = "ms", fractional = true) {
    unit = DH.normalizeUnit(unit);
    if (!start || !end)
      return 0;
    let amount;
    switch (unit) {
      case "year":
        amount = DH.diff(start, end, "month") / 12;
        break;
      case "quarter":
        amount = DH.diff(start, end, "month") / 3;
        break;
      case "month":
        amount = (end.getFullYear() - start.getFullYear()) * 12 + (end.getMonth() - start.getMonth());
        if (amount === 0 && fractional) {
          amount = DH.diff(start, end, "day", fractional) / DH.daysInMonth(start);
        }
        break;
      case "week":
        amount = DH.diff(start, end, "day") / 7;
        break;
      case "day": {
        const dstDiff = start.getTimezoneOffset() - end.getTimezoneOffset();
        amount = (end - start + dstDiff * 60 * 1e3) / 864e5;
        break;
      }
      case "hour":
        amount = (end - start) / 36e5;
        break;
      case "minute":
        amount = (end - start) / 6e4;
        break;
      case "second":
        amount = (end - start) / 1e3;
        break;
      case "millisecond":
        amount = end - start;
        break;
    }
    return fractional ? amount : Math.round(amount);
  }
  /**
   * Sets the date to the start of the specified unit, by default returning a clone of the date instead of changing it
   * in place.
   * @param {Date} date Original date
   * @param {String} [unit='day'] Start of this unit, `'day'`, `'month'` etc
   * @param {Boolean} [clone=true] Manipulate a copy of the date
   * @param {Number} [weekStartDay] The first day of week, `0-6` (Sunday-Saturday). Defaults to the {@link #property-weekStartDay-static}
   * @returns {Date} Manipulated date
   * @category Manipulate
   */
  static startOf(date, unit = "day", clone = true, weekStartDay = DH.weekStartDay) {
    if (!date) {
      return null;
    }
    unit = DH.normalizeUnit(unit);
    if (clone) {
      date = DH.clone(date);
    }
    switch (unit) {
      case "year":
        date.setMonth(0, 1);
        date.setHours(0, 0, 0, 0);
        return date;
      case "quarter":
        date.setMonth((DH.get(date, "quarter") - 1) * 3, 1);
        date.setHours(0, 0, 0, 0);
        return date;
      case "month":
        date.setDate(1);
        date.setHours(0, 0, 0, 0);
        return date;
      case "week": {
        const delta = date.getDay() - weekStartDay;
        date.setDate(date.getDate() - delta);
        date.setHours(0, 0, 0, 0);
        return date;
      }
      case "day":
        date.setHours(0, 0, 0, 0);
        return date;
      case "hour":
        date.getMinutes() > 0 && date.setMinutes(0);
      case "minute":
        date.getSeconds() > 0 && date.setSeconds(0);
      case "second":
        date.getMilliseconds() > 0 && date.setMilliseconds(0);
      case "millisecond":
        return date;
    }
  }
  /**
   * Returns the end point of the passed date, that is 00:00:00 of the day after the passed date.
   * @param {Date} date The date to return the end point of
   * @returns {Date} Manipulated date
   */
  static endOf(date) {
    return new Date(date.getFullYear(), date.getMonth(), date.getDate() + 1);
  }
  /**
   * Creates a clone of the specified date
   * @param {Date} date Original date
   * @returns {Date} Cloned date
   * @category Manipulate
   */
  static clone(date) {
    return new Date(date.getTime());
  }
  /**
   * Removes time from a date (same as calling {@link #function-startOf-static startOf(date)}).
   * @param {Date} date Date to remove time from
   * @param {Boolean} [clone=true] Manipulate a copy of the date
   * @returns {Date} Manipulated date
   * @category Manipulate
   */
  static clearTime(date, clone = true) {
    if (!date) {
      return null;
    }
    if (clone) {
      date = new Date(date.getTime());
    }
    date.setHours(0, 0, 0, 0);
    return date;
  }
  static midnight(date, inclusive) {
    let ret = DH.clearTime(date);
    if (inclusive && ret < date) {
      ret = DH.add(ret, 1, "d");
    }
    return ret;
  }
  /**
   * Returns the elapsed milliseconds from the start of the specified date.
   * @param {Date} date Date to remove date from
   * @param {String} [unit='ms'] The time unit to return
   * @returns {Number} The elapsed milliseconds from the start of the specified date
   * @category Manipulate
   */
  static getTimeOfDay(date, unit = "ms") {
    const t = date.getHours() * validConversions.hour.millisecond + date.getMinutes() * validConversions.minute.millisecond + date.getSeconds() * validConversions.second.millisecond + date.getMilliseconds();
    return unit === "ms" ? t : DH.as(unit, t, "ms");
  }
  /**
   * Sets a part of a date (in place).
   * @param {Date} date Date to manipulate
   * @param {String|Object} unit Part of date to set, for example `'minute'`. Or an object like `{ second: 1, minute: 1 }`
   * @param {Number} amount Value to set
   * @returns {Date} Passed date instance modified according to the arguments
   * @category Manipulate
   */
  static set(date, unit, amount) {
    if (!unit) {
      return date;
    }
    if (typeof unit === "string") {
      switch (DH.normalizeUnit(unit)) {
        case "millisecond":
          if (amount !== 0 || date.getMilliseconds() > 0) {
            date.setMilliseconds(amount);
          }
          break;
        case "second":
          if (amount !== 0 || date.getSeconds() > 0) {
            date.setSeconds(amount);
          }
          break;
        case "minute":
          if (amount !== 0 || date.getMinutes() > 0) {
            date.setMinutes(amount);
          }
          break;
        case "hour":
          date.setHours(amount);
          break;
        case "day":
        case "date":
          date.setDate(amount);
          break;
        case "week":
          throw new Error("week not implemented");
        case "month":
          date.setMonth(amount);
          break;
        case "quarter":
          date.setDate(1);
          date.setMonth((amount - 1) * 3);
          break;
        case "year":
          date.setFullYear(amount);
          break;
      }
    } else {
      Object.entries(unit).sort((a, b) => unitMagnitudes[a[0]] - unitMagnitudes[b[0]]).forEach(([unit2, amount2]) => {
        DH.set(date, unit2, amount2);
      });
    }
    return date;
  }
  static setDateToMidday(date, clone = true) {
    return DH.set(DH.clearTime(date, clone), "hour", 12);
  }
  /**
   * Constrains the date within a min and a max date.
   * @param {Date} date The date to constrain
   * @param {Date} [min] Min date
   * @param {Date} [max] Max date
   * @returns {Date} The constrained date
   * @category Manipulate
   */
  static constrain(date, min2, max) {
    if (min2 != null) {
      date = DH.max(date, min2);
    }
    return max == null ? date : DH.min(date, max);
  }
  /**
   * Returns time with default year, month, and day (Jan 1, 2020).
   * @param {Number|Date} hours Hours value or the full date to extract the time of
   * @param {Number} [minutes=0] Minutes value
   * @param {Number} [seconds=0] Seconds value
   * @param {Number} [ms=0] Milliseconds value
   * @returns {Date} A new default date with the time extracted from the given date or from the time values provided individually
   * @category Manipulate
   */
  static getTime(hours, minutes = 0, seconds = 0, ms = 0) {
    if (hours instanceof Date) {
      ms = hours.getMilliseconds();
      seconds = hours.getSeconds();
      minutes = hours.getMinutes();
      hours = hours.getHours();
    }
    return new Date(DEFAULT_YEAR, DEFAULT_MONTH, DEFAULT_DAY, hours, minutes, seconds, ms);
  }
  /**
   * Copies hours, minutes, seconds, milliseconds from one date to another.
   *
   * @param {Date} targetDate The target date
   * @param {Date} sourceDate The source date
   * @returns {Date} The adjusted target date
   * @category Manipulate
   * @static
   */
  static copyTimeValues(targetDate, sourceDate) {
    targetDate.setHours(sourceDate.getHours());
    targetDate.setMinutes(sourceDate.getMinutes());
    targetDate.setSeconds(sourceDate.getSeconds());
    targetDate.setMilliseconds(sourceDate.getMilliseconds());
    return targetDate;
  }
  //endregion
  //region Comparison
  static get isDSTEnabled() {
    const year = (/* @__PURE__ */ new Date()).getFullYear(), jan = new Date(year, 0, 1), jul = new Date(year, 6, 1);
    return jan.getTimezoneOffset() !== jul.getTimezoneOffset();
  }
  static isDST(date) {
    const year = date.getFullYear(), jan = new Date(year, 0, 1), jul = new Date(year, 6, 1);
    return date.getTimezoneOffset() < Math.max(jan.getTimezoneOffset(), jul.getTimezoneOffset());
  }
  /**
   * Determines if a date precedes another.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Boolean} `true` if first precedes second, otherwise false
   * @category Comparison
   */
  static isBefore(first, second) {
    return first < second;
  }
  /**
   * Determines if a date succeeds another.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Boolean} `true` if first succeeds second, otherwise false
   * @category Comparison
   */
  static isAfter(first, second) {
    return first > second;
  }
  /**
   * Checks if two dates are equal.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @param {String} [unit] Unit to calculate difference in. If not given, the comparison will be done up to a millisecond
   * @returns {Boolean} `true` if the dates are equal
   * @category Comparison
   */
  static isEqual(first, second, unit = null) {
    if (unit === null) {
      return first && second && first.getTime() === second.getTime();
    }
    return DH.startOf(first, unit) - DH.startOf(second, unit) === 0;
  }
  /**
   * Compares two dates using the specified precision.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @param {String} [unit] Unit to calculate difference in. If not given, the comparison will be done up to a millisecond
   * @returns {Number} `0` = equal, `-1` = first before second, `1` = first after second
   * @category Comparison
   */
  static compare(first, second, unit = null) {
    if (unit) {
      first = DH.startOf(first, unit);
      second = DH.startOf(second, unit);
    }
    if (first < second)
      return -1;
    if (first > second)
      return 1;
    return 0;
  }
  /**
   * Coerces the passed Date between the passed minimum and maximum values.
   * @param {Date} date The date to clamp between the `min` and `max`
   * @param {Date} min The minimum Date
   * @param {Date} max The maximum Date
   * @returns {Date} If the passed `date` is valid, a *new* Date object which is clamped between the `min` and `max`
   */
  static clamp(date, min2, max) {
    if (!isNaN(date)) {
      if (min2 != null) {
        date = Math.max(date, min2);
      }
      if (max != null) {
        date = Math.min(date, max);
      }
      return new Date(date);
    }
  }
  static isSameDate(first, second) {
    return DH.compare(first, second, "d") === 0;
  }
  static isSameTime(first, second) {
    return first.getHours() === second.getHours() && first.getMinutes() === second.getMinutes() && first.getSeconds() === second.getSeconds() && first.getMilliseconds() === second.getMilliseconds();
  }
  /**
   * Checks if date is the start of specified unit.
   * @param {Date} date Date
   * @param {String} unit Time unit
   * @returns {Boolean} `true` if date is the start of specified unit
   * @category Comparison
   */
  static isStartOf(date, unit) {
    return DH.isEqual(date, DH.startOf(date, unit));
  }
  /**
   * Checks if this date is `>= start` and `< end`.
   * @param {Date} date The source date
   * @param {Date} start Start date
   * @param {Date} end End date
   * @returns {Boolean} `true` if this date falls on or between the given start and end dates
   * @category Comparison
   */
  static betweenLesser(date, start, end) {
    return start.getTime() <= date.getTime() && date.getTime() < end.getTime();
  }
  /**
   * Checks if this date is `>= start` and `<= end`.
   * @param {Date} date The source date
   * @param {Date} start Start date
   * @param {Date} end End date
   * @returns {Boolean} `true` if this date falls on or between the given start and end dates
   * @category Comparison
   */
  static betweenLesserEqual(date, start, end) {
    return start.getTime() <= date.getTime() && date.getTime() <= end.getTime();
  }
  /**
   * Returns `true` if dates intersect.
   * @param {Date} date1Start Start date of first span
   * @param {Date} date1End End date of first span
   * @param {Date} date2Start Start date of second span
   * @param {Date} date2End End date of second span
   * @returns {Boolean} Returns `true` if dates intersect
   * @category Comparison
   */
  static intersectSpans(date1Start, date1End, date2Start, date2End) {
    return DH.betweenLesser(date1Start, date2Start, date2End) || DH.betweenLesser(date2Start, date1Start, date1End);
  }
  /**
   * Compare two units. Returns `1` if first param is a greater unit than second param, `-1` if the opposite is true or `0` if they're equal.
   * @param {String} unit1 The 1st unit
   * @param {String} unit2 The 2nd unit
   * @returns {Number} Returns `1` if first param is a greater unit than second param, `-1` if the opposite is true or `0` if they're equal
   * @category Comparison
   */
  static compareUnits(unit1, unit2) {
    return Math.sign(unitMagnitudes[DH.normalizeUnit(unit1)] - unitMagnitudes[DH.normalizeUnit(unit2)]);
  }
  /**
   * Returns `true` if the first time span completely 'covers' the second time span.
   *
   * @example
   * DateHelper.timeSpanContains(
   *     new Date(2010, 1, 2),
   *     new Date(2010, 1, 5),
   *     new Date(2010, 1, 3),
   *     new Date(2010, 1, 4)
   * ) ==> true
   * DateHelper.timeSpanContains(
   *   new Date(2010, 1, 2),
   *   new Date(2010, 1, 5),
   *   new Date(2010, 1, 3),
   *   new Date(2010, 1, 6)
   * ) ==> false
   *
   * @param {Date} spanStart The start date for initial time span
   * @param {Date} spanEnd The end date for initial time span
   * @param {Date} otherSpanStart The start date for the 2nd time span
   * @param {Date} otherSpanEnd The end date for the 2nd time span
   * @returns {Boolean} `true` if the first time span completely 'covers' the second time span
   * @category Comparison
   */
  static timeSpanContains(spanStart, spanEnd, otherSpanStart, otherSpanEnd) {
    return otherSpanStart - spanStart >= 0 && spanEnd - otherSpanEnd >= 0;
  }
  //endregion
  //region Query
  /**
   * Get the first day of week, 0-6 (Sunday-Saturday).
   * This is determined by the current locale's `DateHelper.weekStartDay` parameter.
   * @property {Number}
   * @readonly
   */
  static get weekStartDay() {
    if (DH._weekStartDay == null) {
      DH._weekStartDay = this.localize("L{weekStartDay}") || 0;
    }
    return DH._weekStartDay;
  }
  /**
   * Get non-working days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
   * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
   *
   * For example:
   * ```javascript
   * {
   *     0 : true, // Sunday
   *     6 : true  // Saturday
   * }
   * ```
   *
   * @property {Object<Number,Boolean>}
   * @readonly
   */
  static get nonWorkingDays() {
    return { ...this.localize("L{nonWorkingDays}") };
  }
  /**
   * Get non-working days as an array of day indices, 0-6 (Sunday-Saturday).
   * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
   *
   * For example:
   *
   * ```javascript
   * [ 0, 6 ] // Sunday & Saturday
   * ```
   *
   * @property {Number[]}
   * @readonly
   * @internal
   */
  static get nonWorkingDaysAsArray() {
    return Object.keys(this.nonWorkingDays).map(Number);
  }
  /**
   * Get weekend days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
   * Weekends are days which are declared as weekend days by the selected country and defined by the current locale's
   * `DateHelper.weekends` parameter.
   * To get non-working days see {@link #property-nonWorkingDays-static}.
   *
   * For example:
   * ```javascript
   * {
   *     0 : true, // Sunday
   *     6 : true  // Saturday
   * }
   * ```
   * @property {Object<Number,Boolean>}
   * @readonly
   * @internal
   */
  static get weekends() {
    return { ...this.localize("L{weekends}") };
  }
  /**
   * Get the specified part of a date.
   * @param {Date} date
   * @param {String} unit Part of date, hour, minute etc.
   * @returns {Number} The requested part of the specified date
   * @category Query
   */
  static get(date, unit) {
    switch (DH.normalizeUnit(unit)) {
      case "millisecond":
        return date.getMilliseconds();
      case "second":
        return date.getSeconds();
      case "minute":
        return date.getMinutes();
      case "hour":
        return date.getHours();
      case "date":
      case "day":
        return date.getDate();
      case "week":
        return formats.W(date);
      case "month":
        return date.getMonth();
      case "quarter":
        return Math.floor(date.getMonth() / 3) + 1;
      case "year":
        return date.getFullYear();
    }
    return null;
  }
  /**
   * Get number of days in the current year for the supplied date.
   * @param {Date} date Date to check
   * @returns {Number} Days in year
   * @category Query
   * @internal
   */
  static daysInYear(date) {
    const fullYear = date.getFullYear(), duration = new Date(fullYear + 1, 0, 1) - new Date(fullYear, 0, 1);
    return this.as("day", duration);
  }
  /**
   * Get number of days in the current month for the supplied date.
   * @param {Date} date Date which month should be checked
   * @returns {Number} Days in month
   * @category Query
   */
  static daysInMonth(date) {
    return 32 - new Date(date.getFullYear(), date.getMonth(), 32).getDate();
  }
  /**
   * Get number of hours in the current day for the supplied date.
   * @param {Date} date Date to check
   * @returns {Number} Hours in day
   * @category Query
   * @internal
   */
  static hoursInDay(date) {
    const fullYear = date.getFullYear(), month2 = date.getMonth(), day2 = date.getDate(), duration = new Date(fullYear, month2, day2 + 1) - new Date(fullYear, month2, day2);
    return this.as("hour", duration);
  }
  /**
   * Converts unit related to the date to actual amount of milliseconds in it. Takes into account leap years and
   * different duration of months.
   * @param {Date} date Date
   * @param {String} unit Time unit
   * @returns {Number} Returns amount of milliseconds
   * @internal
   */
  static getNormalizedUnitDuration(date, unit) {
    let result;
    switch (unit) {
      case "month":
        result = DH.asMilliseconds(DH.daysInMonth(date), "day");
        break;
      case "year":
        result = DH.asMilliseconds(DH.daysInYear(date), "day");
        break;
      case "day":
        result = DH.asMilliseconds(DH.hoursInDay(date), "hour");
        break;
      default:
        result = DH.asMilliseconds(unit);
    }
    return result;
  }
  /**
   * Get the first date of the month for the supplied date.
   * @param {Date} date Date
   * @returns {Date} New Date instance
   * @category Query
   */
  static getFirstDateOfMonth(date) {
    return new Date(date.getFullYear(), date.getMonth(), 1);
  }
  /**
   * Get the last date of the month for the supplied date.
   * @param {Date} date Date
   * @returns {Date} New Date instance
   * @category Query
   */
  static getLastDateOfMonth(date) {
    return new Date(date.getFullYear(), date.getMonth() + 1, 0);
  }
  /**
   * Get the earliest of two dates.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Date} Earliest date
   * @category Query
   */
  static min(first, second) {
    return first.getTime() < second.getTime() ? first : second;
  }
  /**
   * Get the latest of two dates.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Date} Latest date
   * @category Query
   */
  static max(first, second) {
    return first.getTime() > second.getTime() ? first : second;
  }
  /**
   * Get an incremented date. Incrementation based on specified unit and optional amount.
   * @param {Date} date Date
   * @param {String} unit Time unit
   * @param {Number} [increment=1] Increment amount
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   * @category Query
   */
  static getNext(date, unit, increment = 1, weekStartDay = DH.weekStartDay) {
    if (unit === "week") {
      const dt = DH.clone(date), day2 = dt.getDay();
      DH.startOf(dt, "day", false);
      DH.add(dt, weekStartDay - day2 + 7 * (increment - (weekStartDay <= day2 ? 0 : 1)), "day", false);
      if (dt.getDay() !== weekStartDay) {
        DH.add(dt, 1, "hour");
      }
      return dt;
    }
    return DH.startOf(DH.add(date, increment, unit), unit, false);
  }
  /**
   * Checks if date object is valid.
   *
   * For example:
   *
   * ```javascript
   * date = new Date('foo')
   * date instanceof Date // true
   * date.toString() // Invalid Date
   * isNaN(date) // true
   * DateHelper.isValidDate(date) // false
   *
   * date = new Date()
   * date instanceof Date // true
   * date.toString() // Mon Jan 13 2020 18:27:38 GMT+0300 (GMT+03:00)
   * isNaN(date) // false
   * DateHelper.isValidDate(date) // true
   * ```
   *
   * @param {Date} date Date
   * @returns {Boolean} `true` if date object is valid
   */
  static isValidDate(date) {
    return DH.isDate(date) && !isNaN(date);
  }
  /**
   * Checks if value is a date object. Allows to recognize date object even from another context,
   * like the top frame when used in an iframe.
   *
   * @param {*} value Value to check
   * @returns {Boolean} `true` if value is a date object
   */
  static isDate(value) {
    return value && toString3.call(value) === DATE_TYPE2;
  }
  /**
   * Get the start of the next day.
   * @param {Date} date Date
   * @param {Boolean} [clone=false] Clone date
   * @param {Boolean} [noNeedToClearTime=false] Flag to not clear time from the result
   * @returns {Date} Passed Date or new Date instance, depending on the `clone` flag
   * @category Query
   */
  static getStartOfNextDay(date, clone = false, noNeedToClearTime = false) {
    let nextDay = DH.add(noNeedToClearTime ? date : DH.clearTime(date, clone), 1, "day");
    if (nextDay.getDate() === date.getDate()) {
      const offsetNextDay = DH.add(DH.clearTime(date, clone), 2, "day").getTimezoneOffset(), offsetDate = date.getTimezoneOffset();
      nextDay = DH.add(nextDay, offsetDate - offsetNextDay, "minute");
    }
    return nextDay;
  }
  /**
   * Get the end of previous day.
   * @param {Date} date Date
   * @param {Boolean} [noNeedToClearTime=false] Flag to not clear time from the result
   * @returns {Date} New Date instance
   * @category Query
   */
  static getEndOfPreviousDay(date, noNeedToClearTime = false) {
    const dateOnly = noNeedToClearTime ? date : DH.clearTime(date, true);
    if (dateOnly - date) {
      return dateOnly;
    } else {
      return DH.add(dateOnly, -1, "day");
    }
  }
  /**
   * Returns a string describing the specified week. For example, `'39, September 2020'` or `'40, Sep - Oct 2020'`.
   * @param {Date} startDate Start date
   * @param {Date} [endDate] End date
   * @returns {String} String describing the specified week
   * @internal
   */
  static getWeekDescription(startDate, endDate = startDate) {
    const monthDesc = startDate.getMonth() === endDate.getMonth() ? _DateHelper.format(startDate, "MMMM") : `${_DateHelper.format(startDate, "MMM")} - ${_DateHelper.format(endDate, "MMM")}`, week = _DateHelper.getWeekNumber(startDate);
    return `${week[1]}, ${monthDesc} ${week[0]}`;
  }
  /**
   * Get week number for the date.
   * @param {Date} date The date
   * @param {Number} [weekStartDay] The first day of week, 0-6 (Sunday-Saturday). Defaults to the {@link #property-weekStartDay-static}
   * @returns {Number[]} year and week number
   * @category Query
   */
  static getWeekNumber(date, weekStartDay = _DateHelper.weekStartDay) {
    const jan01 = new Date(date.getFullYear(), 0, 1), dec31 = new Date(date.getFullYear(), 11, 31), firstDay = normalizeDay(jan01.getDay() - weekStartDay), lastDay = normalizeDay(dec31.getDay() - weekStartDay), dayNumber = getDayDiff(date, jan01);
    let weekNumber;
    if (firstDay < 4) {
      weekNumber = Math.floor((dayNumber + firstDay - 1) / 7) + 1;
    } else {
      weekNumber = Math.floor((dayNumber + firstDay - 1) / 7);
    }
    if (weekNumber) {
      let year = date.getFullYear();
      if (weekNumber === 53 && lastDay < 3) {
        year++;
        weekNumber = 1;
      }
      return [year, weekNumber];
    }
    const lastWeekOfLastYear = _DateHelper.getWeekNumber(new Date(date.getFullYear() - 1, 11, 31))[1];
    return [date.getFullYear() - 1, lastWeekOfLastYear];
  }
  //endregion
  //region Unit helpers
  /**
   * Turns `(10, 'day')` into `'10 days'` etc.
   * @param {Number} count Amount of unit
   * @param {String} unit Unit, will be normalized (days, d -> day etc.)
   * @returns {String} Amount formatted to string
   * @category Unit helpers
   */
  static formatCount(count, unit) {
    unit = DH.normalizeUnit(unit);
    if (count !== 1)
      unit += "s";
    return count + " " + unit;
  }
  /**
   * Get the ratio between two units ( year, month -> 1/12 ).
   * @param {String} baseUnit Base time unit
   * @param {String} unit Time unit
   * @param {Boolean} [acceptEstimate=false] If `true`, process negative values of validConversions
   * @returns {Number} Ratio
   * @category Unit helpers
   */
  static getUnitToBaseUnitRatio(baseUnit, unit, acceptEstimate = false) {
    baseUnit = DH.normalizeUnit(baseUnit);
    unit = DH.normalizeUnit(unit);
    if (baseUnit === unit)
      return 1;
    if (validConversions[baseUnit] && validConversions[baseUnit][unit] && (acceptEstimate || validConversions[baseUnit][unit] > 0)) {
      return 1 / DH.as(unit, 1, baseUnit);
    }
    if (validConversions[unit] && validConversions[unit][baseUnit] && (acceptEstimate || validConversions[unit][baseUnit] > 0)) {
      return DH.as(baseUnit, 1, unit);
    }
    return -1;
  }
  /**
   * Returns a localized abbreviated form of the name of the duration unit.
   * For example in the `EN` locale, for `'qrt'` it will return `'q'`.
   * @param {String} unit Duration unit
   * @returns {String} Localized abbreviated form of the name of the duration unit
   * @category Unit helpers
   */
  static getShortNameOfUnit(unit) {
    unit = DH.parseTimeUnit(unit);
    return DH.unitLookup[unit].abbrev;
  }
  /**
   * Returns a localized full name of the duration unit.
   *
   * For example in the `EN` locale, for `'d'` it will return either
   * `'day'` or `'days'`, depending on the `plural` argument
   *
   * Preserves casing of first letter.
   *
   * @static
   * @param {String} unit Time unit
   * @param {Boolean} [plural=false] Whether to return a plural name or singular
   * @returns {String} Localized full name of the duration unit
   * @category Unit helpers
   */
  static getLocalizedNameOfUnit(unit, plural = false) {
    const capitalize = unit.charAt(0) === unit.charAt(0).toUpperCase();
    unit = DH.normalizeUnit(unit);
    unit = DH.parseTimeUnit(unit);
    unit = DH.unitLookup[unit][plural ? "plural" : "single"];
    if (capitalize) {
      unit = StringHelper.capitalize(unit);
    }
    return unit;
  }
  /**
   * Normalizes a unit for easier usage in conditionals. For example `'year'`, `'years'`, `'y'` -> `'year'`.
   * @param {String} unit Time unit
   * @returns {String} Normalized unit name
   * @category Unit helpers
   */
  static normalizeUnit(unit) {
    if (!unit) {
      return null;
    }
    const unitLower = unit.toLowerCase();
    if (unitLower === "date") {
      return unitLower;
    }
    return canonicalUnitNames.includes(unitLower) ? unitLower : normalizedUnits[unit] || normalizedUnits[unitLower];
  }
  static getUnitByName(name) {
    return DH.normalizeUnit(name) || DH.normalizeUnit(DH.parseTimeUnit(name));
  }
  /**
   * Returns a duration of the timeframe in the given unit.
   * @param {Date} start The start date of the timeframe
   * @param {Date} end The end date of the timeframe
   * @param {String} unit Duration unit
   * @privateparam {Boolean} [doNotRound]
   * @returns {Number} The duration in the units
   * @category Unit helpers
   * @ignore
   */
  static getDurationInUnit(start, end, unit, doNotRound) {
    return DH.diff(start, end, unit, doNotRound);
  }
  /**
   * Checks if two date units align.
   * @private
   * @param {String} majorUnit Major time unit
   * @param {String} minorUnit Minor time unit
   * @returns {Boolean} `true` if two date units align
   * @category Unit helpers
   */
  static doesUnitsAlign(majorUnit, minorUnit) {
    return !(majorUnit !== minorUnit && minorUnit === "week");
  }
  static getSmallerUnit(unit) {
    return canonicalUnitNames[unitMagnitudes[DH.normalizeUnit(unit)] - 1] || null;
  }
  static getLargerUnit(unit) {
    return canonicalUnitNames[unitMagnitudes[DH.normalizeUnit(unit)] + 1] || null;
  }
  /**
   *
   * Rounds the passed Date value to the nearest `increment` value.
   *
   * Optionally may round relative to a certain base time point.
   *
   * For example `DH.round(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would round to 9:45 because that's the nearest integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when rounding to weeks. The configured {@link #property-weekStartDay-static}
   * dictates what the base of a week is.
   *
   * @param {Date} time The time to round
   * @param {String|Number} increment A millisecond value by which to round the time
   * May be specified in string form eg: `'15 minutes'`
   * @param {Date} [base] The start from which to apply the rounding
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */
  static round(time, increment, base, weekStartDay) {
    return DH.snap("round", time, increment, base, weekStartDay);
  }
  /**
   *
   * Floor the passed Date value to the nearest `increment` value.
   *
   * Optionally may floor relative to a certain base time point.
   *
   * For example `DH.floor(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would floor to 9:15 because that's the closest lower integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when flooring to weeks. The configured {@link #property-weekStartDay-static}
   * dictates what the base of a week is.
   *
   * @param {Date} time The time to floor
   * @param {String|Number|DurationConfig|Object} increment A numeric millisecond value by which to floor the time.
   * or a duration in string form eg `'30 min'` or object form : `{unit: 'minute', magnitude: 30}`
   * or `{unit: 'minute', increment: 30}`
   * @param {Date} [base] The start from which to apply the flooring
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */
  static floor(time, increment, base, weekStartDay) {
    return DH.snap("floor", time, increment, base, weekStartDay);
  }
  /**
   *
   * Ceils the passed Date value to the nearest `increment` value.
   *
   * Optionally may ceil relative to a certain base time point.
   *
   * For example `DH.ceil(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would ceil to 9:45 because that's the closest higher integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when ceiling to weeks. Use weekStartDay argument which default to the configured
   * {@link #property-weekStartDay-static} dictates what the base of a week is
   *
   * @param {Date} time The time to ceil
   * @param {String|Number|DurationConfig|Object} increment A numeric millisecond value by which to ceil the time
   * or a duration in string form eg `'30 min'` or object form : `{unit: 'minute', magnitude: 30}`
   * or `{unit: 'minute', increment: 30}`
   * @param {Date} [base] The start from which to apply the ceiling
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */
  static ceil(time, increment, base, weekStartDay) {
    return DH.snap("ceil", time, increment, base, weekStartDay);
  }
  /**
   * Implementation for round, floor and ceil.
   * @internal
   */
  static snap(operation, time, increment, base, weekStartDay = DH.weekStartDay) {
    const snapFn = snapFns[operation];
    if (typeof increment === "string") {
      increment = DH.parseDuration(increment);
    }
    if (Objects.isObject(increment)) {
      const magnitude = increment.magnitude || increment.increment;
      switch (increment.unit) {
        case "week": {
          const weekDay = time.getDay();
          base = DH.add(
            DH.clearTime(time),
            weekDay >= weekStartDay ? weekStartDay - weekDay : -(weekDay - weekStartDay + 7),
            "day"
          );
          return DH[operation](time, `${magnitude * 7} days`, base);
        }
        case "month": {
          time = DH.asMonths(time);
          let resultMonths;
          if (base) {
            base = DH.asMonths(base);
            resultMonths = time + snapFn(time - base, magnitude);
          } else {
            resultMonths = snapFn(time, magnitude);
          }
          return DH.monthsToDate(resultMonths);
        }
        case "quarter":
          return DH[operation](time, `${magnitude * 3} months`, base);
        case "year":
          return DH[operation](time, `${magnitude * 12} months`, base);
        case "decade":
          return DH[operation](time, `${magnitude * 10} years`, base);
      }
      increment = DH.as("ms", magnitude, increment.unit);
    }
    if (base) {
      const tzChange = DH.as("ms", base.getTimezoneOffset() - time.getTimezoneOffset(), "ms");
      return new Date(base.valueOf() + snapFn(DH.diff(base, time, "ms") + tzChange, increment));
    } else {
      const offset = time.getTimezoneOffset() * 60 * 1e3;
      return new Date(snapFn(time.valueOf() - offset, increment) + offset);
    }
  }
  //endregion
  //region Date picker format
  /**
   * Parses a typed duration value according to locale rules.
   *
   * The value is taken to be a string consisting of the numeric magnitude and the units:
   * - The numeric magnitude can be either an integer or a float value. Both `','` and `'.'` are valid decimal separators.
   * - The units may be a recognised unit abbreviation of this locale or the full local unit name.
   *
   * For example:
   * `'2d'`, `'2 d'`, `'2 day'`, `'2 days'` will be turned into `{ magnitude : 2, unit : 'day' }`
   * `'2.5d'`, `'2,5 d'`, `'2.5 day'`, `'2,5 days'` will be turned into `{ magnitude : 2.5, unit : 'day' }`
   *
   * **NOTE:** Doesn't work with complex values like `'2 days, 2 hours'`
   *
   * @param {String} value The value to parse
   * @param {Boolean} [allowDecimals=true] Decimals are allowed in the magnitude
   * @param {String} [defaultUnit] Default unit to use if only magnitude passed
   * @returns {DurationConfig} If successfully parsed, the result contains two properties, `magnitude` being a number, and
   * `unit` being the canonical unit name, *NOT* a localized name. If parsing was unsuccessful, `null` is returned
   * @category Parse & format
   */
  static parseDuration(value, allowDecimals = true, defaultUnit) {
    var _a2;
    const durationRegEx = allowDecimals ? withDecimalsDurationRegex : noDecimalsDurationRegex, match = durationRegEx.exec(value);
    if (value == null || !match) {
      return null;
    }
    const magnitude = parseNumber((_a2 = match[1]) == null ? void 0 : _a2.replace(",", ".")), unit = DH.parseTimeUnit(match[2]) || defaultUnit;
    if (!unit) {
      return null;
    }
    return {
      magnitude,
      unit
    };
  }
  /**
   * Parses a typed unit name, for example `'ms'` or `'hr'` or `'yr'` into the
   * canonical form of the unit name which may be passed to {@link #function-add-static}
   * or {@link #function-diff-static}.
   * @param {*} unitName Time unit name
   * @category Parse & format
   */
  static parseTimeUnit(unitName) {
    const unitMatch = unitName == null ? null : DH.durationRegEx.exec(unitName.toLowerCase());
    if (!unitMatch) {
      return null;
    }
    for (let unitOrdinal = 0; unitOrdinal < canonicalUnitNames.length; unitOrdinal++) {
      if (unitMatch[unitOrdinal + 1]) {
        return canonicalUnitNames[unitOrdinal];
      }
    }
  }
  //endregion
  //region Internal
  static getGMTOffset(date = /* @__PURE__ */ new Date()) {
    if (!date) {
      return;
    }
    const offsetInMinutes = date.getTimezoneOffset();
    if (!offsetInMinutes)
      return "Z";
    return (offsetInMinutes > 0 ? "-" : "+") + Math.abs(Math.trunc(offsetInMinutes / 60)).toString().padStart(2, "0") + ":" + Math.abs(offsetInMinutes % 60).toString().padStart(2, "0");
  }
  static fillDayNames() {
    const tempDate3 = /* @__PURE__ */ new Date("2000-01-01T12:00:00"), dayNames = DH._dayNames || [], dayShortNames = DH._dayShortNames || [];
    dayNames.length = 0;
    dayShortNames.length = 0;
    for (let day2 = 2; day2 < 9; day2++) {
      tempDate3.setDate(day2);
      dayNames.push(DH.format(tempDate3, "dddd"));
      dayShortNames.push(DH.format(tempDate3, "ddd"));
    }
    DH._dayNames = dayNames;
    DH._dayShortNames = dayShortNames;
  }
  static getDayNames() {
    return DH._dayNames;
  }
  static getDayName(day2) {
    return DH._dayNames[day2];
  }
  static getDayShortNames() {
    return DH._dayShortNames;
  }
  static getDayShortName(day2) {
    return DH._dayShortNames[day2];
  }
  static fillMonthNames() {
    const tempDate3 = /* @__PURE__ */ new Date("2000-01-15T12:00:00"), monthNames = DH._monthNames || [], monthShortNames = DH._monthShortNames || [], monthNamesIndex = {}, monthShortNamesIndex = {};
    monthNames.length = 0;
    monthShortNames.length = 0;
    for (let month2 = 0; month2 < 12; month2++) {
      tempDate3.setMonth(month2);
      const monthName = DH.format(tempDate3, "MMMM");
      monthNames.push(monthName);
      const monthShortName = DH.format(tempDate3, "MMM");
      monthShortNames.push(monthShortName);
      monthNamesIndex[monthName.toLowerCase()] = { name: monthName, value: month2 };
      monthShortNamesIndex[monthShortName.toLowerCase()] = { name: monthShortName, value: month2 };
    }
    DH._monthNames = monthNames;
    DH._monthShortNames = monthShortNames;
    DH._monthNamesIndex = monthNamesIndex;
    DH._monthShortNamesIndex = monthShortNamesIndex;
  }
  static getMonthShortNames() {
    return DH._monthShortNames;
  }
  static getMonthShortName(month2) {
    return DH._monthShortNames[month2];
  }
  static getMonthNames() {
    return DH._monthNames;
  }
  static getMonthName(month2) {
    return DH._monthNames[month2];
  }
  static set locale(name) {
    locale2 = name;
    intlFormatterCache = {};
    formatCache = {};
    formatRedirects = {};
  }
  static get locale() {
    return locale2;
  }
  static setupDurationRegEx(unitNames = [], unitAbbreviations = []) {
    const me = this, unitLookup = {};
    let unitAbbrRegEx = "";
    for (let i = 0; i < unitAbbreviations.length; i++) {
      const abbreviations = unitAbbreviations[i], unitNamesCfg = unitNames[i];
      unitNamesCfg.canonicalUnitName = canonicalUnitNames[i];
      unitLookup[unitNamesCfg.single] = unitLookup[unitNamesCfg.single.toUpperCase()] = unitLookup[unitNamesCfg.canonicalUnitName] = unitLookup[unitNamesCfg.canonicalUnitName.toUpperCase()] = unitNamesCfg;
      unitAbbrRegEx += `${i ? "|" : ""}(`;
      for (let j = 0; j < abbreviations.length; j++) {
        unitAbbrRegEx += `${abbreviations[j]}|`;
      }
      locale2 = me.localize("L{locale}") || "en-US";
      if (locale2 !== "en-US") {
        const canonicalAbbreviations = canonicalUnitAbbreviations[i];
        for (let j = 0; j < canonicalAbbreviations.length; j++) {
          unitAbbrRegEx += `${canonicalAbbreviations[j]}|`;
        }
      }
      unitAbbrRegEx += `${unitNamesCfg.single}|${unitNamesCfg.plural}|${unitNamesCfg.canonicalUnitName}|${unitNamesCfg.canonicalUnitName}s)`;
    }
    me.unitLookup = unitLookup;
    me.durationRegEx = new RegExp(`^(?:${unitAbbrRegEx})$`);
  }
  static applyLocale() {
    const me = this, unitAbbreviations = me.localize("L{unitAbbreviations}") || [], unitNames = me.unitNames = me.localize("L{unitNames}");
    if (unitNames === "unitNames") {
      return;
    }
    locale2 = me.localize("L{locale}") || "en-US";
    if (locale2 === "en-US") {
      ordinalSuffix = enOrdinalSuffix;
    } else {
      ordinalSuffix = me.localize("L{ordinalSuffix}") || ordinalSuffix;
    }
    formatCache = {};
    formatRedirects = {};
    parserCache = {};
    intlFormatterCache = {};
    DH._weekStartDay = null;
    DH.setupDurationRegEx(unitNames, unitAbbreviations);
    DH.fillDayNames();
    DH.fillMonthNames();
  }
  //endregion
};
var DateHelper = _DateHelper;
__publicField(DateHelper, "MS_PER_DAY", MS_PER_HOUR * 24);
var DH = DateHelper;
DH.useIntlFormat = useIntlFormat;
LocaleManager_default.ion({
  locale: "applyLocale",
  prio: 1e3,
  thisObj: DH
});
if (LocaleManager_default.locale) {
  DH.applyLocale();
}
DateHelper._$name = "DateHelper";

// ../Core/lib/Core/helper/ObjectHelper.js
var { hasOwn: hasOwn2 } = Objects;
var toFixedFix = 1.005.toFixed(2) === "1.01" ? null : function(number, fractionDigits) {
  const split2 = number.toString().split("."), newNumber = +(!split2[1] ? split2[0] : split2.join(".") + "1");
  return number.toFixed.call(newNumber, fractionDigits);
};
var ObjectHelper = class extends Objects {
  // These methods are inherited from Objects (an internal class) but need to be documented here for public use.
  // This is primarily because static methods, while inherited by JavaScript classes, are not displayed in derived
  // classes in the docs.
  /**
   * Copies all enumerable properties from the supplied source objects to `dest`. Unlike `Object.assign`, this copy
   * also includes inherited properties.
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method assign
   * @static
   */
  /**
   * Copies all enumerable properties from the supplied source objects to `dest`, only including properties that does
   * not already exist on `dest`. Unlike `Object.assign`, this copy also includes inherited properties.
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method assignIf
   * @static
   */
  /**
   * Creates a deep copy of the `value`. Simple objects ({@link #function-isObject-static}, arrays and `Date` objects
   * are cloned. The enumerable properties of simple objects and the elements of arrays are cloned recursively.
   * @param {*} value The value to clone.
   * @param {Function} [handler] An optional function to call for values of types other than simple object, array or
   * `Date`. This function should return the clone of the `value` passed to it. It is only called for truthy values
   * whose `typeof` equals `'object'`.
   * @param {*} handler.value The value to clone.
   * @returns {*} The cloned value.
   * @method clone
   * @static
   */
  /**
   * Converts a list of names (either a space separated string or an array), into an object with those properties
   * assigned truthy values. The converse of {@link #function-getTruthyKeys-static}.
   * @param {String|String[]} source The list of names to convert to object form.
   * @method createTruthyKeys
   * @static
   */
  /**
   * Gathers the names of properties which have truthy values into an array.
   *
   * This is useful when gathering CSS class names for complex element production.
   * Instead of appending to an array or string which may already contain the
   * name, and instead of contending with space separation and concatenation
   * and conditional execution, just set the properties of an object:
   *
   *     cls = {
   *         [this.selectedCls] : this.isSelected(thing),
   *         [this.dirtyCls] : this.isDirty(thing)
   *     };
   *
   * @param {Object} source Source of keys to gather into an array.
   * @returns {String[]} The keys which had a truthy value.
   * @method getTruthyKeys
   * @static
   */
  /**
   * Gathers the values of properties which are truthy into an array.
   * @param {Object} source Source of values to gather into an array.
   * @returns {String[]} The truthy values from the passed object.
   * @method getTruthyValues
   * @static
   */
  /**
   * Tests whether a passed object has any enumerable properties.
   * @param {Object} object
   * @returns {Boolean} `true` if the passed object has no enumerable properties.
   * @method isEmpty
   * @static
   */
  /**
   * Returns `true` if the `value` is a simple `Object`.
   * @param {Object} value
   * @returns {Boolean} `true` if the `value` is a simple `Object`.
   * @method isObject
   * @static
   */
  /**
   * Copies all enumerable properties from the supplied source objects to `dest`, recursing when the properties of
   * both the source and `dest` are objects.
   * ```
   *  const o = {
   *      a : 1,
   *      b : {
   *          c : 2
   *      }
   *  };
   *  const o2 = {
   *      b : {
   *          d : 3
   *      }
   *  }
   *
   *  console.log(merge(o, o2));
   *
   *  > { a : 1, b : { c : 2, d : 3 } }
   * ```
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method merge
   * @static
   */
  /**
   * Returns the specific type of the given `value`. Unlike the `typeof` operator, this function returns the text
   * from the `Object.prototype.toString` result allowing `Date`, `Array`, `RegExp`, and others to be differentiated.
   * ```
   *  console.log(typeOf(null));
   *  > null
   *
   *  console.log(typeOf({}));
   *  > object
   *
   *  console.log(typeOf([]));
   *  > array
   *
   *  console.log(typeOf(new Date()));
   *  > date
   *
   *  console.log(typeOf(NaN));
   *  > nan
   *
   *  console.log(typeOf(/a/));
   *  > regexp
   * ```
   * @param {*} value
   * @returns {String}
   * @method typeOf
   * @static
   */
  /**
   * Returns value for a given path in the object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {*} Value associated with passed key
   * @method getPath
   * @static
   */
  /**
   * Sets value for a given path in the object
   * @param {Object} object Target object
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @param {*} value Value for a given path
   * @returns {Object} Returns passed object
   * @method setPath
   * @static
   */
  /**
   * Creates a new object where key is a property in array item (`ref` by default) or index in the array and value is array item.
   *
   * From:
   * ```
   * [
   *     {
   *          text : 'foo',
   *          ref : 'fooItem'
   *     },
   *     {
   *          text : 'bar'
   *     }
   * ]
   * ```
   *
   * To:
   * ```
   * {
   *     fooItem : {
   *         text : 'foo',
   *         ref  : 'fooItem'
   *     },
   *     1 : {
   *         text : 'bar'
   *     }
   * }
   * ```
   *
   * @param {Object[]} arrayOfItems Array to transform.
   * @param {String} [prop] Property to read the key from. `ref` by default.
   * @returns {Object} namedItems
   */
  static transformArrayToNamedObject(arrayOfItems, prop = "ref") {
    const namedItems = {};
    arrayOfItems.forEach((item, index) => {
      const key = item[prop] != null && item[prop].toString().length ? item[prop] : index;
      namedItems[key] = item;
    });
    return namedItems;
  }
  /**
   * Creates a new array from object values and saves key in a property (`ref` by default) of each item.
   *
   * From:
   * ```
   * {
   *     fooItem : {
   *         text : 'foo'
   *     },
   *     1 : {
   *         text : 'bar'
   *     },
   *     barItem : false // will be ignored
   * }
   * ```
   *
   * To:
   * ```
   * [
   *     {
   *          text : 'foo',
   *          ref : 'fooItem'
   *     },
   *     {
   *          text : 'bar',
   *          ref : 1
   *     }
   * ]
   * ```
   *
   * @param {Object} namedItems Object to transform.
   * @param {String} [prop] Property to save the key to. `ref` by default.
   * @returns {Object[]} arrayOfItems
   */
  static transformNamedObjectToArray(namedItems, prop = "ref") {
    return Object.keys(namedItems).filter((key) => namedItems[key]).map((key) => {
      const item = namedItems[key];
      item[prop] = key;
      return item;
    });
  }
  /**
   * Checks if two values are equal. Basically === but special handling of dates.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {*} true if values are equal, otherwise false
   */
  static isEqual(a, b, useIsDeeply = false) {
    if (a === null && b !== null || a === void 0 && b !== void 0 || b === null && a !== null || b === void 0 && a !== void 0) {
      return false;
    }
    if (a == null && b == null) {
      return true;
    }
    if (a === b) {
      return true;
    }
    const typeA = typeof a, typeB = typeof b;
    if (typeA === typeB) {
      switch (typeA) {
        case "number":
        case "string":
        case "boolean":
          return a === b;
      }
      switch (true) {
        case (a instanceof Date && b instanceof Date):
          return a.getTime() === b.getTime();
        case (Array.isArray(a) && Array.isArray(b)):
          return a.length === b.length ? a.every((v, idx) => OH.isEqual(v, b[idx], useIsDeeply)) : false;
        case (typeA === "object" && a.constructor.prototype === b.constructor.prototype):
          return useIsDeeply ? OH.isDeeplyEqual(a, b, useIsDeeply) : StringHelper.safeJsonStringify(a) === StringHelper.safeJsonStringify(b);
      }
    }
    return String(a) === String(b);
  }
  /**
   * Checks if two objects are deeply equal
   * @param {Object} a
   * @param {Object} b
   * @param {Object} [options] Additional comparison options
   * @param {Object} [options.ignore] Map of property names to ignore when comparing
   * @param {Function} [options.shouldEvaluate] Function used to evaluate if a property should be compared or not.
   * Return false to prevent comparison
   * @param {Function} [options.evaluate] Function used to evaluate equality. Return `true`/`false` as evaluation
   * result or anything else to let `isEqual` handle the comparison
   * @returns {Boolean}
   */
  static isDeeplyEqual(a, b, options = {}) {
    if (a === b) {
      return true;
    }
    if (!a || !b) {
      return false;
    }
    const aKeys = OH.keys(a, options.ignore), bKeys = OH.keys(b, options.ignore);
    if (aKeys.length !== bKeys.length) {
      return false;
    }
    for (let i = 0; i < aKeys.length; i++) {
      const aKey = aKeys[i], bKey = bKeys[i];
      if (aKey !== bKey) {
        return false;
      }
      const aVal = a[aKey], bVal = b[bKey];
      if (options.shouldEvaluate) {
        if (options.shouldEvaluate(
          aKey,
          {
            value: aVal,
            object: a
          },
          {
            value: bVal,
            object: b
          }
        ) === false) {
          continue;
        }
      }
      if (options.evaluate) {
        const result = options.evaluate(aKey, {
          value: aVal,
          object: a
        }, {
          value: bVal,
          object: b
        });
        if (result === false) {
          return false;
        }
        if (result === true) {
          continue;
        }
      }
      if (!OH.isEqual(aVal, bVal, options)) {
        return false;
      }
    }
    return true;
  }
  /**
   * Checks if value B is partially equal to value A.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if values are partially equal, false otherwise
   */
  static isPartial(a, b) {
    a = String(a).toLowerCase();
    b = String(b).toLowerCase();
    return a.indexOf(b) !== -1;
  }
  /**
   * Checks if value a is smaller than value b.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if a < b
   */
  static isLessThan(a, b) {
    if (a instanceof Date && b instanceof Date) {
      return DateHelper.isBefore(a, b);
    }
    return a < b;
  }
  /**
   * Checks if value a is bigger than value b.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if a > b
   */
  static isMoreThan(a, b) {
    if (a instanceof Date && b instanceof Date) {
      return DateHelper.isAfter(a, b);
    }
    return a > b;
  }
  /**
   * Used by the Base class to make deep copies of defaultConfig blocks
   * @private
   */
  static fork(obj) {
    let ret, key, value;
    if (obj && obj.constructor === Object) {
      ret = Object.setPrototypeOf({}, obj);
      for (key in obj) {
        value = obj[key];
        if (value) {
          if (value.constructor === Object) {
            ret[key] = OH.fork(value);
          } else if (value instanceof Array) {
            ret[key] = value.slice();
          }
        }
      }
    } else {
      ret = obj;
    }
    return ret;
  }
  /**
   * Copies the named properties from the `source` parameter into the `dest` parameter.
   * @param {Object} dest The destination into which properties are copied.
   * @param {Object} source The source from which properties are copied.
   * @param {String[]} props The list of property names.
   * @returns {Object} The `dest` object.
   */
  static copyProperties(dest, source, props) {
    let prop, i;
    for (i = 0; i < props.length; i++) {
      prop = props[i];
      if (prop in source) {
        dest[prop] = source[prop];
      }
    }
    return dest;
  }
  /**
   * Copies the named properties from the `source` parameter into the `dest` parameter
   * unless the property already exists in the `dest`.
   * @param {Object} dest The destination into which properties are copied.
   * @param {Object} source The source from which properties are copied.
   * @param {String[]} props The list of property names.
   * @returns {Object} The `dest` object.
   */
  static copyPropertiesIf(dest, source, props) {
    if (source) {
      for (const prop of props) {
        if (!(prop in dest) && prop in source) {
          dest[prop] = source[prop];
        }
      }
    }
    return dest;
  }
  /**
   * Returns an array containing the keys and values of all enumerable properties from every prototype level for the
   * object. If `object` is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve entries.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @returns {Array}
   * @internal
   */
  static entries(object, ignore) {
    const result = [], call = typeof ignore === "function";
    if (object) {
      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore == null ? void 0 : ignore[p])) {
          result.push([p, object[p]]);
        }
      }
    }
    return result;
  }
  /**
   * Populates an `object` with the provided `entries`.
   * @param {Array} entries The key/value pairs (2-element arrays).
   * @param {Object} [object={}] The object onto which to add `entries`.
   * @returns {Object} The passed `object` (by default, a newly created object).
   * @internal
   */
  static fromEntries(entries, object) {
    object = object || {};
    if (entries) {
      for (let i = 0; i < entries.length; ++i) {
        object[entries[i][0]] = entries[i][1];
      }
    }
    return object;
  }
  /**
   * Returns an array containing all enumerable property names from every prototype level for the object. If `object`
   * is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve property names.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
   * function is stored in the returned array. It is called with the array element as the first parameter, and the
   * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
   * on).
   * @returns {String[]}
   */
  static keys(object, ignore, mapper) {
    const result = [], call = typeof ignore === "function";
    if (object) {
      let index = 0;
      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore == null ? void 0 : ignore[p])) {
          result.push(mapper ? mapper(p, index) : p);
          ++index;
        }
      }
    }
    return result;
  }
  /**
   * Returns an array containing the values of all enumerable properties from every prototype level for the object.
   * If `object` is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve values.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
   * function is stored in the returned array. It is called with the array element as the first parameter, and the
   * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
   * on).
   * @returns {Array}
   * @internal
   */
  static values(object, ignore, mapper) {
    const result = [], call = typeof ignore === "function";
    if (object) {
      let index = 0;
      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore == null ? void 0 : ignore[p])) {
          result.push(mapper ? mapper(object[p], index) : object[p]);
          ++index;
        }
      }
    }
    return result;
  }
  //region Path
  /**
   * Checks if a given path exists in an object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {Boolean} Returns `true` if path exists or `false` if it does not
   */
  static pathExists(object, path) {
    const properties = path.split(".");
    return properties.every((property) => {
      if (!object || !(property in object)) {
        return false;
      }
      object = object[property];
      return true;
    });
  }
  /**
   * Creates a simple single level key-value object from complex deep object.
   * @param {Object} object Object to extract path and values from
   * @returns {Object} Key-value object where key is a path to the corresponding value
   * @internal
   *
   * ```javascript
   * // converts deep object
   * {
   *     foo : {
   *         bar : {
   *             test : 1
   *         }
   *     }
   * }
   * // into a single level object
   * {
   *     'foo.bar.test' : 1
   * }
   * ```
   */
  static pathifyKeys(object, fieldDataSourceMap) {
    const result = {};
    for (const key in object) {
      if (hasOwn2(object, key)) {
        const field2 = fieldDataSourceMap == null ? void 0 : fieldDataSourceMap[key];
        const usesPathKeys = (field2 == null ? void 0 : field2.type) === "object" || (field2 == null ? void 0 : field2.complexMapping) || !Boolean(fieldDataSourceMap);
        if (usesPathKeys && Array.isArray(object[key])) {
          result[key] = object[key].slice();
        } else if (usesPathKeys && object[key] instanceof Object) {
          const paths = this.pathifyKeys(object[key]);
          for (const path in paths) {
            result[`${key}.${path}`] = paths[path];
          }
        } else {
          result[key] = object[key];
        }
      }
    }
    return result;
  }
  /**
   * Removes value for a given path in the object. Doesn't cleanup empty objects.
   * @param {Object} object
   * @param {String} path Dot-separated path, e.g. `obj.child.someKey`
   * @internal
   */
  static deletePath(object, path) {
    path.split(".").reduce((result, key, index, array) => {
      if (result == null) {
        return null;
      }
      if (hasOwn2(result, key)) {
        if (index === array.length - 1) {
          delete result[key];
        } else {
          return result[key];
        }
      }
    }, object);
  }
  //endregion
  static coerce(from, to) {
    const fromType = Objects.typeOf(from), toType = Objects.typeOf(to), isString = typeof from === "string";
    if (fromType !== toType) {
      switch (toType) {
        case "string":
          return String(from);
        case "number":
          return Number(from);
        case "boolean":
          return isString && (!from || from === "false" || from === "0") ? false : Boolean(from);
        case "null":
          return isString && (!from || from === "null") ? null : false;
        case "undefined":
          return isString && (!from || from === "undefined") ? void 0 : false;
        case "date":
          return isString && isNaN(from) ? DateHelper.parse(from) : Date(Number(from));
      }
    }
    return from;
  }
  static wrapProperty(object, propertyName, newGetter, newSetter, deep = true) {
    const newProperty = {};
    let proto4 = Object.getPrototypeOf(object), existingProperty = Object.getOwnPropertyDescriptor(proto4, propertyName);
    while (!existingProperty && proto4 && deep) {
      proto4 = Object.getPrototypeOf(proto4);
      if (proto4) {
        existingProperty = Object.getOwnPropertyDescriptor(proto4, propertyName);
      }
    }
    if (existingProperty) {
      if (existingProperty.set) {
        newProperty.set = (v) => {
          existingProperty.set.call(object, v);
          newSetter && newSetter.call(object, existingProperty.get.call(object));
        };
      } else {
        newProperty.set = newSetter;
      }
      if (existingProperty.get) {
        newProperty.get = () => {
          let result = existingProperty.get.call(object);
          if (newGetter) {
            result = newGetter.call(object, result);
          }
          return result;
        };
      } else {
        newProperty.get = newGetter;
      }
    } else {
      newProperty.set = (v) => {
        object[`_${propertyName}`] = v;
        newSetter && newSetter.call(object, v);
      };
      newProperty.get = () => {
        let result = object[`_${propertyName}`];
        if (newGetter) {
          result = newGetter.call(object, result);
        }
        return result;
      };
    }
    Object.defineProperty(object, propertyName, newProperty);
  }
  /**
   * Intercepts access to a `property` of a given `object`.
   *
   * ```javascript
   *      ObjectHelper.hookProperty(object, 'prop', class {
   *          get value() {
   *              return super.value;
   *          }
   *          set value(v) {
   *              super.value = v;
   *          }
   *      });
   * ```
   * The use of `super` allows the hook's getter and setter to invoke the object's existing get/set.
   *
   * @param {Object} object
   * @param {String} property
   * @param {Function} hook A `class` defining a `value` property getter and/or setter.
   * @returns {Function} A function that removes the hook when called.
   * @internal
   */
  static hookProperty(object, property, hook) {
    const desc = ObjectHelper.getPropertyDescriptor(hook.prototype, "value"), existingDesc = ObjectHelper.getPropertyDescriptor(object, property), fieldName = `_${property}`, base = class {
      get value() {
        return existingDesc ? existingDesc.get.call(this) : this[fieldName];
      }
      set value(v) {
        if (existingDesc) {
          existingDesc.set.call(this, v);
        } else {
          this[fieldName] = v;
        }
      }
    }, baseDesc = ObjectHelper.getPropertyDescriptor(base.prototype, "value");
    Object.setPrototypeOf(hook.prototype, base.prototype);
    Object.defineProperty(object, property, {
      configurable: true,
      get: desc.get || baseDesc.get,
      set: desc.set || baseDesc.set
    });
    return () => delete object[property];
  }
  /**
   * Finds a property descriptor for the passed object from all inheritance levels.
   * @param {Object} object The Object whose property to find.
   * @param {String} propertyName The name of the property to find.
   * @returns {Object} An ECMA property descriptor is the property was found, otherwise `null`
   */
  static getPropertyDescriptor(object, propertyName) {
    let result = null;
    for (let o = object; o && !result && !hasOwn2(o, "isBase"); o = Object.getPrototypeOf(o)) {
      result = Object.getOwnPropertyDescriptor(o, propertyName);
    }
    return result;
  }
  /**
   * Changes the passed object and removes all null and undefined properties from it
   * @param {Object} object Target object
   * @param {Boolean} [keepNull] Pass true to only remove undefined properties
   * @returns {Object} Passed object
   */
  static cleanupProperties(object, keepNull = false) {
    Object.entries(object).forEach(([key, value]) => {
      if (keepNull) {
        value === void 0 && delete object[key];
      } else {
        value == null && delete object[key];
      }
    });
    return object;
  }
  /**
   * Changes the passed object and removes all properties from it.
   * Used while mutating when need to keep reference to the object but replace its properties.
   * @param {Object} object Target object
   * @returns {Object} Passed object
   */
  static removeAllProperties(obj) {
    Object.keys(obj).forEach((key) => delete obj[key]);
    return obj;
  }
  //region Assert type
  /**
   * Checks that the supplied value is of the specified type.Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} type Expected type
   * @param {String} name Name of the value, used in error message
   * @param {Boolean} [allowNull] Accept `null` without throwing
   */
  static assertType(value, type, name) {
    const valueType = Objects.typeOf(value);
    if (value != null && valueType !== type) {
      throw new Error(`Incorrect type "${valueType}" for ${name}, expected "${type}"`);
    }
  }
  /**
   * Checks that the supplied value is a plain object. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertObject(value, name) {
    OH.assertType(value, "object", name);
  }
  /**
   * Checks that the supplied value is an instance of a Bryntum class. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertInstance(value, name) {
    OH.assertType(value, "instance", name);
  }
  /**
   * Checks that the supplied value is a Bryntum class. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertClass(value, name) {
    OH.assertType(value, "class", name);
  }
  /**
   * Checks that the supplied value is a function. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertFunction(value, name) {
    if (typeof value !== "function" || value.isBase || value.$$name) {
      throw new Error(`Incorrect type for ${name}, got "${value}" (expected a function)`);
    }
  }
  /**
   * Checks that the supplied value is a number. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertNumber(value, name) {
    const asNumber = Number(value);
    if (typeof value !== "number" || isNaN(asNumber)) {
      throw new Error(`Incorrect type for ${name}, got "${value}" (expected a Number)`);
    }
  }
  /**
   * Checks that the supplied value is a boolean. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertBoolean(value, name) {
    OH.assertType(value, "boolean", name);
  }
  /**
   * Checks that the supplied value is a string. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertString(value, name) {
    OH.assertType(value, "string", name);
  }
  /**
   * Checks that the supplied value is an array. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */
  static assertArray(value, name) {
    OH.assertType(value, "array", name);
  }
  //endregion
  /**
   * Number.toFixed(), with polyfill for browsers that needs it
   * @param {Number} number
   * @param {Number} digits
   * @returns {String} A fixed point string representation of the passed number.
   */
  static toFixed(number, digits) {
    if (toFixedFix) {
      return toFixedFix(number, digits);
    }
    return number.toFixed(digits);
  }
  /**
   * Round the passed number to closest passed step value.
   * @param {Number} number The number to round.
   * @param {Number} [step] The step value to round to.
   * @returns {Number} The number rounded to the closest step.
   */
  static roundTo(number, step = 1) {
    return Math.round(number / step) * step;
  }
  /**
   * Round the passed number to the passed number of decimals.
   * @param {Number} number The number to round.
   * @param {Number} digits The number of decimal places to round to.
   * @returns {Number} The number rounded to the passed number of decimal places.
   */
  static round(number, digits) {
    if (digits == null) {
      return number;
    }
    const factor = 10 ** digits;
    return Math.round(number * factor) / factor;
  }
  /**
   * Returns a non-null entry from a Map for a given key path. This enables a specified defaultValue to be added "just
   * in time" which is returned if the key is not already present.
   * @param {Map} map The Map to find the key in (and potentially add to).
   * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
   * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
   * @param {Object} [defaultValue] Optionally the value to insert if the key is not found.
   */
  static getMapPath(map2, path, defaultValue2) {
    const keyPath = Array.isArray(path) ? path : typeof path === "string" ? path.split(".") : [path], simpleKey = keyPath.length === 1, topKey = keyPath[0], topValue = map2.has(topKey) ? map2.get(topKey) : map2.set(topKey, simpleKey ? defaultValue2 : {}).get(topKey);
    if (simpleKey) {
      return topValue;
    }
    return OH.getPathDefault(topValue, keyPath.slice(1), defaultValue2);
  }
};
var OH = ObjectHelper;
ObjectHelper._$name = "ObjectHelper";

// ../Core/lib/Core/helper/util/Rectangle.js
var allBorders = ["border-top-width", "border-right-width", "border-bottom-width", "border-left-width"];
var allMargins = ["margin-top", "margin-right", "margin-bottom", "margin-left"];
var allPaddings = ["padding-top", "padding-right", "padding-bottom", "padding-left"];
var borderNames = {
  t: "border-top-width",
  r: "border-right-width",
  b: "border-bottom-width",
  l: "border-left-width"
};
var paddingNames = {
  t: "padding-top",
  r: "padding-right",
  b: "padding-bottom",
  l: "padding-left"
};
var zeroBased = Object.freeze({
  x: 0,
  y: 0
});
var alignSpecRe = /^([trblc])(\d*)-([trblc])(\d*)$/i;
var alignPointRe = /^([trblc])(\d*)$/i;
var edgeNames = [
  "top",
  "right",
  "bottom",
  "left"
];
var edgeIndices = {
  t: 0,
  r: 1,
  b: 2,
  l: 3
};
var defaultAlignments = [
  "b-t",
  "l-r",
  "t-b",
  "r-l"
];
var edgeAligments = {
  bt: 1,
  tb: 1,
  lr: 2,
  rl: 2
};
var zeroOffsets = Object.freeze([0, 0]);
var matchDimensions = ["width", "height"];
var parseEdges = (top, right = top, bottom = top, left = right) => {
  return Array.isArray(top) ? parseEdges.apply(null, top) : [top, right, bottom, left];
};
function parseAlign(alignSpec, rtl) {
  const parts = alignSpecRe.exec(alignSpec), myEdge = parts[1], targetEdge = parts[3], mO = parseInt(parts[2] || 50), tO = parseInt(parts[4] || 50), myOffset = rtl && !(edgeIndices[myEdge] & 1) ? 100 - mO : mO, targetOffset = rtl && !(edgeIndices[targetEdge] & 1) ? 100 - tO : tO, edgeAligned = edgeAligments[myEdge + targetEdge];
  return {
    myAlignmentPoint: myEdge + myOffset,
    // l0
    myEdge,
    // l
    myOffset,
    // 0
    targetAlignmentPoint: targetEdge + targetOffset,
    // r0
    targetEdge,
    // r
    targetOffset,
    // 0
    startZone: edgeIndices[targetEdge],
    // 1 - start trying zone 1 in TRBL order
    edgeAligned
    // Edge-to-edge align requested
  };
}
function flipAlign(align) {
  return `${edgeNames[(edgeIndices[align.myEdge] + 2) % 4][0]}${align.myOffset}-${edgeNames[(edgeIndices[align.targetEdge] + 2) % 4][0]}${align.targetOffset}`;
}
function createOffsets(offset) {
  if (offset == null) {
    return zeroOffsets;
  } else if (typeof offset === "number") {
    return [offset, offset];
  }
  return offset;
}
var Rectangle = class {
  // Class does not extend Base, so we need to define this
  get isRectangle() {
    return true;
  }
  /**
   * Returns the Rectangle in document based coordinates of the passed element.
   *
   * *Note:* If the element passed is the `document` or `window` the `window`'s
   * rectangle is returned which is always at `[0, 0]` and encompasses the
   * browser's entire document viewport.
   * @param {HTMLElement|Core.widget.Widget|Core.widget.Mask} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll=false] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */
  static from(element, relativeTo, ignorePageScroll) {
    if (typeof element === "string") {
      element = document.querySelector(element);
    } else if ((element == null ? void 0 : element.nodeType) === Node.DOCUMENT_FRAGMENT_NODE) {
      element = element.host || element.ownerDocument;
    }
    if (typeof relativeTo === "string") {
      relativeTo = document.querySelector(relativeTo);
    }
    if (element == null || element.isRectangle) {
      return element;
    }
    element = element.element || element;
    if (ignorePageScroll === void 0 && typeof relativeTo === "boolean") {
      ignorePageScroll = relativeTo;
      relativeTo = null;
    }
    if (!(relativeTo == null ? void 0 : relativeTo.isRectangle)) {
      if (relativeTo) {
        let { scrollLeft, scrollTop } = relativeTo;
        if (BrowserHelper.isSafari && relativeTo === document.body) {
          scrollLeft = scrollTop = 0;
        }
        relativeTo = Rectangle.from(relativeTo).translate(-scrollLeft, -scrollTop);
      } else {
        relativeTo = zeroBased;
      }
    }
    const isViewport = element === document || element === globalThis, isSFViewport = element === document.body && document.body.offsetHeight === 0, sfElRect = isSFViewport && element.getBoundingClientRect(), viewRect = isSFViewport ? new Rectangle(sfElRect.left, sfElRect.top, sfElRect.width, document.body.parentElement.scrollHeight) : isViewport ? new Rectangle(0, 0, globalThis.innerWidth, globalThis.innerHeight) : element.getBoundingClientRect(), scrollOffset = ignorePageScroll || isViewport ? [0, 0] : [globalThis.pageXOffset, globalThis.pageYOffset];
    return new Rectangle(
      viewRect.left + scrollOffset[0] - relativeTo.x,
      viewRect.top + scrollOffset[1] - relativeTo.y,
      viewRect.width,
      viewRect.height
    );
  }
  /**
   * Returns the Rectangle in viewport coordinates of the passed element.
   *
   * *Note:* If the element passed is the `document` or `window` the `window`'s rectangle is returned which is always
   * at `[0, 0]` and encompasses the browser's entire document viewport.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @returns {Core.helper.util.Rectangle} The Rectangle in viewport based coordinates. Relative to the _relativeTo_
   * parameter if provided.
   */
  static fromScreen(element, relativeTo) {
    return Rectangle.from(
      element,
      relativeTo,
      /* ignorePageScroll = */
      true
    );
  }
  /**
   * Returns the inner Rectangle (within border) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */
  static inner(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll);
    if (document.body.contains(element)) {
      const borders = DomHelper.getStyleValue(element, allBorders);
      result.x += parseFloat(borders[borderNames.l]);
      result.y += parseFloat(borders[borderNames.t]);
      result.right -= parseFloat(borders[borderNames.r]);
      result.bottom -= parseFloat(borders[borderNames.b]);
    }
    return result;
  }
  /**
   * Returns the content Rectangle (within border and padding) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */
  static content(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll);
    if (document.body.contains(element)) {
      const borders = DomHelper.getStyleValue(element, allBorders), padding = DomHelper.getStyleValue(element, allPaddings);
      result.x += parseFloat(borders[borderNames.l]) + parseFloat(padding[paddingNames.l]);
      result.y += parseFloat(borders[borderNames.t]) + parseFloat(padding[paddingNames.t]);
      result.right -= parseFloat(borders[borderNames.r]) + parseFloat(padding[paddingNames.r]);
      result.bottom -= parseFloat(borders[borderNames.b]) + parseFloat(padding[paddingNames.b]);
    }
    return result;
  }
  /**
   * Returns the client Rectangle (within border and padding and scrollbars) in document based coordinates of the
   * passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */
  static client(element, relativeTo, ignorePageScroll = false) {
    const result = this.content(element, relativeTo, ignorePageScroll), scrollbarWidth = DomHelper.scrollBarWidth;
    let padding;
    if (scrollbarWidth) {
      if (element.scrollHeight > element.clientHeight && DomHelper.getStyleValue(element, "overflow-y") !== "hidden") {
        padding = parseFloat(DomHelper.getStyleValue(element, "padding-right"));
        result.right += padding - Math.max(padding, scrollbarWidth);
      }
      if (element.scrollWidth > element.clientWidth && DomHelper.getStyleValue(element, "overflow-x") !== "hidden") {
        padding = parseFloat(DomHelper.getStyleValue(element, "padding-bottom"));
        result.bottom += padding - Math.max(padding, scrollbarWidth);
      }
    }
    return result;
  }
  /**
   * Returns the outer Rectangle (including margin) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates.
   * Relative to the _relativeTo_ parameter if passed.
   * @internal
   */
  static outer(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll);
    if (document.body.contains(element)) {
      const margin = DomHelper.getStyleValue(element, allMargins);
      result.x -= parseFloat(margin["margin-left"]);
      result.y -= parseFloat(margin["margin-top"]);
      result.right += parseFloat(margin["margin-right"]);
      result.bottom += parseFloat(margin["margin-bottom"]);
    }
    return result;
  }
  /**
   * Returns a new rectangle created as the union of all supplied rectangles.
   * @param {Core.helper.util.Rectangle[]} rectangles
   * @returns {Core.helper.util.Rectangle}
   */
  static union(...rectangles) {
    let { x, y, right, bottom } = rectangles[0], current;
    if (rectangles.length > 1) {
      for (let i = 1; i < rectangles.length; i++) {
        current = rectangles[i];
        if (current.x < x) {
          x = current.x;
        }
        if (current.y < y) {
          y = current.y;
        }
        if (current.right > right) {
          right = current.right;
        }
        if (current.bottom > bottom) {
          bottom = current.bottom;
        }
      }
    }
    return new Rectangle(x, y, right - x, bottom - y);
  }
  /**
   * Rounds this Rectangle to the pixel resolution of the current display or to the nearest
   * passed unit which defaults to the current display's [`devicePixelRatio`](https://developer.mozilla.org/en-US/docs/Web/API/Window/devicePixelRatio).
   * @param {Number} [devicePixelRatio] device pixel ratio which defaults to `window.devicePixelRatio`
   */
  roundPx(devicePixelRatio = globalThis.devicePixelRatio || 1) {
    const me = this;
    me._x = DomHelper.roundPx(me._x, devicePixelRatio);
    me._y = DomHelper.roundPx(me._y, devicePixelRatio);
    me._width = DomHelper.roundPx(me._width, devicePixelRatio);
    me._height = DomHelper.roundPx(me._height, devicePixelRatio);
    return me;
  }
  // This class doesn't extend Base and extending doesn't seem to be
  // the way to go. Instead we duplicate smallest piece of logic here
  static get $$name() {
    return hasOwnProperty.call(this, "$name") && this.$name || // _$name is filled by webpack for every class (cls._$name = '...')
    hasOwnProperty.call(this, "_$name") && this._$name || this.name;
  }
  get $$name() {
    return this.constructor.$$name;
  }
  /**
   * Constructs a Rectangle
   * @param {Number} x The X coordinate
   * @param {Number} y The Y coordinate
   * @param {Number} width The width
   * @param {Number} height The height
   */
  constructor(x, y, width, height) {
    ObjectHelper.assertNumber(x, "Rectangle.x");
    ObjectHelper.assertNumber(y, "Rectangle.y");
    ObjectHelper.assertNumber(width, "Rectangle.width");
    ObjectHelper.assertNumber(height, "Rectangle.height");
    const me = this;
    if (width < 0) {
      x += width;
      width = -width;
    }
    if (height < 0) {
      y += height;
      height = -height;
    }
    me._x = x;
    me._y = y;
    me._width = width;
    me._height = height;
  }
  /**
   * Creates a copy of this Rectangle.
   */
  clone() {
    const me = this, result = new Rectangle(me.x, me.y, me.width, me.height);
    result.isAlignRectangle = me.isAlignRectangle;
    result.minHeight = me.minHeight;
    result.minWidth = me.minWidth;
    return result;
  }
  /**
   * Returns `true` if this Rectangle wholly contains the passed rectangle.
   *
   * Note that a {@link Core.helper.util.Point} may be passed.
   * @param {Core.helper.util.Rectangle} other The Rectangle to test for containment within this Rectangle
   * @returns {Boolean} `true` if the other Rectangle is wholly contained within this Rectangle
   */
  contains(other) {
    const me = this;
    if (other.isRectangle) {
      return other._x >= me._x && other._y >= me._y && other.right <= me.right && other.bottom <= me.bottom;
    } else {
      return false;
    }
  }
  /**
   * Checks if this Rectangle intersects the passed Rectangle
   * @param {Core.helper.util.Rectangle} other The Rectangle to intersect with this.
   * @param {Boolean} [useBoolean] Specify `true` to return a boolean value instead of constructing a new Rectangle
   * @param {Boolean} [allowZeroDimensions] `true` to consider zero-width or zero-hight rectangles as intersecting if coordinates indicate the intersection
   * @returns {Core.helper.util.Rectangle|Boolean} Returns the intersection Rectangle or `false` if there is no intersection.
   */
  intersect(other, useBoolean = false, allowZeroDimensions = false) {
    const me = this, y = Math.max(me.y, other.y), r = Math.min(me.right, other.right), b = Math.min(me.bottom, other.bottom), x = Math.max(me.x, other.x), intersect = allowZeroDimensions ? b >= y && r >= x : b > y && r > x;
    if (intersect) {
      return useBoolean ? true : new Rectangle(x, y, r - x, b - y);
    } else {
      return false;
    }
  }
  equals(other, round2 = false) {
    const processor = round2 ? (x) => Math.round(x) : (x) => x;
    return other.isRectangle && processor(other.x) === processor(this.x) && processor(other.y) === processor(this.y) && processor(other.width) === processor(this.width) && processor(other.height) === processor(this.height);
  }
  /**
   * Translates this Rectangle by the passed vector. Size is maintained.
   * @param {Number} x The X translation vector.
   * @param {Number} y The Y translation vector.
   * @returns {Core.helper.util.Rectangle} This Rectangle;
   */
  translate(x, y) {
    this._x += x || 0;
    this._y += y || 0;
    return this;
  }
  /**
   * Moves this Rectangle to the passed `x`, `y` position. Size is maintained.
   * @param {Number} x The new X position.
   * @param {Number} y The new Y position.
   * @returns {Core.helper.util.Rectangle}  This Rectangle;
   */
  moveTo(x, y) {
    if (x != null) {
      this._x = x;
    }
    if (y != null) {
      this._y = y;
    }
    return this;
  }
  /**
   * Returns the vector which would translate this Rectangle (or Point) to the same position as the other Rectangle (or point)
   * @param {Core.helper.util.Rectangle|Core.helper.util.Point} other The Rectangle or Point to calculate the delta to.
   * @returns {Array} Returns a vector using format `[deltaX, deltaY]`
   * @internal
   */
  getDelta(other) {
    return [other.x - this.x, other.y - this.y];
  }
  /**
   * The center point of this rectangle.
   * @property {Core.helper.util.Point}
   */
  get center() {
    return new Rectangle.Point(this.x + this.width / 2, this.y + this.height / 2);
  }
  /**
   * Get/sets the X coordinate of the Rectangle. Note that this does *not* translate the
   * Rectangle. The requested {@link #property-width} will change.
   * @property {Number}
   */
  set x(x) {
    const xDelta = x - this._x;
    this._x = x;
    this._width -= xDelta;
  }
  get x() {
    return this._x;
  }
  get start() {
    return this.left;
  }
  /**
   * Alias for x. To match DOMRect.
   * @property {Number}
   */
  set left(x) {
    this.x = x;
  }
  get left() {
    return this.x;
  }
  /**
   * Alias for y. To match DOMRect.
   * @property {Number}
   */
  set top(y) {
    this.y = y;
  }
  get top() {
    return this.y;
  }
  /**
   * Get/sets the Y coordinate of the Rectangle. Note that this does *not* translate the
   * Rectangle. The requested {@link #property-height} will change.
   * @property {Number}
   */
  set y(y) {
    const yDelta = y - this._y;
    this._y = y;
    this._height -= yDelta;
  }
  get y() {
    return this._y;
  }
  /**
   * Get/sets the width of the Rectangle. Note that the requested {@link #property-right} will change.
   * @property {Number}
   */
  set width(width) {
    this._width = width;
  }
  get width() {
    return this._width;
  }
  /**
   * Get/sets the height of the Rectangle. Note that the requested {@link #property-bottom} will change.
   * @property {Number}
   */
  set height(height) {
    this._height = height;
  }
  get height() {
    return this._height;
  }
  /**
   * Get/sets the right edge of the Rectangle. Note that the requested {@link #property-width} will change.
   *
   * The right edge value is exclusive of the calculated rectangle width. So x=0 and right=10
   * means a width of 10.
   * @property {Number}
   */
  set right(right) {
    this._width = right - this._x;
  }
  get right() {
    return this._x + this._width;
  }
  get end() {
    return this.right;
  }
  /**
   * Get/sets the bottom edge of the Rectangle. Note that the requested {@link #property-height} will change.
   *
   * The bottom edge value is exclusive of the calculated rectangle height. So y=0 and bottom=10
   * means a height of 10.
   * @property {Number}
   */
  set bottom(bottom) {
    this._height = bottom - this._y;
  }
  get bottom() {
    return this._y + this._height;
  }
  getStart(rtl, horizontal = true) {
    if (horizontal) {
      return rtl ? this.right : this.left;
    }
    return this.top;
  }
  getEnd(rtl, horizontal = true) {
    if (horizontal) {
      return rtl ? this.left : this.right;
    }
    return this.bottom;
  }
  get area() {
    return this.width * this.height;
  }
  set minWidth(minWidth) {
    const me = this;
    if (isNaN(minWidth)) {
      me._minWidth = null;
    } else {
      me._minWidth = Number(minWidth);
      if (!me.isAlignRectangle) {
        me.width = Math.max(me.width, me._minWidth);
      }
    }
  }
  get minWidth() {
    return this._minWidth;
  }
  set minHeight(minHeight) {
    const me = this;
    if (isNaN(minHeight)) {
      me._minHeight = null;
    } else {
      me._minHeight = Number(minHeight);
      if (!me.isAlignRectangle) {
        me.height = Math.max(me.height, me._minHeight);
      }
    }
  }
  get minHeight() {
    return this._minHeight;
  }
  /**
   * Modifies the bounds of this Rectangle by the specified deltas.
   * @param {Number} x How much to *add* to the x position.
   * @param {Number} y  How much to *add* to the y position.
   * @param {Number} width  How much to add to the width.
   * @param {Number} height  How much to add to the height.
   * @returns {Core.helper.util.Rectangle} This Rectangle
   */
  adjust(x, y, width, height) {
    const me = this;
    me.x += x;
    me.y += y;
    me.width += width;
    me.height += height;
    return me;
  }
  /**
   * Modifies the bounds of this rectangle by expanding them by the specified amount in all directions.
   * The parameters are read the same way as CSS margin values.
   *
   * - If one value is passed, all edges are inflated by that value.
   * - If two values are passed they are top/bottom inflation and left/right inflation.
   * - If four values are passed, that's top, right, bottom, left.
   * @param {Number} amount How much to inflate, or the top value if more than one values passed.
   * @param {Number} right How much to inflate the right side, or both left and right is only two values passed.
   * @param {Number} bottom How much to inflate the bottom side.
   * @param {Number} left How much to inflate the left side.
   * @returns {Core.helper.util.Rectangle} This Rectangle
   * @internal
   */
  inflate(top, right = top, bottom = top, left = right) {
    if (Array.isArray(top)) {
      [top, right, bottom, left] = parseEdges(top);
    }
    return this.adjust(-left, -top, right, bottom);
  }
  /**
   * Modifies the bounds of this rectangle by reducing them by the specified amount in all directions.
   * The parameters are read the same way as CSS margin values.
   *
   * - If one value is passed, all edges are deflated by that value.
   * - If two values are passed they are top/bottom deflation and left/right deflation.
   * - If four values are passed, that's top, right, bottom, left.
   * @param {Number} amount How much to deflate, or the top value if more than one values passed.
   * @param {Number} right How much to deflate the right side, or both left and right is only two values passed.
   * @param {Number} bottom How much to deflate the bottom side.
   * @param {Number} left How much to deflate the left side.
   * @returns {Core.helper.util.Rectangle} This Rectangle
   * @internal
   */
  deflate(top, right = top, bottom = top, left = right) {
    if (Array.isArray(top)) {
      [top, right, bottom, left] = parseEdges(top);
    }
    return this.adjust(left, top, -right, -bottom);
  }
  /**
   * Attempts constrain this Rectangle into the passed Rectangle. If the `strict` parameter is `true`
   * then this method will return `false` if constraint could not be achieved.
   *
   * If this Rectangle has a `minHeight` or `minWidth` property, size will be adjusted while attempting to constrain.
   *
   * Right and bottom are adjusted first leaving the top and bottom sides to "win" in the case that this Rectangle overflows
   * the constrainTo Rectangle.
   * @param {Core.helper.util.Rectangle} constrainTo The Rectangle to constrain this Rectangle into if possible.
   * @param {Boolean} strict Pass `true` to return false, and leave this Rectangle unchanged if constraint
   * could not be achieved.
   * @returns {Core.helper.util.Rectangle|Boolean} This Rectangle. If `strict` is true, and constraining was not successful, `false`.
   */
  constrainTo(constrainTo, strict) {
    const me = this, originalHeight = me.height, originalY = me.y, minWidth = me.minWidth || me.width, minHeight = me.minHeight || me.height;
    if (me.height >= constrainTo.height) {
      if (strict && minHeight > constrainTo.height) {
        return false;
      }
      me._y = constrainTo.y;
      me.height = constrainTo.height;
    }
    if (me.width >= constrainTo.width) {
      if (strict && minWidth > constrainTo.width) {
        me.y = originalY;
        me.height = originalHeight;
        return false;
      }
      me._x = constrainTo.x;
      me.width = constrainTo.width;
    }
    me.translate.apply(me, me.constrainVector = [
      Math.min(constrainTo.right - me.right, 0),
      Math.min(constrainTo.bottom - me.bottom, 0)
    ]);
    me.translate(Math.max(constrainTo.x - me.x, 0), Math.max(constrainTo.y - me.y, 0));
    return me;
  }
  /**
   * Returns a cloned version of this Rectangle aligned to a target Rectangle, or element or {@link Core.widget.Widget}.
   * @param {Object} spec Alignment specification.
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} spec.target The Widget or element or Rectangle to align to.
   * @param {Number[]} [spec.anchorSize] The `[width, height]` of the anchor pointer when in `top` position. The
   * width is the baseline length, and the height is the height of the arrow. If passed, the anchor position
   * will be calculated to be at the centre of the overlap of the two aligned edges and returned in the `anchor`
   * property of the resulting Rectangle:
   *
   *     {
   *         edge: 'top',         // or 'right' or 'bottom' or 'left'
   *         x/y: offset          // dimension to translate and value to translate by.
   *     }
   *
   * @param {Object} [spec.anchorPosition] an `{ x: n, y: n }` anchor translation to be used *if the requested alignment
   * succeeds without violating constraints*. If a fallback alignment is used, the anchor will be centered in the
   * overlap of the aligned edges as usual.
   * @param {Boolean} [spec.overlap] True to allow this to overlap the target.
   * @param {String} spec.align The edge alignment specification string, specifying two points to bring together.
   *
   * Each point is described by an edge initial (`t` for top edge, `b` for bottom edge etc) followed
   * by a percentage along that edge.
   *
   * So the form would be `[trblc][n]-[trblc][n].` The `n` is the percentage offset along that edge
   * which defines the alignment point. This is not valid for alignment point `c` which means the center point.
   *
   * For example `t0-b0' would align this Rectangle's top left corner with the bottom left corner of the `target`.
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} [spec.constrainTo] The Widget or Element or Rectangle to constrain to.
   * If the requested alignment cannot be constrained (it will first shrink the resulting Rectangle according
   * to the `minWidth` and `minHeight` properties of this rectangle), then it will try aligning at other edges
   * (honouring the `axisLock` option), and pick the fallback alignment which results in the shortest translation.
   * @param {Boolean} [spec.axisLock] Specify as a truthy value to fall back to aligning against the opposite
   * edge first if the requested alignment cannot be constrained into the `constrainTo` option. If specified
   * as `'flexible'`, then fallback will continue searching for solutions on the remaining two sides.
   * @param {Boolean} [spec.matchSize] When aligning edge-to-edge, match the length of the aligned-to
   * edge of the target. This is only honored when `axisLock` is enabled and alignment succeeds on the requested axis.
   * If __not__ aligning edge-to-edge, `matchSize` matches both dimensions of the target.
   * @param {Number|Number[]} [spec.offset] The 'x' and 'y' offset values to create an extra margin round the target
   * to offset the aligned widget further from the target. May be configured as -ve to move the aligned widget
   * towards the target - for example producing the effect of the anchor pointer piercing the target.
   * @param {Number|Number[]} [spec.constrainPadding] The amount of pixels to pad from the `constrainTo` target,
   * either a single value, or an array of values in CSS edge order.
   * @param {Boolean} [spec.rtl] Pass as true if this is being used in an RTL environment, and aligning 0% to
   * 100% along a horizontal edge must proceed from right to left.
   * @returns {Core.helper.util.Rectangle} A new Rectangle aligned as requested if possible, but if the requested position violates
   * the `constrainTo` Rectangle, the shortest translation from the requested position which obeys constraints will be used.
   */
  alignTo(spec) {
    let result = this.clone(), {
      target,
      constrainTo,
      constrainPadding
    } = spec, calculatedAnchorPosition, zone, resultZone, constrainingToViewport;
    if (target && !target.isRectangle) {
      target = Rectangle.from(target.element ? target.element : target);
    }
    if (constrainTo) {
      if (!constrainTo.isRectangle) {
        constrainingToViewport = constrainTo === globalThis || constrainTo === document;
        const ignorePageScroll = "ignorePageScroll" in spec ? spec.ignorePageScroll : !constrainingToViewport;
        constrainTo = Rectangle.from(constrainTo.element ? constrainTo.element : constrainTo, null, ignorePageScroll);
      }
      if (constrainPadding) {
        constrainPadding = parseEdges(constrainPadding);
        constrainPadding[0] = Math.min(constrainPadding[0], target.top);
        constrainPadding[1] = Math.min(constrainPadding[1], constrainTo.right - target.right);
        constrainPadding[2] = Math.min(constrainPadding[0], constrainTo.bottom - target.bottom);
        constrainPadding[3] = Math.min(constrainPadding[0], target.left);
        constrainTo = constrainTo.deflate.apply(constrainTo.clone(), constrainPadding);
      }
    }
    const me = this, targetOffsets = createOffsets(spec.offset), {
      align,
      axisLock,
      anchorSize,
      anchorPosition,
      matchSize,
      position,
      rtl
    } = spec, alignSpec = parseAlign(align, rtl), targetConstrainRect = constrainTo && constrainTo.clone(), constraintZones = [], zoneOrder = [{
      zone: zone = alignSpec.startZone,
      align
    }], matchDimension = matchSize && matchDimensions[alignSpec.startZone & 1], originalSize = me[matchDimension];
    if (matchDimension && axisLock) {
      result[matchDimension] = target[matchDimension];
    } else if (!alignSpec.edgeAligned && matchSize) {
      result.width = target.width;
      result.height = target.height;
    }
    if (constrainTo) {
      result.constrainTo(constrainTo);
    }
    if (constrainTo && alignSpec.startZone != null) {
      if (axisLock) {
        zoneOrder.push({
          zone: zone = (zone + 2) % 4,
          align: flipAlign(alignSpec)
        });
        if (axisLock === "flexible") {
          zoneOrder.push({
            zone: zone = (alignSpec.startZone + 1) % 4,
            align: defaultAlignments[zone]
          });
          zoneOrder.push({
            zone: zone = (zone + 2) % 4,
            align: defaultAlignments[zone]
          });
        }
      } else {
        for (let i = 1; i < 4; i++) {
          zoneOrder.push({
            zone: zone = (zone + 1) % 4,
            align: defaultAlignments[zone]
          });
        }
      }
    }
    if (anchorPosition) {
      const pos = alignSpec.startZone & 1 ? "y" : "x";
      calculatedAnchorPosition = {
        [pos]: anchorPosition[pos],
        edge: edgeNames[(alignSpec.startZone + 2) % 4]
      };
    }
    if (targetConstrainRect && target) {
      targetConstrainRect.adjust(-target.width, -target.height, target.width, target.height);
      target.constrainTo(targetConstrainRect);
    }
    result.minWidth = me.minWidth;
    result.minHeight = me.minHeight;
    if (position) {
      result.moveTo(position.x, position.y);
      if (constrainTo) {
        result.constrainTo(constrainTo);
      }
    } else {
      const centerAligned = alignSpec.myEdge === "c" || alignSpec.targetEdge === "c", offsets = anchorSize && !centerAligned ? [anchorSize[1] + targetOffsets[0], anchorSize[1] + targetOffsets[1]] : targetOffsets, targetPoint = target.getAlignmentPoint(alignSpec.targetAlignmentPoint, offsets), myPoint = result.getAlignmentPoint(alignSpec.myAlignmentPoint);
      result.translate(targetPoint[0] - myPoint[0], targetPoint[1] - myPoint[1]);
      let overlap = result.intersect(target, true);
      if (overlap) {
        if (constrainTo) {
          result.constrainTo(constrainTo);
        }
        resultZone = alignSpec.startZone;
        result.translate(...offsets);
      } else if (constrainTo && !constrainTo.contains(result)) {
        const requestedResult = result.clone(), solutions = [];
        let zone2, largestZone;
        calculatedAnchorPosition = null;
        constraintZones[0] = zone2 = constrainTo.clone();
        zone2.bottom = target.y - offsets[1];
        constraintZones[1] = zone2 = constrainTo.clone();
        zone2.x = target.right + offsets[0];
        constraintZones[2] = zone2 = constrainTo.clone();
        zone2.y = target.bottom + offsets[1];
        constraintZones[3] = zone2 = constrainTo.clone();
        zone2.right = target.x - offsets[0];
        for (let i = 0; i < zoneOrder.length; i++) {
          if (matchDimension && i == 2) {
            result[matchDimension] = originalSize;
          }
          zone2 = constraintZones[resultZone = zoneOrder[i].zone];
          result = result.alignTo({
            target,
            offsets,
            align: zoneOrder[i].align
          });
          if (result.constrainTo(zone2, true)) {
            solutions.push({
              result,
              zone: resultZone
            });
            if (!largestZone || result.width < me.width || result.height < me.height) {
              result.align = zoneOrder[i].align;
              break;
            }
          }
          if (!largestZone || zone2.area > largestZone.area) {
            const r = result.clone();
            switch (resultZone) {
              case 0:
                r.moveTo(null, zone2.bottom - r.height);
                break;
              case 1:
                r.moveTo(zone2.left);
                break;
              case 2:
                r.moveTo(null, zone2.top);
                break;
              case 3:
                r.moveTo(zone2.right - r.width);
                break;
            }
            largestZone = {
              area: zone2.area,
              result: r,
              zone: resultZone
            };
          }
        }
        if (solutions.length) {
          if (solutions.length > 1 && !axisLock) {
            solutions.sort((s12, s22) => {
              const s1TranslationDistance = Math.sqrt((requestedResult.x - s12.result.x) ** 2 + (requestedResult.y - s12.result.y) ** 2), s2TranslationDistance = Math.sqrt((requestedResult.x - s22.result.x) ** 2 + (requestedResult.y - s22.result.y) ** 2);
              return s1TranslationDistance - s2TranslationDistance;
            });
          }
          result = solutions[0].result;
          resultZone = solutions[0].zone;
        } else {
          result = largestZone.result;
          resultZone = largestZone.zone;
          if (constrainingToViewport) {
            result.constrainTo(constrainTo);
          }
        }
      } else {
        resultZone = alignSpec.startZone;
      }
      result.zone = resultZone;
      result.overlap = overlap = result.intersect(target, true);
      if (anchorSize && !overlap) {
        if (!calculatedAnchorPosition) {
          const isLeftOrRight = resultZone & 1, start = isLeftOrRight ? "y" : "x", end = isLeftOrRight ? "bottom" : "right", startValue = Math.max(target[start], result[start]), endValue = Math.min(target[end], result[end]);
          let anchorStart = startValue + (endValue - startValue) / 2 - anchorSize[0] / 2;
          const anchorEnd = anchorStart + anchorSize[0];
          if (anchorEnd > result[end]) {
            anchorStart -= anchorEnd - result[end];
          }
          if (anchorStart < result[start]) {
            anchorStart += result[start] - anchorStart;
          }
          calculatedAnchorPosition = {
            [start]: anchorStart - result[start],
            edge: edgeNames[(resultZone + 2) % 4]
          };
        }
        result.anchor = calculatedAnchorPosition;
      }
    }
    return result;
  }
  /**
   * Returns the `[x, y]` position of the specified anchor point of this Rectangle in <edge><offset> format.
   * for example passing "t50" will return the centre point of the top edge, passing "r0" will return the start
   * position of the right edge (the top right corner).
   *
   * Note that the offset defaults to 50, so "t" means the centre of the top edge.
   * @param {String} alignmentPoint The alignment point to calculate. Must match the RegExp `[trbl]\d*`
   * @param {Number[]} margins The `[x, y]` margins to add from the left/right, top/bottom edge.
   * @internal
   */
  getAlignmentPoint(alignmentPoint, margins = zeroOffsets) {
    const me = this, parts = alignPointRe.exec(alignmentPoint), edge = parts[1].toLowerCase(), edgeOffset = Math.min(Math.max(parseInt(parts[2] || 50), 0), 100) / 100;
    switch (edge) {
      case "t":
        return [me.x + me.width * edgeOffset, me.y - margins[1]];
      case "r":
        return [me.right + margins[0], me.y + me.height * edgeOffset];
      case "b":
        return [me.x + me.width * edgeOffset, me.bottom + margins[1]];
      case "l":
        return [me.x - margins[0], me.y + me.height * edgeOffset];
      case "c": {
        return [me.x + me.width / 2, me.y + me.height / 2];
      }
    }
  }
  /**
   * Highlights this Rectangle using the highlighting effect of {@link Core.helper.DomHelper}
   * on a transient element which encapsulates the region's area.
   */
  highlight() {
    const me = this, highlightElement = DomHelper.createElement({
      parent: document.body,
      style: `position:absolute;z-index:9999999;pointer-events:none;
                            left:${me.x}px;top:${me.y}px;width:${me.width}px;height:${me.height}px`
    });
    return DomHelper.highlight(highlightElement).then(() => highlightElement.remove());
  }
  /**
   * Visualizes this Rectangle by adding a DOM element which encapsulates the region's area into the provided parent element.
   * @param {DomConfig} config Element config object
   * @returns {Element} The highlight element
   * @internal
   */
  visualize(config, asDomConfig) {
    const me = this, domConfig = ObjectHelper.merge({
      style: {
        left: `${me.x}px`,
        top: `${me.y}px`,
        width: `${me.width}px`,
        height: `${me.height}px`,
        pointerEvents: "none",
        // If this visualization is provided a CSS class, let outside handle position + z-index
        ...config.class ? {} : { position: "absolute", "z-index": 9999999 }
      }
    }, config);
    return asDomConfig ? domConfig : DomHelper.createElement(domConfig);
  }
  toString(delimiter = ",") {
    return [`${this.top}px`, `${this.right}px`, `${this.bottom}px`, `${this.left}px`].join(delimiter);
  }
};
Rectangle._$name = "Rectangle";

// ../Core/lib/Core/helper/util/DomClassList.js
var valueSymbol = Symbol("value");
var lengthSymbol = Symbol("length");
var DomClassList = class {
  static change(cls2, add, remove, as = "string") {
    remove = DomClassList.normalize(remove, "object");
    const after = DomClassList.normalize(cls2, "array").filter((c) => !remove[c]);
    if (add) {
      add = DomClassList.normalize(add, "array");
      for (let i = 0; i < add.length; ++i) {
        if (!after.includes(add[i])) {
          after.push(add[i]);
        }
      }
    }
    return DomClassList.normalize(after, as);
  }
  static from(classes, returnEmpty) {
    if (classes) {
      if (classes.isDomClassList) {
        returnEmpty = returnEmpty != null ? returnEmpty : true;
      } else {
        returnEmpty = returnEmpty != null ? returnEmpty : Objects.isObject(classes) && !Objects.isEmpty(classes);
        classes = new DomClassList(classes);
      }
      if (!classes.value && !returnEmpty) {
        classes = null;
      }
    }
    return classes || (returnEmpty ? new DomClassList() : null);
  }
  /**
   * Converts a class name of any understood type to a desired form.
   * @param {String|String[]|Object|Set|Map|HTMLElement} cls
   * @param {String} as Pass `'object'` to return an object with the class names as its keys (all keys will have a
   * value of `true`), or pass `'array'` to return an array of class names, or pass `'string'` (the default) to
   * return a space-separated string of class names.
   * @returns {String|String[]|Object}
   * @internal
   */
  static normalize(cls2, as = "string") {
    cls2 = cls2 || "";
    const type = typeof cls2, asArray = as === "array", asObject = as === "object", asString = !asArray && !asObject;
    let isString = type === "string", c, i, ret;
    if (type === "object") {
      if (cls2.nodeType === Element.ELEMENT_NODE && typeof cls2.getAttribute === "function") {
        cls2 = cls2.getAttribute("class") || "";
        isString = true;
      } else if (cls2 == null ? void 0 : cls2.isDomClassList) {
        cls2 = cls2.values;
      } else if (cls2 instanceof DOMTokenList) {
        cls2 = Array.from(cls2);
      } else if (cls2 instanceof Map) {
        cls2 = Array.from(cls2.keys()).filter((k) => cls2.get(k));
      } else if (cls2 instanceof Set) {
        cls2 = Array.from(cls2);
      } else if (!Array.isArray(cls2)) {
        cls2 = Objects.getTruthyKeys(cls2);
      }
    }
    if (isString) {
      cls2 = [...new Set(StringHelper.split(cls2))];
    }
    for (i = cls2.length; i-- > 0; ) {
      c = cls2[i];
      if (!c.length) {
        cls2.splice(i, 1);
      } else if (c.includes(" ")) {
        cls2.splice(i, 1, ...StringHelper.split(c));
      }
    }
    if (asArray) {
      ret = cls2;
    } else if (asString) {
      ret = cls2.join(" ");
    } else {
      ret = /* @__PURE__ */ Object.create(null);
      for (i = 0; i < cls2.length; ++i) {
        ret[cls2[i]] = true;
      }
    }
    return ret;
  }
  /**
   * Initializes a new DomClassList.
   * @param {...String|Object} classes The CSS classes as strings or objects.
   * @function constructor
   */
  constructor(...classes) {
    this.process(1, classes);
  }
  /**
   * Clears all class names from this DomClassList instance.
   * @returns {Core.helper.util.DomClassList} this DomClassList.
   */
  clear() {
    for (const key in this) {
      this[key] = false;
    }
    return this;
  }
  /**
   * Sets this DomClassList instance to represent the classes passed as either strings or objects.
   * @returns {Core.helper.util.DomClassList} this DomClassList.
   */
  set(...classes) {
    return this.clear().process(1, classes);
  }
  // To gain some speed in DomHelper.sync(), faster than instanceof etc
  get isDomClassList() {
    return true;
  }
  /**
   * Returns a clone of this DomClassList with all the same keys set.
   * @returns {Core.helper.util.DomClassList} A clone of this DomClassList.
   */
  clone() {
    return new DomClassList(this);
  }
  /**
   * Returns a Boolean value, indicating whether this ClassList has the specified CSS class name.
   * @param {String} className CSS class name to check
   * @returns {Boolean} true if this ClassList contains the passed CSS class name, false otherwise
   */
  contains(className) {
    if (typeof className === "string" && className) {
      return Boolean(this[className]);
    }
    return false;
  }
  // An instance of this class may be assigned directly to an element's className
  // it will be coerced to a string value using this method.
  toString() {
    return this.length ? `${this.value} ` : "";
  }
  toJSON() {
    return this.toString();
  }
  /**
   * Analogous to string.trim, returns the string value of this `DomClassList` with no trailing space.
   * @returns {String} A concatenated string value of all the class names in this `DomClassList`
   * separated by spaces.
   */
  trim() {
    return this.value;
  }
  /**
   * Compares this DomClassList to another DomClassList (or class name string of space separated classes).
   * If the same class names (regardless of order) are present, the two are considered equal.
   *
   * So `new DomClassList('foo bar bletch').isEqual('bletch bar foo')` would return `true`
   * @param {Core.helper.util.DomClassList|String} other The `DomClassList` or string of classes to compare to.
   * @returns {Boolean} `true` if the two contain the same class names.
   */
  isEqual(other) {
    const otherClasses = DomClassList.normalize(other, "array"), len = otherClasses.length;
    if (this.length === len) {
      for (let i = 0; i < len; i++) {
        if (!this[otherClasses[i]]) {
          return false;
        }
      }
      return true;
    }
    return false;
  }
  /**
   * Get/set string value.
   * Class names separated with space.
   * @property {String}
   */
  get value() {
    let value = this[valueSymbol], keys;
    if (value == null) {
      keys = Objects.getTruthyKeys(this);
      this[lengthSymbol] = keys.length;
      this[valueSymbol] = value = keys.join(" ");
    }
    return value;
  }
  set value(value) {
    const me = this, keys = Object.keys(me), len = keys.length;
    for (let i = 0; i < len; i++) {
      delete me[keys[i]];
    }
    if (value) {
      me.process(1, [value]);
    } else {
      delete me[valueSymbol];
    }
  }
  /**
   * Returns string values as an array.
   * @readonly
   * @property {String[]}
   */
  get values() {
    return Objects.getTruthyKeys(this);
  }
  get length() {
    return this.value ? this[lengthSymbol] : 0;
  }
  process(value, classes) {
    for (let cls2, k, i = 0; i < classes.length; i++) {
      if (classes[i]) {
        cls2 = classes[i];
        if (cls2.isDomClassList || Objects.isObject(cls2)) {
          for (k in cls2) {
            this[k] = value ? cls2[k] : !cls2[k];
          }
        } else {
          cls2 = DomClassList.normalize(classes[i], "array");
          for (k = 0; k < cls2.length; ++k) {
            this[cls2[k]] = value;
          }
        }
      }
    }
    delete this[valueSymbol];
    return this;
  }
  /**
   * Adds/removes class names according to the passed object's properties.
   *
   * Properties with truthy values are added.
   * Properties with falsy values are removed.
   * @param {Object} classList Object containing properties to set/clear
   */
  assign(classList) {
    for (const cls2 in classList) {
      if (!this[cls2] !== !classList[cls2]) {
        this[cls2] = classList[cls2];
        delete this[valueSymbol];
      }
    }
    return this;
  }
  /**
   * Adds/removes this objects classes to the passed `classList` or element.
   *
   * Properties with truthy values are added.
   * Properties with falsy values are removed.
   * @param {HTMLElement|DOMTokenList} element The element or the element's `classList` to be updated.
   */
  assignTo(element) {
    const classList = element.nodeType === Element.ELEMENT_NODE ? element.classList : element, temp = DomClassList.from(classList);
    temp.add(this);
    classList.value = temp.value;
  }
  /**
   * Add CSS class(es)
   * ```
   * myClassList.add('bold', 'small');
   * ```javascript
   * @param {String|String[]|Object} classes CSS classes to add
   */
  add(...classes) {
    return this.process(1, classes);
  }
  /**
   * Remove CSS class(es)
   * ```javascript
   * myClassList.remove('bold', 'small');
   * ```
   * @param {String} classes CSS classes to remove
   */
  remove(...classes) {
    return this.process(0, classes);
  }
  /**
   * Toggles the passed CSS class name.
   *
   * If the `force` parameter is passed, `true` means add the class name, `false` means remove it.
   *
   * ```javascript
   * myClassList.toggle('bold', isImportant);
   * ```
   * @param {String} className CSS class to toggle
   * @param {Boolean} [force] `true` to add the class, `false` to remove it.
   * @returns {Boolean} `true` if the operation changed the value.
   */
  toggle(className, flag = Boolean(!this[className])) {
    flag = Boolean(flag);
    if (Boolean(this[className]) !== flag) {
      this[className] = flag;
      delete this[valueSymbol];
      return true;
    }
  }
  /**
   * Analogous to the `String#split` method, but with no delimiter
   * parameter. This method returns an array containing the individual
   * CSS class names set.
   * @returns {String[]} The individual class names in this `DomClassList`
   */
  split() {
    return Objects.getTruthyKeys(this);
  }
  forEach(fn2) {
    return Objects.getTruthyKeys(this).forEach(fn2);
  }
};
DomClassList.prototype[valueSymbol] = null;
DomClassList._$name = "DomClassList";

// ../Core/lib/Core/helper/util/Point.js
var Point = class extends Rectangle {
  /**
   * Creates a new Point encapsulating the event's page position.
   * @param {Event} event
   * @returns {Core.helper.util.Point}
   * @typings ignore
   */
  static from(event) {
    const touchPoints = event.changedTouches;
    return new Point(touchPoints ? touchPoints[0].screenX : event.screenX, touchPoints ? touchPoints[0].screenY : event.pageY);
  }
  /**
   * Constructs a Point
   * @param x The X coordinate
   * @param y The Y coordinate
   */
  constructor(x, y) {
    super(x, y, 0, 0);
  }
  /**
   * Coerces this Point to be within the passed Rectangle. Translates it into the bounds.
   * @param {Core.helper.util.Rectangle} into The Rectangle into which to coerce this Point.
   */
  constrain(into) {
    this.x = Math.min(Math.max(this.x, into.x), into.right - 1);
    this.y = Math.min(Math.max(this.y, into.y), into.bottom - 1);
    return this;
  }
  toArray() {
    return [this.x, this.y];
  }
};
Object.getPrototypeOf(Point).Point = Point;
Point._$name = "Point";

// ../Core/lib/Core/helper/EventHelper.js
var touchProperties = [
  "clientX",
  "clientY",
  "pageX",
  "pageY",
  "screenX",
  "screenY"
];
var isOption = {
  element: 1,
  thisObj: 1,
  once: 1,
  delegate: 1,
  delay: 1,
  capture: 1,
  passive: 1,
  throttled: 1,
  autoDetach: 1,
  expires: 1,
  block: 1
};
var configurable = true;
var returnTrueProp = {
  configurable,
  value: true
};
var normalizedKeyNames = {
  Spacebar: "Space",
  Del: "Delete",
  Esc: "Escape",
  Left: "ArrowLeft",
  Up: "ArrowUp",
  Right: "ArrowRight",
  Down: "ArrowDown"
};
var specialKeys = {
  Control: "ctrl",
  Alt: "alt",
  Shift: "shift"
};
var specialKeyRe = /^(ctrl|shift|alt|meta)$/;
var eventProps = [
  "altKey",
  "bubbles",
  "button",
  "buttons",
  "cancelBubble",
  "cancelable",
  "clientX",
  "clientY",
  "ctrlKey",
  "layerX",
  "layerY",
  "metaKey",
  "pageX",
  "pageY",
  "returnValue",
  "screenX",
  "screenY",
  "shiftKey"
];
var _EventHelper = class {
  static normalizeEvent(event) {
    return ObjectHelper.copyPropertiesIf(event, event.changedTouches[0] || event.touches[0], touchProperties);
  }
  /**
   * For use when synthesizing events from native DOM events. Copies valid properties from the passed
   * event into the destination object;
   * @param {Object} dest Destination object
   * @param {Event} event The event whose properties to copy
   * @returns {Object} An event construction object.
   * @internal
   */
  static copyEvent(dest, event) {
    return ObjectHelper.copyProperties(dest, event, eventProps);
  }
  /**
   * Returns the `[x, y]` coordinates of the event in the viewport coordinate system.
   * @param {Event} event The event
   * @returns {Number[]} The coordinate.
   */
  static getXY(event) {
    if (event.touches) {
      event = event.touches[0];
    }
    return [event.clientX, event.clientY];
  }
  /**
   * Returns the pixel distance between two mouse/touch/pointer events.
   * @param {Event} event1 The first event.
   * @param {Event} event2 The second event.
   * @returns {Number} The distance in pixels between the two events.
   */
  static getDistanceBetween(event1, event2) {
    const xy1 = EH.getXY(event1), xy2 = EH.getXY(event2);
    return Math.sqrt(Math.pow(xy1[0] - xy2[0], 2) + Math.pow(xy1[1] - xy2[1], 2));
  }
  /**
   * Returns a {@link Core.helper.util.Point} which encapsulates the `pageX/Y` position of the event.
   * May be used in {@link Core.helper.util.Rectangle} events.
   * @param {Event} event A browser mouse/touch/pointer event.
   * @returns {Core.helper.util.Point} The page point.
   */
  static getPagePoint(event) {
    return new Rectangle.Point(event.pageX, event.pageY);
  }
  /**
   * Returns a {@link Core.helper.util.Point} which encapsulates the `clientX/Y` position of the event.
   * May be used in {@link Core.helper.util.Rectangle} events.
   * @param {Event} event A browser mouse/touch/pointer event.
   * @returns {Core.helper.util.Point} The page point.
   */
  static getClientPoint(event) {
    return new Rectangle.Point(event.clientX, event.clientY);
  }
  /**
   * Add a listener or listeners to an element
   * The `options` parameter allows supplying options for the listener(s), for available options see {@link #typedef-ElementListenerConfig}.
   *
   * @param {HTMLElement} element The element to add a listener/listeners to.
   * @param {String|Object} eventName Either a string, being the name of the event to listen for,
   * or an options object containing event names and options as keys. See the options parameter
   * for details, or the {@link #function-on-static} method for details.
   * @param {Function} [handler] If the second parameter is a string event name, this is the handler function.
   * @param {ElementListenerConfig} [options] If the second parameter is a string event name, this is the options.
   * @returns {Function} A detacher function which removes all the listeners when called.
   */
  static addListener(element, eventName, handler, options) {
    if (element.nodeType) {
      if (typeof eventName === "string") {
        options = Object.assign({
          element,
          [eventName]: handler
        }, options);
      } else {
        options = Object.assign({
          element
        }, eventName);
      }
    } else {
      options = element;
    }
    return EH.on(options);
  }
  /**
   * Adds a listener or listeners to an element.
   * all property names other than the options listed below are taken to be event names,
   * and the values as handler specs.
   *
   * A handler spec is usually a function reference or the name of a function in the `thisObj`
   * option.
   *
   * But a handler spec may also be an options object containing a `handler` property which is
   * the function or function name, and local options, including `element` and `thisObj`
   * which override the top level options.
   *
   * The `options` parameter allows supplying options for the listener(s), for available options see {@link #typedef-ElementListenerConfig}.
   *
   *  Usage example
   *
   * ```javascript
   * construct(config) {
   *     super.construct(config);
   *
   *     // Add auto detaching event handlers to this Widget's reference elements
   *     EventHelper.on({
   *         element : this.iconElement,
   *         click   : '_handleIconClick',
   *         thisObj : this,
   *         contextmenu : {
   *             element : document,
   *             handler : '_handleDocumentContextMenu'
   *         }
   *     });
   * }
   *```
   *
   * The `click` handler on the `iconElement` calls `this._handleIconClick`.
   *
   * The `contextmenu` handler is added to the `document` element, but the `thisObj`
   * is defaulted in from the top `options` and calls `this._handleDocumentContextMenu`.
   *
   * Note that on touch devices, `dblclick` and `contextmenu` events are synthesized.
   * Synthesized events contain a `browserEvent` property containing the final triggering
   * event of the gesture. For example a synthesized `dblclick` event would contain a
   * `browserEvent` property which is the last `touchend` event. A synthetic `contextmenu`
   * event will contain a `browserEvent` property which the longstanding `touchstart` event.
   *
   * @param {ElementListenerConfig} options The full listener specification.
   * @returns {Function} A detacher function which removes all the listeners when called.
   */
  static on(options) {
    const element = options.element, thisObj = options.thisObj, handlerDetails = [];
    for (const eventName in options) {
      if (!isOption[eventName]) {
        let handlerSpec = options[eventName];
        if (typeof handlerSpec !== "object") {
          handlerSpec = {
            handler: handlerSpec
          };
        }
        const targetElement = handlerSpec.element || element;
        handlerDetails.push(EH.addElementListener(targetElement, eventName, handlerSpec, options));
      }
    }
    const detacher2 = () => {
      for (let handlerSpec, i = 0; i < handlerDetails.length; i++) {
        handlerSpec = handlerDetails[i];
        EH.removeEventListener(handlerSpec[0], handlerSpec[1], handlerSpec[2]);
      }
      handlerDetails.length = 0;
    };
    if (thisObj && options.autoDetach !== false) {
      thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, detacher2, thisObj);
    }
    return detacher2;
  }
  /**
   * Used internally to add a single event handler to an element.
   * @param {HTMLElement} element The element to add the handler to.
   * @param {String} eventName The name of the event to add a handler for.
   * @param {Function|String|Object} handlerSpec Either a function to call, or
   * the name of a function to call in the `thisObj`, or an object containing
   * the handler local options.
   * @param {Function|String} [handlerSpec.handler] Either a function to call, or
   * the name of a function to call in the `thisObj`.
   * @param {HTMLElement} [handlerSpec.element] Optionally a local element for the listener.
   * @param {Object} [handlerSpec.thisObj] A local `this` specification for the handler.
   * @param {Object} defaults The `options` parameter from the {@link #function-addListener-static} call.
   * @private
   */
  static addElementListener(element, eventName, handlerSpec, defaults) {
    const handler = EH.createHandler(element, eventName, handlerSpec, defaults), { spec } = handler, expires = handlerSpec.expires || defaults.expires, options = spec.capture != null || spec.passive != null ? {
      capture: spec.capture,
      passive: spec.passive
    } : void 0;
    element.addEventListener(eventName, handler, options);
    if (expires) {
      const thisObj = handlerSpec.thisObj || defaults.thisObj, delayable = (thisObj == null ? void 0 : thisObj.isDelayable) ? thisObj : globalThis, { alt: alt2 } = expires, delay3 = alt2 ? expires.delay : expires, { spec: spec2 } = handler;
      spec2.expires = expires;
      spec2.timerId = delayable[typeof delay3 === "number" ? "setTimeout" : "requestAnimationFrame"](() => {
        spec2.timerId = null;
        EH.removeEventListener(element, eventName, handler);
        if (alt2 && !handler.called) {
          (typeof alt2 === "string" ? thisObj[alt2] : alt2).call(thisObj);
        }
      }, delay3, `listener-timer-${performance.now()}`);
    }
    return [element, eventName, handler, options];
  }
  // composedPath throws in salesforce
  // https://github.com/bryntum/support/issues/4432
  static getComposedPathTarget(event) {
    return event.composedPath()[0] || event.path[0];
  }
  static fixEvent(event) {
    var _a2, _b, _c;
    if (event.fixed) {
      return event;
    }
    const { type, target } = event;
    if (((target == null ? void 0 : target.shadowRoot) || ((_b = (_a2 = target == null ? void 0 : target.getRootNode) == null ? void 0 : _a2.call(target)) == null ? void 0 : _b.host)) && event.composedPath) {
      const targetElement = this.getComposedPathTarget(event), originalTarget = target;
      Object.defineProperty(event, "target", {
        value: targetElement,
        configurable
      });
      Object.defineProperty(event, "originalTarget", {
        value: originalTarget,
        configurable
      });
    }
    Object.defineProperty(event, "fixed", returnTrueProp);
    if (type.startsWith("key")) {
      const normalizedKeyName = normalizedKeyNames[event.key];
      if (normalizedKeyName) {
        Object.defineProperty(event, "key", {
          value: normalizedKeyName,
          configurable
        });
      }
      if (event.key === " " && !event.code) {
        Object.defineProperty(event, "code", {
          value: "Space",
          configurable
        });
      }
    }
    if (event.metaKey && !event.ctrlKey) {
      Object.defineProperty(event, "ctrlKey", returnTrueProp);
    }
    if (target && "offsetX" in event) {
      if (!Object.getOwnPropertyDescriptor(event, "borderOffsetX")) {
        Object.defineProperty(event, "borderOffsetX", {
          get: () => {
            return event.offsetX + (BrowserHelper.isSafari ? 0 : parseInt(target.ownerDocument.defaultView.getComputedStyle(target).getPropertyValue("border-left-width")));
          }
        });
      }
      if (!Object.getOwnPropertyDescriptor(event, "borderOffsetY")) {
        Object.defineProperty(event, "borderOffsetY", {
          get: () => {
            return event.offsetY + (BrowserHelper.isSafari ? 0 : parseInt(target.ownerDocument.defaultView.getComputedStyle(target).getPropertyValue("border-top-width")));
          }
        });
      }
    }
    if ((target == null ? void 0 : target.nodeType) === Element.DOCUMENT_NODE && "clientX" in event) {
      const targetElement = DomHelper.elementFromPoint(event.clientX, event.clientY);
      Object.defineProperty(event, "target", {
        value: targetElement,
        configurable
      });
    }
    if ((target == null ? void 0 : target.nodeType) === Element.TEXT_NODE) {
      const targetElement = event.target.parentElement;
      Object.defineProperty(event, "target", {
        value: targetElement,
        configurable
      });
    }
    if (((_c = event.relatedTarget) == null ? void 0 : _c.nodeType) === Element.TEXT_NODE) {
      const relatedTargetElement = event.target.parentElement;
      Object.defineProperty(event, "relatedTarget", {
        value: relatedTargetElement,
        configurable
      });
    }
    if (type.startsWith("touch") && event.touches.length) {
      this.normalizeEvent(event);
    }
    return event;
  }
  static createHandler(element, eventName, handlerSpec, defaults) {
    const delay3 = handlerSpec.delay || defaults.delay, throttled = handlerSpec.throttled || defaults.throttled, block = handlerSpec.block || defaults.block, once = "once" in handlerSpec ? handlerSpec.once : defaults.once, capture = "capture" in handlerSpec ? handlerSpec.capture : defaults.capture, passive = "passive" in handlerSpec ? handlerSpec.passive : defaults.passive, delegate = "delegate" in handlerSpec ? handlerSpec.delegate : defaults.delegate, wrappedFn = handlerSpec.handler, expires = handlerSpec.expires, thisObj = handlerSpec.thisObj || defaults.thisObj, { rtlSource = thisObj } = thisObj || {};
    let handler = (event, ...args) => {
      if (EH.playingDemo && event.isTrusted) {
        return;
      }
      if (thisObj == null ? void 0 : thisObj.isDestroyed) {
        return;
      }
      event = EH.fixEvent(event, rtlSource == null ? void 0 : rtlSource.rtl);
      handler.called = true;
      (typeof wrappedFn === "string" ? thisObj[wrappedFn] : wrappedFn).call(thisObj, event, ...args);
      delete event.target;
      delete event.relatedTarget;
      delete event.originalarget;
      delete event.key;
      delete event.code;
      delete event.ctrlKey;
      delete event.fixed;
    };
    if (block) {
      const wrappedFn2 = handler;
      let lastCallTime, lastTarget;
      handler = (e, ...args) => {
        const now2 = performance.now();
        if (!lastCallTime || e.target !== lastTarget || now2 - lastCallTime > block) {
          lastTarget = e.target;
          lastCallTime = now2;
          wrappedFn2(e, ...args);
        }
      };
    }
    if (delay3 != null) {
      const wrappedFn2 = handler, delayable = (thisObj == null ? void 0 : thisObj.setTimeout) ? thisObj : globalThis;
      handler = (...args) => {
        delayable.setTimeout(() => {
          wrappedFn2(...args);
        }, delay3);
      };
    }
    if (throttled != null) {
      let alt2, buffer = throttled;
      if (throttled.buffer) {
        alt2 = (e) => {
          return throttled.alt.call(EH, EH.fixEvent(e, rtlSource == null ? void 0 : rtlSource.rtl));
        };
        buffer = throttled.buffer;
      }
      if (thisObj == null ? void 0 : thisObj.isDelayable) {
        handler = thisObj.throttle(handler, {
          delay: buffer,
          throttled: alt2
        });
      } else {
        handler = FunctionHelper.createThrottled(handler, buffer, thisObj, null, alt2);
      }
    }
    if (once) {
      const wrappedFn2 = handler;
      handler = (...args) => {
        EH.removeEventListener(element, eventName, handler);
        wrappedFn2(...args);
      };
    }
    if (delegate) {
      const wrappedFn2 = handler;
      handler = (event, ...args) => {
        var _a2;
        event = EH.fixEvent(event, rtlSource == null ? void 0 : rtlSource.rtl);
        const delegatedTarget = ((_a2 = event.target.closest) == null ? void 0 : _a2.call) && event.target.closest(delegate);
        if (!delegatedTarget) {
          return;
        }
        Object.defineProperty(event, "currentTarget", {
          get: () => delegatedTarget,
          configurable: true
        });
        wrappedFn2(event, ...args);
      };
    }
    if (handlerSpec.thisObj && handlerSpec.autoDetach !== false) {
      thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, () => EH.removeEventListener(element, eventName, handler), thisObj);
    }
    handler.spec = {
      delay: delay3,
      throttled,
      block,
      once,
      thisObj,
      capture,
      expires,
      passive,
      delegate
    };
    return handler;
  }
  static removeEventListener(element, eventName, handler) {
    const { expires, timerId, thisObj, capture } = handler.spec;
    if ((expires == null ? void 0 : expires.alt) && timerId) {
      const delayable = (thisObj == null ? void 0 : thisObj.isDelayable) ? thisObj : globalThis;
      delayable[typeof expires.delay === "number" ? "clearTimeout" : "cancelAnimationFrame"](timerId);
    }
    element.removeEventListener(eventName, handler, capture);
  }
  /**
   * Calls a callback when the described animation completes.
   *
   * @param {Object} detail
   * @param {HTMLElement} detail.element The element which is being animated.
   * @param {String|RegExp} [detail.animationName] The name of the animation to wait for.
   * @param {String} [detail.property] If no `animationName` specified, the CSS property
   * which is being animated.
   * @param {Function} detail.handler The function to call on animation end.
   * @param {Number} [detail.duration] Optional fallback time to wait until calling the callback.
   * @param {Object} [detail.thisObj] The `this` reference to call the callback with.
   * @param {Array} [detail.args] Optional arguments to call the callback with.
   * @param {Core.mixin.Delayable} [detail.timerSource] A Delayable to provide the fallback timeout.
   * @param {Boolean} [detail.runOnDestroy] If `timerSource` is a {@link Core.mixin.Delayable},
   * `true` to invoke the callback if it is destroyed during the animation.
   * @returns {Function} a function which detaches the animation end listener.
   */
  static onTransitionEnd({
    element,
    animationName,
    property,
    handler,
    mode = animationName ? "animation" : "transition",
    duration = DomHelper[`get${mode === "transition" ? "Property" : ""}${StringHelper.capitalize(mode)}Duration`](element, property),
    thisObj = globalThis,
    args = [],
    timerSource,
    runOnDestroy
  }) {
    let timerId;
    timerSource = timerSource || (thisObj.isDelayable ? thisObj : globalThis);
    const callbackArgs = [element, property, ...args], doCallback = () => {
      detacher2();
      if (!thisObj.isDestroyed) {
        if (thisObj.callback) {
          thisObj.callback(handler, thisObj, callbackArgs);
        } else {
          handler.apply(thisObj, callbackArgs);
        }
      }
    }, detacher2 = EH.on({
      element,
      [`${mode}end`]({ animationName: endedAnimation, propertyName, target }) {
        var _a2;
        if (target === element) {
          if (propertyName === property || (endedAnimation == null ? void 0 : endedAnimation.match(animationName))) {
            if (timerId) {
              (_a2 = timerSource.clearTimeout) == null ? void 0 : _a2.call(timerSource, timerId);
              timerId = null;
            }
            doCallback();
          }
        }
      }
    });
    if (duration != null) {
      timerId = timerSource.setTimeout(doCallback, duration + 50, "onTransitionEnd", runOnDestroy);
    }
    return detacher2;
  }
  /**
   * Waits for the described animation completes.
   *
   * @param {Object} config
   * @param {HTMLElement} config.element The element which is being animated.
   * @param {String|RegExp} [config.animationName] The name of the animation to wait for.
   * @param {String} [config.property] If no `animationName` specified, the CSS property
   * which is being animated.
   * @param {Number} [config.duration] Optional fallback time to wait until calling the callback.
   * @param {Core.mixin.Delayable} [config.timerSource] A Delayable to provide the fallback timeout.
   * @param {Boolean} [config.runOnDestroy] If `timerSource` is a {@link Core.mixin.Delayable},
   * `true` to invoke the callback if it is destroyed during the animation.
   * @async
   */
  static async waitForTransitionEnd(config) {
    return new Promise((resolve) => {
      config.handler = resolve;
      _EventHelper.onTransitionEnd(config);
    });
  }
  /**
   * Private function to wrap the passed function. The returned wrapper function to be used as
   * a `touchend` handler which will call the passed function passing a fabricated `dblclick`
   * event if there is a `click` within 300ms.
   * @param {Element} element element
   * @param {String|Function} handler The handler to call.
   * @param {Object} thisObj The owner of the function.
   * @private
   */
  static createDblClickWrapper(element, handler, thisObj) {
    let startId, secondListenerDetacher, tapholdTimer;
    return () => {
      if (!secondListenerDetacher) {
        secondListenerDetacher = EH.on({
          element,
          // We only get here if a touchstart arrives within 300ms of a click
          touchstart: (secondStart) => {
            startId = secondStart.changedTouches[0].identifier;
            secondStart.preventDefault();
          },
          touchend: (secondClick) => {
            if (secondClick.changedTouches[0].identifier === startId) {
              secondClick.preventDefault();
              clearTimeout(tapholdTimer);
              startId = secondListenerDetacher = null;
              const targetRect = Rectangle.from(secondClick.changedTouches[0].target, null, true), offsetX = secondClick.changedTouches[0].pageX - targetRect.x, offsetY = secondClick.changedTouches[0].pageY - targetRect.y, dblclickEventConfig = Object.assign({
                browserEvent: secondClick
              }, secondClick), dblclickEvent = new MouseEvent("dblclick", dblclickEventConfig);
              Object.defineProperties(dblclickEvent, {
                target: { value: secondClick.target },
                offsetX: { value: offsetX },
                offsetY: { value: offsetY }
              });
              if (typeof handler === "string") {
                handler = thisObj[handler];
              }
              handler.call(thisObj, dblclickEvent);
            }
          },
          once: true
        });
        tapholdTimer = setTimeout(() => {
          secondListenerDetacher();
          startId = secondListenerDetacher = null;
        }, EH.dblClickTime);
      }
    };
  }
  /**
   * Handles various inputs to figure out the name of the special key of the event.
   *
   * ```javascript
   * EventHelper.toSpecialKey('ctrl') // 'ctrlKey'
   * EventHelper.toSpecialKey(true)   // 'ctrlKey', default for PC (Cmd for Mac)
   * EventHelper.toSpecialKey(false)  // false
   * EventHelper.toSpecialKey('foo')  // false
   * ```
   *
   * @param {*} value User input value to process.
   * @param {String} defaultValue Default value to fall back to if `true` value is passed.
   * @returns {Boolean|String} Returns `false` if provided value cannot be converted to special key and special key
   * name otherwise.
   * @internal
   */
  static toSpecialKey(value, defaultValue2 = BrowserHelper.isMac ? "metaKey" : "ctrlKey") {
    let result = false;
    if (value === true) {
      result = defaultValue2;
    } else if (typeof value === "string") {
      value = value.toLowerCase();
      if (value.match(specialKeyRe)) {
        result = `${value}Key`;
      }
    }
    return result;
  }
  /**
   * If keyup event is triggered when special key is pressed, we don't get special key value from properties like
   * `ctrlKey`. Instead we need to read `event.key`. That property uses full name and we use abbreviations, so we
   * need to convert the key.
   * @param {String} code
   * @returns {String}
   * @internal
   */
  static specialKeyFromEventKey(code) {
    return specialKeys[code] || "no-special-key";
  }
};
var EventHelper = _EventHelper;
/**
 * DOM event to trigger name mapping.
 * @internal
 */
__publicField(EventHelper, "eventNameMap", {
  mousedown: "MouseDown",
  mouseup: "MouseUp",
  click: "Click",
  dblclick: "DblClick",
  contextmenu: "ContextMenu",
  mouseenter: "MouseEnter",
  mouseleave: "MouseLeave",
  mousemove: "MouseMove",
  mouseover: "MouseOver",
  mouseout: "MouseOut",
  keyup: "KeyUp",
  keydown: "KeyDown",
  keypress: "KeyPress"
});
var EH = EventHelper;
EH.longPressTime = 700;
EH.dblClickTime = 300;
if (BrowserHelper.isTouchDevice) {
  EH.on({
    element: document,
    touchmove: (event) => {
      if (event.target.closest(".b-dragging")) {
        event.preventDefault();
      }
    },
    passive: false,
    capture: true
  });
}
EventHelper._$name = "EventHelper";

// ../Core/lib/Core/helper/DomHelper.js
var DEFAULT_FONT_SIZE = 14;
var t0t0 = { align: "t0-t0" };
var ELEMENT_NODE = Node.ELEMENT_NODE;
var TEXT_NODE = Node.TEXT_NODE;
var { isObject } = ObjectHelper;
var numberRe = /[+-]?\d*\.?\d+[eE]?-?\d*/g;
var numberReSrc = numberRe.source;
var translateMatrix2dRe = new RegExp(`matrix\\((?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(${numberReSrc}),\\s?(${numberReSrc})`);
var translateMatrix3dRe = new RegExp(`matrix3d\\((?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(-?\\d*),\\s?(-?\\d*)`);
var translateMatrixRe = new RegExp(`(?:${translateMatrix2dRe.source})|(?:${translateMatrix3dRe.source})`);
var pxTtranslateXRe = new RegExp(`translate(3d|X)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`);
var pxTtranslateYRe = new RegExp(`translate(3d|Y)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`);
var whiteSpaceRe2 = /\s+/;
var semicolonRe = /\s*;\s*/;
var colonRe = /\s*:\s*/;
var digitsRe = /^-?((\d+(\.\d*)?)|(\.?\d+))$/;
var elementPropKey = "$bryntum";
var elementCreateExpandos = {
  elementData: "",
  for: "htmlFor",
  retainElement: ""
};
var elementCreateProperties = {
  // these two are handled by being in elementCreateExpands:
  // elementData  : 1,
  // for          : 1,
  tag: 1,
  html: 1,
  text: 1,
  children: 1,
  tooltip: 1,
  style: 1,
  dataset: 1,
  parent: 1,
  nextSibling: 1,
  ns: 1,
  reference: 1,
  class: 1,
  className: 1,
  unmatched: 1,
  // Used by syncId approach
  onlyChildren: 1,
  // Used by sync to not touch the target element itself,
  listeners: 1,
  // eslint-disable-line bryntum/no-listeners-in-lib
  compareHtml: 1,
  // Sync
  syncOptions: 1,
  // Sync
  keepChildren: 1
  // Sync
};
var styleIgnoreProperties = {
  length: 1,
  parentRule: 1,
  style: 1
};
var nativeEditableTags = {
  INPUT: 1,
  TEXTAREA: 1
};
var nativeFocusableTags = {
  BUTTON: 1,
  IFRAME: 1,
  EMBED: 1,
  INPUT: 1,
  OBJECT: 1,
  SELECT: 1,
  TEXTAREA: 1,
  BODY: 1
};
var win = globalThis;
var doc = document;
var emptyObject4 = Object.freeze({});
var arraySlice = Array.prototype.slice;
var immediatePromise = Promise.resolve();
var fontProps = [
  "font-size",
  "font-size-adjust",
  "font-style",
  "font-weight",
  "font-family",
  "font-kerning",
  "font-stretch",
  "line-height",
  "text-transform",
  "text-decoration",
  "letter-spacing",
  "word-break"
];
var isHiddenWidget = (e) => e._hidden;
var parentNode = (el) => el.parentNode || el.host;
var mergeChildren = (dest, src, options) => {
  if (options.key === "children") {
    return ObjectHelper.mergeItems(dest, src, options);
  }
  return ObjectHelper.blend(dest, src, options);
};
var isVisible = (e) => {
  const style = e.ownerDocument.defaultView.getComputedStyle(e);
  return style.getPropertyValue("display") !== "none" && style.getPropertyValue("visibility") !== "hidden";
};
var hasLayout = (el) => el && (el === doc.body || Boolean(el.offsetParent) || (el.ownerDocument.defaultView && ("offsetParent" in el && DomHelper.getStyleValue(el, "position") !== "fixed") ? el.offsetParent : hasLayout(el.parentNode)));
var elementOrConfigToElement = (elementOrConfig) => {
  if (elementOrConfig instanceof Node) {
    return elementOrConfig;
  }
  if (typeof elementOrConfig === "string") {
    return DH2.createElementFromTemplate(elementOrConfig);
  }
  return DH2.createElement(elementOrConfig);
};
var canonicalStyles = /* @__PURE__ */ Object.create(null);
var canonicalizeStyle = (name, hasUnit) => {
  const entry = canonicalStyles[name] || [StringHelper.hyphenate(name), hasUnit];
  if (!canonicalStyles[name]) {
    canonicalStyles[entry[0]] = canonicalStyles[name] = entry;
  }
  return entry;
};
var getOffsetParent = (node) => node.ownerSVGElement ? node.ownerSVGElement.parentNode : node.offsetParent;
var slideInAnimationName = /b-slide-in-from-\w+/;
[
  "top",
  "right",
  "bottom",
  "left",
  "width",
  "height",
  "maxWidth",
  "maxHeight",
  "minWidth",
  "minHeight",
  "borderSpacing",
  "borderWidth",
  "borderTopWidth",
  "borderRightWidth",
  "borderBottomWidth",
  "borderLeftWidth",
  "marginTop",
  "marginRight",
  "marginBottom",
  "marginLeft",
  "paddingTop",
  "paddingRight",
  "paddingBottom",
  "paddingLeft",
  "fontSize",
  "letterSpacing",
  "lineHeight",
  "outlineWidth",
  "textIndent",
  "wordSpacing"
].forEach((name) => canonicalizeStyle(name, true));
var scrollBarWidth = null;
var idCounter = 0;
var themeInfo = null;
var templateElement;
var htmlParser;
var scrollBarMeasureElement;
var DomHelper = class {
  /**
   * Animates the specified element to slide it into view within the visible viewport
   * of its parentElement from the direction of movement.
   *
   * So in a left-to-right Widget, `direction` 1 means it slides in from the right
   * and `direction` -1 means it slides in from the left. RTL reverses the movement.
   *
   * See the forward/backward navigations in {@link Core.widget.DatePicker} for an example
   * of this in action.
   *
   * If "next" should arrive from below and "previous" should arrive from above, add the
   * class `b-slide-vertical` to the element.
   * @param {HTMLElement} element The element to slide in.
   * @param {Number} direction
   * * `1` to slide in from the "next" direction.
   * * `-1` to slide in from the "previous" direction.
   *
   * If the element is inside an RTL widget the directions are reversed.
   * @async
   */
  static async slideIn(element, direction = 1) {
    const cls2 = `b-slide-in-${direction > 0 ? "next" : "previous"}`, { classList } = element, { style } = element.parentNode, {
      overflow,
      overflowX,
      overflowY
    } = style;
    style.overflow = "hidden";
    classList.add(cls2);
    await EventHelper.waitForTransitionEnd({
      element,
      animationName: slideInAnimationName
    });
    style.overflow = overflow;
    style.overflowX = overflowX;
    style.overflowY = overflowY;
    classList.remove(cls2);
  }
  /**
   * Returns `true` if the passed element is focusable either programmatically or through pointer gestures.
   * @param {HTMLElement} element The element to test.
   * @returns {Boolean} Returns `true` if the passed element is focusable
   */
  static isFocusable(element, skipAccessibilityCheck = false) {
    if (!skipAccessibilityCheck) {
      if (!DH2.isVisible(element) || DH2.Widget.fromElement(element, isHiddenWidget)) {
        return false;
      }
    }
    const nodeName = element.nodeName;
    return nativeFocusableTags[nodeName] || (nodeName === "A" || nodeName === "LINK") && !!element.href || element.getAttribute("tabIndex") != null || element.contentEditable === "true";
  }
  /**
   * Returns `true` if the passed element accepts keystrokes to edit its contents.
   * @returns {Boolean} Returns `true` if the passed element is editable.
   */
  static isEditable(element) {
    return element.isContentEditable || nativeEditableTags[element.nodeName];
  }
  /**
   * Returns the rectangle of the element which is currently visible in the browser viewport, i.e. user can find it on
   * screen, or `false` if it is scrolled out of view.
   * @param {HTMLElement} target The element to test.
   * @param {Boolean} [whole=false] Whether to check that whole element is visible, not just part of it.
   * If this is passed as true, the result will be a boolean, `true` or `false`.
   * @privateparam {Core.widget.Widget} [caller] the Widget aligning to the target.
   * @returns {Core.helper.util.Rectangle|Boolean} Returns the rectangle of the element which is currently visible in
   * the browser viewport, or `false` if it is out of view.
   */
  static isInView(target, whole = false, caller) {
    if (!hasLayout(target)) {
      return false;
    }
    const positioned = (caller == null ? void 0 : caller.positioned) && DomHelper.getStyleValue(caller.element, "position") !== "fixed", docRect = Rectangle.from(globalThis), method = whole ? "contains" : "intersect", cOp = positioned && caller.element.offsetParent, cOpR = positioned && Rectangle.from(cOp);
    docRect.height = doc.scrollingElement.scrollHeight;
    if (target === doc.body) {
      return docRect;
    }
    const result = this.getViewportIntersection(target, docRect, method);
    if (result && positioned) {
      result.translate(doc.scrollingElement.scrollLeft, doc.scrollingElement.scrollTop);
    }
    return positioned && result ? result.translate(-cOpR.x + cOp.scrollLeft, -cOpR.y + cOp.scrollTop) : result;
  }
  /**
   * This method goes up the DOM tree checking that all ancestors are visible in the viewport
   * @param {HTMLElement} target Starting html element
   * @param {Core.helper.util.Rectangle} docRect Window rectangle
   * @param {String} method 'contains' or 'intersect'
   * @returns {Core.helper.util.Rectangle}
   */
  static getViewportIntersection(target, docRect, method) {
    const { parentNode: parentNode2 } = target, { parentElement } = parentNode2.nodeType === Node.DOCUMENT_FRAGMENT_NODE ? target.getRootNode().host : target, peStyle = parentElement.ownerDocument.defaultView.getComputedStyle(parentElement), parentScroll = peStyle.overflowX !== "visible" || peStyle.overflowY !== "visible", offsetParent = getOffsetParent(target);
    let result = Rectangle.from(target, null, true);
    for (let viewport = parentScroll ? target.parentNode : offsetParent; result && viewport !== doc.documentElement; viewport = viewport.parentNode) {
      if (viewport.nodeType === Node.DOCUMENT_FRAGMENT_NODE && viewport.host) {
        viewport = viewport.host.parentNode;
      }
      const isTop = viewport === doc.body, style = viewport.ownerDocument.defaultView.getComputedStyle(viewport), viewportRect = isTop ? docRect : Rectangle.inner(viewport, null, true);
      if (isTop || style.overflow !== "visible") {
        result = viewportRect[method](result, false, true);
      }
    }
    return result;
  }
  /**
   * Returns `true` if the passed element is deeply visible. Meaning it is not hidden using `display`
   * or `visibility` and no ancestor node is hidden.
   * @param {HTMLElement} element The element to test.
   * @returns {Boolean} `true` if deeply visible.
   */
  static isVisible(element) {
    const document2 = element.ownerDocument;
    for (; element; element = parentNode(element)) {
      if (element === document2) {
        return true;
      }
      if (element.nodeType === element.ELEMENT_NODE && !isVisible(element)) {
        return false;
      }
    }
    return false;
  }
  /**
   * Returns true if DOM Event instance is passed. It is handy to override to support Locker Service.
   * @param event
   * @internal
   * @returns {Boolean}
   */
  static isDOMEvent(event) {
    return event instanceof Event;
  }
  /**
   * Merges specified source DOM config objects into a `dest` object.
   * @param {DomConfig} dest The destination DOM config object.
   * @param {...DomConfig} sources The DOM config objects to merge into `dest`.
   * @returns {DomConfig} The `dest` object.
   * @internal
   */
  static merge(dest, ...sources) {
    return ObjectHelper.blend(dest, sources, { merge: mergeChildren });
  }
  /**
   * Updates in-place a DOM config object whose `children` property may be an object instead of the typical array.
   * The keys of such objects become the `reference` property upon conversion.
   *
   * @param {DomConfig} domConfig
   * @param {Function} [namedChildren] A function to call for each named child element.
   * @privateparam {Boolean} [ignoreRefs] Not meant to be manually set, used when recursing.
   * @returns {DomConfig} Returns the altered DOM config
   * @internal
   */
  static normalizeChildren(domConfig, namedChildren, ignoreRefs) {
    var _a2;
    let children = domConfig == null ? void 0 : domConfig.children, child, i, name, kids, ref;
    if ((_a2 = domConfig == null ? void 0 : domConfig.syncOptions) == null ? void 0 : _a2.ignoreRefs) {
      ignoreRefs = true;
    }
    if (children && !(domConfig instanceof Node)) {
      if (Array.isArray(children)) {
        for (i = 0; i < children.length; ++i) {
          DH2.normalizeChildren(children[i], namedChildren, ignoreRefs);
        }
      } else {
        kids = children;
        domConfig.children = children = [];
        for (name in kids) {
          child = kids[name];
          if (child == null ? void 0 : child.isWidget) {
            child = child.element;
          }
          ref = !name.startsWith("$") && !DH2.isElement(child);
          ref && (namedChildren == null ? void 0 : namedChildren(
            name,
            /* hoist = */
            !ignoreRefs
          ));
          if (child) {
            if (!(child instanceof Node)) {
              if (child.reference === false) {
                delete child.reference;
              } else if (ref && typeof child !== "string") {
                child.reference = name;
              }
              DH2.normalizeChildren(child, namedChildren, ignoreRefs);
            }
            children.push(child);
          }
        }
      }
    }
    return domConfig;
  }
  static roundPx(px, devicePixelRatio = globalThis.devicePixelRatio || 1) {
    const multiplier = 1 / devicePixelRatio;
    return Math.round(px / multiplier) * multiplier;
  }
  // For use when we are dividing a DOM element into even parts. The resulting value
  // must be floored to prevent overflow. But only floored to the device's resolution,
  // so raw Math.floor will not work - it would leave empty space in hi resolution screens.
  static floorPx(px, devicePixelRatio = globalThis.devicePixelRatio || 1) {
    const multiplier = 1 / devicePixelRatio;
    return Math.floor(px * multiplier) / multiplier;
  }
  /**
   * Returns true if element has opened shadow root
   * @param {HTMLElement} element Element to check
   * @returns {Boolean}
   */
  static isCustomElement(element) {
    return Boolean(element == null ? void 0 : element.shadowRoot);
  }
  /**
   * Resolves element from point, checking shadow DOM if required
   * @param {Number} x
   * @param {Number} y
   * @returns {HTMLElement}
   */
  static elementFromPoint(x, y) {
    let el = document.elementFromPoint(x, y);
    if (DH2.isCustomElement(el)) {
      el = el.shadowRoot.elementFromPoint(x, y) || el;
    }
    return el;
  }
  /**
   * Resolves child element from point __in the passed element's coordinate space__.
   * @param {HTMLElement} parent The element to find the occupying element in.
   * @param {Number|Core.helper.util.Point} x Either the `X` part of a point, or the point to find.
   * @param {Number} [y] The `Y` part of the point.
   * @returns {HTMLElement}
   * @internal
   */
  static childFromPoint(el, x, y, parent = el) {
    const p = y == null ? x : new Rectangle(x, y, 0, 0);
    let result = null;
    Array.from(el.children).reverse().some((el2) => {
      if (Rectangle.from(el2, parent).contains(p)) {
        result = el2.children.length && DH2.childFromPoint(el2, p, null, parent) || el2;
        return true;
      }
    });
    return result;
  }
  /**
   * Converts a name/value pair of a style name and its value into the canonical (hyphenated) name of the style
   * property and a value with the `defaultUnit` suffix appended if no unit is already present in the `value`.
   *
   * For example:
   * ```javascript
   *  const [property, value] = DomHelper.unitize('marginLeft', 50);
   *  console.log(property, value);
   * ```
   *
   * ```
   *  > margin-left 50px
   * ```
   * @param {String} name
   * @param {String|Number} value
   * @param {String} [defaultUnit]
   * @returns {String[]}
   * @internal
   */
  static unitize(name, value, defaultUnit = "px") {
    const [trueName, hasUnits] = canonicalizeStyle(name);
    if (value != null) {
      value = String(value);
      value = hasUnits && digitsRe.test(value) ? value + defaultUnit : value;
    }
    return [trueName, value];
  }
  /**
   * Returns active element checking shadow dom too
   * @readonly
   * @property {HTMLElement}
   */
  static get activeElement() {
    let el = document.activeElement;
    while (el.shadowRoot) {
      el = el.shadowRoot.activeElement;
    }
    return el;
  }
  // returns active element for DOM tree / shadow DOM tree to which element belongs
  static getActiveElement(element) {
    if (element == null ? void 0 : element.isWidget) {
      element = element.element;
    }
    let el = ((element == null ? void 0 : element.getRootNode()) || document).activeElement;
    while (el == null ? void 0 : el.shadowRoot) {
      el = el.shadowRoot.activeElement;
    }
    return el;
  }
  // Returns the visible root (either document.body or a web component shadow root)
  static getRootElement(element) {
    var _a2;
    const root = (_a2 = element.getRootNode) == null ? void 0 : _a2.call(element), { nodeType } = root;
    return nodeType === Node.DOCUMENT_NODE ? root.body : nodeType === Node.DOCUMENT_FRAGMENT_NODE ? root : element.ownerDocument.contains(element) ? element.ownerDocument.body : null;
  }
  // Returns the topmost HTMLElement inside the current context (either document.body or a direct child of a web component shadow root)
  static getOutermostElement(element) {
    var _a2;
    const root = (_a2 = element.getRootNode) == null ? void 0 : _a2.call(element);
    if (root == null ? void 0 : root.body) {
      return root == null ? void 0 : root.body;
    }
    while (element.parentNode !== root && element.parentNode) {
      element = element.parentNode;
    }
    return element;
  }
  static isValidFloatRootParent(target) {
    return target === document.body || target.constructor.name === "ShadowRoot";
  }
  /**
   * Returns the `id` of the passed element. Generates a unique `id` if the element does not have one.
   * @param {HTMLElement} element The element to return the `id` of.
   */
  static getId(element) {
    return element.id || (element.id = "b-element-" + ++idCounter);
  }
  /**
   * Returns common widget/node ancestor for from/to arguments
   * @param {Core.widget.Widget|HTMLElement} from
   * @param {Core.widget.Widget|HTMLElement} to
   * @returns {Core.widget.Widget|HTMLElement}
   * @internal
   */
  static getCommonAncestor(from, to) {
    var _a2;
    if (from === to) {
      return from;
    }
    while (from && !(((_a2 = from[from.isWidget ? "owns" : "contains"]) == null ? void 0 : _a2.call(from, to)) || from === to)) {
      from = from.owner || from.parentNode;
    }
    return from;
  }
  //region Internal
  /**
   * Internal convenience fn to allow specifying either an element or a CSS selector to retrieve one
   * @private
   * @param {String|HTMLElement} elementOrSelector element or selector to lookup in DOM
   * @returns {HTMLElement}
   */
  static getElement(elementOrSelector) {
    if (elementOrSelector instanceof Element) {
      return elementOrSelector;
    }
    return doc.querySelector(elementOrSelector);
  }
  /**
   * Sets attributes passed as object to given element
   * @param {String|Element} elementOrSelector
   * @param {Object} attributes
   * @internal
   */
  static setAttributes(elementOrSelector, attributes) {
    const element = DH2.getElement(elementOrSelector);
    if (element && attributes) {
      for (const key in attributes) {
        if (attributes[key] == null) {
          element.removeAttribute(key);
        } else {
          element.setAttribute(key, attributes[key]);
        }
      }
    }
  }
  /**
   * Sets a CSS [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length) style value.
   * @param {String|HTMLElement} element The element to set the style in, or, if just the result is required,
   * the style magnitude to return with units added. If a nullish value is passed, an empty string
   * is returned.
   * @param {String} [style] The name of a style property which specifies a [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length)
   * @param {Number|String} [value] The magnitude. If a number is used, the value will be set in `px` units.
   * @returns {String} The style value string.
   */
  static setLength(element, style, value) {
    if (arguments.length === 1) {
      value = typeof element === "number" ? `${element}px` : element != null ? element : "";
    } else {
      element = DH2.getElement(element);
      value = element.style[style] = typeof value === "number" ? `${value}px` : value != null ? value : "";
    }
    return value;
  }
  /**
   * Returns string percentified and rounded value for setting element's height, width etc.
   * @param {String|Number} value percent value
   * @param {Number} digits number of decimal digits for rounding
   * @returns {string} percentified value or empty string if value can not be parsed
   * @internal
   */
  static percentify(value, digits = 2) {
    const mult = Math.pow(10, digits);
    return value == null || value === "" || isNaN(value) ? "" : `${Math.round(value * mult) / mult}%`;
  }
  //endregion
  //region Children, going down...
  /**
   * Gets the first direct child of `element` that matches `selector`.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {HTMLElement}
   * @category Query children
   */
  static getChild(element, selector) {
    return element.querySelector(":scope>" + selector);
  }
  /**
   * Checks if `element` has any child that matches `selector`.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {Boolean} true if any child matches selector
   * @category Query children
   */
  static hasChild(element, selector) {
    return DH2.getChild(element, selector) != null;
  }
  /**
   * Returns all child elements (not necessarily direct children) that matches `selector`.
   *
   * If `selector` starts with `'>'` or `'# '`, then all components of the `selector` must match inside of `element`.
   * The scope selector, `:scope` is prepended to the selector (and if `#` was used, it is removed).
   *
   * These are equivalent:
   *
   *      DomHelper.children(el, '# .foo .bar');
   *
   *      el.querySelectorAll(':scope .foo .bar');
   *
   * These are also equivalent:
   *
   *      DomHelper.children(el, '> .foo .bar');
   *
   *      el.querySelectorAll(':scope > .foo .bar');
   *
   * @param {HTMLElement} element The parent element
   * @param {String} selector The CSS selector
   * @returns {HTMLElement[]} Matched elements, somewhere below `element`
   * @category Query children
   */
  static children(element, selector) {
    if (selector[0] === ">" || selector.startsWith("# ")) {
      if (selector[0] === "#") {
        selector = selector.substr(2);
      }
      selector = ":scope " + selector;
    }
    return Array.from(element.querySelectorAll(selector));
  }
  // Salesforce doesn't yet support childElementCount. So we relace all native usages with this wrapper and
  // override it for salesforce environment.
  // https://github.com/bryntum/support/issues/3008
  static getChildElementCount(element) {
    return element.childElementCount;
  }
  /**
   * Looks at the specified `element` and all of its children for the one that first matches `selector.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {HTMLElement} Matched element, either element or an element below it
   * @category Query children
   */
  static down(element, selector) {
    if (!element) {
      return null;
    }
    if (element.matches && element.matches(selector)) {
      return element;
    }
    selector = ":scope " + selector;
    return element.querySelector(selector);
  }
  /**
   * Checks if childElement is a descendant of parentElement (contained in it or a sub element)
   * @param {HTMLElement} parentElement Parent element
   * @param {HTMLElement} childElement Child element, at any level below parent (includes nested shadow roots)
   * @returns {Boolean}
   * @category Query children
   */
  static isDescendant(parentElement, childElement) {
    const parentRoot = DH2.getRootElement(parentElement), childRoot = DH2.getRootElement(childElement);
    if (childRoot && parentRoot !== childRoot && childRoot.host) {
      return DH2.isDescendant(parentRoot, childRoot.host);
    }
    return parentElement.contains(childElement);
  }
  /**
   * Returns the specified element of the given `event`. If the `event` is an `Element`, it is returned. Otherwise,
   * the `eventName` argument is used to retrieve the desired element property from `event` (this defaults to the
   * `'target'` property).
   * @param {Event|Element} event
   * @param {String} [elementName]
   * @returns {Element}
   */
  static getEventElement(event, elementName = "target") {
    return !event || DH2.isElement(event) ? event : event[elementName];
  }
  /**
   * Returns `true` if the provided value is _likely_ a DOM element. If the element can be assured to be from the
   * same document, `instanceof Element` is more reliable.
   * @param {*} value
   * @returns {Boolean}
   */
  static isElement(value) {
    return (value == null ? void 0 : value.nodeType) === document.ELEMENT_NODE && DH2.isNode(value);
  }
  /**
   * Returns `true` if the provided element is an instance of React Element.
   * All React elements require an additional $$typeof: Symbol.for('react.element') field declared on the object for security reasons.
   * The object which React.createElement() return has $$typeof property equals to Symbol.for('react.element')
   *
   * Sources:
   * https://reactjs.org/blog/2015/12/18/react-components-elements-and-instances.html
   * https://github.com/facebook/react/pull/4832
   *
   * @param {*} element
   * @returns {Boolean}
   * @internal
   */
  static isReactElement(element) {
    return (element == null ? void 0 : element.$$typeof) === Symbol.for("react.element");
  }
  /**
   * Returns `true` if the provided value is _likely_ a DOM node. If the node can be assured to be from the same
   * document, `instanceof Node` is more reliable.
   * @param {*} value
   * @returns {Boolean}
   */
  static isNode(value) {
    return Boolean(value) && typeof value.nodeType === "number" && !isObject(value);
  }
  /**
   * Iterates over each result returned from `element.querySelectorAll(selector)`. First turns it into an array to
   * work in IE. Can also be called with only two arguments, in which case the first argument is used as selector and
   * document is used as the element.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @param {Function} fn Function called for each found element
   * @category Query children
   */
  static forEachSelector(element, selector, fn2) {
    if (typeof element === "string") {
      throw new Error("DomHelper.forEachSelector must provide a root element context (for shadow root scenario)");
    }
    DH2.children(element, selector).forEach(fn2);
  }
  /**
   * Iterates over the direct child elements of the specified element. First turns it into an array to
   * work in IE.
   * @param {HTMLElement} element Parent element
   * @param {Function} fn Function called for each child element
   * @category Query children
   */
  static forEachChild(element, fn2) {
    Array.from(element.children).forEach(fn2);
  }
  /**
   * Removes each element returned from `element.querySelectorAll(selector)`.
   * @param {HTMLElement} element
   * @param {String} selector
   * @category Query children
   */
  static removeEachSelector(element, selector) {
    DH2.forEachSelector(element, selector, (child) => child.remove());
  }
  static removeClsGlobally(element, ...classes) {
    classes.forEach((cls2) => DH2.forEachSelector(element, "." + cls2, (child) => child.classList.remove(cls2)));
  }
  //endregion
  //region Parents, going up...
  /**
   * Looks at the specified element and all of its parents for the one that first matches selector.
   * @deprecated Since 5.3.9, use native `element.closest()` instead
   * @param {HTMLElement} element Element
   * @param {String} selector CSS selector
   * @returns {HTMLElement} Matched element, either the passed in element or an element above it
   * @category Query parents
   */
  static up(element, selector) {
    VersionHelper.deprecate("Core", "6.0.0", "DomHelper.up() deprecated, use native `element.closest()` instead");
    return element.closest(selector);
  }
  static getAncestor(element, possibleAncestorParents, outerElement = null) {
    let found = false, ancestor, parent = element;
    possibleAncestorParents = ArrayHelper.asArray(possibleAncestorParents);
    while (parent = parent.parentElement) {
      if (possibleAncestorParents.includes(parent)) {
        found = true;
        break;
      }
      if (outerElement && parent === outerElement)
        break;
      ancestor = parent;
    }
    if (!found)
      return null;
    return ancestor || element;
  }
  /**
   * Retrieves all parents to the specified element.
   * @param {HTMLElement} element Element
   * @returns {HTMLElement[]} All parent elements, bottom up
   * @category Query parents
   */
  static getParents(element) {
    const parents = [];
    while (element.parentElement) {
      parents.push(element.parentElement);
      element = element.parentElement;
    }
    return parents;
  }
  //endregion
  //region Creation
  /**
   * Converts the passed id to an id valid for usage as id on a DOM element.
   * @param {String} id
   * @returns {String}
   */
  static makeValidId(id, replaceValue = "") {
    return StringHelper.makeValidDomId(id, replaceValue);
  }
  /**
   * Creates an Element, accepts a {@link #typedef-DomConfig} object. Example usage:
   *
   * ```javascript
   * DomHelper.createElement({
   *   tag         : 'table', // defaults to 'div'
   *   className   : 'nacho',
   *   html        : 'I am a nacho',
   *   children    : [ { tag: 'tr', ... }, myDomElement ],
   *   parent      : myExistingElement // Or its id
   *   style       : 'font-weight: bold;color: red',
   *   dataset     : { index: 0, size: 10 },
   *   tooltip     : 'Yay!',
   *   ns          : 'http://www.w3.org/1999/xhtml'
   * });
   * ```
   *
   * @param {DomConfig} config Element config object
   * @param {Object} [options] An object specifying creation options. If this is a boolean value, it is
   * understood to be the `returnAll` option.
   * @param {Boolean} [options.ignoreRefs] Pass `true` to ignore element references.
   * @param {Boolean} [options.returnAll] Specify true to return all elements & child elements
   * created as an array.
   * @returns {HTMLElement|HTMLElement[]|Object<String,HTMLElement>} Single element or array of elements `returnAll` was set to true.
   * If any elements had a `reference` property, this will be an object containing a reference to
   * all those elements, keyed by the reference name.
   * @category Creation
   */
  static createElement(config = {}, options) {
    var _a2, _b;
    let returnAll = options, element, i, ignoreChildRefs, ignoreRefOption, ignoreRefs, key, name, value, refOwner, refs, syncIdField;
    if (typeof returnAll === "boolean") {
      throw new Error("Clean up");
    } else if (options) {
      ignoreRefs = options.ignoreRefs;
      refOwner = options.refOwner;
      refs = options.refs;
      returnAll = options.returnAll;
      syncIdField = options.syncIdField;
      if (ignoreRefs) {
        ignoreChildRefs = true;
        ignoreRefs = ignoreRefs !== "children";
      }
    }
    if (typeof config.parent === "string") {
      config.parent = document.getElementById(config.parent);
    }
    const parent = config.parent || config.nextSibling && config.nextSibling.parentNode, { dataset, html, reference: reference2, syncOptions, text } = config;
    if (syncOptions) {
      syncIdField = syncOptions.syncIdField || syncIdField;
      ignoreRefOption = syncOptions.ignoreRefs;
      if (ignoreRefOption) {
        ignoreChildRefs = true;
        ignoreRefs = ignoreRefOption !== "children";
        options = {
          ...options,
          ignoreRefs: true
        };
      }
    }
    if (ignoreRefs) {
      refOwner = null;
    }
    if (config.ns) {
      element = doc.createElementNS(config.ns, config.tag || "svg");
    } else {
      element = doc.createElement(config.tag || "div");
    }
    if (text != null) {
      DH2.setInnerText(element, text);
    } else if (html != null) {
      if (html instanceof DocumentFragment) {
        element.appendChild(html);
      } else {
        element.innerHTML = html;
      }
    }
    if (config.tooltip) {
      DH2.Widget.attachTooltip(element, config.tooltip);
    }
    if (config.style) {
      DH2.applyStyle(element, config.style);
    }
    if (dataset) {
      for (name in dataset) {
        value = dataset[name];
        if (value != null) {
          element.dataset[name] = value;
        }
      }
    }
    if (parent) {
      this.addChild(parent, element, config.nextSibling);
    }
    if (refOwner) {
      element.$refOwnerId = refOwner.id;
    }
    if (reference2 && !ignoreRefs) {
      if (refOwner) {
        element.$reference = reference2;
        refOwner.attachRef(reference2, element, config);
      } else {
        if (!refs) {
          options = Object.assign({}, options);
          options.refs = refs = {};
        }
        refs[reference2] = element;
        element.setAttribute("data-reference", reference2);
      }
    }
    const className = config.className || config.class, keys = Object.keys(config);
    if (className) {
      element.setAttribute("class", DomClassList.normalize(className));
    }
    for (i = 0; i < keys.length; ++i) {
      name = keys[i];
      value = config[name];
      if ((key = elementCreateExpandos[name]) != null) {
        element[key || name] = value;
      } else if (!elementCreateProperties[name] && name && value != null) {
        element.setAttribute(name, value);
      }
    }
    if (!config["aria-hidden"] && !config.role && !config.tabIndex && !DomHelper.isFocusable(element, true) && !element.htmlFor) {
      element.setAttribute("role", "presentation");
    }
    (_b = options == null ? void 0 : options.callback) == null ? void 0 : _b.call(options, {
      action: "newElement",
      domConfig: config,
      targetElement: element,
      syncId: refOwner ? reference2 : options.syncIdField && ((_a2 = config.dataset) == null ? void 0 : _a2[options.syncIdField])
    });
    if (returnAll === true) {
      options.returnAll = returnAll = [element];
    } else if (Array.isArray(returnAll)) {
      returnAll.push(element);
    }
    if (config.children) {
      if (syncIdField) {
        element.syncIdMap = {};
      }
      config.children.forEach((child) => {
        var _a3, _b2, _c;
        if (child) {
          if (typeof child === "string") {
            const textNode = document.createTextNode(child);
            if (refOwner) {
              textNode.$refOwnerId = refOwner.id;
            }
            element.appendChild(textNode);
          } else if (isNaN(child.nodeType)) {
            child.parent = element;
            if (!child.ns && config.ns) {
              child.ns = config.ns;
            }
            const childElement = DH2.createElement(child, {
              ...options,
              ignoreRefs: (_b2 = (_a3 = config.syncOptions) == null ? void 0 : _a3.ignoreRef) != null ? _b2 : ignoreChildRefs
            }), syncId = (_c = child.dataset) == null ? void 0 : _c[syncIdField];
            if (syncId != null) {
              element.syncIdMap[syncId] = childElement;
            }
            delete child.parent;
          } else {
            element.appendChild(child);
          }
        }
      });
    }
    element.lastDomConfig = config;
    return refs || returnAll || element;
  }
  /**
   * Create element(s) from a template (html string). Note that
   * `textNode`s are discarded unless the `raw` option is passed
   * as `true`.
   *
   * If the template has a single root element, then the single element will be returned
   * unless the `array` option is passed as `true`.
   *
   * If there are multiple elements, then an Array will be returned.
   *
   * @param {String} template The HTML string from which to create DOM content
   * @param {Object} [options] An object containing properties to modify how the DOM is created and returned.
   * @param {Boolean} [options.array] `true` to return an array even if there's only one resulting element.
   * @param {Boolean} [options.raw] Return all child nodes, including text nodes.
   * @param {Boolean} [options.fragment] Return a DocumentFragment.
   * @private
   */
  static createElementFromTemplate(template, options = emptyObject4) {
    const { array, raw, fragment } = options;
    let result;
    if (DH2.supportsTemplate) {
      (templateElement || (templateElement = doc.createElement("template"))).innerHTML = template;
      result = templateElement.content;
      if (fragment) {
        return result.cloneNode(true);
      }
    } else {
      result = (htmlParser || (htmlParser = new DOMParser())).parseFromString(template, "text/html").body;
      if (fragment) {
        const nodes = result.childNodes;
        result = document.createDocumentFragment();
        while (nodes.length) {
          result.appendChild(nodes[0]);
        }
        return result;
      }
    }
    if (raw) {
      result = result.childNodes;
    } else {
      result = result.children;
    }
    return result.length === 1 && !array ? result[0] : arraySlice.call(result);
  }
  /**
   * Dispatches a MouseEvent of the passed type to the element at the visible centre of the passed element.
   * @param {HTMLElement} targetElement The element whose center receives the mouse event.
   * @param {String} [type=contextmenu] The mouse event type to dispatch.
   * @internal
   */
  static triggerMouseEvent(targetElement, type = "contextmenu") {
    const isInView = this.isInView(targetElement), targetRect = isInView || Rectangle.from(targetElement), targetPoint = targetRect.center, contextmenuEvent = new MouseEvent(type, {
      clientX: targetPoint.x,
      clientY: targetPoint.y,
      bubbles: true
    });
    targetElement.dispatchEvent(contextmenuEvent);
  }
  /**
   * Inserts an `element` at first position in `into`.
   * @param {HTMLElement} into Parent element
   * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
   * @returns {HTMLElement}
   * @category Creation
   */
  static insertFirst(into, element) {
    if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
      element = DH2.createElement(element);
    }
    return into.insertBefore(element, into.firstElementChild);
  }
  /**
   * Inserts a `element` before `beforeElement` in `into`.
   * @param {HTMLElement} into Parent element
   * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
   * @param {HTMLElement} beforeElement Element before which passed element should be inserted
   * @returns {HTMLElement}
   * @category Creation
   */
  static insertBefore(into, element, beforeElement) {
    if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
      element = DH2.createElement(element);
    }
    return beforeElement ? into.insertBefore(element, beforeElement) : DH2.insertFirst(into, element);
  }
  static insertAt(parentElement, newElement, index) {
    const siblings = Array.from(parentElement.children);
    if (index >= siblings.length) {
      return DH2.append(parentElement, newElement);
    }
    const beforeElement = siblings[index];
    return DH2.insertBefore(parentElement, newElement, beforeElement);
  }
  /**
   * Appends element to parentElement.
   * @param {HTMLElement} parentElement Parent element
   * @param {HTMLElement|DomConfig|String} elementOrConfig Element to insert, or an element config passed on to
   * `createElement()`, or an html string passed to `createElementFromTemplate()`
   * @returns {HTMLElement}
   * @category Creation
   */
  static append(parentElement, elementOrConfig) {
    if (elementOrConfig.forEach) {
      if (Array.isArray(elementOrConfig)) {
        elementOrConfig = elementOrConfig.map((elementOrConfig2) => elementOrConfigToElement(elementOrConfig2));
      }
      if (parentElement.append) {
        parentElement.append(...elementOrConfig);
      } else {
        const docFrag = document.createDocumentFragment();
        elementOrConfig.forEach(function(child) {
          docFrag.appendChild(child);
        });
        parentElement.appendChild(docFrag);
      }
      return elementOrConfig;
    } else {
      return parentElement.appendChild(elementOrConfigToElement(elementOrConfig));
    }
  }
  //endregion
  //region Get position
  /**
   * Returns the element's `transform translateX` value in pixels.
   * @param {HTMLElement} element
   * @returns {Number} X transform
   * @category Position, get
   */
  static getTranslateX(element) {
    const transformStyle = element.style.transform;
    let matches = pxTtranslateXRe.exec(transformStyle);
    if (matches) {
      return parseFloat(matches[2]);
    } else {
      matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH2.getStyleValue(element, "transform"));
      return matches ? parseFloat(matches[1] || matches[3]) : 0;
    }
  }
  /**
   * Returns the element's `transform translateY` value in pixels.
   * @param {HTMLElement} element
   * @returns {Number} Y coordinate
   * @category Position, get
   */
  static getTranslateY(element) {
    const transformStyle = element.style.transform;
    let matches = pxTtranslateYRe.exec(transformStyle);
    if (matches) {
      const y = parseFloat(matches[matches[1] === "Y" ? 2 : 3]);
      return isNaN(y) ? 0 : y;
    } else {
      matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH2.getStyleValue(element, "transform"));
      return matches ? parseFloat(matches[2] || matches[4]) : 0;
    }
  }
  /**
   * Gets both X and Y coordinates as an array [x, y]
   * @param {HTMLElement} element
   * @returns {Number[]} [x, y]
   * @category Position, get
   */
  static getTranslateXY(element) {
    return [DH2.getTranslateX(element), DH2.getTranslateY(element)];
  }
  /**
   * Get elements X offset within a containing element
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number} X offset
   * @category Position, get
   */
  static getOffsetX(element, container = null) {
    return container ? element.getBoundingClientRect().left - container.getBoundingClientRect().left : element.offsetLeft;
  }
  /**
   * Get elements Y offset within a containing element
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number} Y offset
   * @category Position, get
   */
  static getOffsetY(element, container = null) {
    return container ? element.getBoundingClientRect().top - container.getBoundingClientRect().top : element.offsetTop;
  }
  /**
   * Gets elements X and Y offset within containing element as an array [x, y]
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number[]} [x, y]
   * @category Position, get
   */
  static getOffsetXY(element, container = null) {
    return [DH2.getOffsetX(element, container), DH2.getOffsetY(element, container)];
  }
  /**
   * Focus element without scrolling the element into view.
   * @param {HTMLElement} element
   */
  static focusWithoutScrolling(element) {
    function resetScroll(scrollHierarchy) {
      scrollHierarchy.forEach(({ element: element2, scrollLeft, scrollTop }) => {
        if (element2.scrollLeft !== scrollLeft) {
          element2.scrollLeft = scrollLeft;
        }
        if (element2.scrollTop !== scrollTop) {
          element2.scrollTop = scrollTop;
        }
      });
    }
    const preventScrollSupported = !BrowserHelper.isSafari;
    if (preventScrollSupported) {
      element.focus({ preventScroll: true });
    } else {
      const parents = DH2.getParents(element), scrollHierarchy = parents.map((parent) => ({
        element: parent,
        scrollLeft: parent.scrollLeft,
        scrollTop: parent.scrollTop
      }));
      element.focus();
      setTimeout(() => resetScroll(scrollHierarchy), 0);
    }
  }
  /**
   * Get elements X position on page
   * @param {HTMLElement} element
   * @returns {Number}
   * @category Position, get
   */
  static getPageX(element) {
    return element.getBoundingClientRect().left + win.pageXOffset;
  }
  /**
   * Get elements Y position on page
   * @param {HTMLElement} element
   * @returns {Number}
   * @category Position, get
   */
  static getPageY(element) {
    return element.getBoundingClientRect().top + win.pageYOffset;
  }
  /**
   * Returns extremal (min/max) size (height/width) of the element in pixels
   * @param {HTMLElement} element
   * @param {String} style minWidth/minHeight/maxWidth/maxHeight
   * @returns {Number}
   * @internal
   */
  static getExtremalSizePX(element, style) {
    const prop = StringHelper.hyphenate(style), measure = prop.split("-")[1];
    let value = DH2.getStyleValue(element, prop);
    if (/%/.test(value)) {
      if (element.parentElement) {
        value = parseInt(DH2.getStyleValue(element.parentElement, measure), 10);
      } else {
        value = NaN;
      }
    } else {
      value = parseInt(value, 10);
    }
    return value;
  }
  //endregion
  //region Set position
  /**
   * Set element's `scale`.
   * @param {HTMLElement} element
   * @param {Number} scaleX The value by which the element should be scaled in the X axis (0 to 1)
   * @param {Number} [scaleY] The value by which the element should be scaled in the Y axis (0 to 1).
   * Defaults to `scaleX`
   * @category Position, set
   * @internal
   */
  static setScale(element, scaleX, scaleY = scaleX) {
    const t = DH2.getStyleValue(element, "transform").split(/,\s*/);
    if (t.length > 1) {
      if (t[0].startsWith("matrix3d")) {
        t[0] = `matrix3d(${scaleX}`;
        t[5] = scaleY;
      } else {
        t[0] = `matrix(${scaleX}`;
        t[3] = scaleY;
      }
      element.style.transform = t.join(",");
    } else {
      element.style.transform = `scale(${scaleX}, ${scaleY})`;
    }
  }
  /**
   * Set element's `X` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} x The value by which the element should be translated from its default position.
   * @category Position, set
   */
  static setTranslateX(element, x) {
    const t = DH2.getStyleValue(element, "transform").split(/,\s*/);
    x = DH2.roundPx(x);
    if (t.length > 1) {
      t[t[0].startsWith("matrix3d") ? 12 : 4] = x;
      element.style.transform = t.join(",");
    } else {
      element.style.transform = `translateX(${x}px)`;
    }
  }
  /**
   * Set element's `Y` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} y  The value by which the element should be translated from its default position.
   * @category Position, set
   */
  static setTranslateY(element, y) {
    const t = DH2.getStyleValue(element, "transform").split(/,\s*/);
    y = DH2.roundPx(y);
    if (t.length > 1) {
      t[t[0].startsWith("matrix3d") ? 13 : 5] = y;
      element.style.transform = t.join(",") + ")";
    } else {
      element.style.transform = `translateY(${y}px)`;
    }
  }
  /**
   * Set element's style `top`.
   * @param {HTMLElement} element
   * @param {Number|String} y The top position. If numeric, `'px'` is used as the unit.
   * @category Position, set
   */
  static setTop(element, y) {
    DH2.setLength(element, "top", y);
  }
  /**
   * Set element's style `left`.
   * @param {HTMLElement} element
   * @param {Number|String} x The top position. If numeric, `'px'` is used as the unit.
   * @category Position, set
   */
  static setLeft(element, x) {
    DH2.setLength(element, "left", x);
  }
  static setTopLeft(element, y, x) {
    DH2.setLength(element, "top", y);
    DH2.setLength(element, "left", x);
  }
  static setRect(element, { x, y, width, height }) {
    DH2.setTopLeft(element, y, x);
    DH2.setLength(element, "width", width);
    DH2.setLength(element, "height", height);
  }
  /**
   * Set elements `X` and `Y` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} [x] The `X translation.
   * @param {Number} [y] The `Y translation.
   * @category Position, set
   */
  static setTranslateXY(element, x, y) {
    if (x == null) {
      return DH2.setTranslateY(element, y);
    }
    if (y == null) {
      return DH2.setTranslateX(element, x);
    }
    x = DH2.roundPx(x);
    y = DH2.roundPx(y);
    const t = DH2.getStyleValue(element, "transform").split(/,\s*/), is3d = t[0].startsWith("matrix3d");
    if (t.length > 1) {
      t[is3d ? 12 : 4] = x;
      t[is3d ? 13 : 5] = y;
      element.style.transform = t.join(",") + ")";
    } else {
      element.style.transform = `translate(${x}px, ${y}px)`;
    }
  }
  /**
   * Increase `X` translation
   * @param {HTMLElement} element
   * @param {Number} x The number of pixels by which to increase the element's `X` translation.
   * @category Position, set
   */
  static addTranslateX(element, x) {
    DH2.setTranslateX(element, DH2.getTranslateX(element) + x);
  }
  /**
   * Increase `Y` position
   * @param {HTMLElement} element
   * @param {Number} y The number of pixels by which to increase the element's `Y` translation.
   * @category Position, set
   */
  static addTranslateY(element, y) {
    DH2.setTranslateY(element, DH2.getTranslateY(element) + y);
  }
  /**
   * Increase X position
   * @param {HTMLElement} element
   * @param {Number} x
   * @category Position, set
   */
  static addLeft(element, x) {
    DH2.setLeft(element, DH2.getOffsetX(element) + x);
  }
  /**
   * Increase Y position
   * @param {HTMLElement} element
   * @param {Number} y
   * @category Position, set
   */
  static addTop(element, y) {
    DH2.setTop(element, DH2.getOffsetY(element) + y);
  }
  /**
   * Align the passed element with the passed target according to the align spec.
   * @param {HTMLElement} element The element to align.
   * @param {HTMLElement|Core.helper.util.Rectangle} target The target element or rectangle to align to
   * @param {Object} [alignSpec] See {@link Core.helper.util.Rectangle#function-alignTo} Defaults to `{ align : 't0-t0' }`
   * @param {Boolean} [round] Round the calculated Rectangles (for example if dealing with scrolling which
   * is integer based).
   */
  static alignTo(element, target, alignSpec = t0t0, round2) {
    target = target instanceof Rectangle ? target : Rectangle.from(target, true);
    const elXY = DH2.getTranslateXY(element), elRect = Rectangle.from(element, true);
    if (round2) {
      elRect.roundPx();
      target.roundPx();
    }
    const targetRect = elRect.alignTo(Object.assign(alignSpec, {
      target
    }));
    DH2.setTranslateXY(element, elXY[0] + targetRect.x - elRect.x, elXY[1] + targetRect.y - elRect.y);
  }
  //endregion
  //region Styles & CSS
  /**
   * Returns a style value or values for the passed element.
   * @param {HTMLElement} element The element to read styles from
   * @param {String|String[]} propName The property or properties to read
   * @param {Boolean} [inline=false] Pass as `true` to read the element's inline style.
   * Note that this could return inaccurate results if CSS rules apply to this element.
   * @returns {String|Object} The value or an object containing the values keyed by the requested property name.
   * @category CSS
   */
  static getStyleValue(element, propName, inline, pseudo) {
    const styles = inline ? element.style : element.ownerDocument.defaultView.getComputedStyle(element, pseudo);
    if (Array.isArray(propName)) {
      const result = {};
      for (const prop of propName) {
        result[prop] = styles.getPropertyValue(StringHelper.hyphenate(prop));
      }
      return result;
    }
    return styles.getPropertyValue(StringHelper.hyphenate(propName));
  }
  /**
   * Returns an object with the parse style values for the top, right, bottom, and left
   * components of the given edge style.
   *
   * The return value is an object with `top`, `right`, `bottom`, and `left` properties
   * for the respective components of the edge style, as well as `width` (the sum of
   * `left` and `right`) and `height` (the sum of `top` and `bottom`).
   *
   * @param {HTMLElement} element
   * @param {String} edgeStyle The element's desired edge style such as 'padding', 'margin',
   * or 'border'.
   * @param {String} [edges='trbl'] A string with one character codes for each edge. Only
   * those edges will be populated in the returned object. By default, all edges will be
   * populated.
   * @returns {Object}
   */
  static getEdgeSize(element, edgeStyle, edges) {
    const suffix = edgeStyle === "border" ? "-width" : "", ret = {
      raw: {}
    };
    for (const edge of ["top", "right", "bottom", "left"]) {
      if (!edges || edges.includes(edge[0])) {
        ret[edge] = parseFloat(
          ret.raw[edge] = DH2.getStyleValue(element, `${edgeStyle}-${edge}${suffix}`)
        );
      }
    }
    ret.width = (ret.left || 0) + (ret.right || 0);
    ret.height = (ret.top || 0) + (ret.bottom || 0);
    return ret;
  }
  /**
   * Splits a style string up into object form. For example `'font-weight:bold;font-size:150%'`
   * would convert to
   *
   * ```javascript
   * {
   *     font-weight : 'bold',
   *     font-size : '150%'
   * }
   * ```
   * @param {String} style A DOM style string
   * @returns {Object} the style declaration in object form.
   */
  static parseStyle(style) {
    if (typeof style === "string") {
      const styles = style.split(semicolonRe);
      style = {};
      for (let i = 0, { length } = styles; i < length; i++) {
        const propVal = styles[i].split(colonRe);
        style[propVal[0]] = propVal[1];
      }
    }
    return style || {};
  }
  /**
   * Applies specified style to the passed element. Style can be an object or a string.
   * @param {HTMLElement} element Target element
   * @param {String|Object} style Style to apply, 'border: 1px solid black' or { border: '1px solid black' }
   * @param {Boolean} [overwrite] Specify `true` to replace style instead of applying changes
   * @category CSS
   */
  static applyStyle(element, style, overwrite = false) {
    if (typeof style === "string") {
      if (overwrite) {
        if (style.length || element.style.cssText.length) {
          element.style.cssText = style;
        }
      } else {
        element.style.cssText += style;
      }
    } else if (style) {
      if (overwrite) {
        element.style.cssText = "";
      }
      if (style.style && typeof style.style !== "string") {
        style = ObjectHelper.assign({}, style, style.style);
      }
      let key, value;
      for (key in style) {
        if (!styleIgnoreProperties[key]) {
          [key, value] = DH2.unitize(key, style[key]);
          if (value == null) {
            element.style.removeProperty(key);
          } else {
            element.style.setProperty(key, value);
          }
        }
      }
      if (typeof style.style === "string") {
        element.style.cssText += style.style;
      }
    }
  }
  static getCSSText(style) {
    if (typeof style === "string") {
      return style;
    }
    let cssText = "";
    for (const key in style) {
      if (!styleIgnoreProperties[key]) {
        cssText += `${StringHelper.hyphenate(key)}:${style[key]};`;
      }
    }
    return cssText;
  }
  /**
   * Add multiple classes to elements classList.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @deprecated Since 5.0. Use {@link https://developer.mozilla.org/en-US/docs/Web/API/DOMTokenList/add add} method
   * for {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/classList Element.classlist}
   * @category CSS
   */
  static addClasses(element, classes) {
    VersionHelper.deprecate("Core", "6.0.0", "DomHelper.addClasses should be replaced by native classList.add");
    element.classList.add(...classes);
  }
  /**
   * Remove multiple classes from elements classList.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @deprecated Since 5.0. Use {@link https://developer.mozilla.org/en-US/docs/Web/API/DOMTokenList/remove remove} method
   * for {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/classList Element.classlist}
   * @category CSS
   */
  static removeClasses(element, classes) {
    VersionHelper.deprecate("Core", "6.0.0", "DomHelper.removeClasses should be replaced by native classList.remove");
    element.classList.remove(...classes);
  }
  /**
   * Toggle multiple classes in elements classList. Helper for toggling multiple classes at once.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @param {Boolean} [force] Specify true to add classes, false to remove. Leave blank to toggle
   * @category CSS
   */
  static toggleClasses(element, classes, force = null) {
    classes = ArrayHelper.asArray(classes);
    if (force === true) {
      element.classList.add(...classes);
    } else if (force === false) {
      element.classList.remove(...classes);
    } else {
      classes.forEach((cls2) => element.classList.toggle(cls2));
    }
  }
  /**
   * Adds a CSS class to an element during the specified duration
   * @param {HTMLElement} element Target element
   * @param {String} cls CSS class to add temporarily
   * @param {Number} duration Duration in ms, 0 means cls will not be applied
   * @param {Core.mixin.Delayable} delayable The delayable to tie the setTimeout call to
   * @typings delayable -> {typeof Delayable}
   * @category CSS
   */
  static addTemporaryClass(element, cls2, duration, delayable = globalThis) {
    if (duration > 0) {
      element.classList.add(cls2);
      delayable.setTimeout({
        fn: (cls3) => element.classList.remove(cls3),
        delay: duration,
        name: cls2,
        args: [cls2],
        cancelOutstanding: true
      });
    }
  }
  /**
   * Reads computed style from the element and returns transition duration for a given property in milliseconds
   * @param {HTMLElement} element Target DOM element
   * @param {String} property Animated property name
   * @returns {Number} Duration in ms
   * @internal
   */
  static getPropertyTransitionDuration(element, property) {
    const style = globalThis.getComputedStyle(element), properties = style.transitionProperty.split(", "), durations = style.transitionDuration.split(", "), index = properties.indexOf(StringHelper.hyphenate(property));
    let result;
    if (index !== -1) {
      result = parseFloat(durations[index]) * 1e3;
    }
    return result;
  }
  /**
   * Reads computed style from the element and returns the animation duration for any
   * attached animation in milliseconds
   * @param {HTMLElement} element Target DOM element
   * @returns {Number} Duration in ms
   * @internal
   */
  static getAnimationDuration(element) {
    return parseFloat(DH2.getStyleValue(element, "animation-duration")) * 1e3;
  }
  //endregion
  //region Effects
  /**
   * Highlights the passed element or Rectangle according to the theme's highlighting rules.
   * Usually an animated framing effect.
   *
   * The framing effect is achieved by adding the CSS class `b-fx-highlight` which references
   * a `keyframes` animation named `b-fx-highlight-animation`. You may override the animation
   * name referenced, or the animation itself in your own CSS.
   *
   * @param {HTMLElement|Core.helper.util.Rectangle} element The element or Rectangle to highlight.
   */
  static highlight(element, delayable = globalThis) {
    if (element instanceof Rectangle) {
      return element.highlight();
    }
    return new Promise((resolve) => {
      delayable.setTimeout(() => {
        element.classList.add("b-fx-highlight");
        delayable.setTimeout(() => {
          element.classList.remove("b-fx-highlight");
          resolve();
        }, 1e3);
      }, 0);
    });
  }
  //endregion
  //region Measuring / Scrollbar
  /**
   * Measures the scrollbar width using a hidden div. Caches result
   * @property {Number}
   * @readonly
   */
  static get scrollBarWidth() {
    if (scrollBarWidth === null) {
      const element = scrollBarMeasureElement || (scrollBarMeasureElement = DH2.createElement({
        parent: doc.documentElement,
        style: "position:absolute;top:-9999em;height:100px;overflow-y:scroll"
      }));
      if (element.parentNode !== doc.documentElement) {
        doc.documentElement.appendChild(element);
      }
      scrollBarWidth = element.offsetWidth;
    }
    return scrollBarWidth;
  }
  static get scrollBarPadElement() {
    return {
      className: "b-yscroll-pad",
      children: [{
        className: "b-yscroll-pad-sizer"
      }]
    };
  }
  /**
   * Resets DomHelper.scrollBarWidth cache, triggering a new measurement next time it is read
   */
  static resetScrollBarWidth() {
    scrollBarWidth = null;
  }
  /**
   * Measures the text width using a hidden div
   * @param {String} text
   * @param {HTMLElement} sourceElement
   * @returns {Number} width
   * @category Measure
   */
  static measureText(text, sourceElement, useHTML = false, parentElement = void 0) {
    const offScreenDiv = DH2.getMeasureElement(sourceElement, parentElement);
    offScreenDiv[useHTML ? "innerHTML" : "innerText"] = text;
    const result = offScreenDiv.clientWidth;
    offScreenDiv.className = "";
    return result;
  }
  /**
   * Measures a relative size, such as a size specified in `em` units for the passed element.
   * @param {String} size The CSS size value to measure.
   * @param {HTMLElement} sourceElement
   * @param {Boolean} [round] Pass true to return exact width, not rounded value
   * @returns {Number} size The size in pixels of the passed relative measurement.
   * @category Measure
   */
  static measureSize(size, sourceElement, round2 = true) {
    if (!size) {
      return 0;
    }
    if (typeof size === "number") {
      return size;
    }
    if (!size.length) {
      return 0;
    }
    if (/^\d+(px)?$/.test(size)) {
      return parseInt(size);
    }
    if (sourceElement) {
      const offScreenDiv = DH2.getMeasureElement(sourceElement);
      offScreenDiv.innerHTML = "";
      offScreenDiv.style.width = DH2.setLength(size);
      const result = round2 ? offScreenDiv.offsetWidth : offScreenDiv.getBoundingClientRect().width;
      offScreenDiv.style.width = offScreenDiv.className = "";
      return result;
    }
    if (/^\d+em$/.test(size)) {
      return parseInt(size) * DEFAULT_FONT_SIZE;
    }
    return isNaN(size) ? 0 : parseInt(size);
  }
  // parentElement allows measurement to happen inside a specific element, allowing scoped css rules to match
  static getMeasureElement(sourceElement, parentElement = doc.body) {
    const sourceElementStyle = win.getComputedStyle(sourceElement), offScreenDiv = parentElement.offScreenDiv = parentElement.offScreenDiv || DH2.createElement({
      parent: parentElement,
      style: "position:fixed;top:-10000px;left:-10000px;visibility:hidden;contain:strict",
      className: "b-measure-element",
      children: [{
        style: "white-space:nowrap;display:inline-block;will-change:contents;width:auto;contain:none"
      }]
    }, { returnAll: true })[1];
    fontProps.forEach((prop) => {
      if (offScreenDiv.style[prop] !== sourceElementStyle[prop]) {
        offScreenDiv.style[prop] = sourceElementStyle[prop];
      }
    });
    offScreenDiv.className = sourceElement.className;
    if (offScreenDiv.parentElement.parentElement !== parentElement) {
      parentElement.appendChild(offScreenDiv.parentElement);
    }
    return offScreenDiv;
  }
  /**
   * Strips the tags from a html string, returning text content.
   *
   * ```javascript
   * DomHelper.stripTags('<div class="custom"><b>Bold</b><i>Italic</i></div>'); // -> BoldItalic
   * ```
   *
   * @internal
   * @param {String} htmlString HTML string
   * @returns {String} Text content
   */
  static stripTags(htmlString) {
    const parser = DH2.$domParser || (DH2.$domParser = new DOMParser()), doc2 = parser.parseFromString(htmlString, "text/html");
    return doc2.body.textContent;
  }
  //endregion
  //region Sync
  /**
   * Sync one source element attributes, children etc. to a target element. Source element can be specified as a html
   * string or an actual HTMLElement.
   *
   * NOTE: This function is superseded by {@link Core/helper/DomSync#function-sync-static DomSync.sync()}, which works
   * with DOM configs. For most usecases, use it instead.
   *
   * @param {String|HTMLElement} sourceElement Source "element" to copy from
   * @param {HTMLElement} targetElement Target element to apply to, can also be specified as part of the config object
   * @returns {HTMLElement} Returns the updated targetElement (which is also updated in place)
   */
  static sync(sourceElement, targetElement) {
    if (typeof sourceElement === "string") {
      if (sourceElement === "") {
        targetElement.innerHTML = "";
        return;
      } else {
        sourceElement = DH2.createElementFromTemplate(sourceElement);
      }
    }
    DH2.performSync(sourceElement, targetElement);
    return targetElement;
  }
  // Internal helper used for recursive syncing
  static performSync(sourceElement, targetElement) {
    if (sourceElement.outerHTML !== targetElement.outerHTML) {
      DH2.syncAttributes(sourceElement, targetElement);
      DH2.syncContent(sourceElement, targetElement);
      DH2.syncChildren(sourceElement, targetElement);
      return true;
    }
    return false;
  }
  // Attributes as map { attr : value, ... }, either from an html element or from a config
  static getSyncAttributes(element) {
    const attributes = {}, names = [];
    for (let i = 0; i < element.attributes.length; i++) {
      const attr = element.attributes[i];
      if (attr.specified) {
        const name = attr.name.toLowerCase();
        attributes[name] = attr.value;
        names.push(name);
      }
    }
    return { attributes, names };
  }
  /**
   * Syncs attributes from sourceElement to targetElement.
   * @private
   * @param {HTMLElement} sourceElement
   * @param {HTMLElement} targetElement
   */
  static syncAttributes(sourceElement, targetElement) {
    const {
      attributes: sourceAttributes,
      names: sourceNames
    } = DH2.getSyncAttributes(sourceElement), {
      attributes: targetAttributes,
      names: targetNames
    } = DH2.getSyncAttributes(targetElement), hasDataset = sourceNames.includes("dataset"), toAdd = sourceNames.filter((attr) => !targetNames.includes(attr)), toRemove = targetNames.filter((attr) => !sourceNames.includes(attr) && (!hasDataset || !attr.startsWith("data-"))), toSync = sourceNames.filter((attr) => targetNames.includes(attr));
    if (toAdd.length > 0) {
      for (let i = 0; i < toAdd.length; i++) {
        const attr = toAdd[i];
        if (attr === "style") {
          DH2.applyStyle(targetElement, sourceAttributes.style, true);
        } else if (attr === "dataset") {
          Object.assign(targetElement.dataset, sourceAttributes.dataset);
        } else {
          targetElement.setAttribute(attr, sourceAttributes[attr]);
        }
      }
    }
    if (toRemove.length > 0) {
      for (let i = 0; i < toRemove.length; i++) {
        targetElement.removeAttribute(toRemove[i]);
      }
    }
    if (toSync.length > 0) {
      for (let i = 0; i < toSync.length; i++) {
        const attr = toSync[i];
        if (attr === "style") {
          DH2.applyStyle(targetElement, sourceAttributes.style, true);
        } else if (attr === "dataset") {
          Object.assign(targetElement.dataset, sourceAttributes.dataset);
        } else if (attr === "class" && (sourceAttributes.class.isDomClassList || typeof sourceAttributes.class === "object")) {
          let classList;
          if (sourceAttributes.class.isDomClassList) {
            classList = sourceAttributes.class;
          } else {
            classList = new DomClassList(sourceAttributes.class);
          }
          if (!classList.isEqual(targetAttributes.class)) {
            targetElement.setAttribute("class", classList);
          }
        } else if (targetAttributes[attr] !== sourceAttributes[attr]) {
          targetElement.setAttribute(attr, sourceAttributes[attr]);
        }
      }
    }
  }
  /**
   * Sync content (innerText) from sourceElement to targetElement
   * @private
   * @param {HTMLElement} sourceElement
   * @param {HTMLElement} targetElement
   */
  static syncContent(sourceElement, targetElement) {
    if (DH2.getChildElementCount(sourceElement) === 0) {
      targetElement.innerText = sourceElement.innerText;
    }
  }
  static setInnerText(targetElement, text) {
    const { firstChild } = targetElement;
    if ((firstChild == null ? void 0 : firstChild.nodeType) === Element.TEXT_NODE) {
      firstChild.data = text;
    } else {
      targetElement.textContent = text;
    }
  }
  /**
   * Sync traversing children
   * @private
   * @param {HTMLElement} sourceElement Source element
   * @param {HTMLElement} targetElement Target element
   */
  static syncChildren(sourceElement, targetElement) {
    const me = this, sourceNodes = arraySlice.call(sourceElement.childNodes), targetNodes = arraySlice.call(targetElement.childNodes);
    while (sourceNodes.length) {
      const sourceNode = sourceNodes.shift(), targetNode = targetNodes.shift();
      if (sourceNode && sourceNode.nodeType !== TEXT_NODE && sourceNode.nodeType !== ELEMENT_NODE) {
        throw new Error(`Source node type ${sourceNode.nodeType} not supported by DomHelper.sync()`);
      }
      if (targetNode && targetNode.nodeType !== TEXT_NODE && targetNode.nodeType !== ELEMENT_NODE) {
        throw new Error(`Target node type ${targetNode.nodeType} not supported by DomHelper.sync()`);
      }
      if (!targetNode) {
        targetElement.appendChild(sourceNode);
      } else {
        if (sourceNode.nodeType === targetNode.nodeType) {
          if (sourceNode.nodeType === TEXT_NODE) {
            targetNode.data = sourceNode.data;
          } else {
            if (sourceNode.tagName === targetNode.tagName) {
              me.performSync(sourceNode, targetNode);
            } else {
              targetElement.insertBefore(sourceNode, targetNode);
              targetNode.remove();
            }
          }
        } else if (sourceNode.nodeType === TEXT_NODE && targetNode.nodeType === ELEMENT_NODE) {
          targetElement.innerText = sourceNode.data.trim();
        } else {
          const logElement = sourceNode.parentElement || sourceNode;
          throw new Error(`Currently no support for transforming nodeType.
${logElement.outerHTML}`);
        }
      }
    }
    targetNodes.forEach((targetNode) => {
      targetNode.remove();
    });
  }
  /**
   * Replaces the passed element's `className` with the class names
   * passed in either Array or String format or Object.
   *
   * This method compares the existing class set with the incoming class set and
   * avoids mutating the element's class name set if possible.
   *
   * This can avoid browser style invalidations.
   * @param {HTMLElement} element The element whose class list to synchronize.
   * @param {String[]|String|Object} newClasses The incoming class names to set on the element.
   * @returns {Boolean} `true` if the DOM class list was changed.
   * @category CSS
   */
  static syncClassList(element, newClasses) {
    const { classList } = element, isString = typeof newClasses === "string", newClsArray = isString ? newClasses.split(whiteSpaceRe2) : DomClassList.normalize(newClasses, "array"), classCount = newClsArray.length;
    let changed = classList.length !== classCount, i;
    for (i = 0; !changed && i < classCount; i++) {
      changed = !classList.contains(newClsArray[i]);
    }
    if (changed) {
      element.className = isString ? newClasses : newClsArray.join(" ");
    }
    return changed;
  }
  /**
   * Applies the key state of the passed object or DomClassList to the passed element.
   *
   * Properties with a falsy value mean that property name is *removed* as a class name.
   *
   * Properties with a truthy value mean that property name is *added* as a class name.
   *
   * This is different from {@link #function-syncClassList-static}. That sets the `className` of the element to the
   * sum of all its truthy keys, regardless of what the pre-existing value of the `className` was, and ignoring falsy
   * keys.
   *
   * This _selectively_ updates the classes in the `className`. If there is a truthy key, the name is added. If there
   * is a falsy key, the name is removed.
   * @param {HTMLElement} element The element to apply the class list to .
   * @param {Object|Core.helper.util.DomClassList} classes The classes to add or remove.
   * @returns {Boolean} `true` if the DOM class list was changed.
   * @category CSS
   */
  static updateClassList(element, classes) {
    const { classList } = element;
    let cls2, add, changed = false;
    for (cls2 in classes) {
      add = Boolean(classes[cls2]);
      if (classList.contains(cls2) !== add) {
        classList[add ? "add" : "remove"](cls2);
        changed = true;
      }
    }
    return changed;
  }
  /**
   * Changes the theme to the passed theme name if possible.
   *
   * Theme names are case insensitive. The `href` used is all lower case.
   *
   * To use this method, the `<link rel="stylesheet">` _must_ use the default,
   * Bryntum-supplied CSS files where the `href` end with `<themeName>.css`, so that
   * it can be found in the document, and switched out for a new link with
   * the a modified `href`. The new `href` will use the same path, just
   * with the `themeName` portion substituted for the new name.
   *
   * If no `<link>` with that name pattern can be found, an error will be thrown.
   *
   * If you use this method, you  must ensure that the theme files are
   * all accessible on your server.
   *
   * Because this is an asynchronous operation, a `Promise` is returned.
   * The theme change event is passed to the success function. If the
   * theme was not changed, because the theme name passed is the current theme,
   * nothing is passed to the success function.
   *
   * The theme change event contains two properties:
   *
   *  - `prev` The previous Theme name.
   *  - `theme` The new Theme name.
   *
   * @param {String} newThemeName the name of the theme that should be applied
   * @privateparam {String} [defaultTheme] Optional, the name of the theme that should be used in case of fail
   * @returns {Promise} A promise who's success callback receives the theme change
   * event if the theme in fact changed. If the theme `href` could not be loaded,
   * the failure callback is called, passing the error event caught.
   * @async
   */
  static setTheme(newThemeName, defaultTheme) {
    newThemeName = newThemeName.toLowerCase();
    const { head } = document, oldThemeName = DH2.getThemeInfo(defaultTheme).name.toLowerCase();
    let oldThemeLinks = head.querySelectorAll("[data-bryntum-theme]:not([data-loading])"), loaded = 0;
    if (oldThemeName === newThemeName) {
      return immediatePromise;
    }
    DH2.removeEachSelector(head, "#bryntum-theme[data-loading],link[data-bryntum-theme][data-loading]");
    const themeEvent = {
      theme: newThemeName,
      prev: oldThemeName
    };
    function replaceTheme(oldThemeLink, resolve, reject) {
      const newThemeLink = DomHelper.createElement({
        tag: "link",
        rel: "stylesheet",
        dataset: {
          loading: true,
          bryntumTheme: true
        },
        href: oldThemeLink.href.replace(oldThemeName, newThemeName),
        nextSibling: oldThemeLink
      });
      newThemeLink.addEventListener("load", () => {
        delete newThemeLink.dataset.loading;
        themeInfo = null;
        if (++loaded === oldThemeLinks.length) {
          oldThemeLinks.forEach((link) => link.remove());
          GlobalEvents_default.trigger("theme", themeEvent);
          resolve(themeEvent);
        }
      });
      newThemeLink.addEventListener("error", (e) => {
        delete newThemeLink.dataset.loading;
        reject(e);
      });
    }
    if (oldThemeLinks.length) {
      return new Promise((resolve, reject) => {
        oldThemeLinks.forEach((oldThemeLink, i) => {
          replaceTheme(oldThemeLink, resolve, reject, i === oldThemeLinks.length - 1);
        });
      });
    } else {
      const oldThemeLink = head.querySelector("#bryntum-theme:not([data-loading])") || head.querySelector(`[href*="${oldThemeName}.css"]:not([data-loading])`);
      if (!(oldThemeLink == null ? void 0 : oldThemeLink.href.includes(`${oldThemeName}.css`))) {
        throw new Error(`Theme link for ${oldThemeName} not found`);
      }
      oldThemeLinks = [oldThemeLink];
      return new Promise((resolve, reject) => replaceTheme(oldThemeLink, resolve, reject));
    }
  }
  /**
   * A theme information object about the current theme.
   *
   * Currently, this has only one property:
   *
   *   - `name` The current theme name.
   * @property {Object}
   * @readonly
   */
  static get themeInfo() {
    return DomHelper.getThemeInfo();
  }
  /**
   * A theme information object about the current theme.
   *
   * Currently this has only one property:
   *
   *   - `name` The current theme name.
   * @param {String} defaultTheme the name of the theme used as backup value in case of fail
   * @param {HTMLElement} contextElement The element for which to find the theme. If using a
   * web component, the theme will be encapsulated in the web component's encapsulated style
   * so a context element is required. If no web components are in use, this may be omitted and
   * `document.body` will be used.
   * @returns {Object} info, currently it contains only one property - 'name'.
   * @private
   */
  static getThemeInfo(defaultTheme) {
    if (!themeInfo) {
      const testDiv = DH2.createElement({
        parent: document.body,
        className: "b-theme-info"
      }), themeData = DH2.getStyleValue(testDiv, "content", false, ":before");
      if (themeData) {
        try {
          themeInfo = JSON.parse(themeData.replace(/^["']|["']$|\\/g, ""));
        } catch (e) {
          themeInfo = null;
        }
      }
      themeInfo = themeInfo || (defaultTheme ? { name: defaultTheme } : null);
      testDiv.remove();
    }
    return themeInfo;
  }
  //endregion
  //region Transition
  static async transition({
    element: outerElement,
    selector = "[data-dom-transition]",
    duration,
    action,
    thisObj = this,
    addTransition = {},
    removeTransition = {}
  }) {
    const scrollers = /* @__PURE__ */ new Set(), beforeElements = Array.from(outerElement.querySelectorAll(selector)), beforeMap = new Map(beforeElements.map((element) => {
      let depth = 0, parent = element.parentElement;
      while (parent && parent !== outerElement) {
        depth++;
        parent = parent.parentElement;
      }
      element.$depth = depth;
      if (element.scrollHeight > element.offsetHeight && getComputedStyle(element).overflow === "auto") {
        element.$scrollTop = element.scrollTop;
        scrollers.add(element);
      }
      const { parentElement } = element, globalBounds = Rectangle.from(element, outerElement), localBounds = Rectangle.from(element, parentElement), style = getComputedStyle(parentElement), borderLeftWidth = parseFloat(style.borderLeftWidth);
      if (borderLeftWidth) {
        globalBounds.left -= borderLeftWidth;
        localBounds.left -= borderLeftWidth;
      }
      return [
        element.id,
        { element, globalBounds, localBounds, depth, parentElement }
      ];
    }));
    action.call(thisObj);
    const afterElements = Array.from(outerElement.querySelectorAll(selector)), afterMap = new Map(afterElements.map((element) => {
      const globalBounds = Rectangle.from(element, outerElement), localBounds = Rectangle.from(element, element.parentElement), style = globalThis.getComputedStyle(element.parentElement), borderLeftWidth = parseFloat(style.borderLeftWidth);
      if (borderLeftWidth) {
        globalBounds.left -= borderLeftWidth;
        localBounds.left -= borderLeftWidth;
      }
      return [
        element.id,
        { element, globalBounds, localBounds }
      ];
    })), styleProps = ["position", "top", "left", "width", "height", "padding", "margin", "zIndex", "minWidth", "minHeight", "opacity", "overflow"];
    for (const [id, before] of beforeMap) {
      const after = afterMap.get(id);
      if (after) {
        const { element } = after, { style, parentElement } = element, zIndex = parseInt(DH2.getStyleValue(element, "zIndex")), {
          globalBounds,
          localBounds,
          depth,
          parentElement: beforeParent
        } = before, parentChanged = beforeParent !== parentElement;
        ObjectHelper.copyProperties(element.$initial = { parentElement }, style, styleProps);
        let bounds;
        if (parentChanged) {
          after.bounds = after.globalBounds;
          bounds = globalBounds;
          outerElement.appendChild(element);
        } else {
          after.bounds = after.localBounds;
          bounds = localBounds;
          beforeParent.appendChild(element);
        }
        let overflow = "hidden";
        if (scrollers.has(element)) {
          element.$scrollPlaceholder = DH2.createElement({
            parent: element,
            style: {
              height: element.scrollHeight
            }
          });
          overflow = "auto";
        }
        const targetStyle = {
          position: "absolute",
          top: `${bounds.top}px`,
          left: `${bounds.left}px`,
          width: `${bounds.width}px`,
          height: `${bounds.height}px`,
          minWidth: 0,
          minHeight: 0,
          margin: 0,
          zIndex: depth + (zIndex || 0),
          overflow
        };
        if (element.dataset.domTransition !== "preserve-padding") {
          targetStyle.padding = 0;
        }
        Object.assign(style, targetStyle);
        after.processed = true;
      } else {
        const { element, localBounds: bounds, depth, parentElement } = before;
        element.$initial = { removed: true };
        Object.assign(element.style, {
          position: "absolute",
          top: `${bounds.top}px`,
          left: `${bounds.left}px`,
          width: `${bounds.width}px`,
          height: `${bounds.height}px`,
          minWidth: 0,
          minHeight: 0,
          padding: 0,
          margin: 0,
          zIndex: depth,
          overflow: "hidden"
          // Looks weird with content sticking out if height is transitioned
        });
        parentElement.appendChild(element);
        afterMap.set(id, { element, bounds, removed: true, processed: true });
        afterElements.push(element);
      }
    }
    for (const [, after] of afterMap) {
      if (!after.processed) {
        const { element } = after, { style, parentElement } = element, bounds = after.bounds = after.localBounds;
        element.classList.add("b-dom-transition-adding");
        ObjectHelper.copyProperties(element.$initial = { parentElement }, style, styleProps);
        Object.assign(style, {
          position: "absolute",
          top: addTransition.top ? 0 : `${bounds.top}px`,
          left: addTransition.left ? 0 : `${bounds.left}px`,
          width: addTransition.width ? 0 : `${bounds.width}px`,
          height: addTransition.height ? 0 : `${bounds.height}px`,
          opacity: addTransition.opacity ? 0 : null,
          zIndex: parentElement.$depth + 1,
          overflow: "hidden"
          // Looks weird with content sticking out if height is transitioned
        });
      }
    }
    for (const element of scrollers) {
      element.scrollTop = element.$scrollTop;
    }
    outerElement.classList.add("b-dom-transition");
    outerElement.firstElementChild.offsetWidth;
    for (const [, { element, bounds: afterBounds, removed }] of afterMap) {
      if (removed) {
        Object.assign(element.style, {
          top: removeTransition.top ? 0 : `${afterBounds.top}px`,
          left: removeTransition.left ? 0 : `${afterBounds.left}px`,
          width: removeTransition.width ? 0 : `${afterBounds.width}px`,
          height: removeTransition.height ? 0 : `${afterBounds.height}px`,
          opacity: removeTransition.opacity ? 0 : element.$initial.opacity
        });
      } else {
        Object.assign(element.style, {
          top: `${afterBounds.top}px`,
          left: `${afterBounds.left}px`,
          width: `${afterBounds.width}px`,
          height: `${afterBounds.height}px`,
          opacity: element.$initial.opacity
        });
      }
    }
    await AsyncHelper.sleep(duration);
    outerElement.classList.remove("b-dom-transition");
    for (const element of afterElements) {
      if (element.$initial) {
        if (element.$initial.removed) {
          element.remove();
        } else {
          ObjectHelper.copyProperties(element.style, element.$initial, styleProps);
          if (element.$scrollPlaceholder) {
            element.$scrollPlaceholder.remove();
            delete element.$scrollPlaceholder;
          }
          element.classList.remove("b-dom-transition-adding");
          element.$initial.parentElement.appendChild(element);
        }
      }
    }
    for (const element of scrollers) {
      element.scrollTop = element.$scrollTop;
      delete element.$scrollTop;
    }
  }
  //endregion
  static async loadScript(url) {
    return new Promise((resolve, reject) => {
      const script = document.createElement("script");
      script.src = url;
      script.onload = resolve;
      script.onerror = reject;
      document.head.appendChild(script);
    });
  }
  static isNamedColor(color) {
    return color && !/^(#|hsl|rgb|hwb)/.test(color);
  }
  //#region Salesforce hooks
  // Wrap NodeFilter to support salesforce
  static get NodeFilter() {
    return NodeFilter;
  }
  static addChild(parent, child, sibling) {
    parent.insertBefore(child, sibling);
  }
  static cloneStylesIntoShadowRoot(shadowRoot, removeExisting) {
    return new Promise((resolve, reject) => {
      if (removeExisting) {
        shadowRoot.querySelectorAll('style, link[rel="stylesheet"]').forEach((el) => el.remove());
      }
      const links = document.querySelectorAll('link[rel="stylesheet"]');
      let loadCount = 0;
      links.forEach((node) => {
        const clone = node.cloneNode();
        clone.addEventListener("load", () => {
          loadCount += 1;
          if (loadCount === links.length) {
            resolve();
          }
        });
        clone.addEventListener("error", (e) => {
          reject(clone.href);
        });
        shadowRoot.appendChild(clone);
      });
      document.querySelectorAll("style").forEach((node) => {
        shadowRoot.appendChild(node.cloneNode());
      });
    });
  }
  //#endregion
};
var DH2 = DomHelper;
var clearTouchTimer;
var clearTouchEvent = () => DH2.isTouchEvent = false;
var setTouchEvent = () => {
  DH2.isTouchEvent = true;
  clearTimeout(clearTouchTimer);
  clearTouchTimer = setTimeout(clearTouchEvent, 400);
};
doc.addEventListener("touchstart", setTouchEvent, true);
doc.addEventListener("touchend", setTouchEvent, true);
DH2.canonicalStyles = canonicalStyles;
DH2.supportsTemplate = "content" in doc.createElement("template");
DH2.elementPropKey = elementPropKey;
DH2.numberRe = numberRe;
if (!("children" in Node.prototype)) {
  const elementFilter = (node) => node.nodeType === node.ELEMENT_NODE;
  Object.defineProperty(Node.prototype, "children", {
    get: function() {
      return Array.prototype.filter.call(this.childNodes, elementFilter);
    }
  });
}
if (!Element.prototype.matches) {
  Element.prototype.matches = Element.prototype.matchesSelector || Element.prototype.mozMatchesSelector || Element.prototype.msMatchesSelector || Element.prototype.oMatchesSelector || Element.prototype.webkitMatchesSelector || function(s) {
    const matches = (this.document || this.ownerDocument).querySelectorAll(s);
    let i = matches.length;
    while (--i >= 0 && matches.item(i) !== this) {
    }
    return i > -1;
  };
}
if (win.Element && !Element.prototype.closest) {
  Node.prototype.closest = Element.prototype.closest = function(s) {
    let el = this;
    if (!doc.documentElement.contains(el))
      return null;
    do {
      if (el.matches(s))
        return el;
      el = el.parentElement || el.parentNode;
    } while (el !== null && el.nodeType === el.ELEMENT_NODE);
    return null;
  };
} else {
  Node.prototype.closest = function(selector) {
    var _a2;
    return (_a2 = this.parentNode) == null ? void 0 : _a2.closest(selector);
  };
}
(function(arr) {
  arr.forEach(function(item) {
    if (Object.prototype.hasOwnProperty.call(item, "remove")) {
      return;
    }
    Object.defineProperty(item, "remove", {
      configurable: true,
      enumerable: true,
      writable: true,
      value: function remove() {
        this.parentNode && this.parentNode.removeChild(this);
      }
    });
  });
})([Element.prototype, CharacterData.prototype, DocumentType.prototype]);
globalThis.addEventListener("resize", () => scrollBarWidth = null);
DomHelper._$name = "DomHelper";

// ../Core/lib/Core/GlobalEvents.js
var longpressMoveThreshold = 5;
var isFloatingWidget = (w) => w.floating;
var longPressCancelEvents = {
  touchend: 1,
  pointerup: 1
};
var ignoreModifierKeys = {
  Meta: 1,
  Control: 1,
  Alt: 1
};
var GlobalEvents = new class GlobalEventsHandler extends Base.mixin(Events_default) {
  suspendFocusEvents() {
    focusEventsSuspended = true;
  }
  resumeFocusEvents() {
    focusEventsSuspended = false;
  }
  setupFocusListenersOnce(rootElement, EventHelper2) {
    if (rootElement && !GlobalEvents.observedElements.has(rootElement)) {
      GlobalEvents.setupFocusListeners(rootElement, EventHelper2);
      GlobalEvents.observedElements.add(rootElement);
    }
  }
  // This is imported by EventHelper and that makes the call to set up the listeners
  // `detach` argument is required to not setup more listeners than we need to. In case of salesforce we include floatroot
  // inside the webcomponent element and thus don't need default listeners on document. In regular webcomponents demo we
  // don't need to do it, because with multiple components on one page that would force us to make more complex lookups.
  setupFocusListeners(element = document, EventHelper2, detach = false) {
    const listeners = {
      element,
      touchstart(touchstart) {
        if (!globaltouchStart && touchstart.changedTouches.length === 1) {
          globaltouchStart = touchstart.changedTouches[0];
          if (!BrowserHelper.isAndroid) {
            const onMoveOrPointerUp = ({
              clientX,
              clientY,
              type
            }) => {
              if (longPressCancelEvents[type] || Math.max(Math.abs(clientX - globaltouchStart.clientX), Math.abs(clientY - globaltouchStart.clientY)) > longpressMoveThreshold) {
                contextMenuTouchId = null;
                touchMoveRemover();
                clearTimeout(tapholdTimer);
              }
            }, touchMoveRemover = EventHelper2.on({
              element: document,
              touchmove: onMoveOrPointerUp,
              touchend: onMoveOrPointerUp,
              pointermove: onMoveOrPointerUp,
              pointerup: onMoveOrPointerUp,
              capture: true
            }), tapholdTimer = setTimeout(() => {
              contextMenuTouchId = globaltouchStart.identifier;
              touchMoveRemover();
              touchstart.target.dispatchEvent(new MouseEvent("contextmenu", EventHelper2.copyEvent({}, touchstart)));
            }, EventHelper2.longPressTime);
          }
        } else {
          globaltouchStart = null;
        }
      },
      // Just this one has to be passive: false so that we are allowed to preventDefault
      // if we are part of a contextmenu longpress emulation. Otherwise the gesture will
      // proceed to cause a mousedown event.
      touchend: {
        handler: (event) => {
          if (globaltouchStart) {
            if (event.changedTouches[0].identifier === contextMenuTouchId) {
              event.stopImmediatePropagation();
              if (event.cancelable !== false) {
                event.preventDefault();
              }
            } else if (event.changedTouches.length === 1 && event.changedTouches[0].identifier === globaltouchStart.identifier) {
              GlobalEvents.trigger("globaltap", { event });
            }
            globaltouchStart = null;
          }
        },
        passive: false
      },
      mousedown: {
        handler: (event) => {
          lastInteractionType = "mouse";
          if (!globaltouchStart) {
            GlobalEvents.trigger("globaltap", { event });
          }
          currentMouseDown = event;
          const hasModifierKey = event.ctrlKey || event.altKey || event.shiftKey || event.metaKey;
          if (!currentKeyDown && hasModifierKey) {
            currentKeyDown = new KeyboardEvent("keydown", {
              key: event.ctrlKey ? "Control" : event.shiftKey ? "Shift" : event.altKey ? "Alt" : "Meta",
              ctrlKey: event.ctrlKey,
              altKey: event.altKey,
              shiftKey: event.shiftKey,
              metaKey: event.metaKey
            });
          } else if (currentKeyDown && !hasModifierKey) {
            currentKeyDown = null;
          }
        },
        passive: false
      },
      mouseup() {
        currentMouseDown = null;
      },
      pointerdown: {
        passive: false,
        handler: (event) => {
          var _a2;
          currentPointerDown = event;
          DomHelper.usingKeyboard = false;
          (_a2 = element.classList) == null ? void 0 : _a2.remove("b-using-keyboard");
          if (element.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
            DomHelper.removeClsGlobally(element, "b-using-keyboard");
          }
        }
      },
      pointerup: {
        passive: false,
        handler: (event) => {
          if ((currentPointerDown == null ? void 0 : currentPointerDown.pointerId) === event.pointerId) {
            currentPointerDown = null;
          }
        }
      },
      keydown(ev) {
        lastInteractionType = "key";
        currentKeyDown = ev;
        if (!ignoreModifierKeys[ev.key]) {
          DomHelper.usingKeyboard = true;
          if (element.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
            for (const node of element.children) {
              if (node.matches(".b-outer")) {
                node.classList.add("b-using-keyboard");
              }
            }
          } else {
            element.classList.add("b-using-keyboard");
          }
        }
      },
      keypress() {
        lastInteractionType = "key";
      },
      keyup() {
        currentKeyDown = null;
      },
      focusin(focusin) {
        var _a2, _b, _c;
        const { Widget: Widget2 } = GlobalEvents;
        if (((_a2 = focusin.target) == null ? void 0 : _a2.shadowRoot) || ((_b = focusin.target) == null ? void 0 : _b._shadowRoot)) {
          return;
        }
        Widget2.resetFloatRootScroll();
        if (focusEventsSuspended) {
          return;
        }
        const fromElement = !focusin.relatedTarget ? null : focusin.relatedTarget instanceof HTMLElement ? focusin.relatedTarget : document.body, toElement = focusin.target || document.body, fromWidget = Widget2.fromElement(fromElement), toWidget = Widget2.fromElement(toElement), commonAncestor = DomHelper.getCommonAncestor(fromWidget, toWidget), backwards = !!(fromElement && toElement.compareDocumentPosition(fromElement) & 4), topVisibleModal = Widget2.query(isTopVisibleModal);
        let currentFocus = null;
        if (toElement && toElement !== document.body) {
          currentFocus = DomHelper.getActiveElement(toElement);
        } else {
          currentFocus = DomHelper.getActiveElement(document);
        }
        if (topVisibleModal && !topVisibleModal._isRevertingFocus) {
          if (!toWidget || !topVisibleModal.owns(toWidget) && !(topVisibleModal.element.compareDocumentPosition(toWidget.element) & 4 && toWidget.up(isFloatingWidget))) {
            return topVisibleModal.focus();
          }
        }
        let event = createWidgetEvent("focusout", fromElement, focusin.target, fromWidget, toWidget, backwards);
        for (let target = fromWidget, owner; target && target !== commonAncestor; target = owner) {
          owner = target.owner;
          if (!target.isDestroying && target.onFocusOut) {
            target.onFocusOut(event);
            if (focusin.target && currentFocus !== DomHelper.getActiveElement(focusin.target)) {
              return;
            }
          }
        }
        if (commonAncestor && focusin.target === commonAncestor.element) {
          if (!commonAncestor.isDestroying && DomHelper.getActiveElement(commonAncestor) === toElement && commonAncestor.focusElement && commonAncestor.focusElement !== commonAncestor.element) {
            if (!commonAncestor.element.contains(currentFocus) || commonAncestor.focusDescendant) {
              commonAncestor.setTimeout(() => {
                var _a3;
                return (_a3 = commonAncestor.focus) == null ? void 0 : _a3.call(commonAncestor);
              }, 0);
            }
          }
        } else {
          event = createWidgetEvent("focusin", toElement, fromElement, fromWidget, toWidget, backwards);
          for (let target = toWidget; target && target !== commonAncestor; target = target.owner) {
            if (!target.isDestroying) {
              (_c = target.onFocusIn) == null ? void 0 : _c.call(target, event);
            }
          }
        }
        const commonAncestorEl = DomHelper.getCommonAncestor((fromElement == null ? void 0 : fromElement.nodeType) === Element.ELEMENT_NODE ? fromElement : null, toElement) || toElement.parentNode;
        if (commonAncestorEl) {
          event = createWidgetEvent("focusmove", toElement, fromElement, fromWidget, toWidget, backwards, { bubbles: true });
          commonAncestorEl.dispatchEvent(event);
        }
      },
      focusout(focusout) {
        if (focusEventsSuspended) {
          return;
        }
        if (!focusout.relatedTarget || !GlobalEvents.Widget.fromElement(focusout.relatedTarget)) {
          const target = focusout.relatedTarget && focusout.relatedTarget instanceof HTMLElement ? focusout.relatedTarget : null;
          listeners.focusin({
            target,
            relatedTarget: focusout.target
          });
          currentKeyDown = currentMouseDown = null;
        }
      },
      // This will clear keydown and mousedown status on window blur
      blur: {
        element: globalThis,
        handler(event) {
          if (event.target === globalThis) {
            currentKeyDown = null;
            currentMouseDown = null;
          }
        }
      },
      capture: true,
      passive: true
    };
    detach && (detacher == null ? void 0 : detacher());
    detacher = this.detachEvents = EventHelper2.on(listeners);
  }
  get lastInteractionType() {
    return lastInteractionType;
  }
  get shiftKeyDown() {
    return currentKeyDown == null ? void 0 : currentKeyDown.shiftKey;
  }
  get ctrlKeyDown() {
    return (currentKeyDown == null ? void 0 : currentKeyDown.ctrlKey) || (currentKeyDown == null ? void 0 : currentKeyDown.metaKey);
  }
  get altKeyDown() {
    return currentKeyDown == null ? void 0 : currentKeyDown.altKey;
  }
  isKeyDown(key) {
    return !key ? Boolean(currentKeyDown) : (currentKeyDown == null ? void 0 : currentKeyDown.key) === key || currentKeyDown[(key == null ? void 0 : key.toLowerCase()) + "Key"] === true;
  }
  isMouseDown(button = 0) {
    return (currentMouseDown == null ? void 0 : currentMouseDown.button) === button;
  }
  get currentMouseDown() {
    return currentMouseDown;
  }
  get currentPointerDown() {
    return currentPointerDown;
  }
  get currentTouch() {
    return globaltouchStart;
  }
  get currentKeyDown() {
    return currentKeyDown;
  }
}();
var isTopVisibleModal = (w) => w.isVisible && w.isTopModal;
GlobalEvents.observedElements = /* @__PURE__ */ new Set();
var globaltouchStart;
var contextMenuTouchId;
var focusEventsSuspended = false;
var lastInteractionType;
var currentKeyDown;
var currentMouseDown;
var currentPointerDown;
var detacher;
function createWidgetEvent(eventName, target, relatedTarget, fromWidget, toWidget, backwards, options) {
  const result = new CustomEvent(eventName, options);
  Object.defineProperty(result, "_target", {
    get() {
      return target;
    }
  });
  Object.defineProperty(result, "relatedTarget", {
    get() {
      return relatedTarget;
    }
  });
  result.fromWidget = fromWidget;
  result.toWidget = toWidget;
  result.backwards = backwards;
  return result;
}
var GlobalEvents_default = GlobalEvents;

// ../Core/lib/Core/mixin/InstancePlugin.js
function getDescriptor(me, fnName) {
  const property = ObjectHelper.getPropertyDescriptor(me, fnName);
  return property && (property.get || property.set) ? property : null;
}
var InstancePlugin = class extends Base.mixin(Events_default, Localizable_default) {
  //region Config
  static get configurable() {
    return {
      clientListeners: null,
      /**
       * The plugin/feature `disabled` state.
       *
       * For a feature that is **off** by default that you want to enable later during runtime,
       * configure it with `disabled : true`.
       * ```javascript
       * const grid = new Grid({
       *      features : {
       *          featureName : {
       *              disabled : true // on and disabled, can be enabled later
       *          }
       *      }
       * });
       *
       * // enable the feature
       * grid.features.featureName.disabled = false;
       * ```
       *
       * If the feature is **off** by default, and you want to include and enable the feature, configure it as `true`:
       * ```javascript
       * const grid = new Grid({
       *      features : {
       *          featureName : true // on and enabled, can be disabled later
       *      }
       * });
       *
       * // disable the feature
       * grid.features.featureName.disabled = true;
       * ```
       *
       * If the feature is **on** by default, but you want to turn it **off**, configure it as `false`:
       * ```javascript
       * const grid = new Grid({
       *      features : {
       *          featureName : false // turned off, not included at all
       *      }
       * });
       * ```
       *
       * If the feature is **enabled** by default and you have no need of reconfiguring it,
       * you can omit the feature configuration.
       *
       * @prp {Boolean}
       * @default
       * @category Common
       */
      disabled: false,
      /**
       * The Widget which was passed into the constructor,
       * which is the Widget we are providing extra services for.
       * @member {Core.widget.Widget} client
       * @readonly
       * @category Misc
       * @advanced
       */
      /**
       * The widget which this plugin is to attach to.
       * @config {Core.widget.Widget}
       * @category Misc
       * @advanced
       */
      client: null,
      /**
       * @hideconfigs bubbleEvents, callOnFunctions
       */
      /**
       * @hidefunctions downloadTestCase, destroy
       */
      /**
       * @hideproperties isDestroyed
       */
      /**
       * @hideevents destroy, beforeDestroy
       */
      // The plugins can define their own keyMap which will then be merged with their client's keyMap.
      keyMap: null
    };
  }
  //endregion
  updateClient(client) {
    if (!this.owner) {
      this.owner = client;
    }
  }
  /**
   * This will merge a feature's (subclass of InstancePlugin) keyMap with it's client's keyMap.
   * @private
   */
  updateKeyMap(keyMap) {
    const { client } = this;
    client.keyMap = client.mergeKeyMaps(client.keyMap, keyMap, StringHelper.uncapitalize(this.constructor.$name));
  }
  //region Init
  /**
   * Call from another instance to add plugins to it.
   * @example
   * InstancePlugin.initPlugins(this, Search, Stripe);
   * @param plugInto Instance to mix into (usually this)
   * @param plugins Classes to plug in
   * @internal
   */
  static initPlugins(plugInto, ...plugins) {
    const property = plugInto.plugins || (plugInto.plugins = {});
    for (const PluginClass of plugins) {
      property[PluginClass.$$name] = new PluginClass(plugInto);
    }
  }
  /**
   * Simple wrapper for {@link #property-disabled} to make optional chaining simple:
   *
   * ```javascript
   * grid.features.myFeature?.enabled // returns true when feature exists and is enabled
   * ```
   * @returns {Boolean}
   * @internal
   */
  get enabled() {
    return !this.disabled;
  }
  // We can act as an owner of a widget, so must be able to participate in focus reversion
  getFocusRevertTarget() {
    var _a2;
    return (_a2 = this.client) == null ? void 0 : _a2.getFocusRevertTarget();
  }
  construct(...args) {
    const me = this;
    let [plugInto, config] = args, listeners;
    if (args.length === 1) {
      if (ObjectHelper.isObject(plugInto)) {
        config = plugInto;
        plugInto = config.client;
      }
    } else {
      config = ObjectHelper.assign({}, config);
      delete config.client;
    }
    me.client = plugInto;
    super.construct(config);
    me.applyPluginConfig(plugInto);
    listeners = me.clientListeners;
    if (listeners) {
      listeners = ObjectHelper.assign({}, listeners);
      listeners.thisObj = me;
      plugInto.ion(listeners);
    }
  }
  /**
   * Applies config as found in plugInto.pluginConfig, or published all if no config found.
   * @private
   * @param plugInto Target instance to plug into
   */
  applyPluginConfig(plugInto) {
    const me = this, config = me.pluginConfig || me.constructor.pluginConfig;
    if (config) {
      const { assign: assign2, chain, after, before, override } = config;
      assign2 && me.applyAssign(plugInto, assign2);
      (chain || after) && me.applyChain(plugInto, chain || after);
      before && me.applyChain(plugInto, before, false);
      override && me.applyOverride(plugInto, override);
    }
  }
  /**
   * Applies assigning for specified functions.
   * @private
   * @param plugInto
   * @param fnNames
   */
  applyAssign(plugInto, fnNames) {
    fnNames.forEach((fnName) => this.assign(plugInto, fnName));
  }
  /**
   * Applies chaining for specified functions.
   * @private
   * @param plugInto
   * @param functions
   * @param after
   */
  applyChain(plugInto, functions, after = true) {
    if (Array.isArray(functions)) {
      for (const fnName of functions) {
        this.chain(plugInto, fnName, fnName, after);
      }
    } else {
      for (const intoName in functions) {
        this.chain(plugInto, intoName, functions[intoName], after);
      }
    }
  }
  /**
   * Applies override for specified functions.
   * @private
   * @param plugInto
   * @param fnNames
   */
  applyOverride(plugInto, fnNames) {
    const me = this;
    if (!me.overridden) {
      me.overridden = {};
    }
    fnNames.forEach((fnName) => {
      if (!me[fnName]) {
        throw new Error(`Trying to chain fn ${plugInto.$$name}#${fnName}, but plugin fn ${me.$$name}#${fnName} does not exist`);
      }
      if (typeof plugInto[fnName] === "function") {
        me.overridden[fnName] = plugInto[fnName].bind(plugInto);
      }
      plugInto[fnName] = me[fnName].bind(me);
    });
  }
  /**
   * Assigns specified functions.
   * @private
   * @param plugInto
   * @param fnName
   */
  assign(plugInto, fnName) {
    const me = this, property = getDescriptor(me, fnName);
    if (property) {
      Object.defineProperty(plugInto, fnName, {
        configurable: true,
        enumerable: true,
        get: property.get && property.get.bind(me),
        set: property.set && property.set.bind(me)
      });
    } else {
      plugInto[fnName] = me[fnName].bind(me);
    }
  }
  //endregion
  //region Chaining
  /**
   * Chains functions. When the function is called on the target class all functions in the chain will be called in
   * the order they where added.
   * @private
   * @param plugInto
   * @param intoName
   * @param hookName
   * @param after
   */
  chain(plugInto, intoName, hookName, after = true) {
    let prio = 0;
    if (typeof intoName === "object") {
      intoName = intoName.fn;
    }
    if (typeof hookName === "object") {
      prio = hookName.prio || 0;
      hookName = hookName.fn;
    }
    const me = this, chains = plugInto.pluginFunctionChain || (plugInto.pluginFunctionChain = {}), hookFn = me[hookName] && me[hookName].bind(me), functionChainRunner = me.functionChainRunner;
    if (!hookFn) {
      throw new Error(`Trying to chain fn ${plugInto.$$name}#${hookName}, but plugin fn ${me.$$name}#${hookName} does not exist`);
    }
    if (!chains[intoName]) {
      let intoFn = plugInto[intoName];
      if (intoFn) {
        intoFn = intoFn.bind(plugInto);
        intoFn.$this = plugInto;
        intoFn.$prio = 0;
      }
      chains[intoName] = intoFn ? [intoFn] : [];
      plugInto[intoName] = (...params) => functionChainRunner(chains[intoName], params);
    }
    hookFn.$this = me;
    hookFn.$prio = prio;
    chains[intoName][after ? "push" : "unshift"](hookFn);
    chains[intoName].$sorted = false;
  }
  /**
   * Used to run multiple plugged in functions with the same name, see chain above. Returning false from a
   * function will abort chain.
   * @private
   * @param {Array} chain
   * @param {Array} params
   * @returns {*} value returned from last function in chain (or false if any returns false)
   */
  functionChainRunner(chain, params) {
    let fn2, i, returnValue;
    if (!chain.$sorted) {
      chain.sort((a, b) => b.$prio - a.$prio);
      chain.$sorted = true;
    }
    for (i = 0; i < chain.length; i++) {
      fn2 = chain[i];
      if (!fn2.$this.isDestroyed) {
        returnValue = fn2(...params);
        if (returnValue === false) {
          break;
        }
      }
    }
    return returnValue;
  }
  //endregion
  /**
   * Called when disabling/enabling the plugin/feature, not intended to be called directly. To enable or disable a
   * plugin/feature, see {@link #property-disabled}.
   *
   * By default removes the cls of the plugin from its client. Override in subclasses to take any other actions necessary.
   * @category Misc
   * @advanced
   */
  doDisable(disable) {
    var _a2, _b, _c, _d;
    const me = this, { constructor } = me, cls2 = "featureClass" in constructor ? constructor.featureClass : `b-${constructor.$$name.toLowerCase()}`, key = StringHelper.uncapitalize(constructor.$$name);
    if (cls2) {
      (_b = (_a2 = me.client) == null ? void 0 : _a2._element) == null ? void 0 : _b.classList[disable ? "remove" : "add"](cls2);
    }
    if (!me.isConfiguring) {
      if (disable) {
        me.trigger("disable");
      } else {
        me.trigger("enable");
      }
      (_d = (_c = me.client).syncSplits) == null ? void 0 : _d.call(_c, (other) => {
        const otherFeature = other.features[key];
        if (otherFeature) {
          otherFeature.disabled = disable;
        }
      });
    }
  }
  updateDisabled(disabled) {
    this.doDisable(disabled);
  }
  throwOverrideIsMissing(data) {
    throw new Error(`Trying to override fn ${data.plugIntoName}#${data.fnName}, but plugin fn ${data.pluginName}#${data.fnName} does not exist`);
  }
  // Convenience method to read the rootElement from the owner widget
  get rootElement() {
    return this.client.rootElement;
  }
};
__publicField(InstancePlugin, "$name", "InstancePlugin");
InstancePlugin._$name = "InstancePlugin";

// ../Core/lib/Core/mixin/Pluggable.js
var Pluggable_default = (Target) => class Pluggable extends (Target || Base) {
  static get $name() {
    return "Pluggable";
  }
  /**
   * Specify plugins (an array of classes) in config
   * @config {Function[]} plugins
   * @category Misc
   * @advanced
   */
  /**
   * Map of applied plugins
   * @property {Object<String,Core.mixin.InstancePlugin>}
   * @readonly
   * @category Misc
   * @advanced
   */
  get plugins() {
    if (!this._plugins) {
      this._plugins = {};
    }
    return this._plugins;
  }
  set plugins(plugins) {
    if (plugins) {
      if (!Array.isArray(plugins))
        plugins = [plugins];
      InstancePlugin.initPlugins(this, ...plugins);
    }
    this.initPlugins();
  }
  /**
   * Template method which may be implemented in subclasses to initialize any plugins.
   * This method is empty in the `Pluggable` base class.
   * @internal
   */
  initPlugins() {
  }
  /**
   * Adds plugins to an instance.
   * @param {Function[]} plugins The plugins to add
   * @category Misc
   * @advanced
   */
  addPlugins(...plugins) {
    InstancePlugin.initPlugins(this, ...plugins);
  }
  /**
   * Checks if instance has plugin.
   * @param {String|Function} pluginClassOrName Plugin or name to check for
   * @returns {Boolean}
   * @category Misc
   * @advanced
   */
  hasPlugin(pluginClassOrName) {
    return this.getPlugin(pluginClassOrName) != null;
  }
  /**
   * Get a plugin instance.
   * @param {String|Function} pluginClassOrName
   * @returns {Core.mixin.InstancePlugin}
   * @category Misc
   * @advanced
   */
  getPlugin(pluginClassOrName) {
    var _a2;
    if (typeof pluginClassOrName === "function") {
      pluginClassOrName = pluginClassOrName.$$name;
    }
    return (_a2 = this.plugins) == null ? void 0 : _a2[pluginClassOrName];
  }
  // This does not need a className on Widgets.
  // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
  // to the Widget it's mixed in to should implement thus.
  get widgetClass() {
  }
};

// ../Core/lib/Core/mixin/Delayable.js
var { defineProperty: defineProperty3 } = Reflect;
var performance2;
if (BrowserHelper.isBrowserEnv) {
  performance2 = globalThis.performance;
} else {
  performance2 = {
    now() {
      return (/* @__PURE__ */ new Date()).getTime();
    }
  };
}
var globalDelays = null;
if (VersionHelper.isTestEnv) {
  const bryntum = globalThis.bryntum || (globalThis.bryntum = {});
  globalDelays = bryntum.globalDelays = {
    timeouts: /* @__PURE__ */ new Map(),
    intervals: /* @__PURE__ */ new Map(),
    animationFrames: /* @__PURE__ */ new Map(),
    idleCallbacks: /* @__PURE__ */ new Map(),
    isEmpty(includeIntervals = false) {
      return globalDelays.timeouts.size + globalDelays.animationFrames.size + globalDelays.idleCallbacks.size + (includeIntervals ? globalDelays.intervals.size : 0) === 0;
    },
    /**
     * Returns filtered delays array
     * @param {Object} options
     * @param {String[]} [options.ignoreTimeouts] array of delays names to ignore
     * @param {Number} [options.maxDelay] maximum delay in milliseconds. Timeouts with bigger delay will be filtered out
     * @param {Boolean} [options.includeIntervals] include intervals
     * @returns {Object[]} array of filtered delays
     * @internal
     */
    getFiltered({ ignoreTimeouts = [], maxDelay = 5e3, includeIntervals = false }) {
      const result = [], scopes = ["timeouts", "animationFrames", "idleCallbacks"];
      if (includeIntervals) {
        scopes.push("intervals");
      }
      for (const scope of scopes) {
        const map2 = globalDelays[scope];
        for (const [, entry] of map2.entries()) {
          if (!ignoreTimeouts.includes(entry.name) && (!Number.isInteger(entry.delay) || entry.delay < maxDelay)) {
            result.push(entry);
          }
        }
      }
      return result;
    }
  };
}
var makeInvoker = (me, fn2, wrapFn, options) => {
  const named = typeof fn2 === "string", appendArgs = (options == null ? void 0 : options.appendArgs) || [], invoker = () => {
    wrapFn.timerId = null;
    wrapFn.lastCallTime = performance2.now();
    const args = wrapFn.args;
    wrapFn.args = null;
    if (named) {
      me[fn2](...args, ...appendArgs);
    } else {
      fn2.call(me, ...args, ...appendArgs);
    }
    wrapFn.called = true;
    ++wrapFn.calls;
  };
  if (options) {
    me = options.thisObj || me;
  }
  wrapFn.lastCallTime = -9e9;
  wrapFn.calls = 0;
  wrapFn.invoker = invoker;
  invoker.wrapFn = wrapFn;
  return invoker;
};
var decorateWrapFn = (me, wrapFn, cancelFn = "clearTimeout") => {
  wrapFn.cancel = () => {
    if (wrapFn.isPending) {
      me[cancelFn](wrapFn.timerId);
      wrapFn.args = wrapFn.timerId = null;
    }
  };
  wrapFn.flush = () => {
    if (wrapFn.isPending) {
      me[cancelFn](wrapFn.timerId);
      wrapFn.timerId = null;
      wrapFn.invoker();
    }
  };
  wrapFn.now = (...args) => {
    wrapFn.cancel();
    wrapFn.args = args;
    wrapFn.invoker();
  };
  wrapFn.resume = (all) => {
    const n = wrapFn.suspended;
    wrapFn.suspended = all || n < 1 ? 0 : n - 1;
  };
  wrapFn.suspend = () => {
    ++wrapFn.suspended;
  };
  wrapFn.immediate = false;
  wrapFn.suspended = 0;
  wrapFn.timerId = null;
  defineProperty3(wrapFn, "isPending", {
    get() {
      return wrapFn.timerId !== null;
    }
  });
  return wrapFn;
};
var Delayable_default = (Target) => class Delayable extends (Target || Base) {
  static get $name() {
    return "Delayable";
  }
  static get declarable() {
    return [
      /**
       * This class property returns an object that specifies methods to wrap with configurable timer behaviors.
       *
       * It is used like so:
       * ```javascript
       *  class Foo extends Base.mixin(Delayable) {
       *      static get delayable() {
       *          return {
       *              expensiveMethod : 500
       *          };
       *      }
       *
       *      expensiveMethod() {
       *          this.things();
       *          this.moreThings();
       *          this.evenMoreThings();
       *      }
       *  }
       * ```
       * With the above in place, consider:
       * ```javascript
       *  let instance = new Foo();
       *
       *  instance.expensiveMethod();
       * ```
       * Instead of the above code immediately calling the `expensiveMethod()`, it will start a timer that will
       * invoke the method 500ms later. Because `expensiveMethod()` is an instance method, each instance of `Foo`
       * will have its own timer.
       *
       * NOTE: Only instance methods are currently supported (i.e., only non-`static` methods).
       *
       * #### Options
       * The value of each key configures how the method will be scheduled. If the value is a number, it is
       * promoted to a config object of `type='buffer'` as in the following:
       * ```javascript
       *  class Foo extends Base.mixin(Delayable) {
       *      static get delayable() {
       *          return {
       *              expensiveMethod : {
       *                  type  : 'buffer',
       *                  delay : 500
       *              }
       *          };
       *      }
       *  }
       * ```
       * The `type` property of the config object must be one of three values. Other options can be provided
       * depending on the `type`:
       *
       *  - `buffer`<br>
       *    Other options:
       *     - `delay` (Number) : The number of milliseconds to wait before calling the underlying method. A
       *       value of 0 is equivalent to setting `immediate: true`.
       *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the buffer).
       *  - `raf` (short for "request animation frame")<br>
       *  - `idle` (short for "request idle callback") __Not available on Safari__ <br>
       *    Other options:
       *     - `cancelOutstanding` (Boolean) : Set to `true` to cancel any pending animation frame requests and
       *       schedule a new one on each call.
       *     - `immediate` (Boolean) : Set to `true` to call immediately.
       *  - `throttle`<br>
       *    Other options:
       *     - `delay` (Number) : The number of milliseconds to wait after each execution before another
       *       execution takes place. A value of 0 is equivalent to setting `immediate: true`.
       *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the throttle).
       *
       * While `immediate: true` can be specified at the class level, it is more typical to set it on the
       * instance's method as described below.
       *
       * #### Delayable Method API
       * Delayable methods have a consistent API to manage their scheduling. This API is added to the methods
       * themselves.
       *
       * For example:
       * ```javascript
       *  let instance = new Foo();
       *
       *  instance.expensiveMethod();         // schedule a call in 500ms
       *  instance.expensiveMethod.isPending; // true
       *  instance.expensiveMethod.cancel();
       *  instance.expensiveMethod.flush();
       *  instance.expensiveMethod.now();
       *
       *  instance.expensiveMethod.delay = 10;
       *  instance.expensiveMethod();         // schedule a call in 10ms
       * ```
       *
       * ##### `isPending` (Boolean, readonly)
       * This boolean property will be `true` if a call has been scheduled, and false otherwise.
       *
       * ##### `cancel()`
       * Cancels a pending call if one has been scheduled. Otherwise this method does nothing.
       *
       * ##### `flush()`
       * Cancels the timer and causes the pending call to execute immediately. If there is no pending call, this
       * method does nothing.
       *
       * ##### `now()`
       * Cancels the timer (if one is pending) and executes the method immediately. If there is no pending call,
       * this method will still call the underlying method.
       *
       * @static
       * @member {Object<String,'raf'|Number|DelayableConfig>} delayable
       * @internal
       */
      "delayable"
    ];
  }
  doDestroy() {
    const me = this;
    super.doDestroy();
    if (me.timeoutIds) {
      me.timeoutIds.forEach((fn2, id) => {
        if (typeof fn2 === "function") {
          fn2();
        }
        clearTimeout(id);
        globalDelays == null ? void 0 : globalDelays.timeouts.delete(id);
      });
      me.timeoutIds = null;
    }
    if (me.timeoutMap) {
      me.timeoutMap.forEach((name, id) => clearTimeout(id));
      me.timeoutMap = null;
    }
    if (me.intervalIds) {
      me.intervalIds.forEach((id) => {
        clearInterval(id);
        globalDelays == null ? void 0 : globalDelays.intervals.delete(id);
      });
      me.intervalIds = null;
    }
    if (me.animationFrameIds) {
      me.animationFrameIds.forEach((id) => {
        cancelAnimationFrame(id);
        globalDelays == null ? void 0 : globalDelays.animationFrames.delete(id);
      });
      me.animationFrameIds = null;
    }
    if (me.idleCallbackIds) {
      me.idleCallbackIds.forEach((id) => {
        cancelIdleCallback(id);
        globalDelays == null ? void 0 : globalDelays.idleCallbacks.delete(id);
      });
      me.idleCallbackIds = null;
    }
  }
  /**
   * Check if a named timeout is active
   * @param name
   * @internal
   */
  hasTimeout(name) {
    var _a2;
    return Boolean((_a2 = this.timeoutMap) == null ? void 0 : _a2.has(name));
  }
  /**
   * Same as native setTimeout, but will be cleared automatically on destroy. If a propertyName is supplied it will
   * be used to store the timeout id.
   * @param {Object} timeoutSpec An object containing the details about that function, and the time delay.
   * @param {Function|String} timeoutSpec.fn The function to call, or name of function in this object to call. Used as the `name` parameter if a string.
   * @param {Number} timeoutSpec.delay The milliseconds to delay the call by.
   * @param {Object[]} timeoutSpec.args The arguments to pass.
   * @param {String} [timeoutSpec.name] The name under which to register the timer. Defaults to `fn.name`.
   * @param {Boolean} [timeoutSpec.runOnDestroy] Pass `true` if this function should be executed if the Delayable instance is destroyed while function is scheduled.
   * @param {Boolean} [timeoutSpec.cancelOutstanding] Pass `true` to cancel any outstanding invocation of the passed function.
   * @returns {Number}
   * @internal
   */
  setTimeout({ fn: fn2, delay: delay3, name, runOnDestroy, cancelOutstanding, args }) {
    if (arguments.length > 1 || typeof arguments[0] === "function") {
      [fn2, delay3, name, runOnDestroy] = arguments;
    }
    if (typeof fn2 === "string") {
      name = fn2;
    } else if (!name) {
      name = fn2.name || fn2;
    }
    if (cancelOutstanding) {
      this.clearTimeout(name);
    }
    const me = this, timeoutIds = me.timeoutIds || (me.timeoutIds = /* @__PURE__ */ new Map()), timeoutMap = me.timeoutMap || (me.timeoutMap = /* @__PURE__ */ new Map()), timeoutId = setTimeout(() => {
      if (typeof fn2 === "string") {
        fn2 = me[name];
      }
      timeoutIds == null ? void 0 : timeoutIds.delete(timeoutId);
      timeoutMap == null ? void 0 : timeoutMap.delete(name);
      globalDelays == null ? void 0 : globalDelays.timeouts.delete(timeoutId);
      fn2.apply(me, args);
    }, delay3);
    timeoutIds.set(timeoutId, runOnDestroy ? fn2 : true);
    globalDelays == null ? void 0 : globalDelays.timeouts.set(timeoutId, {
      fn: fn2,
      delay: delay3,
      name
      /*, stack : new Error().stack*/
    });
    if (name) {
      timeoutMap.set(name, timeoutId);
    }
    return timeoutId;
  }
  /**
   * clearTimeout wrapper, either call with timeout id as normal clearTimeout or with timeout name (if you specified
   * a name to setTimeout())
   * property to null.
   * @param {Number|String} idOrName timeout id or name
   * @internal
   */
  clearTimeout(idOrName) {
    var _a2;
    let id = idOrName;
    if (typeof id === "string") {
      if (this.timeoutMap) {
        id = this.timeoutMap.get(idOrName);
        this.timeoutMap.delete(idOrName);
      } else {
        return;
      }
    }
    clearTimeout(id);
    (_a2 = this.timeoutIds) == null ? void 0 : _a2.delete(id);
    globalDelays == null ? void 0 : globalDelays.timeouts.delete(id);
  }
  /**
   * clearInterval wrapper
   * @param {Number} id
   * @internal
   */
  clearInterval(id) {
    var _a2;
    clearInterval(id);
    (_a2 = this.intervalIds) == null ? void 0 : _a2.delete(id);
    globalDelays == null ? void 0 : globalDelays.intervals.delete(id);
  }
  /**
   * Same as native setInterval, but will be cleared automatically on destroy
   * @param {Function} fn callback method
   * @param {Number} delay delay in milliseconds
   * @param {String} name delay name for debugging
   * @returns {Number}
   * @internal
   */
  setInterval(fn2, delay3, name) {
    const intervalId = setInterval(fn2, delay3);
    (this.intervalIds || (this.intervalIds = /* @__PURE__ */ new Set())).add(intervalId);
    globalDelays == null ? void 0 : globalDelays.intervals.set(intervalId, { fn: fn2, delay: delay3, name });
    return intervalId;
  }
  /**
   * Relays to native requestAnimationFrame and adds to tracking to have call automatically canceled on destroy.
   * @param {Function} fn
   * @param {Object[]} [extraArgs] The argument list to append to those passed to the function.
   * @param {Object} [thisObj] `this` reference for the function.
   * @returns {Number}
   * @internal
   */
  requestAnimationFrame(fn2, extraArgs = [], thisObj = this) {
    const animationFrameIds = this.animationFrameIds || (this.animationFrameIds = /* @__PURE__ */ new Set()), frameId = requestAnimationFrame(() => {
      globalDelays == null ? void 0 : globalDelays.animationFrames.delete(frameId);
      animationFrameIds.delete(frameId) && fn2.apply(thisObj, extraArgs);
    });
    animationFrameIds.add(frameId);
    globalDelays == null ? void 0 : globalDelays.animationFrames.set(frameId, { fn: fn2, extraArgs, thisObj });
    return frameId;
  }
  /**
   * Relays to native requestIdleCallback and adds to tracking to have call automatically canceled on destroy.
   * @param {Function} fn
   * @param {Object[]} [extraArgs] The argument list to append to those passed to the function.
   * @param {Object} [thisObj] `this` reference for the function.
   * @returns {Number}
   * @internal
   */
  requestIdleCallback(fn2, extraArgs = [], thisObj = this) {
    const idleCallbackIds = this.idleCallbackIds || (this.idleCallbackIds = /* @__PURE__ */ new Set()), frameId = requestIdleCallback(() => {
      globalDelays == null ? void 0 : globalDelays.idleCallbacks.delete(frameId);
      idleCallbackIds.delete(frameId) && fn2.apply(thisObj, extraArgs);
    });
    idleCallbackIds.add(frameId);
    globalDelays == null ? void 0 : globalDelays.idleCallbacks.set(frameId, { fn: fn2, extraArgs, thisObj });
    return frameId;
  }
  /**
   * Creates a function which will execute once, on the next animation frame. However many time it is
   * called in one event run, it will only be scheduled to run once.
   * @param {Function|String} fn The function to call, or name of function in this object to call.
   * @param {Object[]} [args] The argument list to append to those passed to the function.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Boolean} [cancelOutstanding] Cancel any outstanding queued invocation upon call.
   * @internal
   */
  createOnFrame(fn2, args = [], thisObj = this, cancelOutstanding) {
    let rafId;
    const result = (...callArgs) => {
      if (rafId != null && cancelOutstanding) {
        this.cancelAnimationFrame(rafId);
        rafId = null;
      }
      if (rafId == null) {
        rafId = this.requestAnimationFrame(() => {
          if (typeof fn2 === "string") {
            fn2 = thisObj[fn2];
          }
          rafId = null;
          callArgs.push(...args);
          fn2.apply(thisObj, callArgs);
        });
      }
    };
    result.cancel = () => this.cancelAnimationFrame(rafId);
    return result;
  }
  /**
   * Relays to native cancelAnimationFrame and removes from tracking.
   * @param {Number} handle
   * @internal
   */
  cancelAnimationFrame(handle) {
    var _a2;
    cancelAnimationFrame(handle);
    (_a2 = this.animationFrameIds) == null ? void 0 : _a2.delete(handle);
    globalDelays == null ? void 0 : globalDelays.animationFrames.delete(handle);
  }
  /**
   * Relays to native cancelIdleCallback and removes from tracking.
   * @param {Number} handle
   * @internal
   */
  cancelIdleCallback(handle) {
    var _a2;
    cancelIdleCallback(handle);
    (_a2 = this.idleCallbackIds) == null ? void 0 : _a2.delete(handle);
    globalDelays == null ? void 0 : globalDelays.idleCallbacks.delete(handle);
  }
  async nextAnimationFrame() {
    return new Promise((resolve) => this.requestAnimationFrame(resolve));
  }
  /**
   * Wraps a function with another function that delays it specified amount of time, repeated calls to the wrapper
   * resets delay.
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Object|Number} options The delay in milliseconds or an options object.
   * @param {Number} options.delay The delay in milliseconds.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Object} [options.thisObj] The `this` reference for the function.
   * @returns {Function} Wrapped function to call.
   * @internal
   */
  buffer(fn2, options) {
    let delay3 = options;
    if (options && typeof options !== "number") {
      delay3 = options.delay;
    } else {
      options = null;
    }
    const bufferWrapFn = (...params) => {
      if (bufferWrapFn.suspended) {
        return;
      }
      const { delay: delay4 } = bufferWrapFn;
      bufferWrapFn.cancel();
      bufferWrapFn.called = false;
      bufferWrapFn.args = params;
      if (bufferWrapFn.immediate || !delay4) {
        invoker();
      } else {
        bufferWrapFn.timerId = this.setTimeout(invoker, delay4);
      }
    }, invoker = makeInvoker(this, fn2, bufferWrapFn, options);
    bufferWrapFn.delay = delay3;
    return decorateWrapFn(this, bufferWrapFn);
  }
  /**
   * Returns a function that when called will schedule a call to `fn` on the next animation frame.
   *
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Boolean|Object} [options] An options object or the `cancelOutstanding` boolean property of it.
   * @param {Boolean} [options.cancelOutstanding] Pass `true` to cancel any pending animation frame requests and
   * schedule a new one on each call to the returned function.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Object} [options.thisObj] The `this` reference for the function.
   * @returns {Function}
   * @internal
   */
  raf(fn2, options) {
    let cancelOutstanding = options;
    if (options && typeof options !== "boolean") {
      cancelOutstanding = options.cancelOutstanding;
    } else {
      options = null;
    }
    const rafWrapFn = (...params) => {
      if (rafWrapFn.suspended) {
        return;
      }
      if (rafWrapFn.cancelOutstanding) {
        rafWrapFn.cancel();
      }
      rafWrapFn.called = false;
      rafWrapFn.args = params;
      if (rafWrapFn.immediate) {
        invoker();
      } else if (!rafWrapFn.isPending) {
        rafWrapFn.timerId = this.requestAnimationFrame(invoker);
      }
    }, invoker = makeInvoker(this, fn2, rafWrapFn, options);
    rafWrapFn.cancelOutstanding = cancelOutstanding;
    return decorateWrapFn(this, rafWrapFn, "cancelAnimationFrame");
  }
  idle(fn2, options) {
    let cancelOutstanding = options;
    if (options && typeof options !== "boolean") {
      cancelOutstanding = options.cancelOutstanding;
    } else {
      options = null;
    }
    const idleWrapFn = (...params) => {
      if (idleWrapFn.suspended) {
        return;
      }
      if (idleWrapFn.cancelOutstanding) {
        idleWrapFn.cancel();
      }
      idleWrapFn.called = false;
      idleWrapFn.args = params;
      if (idleWrapFn.immediate) {
        invoker();
      } else if (!idleWrapFn.isPending) {
        idleWrapFn.timerId = this.requestIdleCallback(invoker);
      }
    }, invoker = makeInvoker(this, fn2, idleWrapFn, options);
    idleWrapFn.cancelOutstanding = cancelOutstanding;
    this.setTimeout(() => this.idleCallbackIds.delete(idleWrapFn.timerId) && idleWrapFn.now(), 100);
    return decorateWrapFn(this, idleWrapFn, "cancelIdleCallback");
  }
  /**
   * Create a "debounced" function which will call on the "leading edge" of a timer period.
   * When first invoked will call immediately, but invocations after that inside its buffer
   * period will be rejected, and *one* invocation will be made after the buffer period has expired.
   *
   * This is useful for responding immediately to a first mousemove, but from then on, only
   * calling the action function on a regular timer while the mouse continues to move.
   *
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Number|Object} options The milliseconds to wait after each execution before another execution takes place
   * or a object containing options.
   * @param {Object} [options.thisObj] `this` reference for the function.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Function|String} [options.throttled] A function to call when the invocation is delayed due to buffer
   * time not having expired. If this is a string, it is looked up as a method on `this` instance (or `options.thisObj`
   * instead, if provided). When called, the same arguments are passed as would have been passed to `fn`, including
   * any `options.appendArgs`.
   * @internal
   */
  throttle(fn2, options) {
    let delay3 = options, throttled;
    if (options && typeof options !== "number") {
      delay3 = options.delay;
      throttled = options.throttled;
    } else {
      options = null;
    }
    const me = this, throttleWrapFn = (...args) => {
      if (throttleWrapFn.suspended) {
        return;
      }
      const { delay: delay4 } = throttleWrapFn, elapsed = performance2.now() - throttleWrapFn.lastCallTime;
      throttleWrapFn.args = args;
      if (throttleWrapFn.immediate || elapsed >= delay4) {
        me.clearTimeout(throttleWrapFn.timerId);
        invoker();
      } else {
        if (!throttleWrapFn.isPending) {
          throttleWrapFn.timerId = me.setTimeout(invoker, delay4 - elapsed);
          throttleWrapFn.called = false;
        }
        if (throttled) {
          throttled.wrapFn.args = args;
          throttled();
        }
      }
    }, invoker = makeInvoker(me, fn2, throttleWrapFn, options);
    throttleWrapFn.delay = delay3;
    if (throttled) {
      throttled = makeInvoker(me, throttled, () => {
      }, options);
    }
    return decorateWrapFn(me, throttleWrapFn);
  }
  static setupDelayable(cls2) {
    cls2.setupDelayableMethods(cls2.delayable);
  }
  /**
   * This method initializes the `delayable` methods on this class.
   * @param {Object} delayable The `delayable` property.
   * @param {Function} [cls] This parameter will be used internally to process static methods.
   * @private
   */
  static setupDelayableMethods(delayable, cls2 = null) {
    const me = this, statics = delayable.static, target = cls2 || me.prototype;
    if (statics) {
      delete delayable.static;
    }
    for (const name in delayable) {
      let options = delayable[name];
      const implName = name + "Now", type = typeof options;
      if (!target[implName]) {
        target[implName] = target[name];
      }
      if (type === "number") {
        options = {
          type: "buffer",
          delay: options
        };
      } else if (type === "string") {
        options = {
          type: options
        };
      }
      defineProperty3(target, name, {
        get() {
          const value = this[options.type]((...params) => {
            this[implName](...params);
          }, options);
          defineProperty3(this, name, { value });
          return value;
        }
      });
    }
  }
  // This does not need a className on Widgets.
  // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
  // to the Widget it's mixed in to should implement thus.
  get widgetClass() {
  }
};

// ../Core/lib/Core/state/StateStorage.js
var StateStorage = class {
  /**
   * Returns an object with all stored keys and their values as its properties
   * @member {Object}
   */
  get data() {
    return /* @__PURE__ */ Object.create(null);
  }
  /**
   * Returns the stored keys as set by {@link #function-setItem}
   * @member {String[]}
   */
  get keys() {
    return [];
  }
  /**
   * Remove all stored keys
   */
  clear() {
  }
  /**
   * Returns key value as set by {@link #function-setItem}
   * @param {String} key
   * @returns {*}
   */
  getItem(key) {
    return null;
  }
  /**
   * Removes the specified key
   * @param {String} key
   */
  removeItem(key) {
  }
  /**
   * Sets the specified key to the given value
   * @param {String} key
   * @param {*} value The item value
   */
  setItem(key, value) {
  }
};
StateStorage._$name = "StateStorage";

// ../Core/lib/Core/state/StateProvider.js
var Local = class extends StateStorage {
  constructor(stateProvider) {
    super();
    this.prefix = stateProvider.prefix || "";
  }
  get isLocal() {
    return true;
  }
  get data() {
    const data = empty(), keys = this.keys;
    for (const key of keys) {
      data[key] = this.getItem(key);
    }
    return data;
  }
  get keys() {
    return getKeys(this.prefix, this.prefix.length);
  }
  clear() {
    const keys = getKeys(this.prefix);
    for (const key of keys) {
      localStorage.removeItem(key);
    }
  }
  getItem(key) {
    const value = localStorage.getItem(this.prefix + key);
    return value === null ? value : JSON.parse(value);
  }
  removeItem(key) {
    return localStorage.removeItem(this.prefix + key);
  }
  setItem(key, value) {
    return localStorage.setItem(this.prefix + key, JSON.stringify(value));
  }
};
var Memory = class extends StateStorage {
  constructor() {
    super();
    this.clear();
  }
  get isMemory() {
    return true;
  }
  get data() {
    return ObjectHelper.clone(this._data);
  }
  get keys() {
    return Object.keys(this._data);
  }
  clear() {
    this._data = empty();
  }
  getItem(key) {
    return key in this._data ? this._data[key] : null;
  }
  removeItem(key) {
    delete this._data[key];
  }
  setItem(key, value) {
    this._data[key] = value;
  }
};
var empty = () => /* @__PURE__ */ Object.create(null);
var getKeys = (prefix, pos = 0) => {
  const keys = [], count = localStorage.length;
  for (let key, i = 0; i < count; ++i) {
    key = localStorage.key(i);
    key.startsWith(prefix) && keys.push(key.slice(pos));
  }
  return keys;
};
var nullStorage = new StateStorage();
var storageTypes = {
  local: Local,
  memory: Memory
};
var StateProvider = class extends Base.mixin(Delayable_default, Events_default) {
  static get $name() {
    return "StateProvider";
  }
  static get configurable() {
    return {
      /**
       * The key prefix applied when using the `'local'` {@link #config-storage} type.
       * @config {String}
       * @default
       */
      prefix: "bryntum-state:",
      /**
       * Storage instance
       * @member {Core.state.StateStorage} storage
       */
      /**
       * One of the following storage types:
       *  - `local` : Stores data in the browser's `localStorage` using the {@link #config-prefix}.
       *  - `memory` : Stores data in the provider's memory.
       *
       * @config {'local'|'memory'|Core.state.StateStorage}
       * @default
       */
      storage: "local"
    };
  }
  static get delayable() {
    return {
      writeStatefuls: 50
    };
  }
  /**
   * The default {@link Core.mixin.State#config-stateProvider} for stateful objects.
   * @property {Core.state.StateProvider}
   */
  static get instance() {
    return this._instance;
  }
  static set instance(inst) {
    if (inst == null) {
      inst = nullProvider;
    } else {
      if (typeof inst === "string" || ObjectHelper.isClass(inst) || inst instanceof StateStorage) {
        inst = {
          storage: inst
        };
      }
      if (ObjectHelper.isObject(inst)) {
        inst = new StateProvider(inst);
      }
    }
    this._instance = inst;
  }
  /**
   * Initializes the default `StateProvider` instance for the page. This method can be passed an instance or one of
   * the following type aliases:
   *
   *  - `'local'` : use `localStorage` to store application state (most common)
   *  - `'memory'` : holds application state in the `StateProvider` instance (used when state is saved to a server)
   *
   * Once the `StateProvider` is initialized, components that use {@link Core.mixin.State} and assign components a
   * {@link Core.mixin.State#config-stateId} will use this default provider to automatically save and restore their
   * state.
   *
   * @param {'local'|'memory'|Core.state.StateProvider} inst The state provider storage type ('local' or 'memory') or
   * the `StateProvider` instance.
   * @returns {Core.state.StateProvider}
   */
  static setup(inst) {
    this.instance = inst;
    return this.instance;
  }
  doDestroy() {
    self.writeStatefuls.flush();
    super.doDestroy();
  }
  /**
   * On read, this property returns all state data stored in the provider. On write, this property _adds_ all the
   * given values to the state provider's data. To replace the data, call {@link #function-clear} before assigning
   * this property. This is used to bulk populate this `StateProvider` with data for stateful components.
   * @member {Object}
   */
  get data() {
    return this.storage.data;
  }
  set data(data) {
    if (!data) {
      this.clear();
    } else {
      for (const key in data) {
        this.setValue(key, data[key]);
      }
    }
  }
  /**
   * Clears all state date
   * @returns {Core.state.StateProvider} this instance
   */
  clear() {
    this.storage.clear();
    return this;
  }
  changeStorage(storage) {
    if (storage == null) {
      storage = nullStorage;
    } else {
      if (typeof storage === "string") {
        if (!storageTypes[storage]) {
          throw new Error(`Invalid storage type "${storage}" (expected one of: "${Object.keys(storageTypes).join('", "')}")`);
        }
        storage = storageTypes[storage];
      }
      if (ObjectHelper.isClass(storage)) {
        storage = new storage(this);
      }
    }
    return storage;
  }
  /**
   * This method is called to schedule saving the given `stateful` object.
   * @param {Core.mixin.State} stateful The stateful object to save.
   * @param {Object} [options] An object of options that affect the state saving process.
   * @param {String} [options.id] The key for the saved state.
   * @param {Boolean} [options.immediate] Pass `true` to save the data synchronously instead of on a delay.
   * @internal
   */
  saveStateful(stateful, options) {
    (this.pendingSaves || (this.pendingSaves = [])).push([stateful, options]);
    this.writeStatefuls();
  }
  /**
   * A delayable method that flushes pending stateful objects.
   * @private
   */
  writeStatefuls() {
    const me = this, { pendingSaves } = me, n = pendingSaves == null ? void 0 : pendingSaves.length, stateIds = [], saved = [];
    me.pendingSaves = null;
    if (n) {
      for (let options, stateful, stateId, i = 0; i < n; ++i) {
        [stateful, options] = pendingSaves[i];
        if (!stateful.isDestroying && stateful.isSaveStatePending) {
          stateId = stateful.saveState({
            ...options,
            immediate: true
          });
          if (stateId) {
            stateIds.push(stateId);
            saved.push(stateful);
          }
        }
      }
      if (stateIds.length) {
        me.trigger("save", {
          stateIds,
          saved
        });
      }
    }
  }
  /**
   * Returns the stored state given its `key`.
   * @param {String} key The identifier of the state to return.
   * @returns {Object}
   */
  getValue(key) {
    this.writeStatefuls.flush();
    return this.storage.getItem(key);
  }
  /**
   * Stores the given state `value` under the specified `key`.
   * @param {String} key The identifier of the state value.
   * @param {Object} value The state value to set.
   * @returns {Core.state.StateProvider} this instance
   */
  setValue(key, value) {
    const me = this, { storage } = me, was = me.getValue(key);
    if (value != null) {
      storage.setItem(key, value);
      me.trigger("set", { key, value, was });
    } else if (was !== null) {
      storage.removeItem(key);
      me.trigger("remove", { key, was });
    }
    return me;
  }
};
var nullProvider = new StateProvider({
  storage: nullStorage
});
StateProvider._instance = nullProvider;
StateProvider._$name = "StateProvider";

// ../Core/lib/Core/mixin/State.js
var primitiveRe = /boolean|number|string/;
var State_default = (Target) => {
  var _a2;
  return _a2 = class extends (Target || Base) {
    afterConstruct() {
      super.afterConstruct();
      this.loadState();
    }
    finalizeInit() {
      this.loadState();
      super.finalizeInit();
    }
    /**
     * Returns `true` if this instance implements the {@link Core.mixin.State} interface.
     * @property {Boolean}
     * @readonly
     * @advanced
     */
    get isStateful() {
      return true;
    }
    /**
     * Returns `true` if this instance is ready to participate in state activities.
     * @property {Boolean}
     * @readonly
     * @internal
     */
    get isStatefulActive() {
      return !this.statefulSuspended && !this.isResponsivePending && !this.isResponsiveUpdating;
    }
    // state
    /**
     * Gets or sets a component's state
     * @property {Object}
     * @category State
     */
    get state() {
      return this._state = this.getState();
    }
    set state(state) {
      this._state = state;
      if (state) {
        this.applyState(state);
      }
    }
    // statefulEvents
    updateStatefulEvents(events) {
      var _a3;
      const me = this, listeners = {
        name: "statefulEvents",
        thisObj: me
      };
      me.detachListeners(listeners.name);
      if (events) {
        if (typeof events === "string") {
          events = StringHelper.split(events);
        } else if (!Array.isArray(events)) {
          events = ObjectHelper.getTruthyKeys(events);
        }
        if (events.length) {
          for (const event of events) {
            listeners[event] = "onStatefulEvent";
          }
          (_a3 = me.ion) == null ? void 0 : _a3.call(me, listeners);
        }
      }
    }
    // statefulId
    /**
     * Returns the state key to use for this instance. This will be either the {@link #config-stateId} or the
     * {@link Core.widget.Widget#config-id} (if explicitly specified and {@link #config-stateful} is not `false`).
     * @property {String}
     * @category State
     * @internal
     */
    get statefulId() {
      const me = this, { responsiveState } = me;
      let statefulId = me.stateId;
      if (statefulId == null && me.hasGeneratedId === false && me.stateful !== false) {
        statefulId = me.id;
      }
      if (statefulId && responsiveState) {
        statefulId = `${statefulId}[${responsiveState}]`;
      }
      return statefulId;
    }
    // statefulness
    /**
     * Returns an object whose truthy keys are the config properties to include in this object's {@link #property-state}.
     * @property {Object}
     * @category State
     * @readonly
     * @private
     */
    get statefulness() {
      const { stateful } = this;
      return Array.isArray(stateful) ? ObjectHelper.createTruthyKeys(stateful) : stateful;
    }
    // stateProvider
    get stateProvider() {
      return this._stateProvider || StateProvider.instance;
    }
    //---------------------------------------------------------------------------------------------------------------
    // Methods
    /**
     * Applies the given `state` to this instance.
     *
     * This method is not called directly, but is called when the {@link #property-state} property is assigned a value.
     *
     * This method is implemented by derived classes that have complex state which exceeds the simple list of config
     * properties provided by {@link #config-stateful}. In these cases, the `super` method can be called to handle any
     * config properties that are part of the complex state. The default implementation of this method will only assign
     * those config properties listed in {@link #config-stateful} from the provided `state` object.
     *
     * @param {Object} state The state object to apply to this instance.
     * @category State
     * @advanced
     */
    applyState(state) {
      state = this.pruneState(state);
      if (state) {
        this.setConfig(state);
      }
    }
    /**
     * Returns this object's state information.
     *
     * This method is not called directly, but is called to return the value of the {@link #property-state} property.
     *
     * This method is implemented by derived classes that have complex state which exceeds the simple list of config
     * properties provided by {@link #config-stateful}. In these cases, the `super` method can be called to gather the
     * config properties that are part of the complex state. The default implementation of this method will only copy
     * those config properties listed in {@link #config-stateful} to the returned `state` object.
     *
     * @returns {Object}
     * @category State
     * @advanced
     */
    getState() {
      const me = this, {
        initialConfig,
        statefulness,
        isConstructing: defaultState
      } = me, { configs } = me.$meta, source = defaultState ? Object.setPrototypeOf(initialConfig, me.$meta.config) : me;
      let state = null, key, value;
      if (statefulness) {
        state = {};
        for (key in statefulness) {
          if (statefulness[key]) {
            value = source[key];
            if (value == null ? void 0 : value.isStateful) {
              value = value.state;
            } else if (!defaultState) {
              if (ObjectHelper.isDate(value)) {
                value = DateHelper.format(value, "YYYY-MM-DDTHH:mm:ssZ");
              }
              if (configs[key].equal(value, initialConfig == null ? void 0 : initialConfig[key]) || !primitiveRe.test(typeof value)) {
                continue;
              }
            }
            state[key] = value;
          }
        }
      }
      return state;
    }
    /**
     * Loads this object's state from its {@link #config-stateProvider} and applies it to its {@link #property-state}.
     *
     * This method only acts upon its first invocation for a given instance (unless `true` is passed for the `reload`
     * parameter). This allows for flexibility in the timing of that call during the early stages of the instances'
     * lifecycle. To reload the state after this time, manually assign the desired value to the {@link #property-state}
     * property or call this method and pass `reload` as `true`.
     *
     * This method is called automatically during construction when a {@link #config-stateId} or (in some cases) an
     * explicit {@link Core.widget.Widget#config-id} is provided.
     *
     * @param {String} [stateId] An overriding key to use instead of this object's {@link #config-stateId}.
     * @param {Boolean} [reload=false] Pass `true` to load the state even if previously loaded.
     * @category State
     */
    loadState(stateId, reload) {
      if (typeof stateId === "boolean") {
        reload = stateId;
        stateId = null;
      }
      const me = this, { statefulLoaded } = me;
      if (me.isStatefulActive && (reload || !statefulLoaded)) {
        const state = me.loadStatefulData(stateId || (stateId = me.statefulId));
        if (!statefulLoaded && stateId) {
          me.defaultState = me.state;
          me.statefulLoaded = true;
        }
        if (state) {
          me.state = state;
        }
      }
    }
    loadStatefulData(stateId) {
      var _a3;
      stateId = this.isStatefulActive ? stateId || this.statefulId : null;
      return stateId && ((_a3 = this.stateProvider) == null ? void 0 : _a3.getValue(stateId));
    }
    resetDefaultState() {
      if (this.defaultState) {
        this.state = this.defaultState;
      }
    }
    resumeStateful(full = false) {
      this.statefulSuspended = full ? 0 : Math.max(this.statefulSuspended - 1, 0);
    }
    /**
     * Saves this object's state to its {@link #config-stateProvider}.
     *
     * When a {@link #config-stateId} or (in some cases) an explicit {@link Core.widget.Widget#config-id} is provided,
     * this method will be called automatically any time a config property listed in {@link #config-stateful} changes or
     * when a {@link #config-statefulEvents stateful event} is fired.
     *
     * Derived classes are responsible for calling this method whenever the persistent {@link #property-state} of the
     * object changes.
     *
     * @param {Object|String} [options] Options that affect the state saving process or, if a string, the state `id`.
     * @param {String} [options.id] The state id for the saved state (overrides {@link #config-stateId}).
     * @param {Boolean} [options.immediate] Pass `true` to save the data synchronously instead of on a delay.
     * @category State
     */
    saveState(options) {
      if (typeof options === "string") {
        options = {
          id: options
        };
      } else {
        options = options || {};
      }
      const me = this, { stateProvider } = me, statefulId = options.id || me.isStatefulActive && me.statefulId;
      if (statefulId && stateProvider) {
        if (options.immediate) {
          me.isSaveStatePending = false;
          stateProvider.setValue(statefulId, me.state);
        } else if (!me.isSaveStatePending) {
          me.isSaveStatePending = true;
          stateProvider.saveStateful(me, options);
        }
        return statefulId;
      }
    }
    suspendStateful() {
      ++this.statefulSuspended;
    }
    //---------------------------------------------------------------------------------------------------------------
    // Private / Internal
    onConfigChange({ name, value, was, config }) {
      super.onConfigChange({ name, value, was, config });
      if (!this.isConstructing && this.isStatefulActive && this.statefulId) {
        const { stateful } = this;
        if (Array.isArray(stateful) ? stateful.includes(name) : stateful == null ? void 0 : stateful[name]) {
          this.saveState();
        }
      }
    }
    onStatefulEvent() {
      if (!this.isConstructing) {
        this.saveState();
      }
    }
    /**
     * Returns an object that copies the {@link #config-stateful} config properties from the provided `state` object.
     *
     * @param {Object} state A state object from which to copy stateful configs.
     * @returns {Object}
     * @category State
     * @private
     */
    pruneState(state) {
      const { statefulness } = this;
      if (statefulness) {
        const pruned = {};
        for (const key in state) {
          if (statefulness[key]) {
            pruned[key] = state[key];
          }
        }
        state = pruned;
      }
      return state;
    }
    //---------------------------------------------------------------------------------------------------------------
    // This does not need a className on Widgets.
    // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
    // to the Widget it's mixed in to should implement thus.
    get widgetClass() {
    }
  }, __publicField(_a2, "$name", "State"), __publicField(_a2, "configurable", {
    /**
     * This value can be one of the following:
     *
     * - `false` to not use an explicitly assigned {@link Core.widget.Widget#config-id} as the component's
     * {@link #config-stateId} (this is only necessary when there is a {@link #config-stateProvider}).
     * - An array of strings naming the config properties to save in the component's {@link #property-state}
     * object.
     * - An object whose truthy keys are the config properties to save in the component's {@link #property-state}
     * object.
     *
     * These last two uses of the `stateful` config property do not apply to components that have a complex
     * state, as described in the {@link Core.mixin.State State mixin documentation}.
     *
     * This config property is typically set by derived classes to a value including any config property that
     * the user can affect via the user interface. For example, the {@link Core.widget.Panel#config-collapsed}
     * config property is listed for a {@link Core.widget.Panel} since the user can toggle this config property
     * using the {@link Core.widget.panel.PanelCollapser#config-tool collapse tool}.
     *
     * @config {Boolean|Object|String[]}
     * @category State
     */
    stateful: {
      value: null,
      $config: {
        merge: "classList"
      }
    },
    /**
     * The events that, when fired by this component, should trigger it to save its state by calling
     * {@link #function-saveState}.
     *
     * ```javascript
     *  class MyStatefulComponent extends Base.mixin(State) {
     *      static get configurable() {
     *          return {
     *              statefulEvents : [ 'change', 'resize' ]
     *          };
     *      }
     *  }
     * ```
     * In the above example, {@link #function-saveState} will be called any time an instance of this class
     * fires the `change` or `resize` event.
     *
     * This config is typically set by derived classes as a way to ensure {@link #function-saveState} is called
     * whenever their persistent state changes.
     *
     * @config {Object|String[]}
     * @category State
     * @default
     */
    statefulEvents: {
      $config: {
        merge: "classList"
      },
      value: ["stateChange"]
    },
    /**
     * The key to use when saving this object's state in the {@link #config-stateProvider}. If this config is
     * not assigned, and {@link #config-stateful} is not set to `false`, the {@link Core.widget.Widget#config-id}
     * (if explicitly specified) will be used as the `stateId`.
     *
     * If neither of these is given, the {@link #function-loadState} and {@link #function-saveState} methods
     * will need to be called directly to make use of the `stateProvider`.
     *
     * For single page applications (SPA's), or multi-page applications (MPA's) that have common, stateful
     * components on multiple pages, the `stateId` should be unique across all stateful components (similar to DOM
     * element id's). MPA's that want each page to be isolated can more easily achieve that isolation using the
     * {@link Core.state.StateProvider#config-prefix}.
     *
     * @config {String}
     * @category State
     */
    stateId: null,
    /**
     * The `StateProvider` to use to save and restore this object's {@link #property-state}. By default, `state`
     * will be saved using the {@link Core.state.StateProvider#property-instance-static default state provider}.
     *
     * This config is useful for multi-page applications that have a set of common components that want to share
     * state across pages, as well as other components that want their state to be isolated. One of these groups
     * of stateful components could be assigned an explicit `stateProvider` while the other group could use the
     * default state provider.
     *
     * @config {Core.state.StateProvider}
     * @category State
     */
    stateProvider: null
  }), __publicField(_a2, "prototypeProperties", {
    statefulLoaded: false,
    statefulSuspended: 0
  }), _a2;
};

// ../Core/lib/Core/mixin/Identifiable.js
var idCounts = ObjectHelper.getPathDefault(globalThis, "bryntum.idCounts", /* @__PURE__ */ Object.create(null));
var idTypes = {
  string: 1,
  number: 1
};
var Identifiable_default = (Target) => class Identifiable extends (Target || Base) {
  static get $name() {
    return "Identifiable";
  }
  static get declarable() {
    return [
      "identifiable"
    ];
  }
  static get configurable() {
    return {
      /**
       * The id of this object.  If not specified one will be generated. Also used for lookups through the
       * static `getById` of the class which mixes this in. An example being {@link Core.widget.Widget}.
       *
       * For a {@link Core.widget.Widget Widget}, this is assigned as the `id` of the DOM
       * {@link Core.widget.Widget#config-element element} and must be unique across all elements
       * in the page's `document`.
       * @config {String}
       * @category Common
       */
      id: ""
    };
  }
  static setupIdentifiable(cls2, meta) {
    const { identifiable } = cls2;
    identifiable.idMap = /* @__PURE__ */ Object.create(null);
    Reflect.defineProperty(cls2, "identifiable", {
      get() {
        return identifiable;
      }
    });
  }
  doDestroy() {
    this.constructor.unregisterInstance(this);
    super.doDestroy();
  }
  changeId(id) {
    return (this.hasGeneratedId = !id) ? this.generateAutoId() : id;
  }
  updateId(id, oldId) {
    const me = this, C = me.constructor;
    oldId && C.unregisterInstance(me, oldId);
    if (!me.hasGeneratedId || C.identifiable.registerGeneratedId !== false) {
      C.registerInstance(me, id);
    }
  }
  /**
   * This method generates an id for this instance.
   * @returns {String}
   * @internal
   */
  generateAutoId() {
    return this.constructor.generateId(`b-${this.$$name.toLowerCase()}-`);
  }
  static get all() {
    return Object.values(this.identifiable.idMap);
  }
  /**
   * Generate a new id, using an internal counter and a prefix.
   * @param {String} prefix Id prefix
   * @returns {String} Generated id
   */
  static generateId(prefix = "generatedId") {
    return prefix + (idCounts[prefix] = (idCounts[prefix] || 0) + 1);
  }
  static registerInstance(instance, instanceId = instance.id) {
    const { idMap } = this.identifiable;
    if (instanceId in idMap && !this.disableThrow) {
      throw new Error("Id " + instanceId + " already in use");
    }
    idMap[instanceId] = instance;
  }
  /**
   * Unregister Identifiable instance, normally done on destruction
   * @param {Object} instance Object to unregister
   * @param {String} id The id of the instance to unregister.
   */
  static unregisterInstance(instance, id = instance.id) {
    const { idMap } = this.identifiable;
    if (idTypes[typeof instance]) {
      delete idMap[instance];
    } else if (idMap[id] === instance) {
      delete idMap[id];
    }
  }
  static getById(id) {
    const idMap = this.identifiable.idMap;
    if (idMap) {
      return idMap[id];
    }
  }
  static get registeredInstances() {
    const idMap = this.identifiable.idMap;
    return idMap ? Object.values(idMap) : [];
  }
};

// ../Core/lib/Core/data/stm/mixin/ModelStm.js
var STM_PROP = Symbol("STM_PROP");
var unrecordedFields = {
  // This field's value is a by product of node insertion and must not be recorded here.
  // It's the node insertion operation which is recorded by STM.
  parentIndex: 1
};
var ModelStm_default = (Target) => class ModelStm extends (Target || Base) {
  static get $name() {
    return "ModelStm";
  }
  static get defaultConfig() {
    return {
      stm: null
    };
  }
  joinStore(store) {
    if (!this.stm) {
      this.stm = store.stm;
    }
  }
  unjoinStore(store, isReplacing = false) {
    var _a2;
    if (this.stm === store.stm) {
      this.stm = null;
    }
    (_a2 = super.unjoinStore) == null ? void 0 : _a2.call(this, store, isReplacing);
  }
  /**
   * Reference to STM manager, if used
   * @member {Core.data.stm.StateTrackingManager}
   * @category Misc
   */
  get stm() {
    return this[STM_PROP];
  }
  set stm(stm) {
    this[STM_PROP] = stm;
  }
  // Hook for chronograph entity field accessors, for example; task.duration = 123.
  // Triggers before setting the value.
  beforeChronoFieldSet(fieldName, value) {
    var _a2;
    const me = this;
    if (!me.inSetting && ((_a2 = me.stm) == null ? void 0 : _a2.enabled) && !unrecordedFields[fieldName] && !me.constructor.nonPersistableFields[fieldName]) {
      if (me.getFieldDefinition(fieldName)) {
        return { [fieldName]: { value, oldValue: me[fieldName] } };
      }
    }
    return null;
  }
  // Hook for chronograph entity field accessors, for example; task.duration = 123
  // Triggers after setting the value.
  afterChronoFieldSet(fieldName, value, wasSet) {
    wasSet && this.afterSet(fieldName, value, false, false, wasSet, true);
  }
  shouldRecordFieldChange(fieldName, oldValue, newValue) {
    const store = this.firstStore;
    return !(this.constructor.nonPersistableFields[fieldName] || oldValue == null && newValue == null || oldValue === newValue || oldValue instanceof Date && newValue instanceof Date && oldValue.getTime() === newValue.getTime() || fieldName === "id" || fieldName === "$PhantomId" || fieldName === "parentId" && store && store.oldIdMap[oldValue] === store.getById(newValue));
  }
  /**
   * Overridden to store initial data of the changed fields and to notify STM
   * manager about the change action if anything has been changed in result.
   *
   * The method is called from within {@link Core/data/Model#function-set} method.
   *
   * @private
   */
  afterSet(field2, value, silent, fromRelationUpdate, wasSet, isChronoFieldSet) {
    const { stm } = this, nonPersistableFields = this.constructor.nonPersistableFields;
    if ((stm == null ? void 0 : stm.isBase) && stm.enabled && !unrecordedFields[field2] && !nonPersistableFields[field2]) {
      if (wasSet) {
        let shouldRecord;
        const [newData, oldData] = Object.keys(wasSet).reduce(
          (data, fieldName) => {
            const { value: value2, oldValue } = wasSet[fieldName];
            if (this.shouldRecordFieldChange(fieldName, oldValue, value2)) {
              shouldRecord = true;
              data[0][fieldName] = value2;
              data[1][fieldName] = oldValue;
            }
            return data;
          },
          [{}, {}]
        );
        if (shouldRecord) {
          stm.onModelUpdate(this, newData, oldData, isChronoFieldSet);
        }
      }
    }
  }
  /**
   * Called from {@link Core/data/mixin/TreeNode#function-insertChild} to obtain inserted
   * records initial parents and parent index, to be able to restore the state back upon undo.
   *
   * @param {Core.data.Model[]} childRecords
   * @returns {Array} Array of results from this call and any of super calls if any.
   *               This result is consumed by {@link #function-afterInsertChild} which pops
   *               from the result array to take only results of this method call and leave
   *               results from super calls untouched.
   *
   * @private
   */
  beforeInsertChild(childRecords) {
    var _a2;
    const preResult = ((_a2 = super.beforeInsertChild) == null ? void 0 : _a2.call(this, childRecords)) || [], { stm } = this;
    if (stm == null ? void 0 : stm.enabled) {
      preResult.push(
        childRecords.reduce((result, childRecord) => {
          if (childRecord.root === this.root) {
            result.set(childRecord, {
              parent: childRecord.parent,
              index: childRecord.parent ? childRecord.parentIndex : void 0
            });
          }
          return result;
        }, /* @__PURE__ */ new Map())
      );
    }
    return preResult;
  }
  /**
   * Called from {@link Core/data/mixin/TreeNode#function-insertChild} to notify {@link Core/data/stm/StateTrackingManager}
   * about children insertion. Provides it with all necessary context information collected
   * in {@link #function-beforeInsertChild} required to undo/redo the action.
   *
   * @private
   */
  afterInsertChild(index, childRecords, beforeResult, inserted) {
    var _a2;
    const { stm } = this;
    if (stm == null ? void 0 : stm.enabled) {
      const context = beforeResult.pop();
      if (inserted) {
        stm.onModelInsertChild(this, index, inserted, context);
      }
    }
    (_a2 = super.afterInsertChild) == null ? void 0 : _a2.call(this, index, childRecords, beforeResult, inserted);
  }
  /**
   * Called from {@link Core/data/mixin/TreeNode#function-removeChild} to obtain removed
   * records initial parent index, to be able to restore the state back upon undo.
   *
   * @param {Core.data.Model[]} childRecords
   * @param {Boolean} isMove
   * @returns {Array} Array of results from this call and any of super calls if any.
   *               This result is consumed by {@link #function-afterRemoveChild} which pops
   *               from the result array to take only results of this method call and leave
   *               results from super calls untouched.
   *
   * @private
   */
  beforeRemoveChild(childRecords, isMove) {
    const preResult = super.beforeRemoveChild ? super.beforeRemoveChild(childRecords, isMove) : [], { stm } = this;
    if ((stm == null ? void 0 : stm.enabled) && !isMove) {
      preResult.push(
        childRecords.reduce((result, childRecord) => {
          result.set(childRecord, { parentIndex: childRecord.parentIndex, orderedParentIndex: childRecord.orderedParentIndex });
          return result;
        }, /* @__PURE__ */ new Map())
      );
    }
    return preResult;
  }
  /**
   * Called from {@link Core/data/mixin/TreeNode#function-removeChild} to notify {@link Core/data/stm/StateTrackingManager}
   * about children removing. Provides it with all necessary context information collected
   * in {@link #function-beforeRemoveChild} required to undo/redo the action.
   *
   * @private
   */
  afterRemoveChild(childRecords, beforeResult, isMove) {
    var _a2;
    const { stm } = this;
    if ((stm == null ? void 0 : stm.enabled) && !isMove) {
      const context = beforeResult.pop();
      if (childRecords && childRecords.length) {
        stm.onModelRemoveChild(this, childRecords, context);
      }
    }
    (_a2 = super.afterRemoveChild) == null ? void 0 : _a2.call(this, childRecords, beforeResult, isMove);
  }
};

// ../Core/lib/Core/data/mixin/TreeNode.js
var defaultTraverseOptions = {
  includeFilteredOutRecords: false
};
var fixTraverseOptions = (options) => {
  options = options || false;
  if (typeof options === "boolean") {
    options = {
      includeFilteredOutRecords: options
    };
  }
  return options || defaultTraverseOptions;
};
var TreeNode_default = (Target) => class TreeNode extends (Target || Base) {
  static get $name() {
    return "TreeNode";
  }
  /**
   * This static configuration option allows you to control whether an empty parent task should be converted into a
   * leaf. Enable/disable it for a whole class:
   *
   * ```javascript
   * Model.convertEmptyParentToLeaf = false;
   * ```
   *
   * By specifying `true`, all empty parents will be considered leafs. Can also be assigned a configuration object
   * with the following Boolean properties to customize the behaviour:
   *
   * ```javascript
   * Model.convertEmptyParentToLeaf = {
   *     onLoad   : false,
   *     onRemove : true
   * }
   * ```
   *
   * @member {Boolean|{ onLoad : Boolean, onRemove : Boolean }} convertEmptyParentToLeaf
   * @property {Boolean} onLoad Apply the transformation on load to any parents without children (`children : []`)
   * @property {Boolean} onRemove Apply the transformation when all children have been removed from a parent
   * @default false
   * @static
   * @category Parent & children
   * */
  static set convertEmptyParentToLeaf(value) {
    if (value === true) {
      value = {
        onLoad: true,
        onRemove: true
      };
    } else if (value === false) {
      value = {
        onLoad: false,
        onRemove: false
      };
    }
    this._convertEmptyParentToLeaf = value;
  }
  constructor(...args) {
    super(...args);
    if (this.children) {
      this.orderedChildren = this.orderedChildren || [];
    }
  }
  static get convertEmptyParentToLeaf() {
    return this._convertEmptyParentToLeaf || { onLoad: false, onRemove: false };
  }
  /**
   * This is a read-only property providing access to the parent node.
   * @member {Core.data.Model} parent
   * @readonly
   * @category Parent & children
   */
  /**
   * This is a read-only field provided in server synchronization packets to specify
   * which record id is the parent of the record.
   * @readonly
   * @field {String|Number|null} parentId
   * @category Tree
   */
  /**
   * This is a read-only field provided in server synchronization packets to specify
   * which position the node takes in the parent's children array.
   * This index is set on load and gets updated automatically after row reordering, sorting, etc.
   * To save the order, need to persist the field on the server and when data is fetched to be loaded,
   * need to sort by this field.
   * @readonly
   * @field {Number} parentIndex
   * @category Tree
   */
  /**
   * This is a read-only field provided in server synchronization packets to specify
   * which position the node takes in the parent's ordered children array.
   * This index is set on load and gets updated on reordering nodes in tree. Sorting and filtering
   * have no effect on it.
   * @readonly
   * @field {Number} orderedParentIndex
   * @category Tree
   */
  ingestChildren(childRecord, stores2 = this.stores) {
    const { inProcessChildren, constructor: MyClass } = this, store = stores2[0];
    if (childRecord === true) {
      if (inProcessChildren) {
        return true;
      }
      return [];
    }
    if (childRecord) {
      childRecord = ArrayHelper.asArray(childRecord);
      const len = childRecord.length, result = [];
      for (let i = 0, child; i < len; i++) {
        child = childRecord[i];
        child = child.isModel ? child : store ? store.createRecord(child, false, true) : new MyClass(child, null, null, true);
        child = store ? store.processRecord(child) : child;
        result.push(child);
      }
      if (this.children === true && store) {
        const sorter = store.createSorterFn(store.sorters);
        result.sort(sorter);
      }
      return result;
    }
  }
  /**
   * Child nodes. To allow loading children on demand, specify `children : true` in your data. Omit the field for leaf
   * tasks.
   *
   * Note, if the tree store loads data from a remote origin, make sure {@link Core/data/AjaxStore#config-readUrl}
   * is specified, and optionally {@link Core/data/AjaxStore#config-parentIdParamName} is set, otherwise
   * {@link Core/data/Store#function-loadChildren} has to be implemented.
   *
   * @field {Boolean|Object[]|Core.data.Model[]} children
   * @category Parent & children
   */
  /**
   * Array of sorted tree nodes but without a filter applied
   * @member {Core.data.Model[]|null} unfilteredChildren
   * @category Parent & children
   * @private
   */
  /**
   * Array of children unaffected by sorting and filtering, keeps original tree structure
   * @member {Core.data.Model[]|null} orderedChildren
   * @category Parent & children
   * @private
   */
  /**
   * Called during creation to also turn any children into Models joined to the same stores as this model
   * @internal
   * @category Parent & children
   */
  processChildren(stores2 = this.stores) {
    const me = this, { meta } = me;
    me.inProcessChildren = true;
    const children = me.ingestChildren(me.data[me.constructor.childrenField], stores2);
    if (children) {
      const { convertEmptyParentToLeaf } = me.constructor, shouldConvert = convertEmptyParentToLeaf === true || convertEmptyParentToLeaf.onLoad;
      if (shouldConvert ? children.length : Array.isArray(children)) {
        meta.isLeaf = false;
        if (me.children === true) {
          me.children = [];
        } else if (children.length === 0) {
          me.children = children;
          return;
        }
        me.insertChild(children);
      } else if (children === true) {
        meta.isLeaf = false;
        me.children = true;
      } else if (!me.isRoot) {
        meta.isLeaf = me.constructor.convertEmptyParentToLeaf.onLoad;
      }
    }
    me.inProcessChildren = false;
  }
  /**
   * This method returns `true` if this record has all expanded ancestors and is therefore
   * eligible for inclusion in a UI.
   * @param {Core.data.Store} [store] Optional store, defaults to nodes first store
   * @returns {Boolean}
   * @readonly
   * @category Parent & children
   * @returns {Boolean}
   */
  ancestorsExpanded(store = this.firstStore) {
    const { parent } = this;
    return !parent || parent.isExpanded(store) && parent.ancestorsExpanded(store);
  }
  /**
   * Used by stores to assess the record's collapsed/expanded state in that store.
   * @param {Core.data.Store} store
   * @category Parent & children
   * @returns {Boolean}
   */
  isExpanded(store = this.firstStore) {
    const mapMeta = this.instanceMeta(store.id);
    if (!Object.prototype.hasOwnProperty.call(mapMeta, "collapsed")) {
      mapMeta.collapsed = !this.expanded;
    }
    return !mapMeta.collapsed;
  }
  // A read-only property. It provides the initial state upon load
  // The UI's expanded/collapsed state is in the store's meta map.
  get expanded() {
    return this.data.expanded;
  }
  /**
   * Depth in the tree at which this node exists. First visual level of nodes are at level 0, their direct children at
   * level 1 and so on.
   * @property {Number}
   * @readonly
   * @category Parent & children
   */
  get childLevel() {
    let node = this, ret = -1;
    while (node && !node.isRoot) {
      ++ret;
      node = node.parent;
    }
    return ret;
  }
  /**
   * Is a leaf node in a tree structure?
   * @property {Boolean}
   * @readonly
   * @category Parent & children
   */
  get isLeaf() {
    return this.meta.isLeaf !== false && !this.isRoot;
  }
  /**
   * Returns `true` if this node is the root of the tree
   * @member {Boolean} isRoot
   * @readonly
   * @category Parent & children
   */
  /**
   * Is a parent node in a tree structure?
   * @property {Boolean}
   * @readonly
   * @category Parent & children
   */
  get isParent() {
    return !this.isLeaf;
  }
  /**
   * Returns true for parent nodes with children loaded (there might still be no children)
   * @property {Boolean}
   * @readonly
   * @category Parent & children
   */
  get isLoaded() {
    return this.isParent && Array.isArray(this.children);
  }
  /**
   * Count all children (including sub-children) for a node (in its `firstStore´)
   * @member {Number}
   * @category Parent & children
   */
  get descendantCount() {
    return this.getDescendantCount();
  }
  /**
   * Count visible (expanded) children (including sub-children) for a node (in its `firstStore`)
   * @member {Number}
   * @category Parent & children
   */
  get visibleDescendantCount() {
    return this.getDescendantCount(true);
  }
  /**
   * Count visible (expanded)/all children for this node, optionally specifying for which store.
   * @param {Boolean} [onlyVisible] Specify `true` to only count visible (expanded) children.
   * @param {Core.data.Store} [store] A Store to which this node belongs
   * @returns {Number}
   * @category Parent & children
   */
  getDescendantCount(onlyVisible = false, store = this.firstStore) {
    const { children } = this;
    if (!children || !Array.isArray(children) || onlyVisible && !this.isExpanded(store)) {
      return 0;
    }
    return children.reduce((count, child) => count + child.getDescendantCount(onlyVisible), children.length);
  }
  /**
   * Retrieve all children, not including filtered out nodes (by traversing sub nodes)
   * @property {Core.data.Model[]}
   * @category Parent & children
   */
  get allChildren() {
    return this.getAllChildren(false);
  }
  /**
   * Retrieve all children, including filtered out nodes (by traversing sub nodes)
   * @property {Core.data.Model[]}
   * @private
   * @category Parent & children
   */
  get allUnfilteredChildren() {
    return this.getAllChildren(true);
  }
  getAllChildren(unfiltered = false) {
    const { [unfiltered ? "unfilteredChildren" : "children"]: children } = this;
    if (!children || children === true) {
      return [];
    }
    return children.reduce((all, child) => {
      all.push(child);
      all.push.apply(all, unfiltered ? child.allUnfilteredChildren : child.allChildren);
      return all;
    }, []);
  }
  /**
   * Get the first child of this node
   * @property {Core.data.Model}
   * @readonly
   * @category Parent & children
   */
  get firstChild() {
    const { children } = this;
    return (children == null ? void 0 : children.length) && children[0] || null;
  }
  /**
   * Get the last child of this node
   * @property {Core.data.Model}
   * @readonly
   * @category Parent & children
   */
  get lastChild() {
    const { children } = this;
    return (children == null ? void 0 : children.length) && children[children.length - 1] || null;
  }
  /**
   * Get the previous sibling of this node
   * @member {Core.data.Model} previousSibling
   * @readonly
   * @category Parent & children
   */
  /**
   * Get the next sibling of this node
   * @member {Core.data.Model} nextSibling
   * @readonly
   * @category Parent & children
   */
  /**
   * Returns count of all preceding sibling nodes (including their children).
   * @property {Number}
   * @category Parent & children
   */
  get previousSiblingsTotalCount() {
    let task = this.previousSibling, count = this.parentIndex;
    while (task) {
      count += task.descendantCount;
      task = task.previousSibling;
    }
    return count;
  }
  get previousOrderedSibling() {
    var _a2;
    return (_a2 = this.parent) == null ? void 0 : _a2.orderedChildren[this.orderedParentIndex - 1];
  }
  get nextOrderedSibling() {
    var _a2;
    return (_a2 = this.parent) == null ? void 0 : _a2.orderedChildren[this.orderedParentIndex + 1];
  }
  get root() {
    var _a2;
    return ((_a2 = this.parent) == null ? void 0 : _a2.root) || this;
  }
  /**
   * Reading this property returns the id of the parent node, if this record is a child of a node.
   *
   * Setting this property appends this record to the record with the passed id **in the same store that this record
   * is already in**.
   *
   * Note that setting this property is **only valid if this record is already part of a tree store**.
   *
   * This is not intended for general use. This is for when a server responds to a record mutation and the server
   * decides to move a record to a new parent. If a `parentId` property is passed in the response data for a record,
   * that record will be moved.
   *
   * @property {Number|String|null}
   * @category Parent & children
   */
  get parentId() {
    return this.parent && !this.parent.isAutoRoot ? this.parent.id : null;
  }
  set parentId(parentId) {
    const me = this, { parent } = me, newParent = parentId === null ? me.firstStore.rootNode : me.firstStore.getById(parentId);
    if (!(newParent === parent || !parent && !newParent)) {
      if (me.isBatchUpdating) {
        me.meta.batchChanges.parentId = parentId;
      } else {
        if (newParent) {
          newParent.appendChild(me);
        } else {
          me.parent.removeChild(me);
        }
      }
    }
  }
  static set parentIdField(parentIdField) {
    this._parentIdField = parentIdField;
    Object.defineProperty(this.prototype, parentIdField, {
      set: function(parentId) {
        this.parentId = parentId;
      },
      get: function() {
        return this.parentId;
      }
    });
  }
  static get parentIdField() {
    return this._parentIdField || "parentId";
  }
  getChildren(options) {
    let result;
    if (options.includeFilteredOutRecords) {
      result = this.unfilteredChildren || this.children;
    } else if (options.useOrderedTree) {
      result = this.orderedChildren;
    } else {
      result = this.children;
    }
    return result;
  }
  /**
   * Traverses all child nodes recursively calling the passed function
   * on a target node **before** iterating the child nodes.
   * @param {Function} fn The function to call
   * @param {Boolean} [skipSelf=false] True to ignore self
   * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
   * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
   * @param {Boolean} [options.useOrderedTree] True to traverse unsorted/unfiltered tree
   * @category Parent & children
   */
  traverse(fn2, skipSelf, options) {
    options = fixTraverseOptions(options);
    const me = this, children = me.getChildren(options);
    if (!skipSelf) {
      fn2.call(me, me);
    }
    for (let i = 0, l = children == null ? void 0 : children.length; i < l; i++) {
      children[i].traverse(fn2, false, options);
    }
  }
  /**
   * Traverses all child nodes recursively calling the passed function
   * on child nodes of a target **before** calling it on the node.
   * @param {Function} fn The function to call
   * @param {Boolean} [skipSelf=false] True to skip this node in the traversal
   * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
   * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
   * @category Parent & children
   */
  traverseBefore(fn2, skipSelf, options) {
    options = fixTraverseOptions(options);
    const me = this, children = me.getChildren(options);
    for (let i = 0, l = children == null ? void 0 : children.length; i < l; i++) {
      children[i].traverse(fn2, false, options);
    }
    if (!skipSelf) {
      fn2.call(me, me);
    }
  }
  /**
   * Traverses child nodes recursively while fn returns true
   * @param {Function} fn
   * @param {Boolean} [skipSelf=false] True to skip this node in the traversal
   * @param {Object|Boolean} [options] A boolean for includeFilteredOutRecords, or an options object
   * @param {Boolean} [options.includeFilteredOutRecords] True to also include filtered out records
   * @category Parent & children
   * @returns {Boolean}
   */
  traverseWhile(fn2, skipSelf, options) {
    options = fixTraverseOptions(options);
    const me = this;
    let goOn = skipSelf || fn2.call(me, me) !== false;
    if (goOn) {
      const children = me.getChildren(options);
      if (children == null ? void 0 : children.length) {
        goOn = children.every((child) => child.traverseWhile(fn2, false, options));
      }
    }
    return goOn;
  }
  /**
   * Bubbles up from this node, calling the specified function with each node.
   *
   * @param {Function} fn The function to call for each node
   * @param {Boolean} [skipSelf] True to skip this node in the traversal
   * @category Parent & children
   */
  bubble(fn2, skipSelf = false) {
    let me = this;
    if (!skipSelf) {
      fn2.call(me, me);
    }
    while (me.parent) {
      me = me.parent;
      fn2.call(me, me);
    }
  }
  /**
   * Bubbles up from this node, calling the specified function with each node,
   * while the function returns true.
   *
   * @param {Function} fn The function to call for each node
   * @param {Boolean} [skipSelf] True to skip this node in the traversal
   * @category Parent & children
   * @returns {Boolean}
   */
  bubbleWhile(fn2, skipSelf = false) {
    let me = this, goOn = true;
    if (!skipSelf) {
      goOn = fn2.call(me, me);
    }
    while (goOn && me.parent) {
      me = me.parent;
      goOn = fn2.call(me, me);
    }
    return goOn;
  }
  /**
   * Checks if this model contains another model as one of it's descendants
   *
   * @param {Core.data.Model|String|Number} childOrId child node or id
   * @param {Boolean} [skipSelf=false] True to ignore self in the traversal
   * @category Parent & children
   * @returns {Boolean}
   */
  contains(childOrId, skipSelf = false) {
    if (childOrId && typeof childOrId === "object") {
      childOrId = childOrId.id;
    }
    return !this.traverseWhile((node) => node.id != childOrId, skipSelf);
  }
  getTopParent(all) {
    let result;
    if (all) {
      result = [];
      this.bubbleWhile((t) => {
        result.push(t);
        return t.parent && !t.parent.isRoot;
      });
    } else {
      result = null;
      this.bubbleWhile((t) => {
        result = t;
        return t.parent && !t.parent.isRoot;
      });
    }
    return result;
  }
  /**
   * Append a child record(s) to any current children.
   * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} childRecord Array of records/data or a single
   * record/data to append
   * @param {Boolean} [silent] Pass `true` to not trigger events during append
   * @returns {Core.data.Model|Core.data.Model[]|null}
   * @category Parent & children
   */
  appendChild(childRecord, silent = false) {
    return this.insertChild(childRecord, null, silent);
  }
  /**
   * Insert a child record(s) before an existing child record.
   * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} childRecord Array of records/data or a single
   * record/data to insert
   * @param {Core.data.Model} [before] Optional record to insert before, leave out to append to the end
   * @param {Boolean} [silent] Pass `true` to not trigger events during append
   * @returns {Core.data.Model|Core.data.Model[]|null}
   * @category Parent & children
   */
  insertChild(childRecord, before = null, silent = false, options = {}) {
    var _a2, _b, _c, _d, _e, _f, _g;
    const me = this, returnArray = Array.isArray(childRecord);
    childRecord = ArrayHelper.asArray(childRecord);
    if (typeof before === "number") {
      before = (_b = (_a2 = me.children) == null ? void 0 : _a2[before]) != null ? _b : null;
    }
    if (!silent && !me.stores.every((s) => s.trigger("beforeAdd", {
      records: childRecord,
      parent: me
    }) !== false)) {
      return null;
    }
    childRecord = me.ingestChildren(childRecord);
    const index = (_e = (_d = before == null ? void 0 : before.parentIndex) != null ? _d : (_c = me.children) == null ? void 0 : _c.length) != null ? _e : 0, preResult = (_f = me.beforeInsertChild) == null ? void 0 : _f.call(me, childRecord), inserted = me.internalAppendInsert(childRecord, before, silent, options);
    if (inserted.length) {
      me.convertToParent(silent);
    }
    (_g = me.afterInsertChild) == null ? void 0 : _g.call(me, index, childRecord, preResult, inserted);
    return returnArray || !inserted ? inserted : inserted[0];
  }
  /**
   * Converts a leaf node to a parent node, assigning an empty array as its children
   * @param {Boolean} [silent] Pass `true` to not trigger any event
   * @category Parent & children
   */
  convertToParent(silent = false) {
    const me = this, wasLeaf = me.isLeaf;
    me.meta.isLeaf = false;
    if (!me.children) {
      me.children = [];
    }
    if (wasLeaf && !me.root.isLoading && !silent) {
      me.signalNodeChanged({
        isLeaf: {
          value: false,
          oldValue: true
        }
      });
    }
  }
  signalNodeChanged(changes, stores2 = this.stores) {
    stores2.forEach((s) => {
      s.trigger("update", { record: this, records: [this], changes });
      s.trigger("change", { action: "update", record: this, records: [this], changes });
    });
  }
  tryInsertChild() {
    return this.insertChild(...arguments);
  }
  internalAppendInsert(recordsToInsert, beforeRecord, silent, options) {
    const me = this, { stores: stores2, root, children } = me, { firstStore: rootStore } = root, { parentIdField } = me.constructor, parentId = me.id;
    let isNoop, start, i, newRecordsCloned, oldParentIndices, isMove;
    if (!root.isLoading && rootStore) {
      isMove = {};
      oldParentIndices = [];
      for (i = 0; i < recordsToInsert.length; i++) {
        const newRecord = recordsToInsert[i];
        isMove[newRecord.id] = newRecord.root === root;
        oldParentIndices[i] = newRecord.parentIndex;
      }
    }
    if (beforeRecord && beforeRecord.parent !== me) {
      beforeRecord = null;
    }
    if (children) {
      const insertAt = beforeRecord ? beforeRecord.parentIndex : children.length;
      if (children[start = insertAt] === recordsToInsert[0] || children[start = insertAt - 1] === recordsToInsert[0]) {
        for (isNoop = true, i = 0; isNoop && i < recordsToInsert.length; i++) {
          if (recordsToInsert[i] !== children[start + i]) {
            isNoop = false;
          }
        }
      }
    }
    if (isNoop) {
      return recordsToInsert;
    }
    for (i = 0; i < recordsToInsert.length; i++) {
      const newRecord = recordsToInsert[i], oldParent = newRecord.parent;
      if (rootStore && !root.isLoading) {
        newRecord.traverse((r) => {
          if (r.root === root) {
            isMove[r.id] = true;
          }
        });
      }
      if ((oldParent == null ? void 0 : oldParent.removeChild(newRecord, isMove == null ? void 0 : isMove[newRecord.id], silent, { isInserting: true, ...options })) === false) {
        if (!newRecordsCloned) {
          recordsToInsert = recordsToInsert.slice();
          newRecordsCloned = true;
        }
        recordsToInsert.splice(i--, 1);
      } else {
        newRecord.parent = me;
        newRecord.data[parentIdField] = parentId;
        const { meta } = newRecord;
        if (meta.modified[parentIdField] === parentId && !oldParent) {
          meta.oldParentId = parentId;
        }
        if (oldParent) {
          meta.oldParentId = oldParent.id;
        }
      }
    }
    if (recordsToInsert.length) {
      if (!Array.isArray(children)) {
        me.children = [];
      }
      if (!Array.isArray(me.orderedChildren)) {
        me.orderedChildren = [];
      }
      const insertAt = me.addToChildren(beforeRecord, recordsToInsert, options);
      stores2.forEach((store) => {
        if (!store.isChained) {
          recordsToInsert.forEach((record) => {
            record.joinStore(store);
          });
          store.onNodeAddChild(me, recordsToInsert, insertAt, isMove, silent);
          recordsToInsert.forEach((record, i2) => {
            if (record.meta.oldParentId != null && !(me.inProcessChildren || me.isLoading)) {
              const toSet = {
                [parentIdField]: parentId,
                [me.getDataSource("parentIndex")]: record.parentIndex
              }, wasSet = {}, { modified, oldParentId } = record.meta, oldParentIndex = oldParentIndices[i2];
              delete record.meta.oldParentId;
              if (me.id !== oldParentId) {
                wasSet[parentIdField] = {
                  value: parentId,
                  oldValue: oldParentId
                };
              }
              if (record.parentIndex !== oldParentIndex) {
                wasSet.parentIndex = {
                  value: record.parentIndex,
                  oldValue: oldParentIndex
                };
              }
              if (modified[parentIdField] === me.id) {
                Reflect.deleteProperty(modified, parentIdField);
              } else if (!(parentIdField in modified)) {
                modified[parentIdField] = oldParentId;
              }
              if (isMove[record.id]) {
                const oldParent = store.getById(oldParentId);
                if (oldParent.isLeaf && !silent) {
                  oldParent.signalNodeChanged({
                    isLeaf: {
                      value: true,
                      oldValue: false
                    }
                  }, [store]);
                }
              }
              record.afterChange(toSet, wasSet);
            }
            record.traverse((node) => {
              if (!node.ignoreBag && !node.isLinked) {
                store.updateModifiedBagForRecord(node);
              }
            });
          });
        }
      });
    }
    return recordsToInsert;
  }
  /**
   * Remove a child record. Only direct children of this node can be removed, others are ignored.
   * @param {Core.data.Model|Core.data.Model[]} childRecords The record(s) to remove.
   * @param {Boolean} [isMove] Pass `true` if the record is being moved within the same store.
   * @param {Boolean} [silent] Pass `true` to not trigger events during remove.
   * @privateparam {Object} [options]
   * @privateparam {Object} [options.isInserting] `true` is passed when removal is part of record inserting (acted on by
   * ModelLink)
   * @returns {Core.data.Model[]} All records (including nested children) removed
   * @category Parent & children
   */
  removeChild(childRecords, isMove = false, silent = false, options = {}) {
    var _a2, _b;
    const me = this, allRemovedRecords = [], wasLeaf = me.isLeaf, {
      children,
      stores: stores2
    } = me;
    childRecords = ArrayHelper.asArray(childRecords);
    childRecords = childRecords.filter((r) => r.parent === me);
    if (!silent) {
      for (const store of stores2) {
        if (!store.isChained && store.trigger("beforeRemove", {
          parent: me,
          records: childRecords,
          isMove
        }) === false) {
          return false;
        }
      }
    }
    const preResult = (_a2 = me.beforeRemoveChild) == null ? void 0 : _a2.call(me, childRecords, isMove);
    for (const childRecord of childRecords) {
      const { parentIdField } = childRecord.constructor, { modified } = childRecord.meta, oldParentId = childRecord.parent ? childRecord.parent.id : null;
      if (!(parentIdField in modified) && !childRecord.isLinked) {
        modified[parentIdField] = oldParentId;
      }
      const index = me.removeFromChildren(childRecord, options);
      stores2.forEach((store) => {
        if (!store.isChained) {
          const { isRemoving } = store;
          store.isRemoving = true;
          allRemovedRecords.push(...store.onNodeRemoveChild(me, [childRecord], index, { isMove, silent }));
          store.isRemoving = isRemoving;
        }
      });
      if (!isMove) {
        childRecord.parent = childRecord.parentIndex = childRecord.unfilteredIndex = childRecord.nextSibling = childRecord.previousSibling = null;
        childRecord.data[parentIdField] = null;
      }
    }
    if ((me.unfilteredChildren || children).length === 0 && me.constructor.convertEmptyParentToLeaf.onRemove && !me.isRoot) {
      me.meta.isLeaf = true;
    }
    if (me.isLeaf !== wasLeaf && !silent && !isMove) {
      me.signalNodeChanged({
        isLeaf: {
          value: true,
          oldValue: false
        }
      });
    }
    (_b = me.afterRemoveChild) == null ? void 0 : _b.call(me, childRecords, preResult, isMove);
    return allRemovedRecords;
  }
  clearParentId() {
    const me = this;
    Reflect.deleteProperty(me.data, me.parentIdField);
    Reflect.deleteProperty(me.originalData, me.parentIdField);
    if (me.meta.modified) {
      Reflect.deleteProperty(me.meta.modified, me.parentIdField);
    }
  }
  /**
   * Replaces all child nodes with the new node set.
   * @param {Core.data.Model|Core.data.Model[]} childRecords The new child record set.
   * @returns {Core.data.Model[]}
   * @category Parent & children
   */
  replaceChildren(newChildren) {
    this.clearChildren();
    this.data[this.constructor.childrenField] = newChildren;
    this.processChildren();
    return this.children;
  }
  /**
   * Removes all child nodes from this node.
   * @param {Boolean} [silent=false] Pass `true` to not fire Store events during the remove.
   * @returns {Core.data.Model[]}
   * @category Parent & children
   */
  clearChildren(silent = false) {
    const me = this, { stores: stores2 } = me, children = me.unfilteredChildren || me.children;
    me.children = [];
    me.orderedChildren = [];
    if (children && children !== true) {
      stores2.forEach((store) => {
        if (!store.isChained) {
          store.onNodeRemoveChild(me, children, 0, { unfiltered: true, silent });
        }
      });
      if (me.unfilteredChildren) {
        me.unfilteredChildren = [];
      }
    }
  }
  /**
   * Removes all records from the rootNode
   * @private
   */
  clear() {
    var _a2;
    const me = this, { stores: stores2 } = me, children = (_a2 = me.children) == null ? void 0 : _a2.slice();
    if (!me.isRoot || !children) {
      return;
    }
    for (const store of stores2) {
      if (!store.isChained) {
        if (store.trigger("beforeRemove", {
          parent: me,
          records: children,
          isMove: false,
          removingAll: true
        }) === false) {
          return false;
        }
      }
    }
    me.children.length = 0;
    if (me.unfilteredChildren) {
      me.unfilteredChildren.length = 0;
    }
    stores2.forEach((store) => {
      children.forEach((child) => {
        if (child.stores.includes(store)) {
          child.unjoinStore(store);
        }
        child.parent = child.parentIndex = child.nextSibling = child.previousSibling = null;
      });
      store.storage.suspendEvents();
      store.storage.clear();
      store.storage.resumeEvents();
      store.added.clear();
      store.modified.clear();
      store.trigger("removeAll");
      store.trigger("change", { action: "removeall" });
    });
  }
  updateChildrenIndices(children, indexName, silent = false) {
    let previousSibling = null;
    for (let i = 0; i < children.length; i++) {
      const child = children[i], oldValue = child[indexName];
      if (indexName === "parentIndex" || indexName === "orderedParentIndex") {
        if (oldValue === void 0 || silent) {
          child.setData(indexName, i);
        } else if (oldValue !== i) {
          child.set(indexName, i, true);
        }
      } else {
        child[indexName] = i;
      }
      if (indexName === "parentIndex") {
        child.previousSibling = previousSibling;
        if (previousSibling) {
          previousSibling.nextSibling = child;
        }
        if (i === children.length - 1) {
          child.nextSibling = null;
        }
        previousSibling = child;
      }
    }
  }
  addToChildren(beforeRecord, newRecords, options = {}) {
    var _a2, _b;
    const me = this, configs = [
      [me.children, "parentIndex", beforeRecord],
      [me.unfilteredChildren, "unfilteredIndex", beforeRecord],
      [
        me.orderedChildren,
        "orderedParentIndex",
        (_a2 = options == null ? void 0 : options.orderedBeforeNode) != null ? _a2 : (options == null ? void 0 : options.orderedParentIndex) !== void 0 ? me.orderedChildren[options == null ? void 0 : options.orderedParentIndex] : beforeRecord
      ]
    ];
    for (const config of configs) {
      const [children, indexName, beforeRecord2] = config;
      if (children) {
        const index = beforeRecord2 ? indexName === "orderedParentIndex" ? children.indexOf(beforeRecord2) : beforeRecord2[indexName] : children.length;
        config.push(index);
        children.splice(index, 0, ...newRecords);
        if (!((_b = options == null ? void 0 : options[indexName]) == null ? void 0 : _b.skip)) {
          me.updateChildrenIndices(children, indexName);
        }
      }
    }
    return configs[0][3];
  }
  removeFromChildren(childRecord, options) {
    var _a2;
    const configs = [
      [this.children, "parentIndex"],
      [this.unfilteredChildren, "unfilteredIndex"],
      [this.orderedChildren, "orderedParentIndex"]
    ];
    for (const config of configs) {
      const [children, indexName] = config;
      if (children) {
        const index = children.indexOf(childRecord);
        config.push(index);
        if (index > -1) {
          children.splice(index, 1);
          if (!((_a2 = options == null ? void 0 : options[indexName]) == null ? void 0 : _a2.skip)) {
            this.updateChildrenIndices(children, indexName);
          }
        }
      }
    }
    return configs[0][2];
  }
  /**
   * Iterates orderedChildren array to apply sorting order according to `orderedParentIndex`.
   * Normally sorting is not required because order is maintained on append/insert. But is useful
   * when pasting number of records to restore their original order.
   * @param {Boolean} [deep=true] True to dive into children. False to sort own children.
   * @param {Boolean} [usePreviousOrder=false] Enable to use previous value of `orderedParentIndex`.
   * @returns {Set} Returns Set of moved nodes which require WBS update
   * @private
   */
  sortOrderedChildren(deep = true, usePreviousOrder = false) {
    const movedNodes = [];
    if (!this.isLeaf) {
      this.orderedChildren.sort((a, b) => {
        var _a2, _b;
        if (usePreviousOrder) {
          const aPrevIndex = (_a2 = a.meta.modified.orderedParentIndex) != null ? _a2 : a.orderedParentIndex, bPrevIndex = (_b = b.meta.modified.orderedParentIndex) != null ? _b : b.orderedParentIndex, result = aPrevIndex - bPrevIndex;
          if (result !== 0) {
            movedNodes.push(a);
            movedNodes.push(b);
          }
          return result;
        } else {
          return a.orderedParentIndex - b.orderedParentIndex;
        }
      });
      if (deep) {
        this.orderedChildren.forEach((child) => {
          movedNodes.push(...child.sortOrderedChildren(deep, usePreviousOrder));
        });
      }
      this.updateChildrenIndices(this.orderedChildren, "orderedParentIndex", true);
    }
    return new Set(movedNodes);
  }
  unjoinStore(store, isReplacing = false) {
    var _a2;
    const me = this;
    if (me.unfilteredChildren) {
      me.children = me.unfilteredChildren.slice();
      me.unfilteredChildren = null;
    }
    (_a2 = super.unjoinStore) == null ? void 0 : _a2.call(this, store, isReplacing);
  }
};

// ../Core/lib/Core/data/mixin/ModelLink.js
var propertyOverrides = {
  id: 1,
  stores: 1,
  parentIndex: 1,
  parent: 1,
  previousSibling: 1,
  nextSibling: 1,
  unfilteredIndex: 1
};
var proxyConfig = {
  get(target, prop) {
    if (prop === "proxyMeta") {
      return this.proxyMeta;
    }
    if (prop === "constructor") {
      return target.constructor;
    }
    if (prop === "setData") {
      return this.setDataOverride;
    }
    if (prop === "set") {
      return this.setOverride;
    }
    if (propertyOverrides[prop]) {
      return this.proxyMeta.data[prop];
    }
    return Reflect.get(target, prop, this.proxyRecord);
  },
  set(target, prop, value) {
    if (propertyOverrides[prop]) {
      this.proxyMeta.data[prop] = value;
    } else {
      target[prop] = value;
    }
    return true;
  },
  // Override setData & set to reroute parentIndex updates
  setDataOverride(toSet, value) {
    if (toSet === "parentIndex") {
      this.proxyMeta.data.parentIndex = value;
    } else {
      this.proxyMeta.originalRecord.setData(toSet, value);
    }
  },
  setOverride(field2, value, ...args) {
    if (field2 === "parentIndex") {
      this.proxyMeta.data.parentIndex = value;
    } else {
      this.proxyMeta.originalRecord.set(field2, value, ...args);
    }
  }
};
var ModelLink_default = (Target) => {
  var _a2;
  return _a2 = class extends (Target || Base) {
    /**
     * Creates a proxy record (using native Proxy) linked to this record (the original). The proxy records shares most
     * data with the original, except for its `id` (which is always generated), and ordering fields such as
     * `parentIndex` and `parentId` etc.
     *
     * Any change to the proxy record will be reflected on the original, and vice versa. A proxy record is not meant to
     * be persisted, only the original record should be persisted. Thus, proxy records are not added to stores change
     * tracking (added, modified and removed records).
     *
     * Removing the original record removes all proxies.
     *
     * Creating a proxy record allows a Store to seemingly contain the record multiple times, something that is
     * otherwise not possible. It also allows a record to be used in both a tree store and in a flat store.
     *
     * <div class="note">Note that not all UI features support linked records</div>
     *
     * @returns {Proxy} Proxy record linked to the original record
     * @category Misc
     */
    link() {
      if (this.isLinked) {
        return this.$original.link();
      }
      const me = this, useConfig = {
        ...proxyConfig,
        // Data not shared with the original record
        proxyMeta: {
          originalRecord: me,
          data: {
            id: `${me.id}_link_${StringHelper.generateUUID()}`,
            stores: []
          }
        }
      }, proxyRecord = new Proxy(me, useConfig);
      useConfig.proxyRecord = proxyRecord;
      (me.meta.linkedRecords || (me.meta.linkedRecords = [])).push(proxyRecord);
      return proxyRecord;
    }
    /**
     * Is this record linked to another record?
     * @member {Boolean}
     * @readonly
     * @category Misc
     */
    get isLinked() {
      var _a3;
      return Boolean((_a3 = this.proxyMeta) == null ? void 0 : _a3.originalRecord);
    }
    /**
     * Are other records linked to this record?
     * @member {Boolean}
     * @readonly
     * @category Misc
     */
    get hasLinks() {
      return Boolean(!this.proxyMeta && this.$links.length);
    }
    // Logic to remove a link shared between removing in a flat store and a tree store
    removeLink(link, records = null, silent = false) {
      if (link.hasLinks) {
        for (const linked of link.$links.slice()) {
          if (records) {
            ArrayHelper.include(records, linked);
          } else {
            linked.remove(silent);
          }
        }
      } else if (link.isLinked) {
        ArrayHelper.remove(link.$original.$links, link);
      }
    }
    // Overrides beforeRemove in Model, to remove all linked records when original record is removed.
    beforeRemove(records) {
      this.removeLink(this, records);
    }
    // Overrides removeChild in TreeNode, to remove the original node and all linked nodes when either a linked or
    // original node is removed.
    removeChild(childRecords, isMove, silent, options) {
      if (!(options == null ? void 0 : options.isInserting)) {
        childRecords = ArrayHelper.asArray(childRecords);
        for (const child of childRecords) {
          this.removeLink(child, null, silent);
        }
      }
      return super.removeChild(childRecords, isMove, silent, options);
    }
    // Convenience getter for code keying by id that needs to work with both link and original
    get $originalId() {
      return this.$original.id;
    }
    // Convenience getter to retrieve linked records
    get $links() {
      var _a3;
      return (_a3 = this.meta.linkedRecords) != null ? _a3 : [];
    }
  }, __publicField(_a2, "$name", "ModelLink"), _a2;
};

// ../Core/lib/Core/mixin/Factoryable.js
var { defineProperty: defineProperty4 } = Reflect;
var ownerSymbol = Symbol("owner");
var typeSplitRe = /[\s,]+/;
var Factoryable_default = (Target) => class Factoryable extends (Target || Base) {
  static get $name() {
    return "Factoryable";
  }
  static get declarable() {
    return [
      /**
       * This property getter returns options that control the factory process. This property getter must be
       * defined by the class that mixes in `Factoryable` in order to initialize the factory properly.
       * ```
       *  static get factoryable() {
       *      return {
       *          defaultType : 'default'
       *      };
       *  }
       * ```
       * If there are no special options to provide, this method can return nothing (`undefined`):
       * ```
       *  static get factoryable() {
       *      // initialize the factory with all default options
       *  }
       * ```
       * @static
       * @member {Object} factoryable
       * @property {Boolean} [factoryable.caseless=true] Specify `false` to use case-sensitive type names. The
       * default is to ignore case.
       * @property {String} [factoryable.defaultType=null] The default type to create when a config object has
       * no `typeKey` property.
       * @property {Function|Function[]} [factoryable.extends] One or more classes that mix in `Factoryable` to
       * use for resolving type names when a type name is not found in this factory.
       * @property {String} [factoryable.typeKey='type'] The name of the property in a config object that holds
       * the type name.
       * @internal
       */
      "factoryable",
      /**
       * One or more additional type name aliases for this class. This can be useful for renaming and maintaining
       * a previous type name.
       * ```
       *  class Fit extends Layout {
       *      static get type() {
       *          return 'fit';
       *      }
       *
       *      static get alias() {
       *          return 'fill';  // deprecated type name (now known as 'fit')
       *      }
       *  }
       * ```
       * @static
       * @member {String|String[]} alias
       * @internal
       */
      "alias",
      /**
       * The (canonical) type name for this class by which instances can be created using the static
       * {@link #function-create-static create()} method.
       * @static
       * @member {String} type
       */
      "type"
    ];
  }
  /**
   * Registers a class (`cls`) associated with the given `type`.
   * @param {String|String[]} type A string, array of strings or a comma-separated string containing the type names
   * for the specified `cls` class.
   * @param {Function} cls The class (constructor function)
   * @param {Boolean} [replace] Pass `true` to overwrite existing registered types. Otherwise, this method will throw
   * an exception if the `type` is already registered with this factory.
   * @internal
   */
  static register(type, cls2, replace = globalThis.__BRYNTUM_EXAMPLE) {
    const { factoryable } = this.initClass(), { caseless, registry: registry2 } = factoryable, types = StringHelper.split(type, typeSplitRe);
    for (let lower, name, i = 0; i < types.length; ++i) {
      name = types[i];
      lower = caseless ? name.toLowerCase() : name;
      if (!replace && lower in registry2) {
        throw new Error(`Type "${name}" already registered with ${factoryable.class.name} factory`);
      }
      registry2[name] = registry2[lower] = cls2.initClass();
    }
  }
  /**
   * Returns `true` if the passed instance is of the passed type or of a derived class.
   * @param {Object} instance The object to test.
   * @param {String} type The type to test against
   */
  static isA(instance, type) {
    return this.isType(instance, type, true);
  }
  /**
   * Returns `true` if the passed instance is of the passed type.
   * @param {Object} instance The object to test.
   * @param {String} type The type to test against
   * @param {Boolean} [deep] Pass `true` to return `true` if the class is a subclass of the passed type.
   */
  static isType(instance, type, deep) {
    const { factoryable } = this, { caseless, registry: registry2 } = factoryable, typeCls = registry2[caseless ? type.toLowerCase() : type];
    if (typeCls) {
      if (deep) {
        return instance instanceof typeCls;
      }
      return instance.constructor === typeCls;
    }
    return false;
  }
  static setupAlias(cls2) {
    cls2.register(cls2.alias, cls2);
  }
  static setupFactoryable(cls2, meta) {
    const superClass = meta.super.class;
    let { factoryable } = cls2;
    factoryable = {
      caseless: true,
      defaultType: null,
      extends: superClass.factoryable ? [superClass] : null,
      typeKey: "type",
      ...factoryable
    };
    factoryable.class = cls2;
    factoryable.registry = /* @__PURE__ */ Object.create(null);
    if (factoryable.extends && !Array.isArray(factoryable.extends)) {
      factoryable.extends = [factoryable.extends];
    }
    defineProperty4(cls2, "factoryable", {
      get() {
        return factoryable;
      }
    });
  }
  static setupType(cls2, meta) {
    const { type } = cls2;
    cls2.register(type, cls2, meta.replaceType);
    defineProperty4(cls2.prototype, "type", {
      value: type
    });
  }
  /**
   * Creates an instance from this factory, given the type name or a config object.
   * @param {String|Object} config The type name string or config object.
   * @param {String|Function|Object} [options] Creation options (for details see {@link #function-reconfigure-static}).
   * @returns {Object}
   */
  static create(config, options) {
    return this.reconfigure(null, config, options);
  }
  /**
   * Reconfigures an optional existing instance based on the provided config and returns the correctly configured
   * instance. This will be the `existingInstance` if the `config` does not specify a different type.
   *
   * If `config` is `null` (or simply falsy), this method will destroy the `existingInstance` (if any) and return
   * `null`.
   *
   * If there is no `existingInstance`, the config must specify a type. That is, it must be a string (the type name)
   * or an object containing a `type` property, the `defaultType` must be provided or the factory itself must have
   * a `defaultType` specified (see {@link #property-factoryable-static}).
   *
   * When an `existingInstance` is provided and a type is specified, the instance will be reconfigured via `setConfig`
   * if it is of that type. Otherwise, the `existingInstance` is destroyed (if it is owned by the `options.owner`)
   * and a new instance of the correct type is created.
   *
   * @param {Object} existingInstance The instance to reconfigure. This can be `null`.
   *
   * @param {String|Object} config The type name string or config object.
   *
   * @param {String|Function|Object} [options] Additional options to control the reconfiguration process. If this
   * value is a string or a class constructor, it treated as `options.type`. If this value is a class instance, it
   * is used as the `options.owner`. If this is a function, it is treated as `options.setup`. NOTE: functions declared
   * using the `function` keyword are equivalent to class constructors. Use an arrow function or a class method to
   * avoid this when a `setup` function is intended.
   *
   * @param {String|Function} [options.type] The default type to use if the `config` object does not specify a type.
   *
   * @param {Object} [options.owner] The owner of any created instances. If the `existingInstance` is being replaced,
   * this value determines if the instance will be destroyed.
   *
   * @param {Object} [options.defaults] A config object of default values to use when creating a new instance.
   *
   * @param {Function|String} [options.setup] A function or the name of a method (on the `options.owner`) to call
   * prior to creating a new instance. It is passed the config object that will be used to create the instance. If a
   * truthy value is returned, that value is passed to the constructor instead of the provided config object.
   *
   * @param {Function|String} [options.transform] A function or the name of a method (on the `options.owner`) to call
   * with the raw config object prior to processing and the value it returns replaces the raw value. This function is
   * used to transform strings or arrays (for example) into proper config objects.
   *
   * @param {Function|String} [options.cleanup] A function or the name of a method (on the `options.owner`) to call
   * prior to destroying the `existingInstance`. The `existingInstance` is passed as the sole argument.
   *
   * @returns {Object} The reconfigured instance (either `existingInstance` or a new instance of the desired type)
   */
  static reconfigure(existingInstance, config, options) {
    const me = this, { factoryable } = me, { typeKey } = factoryable;
    let defaultType = options, cleanup, defaults, mergeType, owner, prepared, setup, t, transform, type;
    if (options && !ObjectHelper.isClass(options)) {
      defaultType = null;
      t = typeof options;
      if (t === "function") {
        setup = options;
      } else if (t === "string") {
        defaultType = options;
      } else if (ObjectHelper.isObject(options)) {
        cleanup = options.cleanup;
        defaults = options.defaults;
        owner = options.owner;
        setup = options.setup;
        defaultType = options.type;
        transform = options.transform;
      } else {
        owner = options;
      }
    }
    if (transform) {
      config = typeof transform === "string" ? owner[transform](config) : transform(config);
    }
    type = config;
    if (typeof type === "string") {
      config = {};
    } else if (config) {
      if (config === true) {
        config = {};
      }
      if (!ObjectHelper.isObject(config)) {
        if (owner && config !== existingInstance && (existingInstance == null ? void 0 : existingInstance[ownerSymbol]) === owner) {
          typeof cleanup === "string" ? owner[cleanup](existingInstance) : cleanup == null ? void 0 : cleanup(existingInstance);
          existingInstance.destroy();
        }
        return config;
      }
      type = config[typeKey];
    }
    type = type && me.resolveType(type);
    if (existingInstance) {
      if (config && (!type || existingInstance.constructor === type)) {
        if (typeKey in config) {
          config = ObjectHelper.assign({}, config);
          delete config[typeKey];
        }
        existingInstance.setConfig(config);
        return existingInstance;
      }
      if (owner && existingInstance[ownerSymbol] === owner) {
        typeof cleanup === "string" ? owner[cleanup](existingInstance) : cleanup == null ? void 0 : cleanup(existingInstance);
        existingInstance.destroy();
      }
    }
    if (config) {
      if (defaults) {
        if (!(mergeType = type)) {
          if (!(mergeType = defaults[typeKey] || defaultType || factoryable.defaultType)) {
            throw new Error(`No default mergeType defined for ${factoryable.class.name} factory`);
          }
          mergeType = me.resolveType(mergeType);
        }
        if (mergeType) {
          config = mergeType.mergeConfigs(defaults, config);
        }
      }
      if (setup) {
        prepared = typeof setup === "string" ? owner[setup](config, type, defaults) : setup(config, type, defaults);
        if (prepared === null) {
          return prepared;
        }
        config = prepared || config;
      }
      if (!type) {
        if (!(type = config[typeKey] || defaultType || factoryable.defaultType)) {
          throw new Error(`No default type defined for ${factoryable.class.name} factory`);
        }
        type = me.resolveType(type);
      }
      if (defaults && !mergeType) {
        config = type.mergeConfigs(defaults, config);
      }
      if (typeKey in config) {
        config = ObjectHelper.assign({}, config);
        delete config[typeKey];
      }
      config = new type(config);
      if (owner) {
        config[ownerSymbol] = owner;
      }
    }
    return config || null;
  }
  /**
   * This method returns the constructor of the class registered for the given type name.
   * @param {String} type The type name to look up.
   * @param {Boolean} [optional] Pass `true` to return `null` if `type` is not found instead of throwing an exception.
   * @returns {Function}
   */
  static resolveType(type, optional) {
    if (typeof type !== "string") {
      return type;
    }
    const { factoryable } = this, bases = factoryable.extends;
    let result = factoryable.registry[factoryable.caseless ? type.toLowerCase() : type], i;
    for (i = 0; !result && bases && i < bases.length; ++i) {
      result = bases[i].resolveType(
        type,
        /* optional = */
        true
      );
    }
    if (!result && !optional) {
      throw new Error(`Invalid type name "${type}" passed to ${factoryable.class.name} factory`);
    }
    return result;
  }
};

// ../Core/lib/Core/data/field/DataField.js
var { getOwnPropertyDescriptor: getOwnPropertyDescriptor2 } = Reflect;
var DataField = class extends Base.mixin(Factoryable_default) {
  static get $name() {
    return "DataField";
  }
  static get type() {
    return "auto";
  }
  static get factoryable() {
    return {
      defaultType: "auto"
    };
  }
  static get prototypeProperties() {
    return {
      /**
       * The name of the field.
       * @config {String} name
       */
      /**
       * The label text for a form item generated for this field. This is also used to create
       * a column header for a {@link #config-column} for this field.
       * @config {String} label
       */
      /**
       * A column config object for a column to display this field in a grid. For simple, atomic
       * data types, such as `date`, `string`, `boolean`, `number` and `integer`, this is optional
       * and the appropriate column type can be inferred.
       *
       * This also provides default values for column configuration if a configured column definition
       * for a grid lacks a property.
       *
       * For complex fields, such as identifiers which link to other records, a more capable
       * column type may be specified, for example a `type : `number'` field may be configured
       * with
       *
       * ```javascript
       * column : 'percent'
       * ```
       * or
       * ```javascript
       * column : {
       *     type : 'percent',
       *     width : 100
       * }
       * ```
       * if it represents a percentage vaue and needs appropriate rendering and editing.
       * @config {String|Object} column
       */
      /**
       * A config object for a widget to edit this field in a form. For simple, atomic
       * data types, such as `date`, `string`, `boolean`, `number` and `integer`, this is optional
       * and the appropriate input widget type can be inferred.
       *
       * For complex fields, such as identifiers which link to other records, a more capable
       * widget may be specified.
       * @config {String|InputFieldConfig} editor
       * @private
       */
      /**
       * A function that compares two values and returns a value < 0 if the first is less than the second, or 0
       * if the values are equal, or a value > 0 if the first is greater than the second.
       * @config {Function}
       * @default
       */
      compare: null,
      /**
       * A function that compares two objects or records using the `compare` function on the properties of each
       * objects based on the `name` of this field.
       * @config {Function}
       * @default
       * @internal
       */
      compareItems: null,
      /**
       * The property in a record's data object that contains the field's value.
       * Defaults to the field's `name`.
       * @config {String}
       */
      dataSource: null,
      /**
       * The default value to assign to this field in a record if no value is provided.
       * @config {*} defaultValue
       */
      /**
       * Setting to `true` will ensure this field is included in any update/insert request payload
       * when a Store / Project / CrudManager performs a request.
       * @config {Boolean}
       * @default
       */
      alwaysWrite: false,
      /**
       * Setting to `false` indicates that `null` is not a valid value.
       * @config {Boolean}
       * @default
       */
      nullable: true,
      /**
       * The value to return from {@link #function-print} for a `null` or `undefined` value.
       * @config {String}
       * @default
       */
      nullText: null,
      /**
       * The value to replace `null` when the field is not `nullable`.
       * @config {*}
       * @default
       */
      nullValue: void 0,
      /**
       * Set to `false` to exclude this field when saving records to a server.
       * @config {Boolean}
       * @default
       */
      persist: true,
      /**
       * Set to `true` for the field's set accessor to ignore attempts to set this field.
       * @config {Boolean}
       * @default
       */
      readOnly: false,
      /**
       * By default, defined {@link Core.data.Model} fields may be used to create a grid column
       * suitable for diplaying that field in a grid cell. Some fields may not be suitable for
       * features which automatically generate columns for view. These fields are created using
       * `internal : true`. Some examples are the `expanded` and `rowHeight` fields which are used
       * internally.
       * @config {Boolean}
       * @default
       */
      internal: false,
      useProp: null
    };
  }
  /**
   * The class that first defined this field. Derived classes that override a field do not change this property.
   * @member {Core.data.Model} definedBy
   * @private
   * @readonly
   */
  /**
   * The class that most specifically defined this field. Derived classes that override a field set this property to
   * themselves.
   * @member {Core.data.Model} owner
   * @private
   * @readonly
   */
  // NOTE: Since we create lots of instances, they have no life cycle (they are not destroyed) and are readonly after
  // creation, this class does not use configurable.
  construct(config) {
    const me = this;
    if (config) {
      me.name = config.name;
      Object.assign(me, config);
    }
    if (me.compare) {
      me.compareItems = (itemA, itemB) => me.compare(itemA == null ? void 0 : itemA[me.name], itemB == null ? void 0 : itemB[me.name]);
    }
  }
  /**
   * This method transforms a data value into the desired form for storage in the record's data object.
   *
   * ```javascript
   * export default class Task extends TaskModel {
   *    static get fields() {
   *        return [
   *            {
   *                name    : 'status',
   *                convert : (value, data) => {
   *                    if (value >= 100) {
   *                        return 'done';
   *                    }
   *                    else if (value > 0) {
   *                        return 'started';
   *                    }
   *                }
   *            }
   *        ];
   *    }
   * }
   * ```
   *
   * @method convert
   * @param {*} value The value to convert for storage in a record.
   * @param {Object} data The raw record data object
   * @returns {*} The converted value.
   */
  /**
   * This method transforms a data value into the desired form for transmitting to a server.
   * @method serialize
   * @param {*} value The value to serialize
   * @param {Core.data.Model} record The record that contains the value being serialized.
   * @returns {*} The serialized value.
   */
  /**
   * This optional method is called when setting a data value on a record.
   * @method set
   * @param {*} value The value to set
   * @param {Object} data The records future or current data object to set value to
   * @param {Core.data.Model} record The record that owns or will own the data object
   * @internal
   */
  /**
   * This optional method is called when a record using this field is created.
   * @method init
   * @param {Core.data.Model} record The record being created
   * @internal
   */
  /**
   * Create getter and setter functions for the specified field name under the specified key.
   * @internal
   */
  defineAccessor(target, force) {
    const { name, dataSource } = this;
    if (!force && name in target && target.$meta.hierarchy.some((current) => {
      var _a2;
      return ((_a2 = getOwnPropertyDescriptor2(current.prototype, name)) == null ? void 0 : _a2.enumerable) === false;
    })) {
      return;
    }
    Reflect.defineProperty(target, name, {
      configurable: true,
      // To allow removing it later
      enumerable: true,
      // no arrow functions here, need `this` to change to instance
      get: this.complexMapping ? function() {
        return this.complexGet(name, dataSource);
      } : function() {
        if (this.batching && name in this.meta.batchChanges) {
          return this.meta.batchChanges[name];
        }
        return dataSource in this.data ? this.data[dataSource] : this.data[name];
      },
      // no arrow functions here, need `this` to change to instance
      set(value) {
        const field2 = this.$meta.fields.map[name];
        if (!(field2 && field2.readOnly)) {
          this.set(name, value);
        }
      }
    });
  }
  /**
   * Compares two values for this field and returns `true` if they are equal, and `false` if not.
   * @param {*} first The first value to compare for equality.
   * @param {*} second The second value to compare for equality.
   * @returns {Boolean} `true` if `first` and `second` are equal.
   */
  isEqual(first, second) {
    return ObjectHelper.isEqual(first, second);
  }
  /**
   * Returns the given field value as a `String`. If `value` is `null` or `undefined`, the value specified by
   * {@link #config-nullText} is returned.
   * @param {*} value The value to convert to a string.
   * @returns {String}
   */
  print(value) {
    return value == null ? this.nullText : this.printValue(value);
  }
  /**
   * Returns the given, non-null field value as a `String`.
   * @param {*} value The value to convert to a string (will not be `null` or `undefined`).
   * @returns {String}
   * @protected
   */
  printValue(value) {
    return String(value);
  }
};
DataField._$name = "DataField";

// ../Core/lib/Core/data/field/ArrayDataField.js
var ArrayDataField = class extends DataField {
  isEqual(a, b) {
    return a === b;
  }
  getAt(record, index) {
    return record.get(this.name)[index];
  }
};
__publicField(ArrayDataField, "$name", "ArrayDataField");
__publicField(ArrayDataField, "type", "array");
ArrayDataField.initClass();
ArrayDataField._$name = "ArrayDataField";

// ../Core/lib/Core/data/field/BooleanDataField.js
var BooleanDataField = class extends DataField {
  static get $name() {
    return "BooleanDataField";
  }
  static get type() {
    return "boolean";
  }
  static get alias() {
    return "bool";
  }
  static get prototypeProperties() {
    return {
      /**
       * The value to replace `null` when the field is not `nullable`.
       * @config {Boolean}
       * @default
       */
      nullValue: false
    };
  }
  isEqual(first, second) {
    if (first == null && second == null) {
      return true;
    }
    return super.isEqual(first, second);
  }
  convert(value) {
    var _a2;
    if (value == null) {
      return this.nullable ? value : this.nullValue;
    }
    if (((_a2 = value.toLowerCase) == null ? void 0 : _a2.call(value)) === "false") {
      return false;
    }
    return Boolean(value);
  }
};
BooleanDataField.initClass();
BooleanDataField._$name = "BooleanDataField";

// ../Core/lib/Core/data/field/DateDataField.js
var DateDataField = class extends DataField {
  static get $name() {
    return "DateDataField";
  }
  static get type() {
    return "date";
  }
  static get prototypeProperties() {
    return {
      /**
       * The format of the date field.
       *
       * See {@link Core.helper.DateHelper DateHelper} for details.
       * @config {String} format
       * @default DateHelper.defaultFormat
       */
      format: null
    };
  }
  convert(value) {
    if (value == null) {
      if (!this.nullable) {
        value = this.nullValue;
      }
    } else if (value === "now") {
      value = /* @__PURE__ */ new Date();
    } else if (!(value instanceof Date)) {
      value = DateHelper.parse(value, this.format || DateHelper.defaultParseFormat);
      if (!value || isNaN(value)) {
        value = void 0;
      }
    }
    return value;
  }
  serialize(value) {
    if (value instanceof Date) {
      value = DateHelper.format(value, this.format || DateHelper.defaultFormat);
    }
    return value;
  }
  printValue(value) {
    return DateHelper.format(value, this.format || DateHelper.defaultFormat);
  }
};
DateDataField.initClass();
DateDataField._$name = "DateDataField";

// ../Core/lib/Core/data/field/StringDataField.js
var StringDataField = class extends DataField {
  static get $name() {
    return "StringDataField";
  }
  static get type() {
    return "string";
  }
  static get prototypeProperties() {
    return {
      /**
       * The value to replace `null` when the field is not `nullable`.
       * @config {String}
       * @default
       */
      nullValue: ""
    };
  }
  convert(value) {
    return value == null ? this.nullable ? value : this.nullValue : String(value);
  }
};
StringDataField.initClass();
StringDataField._$name = "StringDataField";

// ../Core/lib/Core/data/field/DurationUnitDataField.js
var DurationUnitDataField = class extends StringDataField {
  static get $name() {
    return "DurationUnitDataField";
  }
  static get type() {
    return "durationunit";
  }
  isEqual(first, second) {
    return DateHelper.compareUnits(first, second) === 0;
  }
};
DurationUnitDataField.initClass();
DurationUnitDataField._$name = "DurationUnitDataField";

// ../Core/lib/Core/data/field/IntegerDataField.js
var IntegerDataField = class extends DataField {
  static get $name() {
    return "IntegerDataField";
  }
  static get type() {
    return "integer";
  }
  static get alias() {
    return "int";
  }
  static get prototypeProperties() {
    return {
      /**
       * The value to replace `null` when the field is not `nullable`.
       * @config {Number}
       * @default
       */
      nullValue: 0,
      /**
       * The `Math` method to use to ensure fractional component is removed.
       * @config {'round'|'floor'|'ceil'}
       * @default
       */
      rounding: "round"
    };
  }
  convert(value) {
    return value == null ? this.nullable ? value : this.nullValue : Math[this.rounding](Number(value));
  }
};
IntegerDataField.initClass();
IntegerDataField._$name = "IntegerDataField";

// ../Core/lib/Core/data/field/ModelDataField.js
var ModelDataField = class extends DataField {
  static get $name() {
    return "ModelDataField";
  }
  static get type() {
    return "model";
  }
  static get prototypeProperties() {
    return {
      complexMapping: true
    };
  }
  isEqual(first, second) {
    return first && second && second instanceof first.constructor && second.id == first.id;
  }
};
ModelDataField.initClass();
ModelDataField._$name = "ModelDataField";

// ../Core/lib/Core/data/field/NumberDataField.js
var NumberDataField = class extends DataField {
  static get $name() {
    return "NumberDataField";
  }
  static get type() {
    return "number";
  }
  static get alias() {
    return "float";
  }
  static get prototypeProperties() {
    return {
      /**
       * The value to replace `null` when the field is not `nullable`.
       * @config {Number}
       * @default
       */
      nullValue: 0,
      /**
       * The numeric precision of this field. Values are rounded to the specified number of digits. If `null`,
       * the default, no rounding is performed.
       * @config {Number}
       * @default
       */
      precision: null
    };
  }
  isEqual(first, second) {
    return isNaN(Number(first)) && isNaN(Number(second)) || super.isEqual(first, second);
  }
  convert(value) {
    if (value == null) {
      return this.nullable ? value : this.nullValue;
    }
    value = Number(value);
    if (isNaN(value)) {
      return;
    }
    let scale = this.precision;
    if (scale) {
      scale = 10 ** scale;
      value = Math.round(value * scale) / scale;
    } else if (scale === 0) {
      value = Math.round(value);
    }
    return value;
  }
};
NumberDataField.initClass();
NumberDataField._$name = "NumberDataField";

// ../Core/lib/Core/data/field/ObjectDataField.js
var ObjectDataField = class extends DataField {
  static get $name() {
    return "ObjectDataField";
  }
  static get type() {
    return "object";
  }
  static get prototypeProperties() {
    return {
      complexMapping: true
    };
  }
};
ObjectDataField.initClass();
ObjectDataField._$name = "ObjectDataField";

// ../Core/lib/Core/data/field/StoreDataField.js
var StoreDataField = class extends DataField {
  /**
   * Store class to use when creating the store.
   *
   * ```javascript
   * class TodoStore extends Store {
   *     ...
   * }
   *
   * const task = new Store({
   *     static fields = [
   *         { type : 'store', name: 'todoItems', storeClass : TodoStore }
   *     ]
   * });
   * ```
   *
   * @config {Class} storeClass
   * @typings {typeof Store}
   */
  /**
   * Model class to use for the store (can also be configured as usual on the store class, this config is for
   * convenience).
   *
   * ```javascript
   * class TodoItem extends Model {
   *   ...
   * }
   *
   * const task = new Store({
   *     static fields = [
   *         { type : 'store', name: 'todoItems', storeClass : Store, modelClass : TodoItem }
   *     ]
   * });
   * ```
   *
   * @config {Class} modelClass
   * @typings {typeof Model}
   */
  /**
   * Optional store configuration object to apply when creating the store.
   *
   * ```javascript
   * const task = new Store({
   *     static fields = [
   *         {
   *             type       : 'store',
   *             name       : 'todoItems',
   *             storeClass : Store
   *             store      : {
   *                  syncDataOnLoad : false
   *             }
   *         }
   *     ]
   * });
   * ```
   *
   * @config {StoreConfig} store
   */
  // Initializer, called when creating a record. Sets up the store and populates it with any initial data
  init(data, record) {
    var _a2;
    const me = this, storeName = `${me.name}Store`, config = { skipStack: true, syncDataOnLoad: true };
    if (me.store) {
      ObjectHelper.assign(config, me.store);
    }
    if (me.modelClass) {
      config.modelClass = me.modelClass;
    }
    (_a2 = record[`init${StringHelper.capitalize(storeName)}`]) == null ? void 0 : _a2.call(record, config);
    if (!config.storeClass && !me.storeClass) {
      throw new Error(`Field '${me.name}' with type 'store' must have a storeClass configured`);
    }
    const store = record.meta[storeName] = new (config.storeClass || me.storeClass)(config);
    if (me.complexMapping) {
      ObjectHelper.setPath(data, me.dataSource, store);
    } else {
      data[me.dataSource] = store;
    }
    store.verifyNoGeneratedIds = false;
    store.usesId = !store.count || !store.every((record2) => record2.hasGeneratedId);
    store.$currentValue = me.getValue(store);
    store.ion({
      change: ({ action }) => {
        const value = me.getValue(store);
        if (!store.$isSettingStoreFieldData) {
          const oldPreserveCurrentDataset = store.$preserveCurrentDataset;
          store.$preserveCurrentDataset = me.subStore && (action === "update" || action === "remove" || action === "add");
          me.$isUpdatingRecord = true;
          record.set(me.name, value);
          me.$isUpdatingRecord = false;
          store.$preserveCurrentDataset = oldPreserveCurrentDataset;
        }
        store.$currentValue = value;
      }
    });
  }
  // Called when setting a new value to the field on a record
  set(value, data, record) {
    var _a2, _b;
    const me = this, storeName = `${me.name}Store`, { [storeName]: store } = record.meta;
    if (!store) {
      record.meta.initableValues.set(me, value);
      return false;
    }
    if (store.$isSettingStoreFieldData) {
      return;
    }
    store.$isSettingStoreFieldData = true;
    value = (_b = (_a2 = record[`process${StringHelper.capitalize(storeName)}Data`]) == null ? void 0 : _a2.call(record, value, record)) != null ? _b : value;
    if (!store.$preserveCurrentDataset) {
      store.data = value;
    }
    store.$isSettingStoreFieldData = false;
    store.usesId = !store.count || !store.every((record2) => record2.hasGeneratedId);
  }
  serialize(value, record) {
    const store = record.meta[`${this.name}Store`];
    return this.$isUpdatingRecord ? this.getValue(store) : store.$currentValue;
  }
  // Extract persistable values, optionally including id depending on if ids are used
  getValue(store) {
    return store.allRecords.map((r) => {
      const data = r.persistableData;
      if (!store.usesId) {
        delete data.id;
      }
      return data;
    });
  }
  isEqual(a, b) {
    if (a == null ? void 0 : a.isStore) {
      a = a.$currentValue;
    }
    if (b == null ? void 0 : b.isStore) {
      b = b.$currentValue;
    }
    return ObjectHelper.isDeeplyEqual(a, b);
  }
  // Cloned value to be able to restore it later using STM
  getOldValue(record) {
    const store = record.meta[`${this.name}Store`];
    return store ? ObjectHelper.clone(store.$currentValue) : null;
  }
  getAt(record, index) {
    const store = record.meta[`${this.name}Store`];
    return store == null ? void 0 : store.getAt(index);
  }
};
__publicField(StoreDataField, "$name", "StoreDataField");
__publicField(StoreDataField, "type", "store");
StoreDataField.initClass();
StoreDataField._$name = "StoreDataField";

// ../Core/lib/Core/data/Model.js
var nestedRe = new RegExp(/^(.*?)\.(.*)/);
var arrayRe = /(.*)\[(.*)]\.?(.*)/;
var { defineProperty: defineProperty5 } = Reflect;
var { hasOwn: hasOwn3 } = ObjectHelper;
var _undefined = void 0;
var internalProps = {
  children: 1,
  data: 1,
  meta: 1
};
var abbreviationFields = [
  "name",
  "title",
  "text",
  "label",
  "description"
];
var fieldDataTypes = {
  boolean: 1,
  number: 1,
  date: 1,
  object: 1
};
var fieldsOrder = {
  parentId: 1,
  $PhantomId: 2,
  id: 3
};
var _Model = class extends Base.mixin(ModelStm_default, TreeNode_default, ModelLink_default) {
  static get $name() {
    return "Model";
  }
  static get declarable() {
    return [
      /**
       * Array of defined fields for this model class. Subclasses add new fields by implementing this static
       * getter:
       *
       * ```javascript
       * // Model defining two fields
       * class Person extends Model {
       *     static get fields() {
       *         return [
       *             { name : 'username', defaultValue : 'New person' },
       *             { name : 'birthdate', type : 'date' }
       *         ];
       *     }
       * }
       *
       * // Subclass overriding one of the fields
       * class Bot extends Person {
       *     static get fields() {
       *         return [
       *             // Default value of 'username' field is overridden, any other setting from the parents
       *             // definition is preserved
       *             { name : 'username', defaultValue : 'Bot' }
       *         ];
       *     }
       * }
       * ```
       *
       * Fields in a subclass are merged with those from the parent class, making it easy to override mappings,
       * formats etc.
       *
       * @member {Array<String|ModelFieldConfig|Core.data.field.DataField>} fields
       * @readonly
       * @static
       * @category Fields
       */
      "fields"
    ];
  }
  static get fields() {
    return [
      // The index of this item in its parent (respects filtering)
      {
        name: "parentIndex",
        type: "number",
        persist: false,
        internal: true
      },
      // The index of this item in its parent ghost (non-sortable) children array
      {
        name: "orderedParentIndex",
        type: "number",
        persist: false,
        internal: true
      },
      /**
       * Flag the record as read-only on the UI level, preventing the end user from manipulating it using editing
       * features such as cell editing and event dragging.
       *
       * Does not prevent altering the record programmatically, it can still be manipulated by application code.
       *
       * For more info, see the "Read-only records" section above.
       *
       * @field {Boolean} readOnly
       * @category Common
       */
      {
        name: "readOnly",
        type: "boolean"
      },
      /**
       * Start expanded or not (only valid for tree data)
       * @readonly
       * @field {Boolean} expanded
       * @category Tree
       */
      {
        name: "expanded",
        internal: true
      }
    ];
  }
  /**
   * Template static getter which is supposed to be overridden to define default field values for the Model class.
   * Overrides `defaultValue` config specified by the {@link #property-fields-static} getter.
   * Returns a named object where key is a field name and value is a default value for the field.
   *
   * NOTE: This is a legacy way of defining default values, we recommend using {@link #property-fields-static} moving
   * forward.
   *
   * ```javascript
   * class Person extends Model {
   *     static get fields() {
   *         return [
   *             { name : 'username', defaultValue : 'New person' }
   *         ];
   *     }
   * }
   *
   * class Bot extends Person {
   *     static get defaults() {
   *         return {
   *             username : 'Bot' // default value of 'username' field is overridden
   *         };
   *     }
   * }
   * ```
   *
   * @member {Object} defaults
   * @static
   * @category Fields
   */
  /**
   * The data source for the id field which provides the ID of instances of this Model.
   * @property {String}
   * @category Fields
   */
  static set idField(idField) {
    this._assignedIdField = true;
    this._idField = idField;
  }
  static get idField() {
    return this._idField;
  }
  /**
   * The name of the data field which holds children of this Model when used in a tree structure
   * ```javascript
   * MyModel.childrenField = 'kids';
   * const parent = new MyModel({
   *     name : 'Dad',
   *     kids : [
   *         { name : 'Daughter' },
   *         { name : 'Son' }
   *     ]
   * });
   * ```
   * @property {String}
   * @category Fields
   */
  static set childrenField(childrenField) {
    this._childrenField = childrenField;
  }
  static get childrenField() {
    if (!this._childrenField) {
      const dataField = this.fieldMap.children;
      this._childrenField = (dataField == null ? void 0 : dataField.dataSource) || "children";
    }
    return this._childrenField;
  }
  /**
   * Returns index path to this node. This is the index of each node in the node path
   * starting from the topmost parent. (only relevant when its part of a tree store).
   * @returns {Number[]} The index of each node in the path from the topmost parent to this node.
   * @category Parent & children
   * @private
   */
  get indexPath() {
    const indices = [];
    let node = this, depth = node.childLevel;
    for (node = this; node && !node.isRoot; node = node.parent) {
      indices[depth--] = node.parentIndex + 1;
    }
    return indices;
  }
  /**
   * Unique identifier for the record. Might be mapped to another dataSource using idField, but always exposed as
   * record.id. Will get a generated value if none is specified in records data.
   *
   * {@note}Note that generated ids are meant to be temporary (phantom ids), they should not be serialized
   * but instead replaced by the backend on commit{/@note}
   *
   * @field {String|Number} id
   * @category Common
   */
  //region Init
  /**
   * Constructs a new record from the supplied data config.
   * @param {Object} [config] Raw model config
   * @param {Core.data.Store} [store] Data store
   * @param {Object} [meta] Meta data
   * @privateparam {Boolean} [skipExpose] Skip exposing properties from data
   * @privateparam {Boolean} [forceUseRaw] Force using raw data, used by copy to not clone data twice
   * @function constructor
   * @category Lifecycle
   */
  construct(config = {}, store = null, meta = null, skipExpose = false, forceUseRaw = false, rawData = false) {
    var _a2, _b;
    const me = this, stores2 = (_a2 = ArrayHelper.asArray(store)) != null ? _a2 : [], { constructor, fieldMap } = me;
    let configs = null;
    store = stores2[0];
    me.meta = {
      modified: {},
      ...constructor.metaConfig,
      ...meta
    };
    if (constructor.applyConfigs) {
      for (const key in me.getDefaultConfiguration()) {
        if (!configs) {
          configs = {};
          if (!me.useRawData || !me.useRawData.enabled) {
            config = { ...config };
          }
        }
        if (key in config) {
          if (config[key] !== void 0) {
            configs[key] = config[key];
          }
          delete config[key];
        }
      }
    }
    super.construct(configs);
    if (!skipExpose) {
      constructor.exposeProperties(config, rawData);
    }
    if (!hasOwn3(constructor, "idFieldProcessed")) {
      let overriddenIdField = me.meta.idField;
      if (!overriddenIdField) {
        if (constructor._assignedIdField) {
          overriddenIdField = constructor.idField;
        } else if (store) {
          overriddenIdField = store.idField;
        }
      }
      if (overriddenIdField && overriddenIdField !== fieldMap.id.dataSource) {
        constructor.addField({
          name: "id",
          dataSource: overriddenIdField,
          internal: true
        });
      }
      constructor._idField = fieldMap.id.dataSource;
      constructor.idFieldProcessed = true;
    }
    me._internalId = _Model._internalIdCounter++;
    me.stores = [];
    me.unjoinedStores = [];
    if (!me.originalData) {
      me.originalData = config;
    }
    me.data = constructor.processData(config, false, store, me, forceUseRaw);
    ((_b = me.meta.initableValues) == null ? void 0 : _b.size) && me.assignInitables();
    if (me.id == null) {
      me.setData("id", me.generateId(store));
    }
    if (me.data[constructor.childrenField]) {
      me.processChildren(stores2);
    }
    me.generation = 0;
  }
  /**
   * Set this property to `true` when adding a record on a conditional basis, that is, it is yet
   * to be confirmed as an addition.
   *
   * When this is set, the {@link #property-isPersistable} value of the record is **false**, and upon being
   * added to a Store it will *not* be eligible to be synced with the server as an added record.
   *
   * Subsequently, *clearing* this property means this record will become persistable and eligible
   * for syncing as an added record.
   * @property {Boolean}
   * @category Editing
   */
  set isCreating(isCreating) {
    const me = this;
    if (Boolean(me.meta.isCreating) !== isCreating) {
      me.meta.isCreating = isCreating;
      me.stores.forEach((s) => {
        s.onIsCreatingToggle(me, isCreating);
      });
    }
  }
  get isCreating() {
    return Boolean(this.meta.isCreating);
  }
  /**
   * Compares this Model instance to the passed instance. If they are of the same type, and all fields
   * (except, obviously, `id`) are equal, this returns `true`.
   * @param {Core.data.Model} other The record to compare this record with.
   * @returns {Boolean} `true` if the other is of the same class and has all fields equal.
   * @category Misc
   */
  equals(other) {
    if (other instanceof this.constructor) {
      for (let fields = this.$meta.fields.defs, i = 0, { length } = fields; i < length; i++) {
        const field2 = fields[i], { name } = field2;
        if (name !== "id" && !field2.isEqual(this.getValue(name), other.getValue(name))) {
          return false;
        }
      }
      return true;
    }
    return false;
  }
  get subclass() {
    return new this.constructor(Object.setPrototypeOf({
      id: _undefined
    }, this.data), this.stores[0], null, true);
  }
  /**
   * Processes raw data, converting values and setting defaults.
   * @private
   * @param {Object} data Raw data
   * @param {Boolean} [ignoreDefaults] Ignore setting default values, used when updating
   * @returns {Object} Processed data
   * @category Fields
   */
  static processData(data, ignoreDefaults = false, store, record, forceUseRaw) {
    const { fieldMap, defaultValues } = this, { useRawData = { enabled: false } } = store || {}, processed = forceUseRaw || useRawData.enabled ? data : ObjectHelper.clone(data);
    let fieldName;
    ignoreDefaults = ignoreDefaults || useRawData.disableDefaultValue || forceUseRaw;
    if (!ignoreDefaults) {
      for (fieldName in defaultValues) {
        if (processed[fieldName] === _undefined) {
          let defaultValue2 = defaultValues[fieldName];
          if (Array.isArray(defaultValue2)) {
            defaultValue2 = defaultValue2.slice();
          }
          processed[fieldName] = defaultValue2;
        }
      }
    }
    if (!useRawData.disableTypeConversion && !forceUseRaw) {
      for (fieldName in fieldMap) {
        const field2 = fieldMap[fieldName], { name, dataSource } = field2, hasSource = dataSource !== name, complex = field2.complexMapping, sourceExists = hasSource && (complex ? ObjectHelper.pathExists(data, dataSource) : dataSource in data), useNameForValue = name in data && (!hasSource || !sourceExists), convert = !useRawData.disableTypeConversion && field2.convert;
        if (useNameForValue || convert) {
          if (!ignoreDefaults || useNameForValue || sourceExists) {
            const value = useNameForValue ? processed[name] : complex ? ObjectHelper.getPath(processed, dataSource) : processed[dataSource], converted = convert ? field2.convert(value, data, record) : value;
            if (complex) {
              ObjectHelper.setPath(processed, dataSource, converted);
            } else {
              processed[dataSource] = converted;
            }
            if (hasSource) {
              delete processed[name];
            }
          }
        }
      }
    }
    this.$meta.fields.initable.length && this.initInitables(record, processed);
    return processed;
  }
  static setupClass(meta) {
    super.setupClass(meta);
    if (!meta.fields) {
      this.setupFields(this, meta);
    }
  }
  static setupFields(cls2, meta) {
    var _a2, _b, _c, _d, _e;
    const classFields = hasOwn3(cls2, "fields") && cls2.fields, base = meta.super.fields, fieldsInfo = meta.fields = {
      defs: (_a2 = base == null ? void 0 : base.defs.slice()) != null ? _a2 : [],
      // Set to true when an instance's data object is run through exposeProperties
      exposedData: false,
      // These objects are all keyed by field name:
      defaults: base ? { ...base.defaults } : {},
      // value=field.defaultValue
      exposed: Object.create((_b = base == null ? void 0 : base.exposed) != null ? _b : null),
      // value=true if we've done defineProperty
      ordinals: Object.create((_c = base == null ? void 0 : base.ordinals) != null ? _c : null),
      // value=index in the defs array
      map: Object.create((_d = base == null ? void 0 : base.map) != null ? _d : null),
      // value=definition object
      sources: Object.create((_e = base == null ? void 0 : base.sources) != null ? _e : null)
      // value=source definition object
    };
    if (hasOwn3(cls2, "defaults")) {
      Object.assign(fieldsInfo.defaults, cls2.defaults);
    }
    if (hasOwn3(cls2, "idField")) {
      cls2.addField({
        name: "id",
        dataSource: cls2.idField,
        internal: true
      });
      fieldsInfo.exposed[cls2.idField] = true;
    }
    if (classFields == null ? void 0 : classFields.length) {
      classFields.map(cls2.addField, cls2);
    }
    fieldsInfo.initable = fieldsInfo.defs.filter((field2) => field2.init);
    cls2.exposeRelations();
  }
  static get defaultValues() {
    return this.$meta.fields.defaults;
  }
  /**
   * An array containing all the _defined_ fields for this Model class. This will include all superclass's
   * defined fields.
   * @property {Core.data.field.DataField[]}
   * @static
   * @readonly
   * @category Fields
   */
  static get allFields() {
    return this.$meta.fields.defs;
  }
  /**
   * Same as {@link #property-allFields-static}.
   * @property {Core.data.field.DataField[]}
   * @readonly
   * @category Fields
   */
  get allFields() {
    return this.$meta.fields.defs;
  }
  /**
   * An object containing all the _defined_ fields for this Model class. This will include all superclass's
   * defined fields through its prototype chain. So be aware that `Object.keys` and `Object.entries` will only
   * access this class's defined fields.
   * @property {Object<String,Core.data.field.DataField>}
   * @static
   * @readonly
   * @category Fields
   */
  static get fieldMap() {
    return this.$meta.fields.map;
  }
  /**
   * Same as {@link #property-fieldMap-static}.
   * @property {Object<String,Core.data.field.DataField>}
   * @readonly
   * @category Fields
   */
  get fieldMap() {
    return this.$meta.fields.map;
  }
  static get fieldDataSourceMap() {
    return this.$meta.fields.sources;
  }
  /**
   * Makes getters and setters for fields (from definitions and data). Called once when class is defined and once when
   * data is loaded first time.
   * @internal
   * @param {Object} [data] Raw data
   * @param {Boolean} [raw=true] True if data is raw (contains data sources), False if data contains field names
   * @category Fields
   */
  static exposeProperties(data, raw = true) {
    const me = this, fieldsInfo = me.$meta.fields, fieldMapProperty = raw ? "exposed" : "map";
    if (data && me.autoExposeFields && !fieldsInfo.exposedData) {
      let dataProperty, fieldDef, type;
      for (dataProperty in data) {
        if (!fieldsInfo[fieldMapProperty][dataProperty] && dataProperty !== me.childrenField) {
          type = ObjectHelper.typeOf(data[dataProperty]);
          fieldDef = {
            name: dataProperty,
            dataSource: dataProperty,
            fromData: true
          };
          if (fieldDataTypes[type]) {
            fieldDef.type = type;
          }
          me.addField(fieldDef);
        }
      }
      fieldsInfo.exposedData = true;
    }
    me.exposeRelations();
  }
  /**
   * Add a field definition in addition to those predefined in `fields`.
   * @param {String|ModelFieldConfig} fieldDef A field name or definition
   * @category Fields
   */
  static addField(fieldDef) {
    if (fieldDef == null) {
      return;
    }
    if (typeof fieldDef === "string") {
      fieldDef = {
        name: fieldDef
      };
    }
    const me = this.initClass(), fieldsInfo = me.$meta.fields, { ordinals } = fieldsInfo, propertiesExposed = fieldsInfo.exposed, { name } = fieldDef, existing = fieldsInfo.map[name], dataSource = fieldDef.dataSource || (fieldDef.dataSource = name);
    let field2, key;
    if (!existing || fieldDef.type && fieldDef.type !== existing.type) {
      field2 = DataField.create(fieldDef);
      field2.definedBy = existing ? existing.definedBy : me;
      field2.ordinal = existing ? existing.ordinal : ordinals[name] = fieldsInfo.defs.length;
    } else {
      field2 = Object.create(existing);
      for (key in fieldDef) {
        if (key !== "type") {
          field2[key] = fieldDef[key];
        }
      }
    }
    field2.owner = me;
    fieldsInfo.defs[field2.ordinal] = field2;
    fieldsInfo.map[name] = field2;
    if (!fieldsInfo.sources[dataSource]) {
      fieldsInfo.sources[dataSource] = field2;
    }
    if (dataSource.includes(".")) {
      field2.complexMapping = true;
    }
    if (field2.complexMapping) {
      propertiesExposed[dataSource.split(".")[0]] = true;
    } else {
      propertiesExposed[dataSource] = true;
    }
    if ("defaultValue" in field2) {
      fieldsInfo.defaults[dataSource] = field2.defaultValue;
    }
    if (!internalProps[name]) {
      field2.defineAccessor(me.prototype);
    }
    me._nonPersistableFields = null;
    me._alwaysWriteFields = null;
    return field2;
  }
  /**
   * Remove a field definition by name.
   * @param {String} fieldName Field name
   * @category Fields
   */
  static removeField(fieldName) {
    const me = this.initClass(), fieldsInfo = me.$meta.fields, definition = fieldsInfo.map[fieldName], { ordinals } = fieldsInfo, index = ordinals[fieldName];
    if (definition) {
      fieldsInfo.defs.splice(index, 1);
      delete ordinals[fieldName];
      delete fieldsInfo.defaults[fieldName];
      delete fieldsInfo.exposed[fieldName];
      delete fieldsInfo.map[fieldName];
      delete fieldsInfo.sources[definition.dataSource];
      for (const name in ordinals) {
        if (ordinals[name] > index) {
          --ordinals[name];
        }
      }
      delete me.prototype[fieldName];
    }
  }
  /**
   * Makes getters and setters for related records. Populates a Model#relation array with the relations, to allow it
   * to be modified later when assigning stores.
   * @internal
   * @category Relations
   */
  static exposeRelations() {
    const me = this;
    if (hasOwn3(me, "exposedRelations")) {
      return;
    }
    if (me.relations) {
      me.exposedRelations = [];
      for (const relationName in me.relations) {
        const relation = me.relations[relationName];
        relation.relationName = relationName;
        me.exposedRelations.push(relation);
        if (!Reflect.ownKeys(me.prototype).includes(relationName)) {
          defineProperty5(me.prototype, relationName, {
            enumerable: true,
            get: function() {
              return this.getForeign(relationName);
            },
            set: function(value) {
              this.setForeign(relationName, value, relation);
            }
          });
        }
      }
    }
  }
  //endregion
  //region Initable fields
  // Initializes any fields using a data type that has an init method, and caches the value to assign to that field to
  // be able to assign it after all others. That allows the initter to reference the records other data if needed
  // (baselines use that to reference the task to get default values)
  static initInitables(record, processedData) {
    const laterValues = record.meta.initableValues = /* @__PURE__ */ new Map();
    for (const field2 of this.$meta.fields.initable) {
      const value = ObjectHelper.getPath(processedData, field2.dataSource);
      value !== void 0 && laterValues.set(field2, value);
      !field2.lazy && field2.init(processedData, record);
    }
  }
  // Assigns values to the fields that were initialized earlier (see initInitables above)
  assignInitables() {
    const { initableValues } = this.meta;
    for (const [field2, value] of initableValues) {
      if (field2.set(value, this.data, this) !== false) {
        initableValues.delete(field2);
      }
    }
  }
  //endregion
  //region Fields
  /**
   * Flag checked from Store when loading data that determines if fields found in first records should be exposed in
   * same way as predefined fields.
   *
   * {@note}Note that we for all but the most basic use cases recommend explicitly defining the fields.
   * Having them auto exposed can lead to unexpected behavior, if the first record is not complete (fields missing,
   * null etc).
   * {/@note}
   *
   * @property {Boolean}
   * @category Fields
   */
  static get autoExposeFields() {
    return true;
  }
  /**
   * This function forces correct field order. Correct order is parentId before id. If we process id field before
   * parentId, idMap won't be updated and changing parent node will lead to duplicated records in storage
   * @param {String} a
   * @param {String} b
   * @returns {number}
   * @private
   */
  static fieldSorter(a, b) {
    return (fieldsOrder[a] || 100) - (fieldsOrder[b] || 100);
  }
  /**
   * Convenience getter to get field definitions from class.
   * @property {Core.data.field.DataField[]}
   * @readonly
   * @category Fields
   */
  get fields() {
    return this.$meta.fields.defs;
  }
  /**
   * Convenience function to get the definition for a field from class.
   * @param {String} fieldName Field name
   * @returns {Core.data.field.DataField}
   * @category Fields
   */
  getFieldDefinition(fieldName) {
    return this.$meta.fields.map[fieldName];
  }
  getFieldDefinitionFromDataSource(dataSource) {
    return this.$meta.fields.sources[dataSource];
  }
  /**
   * Get the names of all fields in data.
   * @property {String[]}
   * @readonly
   * @category Fields
   */
  get fieldNames() {
    return Object.keys(this.data);
  }
  /**
   * Get the definition for a field by name. Caches results.
   * @param {String} fieldName Field name
   * @returns {Core.data.field.DataField} Field definition or null if none found
   * @category Fields
   */
  static getFieldDefinition(fieldName) {
    return this.$meta.fields.map[fieldName];
  }
  /**
   * Returns dataSource configuration for a given field name
   * @param {String} fieldName
   * @returns {String} Field `dataSource` mapping
   * @internal
   */
  static getFieldDataSource(fieldName) {
    var _a2;
    return ((_a2 = this.getFieldDefinition(fieldName)) == null ? void 0 : _a2.dataSource) || fieldName;
  }
  /**
   * Get the data source used by specified field. Returns the fieldName if no data source specified.
   * @param {String} fieldName Field name
   * @returns {String}
   * @category Fields
   */
  getDataSource(fieldName) {
    const def = this.constructor.getFieldDefinition(fieldName);
    return (def == null ? void 0 : def.dataSource) || (def == null ? void 0 : def.name);
  }
  /**
   * Processes input to a field, converting to expected type.
   * @param {String} fieldName Field name
   * @param {*} value Value to process
   * @returns {*} Converted value
   * @category Fields
   */
  static processField(fieldName, value, record) {
    const field2 = this.fieldMap[fieldName];
    return (field2 == null ? void 0 : field2.convert) ? field2.convert(value, this.data, record) : value;
  }
  //endregion
  //region Relations
  /**
   * Initializes model relations. Called from store when adding a record.
   * @private
   * @category Relations
   */
  initRelations() {
    const me = this, relations = me.constructor.exposedRelations;
    if (!relations) {
      return;
    }
    me.stores.forEach((store) => {
      var _a2;
      if (!store.modelRelations) {
        store.initRelations();
      }
      const relatedRecords = [];
      (_a2 = store.modelRelations) == null ? void 0 : _a2.forEach((config) => {
        relatedRecords.push({ related: me.initRelation(config), config });
      });
      store.updateRecordRelationCache(me, relatedRecords);
    });
  }
  /**
   * Initializes/updates a single relation.
   * @param config Relation config
   * @returns {Core.data.Model} Related record
   * @private
   * @category Relations
   */
  initRelation(config) {
    const me = this, foreignId = me.get(config.foreignKey), foreign = foreignId !== _undefined && typeof config.foreignStore !== "string" && config.foreignStore.getById(foreignId), relationCache = me.meta.relationCache || (me.meta.relationCache = {});
    relationCache[config.relationName] = foreign || (foreignId != null ? { id: foreignId, placeHolder: true } : null);
    return foreign;
  }
  removeRelation(config) {
    const { relationName, foreignKey, nullFieldOnRemove } = config;
    if (this.meta.relationCache[relationName]) {
      delete this.meta.relationCache[relationName];
      if (nullFieldOnRemove) {
        this.setData(foreignKey, null);
      }
    }
  }
  getForeign(name) {
    var _a2;
    return (_a2 = this.meta.relationCache) == null ? void 0 : _a2[name];
  }
  setForeign(name, value, config) {
    const id = _Model.asId(value);
    return this.set(config.foreignKey, id);
  }
  /**
   * Get a relation config by name, from the first store.
   * @param {String} name
   * @returns {Object}
   * @private
   * @category Relations
   */
  getRelationConfig(name) {
    var _a2, _b;
    return (_b = (_a2 = this.firstStore) == null ? void 0 : _a2.modelRelations) == null ? void 0 : _b.find((r) => r.foreignKey === name);
  }
  //endregion
  //region Get/set values, data handling
  flatGet(fieldName, dataSource) {
    if (this.batching && fieldName in this.meta.batchChanges) {
      return this.meta.batchChanges[fieldName];
    }
    return dataSource in this.data ? this.data[dataSource] : this.data[fieldName];
  }
  complexGet(fieldName, dataSource) {
    if (this.batching && fieldName in this.meta.batchChanges) {
      return this.meta.batchChanges[fieldName];
    }
    return ObjectHelper.getPath(this.data, dataSource);
  }
  /**
   * Get value for specified field name. You can also use the generated getters if loading through a Store.
   * If model is currently in batch operation this will return updated batch values which are not applied to Model
   * until endBatch() is called.
   * @param {String} fieldName Field name to get value from
   * @returns {*} Fields value
   * @category Fields
   */
  get(fieldName) {
    if (!fieldName) {
      return;
    }
    const me = this, field2 = me.fieldMap[fieldName];
    if (!field2) {
      if (fieldName.includes("[")) {
        const [, arrayFieldName, index, path] = fieldName.match(arrayRe);
        const arrayField = me.fieldMap[arrayFieldName];
        if (arrayField == null ? void 0 : arrayField.getAt) {
          me._thisIsAUsedExpression(me[arrayFieldName]);
          const subRecord = arrayField.getAt(me, index);
          if (subRecord && path) {
            if (subRecord.isModel) {
              return subRecord.getValue(path);
            }
            return subRecord[path];
          }
          return subRecord;
        }
        return null;
      }
      if (fieldName.includes(".")) {
        if (!ObjectHelper.hasPath(me.data, fieldName)) {
          return ObjectHelper.getPath(me, fieldName);
        }
        return me.complexGet(fieldName, fieldName);
      }
    }
    if (field2 == null ? void 0 : field2.complexMapping) {
      return me.complexGet(fieldName, field2.dataSource);
    }
    return me.flatGet(fieldName, (field2 == null ? void 0 : field2.dataSource) || fieldName);
  }
  // Used to get field values, replaces `record[fieldName]` in internal code to allow relations etc.
  getValue(fieldName) {
    if (!fieldName) {
      return;
    }
    if (fieldName in this) {
      return this[fieldName];
    }
    return this.get(fieldName);
  }
  // Used to set field values, replacing `record[fieldName] = value` to handle dot notation
  setValue(fieldName, value) {
    if (fieldName in this) {
      this[fieldName] = value;
    } else {
      this.set(fieldName, value);
    }
  }
  /**
   * Internal function used to update a records underlying data block (record.data) while still respecting field
   * mappings. Needed in cases where a field needs setting without triggering any associated behaviour and it has a
   * dataSource with a different name.
   *
   * For example:
   * ```javascript
   * // startDate mapped to data.beginDate
   * { name : 'startDate', dataSource : 'beginDate' }
   *
   * // Some parts of our code needs to update the data block without triggering any of the behaviour associated with
   * // calling set. This would then not update "beginDate":
   * record.data.startDate = xx;
   *
   * // But this would
   * record.setData('startDate', xx);
   * ```
   * @internal
   * @category Editing
   */
  setData(toSet, value) {
    var _a2, _b;
    const { data, fieldMap } = this;
    if (typeof toSet === "string") {
      const field2 = fieldMap[toSet], dataSource = (_a2 = field2 == null ? void 0 : field2.dataSource) != null ? _a2 : toSet;
      if (field2 == null ? void 0 : field2.set) {
        field2.set(value, this.data, this);
      } else if (field2 == null ? void 0 : field2.complexMapping) {
        ObjectHelper.setPath(data, dataSource, value);
      } else {
        data[dataSource] = value;
      }
    } else {
      const keys = Object.keys(toSet);
      for (let i = 0; i < keys.length; i++) {
        const fieldName = keys[i], field2 = fieldMap[fieldName], dataSource = (_b = field2 == null ? void 0 : field2.dataSource) != null ? _b : fieldName;
        if (field2 == null ? void 0 : field2.set) {
          field2.set(value, this.data, this);
        } else if (dataSource) {
          if (field2 == null ? void 0 : field2.complexMapping) {
            ObjectHelper.setPath(data, dataSource, toSet[fieldName]);
          } else {
            data[dataSource] = toSet[fieldName];
          }
        }
      }
    }
  }
  /**
   * Returns raw data from the encapsulated data object for the passed field name
   * @param {String} fieldName The field to get data for.
   * @returns {*} The raw data value for the field.
   * @category Editing
   */
  getData(fieldName) {
    var _a2;
    const field2 = this.fieldMap[fieldName], dataSource = (_a2 = field2 == null ? void 0 : field2.dataSource) != null ? _a2 : fieldName;
    if (dataSource) {
      if (field2 == null ? void 0 : field2.complexMapping) {
        return ObjectHelper.getPath(this.data, dataSource);
      }
      return this.data[dataSource];
    }
  }
  /**
   * Silently updates record's id with no flagging the property as modified.
   * Triggers onModelChange event for changed id.
   * @param {String|Number} value id value
   * @private
   */
  syncId(value) {
    const oldValue = this.id;
    if (oldValue !== value) {
      this.setData("id", value);
      const data = { id: { value, oldValue } };
      this.afterChange(data, data);
    }
  }
  /**
   * Set value for the specified field. You can also use the generated setters if loading through a Store.
   *
   * Setting a single field, supplying name and value:
   *
   * ```javascript
   * record.set('name', 'Clark');
   * ```
   *
   * Setting multiple fields, supplying an object:
   *
   * ```javascript
   * record.set({
   *     name : 'Clark',
   *     city : 'Metropolis'
   * });
   * ```
   *
   * @param {String|Object} field The field to set value for, or an object with multiple values to set in one call
   * @param {*} [value] Value to set
   * @param {Boolean} [silent] Set to true to not trigger events. If event is recurring, occurrences won't be updated automatically.
   * @privateparam {Boolean} [validOnly] If set to `true` it will ignore setting a `undefined` value, allowing conversion functions to invalidate a data input
   * @privateparam {Boolean} [triggerBeforeUpdate]
   * @fires Store#idChange
   * @fires Store#update
   * @fires Store#change
   * @category Editing
   */
  set(field2, value, silent = false, fromRelationUpdate = false, skipAccessors = false, validOnly = false, triggerBeforeUpdate = true) {
    var _a2;
    const me = this;
    if (me.isBatchUpdating) {
      me.inBatchSet(field2, value, silent || me.$silenceBatch);
      return null;
    } else {
      const wasSet = me.inSet(field2, value, silent, fromRelationUpdate, skipAccessors, validOnly, triggerBeforeUpdate);
      (_a2 = me.afterSet) == null ? void 0 : _a2.call(me, field2, value, silent, fromRelationUpdate, wasSet);
      return wasSet;
    }
  }
  fieldToKeys(field2, value) {
    if (typeof field2 !== "string") {
      return ObjectHelper.assign({}, field2);
    }
    return {
      [field2]: value
    };
  }
  inBatchSet(field2, value, silent) {
    const me = this, {
      meta,
      constructor,
      fieldMap
    } = me, wasSet = {};
    let cmp, changed = false;
    if (typeof field2 !== "string") {
      Object.keys(me.fieldToKeys(field2, value)).forEach((key) => {
        cmp = fieldMap[key] || ObjectHelper;
        value = constructor.processField(key, field2[key], me);
        if (!cmp.isEqual(meta.batchChanges[key], value)) {
          wasSet[key] = {
            value,
            oldValue: me.get(key)
          };
          meta.batchChanges[key] = value;
          changed = true;
        }
      });
    } else {
      cmp = fieldMap[field2] || ObjectHelper;
      if (!cmp.isEqual(meta.batchChanges[field2], value)) {
        wasSet[field2] = {
          value,
          oldValue: me.get(field2)
        };
        meta.batchChanges[field2] = value;
        changed = true;
      }
    }
    if (changed) {
      me.generation++;
      if (!silent) {
        const event = {
          action: "update",
          record: me,
          records: [me],
          changes: wasSet
        };
        me.stores.forEach((store) => {
          store.trigger("batchedUpdate", { ...event });
        });
        me.forEachLinked(
          (store, record) => store.trigger("batchedUpdate", { ...event, record, records: [record] })
        );
      }
    }
  }
  inSet(fieldNameOrObject, value, silent, fromRelationUpdate, skipAccessors = false, validOnly = false, triggerBeforeUpdate = true) {
    var _a2, _b, _c;
    const me = this, {
      data,
      meta,
      fieldMap,
      constructor
    } = me, {
      prototype: myProto,
      childrenField,
      relations
    } = constructor, wasSet = {}, toSet = me.fieldToKeys(fieldNameOrObject, value), keys = Object.keys(toSet).sort(constructor.fieldSorter);
    let changed = false;
    if (!silent && !me.triggerBeforeUpdate(toSet)) {
      return null;
    }
    me.inSetting = true;
    for (let i = 0; i < keys.length; i++) {
      const key = keys[i];
      if (key === childrenField) {
        continue;
      }
      if (key.includes("[")) {
        const [, arrayFieldName, index, path] = key.match(arrayRe), field3 = me.fieldMap[arrayFieldName];
        if (field3 == null ? void 0 : field3.getAt) {
          const subRecord = field3.getAt(me, index);
          if (subRecord.isModel) {
            subRecord.set(path, toSet[key]);
          } else {
            ObjectHelper.setPath(subRecord, path, toSet[key]);
          }
          continue;
        }
      }
      const complexKey = key.includes(".");
      if (relations && complexKey) {
        const [, relationName, prop] = key.match(nestedRe);
        if (relations[relationName]) {
          me[relationName].set(prop, toSet[key]);
          continue;
        }
      }
      const field2 = fieldMap[key], cmp = field2 || ObjectHelper, readOnly = field2 == null ? void 0 : field2.readOnly, mapping = (_a2 = field2 == null ? void 0 : field2.dataSource) != null ? _a2 : key, useProp = !skipAccessors && !field2 && key in myProto || (field2 == null ? void 0 : field2.useProp), oldValue = useProp ? me[mapping] : (field2 == null ? void 0 : field2.complexMapping) || complexKey ? ObjectHelper.getPath(data, mapping) : data[mapping], value2 = constructor.processField(key, toSet[key], me), val = toSet[key] = { value: value2 }, relation = me.getRelationConfig(key);
      if (!readOnly && !cmp.isEqual(oldValue, value2) && (!validOnly || value2 !== void 0)) {
        me.generation++;
        val.oldValue = (_c = (_b = field2 == null ? void 0 : field2.getOldValue) == null ? void 0 : _b.call(field2, me)) != null ? _c : oldValue;
        changed = true;
        if (key in meta.modified && cmp.isEqual(meta.modified[key], value2)) {
          Reflect.deleteProperty(meta.modified, key);
          if (me.isReverting) {
            me.data[mapping] = value2;
          }
        } else if (!me.ignoreBag) {
          if (!(key in meta.modified)) {
            me.storeFieldChange(key, oldValue);
          }
          if (val.oldValue === _undefined) {
            Reflect.deleteProperty(val, "oldValue");
          }
        }
        wasSet[key] = val;
        me.applyValue(useProp, mapping, value2, skipAccessors, field2);
        if (relation && !fromRelationUpdate) {
          me.initRelation(relation);
          me.stores.forEach((store) => store.cacheRelatedRecord(me, value2, relation.relationName, val.oldValue));
        }
      }
    }
    if (changed) {
      me.afterChange(toSet, wasSet, silent, fromRelationUpdate, skipAccessors);
    }
    me.inSetting = false;
    return changed ? wasSet : null;
  }
  // Provided as a hook for Engine to do what needs to be done which ever way a field value is changed
  applyValue(useProp, key, value, skipAccessors, field2) {
    var _a2;
    const me = this;
    if (((field2 == null ? void 0 : field2.name) || key) === me.constructor.parentIdField) {
      if (!(((_a2 = me.parent) == null ? void 0 : _a2.isRoot) && (value == null || !me.firstStore.getById(value)))) {
        (me.firstStore.getById(value) || me.firstStore.rootNode).insertChild(me, null, false, {
          orderedParentIndex: { skip: true }
        });
      }
    } else if (useProp) {
      me[(field2 == null ? void 0 : field2.name) || key] = value;
    } else if (field2 == null ? void 0 : field2.set) {
      field2.set(value, me.data, me);
    } else {
      let complexMapping = field2 == null ? void 0 : field2.complexMapping;
      if (!field2 && key.includes(".")) {
        const nestedName = key.split(".")[0];
        field2 = me.constructor.fieldMap[nestedName];
        complexMapping = (field2 == null ? void 0 : field2.complexMapping) || (field2 == null ? void 0 : field2.type) === "object";
      }
      if (complexMapping) {
        ObjectHelper.setPath(me.data, key, value);
      } else {
        me.data[key] = value;
      }
    }
  }
  // skipAccessors argument is used in the engine override
  afterChange(toSet, wasSet, silent, fromRelationUpdate, skipAccessors) {
    for (const store of this.stores) {
      store.onModelChange(this, toSet, wasSet, silent, fromRelationUpdate, skipAccessors);
    }
    this.forEachLinked(
      (store, record) => store.onModelChange(record, toSet, wasSet, silent, fromRelationUpdate, skipAccessors)
    );
  }
  // Run fn for each store on each linked record
  forEachLinked(fn2) {
    for (const linked of this.$links) {
      for (const store of linked.stores) {
        fn2(store, linked);
      }
    }
  }
  /**
   * This yields `true` if this record is eligible for syncing with the server.
   * It can yield `false` if the record is in the middle of a {@link #property-isBatchUpdating batched update},
   * or if it is a {@link #property-isCreating tentative record} yet to be confirmed as a new addition.
   * @property {Boolean}
   * @readonly
   */
  get isPersistable() {
    return !this.isBatchUpdating && !this.isCreating;
  }
  /**
   * True if this model has any uncommitted changes.
   * @property {Boolean}
   * @readonly
   * @category Editing
   */
  get isModified() {
    return Boolean(this.meta.modified && Object.keys(this.meta.modified).length > 0);
  }
  get hasPersistableChanges() {
    return this.isPersistable && !ObjectHelper.isEmpty(this.rawModificationData);
  }
  /**
   * Returns true if this model has uncommitted changes for the provided field.
   * @param {String} fieldName Field name
   * @returns {Boolean} True if the field is changed
   * @category Editing
   */
  isFieldModified(fieldName) {
    return this.isModified && fieldName in this.meta.modified;
  }
  /**
   * Returns field value that should be persisted, or `undefined` if field is configured with `persist: false`.
   * @param {String|Core.data.field.DataField} nameOrField Name of the field to get value for, or its field definition
   * @private
   * @category Fields
   */
  getFieldPersistentValue(nameOrField) {
    const field2 = typeof nameOrField === "string" ? this.getFieldDefinition(nameOrField) : nameOrField, name = (field2 == null ? void 0 : field2.name) || nameOrField;
    let result;
    if (!field2 || field2.persist) {
      result = this.getValue(name);
      if (field2 == null ? void 0 : field2.serialize) {
        result = field2.serialize(result, this);
      }
    }
    return result;
  }
  /**
   * Get a map of the modified fields in form of an object. The field *names* are used as the property names
   * in the returned object.
   * @property {Object}
   * @readonly
   * @category Editing
   */
  get modifications() {
    const data = this.rawModifications;
    if (data && Object.keys(data).length) {
      data[this.constructor.idField] = this.id;
    }
    return data;
  }
  get rawModifications() {
    const me = this, data = {};
    if (!me.isModified) {
      return null;
    }
    let keySet = false;
    Object.keys(me.meta.modified).forEach((key) => {
      const value = me.getFieldPersistentValue(key);
      if (value !== _undefined) {
        data[key] = value;
        keySet = true;
      }
    });
    return keySet ? data : null;
  }
  /**
   * Get a map of the modified fields in form of an object. The field´s {@link Core.data.field.DataField#config-dataSource}
   * is used as the property name in the returned object. The record´s id is always included.
   * @property {Object}
   * @readonly
   * @category Editing
   */
  get modificationData() {
    const data = this.rawModificationData;
    if (data && Object.keys(data).length) {
      ObjectHelper.setPath(data, this.constructor.idField, this.id);
    }
    return data;
  }
  /**
   * Returns a map of the modified persistable fields
   * @internal
   * @property {Object}
   * @category Editing
   */
  get rawModificationData() {
    const me = this, { fieldMap } = me.constructor, data = {};
    if (!me.isModified) {
      return null;
    }
    let keySet = false;
    Object.keys(me.meta.modified).forEach((fieldName) => {
      let field2 = fieldMap[fieldName], dataSource = field2 == null ? void 0 : field2.dataSource;
      if (fieldName.includes(".")) {
        const topLevelFieldName = fieldName.match(nestedRe)[1];
        field2 = fieldMap[topLevelFieldName];
        dataSource = fieldName;
      }
      if (field2 == null ? void 0 : field2.persist) {
        const value = me.getFieldPersistentValue(fieldName);
        if (value !== _undefined) {
          ObjectHelper.setPath(data, dataSource, value);
          keySet = true;
        }
      }
    });
    return keySet ? data : null;
  }
  /**
   * Get a map of the modified data fields along with any {@link Core/data/field/DataField#config-alwaysWrite} fields,
   * in form of an object. The field´s *dataSource* is used as the property name in the returned object.
   * Used internally by AjaxStore / CrudManager when sending updates.
   * @property {Object}
   * @readonly
   * @category Editing
   */
  get modificationDataToWrite() {
    const alwaysWriteFields = this.constructor.alwaysWriteFields, recordData = this.modificationData;
    alwaysWriteFields.forEach((fieldName) => {
      recordData[this.getFieldDefinition(fieldName).dataSource] = this.getFieldPersistentValue(fieldName);
    });
    return recordData;
  }
  /**
   * Returns data for **all** {@link Core.data.field.DataField#config-persist persistable} fields in form of an
   * object, using dataSource if present.
   * @property {Object}
   * @internal
   * @readonly
   * @category Editing
   */
  get persistableData() {
    const me = this, data = {};
    me.fields.forEach((field2) => {
      const value = me.getFieldPersistentValue(field2);
      if (value !== _undefined) {
        if (field2 == null ? void 0 : field2.complexMapping) {
          ObjectHelper.setPath(data, field2.dataSource, value);
        } else {
          data[field2.dataSource] = value;
        }
      }
    });
    return data;
  }
  get dataByFieldName() {
    const { data } = this;
    return this.fields.reduce((result, field2) => {
      if (ObjectHelper.hasPath(data, field2.dataSource)) {
        result[field2.name] = data[field2.dataSource];
      }
      return result;
    }, {});
  }
  /**
   * True if this models changes are currently being committed.
   * @property {Boolean}
   * @category Editing
   */
  get isCommitting() {
    return Boolean(this.meta.committing);
  }
  /**
   * Clear stored changes, used on commit. Does not revert changes.
   * @param {Boolean} [includeDescendants] Supply `false` to not clear node descendants
   * @privateparam {Boolean} [removeFromStoreChanges] Update related stores modified collection or not
   * @privateparam {Object|null} [changes] Set of changes to clear on the record
   * @category Editing
   */
  clearChanges(includeDescendants = true, removeFromStoreChanges = true, changes = null) {
    const me = this, { meta } = me;
    if (changes) {
      for (const key in changes) {
        delete meta.modified[key];
      }
    } else {
      meta.modified = {};
    }
    const noChanges = ObjectHelper.isEmpty(meta.modified);
    meta.committing = false;
    if (removeFromStoreChanges) {
      me.stores.forEach((store) => {
        noChanges && store.modified.remove(me);
        store.added.remove(me);
        if (includeDescendants) {
          const descendants = store.collectDescendants(me).all;
          store.added.remove(descendants);
          noChanges && store.modified.remove(descendants);
        }
      });
    }
  }
  storeFieldChange(key, oldValue) {
    this.meta.modified[key] = oldValue;
  }
  /**
   * Reverts changes in this back to their original values.
   * @privateparam {Boolean} [silent] Specify `true` to not trigger events.
   * @category Editing
   */
  revertChanges(silent = false) {
    this.isReverting = true;
    this.set(this.meta.modified, _undefined, silent);
    this.isReverting = false;
  }
  applyChangeset(rawChanges, phantomIdField = "$PhantomId", remote = true) {
    var _a2;
    const me = this, modelClass = me.constructor, {
      idField,
      fieldDataSourceMap
    } = modelClass, rawChangesSimplePaths = ObjectHelper.pathifyKeys(rawChanges, fieldDataSourceMap), ownChangesSimplePaths = ObjectHelper.pathifyKeys(me.modificationData), changes = {}, idChanged = phantomIdField in rawChanges;
    for (const dataSource in rawChangesSimplePaths) {
      const field2 = fieldDataSourceMap[dataSource], propName = (_a2 = field2 == null ? void 0 : field2.name) != null ? _a2 : dataSource;
      if (remote || ((field2 == null ? void 0 : field2.name) === idField ? idChanged : rawChangesSimplePaths[dataSource] === ownChangesSimplePaths[dataSource])) {
        changes[propName] = rawChangesSimplePaths[dataSource];
      }
    }
    const wasSet = me.set(changes);
    me.clearChanges(false, true, remote ? null : changes);
    return wasSet ? Object.entries(wasSet).reduce((result, [field2, change]) => {
      result[field2] = change.value;
      return result;
    }, {}) : {};
  }
  //endregion
  //region Id
  /**
   * Gets the records internalId. It is assigned during creation, guaranteed to be globally unique among models.
   * @property {Number}
   * @category Identification
   */
  get internalId() {
    return this._internalId;
  }
  /**
   * Returns true if the record is new and has not been persisted (and received a proper id).
   * @property {Boolean}
   * @readonly
   * @category Identification
   */
  get isPhantom() {
    return this.id === "" || this.id == null || this.hasGeneratedId;
  }
  get isModel() {
    return true;
  }
  /**
   * Checks if record has a generated id.
   *
   * New records are assigned a generated id based on a UUID (starting with `_generated`), which is intended to be
   * temporary and should be replaced by the backend on commit.
   *
   * @property {Boolean}
   * @category Identification
   */
  get hasGeneratedId() {
    return typeof this.id === "string" && this.id.startsWith("_generated");
  }
  static generateId(text = this.$$name) {
    return `_generated${text}_${StringHelper.generateUUID()}`;
  }
  /**
   * Generates an id for a new record (a phantom id), based on a UUID (starting with `_generated`).
   *
   * Generated ids are intended to be temporary and should be replaced by the backend on commit.
   *
   * @category Identification
   * @returns {String}
   */
  generateId() {
    return this.constructor.generateId();
  }
  /**
   * Gets the id of specified model or model data object, or the value if passed string/number.
   * @param {Core.data.Model|String|Number} model
   * @returns {String|Number} id
   * @category Identification
   */
  static asId(model) {
    return (model == null ? void 0 : model.isModel) ? model.id : ObjectHelper.isObject(model) ? model[this.fieldMap.id.dataSource] : model;
  }
  //endregion
  //region JSON
  /**
   * Get the records data as a json string.
   *
   * ```javascript
   * const record = new Model({
   *     title    : 'Hello',
   *     children : [
   *         ...
   *     ]
   * });
   *
   * const jsonString = record.json;
   *
   * //jsonString:
   * '{"title":"Hello","children":[...]}'
   * ```
   *
   * @member {String}
   * @category JSON
   */
  get json() {
    return StringHelper.safeJsonStringify(this);
  }
  /**
   * Used by `JSON.stringify()` to correctly convert this record to json.
   *
   * In most cases no point in calling it directly.
   *
   * ```
   * // This will call `toJSON()`
   * const json = JSON.stringify(record);
   * ```
   *
   * If called manually, the resulting object is a clone of `record.data` + the data of any children:
   *
   * ```
   * const record = new Model({
   *     title    : 'Hello',
   *     children : [
   *         ...
   *     ]
   * });
   *
   * const jsonObject = record.toJSON();
   *
   * // jsonObject:
   * {
   *     title : 'Hello',
   *     children : [
   *         ...
   *     ]
   * }
   * ```
   *
   * @returns {Object}
   * @category JSON
   */
  toJSON() {
    const { children, unfilteredChildren } = this, jsonData = this.persistableData;
    if (unfilteredChildren || children) {
      jsonData[this.constructor.childrenField] = (unfilteredChildren || children).map((c) => c.toJSON());
    }
    return jsonData;
  }
  /**
   * Represent the record as a string, by default as a JSON string. Tries to use an abbreviated version of the
   * object's data, using id + name/title/text/label/description. If no such field exists, the full data is used.
   *
   * ```javascript
   * const record = new Model({ id : 1, name : 'Steve Rogers', alias : 'Captain America' });
   * console.log(record.toString()); // logs { "id" : 1, "name" : "Steve Rogers" }
   * ```
   *
   * @returns {String}
   * @category JSON
   */
  toString() {
    const me = this, nameField = abbreviationFields.find((field2) => field2 in me.constructor.fieldMap), data = nameField ? { [me.constructor.idField]: me.id, [nameField]: me[nameField] } : me.data;
    return StringHelper.safeJsonStringify(data);
  }
  //endregion
  //region Batch
  /**
   * True if this Model is currently batching its changes.
   * @property {Boolean}
   * @readonly
   * @category Editing
   */
  get isBatchUpdating() {
    return Boolean(this.batching);
  }
  /**
   * Returns `true` if this Model currently has outstanding batched changes for the specified field name.
   * @param {String} fieldName The field name to check for batched updates on.
   * @returns {Boolean}
   * @category Editing
   */
  hasBatchedChange(fieldName) {
    var _a2, _b;
    return (_b = (_a2 = this.meta) == null ? void 0 : _a2.batchChanges) == null ? void 0 : _b[fieldName];
  }
  /**
   * Begin a batch, which stores changes and commits them when the batch ends.
   * Prevents events from being fired during batch.
   *
   * ```javascript
   * record.beginBatch();
   * record.name = 'Mr Smith';
   * record.team = 'Golden Knights';
   * record.endBatch();
   * ```
   *
   * Please note that you can also set multiple fields in a single call using {@link #function-set}, which in many
   * cases can replace using a batch:
   *
   * ```javascript
   * record.set({
   *   name : 'Mr Smith',
   *   team : 'Golden Knights'
   * });
   * ```
   * @category Editing
   * @privateparam {Boolean} silentUpdates Suppress firing the `batchUpdatedEvent`
   */
  beginBatch(silentUpdates = false) {
    const me = this;
    if (!me.batching) {
      me.batching = 0;
      me.meta.batchChanges = {};
    }
    if (silentUpdates) {
      me.$silenceBatch = (me.$silenceBatch || 0) + 1;
    }
    me.batching++;
  }
  /**
   * End a batch, triggering events if data has changed.
   * @param {Boolean} [silent] Specify `true` to not trigger events. If event is recurring, occurrences won't be updated automatically.
   * @privateparam {Boolean} [silent=false] Specify `true` to not trigger events. If event is recurring, occurrences won't be updated automatically.
   * @privateparam {Boolean} [triggerBeforeUpdate=true]
   * @category Editing
  */
  endBatch(silent = false, skipAccessors = false, triggerBeforeUpdate = true) {
    const me = this, { parentIdField } = me.constructor;
    if (!me.batching) {
      return;
    }
    me.batching--;
    me.$silenceBatch && me.$silenceBatch--;
    if (me.batching > 0) {
      return;
    }
    if (!ObjectHelper.isEmpty(me.meta.batchChanges)) {
      let { batchChanges } = me.meta;
      me.meta.batchChanges = null;
      if (batchChanges[parentIdField]) {
        batchChanges = { ...batchChanges };
        me.parentId = batchChanges[parentIdField];
        delete batchChanges[parentIdField];
      }
      me.set(batchChanges, _undefined, silent, false, skipAccessors, void 0, triggerBeforeUpdate);
    }
  }
  /**
   * Cancels current batch operation. Any changes during the batch are discarded.
   * @category Editing
   */
  cancelBatch() {
    if (this.batching) {
      const me = this, { batchChanges } = me.meta, wasSet = {};
      me.batching = null;
      me.meta.batchChanges = null;
      me.generation++;
      if (!me.$silenceBatch) {
        Object.entries(batchChanges).forEach(([key, oldValue]) => {
          wasSet[key] = {
            oldValue,
            value: me[key]
          };
        });
        const event = {
          action: "update",
          record: me,
          records: [me],
          changes: wasSet
        };
        me.stores.forEach((store) => {
          store.trigger("batchedUpdate", { ...event });
        });
        me.forEachLinked((store, record) => {
          store.trigger("batchedUpdate", { ...event, record, records: [record] });
        });
      }
      me.$silenceBatch && me.$silenceBatch--;
    }
  }
  //endregion
  //region Events
  /**
   * Triggers beforeUpdate event for each store and checks if changes can be made from event return value.
   * @param {Object} changes Data changes
   * @returns {Boolean} returns true if data changes are accepted
   * @private
   */
  triggerBeforeUpdate(changes) {
    var _a2;
    return !((_a2 = this.stores) == null ? void 0 : _a2.some((s) => s.trigger("beforeUpdate", { record: this, changes }) === false));
  }
  //endregion
  //region Additional functionality
  /**
   * Makes a copy of this model, assigning the specified id or a generated id and also allowing you to pass field values to
   * the created copy.
   *
   * ```
   * const record = new Model({ name : 'Super model', hairColor : 'Brown' });
   * const clone = record.copy({ name : 'Super model clone' });
   * ```
   * @param {Number|String|Object} [newId] The id for the copied instance, or any field values to apply
   * (overriding the values from the source record). If no id provided, one will be auto-generated
   * @param {Boolean} [deep] True to also clone children
   * @returns {Core.data.Model} Copy of this model
   * @category Editing
   */
  copy(newId = null, deep) {
    const me = this, data = ObjectHelper.clone(me.data), idField = me.constructor.idField, useDeep = ObjectHelper.isObject(deep) ? deep.deep : deep;
    let id;
    if (newId && typeof newId === "object") {
      id = newId[idField];
      Object.assign(data, newId);
    } else {
      id = newId;
    }
    if (useDeep && me.children) {
      data.children = me.children.map((child) => child.copy(void 0, deep));
    } else {
      delete data.children;
      delete data.expanded;
    }
    if (me.$meta.fields.initable.length > 0) {
      const json = me.toJSON();
      for (const field2 of me.$meta.fields.initable) {
        data[field2.name] = json[field2.name];
      }
    }
    if (newId !== false) {
      data[idField] = id || me.generateId(me.firstStore);
    }
    const copy = new me.constructor(data, null, null, false, true);
    copy.originalInternalId = me.internalId;
    return copy;
  }
  // Copies data using the real field names to trigger setters
  copyData(fromRecord, raw, silent) {
    const propertiesAndValues = {};
    fromRecord.allFields.forEach(({ name: fieldName }) => {
      if (fieldName !== fromRecord.constructor.idField) {
        propertiesAndValues[fieldName] = raw ? fromRecord.get(fieldName) : fromRecord.getValue(fieldName);
      }
    });
    this.set(propertiesAndValues, null, silent);
  }
  /**
   * Removes this record from all stores (and in a tree structure, also from its parent if it has one).
   * @param {Boolean} [silent] Specify `true` to not trigger events. If event is recurring, occurrences won't be updated automatically.
   * @category Editing
   */
  remove(silent = false) {
    const me = this, { parent } = this;
    if (parent) {
      parent.removeChild(me);
    } else if (me.stores.length && !me.isSpecialRow) {
      me.stores.forEach((s) => s.remove(me, silent, false, true));
    }
  }
  // Called by stores before removing the record from the store. Returning false prevents the removal (overridden in
  // ModelLink.js)
  beforeRemove(stores2, records) {
    return super.beforeRemove(stores2, records);
  }
  //endregion
  //region Validation
  /**
   * Check if record has valid data. Default implementation returns true, override in your model to do actual validation.
   * @property {Boolean}
   * @category Editing
   */
  get isValid() {
    return true;
  }
  //endregion
  //region Store
  /**
   * Get the first store that this model is assigned to.
   * @property {Core.data.Store}
   * @readonly
   * @category Misc
   */
  get firstStore() {
    return this.stores.length > 0 && this.stores[0];
  }
  /**
   * Joins this record and any children to specified store, if not already joined.
   * @internal
   * @param {Core.data.Store} store Store to join
   * @category Misc
   */
  joinStore(store) {
    var _a2;
    const me = this, { stores: stores2 } = me;
    if (!stores2.includes(store)) {
      const { unjoinedStores } = me;
      super.joinStore(store);
      store.register(me);
      stores2.push(store);
      if (unjoinedStores.includes(store)) {
        unjoinedStores.splice(unjoinedStores.indexOf(store), 1);
      }
      me.isLoaded && me.children.forEach((child) => child.joinStore(store));
      me.initRelations();
      if (store.tree && !me.isRoot && !((_a2 = store.stm) == null ? void 0 : _a2.isRestoring)) {
        me.instanceMeta(store.id).collapsed = !me.expanded;
      }
    }
  }
  /**
   * Unjoins this record and any children from specified store, if already joined.
   * @internal
   * @param {Core.data.Store} store Store to join
   * @param {Boolean} [isReplacing] `true` if this record is being replaced
   * @category Misc
   */
  unjoinStore(store, isReplacing = false) {
    var _a2, _b, _c;
    const me = this, { stores: stores2, unjoinedStores } = me;
    if (stores2.includes(store)) {
      if (!store.isDestroying) {
        store.unregister(me);
        unjoinedStores.push(store);
      }
      (_b = (_a2 = me.unfilteredChildren || me.children) == null ? void 0 : _a2.forEach) == null ? void 0 : _b.call(_a2, (child) => child.unjoinStore(store, isReplacing));
      stores2.splice(stores2.indexOf(store), 1);
      (_c = super.unjoinStore) == null ? void 0 : _c.call(this, store, isReplacing);
      store.uncacheRelatedRecord(me);
    }
  }
  /**
   * Returns true if this record is contained in the specified store, or in any store if store param is omitted.
   * @internal
   * @param {Core.data.Store} store Store to join
   * @returns {Boolean}
   * @category Misc
   */
  isPartOfStore(store) {
    if (store) {
      return store.includes(this);
    }
    return this.stores.length > 0;
  }
  /**
   * Returns true if this record is not part of any store.
   * @property {Boolean}
   * @readonly
   * @internal
   */
  get isRemoved() {
    return !this.isPartOfStore();
  }
  //endregion
  //region Per instance meta
  /**
   * Used to set per external instance meta data. For example useful when using a record in multiple grids to store some state
   * per grid.
   * @param {String|Object} instanceOrId External instance id or the instance itself, if it has id property
   * @private
   * @category Misc
   */
  instanceMeta(instanceOrId) {
    const { meta } = this, id = instanceOrId.id || instanceOrId;
    if (!meta.map) {
      meta.map = {};
    }
    return meta.map[id] || (meta.map[id] = {});
  }
  /**
   * When called on a group header row returns list of records in that group. Returns `undefined` otherwise.
   * @member {Core.data.Model[]|undefined} groupChildren
   * @category Grouping
   * @readonly
   */
  /**
   * Returns true for a group header record
   * @member {Boolean}
   * @category Grouping
   * @readonly
   */
  get isGroupHeader() {
    return "groupRowFor" in this.meta;
  }
  get isGroupFooter() {
    return "groupFooterFor" in this.meta;
  }
  get isSpecialRow() {
    var _a2;
    return Boolean((_a2 = this.meta) == null ? void 0 : _a2.specialRow);
  }
  get $original() {
    return this.isLinked ? this.proxyMeta.originalRecord : this;
  }
  //endregion
  static get nonPersistableFields() {
    const me = this;
    if (!me._nonPersistableFields) {
      me._nonPersistableFields = {};
      me.allFields.forEach((field2) => {
        if (!field2.persist || field2.calculated) {
          me._nonPersistableFields[field2.name] = 1;
        }
      });
    }
    return me._nonPersistableFields;
  }
  static get alwaysWriteFields() {
    const me = this;
    if (!me._alwaysWriteFields) {
      me._alwaysWriteFields = [];
      me.allFields.forEach((field2) => {
        if (field2.alwaysWrite) {
          me._alwaysWriteFields.push(field2.name);
        }
      });
    }
    return me._alwaysWriteFields;
  }
  // Id with spaces and dots replaced by -, for safe usage as an id in DOM
  get domId() {
    return typeof this.id === "string" ? this.id.replace(/[ .]/g, "-") : this.id;
  }
  //region Extract config
  // These functions are not meant to be called by any code other than Base#getCurrentConfig()
  // Convert custom modelClass to string, keeping custom fields
  static toJavaScriptValue(options) {
    const { names } = this.$meta, className = names[names.length - 2], superName = names[names.length - 3];
    return `class ${className} extends ${superName} { static fields = ${StringHelper.toJavaScriptValue(this.fields, options)}; }`;
  }
  // Get fields current values
  getCurrentConfig(options) {
    const { data, children } = this, { defaultValues, applyConfigs } = this.constructor, result = applyConfigs ? super.getCurrentConfig(options) : {};
    if (result) {
      for (const field2 of this.fields) {
        if (field2.persist) {
          const value = ObjectHelper.getPath(data, field2.dataSource);
          if (!field2.isEqual(value, defaultValues[field2.name])) {
            ObjectHelper.setPath(result, field2.dataSource, Base.processConfigValue(value, options));
          }
        }
      }
      if (children) {
        if (Array.isArray(children)) {
          result.children = [];
          for (const child of children) {
            result.children.push(child.getCurrentConfig(options));
          }
        } else {
          result.children = children;
        }
      }
      if (this.hasGeneratedId) {
        delete result.id;
      }
      delete result.parentId;
      delete result.parentIndex;
    }
    return result;
  }
  //endregion
};
var Model = _Model;
/**
 * Override in a subclass of Model to define relations to records in other stores.
 *
 * Always defined on the "one" side, not the "many" side.
 *
 * Expects an object where keys are relation names and values are {@link #typedef-RelationConfig relation configs}.
 *
 * This snippet will define a relation called `team`, allowing access to the foreign record via `player.team`. It
 * will point to a record in the `teamStore` (must be available as `record.firstStore.teamStore)` with an id
 * matching the players `teamId` field. The team record in turn, will have a field called `players` which is a
 * collection of all players in the team.
 *
 * ```javascript
 * class Player extends Model {
 *     static relations = {
 *         // Define a relation between a player and a team
 *         team : {
 *             foreignKey            : 'teamId',
 *             foreignStore          : 'teamStore',
 *             relatedCollectionName : 'players'
 *         }
 *     }
 * }
 *
 * const teamStore = new Store({
 *     data : [
 *         { id : 1, name : 'Brynas' },
 *         { id : 2, name : 'Leksand' }
 *     ]
 * });
 *
 * const playerStore = new Store({
 *     modelClass : Player,
 *     // Matches foreignStore, allowing records of playerStore to find the related store
 *     teamStore,
 *     data       : [
 *         // teamId is specified as foreignKey, will be used to match the team
 *         { id : 1, name : 'Nicklas Backstrom', teamId : 1  },
 *         { id : 2, name : 'Elias Lindholm',   teamId : 1  },
 *         { id : 3, name : 'Filip Forsberg',  teamId : 2  }
 *     ],
 * }
 *
 * playerStore.first.team.name // > Brynas
 * playerStore.last.team.name // > Leksand
 * teamStore.first.players // > [nick, elias]
 * teamStore.last.players // > [filip]
 * ```
 *
 * To access the related record from the many side, use dot notation for the field name. For example in a Grid
 * column:
 *
 * ```javascript
 * const grid = new Grid({
 *    store : playerStore,
 *    columns : [
 *        { field : 'name', text : 'Name' },
 *        { field : 'team.name', text : 'Team' }
 *    ]
 * });
 * ```
 *
 * @member {Object<String,RelationConfig>} relations
 * @static
 */
__publicField(Model, "relations", null);
Model._idField = "id";
Model._internalIdCounter = 1;
Model._assignedIdField = false;
Model.exposeProperties();
Model._$name = "Model";

// ../Core/lib/Core/data/Duration.js
var Duration = class {
  /**
   * Duration constructor.
   * @param {Number|String} magnitude Duration magnitude value or a duration + magnitude string ('2h', '4d')
   * @param {String} [unit] Duration unit value
   */
  constructor(magnitude, unit) {
    if (typeof magnitude === "number" || magnitude === null) {
      this._magnitude = magnitude;
      this._unit = unit;
    } else {
      if (typeof magnitude === "string") {
        Object.assign(this, DateHelper.parseDuration(magnitude));
      }
      if (typeof magnitude === "object") {
        Object.assign(this, magnitude);
      }
    }
  }
  /**
   * Get/Set numeric magnitude `value`.
   * @property {Number}
   */
  get magnitude() {
    return this._magnitude;
  }
  set magnitude(value) {
    this._magnitude = typeof value === "number" && value;
  }
  /**
   * Get/Set duration unit to use with the current magnitude value.
   * Valid values are:
   * - "millisecond" - Milliseconds
   * - "second" - Seconds
   * - "minute" - Minutes
   * - "hour" - Hours
   * - "day" - Days
   * - "week" - Weeks
   * - "month" - Months
   * - "quarter" - Quarters
   * - "year"- Years
   *
   * @member {'millisecond'|'second'|'minute'|'hour'|'day'|'week'|'month'|'quarter'|'year'}
   */
  get unit() {
    return this._unit;
  }
  set unit(value) {
    this._unit = DateHelper.parseTimeUnit(value);
  }
  get isValid() {
    return this._magnitude != null && Boolean(DateHelper.normalizeUnit(this._unit));
  }
  /**
   * The `milliseconds` property is a read only property which returns the number of milliseconds in this Duration
   * @property {Number}
   * @readonly
   */
  get milliseconds() {
    return this.isValid ? Math.round(DateHelper.asMilliseconds(this._magnitude, this._unit)) : 0;
  }
  /**
   * Returns truthy value if this Duration equals the passed value.
   * @param {Core.data.Duration} value
   * @returns {Boolean}
   */
  isEqual(value) {
    return Boolean(value) && this._magnitude != null && value._magnitude != null && this.milliseconds === value.milliseconds;
  }
  toString(useAbbreviation) {
    const me = this, abbreviationFn = useAbbreviation ? "getShortNameOfUnit" : "getLocalizedNameOfUnit";
    return me.isValid ? `${me._magnitude} ${DateHelper[abbreviationFn](me._unit, me._magnitude !== 1)}` : "";
  }
  toJSON() {
    return this.toString();
  }
  valueOf() {
    return this.milliseconds;
  }
  diff(otherDuration) {
    return new Duration({
      unit: this.unit,
      magnitude: DateHelper.as(this.unit, this.milliseconds - otherDuration.milliseconds)
    });
  }
};
Duration._$name = "Duration";

// ../Core/lib/Core/util/CollectionFilter.js
var nestedValueReducer = (object, path) => object == null ? void 0 : object[path];
var relativeDateUnitRegExp = /^is(this|next|last)(week|month|year)$/i;
var relativeDateOperators = [
  "isToday",
  "isTomorrow",
  "isYesterday",
  "isThisWeek",
  "isNextWeek",
  "isLastWeek",
  "isThisMonth",
  "isNextMonth",
  "isLastMonth",
  "isThisYear",
  "isNextYear",
  "isLastYear",
  "isYearToDate"
];
var _CollectionFilter = class extends Base.mixin(Identifiable_default) {
  static get defaultConfig() {
    return {
      /**
       * The value against which to compare the {@link #config-property} of candidate objects.
       * @config {*}
       */
      value: null,
      /**
       * The operator to use when comparing a candidate object's {@link #config-property} with this CollectionFilter's {@link #config-value}.
       * May be:
       * `'='`, `'!='`, `'>'`, `'>='`, `'<'`, `'<='`, `'*'`,
       * `'startsWith'`, `'endsWith'`, `'isIncludedIn'`, `'includes'`, `'doesNotInclude'`,
       * `'empty'`, `'notEmpty'`, `'between'`, `'notBetween'`, `'sameDay'`,
       * `'isToday'`, `'isTomorrow'`, `'isYesterday'`, `'isThisWeek'`, `'isLastWeek'`, `'isNextWeek'`, `'isThisMonth'`,
       * `'isLastMonth'`, `'isNextMonth'`, `'isThisYear'`, `'isLastYear'`, `'isNextYear'`, `'isYearToDate`',
       * `'isTrue'`, `'isFalse'`
       * @config {'='|'!='|'>'|'>='|'<'|'<='|'*'|'startsWith'|'endsWith'|'isIncludedIn'|'isNotIncludedIn'|'includes'|'doesNotInclude'|'empty'|'notEmpty'|'between'|'notBetween'|'sameDay'|'isToday'|'isTomorrow'|'isYesterday'|'isThisWeek'|'isLastWeek'|'isNextWeek'|'isThisMonth'|'isLastMonth'|'isNextMonth'|'isThisYear'|'isLastYear'|'isNextYear'|'isYearToDate'|'isTrue'|'isFalse'}
       */
      operator: null,
      /**
       * May be used in place of the {@link #config-property}, {@link #config-value} and {@link #config-property} configs. A function which
       * accepts a candidate object and returns `true` or `false`
       * @config {Function}
       */
      filterBy: null,
      /**
       * A function which accepts a value extracted from a candidate object using the {@link #config-property} name, and
       * returns the value which the filter should use to compare against its {@link #config-value}.
       * @config {Function}
       */
      convert: null,
      /**
       * Configure as `false` to have string comparisons case insensitive.
       * @config {Boolean}
       * @default
       */
      caseSensitive: true,
      /**
       * The `id` of this Filter for when used by a {@link Core.util.Collection} Collection.
       * By default the `id` is the {@link #config-property} value.
       * @config {String}
       */
      id: null,
      // Type is required to process the Date value in State API. Store doesn't always know about field type to
      // process filter value, when it applies it from the state, e.g. when you don't declare model field as `date`
      // type but provide a Date instance there. When DateColumn is used to shows this field, it could add date
      // filters to the store. When store is applying state it cannot just infer type, because model doesn't
      // declare it. Only column knows. So to properly process the Date instance for the filter State API would
      // have to process the field additionally, checking model field type and column type. So it is simpler to
      // make Filter to put this information. That way when filter is instantiated by the store, it can gracefully
      // handle value processing, converting date string to the Date instance.
      // Date is the only known value type so far which requires this processing.
      type: null,
      /**
       * Setting the `internal` config on a filter means that it is a fixed part of your store's operation.
       *
       * {@link Core.data.Store#function-clearFilters} does not remove `internal` filters. If you add an
       * `internal` filter, you must explicitly remove it if it is no longer required.
       *
       * Grid features which offer column-based filtering do *not* ingest existing store filters on
       * their data field if the filter is `internal`
       * @config {Boolean}
       * @default false
       */
      internal: null,
      /**
       * When `true`, the filter will not be applied.
       * @config {Boolean}
       * @default
       */
      disabled: false
    };
  }
  static get configurable() {
    return {
      /**
       * The name of a property of candidate objects which yields the value to compare against this CollectionFilter's {@link #config-value}.
       * @member {String} property
       */
      /**
       * The name of a property of candidate objects which yields the value to compare against this CollectionFilter's {@link #config-value}.
       * @config {String}
       */
      property: null
    };
  }
  construct(config) {
    if (typeof config === "function") {
      config = {
        filterBy: config
      };
    }
    if (!config.type) {
      if (DateHelper.isDate(config.value) || Array.isArray(config.value) && config.value.every(DateHelper.isDate)) {
        config.type = "date";
      } else if (config.value instanceof Duration) {
        config.type = "duration";
      }
    } else {
      if (config.type === "date" && config.value != null && !Array.isArray(config.value)) {
        config.value = new Date(config.value);
      } else if (config.type === "duration" && config.value != null && !Array.isArray(config.value)) {
        config.value = new Duration(config.value);
      }
    }
    super.construct(config);
  }
  /**
   * When in a Collection (A Collection holds its Filters in a Collection), we need an id.
   * @property {String}
   * @private
   */
  get id() {
    if (!this._id) {
      if (this.internal) {
        this._id = _CollectionFilter.generateId(`b-internal-${this.property}-filter-`);
      } else {
        this._id = this.property || _CollectionFilter.generateId("b-filter-");
      }
    }
    return this._id;
  }
  set id(id) {
    this._id = id;
  }
  onChange(propertyChanged) {
    var _a2;
    const me = this;
    if (!me.isConfiguring && ((_a2 = me.owner) == null ? void 0 : _a2.onFilterChanged) && !me.owner.isConfiguring) {
      me.owner.onFilterChanged(me, propertyChanged);
    }
  }
  get filterBy() {
    return this._filterBy || this.defaultFilterBy;
  }
  /**
   * May be used in place of the {@link #config-property}, {@link #config-value} and {@link #config-property} configs. A function which
   * accepts a candidate object and returns `true` or `false`
   * @type {Function}
   */
  set filterBy(filterBy) {
    this._filterBy = filterBy;
  }
  defaultFilterBy(candidate) {
    const me = this;
    let candidateValue;
    if (candidate.isModel) {
      candidateValue = candidate.getValue(me.property);
    } else if (me._propertyItems.length > 1) {
      candidateValue = me._propertyItems.reduce(nestedValueReducer, candidate);
    } else {
      candidateValue = candidate[me.property];
    }
    return me[me.operator](me.convert(candidateValue));
  }
  updateProperty(property) {
    this._propertyItems = property.split(".");
    this.onChange("property");
  }
  /**
   * The value against which to compare the {@link #config-property} of candidate objects.
   * @type {*}
   */
  set value(value) {
    const me = this;
    me._value = value;
    if (Array.isArray(value) && ({ date: 1, duration: 1 }[me.type] || value.length > 0 && typeof value[0] === "string")) {
      me._filterValue = value.map((v) => me.convert(v));
    } else if (!me.caseSensitive && Array.isArray(value) && value.length > 0 && typeof value[0] === "string") {
      me._filterValue = value.map((s) => s == null ? void 0 : s.toLowerCase());
    } else if (!me.caseSensitive && typeof value === "string") {
      me._filterValue = value.toLowerCase();
    } else {
      me._filterValue = me.convert(value);
    }
    me.onChange("value");
  }
  get value() {
    return this._value;
  }
  get filterValue() {
    return this._filterValue;
  }
  /**
   * The operator to use when comparing a candidate object's {@link #config-property} with this CollectionFilter's {@link #config-value}.
   * May be:
   * `'='`, `'!='`, `'>'`, `'>='`, `'<'`, `'<='`, `'*'`,
   * `'startsWith'`, `'endsWith'`, `'isIncludedIn'`, `'includes'`, `'doesNotInclude'`,
   * `'empty'`, `'notEmpty'`, `'between'`, `'notBetween'`,
   * `'isToday'`, `'isTomorrow'`, `'isYesterday'`, `'isThisWeek'`, `'isLastWeek'`, `'isNextWeek'`, `'isThisMonth'`,
   * `'isLastMonth'`, `'isNextMonth'`, `'isThisYear'`, `'isLastYear'`, `'isNextYear'`, `'isYearToDate`',
   * `'isTrue'`, `'isFalse'`
   * @type {'='|'!='|'>'|'>='|'<'|'<='|'*'|'startsWith'|'endsWith'|'isIncludedIn'|'isNotIncludedIn'|'includes'|'doesNotInclude'|'empty'|'notEmpty'|'between'|'notBetween'|'isToday'|'isTomorrow'|'isYesterday'|'isThisWeek'|'isLastWeek'|'isNextWeek'|'isThisMonth'|'isLastMonth'|'isNextMonth'|'isThisYear'|'isLastYear'|'isNextYear'|'isYearToDate'|'isTrue'|'isFalse'}
   */
  set operator(operator) {
    this._operator = operator;
    this.onChange("operator");
  }
  get operator() {
    const me = this;
    if (me._operator) {
      return me._operator;
    }
    if (Array.isArray(me.filterValue)) {
      return "isIncludedIn";
    }
    return typeof me.filterValue === "string" ? "*" : "=";
  }
  convert(value) {
    var _a2;
    if (this.operator !== "sameTime" && !(typeof this.filterValue === "string" && value instanceof Date)) {
      if (this.operator === "sameDay") {
        value = DateHelper.clearTime(value);
      }
      value = (_a2 = value == null ? void 0 : value.valueOf()) != null ? _a2 : value;
    }
    value = !this.caseSensitive && typeof value === "string" ? value.toLowerCase() : value;
    return value;
  }
  filter(candidate) {
    return this.filterBy(candidate);
  }
  startsWith(v) {
    return String(v).startsWith(this.filterValue);
  }
  endsWith(v) {
    return String(v).endsWith(this.filterValue);
  }
  isIncludedIn(v) {
    return this.filterValue.length === 0 || this.filterValue.includes(v);
  }
  isNotIncludedIn(v) {
    return !this.isIncludedIn(v);
  }
  includes(v) {
    return this.filterValue.length === 0 || String(v).includes(this.filterValue);
  }
  doesNotInclude(v) {
    return !this.includes(v);
  }
  sameTime(v) {
    return DateHelper.isSameTime(v, this.filterValue);
  }
  sameDay(v) {
    return v === this.filterValue;
  }
  "="(v) {
    return ObjectHelper.isEqual(v, this.filterValue);
  }
  "!="(v) {
    return !ObjectHelper.isEqual(v, this.filterValue);
  }
  ">"(v) {
    return ObjectHelper.isMoreThan(v, this.filterValue);
  }
  ">="(v) {
    return ObjectHelper.isMoreThan(v, this.filterValue) || ObjectHelper.isEqual(v, this.filterValue);
  }
  "<"(v) {
    return ObjectHelper.isLessThan(v, this.filterValue);
  }
  "<="(v) {
    return ObjectHelper.isLessThan(v, this.filterValue) || ObjectHelper.isEqual(v, this.filterValue);
  }
  "*"(v) {
    return ObjectHelper.isPartial(v, this.filterValue);
  }
  between(v) {
    const [start, end] = this._filterValue;
    return (ObjectHelper.isMoreThan(v, start) || ObjectHelper.isEqual(v, start)) && (ObjectHelper.isLessThan(v, end) || ObjectHelper.isEqual(v, end));
  }
  notBetween(v) {
    return !this.between(v);
  }
  empty(v) {
    return v === void 0 || v === null || String(v).length === 0;
  }
  notEmpty(v) {
    return !this.empty(v);
  }
  isToday(v) {
    return this.between(v);
  }
  isTomorrow(v) {
    return this.between(v);
  }
  isYesterday(v) {
    return this.between(v);
  }
  isThisWeek(v) {
    return this.between(v);
  }
  isNextWeek(v) {
    return this.between(v);
  }
  isLastWeek(v) {
    return this.between(v);
  }
  isThisMonth(v) {
    return this.between(v);
  }
  isNextMonth(v) {
    return this.between(v);
  }
  isLastMonth(v) {
    return this.between(v);
  }
  isThisYear(v) {
    return this.between(v);
  }
  isNextYear(v) {
    return this.between(v);
  }
  isLastYear(v) {
    return this.between(v);
  }
  isYearToDate(v) {
    return this.between(v);
  }
  isTrue(v) {
    return v === true;
  }
  isFalse(v) {
    return v === false;
  }
  // Fill in actual dates relative to now
  setRelativeDateValues() {
    this._filterValue = _CollectionFilter.getRelativeDateRange(this._operator).map((date) => date.valueOf());
  }
  static getRelativeDateRange(relativeExpr, now2 = /* @__PURE__ */ new Date()) {
    let todayStart, tomorrowStart, parts, oneTimeUnit, unitStart, which, timeUnit;
    switch (relativeExpr) {
      case "isYearToDate":
        return [DateHelper.floor(now2, "1 year"), now2];
      case "isToday":
        todayStart = DateHelper.floor(now2, "1 day");
        return [todayStart, DateHelper.add(todayStart, 1, "day")];
      case "isYesterday":
        todayStart = DateHelper.floor(now2, "1 day");
        return [DateHelper.add(todayStart, -1, "day"), todayStart];
      case "isTomorrow":
        tomorrowStart = DateHelper.getStartOfNextDay(now2);
        return [tomorrowStart, DateHelper.add(tomorrowStart, 1, "day")];
      case "isThisWeek":
      case "isNextWeek":
      case "isLastWeek":
      case "isThisMonth":
      case "isNextMonth":
      case "isLastMonth":
      case "isThisYear":
      case "isNextYear":
      case "isLastYear":
        parts = relativeExpr.toLowerCase().match(relativeDateUnitRegExp);
        if (!parts) {
          throw new Error(`Unrecognized relative date expression: ${relativeExpr}`);
        }
        [, which, timeUnit] = parts;
        oneTimeUnit = `1 ${timeUnit}`;
        unitStart = DateHelper.floor(now2, oneTimeUnit);
        if (which === "next") {
          unitStart = DateHelper.add(unitStart, 1, timeUnit);
        } else if (which === "last") {
          unitStart = DateHelper.add(unitStart, -1, timeUnit);
        }
        return [unitStart, DateHelper.add(unitStart, 1, timeUnit)];
    }
  }
  // Accepts an array or a Collection
  static generateFiltersFunction(filters) {
    if (!filters || !filters.length && !filters.count) {
      return FunctionHelper.returnTrue;
    }
    for (const filter2 of filters) {
      if (filter2.type === "date" && relativeDateOperators.includes(filter2._operator)) {
        filter2.setRelativeDateValues();
      }
    }
    return function(candidate) {
      let match = true;
      for (const filter2 of filters) {
        if (!filter2.disabled) {
          match = filter2.filter(candidate);
        }
        if (!match) {
          break;
        }
      }
      return match;
    };
  }
};
var CollectionFilter = _CollectionFilter;
__publicField(CollectionFilter, "$name", "CollectionFilter");
CollectionFilter._$name = "CollectionFilter";

// ../Core/lib/Core/helper/IdHelper.js
var idCounts2 = ObjectHelper.getPathDefault(globalThis, "bryntum.idCounts", /* @__PURE__ */ Object.create(null));
var IdHelper = class {
  /**
   * Generate a new id, using IdHelpers internal counter and a prefix
   * @param {String} prefix Id prefix
   * @returns {String} Generated id
   */
  static generateId(prefix = "generatedId") {
    return prefix + (idCounts2[prefix] = (idCounts2[prefix] || 0) + 1);
  }
};
IdHelper._$name = "IdHelper";

// ../Core/lib/Core/util/CollectionSorter.js
var CollectionSorter = class extends Base {
  static get defaultConfig() {
    return {
      /**
       * The name of a property of collection objects which yields the value to sort by.
       * @config {String}
       */
      property: null,
      /**
       * The direction to sort in, `'ASC'` or `'DESC'`
       * @config {'ASC'|'DESC'}
       * @default
       */
      direction: "ASC",
      /**
       * A function which takes the place of using {@link #config-property} and {@link #config-direction}.
       * The function is passed two objects from the collection to compare and must return the comparison result.
       * @config {Function}
       */
      sortFn: null,
      /**
       * When using {@link #config-property}, this may be specified as a function which takes the raw
       * property value and returns the value to actually sort by.
       * @config {Function}
       */
      convert: null,
      /**
       * The `id` of this Sorter for when used by a {@link Core.util.Collection} Collection.
       * By default the `id` is the {@link #config-property} value.
       * @config {String}
       */
      id: null,
      /**
       * Use `localeCompare()` when sorting, which lets the browser sort in a locale specific order. Set to `true`,
       * a locale string or a locale config to enable.
       *
       * Enabling this has big negative impact on sorting performance. For more info on `localeCompare()`, see
       * [MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/localeCompare).
       *
       * ```javascript
       * collection.addSorter({ field: 'name', useLocaleSort : 'sv-SE' });
       * ```
       *
       * @config {Boolean|String|Object}
       * @default false
       */
      useLocaleSort: null
    };
  }
  construct(config) {
    if (typeof config === "function") {
      config = {
        sortFn: config
      };
    }
    super.construct(config);
  }
  /**
   * When in a Collection (A Collection holds its Sorters in a Collection), we need an id.
   * @property {String}
   * @private
   */
  get id() {
    return this._id || (this._id = this.property || IdHelper.generateId("b-sorter"));
  }
  set id(id) {
    this._id = id;
  }
  set sortFn(sortFn2) {
    this._sortFn = sortFn2;
  }
  get sortFn() {
    if (this._sortFn) {
      return this._sortFn;
    }
    return this.defaultSortFn;
  }
  /**
   * Default sortFn used when no sortFn specified. Uses the {@link #config-property},
   * {@link #config-direction}, and {@link #config-convert}.
   * @private
  */
  defaultSortFn(lhs, rhs) {
    const me = this, { convert, property, useLocaleSort } = me, multiplier = me.direction.toLowerCase() === "desc" ? -1 : 1;
    lhs = lhs[property];
    rhs = rhs[property];
    if (convert) {
      lhs = convert(lhs);
      rhs = convert(rhs);
    }
    if (useLocaleSort && lhs != null && rhs != null && typeof lhs === "string") {
      if (useLocaleSort === true) {
        return String(lhs).localeCompare(rhs);
      }
      if (typeof useLocaleSort === "string") {
        return String(lhs).localeCompare(rhs, useLocaleSort);
      }
      if (typeof useLocaleSort === "object") {
        return String(lhs).localeCompare(rhs, useLocaleSort.locale, useLocaleSort);
      }
    }
    return (lhs > rhs ? 1 : lhs < rhs ? -1 : 0) * multiplier;
  }
  static generateSortFunction(sorters, tieBreaker) {
    const items2 = sorters.isCollection ? sorters.values : sorters, n = items2.length;
    return (lhs, rhs) => {
      let comp, i;
      for (i = 0; i < n; ++i) {
        comp = items2[i].sortFn(lhs, rhs);
        if (comp) {
          return comp;
        }
      }
      return tieBreaker ? tieBreaker(lhs, rhs) : 0;
    };
  }
};
CollectionSorter._$name = "CollectionSorter";

// ../Core/lib/Core/util/Collection.js
var return0 = () => 0;
var reverseNumericSortFn = (a, b) => b - a;
var filteredIndicesProperty = Symbol("filteredIndicesProperty");
var emptyArray2 = Object.freeze([]);
var sortEvent = Object.freeze({
  action: "sort",
  added: emptyArray2,
  removed: emptyArray2,
  replaced: emptyArray2
});
var filterEvent = Object.freeze({
  action: "filter",
  added: emptyArray2,
  removed: emptyArray2,
  replaced: emptyArray2
});
var keyTypes = {
  string: 1,
  number: 1
};
function addItemToIndex(item, index, key) {
  if (index.unique !== false) {
    index.set(key, item);
  } else {
    let set = index.get(key);
    if (!set) {
      set = /* @__PURE__ */ new Set();
      index.set(key, set);
    }
    set.add(item);
  }
}
function removeItemFromIndex(item, index, key) {
  if (index.unique !== false) {
    index.delete(key);
  } else if (index.has(key)) {
    index.get(key).delete(item);
    if (!index.get(key).size) {
      index.delete(key);
    }
  }
}
function doRebuildIndices(values, indices, keyProps, indexCount) {
  for (let i = 0; i < values.length; i++) {
    const item = values[i];
    for (let j = 0; j < indexCount; j++) {
      const keyProp = keyProps[j], key = item[keyProp], index = indices[keyProp];
      addItemToIndex(item, index, key);
    }
  }
}
var Collection = class extends Base.mixin(Events_default) {
  constructor() {
    super(...arguments);
    __publicField(this, "_sortFunction", null);
    __publicField(this, "_addedValues", null);
  }
  //region Config
  static get configurable() {
    return {
      /**
       * Specify the name of the property of added objects which provides the lookup key
       * @config {String}
       * @default
       */
      idProperty: "id",
      /**
       * Specify the names or index configs of properties which are to be indexed for fast lookup.
       *
       * Index configs use the format `{ property : string, unique : boolean }`. Unique indices stores one index
       * per entry, non-unique stores a `Set`. If left out, `unique` defaults to `true`
       *
       * @config {String[]|Object[]}
       * @property {String} property Property to index by
       * @property {Boolean} [unique=true] `true` for unique keys (~primary keys), `false` for non-unique keys
       * (~foreign keys)
       */
      extraKeys: null,
      /**
       * Automatically apply filters on item add.
       * @config {Boolean}
       * @default
       */
      autoFilter: true,
      /**
       * Automatically apply sorters on item add.
       * @config {Boolean}
       * @default
       */
      autoSort: true,
      /**
       * A {@link Core.util.CollectionSorter Sorter}, or Sorter config object, or
       * an array of these, to use to sort this Collection.
       * @config {CollectionSorterConfig[]}
       * @default
       */
      sorters: {
        $config: ["lazy"],
        value: []
      }
    };
  }
  get isCollection() {
    return true;
  }
  //endregion
  //region Init & destroy
  construct(config) {
    this.generation = 0;
    this._values = [];
    super.construct(config);
  }
  doDestroy() {
    var _a2;
    super.doDestroy();
    const me = this;
    me._values.length = 0;
    if (me.isFiltered) {
      me._filteredValues.length = 0;
      me.filters.destroy();
    }
    (_a2 = me._sorters) == null ? void 0 : _a2.destroy();
  }
  //endregion
  //region "CRUD"
  /**
   * Clears this collection.
   */
  clear() {
    const me = this, removed = me._values.slice();
    if (me.totalCount) {
      me._values.length = 0;
      if (me._filteredValues) {
        me._filteredValues.length = 0;
      }
      me._indicesInvalid = true;
      me.generation++;
      me.trigger("change", {
        action: "clear",
        removed
      });
    }
  }
  /**
   * Compares the content of this Collection with the content of the passed Collection or
   * with the passed array. Order insensitive. This returns `true` if the two objects passed
   * contain the same set of items.
   * @param {Core.util.Collection|Array} other The Collection or array to compare with.
   * @param {Function} [map] Optionally a function to convert the items into a comparable object
   * to compare. For example `item => item.id` could be used to compare the ids of the
   * constituent items.
   * @returns {Boolean} `true` if the two objects passed have the same content.
   */
  equals(other, map2) {
    if (other.isCollection) {
      other = other.values;
    }
    if (other.length === this.count) {
      let { values } = this;
      if (map2) {
        other = other.map(map2);
        values = values.map(map2);
      }
      return ArrayHelper.delta(other, values).inBoth.length === this.count;
    }
    return false;
  }
  /**
   * Replaces the internal `values` array with the passed `values`, or `filteredValues` array with the passed `filteredValues`.
   * If `filteredValues` are not passed explicitly, but storage is filtered, decides internally `values` or `filteredValues` should
   * be replaced by passed `values`.
   *
   * Note that this takes ownership of the array, and the array must not be mutated by outside code.
   *
   * This is an internal utility method, not designed for use by application code.
   *
   * @param {Object} params Values and parameters to replace
   * @param {Object[]} params.values The new `values` array
   * @param {Object[]} [params.filteredValues] The new `filteredValues` array. Applicable only when storage is filtered.
   * @param {Boolean} [params.silent=false] If true, `change` event will not be fired
   * @param {Boolean} [params.isNewDataset=false] If true, `values` is a new dataset
   * @fires change
   * @internal
   */
  replaceValues({ values, filteredValues, silent = false, isNewDataset = false }) {
    const me = this;
    let replacedValues, replacedFilteredValues;
    if (me.isFiltered && !isNewDataset) {
      const filteredPassed = Boolean(filteredValues);
      if (!filteredPassed) {
        filteredValues = values.slice();
        values = null;
      } else if (values) {
        replacedValues = me._values;
        me._values = values.slice();
      }
      replacedFilteredValues = me._filteredValues;
      me._filteredValues = filteredValues.slice();
    } else {
      replacedValues = me._values;
      me._values = values.slice();
      filteredValues = null;
      if (me.isFiltered && isNewDataset && me.autoFilter) {
        me._filterFunction = null;
        me._filteredValues = me._values.filter(me.filterFunction);
      } else if (me._filteredValues) {
        me._filteredValues.length = 0;
      }
    }
    me._indicesInvalid = true;
    me._addedValues = void 0;
    me.generation++;
    if (!silent) {
      me.trigger("change", {
        action: "replaceValues",
        replacedValues,
        replacedFilteredValues,
        values,
        filteredValues
      });
    }
  }
  set values(values) {
    this.invalidateIndices();
    this.splice(0, this._values.length, values);
  }
  /**
   * The set of values of this Collection. If this Collection {@link #property-isFiltered},
   * this yields the filtered data set.
   *
   * Setting this property replaces the data set.
   * @property {Object[]}
   */
  get values() {
    return this.isFiltered ? this._filteredValues : this._values;
  }
  /**
   * The set of filtered values of this Collection (those matching the current filters).
   * @property {Object[]}
   * @private
   */
  get filteredValues() {
    return this._filteredValues;
  }
  /**
   * Iterator that allows you to do `for (const item of collection)`
   */
  [Symbol.iterator]() {
    return this.values[Symbol.iterator]();
  }
  /**
   * Executes the passed function for each item in this Collection, passing in the item,
   * ths index, and the full item array.
   * @param {Function} fn The function to execute.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to include all items, bypassing filters.
   */
  forEach(fn2, ignoreFilters = false) {
    (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).forEach(fn2);
  }
  /**
   * Extracts ths content of this Collection into an array based upon the passed
   * value extraction function.
   * @param {Function} fn A function, which, when passed an item, returns a value to place into the resulting array.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to process an item even if it is filtered out.
   * @returns {Object[]} An array of values extracted from this Collection.
   */
  map(fn2, ignoreFilters = false) {
    return (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).map(fn2);
  }
  /**
   * Returns the first item in this Collection which elicits a *truthy* return value from the passed function.
   * @param {Function} fn A function, which, when passed an item, returns `true` to select it as the item to return.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to include filtered out items.
   * @returns {Object} The matched item, or `undefined`.
   */
  find(fn2, ignoreFilters = false) {
    return (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).find(fn2);
  }
  get first() {
    return this.values[0];
  }
  get last() {
    return this.values[this.count - 1];
  }
  /**
   * The set of all values of this Collection regardless of filters applied.
   * @readonly
   * @property {Object[]}
   */
  get allValues() {
    return this._values;
  }
  /**
   * The set of values added to this Collection since the last sort or replaceValues operation.
   * @private
   * @readonly
   * @property {Object[]}
   */
  get addedValues() {
    return this._addedValues;
  }
  /**
   * This method ensures that every item in this Collection is replaced by the matched by
   * `id` item in the other Collection.
   *
   * By default, any items in this Collection which are __not__ in the other Collection are removed.
   *
   * If the second parameter is passed as `false`, then items which are not in the other
   * Collection are not removed.
   *
   * This can be used for example when updating a selected record Collection when a new
   * Store or new store dataset arrives. The selected Collection must reference the latest
   * versions of the selected record `id`s
   * @param {Core.util.Collection} other The Collection whose items to match.
   */
  match(other, allowRemove = true) {
    const me = this, { _values } = me, toRemove = [];
    me.forEach((item) => {
      const newInstance = other.get(item.id, true);
      if (newInstance) {
        const index = me.indexOf(item, true), oldInstance = _values[index];
        _values[index] = newInstance;
        me.removeFromIndices(oldInstance);
        me.addToIndices(newInstance);
      } else if (allowRemove) {
        toRemove.push(item);
      }
    });
    if (toRemove.length) {
      me.remove(toRemove);
    }
    if (me.isFiltered) {
      me._filteredValues = me._values.filter(me.filterFunction);
    }
    return toRemove;
  }
  /**
   * Adds items to this Collection. Multiple new items may be passed.
   *
   * By default, new items are appended to the existing values.
   *
   * Any {@link #property-sorters} {@link #property-sorters} present are re-run.
   *
   * Any {@link #property-filters} {@link #property-filters} present are re-run.
   *
   * *Note that if application functionality requires add and remove, the
   * {@link #function-splice} operation is preferred as it performs both
   * operations in an atomic manner*
   * @param  {...Object} items The item(s) to add.
   */
  add(...items2) {
    if (items2.length === 1) {
      this.splice(this._values.length, null, ...items2);
    } else {
      this.splice(this._values.length, null, items2);
    }
  }
  /**
   * Removes items from this Collection. Multiple items may be passed.
   *
   * Any {@link #property-sorters} {@link #property-sorters} present are re-run.
   *
   * Any {@link #property-filters} {@link #property-filters} present are re-run.
   *
   * *Note that if application functionality requires add and remove, the
   * {@link #function-splice} operation is preferred as it performs both
   * operations in an atomic manner*
   * @param  {...Object} items The item(s) to remove.
   */
  remove(...items2) {
    if (items2.length === 1) {
      this.splice(0, ...items2);
    } else {
      this.splice(0, items2);
    }
  }
  /**
   * Moves an individual item, or a block of items to another location.
   * @param {Object|Object[]} items The item/items to move.
   * @param {Object} [beforeItem] the item to insert the first item before. If omitted, the `item`
   * is moved to the end of the Collection.
   * @returns {Number} The new index of the `item`.
   */
  move(items2, beforeItem) {
    items2 = ArrayHelper.asArray(items2);
    while (items2.length && items2[0] === beforeItem) {
      items2.shift();
    }
    if (!items2.length) {
      return;
    }
    const me = this, { _values } = me, itemIndex = me.indexOf(items2[0], true);
    if (items2.length === 1 && _values[itemIndex + 1] === beforeItem) {
      return;
    }
    me.suspendEvents();
    me.remove(items2);
    me.resumeEvents();
    const beforeIndex = beforeItem ? me.indexOf(beforeItem, true) : _values.length;
    if (beforeIndex === -1) {
      throw new Error("Collection move beforeItem parameter must be present in Collection");
    }
    _values.splice(beforeIndex, 0, ...items2);
    me._indicesInvalid = 1;
    me.trigger("change", {
      action: "move",
      items: items2,
      from: itemIndex,
      to: beforeIndex
    });
    return beforeIndex;
  }
  /**
   * The core data set mutation method. Removes and adds at the same time. Analogous
   * to the `Array` `splice` method.
   *
   * Note that if items that are specified for removal are also in the `toAdd` array,
   * then those items are *not* removed then appended. They remain in the same position
   * relative to all remaining items.
   *
   * @param {Number} index Index at which to remove a block of items. Only valid if the
   * second, `toRemove` argument is a number.
   * @param {Object[]|Number} [toRemove] Either the number of items to remove starting
   * at the passed `index`, or an array of items to remove (If an array is passed, the `index` is ignored).
   * @param  {Object[]|Object} [toAdd] An item, or an array of items to add.
   */
  splice(index = 0, toRemove, ...toAdd) {
    const me = this, idProperty = me.idProperty, values = me._values, newIds = {}, removed = [], replaced = [], oldCount = me.totalCount;
    let added, mutated;
    if (me.trigger("beforeSplice", { index, toRemove, toAdd }) === false) {
      return;
    }
    if (toAdd) {
      if (toAdd.length === 1 && Array.isArray(toAdd[0])) {
        toAdd = toAdd[0];
      }
      if (oldCount && toAdd.length) {
        const idIndex = me.indices[idProperty];
        added = [];
        for (let i = 0; i < toAdd.length; i++) {
          const newItem = toAdd[i], id = newItem[idProperty], existingItem = idIndex.get(id), existingIndex = existingItem ? values.indexOf(existingItem) : -1;
          newIds[id] = true;
          if (existingIndex !== -1) {
            if (values[existingIndex] !== newItem) {
              replaced.push([values[existingIndex], newItem]);
              values[existingIndex] = newItem;
            }
          } else {
            added.push(newItem);
          }
        }
      } else {
        added = toAdd;
      }
    }
    if (toRemove) {
      if (typeof toRemove === "number") {
        toRemove = Math.min(toRemove, values.length - index);
        for (let removeIndex = index; toRemove; --toRemove) {
          const id = values[removeIndex][idProperty];
          if (newIds[id]) {
            index++;
            removeIndex++;
          } else {
            removed.push(values[removeIndex]);
            values.splice(removeIndex, 1);
            mutated = true;
          }
        }
      } else {
        let contiguous = added.length === 0, lastIdx;
        toRemove = ArrayHelper.asArray(toRemove);
        const removeIndices = toRemove.reduce((result, item) => {
          const isNumeric = typeof item === "number", idx = isNumeric ? item : me.indexOf(item, true);
          if (contiguous && (lastIdx != null && idx !== lastIdx + 1 || isNumeric)) {
            contiguous = false;
          }
          if (idx >= 0 && idx < oldCount) {
            result.push(idx);
          }
          lastIdx = idx;
          return result;
        }, []).sort(reverseNumericSortFn);
        if (contiguous) {
          if (removeIndices.length) {
            removed.push.apply(removed, toRemove);
            values.splice(removeIndices[removeIndices.length - 1], removeIndices.length);
            mutated = true;
          }
        } else {
          for (let i = 0; i < removeIndices.length; i++) {
            const removeIndex = removeIndices[i];
            if (removeIndex !== -1) {
              const id = values[removeIndex][idProperty];
              if (!newIds[id]) {
                removed.unshift(values[removeIndex]);
                values.splice(removeIndex, 1);
                mutated = true;
              }
            }
          }
        }
      }
      if (removed.length && !me._indicesInvalid) {
        removed.forEach(me.removeFromIndices, me);
      }
    }
    if (added.length) {
      values.splice(Math.min(index, values.length), 0, ...added);
      mutated = true;
      if (!me._indicesInvalid) {
        added.forEach(me.addToIndices, me);
      }
      if (!me._addedValues) {
        me._addedValues = /* @__PURE__ */ new Set();
      }
      for (const value of added) {
        me._addedValues.add(value);
      }
    }
    if (removed.length && me._addedValues) {
      for (const value of removed) {
        me._addedValues.delete(value);
      }
    }
    if (replaced.length && !me._indicesInvalid) {
      replaced.forEach((rep) => {
        me.removeFromIndices(rep[0]);
        me.addToIndices(rep[1]);
      });
    }
    if (mutated || replaced.length) {
      if (me.isSorted) {
        me.onSortersChanged();
      } else if (me.isFiltered) {
        if (me.autoFilter) {
          me.onFiltersChanged({ action: "splice", oldCount: 1 });
        } else {
          me._filteredValues.splice(Math.min(index, me._filteredValues.length), 0, ...added);
        }
      }
      me.generation++;
      me.trigger("change", {
        action: "splice",
        removed,
        added,
        replaced,
        oldCount
      });
    } else {
      me.trigger("noChange", {
        index,
        toRemove,
        toAdd
      });
    }
  }
  /**
   * Change the id of an existing member by mutating its {@link #config-idProperty}.
   * @param {String|Number|Object} item The item or id of the item to change.
   * @param {String|Number} newId The id to set in the existing member.
   */
  changeId(item, newId) {
    const me = this, { idProperty } = me, oldId = keyTypes[typeof item] ? item : item[idProperty], member = me.get(oldId);
    if (member) {
      const existingMember = me.get(newId);
      if (existingMember && member !== existingMember) {
        throw new Error(`Attempt to set item ${oldId} to already existing member's id ${newId}`);
      }
      me.removeIndexEntry(item, idProperty, oldId);
      me.addIndexEntry(item, idProperty, newId);
      member[idProperty] = newId;
    }
  }
  /**
   * Returns the item with the passed `id`. By default, filtered are honoured, and
   * if the item with the requested `id` is filtered out, nothing will be returned.
   *
   * To return the item even if it has been filtered out, pass the second parameter as `true`.
   * @param {*} id The `id` to find.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to return an item even if it is filtered out.
   * @returns {Object} The found item, or `undefined`.
   */
  get(id, ignoreFilters = false) {
    return this.getBy(this.idProperty, id, ignoreFilters);
  }
  getAt(index, ignoreFilters = false) {
    if (this.isFiltered && !ignoreFilters) {
      return this._filteredValues[index];
    } else {
      return this._values[index];
    }
  }
  /**
   * Returns the item with passed property name equal to the passed value. By default,
   * filtered are honoured, and if the item with the requested `id` is filtered out,
   * nothing will be returned.
   *
   * To return the item even if it has been filtered out, pass the third parameter as `true`.
   * @param {String} propertyName The property to test.
   * @param {*} value The value to find.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to return an item even if it is filtered out.
   * @returns {Object} The found item, or `undefined`.
   */
  getBy(propertyName, value, ignoreFilters = false) {
    return this.findItem(propertyName, value, this.isFiltered && ignoreFilters);
  }
  /**
   * The number of items in this collection. Note that this honours filtering.
   * See {@link #property-totalCount};
   * @property {Number}
   * @readonly
   */
  get count() {
    return this.values.length;
  }
  /**
   * The number of items in this collection regardless of filtering.
   * @property {Number}
   * @readonly
   */
  get totalCount() {
    return this._values.length;
  }
  /**
   * The property name used to extract item `id`s from added objects.
   * @member {String} idProperty
   */
  updateIdProperty(idProperty) {
    this.addIndex({ property: idProperty, unique: true });
  }
  //endregion
  //region Sorting
  /**
   * The Collection of {@link Core.util.CollectionSorter Sorters} for this Collection.
   * @member {Core.util.Collection} sorters
   */
  changeSorters(sorters) {
    return new Collection({
      values: ArrayHelper.asArray(sorters),
      internalListeners: {
        change: "onSortersChanged",
        thisObj: this
      }
    });
  }
  /**
   * Adds a Sorter to the Collection of Sorters which are operating on this Collection.
   *
   * A Sorter may be specified as an instantiated {@link Core.util.CollectionSorter}, or a config object for a
   * CollectionSorter of the form
   *
   *     {
   *         property  : 'age',
   *         direction : 'desc'
   *     }
   *
   * Note that by default, a Sorter *replaces* a Sorter with the same `property` to make
   * it easy to change existing Sorters. A Sorter's `id` is its `property` by default. You
   * can avoid this and add multiple Sorters for one property by configuring Sorters with `id`s.
   *
   * A Sorter may also be specified as a function which compares two objects eg:
   *
   *     (lhs, rhs) => lhs.customerDetails.age - rhs.customerDetails.age
   *
   * @param {CollectionSorterConfig} sorter A Sorter configuration object to add to the Collection
   * of Sorters operating on this Collection.
   * @returns {Core.util.CollectionSorter} The resulting Sorter to make it easy to remove Sorters.
   */
  addSorter(sorter) {
    const result = sorter instanceof CollectionSorter ? sorter : new CollectionSorter(sorter);
    this.sorters.add(result);
    return result;
  }
  /**
   * A flag which is `true` if this Collection has active {@link #property-sorters}.
   * @property {Boolean}
   * @readonly
   */
  get isSorted() {
    var _a2;
    return Boolean((_a2 = this._sorters) == null ? void 0 : _a2.count);
  }
  onSortersChanged() {
    const me = this;
    me._sortFunction = null;
    me._addedValues = null;
    me._values.sort(me.sortFunction);
    me.trigger("change", sortEvent);
  }
  /**
   * A sorter function which encapsulates the {@link Core.util.CollectionSorter Sorters}
   * for this Collection.
   * @property {Function}
   * @readonly
   */
  get sortFunction() {
    if (!this._sortFunction) {
      if (this.isSorted) {
        this._sortFunction = CollectionSorter.generateSortFunction(this.sorters.values);
      } else {
        this._sortFunction = return0;
      }
    }
    return this._sortFunction;
  }
  //endregion
  //region Filtering
  /**
   * The Collection of {@link Core.util.CollectionFilter Filters} for this Collection.
   * @property {Core.util.Collection}
   * @readonly
   */
  get filters() {
    if (!this._filters) {
      this._filters = new Collection({
        internalListeners: {
          change: "onFiltersChanged",
          thisObj: this
        }
      });
    }
    return this._filters;
  }
  /**
   * Adds a Filter to the Collection of Filters which are operating on this Collection.
   *
   * A Filter may be an specified as an instantiated {@link Core.util.CollectionFilter
   * CollectionFilter}, or a config object for a CollectionFilter of the form
   *
   *     {
   *         property : 'age',
   *         operator : '>=',
   *         value    : 21
   *     }
   *
   * Note that by default, a Filter *replaces* a Filter with the same `property` to make
   * it easy to change existing Filters. A Filter's `id` is its `property` by default. You
   * can avoid this and add multiple Filters for one property by configuring Filters with `id`s.
   *
   * A Filter may also be specified as a function which filters candidate objects eg:
   *
   *     candidate => candidate.customerDetails.age >= 21
   *
   * @param {CollectionFilterConfig|Core.util.CollectionFilter} filter A Filter or Filter configuration object to add
   * to the Collection of Filters operating on this Collection.
   * @returns {Core.util.CollectionFilter} The resulting Filter to make it easy to remove Filters.
   */
  addFilter(filter2) {
    const result = filter2 instanceof CollectionFilter ? filter2 : new CollectionFilter(filter2);
    this.filters.add(result);
    return result;
  }
  removeFilter(filter2) {
    const { filters } = this;
    if (!filter2.isCollectionFilter) {
      filter2 = filters.get(filter2);
    }
    filters.remove(filter2);
  }
  clearFilters() {
    this.filters.clear();
  }
  /**
   * A flag which is `true` if this Collection has active {@link #property-filters}.
   * @property {Boolean}
   * @readonly
   */
  get isFiltered() {
    return Boolean(this._filters && this._filters.count);
  }
  onFiltersChanged({ action, removed: gone, oldCount }) {
    const me = this, oldDataset = oldCount || action === "clear" && gone.length ? me._filteredValues : me._values;
    me._filterFunction = null;
    me._filteredValues = me._values.filter(me.filterFunction);
    me._indicesInvalid = true;
    const {
      toAdd: added,
      toRemove: removed
    } = ArrayHelper.delta(me._filteredValues, oldDataset, true);
    me.trigger("change", { ...filterEvent, added, removed });
  }
  /**
   * A filter function which encapsulates the {@link Core.util.CollectionFilter Filters}
   * for this Collection.
   * @property {Function}
   * @readonly
   */
  get filterFunction() {
    if (!this._filterFunction) {
      if (this.isFiltered) {
        this._filterFunction = CollectionFilter.generateFiltersFunction(this.filters.values);
      } else {
        this._filterFunction = FunctionHelper.returnTrue;
      }
    }
    return this._filterFunction;
  }
  //endregion
  //region Indexing
  changeExtraKeys(extraKeys) {
    const keys = ArrayHelper.asArray(extraKeys);
    return keys.map((config) => {
      if (typeof config === "string") {
        return { property: config, unique: true };
      }
      return config;
    });
  }
  updateExtraKeys(extraKeys) {
    for (let i = 0; i < extraKeys.length; i++) {
      this.addIndex(extraKeys[i]);
    }
  }
  /**
   * Adds a lookup index for the passed property name or index config. The index is built lazily when an index is
   * searched
   * @internal
   * @param {Object} indexConfig An index config
   * @param {String} indexConfig.property The property name to add an index for
   * @param {Boolean} [indexConfig.unique] Specify `false` to allow multiple entries of the same index, turning
   *   entries into sets
   * @param {Object} [indexConfig.dependentOn] The properties that make the key
   */
  addIndex(indexConfig) {
    const me = this;
    if (indexConfig) {
      (me._indices || (me._indices = {}))[indexConfig.property] = /* @__PURE__ */ new Map();
      Object.assign(me._indices[indexConfig.property], indexConfig);
      me.invalidateIndices();
      if (indexConfig.dependentOn) {
        me.hasCompositeIndex = true;
      }
    }
  }
  /**
   * Return the index of the item with the specified key having the specified value.
   *
   * By default, filtering is taken into account and this returns the index in the filtered dataset if present. To
   * bypass this, pass the third parameter as `true`.
   *
   * Only useful for indices configured with `unique: true`.
   *
   * @param {String} propertyName The name of the property to test.
   * @param {*} value The value to test for.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to return the index in
   * the original data set if the item is filtered out.
   * @returns {Number} The index of the item or `-1` if not found for unique indices
   */
  findIndex(propertyName, value, ignoreFilters = false) {
    const item = this.findItem(propertyName, value, ignoreFilters);
    if (!item) {
      return -1;
    }
    const values = this.isFiltered && !ignoreFilters ? this._filteredValues : this._values;
    return values.indexOf(item);
  }
  /**
   * Return the item with the specified key having the specified value.
   *
   * By default, filtering is taken into account. To bypass this, pass the third parameter as `true`.
   *
   * For indices configured with `unique: false`, a Set of items will be returned.
   *
   * @param {String} propertyName The name of the property to test.
   * @param {*} value The value to test for.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to return the index in
   * the original data set if the item is filtered out.
   * @returns {Object|Set} The found item or Set of items or null
   */
  findItem(propertyName, value, ignoreFilters = false) {
    var _a2;
    const me = this, { isFiltered } = me, index = isFiltered && !ignoreFilters ? me.indices[filteredIndicesProperty][propertyName] : me.indices[propertyName];
    if (index) {
      const item = (_a2 = index.get(value)) != null ? _a2 : typeof value === "string" && value.length && !isNaN(value) && index.get(Number(value)) || null;
      if (item != null) {
        return item;
      }
    } else {
      const values = isFiltered && !ignoreFilters ? me._filteredValues : me._values, count = values.length;
      for (let i = 0; i < count; i++) {
        const item = values[i];
        if (item[propertyName] == value) {
          return item;
        }
      }
    }
    return null;
  }
  removeIndex(propertyName) {
    delete this._indices[propertyName];
    this.hasCompositeIndex = Object.values(this.indices).some((index) => index.dependentOn);
  }
  /**
   * Returns the index of the item with the same `id` as the passed item.
   *
   * By default, filtering is honoured, so if the item in question has been added, but is currently filtered out of
   * visibility, `-1` will be returned.
   *
   * To find the index in the master, unfiltered dataset, pass the second parameter as `true`;
   * @param {Object|String|Number} item The item to find, or an `id` to find.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to find the index in the master, unfiltered data set.
   * @returns {Number} The index of the item, or `-1` if not found.
   */
  indexOf(item, ignoreFilters = false) {
    return this.findIndex(this.idProperty, keyTypes[typeof item] ? item : item[this.idProperty], ignoreFilters);
  }
  /**
   * Returns `true` if this Collection includes an item with the same `id` as the passed item.
   *
   * By default, filtering is honoured, so if the item in question has been added,
   * but is currently filtered out of visibility, `false` will be returned.
   *
   * To query inclusion in the master, unfiltered dataset, pass the second parameter as `true`;
   * @param {Object|String|Number} item The item to find, or an `id` to find.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to find the index in the master, unfiltered data set.
   * @returns {Boolean} True if the passed item is found.
   */
  includes(item, ignoreFilters = false) {
    if (Array.isArray(item)) {
      return item.some((item2) => this.includes(item2));
    }
    return Boolean(this.findItem(this.idProperty, keyTypes[typeof item] ? item : item[this.idProperty], ignoreFilters));
  }
  get indices() {
    if (this._indicesInvalid) {
      this.rebuildIndices();
    }
    return this._indices;
  }
  invalidateIndices() {
    this._indicesInvalid = true;
  }
  /**
   * Called when the Collection is mutated and the indices have been flagged as invalid.
   *
   * Rebuilds the indices object to allow lookup by keys.
   * @internal
   */
  rebuildIndices() {
    const me = this, isFiltered = me.isFiltered, indices = me._indices || (me._indices = {}), keyProps = Object.keys(indices), indexCount = keyProps.length, values = me._values;
    let filteredIndices;
    if (isFiltered) {
      filteredIndices = indices[filteredIndicesProperty] = {};
    }
    for (let i = 0; i < indexCount; i++) {
      const index = indices[keyProps[i]];
      index.clear();
      if (isFiltered) {
        let filteredIndex = filteredIndices[keyProps[i]];
        if (filteredIndex) {
          filteredIndex.clear();
        } else {
          filteredIndex = filteredIndices[keyProps[i]] = /* @__PURE__ */ new Map();
          filteredIndex.unique = index.unique;
        }
      }
    }
    doRebuildIndices(values, indices, keyProps, indexCount);
    if (isFiltered) {
      doRebuildIndices(me._filteredValues, filteredIndices, keyProps, indexCount);
    }
    me._indicesInvalid = false;
  }
  // Returns an array with [indices] or [indices, filteredIndices] if filtering is used
  getIndices(propertyName) {
    const indices = [this.indices[propertyName]];
    if (this.isFiltered) {
      indices.push(this.indices[filteredIndicesProperty][propertyName]);
    }
    return indices;
  }
  /**
   * Add an item to all indices
   * @param {*} item Item already available in the Collection
   * @private
   */
  addToIndices(item) {
    Object.keys(this.indices).forEach((propertyName) => {
      this.addIndexEntry(item, propertyName, item[propertyName]);
    });
  }
  /**
   * Remove an item from all indices
   * @param {*} item Item already available in the Collection
   * @private
   */
  removeFromIndices(item) {
    Object.keys(this.indices).forEach((propertyName) => {
      this.removeIndexEntry(item, propertyName, item[propertyName]);
    });
  }
  /**
   * Remove an entry from an index, and if filtering is used also from the filtered index.
   * @param {*} item Item already available in the Collection
   * @param {String} propertyName Property of the item, will be matched with configured indices
   * @param {*} oldValue Value to remove
   * @private
   */
  removeIndexEntry(item, propertyName, oldValue) {
    this.getIndices(propertyName).forEach((index) => removeItemFromIndex(item, index, oldValue));
  }
  /**
   * Add a new entry to an index, and if filtering is used also to the filtered index.
   * @param {*} item Item already available in the Collection
   * @param {String} propertyName Property of the item, will be matched with configured indices
   * @param {*} value Value to store
   * @private
   */
  addIndexEntry(item, propertyName, value) {
    this.getIndices(propertyName).forEach((index) => addItemToIndex(item, index, value));
  }
  /**
   * Call externally to update indices on item mutation (from Store)
   * @param {*} item Item already available in the Collection
   * @param {Object} wasSet Uses the `wasSet` format from Store, `{ field : { oldValue, newValue } }`
   * @internal
   */
  onItemMutation(item, wasSet) {
    const me = this;
    if (!me._indicesInvalid && Object.keys(me.indices).length > 1) {
      Object.keys(wasSet).forEach((propertyName) => {
        var _a2;
        const indexConfig = me.indices[propertyName];
        if (indexConfig) {
          const { value, oldValue } = wasSet[propertyName];
          me.removeIndexEntry(item, propertyName, oldValue);
          me.addIndexEntry(item, propertyName, value);
        }
        if (me.hasCompositeIndex) {
          const dependentIndex = Object.values(me.indices).find((index) => {
            var _a3;
            return (_a3 = index.dependentOn) == null ? void 0 : _a3[propertyName];
          });
          if (dependentIndex) {
            const keysAndOldValues = {};
            for (const o in dependentIndex.dependentOn) {
              keysAndOldValues[o] = ((_a2 = wasSet[o]) == null ? void 0 : _a2.oldValue) || item[o];
            }
            const oldIndex = item.buildIndexKey(keysAndOldValues);
            me.removeIndexEntry(item, dependentIndex.property, oldIndex);
            me.addIndexEntry(item, dependentIndex.property, item[dependentIndex.property]);
          }
        }
      });
    }
  }
  //endregion
};
Collection._$name = "Collection";

// ../Core/lib/Core/util/Bag.js
var nonPrimitives = /* @__PURE__ */ new WeakMap();
var safeIndexKey = (value) => {
  if (value && typeof value === "object") {
    let substitute = nonPrimitives.get(value);
    if (substitute === void 0) {
      substitute = Symbol("bscik");
      nonPrimitives.set(value, substitute);
    }
    value = substitute;
  }
  return value;
};
var Bag = class {
  constructor(config) {
    const me = this;
    me.generation = 0;
    me.items = /* @__PURE__ */ new Set();
    me.idMap = {};
    me.idProperty = "id";
    if (config) {
      if (config.idProperty) {
        me.idProperty = config.idProperty;
      }
      if (config.values) {
        me.values = config.values;
      }
    }
  }
  /**
   * Returns the item with the passed `id`.
   *
   * @param {*} id The `id` to find.
   * @returns {Object} The found item, or `undefined`.
   */
  get(key) {
    return this.idMap[safeIndexKey(key)];
  }
  /**
   * The number of items in this Bag.
   * @property {Number}
   * @readonly
   */
  get count() {
    return this.items.size;
  }
  /**
   * Adds the passed item(s) to this Bag. Existing items with the same ID
   * will be replaced.
   * @param {Object|Object[]} toAdd Item(s) to add.
   */
  add(...toAdd) {
    if (toAdd.length === 1 && Array.isArray(toAdd[0])) {
      toAdd = toAdd[0];
    }
    const me = this, {
      items: items2,
      idMap,
      idProperty
    } = me, len = toAdd.length;
    for (let i = 0; i < len; i++) {
      const item = toAdd[i], key = keyTypes[typeof item] ? item : safeIndexKey(item[idProperty]), existingItem = idMap[key];
      if (existingItem == null) {
        idMap[key] = item;
        items2.add(item);
        me.generation++;
      } else if (existingItem !== item) {
        idMap[key] = item;
        items2.delete(existingItem);
        items2.add(item);
      }
    }
  }
  /**
   * Returns `nth` item in this Bag.
   * @param {Number} nth The index of the matching item. Negative numbers index for the last item. For example, -1
   * returns the last item, -2 the 2nd to last item etc.
   * @returns {Object} The matched item, or `undefined`.
   */
  at(nth) {
    let item, ret;
    if (nth < 0) {
      nth += this.count;
    }
    for (item of this.items) {
      if (!nth--) {
        ret = item;
        break;
      }
    }
    return ret;
  }
  /**
   * Removes the passed item(s) from this Bag.
   * @param {Object|Object[]} toRemove Item(s) to remove.
   */
  remove(toRemove) {
    toRemove = ArrayHelper.asArray(toRemove);
    const { items: items2, idMap, idProperty } = this, len = toRemove.length;
    for (let i = 0; i < len; i++) {
      const item = toRemove[i], key = keyTypes[typeof item] ? item : safeIndexKey(item[idProperty]), existingItem = idMap[key];
      if (existingItem != null) {
        items2.delete(existingItem);
        delete idMap[key];
        this.generation++;
      }
    }
  }
  clear() {
    this.items.clear();
    this.idMap = {};
    this.generation++;
  }
  /**
   * Returns the number of items in this Bag which elicits a truthy return value from the passed function.
   * @param {Function} fn A function, which, when passed an item, returns `true` to select it as the item to return.
   * @returns {Number} The number of matched items.
   */
  countOf(fn2) {
    let ret = 0, item;
    for (item of this.items) {
      if (fn2(item)) {
        ++ret;
      }
    }
    return ret;
  }
  /**
   * Change the id of an existing member by mutating its idProperty.
   * @param {String|Number|Object} item The item or id of the item to change.
   * @param {String|Number} newId The id to set in the existing member.
   */
  changeId(item, newId) {
    const me = this, { idMap, idProperty } = me, oldId = keyTypes[typeof item] ? item : safeIndexKey(item[idProperty]), member = me.get(oldId);
    if (member) {
      const existingMember = me.get(newId);
      if (existingMember && member !== existingMember) {
        throw new Error(`Attempt to set item ${oldId} to already existing member's id ${newId}`);
      }
      member[idProperty] = newId;
      delete idMap[oldId];
      idMap[newId] = member;
    }
  }
  /**
   * Extracts the matching items from this Bag into an array based upon the passed value filter function.
   * @param {Function} fn A function, which, when passed an item, returns a `true` to place into the resulting array.
   * @param {Object} [thisObj] The `this` reference when the function is called.
   * @returns {Object[]} An array of values extracted from this Bag.
   */
  filter(fn2, thisObj) {
    const { items: items2 } = this, result = [];
    let i = 0;
    items2.forEach((item) => {
      if (fn2.call(thisObj, item, i++, items2)) {
        result.push(item);
      }
    });
    return result;
  }
  /**
   * Returns `true` if this Collection includes an item with the same `id` as the passed item.
   *
   * @param {Object|String|Number} item The item to find, or an `id` to find.
   * @returns {Boolean} True if the passed item is found.
   */
  includes(item) {
    const key = keyTypes[typeof item] ? item : safeIndexKey(item[this.idProperty]);
    return Boolean(this.idMap[key]);
  }
  /**
   * Extracts the content of this Bag into an array based upon the passed
   * value extraction function.
   * @param {Function} fn A function, which, when passed an item, returns a value to place into the resulting array.
   * @param {Object} [thisObj] The `this` reference when the function is called.
   * @returns {Object[]} An array of values extracted from this Bag.
   */
  map(fn2, thisObj) {
    const { items: items2 } = this, result = new Array(items2.size);
    let i = 0;
    items2.forEach((item) => {
      result[i] = fn2.call(thisObj, item, i++, items2);
    });
    return result;
  }
  /**
   * Executes the passed function for each item in this Bag, passing in the item.
   * @param {Function} fn The function to execute.
   * @param {Object} [thisObj] The `this` reference when the function is called.
   */
  forEach(fn2, thisObj) {
    return this.items.forEach(fn2, thisObj);
  }
  /**
   * Returns `nth` item in this Bag which elicits a truthy return value from the provided matcher function `fn`.
   * @param {Function} fn A function which, when passed an item, returns `true` to select the item as a match.
   * @param {Number} [nth=0] The index of the matching item. By default, 0 returns the first item that matches
   * according to `fn`. Negative numbers index for the last item. For example, -1 returns the last matching item,
   * -2 the 2nd to last matching item etc..
   * @returns {Object} The matched item, or `undefined`.
   */
  find(fn2, nth = 0) {
    let item, ret;
    if (nth < 0) {
      nth += this.countOf(fn2);
    }
    for (item of this.items) {
      if (fn2(item) && !nth--) {
        ret = item;
        break;
      }
    }
    return ret;
  }
  /**
   * Iterator that allows you to do for (let item of bag)
   */
  [Symbol.iterator]() {
    return this.items[Symbol.iterator]();
  }
  indexOf(item, matchFn) {
    let index = -1, it;
    for (it of this.items) {
      if (!matchFn || matchFn(it)) {
        ++index;
        if (it === item) {
          return index;
        }
      }
    }
    return -1;
  }
  /**
   * The set of values of this Bag.
   *
   * Setting this property replaces the data set.
   * @property {Object[]}
   */
  get values() {
    return [...this.items];
  }
  set values(values) {
    values = ArrayHelper.asArray(values);
    this.clear();
    this.add.apply(this, values);
    this.generation++;
  }
  /**
   * Sort the values of this Bag using the passed comparison function.
   *
   * Setting this property replaces the data set.
   * @param {Function} fn Comparison function which returns -ve, 0, or +ve
   */
  sort(fn2) {
    this.values = this.values.sort(fn2);
  }
  some(fn2, thisObj) {
    return this.values.some(fn2, thisObj);
  }
};
Bag._$name = "Bag";

// ../Core/lib/Core/data/StoreBag.js
var StoreBag = class extends Bag {
  add(...toAdd) {
    if (toAdd.length === 1 && Array.isArray(toAdd[0])) {
      toAdd = toAdd[0];
    }
    return super.add(...toAdd.filter((record) => record.isPersistable));
  }
};
StoreBag._$name = "StoreBag";

// ../Core/lib/Core/data/mixin/StoreCRUD.js
var StoreCRUD_default = (Target) => class StoreCRUD extends (Target || Base) {
  static get $name() {
    return "StoreCRUD";
  }
  //region Config
  static get defaultConfig() {
    return {
      /**
       * Commit changes automatically
       * @config {Boolean}
       * @default
       * @category Common
       */
      autoCommit: false
    };
  }
  static get properties() {
    return {
      isRemoving: false,
      suspendCount: 0
    };
  }
  //endregion
  //region Events
  /**
   * Fired after removing all records
   * @event removeAll
   * @param {Core.data.Store} source This Store
   */
  /**
   * Fired before committing changes. Return false from handler to abort commit
   * @event beforeCommit
   * @param {Core.data.Store} source This Store
   * @param {Object} changes Modification data
   */
  /**
   * Fired after committing changes
   * @event commit
   * @param {Core.data.Store} source This Store
   * @param {Object} changes Modification data
   */
  /**
   * Fired before records are removed from this store by the {@link #function-remove} or {@link #function-removeAll}.
   * Also fired when removing a child record in a tree store using {@link Core.data.mixin.TreeNode#function-removeChild}.
   * The remove may be vetoed by returning `false` from a handler.
   * @event beforeRemove
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model[]} records The records which are to be removed.
   * @param {Core.data.Model} parent The record from which children are being removed when using a tree store. Only
   * provided when removing a single node.
   * @param {Boolean} isMove This flag is `true` if the child node is being removed by
   * {@link Core.data.mixin.TreeNode#function-appendChild appendChild} to be moved
   * _within the same tree_.
   * @param {Boolean} removingAll This flag is `true` if the operation is removing the store's entire data set.
   * @preventable
   */
  /**
   * Fired before records are added to this store by the {@link #function-add} or {@link #function-insert}. In a tree
   * store, also fired by {@link Core.data.mixin.TreeNode#function-appendChild} and
   * {@link Core.data.mixin.TreeNode#function-insertChild}. The add or insert may be vetoed by returning `false`
   * from a handler.
   * @event beforeAdd
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model[]} records The records which are to be added
   * @param {Core.data.Model} parent The parent node when using a tree store
   * @preventable
   */
  /**
   * Fired after adding/inserting record(s). If the record was added to a parent, the `isChild` flag is set on the
   * event. If it was inserted, event contains `index`
   * @event add
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model[]} records Added records. In case of tree store, if branch is added, only branch root
   * is returned
   * @param {Core.data.Model[]} [allRecords] Flat list of all added records. In case of tree store, if branch is
   * added, all new records are returned, not only branch root
   * @param {Core.data.Model} [parent] If due to an {@link Core/data/mixin/TreeNode#function-appendChild}
   * call, this is the parent node added to.
   * @param {Number} [index] Insertion point in the store's {@link Core.data.Store#config-storage Collection}.
   * @param {Number} [oldIndex] Not used for tree stores. The index of the first record moved.
   * @param {Boolean} [isChild] Flag which is set to true if the records are added to a parent record
   * @param {Boolean} [isExpand] Flag which is set to true if records are added to the store by expanding parent
   * @param {Object} [isMove] An object keyed by the ids of the records which were moved from another
   * position in the store, or from another parent node in the store. The ids of moved records will be
   * property names with a value `true`.
   */
  /**
   * Fired when one or more records are removed
   * @event remove
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model[]} records Array of removed records. In case of tree store, if branch is removed, only branch root
   * is returned
   * @param {Core.data.Model[]} [allRecords] Flat array of all removed records. In case of tree store, if branch is
   * removed, all removed records are returned, not only branch root
   * @param {Core.data.Model} [parent] If due to a {@link Core.data.mixin.TreeNode#function-removeChild removeChild}
   * call, this is the parent node removed from. Only applicable when removing a single tree node.
   * @param {Number} [index] Visible index at which record was removed. In case the record was removed from a collapsed
   * branch, -1 is returned. For tree store, this is only provided when removing a single node.
   * @param {Boolean} [isChild] Flag which is set to true if the record was removed from a parent record
   * @param {Boolean} [isCollapse] Flag which is set to true if records were removed from the store by collapsing parent
   * @param {Boolean} [isMove] Passed as `true` if the remove was part of a move operation within this Store.
   */
  //endregion
  //region Add, insert & remove
  /**
   * Removes a record from this store. Fires a single {@link #event-remove} event passing the removed records.
   * @param {String|String[]|Number|Number[]|Core.data.Model|Core.data.Model[]} records Record/array of records (or record ids) to remove
   * @param {Boolean} [silent] Specify `true` to suppress events/autoCommit
   * @returns {Core.data.Model[]} Removed records
   * @fires beforeRemove
   * @fires remove
   * @fires change
   * @category CRUD
   */
  remove(records, silent = false) {
    const me = this, { storage } = me;
    records = ArrayHelper.asArray(records).reduce((result, r) => {
      r = me.getById(r);
      if (r) {
        result.push(r);
      }
      return result;
    }, []);
    if (records.length === 0) {
      return records;
    }
    if (!me.tree && !silent && me.trigger("beforeRemove", { records }) === false) {
      return [];
    }
    me.isRemoving = true;
    if (me.isGrouped) {
      const oldCount = storage.count, recordsInCollapsedGroups = [], changedGroupParents = /* @__PURE__ */ new Set();
      for (const rec of records) {
        const { groupParent } = rec.instanceMeta(me);
        if (groupParent.meta.collapsed) {
          recordsInCollapsedGroups.push(rec);
        }
        ArrayHelper.remove(groupParent.groupChildren, rec);
        ArrayHelper.remove(groupParent.unfilteredGroupChildren, rec);
        groupParent.meta.childCount--;
        changedGroupParents.add(groupParent);
      }
      for (const groupParent of changedGroupParents) {
        if (groupParent.groupChildren.length > 0) {
          me.onModelChange(groupParent, {}, {});
        }
      }
      if (recordsInCollapsedGroups.length) {
        storage.trigger("change", {
          action: "splice",
          removed: recordsInCollapsedGroups,
          added: [],
          replaced: [],
          oldCount
        });
      }
    } else if (me.tree) {
      const allRemovedRecords = [], removingMultiple = records.length > 1, firstRemoved = records[0], index = removingMultiple ? void 0 : storage.indexOf(firstRemoved), removeChildArgs = records.reduce((result, child) => {
        const { parent } = child;
        if (parent) {
          if (!result[parent.id]) {
            result[parent.id] = [parent, []];
          }
          result[parent.id][1].push(child);
        }
        return result;
      }, {}), removals = Array.from(Object.values(removeChildArgs));
      if (!silent && me.trigger("beforeRemove", { records, isMove: false, parent: removingMultiple ? void 0 : firstRemoved.parent }) === false) {
        me.isRemoving = false;
        return [];
      }
      removals.sort((a, b) => b[0].childLevel - a[0].childLevel);
      for (const [parent, children] of removals) {
        allRemovedRecords.push(...parent.removeChild(children, false, true));
      }
      if (!silent) {
        me.trigger("remove", {
          // parent is only relevant when removing single node
          parent: removingMultiple ? void 0 : firstRemoved.parent,
          index,
          isChild: true,
          allRecords: allRemovedRecords,
          isMove: false,
          records
        });
        me.trigger("change", {
          action: "remove",
          records
        });
      }
      me.isRemoving = false;
      return records;
    }
    if (records.length === 0) {
      me.isRemoving = false;
      return records;
    }
    for (const record of records) {
      record.beforeRemove(records);
    }
    if (silent) {
      me.suspendEvents();
    }
    storage.remove(records);
    if (silent) {
      me.resumeEvents();
    }
    if (me.autoCommit) {
      me.doAutoCommit();
    }
    me.isRemoving = false;
    return records;
  }
  /**
   * Clears store data. Used by removeAll, separate function for using with chained stores.
   * @private
   * @category CRUD
   */
  clear(isLoading = false) {
    var _a2, _b;
    const me = this, { storage } = me;
    if (me.storage.totalCount || ((_b = (_a2 = me.rootNode) == null ? void 0 : _a2.unfilteredChildren) == null ? void 0 : _b.length)) {
      if (!isLoading && me.trigger("beforeRemove", { records: storage.allValues, removingAll: true }) === false) {
        return null;
      }
      if (me.rootNode) {
        if (!me.isChained) {
          me.rootNode.clearChildren(isLoading);
        }
      } else if (isLoading) {
        const allRecords = me.registeredRecords;
        for (let i = allRecords.length - 1, rec; i >= 0; i--) {
          rec = allRecords[i];
          if (rec && !rec.isDestroyed) {
            rec.unjoinStore(me);
          }
        }
      }
      if (isLoading) {
        me.removed.clear();
        storage.suspendEvents();
      }
      storage.clear();
      if (isLoading) {
        storage.resumeEvents();
      }
      me.added.clear();
      me.modified.clear();
    }
  }
  /**
   * Removes all records from the store.
   * @param {Boolean} [silent] Specify `true` to suppress events
   * @returns {Boolean} `true` unless the action was prevented, in which case it returns `false`
   * @fires beforeRemove
   * @fires removeAll
   * @fires change
   * @category CRUD
   */
  removeAll(silent = false) {
    const me = this, storage = me.storage;
    let result;
    me.isRemoving = true;
    if (silent) {
      storage.suspendEvents();
      const allRecords = me.registeredRecords;
      for (let i = allRecords.length - 1, rec; i >= 0; i--) {
        rec = allRecords[i];
        if (rec && !rec.isDestroyed && !rec.isRoot) {
          rec.unjoinStore(me);
        }
      }
    }
    if (me.tree) {
      result = me.rootNode.clear() !== false;
    } else {
      result = me.clear() !== null;
    }
    if (silent) {
      storage.resumeEvents();
    }
    me.isRemoving = false;
    return result;
  }
  /**
   * Add records to store.
   * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} records Array of records/data or a single record/data to add to store
   * @param {Boolean} [silent] Specify `true` to suppress events
   * @returns {Core.data.Model[]} Added records
   * @fires add
   * @fires change
   * @category CRUD
   */
  add(records, silent = false, options = {}) {
    const me = this, { storage } = me;
    records = ArrayHelper.asArray(records);
    if (!(records == null ? void 0 : records.length)) {
      return;
    }
    me.tree = me.tree || Boolean(me.autoTree && records[0].children);
    if (me.tree) {
      const parentIdMap = /* @__PURE__ */ new Map(), parentIndexField = me.modelClass.getFieldDataSource("parentIndex"), { parentIdField } = me.modelClass, added2 = [];
      records.forEach((node) => {
        const parentId = node[parentIdField];
        if (!parentIdMap.has(parentId)) {
          parentIdMap.set(parentId, { append: [], insert: [] });
        }
        const entry = parentIdMap.get(parentId);
        if (!node.isModel && parentIndexField in node) {
          entry.insert.push(node);
        } else {
          entry.append.push(node);
        }
      });
      parentIdMap.forEach(({ append, insert }, parentId) => {
        const parentNode2 = parentId == null ? me.rootNode : me.getById(parentId);
        if (!parentNode2) {
          throw new Error(`Parent node with id ${parentId} not found, cannot add children.`);
        }
        if (append.length) {
          added2.push(...parentNode2.appendChild(append, silent));
        }
        if (insert.length) {
          for (const toInsert of insert) {
            added2.push(parentNode2.insertChild(toInsert, toInsert[parentIndexField], silent, options));
          }
        }
      });
      if (me.isFiltered && me.reapplyFilterOnAdd) {
        me.filter({
          silent: true
        });
      }
      if (me.reapplySortersOnAdd && !me.isSyncingDataOnLoad) {
        me.sort();
      }
      return added2;
    }
    if (!silent && me.trigger("beforeAdd", { records }) === false) {
      return null;
    }
    me.tree = me.tree || Boolean(me.autoTree && records[0].children);
    if (me.tree) {
      return me.rootNode.appendChild(records);
    }
    const added = me.processRecords(records);
    if (silent) {
      me.suspendEvents();
    }
    storage.add(added);
    if (silent) {
      me.resumeEvents();
    }
    if (me.autoCommit) {
      me.doAutoCommit();
    }
    return added;
  }
  processRecords(records, onRecordCreated) {
    return records.map((data) => {
      const record = this.processRecord(data.isModel ? data : this.createRecord(data));
      onRecordCreated == null ? void 0 : onRecordCreated.call(this, record);
      return record;
    });
  }
  /**
   * Insert records into the store.
   * @param {Number} index Index to insert at
   * @param {Core.data.Model|Core.data.Model[]|Object|Object[]} records Array of records/data or a single record/data to insert to store
   * @param {Boolean} [silent] Specify `true` to suppress events
   * @returns {Core.data.Model[]} Inserted records
   * @fires add
   * @fires change
   * @category CRUD
   */
  insert(index, records, silent = false) {
    var _a2;
    const me = this, storage = me.storage, insertBefore = me.getAt(index), _records = storage.values, removeIndices = [];
    records = ArrayHelper.asArray(records);
    if (me.tree) {
      const root = me.rootNode;
      return root.insertChild(records, (_a2 = root.children) == null ? void 0 : _a2[index], silent);
    }
    if (!silent && me.trigger("beforeAdd", { records }) === false) {
      return null;
    }
    let isNoop, start, i;
    if (_records[start = index] === records[0] || _records[start = index - 1] === records[0]) {
      for (isNoop = true, i = 0; isNoop && i < records.length; i++) {
        if (records[i] !== _records[start + i]) {
          isNoop = false;
        }
      }
    }
    if (isNoop) {
      return null;
    }
    const added = me.processRecords(records, (record) => {
      const removedAtIndex = storage.indexOf(record);
      if (record.children && record.children.length && me.autoTree) {
        me.tree = true;
      }
      if (removedAtIndex > -1) {
        if (removedAtIndex < index && insertBefore)
          index--;
        removeIndices.push(removedAtIndex);
      }
      record.meta.previousIndex = removedAtIndex;
    });
    me.suspendEvents();
    me.storage.remove(removeIndices);
    me.resumeEvents();
    if (silent) {
      me.suspendEvents();
    }
    storage.splice(index, 0, ...added);
    if (silent) {
      me.resumeEvents();
    }
    if (me.autoCommit) {
      me.doAutoCommit();
    }
    return added;
  }
  /**
   * Moves a record, or block of records to another location.
   * @param {Core.data.Model|Core.data.Model[]} records The record(s) to move.
   * @param {Core.data.Model} beforeRecord the record to insert the first record(s) before.
   * @fires change
   * @category CRUD
   */
  move(records, beforeRecord) {
    if (this.isTree) {
      beforeRecord.parent.insertChild(records, beforeRecord);
    } else {
      this.storage.move(records, beforeRecord);
    }
  }
  //endregion
  //region Update multiple
  setMultiple(filterFn, field2, value) {
    const me = this, records = [], changes = [];
    me.forEach((r) => {
      if (filterFn(r)) {
        changes.push(r.set(field2, value, true));
        records.push(r);
      }
    });
    me.trigger("updateMultiple", { records, all: me.records.length === records.length });
    me.trigger("change", { action: "updatemultiple", records, all: me.records.length === records.length });
    if (me.reapplyFilterOnUpdate && me.isFiltered)
      me.filter();
  }
  setAll(field2, value) {
    const me = this, changes = [];
    me.forEach((r) => {
      changes.push(r.set(field2, value, true));
    });
    me.trigger("updateMultiple", { records: me.records, all: true });
    me.trigger("change", { action: "updatemultiple", records: me.records, all: true });
    if (me.reapplyFilterOnUpdate && me.isFiltered)
      me.filter();
  }
  //endregion
  //region Commit
  /**
   * Accepts all changes, resets the modification tracking:
   * * Clears change tracking for all records
   * * Clears added
   * * Clears modified
   * * Clears removed
   * Leaves the store in an "unmodified" state.
   * @internal
   */
  acceptChanges() {
    const me = this;
    me.added.forEach((r) => r.clearChanges(true, false));
    me.modified.forEach((r) => r.clearChanges(true, false));
    me.added.clear();
    me.modified.clear();
    me.removed.clear();
  }
  /**
   * Commits changes, per default only returns changes and resets tracking.
   * @param {Boolean} [silent] Specify `true` to suppress events
   * @returns {Object} Changes, see {@link #property-changes}
   * @fires beforeCommit
   * @fires commit
   * @category CRUD
   */
  commit(silent = false) {
    const { changes } = this;
    if (!silent && this.trigger("beforeCommit", { changes }) === false) {
      return false;
    }
    this.acceptChanges();
    if (!silent) {
      this.trigger("commit", { changes });
    }
    return changes;
  }
  /**
   * Reverts all changes in the store (adds removed records back, and removes newly added records).
   * @fires change
   * @fires refresh
   * @category CRUD
   */
  revertChanges() {
    const me = this, { changes } = me;
    if (changes) {
      const event = { action: "clearchanges", changes };
      me.add(me.removed.values, true);
      me.remove(me.added.values, true);
      me.modified.forEach((r) => r.revertChanges(true));
      me.added.clear();
      me.modified.clear();
      me.removed.clear();
      me.trigger("change", event);
      me.trigger("refresh", event);
    }
  }
  /**
   * Get uncommitted changes as an object of added/modified/removed arrays of records.
   *
   * ```javascript
   * // Format:
   * {
   *      added: [], // array of Core.data.Model
   *      modified: [], // array of Core.data.Model
   *      removed: [] // array of Core.data.Model
   * }
   * ```
   *
   * @member {Object} changes
   * @property {Core.data.Model[]} changes.added Records that have been added
   * @property {Core.data.Model[]} changes.modified Records that have changes to persistable fields
   * @property {Core.data.Model[]} changes.removed Records that have been removed
   * @readonly
   * @category Records
   */
  get changes() {
    const me = this, modified = me.modified.values.filter((r) => r.rawModifications);
    return me.added.count || modified.length || me.removed.count ? {
      // Slicing to have changes intact when triggering commit
      added: me.added.values.slice(),
      modified,
      removed: me.removed.values.slice()
    } : null;
  }
  /**
   * Setting autoCommit to true automatically commits changes to records.
   * @property {Boolean}
   * @category Records
   */
  get autoCommit() {
    return this._autoCommit;
  }
  set autoCommit(auto) {
    this._autoCommit = auto;
    if (auto && this.changes) {
      this.commit();
    }
  }
  /**
   * Suspends automatic commits upon store changes. Can be called multiple times (it uses an internal counter).
   * @category Records
   */
  suspendAutoCommit() {
    this.suspendCount++;
  }
  /**
   * Resumes automatic commits upon store changes. Will trigger commit if the internal counter is 0.
   * @category Records
   */
  resumeAutoCommit(doCommit = true) {
    this.suspendCount--;
    if (this.autoCommit && doCommit) {
      this.doAutoCommit();
    }
  }
  doAutoCommit() {
    if (this.suspendCount <= 0) {
      this.commit();
    }
  }
  //endregion
};

// ../Core/lib/Core/data/mixin/StoreChanges.js
var StoreChanges_default = (Target) => class StoreChanges extends (Target || Base) {
  static get $name() {
    return "StoreChanges";
  }
  static get configurable() {
    return {
      /**
       * Specifies target to filter and sort after applying changeset:
       * * `'changes'` - apply sort and filter to changeset only (see more below)
       * * `'none'` - do not apply sort and filter
       *
       * ### `changes` behavior
       * If the store has filters in effect when the changeset is applied, the following rules will determine how the
       * filtered values are affected:
       * - Among added records, only those that match the filter will be included in the filtered set
       * - Among updated records, those that did not previously match the filter but now do will be added to the filtered set,
       *   and those that did match but no longer do will also remain in the filtered set. This means that new records may
       *   appear in the filtered set as a result of `applyChanges`, but records will not disappear until filters are
       *   re-applied.
       *
       * @default
       * @prp {'changes'|'none'}
       * @category Advanced
       */
      applyChangesetFilterSortTarget: "changes"
    };
  }
  /**
   * Applies changes from another store to this store. Useful if cloning records in one store to display in a
   * grid in a popup etc. to reflect back changes.
   * @param {Core.data.Store} otherStore
   * @category CRUD
   */
  applyChangesFromStore(otherStore) {
    const me = this, { changes } = otherStore;
    if (!changes) {
      return;
    }
    if (changes.added) {
      me.add(changes.added);
    }
    if (changes.removed) {
      me.remove(changes.removed.map((r) => r.id));
    }
    if (changes.modified) {
      changes.modified.forEach((record) => {
        const localRecord = me.getById(record.id);
        localRecord == null ? void 0 : localRecord.set(record.modifications);
      });
    }
  }
  /**
   * Applies a set of changes (presumable from a backend) expressed as an object matching the format outputted by the
   * {@link Core/data/Store#property-changes} property: `{ added : [], modified/updated : [], removed : [] }`
   *
   * `added` is expected to be an array of raw data objects consumable by the stores model class for records to add to
   * the store (see example snippet below).
   *
   * `modified` (or `updated` for compatibility with Schedulers CrudManager) is expected to have the same format as
   * `added`, but should always include the `id` of the record to update.
   *
   * Records that have been created locally and gets assigned a proper id by the backend are expected to also pass a
   * `phantomId` field (name of the field is configurable using the `phantomIdField` arg, more info on phantom ids
   * below), to match it with the current id of a local record (`id` will contain the new id).
   *
   * Note that it is also possible to pass this `phantomId` -> `id` mapping in the `added` array. When encountering a
   * record in that array that already exists in the local store, it will be treated the same was as a record in the
   * `modified` array.
   *
   * `removed` is expected to be an array of objects with the `{ id : xx }` shape. Any matches on an id in the store
   * will be removed, those and any non matches will also be cleared from the change tracking of the store.
   *
   * If the store has filters in effect when the changeset is applied, the following rules will determine how the
   * filtered values are affected:
   * - Among added records, only those that match the filter will be included in the filtered set
   * - Among updated records, those that did not previously match the filter but now do will be added to the filtered set,
   *   and those that did match but no longer do will also remain in the filtered set. This means that new records may
   *   appear in the filtered set as a result of `applyChanges`, but records will not disappear until filters are
   *   re-applied.
   *
   * As an example, consider a store with the following initial state and some operations performed on it:
   *
   * ```javascript
   * // Load some data into the store
   * store.data = [
   *     { id : 1, name : 'Minerva' },
   *     { id : 2, name : 'Mars' },
   *     { id : 3, name : 'Jupiter' }
   * ];
   * // Add a new record. It gets assigned a generated id,
   * // for example 'generated56'
   * store.add({ name : 'Artemis' });
   * // Remove Jupiter
   * store.remove(3);
   * ```
   *
   * After syncing those operations to a custom backend (however you chose to solve it in your application) we might
   * get the following response (see "Transforming a response to the correct format" below if your backend responds
   * in another format):
   *
   * ```javascript
   * const serverResponse = {
   *     added : [
   *         // Added by the backend, will be added locally
   *         { id : 5, name : 'Demeter' }
   *     ],
   *
   *     updated : [
   *         // Will change the name of Minerva -> Athena
   *         { id : 1, name : 'Athena' },
   *         // Will set proper id 4 for Artemis
   *         { $PhantomId : 'generated56', id : 4 }
   *     ],
   *
   *     removed : [
   *         // Confirmed remove of Jupiter
   *         { id : 3 },
   *         // Removed by the backend, Mars will be removed locally
   *         { id : 2 }
   *     ]
   * };
   * ```
   *
   * If that response is then passed to this function:
   *
   * ```javascript
   * store.applyChangeSet(serverResponse);
   * ```
   *
   * The end result will be the following data in the store:
   *
   * ```javascript
   * [
   *     { id : 1, name : 'Athena' }, // Changed name
   *     { id : 4, name : 'Artemis' }, // Got a proper id
   *     { id : 5, name : 'Demeter' } // Added by the backend
   * ]
   * ```
   *
   * ### Phantom ids
   *
   * When a record is created locally, it is always assigned a generated id. That id is called a phantom id (note that
   * it is assigned to the normal id field). When passing the new record to the backend, the id is sent with it. When
   * the backend inserts the record into the database, it (normally) gets a proper id assigned. That id then needs to
   * be passed back in the response, to update the local record with the correct id. Making sure that future updates
   * match the correct row in the database.
   *
   * For example a newly created record should be passed similar to this to the backend (pseudo format, up to the
   * application/backend to decide):
   *
   * ```json
   * {
   *     "added" : {
   *         "id" : "generated79",
   *         "name" : "Hercules",
   *         ...
   *     }
   * }
   * ```
   *
   * For the backend response to be applicable for this function, it should then respond with:
   *
   * ```json
   * {
   *     "updated" : {
   *         {
   *             "$PhantomId" : "generated79",
   *             "id" : 465
   *         }
   *     }
   * }
   * ```
   *
   * (Or, as stated above, it can also be passed in the "added" array. Which ever suits your backend best).
   *
   * This function will then change the id of the local record using the phantom id `generated79` to `465`.
   *
   * ### Transforming a response to the correct format
   *
   * This function optionally accepts a `transformFn`, a function that will be called with the `changes`. It is
   * expected to return a changeset in the format described above (`{ added : [], updated : [], removed : [] }`),
   * which then will be used to apply the changes.
   *
   * Consider the following "non standard" (made up) changeset:
   *
   * ```javascript
   * const changes = {
   *     // Database ids for records previously added locally
   *     assignedIds : {
   *         'phantom1' : 10,
   *         'phantom2' : 15
   *     },
   *     // Ids records removed by the backend
   *     removed : [11, 27],
   *     // Modified records, keyed by id
   *     altered : {
   *         12 : { name : 'Changed' }
   *     },
   *     // New records, keyed by id
   *     inserted : {
   *         20  : { name : 'New' }
   *     }
   * }
   * ```
   *
   * Since it does not match the expected format it has to be transformed:
   *
   * ```javascript
   * store.applyChangeset(changes, ({ assignedIds, inserted, altered, removed }) => ({
   *    // Convert inserted to [{ id : 20, name : 'New' }]
   *    added : Object.entries(inserted).map(([id, data] => ({ id, ...data }),
   *    updated : [
   *        // Convert assignedIds to [{ $PhantomId : 'phantom1', id : 10 }, ...]
   *       ...Object.entries(assignedIds).map(([phantomId, id])) => ({ $PhantomId : phantomId, id }),
   *       // Convert altered to [{ id : 12, name : 'Changed' }]
   *       ...Object.entries(modified).map(([id, data] => ({ id, ...data })
   *    ],
   *    // Convert removed to [{ id : 11 }, ...]
   *    removed : removed.map(id => ({ id }))
   * }));
   * ```
   *
   * The transform function above would output:
   *
   * ```javascript
   * {
   *     added : [
   *         {  id : 20, name : 'New' }
   *     ],
   *     updated : [
   *         { $PhantomId : 'phantom1', id : 10 },
   *         { $PhantomId : 'phantom2', id : 15 },
   *         {  id : 12, name : 'Changed' }
   *     ],
   *     removed : [
   *        { id : 11 },
   *        { id : 12 }
   *     ]
   * }
   * ```
   *
   * And that format can then be applied.
   *
   * @param {Object} changes Changeset to apply to the store, see specification above
   * @param {Function} [transformFn] Optional function used to preprocess a changeset in a different format,
   * should return an object with the format expected by this function (see above)
   * @param {String} [phantomIdField] Field used by the backend when communicating a record being assigned a proper id
   * instead of a phantom id (see above)
   * @privateparam {Boolean} [remote] Set to true to indicate changes are from the remote source. Remote changes have
   * precedence over local.
   * @privateparam {Boolean} [logChanges] Used by CrudManager to be able to revert specific changes later
   * @category CRUD
   */
  applyChangeset(changes, transformFn = null, phantomIdField = "$PhantomId", remote = true, logChanges = false) {
    var _a2, _b, _c, _d;
    const me = this, { added, updated, modified, removed } = (_a2 = transformFn == null ? void 0 : transformFn(changes, me)) != null ? _a2 : changes, altered = (_b = updated != null ? updated : modified) != null ? _b : [], idDataSource = me.modelClass.getFieldDataSource("id"), log = logChanges ? /* @__PURE__ */ new Map() : null, allAdded = [], allAltered = [];
    let rootUpdated = false, modifiedParents = [];
    me._groupVisibleRecordIds = [];
    me.isGrouped && me.forEach((record) => {
      me._groupVisibleRecordIds.push(record.id);
    });
    if ((added == null ? void 0 : added.length) > 0) {
      const toUpdate = [], toAdd = [];
      for (const data of added) {
        if (me.getById((_c = data[phantomIdField]) != null ? _c : ObjectHelper.getPath(data, idDataSource))) {
          toUpdate.push(data);
        } else {
          toAdd.push(data);
        }
      }
      altered.unshift.apply(altered, toUpdate);
      const addedRecords = (_d = me.add(toAdd, false, { orderedParentIndex: { skip: true } })) != null ? _d : [];
      allAdded.push(...addedRecords);
      if (me.tree) {
        for (const record of addedRecords) {
          const { parent } = record;
          if (parent.isRoot) {
            rootUpdated = true;
            modifiedParents = [parent];
            break;
          }
          if (!parent.isRoot && modifiedParents.every((r) => !r.contains(parent))) {
            modifiedParents.push(parent);
          }
        }
      }
      for (const record of addedRecords) {
        log == null ? void 0 : log.set(record.id, record.data);
        record.clearChanges();
      }
    }
    if ((altered == null ? void 0 : altered.length) > 0) {
      for (const data of altered) {
        const phantomId = data[phantomIdField], id = ObjectHelper.getPath(data, idDataSource), record = me.getById(phantomId != null ? phantomId : id);
        if (record) {
          const changes2 = record.applyChangeset(data, phantomIdField, remote);
          if (me.tree && !rootUpdated && modifiedParents.every((r) => !r.contains(record))) {
            if (record.parent.isRoot) {
              rootUpdated = true;
              modifiedParents = [record.parent];
            } else {
              modifiedParents.push(record.parent);
            }
          }
          log == null ? void 0 : log.set(id, changes2);
          allAltered.push(record);
        }
      }
    }
    if ((removed == null ? void 0 : removed.length) > 0) {
      me.applyRemovals(removed);
    }
    if (me.applyChangesetFilterSortTarget === "changes") {
      const parentsModifiedByFilter = me.filterChangeset(allAdded, allAltered);
      modifiedParents.push(...parentsModifiedByFilter);
    }
    me.afterChangesetApplied(modifiedParents);
    me._groupVisibleRecordIds = null;
    return log;
  }
  afterChangesetApplied(modifiedParents) {
    modifiedParents.forEach((parent) => {
      parent.traverse((record) => {
        record.sortOrderedChildren(false, false);
        if (record.children) {
          record.updateChildrenIndices(record.children, "parentIndex", true);
        }
        if (record.unfilteredChildren) {
          record.updateChildrenIndices(record.unfilteredChildren, "unfilteredIndex", true);
        }
      });
    });
  }
  // Apply removals, removing records and updating the `removed` bag to match.
  //
  // Accepts an array of objects containing an `id` property. Records in the store matching an entry in the array
  // will be removed from the store and the `removed` bag. Unmatched entries will be removed from the `removed` bag.
  applyRemovals(removals) {
    const me = this, { removed: removedStash } = me, idDataSource = me.modelClass.idField, toRemove = [];
    for (const removedEntry of removals) {
      const id = ObjectHelper.getPath(removedEntry, idDataSource);
      if (removedStash.includes(id)) {
        removedStash.remove(id);
      } else {
        toRemove.push(id);
      }
    }
    me.remove(toRemove);
    for (const record of toRemove) {
      removedStash.remove(record);
    }
  }
  /**
   * Filters records that have been added/updated as part of a changeset. The `added` and `updated` parameters
   * are arrays of values that have already been added/updated in the Collection's values. This method brings
   * the Collection's `_filteredValues` in sync without performing a full sort or filter, using the following rules:
   *
   * - Added records that do not match the filter are removed from _filteredValues
   *
   * - Updated records that now match the filter are inserted at the correct position in _filteredValues
   *   if they were not formerly included
   *
   * - Updated records that formerly matched the filter, but now do not, are NOT removed from _filteredValues
   *
   * If the collection is sorted, either on its own or via a sort applied at the store level, that sort order is
   * respected when adding items to _filteredValues. If not, items are inserted in the same order they occur in
   * _values.
   *
   * @param {Object[]} added An array of unique values that were added as part of the changeset.
   * @param {Object[]} updated An array of unique values that were updated as part of the changeset.
   * @returns {Object[]} Any records that were added or removed from view, or whose children were modified.
   * @private
   */
  filterChangeset(added, updated) {
    const me = this, {
      isFiltered,
      tree,
      isGrouped,
      filtersFunction
    } = me, storeSortFunction = me.isSorted ? me.createSorterFn(me.sorters) : void 0, {
      allValues,
      addedValues,
      isSorted
    } = me.storage, sorter = storeSortFunction != null || isSorted ? storeSortFunction != null ? storeSortFunction : me.storage.sortFunction : null, modifiedParents = /* @__PURE__ */ new Set();
    if (!isFiltered) {
      return [];
    }
    let trigger = false, groupers;
    if (isGrouped) {
      groupers = me.groupers;
      me.clearGroupers(true);
    }
    if (tree) {
      const nodesToInclude = new Set(updated.filter(filtersFunction));
      for (const matchingAdd of added.filter(filtersFunction)) {
        nodesToInclude.add(matchingAdd);
      }
      nodesToInclude.forEach((node) => node.bubble((ancestor) => nodesToInclude.add(ancestor)));
      nodesToInclude.delete(me.rootNode);
      const nodesToIncludeByParent = ArrayHelper.groupBy(Array.from(nodesToInclude), "parentId");
      for (const siblingsToInclude of Object.values(nodesToIncludeByParent)) {
        const { parent } = siblingsToInclude[0];
        if (parent.unfilteredChildren) {
          parent.children.push(...siblingsToInclude.filter((child) => !parent.children.includes(child)));
          modifiedParents.add(parent);
        }
      }
    } else if (sorter && !isGrouped) {
      const { filteredValues } = me.storage, sortedLength = addedValues ? filteredValues.findLastIndex((value) => !addedValues.has(value)) + 1 : filteredValues.length, sorted = filteredValues.slice(0, sortedLength), updatedMatches = new Set(updated.filter(filtersFunction));
      for (const value of filteredValues) {
        if (updatedMatches.has(value)) {
          updatedMatches.delete(value);
        }
      }
      for (const newMatch of updatedMatches) {
        sorted.push(newMatch);
      }
      sorted.sort(sorter);
      filteredValues.splice(0, sortedLength, ...sorted);
      trigger = true;
    } else {
      const updatedMatches = updated.filter((item) => filtersFunction(item) && !me.storage.includes(item));
      if (updatedMatches.length > 0) {
        me.includeInSubset(allValues, me.storage.filteredValues, updatedMatches);
        trigger = true;
      }
    }
    const nonMatchingAdds = new Set(added.filter((value) => !filtersFunction(value)));
    if (nonMatchingAdds.size > 0) {
      if (tree) {
        for (const addedChild of nonMatchingAdds) {
          ArrayHelper.remove(addedChild.parent.children, addedChild);
          modifiedParents.add(addedChild.parent);
        }
      } else {
        ArrayHelper.remove(me.storage.filteredValues, nonMatchingAdds);
      }
      trigger = true;
    }
    if (groupers) {
      me.group(groupers[0], null, false, true, true);
      trigger = true;
    }
    if (tree && modifiedParents.size > 0) {
      me.storage.replaceValues({
        values: me.collectDescendants(me.rootNode).visible,
        silent: true
      });
    } else if (trigger) {
      me._idMap = null;
      me.trigger("refresh");
    }
    return [...modifiedParents];
  }
  /**
   * Given an array `all`, an array `subset` that is a subset of `all` in the same order, and another array
   * `toInclude` that is a different subset of `all` disjoint with `subset`, add each item from `toInclude`
   * to `subset`, in an order matching the order in `all`. The order of `subset` must match the order of `all`.
   * The order of `toInclude` is unimportant.
   *
   * Modifies `subset` in-place.
   *
   * @param {Array} all An array of unique items (e.g. records)
   * @param {Array} subset An array containing a subset of the items in `all` (same order as `all`)
   * @param {Array} toInclude An array or items from `all` that should be included in `subset` (unordered)
   * @returns {Array} The subset modified in-place.
   * @private
   */
  includeInSubset(all, subset, toInclude) {
    const toIncludeSet = new Set(toInclude);
    let subsetIndex = 0, allIndex = 0, done = toIncludeSet.size === 0;
    while (allIndex < all.length && !done) {
      const subsetItem = subset[subsetIndex];
      let allItem = all[allIndex];
      while (subsetItem !== allItem) {
        if (toIncludeSet.has(allItem)) {
          subset.splice(subsetIndex, 0, allItem);
          subsetIndex++;
          toIncludeSet.delete(allItem);
          done = toIncludeSet.size === 0;
        }
        allItem = all[++allIndex];
      }
      if (toIncludeSet.has(subsetItem)) {
        toIncludeSet.delete(subsetItem);
      }
      if (subsetIndex < subset.length) {
        subsetIndex++;
      }
    }
    return subset;
  }
};

// ../Core/lib/Core/data/mixin/StoreFilter.js
var StoreFilter_default = (Target) => class StoreFilter extends (Target || Base) {
  static get $name() {
    return "StoreFilter";
  }
  //region Config
  static get defaultConfig() {
    return {
      /**
       * Specify one or more {@link Core/util/CollectionFilter} config objects to apply initially.
       *
       * For example:
       *
       * ```javascript
       *  // Configure the store to filter in clients over the age of 30
       *  new Store({
       *      ...,
       *      filters : [{
       *          property : 'age',
       *          value    : 30,
       *          operator : '>'
       *      }],
       *      ...
       *  })
       * ```
       *
       * or:
       *
       * ```javascript
       *  // Configure the store to filter based on a complex operation
       *  new Store({
       *      ...,
       *      filters : [{
       *          filterBy(record) {
       *              // Return true or false for filtering in or out
       *              return shouldClientBeVisible(record);
       *          }
       *      }],
       *      ...
       *  })
       * ```
       *
       * @config {CollectionFilterConfig|CollectionFilterConfig[]}
       * @category Filtering
       */
      filters: null,
      /**
       * Specify true to reapply filters when a record is added to the store.
       * @config {Boolean}
       * @default
       * @category Filtering
       */
      reapplyFilterOnAdd: false,
      /**
       * Specify true to reapply filters when a record is updated in the store.
       * @config {Boolean}
       * @default
       * @category Filtering
       */
      reapplyFilterOnUpdate: false
    };
  }
  //endregion
  //region Events
  /**
   * Fired after applying filters to the store
   * @event filter
   * @param {Core.data.Store} source This Store
   * @param {Core.util.Collection} filters Filters used by this Store
   * @param {Core.data.Model[]} removed The records which were filtered out by the action.
   * @param {Core.data.Model[]} added The records which were filtered back in by the action.
   * @param {Core.data.Model[]} records Filtered records
   */
  //endregion
  //region Properties
  set reapplyFilterOnAdd(enable) {
    this.storage.autoFilter = enable;
  }
  get reapplyFilterOnAdd() {
    return this.storage.autoFilter;
  }
  /**
   * Currently applied filters. A collection of {@link Core.util.CollectionFilter} instances.
   * @type {Core.util.Collection}
   * @readonly
   * @category Sort, group & filter
   */
  set filters(filters) {
    const me = this, collection = me.filters;
    collection.clear();
    me._filtersFunction = null;
    if (filters) {
      if (filters.constructor.name === "Object") {
        for (const f of Object.entries(filters)) {
          if (f[0] === "filterBy" && typeof f[1] === "function") {
            collection.add(new CollectionFilter({
              filterBy: f[1]
            }));
          } else {
            collection.add(new CollectionFilter(f[1].constructor.name === "Object" ? Object.assign({
              property: f[0]
            }, f[1]) : {
              property: f[0],
              value: f[1]
            }));
          }
        }
      } else if (Array.isArray(filters)) {
        collection.add(...filters.map((filterConfig) => {
          if (filterConfig instanceof CollectionFilter) {
            return filterConfig;
          }
          return new CollectionFilter(filterConfig);
        }));
      } else if (filters.isCollection) {
        collection.add(...filters.values);
      } else {
        collection.add(new CollectionFilter({
          filterBy: filters
        }));
      }
      collection.forEach((item) => item.owner = me);
    }
  }
  get filters() {
    return this._filters || (this._filters = new Collection({ extraKeys: ["property"] }));
  }
  set filtersFunction(filtersFunction) {
    this._filtersFunction = filtersFunction;
  }
  get filtersFunction() {
    const me = this, { filters, isGrouped } = me;
    if (!me._filtersFunction) {
      if (filters.count) {
        const generatedFilterFunction = CollectionFilter.generateFiltersFunction(filters);
        me._filtersFunction = (candidate) => {
          if (isGrouped && candidate.isSpecialRow) {
            return candidate.groupChildren.some(generatedFilterFunction);
          }
          return generatedFilterFunction(candidate);
        };
      } else {
        me._filtersFunction = FunctionHelper.returnTrue;
      }
    }
    return me._filtersFunction;
  }
  /**
   * Check if store is filtered
   * @property {Boolean}
   * @readonly
   * @category Sort, group & filter
   */
  get isFiltered() {
    return this.filters.values.some((filter2) => !filter2.disabled);
  }
  //endregion
  traverseFilter(record) {
    const me = this, hitsCurrent = !record.isRoot && me.filtersFunction(record), children = record.unfilteredChildren || record.children;
    if (!children || !children.length) {
      return hitsCurrent;
    }
    if (!record.unfilteredChildren) {
      record.unfilteredChildren = record.children.slice();
    }
    record.children = record.unfilteredChildren.filter((r) => {
      return me.traverseFilter(r);
    });
    record.updateChildrenIndices(record.unfilteredChildren, "unfilteredIndex", true);
    record.updateChildrenIndices(record.children, "parentIndex", true);
    return hitsCurrent || Boolean(record.children.length);
  }
  traverseClearFilter(record) {
    const me = this;
    if (record.unfilteredChildren) {
      record.children = record.orderedChildren.slice();
      record.unfilteredChildren = null;
    }
    if (record.children) {
      record.children.forEach((r) => me.traverseClearFilter(r));
      record.updateChildrenIndices(record.children, "parentIndex", true);
    }
  }
  get latestFilterField() {
    return this.filters.last ? this.filters.last.property : null;
  }
  /**
   * Adds a single filter to the {@link #config-filters} collection. By default, filters are reevaluated
   * and a Store change event fired.
   *
   * If the `silent` parameter is passed as `true`, multiple filters can be added without causing data changes.
   *
   * When the filters are as required, call {@link #function-filter} with no parameters
   * to apply the filters to the store.
   *
   * @param {CollectionFilterConfig|Function} newFilter A {@link Core.util.CollectionFilter filter} config,
   * or a function to use for filtering.
   * @param {Boolean} [silent] Pass `true` to *not* refilter the store immediately. Such as when
   * adding multiple filters.
   * @returns {Core.util.CollectionFilter} The Filter instance that was added.
   * @returns {Promise|Core.util.CollectionFilter} If {@link Core/data/AjaxStore#config-filterParamName} is set on store, this method
   * returns Collection filter inside a `Promise` which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  addFilter(filter2, silent = false) {
    const me = this;
    filter2 = filter2 instanceof CollectionFilter ? filter2 : new CollectionFilter(filter2);
    filter2.owner = me;
    me.filters.add(filter2);
    if (!silent) {
      if (me.remoteFilter) {
        return me.filter().then(() => filter2);
      } else {
        me.filter();
      }
    }
    return filter2;
  }
  /**
   * Filters the store by **adding** the specified filter(s) to the existing filters collection applied to this Store.
   * If a filter has an {@link Core.util.CollectionFilter#config-id id} specified,
   * or a {@link Core.util.CollectionFilter#config-property property} specified,
   * it will search for corresponding filter(s) in the existing filters first and replace it with a new filter.
   * **It will not remove other filters applied to the store!**
   *
   * To **add** a new filter:
   * ```
   * // Filter using simple object
   * store.filter({
   *     property : 'age',
   *     operator : '>',
   *     value    : 90
   * });
   *
   * // Filter using function
   * store.filter(r => r.age < 90);
   *
   * // Filter using a named filter as a function
   * store.filter({
   *     id : 'my-filter',
   *     filterBy : record => record.score > 10
   * });
   * ```
   *
   * To **remove** a specific filter, but keep other filters applied
   * ```
   * // Remove by filter `id` or `property`. Filter `id` defaults to the `property` name.
   * store.removeFilter('age');
   * store.removeFilter('my-filter');
   * ```
   *
   * To **replace** all existing filters with a new filter
   * ```
   * // Remove all filters and filter using simple object
   * store.filter({
   *     filters : {
   *         property : 'age',
   *         operator : '<',
   *         value    : 90
   *     },
   *     replace : true
   * });
   *
   * // Remove all filters and filter using function
   * store.filter({
   *     filters : r => r.age > 90,
   *     replace : true
   * });
   *
   * // Remove all filters and filter using a named filter as a function
   * store.filter({
   *     filters : {
   *         id : 'my-filter',
   *         filterBy : record => record.score > 10
   *     },
   *     replace : true
   * });
   * ```
   *
   * Basically filters replacing is an equivalent of having two sequenced calls:
   * {@link #function-clearFilters clearFilters} and {@link #function-filter filter}.
   *
   * Call without arguments to reapply filters.
   * ```
   * // Re-filter the store
   * store.filter();
   * ```
   *
   * @param {Object|CollectionFilterConfig|CollectionFilterConfig[]|Function} newFilters
   *        A {@link Core.util.CollectionFilter filter} config,
   *        or an array of {@link Core.util.CollectionFilter filter} configs,
   *        or a function to use for filtering,
   *        or a special object like: ```{ replace : true, filters : newFilters }```
   * @param {Boolean} [newFilters.replace]
   *        A flag, indicating whether or not the previous filters should be removed.
   * @param {Boolean} [newFilters.silent]
   *        Set as true to not fire events. UI will not be informed about the changes.
   * @param {CollectionFilterConfig|CollectionFilterConfig[]|Function} [newFilters.filters]
   *        If `newFilters` is an object and `replace` property is defined in the `newFilters`,
   *        it means that special object is used and real filter configuration must be nested down to this `filters` property.
   *        It can be:
   *        A {@link Core.util.CollectionFilter filter} config,
   *        or an array of {@link Core.util.CollectionFilter filter} configs,
   *        or a function to use for filtering.
   * @fires filter
   * @fires change
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-filterParamName} is set on store, this method returns Promise
   * which is resolved after data is loaded from remote server, otherwise it returns null value
   * @async
   * @category Sort, group & filter
   */
  filter(newFilters) {
    const me = this;
    let silent = false, internal;
    if (newFilters) {
      let fieldType = typeof newFilters;
      if (fieldType === "object") {
        if ("silent" in newFilters || "replace" in newFilters || newFilters.filters) {
          silent = newFilters.silent;
          if (newFilters.replace) {
            me.clearFilters(newFilters.filters.length === 0);
          }
          internal = newFilters.internal;
          newFilters = newFilters.filters;
          fieldType = typeof newFilters;
        }
      }
      if (newFilters) {
        const wasFiltered = me.isFiltered;
        me.isConfiguring = true;
        if (Array.isArray(newFilters)) {
          newFilters.forEach((f) => me.addFilter(f, true), me);
        } else if (fieldType === "function") {
          const filter2 = new CollectionFilter(newFilters);
          filter2.internal = internal;
          me.addFilter(filter2, true);
        } else if (fieldType === "string") {
          me.addFilter({
            property: newFilters,
            value: arguments[1]
          }, true);
        } else {
          me.addFilter(newFilters, true);
        }
        me.isConfiguring = false;
        if (!me.isFiltered && !wasFiltered) {
          return null;
        }
      }
    }
    me.filtersFunction = null;
    const result = me.performFilter(silent);
    me._idMap = null;
    return result;
  }
  /**
   * Perform filtering according to the {@link #property-filters} Collection.
   * This is the internal implementation which is overridden in {@link Core.data.AjaxStore} and
   * must not be overridden.
   * @private
   */
  performFilter(silent) {
    const me = this, { storage, filters, rootNode } = me, oldCount = me.count;
    me.trigger("beforeFilter", { filters });
    let added, removed;
    if (me.tree) {
      const oldDataset = storage.values;
      if (me.isFiltered) {
        me.traverseFilter(rootNode);
      } else {
        me.traverseClearFilter(rootNode);
      }
      const newDataset = me.collectDescendants(rootNode).visible;
      storage.replaceValues({
        values: newDataset,
        silent: true
      });
      const delta = ArrayHelper.delta(newDataset, oldDataset, true);
      added = delta.toAdd;
      removed = delta.toRemove;
    } else {
      storage.ion({
        change({ removed: r, added: a }) {
          removed = r;
          added = a;
        },
        once: true
      });
      if (me.isFiltered) {
        me.isGrouped && me.includeCollapsed();
        storage.addFilter({
          id: "primary-filter",
          filterBy: me.filtersFunction
        });
        me.isGrouped && me.excludeCollapsed();
      } else {
        storage.filters.clear();
      }
    }
    me.afterPerformFilter(silent || me.isRemoteDataLoading ? null : {
      action: "filter",
      filters,
      oldCount,
      added,
      removed,
      records: me.storage.values
    });
  }
  afterPerformFilter(event) {
    this.resetRelationCache();
    if (event) {
      this.triggerFilterEvent(event);
    }
  }
  // Used from filter() and StoreCRUD when reapplying filters
  triggerFilterEvent(event) {
    this.trigger("filter", event);
    if (!this.remoteFilter) {
      this.trigger("refresh", event);
      this.trigger("change", event);
    }
  }
  /**
   * *Adds* a function used to filter the store. Alias for calling `filter(fn)`. Return `true` from the function to
   * include record in filtered set
   *
   * ```javascript
   * store.filterBy(record => record.age > 25 && record.name.startsWith('A'));
   * ```
   *
   * @param {Function} fn Function used to test records
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-filterParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  filterBy(fn2) {
    return this.filter(fn2);
  }
  /**
   * Removes the passed filter, or the filter by the passed ID from the {@link #config-filters} collection.
   * By default, filters are reevaluated and a Store change event fired.
   *
   * If the `silent` parameter is passed as `true`, multiple filters can be removed without causing data changes.
   *
   * When the filters are as required, call {@link #function-filter} with no parameters
   * to apply the filters to the store.
   *
   * ```javascript
   * // Only view top priority events
   * myEventStore.filter({
   *     id       : 'priorityFilter',
   *     property : 'priority',
   *     value    : 1,
   *     operator : '='
   * });
   *
   * // That individual filter can be removed like this
   * myEventStore.removeFilter('priorityFilter');
   *
   * // Add named filter as a function
   * store.filter({
   *     id : 'my filter',
   *     filterBy : record => record.score > 10
   * });
   *
   * // Remove named filter function
   * store.removeFilter('my filter');
   * ```
   *
   * @param {String|Core.util.CollectionFilter} idOrInstance Filter to remove, or ID of the filter to remove. By default,
   * filters are reevaluated and a change event fired.
   * @param {Boolean} [silent] Pass `true` to *not* refilter the store immediately. Such as when
   * removing multiple filters.
   * @returns {Promise|Core.util.CollectionFilter} If {@link Core/data/AjaxStore#config-filterParamName} is set on store, this method
   * returns Collection filter inside a `Promise` which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  removeFilter(idOrInstance, silent = false) {
    const me = this, filter2 = idOrInstance instanceof CollectionFilter ? idOrInstance : me.filters.get(idOrInstance);
    if (filter2) {
      me.filters.remove(filter2);
      me._filtersFunction = null;
      if (!silent) {
        if (me.remoteFilter) {
          return me.filter().then(() => filter2);
        } else {
          me.filter();
        }
      }
      return filter2;
    }
  }
  /**
   * Removes all filters from the store.
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-filterParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  clearFilters(apply = true) {
    this.filters.remove(this.filters.values.filter((f) => !f.internal));
    if (apply) {
      return this.filter();
    }
  }
  convertFilterToString(field2) {
    const filter2 = this.filters.getBy("property", field2);
    return filter2 && !filter2.filterBy ? String(filter2) : "";
  }
  doDestroy() {
    var _a2;
    (_a2 = this._filters) == null ? void 0 : _a2.destroy();
    super.doDestroy();
  }
};

// ../Core/lib/Core/data/mixin/StoreGroup.js
var resortActions = {
  add: 1,
  replace: 1
};
var StoreGroup_default = (Target) => {
  var _a2;
  return _a2 = class extends (Target || Base) {
    static get properties() {
      return {
        collapsedGroups: /* @__PURE__ */ new Set()
      };
    }
    //endregion
    //region Init
    construct(config) {
      super.construct(config);
      this.ion({ change: "onDataChanged", thisObj: this });
    }
    updateGroupers(groupers) {
      this.setGroupers(groupers);
    }
    /**
     * Set groupers.
     * @param {Grouper[]} groupers Array of groupers to apply to store
     * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns
     * `Promise` which is resolved after data is loaded from remote server, otherwise it returns `null`
     * @async
     * @category Sort, group & filter
     */
    setGroupers(groupers, options = null) {
      const me = this, { storage } = me;
      let result;
      if (groupers == null ? void 0 : groupers.length) {
        me._groupers = groupers;
      } else if (me.groupers) {
        delete me._groupers;
        me.includeCollapsed();
        storage.replaceValues({
          values: me.removeHeadersAndFooters(storage._values),
          filteredValues: storage.isFiltered ? me.removeHeadersAndFooters(storage._filteredValues) : null,
          silent: true
        });
        result = me.group(null, null, null, false, options == null ? void 0 : options.silent);
      }
      me._idMap = null;
      return result;
    }
    // Collects group headers/footers on the fly. Not used in any performance sensitive code, but if that need arises
    // it should be cached and invalidated on record remove, add, update, grouping changes, filter and sorting...
    get groupRecords() {
      const groupRecords = [];
      if (this.isGrouped) {
        for (const record of this) {
          if (record.isSpecialRow) {
            groupRecords.push(record);
          }
        }
      }
      return groupRecords;
    }
    get unfilteredGroupRecords() {
      var _a3;
      const me = this;
      if (me.isGrouped) {
        const { generation } = me.storage;
        if (((_a3 = me._unfilteredGroupRecords) == null ? void 0 : _a3.generation) !== generation) {
          me._unfilteredGroupRecords = me.storage.allValues.filter((r) => r.isSpecialRow);
          me._unfilteredGroupRecords.generation = generation;
        }
      }
      return me._unfilteredGroupRecords || [];
    }
    /**
     * Returns group header record for the passed record or last group header in the store
     * @param {Core.data.Model} [targetRecord]
     * @param {Boolean} [ignoreFilter] Pass true to search in the complete collection
     * @returns {Core.data.Model}
     * @internal
     */
    getGroupHeaderForRecord(targetRecord, ignoreFilter = false) {
      if (this.isGrouped) {
        let result;
        const collection = ignoreFilter ? this.storage._values : this.storage.values;
        for (const record of collection) {
          if (record.isGroupHeader) {
            if (!targetRecord) {
              result = record;
            } else if (record === targetRecord || record.unfilteredGroupChildren.includes(targetRecord)) {
              result = record;
              break;
            }
          }
        }
        return result;
      }
    }
    // Temporarily include records from collapsed groups, for example prior to filtering
    includeCollapsed() {
      for (const groupId of this.collapsedGroups) {
        this.expand(this.getById(groupId), false);
      }
    }
    // Exclude records in collapsed groups, intended to be used after a call to includeCollapsed()
    excludeCollapsed() {
      for (const groupId of this.collapsedGroups) {
        this.collapse(this.getById(groupId));
      }
    }
    onDataChange({ source: storage, action, removed }) {
      var _a3;
      const me = this, { groupers } = me;
      if (groupers) {
        if (groupers.length) {
          if (action === "splice" && (removed == null ? void 0 : removed.length) || action === "move") {
            storage.replaceValues({
              ...me.prepareGroupRecords(),
              silent: true
            });
          }
        } else {
          storage.replaceValues({
            values: me.removeHeadersAndFooters(storage._values),
            filteredValues: storage.isFiltered ? me.removeHeadersAndFooters(storage._filteredValues) : null,
            silent: true
          });
        }
      }
      (_a3 = super.onDataChange) == null ? void 0 : _a3.call(this, ...arguments);
    }
    move(records, beforeRecord) {
      const me = this;
      if (me.isGrouped && !me.tree) {
        let prevRecord = beforeRecord;
        if (beforeRecord == null ? void 0 : beforeRecord.isSpecialRow) {
          prevRecord = me.getPrev(beforeRecord, false, false);
          if (!prevRecord) {
            return;
          }
        }
        const targetGroupHeader = me.getGroupHeaderForRecord(prevRecord), groupField = me.groupers[0].field, newGroupValue = targetGroupHeader.meta.groupRowFor, { reapplyFilterOnUpdate } = me;
        me.reapplyFilterOnUpdate = false;
        me.beginBatch();
        records.forEach((record) => record.setValue(groupField, newGroupValue));
        me.endBatch();
        me.reapplyFilterOnUpdate = reapplyFilterOnUpdate;
        if (me.isFiltered && (beforeRecord == null ? void 0 : beforeRecord.isSpecialRow)) {
          const { unfilteredGroupRecords } = me, index = unfilteredGroupRecords.indexOf(targetGroupHeader);
          beforeRecord = unfilteredGroupRecords[index + 1];
        }
      }
      super.move(records, beforeRecord);
    }
    // private function that collapses on the data level
    collapse(groupRecord) {
      if (groupRecord && !groupRecord.meta.collapsed) {
        this.excludeGroupRecords(groupRecord);
        groupRecord.meta.collapsed = true;
        this.collapsedGroups.add(groupRecord.id);
        this.trigger("toggleGroup", { groupRecord, collapse: true });
        return true;
      }
      return false;
    }
    // private function that expands on the data level
    expand(groupRecord, updateMap = true) {
      if (groupRecord == null ? void 0 : groupRecord.meta.collapsed) {
        this.includeGroupRecords(groupRecord);
        groupRecord.meta.collapsed = false;
        updateMap && this.collapsedGroups.delete(groupRecord.id);
        updateMap && this.trigger("toggleGroup", { groupRecord, collapse: false });
        return true;
      }
      return false;
    }
    removeHeadersAndFooters(records) {
      return records.filter((r) => {
        if (r.isSpecialRow) {
          this.unregister(r);
          return false;
        } else {
          return true;
        }
      });
    }
    prepareGroupRecords(sorter) {
      const me = this, {
        isFiltered,
        reapplyFilterOnUpdate,
        startGroupsCollapsed
      } = me, toCollapse = me.collapsedGroups, { allValues } = me.storage, toExpand = [], visibleRecordsIds = me._groupVisibleRecordIds || [], isVisible2 = (record) => {
        const matchesFilter = !isFiltered || me.filtersFunction(record);
        return reapplyFilterOnUpdate ? matchesFilter : matchesFilter || visibleRecordsIds.includes(record.id);
      };
      for (const record of allValues) {
        if (record.isGroupHeader && (record.meta.collapsed || toCollapse.has(record.id))) {
          toCollapse.add(record.id);
          toExpand.push(record);
        }
      }
      for (const record of toExpand) {
        me.includeGroupRecords(record);
      }
      const records = me.removeHeadersAndFooters(me.storage._values);
      if (sorter) {
        records.sort(sorter);
      }
      if (isFiltered) {
        me.filtersFunction = null;
      }
      const groupedRecords = [], field2 = me.groupers[0].field;
      let curGroup = null, curGroupRecord = null, childCount = 0;
      function addFooter() {
        const val = curGroupRecord.meta.groupRowFor, id = `group-footer-${typeof val === "number" ? val : StringHelper.createId(val)}`, footer = me.getById(id) || new me.modelClass({ id }, me, {
          specialRow: true,
          groupFooterFor: val,
          groupRecord: curGroupRecord
        });
        footer.stores = [me];
        me.register(footer);
        footer.groupChildren = curGroupRecord.groupChildren;
        if (!curGroupRecord.meta.collapsed) {
          groupedRecords.push(footer);
        }
        me.allRecords.push(footer);
        curGroupRecord.groupChildren.push(footer);
        curGroupRecord.unfilteredGroupChildren.push(footer);
        childCount++;
        return footer;
      }
      records.forEach((record) => {
        var _a3;
        const fieldValue = record.getValue(field2), val = fieldValue == void 0 ? "__novalue__" : fieldValue, id = `group-header-${typeof val === "number" ? val : StringHelper.createId(val)}`;
        if (((_a3 = record.unfilteredGroupChildren) == null ? void 0 : _a3.length) === 0) {
          me.unregister(record);
          return;
        }
        if (!ObjectHelper.isEqual(val, curGroup)) {
          if (curGroupRecord) {
            if (me.useGroupFooters) {
              addFooter(curGroupRecord);
            }
            curGroupRecord.meta.childCount = childCount;
          }
          curGroupRecord = me.getById(id);
          if (!curGroupRecord) {
            curGroupRecord = new me.modelClass({ id }, me, {
              specialRow: true,
              groupRowFor: val,
              groupField: field2
            });
            if (startGroupsCollapsed) {
              toCollapse.add(id);
              me.startGroupsCollapsed = false;
            }
          }
          curGroupRecord.meta.collapsed = toCollapse.has(id);
          curGroupRecord.stores = [me];
          me.register(curGroupRecord);
          curGroupRecord.groupChildren = [];
          curGroupRecord.unfilteredGroupChildren = [];
          groupedRecords.push(curGroupRecord);
          me.allRecords.push(curGroupRecord);
          curGroup = val;
          childCount = 0;
        }
        record.instanceMeta(me.id).groupParent = curGroupRecord;
        if (!toCollapse.has(id)) {
          groupedRecords.push(record);
        }
        if (isVisible2(record)) {
          curGroupRecord.groupChildren.push(record);
          childCount++;
        }
        curGroupRecord.unfilteredGroupChildren.push(record);
      });
      if (curGroupRecord) {
        if (me.useGroupFooters) {
          addFooter();
        }
        curGroupRecord.meta.childCount = childCount;
      }
      me._idMap = null;
      const result = {
        values: groupedRecords
      };
      if (isFiltered) {
        result.filteredValues = groupedRecords.filter(isVisible2);
      }
      return result;
    }
    //endregion
    //region Group and ungroup
    /**
     * Is store currently grouped?
     * @property {Boolean}
     * @readonly
     * @category Sort, group & filter
     */
    get isGrouped() {
      var _a3;
      return Boolean((_a3 = this.groupers) == null ? void 0 : _a3.length);
    }
    /**
     * Group records, either by replacing current sorters or by adding to them.
     * A grouper can specify a **_custom sorting function_** which will be called with arguments (recordA, recordB).
     * Works in the same way as a standard array sorter, except that returning `null` triggers the stores
     * normal sorting routine. Grouped store **must** always be sorted by the same field.
     *
     * ```javascript
     * // simple grouper
     * store.group('city');
     *
     * // grouper as object, descending order
     * store.group({ field : 'city', ascending : false });
     *
     * // using custom sorting function
     * store.group({
     *     field : 'city',
     *     fn : (recordA, recordB) => {
     *         // apply custom logic, for example:
     *         return recordA.city.length < recordB.city.length ? -1 : 1;
     *     }
     * });
     * ```
     *
     * @param {String|Object} field Field to group by.
     * Can also be a config containing a field to group by and a custom sorting function called `fn`.
     * @param {Boolean} [ascending] Sort order of the group titles
     * @param {Boolean} [add] Add a grouper (true) or use only this grouper (false)
     * @param {Boolean} [performSort] Trigger sort directly, which does the actual grouping
     * @param {Boolean} [silent] Set as true to not fire events
     * @category Sort, group & filter
     * @fires group
     * @fires refresh
     * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
     * which is resolved after data is loaded from remote server, otherwise it returns `null`
     * @async
     */
    group(field2, ascending, add = false, performSort = true, silent = false) {
      var _a3, _b;
      const me = this;
      let newGrouper, fn2;
      if (field2 && typeof field2 === "object") {
        ascending = field2.ascending;
        fn2 = field2.fn;
        field2 = field2.field;
      }
      if (add) {
        me.groupers.push(newGrouper = {
          field: field2,
          ascending,
          complexMapping: field2.includes(".")
        });
      } else if (field2) {
        if (ascending == null) {
          ascending = ((_a3 = me.groupInfo) == null ? void 0 : _a3.field) === field2 && ((_b = me.groupInfo) == null ? void 0 : _b.fn) === fn2 ? !me.groupInfo.ascending : true;
        }
        me.groupInfo = newGrouper = {
          field: field2,
          ascending,
          fn: fn2,
          complexMapping: field2.includes(".")
        };
        me.groupers = [me.groupInfo];
      }
      if (newGrouper) {
        const { prototype: prototype2 } = me.modelClass;
        if (newGrouper.complexMapping && !Object.prototype.hasOwnProperty.call(prototype2, field2)) {
          Object.defineProperty(prototype2, field2, {
            get() {
              return ObjectHelper.getPath(this, field2);
            }
          });
        }
      }
      if (performSort !== false) {
        if (me.remoteSort && !me.isRemoteDataLoading) {
          return me.sort(null, null, false, true).then(() => me.onAfterGrouping(silent));
        } else {
          me.sort(null, null, false, true);
        }
      }
      me.onAfterGrouping(silent);
    }
    onAfterGrouping(silent) {
      if (silent) {
        return;
      }
      const me = this, groupers = me.groupers || [];
      me.trigger("group", { isGrouped: me.isGrouped, groupers, records: me.storage.values });
      me.trigger("refresh", { action: "group", isGrouped: me.isGrouped, groupers, records: me.storage.values });
    }
    // Internal since UI does not support multi grouping yet
    /**
     * Add a grouping level (a grouper).
     * @param {String} field Field to group by
     * @param {Boolean} ascending Group direction
     * @category Sort, group & filter
     * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
     * which is resolved after data is loaded from remote server, otherwise it returns `null`
     * @async
     * @internal
     */
    addGrouper(field2, ascending = true) {
      return this.group(field2, ascending, true);
    }
    // Internal since UI does not support multi grouping yet
    /**
     * Removes a grouping level (a grouper)
     * @param {String} field Grouper to remove
     * @category Sort, group & filter
     * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
     * which is resolved after data is loaded from remote server, otherwise it returns `null`
     * @async
     * @internal
     */
    removeGrouper(field2) {
      const me = this, { groupers } = me;
      if (!groupers) {
        return;
      }
      const index = groupers.findIndex((grouper) => grouper.field === field2);
      if (index > -1) {
        groupers.splice(index, 1);
        if (!groupers.length) {
          return me.clearGroupers();
        } else {
          return me.group();
        }
      }
    }
    /**
     * Removes all groupers, turning store grouping off.
     * @privateparam {Boolean} [silent=false] Pass true to suppress events.
     * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
     * which is resolved after data is loaded from remote server, otherwise it returns `null`
     * @async
     * @category Sort, group & filter
     */
    clearGroupers(silent = false) {
      return this.setGroupers(null, { silent });
    }
    //endregion
    //region Get and check
    /**
     * Check if a record belongs to a certain group (only for the first grouping level)
     * @param {Core.data.Model} record The Record
     * @param {*} groupValue The group value
     * @returns {Boolean} True if the record belongs to the group, otherwise false
     * @category Sort, group & filter
     */
    isRecordInGroup(record, groupValue) {
      var _a3;
      if (!this.isGrouped) {
        return null;
      }
      const groupField = (_a3 = this.groupers[0]) == null ? void 0 : _a3.field;
      return record.getValue(groupField) === groupValue && !record.isSpecialRow;
    }
    isInCollapsedGroup(record) {
      const parentGroupRec = record.instanceMeta(this).groupParent;
      return parentGroupRec == null ? void 0 : parentGroupRec.meta.collapsed;
    }
    /**
     * Returns all records in the group with specified groupValue.
     * @param {*} groupValue
     * @returns {Core.data.Model[]} Records in specified group or null if store not grouped
     * @category Sort, group & filter
     */
    getGroupRecords(groupValue) {
      if (!this.isGrouped) {
        return null;
      }
      return this.storage.values.filter((record) => this.isRecordInGroup(record, groupValue));
    }
    /**
     * Get all group titles.
     * @returns {String[]} Group titles
     * @category Sort, group & filter
     */
    getGroupTitles() {
      if (!this.isGrouped) {
        return null;
      }
      return this.getDistinctValues(this.groupers[0].field);
    }
    //endregion
    onDataChanged({ changes, action }) {
      if (this.isGrouped && // If an action flagged as requiring resort is performed...
      (!changes && resortActions[action] || // ...or if the group field has changes...
      changes && this.groupers.some((grouper) => grouper.field in changes))) {
        this.sort();
      }
    }
    /**
     * Adds or removes records in a group from storage. Used when expanding/collapsing groups.
     * @private
     * @param {Core.data.Model} groupRecord Group which records should be added or removed
     * @param {Boolean} include Include (true) or exclude (false) records
     * @category Grouping
     */
    internalIncludeExcludeGroupRecords(groupRecord, include) {
      const me = this, index = me.indexOf(groupRecord), allIndex = me.allIndexOf(groupRecord), { id: mapId, storage } = me, {
        _filteredValues,
        _values
      } = storage, {
        meta,
        groupChildren,
        unfilteredGroupChildren
      } = groupRecord;
      if (allIndex === -1 || meta.collapsed && !include || !meta.collapsed && include) {
        return;
      }
      unfilteredGroupChildren.forEach(
        (child) => child.instanceMeta(mapId).hiddenByCollapse = !include
      );
      if (include) {
        if (_filteredValues) {
          _filteredValues.splice(index + 1, 0, ...groupChildren.filter((r) => !me.isAvailable(r)));
        }
        storage._values.splice(allIndex + 1, 0, ...unfilteredGroupChildren.filter((r) => !me.isAvailable(r)));
      } else {
        if (_filteredValues) {
          _filteredValues.splice(index + 1, groupChildren.length);
        }
        _values.splice(allIndex + 1, unfilteredGroupChildren.length);
      }
      storage._indicesInvalid = true;
      me._idMap = null;
    }
    /**
     * Removes records in a group from storage. Used when collapsing a group.
     * @private
     * @param groupRecord Group which records should be removed
     * @category Grouping
     */
    excludeGroupRecords(groupRecord) {
      this.internalIncludeExcludeGroupRecords(groupRecord, false);
    }
    /**
     * Adds records in a group to storage. Used when expanding a group.
     * @private
     * @param groupRecord Group which records should be added
     * @category Grouping
     */
    includeGroupRecords(groupRecord) {
      this.internalIncludeExcludeGroupRecords(groupRecord, true);
    }
    /**
     * Collects all group headers + children, whether expanded or not
     * @private
     * @param {Boolean} allRecords True to include filtered out records
     * @param {Boolean} includeHeaders True to also include group headers
     * @returns {Core.data.Model[]}
     */
    collectGroupRecords(allRecords, includeHeaders = true) {
      const records = allRecords ? this.storage.allValues : this.storage.values;
      return records.reduce((records2, record) => {
        if (record.isSpecialRow) {
          if (includeHeaders && !record.isGroupFooter) {
            records2.push(record);
          }
          if (record.isGroupHeader) {
            records2.push.apply(records2, record.groupChildren);
          }
        }
        return records2;
      }, []);
    }
  }, __publicField(_a2, "$name", "StoreGroup"), //region Config
  __publicField(_a2, "configurable", {
    /**
     * Currently used groupers.
     * To set groupers when remote sorting is enabled by {@link Core/data/AjaxStore#config-sortParamName} you should
     * use {@link #function-setGroupers} instead to be able to wait for the operation to finish.
     * @member {Grouper[]} groupers
     * @category Sort, group & filter
     */
    /**
     * Initial groupers, specify to have store grouped automatically after initially setting data
     * @config {Grouper[]}
     * @category Common
     */
    groupers: null,
    useGroupFooters: false,
    /**
     * To have all groups __initially loaded__ start collapsed, configure this as `true`.
     *
     * Note that this only affects the initial load of the store. Subsequent reloads maintain
     * current group state where possible.
     * @config {Boolean}
     * @default false
     * @category Advanced
     */
    startGroupsCollapsed: null
  }), _a2;
};

// ../Core/lib/Core/data/mixin/StoreProxy.js
var StoreProxy_default = (Target) => class StoreProxy extends (Target || Base) {
  static get configurable() {
    return {
      /**
       * Allow object like interaction with the Store. For example:
       *
       * ```javascript
       * const store = new Store({
       *    objectify : true,
       *    data      : [
       *        { id : 'batman', name : 'Bruce' }
       *    ]
       * });
       *
       * // retrieve using id as property
       * const record = store.batman;
       *
       * // add as property
       * store.superman = { name : 'Clark' };
       *
       * // delete to remove
       * delete store.batman;
       * ``
       *
       * @config {Boolean}
       * @default false
       */
      objectify: null
    };
  }
  initProxy() {
    if (!globalThis.Proxy) {
      throw new Error("Proxy not supported");
    }
    const proxy = new Proxy(this, {
      // Support getting records using `store[id/index]
      get(target, property) {
        if (property in target) {
          return target[property];
        }
        if (property === "$store") {
          return target;
        }
        let record = target.getById(property);
        if (!record && !isNaN(parseInt(property))) {
          record = target.getAt(parseInt(property));
        }
        return record;
      },
      // Support adding/replacing records using `store.id = { ...data }`
      set(target, property, value) {
        if (property in target || target.isDestroying) {
          target[property] = value;
        } else {
          target.add({ [target.modelClass.idField]: property, ...value });
        }
        return true;
      },
      // Support deleting records using `delete store.id`
      deleteProperty(target, property) {
        if (target.isDestroying) {
          delete target[property];
          return true;
        }
        return Boolean(target.remove(property).length);
      },
      // Support `id in store`
      has(target, property) {
        if (property in target) {
          return true;
        }
        if (property.startsWith("{") && property.endsWith("}")) {
          const data = StringHelper.safeJsonParse(property);
          property = data == null ? void 0 : data.id;
        }
        return target.includes(property);
      }
    });
    return proxy;
  }
};

// ../Core/lib/Core/data/mixin/StoreRelation.js
var StoreRelation_default = (Target) => {
  var _a2;
  return _a2 = class extends (Target || Base) {
    //region Init
    /**
     * Initialized relations, called from constructor
     * @private
     */
    initRelations(reset) {
      const me = this, relations = me.modelClass.exposedRelations;
      if (reset && me.modelRelations) {
        me.modelRelations.forEach((relation) => {
          var _a3;
          return (_a3 = relation.storeDetacher) == null ? void 0 : _a3.call(relation);
        });
      }
      if ((!me.modelRelations || me.modelRelations.length === 0 || reset) && relations) {
        me.modelRelations = [];
        relations == null ? void 0 : relations.forEach((modelRelationConfig) => {
          const config = { ...modelRelationConfig }, {
            foreignStore,
            relationName,
            relatedCollectionName
          } = config, relatedStore = typeof foreignStore === "string" ? me[foreignStore] : foreignStore;
          config.dependentStore = me;
          me.modelRelations.push(config);
          if (relatedStore) {
            config.foreignStoreProperty = config.foreignStore;
            config.foreignStore = relatedStore;
            const dependentStoreConfigs = relatedStore.dependentStoreConfigs;
            if (dependentStoreConfigs.has(me)) {
              const dependentConfigs = dependentStoreConfigs.get(me);
              if (reset) {
                const existingConfig = dependentConfigs.find((c) => c.relationName === relationName);
                if (existingConfig) {
                  ArrayHelper.remove(dependentConfigs, existingConfig);
                }
              }
              dependentConfigs.push(config);
            } else {
              dependentStoreConfigs.set(me, [config]);
            }
            if (relatedCollectionName) {
              relatedStore.initRelationCollection(config, me);
            }
            if (relatedStore.count > 0) {
              relatedStore.updateDependentStores("dataset", relatedStore.records);
            }
          }
        });
      }
    }
    /**
     * Called from other end of an relation when this store should hold a collection of related records.
     * @private
     * @param config
     * @param collectionStore
     */
    initRelationCollection(config, collectionStore) {
      const me = this, name = config.relatedCollectionName, collectionStores = me.collectionStores || (me.collectionStores = {});
      collectionStores[name] = {
        store: collectionStore,
        config
      };
      if (!me[name + "Store"]) {
        me[name + "Store"] = collectionStore;
      }
      if (me.count > 0) {
        me.initModelRelationCollection(name, me.records);
      }
    }
    initModelRelationCollection(name, records) {
      const me = this;
      records.forEach((record) => {
        record.traverse((node) => {
          const useName = name in node ? `$related${StringHelper.capitalize(name)}` : name;
          Object.defineProperty(node, useName, {
            enumerable: true,
            configurable: true,
            get: function() {
              return me.getCollection(this, name);
            },
            set: function(value) {
              return me.setCollection(this, name, value);
            }
          });
        });
      });
    }
    /**
     * Updates relationCache for all records.
     * @private
     */
    resetRelationCache() {
      this.relationCache = {};
      this.forEach((record) => record.initRelations());
    }
    /**
     * Caches related records from related store on the local store.
     * @private
     * @param record Local record
     * @param relations Relations to related store
     */
    updateRecordRelationCache(record, relations) {
      relations == null ? void 0 : relations.forEach((relation) => {
        const { config } = relation, foreignId = relation.related ? relation.related.id : record.getValue(config.foreignKey);
        foreignId !== void 0 && this.cacheRelatedRecord(record, foreignId, config.relationName, foreignId);
      });
    }
    //endregion
    //region Getters
    /**
     * Returns records the relation cache. Same result as if retrieving the collection on the dependent store, but
     * without the need of accessing that store.
     * @internal
     * @param {String} name
     * @param {Core.data.Model|String|Number} recordOrId
     * @returns {Array}
     */
    getRelationCollection(name, recordOrId) {
      var _a3;
      const id = Model.asId(recordOrId);
      return ((_a3 = this.relationCache[name]) == null ? void 0 : _a3[id]) || [];
    }
    /**
     * Returns records from a collection of related records. Not to be called directly, called from Model getter.
     * @private
     * @param {Core.data.Model} record
     * @param {String} name
     * @returns {Array}
     */
    getCollection(record, name) {
      var _a3;
      const { config, store } = this.collectionStores[name];
      return ((_a3 = store.relationCache[config.relationName]) == null ? void 0 : _a3[record.id]) || [];
    }
    /**
     * Sets a collection of related records. Will updated the related store and trigger events from it. Not to be called
     * directly, called from Model setter.
     * @private
     */
    setCollection(model, name, records) {
      const { config, store } = this.collectionStores[name], relationCache = store.relationCache[config.relationName] || (store.relationCache[config.relationName] = {}), old = (relationCache[model.id] || []).slice(), added = [], removed = [];
      store.suspendEvents();
      old.forEach((record) => {
        if (!records.includes(record)) {
          record[config.foreignKey] = null;
          store.remove(record);
          removed.push(record);
        }
      });
      records.forEach((record) => {
        if (record.isModel instanceof Model) {
          if (!record.stores.includes(store)) {
            store.add(record);
            added.push(record);
          }
        } else {
          [record] = store.add(record);
          added.push(record);
        }
        record[config.foreignKey] = model.id;
      });
      store.resumeEvents();
      if (removed.length) {
        store.trigger("remove", { records: removed });
        store.trigger("change", { action: "remove", records: removed });
      }
      if (added.length) {
        store.trigger("add", { records: added });
        store.trigger("change", { action: "add", records: added });
      }
    }
    //endregion
    //region Caching
    /**
     * Adds a record to relation cache, optionally removing it if already there.
     * @private
     * @param record
     * @param id
     * @param name
     * @param uncacheId
     */
    cacheRelatedRecord(record, id, name, uncacheId = null) {
      const me = this, cache = me.relationCache[name] || (me.relationCache[name] = {});
      if (uncacheId !== null) {
        me.uncacheRelatedRecord(record, name, uncacheId);
      }
      if (id != null) {
        ArrayHelper.include(cache[id] || (cache[id] = []), record);
      }
    }
    /**
     * Removes a record from relation cache, for a specific relation (specify relation name and id) or for all relations
     * @private
     * @param record Record to remove from cache
     * @param name Optional, relation name
     * @param id Optional, id
     */
    uncacheRelatedRecord(record, name = null, id = null) {
      const me = this;
      function remove(relationName, relatedId) {
        const cache = me.relationCache[relationName], oldCache = cache == null ? void 0 : cache[relatedId];
        if (oldCache) {
          const uncacheIndex = oldCache.indexOf(record);
          uncacheIndex >= 0 && oldCache.splice(uncacheIndex, 1);
          if (oldCache.length === 0) {
            delete cache[relatedId];
          }
        }
      }
      if (id != null) {
        remove(name, id);
      } else {
        if (record.meta.relationCache) {
          Object.entries(record.meta.relationCache).forEach(
            ([relationName, relatedRecord]) => remove(relationName, relatedRecord == null ? void 0 : relatedRecord.id)
          );
        }
      }
    }
    /**
     * Updates related stores when store is cleared, a record is removed or added.
     * @private
     * @param {String} action
     * @param {Core.data.Model[]} records
     */
    updateDependentStores(action, records) {
      this.dependentStoreConfigs.forEach((configs) => {
        configs.forEach((config) => {
          const {
            dependentStore,
            relatedCollectionName,
            relationName,
            foreignKey
          } = config, cache = dependentStore.relationCache[relationName];
          if (action === "dataset") {
            relatedCollectionName && this.initModelRelationCollection(relatedCollectionName, records);
            dependentStore.forEach((record) => {
              const foreign = record.initRelation(config);
              foreign && dependentStore.cacheRelatedRecord(record, foreign.id, relationName, foreign.id);
            });
            return;
          }
          if (action === "removeall") {
            dependentStore.forEach((record) => record.removeRelation(config));
            delete dependentStore.relationCache[relationName];
            return;
          }
          if (action === "add") {
            relatedCollectionName && this.initModelRelationCollection(relatedCollectionName, records);
          }
          if (action === "add" || action === "remove") {
            records.forEach((record) => {
              const dependentRecords = cache == null ? void 0 : cache[record.id];
              switch (action) {
                case "remove":
                  if (dependentRecords) {
                    dependentRecords.forEach((dependentRecord) => dependentRecord.removeRelation(config));
                  }
                  break;
                case "add":
                  dependentStore.forEach((dependentRecord) => {
                    if (dependentRecord.getValue(foreignKey) == record.id) {
                      dependentRecord.initRelation(config);
                      dependentStore.cacheRelatedRecord(dependentRecord, record.id, relationName);
                    }
                  });
                  break;
              }
            });
          }
        });
      });
    }
    /**
     * Updates relation cache and foreign key value when a related objects id is changed.
     * @private
     */
    updateDependentRecordIds(oldValue, value) {
      var _a3;
      (_a3 = this.dependentStoreConfigs) == null ? void 0 : _a3.forEach((configs) => {
        configs.forEach((config) => {
          var _a4;
          const {
            dependentStore,
            relationName,
            foreignKey
          } = config, cache = dependentStore.relationCache[relationName], localRecords = (_a4 = cache == null ? void 0 : cache[oldValue]) == null ? void 0 : _a4.slice();
          localRecords == null ? void 0 : localRecords.forEach((localRecord) => {
            dependentStore.cacheRelatedRecord(localRecord, value, relationName, oldValue);
            localRecord.set(foreignKey, value, false, true);
          });
        });
      });
    }
    //endregion
  }, __publicField(_a2, "$name", "StoreRelation"), _a2;
};

// ../Core/lib/Core/data/mixin/StoreSum.js
var StoreSum_default = (Target) => class StoreSum extends (Target || Base) {
  static get $name() {
    return "StoreSum";
  }
  /**
   * Returns sum calculated by adding value of specified field for specified records. Defaults to using all records
   * in store
   * @param {String} field Field to summarize by
   * @param {Core.data.Model[]} records Records to summarize, uses all records if unspecified.
   * @returns {Number}
   * @category Sum
   */
  sum(field2, records = this.storage.values) {
    if (!records) {
      return 0;
    }
    return records.reduce((sum, record) => {
      if (record.isSpecialRow) {
        return sum;
      }
      const v = Number(record.getValue(field2));
      return isNaN(v) ? sum : sum + v;
    }, 0);
  }
  /**
   * Returns min value for the specified field, can be used with Date or Number values. Defaults to look through all records in store
   * @param {String} field Field to find min value for
   * @param {Core.data.Model[]} records Records to process, uses all records if unspecified
   * @returns {Number|Date}
   * @category Sum
   */
  min(field2, records = this.storage.values) {
    if (!(records == null ? void 0 : records.length)) {
      return 0;
    }
    return records.reduce((min2, record) => {
      const fieldValue = record.getValue(field2), type = typeof (fieldValue == null ? void 0 : fieldValue.valueOf());
      if (type === "number" && fieldValue < min2) {
        min2 = fieldValue;
      }
      return min2;
    }, records[0].getValue(field2));
  }
  /**
   * Returns max value for the specified field, can be used with Date or Number values. Defaults to look through all records in store
   * @param {String} field Field to find max value for
   * @param {Core.data.Model[]} records Records to process, uses all records if unspecified
   * @returns {Number|Date}
   * @category Sum
   */
  max(field2, records = this.storage.values) {
    if (!(records == null ? void 0 : records.length)) {
      return 0;
    }
    return records.reduce((max, record) => {
      const fieldValue = record.getValue(field2), type = typeof (fieldValue == null ? void 0 : fieldValue.valueOf());
      if (type === "number" && fieldValue > max) {
        max = fieldValue;
      }
      return max;
    }, records[0].getValue(field2));
  }
  /**
   * Returns the average value for the specified field. Defaults to look through all records in store
   * @param {String} field Field to calculate average value for
   * @param {Core.data.Model[]} records Records to process, uses all records if unspecified
   * @returns {Number}
   * @category Sum
   */
  average(field2, records = this.storage.values) {
    if (!(records == null ? void 0 : records.length)) {
      return 0;
    }
    let count = 0;
    const sum = records.reduce((sum2, record) => {
      if (record.isSpecialRow) {
        return sum2;
      }
      const v = parseFloat(record.getValue(field2));
      if (!isNaN(v)) {
        count++;
        return sum2 + v;
      } else {
        return sum2;
      }
    }, 0);
    return count > 0 ? sum / count : 0;
  }
  /**
   * Returns sum by adding value of specified field for records in the group with the specified groupValue.
   * @param {*} groupValue The group to summarize
   * @param {String} field Field to summarize by
   * @returns {Number} Sum or null if store not grouped
   * @category Sum
   */
  groupSum(groupValue, field2) {
    return this.sum(field2, this.getGroupRecords(groupValue));
  }
};

// ../Core/lib/Core/data/mixin/StoreSearch.js
var findInString = (value, text) => String(value).toLowerCase().includes(text);
var matchFns = {
  string: findInString,
  number: findInString,
  boolean: findInString,
  date: (value, text) => {
    if (value instanceof Date && text instanceof Date) {
      return value - text === 0;
    }
    return String(value.getMonth() + 1).includes(text) || String(value.getDate()).includes(text) || String(value.getFullYear()).includes(text);
  },
  object: (value, text) => value === text,
  // typeof null === object
  undefined: (value, text) => value === text
};
var StoreSearch_default = (Target) => class StoreSearch extends (Target || Base) {
  static get $name() {
    return "StoreSearch";
  }
  //region Search (multiple hits)
  /**
   * Find all hits matching the specified input
   * @param {String} text Value to search for
   * @param {String[]} fields Fields to search value in
   * @param {Function[]} [formatters] An array of field formatting functions to format the found value
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {StoreSearchResult[]} Array of hits, in the format { index: x, data: record }
   * @category Search
   */
  search(text, fields = null, formatters, searchAllRecords) {
    const records = this.isTree && !searchAllRecords ? this.rootNode.allChildren : this.getAllDataRecords(searchAllRecords), len = records.length, found = [];
    if (text == null) {
      return [];
    }
    if (typeof text === "string") {
      text = text.toLowerCase();
    }
    let i, j, record, value, valueType, comparison;
    for (i = 0; i < len; i++) {
      record = records[i];
      j = 0;
      for (const key of fields || record.fieldNames) {
        value = record.getValue(key);
        valueType = value instanceof Date ? "date" : typeof value;
        const formatter = formatters == null ? void 0 : formatters[j];
        if (formatter) {
          value = formatter(value);
          valueType = "string";
        }
        comparison = matchFns[valueType];
        if (value && (comparison == null ? void 0 : comparison(value, text))) {
          found.push({
            index: i,
            data: record,
            field: key,
            id: record.id
          });
        }
        j++;
      }
    }
    return found;
  }
  /**
   * Find occurrences of the specified `value` in the specified `field` on all records in the store
   * @param {String} field The record field to search in
   * @param {*} value Value to search for
   * @param {Boolean} distinct True to only return distinct matches, no duplicates
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {StoreSearchResult[]} Array of hits, in the format { index: x, data: record }
   * @category Search
   */
  findByField(field2, value, distinct2 = false, searchAllRecords = false) {
    const records = this.getAllDataRecords(searchAllRecords), len = records.length, usedValues = /* @__PURE__ */ new Set(), found = [];
    let i, record, fieldValue;
    if (value != null) {
      value = String(value).toLowerCase();
    }
    for (i = 0; i < len; i++) {
      record = records[i];
      fieldValue = record.getValue(field2);
      if (!distinct2 || !usedValues.has(fieldValue)) {
        const type = fieldValue instanceof Date ? "date" : typeof fieldValue, matchFn = matchFns[type];
        if (value == null && fieldValue === value || value && matchFn(fieldValue, value)) {
          found.push({
            id: record.id,
            index: i,
            data: record
          });
          if (distinct2) {
            usedValues.add(fieldValue);
          }
        }
      }
    }
    return found;
  }
  //endregion
  //region Find (single hit)
  /**
   * Finds the first record for which the specified function returns true
   * @param {Function} fn Comparison function, called with record as parameter
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {Core.data.Model} Record or undefined if none found
   *
   * @example
   * store.find(record => record.color === 'blue');
   * @category Search
   */
  find(fn2, searchAllRecords = false) {
    return this.getAllDataRecords(searchAllRecords).find(fn2);
  }
  /**
   * Finds the first record for which the specified field has the specified value
   * @param {String} fieldName Field name
   * @param {*} value Value to find
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {Core.data.Model} Record or undefined if none found
   * @category Search
   */
  findRecord(fieldName, value, searchAllRecords = false) {
    const matchFn = (r) => ObjectHelper.isEqual(r[fieldName], value);
    if (this.isTree) {
      return this.query(matchFn, searchAllRecords)[0];
    }
    return this.getAllDataRecords(searchAllRecords).find(matchFn);
  }
  /**
   * Searches the Store records using the passed function.
   * @param {Function} fn A function that is called for each record. Return true to indicate a match
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {Core.data.Model[]} An array of the matching Records
   * @category Search
   */
  query(fn2, searchAllRecords = false) {
    if (this.isTree) {
      const matches = [];
      this.traverse((node) => {
        if (fn2(node)) {
          matches.push(node);
        }
      }, void 0, void 0, searchAllRecords);
      return matches;
    }
    return this.getAllDataRecords(searchAllRecords).filter(fn2);
  }
  //endregion
  //region Others
  /**
   * Returns true if the supplied function returns true for any record in the store
   * @param {Function} fn A function that should return true to indicate a match
   * @param {Boolean} [searchAllRecords] True to ignore any applied filters when searching
   * @returns {Boolean}
   *
   * @example
   * store.some(record => record.age > 95); // true if any record has age > 95
   * @category Search
   */
  some(fn2, searchAllRecords = false) {
    return this.getAllDataRecords(searchAllRecords).some(fn2);
  }
  //endregion
};

// ../Core/lib/Core/data/mixin/StoreSort.js
var StoreSort_default = (Target) => class StoreSort extends (Target || Base) {
  static get $name() {
    return "StoreSort";
  }
  //region Config
  static get defaultConfig() {
    return {
      /**
       * Use `localeCompare()` when sorting, which lets the browser sort in a locale specific order. Set to `true`,
       * a locale string or a locale config to enable.
       *
       * Enabling this has big negative impact on sorting
       * performance. For more info on `localeCompare()`, see [MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/localeCompare).
       *
       * Examples:
       *
       * ```javascript
       * const store = new Store({
       *     // Swedish sorting
       *     useLocaleSort : 'sv-SE'
       * });
       *
       * const store = new Store({
       *     // Swedish sorting with custom casing order
       *     useLocaleSort : {
       *         locale    : 'sv-SE',
       *         caseFirst : 'upper'
       *     }
       * });
       * ```
       *
       * Can also be configured on a per-sorter basis:
       *
       * ```javascript
       * store.sort({ field: 'name', useLocaleSort : 'sv-SE' });
       * ```
       *
       * @config {Boolean|String|Object}
       * @default false
       * @category Advanced
       */
      useLocaleSort: null
    };
  }
  static get configurable() {
    return {
      /**
       * Initial sorters, format is [{ field: 'name', ascending: false }, ...]
       * @config {Sorter[]|String[]}
       * @category Common
       */
      sorters: [],
      /**
       * Specify true to sort this store after records are added.
       * @config {Boolean}
       * @default
       * @category Sorting
       */
      reapplySortersOnAdd: false
    };
  }
  //endregion
  //region Events
  /**
   * Fired before sorting
   * @event beforeSort
   * @param {Core.data.Store} source This Store
   * @param {Sorter[]} sorters Sorter configs
   * @param {Core.data.Model[]} records Records to sort
   */
  /**
   * Fired after sorting
   * @event sort
   * @param {Core.data.Store} source This Store
   * @param {Sorter[]} sorters Sorter configs
   * @param {Core.data.Model[]} records Sorted records
   */
  //endregion
  //region Properties
  /**
   * Currently applied sorters
   * @member {Sorter[]} sorters
   * @readonly
   * @category Sort, group & filter
   */
  /**
   * Is store sorted?
   * @property {Boolean}
   * @readonly
   * @category Sort, group & filter
   */
  get isSorted() {
    return Boolean(this.sorters.length) || this.isGrouped;
  }
  changeSorters(sorters) {
    return sorters.map((sorter) => this.normalizeSorterConfig(sorter, true));
  }
  updateReapplySortersOnAdd(enable) {
    this.storage.autoSort = enable;
  }
  //endregion
  //region Add & remove sorters
  /**
   * Sort records, either by replacing current sorters or by adding to them.
   * A sorter can specify a **_custom sorting function_** which will be called with arguments (recordA, recordB).
   * Works in the same way as a standard array sorter, except that returning `null` triggers the stores
   * normal sorting routine.
   *
   * ```javascript
   * // single sorter
   * store.sort('age');
   *
   * // single sorter as object, descending order
   * store.sort({ field : 'age', ascending : false });
   *
   * // multiple sorters
   * store.sort(['age', 'name']);
   *
   * // using custom sorting function
   * store.sort((recordA, recordB) => {
   *     // apply custom logic, for example:
   *     return recordA.name.length < recordB.name.length ? -1 : 1;
   * });
   *
   * // using locale specific sort (slow)
   * store.sort({ field : 'name', useLocaleSort : 'sv-SE' });
   * ```
   *
   * @param {String|Sorter[]|Sorter|Function} field Field to sort by.
   * Can also be an array of {@link Core.util.CollectionSorter sorter} config objects, or a sorting function, or a
   * {@link Core.util.CollectionSorter sorter} config.
   * @param {Boolean} [ascending] Sort order.
   * Applicable when the `field` is a string (if not specified and already sorted by the field, reverts direction),
   * or an object and `ascending` property is not specified for the object. `true` by default.
   * Not applicable when `field` is a function. `ascending` is always `true` in this case.
   * @param {Boolean} [add] If `true`, adds a sorter to the sorters collection.
   * Not applicable when `field` is an array. In this case always replaces active sorters.
   * @param {Boolean} [silent] Set as true to not fire events. UI will not be informed about the changes.
   * @category Sort, group & filter
   * @fires beforeSort
   * @fires sort
   * @fires refresh
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   */
  sort(field2, ascending, add = false, silent = false) {
    const me = this, records = me.allRecords, currentSorters = me.sorters ? me.sorters.slice() : [];
    let currentDir = null, curSort;
    if (field2) {
      if (Array.isArray(field2)) {
        me.sorters = field2.map((sorter) => me.normalizeSorterConfig(sorter, typeof sorter === "string" ? true : ascending));
      } else {
        const sorter = me.normalizeSorterConfig(field2, ascending);
        if (add) {
          curSort = me.getCurrentSorterByField(sorter.field);
          if (curSort) {
            currentDir = curSort.ascending;
            curSort.ascending = sorter.ascending;
          } else {
            me.sorters.push(sorter);
          }
        } else {
          me.sorters = [sorter];
        }
      }
    }
    if (!silent && me.trigger("beforeSort", { sorters: me.sorters, records, currentSorters }) === false) {
      me.sorters = currentSorters;
      if (currentDir !== null) {
        curSort.ascending = currentDir;
      }
      return null;
    }
    return me.performSort(silent);
  }
  normalizeSorterConfig(field2, ascending) {
    var _a2, _b, _c;
    const me = this, sorter = { ascending };
    if (typeof field2 === "object") {
      ObjectHelper.assign(sorter, field2);
      if (field2.fn) {
        delete sorter.fn;
        sorter.sortFn = field2.fn;
      }
      sorter.ascending = (_a2 = field2.ascending) != null ? _a2 : ascending;
    } else if (typeof field2 === "function") {
      sorter.sortFn = field2;
    } else {
      sorter.field = field2;
    }
    if (sorter.ascending == null) {
      const curSort = me.getCurrentSorterByField(sorter.field);
      sorter.ascending = curSort ? !curSort.ascending : true;
    }
    if (sorter.sortFn == null) {
      const compareItems = (_c = (_b = me.modelClass) == null ? void 0 : _b.$meta.fields.map[sorter.field]) == null ? void 0 : _c.compareItems;
      if (compareItems) {
        sorter.sortFn = compareItems;
      }
    }
    return sorter;
  }
  getCurrentSorterByField(field2) {
    return typeof field2 === "string" && this.sorters.find((s) => s.field === field2) || null;
  }
  /**
   * Add a sorting level (a sorter).
   * @param {String|Sorter[]|Sorter|Function} field Field to sort by. Can also be an array of sorters, or a sorting
   * function, or a {@link Core.util.CollectionSorter sorter} config.
   * @param {Boolean} [ascending] Sort order (used only if field specified as string)
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  addSorter(field2, ascending = true) {
    return this.sort(field2, ascending, true);
  }
  /**
   * Remove a sorting level (a sorter)
   * @param {String|Function} field Stop sorting by this field (or sorter function)
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  removeSorter(field2) {
    const sorterIndex = this.sorters.findIndex((sorter) => sorter.field === field2 || sorter.sortFn === field2);
    if (sorterIndex > -1) {
      this.sorters.splice(sorterIndex, 1);
      return this.sort();
    }
  }
  /**
   * Removes all sorters, turning store sorting off.
   * @returns {Promise|null} If {@link Core/data/AjaxStore#config-sortParamName} is set on store, this method returns `Promise`
   * which is resolved after data is loaded from remote server, otherwise it returns `null`
   * @async
   * @category Sort, group & filter
   */
  clearSorters(silent = false) {
    if (this.sorters.length) {
      this.sorters.length = 0;
      return this.sort(void 0, void 0, void 0, silent);
    }
  }
  //region
  //region Sorting logic
  /**
   * Creates a function used with Array#sort when sorting the store. Override to use your own custom sorting logic.
   * @param {Sorter[]} sorters An array of sorter config objects
   * @returns {Function}
   * @category Sort, group & filter
   */
  createSorterFn(sorters) {
    const storeLocaleSort = this.useLocaleSort;
    return (lhs, rhs) => {
      for (let i = 0; i < sorters.length; i++) {
        const sorter = sorters[i], { field: field2, ascending = true, useLocaleSort = storeLocaleSort } = sorter, fn2 = sorter.fn || sorter.sortFn, direction = ascending ? 1 : -1;
        if (fn2) {
          const val = fn2.call(sorter, lhs, rhs);
          if (val === 0) {
            continue;
          }
          if (val !== null) {
            return val * direction;
          }
        }
        const lhsValue = lhs.isModel ? lhs.getValue(field2) : lhs[field2], rhsValue = rhs.isModel ? rhs.getValue(field2) : rhs[field2];
        if (lhsValue === rhsValue) {
          continue;
        }
        if (lhsValue == null) {
          return -direction;
        }
        if (rhsValue == null) {
          return direction;
        }
        if (useLocaleSort && typeof lhsValue === "string") {
          if (useLocaleSort === true) {
            return String(lhsValue).localeCompare(rhsValue) * direction;
          }
          if (typeof useLocaleSort === "string") {
            return String(lhsValue).localeCompare(rhsValue, useLocaleSort) * direction;
          }
          if (typeof useLocaleSort === "object") {
            return String(lhsValue).localeCompare(rhsValue, useLocaleSort.locale, useLocaleSort) * direction;
          }
        }
        if (lhsValue > rhsValue) {
          return direction;
        }
        if (lhsValue < rhsValue) {
          return -direction;
        }
      }
      return 0;
    };
  }
  /**
   * The sorter function for sorting records in the store.
   * @member {Function}
   * @internal
   * @readonly
   */
  get sorterFn() {
    const me = this, { sorters } = me;
    return me.createSorterFn(me.remoteSort ? [{ field: "_remoteSortIndex" }] : me.isGrouped ? me.groupers.concat(sorters) : sorters);
  }
  /**
   * Perform sorting according to the {@link #config-sorters} configured.
   * This is the internal implementation which is overridden in {@link Core.data.AjaxStore} and
   * must not be overridden.
   * @async
   * @private
   * @category Sort, group & filter
   */
  performSort(silent) {
    const me = this, { rootNode, storage, sorterFn: sorter } = me;
    if (me.tree) {
      !me.isChained && rootNode.traverse((node) => {
        if (node.isLoaded && node.isParent) {
          node.children.sort(sorter);
          node.updateChildrenIndices(node.children, "parentIndex", true);
        }
      });
      storage.replaceValues({
        values: me.collectDescendants(rootNode).visible,
        silent: true
      });
    } else if (me.isGrouped) {
      storage.replaceValues({
        ...me.prepareGroupRecords(sorter),
        silent: true
      });
    } else {
      storage.replaceValues({
        values: storage.values.sort(sorter),
        silent: true
      });
    }
    me.afterPerformSort(silent || me.isRemoteDataLoading);
  }
  afterPerformSort(silent) {
    if (silent) {
      return;
    }
    const me = this;
    me._idMap = null;
    const event = {
      action: "sort",
      sorters: me.sorters,
      records: me.allRecords
    };
    me.trigger("sort", event);
    me.trigger("refresh", event);
  }
  //endregion
};

// ../Core/lib/Core/data/mixin/StoreChained.js
var returnTrue = () => true;
var StoreChained_default = (Target) => class StoreChained extends (Target || Base) {
  static get $name() {
    return "StoreChained";
  }
  //region Config
  static get defaultConfig() {
    return {
      /**
       * Function used to filter records in the masterStore into a chained store. If not provided,
       * all records from the masterStore will be included in the chained store.
       * @config {Function}
       * @category Chained store
       */
      chainedFilterFn: null,
      /**
       * Array of field names that should trigger filtering of chained store when the fields are updated.
       * @config {String[]}
       * @category Chained store
       */
      chainedFields: null,
      /**
       * Master store that a chained store gets its records from.
       * @config {Core.data.Store}
       * @category Chained store
       */
      masterStore: null,
      /**
       * Method names calls to which should be relayed to master store.
       * @config {String[]}
       * @category Chained store
       */
      doRelayToMaster: ["add", "remove", "insert"],
      /**
       * Method names calls to which shouldn't be relayed to master store.
       * @config {String}
       * @category Chained store
       */
      dontRelayToMaster: [],
      /**
       * If true, collapsed records in original tree will be excluded from the chained store.
       * @config {Boolean}
       * @category Chained store
       */
      excludeCollapsedRecords: true,
      chainSuspended: 0
    };
  }
  // All props should be predefined to work properly with objectified stores
  static get properties() {
    return {
      chainedStores: null
    };
  }
  //endregion
  construct(config) {
    super.construct(config);
    const me = this, { masterStore } = me, sort = me.syncOrder ? "sort" : "";
    if (masterStore) {
      me.methodNamesToRelay.forEach((fnName) => me[fnName] = (...params) => me.relayToMaster(fnName, params));
      me.removeAll = (...params) => {
        masterStore.remove(me.getRange(), ...params);
      };
      masterStore.ion({
        // HACK to have chained stores react early in a async events scenario (with engine). Could be turned
        // into a config, but this way one does not have to think about it
        changePreCommit: me.onMasterDataChangedPreCommit,
        change: me.onMasterDataChanged,
        [sort]: me.onMasterDataChanged,
        prio: 1,
        thisObj: me
      });
      if (!masterStore.chainedStores) {
        masterStore.chainedStores = [];
      }
      masterStore.chainedStores.push(me);
      me.fillFromMaster();
    }
  }
  //region Properties
  // For accessing the full set of records, whether chained or not
  get $master() {
    return this.masterStore || this;
  }
  /**
   * Is this a chained store?
   * @property {Boolean}
   * @readonly
   * @category Advanced
   */
  get isChained() {
    return Boolean(this.masterStore);
  }
  set chainedFilterFn(chainedFilterFn) {
    this._chainedFilterFn = this.thisObj ? chainedFilterFn.bind(this.thisObj) : chainedFilterFn;
  }
  get chainedFilterFn() {
    return this._chainedFilterFn || returnTrue;
  }
  get methodNamesToRelay() {
    const doIsArray = Array.isArray(this.doRelayToMaster), dontIsArray = Array.isArray(this.dontRelayToMaster);
    return doIsArray && this.doRelayToMaster.filter((name) => !dontIsArray || !this.dontRelayToMaster.includes(name)) || [];
  }
  //endregion
  //region Internal
  updateChainedStores() {
    if (this.chainedStores) {
      this.chainedStores.forEach((store) => store.fillFromMaster());
    }
  }
  /**
   * Updates records available in a chained store by filtering the master store records using
   * {@link #config-chainedFilterFn}
   * @category Chained store
   */
  fillFromMaster() {
    const me = this, { masterStore, isTree } = me;
    let records = [];
    if (!me.isChained) {
      throw new Error("fillFromMaster only allowed on chained store");
    }
    if (me.isChainSuspended) {
      return;
    }
    if (masterStore.isGrouped && masterStore.isFiltered) {
      masterStore.forEach((r) => records.push(r), masterStore, { includeFilteredOutRecords: true, includeCollapsedGroupRecords: true });
    } else {
      records = masterStore.allRecords.filter((r) => !r.isSpecialRow && me.chainedFilterFn(r));
    }
    if (isTree) {
      me.idRegister = {};
      me.internalIdRegister = {};
      records.forEach((r) => {
        if (r.stores.includes(me)) {
          me.register(r);
        } else {
          r.joinStore(me);
        }
      });
      if (me.excludeCollapsedRecords) {
        const children = me.getChildren(me.rootNode);
        records = me.doIncludeExclude(children, true);
      }
    }
    me.isFillingFromMaster = true;
    me.data = records;
    me.isFillingFromMaster = false;
  }
  /**
   * Commits changes back to master.
   * - the records deleted from chained store and present in master will be deleted from master
   * - the records added to chained store and missing in master will added to master
   * Internally calls {Store#function-commit commit()}.
   * @returns {Object} Changes, see Store#changes
   * @internal
   */
  commitToMaster() {
    const me = this, master = me.masterStore;
    if (!me.isChained) {
      throw new Error("commitToMaster only allowed on chained store");
    }
    master.beginBatch();
    master.remove(me.removed.values);
    master.add(me.added.values);
    master.endBatch();
    return me.commit();
  }
  /**
   * Relays some function calls to the master store
   * @private
   */
  relayToMaster(fnName, params) {
    return this.masterStore[fnName](...params);
  }
  // HACK, when used with engine the chained store will catch events early (sync) and prevent late (async) listeners
  onMasterDataChangedPreCommit(event) {
    this.onMasterDataChanged(event);
    this.$masterEventhandled = true;
  }
  /**
   * Handles changes in master stores data. Updates the chained store accordingly
   * @private
   */
  onMasterDataChanged({ action, changes, $handled, isMove }) {
    var _a2;
    if (this.$masterEventhandled) {
      this.$masterEventhandled = false;
      return;
    }
    if (isMove && action === "remove") {
      return;
    }
    if (action !== "update" || ((_a2 = this.chainedFields) == null ? void 0 : _a2.some((field2) => field2 in changes))) {
      this.fillFromMaster();
    }
  }
  //endregion
  //region public API
  /**
   * Creates a chained store, a new Store instance that contains a subset of the records from current store.
   * Which records is determined by a filtering function, which is reapplied when data in the base store changes.
   *
   * ```javascript
   * const oldies = store.makeChained(record => record.age > 50);
   * // or use a simple query
   * const ages = store.makeChained(() => store.allRecords.distinct('age')));
   * ```
   *
   * If this store is a {@link Core.data.mixin.StoreTree#property-isTree tree} store, then the resulting chained store
   * will be a tree store sharing the same root node, but only child nodes which pass the `chainedFilterFn` will be
   * considered when iterating the tree through the methods such as
   * {@link Core.data.Store#function-traverse} or {@link Core.data.Store#function-forEach}.
   *
   * @param {Function} [chainedFilterFn] Either a filter function called for every record to determine if it should be
   * included (return true / false), or a query function called with no arguments (see example below). Defaults to
   * including all records (fn always returning true)
   * @param {String[]} [chainedFields] Array of fields that trigger filtering when they are updated
   * @param {StoreConfig} [config] Additional chained store configuration. See {@link Core.data.Store#configs}
   * @param {Class} [config.storeClass] The Store class to use if this Store type is not required.
   * @returns {Core.data.Store}
   * @category Chained store
   */
  makeChained(chainedFilterFn = returnTrue, chainedFields, config) {
    return new ((config == null ? void 0 : config.storeClass) || this.constructor)({
      ...config || {},
      tree: false,
      autoTree: false,
      // If someone ever chains a chained store, chain master instead
      masterStore: this.$master,
      modelClass: this.modelClass,
      // Chained store should never use syncDataOnLoad, that will create an infinite loop when they determine
      // that a record is added and then add it to master, repopulating this store and round we go
      syncDataOnLoad: false,
      chainedFilterFn,
      chainedFields
    });
  }
  /**
   * Alias for {@link Core.data.Store#function-makeChained}
   *
   * @param {Function} [chainedFilterFn] Either a filter function called for every record to determine if it should be
   * included (return true / false), or a query function called with no arguments (see example below). Defaults to
   * including all records (fn always returning true)
   * @param {String[]} [chainedFields] Array of fields that trigger filtering when they are updated
   * @param {StoreConfig} [config] Additional chained store configuration. See {@link Core.data.Store#configs}
   * @param {Class} [config.storeClass] The Store class to use if this Store type is not required.
   * @returns {Core.data.Store}
   * @category Chained store
   */
  chain() {
    return this.makeChained(...arguments);
  }
  //endregion
  doDestroy() {
    var _a2;
    (_a2 = this.chainedStores) == null ? void 0 : _a2.forEach((chainedStore) => chainedStore.destroy());
    super.doDestroy();
  }
  suspendChain() {
    this.chainSuspended++;
  }
  resumeChain(refill = false) {
    if (this.chainSuspended && !--this.chainSuspended && refill) {
      this.fillFromMaster();
    }
  }
  get isChainSuspended() {
    return this.chainSuspended > 0;
  }
};

// ../Core/lib/Core/data/mixin/StoreState.js
var StoreState_default = (Target) => class StoreState extends (Target || Base) {
  static get $name() {
    return "StoreState";
  }
  /**
   * Get store state. Used by State-plugin to serialize state
   * @private
   * @returns {{ sorters, groupers }}
   */
  getState() {
    const { sorters, groupers, filters } = this, state = {};
    if (sorters == null ? void 0 : sorters.length) {
      state.sorters = sorters.map((sorter) => {
        const clone = ObjectHelper.cleanupProperties(ObjectHelper.clone(sorter));
        delete clone.fn;
        delete clone.sortFn;
        return clone;
      });
    }
    if (groupers == null ? void 0 : groupers.length) {
      state.groupers = groupers.map((grouper) => {
        const clone = ObjectHelper.cleanupProperties(ObjectHelper.clone(grouper));
        delete clone.fn;
        return clone;
      });
    }
    if (filters == null ? void 0 : filters.values.length) {
      state.filters = filters.values.map((filter2) => {
        const clone = ObjectHelper.cleanupProperties(ObjectHelper.clone(filter2.config));
        clone.value = filter2.value;
        if (clone.caseSensitive) {
          delete clone.caseSensitive;
        }
        return clone;
      });
    }
    return state;
  }
  /**
   * Apply store state. Used by State-plugin to restore a previously serialized state
   * @private
   * @param {{ sorters, groupers }} state
   */
  applyState(state) {
    const me = this, {
      sorters = [],
      groupers = [],
      filters = []
    } = state, {
      sortParamName,
      filterParamName
    } = me;
    me.sorters = sorters.filter((sorter) => sorter.field || sorter.sortFn && !sortParamName);
    me.groupers = groupers.slice();
    me.sort();
    me.filters = filters.filter((filter2) => filter2.property || filter2.filterBy && !filterParamName);
    me.filter();
  }
};

// ../Core/lib/Core/data/Wbs.js
var zeroPad = (v) => String(v).padStart(6, "0");
var Wbs = class {
  /**
   * Wbs constructor.
   * @param {String|Number} value The value of WBS
   */
  constructor(value) {
    this.value = value;
    this._padded = null;
  }
  /**
   * The WBS value
   * @readonly
   * @member {String} value
   */
  set value(value) {
    this._value = String(value != null ? value : "");
  }
  get value() {
    return this._value;
  }
  /**
   * Returns a `Wbs` instance given a `value`. If the `value` is already a `Wbs` object, it is returned. Otherwise,
   * a new `Wbs` is created. If `value` is `null` or `undefined`, that value is returned.
   * @param {String|Number|Core.data.Wbs} value
   * @returns {Core.data.Wbs}
   */
  static from(value) {
    return value == null ? value : value instanceof Wbs ? value : new Wbs(value);
  }
  /**
   * Returns a WBS code where each component is 0-padded on the left to 6 digits. That is "1.2" is padded to be
   * "000001.000002". These values can be compared for proper semantic order (e.g., Wbs.pad('1.2') < Wbs.pad('1.10')).
   * @param {String|Number|Core.data.Wbs} value
   * @returns {String}
   * @private
   */
  static pad(value) {
    return value instanceof Wbs ? value.valueOf() : Wbs.split(value).map(zeroPad).join(".");
  }
  /**
   * Returns an array of digits from a given WBS code `value`. If the value cannot be converted, an empty array is
   * returned.
   * @param {String|Number|Core.data.Wbs} value
   * @returns {Number[]}
   * @private
   */
  static split(value) {
    let i, ret;
    if (value || value === 0) {
      switch (typeof value) {
        case "object":
          value = String(value);
        case "string":
          ret = value.split(".");
          for (i = ret.length; i-- > 0; ) {
            ret[i] = parseInt(ret[i], 10);
          }
          break;
        case "number":
          ret = [value];
          break;
      }
    }
    return ret || [];
  }
  /**
   * Compares two WBS values, returning 0 if equal, -1 if `lhs` is less than `rhs, or 1 if `lhs` is greater than `rhs`.
   * @param {String|Core.data.Wbs} lhs
   * @param {String|Core.data.Wbs} rhs
   * @returns {Number}
   */
  static compare(lhs, rhs) {
    if (lhs === rhs) {
      return 0;
    }
    if (!lhs || !rhs) {
      return lhs ? 1 : rhs ? -1 : 0;
    }
    lhs = Wbs.pad(lhs);
    rhs = Wbs.pad(rhs);
    return lhs < rhs ? -1 : rhs < lhs ? 1 : 0;
  }
  /**
   * Appends a sub-level WBS value to this WBS code and returns a `Wbs` instance for it.
   * @param {String|Number} value
   * @returns {Core.data.Wbs}
   */
  append(value) {
    const s = this.value;
    return Wbs.from(s ? `${s}.${value}` : value);
  }
  /**
   * Returns truthy value if this Wbs equals the passed value.
   * @param {String|Core.data.Wbs} value
   * @returns {Boolean}
   */
  isEqual(value) {
    return !Wbs.compare(this, value);
  }
  /**
   * Compares this WBS value with a specified pattern, returning `true` if they match. If the `pattern` is simply a
   * sequence of digits and decimal points (e.g., "1.2"), it is a match if it is a substring of this WBS code (e.g.,
   * "3.1.2.4"). If the `pattern` starts with `*` (e.g., "*.1.2"), it is a match if this WBS code ends with the text
   * following the `*` (e.g., "4.3.1.2"). If the `pattern` ends with `*`, it is a match if this WBS code starts with
   * the text up to the `*`.
   *
   * Some examples:
   * ```
   *  console.log(Wbs.from('1.2.3.4').match('2.3'));
   *  > true
   *  console.log(Wbs.from('1.2.3.4').match('*.4'));
   *  > true
   *  console.log(Wbs.from('1.2.3.4').match('1.2.*'));
   *  > true
   *
   *  console.log(Wbs.from('1.2.3.4').match('2.4'));
   *  > false
   *  console.log(Wbs.from('1.2.3.4').match('*.3'));
   *  > false
   *  console.log(Wbs.from('1.2.3.4').match('2.*'));
   *  > false
   * ```
   * @param {String} pattern A partial WBS code (e.g., "1.2"), optionally starting or ending with `*`.
   * @returns {Boolean}
   */
  match(pattern) {
    let ret = false;
    if (pattern) {
      const wbs = this.value, globLeft = pattern[0] === "*", globRight = pattern.endsWith("*"), n = pattern.length;
      if (globLeft === globRight) {
        ret = wbs.indexOf(globLeft ? pattern.substr(1, n - 2) : pattern) > -1;
      } else if (globLeft) {
        ret = wbs.endsWith(pattern.substr(1));
      } else {
        ret = wbs.startsWith(pattern.substr(0, n - 1));
      }
    }
    return ret;
  }
  toString() {
    return this.value;
  }
  toJSON() {
    return this.toString();
  }
  valueOf() {
    var _a2;
    return (_a2 = this._padded) != null ? _a2 : this._padded = Wbs.pad(this.value);
  }
};
Wbs._$name = "Wbs";

// ../Core/lib/Core/data/mixin/StoreTree.js
var emptyArray3 = Object.freeze([]);
var StopBranch = Symbol("StopBranch");
var StoreTree_default = (Target) => {
  var _a2;
  return _a2 = class extends (Target || Base) {
    get StopBranch() {
      return StopBranch;
    }
    //region Getters
    /**
     * True if this Store is configured to handle tree data (with `tree : true`) or if this is a
     * {@link Core.data.Store#function-makeChained chained store} and the master store is a tree store.
     * @property {Boolean}
     * @readonly
     * @category Tree
     */
    get isTree() {
      return this.tree || this.masterStore && this.masterStore.tree;
    }
    /**
     * Returns all leaf records in a tree store
     * @property {Core.data.Model[]}
     * @category Tree
     */
    get leaves() {
      const me = this, result = [];
      if (me.isTree) {
        me.traverse((record) => {
          if (record.isLeaf) {
            result.push(record);
          }
        });
        return result;
      } else {
        me.allRecords.forEach((r) => {
          if (r.isLeaf) {
            result.push(r);
          }
          r.traverse((record) => {
            if (record.isLeaf) {
              result.push(record);
            }
          }, true);
        });
      }
      return result;
    }
    //endregion
    //region Children
    /**
     * Loads children for a parent node that uses load on demand (when expanding it). Base implementation does nothing,
     * either use AjaxStore which implements it, create your own subclass with an implementation or listen for
     * `toggleNode` and insert records when you have them available.
     * @param {Core.data.Model} parentRecord
     * @returns {Promise} A Promise which will be resolved if the load succeeds, and rejected if the load is
     * vetoed by a {@link Core.data.AjaxStore#event-beforeLoadChildren} handler, or if an {@link Core.data.AjaxStore#event-exception} is detected.
     * The resolved function is passed the event object passed to any event handlers.
     * The rejected function is passed the {@link Core.data.AjaxStore#event-exception} event if an exception occurred,
     * or `false` if the load was vetoed by a {@link Core.data.AjaxStore#event-beforeLoadChildren} handler.
     * @category Tree
     */
    async loadChildren(parentRecord) {
    }
    /**
     * Called from Model when adding children. Not to be called directly, use Model#appendChild() instead.
     * @internal
     * @param {Core.data.mixin.TreeNode} parent
     * @param {Core.data.mixin.TreeNode[]} children
     * @param {Number} index
     * @param {Object} isMove
     * @param {Boolean} [silent]
     * @fires add
     * @fires change
     * @category Tree
     */
    onNodeAddChild(parent, children, index, isMove, silent = false) {
      var _a3;
      const me = this, isRootLoad = parent === me.rootNode && parent.isLoading, { storage } = me, { previousSibling } = children[0];
      let storeInsertionPoint;
      const { visible: toAddToUI, all: toAdd } = me.collectDescendants(children, void 0, void 0, {
        inCollapsedBranch: !(parent.isExpanded(me) && parent.ancestorsExpanded(me)),
        applyFilter: me.isFiltered && me.reapplyFilterOnAdd
      });
      if (!isRootLoad && toAdd.length) {
        for (const record of toAdd) {
          if (!me.modified.includes(record) && !isMove[record.id]) {
            if (me.removed.includes(record)) {
              me.removed.remove(record);
            } else if (!record.isLinked) {
              me.added.add(record);
            }
          }
        }
      }
      if (isRootLoad && me.rootVisible) {
        toAddToUI.unshift(parent);
        toAdd.unshift(parent);
      }
      if (toAddToUI.length) {
        if (index === 0 || !previousSibling) {
          storeInsertionPoint = storage.indexOf(parent);
        } else {
          storeInsertionPoint = storage.indexOf(previousSibling) + previousSibling.getDescendantCount(true, me);
        }
        storage.suspendEvents();
        me.storage.splice(++storeInsertionPoint, 0, toAddToUI);
        storage.resumeEvents();
        me._idMap = null;
      } else {
        me._allRecords = null;
      }
      me.updateDependentStores("add", children);
      if (isRootLoad && toAddToUI.length) {
        if (me.sorters.length) {
          me.sort(null, null, false, true);
        }
        (_a3 = me.afterLoadData) == null ? void 0 : _a3.call(me);
        if (!silent) {
          const event = { action: "dataset", data: me._data, records: toAddToUI };
          me.trigger("refresh", event);
          me.trigger("change", event);
        }
      } else if (!silent) {
        const event = { action: "add", parent, isChild: true, isMove, records: children, allRecords: toAdd, index: storeInsertionPoint };
        me.trigger("add", event);
        me.trigger("change", event);
        if (Object.values(isMove).some((wasMoved) => wasMoved)) {
          const event2 = {
            newParent: parent,
            records: children.filter((record) => isMove[record.id]),
            oldParents: children.map((child) => {
              return me.getById(child.meta.oldParentId);
            })
          };
          me.trigger("move", event2);
        }
      }
    }
    onNodeRemoveChild(parent, children, index, flags = { isMove: false, silent: false, unfiltered: false }) {
      const me = this, { storage } = me, toRemoveFromUI = [], toRemove = [], { isMove, silent, unfiltered } = flags, removeUnfiltered = unfiltered && me.isFiltered, childrenToRemove = removeUnfiltered && parent.unfilteredChildren ? parent.unfilteredChildren : children;
      me.collectDescendants(childrenToRemove, toRemoveFromUI, toRemove, { inCollapsedBranch: !(parent.isExpanded(me) && parent.ancestorsExpanded(me)), unfiltered: removeUnfiltered });
      if (!isMove) {
        for (const record of children) {
          record.unjoinStore(me);
        }
        for (const record of toRemove) {
          if (record.stores.includes(me)) {
            record.unjoinStore(me);
          }
          if (me.added.includes(record)) {
            me.added.remove(record);
          } else if (!record.isLinked) {
            me.removed.add(record);
          }
        }
        me.modified.remove(toRemove);
      }
      if (toRemoveFromUI.length) {
        index = storage.indexOf(toRemoveFromUI[0]);
        if (index > -1) {
          storage.suspendEvents();
          storage.splice(index, toRemoveFromUI.length);
          storage.resumeEvents();
          me._idMap = null;
        }
      } else {
        index = -1;
        me._allRecords = null;
      }
      if (!silent && (me.fireRemoveEventForMoveAction || !isMove)) {
        const event = {
          action: "remove",
          parent,
          isChild: true,
          isMove,
          records: children,
          allRecords: toRemove,
          index
        };
        me.trigger("remove", event);
        me.trigger("change", event);
      }
      return toRemove;
    }
    // IMPORTANT when using `applyFilter` option, should use the return value of this function
    // instead of relying on arguments mutation
    collectDescendants(node, visible = [], all = [], flags = {}) {
      var _a3;
      const me = this, { inCollapsedBranch = false, unfiltered = false, applyFilter = false } = flags, children = Array.isArray(node) ? node : (_a3 = me.getChildren(node, unfiltered)) != null ? _a3 : [];
      if (applyFilter) {
        return {
          visible: children.flatMap((child) => this.collectVisibleNodeDescendantsFiltered(child)),
          all: children.flatMap((child) => child.allChildren)
        };
      } else {
        for (let i = 0, len = children.length, child; i < len; i++) {
          child = children[i];
          if (!inCollapsedBranch) {
            visible.push(child);
          }
          all.push(child);
          me.collectDescendants(child, visible, all, {
            inCollapsedBranch: inCollapsedBranch || !child.isExpanded(me),
            unfiltered
          });
        }
        return { visible, all };
      }
    }
    collectVisibleNodeDescendantsFiltered(node) {
      const children = node.unfilteredChildren || node.children;
      if (!children || children.length === 0 || !node.isLeaf && !node.isExpanded(this)) {
        return this.filtersFunction(node) ? [node] : [];
      }
      const filteredChildren = children.flatMap((child) => this.collectVisibleNodeDescendantsFiltered(child));
      return filteredChildren.length || this.filtersFunction(node) ? [node, ...filteredChildren] : [];
    }
    /**
     * Returns the children of the passed branch node which this store owns. By default, this
     * is the entire `children` array.
     *
     * **If this store {@link Core.data.mixin.StoreChained#property-isChained isChained}**, then
     * this returns only the subset of children which are filtered into this store by the
     * {@link Core.data.mixin.StoreChained#config-chainedFilterFn chainedFilterFn}.
     * @param {Core.data.Model} parent The node to return the children of.
     * @returns {Core.data.Model[]}
     * @category Tree
     */
    getChildren(parent, unfiltered = false) {
      const me = this, children = (unfiltered || me.isChained) && parent.unfilteredChildren || parent.children;
      return !(children == null ? void 0 : children.length) ? emptyArray3 : me.isChained ? children.filter(me.chainedFilterFn).sort(me.sorterFn) : children;
    }
    /**
     * Includes or excludes all records beneath parentRecord in storage. Used when expanding or collapsing
     * nodes.
     * @private
     * @param parentRecord Parent record
     * @param include Include (true) or exclude (false)
     * @category Tree
     */
    internalToggleTreeSubRecords(parentRecord, include) {
      const me = this, { storage } = me, index = storage.indexOf(parentRecord), children = me.doIncludeExclude(me.getChildren(parentRecord), include);
      if (me.isFiltered && include && parentRecord.unfilteredChildren) {
        me.updateChildrenHiddenState(parentRecord);
      }
      if (children.length && index !== false) {
        storage.suspendEvents();
        if (include) {
          storage.splice(index + 1, 0, ...children);
          const event = { action: "add", isExpand: true, records: children, index: index + 1 };
          me.trigger("add", event);
          me.trigger("change", event);
        } else {
          storage.splice(index + 1, children.length);
          const event = { action: "remove", isCollapse: true, records: children, index: index + 1 };
          me.trigger("remove", event);
          me.trigger("change", event);
        }
        storage.resumeEvents();
        me._idMap = null;
      }
    }
    // Updates the hidden flag of its children while store is filtered
    updateChildrenHiddenState(parentRecord) {
      var _a3;
      (_a3 = parentRecord.unfilteredChildren) == null ? void 0 : _a3.forEach((child) => {
        child.instanceMeta(this.id).hidden = false;
        if (!child.isLeaf) {
          this.updateChildrenHiddenState(child);
        }
      });
    }
    doIncludeExclude(children, include, result = []) {
      const me = this, childCount = (children == null ? void 0 : children.length) || 0;
      for (let i = 0; i < childCount; i++) {
        const child = children[i];
        if (!me.isChained || me.chainedFilterFn(child)) {
          const mapMeta = child.instanceMeta(me.id);
          if (include || !mapMeta.hidden) {
            result.push(child);
          }
          mapMeta.hidden = !include;
          if (child.isExpanded(me)) {
            me.doIncludeExclude(me.getChildren(child), include, result);
          }
        }
      }
      return result;
    }
    /**
     * Collapse an expanded record or expand a collapsed. Optionally forcing a certain state.
     * @param {String|Number|Core.data.Model} idOrRecord Record (the record itself) or id of a record to toggle
     * @param {Boolean} [collapse] Force collapse (true) or expand (false)
     * @category Tree
     */
    async toggleCollapse(idOrRecord, collapse) {
      const me = this, record = me.getById(idOrRecord), meta = record.instanceMeta(me);
      if (collapse === void 0) {
        collapse = !meta.collapsed;
      }
      if (!meta.isLoadingChildren && !record.isLeaf && record.isExpanded(me) === collapse) {
        me.trigger("beforeToggleNode", { record, collapse });
        meta.collapsed = collapse;
        if (meta.collapsed) {
          me.onNodeCollapse(record);
          return true;
        } else {
          me.onNodeExpand(record);
          let success2 = true;
          if (!record.isLoaded) {
            meta.isLoadingChildren = true;
            try {
              await me.loadChildren(record);
            } catch (exception) {
              meta.collapsed = true;
              success2 = false;
              me.trigger("loadChildrenException", { record, exception });
            } finally {
              meta.isLoadingChildren = false;
            }
          }
          return success2;
        }
      }
    }
    /**
     * Remove all records beneath parentRecord from storage.
     * @private
     * @param parentRecord Parent record
     * @category Tree
     */
    onNodeCollapse(parentRecord) {
      if (parentRecord.ancestorsExpanded(this)) {
        return this.internalToggleTreeSubRecords(parentRecord, false);
      }
    }
    /**
     * Add all records beneath parentRecord from storage.
     * @private
     * @param parentRecord Parent record
     * @category Tree
     */
    onNodeExpand(parentRecord) {
      if (parentRecord.ancestorsExpanded(this)) {
        return this.internalToggleTreeSubRecords(parentRecord, true);
      }
    }
    //endregion
    //region Transform flat data
    /**
     * Transforms flat data containing parent ids into tree data
     * @param {Object[]} data Flat raw data
     * @returns {Object[]} Tree data
     * @private
     */
    transformToTree(data) {
      const { parentIdField, idField, childrenField } = this.modelClass, indexById = /* @__PURE__ */ new Map(), parentIds = /* @__PURE__ */ new Set(), transformed = [];
      for (const node of data) {
        const id = node[idField];
        if (id != null) {
          indexById.set(id, node);
        }
        if (node[parentIdField] != null) {
          parentIds.add(node[parentIdField]);
        }
      }
      const cloneParent = (node) => {
        const clone = Object.assign({}, node);
        clone[childrenField] = [];
        indexById.set(clone.id, clone);
        return clone;
      };
      for (let node of data) {
        if (parentIds.has(node.id) && !node[childrenField]) {
          node = cloneParent(node);
        }
        const parentId = node[parentIdField];
        if (parentId != null) {
          let parent = indexById.get(parentId);
          if (parent) {
            if (!parent[childrenField]) {
              parent = cloneParent(parent);
            }
            parent[childrenField].push(node);
          }
        } else {
          if (node[childrenField]) {
            transformed.push(node);
          } else if (node[idField] != null) {
            transformed.push(cloneParent(node));
          } else {
            transformed.push(node);
          }
        }
      }
      return transformed;
    }
    /**
     * Transforms data into a tree with parent levels based on supplied fields.
     *
     * ```javascript
     * const newRoot = store.treeify(['name', r => r.age % 10]);
     * ```
     *
     * Generated parent records are indicated with `generatedParent` and `key` properties. The first one is set to
     * `true` and the latter one has a value for the group the parent represents.
     *
     * @param {Array<String|Function>} fields The field names, or a function to call to extract a value to create parent
     * nodes for records with the same value.
     * @param {Function} [parentTransform] A function which is called to allow the caller to transform the raw data
     * object of any newly created parent nodes.
     * @param {Boolean} [convertParents] Pass `true` to convert raw new parent data objects to this Store's
     * {@link Core.data.Store#config-modelClass}.
     * @returns {Core.data.Model} New root node
     * @internal
     */
    treeify(fields, parentTransform, convertParents = false) {
      const { length } = fields, parents = [], orphans = [], newRoot = {};
      let i, lastParent;
      for (i = 0; i < length; i++) {
        let field2 = fields[i];
        field2 = field2.field || field2;
        if (!fields[i].call) {
          fields[i] = (n) => n[field2];
          fields[i].fieldName = field2;
        }
        parents[i] = /* @__PURE__ */ new Map();
      }
      this.rootNode.traverse((n) => {
        lastParent = null;
        if (n.isLeaf) {
          for (i = 0; i < length; i++) {
            const lastParentPath = (lastParent == null ? void 0 : lastParent.path) || "", nodeMap = parents[i], key = fields[i](n);
            if (key === StopBranch) {
              break;
            }
            const path = `${lastParentPath + ((key == null ? void 0 : key.isModel) ? key.id : key)}/`, id = StringHelper.makeValidDomId(`generated_${path}`, "_"), field2 = fields[i].fieldName, parent = nodeMap.get(id) || nodeMap.set(id, {
              id,
              key,
              path,
              expanded: true,
              readOnly: true,
              children: [],
              generatedParent: true,
              field: field2,
              firstGroupChild: n
            }).get(id);
            if (lastParent && !lastParent.children.includes(parent)) {
              lastParent.children.push(parent);
            }
            lastParent = parent;
          }
          if (lastParent) {
            lastParent.children.push(n);
          } else {
            orphans.push(n);
          }
        }
      }, true);
      if (parentTransform || convertParents) {
        parents.forEach((p) => p.forEach((p2, id, map2) => {
          parentTransform == null ? void 0 : parentTransform(p2);
          if (convertParents) {
            p2 = this.createRecord(p2);
            map2.set(id, p2);
          }
        }));
      }
      newRoot.children = [...parents[0].values(), ...orphans];
      const sort = (n) => {
        if (n.children) {
          n.children.sort((lhs, rhs) => {
            if (lhs.isLeaf === rhs.isLeaf) {
              if (lhs.isLeaf) {
                let result;
                for (let i2 = 0; !result && i2 < length; i2++) {
                  const lv = fields[i2](lhs), rv = fields[i2](rhs);
                  if (lv === StopBranch || rv === StopBranch) {
                    return lhs.isLeaf < rhs.isLeaf ? -1 : lhs.isLeaf > rhs.isLeaf ? 1 : 0;
                  }
                  result = lv < rv ? -1 : rv > lv ? 1 : 0;
                }
                return result;
              } else if (typeof lhs.key === "string" && typeof rhs.key === "string") {
                return lhs.key.localeCompare(rhs.key, void 0, { numeric: true });
              } else {
                return lhs.key < rhs.key ? -1 : lhs.key > rhs.key ? 1 : 0;
              }
            } else {
              return lhs.isLeaf < rhs.isLeaf ? -1 : lhs.isLeaf > rhs.isLeaf ? 1 : 0;
            }
          });
          n.children.forEach(sort);
        }
      };
      sort(newRoot);
      return newRoot;
    }
    //endregion
    treeifyFlatData(data) {
      const { childrenField, parentIdField } = this.modelClass;
      let hasParentId = false, shouldTransform = true;
      for (const node of data) {
        if (node.isModel || Array.isArray(node[childrenField])) {
          shouldTransform = false;
          break;
        }
        if (node[parentIdField] != null) {
          hasParentId = true;
        }
      }
      if (shouldTransform && hasParentId) {
        data = this.transformToTree(data);
      }
      return data;
    }
    /**
     * Increase the indentation level of one or more nodes in the tree
     * @param {Core.data.Model|Core.data.Model[]} nodes The nodes to indent.
     * @fires indent
     * @fires change
     * @category Tree
     */
    async indent(nodes) {
      const me = this;
      nodes = Array.isArray(nodes) ? nodes : [nodes];
      nodes = nodes.filter((node) => {
        let result = Boolean(node.previousSibling);
        while (result && !node.isRoot) {
          result = !nodes.includes(node.parent);
          node = node.parent;
        }
        return result;
      });
      if (nodes.length && me.trigger("beforeIndent", { records: nodes }) !== false) {
        nodes.sort((lhs, rhs) => Wbs.compare(lhs.wbsCode, rhs.wbsCode));
        me.beginBatch();
        for (const node of nodes) {
          const newParent = node.previousSibling;
          newParent.appendChild(node);
          me.toggleCollapse(newParent, false);
        }
        me.endBatch();
        me.trigger("indent", { records: nodes });
        me.trigger("change", {
          action: "indent",
          records: nodes
        });
      }
    }
    /**
     * Decrease the indentation level of one or more nodes in the tree
     * @param {Core.data.Model|Core.data.Model[]} nodes The nodes to outdent.
     * @fires outdent
     * @fires change
     * @category Tree
     */
    async outdent(nodes) {
      const me = this;
      nodes = Array.isArray(nodes) ? nodes : [nodes];
      nodes = nodes.filter((node) => {
        const { parent } = node;
        let result = parent && !parent.isRoot;
        while (result && !node.isRoot) {
          result = !nodes.includes(parent);
          node = node.parent;
        }
        return result;
      });
      if (nodes.length && me.trigger("beforeOutdent", { records: nodes }) !== false) {
        nodes.sort((lhs, rhs) => Wbs.compare(lhs.wbsCode, rhs.wbsCode));
        me.beginBatch();
        for (const node of nodes) {
          const { parent } = node, newChildren = parent.children.slice(parent.children.indexOf(node) + 1);
          parent.parent.insertChild(node, parent.nextSibling);
          node.appendChild(newChildren);
          me.toggleCollapse(node, false);
        }
        me.endBatch();
        me.trigger("outdent", { records: nodes });
        me.trigger("change", {
          action: "outdent",
          records: nodes
        });
      }
    }
  }, __publicField(_a2, "$name", "StoreTree"), /**
   * A special `Symbol` signalizing treeify method that the current record grouping should be stopped.
   *
   * ```javascript
   * const newRoot = workerStore.treeify([
   *     // group workers by company
   *     worker => {
   *         // if the worker is unemployed we don't put it in a group
   *         // we just show such record on the root level
   *         if (!worker.company) {
   *             return Store.StopBranch
   *         }
   *
   *         return worker.company;
   *     ]
   * ]);
   * ```
   * @member {Symbol} StopBranch
   * @static
   * @category Advanced
   */
  __publicField(_a2, "StopBranch", StopBranch), __publicField(_a2, "configurable", {
    /**
     * Set to `true` to on load transform a flat dataset with raw objects containing `parentId` into the format
     * expected for tree data.
     *
     * Example input format:
     *
     * ```javascript
     * [
     *   { id : 1, name : 'Parent' },
     *   { id : 2, name : 'Child', parentId : 1 }
     * ]
     * ```
     *
     * Will be transformed into:
     *
     * ```javascript
     * [
     *   {
     *     id       : 1,
     *     name     : 'Parent',
     *     children : [
     *       { id : 2, name : 'Child', parentId : 1 }
     *     ]
     *   }
     * ]
     * ```
     *
     * @config {Boolean}
     * @category Tree
     */
    transformFlatData: null,
    /**
     * This flag prevents firing of 'remove' event when moving a node in the tree. In 6.0 this will be the default
     * behavior and this flag will be removed.
     * @config {Boolean}
     * @category Tree
     */
    fireRemoveEventForMoveAction: VersionHelper.checkVersion("core", "6.0", "<")
  }), _a2;
};

// ../Core/lib/Core/helper/WalkHelper.js
var WalkHelper = class {
  /**
   * Pre-walks any hierarchical data structure
   *
   * @param {Object} data Walking starting point
   * @param {Function} childrenFn Function to return `data` children entries in an array
   *                   or null if no children exists for the entry
   * @param {Function} fn Function to call on each entry
   */
  static preWalk(data, childrenFn, fn2) {
    const walkStack = [data];
    let node, children;
    while (walkStack.length) {
      node = walkStack.pop();
      fn2(node);
      children = childrenFn(node);
      if (children) {
        walkStack.push.apply(walkStack, children.slice().reverse());
      }
    }
  }
  /**
   * Pre-walks any hierarchical data structure, passing along a link to the parent node
   *
   * @param {Object} data Walking starting point
   * @param {Function} childrenFn Function to return `data` children entries in an array
   *                   or null if no children exists for the entry
   * @param {Function} fn Function to call on each entry, called with `parent` and `node`
   */
  static preWalkWithParent(data, childrenFn, fn2) {
    const walkStack = [{ node: data, parent: null }];
    while (walkStack.length) {
      const { parent, node } = walkStack.pop();
      fn2(parent, node);
      const children = childrenFn(node);
      if (Array.isArray(children)) {
        walkStack.push(...children.slice().reverse().map((child) => ({ node: child, parent: node })));
      }
    }
  }
  /**
   * Pre-walk unordered.
   *
   * Like {@link #function-preWalk-static} but doesn't reverse children before walk,
   * thus children will be walked last child first - first child last
   *
   * @param {Object} data Walking starting point
   * @param {Function} childrenFn Function to return `data` children entries in an array
   *                   or null if no children exists for the entry
   * @param {Function} fn Function to call on each entry
   */
  static preWalkUnordered(data, childrenFn, fn2) {
    const walkStack = [data];
    let node, children;
    while (walkStack.length) {
      node = walkStack[walkStack.length - 1];
      fn2(node);
      children = childrenFn(node);
      if (children) {
        walkStack.splice(walkStack.length - 1, 1, ...children);
      } else {
        walkStack.length = walkStack.length - 1;
      }
    }
  }
  /**
   * Post-walks any hierarchical data structure
   *
   * @param {Object} data Walking starting point
   * @param {Function} childrenFn Function to return `data` children entries in an array
   *                   or null if no children exists for the entry
   * @param {Function} fn Function to call on each entry
   */
  static postWalk(data, childrenFn, fn2) {
    const visited = /* @__PURE__ */ new Map(), walkStack = [data];
    let node, children;
    while (walkStack.length) {
      node = walkStack[walkStack.length - 1];
      if (visited.has(node)) {
        fn2(node);
        walkStack.pop();
      } else {
        children = childrenFn(node);
        if (children) {
          walkStack.push(...children.slice().reverse());
        }
        visited.set(node, node);
      }
    }
  }
  /**
   * Pre-/Post-walks any hierarchical data structure calling inFn each node when it walks in,
   * and outFn when it walks out.
   *
   * @param {Object} data Walking starting point
   * @param {Function} childrenFn Function to return `data` children entries in an array
   *                   or null if no children exists for the entry
   * @param {Function} inFn  Function to call on each entry upon enter
   * @param {Function} outFn Function to call on each entry upon exit
   */
  static prePostWalk(data, childrenFn, inFn, outFn) {
    const visited = /* @__PURE__ */ new Map(), walkStack = [data];
    let node, children;
    while (walkStack.length) {
      node = walkStack[walkStack.length - 1];
      if (visited.has(node)) {
        outFn(node);
        walkStack.pop();
      } else {
        inFn(node);
        children = childrenFn(node);
        if (children) {
          walkStack.push(...children.slice().reverse());
        }
        visited.set(node, node);
      }
    }
  }
};
WalkHelper._$name = "WalkHelper";

// ../Core/lib/Core/data/mixin/StoreSync.js
var StoreSync_default = (Target) => class StoreSync extends (Target || Base) {
  static get $name() {
    return "StoreSync";
  }
  static get configurable() {
    return {
      /**
       * Configure with `true` to sync loaded data instead of replacing existing with a new dataset.
       *
       * By default (or when configured with `false`) assigning to `store.data` replaces the entire dataset
       * with a new one, creating all new records:
       *
       * ```javascript
       * store.data = [ { id : 1, name : 'Saitama' } ];
       *
       * const first = store.first;
       *
       * store.data = [ { id : 1, name : 'One-Punch man' } ];
       *
       * // store.first !== first;
       * ```
       *
       * When configured with `true` the new dataset is instead synced against the old, figuring out what was
       * added, removed and updated:
       *
       * ```javascript
       * store.data = [ { id : 1, name : 'Saitama' } ];
       *
       * const first = store.first;
       *
       * store.data = [ { id : 1, name : 'One-Punch man' } ];
       *
       * // store.first === first;
       * ```
       *
       * After the sync, any configured sorters, groupers and filters will be reapplied.
       *
       * #### Threshold
       *
       * The sync operation has a configurable threshold, above which the operation will be treated as a
       * batch/refresh and only trigger a single `refresh` event. If threshold is not reached, individual events
       * will be triggered (single `add`, `remove` and possible multiple `update`). To enable the threshold,
       * supply a config object with a `threshold` property instead of `true`:
       *
       * ```javascript
       * const store = new Store({
       *     syncDataOnLoad : {
       *         threshold : '20%'
       *     }
       * });
       * ```
       *
       * `threshold` accepts numbers or strings. A numeric threshold means number of affected records, while a
       * string is used as a percentage of the whole dataset (appending `%` is optional). By default no threshold
       * is used.
       *
       * #### Missing fields
       *
       * The value of any field not supplied in the new dataset is by default kept as is (if record is not removed
       * by the sync). This behaviour is configurable, by setting `keepMissingValues : false` in a config object
       * it will reset any unspecified field back to their default values:
       *
       * ```javascript
       * const store = new Store({
       *     syncDataOnLoad : {
       *         keepMissingValues : false
       *     }
       * });
       * ```
       *
       * Considering the following sync operation:
       *
       * ```javascript
       * // Existing data
       * { id : 1, name : 'Saitama', powerLevel : 100 }
       * // Sync data
       * { id : 1, name : 'One-Punch Man' }
       * ```
       *
       * The result would by default (or when explicitly configured with `true`)  be:
       *
       * ```javascript
       * { id : 1, name : 'One-Punch Man', powerLevel : 100 }
       * ```
       *
       * If configured with `keepMissingValues : false` it would instead be:
       *
       * ```javascript
       * { id : 1, name : 'One-Punch Man' }
       * ```
       *
       * <div class="note">Never enable `syncDataOnLoad` on a chained store, it will create an infinite loop when
       * it is populated from the main store (the main store can use the setting)</div>
       *
       * @config {Boolean|SyncDataOnLoadOptions} syncDataOnLoad
       * @default false
       * @category Common
       */
      syncDataOnLoad: null,
      shouldSyncDataset: null,
      shouldSyncRecord: null
    };
  }
  /**
   * Syncs a new dataset against the already loaded one, only applying changes.
   * Not intended to be called directly, please configure store with `syncDataOnLoad: true` and assign to
   * `store.data` as usual instead.
   *
   * ```
   * const store = new Store({
   *    syncDataOnLoad : true,
   *    data : [
   *        // initial data
   *    ]
   * });
   *
   * store.data = [ // new data ]; //  Difference between initial data and new data will be applied
   * ```
   *
   * @param {Object[]|Core.data.Model[]} data New dataset, an array of records or data objects
   * @private
   */
  syncDataset(data) {
    var _a2;
    const me = this, { storage } = me, idsToCheck = (_a2 = me.shouldSyncDataset) == null ? void 0 : _a2.call(me, { data });
    if (idsToCheck === false) {
      return;
    }
    me.isSyncingDataOnLoad = true;
    const { toAdd, toRemove, toMove, updated, ids } = me.tree ? me.syncTreeDataset(data, idsToCheck) : me.syncFlatDataset(data, idsToCheck);
    let { threshold } = me.syncDataOnLoad, surpassed = false;
    if (threshold) {
      if (typeof threshold === "string") {
        threshold = parseInt(threshold, 10) / 100 * me.count;
      }
      surpassed = toAdd.length + toRemove.length + toMove.length + updated.length > threshold;
    }
    if (me.tree) {
      if (toAdd.length) {
        const added = me.add(toAdd, surpassed);
        added.forEach((node) => node.clearParentId());
      }
      if (toMove.length) {
        for (const { parent, node, index } of toMove) {
          const newParent = me.getById(parent.id);
          newParent.insertChild(node, index);
        }
      }
      me.remove(toRemove, surpassed);
    } else {
      if (surpassed) {
        me.suspendEvents();
      }
      me.remove(toRemove);
      me.add(toAdd);
      if (surpassed) {
        me.resumeEvents();
      }
    }
    if (threshold && !surpassed) {
      updated.forEach(({ record, toSet, wasSet }) => me.onModelChange(record, toSet, wasSet));
    }
    me.acceptChanges();
    const event = { added: toAdd, removed: toRemove, updated, thresholdSurpassed: surpassed };
    if (me.isFiltered && !me.remoteFilter) {
      me.filter({
        silent: me.isRemoteDataLoading
      });
    }
    if (me.isGrouped) {
      me.group(null, null, false, true, me.isRemoteDataLoading);
    } else if (me.isSorted) {
      if (me.remoteSort) {
        storage.replaceValues({
          values: storage.values.sort(me.createSorterFn(me.sorters)),
          silent: true
        });
      } else {
        me.sort();
      }
    } else if (!me.tree) {
      if (storage.values.some((record, index) => record.id !== ids[index])) {
        storage.replaceValues({
          values: storage.values.sort((a, b) => ids.indexOf(a.id) - ids.indexOf(b.id)),
          silent: true
        });
        !surpassed && me.afterPerformSort();
      }
    } else {
      let unsorted = false, i = 0;
      WalkHelper.preWalk(
        me.rootNode,
        (n) => Array.isArray(n.children) && !unsorted ? n.children : null,
        (node) => {
          if (node.id !== ids[i++]) {
            unsorted = true;
          }
        }
      );
      if (unsorted) {
        me.sort((a, b) => ids.indexOf(a.id) - ids.indexOf(b.id), void 0, void 0, true);
        me.clearSorters(true);
        !surpassed && me.afterPerformSort();
      }
    }
    if (surpassed) {
      me.trigger("refresh", {
        action: "batch",
        data,
        records: storage.values,
        syncInfo: event
      });
    }
    me.isSyncingDataOnLoad = false;
    me.trigger("loadSync", event);
  }
  // Used by syncDataset()
  syncFlatDataset(data, idsToCheck) {
    if (!data) {
      return {
        toRemove: this.records
      };
    }
    const me = this, { idField, allFields } = me.modelClass, toRemove = [], toAdd = [], updated = [], usedIds = {}, ids = [], limitedSet = Array.isArray(idsToCheck);
    const { threshold, keepMissingValues } = me.syncDataOnLoad;
    let hitCount = 0;
    data.forEach((rawData) => {
      var _a2;
      rawData = rawData.isModel ? rawData.data : rawData;
      const id = rawData[idField], record = me.getById(id);
      if (!limitedSet || idsToCheck.includes(id)) {
        if (record) {
          if (((_a2 = me.shouldSyncRecord) == null ? void 0 : _a2.call(me, { record, data: rawData })) !== false) {
            if (keepMissingValues === false) {
              for (const field2 of allFields) {
                if (!(field2.dataSource in rawData) && field2.dataSource in record.data) {
                  rawData[field2.dataSource] = field2.defaultValue;
                }
              }
            }
            const wasSet = record.set(rawData, null, Boolean(threshold));
            if (wasSet) {
              updated.push({
                record,
                wasSet,
                toSet: rawData
              });
            }
          }
        } else {
          toAdd.push(me.processRecord(me.createRecord(rawData)));
        }
      }
      if (record) {
        hitCount++;
      }
      usedIds[id] = 1;
      ids.push(id);
    });
    if (hitCount < me.storage.totalCount) {
      if (idsToCheck) {
        for (const id of idsToCheck) {
          if (!usedIds[id]) {
            toRemove.push(me.getById(id));
          }
        }
      } else {
        me.forEach((record) => {
          if (!usedIds[record.id]) {
            toRemove.push(record);
          }
        });
      }
    }
    return { toAdd, toRemove, toMove: [], updated, ids };
  }
  // Used by syncDataset()
  syncTreeDataset(data) {
    if (!data) {
      return {
        toRemove: this.records
      };
    }
    const me = this, {
      idField,
      parentIdField,
      childrenField,
      allFields
    } = me.modelClass, {
      keepMissingValues,
      threshold
    } = me.syncDataOnLoad, toRemove = [], toAdd = [], toMove = [], updated = [], matchedNodes = /* @__PURE__ */ new Set(), ids = [];
    if (me.transformFlatData) {
      data = me.treeifyFlatData(data);
    }
    WalkHelper.preWalkWithParent({ isRoot: true, id: me.rootNode.id, children: data }, (n) => n.children, (parent, rawData) => {
      var _a2;
      if (parent) {
        const { id, node } = me.resolveSyncNode(rawData);
        if (node) {
          if (((_a2 = me.shouldSyncRecord) == null ? void 0 : _a2.call(me, { record: node, data: rawData })) !== false) {
            let childrenUpdated;
            const oldChildrenValue = node.children;
            if (oldChildrenValue !== true && rawData[childrenField] === true) {
              node.clearChildren();
              node.data[childrenField] = node.children = true;
              delete rawData[childrenField];
              me.toggleCollapse(node, true);
              childrenUpdated = true;
            }
            if (node.parent.id !== parent[idField]) {
              toMove.push({
                node,
                parent,
                index: parent[childrenField].indexOf(rawData)
              });
            }
            if (keepMissingValues === false) {
              for (const field2 of allFields) {
                if (field2.name !== "parentId" && !(field2.dataSource in rawData) && field2.dataSource in node.data) {
                  rawData[field2.dataSource] = field2.defaultValue;
                }
              }
            }
            const wasSet = node.set(rawData, null, Boolean(threshold));
            if (wasSet) {
              updated.push({
                record: node,
                wasSet,
                toSet: rawData
              });
            } else if (childrenUpdated) {
              node.signalNodeChanged({
                [childrenField]: {
                  value: true,
                  oldValue: oldChildrenValue
                }
              });
            }
          }
        } else {
          rawData[parentIdField] = parent[idField];
          toAdd.push({ ...rawData, ...Array.isArray(rawData[childrenField]) ? { children: [] } : void 0 });
        }
        matchedNodes.add(node);
        ids.push(id);
      }
    });
    if (matchedNodes.length !== data.length) {
      me.traverse((node) => {
        if (!matchedNodes.has(node)) {
          toRemove.push(node);
        }
      });
    }
    return { toAdd, toRemove, toMove, updated, ids };
  }
  // ColumnStore overrides this fn to allow syncing by field & type
  resolveSyncNode(rawData) {
    const id = rawData[this.modelClass.idField], node = this.getById(id);
    return { id, node };
  }
};

// ../Core/lib/Core/data/stm/mixin/StoreStm.js
var STM_PROP2 = Symbol("STM_PROP");
var StoreStm_default = (Target) => class StoreStm extends (Target || Base) {
  static get $name() {
    return "StoreStm";
  }
  static get defaultConfig() {
    return {
      /**
       * Reference to STM manager
       *
       * @config {Core.data.stm.StateTrackingManager}
       * @default
       * @category Advanced
       */
      stm: null,
      /**
       * Set to `false` to not record transaction during `applyChangeset` call
       *
       * @prp {Boolean}
       * @default
       * @category Advanced
       */
      ignoreRemoteChangesInSTM: false
    };
  }
  get stm() {
    return this[STM_PROP2];
  }
  set stm(stm) {
    var _a2;
    const me = this;
    if (me.stm !== stm) {
      if ((_a2 = me.stm) == null ? void 0 : _a2.hasStore(me)) {
        me.stm.removeStore(me);
      }
      me[STM_PROP2] = stm;
      if (me.stm && !me.stm.hasStore(me)) {
        me.stm.addStore(me);
      }
    }
  }
  // Overridden to notify STM about flat add action
  add(records, silent = false, options = {}) {
    let result;
    const { stm } = this;
    if (!this.tree && (stm == null ? void 0 : stm.enabled)) {
      result = super.add(records, silent);
      if (result == null ? void 0 : result.length) {
        stm.onStoreModelAdd(this, result, silent);
      }
    } else {
      result = super.add(records, silent, options);
    }
    return result;
  }
  // Overridden to notify STM about flat insert action
  insert(index, records, silent = false) {
    let result;
    const { stm } = this;
    if (!this.tree && (stm == null ? void 0 : stm.enabled)) {
      const context = (Array.isArray(records) ? records : [records]).reduce(
        (context2, r) => {
          const index2 = r instanceof Model ? this.indexOf(r) : void 0;
          if (index2 !== void 0 && index2 !== -1) {
            context2.set(r, index2);
          }
          return context2;
        },
        /* @__PURE__ */ new Map()
      );
      result = super.insert(index, records);
      if (result == null ? void 0 : result.length) {
        index = this.indexOf(result[0]);
        stm.onStoreModelInsert(this, index, result, context, silent);
      }
    } else {
      result = super.insert(index, records, silent);
    }
    return result;
  }
  // Overridden to notify STM about flat removing action
  remove(recordsOrIds, silent = false, fromRemoveChild) {
    let result;
    const { stm } = this;
    if (!this.tree && (stm == null ? void 0 : stm.enabled)) {
      const recordsOrIdsNormalized = (Array.isArray(recordsOrIds) ? recordsOrIds : [recordsOrIds]).map((r) => this.getById(r)).filter((r) => !!r);
      const context = recordsOrIdsNormalized.reduce(
        (context2, r) => {
          const index = this.indexOf(r);
          if (index !== void 0 && index !== -1) {
            context2.set(r, index);
          }
          return context2;
        },
        /* @__PURE__ */ new Map()
      );
      result = super.remove(recordsOrIds, silent, fromRemoveChild);
      if (result == null ? void 0 : result.length) {
        stm.onStoreModelRemove(this, result, context, silent);
      }
    } else {
      result = super.remove(recordsOrIds, silent, fromRemoveChild);
    }
    return result;
  }
  // Overridden to notify STM about flat clear action
  removeAll(silent) {
    const { stm } = this;
    let result;
    if (stm == null ? void 0 : stm.enabled) {
      const { tree, rootNode, allRecords } = this, wasNotEmpty = allRecords.length, records = tree ? rootNode.children.slice() : allRecords.slice();
      result = super.removeAll(silent);
      if (wasNotEmpty && this.count === 0) {
        stm.onStoreRemoveAll(this, records, silent);
      }
    } else {
      result = super.removeAll(silent);
    }
    return result;
  }
  beforeApplyChangeset() {
    const { stm, crudManager } = this;
    let shouldResume = false, transactionId = null;
    if (!(crudManager == null ? void 0 : crudManager.applyingChangeset) && (stm == null ? void 0 : stm.enabled)) {
      shouldResume = true;
      if (stm.isRecording) {
        transactionId = stm.stash();
      }
      if (this.ignoreRemoteChangesInSTM) {
        stm.disable();
      } else {
        stm.startTransaction();
      }
    }
    return { shouldResume, transactionId };
  }
  // When applying changes while STM is in the recording state, first we need to discard local changes, then apply
  // changes from remote, then try to apply local changes. This would emulate starting transaction as if data was
  // already in the correct state
  applyChangeset(changes, transformFn, phantomIdField, remote, logChanges) {
    const {
      shouldResume,
      transactionId
    } = this.beforeApplyChangeset(), log = super.applyChangeset(changes, transformFn, phantomIdField, remote, logChanges);
    this.afterApplyChangeset(shouldResume, transactionId);
    return log;
  }
  afterApplyChangeset(shouldResume, transactionId) {
    if (shouldResume) {
      const { stm } = this;
      if (this.ignoreRemoteChangesInSTM) {
        stm.enable();
      } else {
        stm.stopTransaction();
      }
      stm.applyStash(transactionId);
    }
  }
};

// ../Core/lib/Core/data/Store.js
var dataAddRemoveActions = {
  splice: 1,
  clear: 1
};
var defaultTraverseOptions2 = {
  includeFilteredOutRecords: false,
  includeCollapsedGroupRecords: false
};
var fixTraverseOptions2 = (store, options) => {
  options = options || false;
  if (typeof options === "boolean") {
    options = {
      includeFilteredOutRecords: options,
      includeCollapsedGroupRecords: false
    };
  }
  return options || defaultTraverseOptions2;
};
var Store = class extends Base.mixin(
  Delayable_default,
  Identifiable_default,
  Events_default,
  Pluggable_default,
  State_default,
  StoreFilter_default,
  StoreChanges_default,
  StoreCRUD_default,
  StoreRelation_default,
  // Private
  StoreSum_default,
  StoreSearch_default,
  StoreSort_default,
  StoreGroup_default,
  StoreChained_default,
  StoreState_default,
  StoreTree_default,
  StoreStm_default,
  StoreSync_default,
  StoreProxy_default
  // Private for now, thus not mentioned in @mixes block above
) {
  //region Config & properties
  static get $name() {
    return "Store";
  }
  static get properties() {
    return {
      relationCache: {},
      dependentStoreConfigs: /* @__PURE__ */ new Map()
    };
  }
  static get configurable() {
    return {
      /**
       * Store's unique identifier.
       *
       * @member {String|Number} id
       * @readonly
       * @category Common
       */
      /**
       * Store's unique identifier. When set the store is added to a store map accessible through
       * `Store.getStore(id)`.
       *
       * @config {String|Number}
       * @category Common
       */
      id: true,
      /**
       * Class used to represent records in the store, should be a subclass of {@link Core.data.Model}. Only
       * applies when supplying data to the store (load, add), any supplied record instances are kept as is.
       *
       * ```javascript
       * class MyModel extends Model {
       *     static get fields() {
       *         return [
       *             'name',
       *             'city',
       *             'company'
       *         ]
       *     }
       * }
       *
       * const store = new Store({
       *     modelClass : MyModel,
       *     data : [
       *         { id : 1, name : 'Mark', city : 'London', company : 'Cool inc' },
       *         ...
       *     ]
       * });
       * ```
       *
       * @config {Core.data.Model}
       * @default
       * @typings {typeof Model}
       * @category Common
       */
      modelClass: Model,
      /**
       * Verify that loaded data does not contain any generated ids. If it does, a warning is logged on console.
       *
       * Set this to `false` to disable the check and give a very minor performance boost.
       *
       * @prp {Boolean}
       * @default
       */
      verifyNoGeneratedIds: true
    };
  }
  static get defaultConfig() {
    return {
      /**
       * An array of field definitions used to create a {@link Core.data.Model} (modelClass) subclass. Optional.
       * If the Model already has fields defined, these fields will extend those.
       *
       * ```javascript
       * const store = new Store({
       *     fields : ['name', 'city', 'company'],
       *     data   : [
       *         { id : 1, name : 'Mark', city : 'London', company : 'Cool inc' },
       *         ...
       *     ]
       * });
       * ```
       *
       * See {@link Core.data.Model} for more info on defining fields, changing data source and mapping fields to
       * nested objects.
       *
       * Note that pre-created record instances supplied to the store are kept as is and thus these fields will
       * not apply to them.
       *
       * @config {Array<String|ModelFieldConfig|Core.data.field.DataField>}
       * @category Common
       */
      fields: null,
      /**
       * Automatically detect from set data if used as tree store or flat store
       * @config {Boolean}
       * @default
       * @category Tree
       */
      autoTree: true,
      /**
       * Raw data to load initially.
       *
       * Expects an array of JavaScript objects, with properties matching store's fields (defined on its
       * {@link #config-modelClass model} or in the {@link #config-fields} config).
       *
       * ```javascript
       * const store = new Store({
       *     data : [
       *         { id : 1, name : 'Linda', city : 'NY' },
       *         { id : 2, name : 'Olivia', city : 'Paris' },
       *         ...
       *     ]
       * });
       * ```
       *
       * @config {Object[]|Core.data.Model[]}
       * @category Common
       */
      data: null,
      /**
       * `true` to act as a tree store.
       * @config {Boolean}
       * @category Tree
       */
      tree: false,
      callOnFunctions: true,
      /**
       * A {@link Core/util/Collection}, or Collection config object
       * to use to contain this Store's constituent records.
       * @config {Core.util.Collection|CollectionConfig}
       * @category Advanced
       */
      storage: null,
      /**
       * Retools the loaded data objects instead of making shallow copies of them. This increases performance but
       * pollutes the incoming data and does not allow remapping of fields (dataSource).
       *
       * Also allows disabling certain steps in data loading, to further improve performance. Either accepts an
       * object with the params described below or `true` which equals `disableDuplicateIdCheck` and
       * `disableTypeConversion`.
       *
       * ```javascript
       * // No duplicate id checking, no type conversions
       * new Store({ useRawData : true });
       *
       * new Store({
       *   // No type conversions only
       *   useRawData : {
       *     disableTypeConversion : true
       *   }
       * });
       * ```
       *
       * @config {Boolean|Object}
       * @param {Boolean} [disableDuplicateIdCheck] Data must not contain duplicate ids, check is bypassed.
       * @param {Boolean} [disableDefaultValue] Default values will not be applied to record fields.
       * @param {Boolean} [disableTypeConversion] No type conversions will be performed on record data.
       * @category Advanced
       */
      useRawData: false,
      /**
       * Specify `false` to prevent loading records without ids, a good practise to enforce when syncing with a
       * backend.
       *
       * By default Store allows loading records without ids, in which case a generated id will be assigned.
       *
       * @config {Boolean}
       * @default true
       * @category Advanced
       */
      allowNoId: true,
      /**
       * Prevent dynamically subclassing the modelClass. It does so by default to not pollute it when exposing
       * properties. Should rarely need to be used.
       * @config {Boolean}
       * @default false
       * @private
       * @category Advanced
       */
      preventSubClassingModel: null
    };
  }
  static get identifiable() {
    return {
      registerGeneratedId: false
    };
  }
  /**
   * Class used to represent records. Defaults to class Model.
   * @member {Core.data.Model} modelClass
   * @typings {typeof Model}
   * @category Records
   */
  //endregion
  //region Events
  /**
   * Fired when the id of a record has changed
   * @event idChange
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model} record Modified record
   * @param {String|Number} oldValue Old id
   * @param {String|Number} value New id
   */
  /**
   * Fired before record is modified in this store.
   * Modification may be vetoed by returning `false` from a handler.
   * @event beforeUpdate
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model} record Modified record
   * @param {Object} changes Modification data
   */
  /**
   * Fired when a record is modified
   * @event update
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model} record Modified record
   * @param {Object} changes Modification data
   */
  /**
   * Fired when one of this Store's constituent records is modified while in
   * {@link Core.data.Model#function-beginBatch batched} state. This may be used to keep
   * UIs up to date while "tentative" changes are made to a record which must not be synced with a server.
   * @event batchedUpdate
   * @private
   */
  /**
   * Fired when the root node is set
   * @event rootChange
   * @param {Core.data.Store} source This Store
   * @param {Core.data.Model} oldRoot The old root node.
   * @param {Core.data.Model} rootNode The new root node.
   */
  /**
   * Data in the store was changed. This is a catch-all event which is fired for all changes
   * which take place to the store's data.
   *
   * This includes mutation of individual records, adding and removal of records, as well as
   * setting a new data payload using the {@link #property-data} property, sorting, filtering,
   * and calling {@link Core.data.mixin.StoreCRUD#function-removeAll}.
   *
   * Simple databound widgets may use to the `change` event to refresh their UI without having to add multiple
   * listeners to the {@link #event-update}, {@link Core.data.mixin.StoreCRUD#event-add},
   * {@link Core.data.mixin.StoreCRUD#event-remove}, {@link #event-refresh} and
   * {@link Core.data.mixin.StoreCRUD#event-removeAll} events.
   *
   * A more complex databound widget such as a grid may use the more granular events to perform less
   * destructive updates more appropriate to each type of change. The properties will depend upon the value of the
   * `action` property.
   *
   * @event change
   * @param {Core.data.Store} source This Store.
   * @param {'remove'|'removeAll'|'add'|'updatemultiple'|'clearchanges'|'filter'|'update'|'dataset'|'replace'} action
   * Name of action which triggered the change. May be one of the options listed above
   * @param {Core.data.Model} record Changed record, for actions that affects exactly one record (`'update'`)
   * @param {Core.data.Model[]} records Changed records, passed for all actions except `'removeAll'`
   * @param {Object} changes Passed for the `'update'` action, info on which record fields changed
   */
  // NOTE: When updating params above, also update change event in ProjectModelMixin and dataChange in ProjectConsumer
  /**
   * Data in the store has completely changed, such as by a filter, or sort or load operation.
   * @event refresh
   * @param {Core.data.Store} source This Store.
   * @param {Boolean} batch Flag set to `true` when the refresh is triggered by ending a batch
   * @param {'dataset'|'sort'|'clearchanges'|'filter'|'create'|'update'|'delete'|'group'} action Name of
   * action which triggered the change. May be one of the options listed above.
   */
  //endregion
  /* break doc comment from next method */
  //region Init
  constructor(...args) {
    super(...args);
    if (this.objectify) {
      return this.initProxy();
    }
  }
  construct(config = {}) {
    const me = this;
    Object.assign(me, {
      added: new StoreBag(),
      removed: new StoreBag(),
      modified: new StoreBag(),
      idRegister: {},
      internalIdRegister: {},
      oldIdMap: {}
    });
    if (config.storeId) {
      config = ObjectHelper.assign({ id: config.storeId }, config);
    }
    super.construct(config);
    me.initRelations();
  }
  /**
   * Retrieves/creates a store based on the passed config.
   *
   * | Type              | Result                                                                 |
   * |-------------------|------------------------------------------------------------------------|
   * | Core.data.Store   | Returns supplied store as is                                           |
   * | String            | Retrieves an existing store by id                                      |
   * | Object            | Creates a new store using supplied config object                       |
   * | Object[]          | Creates a new store, populated with records created from supplied data |
   * | Core.data.Model[] | Creates a new store, populated with supplied records                   |
   *
   *
   * @param {Core.data.Store|StoreConfig|String|StoreConfig[]|Core.data.Model[]} config
   * @param {Object} [defaults] Config object to apply when creating a new store for passed data
   * @param {Function} [converterFn] Function called for each data object prior to creating a record from it. The
   * return value is used to create a record.
   * @private
   */
  static from(config, defaults = {}, converterFn = null) {
    if (config && !config.isStore) {
      if (typeof config === "string") {
        config = Store.getStore(config);
      } else {
        if (Array.isArray(config)) {
          if (converterFn) {
            config = config.map((data) => data.isModel ? data : converterFn(data));
          }
          config = ObjectHelper.assign({}, defaults, { data: config });
        }
        config = new Store(config);
      }
    }
    return config;
  }
  doDestroy() {
    var _a2, _b, _c, _d;
    const me = this, allRecords = me.registeredRecords;
    (_b = (_a2 = me.stm) == null ? void 0 : _a2.removeStore) == null ? void 0 : _b.call(_a2, me);
    for (let i = allRecords.length - 1, rec; i >= 0; i--) {
      rec = allRecords[i];
      if (!(rec == null ? void 0 : rec.isDestroyed)) {
        rec.unjoinStore(me);
      }
    }
    (_c = me._storage) == null ? void 0 : _c.destroy();
    if (!me.isChained) {
      (_d = me.rootNode) == null ? void 0 : _d.destroy();
    }
    super.doDestroy();
  }
  /**
   * Stops this store from firing events until {@link #function-endBatch} is called. Multiple calls to `beginBatch`
   * stack up, and will require an equal number of `endBatch` calls to resume events.
   *
   * Upon call of {@link #function-endBatch}, a {@link #event-refresh} event is triggered to allow UIs to
   * update themselves based upon the new state of the store.
   *
   * This is extremely useful when making a large number of changes to a store. It is important not to trigger
   * too many UI updates for performance reasons. Batching the changes ensures that UIs attached to this
   * store are only updated once at the end of the updates.
   */
  beginBatch() {
    this.suspendEvents();
  }
  /**
   * Ends event suspension started by {@link #function-beginBatch}. Multiple calls to {@link #function-beginBatch}
   * stack up, and will require an equal number of `endBatch` calls to resume events.
   *
   * Upon call of `endBatch`, a {@link #event-refresh} event with `action: batch` is triggered to allow UIs to update
   * themselves based upon the new state of the store.
   *
   * This is extremely useful when making a large number of changes to a store. It is important not to trigger
   * too many UI updates for performance reasons. Batching the changes ensures that UIs attached to this
   * store are only updated once at the end of the updates.
   */
  endBatch() {
    if (this.resumeEvents()) {
      this.trigger("refresh", {
        action: "batch",
        data: this.storage.values,
        records: this.storage.values
      });
    }
  }
  set storage(storage) {
    const me = this;
    if (storage == null ? void 0 : storage.isCollection) {
      me._storage = storage;
    } else {
      me._storage = new Collection(storage);
    }
    me._storage.autoFilter = me.reapplyFilterOnAdd;
    me._storage.autoSort = me.reapplySortersOnAdd;
    for (const r of me._storage) {
      r.joinStore(me);
    }
    me._storage.ion({
      change: "onDataChange",
      thisObj: me
    });
  }
  get storage() {
    if (!this._storage) {
      this.storage = {};
    }
    return this._storage;
  }
  /**
   * Returns all records (ignoring any filters) from the store.
   * @property {Core.data.Model[]}
   * @readonly
   * @category Records
   */
  get allRecords() {
    var _a2;
    const me = this;
    if (((_a2 = me._allRecords) == null ? void 0 : _a2.generation) !== me.storage.generation) {
      if (me.isTree) {
        const result = me.collectDescendants(me.rootNode, void 0, void 0, { unfiltered: true }).all;
        if (me.rootVisible) {
          result.unshift(me.rootNode);
        }
        me._allRecords = result;
      } else {
        me._allRecords = me.isGrouped ? me.collectGroupRecords() : me.storage.allValues;
      }
      me._allRecords.generation = me.storage.generation;
    }
    return me._allRecords;
  }
  // All records except special rows such group headers etc
  getAllDataRecords(searchAllRecords) {
    const me = this;
    if (me.tree) {
      return searchAllRecords ? me.allRecords : me.rootNode.allChildren;
    }
    return me.isGrouped ? me.collectGroupRecords(searchAllRecords, false) : searchAllRecords ? me.storage.allValues : me.storage.values;
  }
  /**
   * Called by owned record when the record has its {@link Core.data.Model#property-isCreating}
   * property toggled.
   * @param {Core.data.Model} record The record that is being changed.
   * @param {Boolean} isCreating The new value of the {@link Core.data.Model#property-isCreating} property.
   * @internal
   */
  onIsCreatingToggle(record, isCreating) {
    const me = this, newlyPersistable = record.isPersistable && !isCreating;
    me.added[newlyPersistable ? "add" : "remove"](record);
    if (newlyPersistable) {
      me.trigger("addConfirmed", { record });
      if (me.autoCommit) {
        me.doAutoCommit();
      }
    }
  }
  // Join added records to store, not called when loading
  joinRecordsToStore(records) {
    const { allCount } = this;
    for (let i = 0; i < records.length; i++) {
      const record = records[i];
      record.setData("parentIndex", allCount + i - records.length);
      record.joinStore(this);
    }
  }
  /**
   * Responds to mutations of the underlying storage Collection
   * @param {Object} event
   * @protected
   */
  onDataChange({ source: storage, action, added, removed, replaced, oldCount, items: items2, from, to }) {
    const me = this, isAddRemove = dataAddRemoveActions[action], addedCount = isAddRemove && (added == null ? void 0 : added.length), removedCount = isAddRemove && (removed == null ? void 0 : removed.length);
    let filtersWereReapplied, sortersWereReapplied;
    me._idMap = null;
    if (addedCount) {
      me.joinRecordsToStore(added);
    }
    replaced == null ? void 0 : replaced.forEach(([oldRecord, newRecord]) => {
      oldRecord.unjoinStore(me, true);
      newRecord.joinStore(me);
    });
    super.onDataChange(...arguments);
    if (!me.isTree) {
      if (addedCount) {
        for (const record of added) {
          if (me.removed.includes(record)) {
            me.removed.remove(record);
          } else if (!record.isLinked) {
            me.added.add(record);
          }
        }
        filtersWereReapplied = !me.remoteFilter && me.isFiltered && me.reapplyFilterOnAdd;
        if (filtersWereReapplied) {
          me.filter({
            silent: true
          });
        }
        sortersWereReapplied = !me.remoteSort && me.isSorted && me.reapplySortersOnAdd;
        if (sortersWereReapplied) {
          me.sort(null, null, false, true);
        }
      }
      if (removedCount) {
        for (const record of removed) {
          record.cancelBatch();
          record.unjoinStore(me);
          if (me.added.includes(record)) {
            me.added.remove(record);
          } else if (!record._undoingInsertion && !record.isCreating && !record.isLinked) {
            me.removed.add(record);
          }
        }
        me.modified.remove(removed);
        filtersWereReapplied = !me.remoteFilter && me.isFiltered;
        if (filtersWereReapplied) {
          me.filter({
            silent: true
          });
        }
      }
    }
    switch (action) {
      case "clear":
        me.relationCache = {};
        me.updateDependentStores("removeall");
        me.trigger("removeAll");
        me.trigger("change", {
          action: "removeall"
        });
        break;
      case "splice":
        if (addedCount) {
          me.updateDependentStores("add", added);
          const oldIndex = added.reduce((lowest, record) => {
            const { previousIndex } = record.meta;
            if (previousIndex > -1 && previousIndex < lowest)
              lowest = previousIndex;
            return lowest;
          }, added[0].meta.previousIndex), index = storage.indexOf(added[0], !storage.autoFilter), params = {
            records: added,
            index
          };
          if (oldIndex > -1) {
            params.oldIndex = oldIndex;
          }
          me.trigger("add", params);
          me.trigger("change", Object.assign({ action: "add" }, params));
          if (filtersWereReapplied) {
            me.triggerFilterEvent({
              action: "filter",
              filters: me.filters,
              oldCount,
              records: me.storage.allValues
            });
          }
          if (sortersWereReapplied) {
            me.trigger("sort", { action: "sort", sorters: me.sorters, records: me.storage.allValues });
          }
        }
        if (removed.length) {
          me.updateDependentStores("remove", removed);
          me.trigger("remove", {
            records: removed
          });
          me.trigger("change", {
            action: "remove",
            records: removed
          });
        }
        if (replaced.length) {
          me.trigger("replace", {
            records: replaced,
            all: me.records.length === replaced.length
          });
          me.trigger("change", {
            action: "replace",
            replaced,
            all: me.records.length === replaced.length
          });
        }
        break;
      case "filter":
        if (me.isGrouped || me.isSorted) {
          me.performSort(true);
        }
        break;
      case "move": {
        const start = Math.min(from, to), end = Math.min(me.storage.allValues.length - 1, Math.max(from, to));
        for (let allRecords = me.storage.allValues, i = start; i <= end; i++) {
          allRecords[i].setData("parentIndex", i);
        }
        me.trigger("move", {
          record: items2[0],
          records: items2,
          from,
          to
        });
        if (me.isFiltered) {
          me.performFilter();
        }
        me.trigger("change", {
          action,
          record: items2[0],
          records: items2,
          from,
          to
        });
        break;
      }
    }
  }
  onDataReplaced(action, data) {
    var _a2;
    const me = this, { storage } = me, all = storage.allValues, sorted = Boolean(me.sorters.length > 0);
    for (let i = 0; i < all.length; i++) {
      all[i].joinStore(me);
    }
    if (!me.remoteFilter && me.isFiltered) {
      me.filter({
        silent: true
      });
    }
    if (me.remoteSort) {
      if (me.isGrouped) {
        storage.replaceValues({
          // Need to update group records info (headers and footers)
          ...me.prepareGroupRecords(),
          silent: true
        });
      }
    } else {
      if (me.isGrouped) {
        me.group(null, null, false, !sorted, true);
      }
      if (sorted) {
        me.sort(null, null, false, true);
      }
    }
    if (!me.useRawData.disableDuplicateIdCheck) {
      const { idMap } = me;
      if (Object.keys(idMap).length < storage.values.length) {
        const collisions = [];
        storage.values.forEach((r) => idMap[r.id] ? delete idMap[r.id] : collisions.push(r));
        throw new Error(`Id collision on ${collisions.map((r) => r.id)}`);
      }
    }
    const event = { action, data, records: storage.values };
    me.updateDependentStores(action, event.records);
    (_a2 = me.afterLoadData) == null ? void 0 : _a2.call(me);
    if (!me.isRemoteDataLoading) {
      me.trigger("refresh", event);
    }
    me.trigger("change", event);
  }
  /**
   * This is called from Model after mutating any fields so that Stores can take any actions necessary at that point,
   * and distribute mutation event information through events.
   * @param {Core.data.Model} record The record which has just changed
   * @param {Object} toSet A map of the field names and values that were passed to be set
   * @param {Object} wasSet A map of the fields that were set. Each property is a field name, and
   * the property value is an object containing two properties: `oldValue` and `value` eg:
   * ```javascript
   *     {
   *         name {
   *             oldValue : 'Rigel',
   *             value : 'Nigel'
   *         }
   *     }
   *
   * @param {Boolean} silent Do not trigger events
   * @param {Boolean} fromRelationUpdate Update caused by a change in related model
   * @private
   */
  onModelChange(record, toSet, wasSet, silent, fromRelationUpdate) {
    const me = this, event = {
      record,
      records: [record],
      changes: wasSet,
      // Cannot use isBatching, since change is triggered when batching has reached 0
      // (but before it is set to null)
      batch: record.batching != null,
      fromRelationUpdate
    }, committable = record.ignoreBag || record.isLinked ? false : me.updateModifiedBagForRecord(record);
    me.storage.onItemMutation(record, wasSet);
    if ("id" in wasSet) {
      const { oldValue, value } = toSet.id;
      me.updateDependentRecordIds(oldValue, value);
      me.onRecordIdChange({ record, oldValue, value });
    }
    if (!silent) {
      if ("id" in wasSet) {
        const { oldValue, value } = toSet.id;
        me.trigger("idChange", {
          store: me,
          record,
          oldValue,
          value
        });
      }
      me.onUpdateRecord(record, wasSet);
      me.trigger("update", event);
      me.trigger("change", Object.assign({ action: "update" }, event));
    }
    if (me.autoCommit && committable) {
      me.doAutoCommit();
    }
  }
  updateModifiedBagForRecord(record) {
    const me = this;
    let addedToBag = false;
    if (record.isModified) {
      if (!me.modified.includes(record) && !me.added.includes(record) && record.isPartOfStore(me) && !record.isAutoRoot) {
        if (record.isPhantom) {
          me.added.add(record);
        } else {
          me.modified.add(record);
        }
        addedToBag = true;
      }
    } else {
      me.modified.remove(record);
    }
    return addedToBag;
  }
  get idMap() {
    const me = this, needsRebuild = !me._idMap, idMap = me._idMap || (me._idMap = {});
    if (needsRebuild) {
      const processedRecords = me.storage.values;
      for (let record, index = 0, visibleIndex = 0; index < processedRecords.length; index++) {
        record = processedRecords[index];
        idMap[record.id] = { index, visibleIndex, record };
        if (!record.isSpecialRow) {
          visibleIndex++;
        }
      }
      if (me.isFiltered) {
        for (let index = 0, l = me.storage._values.length; index < l; index++) {
          const record = me.storage._values[index];
          if (record.id in idMap) {
            idMap[record.id].unfilteredIndex = index;
          } else {
            idMap[record.id] = { index: -1, unfilteredIndex: index, record };
          }
        }
      }
    }
    return idMap;
  }
  changeModelClass(ClassDef) {
    const { fields } = this;
    this.originalModelClass = ClassDef;
    let ClassDefEx = ClassDef;
    if (fields == null ? void 0 : fields.length) {
      class ModelClass extends ClassDef {
        static get fields() {
          return fields;
        }
      }
      ClassDefEx = ModelClass;
    } else if (!this.preventSubClassingModel) {
      class ModelClass extends ClassDef {
      }
      ClassDefEx = ModelClass;
    }
    ClassDefEx.initClass();
    return ClassDefEx;
  }
  //endregion
  //region Store id & map
  set storeId(storeId) {
    this.id = storeId;
  }
  get storeId() {
    return this.id;
  }
  changeId(id, oldId) {
    return super.changeId(id !== true && id, oldId);
  }
  updateId(id, oldId) {
    const duplicate = Store.getById(id);
    duplicate && Store.unregisterInst