/*!
 *
 * Bryntum Gantt 5.0.1
 *
 * Copyright(c) 2022 Bryntum AB
 * https://bryntum.com/contact
 * https://bryntum.com/license
 *
 */
const productName = 'gantt'

function ownKeys(object, enumerableOnly) {
  var keys = Object.keys(object);
  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    enumerableOnly && (symbols = symbols.filter(function (sym) {
      return Object.getOwnPropertyDescriptor(object, sym).enumerable;
    })), keys.push.apply(keys, symbols);
  }
  return keys;
}
function _objectSpread2(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = null != arguments[i] ? arguments[i] : {};
    i % 2 ? ownKeys(Object(source), !0).forEach(function (key) {
      _defineProperty(target, key, source[key]);
    }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) {
      Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
    });
  }
  return target;
}
function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }
  return obj;
}
function _objectWithoutPropertiesLoose(source, excluded) {
  if (source == null) return {};
  var target = {};
  var sourceKeys = Object.keys(source);
  var key, i;
  for (i = 0; i < sourceKeys.length; i++) {
    key = sourceKeys[i];
    if (excluded.indexOf(key) >= 0) continue;
    target[key] = source[key];
  }
  return target;
}
function _objectWithoutProperties(source, excluded) {
  if (source == null) return {};
  var target = _objectWithoutPropertiesLoose(source, excluded);
  var key, i;
  if (Object.getOwnPropertySymbols) {
    var sourceSymbolKeys = Object.getOwnPropertySymbols(source);
    for (i = 0; i < sourceSymbolKeys.length; i++) {
      key = sourceSymbolKeys[i];
      if (excluded.indexOf(key) >= 0) continue;
      if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue;
      target[key] = source[key];
    }
  }
  return target;
}
function _toPrimitive(input, hint) {
  if (typeof input !== "object" || input === null) return input;
  var prim = input[Symbol.toPrimitive];
  if (prim !== undefined) {
    var res = prim.call(input, hint || "default");
    if (typeof res !== "object") return res;
    throw new TypeError("@@toPrimitive must return a primitive value.");
  }
  return (hint === "string" ? String : Number)(input);
}
function _toPropertyKey(arg) {
  var key = _toPrimitive(arg, "string");
  return typeof key === "symbol" ? key : String(key);
}

/**
 * @module Core/helper/StringHelper
 */

let charsToEncode, entitiesToDecode, htmlEncodeRe, htmlDecodeRe;

const camelLettersRe = /([a-z])([A-Z])/g,
      crlfRe = /[\n\r]/g,
      escapeRegExpRe = /[.*+?^${}()|[\]\\]/g,
      // same as NPM escape-string-regexp
htmlRe$1 = /[&<]/,
      idRe = /(^[^a-z]+[^\w]+)/gi,
      whiteSpaceRe$1 = /\s+/,
      domIdRe = /^[^a-z]+|[^\w:.-]+/gi,
      htmlDecoder = (m, captured) => entitiesToDecode[captured.toLowerCase()] || String.fromCharCode(parseInt(captured.substr(2), 10)),
      htmlEncoder = (m, captured) => charsToEncode[captured],
      hyphenateCamelLetters = (all, g1, g2) => {
  return `${g1}-${g2.toLowerCase()}`;
},
      separateCamelLetters = (all, g1, g2) => {
  return `${g1} ${g2.toLowerCase()}`;
},
      replaceNonIdChar = c => {
  if (c) {
    return `_x${[...c].map(ch => ch.charCodeAt(0).toString(16)).join('')}`;
  }

  return '__blank__';
},
      hyphenateCache = {},
      separatedCache = {};
/**
 * Helper for string manipulation.
 */

class StringHelper {
  //region Transform

  /**
   * Capitalizes the first letter of a string, "myString" -> "MyString".
   * @param {String} string The string to capitalize
   * @returns {String} The capitalized string or the value of `string` if falsy.
   * @category String formatting
   */
  static capitalize(string) {
    return string && string[0].toUpperCase() + string.substr(1);
  }
  /**
   * Makes the first letter of a string lowercase, "MyString" -> "myString".
   * @param {String} string The string to un-capitalize.
   * @returns {String} The un-capitalized string or the value of `string` if falsy.
   * @category String formatting
   */

  static uncapitalize(string) {
    return string && string[0].toLowerCase() + string.substr(1);
  }
  /**
   * Converts the passed camelCased string to a hyphen-separated string. eg "minWidth" -> "min-width"
   * @param {String} string The string to convert.
   * @return {String} The string with adjoining lower and upper case letters
   * separated by hyphens and converted to lower case.
   * @category String formatting
   * @internal
   */

  static hyphenate(string) {
    // Cached since it is used heavily with DomHelper.sync()
    const cached = hyphenateCache[string];

    if (cached) {
      return cached;
    }

    return hyphenateCache[string] = string.replace(camelLettersRe, hyphenateCamelLetters);
  }
  /**
   * Converts the passed camelCased string to a capitalized, space-separated string. eg "startDate" -> "Start date".
   * @param {String} string The string to convert.
   * @return {String} The string with spaces separating words.
   * @category String formatting
   * @internal
   */

  static separate(string) {
    // Cached since it may be used heavily
    const cached = separatedCache[string];

    if (cached) {
      return cached;
    }

    return separatedCache[string] = this.capitalize(string.replace(camelLettersRe, separateCamelLetters));
  }
  /**
   * Creates an alphanumeric identifier from any passed string. Encodes spaces and non-alpha characters.
   * @param {String} inString The string from which to strip non-identifier characters.
   * @return {String}
   * @category Misc
   * @internal
   */

  static createId(inString) {
    return String(inString).replace(idRe, replaceNonIdChar);
  }

  static makeValidDomId(id, replaceValue = '') {
    if (id == null) {
      return null;
    }

    return String(id).replace(domIdRe, replaceValue);
  } //endregion
  //region Html

  static escapeRegExp(string, flags) {
    // $& means the whole matched string
    let ret = string.replace(escapeRegExpRe, '\\$&');

    if (flags !== undefined) {
      ret = new RegExp(ret, flags);
    }

    return ret;
  }
  /**
   * This method decodes HTML entities and returns the original HTML.
   *
   * See also {@link #function-encodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   */

  static decodeHtml(str) {
    return str && String(str).replace(htmlDecodeRe, htmlDecoder);
  }
  /**
   * This method encodes HTML entities and returns a string that can be placed in the document and produce the
   * original text rather than be interpreted as HTML. Using this method with user-entered values prevents those
   * values from executing as HTML (i.e., a cross-site scripting or "XSS" security issue).
   *
   * See also {@link #function-decodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   */

  static encodeHtml(str = '') {
    return str && String(str).replace(htmlEncodeRe, htmlEncoder);
  }
  /**
   * This method is similar to {@link #function-encodeHtml-static} except that `\n` and `\r` characters in the
   * given `str` are replaced by `<br>` tags _after_ first being encoded by {@link #function-encodeHtml-static}.
   * @param {String} str
   * @returns {String}
   * @category HTML
   * @internal
   */

  static encodeHtmlBR(str) {
    var _StringHelper$encodeH;

    return (_StringHelper$encodeH = StringHelper.encodeHtml(str)) === null || _StringHelper$encodeH === void 0 ? void 0 : _StringHelper$encodeH.replace(crlfRe, '<br>');
  }
  /**
   * Returns `true` if the provided `text` contains special HTML characters.
   * @param {String} text
   * @returns {Boolean}
   * @category HTML
   * @internal
   */

  static isHtml(text) {
    return typeof text === 'string' && htmlRe$1.test(text || '');
  }
  /**
   * Initializes HTML entities used by {@link #function-encodeHtml-static} and {@link #function-decodeHtml-static}.
   * @param {Object} [mappings] An object whose keys are characters that should be encoded and values are the HTML
   * entity for the character.
   * @private
   */

  static initHtmlEntities(mappings) {
    mappings = mappings || {
      '&': '&amp;',
      '>': '&gt;',
      '<': '&lt;',
      '"': '&quot;',
      "'": '&#39;'
    };
    const chars = Object.keys(mappings); // Maps '<' to '&lt;'

    charsToEncode = mappings; // Inverts the mapping so we can convert '&lt;' to '<'

    entitiesToDecode = chars.reduce((prev, val) => {
      prev[mappings[val]] = val;
      return prev;
    }, {}); // Creates a regex char set like /([<&>])/g to match the characters that need to be encoded (escaping any of
    // the regex charset special chars '[', ']' and '-'):

    htmlEncodeRe = new RegExp(`([${chars.map(c => '[-]'.includes(c) ? '\\' + c : c).join('')}])`, 'g'); // Creates a regex like /(&lt;|&amp;|&gt;)/ig to match encoded entities... good news is that (valid) HTML
    // entities do not contain any regex special characters:

    htmlDecodeRe = new RegExp(`(${Object.values(mappings).join('|')}|&#[0-9]+;)`, 'ig');
  } //endregion
  //region JSON

  /**
   * Parses JSON inside a try-catch block. Returns null if the string could not be parsed.
   *
   * @param {String} string String to parse
   * @returns {Object} Resulting object or null if parse failed
   * @category JSON
   */

  static safeJsonParse(string) {
    let parsed = null;

    try {
      parsed = JSON.parse(string);
    } catch (e) {}

    return parsed;
  }
  /**
   * Stringifies an object inside a try-catch block. Returns null if an exception is encountered.
   *
   * See [JSON.stringify on MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify)
   * for more information on the arguments.
   *
   * @param {Object} object The object to stringify
   * @param {Function|String[]|Number[]} [replacer] A function or array of string/number used to determine properties
   * to include in the JSON string
   * @param {String|Number} [space] Number of spaces to indent or string used as whitespace
   * @returns {Object} Resulting object or null if stringify failed
   * @category JSON
   */

  static safeJsonStringify(object, replacer = null, space = null) {
    let result = null;

    try {
      result = JSON.stringify(object, replacer, space);
    } catch (e) {}

    return result;
  } //endregion
  //region Split & join

  /**
   * Joins all given paths together using the separator as a delimiter and normalizes the resulting path.
   * @param paths {Array} array of paths to join
   * @param pathSeparator [{String}] path separator. Default value is '/'
   * @return {String}
   * @category Misc
   * @internal
   */

  static joinPaths(paths, pathSeparator = '/') {
    return paths.join(pathSeparator).replace(new RegExp('\\' + pathSeparator + '+', 'g'), pathSeparator);
  }
  /**
   * Returns the provided string split on whitespace. If the string is empty or consists of only whitespace, the
   * returned array will be empty. If `str` is not a string, it is simply returned. This allows `null` or already
   * split strings (arrays) to be passed through.
   *
   * For example:
   * ```
   *  console.log(StringHelper.split(' abc def xyz   '));
   *  > ['abc', 'def', 'xyz']
   *  console.log(StringHelper.split(''));
   *  > []
   * ```
   * Compare to the standard `split()` method:
   * ```
   *  console.log(' abc def xyz   '.split(/\s+/));
   *  > ['', 'abc', 'def', 'xyz', '']
   *  console.log(''.split(/\s+/));
   *  > ['']
   * ```
   * @param {String} str
   * @param {String|RegExp} delimiter
   * @returns {String[]}
   * @category Misc
   * @internal
   */

  static split(str, delimiter = whiteSpaceRe$1) {
    let ret = str;

    if (typeof ret === 'string') {
      ret = str.trim(); // w/o trim() whitespace on the ends will give us '' in the array

      ret = ret ? ret.split(delimiter) : []; // also ''.split() = ['']
    }

    return ret;
  } //endregion
  //region XSS

  /**
   * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
   * Scripting) attacks.
   *
   * For example:
   *
   * ```javascript
   *  eventRenderer(eventRecord) {
   *      return StringHelper.xss`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
   *  }
   * ```
   *
   * See {@link Core.helper.StringHelper#function-encodeHtml-static}.
   */

  static xss(strings, ...values) {
    const buf = [];
    let i = values.length;
    buf[i] = strings[i];

    while (i-- > 0) {
      buf[i] = strings[i] + StringHelper.encodeHtml(values[i]);
    }

    return buf.join('');
  }
  /**
   * This is a tagged template function that performs HTML encoding on replacement values to avoid XSS (Cross-Site
   * Scripting) attacks. Unlike {@link Core.helper.StringHelper#function-xss-static}, this method converts `\n` and
   * `\r` characters into `<br>` tags.
   *
   * For example:
   *
   * ```javascript
   *  eventRenderer(eventRecord) {
   *      return StringHelper.xssBR`<span class="${eventRecord.attrib}">${eventRecord.name}</span>`;
   *  }
   * ```
   *
   * See {@link Core.helper.StringHelper#function-encodeHtmlBR-static}.
   * @internal
   */

  static xssBR(strings, ...values) {
    const buf = [];
    let i = values.length;
    buf[i] = strings[i];

    while (i-- > 0) {
      buf[i] = strings[i] + StringHelper.encodeHtmlBR(values[i]);
    }

    return buf.join('');
  } //endregion
  //region JavaScript string

  /**
   * Converts a value to a JavaScript string (not JSON).
   *
   * For example a date to `"new Date(y, m, d)"`, an array to `"[...]"` etc.
   *
   * @param {*} value
   * @param {Object} [options]
   * @returns {String}
   * @internal
   */

  static toJavaScriptValue(value, options) {
    const type = typeOf(value);

    if (type === 'boolean' || type === 'string' || type === 'number' || value === null) {
      return StringHelper.safeJsonStringify(value);
    }

    if (value === globalThis) {
      return 'window';
    }

    if (type === 'date') {
      return `new Date(${value.getFullYear()}, ${value.getMonth()}, ${value.getDate()}, ${value.getHours()}, ${value.getMinutes()}, ${value.getSeconds()}, ${value.getMilliseconds()})`;
    }

    if (type === 'array') {
      return `[${value.map(v => StringHelper.toJavaScriptValue(v, options))}]`;
    }

    if (type === 'object' || type === 'instance') {
      return this.toJavaScriptString(value, options);
    }

    if (type === 'function') {
      let contents = value.toString(); // async myFunction() {} => async function() {}

      if (contents.match(/^async (\w+?)\(/)) {
        contents = contents.replace(/^async (\w+?)\(/, 'async function(');
      } // Not an arrow fn? Replace name with function since we always add prop name prior to getting here
      // eventRenderer() {} -> function() {}
      else if (!contents.startsWith('async(') && contents.match(/^(\w+?)\(/)) {
        contents = contents.replace(/^(\w+?)\(/, 'function(');
      }

      return contents;
    }

    if (type === 'class') {
      if (value.toJavaScriptValue) {
        return value.toJavaScriptValue(options);
      }

      return Object.prototype.hasOwnProperty.call(value, '$name') ? value.$name : value.name;
    }
  }
  /**
   * Converts an object into a JavaScript string (not JSON).
   *
   * For example `{ a: 1, b: [2, 3] }` -> `"'{ a: 1, b: [2, 3] }'"`
   *
   * @param {Object} obj
   * @param {Object} [options]
   * @returns {String}
   * @internal
   */

  static toJavaScriptString(obj, options = {}) {
    var _options$level;

    const level = (_options$level = options.level) !== null && _options$level !== void 0 ? _options$level : 0,
          intendSize = 2; // Not using template strings to have control over indentation

    return '{\n' + Object.keys(obj).map(key => // All properties in an object are indented one step further than the object itself
    ' '.repeat((level + 1) * intendSize) + (key.match(/[- *]/) ? `"${key}"` : key) + `: ${StringHelper.toJavaScriptValue(obj[key], _objectSpread2(_objectSpread2({}, options), {}, {
      level: level + 1
    }))}`).join(',\n') + // Closing brace is indented to same level as the object
    '\n' + ' '.repeat(level * intendSize) + '}';
  } //endregion

}
StringHelper.initHtmlEntities();
StringHelper._$name = 'StringHelper';

// IMPORTANT - adding imports here can create problems for Base class
/**
 * @module Core/helper/util/Objects
 */

const {
  toString: toString$3
} = Object.prototype,
      {
  isFrozen
} = Object,
      afterRe = /\s*<\s*/,
      beforeRe = /\s*>\s*/,
      blendOptions = {},
      typeNameRe = /\[object ([^\]]+)]/,
      typeCache = {},
      emptyObject$d = Object.freeze({});

const typeOf = value => {
  const baseType = typeof value; // If not atomic type, we handle date or null

  if (baseType === 'object') {
    if (value === null) {
      return 'null';
    }

    if (Array.isArray(value)) {
      return 'array';
    }

    if (Object.prototype.toString.call(value) === '[object Date]') {
      return 'date';
    }

    if (value.isBase) {
      return 'instance';
    }
  }

  if (baseType === 'function') {
    if (value.isBase) {
      return 'class';
    }
  }

  return baseType;
};
// work around is to copy method documentation to ObjectHelper (the public interface). Also tried making ObjectHelper
// a singleton.

/**
 * Helper for low-level Object manipulation.
 *
 * While documented on {@link Core.helper.ObjectHelper}, the following static methods are implemented by this class:
 *
 * - `{@link Core.helper.ObjectHelper#function-assign-static}`
 * - `{@link Core.helper.ObjectHelper#function-assignIf-static}`
 * - `{@link Core.helper.ObjectHelper#function-clone-static}`
 * - `{@link Core.helper.ObjectHelper#function-createTruthyKeys-static}`
 * - `{@link Core.helper.ObjectHelper#function-getPath-static}`
 * - `{@link Core.helper.ObjectHelper#function-getTruthyKeys-static}`
 * - `{@link Core.helper.ObjectHelper#function-getTruthyValues-static}`
 * - `{@link Core.helper.ObjectHelper#function-isEmpty-static}`
 * - `{@link Core.helper.ObjectHelper#function-isObject-static}`
 * - `{@link Core.helper.ObjectHelper#function-merge-static}`
 * - `{@link Core.helper.ObjectHelper#function-setPath-static}`
 * - `{@link Core.helper.ObjectHelper#function-typeOf-static}`
 * @internal
 */

class Objects {
  static assign(dest, ...sources) {
    for (let source, key, i = 0; i < sources.length; i++) {
      source = sources[i];

      if (source) {
        for (key in source) {
          dest[key] = source[key];
        }
      }
    }

    return dest;
  }

  static assignIf(dest, ...sources) {
    for (let source, key, i = 0; i < sources.length; i++) {
      source = sources[i];

      if (source) {
        for (key in source) {
          if (!(key in dest) || dest[key] === undefined) {
            dest[key] = source[key];
          }
        }
      }
    }

    return dest;
  }

  static blend(dest, source, options) {
    options = options || blendOptions;
    dest = dest || {};
    const {
      clone = Objects.clone,
      merge = Objects.blend
    } = options;

    if (Array.isArray(source)) {
      if (source.length > 1) {
        source.forEach(s => {
          dest = Objects.blend(dest, s, options);
        });
        return dest;
      }

      source = source[0];
    }

    if (source) {
      let destValue, key, value;

      for (key in source) {
        value = source[key];

        if (value && O.isObject(value)) {
          destValue = dest[key];
          options.key = key;

          if (destValue && O.isObject(destValue)) {
            if (isFrozen(destValue)) {
              dest[key] = destValue = clone(destValue, options);
            }

            value = merge(destValue, value, options);
          } else {
            // We don't need to clone frozen objects, but we do clone mutable objects as they get
            // applied to the dest.
            value = isFrozen(value) ? value : clone(value, options);
          }
        }

        dest[key] = value;
      }
    }

    return dest;
  }

  static clone(value, handler) {
    let cloned = value,
        key;

    if (value && typeof value === 'object') {
      const options = handler && typeof handler === 'object' && handler;

      if (options) {
        // When using blend(), the 2nd argument is the options object, so ignore that case
        handler = null;
      }

      if (O.isObject(value)) {
        cloned = {};

        for (key in value) {
          cloned[key] = O.clone(value[key]);
        }
      } else if (Array.isArray(value)) {
        cloned = []; // Loop backwards to:
        //  1. read source.length once
        //  2. get result array sized on first pass (avoid growing)

        for (key = value.length; key-- > 0;) {
          cloned[key] = O.clone(value[key]);
        }
      } else if (O.isDate(value)) {
        cloned = new Date(value.getTime());
      } else if (handler) {
        // Allow other types to be handled (e.g., DOM nodes).
        cloned = handler(value);
      }
    }

    return cloned;
  }

  static createTruthyKeys(source) {
    const keys = StringHelper.split(source),
          result = keys && {};

    if (keys) {
      for (const key of keys) {
        // StringHelper.split won't return empty keys if passed a string, but we
        // could have been passed a String[]
        if (key) {
          result[key] = true;
        }
      }
    }

    return result;
  }
  /**
   * Returns value for a given path in the object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {*} Value associated with passed key
   */

  static getPath(object, path) {
    return path.split('.').reduce((result, key) => {
      return (result || emptyObject$d)[key];
    }, object);
  }
  /**
   * Returns value for a given path in the object, placing a passed default value in at the
   * leaf property and filling in undefined properties all the way down.
   * @param {Object} object Object to get path value for.
   * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
   * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
   * @param {*} [defaultValue] Optionally the value to put in as the `someKey` property.
   * @returns {*} Value at the leaf position of the path.
   */

  static getPathDefault(object, path, defaultValue) {
    const keys = Array.isArray(path) ? path : typeof path === 'string' ? path.split('.') : [path],
          length = keys.length - 1;
    return keys.reduce((result, key, index) => {
      if (defaultValue && !(key in result)) {
        // Can't use emptyObject here, we are creating a node in the object tree
        result[key] = index === length ? defaultValue : {};
      }

      return (result || emptyObject$d)[key];
    }, object);
  }
  /**
   * Determines if the specified path exists
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {Boolean}
   */

  static hasPath(object, path) {
    return path.split('.').every(key => {
      if (key in object) {
        object = object[key];
        return true;
      }

      return false;
    });
  }

  static getTruthyKeys(source) {
    const keys = [];

    for (const key in source) {
      if (source[key]) {
        keys.push(key);
      }
    }

    return keys;
  }

  static getTruthyValues(source) {
    const values = [];

    for (const key in source) {
      if (source[key]) {
        values.push(source[key]);
      }
    }

    return values;
  }

  static isClass(object) {
    var _object$prototype;

    if (typeof object === 'function' && ((_object$prototype = object.prototype) === null || _object$prototype === void 0 ? void 0 : _object$prototype.constructor) === object) {
      // TODO find a way to differentiate function(){} from class{}
      return true;
    }

    return false;
  }

  static isDate(object) {
    // A couple quick rejections but only sure way is typeOf:
    return Boolean(object && object.getUTCDate) && O.typeOf(object) === 'date';
  }
  /**
   * Check if passed object is a Promise or contains `then` method.
   * Used to fix problems with detecting promises in code with `instance of Promise` when
   * Promise class is replaced with any other implementation like `ZoneAwarePromise` in Angular.
   * Related to these issues:
   * https://github.com/bryntum/support/issues/791
   * https://github.com/bryntum/support/issues/2990
   *
   * @param {Object} object object to check
   * @returns {Boolean} truthy value if object is a Promise
   * @internal
   */

  static isPromise(object) {
    if (Promise && Promise.resolve) {
      return Promise.resolve(object) === object || typeof (object === null || object === void 0 ? void 0 : object.then) === 'function';
    }

    throw new Error('Promise not supported in your environment');
  }

  static isEmpty(object) {
    if (object && typeof object === 'object') {
      // noinspection LoopStatementThatDoesntLoopJS
      for (const p in object) {
        // eslint-disable-line no-unused-vars,no-unreachable-loop
        return false;
      }
    }

    return true;
  }

  static isObject(value) {
    const C = value === null || value === void 0 ? void 0 : value.constructor;
    return Boolean(C // Most things have a .constructor property
    ? // An in-frame instance of Object
    C === Object || // Detect cross-frame objects, but exclude instance of custom classes named Object. typeOf(value) is
    // "object" even for instances of a class and typeOf(C) is "function" for all constructors. We'll have
    // to settle for relying on the fact that getPrototypeOf(Object.prototype) === null.
    // NOTE: this issue does come up in Scheduler unit tests at least.
    C.getPrototypeOf && C.prototype && !Object.getPrototypeOf(C.prototype) // Since all classes have a constructor property, an object w/o one is likely from Object.create(null). Of
    // course, primitive types do not have ".constructor"
    : value && typeof value === 'object');
  }

  static isInstantiated(object) {
    return object ? typeof object === 'object' && !Objects.isObject(object) : false;
  }

  static merge(dest, ...sources) {
    return Objects.blend(dest, sources);
  }
  /**
   * Merges two "items" objects. An items object is a simple object whose keys act as identifiers and whose values
   * are "item" objects. An item can be any object type. This method is used to merge such objects while maintaining
   * their property order. Special key syntax is used to allow a source object to insert a key before or after a key
   * in the `dest` object.
   *
   * For example:
   * ```javascript
   *  let dest = {
   *      foo : {},
   *      bar : {},
   *      fiz : {}
   *  }
   *
   *  console.log(Object.keys(dest));
   *  > ["foo", "bar", "fiz"]
   *
   *  dest = mergeItems(dest, {
   *      'zip > bar' : {}    // insert "zip" before "bar"
   *      'bar < zap' : {}    // insert "zap" after "bar"
   *  });
   *
   *  console.log(Object.keys(dest));
   *  > ["foo", "zip", "bar", "zap", "fiz"]
   * ```
   *
   * @param {Object} dest The destination object.
   * @param {Object|Object[]} src The source object or array of source objects to merge into `dest`.
   * @param {Object} [options] The function to use to merge items.
   * @param {Function} [options.merge] The function to use to merge items.
   * @returns {Object} The merged object. This will be the `dest` object.
   * @internal
   */

  static mergeItems(dest, src, options) {
    options = options || blendOptions;
    let anchor, delta, index, indexMap, key, shuffle, srcVal;
    const {
      merge = Objects.blend
    } = options;
    dest = dest || {};

    if (Array.isArray(src)) {
      src.forEach(s => {
        dest = Objects.mergeItems(dest, s, options);
      });
    } else if (src) {
      // https://2ality.com/2015/10/property-traversal-order-es6.html
      // Bottom line: Object keys are iterated in declared/insertion order... unless the key is an integer or
      // Symbol, but we don't care about those generally.
      for (key in src) {
        srcVal = src[key];
        anchor = null; // Allow a key to be added before or after another:
        //
        //  {
        //      'foo > bar' : {
        //          ...
        //      },
        //      'bar < derp' : {
        //          ...
        //      }
        //  }
        //
        // The goal above is to add a 'foo' key before the existing 'bar' key while adding a 'derp' key after
        // 'bar'.

        if (key.includes('>')) {
          [key, anchor] = key.split(beforeRe);
          delta = 0;
        } else if (key.includes('<')) {
          [anchor, key] = key.split(afterRe);
          delta = 1;
        }

        if (key in dest) {
          // Changing the value of a key does not change its iteration order. Since "key in dest" we can do
          // what we need directly.
          if (srcVal && dest[key] && merge) {
            options.key = key;
            srcVal = merge(dest[key], srcVal, options);
          }

          dest[key] = srcVal;
        } else if (!anchor) {
          var _indexMap;

          dest[key] = srcVal;
          (_indexMap = indexMap) === null || _indexMap === void 0 ? void 0 : _indexMap.set(key, indexMap.size);
        } else {
          // Lazily sprout the item index map. When we first merge an item into an items object, we create this
          // Map to control the ordering. This is because any keys we add would necessarily be iterated after
          // the original properties.
          if (!indexMap) {
            indexMap = new Map();
            index = 0;

            for (const k in dest) {
              indexMap.set(k, index++);
            }
          }

          index = indexMap.get(anchor);
          dest[key] = srcVal;

          if (index == null && delta) {
            index = indexMap.size;
          } else {
            shuffle = shuffle || [];
            index = (index || 0) + delta; // Adjust all key indices >= index up by 1 to maintain integer indices (required by the above
            // use case).

            for (const item of indexMap) {
              const [k, v] = item;

              if (index <= v) {
                /*
                Consider object w/the following order:
                    {
                        foo : {}',
                        bar : {},
                        baz : {},
                        zip : {},
                        goo : {},
                        fiz : {}
                    }
                The indexMap is:
                    foo : 0
                    bar : 1
                    baz : 2
                    zip : 3
                    goo : 4
                    fiz : 5
                To insert before goo, we populate shuffle thusly (to set up for popping):
                    +-----+-----+
                    | fiz | goo |
                    +-----+-----+
                      0        1
                      =6-5-1   =6-4-1
                */
                shuffle && (shuffle[indexMap.size - v - 1] = k);
                indexMap.set(k, v + 1);
              }
            } // Delete and re-add the keys that should follow the new key to establish the iteration order
            // we need:

            if (shuffle) {
              while (shuffle.length) {
                const k = shuffle.pop(),
                      v = dest[k];
                delete dest[k];
                dest[k] = v;
              }
            }
          }

          indexMap.set(key, index);
        }
      }
    }

    return dest;
  }
  /**
   * Sets value for a given path in the object
   * @param {Object} object Target object
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @param {*} value Value for a given path
   * @returns {Object} Returns passed object
   */

  static setPath(object, path, value) {
    path.split('.').reduce((result, key, index, array) => {
      const isLast = index === array.length - 1;

      if (isLast) {
        return result[key] = value;
      } else if (!(result[key] instanceof Object)) {
        result[key] = {};
      }

      return result[key];
    }, object);
    return object;
  }

  static typeOf(value) {
    let type = typeof value,
        match,
        trueType; // If not atomic type, we handle date or null

    if (type === 'object') {
      if (value === null) {
        type = 'null';
      } else {
        trueType = toString$3.call(value);

        if (!(type = typeCache[trueType])) {
          match = typeNameRe.exec(trueType);
          typeCache[trueType] = type = match ? match[1].toLowerCase() : trueType;
        }
      }
    } // NaN is the only value that is !== to itself
    else if (value !== value) {
      // eslint-disable-line no-self-compare
      type = 'nan';
    }

    return type;
  }

}
const O = Objects;
Objects._$name = 'Objects';

/**
 * @module Core/helper/BrowserHelper
 */

/**
 * Static helper class that does browser or platform detection and provides other helper functions.
 */
class BrowserHelper {
  //region Init
  static cacheFlags(platform = navigator.platform, userAgent = navigator.userAgent) {
    const me = this; // os

    me._isLinux = Boolean(platform.match(/Linux/));
    me._isMac = Boolean(platform.match(/Mac/));
    me._isWindows = Boolean(platform.match(/Win32/)); // Edge user agent contains webkit too.
    // This is not a typo. Edge has "Safari/537.36 Edg/96.0.1054.34"

    me._isWebkit = Boolean(userAgent.match(/WebKit/) && !userAgent.match(/Edg/));
    me._firefoxVersion = me.getVersion(userAgent, /Firefox\/(\d+)\./);
    me._isFirefox = me._firefoxVersion > 0;
    me._chromeVersion = me.getVersion(userAgent, /Chrom(?:e|ium)\/(\d+)\./);
    me._isChrome = me._chromeVersion > 0;
    me._isSafari = Boolean(userAgent.match(/Safari/)) && !me._isChrome;
    me._isMobileSafari = Boolean(userAgent.match(/Mobile.*Safari/));
    me._safariVersion = me.getVersion(userAgent, /Version\/(.*).Safari/);
    me._isAndroid = Boolean(userAgent.match(/Android/g));
  } //endregion
  //region Device

  /**
   * Yields `true` if the current browser supports CSS style `overflow:clip`.
   * @property {Boolean}
   * @readonly
   * @internal
   */

  static get supportsOverflowClip() {
    if (this._supportsOverflowClip == null) {
      const div = document.createElement('div');
      div.style.overflow = 'clip';
      div.style.display = 'none'; // If we're called before DOMContentLoaded, body won't be available.
      // HTML element works for style calcs.

      document.documentElement.appendChild(div);
      this._supportsOverflowClip = div.ownerDocument.defaultView.getComputedStyle(div).getPropertyValue('overflow') === 'clip';
      div.remove();
    }

    return this._supportsOverflowClip;
  }
  /**
   * Yields `true` if the current browser supports CSS style `position:sticky`.
   * @property {Boolean}
   * @readonly
   * @internal
   */

  static get supportsSticky() {
    return true;
  }
  /**
   * Returns matched version for userAgent.
   * @param {String} versionRe version match regular expression
   * @returns {Number} matched version
   * @readonly
   * @internal
   */

  static getVersion(userAgent, versionRe) {
    const match = userAgent.match(versionRe);
    return match ? parseFloat(match[1]) : 0;
  }
  /**
   * Determines if the user is using a touch device.
   * @property {Boolean}
   * @readonly
   * @internal
   */

  static get isTouchDevice() {
    // Allow tests or client code to set
    if ('_isTouchDevice' in this) {
      return this._isTouchDevice;
    }

    return globalThis.matchMedia('(pointer:coarse)').matches;
  }

  static get supportsPointerEvents() {
    return Boolean(globalThis.PointerEvent || globalThis.MSPointerEvent);
  } // Reports true by default for our tests

  static get isHoverableDevice() {
    if (this._isHoverableDevice === undefined) {
      this._isHoverableDevice = globalThis.matchMedia('(any-hover: hover)').matches;
    }

    return this._isHoverableDevice;
  } //endregion
  //region Platform

  static get isBrowserEnv() {
    // This window reference is left on purpose, globalThis is always defined
    return typeof window !== 'undefined';
  }
  /**
   * Checks if platform is Mac.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */

  static get isMac() {
    return this._isMac;
  }
  /**
   * Checks if platform is Windows.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */

  static get isWindows() {
    return this._isWindows;
  }
  /**
   * Checks if platform is Linux.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */

  static get isLinux() {
    return this._isLinux;
  }
  /**
   * Checks if platform is Android.
   * @property {Boolean}
   * @readonly
   * @category Platform
   */

  static get isAndroid() {
    return this._isAndroid;
  } //endregion
  //region Browser

  /**
   * Checks if browser is Webkit.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */

  static get isWebkit() {
    return this._isWebkit;
  }
  /**
   * Checks if browser is Chrome or Chromium based browser.
   * Returns truthy value for Edge Chromium.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */

  static get isChrome() {
    return this._isChrome;
  }
  /**
   * Returns the major Chrome version or 0 for other browsers.
   * @property {Number}
   * @readonly
   * @category Browser
   */

  static get chromeVersion() {
    return this._chromeVersion;
  }
  /**
   * Checks if browser is Firefox.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */

  static get isFirefox() {
    return this._isFirefox;
  }
  /**
   * Returns the major Firefox version or 0 for other browsers.
   * @property {Number}
   * @readonly
   * @category Browser
   */

  static get firefoxVersion() {
    return this._firefoxVersion;
  }
  /**
   * Checks if browser is Safari.
   * @property {Boolean}
   * @readonly
   * @category Browser
   */

  static get isSafari() {
    return this._isSafari;
  }

  static get safariVersion() {
    return this._safariVersion;
  }
  /**
   * Checks if browser is mobile Safari
   * @property {Boolean}
   * @readonly
   * @category Browser
   */

  static get isMobileSafari() {
    return this._isMobileSafari;
  }
  /**
   * Returns `true` if the browser supports passive event listeners.
   * @property {Boolean}
   * @internal
   * @deprecated 5.0 Use {@link Element} classList.remove method
   * @category Browser
   */

  static get supportsPassive() {
    return true;
  } //endregion
  //region Storage
  // https://developer.mozilla.org/en-US/docs/Web/API/Web_Storage_API/Using_the_Web_Storage_API

  static get storageAvailable() {
    let storage, x;

    try {
      storage = localStorage;
      x = '__storage_test__';
      storage.setItem(x, x);
      storage.removeItem(x);
      return true;
    } catch (e) {
      return e instanceof DOMException && ( // everything except Firefox
      e.code === 22 || // Firefox
      e.code === 1014 || // test name field too, because code might not be present
      // everything except Firefox
      e.name === 'QuotaExceededError' || // Firefox
      e.name === 'NS_ERROR_DOM_QUOTA_REACHED') && // acknowledge QuotaExceededError only if there's something already stored
      storage.length !== 0;
    }
  }

  static setLocalStorageItem(key, value) {
    this.storageAvailable && localStorage.setItem(key, value);
  }

  static getLocalStorageItem(key) {
    return this.storageAvailable && localStorage.getItem(key);
  }

  static removeLocalStorageItem(key) {
    this.storageAvailable && localStorage.removeItem(key);
  } //endregion
  //region Helpers

  /**
   * Returns parameter value from search string by parameter name.
   * @param {String} paramName search parameter name
   * @param {*} [defaultValue] default value if parameter not found
   * @param {String} [search] search string. Defaults to `document.location.search`
   * @category Helper
   */

  static searchParam(paramName, defaultValue = null, search = document.location.search) {
    const re = new RegExp(`[?&]${paramName}=?([^&]*)`),
          match = search.match(re);
    return match && match[1] || defaultValue;
  }
  /**
   * Returns cookie by name.
   * @param {String} name cookie name
   * @return {String} cookie string value
   * @category Helper
   */

  static getCookie(name) {
    const nameEq = encodeURIComponent(name) + '=',
          cookieItems = document.cookie.split(';');

    for (let i = 0; i < cookieItems.length; i++) {
      let c = cookieItems[i];

      while (c.charAt(0) === ' ') {
        c = c.substring(1, c.length);
      }

      if (c.indexOf(nameEq) === 0) {
        return decodeURIComponent(c.substring(nameEq.length, c.length));
      }
    }

    return '';
  }
  /**
   * Triggers a download of a file with the specified name / URL.
   * @param {String} filename The filename of the file to be downloaded
   * @param {String} [url] The URL where the file is to be downloaded from
   * @internal
   * @category Download
   */

  static download(filename, url) {
    const a = document.createElement('a');
    a.download = filename;
    a.href = url || filename;
    a.style.cssText = 'display:none';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
  }
  /**
   * Triggers a download of a Blob with the specified name.
   * @param {Blob} blob The Blob to be downloaded
   * @param {String} filename The filename of the file to be downloaded
   * @internal
   * @category Download
   */

  static downloadBlob(blob, filename) {
    const url = globalThis.URL.createObjectURL(blob);
    this.download(filename, url);
    globalThis.URL.revokeObjectURL(url);
  }

  static get queryString() {
    var _Object$fromEntries;

    // new URLSearchParams throws in salesforce
    // https://github.com/salesforce/lwc/issues/1812
    const params = new URL(globalThis.location.href).searchParams; // ?. to be nice to users with Chrome versions < 73

    return (_Object$fromEntries = Object.fromEntries) === null || _Object$fromEntries === void 0 ? void 0 : _Object$fromEntries.call(Object, params.entries());
  }

  static copyToClipboard(code) {
    let success = true;
    const textArea = document.createElement('textarea');
    textArea.value = code;
    textArea.style.height = textArea.style.width = 0;
    document.body.appendChild(textArea);
    textArea.select();

    try {
      document.execCommand('copy');
    } catch (e) {
      success = false;
    }

    textArea.remove();
    return success;
  }

  static isBryntumOnline(searchStrings) {
    var _searchStrings;

    searchStrings = Array.isArray(searchStrings) ? searchStrings : [searchStrings];
    return Boolean(/^(www\.)?bryntum\.com/.test(globalThis.location.host) || ((_searchStrings = searchStrings) === null || _searchStrings === void 0 ? void 0 : _searchStrings.some(str => this.queryString[str] != null)));
  } //endregion

}

if (BrowserHelper.isBrowserEnv) {
  BrowserHelper.cacheFlags();
}

BrowserHelper._$name = 'BrowserHelper';

/**
 * @module Core/helper/VersionHelper
 */

let isSiesta = false;

try {
  var _globalThis$parent;

  isSiesta = Boolean(((_globalThis$parent = globalThis.parent) === null || _globalThis$parent === void 0 ? void 0 : _globalThis$parent.Siesta) || typeof process !== 'undefined' && (globalThis.StartTest || globalThis.Siesta));
} catch (e) {} // Set Default to hoverable device for our tests to run reliably

if (isSiesta) {
  BrowserHelper._isHoverableDevice = true;
}
/**
 * Helper for version handling
 * @private
 * @example
 *
 * VersionHelper.setVersion('grid', '1.5');
 *
 * if (VersionHelper.getVersion('grid').isNewerThan('1.0')) {
 *   ...
 * }
 */

class VersionHelper {
  /**
   * Set version for specified product
   * @private
   * @param {String} product
   * @param {String} version
   */
  static setVersion(product, version) {
    product = product.toLowerCase();
    VH[product] = {
      version,

      isNewerThan(otherVersion) {
        return otherVersion < version;
      },

      isOlderThan(otherVersion) {
        return otherVersion > version;
      }

    };
    let bundleFor = ''; // Var productName is only defined in bundles, it is internal to bundle so not available on window. Used to
    // tell importing combinations of grid/scheduler/gantt bundles apart from loading same bundle twice
    // eslint-disable-next-line no-undef

    if (typeof productName !== 'undefined') {
      // eslint-disable-next-line no-undef
      bundleFor = productName;
    } // Set "global" flag to detect bundle being loaded twice

    const globalKey = `${bundleFor}.${product}${version.replace(/\./g, '-')}`;

    if (BrowserHelper.isBrowserEnv && !globalThis.bryntum.silenceBundleException) {
      if (globalThis.bryntum[globalKey] === true) {
        if (isSiesta) {
          globalThis.BUNDLE_EXCEPTION = true;
        } else {
          throw new Error('Bryntum bundle included twice, check cache-busters and file types (.js).\n' + 'Simultaneous imports from "*.module.js" and "*.umd.js" bundles are not allowed.');
        }
      } else {
        globalThis.bryntum[globalKey] = true;
      }
    }
  }
  /**
   * Get (previously set) version for specified product
   * @private
   * @param {String} product
   */

  static getVersion(product) {
    product = product.toLowerCase();

    if (!VH[product]) {
      throw new Error('No version specified! Please check that you import VersionHelper right into the class from where you call `deprecate` function.');
    }

    return VH[product].version;
  }
  /**
   * Checks the passed product against the passed version using the passed test.
   * @param {String} product The name of the product to test the version of
   * @param {String} version The version to test against
   * @param {String} test The test operator, `<=`, `<`, `=`, `>` or `>=`.
   * @returns {Boolean} `true` if the test passes.
   * @internal
   */

  static checkVersion(product, version, test) {
    const productVersion = VH.getVersion(product);
    let result;

    switch (test) {
      case '<':
        result = productVersion < version;
        break;

      case '<=':
        result = productVersion <= version;
        break;

      case '=':
        result = productVersion === version;
        break;

      case '>=':
        result = productVersion >= version;
        break;

      case '>':
        result = productVersion > version;
        break;
    }

    return result;
  }
  /**
   * Based on a comparison of current product version and the passed version this method either outputs a console.warn
   * or throws an error.
   * @param {String} product The name of the product
   * @param {String} invalidAsOfVersion The version where the offending code is invalid (when any compatibility layer
   * is actually removed).
   * @param {String} message Required! A helpful warning message to show to the developer using a deprecated API.
   * @internal
   */

  static deprecate(product, invalidAsOfVersion, message) {
    const justWarn = VH.checkVersion(product, invalidAsOfVersion, '<');

    if (justWarn) {
      // During the grace period (until the next major release following the deprecated code), just show a console warning
      console.warn(`Deprecation warning: You are using a deprecated API which will change in v${invalidAsOfVersion}. ${message}`);
    } else {
      throw new Error(`Deprecated API use. ${message}`);
    }
  }
  /**
   * Returns truthy value if environment is in testing mode.
   * @returns {Boolean}
   * @internal
   **/

  static get isTestEnv() {
    return isSiesta;
  }

  static get isDebug() {
    let result = false;
    return result;
  }

}
const VH = VersionHelper;

if (BrowserHelper.isBrowserEnv) {
  (globalThis.bryntum || (globalThis.bryntum = {})).getVersion = VH.getVersion.bind(VH);
  globalThis.bryntum.checkVersion = VH.checkVersion.bind(VH);
  globalThis.bryntum.deprecate = VH.deprecate.bind(VH);
  globalThis.bryntum.isTestEnv = VH.isTestEnv;
}

VersionHelper._$name = 'VersionHelper';

//  ObjectHelper -> DateHelper -> LocaleManager -> Base -> us

/**
 * @module Core/Config
 */

const {
  defineProperty: defineProperty$7,
  getOwnPropertyDescriptor: getOwnPropertyDescriptor$1
} = Reflect,
      {
  hasOwnProperty: hasOwnProperty$8,
  toString: toString$2
} = Object.prototype,
      instancePropertiesSymbol$1 = Symbol('instanceProperties'),
      configuringSymbol$1 = Symbol('configuring'),
      lazyConfigValues = Symbol('lazyConfigValues'),
      DATE_TYPE$1 = toString$2.call(new Date()),
      whitespace = /\s+/,
      createClsProps = (result, cls) => {
  result[cls] = 1;
  return result;
};
/**
 * This class holds the description of a config property. Only one instance of this class is needed for each config
 * name (e.g., "text"). If config options are supplied, however, they also contribute to the cached instance.
 *
 * Instances should always be retrieved by calling `Config.get()`.
 *
 * The **Configs** of this class correspond to `options` that can be supplied to the `get()` method. These affect the
 * behavior of the config property in some way, as descried by their respective documentation.
 *
 * This class is not used directly.
 *
 * ## The Setter
 * The primary functionality provided by `Config` is its standard setter. This setter function ensures consistent
 * behavior when modifying config properties.
 *
 * The standard setter algorithm is as follows (using the `'text'` config for illustration):
 *
 *  - If the class defines a `changeText()` method, call it passing the new value and the current value:
 *    `changeText(newText, oldText)`.<br>
 *    Then:
 *    * If `changeText()` exits without returning a value (i.e., `undefined`), exit and do nothing further. The
 *      assumption is that the changer method has done all that is required.
 *    * Otherwise, the return value of `changeText()` replaces the incoming value passed to the setter.
 *  - If the new value (or the value returned by `changeText()`) is `!==` to the current value:
 *    * Update the stored config value in `this._text`.
 *    * If the class defines an `updateText()` method, call it passing the new value and the previous value.
 *      `updateText(newText, oldText)`
 *    * If the class defines an `onConfigChange()` method, call it passing an object with the following properties:
 *        - `name` - The config's name
 *        - `value` - The new value
 *        - `was` - The previous value
 *        - `config` - The `Config` instance.
 *
 * NOTE: unlike `changeText()` and `updateText()`, the name of the `onConfigChange()` method is unaffected by the
 * config's name.
 *
 * @internal
 */

class Config {
  /**
   * Returns the `Config` instance for the given `name` and `options`.
   * @param {String} name The name of the config (e.g., 'text' for the text config).
   * @param {Object} [options] Config behavior options.
   * @returns {Core.Config}
   * @internal
   */
  static get(name, options) {
    const {
      cache
    } = this,
          baseCfg = cache[name] || (cache[name] = new Config(name));
    let cfg = baseCfg,
        key;

    if (options) {
      key = Config.makeCacheKey(name, options);

      if (!(cfg = key && cache[key])) {
        cfg = baseCfg.extend(options);

        if (key) {
          cache[key] = cfg;
        }
      }
    }

    return cfg;
  }

  constructor(name) {
    const me = this,
          cap = name[0].toUpperCase() + name.substr(1);
    me.base = me; // so extend()ed configs have a link to the base definition

    me.name = name;
    me.field = '_' + name;
    me.capName = cap;
    me.changer = 'change' + cap;
    me.initializing = 'initializing' + cap;
    me.updater = 'update' + cap;
  }
  /**
   * The descriptor to use with `Reflect.defineProperty()` for defining this config's getter and setter.
   * @property {Object}
   * @private
   */

  get descriptor() {
    let descriptor = this._descriptor;

    if (!descriptor || !hasOwnProperty$8.call(this, '_descriptor')) {
      // lazily make the descriptor
      this._descriptor = descriptor = this.makeDescriptor();
    }

    return descriptor;
  }
  /**
   * The descriptor to use with `Reflect.defineProperty()` for defining this config's initter.
   * @property {Object}
   * @private
   */

  get initDescriptor() {
    let descriptor = this._initDescriptor;

    if (!descriptor || !hasOwnProperty$8.call(this, '_initDescriptor')) {
      // lazily make the descriptor
      this._initDescriptor = descriptor = this.makeInitter();
    }

    return descriptor;
  }
  /**
   * This method compares two values for semantic equality. By default, this is based on the `===` operator. This
   * is often overridden for configs that accept `Date` or array values.
   * @param {*} value1
   * @param {*} value2
   * @returns {Boolean}
   * @internal
   */

  equal(value1, value2) {
    return value1 === value2;
  }
  /**
   * Extends this config with a given additional set of options. These objects are just prototype extensions of this
   * instance.
   * @param {Object} options
   * @returns {Core.Config}
   * @internal
   */

  extend(options) {
    const cfg = Object.assign(Object.create(this), options),
          {
      equal,
      merge
    } = options,
          {
      equalityMethods
    } = Config;

    if (typeof equal === 'string') {
      if (equal.endsWith('[]')) {
        cfg.equal = Config.makeArrayEquals(equalityMethods[equal.substr(0, equal.length - 2)]);
      } else {
        cfg.equal = equalityMethods[equal];
      }
    }

    if (typeof merge === 'string') {
      // Base uses { merge : 'replace' } for defaultConfig properties
      cfg.merge = Config.mergeMethods[merge];
    }

    return cfg;
  }
  /**
   * Defines the property on a given target object via `Reflect.defineProperty()`. If the object has its own getter,
   * it will be preserved. It is invalid to define a setter.
   * @param {Object} target
   * @internal
   */

  define(target) {
    const existing = getOwnPropertyDescriptor$1(target, this.name);
    let descriptor = this.descriptor;

    if (existing && existing.get) {
      descriptor = Object.assign({}, descriptor);
      descriptor.get = existing.get;
    }

    defineProperty$7(target, this.name, descriptor);
  }
  /**
   * Defines the property initter on the `target`. This is a property getter/setter that propagates the configured
   * value when the property is read.
   * @param {Object} target
   * @param {*} value
   * @internal
   */

  defineInitter(target, value) {
    const {
      name
    } = this,
          properties = target[instancePropertiesSymbol$1];
    let lazyValues, prop; // If there is an existing property with a getter/setter, *not* a value
    // defined on the object for this config we must call it in our injected getter/setter.

    if (!properties[name] && (
    /* assign */
    prop = getOwnPropertyDescriptor$1(target, name)) && !('value' in prop)) {
      properties[name] = prop;
    } // Set up a temporary instance property which will pull in the value from the initialConfig if the getter
    // is called first.

    defineProperty$7(target, name, this.initDescriptor);

    if (this.lazy) {
      lazyValues = target[lazyConfigValues] || (target[lazyConfigValues] = {
        $counter_: 0
      });
      lazyValues[name] = value;
      ++lazyValues.$counter_;
    }
  }
  /**
   * Returns an equality function for arrays of a base type, for example `'date'`.
   * @param {Function} [fn] The function to use to compare array elements. By default, operator `===` is used.
   * @returns {Function}
   * @private
   */

  static makeArrayEquals(fn) {
    return (value1, value2) => {
      let i,
          equal = value1 && value2 && value1.length === (i = value2.length);

      if (equal && Array.isArray(value1) && Array.isArray(value2)) {
        if (fn) {
          while (equal && i-- > 0) {
            equal = fn(value1[i], value2[i]);
          }
        } else {
          while (equal && i-- > 0) {
            equal = value1[i] === value2[i];
          }
        }
      } else {
        equal = fn ? fn(value1, value2) : value1 === value2;
      }

      return equal;
    };
  }
  /**
   * Returns the key to use in the Config `cache`.
   * @param {String} name The name of the config property.
   * @param {Object} options The config property options.
   * @returns {String}
   * @private
   */

  static makeCacheKey(name, options) {
    const keys = Object.keys(options).sort();

    for (let key, type, value, i = keys.length; i-- > 0;) {
      value = options[key = keys[i]];

      if (value == null && value === false) {
        keys.splice(i, 1);
      } else {
        type = typeof value;

        if (type === 'function') {
          return null;
        }

        if (type === 'string') {
          keys[i] = `${key}:"${value}"`;
        } else if (type === 'number') {
          keys[i] = `${key}:${value}`;
        } // that leaves bool and object, but there are no (valid) config options that are objects... so ignore

      }
    }

    return keys.length ? `${name}>${keys.join('|')}` : name; // eg: 'text>render|merge:v => v|bar'
  }
  /**
   * Creates and returns a property descriptor for this config suitable to be passed to `Reflect.defineProperty()`.
   * @returns {Object}
   * @private
   */

  makeDescriptor() {
    const config = this,
          {
      base,
      field,
      changer,
      updater,
      name
    } = config;

    if (base !== config && base.equal === config.equal) {
      // At present only the equal option affects the setter, so all configs can share the
      // descriptor of the base-most config definition unless their equality test fns differ.
      return base.descriptor;
    }

    return {
      get() {
        var _this$configObserver;

        // Allow folks like Widget.compose() to monitor getter calls
        (_this$configObserver = this.configObserver) === null || _this$configObserver === void 0 ? void 0 : _this$configObserver.get(name, this);
        return this[field];
      },

      set(value) {
        const me = this;
        let was = me[field],
            applied,
            newValue; // Resolve values starting with 'up.' by traversing owners to find it

        if (typeof value === 'string') {
          let resolvedValue = value;

          if (value.startsWith('up.')) {
            var _me$owner;

            resolvedValue = (_me$owner = me.owner) === null || _me$owner === void 0 ? void 0 : _me$owner.resolveProperty(value.substr(3));
          } else if (value.startsWith('this.')) {
            resolvedValue = me.resolveProperty(value.substr(5));
          } // TODO: Ignoring functions for now, we need to handle scope to make those work

          if (resolvedValue !== undefined && typeof resolvedValue !== 'function') {
            value = resolvedValue;
          }
        } // If the "changeTitle()" fellow falls off the end, it must have changed all the needful things.
        // Otherwise, it returned the final config value (it may have changed it instead, for example, making
        // an instance from a config object).

        if (me[changer]) {
          applied = (newValue = me[changer](value, was)) === undefined;

          if (!applied) {
            value = newValue;
            was = me[field]; // in case it was modified by the changer fn...
          }
        } // inline the default equal() for better perf:

        if (!applied && !(config.equal === equal ? was === value : config.equal(was, value))) {
          me[field] = value;
          applied = true; // Check for a "syncTitle()" method and call it if present.

          if (me[updater]) {
            me[updater](value, was);
          }
        }

        if (applied && !me.onConfigChange.$nullFn) {
          me.onConfigChange({
            name,
            value,
            was,
            config
          });
        }
      }

    };
  }
  /**
   * Creates and returns a property descriptor for this config's initter suitable to pass to
   * `Reflect.defineProperty()`.
   * @returns {Object}
   * @private
   */

  makeInitter() {
    const config = this;

    if (config !== config.base) {
      if (config.lazy) {
        return config.makeLazyInitter();
      } // At present no other options affect the setter, so all configs can share the descriptor of the base-most
      // config definition.

      return config.base.initDescriptor;
    }

    return config.makeBasicInitter();
  }

  makeBasicInitter() {
    const config = this,
          {
      initializing,
      name
    } = config;
    return {
      configurable: true,

      get() {
        const me = this;
        config.removeInitter(me); // Set the value from the configuration.

        me[initializing] = true;
        me[name] = me[configuringSymbol$1][name];
        me[initializing] = false; // The property has been *pulled* from the configuration.
        // Prevent the setting loop in configure from setting it again.

        me.configDone[name] = true; // Finally, allow the prototype getter to return the value.

        return me[name];
      },

      set(value) {
        config.removeInitter(this); // The config has been set (some internal code may have called the setter)
        // so prevent it from being called again and overwritten with data from initialConfig.

        this.configDone[name] = true; // Set the property normally (Any prototype setter will be invoked)

        this[name] = value;
      }

    };
  }

  makeLazyInitter() {
    const config = this,
          {
      initializing,
      name
    } = config;
    return {
      configurable: true,

      get() {
        const me = this,
              value = me[lazyConfigValues][name];
        config.removeInitter(me);

        if (!me.isDestroying) {
          // Set the value from the lazy config object.
          me[initializing] = true;
          me[name] = value;
          me[initializing] = false;
        } // Finally, allow the prototype getter to return the value.

        return me[name];
      },

      set(value) {
        config.removeInitter(this); // Set the property normally (Any prototype setter will be invoked)

        this[name] = value;
      }

    };
  }
  /**
   * Removes the property initter and restores the instance to its original form.
   * @param {Object} instance
   * @private
   */

  removeInitter(instance) {
    const {
      name
    } = this,
          instanceProperty = instance[instancePropertiesSymbol$1][name]; // If we took over from an instance property, replace it

    if (instanceProperty) {
      defineProperty$7(instance, name, instanceProperty);
    } // Otherwise just delete the instance property who's getter we are in.
    else {
      delete instance[name];
    }

    if (this.lazy) {
      if (! --instance[lazyConfigValues].$counter_) {
        delete instance[lazyConfigValues];
      }
    }
  }

  setDefault(cls, value) {
    defineProperty$7(cls.prototype, this.field, {
      configurable: true,
      writable: true,
      // or else "this._value = x" will fail
      value
    });
  }
  /**
   * This method combines (merges) two config values. This is called in two cases:
   *
   *  - When a derived class specifies the value of a config defined in a super class.
   *  - When a value is specified in the instance config object.
   *
   * @param {*} newValue In the case of derived classes, this is the config value of the derived class. In the case
   * of the instance config, this is the instance config value.
   * @param {*} currentValue In the case of derived classes, this is the config value of the super class. In the case
   * of the instance config, this is the class config value.
   * @param {Object} meta The class meta object from which the `newValue` is coming. This parameter is `null` if the
   * `newValue` is from an instance configuration.
   * @returns {*}
   * @internal
   */

  merge(newValue, currentValue) {
    if (currentValue && newValue && Objects.isObject(currentValue) && Objects.isObject(newValue)) {
      newValue = Objects.merge(Objects.clone(currentValue), newValue);
    }

    return newValue;
  }

}
const {
  prototype
} = Config,
      {
  equal
} = prototype;
Config.symbols = {
  configuring: configuringSymbol$1,
  instanceProperties: instancePropertiesSymbol$1,
  lazyConfigs: lazyConfigValues
};
/**
 * This object holds `Config` instances keyed by their name. For example:
 * ```
 *  Config.cache = {
 *      disabled : Config.get('disabled'),
 *      text     : Config.get('text'),
 *      title    : Config.get('title')
 *  };
 * ```
 * @member {Object} cache
 * @static
 * @private
 */

Config.cache = Object.create(null); // object w/no properties not even inherited ones

/**
 * This object holds config value equality methods. By default, the `===` operator is used to compare config values for
 * semantic equality. When an `equal` option is specified as a string, that string is used as a key into this object.
 *
 * All equality methods in this object have the same signature as the {@link #function-equal equal()} method.
 *
 * This object has the following equality methods:
 *
 * - `array` : Compares arrays of values using `===` on each element.
 * - `date` : Compares values of `Date` type.
 * - `strict` : The default equal algorithm based on `===` operator.
 * @member {Object} equalityMethods
 * @static
 * @private
 */

Config.equalityMethods = {
  array: Config.makeArrayEquals(),

  date(value1, value2) {
    if (value1 === value2) {
      return true;
    } // see DateHelper.isDate() but cannot import due to circularity

    if (value1 && value2 && toString$2.call(value1) === DATE_TYPE$1 && toString$2.call(value2) === DATE_TYPE$1) {
      // https://jsbench.me/ltkb3vk0ji/1 - getTime is >2x faster vs valueOf/Number/op+
      return value1.getTime() === value2.getTime();
    }

    return false;
  },

  strict: Config.equal = equal
};
/**
 * This object holds config value merge methods. By default, {@link Core.helper.ObjectHelper#function-merge-static} is
 * used to merge object's by their properties. Config merge methods are used to combine config values from derived
 * classes with config values from super classes, as well as instance config values with those of the class.
 *
 * All merge methods in this object have the same signature as the {@link #function-merge merge()} method.
 *
 * This object has the following merge methods:
 *
 * - `distinct`   : Combines arrays of values ensuring that no value is duplicated. When given an object, its truthy
 *   keys are included, while its falsy keys are removed from the result.
 * - `merge`      : The default merge algorithm for `configurable()` properties, based on
 *   {@link Core.helper.ObjectHelper#function-merge-static}.
 * - `items`      : Similar to `merge`, but allows reordering (see `Objects.mergeItems`).
 * - `objects`    : The same as to `merge` except this method promotes `true` to an empty object.
 * - 'classList'  : Incoming strings are converted to an object where the string is a property name with a truthy value.
 * - `replace`    : Always returns `newValue` to replace the super class value with the derived class value, or the
 *   class value with the instance value.
 * @member {Object} mergeMethods
 * @static
 * @internal
 */

Config.mergeMethods = {
  distinct(newValue, oldValue) {
    let ret = oldValue ? oldValue.slice() : [];

    if (newValue != null) {
      if (Objects.isObject(newValue)) {
        if (oldValue === undefined) {
          ret = newValue;
        } else {
          let key, index;

          for (key in newValue) {
            index = ret.indexOf(key);

            if (newValue[key]) {
              if (index < 0) {
                ret.push(key);
              }
            } else if (index > -1) {
              ret.splice(index, 1);
            }
          }
        }
      } else if (Array.isArray(newValue)) {
        newValue.forEach(v => !ret.includes(v) && ret.push(v));
      } else if (!ret.includes(newValue)) {
        ret.push(newValue);
      }
    }

    return ret;
  },

  merge: Config.merge = prototype.merge,

  classList(newValue, oldValue) {
    // 'foo bar' -> { foo : 1, bar : 1 }
    if (typeof newValue === 'string') {
      if (!newValue.length) {
        return oldValue;
      }

      newValue = newValue.split(whitespace).reduce(createClsProps, {});
    }

    return Config.merge(newValue, oldValue);
  },

  objects(newValue, oldValue) {
    return newValue === true ? oldValue || {} : Config.merge(newValue, oldValue);
  },

  replace(newValue) {
    return newValue;
  },

  items(newValue, oldValue) {
    return Objects.mergeItems(oldValue, newValue, {
      merge: (oldValue, newValue) => prototype.merge(newValue, oldValue)
    });
  }

};
Object.assign(prototype, {
  _descriptor: null,
  _initDescriptor: null,

  /**
   * A function that compares values for equality. This test is used to determine if the `update` method should be
   * called when the setter is invoked.
   *
   * To handle `Date` values:
   * ```
   *  class Foo extends Base {
   *      static get configurable() {
   *          return {
   *              date : {
   *                  $config : {
   *                      equal : 'date'
   *                  },
   *
   *                  value : null
   *              }
   *          }
   *      }
   *
   *      updateDate(date) {
   *          // date has changed
   *      }
   *  }
   * ```
   *
   * Also useful for some configs:
   * ```
   *  class Foo extends Base {
   *      static get configurable() {
   *          return {
   *              bar : {
   *                  $config : {
   *                      equal : ObjectHelper.isEqual
   *                  },
   *
   *                  value : null
   *              }
   *          }
   *      }
   *
   *      updateBar(value) {
   *          // value has changed
   *      }
   *  }
   * ```
   * @config {Function} equal
   * @internal
   */

  /**
   * Indicates that this config property should not automatically initialize during construction, but instead should
   * wait for the first explicit use of the property getter.
   * @config {Boolean}
   * @default
   * @internal
   */
  lazy: false,

  /**
   * Indicates that this config property should automatically be set to `null` on destroy.
   * @config {Boolean}
   * @default
   * @internal
   */
  nullify: false,

  /**
   * Indicates that this config participates in rendering. This has does not affect the behavior of the property
   * directly, but allows classes that perform rendering to detect which config changes will affect the rendered
   * result.
   * @config {Boolean}
   * @default
   * @internal
   */
  render: false // TODO FUTURES
  // clear   (bool - set to null during cleanup)
  // destroy (bool - destroy value during cleanup)
  // evented (bool - fire event on change / after updater)

});
Config._$name = 'Config';

class MetaClass {
  constructor(options) {
    options && Object.assign(this, options);
  }

  getInherited(name, create = true) {
    let ret = this[name]; // We use "in this" to allow the object to be set to null

    if (!(name in this)) {
      var _this$super;

      // If there is no object on this meta class, but there may be one on a super class. If create=false, the
      // idea is that a super class object will be "properly" inherited but otherwise no object will be created.
      ret = (_this$super = this.super) === null || _this$super === void 0 ? void 0 : _this$super.getInherited(name, create);

      if (ret || create) {
        this[name] = ret = Object.create(ret || null);
      }
    }

    return ret;
  }

}

const // Using Object.getPrototypeOf instead of Reflect.getPrototypeOf because:
// 1. The are almost the same, according to the MDN difference is handling getPrototypeOf('string')
// 2. It allows to pass security check in SalesForce environment
{
  getPrototypeOf: getPrototypeOf$1
} = Object,
      {
  defineProperty: defineProperty$6
} = Reflect,
      {
  hasOwnProperty: hasOwnProperty$7
} = Object.prototype,
      metaSymbol = Symbol('classMetaData'),
      mixinTagSymbol = Symbol('mixinTag'),
      originalConfigSymbol = Symbol('originalConfig'),
      configuringSymbol = Config.symbols.configuring,
      instancePropertiesSymbol = Config.symbols.instanceProperties,
      lazyConfigsSymbol = Config.symbols.lazyConfigs,
      defaultConfigOptions = {
  merge: 'replace',
  simple: true
},
      emptyFn$3 = () => {},
      newMeta = o => new MetaClass(o),
      setupNames = {
  /* foo : 'setupFoo' */
},
      emptyObject$c = Object.freeze({}),
      emptyArray$a = Object.freeze([]);
/**
 * Base class for all configurable classes.
 *
 * Subclasses do not have to implement a constructor with its restriction of having to call super()
 * before there is a `this` reference. Subclasses instead implement a `construct` method which is
 * called by the `Base` constructor. This may call its `super` implementation at any time.
 *
 * The `Base` constructor applies all configs to properties of the new instance. The instance
 * will have been configured after the `super.construct(config)` is called.
 *
 * See the Class System documentation in the guides for more information.
 *
 * @abstract
 */

class Base$1 {
  static get isBase() {
    return true;
  }

  get isBase() {
    return true;
  } // defaultConfig & properties made private to not spam all other classes

  /**
   * A class property getter to add additional, special class properties.
   *
   * For example, a class adds a `declarable` class property like so:
   * ```
   *  class Something extends Base {
   *      static get declarable() {
   *          return ['extra'];
   *      }
   *
   *      static setupExtra(cls, meta) {
   *          // use cls.extra
   *      }
   *  }
   * ```
   * A derived class can then specify this property like so:
   * ```
   *  class Derived extends Something {
   *      static get extra() {
   *          // return extra information
   *      }
   *  }
   * ```
   * When the `Derived` class is initialized, the `setupExtra()` method is called and `Derived` is passed as the
   * argument. It is also the `this` pointer, but the parameter is minifiable. The second argument passed is the
   * `$meta` object for the class.
   *
   * Classes are initialized at the first occurrence of the following:
   *
   * - An instance is created
   * - The class `$meta` property is accessed
   *
   * @member {String[]} declarable
   * @static
   * @category Configuration
   * @internal
   */

  static get declarable() {
    return ['declarable',
    /**
     * A class property getter for the configuration properties of the class, which can be overridden by
     * configurations passed at construction time.
     *
     * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
     * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
     * individually and then combine them with those of derived classes.
     *
     * For example, a `Label` might declare a `text` config like so:
     * ```javascript
     *  class Label extends Base {
     *      static get configurable() {
     *          return {
     *              text : null
     *          };
     *      }
     *  }
     * ```
     * The `text` config is automatically inherited by classes derived from Label. By implementing
     * `get configurable()`, derived classes can change the default value of inherited configs, or define new
     * configs, or both.
     *
     * When a config property is declared in this way, the class author can also implement either of two
     * special methods that will be called when the config property is assigned a new value:
     *
     *  - `changeText()`
     *  - `updateText()`
     *
     * In the example above, the `Label` class could implement a `changeText()` method, an `updateText()`
     * method, or both. The generated property setter ensures these methods will be called when the `text`
     * property is assigned.
     *
     * The generated setter (for `text` in this example) performs the following steps:
     *
     *  - If the class defines a `changeText()` method, call it passing the new value and the current value:
     *    `changeText(newText, oldText)`.<br>
     *    Then:
     *    * If `changeText()` exits without returning a value (i.e., `undefined`), exit and do nothing
     *      further. The assumption is that the changer method has done all that is required.
     *    * Otherwise, the return value of `changeText()` replaces the incoming value passed to the setter.
     *  - If the new value (or the value returned by `changeText()`) is `!==` to the current value:
     *    * Update the stored config value in `this._text`.
     *    * If the class defines an `updateText()` method, call it passing the new value and the previous value.
     *      `updateText(newText, oldText)`
     *
     * #### Resolving a value from an owner
     * By specifying a value starting with `'up.'` for a config, the config system will resolve that value by
     * examining the ownership hierarchy. It will walk up the hierarchy looking for a property matching the name
     * (or dot separated path) after 'up.'. If one is found, the value will be read and used as the initial
     * value.
     *
     * ```javascript
     * class Parent extends Base {
     *     static get configurable() {
     *         return [
     *           'importantValue'
     *         ]
     *     }
     * }
     *
     * class Child extends Base {
     *     static get configurable() {
     *         return [
     *           'value'
     *         ]
     *     }
     * }
     *
     * const parent = new Parent({
     *     importantValue : 123
     * });
     *
     * const child = new Child({
     *     owner : parent,
     *     // Will be resolved from the owner
     *     value : 'up.importantValue'
     * });
     *
     * console.log(child.value); // logs 123
     * ```
     * Please note that this is for now a one way one time binding, the value will only be read initially and
     * not kept up to date on later changes.
     *
     * #### Value Merging
     * When a config property value is an object, the value declared by the base class is merged with values
     * declared by derived classes and the value passed to the constructor.
     * ```javascript
     *  class Example extends Base {
     *      static get configurable() {
     *          return {
     *              config : {
     *                  foo : 1,
     *                  bar : 2
     *              }
     *          };
     *      }
     *  }
     *
     *  class Example2 extends Example {
     *      static get configurable() {
     *          return {
     *              config : {
     *                  bar : 42,
     *                  zip : 'abc'
     *              }
     *          };
     *      }
     *  }
     *
     *  let ex = new Example2({
     *      config : {
     *          zip : 'xyz'
     *      }
     *  });
     * ```
     * The result of the merge would set `config` to:
     * ```javascript
     *  ex.foo = {
     *      foo : 1,    // from Example
     *      bar : 42,   // from Example2
     *      zip : 'xyz' // from constructor
     *  }
     * ```
     *
     * #### Config Options
     * Some config properties require additional options such as declarative information about the config that
     * may be useful to automate some operation. Consider a `Button`. It could declare that its `text` config
     * affects the rendered HTML by applying a `render` property to the config definition. Its base class could
     * then examine the config definition to find this property.
     *
     * To support this, config options ca be declared like so:
     * ```javascript
     *  class Button extends Widget {
     *      static get configurable() {
     *          return {
     *              text : {
     *                  value   : null,
     *                  $config : {
     *                      render : true
     *                  }
     *              }
     *          };
     *      }
     *  }
     * ```
     * The `$config` property can alternatively be just the names of the options that should be enabled (set
     * to `true`).
     *
     * For example, the following is equivalent to the above:
     * ```javascript
     *  class Button extends Widget {
     *      static get configurable() {
     *          return {
     *              text : {
     *                  value   : null,
     *                  $config : 'render'
     *              }
     *          };
     *  }
     * ```
     *
     * #### Default Value
     * It is common to set a config to a `null` value to take advantage of internal optimizations for `null`
     * values. In most cases the fact that this produces `undefined` as the actual initial value of the config
     * is acceptable. When this is not acceptable, a config can be declared like so:
     * ```javascript
     *  class Widget {
     *      static get configurable() {
     *          return {
     *              disabled : {
     *                  $config : null,
     *                  value   : null,
     *                  default : false
     *              }
     *          };
     *  }
     * ```
     * The `default` property above determines the value of the config while still gaining the benefits of
     * minimal processing due to the `null` value of the `value` property.
     * @member {Object} configurable
     * @static
     * @category Configuration
     * @internal
     */
    'configurable',
    /**
     * A class property getter for the default configuration of the class, which can be overridden by
     * configurations passed at construction time.
     *
     * Unlike a normal `static` property, this property is only ever used for the class that defines it (as in,
     * `hasOwnProperty`). It is retrieved for all classes in a class hierarchy, to gather their configs
     * individually and then combine them with those of derived classes.
     *
     * For example, a `Store` might declare its `url` config like so:
     * ```
     *  class Store extends Base {
     *      static get defaultConfig() {
     *          return {
     *              url : null
     *          };
     *      }
     *  }
     * ```
     * The `url` config is automatically inherited by classes derived from Store. By implementing
     * `get defaultConfig()`, derived classes can change the default value of inherited configs, or define new
     * configs, or both. When defining new configs, however, `configurable` is preferred.
     *
     * Config properties introduced to a class by this declaration do not participate in value merging and do
     * not get a generated setter. Config properties introduced by a base class using `configurable` can be
     * set to a different value using `defaultConfig` and in doing so, the values will be merged as appropriate
     * for `configurable`.
     *
     * @member {Object} defaultConfig
     * @static
     * @category Configuration
     * @internal
     */
    'defaultConfig',
    /**
     * A class property getter for the default values of internal properties for this class.
     * @member {Object} properties
     * @static
     * @category Configuration
     * @internal
     */
    'properties',
    /**
     * A class property getter for properties that will be applied to the class prototype.
     * @member {Object} prototypeProperties
     * @static
     * @category Configuration
     * @internal
     */
    'prototypeProperties'];
  }
  /**
   * Base constructor, passes arguments to {@link #function-construct}.
   * @param {...Object} [args] Usually called with a config object, but accepts any params
   * @function constructor
   * @category Lifecycle
   */

  constructor(...args) {
    const me = this,
          C = me.constructor;

    if (me.$meta.class !== C) {
      // This will happen only once for each class. We need to call the C.$meta getter which puts $meta on our
      // prototype. Since that alone would be optimized away (and would generate IDE and lint warnings), we call
      // emptyFn and simply pass the value.
      emptyFn$3(C.$meta);
    } // Allow subclasses to have a pseudo constructor with "this" already set:

    me.construct(...args);
    me.afterConstruct();
    me.isConstructing = false;
  }
  /**
   * Factory version of the Base constructor. Merges all arguments to create a config object that is passed along to
   * the constructor.
   * @param {...Object} [configs] Allows passing multiple config objects
   * @returns {Core.Base} New instance
   * @private
   */

  static new(...configs) {
    return new this(Objects.merge({}, ...configs));
  }
  /**
   * Base implementation applies configuration.
   *
   * Subclasses need only implement this if they have to initialize instance specific
   * properties required by the class. Often a `construct` method is
   * unnecessary. All initialization of incoming configuration properties can be
   * done in a `set propName` implementation.
   * @param {...Object} [args] Usually called with a config object, but accepts any params
   * @category Lifecycle
   */

  construct(...args) {
    // Passing null to base construct means bypass the config system and stack creation (to gain performance)
    if (args[0] !== null) {
      this.configure(...args);
    }

    this.afterConfigure();
  }
  /**
   * Destroys the provided objects by calling their {@link #function-destroy} method.
   * Skips empty values or objects that are already destroyed.
   *
   * ```javascript
   * Base.destroy(myButton, toolbar1, helloWorldMessageBox);
   * ```
   * @param {...Object} [args] Objects to be destroyed
   * @category Lifecycle
   */

  static destroy(...args) {
    const shredder = object => {
      if (object !== null && object !== void 0 && object.destroy) {
        object.destroy();
      } else if (Array.isArray(object)) {
        object.forEach(shredder);
      }
    };

    shredder(args);
  }
  /**
   * Destroys this object. This is primarily accomplished by calling {@link #function-doDestroy}, however, prior to
   * calling `doDestroy`, {@link #property-isDestroying} is set to `true`. After {@link #function-doDestroy} returns,
   * {@link #property-isDestroyed} is set to `true`.
   *
   * Do not override this method in subclasses. To provide class-specific cleanup, implement {@link #function-doDestroy}
   * instead.
   * @category Lifecycle
   */

  destroy() {
    const me = this,
          {
      id
    } = me; // Let everyone know the object is going inert:

    me.isDestroying = true; // Make calling destroy() harmless:

    me.destroy = emptyFn$3;
    me.doDestroy();
    Object.setPrototypeOf(me, null); // Clear all remaining instance properties.

    for (const key in me) {
      if (key !== 'destroy' && key !== 'isDestroying') {
        delete me[key];
      }
    }

    delete me[originalConfigSymbol]; // Let everyone know the object is inert:

    me.isDestroyed = true;
    me.id = id; // for diagnostic reasons
  }
  /**
   * This method is required to help `unused` getters to survive production build process. Some tools, like angular,
   * will remove `unused` code in production build, making our side-effected getters behind, breaking code heavily.
   * @internal
   * @param getter Getter to evaluate
   */

  _thisIsAUsedExpression(getter) {}

  static get $$name() {
    return hasOwnProperty$7.call(this, '$name') && this.$name || // _$name is filled by webpack for every class (cls._$name = '...')
    hasOwnProperty$7.call(this, '_$name') && this._$name || this.name;
  }

  get $$name() {
    return this.constructor.$$name;
  }
  /**
   * Base implementation so that all subclasses and mixins may safely call super.startConfigure.
   *
   * This is called by the Base class before setting configuration properties, but after
   * the active initial getters have been set, so all configurations are available.
   *
   * This method allows all classes in the hierarchy to force some configs to be evaluated before others.
   * @internal
   * @category Lifecycle
   * @params {Object} config The configuration object use to set the initial state.
   */

  startConfigure(config) {}
  /**
   * Base implementation so that all subclasses and mixins may safely call super.finishConfigure.
   *
   * This is called by the Base class before exiting the {@link #function-configure} method.
   *
   * At this point, all configs have been applied, but the `isConfiguring` property is still set.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * into the config phase.
   * @internal
   * @category Lifecycle
   * @params {Object} config The configuration object use to set the initial state.
   */

  finishConfigure(config) {}
  /**
   * Base implementation so that all subclasses and mixins may safely call `super.afterConfigure`. This is called by the Base class after the {@link #function-configure} method has been
   * called. At this point, all configs have been applied.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * either before or after the super.afterConstruct();
   * @internal
   * @category Lifecycle
   */

  afterConfigure() {}
  /**
   * Base implementation so that all subclasses and mixins may safely call super.afterConstruct.
   *
   * This is called by the Base class after the {@link #function-construct} method has been
   * called.
   *
   * At this point, all configs have been applied.
   *
   * This method allows all classes in the hierarchy to inject functionality
   * either before or after the super.afterConstruct();
   * @internal
   * @function afterConstructor
   * @category Lifecycle
   */

  afterConstruct() {}
  /**
   * Provides a way of calling callbacks which may have been specified as the _name_ of a function
   * and optionally adds scope resolution.
   *
   * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
   * then the function is resolved in this object. This is useful when configuring listeners
   * at the class level.
   *
   * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
   * using the `owner` property until an object with the named function is present, then the
   * named function is called upon that object.
   *
   * If a named function is not found, an error is thrown. If the function should be only called when present,
   * and may not be present, add a `?` as a suffix.
   *
   * @param {String|Function} fn The function to call, or the name of the function to call.
   * @param {Object} thisObject The `this` object of the function.
   * @param {Object[]} args The argument list to pass.
   * @category Misc
   */

  callback(fn, thisObject, args = emptyArray$a) {
    // Maintainer: do not make args ...args. This method may acquire more arguments
    const {
      handler,
      thisObj
    } = this.resolveCallback(fn, thisObject === 'this' ? this : thisObject) || emptyObject$c;
    return handler === null || handler === void 0 ? void 0 : handler.apply(thisObj, args);
  }

  resolveProperty(propertyPath) {
    let thisObj = this;

    while (thisObj) {
      if (Objects.hasPath(thisObj, propertyPath)) {
        return Objects.getPath(thisObj, propertyPath);
      }

      thisObj = thisObj.owner;
    }

    return undefined;
  }
  /**
   * Provides a way of locating callbacks which may have been specified as the _name_ of a function
   * and optionally adds scope resolution.
   *
   * For example, if the callback is specified as a string, then if it is prefixed with `'this.'`
   * then the function is resolved in this object. This is useful when configuring listeners
   * at the class level.
   *
   * If the callback name is prefixed with `'up.'` then the ownership hierarchy is queried
   * using the `owner` property until an object with the named function is present, then the
   * named function is called upon that object.
   * @param {String|Function} handler The function to call, or the name of the function to call.
   * @param {Object} thisObj The `this` object of the function.
   * @param {Boolean} [enforceCallability = true] Pass `false` if the function may not exist, and a null return value is acceptable.
   * @returns {Object} `{ handler, thisObj }`
   * @category Misc
   */

  resolveCallback(handler, thisObj = this, enforceCallability = true) {
    // It's a string, we find it in its own thisObj
    if (handler.substring) {
      if (handler.endsWith('?')) {
        enforceCallability = false;
        handler = handler.substring(0, handler.length - 1);
      }

      if (handler.startsWith('up.')) {
        handler = handler.substring(3); // Empty loop until we find the function owner

        for (thisObj = this.owner; thisObj && !thisObj[handler]; thisObj = thisObj.owner);

        if (!thisObj) {
          return;
        }
      } else if (handler.startsWith('this.')) {
        handler = handler.substring(5);
        thisObj = this;
      }

      if (!thisObj || !(thisObj instanceof Object)) {
        return;
      }

      if (typeof thisObj[handler] !== 'function') {
        return;
      }

      handler = thisObj[handler];
    }

    return {
      handler,
      thisObj
    };
  }

  bindCallback(inHandler, inThisObj = this) {
    if (inHandler) {
      const {
        handler,
        thisObj
      } = this.resolveCallback(inHandler, inThisObj);

      if (handler) {
        return handler.bind(thisObj);
      }
    }
  }
  /**
   * Delays the execution of the passed function by the passed time quantum, or if the time is omitted
   * or not a number, delays until the next animation frame. Note that this will use
   * {@link Core.mixin.Delayable#function-setTimeout} || {@link Core.mixin.Delayable#function-requestAnimationFrame}
   * if this class mixes in `Delayable`, otherwise it uses the global methods. The function will
   * be called using `this` object as its execution scope.
   * @param {Function} fn The function to call on a delay.
   * @param {Number} [by] The number of milliseconds to delay.
   * @returns {Number} The created timeout id.
   * @private
   */

  delay(fn, by) {
    // Force scope on the fn if we are not a Delayable
    fn = this.setTimeout ? fn : fn.bind(this);
    const invoker = this.setTimeout ? this : globalThis;
    return invoker[typeof by === 'number' ? 'setTimeout' : 'requestAnimationFrame'](fn, by);
  }
  /**
   * Classes implement this method to provide custom cleanup logic before calling `super.doDestroy()`. The general
   * pattern is as follows:
   *
   * ```javascript
   *  class Foo extends Base {
   *      doDestroy() {
   *          // perform custom cleanup
   *
   *          super.doDestroy();
   *      }
   *  }
   * ```
   *
   * This method is called by {@link #function-destroy} which also prevents multiple calls from reaching `doDestroy`.
   * Prior to calling `doDestroy`, {@link #property-isDestroying} is set to `true`. Upon return, the object is fully
   * destructed and {@link #property-isDestroyed} is set to `true`.
   *
   * Do not call this method directly. Instead call {@link #function-destroy}.
   * @category Lifecycle
   */

  doDestroy() {
    const me = this,
          {
      nullify
    } = me.$meta;

    if (nullify) {
      for (let i = 0; i < nullify.length; ++i) {
        if (me[nullify[i].field] != null) {
          // if backing property is null/undefined then skip
          me[nullify[i].name] = null; // else, call setter to run through change/update
        }
      }
    }
  }
  /**
   * Destroys the named properties if they have been initialized, and if they have a `destroy` method.
   * Deletes the property from this object. For example:
   *
   *      this.destroyProperties('store', 'resourceStore', 'eventStore', 'dependencyStore', 'assignmentStore');
   *
   * @param {String} properties The names of the properties to destroy.
   * @internal
   * @category Lifecycle
   */

  destroyProperties(...properties) {
    const me = this;
    let key, propertyValue;

    for (key of properties) {
      // If the value has *not* been pulled in from the configuration object yet
      // we must not try to access it, as that will cause the property to be initialized.
      if (key in me && (!me[configuringSymbol] || !me[configuringSymbol][key])) {
        var _propertyValue;

        propertyValue = me[key];

        if ((_propertyValue = propertyValue) !== null && _propertyValue !== void 0 && _propertyValue.destroy) {
          propertyValue.destroy();
        }

        delete me[key];
      }
    }
  }
  /**
   * Called by the Base constructor to apply configs to this instance. This must not be called.
   * @param {Object} config The configuration object from which instance properties are initialized.
   * @private
   * @category Lifecycle
   */

  configure(config = {}) {
    const me = this,
          {
      beforeConfigure
    } = config,
          configs = me.$meta.configs,
          fullConfig = me.getDefaultConfiguration();
    let cfg, key, value;
    me.initialConfig = config; // Important flag for setters to know whether they are being called during
    // configuration when this object is not fully alive, or whether it's being reconfigured.

    me.isConfiguring = true; // Assign any instance properties declared by the class.

    Object.assign(me, me.getProperties()); // Apply configuration to default from class definition. This is safe because it's either chained from or a
    // fork of the class values.

    for (key in config) {
      value = config[key];
      cfg = configs[key];
      fullConfig[key] = cfg ? cfg.merge(value, fullConfig[key], null) : value;
    }

    if (beforeConfigure) {
      delete fullConfig.beforeConfigure; // noinspection JSValidateTypes

      beforeConfigure(me, fullConfig);
    } // Cache me.config for use by get config.

    me.setConfig(me[originalConfigSymbol] = fullConfig, true);
    me.isConfiguring = false;
  }
  /**
   * Returns the value of the specified config property. This is a method to allow
   * property getters to be explicitly called in a way that does not get optimized out.
   *
   * The following triggers the getter call, but optimizers will remove it:
   *
   *      inst.foo;   // also raises "expression has no side-effects" warning
   *
   * Instead, do the following to trigger a getter:
   *
   *      inst.getConfig('foo');
   *
   * @param {String} name
   * @internal
   * @category Configuration
   */

  getConfig(name) {
    return this[name];
  }
  /**
   * Sets configuration options this object with all the properties passed in the parameter object.
   * Timing is taken care of. If the setter of one config is called first, and references
   * the value of another config which has not yet been set, that config will be set just
   * in time, and the *new* value will be used.
   * @param {Object} config An object containing configurations to change.
   * @category Lifecycle
   */

  setConfig(config, isConstructing) {
    const me = this,
          wasConfiguring = me[configuringSymbol],
          configDone = wasConfiguring ? me.configDone : me.configDone = {},
          configs = me.$meta.configs;
    let cfg, key;
    me[instancePropertiesSymbol] = {}; // Cache configuration for use by injected property initializers.

    me[configuringSymbol] = wasConfiguring ? Object.setPrototypeOf(Object.assign({}, config), wasConfiguring) : config; // For each incoming non-null configuration, create a temporary getter which will
    // pull the value in from the initialConfig so that it doesn't matter in
    // which order properties are set. You can access any property at any time.

    for (key in config) {
      // Don't default null configs in unless it's a direct property of the
      // the passed configuration. When used at construct time, defaultConfigs
      // will be prototype-chained onto the config.
      if (config[key] != null || hasOwnProperty$7.call(config, key)) {
        cfg = configs[key] || Config.get(key);
        cfg.defineInitter(me, config[key]);

        if (!isConstructing) {
          configDone[key] = false;
        } else if (cfg.lazy) {
          configDone[key] = true;
        }
      } else {
        configDone[key] = true;
      }
    }

    if (isConstructing) {
      me.startConfigure(config);
    } // Set all our properties from the config object.
    // If one of the properties needs to access a property that has not
    // yet been set, the above temporary property will pull it through.
    // Can't use Object.assign because that only uses own properties.
    // config value blocks are prototype chained subclass->superclass

    for (key in config) {
      // Only push the value through if the property initializer is still present.
      // If it gets triggered to pull the configuration value in, it deleted itself.
      if (!configDone[key]) {
        me[key] = config[key];
      }
    }

    if (wasConfiguring) {
      me[configuringSymbol] = wasConfiguring;
    } else {
      delete me[configuringSymbol];
    }

    if (isConstructing) {
      me.finishConfigure(config);
    }

    return me;
  }
  /**
   * Returns `true` if this instance has a non-null value for the specified config. This will not activate a lazy
   * config.
   *
   * @param {String} name The name of the config property.
   * @returns {Boolean}
   * @internal
   */

  hasConfig(name) {
    const me = this,
          lazyConfig = me[lazyConfigsSymbol],
          config = me[configuringSymbol],
          hasValue = me['_' + name] != null,
          hasLazyValue = (lazyConfig === null || lazyConfig === void 0 ? void 0 : lazyConfig[name]) != null,
          hasConfigValue = Boolean(!me.configDone[name] && config && (config[name] != null || hasOwnProperty$7.call(config, name)));
    return hasValue || hasLazyValue || hasConfigValue;
  }
  /**
   * Returns the value of an uningested config *without* ingesting the config or transforming
   * it from its raw value using its `changeXxxxx` method.
   *
   * @param {String} name The name of the config property.
   * @returns {*} The raw incoming config value.
   * @internal
   */

  peekConfig(name) {
    const me = this,
          lazyConfig = me[lazyConfigsSymbol],
          config = me[configuringSymbol]; // It's waiting in the lazy configs

    if ((lazyConfig === null || lazyConfig === void 0 ? void 0 : lazyConfig[name]) != null) {
      return lazyConfig[name];
    } // It's been read in, so use the current value

    if (me.configDone[name]) {
      return me[name];
    } // Now we have to check the incoming configs and lazyConfigs

    if (config[name] != null || hasOwnProperty$7.call(config, name)) {
      return config[name];
    }
  }
  /**
   * Ensures that the specified config is initialized if it is needed. If there is a config value specified, and it
   * was initialized by this call, this method returns `true`. If there was a config value specified, and it was
   * already initialized, this method returns `false`. If there was no value specified for the given config, this
   * method returns `null`.
   *
   * This is different than just reading the property because some property getters exist that do not actually just
   * read the config value back, but instead produce some result. Reading such properties to incidentally trigger a
   * possible config initializer can lead to incorrect results. For example, the Combo items config.
   *
   * @param {String} name The name of the config property.
   * @returns {Boolean}
   * @internal
   */

  triggerConfig(name) {
    const {
      configDone
    } = this,
          config = this[configuringSymbol],
          triggered = config && (config[name] != null || hasOwnProperty$7.call(config, name)) ? !configDone[name] : null;

    if (triggered) {
      this.getConfig(name);
    }

    return triggered;
  }

  onConfigChange() {} // declared above because lint/IDE get angry about not declaring the args...

  /**
   * This method is called when any config changes.
   * @param {Object} info Object containing information regarding the config change.
   * @param {String} info.name The name of the config that changed.
   * @param {*} info.value The new value of the config.
   * @param {*} info.was The previous value of the config.
   * @param {Core.Config} info.config The `Config` object for the changed config property.
   * @method onConfigChange
   * @internal
   * @category Configuration
   */

  /**
   * Returns a *copy* of the full configuration which was used to configure this object.
   * @property {Object}
   * @category Lifecycle
   * @readonly
   */

  get config() {
    const result = {},
          myConfig = this[originalConfigSymbol]; // The configuration was created as a prototype chain of the class hierarchy's
    // defaultConfig values hanging off a copy of the initialConfig object, so
    // we must loop and copy since Object.assign only copies own properties.

    for (const key in myConfig) {
      result[key] = myConfig[key];
    }

    return result;
  } // region Extract config
  // Recursively get the value of a config. Only intended to be called by getCurrentConfig()

  getConfigValue(name, options) {
    var _this$$meta$configs$n, _this$lazyConfigsSymb;

    function getValue(currentValue) {
      if (currentValue === globalThis) {
        return globalThis;
      } else if (Array.isArray(currentValue)) {
        return currentValue.map(getValue);
      } // Not using isBase to avoid classes (modelClass for example)
      else if (currentValue instanceof Base$1) {
        if (options.visited.has(currentValue)) {
          return;
        }

        return currentValue.getCurrentConfig(options);
      } // appendTo, floatRoot etc
      else if (currentValue instanceof HTMLElement || currentValue instanceof DocumentFragment) {
        return null;
      } // Go deeply into objects, might have instances of our classes in them
      else if (typeOf(currentValue) === 'object') {
        const result = {};

        for (const key in currentValue) {
          result[key] = getValue(currentValue[key]);
        }

        return result;
      }

      return currentValue;
    } // Do not trigger lazy configs

    if (!((_this$$meta$configs$n = this.$meta.configs[name]) !== null && _this$$meta$configs$n !== void 0 && _this$$meta$configs$n.lazy)) {
      return getValue(this[name]);
    } // Instead pull their initial config in
    else if ((_this$lazyConfigsSymb = this[lazyConfigsSymbol]) !== null && _this$lazyConfigsSymb !== void 0 && _this$lazyConfigsSymb[name]) {
      return getValue(this[lazyConfigsSymbol][name]);
    }
  } // Allows removing / adding configs before values are extracted

  preProcessCurrentConfigs() {} // Extract the current values for all initially used configs, in a format that can be used to create a new instance.
  // Not intended to be called by any other code than getConfigString()

  getCurrentConfig(options = {}) {
    const me = this,
          configs = options.configs === 'all' ? me.config : Objects.clone(me.initialConfig),
          visited = options.visited || (options.visited = new Set()),
          depth = options.depth || (options.depth = 0),
          result = {};

    if (visited.has(me)) {
      return undefined;
    }

    visited.add(me);
    this.preProcessCurrentConfigs(configs);

    for (const name in configs) {
      const value = me.getConfigValue(name, _objectSpread2(_objectSpread2({}, options), {}, {
        depth: depth + 1
      }));

      if (value !== undefined) {
        result[name] = value;
      }
    }

    return result;
  } // Extract the current values for all initially used configs and convert them to a JavaScript string

  getConfigString(options) {
    return StringHelper.toJavaScriptString(this.getCurrentConfig(options));
  } // Experimental helper function, extracts the currently used configs and wraps them as an app, returning code as a
  // string.
  //
  // This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
  //

  getTestCase(options = {}) {
    //<remove-on-lwc-release>
    const Product = this.isGantt ? 'Gantt' : this.isSchedulerPro ? 'SchedulerPro' : this.isCalendar ? 'Calendar' : this.isScheduler ? 'Scheduler' : this.isGrid ? 'Grid' : this.isTaskBoard ? 'TaskBoard' : null;

    if (Product) {
      const product = Product.toLowerCase(),
            // bundlePath = `https://bryntum.com/dist/${product}/build/${product}.module.js`,
      bundlePath = `../../build/${product}.module.js`;
      let preamble, postamble;

      if (options.import === 'static') {
        preamble = `import * as module from "${bundlePath}";` + 'Object.assign(window, module);'; // for (const c in module) window[c] = module[c];

        postamble = '';
      } else {
        preamble = `import("${bundlePath}").then(module => { Object.assign(window, module);\n`;
        postamble = '});';
      }

      const version = VersionHelper.getVersion(product);

      if (version) {
        preamble += `\nconsole.log('${Product} ${version}');\n`;
      } // De-indented on purpose

      return `${preamble}      
const ${product} = new ${Product}(${this.getConfigString(options)});
${postamble}`;
    } //</remove-on-lwc-release>

  }
  /**
   * Experimental helper function, extracts the currently used configs and wraps them as an app, downloading the
   * resulting JS file.
   *
   * This function is intended to simplify creating test cases for issue reporting on Bryntum's support forum.
   */

  downloadTestCase(options = {}) {
    options.output = 'return';
    const app = this.getTestCase(options);
    BrowserHelper.download(`app.js`, 'data:application/javascript;charset=utf-8,' + escape(app));
  } //endregion

  /**
   * Registers this class type with its Factory
   */

  static initClass() {
    return this.$meta.class;
  }
  /**
   * The class's {@link #property-$meta-static meta} object.
   * @member {Object} $meta
   * @internal
   * @category Misc
   */

  /**
   * An object owned by this class that does not share properties with its super class.
   *
   * This object may contain other properties which are added as needed and are not documented here.
   *
   * @property {Object} $meta The class meta object.
   * @property {Function} $meta.class The class constructor that owns the meta object.
   * @property {Object} $meta.super The `$meta` object for the super class. This is `null` for `Base`.
   * @property {Object} $meta.config The object holding the default configuration values for this class.
   * @property {Object} $meta.configs An object keyed by config name that holds the defined configs for the class.
   * The value of each property is a {@link Core/Config} instance.
   * @property {Boolean} $meta.forkConfigs This will be `true` if the default configuration values for this class
   * (in the `config` property of the meta object) must be forked to avoid object sharing, or if the object can be
   * passed to `Object.create()` for efficiency.
   * @property {Function[]} $meta.hierarchy The array of classes in the ancestry of this class. This will start with
   * `Base` at index 0 and ends with this class.
   * @property {Function[]} $meta.properties The array of classes that define a "static get properties()" getter.
   * @internal
   * @static
   * @category Misc
   */

  static get $meta() {
    const me = this;
    let meta = me[metaSymbol];

    if (!hasOwnProperty$7.call(me, metaSymbol)) {
      me[metaSymbol] = meta = newMeta();
      meta.class = me;
      me.setupClass(meta);
    }

    return meta;
  }
  /**
   * This optional class method is called when a class is mixed in using the {@link #function-mixin-static mixin()}
   * method.
   * @internal
   */

  static onClassMixedIn() {// empty
  }
  /**
   * Returns the merge of the `baseConfig` and `config` config objects based on the configs defined by this class.
   * @param {Object} baseConfig The base config or defaults.
   * @param {...Object} configs One or more config objects that takes priority over `baseConfig`.
   * @returns {Object}
   * @internal
   */

  static mergeConfigs(baseConfig, ...configs) {
    const classConfigs = this.$meta.configs,
          result = Objects.clone(baseConfig) || {};
    let config, i, key, value;

    for (i = 0; i < configs.length; ++i) {
      config = configs[i];

      if (config) {
        for (key in config) {
          value = config[key];

          if (classConfigs[key]) {
            value = classConfigs[key].merge(value, result[key]);
          } else if (result[key] && value) {
            value = Config.merge(value, result[key]);
          }

          result[key] = value;
        }
      }
    }

    return result;
  }
  /**
   * Applies one or more `mixins` to this class and returns the produced class constructor.
   *
   * For example, instead of writing this:
   * ```
   *  class A extends Delayable(Events(Localizable(Base))) {
   *      // ...
   *  }
   * ```
   *
   * Using this method, one would write this:
   * ```
   *  class A extends Base.mixin(Localizable, Events, Delayable) {
   *      // ...
   *  }
   * ```
   * If one of the mixins specified has already been mixed into the class, it will be ignored and not mixed in a
   * second time.
   * @param {...Function} mixins
   * @returns {Function}
   */

  static mixin(...mixins) {
    // Starting w/the first class C = this
    let C = this,
        i; // wrap each class C using mixins[i] to produce the next class

    for (i = 0; i < mixins.length; ++i) {
      const mixin = mixins[i],
            // Grab or create a unique Symbol for this mixin so we can tell if we've already mixed it in
      tag = mixin[mixinTagSymbol] || (mixin[mixinTagSymbol] = Symbol('mixinTag'));

      if (C[tag]) {
        continue;
      }

      C = mixin(C);
      C[tag] = true; // properties on the constructor are inherited to subclass constructors...

      if (hasOwnProperty$7.call(C, 'onClassMixedIn')) {
        C.onClassMixedIn();
      }
    }

    return C;
  }
  /**
   * This method is called only once for any class. This can occur when the first instance is created or when the
   * `$meta` object is first requested.
   * @param {Object} meta The `$meta` object for the class.
   * @internal
   * @category Misc
   */

  static setupClass(meta) {
    var _base$nullify;

    const cls = meta.class,
          // Trigger setupClass on the super class (if it has yet to happen):
    base = getPrototypeOf$1(cls).$meta,
          name = cls.$$name,
          names = base.names,
          proto = cls.prototype;
    defineProperty$6(proto, '$meta', {
      value: meta
    });
    Object.assign(meta, {
      super: base,
      config: Object.create(base.config),
      configs: Object.create(base.configs),
      declarables: base.declarables,
      forkConfigs: base.forkConfigs,
      hierarchy: Object.freeze([...base.hierarchy, cls]),
      names: names.includes(name) ? names : Object.freeze([...names, name]),
      properties: base.properties,
      nullify: (_base$nullify = base.nullify) === null || _base$nullify === void 0 ? void 0 : _base$nullify.slice()
    });

    if (names !== meta.names) {
      const is = 'is' + name;

      if (!hasOwnProperty$7.call(proto, is)) {
        defineProperty$6(proto, is, {
          value: true
        });
      }

      if (!hasOwnProperty$7.call(cls, is)) {
        defineProperty$6(cls, is, {
          value: true
        });
      }
    } // NOTE: we always use meta.declarables because setupDeclarable() can replace the array on the meta object
    // when new declarable properties are added...

    for (let decl, setupName, i = 0; i < meta.declarables.length; ++i) {
      decl = meta.declarables[i];

      if (hasOwnProperty$7.call(cls, decl)) {
        setupName = setupNames[decl] || (setupNames[decl] = `setup${StringHelper.capitalize(decl)}`);
        cls[setupName](cls, meta);
      }
    }
    /*  Add slash to the front of this line to enable the diagnostic block:
    /**/

  }
  /**
   * This method is called as part of `setupClass()`. It will process the `configurable()` return object and the
   * `defaultConfig` return object.
   * @param {Object} meta The `meta` object for this class.
   * @param {Object} configs The config definition object.
   * @param {Boolean} simple `true` when processing `defaultConfig` and `false` when processing `configurable`.
   * @private
   * @category Configuration
   */

  static setupConfigs(meta, configs, simple) {
    const classConfigValues = meta.config,
          classConfigs = meta.configs,
          cls = meta.class;
    let {
      nullify
    } = meta,
        cfg,
        defaultValue,
        options,
        setDefault,
        value,
        wasNullify;

    for (const name in configs) {
      value = configs[name];

      if (simple) {
        // Using "defaultConfig"
        if (!(cfg = classConfigs[name])) {
          cfg = Config.get(name, defaultConfigOptions);
        } else {
          // The property may be declared in a base class using configurable(), so it may have special
          // merge processing:
          value = cfg.merge(value, classConfigValues[name], meta);
        }
        /*  Add slash to the front of this line to enable the diagnostic block:
        /**/

      } else {
        // Using "configurable"
        defaultValue = options = setDefault = undefined;

        if (value && typeof value === 'object' && '$config' in value) {
          options = value.$config;

          if (options && !Objects.isObject(options)) {
            options = Objects.createTruthyKeys(options);
          }

          setDefault = 'default' in value;
          defaultValue = setDefault ? value.default : defaultValue;
          value = value.value;
        }

        if (!(cfg = classConfigs[name])) {
          cfg = Config.get(name, options);
          cfg.define(cls.prototype);
          setDefault = !(cfg.field in cls.prototype); // reduce object shape changes (helps JIT)

          wasNullify = false;
        } else {
          wasNullify = cfg.nullify;

          if (options) {
            // Defined by a base class, but maybe being adjusted by derived.
            cfg = cfg.extend(options); // In the future, we may need to redefine the property here if options affect the descriptor (such
            // as event firing)
          }

          value = cfg.merge(value, classConfigValues[name], meta);
        }

        if (setDefault) {
          cfg.setDefault(cls, defaultValue);
        }

        if (cfg.nullify && !wasNullify) {
          (nullify || (nullify = meta.nullify || (meta.nullify = []))).push(cfg);
        }
      } // If any default properties are *mutable* Objects or Array we need to clone them.
      // so that instances do not share configured values.

      if (value && (Objects.isObject(value) || Array.isArray(value)) && !Object.isFrozen(value)) {
        meta.forkConfigs = true;
      }

      classConfigs[name] = cfg;
      classConfigValues[name] = value;
    }
  }

  static setupConfigurable(cls, meta) {
    cls.setupConfigs(meta, cls.configurable, false);
  }

  static setupDefaultConfig(cls, meta) {
    cls.setupConfigs(meta, cls.defaultConfig, true);
  }

  static setupDeclarable(cls, meta) {
    const declarable = cls.declarable;
    let all = meta.declarables,
        forked,
        i;

    for (i = 0; i < declarable.length; ++i) {
      if (!all.includes(declarable[i])) {
        if (!forked) {
          meta.declarables = forked = all = all.slice();
        }

        all.push(declarable[i]);
      }
    }
  }

  static setupProperties(cls, meta) {
    meta.properties = meta.super.properties.slice();
    meta.properties.push(cls);
    Object.freeze(meta.properties);
  }

  static setupPrototypeProperties(cls) {
    Object.assign(cls.prototype, cls.prototypeProperties);
  }
  /**
   * Gets the full {@link #property-defaultConfig-static} block for this object's entire inheritance chain
   * all the way up to but not including {@link Core.Base}
   * @return {Object} All default config values for this class.
   * @private
   * @category Configuration
   */

  getDefaultConfiguration() {
    return this.constructor.getDefaultConfiguration();
  }
  /**
   * Gets the full {@link #property-defaultConfig-static} block for the entire inheritance chain for this class
   * all the way up to but not including {@link Core.Base}
   * @return {Object} All default config values for this class.
   * @private
   * @category Configuration
   */

  static getDefaultConfiguration() {
    const meta = this.$meta,
          config = meta.forkConfigs ? Base$1.fork(meta.config) : Object.create(meta.config);

    if (VersionHelper.isTestEnv && BrowserHelper.isBrowserEnv && config.testConfig && globalThis.__applyTestConfigs) {
      for (const o in config.testConfig) {
        config[o] = config.testConfig[o];
      }
    }

    return config;
  }

  static fork(obj) {
    let ret = obj,
        key,
        value;

    if (obj && Objects.isObject(obj) && !Object.isFrozen(obj)) {
      ret = Object.create(obj);

      for (key in obj) {
        value = obj[key];

        if (value) {
          if (Objects.isObject(value)) {
            ret[key] = Base$1.fork(value);
          } else if (Array.isArray(value)) {
            ret[key] = value.slice();
          }
        }
      }
    }

    return ret;
  }
  /**
   * Gets the full {@link #property-properties-static} block for this class's entire inheritance chain
   * all the way up to but not including {@link Core.Base}
   * @return {Object} All default config values for this class.
   * @private
   * @category Configuration
   */

  getProperties() {
    const // The meta.properties array is an array of classes that define "static get properties()"
    hierarchy = this.$meta.properties,
          result = {}; // TODO: if properties block does not change this could be cached? would speed up loading of big data sets
    //  into grid. The problem is that this call creates new (i.e., non-shared) instances of objects, arrays and
    //  Map objects that apply to each instance. These would all need to be cloned so it may be better to just
    //  call the properties getter each time and let it create new instances of whatever types.
    //for (cls of hierarchy) { // for-of transpiles badly and this is called a lot

    for (let i = 0; i < hierarchy.length; i++) {
      // Gather the class result in top-down order so that subclass properties override superclass properties
      Object.assign(result, hierarchy[i].properties);
    }

    return result;
  }

  static get superclass() {
    return getPrototypeOf$1(this);
  }
  /**
   * Used by the Widget and GridFeatureManager class internally. Returns the class hierarchy of this object
   * starting from the `topClass` class (which defaults to `Base`).
   *
   * For example `classHierarchy(Widget)` on a Combo would yield `[Widget, Field, TextField, PickerField, Combo]`
   * @param {Function} [topClass] The topmost class constructor to start from.
   * @returns {Function[]} The class hierarchy of this instance.
   * @private
   * @category Configuration
   */

  classHierarchy(topClass) {
    const hierarchy = this.$meta.hierarchy,
          index = topClass ? hierarchy.indexOf(topClass) : 0;
    return index > 0 ? hierarchy.slice(index) : hierarchy;
  }
  /**
   * Checks if an obj is of type using object's $$name property and doing string comparison of the property with the
   * type parameter.
   *
   * @param {String} type
   * @return {Boolean}
   */

  static isOfTypeName(type) {
    return this.$meta.names.includes(type);
  }
  /**
   * Removes all event listeners that were registered with the given `name`.
   * @param {String} name The name of the event listeners to be removed.
   * @category Events
   */

  detachListeners(name) {
    var _detachers;

    let detachers = this.$detachers;
    detachers = (_detachers = detachers) === null || _detachers === void 0 ? void 0 : _detachers[name];

    if (detachers) {
      while (detachers.length) {
        detachers.pop()();
      }
    }
  }
  /**
   * Tracks a detacher function for the specified listener name.
   * @param {String} name The name assigned to the associated listeners.
   * @param {Function} detacher The detacher function.
   * @private
   */

  trackDetacher(name, detacher) {
    const detachers = this.$detachers || (this.$detachers = {}),
          bucket = detachers[name] || (detachers[name] = []);
    bucket.push(detacher);
  }
  /**
   * Removes all detacher functions for the specified `Events` object. This is called
   * by the `removeAllListeners` method on that object which is typically called by its
   * `destroy` invocation.
   * @param {Core.mixin.Events} eventer The `Events` instance to untrack.
   * @private
   */

  untrackDetachers(eventer) {
    const detachers = this.$detachers;

    if (detachers) {
      for (const name in detachers) {
        const bucket = detachers[name];

        for (let i = bucket.length; i-- > 0;) {
          if (bucket[i].eventer === eventer) {
            bucket.splice(i, 1);
          }
        }
      }
    }
  }

}
const proto$2 = Base$1.prototype; // Informs the standard config setter there is no need to call this fn:

proto$2.onConfigChange.$nullFn = emptyFn$3.$nullFn = true;
Base$1[metaSymbol] = proto$2.$meta = newMeta({
  class: Base$1,
  config: Object.freeze({}),
  configs: Object.create(null),
  declarables: Base$1.declarable,
  forkConfigs: false,
  hierarchy: Object.freeze([Base$1]),
  names: Object.freeze(['Base']),
  nullify: null,
  properties: Object.freeze([]),
  super: null
}); // Avoid some object shape changes:

Object.assign(proto$2, {
  $detachers: null,
  configObserver: null,

  /**
   * This property is set to `true` before the `constructor` returns.
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   */
  isConstructing: true,

  /**
   * This property is set to `true` by {@link #function-destroy} after the {@link #function-doDestroy} method returns.
   * It is also one of the few properties that remains on the object after returning from `destroy()`. This property
   * is often checked in code paths that may encounter a destroyed object (like some event handlers) or in the
   * destruction path during cleanup.
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   */
  isDestroyed: false,

  /**
   * This property is set to `true` on entry to the {@link #function-destroy} method. It remains on the objects after
   * returning from `destroy()`. If {@link #property-isDestroyed} is `true`, this property will also be `true`, so
   * there is no need to test for both (for example, `comp.isDestroying || comp.isDestroyed`).
   * @member {Boolean}
   * @readonly
   * @category Lifecycle
   */
  isDestroying: false
});
Base$1.emptyFn = emptyFn$3;
VersionHelper.setVersion('core', '5.0.1');
Base$1._$name = 'Base';

/**
 * @module Core/helper/ArrayHelper
 */

/**
 * Helper with useful functions for handling Arrays
 * @internal
 */
class ArrayHelper {
  static clean(array) {
    return array.reduce((res, item) => {
      if (item !== null && item !== undefined && !(Array.isArray(item) && item.length === 0) && item !== '') res.push(item);
      return res;
    }, []);
  }
  /**
   * Similar to [`Array.from()`](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/from)
   * this method creates an array from an `iterable` object. Where `Array.from()` accepts a mapper function as the
   * second argument, this method accepts a `filter` function as its second argument. If a mapper function is also
   * needed, it can be passed as the third argument. Unlike `Array.from()`, if this method is passed `null`, it will
   * return an empty array.
   * @param {Array} iterable The iterable object to convert (must support `for-of` loop iteration).
   * @param {Function} [filter] A function to apply to each item of the `iterable` which must return a truthy value
   * to include that item in the resulting array.
   * @param {Function} [map] A function to apply to each item of the `iterable` that returns the actual value to put
   * into the returned array. If a `filter` is also supplied, this method is only called for those items that pass
   * the filter test.
   * @returns {Array}
   */

  static from(iterable, filter, map) {
    const array = [];

    if (iterable) {
      for (const it of iterable) {
        if (!filter || filter(it)) {
          array.push(map ? map(it) : it);
        }
      }
    }

    return array;
  }
  /**
   * Remove one or more items from an array
   * @param {Array} array Array to remove from
   * @param {Object} items One or more items to remove
   * @returns {Boolean} Returns true if any item was removed
   */

  static remove(array, ...items) {
    let index,
        item,
        removed = false;

    for (let i = 0; i < items.length; i++) {
      item = items[i];

      if ((index = array.indexOf(item)) !== -1) {
        array.splice(index, 1);
        removed = true;
      }
    }

    return removed;
  }
  /*
   * Calculates the insertion index of a passed object into the passed Array according
   * to the passed comparator function. Note that the passed Array *MUST* already be ordered.
   * @param {Object} item The item to calculate the insertion index for.
   * @param {Array} The array into which the item is to be inserted.
   * @param {Function} comparatorFn The comparison function. Must return -1 or 0 or 1.
   * @param {Object} comparatorFn.lhs The left object to compare.
   * @param {Object} comparatorFn.rhs The right object to compare.
   * @param {Number} index The possible correct index to try first before a binary
   * search is instigated.
   */

  static findInsertionIndex(item, array, comparatorFn = this.lexicalCompare, index) {
    const len = array.length;
    let beforeCheck, afterCheck;

    if (index < len) {
      beforeCheck = index > 0 ? comparatorFn(array[index - 1], item) : 0;
      afterCheck = index < len - 1 ? comparatorFn(item, array[index]) : 0;

      if (beforeCheck < 1 && afterCheck < 1) {
        return index;
      }
    }

    return this.binarySearch(array, item, comparatorFn);
  }
  /**
   * Similar to the native Array.find call, but this finds the *last* element in the array for which
   * the passed function returns a truthy value.
   * @param {Object[]} array The array to find in.
   * @param {Function} fn The testing function.
   * @param {Object} [thisObj] The scope (`this` reference) in which to call the function.
   */

  static findLast(array, fn, thisObj) {
    for (let {
      length
    } = array, i = length - 1; i >= 0; i--) {
      if (fn.call(thisObj, array[i], i, array)) {
        return array[i];
      }
    }
  }
  /**
   * This method returns the index that a given item would be inserted into the
   * given (sorted) `array`. Note that the given `item` may or may not be in the
   * array. This method will return the index of where the item *should* be.
   *
   * For example:
   *
   *      var array = [ 'A', 'D', 'G', 'K', 'O', 'R', 'X' ];
   *      var index = ArrayHelper.binarySearch(array, 'E');
   *
   *      console.log('index: ' + index);
   *      // logs "index: 2"
   *
   *      array.splice(index, 0, 'E');
   *
   *      console.log('array : ' + array.join(''));
   *      // logs "array: ADEGKORX"
   *
   * @param {Object[]} array The array to search.
   * @param {Object} item The item that you want to insert into the `array`.
   * @param {Number} [begin=0] The first index in the `array` to consider.
   * @param {Number} [end=array.length] The index that marks the end of the range
   * to consider. The item at this index is *not* considered.
   * @param {Function} [compareFn] The comparison function that matches the sort
   * order of the `array`. The default `compareFn` compares items using less-than
   * and greater-than operators.
   * @return {Number} The index for the given item in the given array based on
   * the passed `compareFn`.
   */

  static binarySearch(array, item, begin = 0, end = array.length, compareFn = this.lexicalCompare) {
    const length = array.length;
    let middle, comparison;

    if (begin instanceof Function) {
      compareFn = begin;
      begin = 0;
    } else if (end instanceof Function) {
      compareFn = end;
      end = length;
    }

    --end;

    while (begin <= end) {
      middle = begin + end >> 1;
      comparison = compareFn(item, array[middle]);

      if (comparison >= 0) {
        begin = middle + 1;
      } else if (comparison < 0) {
        end = middle - 1;
      }
    }

    return begin;
  }

  magnitudeCompare(lhs, rhs) {
    return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
  }

  lexicalCompare(lhs, rhs) {
    lhs = String(lhs);
    rhs = String(rhs);
    return lhs < rhs ? -1 : lhs > rhs ? 1 : 0;
  }
  /**
   * Similar to Array.prototype.fill(), but constructs a new array with the specified item count and fills it with
   * clones of the supplied item.
   * @param {Number} count Number of entries to create
   * @param {Object|Array} itemOrArray Item or array of items to clone (uses object spread to create shallow clone)
   * @param {Function} [fn] An optional function that is called for each item added, to allow processing
   * @returns {Array} A new populated array
   */

  static fill(count, itemOrArray = {}, fn = null) {
    const result = [],
          items = Array.isArray(itemOrArray) ? itemOrArray : [itemOrArray];

    for (let i = 0; i < count; i++) {
      for (const item of items) {
        // Using object spread here forces us to use more babel plugins and will make
        // react_typescript demo very difficult to setup
        const processedItem = Object.assign({}, item);

        if (fn) {
          fn(processedItem, i);
        }

        result.push(processedItem);
      }
    }

    return result;
  }
  /**
   * Populates an array with the return value from `fn`.
   * @param {Number} count Number of entries to create
   * @param {Function} fn A function that is called `count` times, return value is added to array
   * @param {Number} fn.index Current index in the array
   * @privateparam {Boolean} [oneBased] Add 1 to the index before calling the fn (making it 1 based)
   * @returns {Array} A new populated array
   */

  static populate(count, fn, oneBased = false) {
    const items = [];

    for (let i = 0; i < count; i++) {
      items.push(fn(i + (oneBased ? 1 : 0)));
    }

    return items;
  }
  /**
   * Pushes `item` on to the `array` if not already included
   * @param {Array}  array Array to push to
   * @param {Object} item Item to push if not already included
   */

  static include(array, item) {
    if (!array.includes(item)) {
      array.push(item);
    }
  }
  /**
   * Returns a new array with the unique items from the supplied array.
   * @param {Array} array Input array
   * @returns {Array} New array with unique items
   */

  static unique(array) {
    return [...new Set(array)];
  } // Kept for future reference : Wanted to create an indexer on Stores.

  static allowNegative(array) {
    // From https://github.com/sindresorhus/negative-array
    return new Proxy(array, {
      get(target, name, receiver) {
        if (typeof name !== 'string') {
          return Reflect.get(target, name, receiver);
        }

        const index = Number(name);

        if (Number.isNaN(index)) {
          return Reflect.get(target, name, receiver);
        }

        return target[index < 0 ? target.length + index : index];
      },

      set(target, name, value, receiver) {
        if (typeof name !== 'string') {
          return Reflect.set(target, name, value, receiver);
        }

        const index = Number(name);

        if (Number.isNaN(index)) {
          return Reflect.set(target, name, value, receiver);
        }

        target[index < 0 ? target.length + index : index] = value;
        return true;
      }

    });
  }

  static delta(a, b, useRelativeNaming = false) {
    // Nicer syntax but about 40% slower (an extra iteration)
    // const
    //     onlyInA = a.filter(item => !b.includes(item)),
    //     onlyInB = b.filter(item => !a.includes(item)),
    //     inBoth  = a.filter(item => b.includes(item));
    // Quick bailout for nonexisting target array
    if (!b) {
      return useRelativeNaming ? {
        toAdd: a,
        toRemove: [],
        toKeep: []
      } : {
        onlyInA: a,
        onlyInB: [],
        inBoth: []
      };
    }

    const onlyInA = [],
          onlyInB = [],
          inBoth = [];

    for (let i = 0; i < a.length; i++) {
      const item = a[i];

      if (b.includes(item)) {
        inBoth.push(item);
      } else {
        onlyInA.push(item);
      }
    }

    for (let i = 0; i < b.length; i++) {
      const item = b[i];

      if (!inBoth.includes(item)) {
        onlyInB.push(item);
      }
    }

    if (useRelativeNaming) {
      return {
        toAdd: onlyInA,
        toRemove: onlyInB,
        toKeep: inBoth
      };
    }

    return {
      onlyInA,
      onlyInB,
      inBoth
    };
  }
  /**
   * Returns the passed object wrapped in an array. Special handling of the following cases:
   * * Passing an array returns it as is
   * * Passing a `Set` returns it converted to an Array
   * * Passing `null`/`undefined` returns the passed value
   *
   * ```javascript
   * const records = ArrayHelper.asArray(record);
   *
   * // { id : 1 } -> [{ id : 1 }]
   * // [{ id : 1 }] -> [{ id : 1 }]
   * ```
   *
   * @param {*} arrayOrObject
   * @returns {Array|null}
   * @internal
   */

  static asArray(arrayOrObject) {
    if (arrayOrObject == null) {
      return arrayOrObject;
    }

    if (arrayOrObject instanceof Set) {
      return Array.from(arrayOrObject);
    }

    return Array.isArray(arrayOrObject) ? arrayOrObject : [arrayOrObject];
  }

}
ArrayHelper._$name = 'ArrayHelper';

/**
 * @module Core/helper/FunctionHelper
 */
const commaSepRe = /,\s*/,
      decompiledSym = Symbol('decompiled'),
      // [async] p => ...
fnRe1 = /^\s*(async\s+)?([a-z_]\w*)\s*=>([\s\S]+)$/i,
      // [async] (p1?[, px]*) => ...
fnRe2 = /^\s*(async\s*)?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)\s+=>([\s\S]+)$/i,
      // [async] [function] [name] (p1?[, px]*) ...
fnRe3 = /^(\s*async)?(?:\s*function)?(?:\s*([a-z_]\w*))?\s*\(((?:[a-z_]\w*(?:, [a-z_]\w*)*)?)\)([\s\S]+)$/i,
      {
  hasOwnProperty: hasOwnProperty$6
} = Object.prototype;
/**
 * Provides functionality for working with functions
 * @internal
 */

class FunctionHelper {
  /**
   * Inserts a function after the specified `method` is called on an `object`. To remove this hook, invoke the
   * function returned by this method.
   * ```
   *  class A {
   *      method() {
   *          console.log('method');
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.after(instance, 'method', () => { console.log('after') });
   *
   *  instance.method();
   *  > method
   *  > after
   *
   *  detach();
   *  instance.method();
   *  > method
   * ```
   * The value returned by the original method is passed as the first argument to `fn` followed by all the arguments
   * passed by the caller.
   *
   * If `fn` returns a value (not `undefined`), that value is returned from the method call instead of the value
   * returned by the original method.
   * ```
   *  class A {
   *      method(x) {
   *          console.log('method', x);
   *          return x * 2
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.after(instance, 'method', (ret, x) => {
   *      console.log('after', ret, x);
   *      return x / 2;
   *  });
   *
   *  console.log(instance.method(50));
   *  > method 50
   *  > after 100 50
   *  > 25
   *
   *  detach();
   *  console.log(instance.method(50));
   *  > method 50
   *  > 100
   * ```
   *
   * @param {Object} object The object to hook.
   * @param {String} method The name of the method on `object` to hook.
   * @param {Function|String} fn The function or method name (on `thisObj`) to call after `method`.
   * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
   * @param {Object} [options] Additional options
   * @param {Boolean} [options.return=true] Specify `false` to not include the return value of the hooked method as
   * the first argument to `fn`.
   * @returns {Function} The function to call to remove the hook.
   */
  static after(object, method, fn, thisObj, options) {
    const named = typeof fn === 'string',
          withReturn = (options === null || options === void 0 ? void 0 : options.return) !== false,
          hook = (...args) => {
      const // if object.destroy() occurs, our hook will be removed, so this fn won't be called in that case
      origResult = hook.$nextHook.call(object, ...args),
            hookResult = thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed ? undefined : withReturn ? named ? thisObj[fn](origResult, ...args) : fn.call(thisObj, origResult, ...args) : named ? thisObj[fn](...args) : fn.call(thisObj, ...args);
      return hookResult === undefined ? origResult : hookResult;
    };

    return FunctionHelper.hookMethod(object, method, hook);
  }
  /**
   * Inserts a function before the specified `method` is called on an `object`. To remove this hook, invoke the
   * function returned by this method.
   * ```
   *  class A {
   *      method() {
   *          console.log('method');
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.before(instance, 'method', () => { console.log('before') });
   *
   *  instance.method();
   *  > before
   *  > method
   *
   *  detach();
   *  instance.method();
   *  > method
   * ```
   * If `fn` returns `false`, the original method is not invoked and `false` is returned to the caller.
   * ```
   *  class A {
   *      method(x) {
   *          console.log('method', x);
   *          return x * 2;
   *      }
   *  }
   *
   *  let instance = new A();
   *
   *  let detach = FunctionHelper.before(instance, 'method', x => {
   *      console.log('before', x);
   *      return false;
   *  });
   *
   *  console.log(instance.method(50));
   *  > before 50
   *  > false
   *
   *  detach();
   *  console.log(instance.method(50));
   *  > method 50
   *  > 100
   * ```
   *
   * @param {Object} object The object to hook.
   * @param {String} method The name of the method on `object` to hook.
   * @param {Function|String} fn The function or method name (on `thisObj`) to call before `method`.
   * @param {Object} [thisObj] The `this` pointer value for calling `fn`.
   * @returns {Function} The function to call to remove the hook.
   */

  static before(object, method, fn, thisObj) {
    const named = typeof fn === 'string',
          hook = (...args) => {
      const ret = thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed ? 0 : named ? thisObj[fn](...args) : fn.call(thisObj, ...args);
      return ret === false ? ret : hook.$nextHook.call(object, ...args);
    };

    return FunctionHelper.hookMethod(object, method, hook);
  }

  static curry(func) {
    return function curried(...args) {
      if (args.length >= func.length) {
        return func.apply(this, args);
      } else {
        return function (...args2) {
          return curried.apply(this, args.concat(args2));
        };
      }
    };
  }

  static bindAll(obj) {
    for (const key in obj) {
      if (typeof obj[key] === 'function') {
        obj[key] = obj[key].bind(obj);
      }
    }
  }
  /**
   * Returns a function which calls the passed `interceptor` function first, and the passed `original` after
   * as long as the `interceptor` does not return `false`.
   * @param {Function} original The function to call second.
   * @param {Function} interceptor The function to call first.
   * @param {Object} [thisObj] The `this` reference when the functions are called.
   * @returns The return value from the `original` function **if it was called**, else `false`.
   */

  static createInterceptor(original, interceptor, thisObj) {
    return (...args) => {
      if (interceptor.call(thisObj, ...args) !== false) {
        return original.call(thisObj, ...args);
      }

      return false;
    };
  }
  /**
   * Returns a function which calls the passed `sequence` function after calling
   * the passed `original`.
   * @param {Function} original The function to call first.
   * @param {Function} sequence The function to call second.
   * @param {Object} [thisObj] The `this` reference when the functions are called.
   * @returns The value returned from the sequence if it returned a value, else the return
   * value from the original function.
   */

  static createSequence(original, sequence, thisObj) {
    return (...args) => {
      const origResult = original.call(thisObj, ...args),
            sequenceResult = sequence.call(thisObj, ...args);
      return sequenceResult === undefined ? origResult : sequenceResult;
    };
  }
  /**
   * Create a "debounced" function which will call on the "leading edge" of a timer period.
   * When first invoked will call immediately, but invocations after that inside its buffer
   * period will be rejected, and *one* invocation will be made after the buffer period has expired.
   *
   * This is useful for responding immediately to a first mousemove, but from then on, only
   * calling the action function on a regular timer while the mouse continues to move.
   *
   * @param {Function} fn The function to call.
   * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Array} [extraArgs] The argument list to append to those passed to the function.
   * @param {Function} [alt] A function to call when the invocation is rejected due to buffer time not having expired.
   */

  static createThrottled(fn, buffer, thisObj, extraArgs, alt) {
    let lastCallTime = -Number.MAX_VALUE,
        callArgs,
        timerId;

    const invoke = () => {
      timerId = 0;
      lastCallTime = performance.now();
      callArgs.push.apply(callArgs, extraArgs);
      fn.apply(thisObj, callArgs);
    },
          result = function (...args) {
      const elapsed = performance.now() - lastCallTime;
      callArgs = args; // If it's been more then the buffer period since we invoked, we can call it now

      if (elapsed >= buffer) {
        clearTimeout(timerId);
        invoke();
      } // Otherwise, kick off a timer for the requested period.
      else {
        if (!timerId) {
          timerId = setTimeout(invoke, buffer - elapsed);
        }

        if (alt) {
          callArgs.push.apply(callArgs, extraArgs);
          alt.apply(thisObj, callArgs);
        }
      }
    };

    result.cancel = () => clearTimeout(timerId);

    return result;
  }
  /**
   * Create a "debounced" function which will call on the "trailing edge" of a timer period.
   * When first invoked will wait until the buffer period has expired to call the function, and
   * more calls within that time will restart the timer.
   *
   * This is useful for responding to keystrokes, but deferring action until the user pauses typing.
   *
   * @param {Function} fn The function to call.
   * @param {Number} buffer The milliseconds to wait after each execution before another execution takes place.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Array} [args] The argument list to append to those passed to the function.
   */

  static createBuffered(fn, buffer, thisObj, args) {
    let callArgs, timerId;

    const invoke = () => {
      timerId = 0;
      result.isPending = false;
      callArgs.push.apply(callArgs, args);
      fn.apply(thisObj, callArgs);
    },
          result = function (...args) {
      callArgs = args; // Cancel any impending invocation. It's pushed out for <buffer> ms from each call

      if (timerId) {
        clearTimeout(timerId);
      }

      result.isPending = true;
      timerId = setTimeout(invoke, buffer);
    };

    result.cancel = () => {
      result.isPending = false;
      clearTimeout(timerId);
    };

    return result;
  }

  static decompile(fn) {
    if (!(decompiledSym in fn)) {
      var _body, _body2;

      const code = fn.toString();
      let m = fnRe1.exec(code),
          args,
          body,
          name,
          decompiled,
          t;

      if (m) {
        // [async] p => ...
        //   [1]   [2]  [3]
        args = [m[2]];
        body = m[3];
      } else if (m
      /* assignment */
      = fnRe2.exec(code)) {
        // [async] (p1?[, px]*) => ...
        //   [1]   [2]             [3]
        t = m[2].trim();
        args = t ? t.split(commaSepRe) : [];
        body = m[3];
      } else if (m
      /* assignment */
      = fnRe3.exec(code)) {
        // [async] [function] [name] (p1?[, px]*) ...
        //   [1]              [2]     [3]         [4]
        name = m[2];
        t = m[3].trim();
        args = t ? t.split(commaSepRe) : [];
        body = m[4];
      }

      body = (_body = body) === null || _body === void 0 ? void 0 : _body.trim();
      fn[decompiledSym] = decompiled = m && {
        args,
        async: Boolean(m[1]),
        body: (_body2 = body) !== null && _body2 !== void 0 && _body2.startsWith('{') ? body.substring(1, body.length - 1).trim() : body
      };

      if (name) {
        decompiled.name = name;
      }
    }

    return fn[decompiledSym];
  }

  static hookMethod(object, method, hook) {
    hook.$nextHook = object[method];
    object[method] = hook;
    return () => {
      // Object will have no hooks on the instance if it is destroyed (perhaps other reasons too)
      if (hasOwnProperty$6.call(object, method)) {
        let f = object[method],
            next;

        if (f === hook) {
          var _Object$getPrototypeO;

          // When this is the outermost hook, we may be the last hook. If $nextHook is found on the object's
          // prototype, simply delete the slot to expose it. Otherwise, there's another hook, so make it the
          // outermost.
          if (((_Object$getPrototypeO = Object.getPrototypeOf(object)) === null || _Object$getPrototypeO === void 0 ? void 0 : _Object$getPrototypeO[method]) === hook.$nextHook) {
            delete object[method];
          } else {
            object[method] = hook.$nextHook;
          }
        } else {
          // Not being the outermost hook means we have outer hooks that should chain to the one we want to
          // remove. Be cautious because the object could be destroyed.
          for (; next = (_f = f) === null || _f === void 0 ? void 0 : _f.$nextHook; f = next) {
            var _f;

            if (next === hook) {
              f.$nextHook = hook.$nextHook;
              break;
            }
          }
        }
      }
    };
  }
  /**
   * Protects the specified `method` on a given `object` such that calling it will not throw exceptions.
   * @param {Object} object The object whose method is to be protected.
   * @param {String} method The name of the method to protect.
   * @param {Function} [handler] An optional function to call for any thrown exceptions.
   * @internal
   */

  static noThrow(object, method, handler) {
    const fn = object[method];

    object[method] = (...args) => {
      try {
        return fn.apply(object, args);
      } catch (e) {
        return handler === null || handler === void 0 ? void 0 : handler(e);
      }
    };
  }

  static returnTrue() {
    return true;
  }

  static animate(duration, fn, thisObj, easing = 'linear') {
    let cancel = false;
    const result = new Promise(resolve => {
      const start = performance.now(),
            iterate = () => {
        const progress = Math.min((performance.now() - start) / duration, 1),
              delayable = thisObj && thisObj.setTimeout ? thisObj : globalThis;

        if (!cancel) {
          if (fn.call(thisObj, this.easingFunctions[easing](progress)) === false) {
            resolve();
          }
        }

        if (cancel || progress === 1) {
          // Push resolution into the next animation frame so that
          // this frame completes before the resolution handler runs.
          delayable.requestAnimationFrame(() => resolve());
        } else {
          delayable.requestAnimationFrame(iterate);
        }
      };

      iterate();
    });

    result.cancel = () => {
      cancel = true;
      return false;
    };

    return result;
  }

}
/* eslint-disable */

const half = 0.5,
      e1 = 1.70158,
      e2 = 7.5625,
      e3 = 1.525,
      e4 = 2 / 2.75,
      e5 = 2.25 / 2.75,
      e6 = 1 / 2.75,
      e7 = 1.5 / 2.75,
      e8 = 2.5 / 2.75,
      e9 = 2.625 / 2.75,
      e10 = 0.75,
      e11 = 0.9375,
      e12 = 0.984375,
      s1 = 1.70158,
      s2 = 1.70158;
FunctionHelper.easingFunctions = {
  linear: t => t,
  easeInQuad: t => Math.pow(t, 2),
  easeOutQuad: t => -(Math.pow(t - 1, 2) - 1),
  easeInOutQuad: t => (t /= half) < 1 ? half * Math.pow(t, 2) : -half * ((t -= 2) * t - 2),
  easeInCubic: t => Math.pow(t, 3),
  easeOutCubic: t => Math.pow(t - 1, 3) + 1,
  easeInOutCubic: t => (t /= half) < 1 ? half * Math.pow(t, 3) : half * (Math.pow(t - 2, 3) + 2),
  easeInQuart: t => Math.pow(t, 4),
  easeOutQuart: t => -(Math.pow(t - 1, 4) - 1),
  easeInOutQuart: t => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
  easeInQuint: t => Math.pow(t, 5),
  easeOutQuint: t => Math.pow(t - 1, 5) + 1,
  easeInOutQuint: t => (t /= half) < 1 ? half * Math.pow(t, 5) : half * (Math.pow(t - 2, 5) + 2),
  easeInSine: t => -Math.cos(t * (Math.PI / 2)) + 1,
  easeOutSine: t => Math.sin(t * (Math.PI / 2)),
  easeInOutSine: t => -half * (Math.cos(Math.PI * t) - 1),
  easeInExpo: t => t === 0 ? 0 : Math.pow(2, 10 * (t - 1)),
  easeOutExpo: t => t === 1 ? 1 : -Math.pow(2, -10 * t) + 1,
  easeInOutExpo: t => t === 0 ? 0 : t === 1 ? 1 : (t /= half) < 1 ? half * Math.pow(2, 10 * (t - 1)) : half * (-Math.pow(2, -10 * --t) + 2),
  easeInCirc: t => -(Math.sqrt(1 - t * t) - 1),
  easeOutCirc: t => Math.sqrt(1 - Math.pow(t - 1, 2)),
  easeInOutCirc: t => (t /= half) < 1 ? -half * (Math.sqrt(1 - t * t) - 1) : half * (Math.sqrt(1 - (t -= 2) * t) + 1),
  easeOutBounce: t => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
  easeInBack: t => t * t * ((e1 + 1) * t - e1),
  easeOutBack: t => (t = t - 1) * t * ((e1 + 1) * t + e1) + 1,
  easeInOutBack: t => {
    let v1 = s1;
    return (t /= half) < 1 ? half * (t * t * (((v1 *= e3) + 1) * t - v1)) : half * ((t -= 2) * t * (((v1 *= e3) + 1) * t + v1) + 2);
  },
  elastic: t => -1 * Math.pow(4, -8 * t) * Math.sin((t * 6 - 1) * (2 * Math.PI) / 2) + 1,
  swingFromTo: t => {
    let v2 = s2;
    return (t /= half) < 1 ? half * (t * t * (((v2 *= e3) + 1) * t - v2)) : half * ((t -= 2) * t * (((v2 *= e3) + 1) * t + v2) + 2);
  },
  swingFrom: t => t * t * ((e1 + 1) * t - e1),
  swingTo: t => (t -= 1) * t * ((e1 + 1) * t + e1) + 1,
  bounce: t => t < e6 ? e2 * t * t : t < e4 ? e2 * (t -= e7) * t + e10 : t < e8 ? e2 * (t -= e5) * t + e11 : e2 * (t -= e9) * t + e12,
  bouncePast: t => t < e6 ? e2 * t * t : t < e4 ? 2 - (e2 * (t -= e7) * t + e10) : t < e8 ? 2 - (e2 * (t -= e5) * t + e11) : 2 - (e2 * (t -= e9) * t + e12),
  easeFromTo: t => (t /= half) < 1 ? half * Math.pow(t, 4) : -half * ((t -= 2) * Math.pow(t, 3) - 2),
  easeFrom: t => Math.pow(t, 4),
  easeTo: t => Math.pow(t, 0.25)
};
FunctionHelper._$name = 'FunctionHelper';

/**
 * @module Core/mixin/Events
 */

const // Used by the config system to flatten configs from the class hierarchy.
// In this case, a pure merge is not wanted. Listener definitions from
// every class level are collected up into an array.
// addListener iterates any passed array, adding each element.
{
  isArray: isArray$1
} = Array,
      {
  hasOwnProperty: hasOwnProperty$5
} = Object.prototype,
      // Used to distinguish event names from listener options in addListener object config.
specialProperties = {
  thisObj: 1,
  detachable: 1,
  once: 1,
  detacher: 1,
  prio: 1,
  args: 1,
  expires: 1,
  buffer: 1,
  throttle: 1,
  name: 1
},
      priorityComparator = (a, b) => b.prio - a.prio;
/**
 * Mix this into another class to enable event handling.
 *
 * ## Basic usage
 * Listeners can be added either through config:
 *
 * ```javascript
 * let button = new Button({
 *   listeners: {
 *     click: () => {},
 *     press: () => {},
 *     ...
 *   }
 * });
 * ```
 *
 * *NOTE*: Do not reuse listeners config object, use new every time:
 * ```javascript
 * // wrong
 * let config = { click : () => {} }
 * new Button({
 *     listeners : config
 * })
 * new Button({
 *     listeners : config
 * })
 * // right
 * new Button({
 *     listeners : { click : () => {} }
 * })
 * new Button({
 *     listeners : { click : () => {} }
 * })
 * ```
 *
 * Or by calling on()/addListener():
 *
 * ```javascript
 * let button = new Button();
 *
 * button.addListener('press', () => {});
 * // on is an alias for addListener
 * button.on('click', () => {});
 * ```
 *
 * This style also accepts multiple listeners in same way as when using config:
 *
 * ```javascript
 * button.on({
 *   click: () => {},
 *   press: () => {},
 *   ...
 * });
 * ```
 *
 * ## Options
 * ### Once
 * Listeners can be configured to automatically deregister after first trigger by specifying config option `once`:
 *
 * ```javascript
 * button.on({
 *   click: () => {},
 *   once: true
 * });
 * ```
 *
 * ### Priority
 * Specifying priority affects the order in which listeners are called when triggering an event. Higher priorities will be
 * called before lower. Default value is 0.
 *
 * ```javascript
 * button.on({
 *   click: this.onClick,
 *   prio: 1
 * });
 * ```
 *
 * ### This reference
 * If desired, you can specify thisObj when configuring listeners. There is no need if you are using arrow functions as
 * listeners, but might be handy in other cases. Of course, you can also use bind to set `this` reference.
 *
 * ```javascript
 * button.on({
 *   click: this.onClick,
 *   thisObj: this
 * });
 *
 * // or
 *
 * button.on({
 *   click: this.onClick.bind(this)
 * });
 * ```
 *
 * ### Buffering
 * By specifying a `buffer` events that fire frequently can be grouped together and delayed. A handler for the event will be called once only, when no new event has been fired during the specified buffer time:
 *
 * ```javascript
 * button.on({
 *   click  : this.onClick,
 *   buffer : 200 // in milliseconds
 * });
 * ```
 * In this example, if a user clicked the button 6 times very fast (<200ms between each click),
 * the `this.onClick` handler would be called only once 200 milliseconds after the last click.
 *
 * ### Throttling
 * Create a "debounced" function which will call on the "leading edge" of a timer period.
 * When first invoked will call immediately, but invocations after that inside its buffer
 * period will be rejected, and *one* invocation will be made after the buffer period has expired.
 *
 * This is useful for responding immediately to a first mousemove, but from then on, only
 * calling the action function on a regular timer while the mouse continues to move.
 * ```javascript
 * button.on({
 *   click    : this.onClick,
 *   throttle : 200 // in milliseconds
 * });
 * ```
 * In this example, if a user clicked the button 6 times very fast, the `this.onClick` handler would be called once immediately on the first click and a second time 200 milliseconds after the **first** click.
 * So in reality the `click` event handler will be called every 200ms independent of amount of click in a middle, if the event was triggered at least once during the `throttle` timeout.
 *
 * ### Detacher
 * A convenient way of unregistering events is to use a detacher, a function returned when adding listeners that you
 * call later to deregister them. As of version 1.0, detachable defaults to true.
 *
 * ```javascript
 * let detacher = button.on({
 *   click: () => {},
 *   press: () => {},
 *   detachable: true
 * });
 *
 * // when you want to detach, for example in destroy()
 * detacher();
 * ```
 *
 * ### Auto detaching
 * When listeners are bound to a class instance using `thisObj`, the `thisObj`'s `doDestroy` method
 * is overridden to remove the listeners before calling the overridden doDestroy.
 *
 * ```javascript
 * class MyClass extends Base {
 *   construct() {
 *     let button = new Button({
 *       listeners: {
 *         click: () => {},
 *         thisObj: this
 *       }
 *     });
 *   }
 *
 *   doDestroy() {
 *     // clean up stuff
 *   }
 * }
 *
 * let myObj = new MyClass();
 * // clean up, also removes listeners
 * myObj.destroy();
 * ```
 *
 * ### On-functions
 * When mixing Events into another class it can be configured to call on-functions when events are triggered.
 * On-functions are functions named 'onEventName', for example 'onClick', 'onPress' declared on the class triggering
 * the event.
 *
 * ```javascript
 * // mix Events in with on-functions activated
 * let button = new Button({
 *   callOnFunctions: true,
 *
 *   onClick: () => {}
 * });
 *
 * // or add a getter in class declaration
 * ```
 *
 * Returning `false` from an on-function will prevent triggering listeners for the event.
 *
 * ### Catching all events
 * By specifying a listener for {@link #event-catchAll catchAll} a function can be notified when any event is triggered:
 *
 * ```javascript
 * const button = new Button({
 *    listeners : {
 *        catchAll(event) {
 *            // All events on the button will pass through here
 *        }
 *    }
 * });
 * ```
 *
 * @mixin
 */

var Events = (Target => class Events extends (Target || Base$1) {
  static get $name() {
    return 'Events';
  } //region Events

  /**
   * Fires before an object is destroyed.
   * @event beforeDestroy
   * @param {Object} source The Object that is being destroyed.
   */

  /**
   * Fires when an object is destroyed.
   * @event destroy
   * @param {Object} source The Object that is being destroyed.
   */

  /**
   * Fires when any other event is fired from the object.
   *
   * **Note**: `catchAll` is fired for both public and private events. Please rely on the public events only.
   * @event catchAll
   * @param {Object} event The Object that contains event details
   * @param {String} event.type The type of the event which is caught by the listener
   */
  //endregion

  static get declarable() {
    return [
    /**
     * The list of deprecated events as an object, where `key` is an event name which is deprecated and
     * `value` is an object which contains values for
     * {@link Core.helper.VersionHelper#function-deprecate-static VersionHelper}:
     * - product {String} The name of the product;
     * - invalidAsOfVersion {String} The version where the offending code is invalid (when any compatibility
     *   layer is actually removed);
     * - message {String} Warning message to show to the developer using a deprecated API;
     *
     * For example:
     *
     * ```javascript
     * return {
     *     click : {
     *         product            : 'Grid',
     *         invalidAsOfVersion : '1.0.0',
     *         message            : 'click is deprecated!'
     *     }
     * }
     * ```
     *
     * @name deprecatedEvents
     * @returns {Object}
     * @static
     * @internal
     */
    'deprecatedEvents'];
  }

  static setupDeprecatedEvents(cls, meta) {
    const all = meta.getInherited('deprecatedEvents'),
          add = cls.deprecatedEvents;

    for (const eventName in add) {
      // Event names are case-insensitive so build our map using toLowerCased names (but keep true case too):
      all[eventName.toLowerCase()] = all[eventName] = add[eventName];
    }
  } //region Config

  static get configurable() {
    return {
      /**
       * Set to true to call onXXX method names (e.g. `onShow`, `onClick`), as an easy way to listen for events.
       *
       * ```javascript
       * const container = new Container({
       *     callOnFunctions : true
       *
       *     onHide() {
       *          // Do something when the 'hide' event is fired
       *     }
       * });
       * ```
       *
       * @config {Boolean} callOnFunctions
       * @category Misc
       * @default false
       */

      /**
       * The listener set for this object.
       *
       * Listeners can be specified in target class config and they will be merged with any listeners specified in
       * the instantiation config. Class listeners will be fired first:
       *
       * ```javascript
       * class MyStore extends Store({
       *     static get configurable() {
       *         return {
       *             listeners : {
       *                 myCustomEvent() {
       *                 },
       *                 load : {
       *                     prio : 10000,
       *                     fn() { // this load listener handles things first }
       *                 }
       *             }
       *         }
       *     }
       * });
       *
       * let store = new MyStore({
       *   listeners: {
       *     load: () => { // This load listener runs after the class's },
       *     ...
       *   }
       * });
       * ```
       *
       * @config {Object}
       * @category Common
       */
      listeners: {
        value: null,
        $config: {
          merge(newValue, currentValue) {
            if (newValue !== null) {
              if (!newValue) {
                return currentValue;
              }

              if (currentValue) {
                newValue = newValue ? [newValue] : [];
                newValue.push[isArray$1(currentValue) ? 'apply' : 'call'](newValue, currentValue);
              }
            }

            return newValue;
          }

        }
      },

      /**
       * An object where property names with a truthy value indicate which events should bubble up the ownership
       * hierarchy when triggered.
       *
       * ```javascript
       * const container = new Container({
       *     items : [
       *        { type : 'text', bubbleEvents : { change : true }}
       *     ],
       *
       *     listeners : {
       *         change() {
       *             // Will catch change event from the text field
       *         }
       *     }
       * });
       * ```
       *
       * @config {Object}
       * @category Misc
       */
      bubbleEvents: null
    };
  }

  destroy() {
    this.trigger('beforeDestroy');
    super.destroy();
  } //endregion
  //region Init

  construct(config, ...args) {
    var _config;

    // Configured listeners use this as the thisObj
    if (this.configuredListeners
    /* assignment */
    = (_config = config) === null || _config === void 0 ? void 0 : _config.listeners) {
      config = _objectSpread2({}, config);
      delete config.listeners;
    }

    super.construct(config, ...args); // Apply configured listeners after construction.
    // Note that some classes invoke this during parts of their construction.
    // Store invokes this prior to setting data so that observers are notified of data load.

    this.processConfiguredListeners();
  }

  processConfiguredListeners() {
    // This can only happen once
    if (this.configuredListeners) {
      const me = this,
            {
        isConfiguring
      } = me; // If called from config ingestion during configuration, listeners must be added
      // so temporarily clear the isConfiguring flag.

      me.isConfiguring = false;
      me.listeners = me.configuredListeners;
      me.configuredListeners = null;
      me.isConfiguring = isConfiguring;
    }
  }
  /**
   * Auto detaches listeners registered from start, if set as detachable
   * @internal
   */

  doDestroy() {
    this.trigger('destroy');
    this.removeAllListeners();
    super.doDestroy();
  }

  static setupClass(meta) {
    super.setupClass(meta);
    Events.prototype.onListen.$nullFn = true;
    Events.prototype.onUnlisten.$nullFn = true;
  } //endregion
  //region Listeners

  /**
   * Adds an event listener. This method accepts parameters in the following format:
   *
   * ```javascript
   *  myObject.addListener({
   *      thisObj    : this,          // The this reference for the handlers
   *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
   *      otherevent : {
   *          fn      : 'handlerFnName',
   *          once    : true          // Just this handler is auto-removed on fire
   *      },
   *      yetanother  : {
   *          fn      : 'yetAnotherHandler',
   *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
   *      },
   *      prio        : 100           // Higher prio listeners are called before lower
   *  });
   * ```
   *
   * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
   * When it is destroyed, those listeners are removed.
   *
   * A simpler signature may be used when only adding a listener for one event and no extra options
   * (such as `once` or `delay`) are required:
   *
   * ```javascript
   * myObject.addListener('click', myController.handleClicks, myController);
   * ```
   *
   * The args in this simple case are `eventName`, `handler` and `thisObj`
   *
   * @param {Object|String} config An object containing listener definitions, or the event name to listen for
   * @param {Object} [config.thisObj] The `this` reference for all listeners.
   * (May be overridden if a handler is specified in object form)
   * @param {Boolean} [config.once] Specify as `true` to remove the listener as soon as it is invoked.
   * @param {Number|Object} [config.expires] The listener only waits for a specified time before
   * being removed. The value may be a number or an object containing an expiry handler.
   * @param {Number} [config.expires.delay] How long to wait for the event for.
   * @param {String|Function} [config.expires.alt] The function to call when the listener expires
   * **without having been triggered**.
   * @param {Object[]} [config.args] An array of arguments to be passed to the handler before the event object.
   * @param {Number} [config.prio] The priority for all listeners; higher priority listeners are called before lower.
   * @param {Number} [config.buffer] A buffer time in milliseconds to wait after last event trigger to call the handler, to reduce the amount of handler calls for frequent events.
   * @param {Number} [config.throttle] A millisecond timeout value to throttle event triggering. With it specified a handler
   * will be called once immediately and then all following calls during the timeout period will be grouped together into one call once per throttle period.
   * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object, or the handler
   * function to call if providing a string as the first arg.
   * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
   * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
   */

  addListener(config, thisObj, oldThisObj) {
    if (isArray$1(config)) {
      for (let i = 0, {
        length
      } = config; i < length; i++) {
        this.addListener(config[i], thisObj);
      }

      return;
    }

    const me = this,
          deprecatedEvents = me.$meta.getInherited('deprecatedEvents');

    if (typeof config === 'string') {
      // arguments[2] is thisObj if (eventname, handler, thisObj) form called.
      // Note that the other side of the if compares to undefined, so this will work.
      return me.addListener({
        [config]: thisObj,
        detachable: thisObj.detachable !== false,
        thisObj: oldThisObj
      });
    } else {
      // Capture the default thisObj.
      thisObj = config.thisObj = config.thisObj !== undefined ? config.thisObj : thisObj;

      for (const key in config) {
        // Skip special properties or events without handlers (convenient syntax with optional handlers)
        if (!specialProperties[key] && config[key] != null) {
          // comparing should be case insensitive
          const // comparing should be case insensitive
          eventName = key.toLowerCase(),
                deprecatedEvent = deprecatedEvents === null || deprecatedEvents === void 0 ? void 0 : deprecatedEvents[eventName],
                events = me.eventListeners || (me.eventListeners = {}),
                listenerSpec = config[key],
                expires = listenerSpec.expires || config.expires,
                listener = {
            fn: typeof listenerSpec === 'object' ? listenerSpec.fn : listenerSpec,
            thisObj: listenerSpec.thisObj !== undefined ? listenerSpec.thisObj : thisObj,
            args: listenerSpec.args || config.args,
            prio: listenerSpec.prio !== undefined ? listenerSpec.prio : config.prio !== undefined ? config.prio : 0,
            once: listenerSpec.once !== undefined ? listenerSpec.once : config.once !== undefined ? config.once : false,
            buffer: listenerSpec.buffer || config.buffer,
            throttle: listenerSpec.throttle || config.throttle,
            catchAll: key === 'catchAll'
          };

          if (deprecatedEvent) {
            const {
              product,
              invalidAsOfVersion,
              message
            } = deprecatedEvent;
            VersionHelper.deprecate(product, invalidAsOfVersion, message);
          }

          if (expires) {
            // Extract expires : { delay : 100, alt : 'onExpireFn' }
            const delayable = me.isDelayable ? me : globalThis,
                  {
              alt
            } = expires,
                  delay = alt ? expires.delay : expires;
            delayable.setTimeout(() => {
              me.un(eventName, listener); // If we make it here and the handler has not been called, invoke the alt handler

              if (alt && !listener.called) {
                me.callback(alt, thisObj);
              }
            }, delay);
          }

          let listeners = events[eventName] || (events[eventName] = []);

          if (listeners.$firing) {
            events[eventName] = listeners = listeners.slice();
          } // Insert listener directly in prio order

          listeners.splice(ArrayHelper.findInsertionIndex(listener, listeners, priorityComparator, listeners.length), 0, listener);

          if (!me.onListen.$nullFn && listeners.length < 2) {
            me.onListen(eventName);
          }
        }
      }

      if (config.relayAll) {
        me.relayAll(config.relayAll);
      } // Hook into the thisObj's destruction sequence to remove these listeners.
      // Pass the default thisObj in for use when it comes to destruction time.

      if (thisObj && thisObj !== me) {
        me.attachAutoDetacher(config, thisObj);
      }

      const detachable = config.detachable !== false,
            name = config.name,
            destroy = config.expires || detachable || name ? () => {
        // drop listeners if not destroyed yet
        if (!me.isDestroyed) {
          me.removeListener(config, thisObj);
        }
      } : null;

      if (destroy) {
        var _thisObj;

        destroy.eventer = me;
        destroy.listenerName = name;

        if (name && (_thisObj = thisObj) !== null && _thisObj !== void 0 && _thisObj.trackDetacher) {
          thisObj.trackDetacher(name, destroy);
        }

        if (config.expires) {
          me.delay(destroy, config.expires);
        }

        if (detachable) {
          return destroy;
        }
      }
    }
  }
  /**
   * Alias for {@link #function-addListener}. Adds an event listener. This method accepts parameters in the following format:
   *
   * ```javascript
   *  myObject.on({
   *      thisObj    : this,          // The this reference for the handlers
   *      eventname2 : 'functionName' // Resolved at invocation time using the thisObj,
   *      otherevent : {
   *          fn      : 'handlerFnName',
   *          once    : true          // Just this handler is auto-removed on fire
   *      },
   *      yetanother  : {
   *          fn      : 'yetAnotherHandler',
   *          args    : [ currentState1, currentState2 ] // Capture info to be passed to handler
   *      },
   *      prio        : 100           // Higher prio listeners are called before lower
   *  });
   * ```
   *
   * When listeners have a `thisObj` option, they are linked to the lifecycle of that object.
   * When it is destroyed, those listeners are removed.
   *
   * A simpler signature may be used when only adding a listener for one event and no extra options
   * (such as `once` or `delay`) are required:
   *
   * ```javascript
   * myObject.on('click', myController.handleClicks, myController);
   * ```
   *
   * The args in this simple case are `eventName`, `handler` and `thisObj`
   *
   * @param {Object|String} config An object containing listener definitions.
   * @param {Object} [config.thisObj] The `this` reference for all listeners.
   * (May be overridden if a handler is specified in object form)
   * @param {Boolean} [config.once] Specify as `true` to remove the listener as soon as it is invoked.
   * @param {Number|Object} [config.expires] The listener only waits for a specified time before
   * being removed. The value may be a number or an object containing an expiry handler.
   * @param {Number} [config.expires.delay] How long to wait for the event for.
   * @param {String|Function} [config.expires.alt] The function to call when the listener expires
   * **without having been triggered**.
   * @param {Object[]} [config.args] An array of arguments to be passed to the handler before the event object.
   * @param {Object|Function} [thisObj] Default `this` reference for all listeners in the config object.
   * @param {Object} [oldThisObj] The `this` reference if the old signature starting with a string event name is used..
   * @returns {Function} Returns a detacher function unless configured with `detachable: false`. Call detacher to remove listeners
   */

  on(...args) {
    return this.addListener(...args);
  }
  /**
   * Shorthand for {@link #function-removeListener}
   * @param {Object|String} config A config object or the event name
   * @param {Object|Function} thisObj `this` reference for all listeners, or the listener function
   * @param {Object} oldThisObj `this` The this object for the legacy way of adding listeners
   */

  un(...args) {
    this.removeListener(...args);
  }

  get listeners() {
    return this.eventListeners;
  }

  changeListeners(listeners) {
    // If we are receiving class listeners, add them early, and they do not become
    // the configured listeners, and are not removed by setting listeners during the lifecycle.
    if (this.isConfiguring) {
      if (listeners) {
        this.on(listeners, this);
      }
    } // Setting listeners after config time clears the old set and adds the new.
    // This will initially happen at the tail end of the constructor when config
    // listeners are set.
    else {
      // Configured listeners use this as the thisObj by default.
      // Flatten using Objects.assign because it may have been part of
      // a prototype chained default configuration of another object.
      // eg: the tooltip config of a Widget.
      // listener object blocks from multiple configuration levels are pushed
      // onto an array (see listeners merge function in configurable block above).
      // If this has happened, each entry must be processed like this.
      if (Array.isArray(listeners)) {
        for (let i = 0, l = listeners[0], {
          length
        } = listeners; i < length; l = listeners[++i]) {
          if (!('thisObj' in l)) {
            listeners[i] = Objects.assign({
              thisObj: this
            }, l);
          }
        }
      } else if (listeners && !('thisObj' in listeners)) {
        listeners = Objects.assign({
          thisObj: this
        }, listeners);
      }

      return listeners;
    }
  }

  updateListeners(listeners, oldListeners) {
    // Only configured listeners get here. Class listeners are added by changeListeners.
    oldListeners && this.un(oldListeners);
    listeners && this.on(listeners);
  }
  /**
   * Removes an event listener. Same API signature as {@link #function-addListener}
   * @param {Object|String} config A config object or the event name
   * @param {Object|Function} thisObj `this` reference for all listeners, or the listener function
   * @param {Object} oldThisObj `this` The this object for the legacy way of adding listeners
   */

  removeListener(config, thisObj = config.thisObj, oldThisObj) {
    const me = this;

    if (typeof config === 'string') {
      return me.removeListener({
        [config]: thisObj
      }, oldThisObj);
    }

    Object.entries(config).forEach(([eventName, listenerToRemove]) => {
      if (!specialProperties[eventName] && listenerToRemove != null) {
        eventName = eventName.toLowerCase();
        const eventListeners = me.eventListeners,
              index = me.findListener(eventName, listenerToRemove, thisObj);

        if (index >= 0) {
          let listeners = eventListeners[eventName];

          if (listeners.length > 1) {
            if (listeners.$firing) {
              eventListeners[eventName] = listeners = listeners.slice();
            } // NOTE: we cannot untrack any detachers here because we may only be
            // removing some of its listeners

            listeners.splice(index, 1);
          } else {
            delete eventListeners[eventName];

            if (!me.onUnlisten.$nullFn) {
              me.onUnlisten(eventName);
            }
          }
        }
      }
    });

    if (config.thisObj && !config.thisObj.isDestroyed) {
      me.detachAutoDetacher(config);
    }
  }
  /**
   * Finds the index of a particular listener to the named event. Returns `-1` if the passed
   * function/thisObj listener is not present.
   * @param {String} eventName The name of an event to find a listener for.
   * @param {String|Function} listenerToFind The handler function to find.
   * @param {Object} defaultThisObj The `thisObj` for the required listener.
   * @internal
   */

  findListener(eventName, listenerToFind, defaultThisObj) {
    var _this$eventListeners;

    const eventListeners = (_this$eventListeners = this.eventListeners) === null || _this$eventListeners === void 0 ? void 0 : _this$eventListeners[eventName],
          fn = listenerToFind.fn || listenerToFind,
          thisObj = listenerToFind.thisObj || defaultThisObj;

    if (eventListeners) {
      for (let listenerEntry, i = 0, {
        length
      } = eventListeners; i < length; i++) {
        listenerEntry = eventListeners[i];

        if (listenerEntry.fn === fn && listenerEntry.thisObj === thisObj) {
          return i;
        }
      }
    }

    return -1;
  }
  /**
   * Check if any listener is registered for the specified eventName
   * @param {String} eventName
   * @returns {Boolean} `true` if listener is registered, otherwise `false`
   */

  hasListener(eventName) {
    var _this$eventListeners2;

    return Boolean((_this$eventListeners2 = this.eventListeners) === null || _this$eventListeners2 === void 0 ? void 0 : _this$eventListeners2[eventName === null || eventName === void 0 ? void 0 : eventName.toLowerCase()]);
  }
  /**
   * Relays all events through another object that also implements Events mixin. Adds a prefix to the event name
   * before relaying, for example add -> storeAdd
   * ```
   * // Relay all events from store through grid, will make it possible to listen for store events prefixed on grid:
   * 'storeLoad', 'storeChange', 'storeRemoveAll' etc.
   * store.relayAll(grid, 'store');
   *
   * grid.on('storeLoad', () => console.log('Store loaded');
   * ```
   * @param {Core.mixin.Events} through Object to relay the events through, needs to mix Events mixin in
   * @param {String} prefix Prefix to add to event name
   * @param {Boolean} [transformCase] Specify false to prevent making first letter of event name uppercase
   */

  relayAll(through, prefix, transformCase = true) {
    if (!this.relayAllTargets) {
      this.relayAllTargets = [];
    }

    const relayAllTargets = this.relayAllTargets;
    through.on('beforeDestroy', ({
      source
    }) => {
      if (source === through) {
        const configs = relayAllTargets.filter(r => r.through === through);
        configs.forEach(config => ArrayHelper.remove(relayAllTargets, config));
      }
    });
    relayAllTargets.push({
      through,
      prefix,
      transformCase
    });
  }
  /**
   * Removes all listeners registered to this object
   */

  removeAllListeners() {
    const listeners = this.eventListeners;
    let i, thisObj;

    for (const event in listeners) {
      const bucket = listeners[event]; // We iterate backwards since we call removeListener which will splice out of
      // this array as we go...

      for (i = bucket.length; i-- > 0;) {
        var _thisObj2;

        const cfg = bucket[i];
        this.removeListener(event, cfg);
        thisObj = cfg.thisObj;

        if ((_thisObj2 = thisObj) !== null && _thisObj2 !== void 0 && _thisObj2.untrackDetachers) {
          thisObj.untrackDetachers(this);
        }
      }
    }
  }

  relayEvents(source, eventNames, prefix = '') {
    const listenerConfig = {
      detachable: true
    };
    eventNames.forEach(eventName => {
      listenerConfig[eventName] = (event, ...params) => {
        return this.trigger(prefix + eventName, event, ...params);
      };
    });
    return source.on(listenerConfig);
  }
  /**
   * This method is called when the first listener for an event is added.
   * @param {String} eventName
   * @internal
   */

  onListen() {}
  /**
   * This method is called when the last listener for an event is removed.
   * @param {String} eventName
   * @internal
   */

  onUnlisten() {}

  destructorInterceptor() {
    const {
      autoDetachers,
      target,
      oldDestructor
    } = this; // Remove listeners first, so that they do not fire during destruction.
    // The observable being listened to by the thisObj may already have
    // been destroyed in a clean up sequence

    for (let i = 0; i < autoDetachers.length; i++) {
      const {
        dispatcher,
        config
      } = autoDetachers[i];

      if (!dispatcher.isDestroyed) {
        dispatcher.removeListener(config, target);
      }
    }

    oldDestructor.call(target);
  }
  /**
   * Internal function used to hook destroy() calls when using thisObj
   * @private
   */

  attachAutoDetacher(config, thisObj) {
    const target = config.thisObj || thisObj,
          // If it's a Bryntum Base subclass, hook doDestroy, otherwise, destroy
    destructorName = 'doDestroy' in target ? 'doDestroy' : 'destroy';

    if (destructorName in target) {
      let {
        $autoDetachers
      } = target;

      if (!$autoDetachers) {
        target.$autoDetachers = $autoDetachers = [];
      }

      if (!target.$oldDestructor) {
        target.$oldDestructor = target[destructorName]; // Binding instead of using closure (used to use FunctionHelper.createInterceptor) to not retain target
        // when detaching manually

        target[destructorName] = this.destructorInterceptor.bind({
          autoDetachers: $autoDetachers,
          oldDestructor: target.$oldDestructor,
          target
        });
      }

      $autoDetachers.push({
        config,
        dispatcher: this
      });
    } else {
      target[destructorName] = () => {
        this.removeListener(config);
      };
    }
  }
  /**
   * Internal function used restore hooked destroy() calls when using thisObj
   * @private
   */

  detachAutoDetacher(config) {
    const target = config.thisObj; // Restore old destructor and remove from auto detachers only if we are not called as part of destruction.
    // (Altering $autoDetachers affects destruction iterating over them, breaking it. It is pointless to clean up
    // during destruction anyway, since everything gets removed)

    if (target.$oldDestructor && !target.isDestroying) {
      ArrayHelper.remove(target.$autoDetachers, target.$autoDetachers.find(detacher => detacher.config === config && detacher.dispatcher === this));

      if (!target.$autoDetachers.length) {
        target['doDestroy' in target ? 'doDestroy' : 'destroy'] = target.$oldDestructor;
        target.$oldDestructor = null;
      }
    }
  } //endregion
  //region Promise based workflow
  // experimental, used in tests to support async/await workflow

  await(eventName, options = {
    checkLog: true,
    resetLog: true,
    args: null
  }) {
    const me = this;

    if (options === false) {
      options = {
        checkLog: false
      };
    }

    const {
      args
    } = options;
    return new Promise(resolve => {
      var _me$_triggered;

      // check if previously triggered?
      if (options.checkLog && (_me$_triggered = me._triggered) !== null && _me$_triggered !== void 0 && _me$_triggered[eventName]) {
        // resolve immediately, no params though...
        resolve(); // reset log to be able to await again

        if (options.resetLog) {
          me.clearLog(eventName);
        }
      } // This branch will listen for events until catches one with specific arguments

      if (args) {
        const detacher = me.on({
          [eventName]: (...params) => {
            const argsOk = Object.keys(args).every(key => {
              return key in params[0] && params[0][key] === args[key];
            });

            if (argsOk) {
              // resolve when event is fired with required arguments
              resolve(...params); // reset log to be able to await again

              if (options.resetLog) {
                me.clearLog(eventName);
              }

              detacher();
            }
          },
          prio: -10000 // Let others do their stuff first

        });
      } else {
        me.on({
          [eventName]: (...params) => {
            // resolve when event is caught
            resolve(...params); // reset log to be able to await again

            if (options.resetLog) {
              me.clearLog(eventName);
            }
          },
          prio: -10000,
          // Let others do their stuff first
          once: true // promises can only be resolved once anyway

        });
      }
    });
  }

  clearLog(eventName) {
    if (this._triggered) {
      if (eventName) {
        delete this._triggered[eventName];
      } else {
        this._triggered = {};
      }
    }
  } //endregion
  //region Trigger

  /**
   * Triggers an event, calling all registered listeners with the supplied arguments. Returning false from any listener
   * makes function return false.
   * @param {String} eventName Event name for which to trigger listeners
   * @param {Object} [param] Single parameter passed on to listeners, source property will be added to it (this)
   * @param {Boolean} [param.bubbles] Pass as `true` to indicate that the event will bubble up the widget
   * ownership hierarchy. For example up a Menu->parent Menu tree, or a Field->Container tree.
   * @returns {Boolean|Promise} Returns false if any listener returned false, or a Promise yielding
   * true / false based on what is returned from the async listener functions, otherwise true
   */

  trigger(eventName, param) {
    var _me$eventListeners, _me$eventListeners2, _me$bubbleEvents, _handlerPromises;

    // TODO: should returning false from a listener really prevent other listeners from executing?
    const me = this,
          name = eventName.toLowerCase(),
          {
      eventsSuspended,
      relayAllTargets,
      callOnFunctions
    } = me;
    let listeners = (_me$eventListeners = me.eventListeners) === null || _me$eventListeners === void 0 ? void 0 : _me$eventListeners[name],
        handlerPromises; // log trigger, used by experimental promise support to resolve immediately when needed

    if (!me._triggered) {
      me._triggered = {};
    }

    me._triggered[eventName] = true;

    if (eventsSuspended) {
      if (eventsSuspended.shouldQueue) {
        eventsSuspended.queue.push(arguments);
      }

      return true;
    } // Include catchall listener for all events.
    // Do not push the catchAll listeners onto the events own listener array.

    if ((_me$eventListeners2 = me.eventListeners) !== null && _me$eventListeners2 !== void 0 && _me$eventListeners2.catchall) {
      (listeners = listeners ? listeners.slice() : []).push(...me.eventListeners.catchall); // The catchAll listeners must honour their prio settings.

      listeners.sort(priorityComparator);
    }

    if (!listeners && !relayAllTargets && !callOnFunctions) {
      return true;
    } // default to include source : this in param

    if (param) {
      if (!('source' in param)) {
        if (Object.isExtensible(param)) {
          param.source = me;
        } else {
          param = Object.setPrototypeOf({
            source: me
          }, param);
        }
      }
    } else {
      param = {
        source: me
      };
    } // Lowercased event name should be the "type" property in keeping with DOM events.

    if (param.type !== name) {
      // Create instance property because "type" is read only
      if (param.constructor !== Object) {
        Reflect.defineProperty(param, 'type', {
          get: () => name
        });
      } else {
        param.type = name;
      }
    } // Bubble according to `bubbleEvents` config if `bubbles` is not explicitly set

    if (!('bubbles' in param) && (_me$bubbleEvents = me.bubbleEvents) !== null && _me$bubbleEvents !== void 0 && _me$bubbleEvents[eventName]) {
      param.bubbles = me.bubbleEvents[eventName];
    }

    if (callOnFunctions) {
      const fnName = 'on' + StringHelper.capitalize(eventName);

      if (fnName in me) {
        var _me$pluginFunctionCha;

        const result = me.callback(me[fnName], me, [param]);
        let inhibit;

        if (Objects.isPromise(result)) {
          (handlerPromises || (handlerPromises = [])).push(result);
        } else {
          inhibit = result === false || inhibit;
        } // See if the called function was injected into the instance
        // masking an implementation in the prototype.
        // we must call the class's implementation after the injected one.
        // unless it's an injected chained function, in which case it will have been called above.
        // Note: The handler may have resulted in destruction.

        if (!me.isDestroyed && hasOwnProperty$5.call(me, fnName) && !((_me$pluginFunctionCha = me.pluginFunctionChain) !== null && _me$pluginFunctionCha !== void 0 && _me$pluginFunctionCha[fnName])) {
          const myProto = Object.getPrototypeOf(me);

          if (fnName in myProto) {
            const result = myProto[fnName].call(me, param);

            if (Objects.isPromise(result)) {
              (handlerPromises || (handlerPromises = [])).push(result);
            } else {
              inhibit = result === false || inhibit;
            } // A handler may have resulted in destruction.

            if (me.isDestroyed) {
              return;
            }
          }
        } // Returning false from an on-function prevents further triggering

        if (inhibit) {
          return false;
        }
      }
    }

    if (listeners) {
      let i = 0,
          ret; // Let add/removeListener know that we're using the array to protect against a situation where an event
      // listener changes the listeners when triggering the event.

      listeners.$firing = true; // If any listener resulted in our destruction, abort.

      for (i; i < listeners.length && !me.isDestroyed && ret !== false; i++) {
        const listener = listeners[i];
        let handler,
            thisObj = listener.thisObj; // Listeners that have thisObj are auto removed when thisObj is destroyed. If thisObj is destroyed from
        // a listener we might still end up here, since listeners are sliced and not affected by the removal

        if (!thisObj || !thisObj.isDestroyed) {
          // Flag for the expiry timer
          listener.called = true;

          if (listener.once) {
            me.removeListener(name, listener);
          } // prepare handler function

          if (typeof listener.fn === 'string') {
            if (thisObj) {
              handler = thisObj[listener.fn];
            } // keep looking for the callback in the hierarchy

            if (!handler) {
              const result = me.resolveCallback(listener.fn);
              handler = result.handler;
              thisObj = result.thisObj;
            }
          } else {
            handler = listener.fn;
          } // if `buffer` option is provided, the handler will be wrapped into buffer function,
          // but only once on the first call

          if (listener.buffer) {
            if (!listener.bufferFn) {
              const buffer = Number(listener.buffer);

              if (typeof buffer !== 'number' || isNaN(buffer)) {
                throw new Error(`Incorrect type for buffer, got "${buffer}" (expected a Number)`);
              }

              listener.bufferFn = FunctionHelper.createBuffered(handler, buffer, listener.thisObj, listener.args);
            }

            handler = listener.bufferFn;
          } // if `throttle` option is provided, the handler will be called immediately, but all the rest calls
          // that happened during time specified in `throttle`, will be delayed and glued into 1 call

          if (listener.throttle) {
            const throttle = Number(listener.throttle);

            if (typeof throttle !== 'number' || isNaN(throttle)) {
              throw new Error(`Incorrect type for throttle, got "${throttle}" (expected a Number)`);
            }

            if (!listener.throttledFn) {
              listener.throttledFn = FunctionHelper.createThrottled(handler, throttle, listener.thisObj, listener.args);
            }

            handler = listener.throttledFn;
          }

          ret = handler.call(thisObj || me, ...(listener.args || []), param);

          if (Objects.isPromise(ret)) {
            // If a handler is async (returns a Promise), then collect all Promises.
            // At the end we return a Promise which encapsulates all returned Promises
            // or, if only one handler was async, *the* Promise.
            // Don't allocate an Array until we have to.
            (handlerPromises || (handlerPromises = [])).push(ret);
          }
        }
      }

      listeners.$firing = false;

      if (ret === false) {
        return ret;
      }
    } // relay all?

    if (relayAllTargets) {
      relayAllTargets.forEach(config => {
        let name = eventName;
        if (config.transformCase) name = StringHelper.capitalize(name);
        if (config.prefix) name = config.prefix + name;
        if (config.through.trigger(name, param) === false) return false;
      });
    } // Use DOM standard event property name to indicate that the event
    // bubbles up the owner axis.
    // False from any handler cancels the bubble.
    // Must also avoid owner if any handlers destroyed the owner.

    if (param.bubbles && me.owner && !me.owner.isDestroyed) {
      return me.owner.trigger(eventName, param);
    } // If any handlers were async functions (returned a Promise), then return a Promise
    // which resolves when they all resolve.

    if ((_handlerPromises = handlerPromises) !== null && _handlerPromises !== void 0 && _handlerPromises.length) {
      return new Promise(resolve => {
        Promise.all(handlerPromises).then(promiseResults => {
          const finalResult = !promiseResults.some(result => result === false);
          resolve(finalResult);
        });
      });
    }

    return true;
  }
  /**
   * Prevents events from being triggered until {@link #function-resumeEvents()} is called. Optionally queues events that are triggered while
   * suspended. Multiple calls stack to require matching calls to `resumeEvents()` before actually resuming.
   * @param {Boolean} queue Specify true to queue events triggered while suspended
   */

  suspendEvents(queue = false) {
    const eventsSuspended = this.eventsSuspended || (this.eventsSuspended = {
      shouldQueue: queue,
      queue: [],
      count: 0
    });
    eventsSuspended.count++;
  }
  /**
   * Resume event triggering after a call to {@link #function-suspendEvents()}. If any triggered events were queued they will be triggered.
   * @returns `true` if events have been resumed (multiple calls to suspend require an equal number of resume calls to resume).
   */

  resumeEvents() {
    const suspended = this.eventsSuspended;

    if (suspended) {
      if (--suspended.count === 0) {
        this.eventsSuspended = null;

        if (suspended.shouldQueue) {
          for (const queued of suspended.queue) {
            this.trigger(...queued);
          }
        }
      }
    }

    return !Boolean(this.eventsSuspended);
  } //endregion

});

/**
 * @module Core/helper/AsyncHelper
 */

/**
 * A helper class to make asynchronous tasks `await` friendly.
 */
class AsyncHelper {
  /**
   * Returns a promise that resolves on next animation frame.
   * ```
   *  async method() {
   *      // do work
   *      await AsyncHelper.animationFrame();
   *      // do more work
   *  }
   * ```
   * @async
   * @returns {Promise}
   */
  static animationFrame() {
    return new Promise(resolve => {
      requestAnimationFrame(resolve);
    });
  }
  /**
   * Returns a promise that resolves after a specified number of milliseconds.
   * ```
   *  async method() {
   *      await AsyncHelper.sleep(10);
   *      // ...
   *  }
   * ```
   * @async
   * @param {Number} millis The number of milliseconds to sleep.
   * @returns {Promise}
   */

  static sleep(millis) {
    return new Promise(resolve => {
      setTimeout(() => {
        resolve();
      }, millis);
    });
  }
  /**
   * Returns a promise that resolves as soon as possible, allowing the browser to minimally process other messages.
   * This is the shortest possible delay the browser offers, so be aware that it does not necessarily allow the
   * browser to paint or reflow if used in a long loop. It does, however, allow other async methods to execute.
   * ```
   *  async method() {
   *      await AsyncHelper.yield();
   *      // ...
   *  }
   * ```
   * @returns {Promise}
   */

  static yield() {
    return Promise.resolve();
  }

}
AsyncHelper._$name = 'AsyncHelper';

var _BrowserHelper$queryS;

const paramValueRegExp = /^(\w+)=(.*)$/,
      // For CodePens in docs, adjust URLs to be absolute
resourceRoot = globalThis.DocsBrowserInstance && ((_BrowserHelper$queryS = BrowserHelper.queryString) === null || _BrowserHelper$queryS === void 0 ? void 0 : _BrowserHelper$queryS.__resourceRoot) || '',
      parseParams = function (paramString) {
  const result = {},
        params = paramString.split('&'); // loop through each 'filter={"field":"name","operator":"=","value":"Sweden","caseSensitive":true}' string
  // So we cannot use .split('=')

  for (const nameValuePair of params) {
    const [match, name, value] = paramValueRegExp.exec(nameValuePair),
          decodedName = decodeURIComponent(name),
          decodedValue = decodeURIComponent(value);

    if (match) {
      let paramValue = result[decodedName];

      if (paramValue) {
        if (!Array.isArray(paramValue)) {
          paramValue = result[decodedName] = [paramValue];
        }

        paramValue.push(decodedValue);
      } else {
        result[decodedName] = decodedValue;
      }
    }
  }

  return result;
};
/**
 * Simplifies Ajax requests. Uses fetch & promises.
 *
 * ```javascript
 * AjaxHelper.get('some-url').then(response => {
 *     // process request response here
 * });
 * ```
 *
 * Uploading file to server via FormData interface.
 * Please visit [FormData](https://developer.mozilla.org/en-US/docs/Web/API/FormData) for details.
 *
 * ```javascript
 * const formData = new FormData();
 * formData.append('file', 'fileNameToUpload');
 * AjaxHelper.post('file-upload-url', formData).then(response => {
 *     // process request response here
 * });
 * ```
 *
 */

class AjaxHelper {
  /**
   * Make a request (using GET) to the specified url.
   * @param {String} url Url
   * @param {Object} [options] The options for the `fetch` API. Please see [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details
   * @param {Object} [options.queryParams] A key-value pair Object containing the params to add to the query string
   * @param {Object} [options.headers] Any headers you want to add to your request, contained within a `Headers object or an object literal with ByteString values
   * @param {Object} [options.body] Any body that you want to add to your request: this can be a Blob, BufferSource, FormData, URLSearchParams, or USVString object. Note that a request using the GET or HEAD method cannot have a body.
   * @param {Object} [options.mode] The mode you want to use for the request, e.g., cors, no-cors, or same-origin.
   * @param {Object} [options.credentials] The request credentials you want to use for the request: omit, same-origin, or include. To automatically send cookies for the current domain, this option must be provided
   * @param {Object} [options.parseJson] Specify `true` to parses the response and attach the resulting object to the `Response` object as `parsedJson`
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   */
  static get(url, options) {
    return this.fetch(url, options);
  }
  /**
   * POST data to the specified URL.
   * @param {String} url The URL
   * @param {String|Object|FormData} payload The data to post. If an object is supplied, it will be stringified
   * @param {Object} options The options for the `fetch` API. Please see [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details
   * @param {Object} [options.queryParams] A key-value pair Object containing the params to add to the query string
   * @param {Object} [options.headers] Any headers you want to add to your request, contained within a `Headers object or an object literal with ByteString values
   * @param {Object} [options.body] Any body that you want to add to your request: this can be a Blob, BufferSource, FormData, URLSearchParams, or USVString object. Note that a request using the GET or HEAD method cannot have a body.
   * @param {Object} [options.mode] The mode you want to use for the request, e.g., cors, no-cors, or same-origin.
   * @param {Object} [options.credentials] The request credentials you want to use for the request: omit, same-origin, or include. To automatically send cookies for the current domain, this option must be provided
   * @param {Object} [options.parseJson] Specify `true` to parses the response and attach the resulting object to the `Response` object as `parsedJson`
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   */

  static post(url, payload, options = {}) {
    if (!(payload instanceof FormData) && !(typeof payload === 'string')) {
      payload = JSON.stringify(payload);
      options.headers = options.headers || {};
      options.headers['Content-Type'] = options.headers['Content-Type'] || 'application/json';
    }

    return this.fetch(url, Object.assign({
      method: 'POST',
      body: payload
    }, options));
  }
  /**
   * Fetch the specified resource using the `fetch` API.
   * @param {String} url object to fetch
   * @param {Object} options The options for the `fetch` API. Please see [fetch API](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch) for details
   * @param {Object} [options.method] The request method, e.g., GET, POST
   * @param {Object} [options.queryParams] A key-value pair Object containing the params to add to the query string
   * @param {Object} [options.headers] Any headers you want to add to your request, contained within a `Headers object or an object literal with ByteString values
   * @param {Object} [options.body] Any body that you want to add to your request: this can be a Blob, BufferSource, FormData, URLSearchParams, or USVString object. Note that a request using the GET or HEAD method cannot have a body.
   * @param {Object} [options.mode] The mode you want to use for the request, e.g., cors, no-cors, or same-origin.
   * @param {Object} [options.credentials] The request credentials you want to use for the request: omit, same-origin, or include. To automatically send cookies for the current domain, this option must be provided
   * @param {Object} [options.parseJson] Specify `true` to parses the response and attach the resulting object to the `Response` object as `parsedJson`
   * @returns {Promise} The fetch Promise, which can be aborted by calling a special `abort` method
   */

  static fetch(url, options = {}) {
    let controller; // AbortController is not supported by LockerService
    // https://github.com/bryntum/support/issues/3689

    if (typeof AbortController !== 'undefined') {
      controller = options.abortController = new AbortController();
      options.signal = controller.signal;
    }

    if (!('credentials' in options)) {
      options.credentials = 'include';
    }

    if (options.queryParams) {
      const params = Object.entries(options.queryParams);

      if (params.length) {
        url += (url.includes('?') ? '&' : '?') + params.map(([param, value]) => `${param}=${encodeURIComponent(value)}`).join('&');
      }
    } // Promise that will be resolved either when network request is finished or when json is parsed

    const promise = new Promise((resolve, reject) => {
      fetch(resourceRoot + url, options).then(response => {
        if (options.parseJson) {
          response.json().then(json => {
            response.parsedJson = json;
            resolve(response);
          }).catch(error => {
            response.parsedJson = null;
            response.error = error;
            reject(response);
          });
        } else {
          resolve(response);
        }
      }).catch(error => {
        error.stack = promise.stack; // Unless calling code aborted the request, reject this operation

        if (error.name !== 'AbortError') {
          reject(error);
        }
      });
    });
    promise.stack = new Error().stack;

    promise.abort = function () {
      var _controller;

      (_controller = controller) === null || _controller === void 0 ? void 0 : _controller.abort();
    };

    return promise;
  }
  /**
   * Registers the passed URL to return the passed mocked up Fetch Response object to the
   * AjaxHelper's promise resolve function.
   * @param {String} url The url to return mock data for
   * @param {Object|Function} response A mocked up Fetch Response object which must contain
   * at least a `responseText` property, or a function to which the `url` and a `params` object
   * and the `Fetch` `options` object is passed which returns that.
   * @param {String} response.responseText The data to return.
   * @param {Boolean} [response.synchronous] resolve the Promise immediately
   * @param {Number} [response.delay=100] resolve the Promise after this number of milliseconds.
   */

  static mockUrl(url, response) {
    const me = this;
    (me.mockAjaxMap || (me.mockAjaxMap = {}))[url] = response; // Inject the override into the AjaxHelper instance

    if (!AjaxHelper.originalFetch) {
      AjaxHelper.originalFetch = AjaxHelper.fetch;
    }

    AjaxHelper.fetch = me.mockAjaxFetch.bind(me);
  }

  static mockAjaxFetch(url, options) {
    let urlAndParams = url.split('?'),
        result = this.mockAjaxMap[urlAndParams[0]],
        parsedJson = null;

    if (result) {
      if (typeof result === 'function') {
        result = result(urlAndParams[0], urlAndParams[1] && parseParams(urlAndParams[1]), options);
      }

      try {
        parsedJson = options.parseJson && JSON.parse(result.responseText);
      } catch (error) {
        parsedJson = null;
        result.error = error;
      }

      result = Object.assign({
        status: 200,
        ok: true,
        headers: new Headers(),
        statusText: 'OK',
        url: url,
        parsedJson: parsedJson,
        text: () => new Promise(resolve => {
          resolve(result.responseText);
        }),
        json: () => new Promise(resolve => {
          resolve(parsedJson);
        })
      }, result);
      return new Promise(function (resolve, reject) {
        if (result.synchronous) {
          resolve(result);
        } else {
          setTimeout(function () {
            resolve(result);
          }, 'delay' in result ? result.delay : 100);
        }
      });
    } else {
      return AjaxHelper.originalFetch(url, options);
    }
  }

}
AjaxHelper._$name = 'AjaxHelper';

/**
 * @module Core/localization/LocaleHelper
 */

/**
 * Provides locale management methods.
 */
class LocaleHelper {
  /**
   * Merges all properties of provided locales into new locale.
   * Locales are merged in order they provided and locales which go later replace same properties of previous locales.
   * @param {...Object} locales Locales to merge
   * @return {Object} Merged locale
   */
  static mergeLocales(...locales) {
    const result = {};
    locales.forEach(locale => {
      Object.keys(locale).forEach(key => {
        if (typeof locale[key] === 'object') {
          result[key] = _objectSpread2(_objectSpread2({}, result[key]), locale[key]);
        } else {
          result[key] = locale[key];
        }
      });
    });
    return result;
  }
  /**
   * Removes all properties from `locale` that are present in the provided `toTrim`.
   * @param {Object} locale Locale to process
   * @param {Object} toTrim Object enumerating properties that should be removed
   * @param {boolean} [silent=true] When `true` ignores missing properties that should be removed (default).
   * When `false` throws exceptions in such cases.
   */

  static trimLocale(locale, toTrim, silent = true) {
    const remove = (key, subKey) => {
      if (locale[key]) {
        if (subKey) {
          if (locale[key][subKey]) {
            delete locale[key][subKey];
          }
        } else {
          delete locale[key];
        }
      }
    };

    Object.keys(toTrim).forEach(key => {
      if (Object.keys(toTrim[key]).length > 0) {
        Object.keys(toTrim[key]).forEach(subKey => remove(key, subKey));
      } else {
        remove(key);
      }
    });
  }
  /**
   * Put the locale under `globalThis.bryntum.locales` to make sure it can be discovered automatically
   * @param {String} localeName Locale name
   * @param {Object} config Locale config
   */

  static publishLocale(localeName, config) {
    const bryntum = globalThis.bryntum = globalThis.bryntum || {},
          locales = bryntum.locales = bryntum.locales || {}; // Avoid registering locales twice

    locales[localeName] = !locales[localeName] ? config : this.mergeLocales(locales[localeName], config);
  }

}
LocaleHelper._$name = 'LocaleHelper';

/**
 * @module Core/localization/LocaleManager
 */
// Documented at the export below, to work for singleton

class LocaleManager extends Events(Base$1) {
  static get defaultConfig() {
    return {
      locales: {},
      // Enable strict locale checking by default for tests
      throwOnMissingLocale: VersionHelper.isTestEnv
    };
  }

  construct(...args) {
    const me = this;
    super.construct(...args);

    if (BrowserHelper.isBrowserEnv) {
      const scriptTag = document.querySelector('script[data-default-locale]'),
            {
        locales
      } = globalThis.bryntum || {};

      if (scriptTag) {
        me.defaultLocaleName = scriptTag.dataset.defaultLocale;
      }

      if (locales) {
        Object.keys(locales).forEach(localeName => {
          // keeping this check in case some client tries to use an old locale
          if (!localeName.startsWith('moment')) {
            const locale = locales[localeName];

            if (locale.extends) {
              me.extendLocale(locale.extends, locale);
            } else if (locale.localeName) {
              me.registerLocale(locale.localeName, {
                desc: locale.localeDesc,
                locale: locale
              });
            }
          }
        });

        if (!me.locale) {
          // English locale is built in, no need to apply it here since it will be applied anyway
          if (me.defaultLocaleName !== 'En') {
            // No locale applied, use default or first found
            me.applyLocale(me.defaultLocaleName || Object.keys(me.locales)[0]);
          }
        }
      }
    }
  }

  set locales(localeConfigs) {
    this._locales = localeConfigs;
  }
  /**
   * Get currently registered locales.
   * Returns an object with locale names (`localeName`) as properties.
   *
   * Example:
   * ```
   * const englishLocale = LocaleManager.locales.En;
   * ```
   *
   * this returns registered English locale object.
   * ```
   * {
   *   "desc": "English",
   *   "locale": {
   *     "localeName": "En",
   *     "localeDesc": "English",
   *
   *     ... (localization goes here)
   *
   *   }
   * }
   * ```
   * @readonly
   * @member {Object} locales
   */

  get locales() {
    return this._locales;
  }
  /**
   * Get/set currently used locale. Set a name of a locale to have it applied, or give a locale configuration to
   * have it registered and then applied
   * @member {Object} locale
   * @accepts {String|Object}
   */

  set locale(locale) {
    if (typeof locale === 'string') {
      this.applyLocale(locale);
    } else {
      if (!locale.locale) {
        locale = {
          locale,
          localeName: locale.localeName || 'custom'
        };
      }

      this.registerLocale(locale.localeName, locale);
      this.applyLocale(locale.localeName);
    }
  }

  get locale() {
    return this._locale;
  }
  /**
   * Register a locale to make it available for applying.
   * Registered locales are available in {@link Core.localization.LocaleManager#property-locales}.
   * @param {String} name Name of locale (for example `En` or `SvSE`)
   * @param {Object} config Object with localized properties
   * @function registerLocale
   */

  registerLocale(name, config) {
    var _me$locale;

    const me = this,
          isDefault = me.defaultLocaleName === name,
          isCurrent = ((_me$locale = me.locale) === null || _me$locale === void 0 ? void 0 : _me$locale.localeName) === name,
          isFirst = Object.keys(me.locales).length === 0;
    let currentLocale = me.locales[name]; // Avoid registering the same locale twice and merge configs

    if (!currentLocale) {
      me.locales[name] = config;
      currentLocale = me.locales[name];
    } else {
      if (config.exclude) {
        currentLocale.exclude = LocaleHelper.mergeLocales(currentLocale.exclude || {}, config.exclude);
      }

      currentLocale.locale = LocaleHelper.mergeLocales(currentLocale.locale, config.locale);
    } // If we have exclude in locale config we remove excluded on registering locale to avoid merging unwanted locale keys

    if (currentLocale.exclude) {
      LocaleHelper.trimLocale(currentLocale.locale, currentLocale.exclude);
    } // If no default locale specified, use the first one. otherwise apply the default when it is registered
    // also reapply if current locale is registered again (first grid, then scheduler etc).

    if (isDefault || !me.defaultLocaleName && (isFirst || isCurrent)) {
      me.internalApplyLocale(currentLocale);
    }
  }
  /**
   * Extend an already loaded locale to add additional translations.
   * @param {String} name Name of locale (for example `En` or `SvSE`)
   * @param {Object} config Object with localized properties
   * @function extendLocale
   */

  extendLocale(name, config) {
    var _this$locale;

    const locale = this.locales[name];

    if (!locale) {
      return false;
    }

    locale.locale = LocaleHelper.mergeLocales(locale.locale, config);
    delete locale.locale.extends; // If current loaded locale is the same then apply to reflect changes

    if (((_this$locale = this.locale) === null || _this$locale === void 0 ? void 0 : _this$locale.localeName) === name) {
      this.applyLocale(name);
    }

    return true;
  }

  internalApplyLocale(localeConfig) {
    this._locale = localeConfig.locale;
    /**
     * Fires when a locale is applied
     * @event locale
     * @param {Core.localization.LocaleManager} source The Locale manager instance.
     * @param {Object} locale Locale configuration
     */

    this.trigger('locale', localeConfig);
  }
  /**
   * Apply a locale. Locale must be defined in {@link Core.localization.LocaleManager#property-locales}.
   * If it is not loaded it will be loaded using AjaxHelper {@link Core.helper.AjaxHelper#function-get-static} request and then applied.
   * @param {String} name Name of locale to apply (for example `En` or `SvSE`)
   * @returns {Boolean|Promise}
   * @fires locale
   * @function applyLocale
   */

  applyLocale(name, forceApply = false, ignoreError = false) {
    const me = this,
          localeConfig = me.locales[name];

    if (localeConfig !== null && localeConfig !== void 0 && localeConfig.locale && me._locale === localeConfig.locale && !forceApply) {
      // no need to apply same locale again
      return true;
    } // ignoreError is used in examples where one example might have defined a locale not available in another

    if (!localeConfig) {
      if (ignoreError) {
        return true;
      }

      throw new Error(`Locale ${name} not registered`);
    }

    function internalApply() {
      me.internalApplyLocale(localeConfig);
    }

    if (!localeConfig.locale) {
      return new Promise((resolve, reject) => {
        me.loadLocale(localeConfig.path).then(response => {
          response.text().then(text => {
            // eslint-disable-next-line no-new-func
            const parseLocale = new Function(text);
            parseLocale();

            if (BrowserHelper.isBrowserEnv) {
              localeConfig.locale = globalThis.bryntum.locales[name];
            }

            internalApply();
            resolve(localeConfig);
          });
        }).catch(response => reject(response));
      });
    }

    internalApply();
    return true;
  }
  /**
   * Loads a locale using AjaxHelper {@link Core.helper.AjaxHelper#function-get-static} request.
   * @private
   * @param {String} path Path to locale file
   * @returns {Promise}
   * @function loadLocale
   */

  loadLocale(path) {
    return AjaxHelper.get(path);
  }
  /**
   * Specifies if {@link Core.localization.Localizable#function-L-static Localizable.L()} function would throw error if no localization found in runtime
   * @member {Boolean} throwOnMissingLocale
   * @default false
   */

  set throwOnMissingLocale(value) {
    this._throwOnMissingLocale = value;
  }

  get throwOnMissingLocale() {
    return this._throwOnMissingLocale;
  }

}

const LocaleManagerSingleton = new LocaleManager();

/**
 * @module Core/localization/Localizable
 */

const ObjectProto = Object.getPrototypeOf(Object),
      localeRe = /L{.*?}/g,
      capturelocaleRe = /L{(.*?)}/g,
      classMatchRe = /((.*?)\.)?(.+)/g,
      escape$1 = txt => txt.replace(/{(\d+)}/gm, '[[$1]]'),
      unescape = txt => txt.replace(/\[\[(\d+)]]/gm, '{$1}'),
      emptyObject$b = Object.freeze(Object.create(null));
/**
 * Mixin that provides localization functionality to a class.
 *
 * ```
 * // Get localized string
 * grid.L('foo');
 * grid.L('L{foo}');
 * ```
 *
 * @mixin
 */

var Localizable = (Target => class Localizable extends (Target || Base$1) {
  static get $name() {
    return 'Localizable';
  }

  static get configurable() {
    return {
      /**
       * A class translations of which are used for translating this entity.
       * This is often used when translations of an item are defined on its container class.
       * For example:
       *
       * ```js
       * // Toolbar class that has some predefined items
       * class MyToolbar extends Toolbar {
       *
       *     static get $name() {
       *         return 'MyToolbar';
       *     }
       *
       *     static get defaultConfig() {
       *         return {
       *             // this specifies default configs for the items
       *             defaults : {
       *                 // will tell items to use the toolbar locale
       *                 localeClass : this
       *             },
       *
       *             items : [
       *                 // The toolbar has 2 buttons and translation for their texts will be searched in
       *                 // the toolbar locales
       *                 { text : 'Agree' },
       *                 { text : 'Disagree' }
       *             ]
       *         };
       *     }
       *
       *    ...
       * }
       * ```
       * So if one makes a locale for the `MyToolbar` class that will include `Agree` and `Disagree` string translations:
       * ```js
       *     ...
       *     MyToolbar : {
       *         Agree    : 'Yes, I agree',
       *         Disagree : 'No, I do not agree'
       *     }
       * ```
       * They will be used for the toolbar buttons and the button captions will say `Yes, I agree` and `No, I do not agree`.
       *
       * @config {Core.Base}
       * @typings {typeof Base}
       */
      localeClass: null,

      /**
       * List of properties which values should be translated automatically upon a locale applying.
       * In case there is a need to localize not typical value (not a String value or a field with re-defined setter/getter),
       * you could use 'localeKey' meta configuration.
       * Example:
       * ```js
       *  static get configurable() {
       *     return {
       *          localizableProperties : ['width'],
       *
       *          width : {
       *              value   : '54em', // default value here
       *              $config : {
       *                  localeKey : 'L{editorWidth}' // name of the property that will be used in localization file
       *              }
       *          }
       *      };
       *  }
       * ```
       * @config {String[]}
       */
      localizableProperties: {
        value: [],
        $config: {
          merge: 'distinct'
        }
      }
    };
  }

  static clsName(cls) {
    var _cls$prototype, _cls$prototype2;

    return typeof cls === 'string' ? cls : cls === ObjectProto ? 'Object' : cls.$$name || cls.name || ((_cls$prototype = cls.prototype) === null || _cls$prototype === void 0 ? void 0 : _cls$prototype.$$name) || ((_cls$prototype2 = cls.prototype) === null || _cls$prototype2 === void 0 ? void 0 : _cls$prototype2.name);
  }

  static parseLocaleString(text) {
    var _text;

    const matches = [];
    let m; // Parse locale text in case it's wrapped with L{foo}

    if ((_text = text) !== null && _text !== void 0 && _text.includes('L{')) {
      // Escape fix for {1}, {2} etc. in locale str
      text = escape$1(text);
      capturelocaleRe.lastIndex = 0;

      while ((m = capturelocaleRe.exec(text)) != null) {
        classMatchRe.lastIndex = 0; // Support for parsing class namespace L{Class.foo}

        const classMatch = classMatchRe.exec(m[1]);
        matches.push({
          match: unescape(m[0]),
          localeKey: unescape(classMatch[3]),
          localeClass: classMatch[2]
        });
      }
    }

    return matches.length > 0 ? matches : [{
      match: text,
      localeKey: text,
      localeClass: undefined
    }];
  }

  construct(config = {}, ...args) {
    // Base class applies configs.
    super.construct(config, ...args);
    LocaleManagerSingleton.on('locale', this.updateLocalization, this);
    this.updateLocalization();
  }

  get localeClass() {
    return this._localeClass || null;
  }

  localizeProperty(property) {
    var _me$$meta$configs$pro, _me$fieldMap, _me$fieldMap$property;

    const me = this,
          currentValue = Objects.getPath(me, property),
          // Grid.column.Column is Localizable too. It uses fields, not configs
    localeKey = ((_me$$meta$configs$pro = me.$meta.configs[property]) === null || _me$$meta$configs$pro === void 0 ? void 0 : _me$$meta$configs$pro.localeKey) || ((_me$fieldMap = me.fieldMap) === null || _me$fieldMap === void 0 ? void 0 : (_me$fieldMap$property = _me$fieldMap[property]) === null || _me$fieldMap$property === void 0 ? void 0 : _me$fieldMap$property.defaultValue);
    let localizedValue; // check if localeKey is defined and try to translate it

    if (localeKey) {
      localizedValue = Localizable.localize(localeKey, me, me.localeClass || me); // if a user set value directly in class definition, his value has a prio

      if (localizedValue && !(property in (me.initialConfig || emptyObject$b))) {
        Objects.setPath(me.isColumn ? me.data : me, property, localizedValue);
      }
    } else if (typeof currentValue === 'string') {
      me.originalLocales = me.originalLocales || {};
      localizedValue = Objects.getPath(me.originalLocales, property); // If we haven't saved original values yet let's do that

      if (localizedValue === undefined) {
        Objects.setPath(me.originalLocales, property, currentValue);
        localizedValue = currentValue;
      } // Doing localization from the original values

      if (localizedValue) {
        Objects.setPath(me, property, localizedValue = me.optionalL(localizedValue, me));
      }
    }

    return localizedValue || currentValue;
  }
  /**
   * Method that is triggered when applying a locale to the instance
   * (happens on the instance construction steps and when switching to another locale).
   *
   * The method can be overridden to dynamically translate the instance when locale is switched.
   * When overriding the method please make sure you call `super.updateLocalization()`.
   * @category Misc
   */

  updateLocalization() {
    var _this$localizableProp;

    (_this$localizableProp = this.localizableProperties) === null || _this$localizableProp === void 0 ? void 0 : _this$localizableProp.forEach(this.localizeProperty, this);
  }

  static getTranslation(text, templateData, localeCls) {
    const {
      locale
    } = LocaleManagerSingleton;
    let result = null,
        clsName,
        cls;

    if (locale) {
      // Iterate over all found localization entries
      for (const {
        match,
        localeKey,
        localeClass
      } of this.parseLocaleString(text)) {
        const translate = clsName => {
          var _locale$clsName;

          const translation = (_locale$clsName = locale[clsName]) === null || _locale$clsName === void 0 ? void 0 : _locale$clsName[localeKey];

          if (translation) {
            if (typeof translation === 'function') {
              result = templateData != null ? translation(templateData) : translation;
            } else if (typeof translation === 'object' || text === match) {
              result = translation;
            } // Likely string
            else {
              result = (result || text).replace(match, translation);
            } // Might have nested L{, recurse

            if (typeof translation === 'string' && translation.includes('L{')) {
              result = this.getTranslation(translation, templateData, localeCls);
            }
          }

          return translation;
        }; // Translate order
        // 1. Try to translate for current class
        // 2. Try to translate by Class hierarchy traversing prototypes
        // 3. Try to translate if Class is in {Class.foo} format

        let success = false;

        for (cls = localeCls; cls && (clsName = Localizable.clsName(cls)); cls = Object.getPrototypeOf(cls)) {
          if (success = translate(clsName)) {
            break;
          } else if (typeof cls === 'string') {
            break;
          }
        }

        if (!success && localeClass) {
          translate(localeClass);
        }
      }
    }

    return result;
  }
  /**
   * Get localized string, returns `null` if no localized string found.
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @internal
   */

  static localize(text, templateData = undefined, ...localeClasses) {
    var _localeClasses;

    // In case this static method is called directly third argument is not provided
    // just fallback to searching locales for the class itself
    if (((_localeClasses = localeClasses) === null || _localeClasses === void 0 ? void 0 : _localeClasses.length) === 0) {
      localeClasses = [this];
    }

    let translation = null;
    localeClasses.some(cls => {
      translation = Localizable.getTranslation(text, templateData, cls);
      return translation != null;
    });
    return translation;
  }
  /**
   * Get localized string, returns value of `text` if no localized string found.
   *
   * If {@link Core.localization.LocaleManager#property-throwOnMissingLocale LocaleManager.throwOnMissingLocale}
   * is `true` then calls to `L()` will throw `Localization is not found for 'text' in 'ClassName'` exception when no
   * localization is found.
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @static
   * @returns {String}
   */

  static L(text, templateData = undefined, ...localeClasses) {
    var _localeClasses2;

    // In case this static method is called directly third argument is not provided
    // just fallback to searching locales for the class itself
    if (((_localeClasses2 = localeClasses) === null || _localeClasses2 === void 0 ? void 0 : _localeClasses2.length) === 0) {
      localeClasses = [this];
    }

    const translation = this.localize(text, templateData, ...localeClasses); // Throw error if not localized and text matches `L{foo}`

    if (translation == null && LocaleManagerSingleton.throwOnMissingLocale && LocaleManagerSingleton.locale && text.includes('L{')) {
      throw new Error(`Localization is not found for '${text}' in '${localeClasses.map(cls => Localizable.clsName(cls)).join(', ')}'. ${LocaleManagerSingleton.locale.localeName ? `Locale : ${LocaleManagerSingleton.locale.localeName}` : ''}`);
    }

    return translation !== null && translation !== void 0 ? translation : text;
  }
  /**
   * Convenience function that can be called directly on the class that mixes Localizable in
   *
   * ```javascript
   * button.text = grid.L('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @category Misc
   */

  L(text, templateData) {
    const {
      localeClass,
      constructor
    } = this; // If we have a different class set as translations provider
    // pass it first and use the class being translated as a fallback provider

    if (localeClass && Localizable.clsName(localeClass) !== Localizable.clsName(constructor)) {
      return Localizable.L(text, templateData, localeClass, constructor);
    } else {
      return Localizable.L(text, templateData, constructor);
    }
  }
  /**
   * Convenience function to get an optional translation. The difference compared to `L()` is that it wont throw
   * an error when the translation is missing even if configured with `throwOnMissingLocale`
   *
   * ```javascript
   * button.text = grid.optionalL('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @static
   * @category Misc
   */

  static optionalL(text, templateData = undefined, ...localeClasses) {
    var _localeClasses3;

    const shouldThrow = LocaleManagerSingleton.throwOnMissingLocale;
    LocaleManagerSingleton.throwOnMissingLocale = shouldThrow && localeRe.test(text); // In case this static method is called directly third argument is not provided
    // just fallback to searching locales for the class itself

    if (((_localeClasses3 = localeClasses) === null || _localeClasses3 === void 0 ? void 0 : _localeClasses3.length) === 0) {
      localeClasses = [this];
    }

    const result = Localizable.L(text, templateData, ...localeClasses);
    LocaleManagerSingleton.throwOnMissingLocale = shouldThrow;
    return result;
  }
  /**
   * Convenience function to get an optional translation. The difference compared to `L()` is that it wont throw
   * an error when the translation is missing even if configured with `throwOnMissingLocale`
   *
   * ```javascript
   * button.text = grid.optionalL('L{group}');
   * ```
   *
   * @param {String} text String key
   * @param {Object} [templateData] Data to supply to template if localized string is a function
   * @returns {String}
   * @category Misc
   * @internal
   */

  optionalL(text, templateData = this) {
    const shouldThrow = LocaleManagerSingleton.throwOnMissingLocale;
    LocaleManagerSingleton.throwOnMissingLocale = shouldThrow && localeRe.test(text);
    const result = this.L(text, templateData);
    LocaleManagerSingleton.throwOnMissingLocale = shouldThrow;
    return result;
  }
  /**
   * Get the global LocaleManager
   * @property {Core.localization.LocaleManager}
   * @typings { typeof LocaleManager }
   * @category Misc
   * @readonly
   */

  get localeManager() {
    return LocaleManagerSingleton;
  }

});

const {
  toString: toString$1
} = Object.prototype,
      DATE_TYPE = toString$1.call(new Date()),
      tempDate$1 = new Date(),
      MS_PER_HOUR = 1000 * 60 * 60,
      defaultValue = (value, defValue) => isNaN(value) || value == null ? defValue : value,
      enOrdinalSuffix = number => {
  const hasSpecialCase = ['11', '12', '13'].find(n => number.endsWith(n));
  let suffix = 'th';

  if (!hasSpecialCase) {
    const lastDigit = number[number.length - 1];
    suffix = {
      1: 'st',
      2: 'nd',
      3: 'rd'
    }[lastDigit] || 'th';
  }

  return number + suffix;
},
      useIntlFormat = (name, options, date) => {
  const formatter = intlFormatterCache[name] || (intlFormatterCache[name] = new Intl.DateTimeFormat(locale$6, options));
  return formatter.format(date);
},
      formatTime = (name, options, date, isShort = false) => {
  let strTime = useIntlFormat(name, options, date); // remove "0" from time when has AM/PM (from 01:00 PM to 1:00 PM): https://github.com/bryntum/support/issues/1483

  if (/am|pm/i.test(strTime)) {
    // remove first character only if is 0
    strTime = strTime.replace(/^0/, ''); // if isShort is true, remove minutes if is :00

    if (isShort) {
      strTime = strTime.replace(/:00/, '');
    }
  }

  return strTime;
},
      getDayDiff = (end, start) => Math.floor((end.getTime() - start.getTime() - (end.getTimezoneOffset() - start.getTimezoneOffset()) * validConversions.minute.millisecond) / validConversions.day.millisecond) + 1,
      normalizeDay = day => day >= 0 ? day : day + 7; // These vars are set when changing locale

let locale$6 = 'en-US',
    ordinalSuffix = enOrdinalSuffix,
    // Used to cache used formats, to not have to parse format string each time
formatCache = {},
    intlFormatterCache = {},
    parserCache = {};

const DEFAULT_YEAR = 2020,
      // 2020 is the year that has no issues in Safari, see: https://github.com/bryntum/support/issues/554
DEFAULT_MONTH = 0,
      DEFAULT_DAY = 1,
      formats = {
  // 1, 2, ... 11, 12
  M: date => date.getMonth() + 1,
  //date.toLocaleDateString(locale, { month : 'numeric' }),
  // 1st, 2nd, 3rd, 4th, ... 11th, 12th
  Mo: date => ordinalSuffix(formats.M(date).toString()),
  // 01, 02, ...
  MM: date => (date.getMonth() + 1).toString().padStart(2, '0'),
  //date.toLocaleDateString(locale, { month : '2-digit' }),
  // Jan, Feb, ...
  MMM: date => useIntlFormat('MMM', {
    month: 'short'
  }, date),
  // January, February, ...
  MMMM: date => useIntlFormat('MMMM', {
    month: 'long'
  }, date),
  // 1, 2, ...
  Q: date => Math.ceil((date.getMonth() + 1) / 3),
  // 1st, 2nd, ...
  Qo: date => ordinalSuffix(formats.Q(date).toString()),
  // 1, 2, ...
  D: date => date.getDate(),
  //date.toLocaleDateString(locale, { day : 'numeric' }),
  // 1st, 2nd, ...
  Do: date => ordinalSuffix(formats.D(date).toString()),
  // 01, 02, ...
  DD: date => date.getDate().toString().padStart(2, '0'),
  //date.toLocaleDateString(locale, { day : '2-digit' }),
  // 1, 2, ..., 365, 365
  DDD: date => Math.ceil((new Date(date.getFullYear(), date.getMonth(), date.getDate(), 12, 0, 0) - new Date(date.getFullYear(), 0, 0, 12, 0, 0)) / validConversions.day.millisecond),
  // 1st, 2nd, ...
  DDDo: date => ordinalSuffix(formats.DDD(date).toString()),
  // 001, 002, ...
  DDDD: date => formats.DDD(date).toString().padStart(3, '0'),
  // 0, 1, ..., 6
  d: date => date.getDay(),
  // 0th, 1st, ...
  do: date => ordinalSuffix(date.getDay().toString()),
  // S, M, ...
  d1: date => useIntlFormat('d1', {
    weekday: 'narrow'
  }, date).substr(0, 1),
  // Su, Mo, ...
  dd: date => formats.ddd(date).substring(0, 2),
  // Sun, Mon, ...
  ddd: date => useIntlFormat('ddd', {
    weekday: 'short'
  }, date),
  // Sunday, Monday, ...
  dddd: date => useIntlFormat('dddd', {
    weekday: 'long'
  }, date),
  e: date => date.getDay(),
  E: date => date.getDay() + 1,
  // ISO week, 1, 2, ...
  W: date => DateHelper.getWeekNumber(date)[1],
  Wo: date => ordinalSuffix(formats.W(date).toString()),
  WW: date => formats.W(date).toString().padStart(2, '0'),
  // 1979, 2018
  Y: date => date.getFullYear(),
  //date.toLocaleDateString(locale, { year : 'numeric' }),
  // 79, 18
  YY: date => date.getFullYear() % 100,
  //date.toLocaleDateString(locale, { year : '2-digit' }),
  // 1979, 2018
  YYYY: date => date.getFullYear(),
  //date.toLocaleDateString(locale, { year : 'numeric' }),
  // AM, PM
  A: date => date.getHours() < 12 ? 'AM' : 'PM',
  a: date => date.getHours() < 12 ? 'am' : 'pm',
  // 0, 1, ... 23
  H: date => date.getHours(),
  // 00, 01, ...
  HH: date => date.getHours().toString().padStart(2, '0'),
  // 1, 2, ... 12
  h: date => date.getHours() % 12 || 12,
  // 01, 02, ...
  hh: date => formats.h(date).toString().padStart(2, '0'),
  // 1, 2, ... 24
  k: date => date.getHours() || 24,
  // 01, 02, ...
  kk: date => formats.k(date).toString().padStart(2, '0'),
  // Locale specific (0 -> 24 or 1 AM -> 12 PM)
  K: date => formatTime('K', {
    hour: 'numeric'
  }, date),
  // Locale specific (00 -> 24 or 1 AM -> 12 PM)
  KK: date => formatTime('KK', {
    hour: '2-digit'
  }, date),
  // 0, 1, ... 59
  m: date => date.getMinutes(),
  //date.toLocaleTimeString(locale, { minute : 'numeric' }),
  // 00, 01, ...
  mm: date => formats.m(date).toString().padStart(2, '0'),
  // 0, 1, ... 59
  s: date => date.getSeconds(),
  //date.toLocaleTimeString(locale, { second : 'numeric' }),
  // 00, 01, ...
  ss: date => formats.s(date).toString().padStart(2, '0'),
  // 0, 1, ... 9 which are 000, 100, 200 ... 900 in milliseconds
  S: date => Math.floor(date.getMilliseconds() / 100).toString(),
  // 00, 01, ... 99 which are 000, 010, 020 ... 990 in milliseconds
  SS: date => Math.floor(date.getMilliseconds() / 10).toString().padStart(2, '0'),
  // 000, 001, ... 999 in milliseconds
  SSS: date => date.getMilliseconds().toString().padStart(3, '0'),
  z: date => useIntlFormat('z', {
    timeZoneName: 'short'
  }, date),
  zz: date => useIntlFormat('zz', {
    timeZoneName: 'long'
  }, date),
  Z: date => DH$2.getGMTOffset(date),
  LT: date => formatTime('LT', {
    hour: '2-digit',
    minute: '2-digit'
  }, date),
  // if minutes is 0, doesn't show it
  LST: date => formatTime('LST', {
    hour: 'numeric',
    minute: '2-digit'
  }, date, true),
  LTS: date => formatTime('LTS', {
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  }, date),
  L: date => useIntlFormat('L', {
    year: 'numeric',
    month: '2-digit',
    day: '2-digit'
  }, date),
  l: date => useIntlFormat('l', {
    year: 'numeric',
    month: 'numeric',
    day: 'numeric'
  }, date),
  LL: date => useIntlFormat('LL', {
    year: 'numeric',
    month: 'long',
    day: 'numeric'
  }, date),
  ll: date => useIntlFormat('ll', {
    year: 'numeric',
    month: 'short',
    day: 'numeric'
  }, date),
  LLL: date => useIntlFormat('LLL', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: 'numeric',
    minute: '2-digit'
  }, date),
  lll: date => useIntlFormat('lll', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: 'numeric',
    minute: '2-digit'
  }, date),
  LLLL: date => useIntlFormat('LLLL', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: 'numeric',
    minute: '2-digit',
    weekday: 'long'
  }, date),
  llll: date => useIntlFormat('llll', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: 'numeric',
    minute: '2-digit',
    weekday: 'short'
  }, date)
},
      // Want longest keys first, to not stop match at L of LTS etc.
formatKeys = Object.keys(formats).sort((a, b) => b.length - a.length),
      formatRegexp = `^(?:${formatKeys.join('|')})`,
      // return empty object, meaning value cannot be processed to a valuable date part
emptyFn$2 = () => ({}),
      isNumber = str => numberRegex.test(str),
      parseMilliseconds = str => isNumber(str) && {
  milliseconds: parseInt(str.padEnd(3, '0').substring(0, 3))
},
      parsers = {
  YYYY: str => ({
    year: parseInt(str)
  }),
  Y: str => ({
    year: parseInt(str)
  }),
  YY: str => {
    const year = parseInt(str);
    return {
      year: year + (year > 1968 ? 1900 : 2000)
    };
  },
  MM: str => ({
    month: parseInt(str) - 1
  }),
  Mo: str => ({
    month: parseInt(str) - 1
  }),
  DD: str => ({
    date: parseInt(str)
  }),
  M: str => ({
    month: parseInt(str) - 1
  }),
  D: str => ({
    date: parseInt(str)
  }),
  Do: str => ({
    date: parseInt(str)
  }),
  DDD: emptyFn$2,
  MMM: emptyFn$2,
  MMMM: emptyFn$2,
  DDDo: emptyFn$2,
  DDDD: emptyFn$2,
  d: emptyFn$2,
  do: emptyFn$2,
  d1: emptyFn$2,
  dd: emptyFn$2,
  ddd: emptyFn$2,
  dddd: emptyFn$2,
  Q: emptyFn$2,
  Qo: emptyFn$2,
  W: emptyFn$2,
  Wo: emptyFn$2,
  WW: emptyFn$2,
  e: emptyFn$2,
  E: emptyFn$2,
  HH: str => ({
    hours: parseInt(str)
  }),
  hh: str => ({
    hours: parseInt(str)
  }),
  mm: str => ({
    minutes: parseInt(str)
  }),
  H: str => ({
    hours: parseInt(str)
  }),
  m: str => ({
    minutes: parseInt(str)
  }),
  ss: str => ({
    seconds: parseInt(str)
  }),
  s: str => ({
    seconds: parseInt(str)
  }),
  S: parseMilliseconds,
  SS: parseMilliseconds,
  SSS: parseMilliseconds,
  A: str => ({
    amPm: str.toLowerCase()
  }),
  a: str => ({
    amPm: str.toLowerCase()
  }),
  L: 'MM/DD/YYYY',
  LT: 'HH:mm A',
  // Can either be Z (=UTC, 0) or +-HH:MM
  Z: str => {
    if (!str || !timeZoneRegEx.test(str) && str !== 'Z') {
      return null;
    }

    let timeZone = 0; // If string being parsed is more "detailed" than the format specified we can have more chars left,
    // thus check the last (for example HH:mmZ with input HH:mm:ssZ -> ssZ)

    if (str !== 'Z') {
      const matches = timeZoneRegEx.exec(str); // If timezone regexp matches, sting has time zone offset like '+02:00'

      if (matches) {
        const sign = matches[1] === '+' ? 1 : -1,
              hours = parseInt(matches[2]) || 0,
              minutes = parseInt(matches[3]) || 0;
        timeZone = sign * (hours * 60 + minutes);
      } // otherwise we just return current time zone, because there's a Z key in the input
      else {
        timeZone = -1 * new Date().getTimezoneOffset();
      }
    }

    return {
      timeZone
    };
  }
},
      parserKeys = Object.keys(parsers).sort((a, b) => b.length - a.length),
      parserRegexp = new RegExp(`(${parserKeys.join('|')})`),
      // Following regexp includes all formats that should be handled by Date class
localeStrRegExp = new RegExp('(l|LL|ll|LLL|lll|LLLL|llll)'),
      //    ISODateRegExp             = new RegExp('YYYY-MM-DD[T ]HH:mm:ss(.s+)?Z'),
// Some validConversions are negative to show that it's not an exact conversion, just an estimate.
validConversions = {
  // The units below assume:
  // 30 days in a month, 91 days for a quarter and 365 for a year
  // 52 weeks per year, 4 per month, 13 per quarter
  // 3652 days per decade (assuming two of the years will be leap with 366 days)
  decade: {
    decade: 1,
    year: 10,
    quarter: 40,
    month: 120,
    week: 520,
    day: 3652,
    hour: 24 * 3652,
    minute: 1440 * 3652,
    second: 86400 * 3652,
    millisecond: 86400000 * 3652
  },
  year: {
    decade: 0.1,
    year: 1,
    quarter: 4,
    month: 12,
    week: 52,
    day: 365,
    hour: 24 * 365,
    minute: 1440 * 365,
    second: 86400 * 365,
    millisecond: 86400000 * 365
  },
  quarter: {
    decade: 1 / 40,
    year: 1 / 4,
    quarter: 1,
    month: 3,
    week: 4,
    day: 91,
    hour: 24 * 91,
    minute: 1440 * 91,
    second: 86400 * 91,
    millisecond: 86400000 * 91
  },
  month: {
    decade: 1 / 120,
    year: 1 / 12,
    quarter: 1 / 3,
    month: 1,
    week: 4,
    day: -30,
    hour: -24 * 30,
    minute: -1440 * 30,
    second: -86400 * 30,
    millisecond: -86400000 * 30
  },
  week: {
    decade: -1 / 520,
    year: -1 / 52,
    quarter: -1 / 13,
    month: -1 / 4,
    day: 7,
    hour: 168,
    minute: 10080,
    second: 604800,
    millisecond: 604800000
  },
  day: {
    decade: -1 / 3652,
    year: -1 / 365,
    quarter: -1 / 91,
    month: -1 / 30,
    week: 1 / 7,
    hour: 24,
    minute: 1440,
    second: 86400,
    millisecond: 86400000
  },
  hour: {
    decade: -1 / (3652 * 24),
    year: -1 / (365 * 24),
    quarter: -1 / (91 * 24),
    month: -1 / (30 * 24),
    week: 1 / 168,
    day: 1 / 24,
    minute: 60,
    second: 3600,
    millisecond: 3600000
  },
  minute: {
    decade: -1 / (3652 * 1440),
    year: -1 / (365 * 1440),
    quarter: -1 / (91 * 1440),
    month: -1 / (30 * 1440),
    week: 1 / 10080,
    day: 1 / 1440,
    hour: 1 / 60,
    second: 60,
    millisecond: 60000
  },
  second: {
    decade: -1 / (3652 * 86400),
    year: -1 / (365 * 86400),
    quarter: -1 / (91 * 86400),
    month: -1 / (30 * 86400),
    week: 1 / 604800,
    day: 1 / 86400,
    hour: 1 / 3600,
    minute: 1 / 60,
    millisecond: 1000
  },
  millisecond: {
    decade: -1 / (3652 * 86400000),
    year: -1 / (365 * 86400000),
    quarter: -1 / (91 * 86400000),
    month: -1 / (30 * 86400000),
    week: 1 / 604800000,
    day: 1 / 86400000,
    hour: 1 / 3600000,
    minute: 1 / 60000,
    second: 1 / 1000
  }
},
      normalizedUnits = {
  ms: 'millisecond',
  milliseconds: 'millisecond',
  s: 'second',
  seconds: 'second',
  m: 'minute',
  mi: 'minute',
  min: 'minute',
  minutes: 'minute',
  h: 'hour',
  hours: 'hour',
  d: 'day',
  days: 'day',
  w: 'week',
  weeks: 'week',
  M: 'month',
  mo: 'month',
  mon: 'month',
  months: 'month',
  q: 'quarter',
  quarters: 'quarter',
  y: 'year',
  years: 'year',
  dec: 'decade',
  decades: 'decade'
},
      withDecimalsDurationRegex = /^\s*([-+]?\d+(?:[.,]\d*)?|[-+]?(?:[.,]\d+))\s*([^\s]+)?/i,
      noDecimalsDurationRegex = /^\s*([-+]?\d+)(?![.,])\s*([^\s]+)?/i,
      canonicalUnitNames = ['millisecond', 'second', 'minute', 'hour', 'day', 'week', 'month', 'quarter', 'year', 'decade'],
      canonicalUnitAbbreviations = [['mil'], ['s', 'sec'], ['m', 'min'], ['h', 'hr'], ['d'], ['w', 'wk'], ['mo', 'mon', 'mnt'], ['q', 'quar', 'qrt'], ['y', 'yr'], ['dec']],
      deltaUnits = ['decade', 'year', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond'],
      // Used when creating a date from an object, to fill in any blanks
dateProperties = ['milliseconds', 'seconds', 'minutes', 'hours', 'date', 'month', 'year'],
      // TODO: Should we provide special number parsing?
parseNumber = n => {
  const result = parseFloat(n);
  return isNaN(result) ? null : result;
},
      numberRegex = /^[0-9]+$/,
      timeZoneRegEx = /([+-])(\d\d):*(\d\d)*$/,
      unitMagnitudes = {
  millisecond: 0,
  second: 1,
  minute: 2,
  hour: 3,
  day: 4,
  week: 5,
  month: 6,
  quarter: 7,
  year: 8,
  decade: 9
},
      snapFns = {
  round(number, step = 1) {
    return Math.round(number / step) * step;
  },

  floor(number, step = 1) {
    return Math.floor(number / step) * step;
  },

  ceil(number, step = 1) {
    return Math.ceil(number / step) * step;
  }

},
      keyCache = {};
/**
 * @module Core/helper/DateHelper
 */

/**
 * Helps with date manipulation, comparison, parsing, formatting etc.
 *
 * ## Parsing strings
 * Use `DateHelper.parse()` to parse strings into dates. It accepts a date string and a format specifier.
 * The format specifier is string built up using the following tokens:
 *
 * | Unit        | Token | Description                       |
 * |-------------|-------|-----------------------------------|
 * | Year        | YYYY  | 2018                              |
 * |             | YY    | < 68 -> 2000, > 68 -> 1900        |
 * | Month       | MM    | 01 - 12                           |
 * | Date        | DD    | 01 - 31                           |
 * | Hour        | HH    | 00 - 23 or 1 - 12                 |
 * | Minute      | mm    | 00 - 59                           |
 * | Second      | ss    | 00 - 59                           |
 * | Millisecond | S     | 0 - 9 [000, 100, 200 .. 900 ]     |
 * |             | SS    | 00 - 99 [000, 010, 020 .. 990 ]   |
 * |             | SSS   | 000 - 999 [000, 001, 002 .. 999 ] |
 * | AM/PM       | A     | AM or PM                          |
 * |             | a     | am or pm                          |
 * | TimeZone    | Z     | Z for UTC or +-HH:mm              |
 * | Predefined  | L     | Long date, MM/DD/YYYY             |
 * |             | LT    | Long time, HH:mm A                |
 *
 * Default parse format is: `'YYYY-MM-DDTHH:mm:ss.SSSZ'` see {@link #property-defaultParseFormat-static}
 *
 * For example:
 * ```
 * DateHelper.parse('2018-11-06', 'YYYY-MM-DD');
 * DateHelper.parse('13:14', 'HH:mm');
 * DateHelper.parse('6/11/18', 'DD/MM/YY');
 * ```
 *
 * ## Formatting dates
 * Use `DateHelper.format()` to create a string from a date using a format specifier. The format specifier is similar to
 * that used when parsing strings. It can use the following tokens (input used for output below is
 * `new Date(2018,8,9,18,7,8,145)`):
 *
 * | Unit                  | Token | Description & output                  |
 * |-----------------------|-------|---------------------------------------|
 * | Year                  | YYYY  | 2018                                  |
 * |                       | YY    | 18                                    |
 * |                       | Y     | 2018                                  |
 * | Quarter               | Q     | 3                                     |
 * |                       | Qo    | 3rd                                   |
 * | Month                 | MMMM  | September                             |
 * |                       | MMM   | Sep                                   |
 * |                       | MM    | 09                                    |
 * |                       | Mo    | 9th                                   |
 * |                       | M     | 9                                     |
 * | Week (iso)            | WW    | 36 (2 digit)                          |
 * |                       | Wo    | 36th                                  |
 * |                       | W     | 36                                    |
 * | Date                  | DDDD  | Day of year, 3 digits                 |
 * |                       | DDDo  | Day of year, ordinal                  |
 * |                       | DDD   | Day of year                           |
 * |                       | DD    | 09                                    |
 * |                       | Do    | 9th                                   |
 * |                       | D     | 9                                     |
 * | Weekday               | dddd  | Sunday                                |
 * |                       | ddd   | Sun                                   |
 * |                       | dd    | Su                                    |
 * |                       | d1    | S                                     |
 * |                       | do    | 0th                                   |
 * |                       | d     | 0                                     |
 * | Hour                  | HH    | 18 (00 - 23)                          |
 * |                       | H     | 18 (0 - 23)                           |
 * |                       | hh    | 06 (00 - 12)                          |
 * |                       | h     | 6 (0 - 12)                            |
 * |                       | KK    | 19 (01 - 24)                          |
 * |                       | K     | 19 (1 - 24)                           |
 * |                       | kk    | 06 or 18, locale determines           |
 * |                       | k     | 6 or 18, locale determines            |
 * | Minute                | mm    | 07                                    |
 * |                       | m     | 7                                     |
 * | Second                | ss    | 08                                    |
 * |                       | s     | 8                                     |
 * | Millisecond           | S     | 1 (100ms)                             |
 * |                       | SS    | 14 (140ms)                            |
 * |                       | SSS   | 145 (145ms)                           |
 * | AM/PM                 | A     | AM or PM                              |
 * |                       | a     | am or pm                              |
 * | Predefined            | LT    | H: 2-digit (2d), m: 2d                |
 * | (uses browser locale) | LTS   | H: 2d, m: 2d, s : 2d                  |
 * |                       | LST   | Depends on 12 or 24 hour clock        |
 * |                       |       | 12h, H : 1d, m : 0 or 2d              |
 * |                       |       | 24h, H : 2d, m : 2d                   |
 * |                       | L     | Y: numeric (n), M : 2d, D : 2d        |
 * |                       | l     | Y: n, M : n, D : n                    |
 * |                       | LL    | Y: n, M : long (l), D : n             |
 * |                       | ll    | Y: n, M : short (s), D : n            |
 * |                       | LLL   | Y: n, M : l, D : n, H: n, m: 2d       |
 * |                       | lll   | Y: n, M : s, D : n, H: n, m: 2d       |
 * |                       | LLLL  | Y: n, M : l, D : n, H: n, m: 2d, d: l |
 * |                       | llll  | Y: n, M : s, D : n, H: n, m: 2d, d: s |
 *
 *
 * Default format is: `'YYYY-MM-DDTHH:mm:ssZ'` see {@link #property-defaultFormat-static}
 *
 * For example:
 *
 * ```javascript
 * DateHelper.format(new Date(2018,10,6), 'YYYY-MM-DD'); // 2018-11-06
 * DateHelper.format(new Date(2018,10,6), 'M/D/YY'); // 11/6/18
 * ```
 *
 * Arbitrary text can be embedded in the format string by wrapping it with {}:
 *
 * ```javascript
 * DateHelper.format(new Date(2019, 7, 16), '{It is }dddd{, yay!}') -> It is Friday, yay!
 * ```
 *
 * ## Unit names
 * Many DateHelper functions (for example add, as, set) accepts a unit among their params. The following units are
 * available:
 *
 * | Unit        | Aliases                       |
 * |-------------|-------------------------------|
 * | millisecond | millisecond, milliseconds, ms |
 * | second      | second, seconds, s            |
 * | minute      | minute, minutes, m            |
 * | hour        | hour, hours, h                |
 * | day         | day, days, d                  |
 * | week        | week, weeks, w                |
 * | month       | month, months, mon, mo, M     |
 * | quarter     | quarter, quarters, q          |
 * | year        | year, years, y                |
 * | decade      | decade, decades, dec          |
 *
 * For example:
 * ```javascript
 * DateHelper.add(date, 2, 'days');
 * DateHelper.as('hour', 7200, 'seconds');
 * ```
 */

class DateHelper extends Localizable() {
  static get $name() {
    return 'DateHelper';
  } //region Parse & format

  /**
   * Get/set the default format used by `format()` and `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ssZ'`
   * (~ISO 8601 Date and time, `'1962-06-17T09:21:34Z'`).
   * @member {String}
   */

  static set defaultFormat(format) {
    DH$2._defaultFormat = format;
  }

  static get defaultFormat() {
    return DH$2._defaultFormat || 'YYYY-MM-DDTHH:mm:ssZ';
  }
  /**
   * Get/set the default format used by `parse()`. Defaults to `'YYYY-MM-DDTHH:mm:ss.SSSZ'` or {@link #property-defaultFormat-static}
   * (~ISO 8601 Date and time, `'1962-06-17T09:21:34.123Z'`).
   * @member {String}
   */

  static set defaultParseFormat(parseFormat) {
    this._defaultParseFormat = parseFormat;
  }

  static get defaultParseFormat() {
    return this._defaultParseFormat || this._defaultFormat || 'YYYY-MM-DDTHH:mm:ss.SSSZ';
  }

  static buildParser(format) {
    // Split input format by regexp, which includes predefined patterns. Normally format would have some
    // splitters, like 'YYYY-MM-DD' or 'D/M YYYY' so output will contain matched patterns as well as splitters
    // which would serve as anchors. E.g. provided format is 'D/M!YYYY' and input is `11/6!2019` algorithm would work like:
    // 1. split format by regexp                // ['', 'D', '/', 'M', '!', 'YYYY', '']
    // 2. find splitters                        // ['/', '!']
    // 3. split input by seps, step by step     // ['11', ['6', ['2019']]]
    // Inputs like 'YYYYY' (5*Y) means 'YYYY' + 'Y', because it matches patterns from longer to shorter,
    // but if few patterns describe same unit the last one is applied, for example
    // DH.parse('20182015', 'YYYYY') equals to new Date(2015, 0, 0)
    const parts = format.split(parserRegexp),
          parser = []; // if length of the parts array is 1 - there are no regexps in the input string. thus - no parsers
    // do same if there are patterns matching locale strings (l, ll, LLLL etc.)
    // returning empty array to use new Date() as parser

    if (parts.length === 1 || localeStrRegExp.test(format)) {
      return [];
    } else {
      parts.reduce((prev, curr, index, array) => {
        // ignore first and last empty string
        if (index !== 0 || curr !== '') {
          // if current element matches parser regexp store it as a parser
          if (parserRegexp.test(curr)) {
            const localeParsers = this.localize('L{parsers}') || {},
                  fn = localeParsers[curr] || parsers[curr]; // Z should be last element in the string that matches regexp. Last array element is always either
            // an empty string (if format ends with Z) or splitter (everything that doesn't match regexp after Z)
            // If there is a pattern after Z, then Z index will be lower than length - 2

            if (curr === 'Z' && index < array.length - 2) {
              throw new Error(`Invalid format ${format} TimeZone (Z) must be last token`);
            } // If fn is a string, we found an alias (L, LLL, l etc.).
            // Need to build parsers from mapped format and merge with existing

            if (typeof fn === 'string') {
              // we are going to merge nested parsers with current, some cleanup required:
              // 1. last element is no longer last
              // 2. need to pass last parser to the next step
              const nestedParsers = DH$2.buildParser(fn),
                    lastItem = nestedParsers.pop();
              delete lastItem.last; // elevate nested parsers

              parser.push(...nestedParsers);
              prev = lastItem;
            } else {
              prev.pattern = curr;
              prev.fn = parsers[curr];
            }
          } // if it doesn't match - we've found a splitter
          else {
            prev.splitter = curr;
            parser.push(prev);
            prev = {};
          }
        } else if (Object.prototype.hasOwnProperty.call(prev, 'pattern')) {
          parser.push(prev);
        }

        return prev;
      }, {});
    }

    parser[parser.length - 1].last = true;
    return parser;
  }
  /**
   * A utility function to create a sortable string key for the passed date or ms timestamp using the `'YYYY-MM-DD'`
   * format.
   * @param {Number|Date} ms The Date instance or ms timestamp to generate a key for
   * @returns {String} Date/timestamp as a string with `'YYYY-M-D'` format
   * @internal
   */

  static makeKey(ms) {
    // Convert Date to ms timestamp
    if (ms.getTime) {
      ms = ms.getTime();
    } // Cache holds ms -> YYYY-MM-DD

    const cached = keyCache[Math.trunc(ms / MS_PER_HOUR)];

    if (cached) {
      return cached;
    }

    tempDate$1.setTime(ms);
    const month = tempDate$1.getMonth() + 1,
          date = tempDate$1.getDate(); // Not using DateHelper.format to save some cycles, hit a lot

    return keyCache[Math.trunc(ms / MS_PER_HOUR)] = `${tempDate$1.getFullYear()}-${month < 10 ? '0' + month : month}-${date < 10 ? '0' + date : date}`;
  }
  /**
   * A utility function to parse a sortable string to a date date using the `'YYYY-MM-DD'` format.
   * @param {String} key The string to return a date for.
   * @returns {Date} new Date instance
   * @internal
   */

  static parseKey(key) {
    return DH$2.parse(key, 'YYYY-MM-DD');
  }
  /**
   * Returns a date created from the supplied string using the specified format. Will try to create even if format
   * is left out, by first using the default format (see {@link #property-defaultFormat-static}, by default
   * `YYYY-MM-DDTHH:mm:ssZ`) and then using `new Date(dateString)`.
   * Supported tokens:
   *
   * | Unit        | Token | Description                       |
   * |-------------|-------|-----------------------------------|
   * | Year        | YYYY  | 2018                              |
   * |             | YY    | < 68 -> 2000, > 68 -> 1900        |
   * | Month       | MM    | 01 - 12                           |
   * | Date        | DD    | 01 - 31                           |
   * | Hour        | HH    | 00 - 23 or 1 - 12                 |
   * | Minute      | mm    | 00 - 59                           |
   * | Second      | ss    | 00 - 59                           |
   * | Millisecond | S     | 0 - 9 [000, 100, 200 .. 900 ]     |
   * |             | SS    | 00 - 99 [000, 010, 020 .. 990 ]   |
   * |             | SSS   | 000 - 999 [000, 001, 002 .. 999 ] |
   * | AM/PM       | A     | AM or PM                          |
   * |             | a     | am or pm                          |
   * | TimeZone    | Z     | Z for UTC or +-HH:mm              |
   * | Predefined  | L     | Long date, MM/DD/YYYY             |
   * |             | LT    | Long time, HH:mm A                |
   *
   * Predefined formats and functions used to parse tokens can be localized, see for example the swedish locale SvSE.js
   *
   * NOTE: If no date parameters are provided then `Jan 01 2020` is used as a default date
   *
   * @param {String} dateString Date string
   * @param {String} format Date format (or {@link #property-defaultParseFormat-static} if left out)
   * @returns {Date} new Date instance parsed from the string
   * @category Parse & format
   */

  static parse(dateString, format = DH$2.defaultParseFormat) {
    if (dateString instanceof Date) {
      return dateString;
    }

    if (typeof dateString !== 'string' || !dateString) {
      return null;
    } // // For ISO 8601 native is faster, but not very forgiving
    // if (format === defaultFormat) {
    //     const dt = new Date(dateString);
    //     if (!isNaN(dt)) {
    //         return dt;
    //     }
    // }

    const config = {
      year: null,
      month: null,
      date: null,
      hours: null,
      minutes: null,
      seconds: null,
      milliseconds: null
    }; // Milliseconds parser is the same for S, SS, SSS

    format = format.replace(/S+/gm, 'SSS');
    let parser = parserCache[format],
        result;

    if (!parser) {
      parser = parserCache[format] = DH$2.buildParser(format);
    } // Each parser knows its pattern and splitter. It looks for splitter in the
    // input string, takes first substring and tries to process it. Remaining string
    // is passed to the next parser.

    parser.reduce((dateString, parser) => {
      if (parser.last) {
        Object.assign(config, parser.fn(dateString));
      } else {
        let splitAt; // ISO 8601 says that T symbol can be replaced with a space

        if (parser.splitter === 'T' && dateString.indexOf('T') === -1) {
          splitAt = dateString.indexOf(' ');
        } else {
          // If splitter specified find its position, otherwise try to determine pattern length
          splitAt = parser.splitter !== '' ? dateString.indexOf(parser.splitter) : parser.pattern && parser.pattern.length || -1;
        }

        let part, rest; // If splitter is not found in the current string we may be dealing with
        // 1. partial input - in that case we just feed all string to current parser and move on
        // 2. time zone (ssZ - splitter is empty string) and pattern is not specified, see comment below
        // 3. parse milliseconds before Z

        if (splitAt === -1 || parser.pattern === 'SSS' && dateString.match(/^\d+Z$/)) {
          // NOTE: parentheses are required here as + and - signs hold valuable information
          // with parentheses we get array like ['00','+','01:00'], omitting them we won't get
          // regexp match in result, loosing information
          const chunks = dateString.split(/([Z\-+])/); // If splitter is not found in the string, we may be dealing with string that contains info about TZ.
          // For instance, if format contains Z as last arg which is not separated (normally it is not indeed),
          // like 'YYYY-MM-DD HH:mm:ssZ', then second to last parser will have string that it cannot just parse, like
          // '2010-01-01 10:00:00'        -> '00'
          // '2010-01-01 10:00:00Z'       -> '00Z'
          // '2010-01-01 10:00:00-01'     -> '00-01'
          // '2010-01-01 10:00:00+01:30'  -> '00+01:30'
          // this cannot be processed by date parsers, so we need to process that additionally. So we
          // split string by symbols that can be found around timezone info: Z,-,+

          if (chunks.length === 1) {
            part = dateString;
            rest = '';
          } else {
            part = chunks[0];
            rest = `${chunks[1]}${chunks[2]}`;
          }
        } else {
          part = dateString.substring(0, splitAt) || dateString;
          rest = dateString.substring(splitAt + parser.splitter.length);
        }

        if (parser.fn) {
          // Run parser and add result to config on successful parse otherwise continue parsing
          const res = parser.fn(part);

          if (res) {
            Object.assign(config, res);
          } else {
            rest = part + rest;
          }
        }

        return rest;
      }
    }, dateString); // If year is specified date has to be greater than 0

    if (config.year && !config.date) {
      config.date = 1;
    }

    const date = DH$2.create(config);

    if (date) {
      result = date;
    } else {
      // Last resort, try if native passing can do it
      result = new Date(dateString);
    }

    return result;
  }
  /**
   * Creates a date from a date definition object. The object can have the following properties:
   * - year
   * - month
   * - date (day in month)
   * - hours
   * - minutes
   * - seconds
   * - milliseconds
   * - amPm : 'am' or 'pm', implies 12 hour clock
   * - timeZone : offset from UTC in minutes
   * @param {Object} definition
   * @returns {Date} new Date instance
   * @category Parse & format
   */

  static create(definition) {
    // Shallow clone to not alter input
    const def = _objectSpread2({}, definition);

    let invalid = isNaN(def.year),
        useUTC = false; // Not much validation yet, only considered invalid if all properties are null

    if (!invalid) {
      let allNull = true;
      dateProperties.forEach(property => {
        if (!(property in def) || isNaN(def[property])) {
          def[property] = 0;
        }

        allNull = allNull && def[property] === null;
      });
      invalid = allNull;
    }

    if (def.amPm === 'pm') {
      def.hours = def.hours % 12 + 12;
    }

    if ('timeZone' in def) {
      useUTC = true;
      def.minutes -= def.timeZone;
    }

    if (invalid) {
      return null;
    }

    const args = [defaultValue(def.year, DEFAULT_YEAR), defaultValue(def.month, DEFAULT_MONTH), defaultValue(def.date, DEFAULT_DAY), def.hours, def.minutes, def.seconds, def.milliseconds];
    return useUTC ? new Date(Date.UTC(...args)) : new Date(...args);
  }

  static toUTC(date) {
    return new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate(), date.getUTCHours(), date.getUTCMinutes(), date.getUTCSeconds(), date.getUTCMilliseconds()));
  }
  /**
   * Converts a date to string with the specified format. Formats heavily inspired by https://momentjs.com.
   * Available formats (input used for output below is `new Date(2018,8,9,18,7,8,145)`):
   *
   * | Unit                  | Token | Description & output                  |
   * |-----------------------|-------|---------------------------------------|
   * | Year                  | YYYY  | 2018                                  |
   * |                       | YY    | 18                                    |
   * |                       | Y     | 2018                                  |
   * | Quarter               | Q     | 3                                     |
   * |                       | Qo    | 3rd                                   |
   * | Month                 | MMMM  | September                             |
   * |                       | MMM   | Sep                                   |
   * |                       | MM    | 09                                    |
   * |                       | Mo    | 9th                                   |
   * |                       | M     | 9                                     |
   * | Week (iso)            | WW    | 36 (2 digit)                          |
   * |                       | Wo    | 36th                                  |
   * |                       | W     | 36                                    |
   * | Date                  | DDDD  | Day of year, 3 digits                 |
   * |                       | DDDo  | Day of year, ordinal                  |
   * |                       | DDD   | Day of year                           |
   * |                       | DD    | 09                                    |
   * |                       | Do    | 9th                                   |
   * |                       | D     | 9                                     |
   * | Weekday               | dddd  | Sunday                                |
   * |                       | ddd   | Sun                                   |
   * |                       | dd    | Su                                    |
   * |                       | d1    | S                                     |
   * |                       | do    | 0th                                   |
   * |                       | d     | 0                                     |
   * | Hour                  | HH    | 18 (00 - 23)                          |
   * |                       | H     | 18 (0 - 23)                           |
   * |                       | hh    | 06 (00 - 12)                          |
   * |                       | h     | 6 (0 - 12)                            |
   * |                       | KK    | 19 (01 - 24)                          |
   * |                       | K     | 19 (1 - 24)                           |
   * |                       | kk    | 06 or 18, locale determines           |
   * |                       | k     | 6 or 18, locale determines            |
   * | Minute                | mm    | 07                                    |
   * |                       | m     | 7                                     |
   * | Second                | ss    | 08                                    |
   * |                       | s     | 8                                     |
   * | Millisecond           | S     | 1 (100ms)                             |
   * |                       | SS    | 14 (140ms)                            |
   * |                       | SSS   | 145 (145ms)                           |
   * | AM/PM                 | A     | AM or PM                              |
   * |                       | a     | am or pm                              |
   * | Predefined            | LT    | H: 2-digit (2d), m: 2d                |
   * | (uses browser locale) | LTS   | H: 2d, m: 2d, s : 2d                  |
   * |                       | LST   | Depends on 12 or 24 hour clock        |
   * |                       |       | 12h, H : 1d, m : 0 or 2d              |
   * |                       |       | 24h, H : 2d, m : 2d                   |
   * |                       | L     | Y: numeric (n), M : 2d, D : 2d        |
   * |                       | l     | Y: n, M : n, D : n                    |
   * |                       | LL    | Y: n, M : long (l), D : n             |
   * |                       | ll    | Y: n, M : short (s), D : n            |
   * |                       | LLL   | Y: n, M : l, D : n, H: n, m: 2d       |
   * |                       | lll   | Y: n, M : s, D : n, H: n, m: 2d       |
   * |                       | LLLL  | Y: n, M : l, D : n, H: n, m: 2d, d: l |
   * |                       | llll  | Y: n, M : s, D : n, H: n, m: 2d, d: s |
   *
   * Some examples:
   *
   * ```
   * DateHelper.format(new Date(2019, 7, 16), 'dddd') -> Friday
   * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'HH:mm') --> 14:27
   * DateHelper.format(new Date(2019, 7, 16, 14, 27), 'L HH') --> 2019-07-16 14
   * ```
   *
   * Arbitrary text can be embedded in the format string by wrapping it with {}:
   *
   * ```
   * DateHelper.format(new Date(2019, 7, 16), '{It is }dddd{, yay!}') -> It is Friday, yay!
   * ```
   *
   * @param {Date} date Date
   * @param {String} format Desired format (uses `defaultFormat` if left out)
   * @returns {String} Formatted string
   * @category Parse & format
   */

  static format(date, format = DH$2.defaultFormat) {
    // Bail out if no date or invalid date
    if (!date || isNaN(date)) {
      return null;
    }

    let formatter = formatCache[format],
        output = '';

    if (!formatter) {
      formatter = formatCache[format] = []; // Build formatter array with the steps needed to format the date

      for (let i = 0; i < format.length; i++) {
        // Matches a predefined format?
        const formatMatch = format.substr(i).match(formatRegexp),
              predefined = formatMatch === null || formatMatch === void 0 ? void 0 : formatMatch[0];

        if (predefined) {
          const localeFormats = this.localize('L{formats}') || {},
                fn = localeFormats[predefined] || formats[predefined];
          formatter.push(fn);
          i += predefined.length - 1;
        } // Start of text block? Append it
        else if (format[i] === '{') {
          // Find closing brace
          const index = format.indexOf('}', i + 1); // No closing brace, grab rest of string

          if (index === -1) {
            formatter.push(format.substr(i + 1));
            i = format.length;
          } // Closing brace found
          else {
            formatter.push(format.substring(i + 1, index)); // Carry on after closing brace

            i = index;
          }
        } // Otherwise append to output (for example - / : etc)
        else {
          formatter.push(format[i]);
        }
      }
    }

    formatter.forEach(step => {
      if (typeof step === 'string') {
        output += step;
      } else {
        output += step(date);
      }
    });
    return output;
  }
  /**
   * Converts the specified amount of desired unit into milliseconds. Can be called by only specifying a unit as the
   * first argument, it then uses amount = 1. For example: asMilliseconds('hour') == asMilliseconds(1, 'hour')
   * @param {Number|String} amount Amount, what of is decided by specifying unit (also takes a unit which implies an amount of 1)
   * @param {String} unit Time unit (s, hour, months etc.)
   * @returns {Number}
   * @category Parse & format
   */

  static asMilliseconds(amount, unit = null) {
    if (typeof amount === 'string') {
      unit = amount;
      amount = 1;
    }

    return DH$2.as('millisecond', amount, unit);
  }
  /**
   * Converts the passed Date to an accurate number of months passed since the epoch start.
   * @param {Date} time The Date to find the month value of.
   * @returns {Number} The number of months since the system time epoch start. May be a fractional value.
   */

  static asMonths(time) {
    const monthLength = DH$2.as('ms', DH$2.daysInMonth(time), 'day'),
          fraction = (time.valueOf() - DH$2.startOf(time, 'month').valueOf()) / monthLength;
    return time.getYear() * 12 + time.getMonth() + fraction;
  }

  static monthsToDate(months) {
    const intMonths = Math.floor(months),
          fraction = months - intMonths,
          result = new Date(0, intMonths),
          msInMonth = DH$2.as('ms', DH$2.daysInMonth(result), 'days');
    result.setTime(result.getTime() + fraction * msInMonth);
    return result;
  }
  /**
   * Converts a millisecond time delta to a human readable form. For example `1000 * 60 * 60 * 50`
   * milliseconds would be rendered as "2 days, 2 hours"
   * @param {Number} delta The millisecond delta value.
   * @param {Object} [options] Formatting options.
   * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg "2d, 2h" for the above example.
   * @param {String} [options.precision] The minimum precision unit.
   * @param {String} [options.separator] The separator to use
   * @param {Boolean} [options.asString] Pass `false` to return the result as an array, eg ['2d', '2h'] for the above example.
   * @returns {String} Formatted string
   * @category Parse & format
   */

  static formatDelta(delta, options) {
    let abbrev, unitName;

    if (typeof options === 'boolean') {
      abbrev = options;
    } else if (options) {
      abbrev = options.abbrev;
    }

    const deltaObj = this.getDelta(delta, options),
          result = [],
          sep = (options === null || options === void 0 ? void 0 : options.separator) || (abbrev ? '' : ' ');

    for (unitName in deltaObj) {
      result.push(`${deltaObj[unitName]}${sep}${unitName}`);
    }

    return (options === null || options === void 0 ? void 0 : options.asString) === false ? result : result.join(', ');
  }
  /**
   * Converts a millisecond time delta to an object structure. For example `1000 * 60 * 60 * 50`
   * milliseconds the result would be as:
   * {
   *     day: 2,
   *     hour: 2
   * }
   *
   * @param {Number} delta The millisecond delta value.
   * @param {Object} [options] Formatting options.
   * @param {Boolean} [options.abbrev] Pass `true` to use abbreviated unit names, eg "{ d: 2, h: 2 }" for the above example.
   * @param {String} [options.precision] The minimum precision unit.
   * @returns {Object} The object with the values for each unit.
   */

  static getDelta(delta, options) {
    let abbrev, d, done, precision, unitName;

    if (typeof options === 'boolean') {
      abbrev = options;
    } else if (options) {
      abbrev = options.abbrev;
      precision = DH$2.normalizeUnit(options.precision);
    }

    const result = {},
          getUnit = abbrev ? DH$2.getShortNameOfUnit : DH$2.getLocalizedNameOfUnit; // Loop downwards through the magnitude of units from year -> ms

    for (unitName of deltaUnits) {
      d = DH$2.as(unitName, delta);
      done = precision === unitName;
      d = Math[done ? 'round' : 'floor'](d); // If there's a non-zero integer quantity of this unit, add it to result
      // and subtract from the delta, then go round to next unit down.

      if (d || done && !result.length) {
        result[getUnit.call(DH$2, unitName, d !== 1)] = d; // TODO 0 is plural in EN but...?

        delta -= DH$2.as('ms', d, unitName);
      }

      if (done || !delta) {
        break;
      }
    }

    return result;
  }
  /**
   * Converts the specified amount of one unit (fromUnit) into an amount of another unit (toUnit)
   * @param {String} toUnit The name of units to convert to, eg: `'ms'`
   * @param {Number|String} amount The time to convert. Either the magnitude number form
   * or a duration string such as '2d'.
   * @param {String} [fromUnit] If the amount was passed as a number, the units to use to convert from.
   * @returns {Number}
   * @category Parse & format
   */

  static as(toUnit, amount, fromUnit = 'ms') {
    // Allow DH.as('ms', '2d')
    if (typeof amount === 'string') {
      amount = DH$2.parseDuration(amount);
    } // Allow DH.as('ms', myDurationObject)

    if (typeof amount === 'object') {
      fromUnit = amount.unit;
      amount = amount.magnitude;
    }

    if (toUnit === fromUnit) {
      return amount;
    }

    toUnit = DH$2.normalizeUnit(toUnit);
    fromUnit = DH$2.normalizeUnit(fromUnit);

    if (toUnit === fromUnit) {
      return amount;
    } // validConversions[][] can be negative to signal that conversion is not exact, ignore sign here
    else if (unitMagnitudes[fromUnit] > unitMagnitudes[toUnit]) {
      return amount * Math.abs(validConversions[fromUnit][toUnit]);
    } else {
      return amount / Math.abs(validConversions[toUnit][fromUnit]);
    }
  }

  static formatContainsHourInfo(format) {
    const stripEscapeRe = /(\\.)/g,
          hourInfoRe = /([HhKkmSsAa]|LT|L{3,}|l{3,})/;
    return hourInfoRe.test(format.replace(stripEscapeRe, ''));
  }
  /**
   * Returns true for 24 hour format.
   * @param {String} format Date format
   * @returns {Boolean}
   * @category Parse & format
   */

  static is24HourFormat(format) {
    return DH$2.format(DH$2.getTime(13, 0, 0), format).includes('13');
  } //endregion
  //region Manipulate

  /**
   * Add days, hours etc. to a date. Always clones the date, original will be left unaffected.
   * @param {Date|String} date Original date
   * @param {Number} amount Amount of days, hours etc
   * @param {String} unit Unit for amount
   * @privateparam {Boolean} [clone] Affect the original
   * @returns {Date} New calculated date
   * @category Manipulate
   */

  static add(date, amount, unit = 'ms', clone = true) {
    let d;

    if (typeof date === 'string') {
      d = DH$2.parse(date);
    } else if (clone) {
      d = new Date(date.getTime());
    } else {
      d = date;
    }

    if (!unit || amount === 0) {
      return d;
    }

    unit = DH$2.normalizeUnit(unit);

    switch (unit) {
      case 'millisecond':
        d.setTime(d.getTime() + amount);
        break;

      case 'second':
        d.setTime(d.getTime() + amount * 1000);
        break;

      case 'minute':
        d.setTime(d.getTime() + amount * 60000);
        break;

      case 'hour':
        d.setTime(d.getTime() + amount * 3600000);
        break;

      case 'day':
        // Integer value added, do calendar calculation to correctly handle DST etc.
        if (amount % 1 === 0) {
          d.setDate(d.getDate() + amount);
        } // No browsers support fractional values for dates any longer, do time based calculation
        else {
          d.setTime(d.getTime() + amount * 86400000);
        } // When crossing DST in Brazil, we expect hours to end up the same

        if (d.getHours() === 23 && date.getHours() === 0) {
          d.setHours(d.getHours() + 1);
        }

        break;

      case 'week':
        d.setDate(d.getDate() + amount * 7);
        break;

      case 'month':
        {
          let day = d.getDate();

          if (day > 28) {
            day = Math.min(day, DH$2.getLastDateOfMonth(DH$2.add(DH$2.getFirstDateOfMonth(d), amount, 'month')).getDate());
          }

          d.setDate(day);
          d.setMonth(d.getMonth() + amount);
          break;
        }

      case 'quarter':
        DH$2.add(d, amount * 3, 'month', false);
        break;

      case 'year':
        d.setFullYear(d.getFullYear() + amount);
        break;

      case 'decade':
        d.setFullYear(d.getFullYear() + amount * 10);
        break;
    }

    return d;
  }
  /**
   * Calculates the difference between two dates, in the specified unit.
   * @param {Date} start First date
   * @param {Date} end Second date
   * @param {String} unit Unit to calculate difference in
   * @param {Boolean} fractional Specify false to round result
   * @returns {Number} Difference in the specified unit
   * @category Manipulate
   */

  static diff(start, end, unit = 'ms', fractional = true) {
    unit = DH$2.normalizeUnit(unit);
    if (!start || !end) return 0;
    let amount;

    switch (unit) {
      case 'year':
        amount = DH$2.diff(start, end, 'month') / 12;
        break;

      case 'quarter':
        amount = DH$2.diff(start, end, 'month') / 3;
        break;

      case 'month':
        amount = (end.getFullYear() - start.getFullYear()) * 12 + (end.getMonth() - start.getMonth());
        break;

      case 'week':
        amount = DH$2.diff(start, end, 'day') / 7;
        break;

      case 'day':
        {
          const dstDiff = start.getTimezoneOffset() - end.getTimezoneOffset();
          amount = (end - start + dstDiff * 60 * 1000) / 86400000;
          break;
        }

      case 'hour':
        amount = (end - start) / 3600000;
        break;

      case 'minute':
        amount = (end - start) / 60000;
        break;

      case 'second':
        amount = (end - start) / 1000;
        break;

      case 'millisecond':
        amount = end - start;
        break;
    }

    return fractional ? amount : Math.round(amount);
  }
  /**
   * Sets the date to the start of the specified unit, by default returning a clone of the date instead of changing it
   * in place.
   * @param {Date} date Original date
   * @param {String} [unit] Start of this unit, 'day', 'month' etc.
   * @param {Boolean} [clone] Manipulate a copy of the date
   * @param {Number} [weekStartDay] The first day of week, 0-6 (Sunday-Saturday). Defaults to the
   * {@link #property-weekStartDay-static}.
   * @returns {Date} Manipulated date
   * @category Manipulate
   */

  static startOf(date, unit = 'day', clone = true, weekStartDay = DH$2.weekStartDay) {
    if (!date) {
      return null;
    }

    unit = DH$2.normalizeUnit(unit);

    if (clone) {
      date = DH$2.clone(date);
    }

    switch (unit) {
      case 'year':
        date.setMonth(0, 1);
        date.setHours(0, 0, 0, 0);
        return date;

      case 'quarter':
        date.setMonth((DH$2.get(date, 'quarter') - 1) * 3, 1);
        date.setHours(0, 0, 0, 0);
        return date;

      case 'month':
        date.setDate(1);
        date.setHours(0, 0, 0, 0);
        return date;

      case 'week':
        {
          const delta = date.getDay() - weekStartDay;
          date.setDate(date.getDate() - delta);
          date.setHours(0, 0, 0, 0);
          return date;
        }

      case 'day':
        date.setHours(0, 0, 0, 0);
        return date;
      // Cant use setMinutes(0, 0, 0) etc for DST transitions

      case 'hour':
        date.getMinutes() > 0 && date.setMinutes(0);
      // eslint-disable-next-line no-fallthrough

      case 'minute':
        date.getSeconds() > 0 && date.setSeconds(0);
      // eslint-disable-next-line no-fallthrough

      case 'second':
        date.getMilliseconds() > 0 && date.setMilliseconds(0);
      // eslint-disable-next-line no-fallthrough

      case 'millisecond':
        return date;
    }
  }
  /**
   * Returns the end point of the passed date, that is 00:00:00 of the day after the passed date.
   * @param {Date} date The date to return the end point of.
   */

  static endOf(date) {
    return new Date(date.getFullYear(), date.getMonth(), date.getDate() + 1);
  }
  /**
   * Creates a clone of the specified date
   * @param {Date} date Original date
   * @returns {Date} Cloned date
   * @category Manipulate
   */

  static clone(date) {
    return new Date(date.getTime());
  }
  /**
   * Removes time from a date (same as calling #startOf(date, 'day'))
   * @param {Date} date Date to remove time from
   * @param {Boolean} [clone=true] Manipulate a copy of the date
   * @returns {Date} Manipulated date
   * @category Manipulate
   */

  static clearTime(date, clone = true) {
    if (!date) {
      return null;
    }

    if (clone) {
      date = new Date(date.getTime());
    }

    date.setHours(0, 0, 0, 0);
    return date;
  }

  static midnight(date, inclusive) {
    let ret = DH$2.clearTime(date);

    if (inclusive && ret < date) {
      ret = DH$2.add(ret, 1, 'd');
    }

    return ret;
  }
  /**
   * Returns the elapsed milliseconds from the start of the specified date.
   * @param {Date} date Date to remove date from.
   * @param {String} [as] The time unit to return.
   * @returns {Number} The elapsed milliseconds from the start of the specified date
   * @category Manipulate
   */

  static getTimeOfDay(date, as = 'ms') {
    const t = date.getHours() * validConversions.hour.millisecond + date.getMinutes() * validConversions.minute.millisecond + date.getSeconds() * validConversions.second.millisecond + date.getMilliseconds();
    return as === 'ms' ? t : DH$2.as(as, t, 'ms');
  }
  /**
   * Sets a part of a date (in place)
   * @param {Date} date Date to manipulate
   * @param {String|Object} unit Part of date to set, for example 'minute'. Or an object like { second: 1, minute: 1 }
   * @param {Number} amount Value to set
   * @returns {Date} Passed date instance modified according to the arguments
   * @category Manipulate
   */

  static set(date, unit, amount) {
    if (!unit) {
      return date;
    }

    if (typeof unit === 'string') {
      switch (DH$2.normalizeUnit(unit)) {
        case 'millisecond':
          // Setting value to 0 when it is 0 at DST crossing messes it up
          if (amount !== 0 || date.getMilliseconds() > 0) {
            date.setMilliseconds(amount);
          }

          break;

        case 'second':
          // Setting value to 0 when it is 0 at DST crossing messes it up
          if (amount !== 0 || date.getSeconds() > 0) {
            date.setSeconds(amount);
          }

          break;

        case 'minute':
          // Setting value to 0 when it is 0 at DST crossing messes it up
          if (amount !== 0 || date.getMinutes() > 0) {
            date.setMinutes(amount);
          }

          break;

        case 'hour':
          date.setHours(amount);
          break;

        case 'day':
        case 'date':
          date.setDate(amount);
          break;

        case 'week':
          throw new Error('week not implemented');

        case 'month':
          date.setMonth(amount);
          break;

        case 'quarter':
          // Setting quarter = first day of first month of that quarter
          date.setDate(1);
          date.setMonth((amount - 1) * 3);
          break;

        case 'year':
          date.setFullYear(amount);
          break;
      }
    } else {
      Object.entries(unit) // Make sure smallest unit goes first, to not change month before changing day
      .sort((a, b) => unitMagnitudes[a[0]] - unitMagnitudes[b[0]]).forEach(([unit, amount]) => {
        DH$2.set(date, unit, amount);
      });
    }

    return date;
  }

  static setDateToMidday(date, clone = true) {
    return DH$2.set(DH$2.clearTime(date, clone), 'hour', 12);
  }
  /**
   * Constrains the date within a min and a max date
   * @param {Date} date The date to constrain
   * @param {Date} [min] Min date
   * @param {Date} [max] Max date
   * @return {Date} The constrained date
   * @category Manipulate
   */

  static constrain(date, min, max) {
    if (min != null) {
      date = DH$2.max(date, min);
    }

    return max == null ? date : DH$2.min(date, max);
  }
  /**
   * Returns time with default year, month, and day (Jan 1, 2020)
   * @param {Number|Date} hours Hours value or the full date to extract the time of.
   * @param {Number} minutes Minutes value
   * @param {Number} seconds Seconds value
   * @param {Number} ms Milliseconds value
   * @returns {Date} A new default date with the time extracted from the given date or from the time values provided individually.
   * @category Manipulate
   */

  static getTime(hours, minutes = 0, seconds = 0, ms = 0) {
    if (hours instanceof Date) {
      ms = hours.getMilliseconds();
      seconds = hours.getSeconds();
      minutes = hours.getMinutes();
      hours = hours.getHours();
    }

    return new Date(DEFAULT_YEAR, DEFAULT_MONTH, DEFAULT_DAY, hours, minutes, seconds, ms);
  }
  /**
   * Copies hours, minutes, seconds, milliseconds from one date to another
   *
   * @param {Date} targetDate The target date
   * @param {Date} sourceDate The source date
   * @return {Date} The adjusted target date
   * @category Manipulate
   * @static
   */

  static copyTimeValues(targetDate, sourceDate) {
    targetDate.setHours(sourceDate.getHours());
    targetDate.setMinutes(sourceDate.getMinutes());
    targetDate.setSeconds(sourceDate.getSeconds());
    targetDate.setMilliseconds(sourceDate.getMilliseconds());
    return targetDate;
  } //endregion
  //region Comparison

  static isDST(date) {
    const year = date.getFullYear(),
          jan = new Date(year, 0, 1),
          jul = new Date(year, 6, 1);
    return date.getTimezoneOffset() < Math.max(jan.getTimezoneOffset(), jul.getTimezoneOffset());
  }
  /**
   * Determines if a date precedes another.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Boolean} true if first precedes second, otherwise false
   * @category Comparison
   */

  static isBefore(first, second) {
    return first < second;
  }
  /**
   * Determines if a date succeeds another.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @returns {Boolean} true if first succeeds second, otherwise false
   * @category Comparison
   */

  static isAfter(first, second) {
    return first > second;
  }
  /**
   * Checks if two dates are equal.
   * @param {Date} first First date
   * @param {Date} second Second date
   * @param {String} unit If not given, the comparison will be done up to a millisecond
   * @returns {Boolean} true if the dates are equal
   * @category Comparison
   */

  static isEqual(first, second, unit = null) {
    if (unit === null) {
      // https://jsbench.me/3jk2bom2r3/1
      // https://jsbench.me/ltkb3vk0ji/1 (more flavors) - getTime is >2x faster vs valueOf/Number/op+
      return first && second && first.getTime() === second.getTime();
    }

    return DH$2.startOf(first, unit) - DH$2.startOf(second, unit) === 0;
  }
  /**
   * Compares two dates using the specified precision
   * @param {Date} first
   * @param {Date} second
   * @param {String} unit
   * @returns {Number} 0 = equal, -1 first before second, 1 first after second
   * @category Comparison
   */

  static compare(first, second, unit = null) {
    // Unit specified, cut the rest out
    if (unit) {
      first = DH$2.startOf(first, unit);
      second = DH$2.startOf(second, unit);
    } // Comparison on ms level

    if (first < second) return -1;
    if (first > second) return 1;
    return 0;
  }

  static isSameDate(first, second) {
    return DH$2.compare(first, second, 'd') === 0;
  }
  /**
   * Checks if date is the start of specified unit
   * @param {Date} date
   * @param {String} unit
   * @returns {Boolean}
   * @category Comparison
   */

  static isStartOf(date, unit) {
    return DH$2.isEqual(date, DH$2.startOf(date, unit));
  }
  /**
   * Checks if this date is >= start and < end.
   * @param {Date} date The source date
   * @param {Date} start Start date
   * @param {Date} end End date
   * @return {Boolean} true if this date falls on or between the given start and end dates.
   * @category Comparison
   */

  static betweenLesser(date, start, end) {
    //return start <= date && date < end;
    return start.getTime() <= date.getTime() && date.getTime() < end.getTime();
  }
  /**
   * Checks if this date is >= start and <= end.
   * @param {Date} date The source date
   * @param {Date} start Start date
   * @param {Date} end End date
   * @return {Boolean} true if this date falls on or between the given start and end dates.
   * @category Comparison
   */

  static betweenLesserEqual(date, start, end) {
    return start.getTime() <= date.getTime() && date.getTime() <= end.getTime();
  }
  /**
   * Returns true if dates intersect
   * @param {Date} date1Start
   * @param {Date} date1End
   * @param {Date} date2Start
   * @param {Date} date2End
   * @return {Boolean} Returns true if dates intersect
   * @category Comparison
   */

  static intersectSpans(date1Start, date1End, date2Start, date2End) {
    return DH$2.betweenLesser(date1Start, date2Start, date2End) || DH$2.betweenLesser(date2Start, date1Start, date1End);
  }
  /**
   * Compare two units. Returns 1 if first param is a greater unit than second param, -1 if the opposite is true or 0 if they're equal
   * @param {String} unit1 The 1st unit
   * @param {String} unit2 The 2nd unit
   * @returns {Number} Returns 1 if first param is a greater unit than second param, -1 if the opposite is true or 0 if they're equal
   * @category Comparison
   */

  static compareUnits(unit1, unit2) {
    return Math.sign(unitMagnitudes[DH$2.normalizeUnit(unit1)] - unitMagnitudes[DH$2.normalizeUnit(unit2)]);
  }
  /**
   * Returns true if the first time span completely 'covers' the second time span.
   * @example
   * DateHelper.timeSpanContains(new Date(2010, 1, 2), new Date(2010, 1, 5), new Date(2010, 1, 3), new Date(2010, 1, 4)) ==> true
   * DateHelper.timeSpanContains(new Date(2010, 1, 2), new Date(2010, 1, 5), new Date(2010, 1, 3), new Date(2010, 1, 6)) ==> false
   * @param {Date} spanStart The start date for initial time span
   * @param {Date} spanEnd The end date for initial time span
   * @param {Date} otherSpanStart The start date for the 2nd time span
   * @param {Date} otherSpanEnd The end date for the 2nd time span
   * @return {Boolean}
   * @category Comparison
   */

  static timeSpanContains(spanStart, spanEnd, otherSpanStart, otherSpanEnd) {
    return otherSpanStart - spanStart >= 0 && spanEnd - otherSpanEnd >= 0;
  } //endregion
  //region Query

  /**
   * Get the first day of week, 0-6 (Sunday-Saturday).
   * This is determined by the current locale's `DateHelper.weekStartDay` parameter.
   * @property {Number}
   * @readonly
   */

  static get weekStartDay() {
    // Cache is reset in applyLocale
    if (DH$2._weekStartDay == null) {
      // Defaults to 0, should not need to happen in real world scenarios when a locale is always loaded
      DH$2._weekStartDay = this.localize('L{weekStartDay}') || 0;
    }

    return DH$2._weekStartDay;
  }
  /**
   * Get non-working days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
   * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
   *
   * For example:
   * ```
   * {
   *     0 : true, // Sunday
   *     6 : true  // Saturday
   * }
   * ```
   * @property {Object}
   * @readonly
   */

  static get nonWorkingDays() {
    return _objectSpread2({}, this.localize('L{nonWorkingDays}'));
  }
  /**
   * Get non-working days as an array of day indices, 0-6 (Sunday-Saturday).
   * This is determined by the current locale's `DateHelper.nonWorkingDays` parameter.
   *
   * For example:
   * ```
   * [ 0, 6 ] // Sunday & Saturday
   * ```
   * @property {Number[]}
   * @readonly
   * @internal
   */

  static get nonWorkingDaysAsArray() {
    // transform string keys to integers
    return Object.keys(this.nonWorkingDays).map(Number);
  }
  /**
   * Get weekend days as an object where keys are day indices, 0-6 (Sunday-Saturday), and the value is `true`.
   * Weekends are days which are declared as weekend days by the selected country and defined by the current locale's `DateHelper.weekends` parameter.
   * To get non-working days see {@link #property-nonWorkingDays-static}.
   *
   * For example:
   * ```
   * {
   *     0 : true, // Sunday
   *     6 : true  // Saturday
   * }
   * ```
   * @property {Object}
   * @readonly
   * @internal
   */

  static get weekends() {
    return _objectSpread2({}, this.localize('L{weekends}'));
  }
  /**
   * Get the specified part of a date
   * @param {Date} date
   * @param {String} unit Part of date, hour, minute etc.
   * @returns {Number} The requested part of the specified date
   * @category Query
   */

  static get(date, unit) {
    switch (DH$2.normalizeUnit(unit)) {
      case 'millisecond':
        return date.getMilliseconds();

      case 'second':
        return date.getSeconds();

      case 'minute':
        return date.getMinutes();

      case 'hour':
        return date.getHours();

      case 'date':
      case 'day':
        // Scheduler has a lot of calculations expecting this to work
        return date.getDate();

      case 'week':
        return formats.W(date);

      case 'month':
        return date.getMonth();

      case 'quarter':
        return Math.floor(date.getMonth() / 3) + 1;

      case 'year':
        return date.getFullYear();
    }

    return null;
  }
  /**
   * Get number of days in the current year for the supplied date.
   * @param {Date} date Date to check
   * @return {Number} Days in year
   * @category Query
   * @internal
   */

  static daysInYear(date) {
    const fullYear = date.getFullYear(),
          duration = new Date(fullYear + 1, 0, 1) - new Date(fullYear, 0, 1);
    return this.as('day', duration);
  }
  /**
   * Get number of days in the current month for the supplied date
   * @param {Date} date Date which month should be checked
   * @returns {Number} Days in month
   * @category Query
   */

  static daysInMonth(date) {
    return 32 - new Date(date.getFullYear(), date.getMonth(), 32).getDate();
  }
  /**
   * Get number of hours in the current day for the supplied date.
   * @param {Date} date Date to check
   * @return {Number} Hours in day
   * @category Query
   * @internal
   */

  static hoursInDay(date) {
    const fullYear = date.getFullYear(),
          month = date.getMonth(),
          day = date.getDate(),
          duration = new Date(fullYear, month, day + 1) - new Date(fullYear, month, day);
    return this.as('hour', duration);
  }
  /**
   * Converts unit related to the date to actual amount of milliseconds in it. Takes into account leap years and
   * different duration of months.
   * @param {Date} date
   * @param {String} unit
   * @return {Number} Returns amount of milliseconds
   * @internal
   */

  static getNormalizedUnitDuration(date, unit) {
    let result;

    switch (unit) {
      case 'month':
        result = DH$2.asMilliseconds(DH$2.daysInMonth(date), 'day');
        break;

      case 'year':
        result = DH$2.asMilliseconds(DH$2.daysInYear(date), 'day');
        break;

      case 'day':
        result = DH$2.asMilliseconds(DH$2.hoursInDay(date), 'hour');
        break;

      default:
        result = DH$2.asMilliseconds(unit);
    }

    return result;
  }
  /**
   * Get the first date of the month for the supplied date
   * @param {Date} date
   * @returns {Date} New Date instance
   * @category Query
   */

  static getFirstDateOfMonth(date) {
    return new Date(date.getFullYear(), date.getMonth(), 1);
  }
  /**
   * Get the last date of the month for the supplied date
   * @param {Date} date
   * @returns {Date} New Date instance
   * @category Query
   */

  static getLastDateOfMonth(date) {
    return new Date(date.getFullYear(), date.getMonth() + 1, 0);
  }
  /**
   * Get the earliest of two dates
   * @param {Date} first
   * @param {Date} second
   * @returns {Date} Earliest date
   * @category Query
   */

  static min(first, second) {
    return first.getTime() < second.getTime() ? first : second;
  }
  /**
   * Get the latest of two dates
   * @param {Date} first
   * @param {Date} second
   * @returns {Date} Latest date
   * @category Query
   */

  static max(first, second) {
    return first.getTime() > second.getTime() ? first : second;
  }
  /**
   * Get an incremented date. Incrementation based on specified unit and optional amount
   * @param {Date} date
   * @param {String} unit
   * @param {Number} [increment]
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   * @category Query
   */

  static getNext(date, unit, increment = 1, weekStartDay = DH$2.weekStartDay) {
    if (unit === 'week') {
      const dt = DH$2.clone(date),
            day = dt.getDay();
      DH$2.startOf(dt, 'day', false);
      DH$2.add(dt, weekStartDay - day + 7 * (increment - (weekStartDay <= day ? 0 : 1)), 'day', false); // For south american timezones, midnight does not exist on DST transitions, adjust...

      if (dt.getDay() !== weekStartDay) {
        DH$2.add(dt, 1, 'hour');
      }

      return dt;
    }

    return DH$2.startOf(DH$2.add(date, increment, unit), unit, false);
  }
  /**
   * Checks if date object is valid.
   *
   * For example:
   *
   * ```javascript
   * date = new Date('foo')
   * date instanceof Date // true
   * date.toString() // "Invalid Date"
   * isNaN(date) // true
   * DateHelper.isValidDate(date) // false
   *
   * date = new Date()
   * date instanceof Date // true
   * date.toString() // "Mon Jan 13 2020 18:27:38 GMT+0300 (GMT+03:00)"
   * isNaN(date) // false
   * DateHelper.isValidDate(date) // true
   * ```
   *
   * @param {Date} date
   * @returns {Boolean}
   */

  static isValidDate(date) {
    return DH$2.isDate(date) && !isNaN(date);
  }
  /**
   * Checks if value is a date object. Allows to recognize date object even from another context,
   * like the top frame when used in an iframe.
   *
   * @param {*} value
   * @returns {Boolean}
   */

  static isDate(value) {
    // see https://jsbench.me/s7kb49w83j/1 (cannot use instanceof cross-frame):
    return value && toString$1.call(value) === DATE_TYPE;
  }
  /**
   * Get the start of the next day
   * @param {Date} date
   * @param {Boolean} [clone]
   * @param {Boolean} [noNeedToClearTime=false]
   * @returns {Date} Passed Date or new Date instance, depending on the `clone` flag
   * @category Query
   */

  static getStartOfNextDay(date, clone, noNeedToClearTime) {
    let nextDay = DH$2.add(noNeedToClearTime ? date : DH$2.clearTime(date, clone), 1, 'day'); // DST case

    if (nextDay.getDate() === date.getDate()) {
      const offsetNextDay = DH$2.add(DH$2.clearTime(date, clone), 2, 'day').getTimezoneOffset(),
            offsetDate = date.getTimezoneOffset();
      nextDay = DH$2.add(nextDay, offsetDate - offsetNextDay, 'minute');
    }

    return nextDay;
  }
  /**
   * Get the end of previous day
   * @param {Date} date
   * @param {Boolean} noNeedToClearTime
   * @returns {Date} New Date instance
   * @category Query
   */

  static getEndOfPreviousDay(date, noNeedToClearTime) {
    const dateOnly = noNeedToClearTime ? date : DH$2.clearTime(date, true); // dates are different

    if (dateOnly - date) {
      return dateOnly;
    } else {
      return DH$2.add(dateOnly, -1, 'day');
    }
  }
  /**
   * Returns a string describing the specified week. For example, "39, September 2020" or "40, Sep - Oct 2020".
   * @param {Date} startDate
   * @param {Date} [endDate]
   * @returns {String}
   * @internal
   */

  static getWeekDescription(startDate, endDate = startDate) {
    const monthDesc = startDate.getMonth() === endDate.getMonth() ? DateHelper.format(startDate, 'MMMM') : `${DateHelper.format(startDate, 'MMM')} - ${DateHelper.format(endDate, 'MMM')}`,
          week = DateHelper.getWeekNumber(startDate);
    return `${week[1]}, ${monthDesc} ${week[0]}`;
  }
  /**
   * Get week number for the date.
   * @param {Date} date The date
   * @param {Number} [weekStartDay] The first day of week, 0-6 (Sunday-Saturday). Defaults to the {@link #property-weekStartDay-static}.
   * @returns {Number[]} year and week number
   * @category Query
   */

  static getWeekNumber(date, weekStartDay = DateHelper.weekStartDay) {
    const jan01 = new Date(date.getFullYear(), 0, 1),
          dec31 = new Date(date.getFullYear(), 11, 31),
          firstDay = normalizeDay(jan01.getDay() - weekStartDay),
          lastDay = normalizeDay(dec31.getDay() - weekStartDay),
          dayNumber = getDayDiff(date, jan01);
    let weekNumber; // Check if the year starts before the middle of a week

    if (firstDay < 4) {
      weekNumber = Math.floor((dayNumber + firstDay - 1) / 7) + 1;
    } else {
      weekNumber = Math.floor((dayNumber + firstDay - 1) / 7);
    }

    if (weekNumber) {
      let year = date.getFullYear(); // Might be week 1 of next year if the year ends before day 3 (0 based)

      if (weekNumber === 53 && lastDay < 3) {
        year++;
        weekNumber = 1;
      }

      return [year, weekNumber];
    } // We're in week zero which is the last week of the previous year, so ask what
    // week encapsulates 31 Dec in the previous year.

    const lastWeekOfLastYear = DateHelper.getWeekNumber(new Date(date.getFullYear() - 1, 11, 31))[1];
    return [date.getFullYear() - 1, lastWeekOfLastYear];
  } //endregion
  //region Unit helpers

  /**
   * Turns (10, 'day') into '10 days' etc.
   * @param {Number} count Amount of unit
   * @param {String} unit Unit, will be normalizes (days, d -> day etc.)
   * @returns {String}
   * @category Unit helpers
   */

  static formatCount(count, unit) {
    unit = DH$2.normalizeUnit(unit);
    if (count !== 1) unit += 's';
    return count + ' ' + unit;
  }
  /**
   * Get the ratio between two units ( year, month -> 1/12 )
   * @param {String} baseUnit
   * @param {String} unit
   * @param {Boolean} acceptEstimate If true, process negative values of validConversions. Defaults to false.
   * @returns {Number} Ratio
   * @category Unit helpers
   */

  static getUnitToBaseUnitRatio(baseUnit, unit, acceptEstimate = false) {
    baseUnit = DH$2.normalizeUnit(baseUnit);
    unit = DH$2.normalizeUnit(unit);
    if (baseUnit === unit) return 1; // Some validConversions have negative sign to signal that it is not an exact conversion.
    // Ignore those here unless acceptEstimate is provided.

    if (validConversions[baseUnit] && validConversions[baseUnit][unit] && (acceptEstimate || validConversions[baseUnit][unit] > 0)) {
      return 1 / DH$2.as(unit, 1, baseUnit);
    }

    if (validConversions[unit] && validConversions[unit][baseUnit] && (acceptEstimate || validConversions[unit][baseUnit] > 0)) {
      return DH$2.as(baseUnit, 1, unit);
    }

    return -1;
  }
  /**
   * Returns a localized abbreviated form of the name of the duration unit.
   * For example in the `EN` locale, for `"qrt"` it will return `"q"`.
   * @param {String} unit Duration unit
   * @return {String}
   * @category Unit helpers
   */

  static getShortNameOfUnit(unit) {
    // Convert abbreviations to the canonical name.
    // See locale file and the applyLocale method below.
    unit = DH$2.parseTimeUnit(unit); // unitLookup is keyed by eg 'DAY', 'day', 'MILLISECOND', 'millisecond' etc

    return DH$2.unitLookup[unit].abbrev;
  }
  /**
   * Returns a localized full name of the duration unit.
   *
   * For for example in the `EN` locale, for `"d"` it will return either
   * `"day"` or `"days"`, depending from the `plural` argument
   *
   * Preserves casing of first letter.
   *
   * @static
   * @param {String} unit Time unit
   * @param {Boolean} [plural] Whether to return a plural name or singular
   * @return {String}
   * @category Unit helpers
   */

  static getLocalizedNameOfUnit(unit, plural = false) {
    const capitalize = unit.charAt(0) === unit.charAt(0).toUpperCase(); // Normalize to not have to have translations for each variation used in code

    unit = DH$2.normalizeUnit(unit); // Convert abbreviations to the canonical name.
    // See locale file and the applyLocale method below.

    unit = DH$2.parseTimeUnit(unit); // Translate
    // unitLookup is keyed by eg 'DAY', 'day', 'MILLISECOND', 'millisecond' etc

    unit = DH$2.unitLookup[unit][plural ? 'plural' : 'single']; // Preserve casing of first letter

    if (capitalize) {
      unit = StringHelper.capitalize(unit);
    }

    return unit;
  }
  /**
   * Normalizes a unit for easier usage in conditionals. For example year, years, y -> year
   * @param {String} unit
   * @returns {String}
   * @category Unit helpers
   */

  static normalizeUnit(unit) {
    if (!unit) {
      return null;
    }

    const unitLower = unit.toLowerCase();

    if (unitLower === 'date') {
      return unitLower;
    }

    return canonicalUnitNames.includes(unitLower) // Already valid
    ? unitLower // Trying specified case first, since we have both "M" for month and "m" for minute
    : normalizedUnits[unit] || normalizedUnits[unitLower];
  }

  static getUnitByName(name) {
    // Allow either a canonical name to be passed, or, if that fails, parse it as a localized name or abbreviation.
    return DH$2.normalizeUnit(name) || DH$2.normalizeUnit(DH$2.parseTimeUnit(name));
  }
  /**
   * Returns a duration of the timeframe in the given unit.
   * @param {Date} start The start date of the timeframe
   * @param {Date} end The end date of the timeframe
   * @param {String} unit Duration unit
   * @return {Number} The duration in the units
   * @category Unit helpers
   * @ignore
   */

  static getDurationInUnit(start, end, unit, doNotRound) {
    return DH$2.diff(start, end, unit, doNotRound);
  }
  /**
   * Checks if two date units align
   * @private
   * @param {String} majorUnit
   * @param {String} minorUnit
   * @returns {Boolean}
   * @category Unit helpers
   */

  static doesUnitsAlign(majorUnit, minorUnit) {
    // TODO: probably needs some fleshing out to be generally useful, otherwise move to TimeAxisViewModel?
    // Maybe also use getUnitToBaseUnitRatio() for assertion?
    return !(majorUnit !== minorUnit && minorUnit === 'week');
  }

  static getSmallerUnit(unit) {
    return canonicalUnitNames[unitMagnitudes[DH$2.normalizeUnit(unit)] - 1] || null;
  }

  static getLargerUnit(unit) {
    return canonicalUnitNames[unitMagnitudes[DH$2.normalizeUnit(unit)] + 1] || null;
  }
  /**
   *
   * Rounds the passed Date value to the nearest `increment` value.
   *
   * Optionally may round relative to a certain base time point.
   *
   * For example `DH.round(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would round to 9:45 because that's the nearest integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when rounding to weeks. The configured {@link #property-weekStartDay-static}
   * dictates what the base of a week is.
   *
   * @param {Date} time The time to round
   * @param {String|Number} increment A millisecond value by which to round the time.
   * May be specified in string form eg: `'15 minutes'`
   * @param {Date} [base] The "start" from which to apply the rounding.
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */

  static round(time, increment, base, weekStartDay) {
    return DH$2.snap('round', time, increment, base, weekStartDay);
  }
  /**
   *
   * Floor the passed Date value to the nearest `increment` value.
   *
   * Optionally may floor relative to a certain base time point.
   *
   * For example `DH.floor(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would floor to 9:15 because that's the closest lower integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when flooring to weeks. The configured {@link #property-weekStartDay-static}
   * dictates what the base of a week is.
   *
   * @param {Date} time The time to floor
   * @param {String|Number} increment A millisecond value by which to floor the time.
   * May be specified in string form eg: `'15 minutes'`
   * @param {Date} [base] The "start" from which to apply the flooring.
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */

  static floor(time, increment, base, weekStartDay) {
    return DH$2.snap('floor', time, increment, base, weekStartDay);
  }
  /**
   *
   * Ceils the passed Date value to the nearest `increment` value.
   *
   * Optionally may ceil relative to a certain base time point.
   *
   * For example `DH.ceil(new Date('2020-01-01T09:35'), '30 min', new Date('2020-01-01T09:15'))`
   * would ceil to 9:45 because that's the closest higher integer number of 30 minute increments
   * from the base.
   *
   * Note that `base` is ignored when ceiling to weeks. Use weekStartDay argument which default to the configured
   * {@link #property-weekStartDay-static} dictates what the base of a week is
   *
   * @param {Date} time The time to ceil
   * @param {String|Number|Object} increment A numeric millisecond value by which to ceil the time.
   * or a duration in string form eg `'30 min'` or object form : `{unit: "minute", magnitude: 30}`
   * or `{unit: "minute", increment: 30}`
   * May be specified in string form eg: `'15 minutes'`
   * @param {Date} [base] The "start" from which to apply the ceiling.
   * @param {Number} [weekStartDay] Will default to what is set in locale
   * @returns {Date} New Date instance
   */

  static ceil(time, increment, base, weekStartDay) {
    return DH$2.snap('ceil', time, increment, base, weekStartDay);
  }
  /**
   * Implementation for round, floor and ceil.
   * @internal
   */

  static snap(operation, time, increment, base, weekStartDay = DH$2.weekStartDay) {
    const snapFn = snapFns[operation];

    if (typeof increment === 'string') {
      increment = DH$2.parseDuration(increment);
    }

    if (Objects.isObject(increment)) {
      // Allow {unit: "minute", increment: 30} or {unit: "minute", magnitude: 30}
      // parseDuration produces "magnitude". The Scheduler's TimeAxis uses "increment"
      // in its resolution object, so we allow that too.
      const magnitude = increment.magnitude || increment.increment; // increment is in weeks, months, quarters or years, then it can't be handled
      // using millisecond arithmetic.

      switch (increment.unit) {
        case 'week':
          {
            const weekDay = time.getDay(); // weekStartDay gives our base
            // Our base is the start of the week

            base = DH$2.add(DH$2.clearTime(time), weekDay >= weekStartDay ? weekStartDay - weekDay : -(weekDay - weekStartDay + 7), 'day');
            return DH$2[operation](time, `${magnitude * 7} days`, base);
          }

        case 'month':
          {
            // Express the time as a number of months from epoch start.
            // May be a fraction, eg the 15th will be 0.5 through a month.
            time = DH$2.asMonths(time);
            let resultMonths; // Snap the month count in the way requested

            if (base) {
              base = DH$2.asMonths(base);
              resultMonths = time + snapFn(time - base, magnitude);
            } else {
              resultMonths = snapFn(time, magnitude);
            } // Convert resulting month value back to a date

            return DH$2.monthsToDate(resultMonths);
          }

        case 'quarter':
          return DH$2[operation](time, `${magnitude * 3} months`, base);

        case 'year':
          return DH$2[operation](time, `${magnitude * 12} months`, base);

        case 'decade':
          // We assume that decades begin with a year divisible by 10
          return DH$2[operation](time, `${magnitude * 10} years`, base);
      } // Convert to a millisecond value

      increment = DH$2.as('ms', magnitude, increment.unit);
    } // It's a simple round to milliseconds

    if (base) {
      const tzChange = DH$2.as('ms', base.getTimezoneOffset() - time.getTimezoneOffset(), 'ms');
      return new Date(base.valueOf() + snapFn(DH$2.diff(base, time, 'ms') + tzChange, increment));
    } else {
      const offset = time.getTimezoneOffset() * 60 * 1000; // Assuming current TZ is GMT+3
      // new Date(2000, 0, 1) / 86400000      -> 10956.875
      // new Date(2000, 0, 1, 3) / 86400000   -> 10957
      // Before calculation we need to align time value of the current timezone to GMT+0
      // And after calculate we need to adjust time back

      return new Date(snapFn(time.valueOf() - offset, increment) + offset);
    }
  } //endregion
  //region Date picker format

  /**
   * Parses a typed duration value according to locale rules.
   *
   * The value is taken to be a string consisting of the numeric magnitude and the units:
   * - The numeric magnitude can be either an integer or a float value. Both "," and "." are valid decimal separators.
   * - The units may be a recognised unit abbreviation of this locale or the full local unit name.
   *
   * For example:
   * "2d", "2 d", "2 day", "2 days" will be turned into `{ magnitude : 2, unit : "day" }`
   * "2.5d", "2,5 d", "2.5 day", "2,5 days" will be turned into `{ magnitude : 2.5, unit : "day" }`
   *
   * **NOTE:** Doesn't work with complex values like "2 days, 2 hours"
   *
   * @param {String} value The value to parse.
   * @param {Boolean} [allowDecimals=true] Decimals are allowed in the magnitude.
   * @param {String} [defaultUnit] Default unit to use if only magnitude passed.
   * @returns {Object} If successfully parsed, the result contains two properties,
   * `magnitude` being a number, and `unit` being the canonical unit name, *NOT*
   * a localized name. If parsing was unsuccessful, `null` is returned.
   * @category Parse & format
   */

  static parseDuration(value, allowDecimals = true, defaultUnit) {
    var _match$;

    const durationRegEx = allowDecimals ? withDecimalsDurationRegex : noDecimalsDurationRegex,
          match = durationRegEx.exec(value);

    if (value == null || !match) {
      return null;
    }

    const magnitude = parseNumber((_match$ = match[1]) === null || _match$ === void 0 ? void 0 : _match$.replace(',', '.')),
          unit = DH$2.parseTimeUnit(match[2]) || defaultUnit;

    if (!unit) {
      return null;
    }

    return {
      magnitude,
      unit
    };
  }
  /**
   * Parses a typed unit name, for example `"ms"` or `"hr"` or `"yr"` into the
   * canonical form of the unit name which may be passed to {@link #function-add-static}
   * or {@link #function-diff-static}
   * @param {*} unitName
   * @category Parse & format
   */

  static parseTimeUnit(unitName) {
    // NOTE: In case you get a crash here when running tests, it is caused by missing locale. Build locales
    // using `scripts/build.js locales` to resolve.
    const unitMatch = unitName == null ? null : DH$2.durationRegEx.exec(unitName.toLowerCase());

    if (!unitMatch) {
      return null;
    } // See which group in the unitAbbrRegEx matched match[2]

    for (let unitOrdinal = 0; unitOrdinal < canonicalUnitNames.length; unitOrdinal++) {
      if (unitMatch[unitOrdinal + 1]) {
        return canonicalUnitNames[unitOrdinal];
      }
    }
  } //endregion
  //region Internal

  static getGMTOffset(date = new Date()) {
    if (!date) {
      return;
    }

    const offsetInMinutes = date.getTimezoneOffset(); // return 'Z' for UTC

    if (!offsetInMinutes) return 'Z';
    return (offsetInMinutes > 0 ? '-' : '+') + Math.abs(Math.trunc(offsetInMinutes / 60)).toString().padStart(2, '0') + ':' + Math.abs(offsetInMinutes % 60).toString().padStart(2, '0');
  }

  static fillDayNames() {
    const tempDate = new Date('2000-01-01T12:00:00'),
          dayNames = DH$2._dayNames || [],
          dayShortNames = DH$2._dayShortNames || [];
    dayNames.length = 0;
    dayShortNames.length = 0;

    for (let day = 2; day < 9; day++) {
      tempDate.setDate(day);
      dayNames.push(DH$2.format(tempDate, 'dddd'));
      dayShortNames.push(DH$2.format(tempDate, 'ddd'));
    }

    DH$2._dayNames = dayNames;
    DH$2._dayShortNames = dayShortNames;
  }

  static getDayNames() {
    return DH$2._dayNames;
  }

  static getDayName(day) {
    return DH$2._dayNames[day];
  }

  static getDayShortNames() {
    return DH$2._dayShortNames;
  }

  static getDayShortName(day) {
    return DH$2._dayShortNames[day];
  }

  static fillMonthNames() {
    const tempDate = new Date('2000-01-15T12:00:00'),
          monthNames = DH$2._monthNames || [],
          monthShortNames = DH$2._monthShortNames || [];
    monthNames.length = 0;
    monthShortNames.length = 0;

    for (let month = 0; month < 12; month++) {
      tempDate.setMonth(month);
      monthNames.push(DH$2.format(tempDate, 'MMMM'));
      monthShortNames.push(DH$2.format(tempDate, 'MMM'));
    }

    DH$2._monthNames = monthNames;
    DH$2._monthShortNames = monthShortNames;
  }

  static getMonthShortNames() {
    return DH$2._monthShortNames;
  }

  static getMonthShortName(month) {
    return DH$2._monthShortNames[month];
  }

  static getMonthNames() {
    return DH$2._monthNames;
  }

  static getMonthName(month) {
    return DH$2._monthNames[month];
  }

  static set locale(name) {
    locale$6 = name;
    intlFormatterCache = {};
    formatCache = {};
  }

  static get locale() {
    return locale$6;
  }

  static setupDurationRegEx(unitNames = [], unitAbbreviations = []) {
    const me = this,
          unitLookup = {};
    let unitAbbrRegEx = '';

    for (let i = 0; i < unitAbbreviations.length; i++) {
      const // for example ['s', 'sec']
      abbreviations = unitAbbreviations[i],
            // for example { single : 'second', plural : 'seconds', abbrev : 's' }
      unitNamesCfg = unitNames[i];
      unitNamesCfg.canonicalUnitName = canonicalUnitNames[i]; // Create a unitLookup object keyed by unit full names
      // both lower and upper case to be able to look up plurals or abbreviations
      // also always include english names, since those are used in sources

      unitLookup[unitNamesCfg.single] = unitLookup[unitNamesCfg.single.toUpperCase()] = unitLookup[unitNamesCfg.canonicalUnitName] = unitLookup[unitNamesCfg.canonicalUnitName.toUpperCase()] = unitNamesCfg;
      unitAbbrRegEx += `${i ? '|' : ''}(`;

      for (let j = 0; j < abbreviations.length; j++) {
        unitAbbrRegEx += `${abbreviations[j]}|`;
      }

      locale$6 = me.localize('L{locale}') || 'en-US';

      if (locale$6 !== 'en-US') {
        // Add canonical values to be able to parse durations specified in configs
        const canonicalAbbreviations = canonicalUnitAbbreviations[i];

        for (let j = 0; j < canonicalAbbreviations.length; j++) {
          unitAbbrRegEx += `${canonicalAbbreviations[j]}|`;
        }
      }

      unitAbbrRegEx += `${unitNamesCfg.single}|${unitNamesCfg.plural}|${unitNamesCfg.canonicalUnitName}|${unitNamesCfg.canonicalUnitName}s)`;
    }

    me.unitLookup = unitLookup;
    me.durationRegEx = new RegExp(`^(?:${unitAbbrRegEx})$`);
  }

  static applyLocale() {
    const me = this,
          unitAbbreviations = me.localize('L{unitAbbreviations}') || [],
          unitNames = me.unitNames = me.localize('L{unitNames}'); // This happens when applying an incomplete locale, as done in Localizable.t.js.
    // Invalid usecase, but return to prevent a crash in that test.

    if (unitNames === 'unitNames') {
      return;
    }

    locale$6 = me.localize('L{locale}') || 'en-US';

    if (locale$6 === 'en-US') {
      // TODO: Include in En locale instead?
      ordinalSuffix = enOrdinalSuffix;
    } else {
      ordinalSuffix = me.localize('L{ordinalSuffix}') || ordinalSuffix;
    }

    formatCache = {};
    parserCache = {};
    intlFormatterCache = {};
    DH$2._weekStartDay = null;
    DH$2.setupDurationRegEx(unitNames, unitAbbreviations); // rebuild day/month names cache

    DH$2.fillDayNames();
    DH$2.fillMonthNames();
  } //endregion

}
const DH$2 = DateHelper;
DH$2.useIntlFormat = useIntlFormat; // to use on tests
// Update when changing locale

LocaleManagerSingleton.on({
  locale: 'applyLocale',
  prio: 1000,
  thisObj: DH$2
}); // Apply default locale

if (LocaleManagerSingleton.locale) {
  DH$2.applyLocale();
}

DateHelper._$name = 'DateHelper';

/**
 * @module Core/helper/ObjectHelper
 */

const {
  hasOwnProperty: hasOwnProperty$4
} = Object.prototype; // Detect if browser has bad implementation of toFixed()

const toFixedFix = 1.005.toFixed(2) === '1.01' ? null : function (number, fractionDigits) {
  const split = number.toString().split('.'),
        newNumber = +(!split[1] ? split[0] : split.join('.') + '1');
  return number.toFixed.call(newNumber, fractionDigits);
};
/**
 * Helper for Object manipulation.
 */

class ObjectHelper extends Objects {
  // These methods are inherited from Objects (an internal class) but need to be documented here for public use.
  // This is primarily because static methods, while inherited by JavaScript classes, are not displayed in derived
  // classes in the docs.

  /**
   * Copies all enumerable properties from the supplied source objects to `dest`. Unlike `Object.assign`, this copy
   * also includes inherited properties.
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method assign
   * @static
   */

  /**
   * Copies all enumerable properties from the supplied source objects to `dest`, only including properties that does
   * not already exist on `dest`. Unlike `Object.assign`, this copy also includes inherited properties.
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method assignIf
   * @static
   */

  /**
   * Creates a deep copy of the `value`. Simple objects ({@link #function-isObject-static}, arrays and `Date` objects
   * are cloned. The enumerable properties of simple objects and the elements of arrays are cloned recursively.
   * @param {*} value The value to clone.
   * @param {Function} [handler] An optional function to call for values of types other than simple object, array or
   * `Date`. This function should return the clone of the `value` passed to it. It is only called for truthy values
   * whose `typeof` equals `'object'`.
   * @param {*} handler.value The value to clone.
   * @returns {*} The cloned value.
   * @method clone
   * @static
   */

  /**
   * Converts a list of names (either a space separated string or an array), into an object with those properties
   * assigned truthy values. The converse of {@link #function-getTruthyKeys-static}.
   * @param {String|String[]} source The list of names to convert to object form.
   * @method createTruthyKeys
   * @static
   */

  /**
   * Gathers the names of properties which have truthy values into an array.
   *
   * This is useful when gathering CSS class names for complex element production.
   * Instead of appending to an array or string which may already contain the
   * name, and instead of contending with space separation and concatenation
   * and conditional execution, just set the properties of an object:
   *
   *     cls = {
   *         [this.selectedCls] : this.isSelected(thing),
   *         [this.dirtyCls] : this.isDirty(thing)
   *     };
   *
   * @param {Object} source Source of keys to gather into an array.
   * @returns {String[]} The keys which had a truthy value.
   * @method getTruthyKeys
   * @static
   */

  /**
   * Gathers the values of properties which are truthy into an array.
   * @param {Object} source Source of values to gather into an array.
   * @returns {String[]} The truthy values from the passed object.
   * @method getTruthyValues
   * @static
   */

  /**
   * Tests whether a passed object has any enumerable properties.
   * @param {Object} object
   * @returns {Boolean} `true` if the passed object has no enumerable properties.
   * @method isEmpty
   * @static
   */

  /**
   * Returns `true` if the `value` is a simple `Object`.
   * @param {Object} value
   * @returns {Boolean} `true` if the `value` is a simple `Object`.
   * @method isObject
   * @static
   */

  /**
   * Copies all enumerable properties from the supplied source objects to `dest`, recursing when the properties of
   * both the source and `dest` are objects.
   * ```
   *  const o = {
   *      a : 1,
   *      b : {
   *          c : 2
   *      }
   *  };
   *  const o2 = {
   *      b : {
   *          d : 3
   *      }
   *  }
   *
   *  console.log(merge(o, o2));
   *
   *  > { a : 1, b : { c : 2, d : 3 } }
   * ```
   * @param {Object} dest The destination object.
   * @param {...Object} sources The source objects.
   * @returns {Object} The `dest` object.
   * @method merge
   * @static
   */

  /**
   * Returns the specific type of the given `value`. Unlike the `typeof` operator, this function returns the text
   * from the `Object.prototype.toString` result allowing `Date`, `Array`, `RegExp`, and others to be differentiated.
   * ```
   *  console.log(typeOf(null));
   *  > null
   *
   *  console.log(typeOf({}));
   *  > object
   *
   *  console.log(typeOf([]));
   *  > array
   *
   *  console.log(typeOf(new Date()));
   *  > date
   *
   *  console.log(typeOf(NaN));
   *  > nan
   *
   *  console.log(typeOf(/a/));
   *  > regexp
   * ```
   * @param {*} value
   * @returns {String}
   * @method typeOf
   * @static
   */

  /**
   * Returns value for a given path in the object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {*} Value associated with passed key
   * @method getPath
   * @static
   */

  /**
   * Sets value for a given path in the object
   * @param {Object} object Target object
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @param {*} value Value for a given path
   * @returns {Object} Returns passed object
   * @method setPath
   * @static
   */

  /**
   * Creates a new object where key is a property in array item (`ref` by default) or index in the array and value is array item.
   *
   * From:
   * ```
   * [
   *     {
   *          text : 'foo',
   *          ref : 'fooItem'
   *     },
   *     {
   *          text : 'bar'
   *     }
   * ]
   * ```
   *
   * To:
   * ```
   * {
   *     fooItem : {
   *         text : 'foo',
   *         ref  : 'fooItem'
   *     },
   *     1 : {
   *         text : 'bar'
   *     }
   * }
   * ```
   *
   * @param {Object[]} arrayOfItems Array to transform.
   * @param {String} [prop] Property to read the key from. `ref` by default.
   * @returns {Object} namedItems
   */
  static transformArrayToNamedObject(arrayOfItems, prop = 'ref') {
    const namedItems = {};
    arrayOfItems.forEach((item, index) => {
      const // 0 is valid value, but empty string in not valid
      key = item[prop] != null && item[prop].toString().length ? item[prop] : index;
      namedItems[key] = item;
    });
    return namedItems;
  }
  /**
   * Creates a new array from object values and saves key in a property (`ref` by default) of each item.
   *
   * From:
   * ```
   * {
   *     fooItem : {
   *         text : 'foo'
   *     },
   *     1 : {
   *         text : 'bar'
   *     },
   *     barItem : false // will be ignored
   * }
   * ```
   *
   * To:
   * ```
   * [
   *     {
   *          text : 'foo',
   *          ref : 'fooItem'
   *     },
   *     {
   *          text : 'bar',
   *          ref : 1
   *     }
   * ]
   * ```
   *
   * @param {Object} namedItems Object to transform.
   * @param {String} [prop] Property to save the key to. `ref` by default.
   * @returns {Object[]} arrayOfItems
   */

  static transformNamedObjectToArray(namedItems, prop = 'ref') {
    return Object.keys(namedItems).filter(key => namedItems[key]).map(key => {
      const item = namedItems[key];
      item[prop] = key;
      return item;
    });
  }
  /**
   * Checks if two values are equal. Basically === but special handling of dates.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {*} true if values are equal, otherwise false
   */

  static isEqual(a, b, useIsDeeply = false) {
    // Eliminate null vs undefined mismatch
    if (a === null && b !== null || a === undefined && b !== undefined || b === null && a !== null || b === undefined && a !== undefined) {
      return false;
    } // Covers undefined === undefined and null === null, since mismatches are eliminated above

    if (a == null && b == null) {
      return true;
    } // The same instance should equal itself.

    if (a === b) {
      return true;
    }

    const typeA = typeof a,
          typeB = typeof b;

    if (typeA === typeB) {
      switch (typeA) {
        case 'number':
        case 'string':
        case 'boolean':
          return a === b;
      }

      switch (true) {
        case a instanceof Date && b instanceof Date:
          // faster than calling DateHelper.isEqual
          // https://jsbench.me/3jk2bom2r3/1
          return a.getTime() === b.getTime();

        case Array.isArray(a) && Array.isArray(b):
          return a.length === b.length ? a.every((v, idx) => OH.isEqual(v, b[idx], useIsDeeply)) : false;

        case typeA === 'object' && a.constructor.prototype === b.constructor.prototype:
          return useIsDeeply ? OH.isDeeplyEqual(a, b, useIsDeeply) : JSON.stringify(a) === JSON.stringify(b);
      }
    }

    return String(a) === String(b);
  }
  /**
   * Checks if two objects are deeply equal
   * @param {Object} a
   * @param {Object} b
   * @param {Object} [options] Additional comparison options
   * @param {Object} [options.ignore] Map of property names to ignore when comparing
   * @param {Function} [options.shouldEvaluate] Function used to evaluate if a property should be compared or not.
   * Return false to prevent comparison
   * @param {Function} [options.evaluate] Function used to evaluate equality. Return `true`/`false` as evaluation
   * result or anything else to let `isEqual` handle the comparison
   * @returns {Boolean}
   */

  static isDeeplyEqual(a, b, options = {}) {
    // Same object, equal :)
    if (a === b) {
      return true;
    } // Nothing to compare, not equal

    if (!a || !b) {
      return false;
    } // Property names excluding ignored

    const aKeys = OH.keys(a, options.ignore),
          bKeys = OH.keys(b, options.ignore); // Property count differs, not equal

    if (aKeys.length !== bKeys.length) {
      return false;
    }

    for (let i = 0; i < aKeys.length; i++) {
      const aKey = aKeys[i],
            bKey = bKeys[i]; // Property name differs, not equal

      if (aKey !== bKey) {
        return false;
      }

      const aVal = a[aKey],
            bVal = b[bKey]; // Allow caller to determine if property values should be evaluated or not
      // TODO: Not currently used

      if (options.shouldEvaluate) {
        if (options.shouldEvaluate(aKey, {
          value: aVal,
          object: a
        }, {
          value: bVal,
          object: b
        }) === false) {
          continue;
        }
      } // Allow caller to determine equality of properties

      if (options.evaluate) {
        const result = options.evaluate(aKey, {
          value: aVal,
          object: a
        }, {
          value: bVal,
          object: b
        }); // Not equal

        if (result === false) {
          return false;
        } // Equal, skip isEqual call below

        if (result === true) {
          continue;
        }
      } // Values differ, not equal (also digs deeper)

      if (!OH.isEqual(aVal, bVal, options)) {
        return false;
      }
    } // Found to be equal

    return true;
  }
  /**
   * Checks if value B is partially equal to value A.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if values are partially equal, false otherwise
   */

  static isPartial(a, b) {
    a = String(a).toLowerCase();
    b = String(b).toLowerCase();
    return a.indexOf(b) !== -1;
  }
  /**
   * Checks if value a is smaller than value b.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if a < b
   */

  static isLessThan(a, b) {
    if (a instanceof Date && b instanceof Date) {
      return DateHelper.isBefore(a, b);
    }

    return a < b;
  }
  /**
   * Checks if value a is bigger than value b.
   * @param {*} a First value
   * @param {*} b Second value
   * @returns {Boolean} true if a > b
   */

  static isMoreThan(a, b) {
    if (a instanceof Date && b instanceof Date) {
      return DateHelper.isAfter(a, b);
    }

    return a > b;
  }
  /**
   * Used by the Base class to make deep copies of defaultConfig blocks
   * @private
   */

  static fork(obj) {
    let ret, key, value;

    if (obj && obj.constructor === Object) {
      ret = Object.setPrototypeOf({}, obj);

      for (key in obj) {
        value = obj[key];

        if (value) {
          if (value.constructor === Object) {
            ret[key] = OH.fork(value);
          } else if (value instanceof Array) {
            ret[key] = value.slice();
          }
        }
      }
    } else {
      ret = obj;
    }

    return ret;
  }
  /**
   * Copies the named properties from the `source` parameter into the `dest` parameter.
   * @param {Object} dest The destination into which properties are copied.
   * @param {Object} source The source from which properties are copied.
   * @param {String[]} props The list of property names.
   * @returns The `dest` object.
   */

  static copyProperties(dest, source, props) {
    let prop, i;

    for (i = 0; i < props.length; i++) {
      prop = props[i];

      if (prop in source) {
        dest[prop] = source[prop];
      }
    }

    return dest;
  }
  /**
   * Copies the named properties from the `source` parameter into the `dest` parameter
   * unless the property already exists in the `dest`.
   * @param {Object} dest The destination into which properties are copied.
   * @param {Object} source The source from which properties are copied.
   * @param {String[]} props The list of property names.
   * @returns The `dest` object.
   */

  static copyPropertiesIf(dest, source, props) {
    if (source) {
      for (const prop of props) {
        if (!(prop in dest)) {
          dest[prop] = source[prop];
        }
      }
    }

    return dest;
  }
  /**
   * Returns an array containing the keys and values of all enumerable properties from every prototype level for the
   * object. If `object` is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve entries.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @returns {Array}
   * @internal
   */

  static entries(object, ignore) {
    const result = [],
          call = typeof ignore === 'function';

    if (object) {
      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
          result.push([p, object[p]]);
        }
      }
    }

    return result;
  }
  /**
   * Populates an `object` with the provided `entries`.
   * @param {Array} entries The key/value pairs (2-element arrays).
   * @param {Object} [object={}] The object onto which to add `entries`.
   * @returns {Object} The passed `object` (by default, a newly created object).
   * @internal
   */

  static fromEntries(entries, object) {
    object = object || {};

    if (entries) {
      for (let i = 0; i < entries.length; ++i) {
        object[entries[i][0]] = entries[i][1];
      }
    }

    return object;
  }
  /**
   * Returns an array containing all enumerable property names from every prototype level for the object. If `object`
   * is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve property names.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
   * function is stored in the returned array. It is called with the array element as the first parameter, and the
   * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
   * on).
   * @returns {String[]}
   */

  static keys(object, ignore, mapper) {
    const result = [],
          call = typeof ignore === 'function';

    if (object) {
      let index = 0;

      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
          result.push(mapper ? mapper(p, index) : p);
          ++index;
        }
      }
    }

    return result;
  }
  /**
   * Returns an array containing the values of all enumerable properties from every prototype level for the object.
   * If `object` is `null`, this method returns an empty array.
   * @param {Object} object Object from which to retrieve values.
   * @param {Object|Function} [ignore] Optional object of names to ignore or a function accepting the name and value
   * which returns `true` to ignore the item.
   * @param {Function} [mapper] Optional function to call for each non-ignored item. If provided, the result of this
   * function is stored in the returned array. It is called with the array element as the first parameter, and the
   * index in the result array as the second argument (0 for the first, non-ignored element, 1 for the second and so
   * on).
   * @returns {Array}
   * @internal
   */

  static values(object, ignore, mapper) {
    const result = [],
          call = typeof ignore === 'function';

    if (object) {
      let index = 0;

      for (const p in object) {
        if (call ? !ignore(p, object[p]) : !(ignore !== null && ignore !== void 0 && ignore[p])) {
          result.push(mapper ? mapper(object[p], index) : object[p]);
          ++index;
        }
      }
    }

    return result;
  } //region Path

  /**
   * Checks if a given path exists in an object
   * @param {Object} object Object to check path on
   * @param {String} path Dot-separated path, e.g. 'object.childObject.someKey'
   * @returns {Boolean} Returns `true` if path exists or `false` if it does not
   */

  static pathExists(object, path) {
    const properties = path.split('.');
    return properties.every(property => {
      if (!object || !(property in object)) {
        return false;
      }

      object = object[property];
      return true;
    });
  }
  /**
   * Creates a simple single level key-value object from complex deep object.
   * @param {Object} object Object to extract path and values from
   * @returns {Object} Key-value object where key is a path to the corresponding value
   * @internal
   *
   * ```javascript
   * // converts deep object
   * {
   *     foo : {
   *         bar : {
   *             test : 1
   *         }
   *     }
   * }
   * // into a single level object
   * {
   *     'foo.bar.test' : 1
   * }
   * ```
   */

  static pathifyKeys(object) {
    const result = {};

    for (const key in object) {
      if (hasOwnProperty$4.call(object, key)) {
        if (object[key] instanceof Object) {
          const paths = this.pathifyKeys(object[key]);

          for (const path in paths) {
            result[`${key}.${path}`] = paths[path];
          }
        } else {
          result[key] = object[key];
        }
      }
    }

    return result;
  }
  /**
   * Removes value for a given path in the object. Doesn't cleanup empty objects.
   * @param {Object} object
   * @param {String} path Dot-separated path, e.g. `obj.child.someKey`
   * @internal
   */

  static deletePath(object, path) {
    path.split('.').reduce((result, key, index, array) => {
      if (result == null) {
        return null;
      } else if (hasOwnProperty$4.call(result, key)) {
        if (index === array.length - 1) {
          delete result[key];
        } else {
          return result[key];
        }
      }
    }, object);
  } //endregion

  static coerce(from, to) {
    const fromType = typeOf(from),
          toType = typeOf(to),
          isString = typeof from === 'string';

    if (fromType !== toType) {
      switch (toType) {
        case 'string':
          return String(from);

        case 'number':
          return Number(from);

        case 'boolean':
          // See http://ecma262-5.com/ELS5_HTML.htm#Section_11.9.3 as to why '0'.
          // TL;DR => ('0' == 0), so if given string '0', we must return boolean false.
          return isString && (!from || from === 'false' || from === '0') ? false : Boolean(from);

        case 'null':
          return isString && (!from || from === 'null') ? null : false;

        case 'undefined':
          return isString && (!from || from === 'undefined') ? undefined : false;

        case 'date':
          return isString && isNaN(from) ? DateHelper.parse(from) : Date(Number(from));
      }
    }

    return from;
  }

  static wrapProperty(object, propertyName, newGetter, newSetter, deep = true) {
    const newProperty = {};
    let proto = Object.getPrototypeOf(object),
        existingProperty = Object.getOwnPropertyDescriptor(proto, propertyName);

    while (!existingProperty && proto && deep) {
      proto = Object.getPrototypeOf(proto);

      if (proto) {
        existingProperty = Object.getOwnPropertyDescriptor(proto, propertyName);
      }
    }

    if (existingProperty) {
      if (existingProperty.set) {
        newProperty.set = v => {
          existingProperty.set.call(object, v); // Must invoke the getter in case "v" has been transformed.

          newSetter && newSetter.call(object, existingProperty.get.call(object));
        };
      } else {
        newProperty.set = newSetter;
      }

      if (existingProperty.get) {
        newProperty.get = () => {
          let result = existingProperty.get.call(object);

          if (newGetter) {
            result = newGetter.call(object, result);
          }

          return result;
        };
      } else {
        newProperty.get = newGetter;
      }
    } else {
      newProperty.set = v => {
        object[`_${propertyName}`] = v;
        newSetter && newSetter.call(object, v);
      };

      newProperty.get = () => {
        let result = object[`_${propertyName}`];

        if (newGetter) {
          result = newGetter.call(object, result);
        }

        return result;
      };
    }

    Object.defineProperty(object, propertyName, newProperty);
  }
  /**
   * Intercepts access to a `property` of a given `object`.
   *
   * ```javascript
   *      ObjectHelper.hookProperty(object, 'prop', class {
   *          get value() {
   *              return super.value;
   *          }
   *          set value(v) {
   *              super.value = v;
   *          }
   *      });
   * ```
   * The use of `super` allows the hook's getter and setter to invoke the object's existing get/set.
   *
   * @param {Object} object
   * @param {String} property
   * @param {Function} hook A `class` defining a `value` property getter and/or setter.
   * @returns {Function} A function that removes the hook when called.
   * @internal
   */

  static hookProperty(object, property, hook) {
    const desc = ObjectHelper.getPropertyDescriptor(hook.prototype, 'value'),
          existingDesc = ObjectHelper.getPropertyDescriptor(object, property),
          fieldName = `_${property}`,
          base = class {
      get value() {
        return existingDesc ? existingDesc.get.call(this) : this[fieldName];
      }

      set value(v) {
        if (existingDesc) {
          existingDesc.set.call(this, v);
        } else {
          this[fieldName] = v;
        }
      }

    },
          baseDesc = ObjectHelper.getPropertyDescriptor(base.prototype, 'value');
    Object.setPrototypeOf(hook.prototype, base.prototype); // direct super calls to our "base" implementation

    Object.defineProperty(object, property, {
      configurable: true,
      get: desc.get || baseDesc.get,
      set: desc.set || baseDesc.set
    });
    return () => delete object[property];
  }
  /**
   * Finds a property descriptor for the passed object from all inheritance levels.
   * @param {Object} object The Object whose property to find.
   * @param {String} propertyName The name of the property to find.
   * @returns {Object} An ECMA property descriptor is the property was found, otherwise `null`
   */

  static getPropertyDescriptor(object, propertyName) {
    let result = null;

    for (let o = object; o && !result && !hasOwnProperty$4.call(o, 'isBase'); o = Object.getPrototypeOf(o)) {
      result = Object.getOwnPropertyDescriptor(o, propertyName);
    }

    return result;
  }
  /**
   * Return `true` if the given `object` defines the specified `property` as defined by
   * [Object.hasOwnProperty](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/hasOwnProperty).
   * @param {Object} object
   * @param {String} property
   * @returns {Boolean}
   * @internal
   */

  static hasOwn(object, property) {
    // Benefits:
    // - Better compression over direct call to Object.prototype.hasOwnProperty.call(object, property)
    // - Almost as compact as "object.hasOwnProperty()" but still works with Object.create(null) objects
    // - Does not throw if object is null/undefined unlike all forms of hasOwnProperty
    //
    // Even given above, if needed often in a module it is probably still better to use:
    //  const { hasOwnProperty } = Object.prototype;
    // and
    //  hasOwnProperty.call()
    return object ? hasOwnProperty$4.call(object, property) : false;
  }
  /**
   * Changes the passed object and removes all null and undefined properties from it
   * @param {Object} object Target object
   * @param {Boolean} [keepNull] Pass true to only remove undefined properties
   * @returns {Object} Passed object
   */

  static cleanupProperties(object, keepNull = false) {
    Object.entries(object).forEach(([key, value]) => {
      if (keepNull) {
        value === undefined && delete object[key];
      } else {
        value == null && delete object[key];
      }
    });
    return object;
  }
  /**
   * Changes the passed object and removes all properties from it.
   * Used while mutating when need to keep reference to the object but replace its properties.
   * @param {Object} object Target object
   * @returns {Object} Passed object
   */

  static removeAllProperties(obj) {
    Object.keys(obj).forEach(key => delete obj[key]);
    return obj;
  } //region Assert type

  /**
   * Checks that the supplied value is of the specified type.Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} type Expected type
   * @param {String} name Name of the value, used in error message
   * @param {Boolean} [allowNull] Accept `null` without throwing
   */

  static assertType(value, type, name) {
    const valueType = typeOf(value);

    if (value != null && valueType !== type) {
      throw new Error(`Incorrect type "${valueType}" for ${name}, expected "${type}"`);
    }
  }
  /**
   * Checks that the supplied value is a plain object. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertObject(value, name) {
    OH.assertType(value, 'object', name);
  }
  /**
   * Checks that the supplied value is an instance of a Bryntum class. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertInstance(value, name) {
    OH.assertType(value, 'instance', name);
  }
  /**
   * Checks that the supplied value is a Bryntum class. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertClass(value, name) {
    OH.assertType(value, 'class', name);
  }
  /**
   * Checks that the supplied value is a function. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertFunction(value, name) {
    if (typeof value !== 'function' || value.isBase || value.$$name) {
      throw new Error(`Incorrect type for ${name}, got "${value}" (expected a function)`);
    }
  }
  /**
   * Checks that the supplied value is a number. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertNumber(value, name) {
    const asNumber = Number(value);

    if (typeof value !== 'number' || isNaN(asNumber)) {
      throw new Error(`Incorrect type for ${name}, got "${value}" (expected a Number)`);
    }
  }
  /**
   * Checks that the supplied value is a boolean. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertBoolean(value, name) {
    OH.assertType(value, 'boolean', name);
  }
  /**
   * Checks that the supplied value is a string. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertString(value, name) {
    OH.assertType(value, 'string', name);
  }
  /**
   * Checks that the supplied value is an array. Throws if it is not
   * @param {Object} value Value to check type of
   * @param {String} name Name of the value, used in error message
   */

  static assertArray(value, name) {
    OH.assertType(value, 'array', name);
  } //endregion

  /**
   * Number.toFixed(), with polyfill for browsers that needs it
   * @param {Number} number
   * @param {Number} digits
   * @returns {String} A fixed point string representation of the passed number.
   */

  static toFixed(number, digits) {
    if (toFixedFix) {
      return toFixedFix(number, digits);
    }

    return number.toFixed(digits);
  }
  /**
   * Round the passed number to closest passed step value.
   * @param {Number} number The number to round.
   * @param {Number} [step] The step value to round to.
   * @returns {Number} The number rounded to the closest step.
   */

  static roundTo(number, step = 1) {
    return Math.round(number / step) * step;
  }
  /**
   * Round the passed number to the passed number of decimals.
   * @param {Number} number The number to round.
   * @param {Number} digits The number of decimal places to round to.
   * @returns {Number} The number rounded to the passed number of decimal places.
   */

  static round(number, digits) {
    // Undefined or null means do not round. NOT round to no decimals.
    if (digits == null) {
      return number;
    }

    const factor = 10 ** digits;
    return Math.round(number * factor) / factor;
  }
  /**
   * Returns a non-null entry from a Map for a given key path. This enables a specified defaultValue to be added "just
   * in time" which is returned if the key is not already present.
   * @param {Map} map The Map to find the key in (and potentially add to).
   * @param {String|Number|String[]|Number[]} path Dot-separated path, e.g. 'firstChild.childObject.someKey',
   * or the key path as an array, e.g. ['firstChild', 'childObject', 'someKey'].
   * @param {Object} [defaultValue] Optionally the value to insert if the key is not found.
   */

  static getMapPath(map, path, defaultValue) {
    const keyPath = Array.isArray(path) ? path : typeof path === 'string' ? path.split('.') : [path],
          simpleKey = keyPath.length === 1,
          topKey = keyPath[0],
          topValue = map.has(topKey) ? map.get(topKey) : map.set(topKey, simpleKey ? defaultValue : {}).get(topKey); // If it was a simple key, we are done.

    if (simpleKey) {
      return topValue;
    } // Go down the property path on the top Object, filling entries in until the leaf.

    return OH.getPathDefault(topValue, keyPath.slice(1), defaultValue);
  }

}
const OH = ObjectHelper;
ObjectHelper._$name = 'ObjectHelper';

const allBorders = ['border-top-width', 'border-right-width', 'border-bottom-width', 'border-left-width'],
      allMargins = ['margin-top', 'margin-right', 'margin-bottom', 'margin-left'],
      allPaddings = ['padding-top', 'padding-right', 'padding-bottom', 'padding-left'],
      borderNames = {
  t: 'border-top-width',
  r: 'border-right-width',
  b: 'border-bottom-width',
  l: 'border-left-width'
},
      paddingNames = {
  t: 'padding-top',
  r: 'padding-right',
  b: 'padding-bottom',
  l: 'padding-left'
},
      zeroBased = Object.freeze({
  x: 0,
  y: 0
}),
      alignSpecRe$1 = /^([trblc])(\d*)-([trblc])(\d*)$/i,
      alignPointRe = /^([trblc])(\d*)$/i,
      edgeNames = ['top', 'right', 'bottom', 'left'],
      edgeIndices = {
  t: 0,
  r: 1,
  b: 2,
  l: 3
},
      defaultAlignments = ['b-t', 'l-r', 't-b', 'r-l'],
      edgeAligments = {
  bt: 1,
  tb: 1,
  lr: 1,
  rl: 1
},
      zeroOffsets = Object.freeze([0, 0]),
      matchDimensions = ['width', 'height'],
      parseEdges = (top, right = top, bottom = top, left = right) => {
  return Array.isArray(top) ? parseEdges.apply(null, top) : [top, right, bottom, left];
}; // Parse a l0-r0 (That's how Menus align to their owning MenuItem) align spec.

function parseAlign(alignSpec) {
  const parts = alignSpecRe$1.exec(alignSpec),
        myOffset = parseInt(parts[2] || 50),
        targetOffset = parseInt(parts[4] || 50),
        myEdge = parts[1],
        targetEdge = parts[3],
        edgeAligned = edgeAligments[myEdge + targetEdge]; // Comments assume the Menu's alignSpec of l0-r0 is used.

  return {
    myAlignmentPoint: myEdge + myOffset,
    // l0
    myEdge,
    // l
    myOffset,
    // 0
    targetAlignmentPoint: targetEdge + targetOffset,
    // r0
    targetEdge,
    // r
    targetOffset,
    // 0
    startZone: edgeIndices[targetEdge],
    // 1 - start trying zone 1 in TRBL order
    edgeAligned // Edge-to-edge align requested

  };
} // Takes a result from the above function and flips edges for the axisLock config

function flipAlign(align) {
  return `${edgeNames[(edgeIndices[align.myEdge] + 2) % 4][0]}${align.myOffset}-${edgeNames[(edgeIndices[align.targetEdge] + 2) % 4][0]}${align.targetOffset}`;
}

function createOffsets(offset) {
  if (offset == null) {
    return zeroOffsets;
  } else if (typeof offset === 'number') {
    return [offset, offset];
  }

  return offset;
}
/**
 * Encapsulates rectangular areas for comparison, intersection etc.
 *
 * Note that the `right` and `bottom` properties are *exclusive*.
 *
 */

class Rectangle {
  // Class does not extend Base, so we need to define this
  get isRectangle() {
    return true;
  }
  /**
   * Returns the Rectangle in document based coordinates of the passed element.
   *
   * *Note:* If the element passed is the `document` or `window` the `window`'s
   * rectangle is returned which is always at `[0, 0]` and encompasses the
   * browser's entire document viewport.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll=false] Use browser viewport based coordinates.
   * @return {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */

  static from(element, relativeTo, ignorePageScroll) {
    if (element.isRectangle) {
      return element;
    }

    if (ignorePageScroll === undefined && typeof relativeTo === 'boolean') {
      ignorePageScroll = relativeTo;
      relativeTo = null;
    }

    if (relativeTo) {
      // TODO: nige should figure out if there is a better solution
      let {
        scrollLeft,
        scrollTop
      } = relativeTo;

      if (BrowserHelper.isSafari && relativeTo === document.body) {
        scrollLeft = scrollTop = 0;
      }

      relativeTo = Rectangle.from(relativeTo).translate(-scrollLeft, -scrollTop);
    } else {
      relativeTo = zeroBased;
    } // Viewport is denoted by requesting window or document.
    // document.body may overflow the viewport, so this must not be evaluated as the viewport.

    const // If body is 0 height we should treat this case as a viewport
    isViewport = element === document || element === globalThis,
          isSFViewport = element === document.body && document.body.offsetHeight === 0,
          sfElRect = isSFViewport && element.getBoundingClientRect(),
          viewRect = isSFViewport // In Salesforce body and html have 0 height so to get correct viewport vertical size we have to use
    // scrollHeight on html element.
    ? new Rectangle(sfElRect.left, sfElRect.top, sfElRect.width, document.body.parentElement.scrollHeight) : isViewport ? new Rectangle(0, 0, globalThis.innerWidth, globalThis.innerHeight) : element.getBoundingClientRect(),
          scrollOffset = ignorePageScroll || isViewport ? [0, 0] : [globalThis.pageXOffset, globalThis.pageYOffset];
    return new Rectangle(viewRect.left + scrollOffset[0] - relativeTo.x, viewRect.top + scrollOffset[1] - relativeTo.y, viewRect.width, viewRect.height);
  }
  /**
   * Returns the Rectangle in viewport coordinates of the passed element.
   *
   * *Note:* If the element passed is the `document` or `window` the `window`'s rectangle is returned which is always
   * at `[0, 0]` and encompasses the browser's entire document viewport.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @return {Core.helper.util.Rectangle} The Rectangle in viewport based coordinates. Relative to the _relativeTo_
   * parameter if provided.
   */

  static fromScreen(element, relativeTo) {
    return Rectangle.from(element, relativeTo,
    /* ignorePageScroll = */
    true);
  }
  /**
   * Returns the inner Rectangle (within border) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @return {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */

  static inner(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll); // Can only ask for the following styles if element is in the document.

    if (document.body.contains(element)) {
      const borders = DomHelper.getStyleValue(element, allBorders);
      result.x += parseFloat(borders[borderNames.l]);
      result.y += parseFloat(borders[borderNames.t]);
      result.right -= parseFloat(borders[borderNames.r]);
      result.bottom -= parseFloat(borders[borderNames.b]);
    }

    return result;
  }
  /**
   * Returns the content Rectangle (within border and padding) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @return {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */

  static content(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll); // Can only ask for the following styles if element is in the document.

    if (document.body.contains(element)) {
      const borders = DomHelper.getStyleValue(element, allBorders),
            padding = DomHelper.getStyleValue(element, allPaddings);
      result.x += parseFloat(borders[borderNames.l]) + parseFloat(padding[paddingNames.l]);
      result.y += parseFloat(borders[borderNames.t]) + parseFloat(padding[paddingNames.t]);
      result.right -= parseFloat(borders[borderNames.r]) + parseFloat(padding[paddingNames.r]);
      result.bottom -= parseFloat(borders[borderNames.b]) + parseFloat(padding[paddingNames.b]);
    }

    return result;
  }
  /**
   * Returns the client Rectangle (within border and padding and scrollbars) in document based coordinates of the
   * passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @return {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates. Relative to the _relativeTo_ parameter if passed.
   */

  static client(element, relativeTo, ignorePageScroll = false) {
    const result = this.content(element, relativeTo, ignorePageScroll),
          scrollbarWidth = DomHelper.scrollBarWidth;
    let padding;

    if (scrollbarWidth) {
      // Capture width taken by any vertical scrollbar.
      // If there is a vertical scrollbar, shrink the box.
      // TODO: We may have to shrink from the left in RTL mode.
      if (element.scrollHeight > element.clientHeight && DomHelper.getStyleValue(element, 'overflow-y') !== 'hidden') {
        padding = parseFloat(DomHelper.getStyleValue(element, 'padding-right'));
        result.right += padding - Math.max(padding, scrollbarWidth);
      } // Capture height taken by any horizontal scrollbar.
      // If there is a horizontal scrollbar, shrink the box.

      if (element.scrollWidth > element.clientWidth && DomHelper.getStyleValue(element, 'overflow-x') !== 'hidden') {
        padding = parseFloat(DomHelper.getStyleValue(element, 'padding-bottom'));
        result.bottom += padding - Math.max(padding, scrollbarWidth);
      }
    } // The client region excluding any scrollbars.

    return result;
  }
  /**
   * Returns the outer Rectangle (including margin) in document based coordinates of the passed element.
   * @param {HTMLElement} element The element to calculate the Rectangle for.
   * @param {HTMLElement} [relativeTo] Optionally, a parent element in whose space to calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll] Use browser viewport based coordinates.
   * @return {Core.helper.util.Rectangle} The Rectangle in document based (or, optionally viewport based) coordinates.
   * Relative to the _relativeTo_ parameter if passed.
   * @internal
   */

  static outer(element, relativeTo, ignorePageScroll = false) {
    const result = this.from(element, relativeTo, ignorePageScroll); // Can only ask for the following styles if element is in the document.

    if (document.body.contains(element)) {
      const margin = DomHelper.getStyleValue(element, allMargins);
      result.x -= parseFloat(margin['margin-left']);
      result.y -= parseFloat(margin['margin-top']);
      result.right += parseFloat(margin['margin-right']);
      result.bottom += parseFloat(margin['margin-bottom']);
    }

    return result;
  }
  /**
   * Returns a new rectangle created as the union of all supplied rectangles.
   * @param {Core.helper.util.Rectangle[]} rectangles
   * @return {Core.helper.util.Rectangle}
   */

  static union(...rectangles) {
    let {
      x,
      y,
      right,
      bottom
    } = rectangles[0],
        current;

    if (rectangles.length > 1) {
      for (let i = 1; i < rectangles.length; i++) {
        current = rectangles[i];

        if (current.x < x) {
          x = current.x;
        }

        if (current.y < y) {
          y = current.y;
        }

        if (current.right > right) {
          right = current.right;
        }

        if (current.bottom > bottom) {
          bottom = current.bottom;
        }
      }
    }

    return new Rectangle(x, y, right - x, bottom - y);
  }
  /**
   * Rounds this Rectangle to the pixel resolution of the current display or to the nearest
   * passed unit which defaults to the current display's [`devicePixelRatio`](https://developer.mozilla.org/en-US/docs/Web/API/Window/devicePixelRatio).
   * @param {Number} [devicePixelRatio] device pixel ratio which defaults to `window.devicePixelRatio`
   */

  roundPx(devicePixelRatio = globalThis.devicePixelRatio || 1) {
    const me = this;
    me._x = DomHelper.roundPx(me._x, devicePixelRatio);
    me._y = DomHelper.roundPx(me._y, devicePixelRatio);
    me._width = DomHelper.roundPx(me._width, devicePixelRatio);
    me._height = DomHelper.roundPx(me._height, devicePixelRatio);
    return me;
  } // This class doesn't extend Base and extending doesn't seem to be
  // the way to go. Instead we duplicate smallest piece of logic here

  static get $$name() {
    return hasOwnProperty.call(this, '$name') && this.$name || // _$name is filled by webpack for every class (cls._$name = '...')
    hasOwnProperty.call(this, '_$name') && this._$name || this.name;
  }

  get $$name() {
    return this.constructor.$$name;
  }
  /**
   * Constructs a Rectangle
   * @param {Number} x The X coordinate
   * @param {Number} y The Y coordinate
   * @param {Number} width The width
   * @param {Number} height The height
   */

  constructor(x, y, width, height) {
    ObjectHelper.assertNumber(x, 'Rectangle.x');
    ObjectHelper.assertNumber(y, 'Rectangle.y');
    ObjectHelper.assertNumber(width, 'Rectangle.width');
    ObjectHelper.assertNumber(height, 'Rectangle.height');
    const me = this;
    me._x = x;
    me._y = y;
    me._width = width;
    me._height = height;
  }
  /**
   * Creates a copy of this Rectangle.
   */

  clone() {
    const me = this,
          result = new Rectangle(me.x, me.y, me.width, me.height);
    result.isAlignRectangle = me.isAlignRectangle;
    result.minHeight = me.minHeight;
    result.minWidth = me.minWidth;
    return result;
  }
  /**
   * Returns `true` if this Rectangle wholly contains the passed rectangle.
   *
   * Note that a {@link Core.helper.util.Point} may be passed.
   * @param {Core.helper.util.Rectangle} other The Rectangle to test for containment within this Rectangle
   * @return {Boolean} `true` if the other Rectangle is wholly contained within this Rectangle
   */

  contains(other) {
    const me = this;

    if (other.isRectangle) {
      return other._x >= me._x && other._y >= me._y && other.right <= me.right && other.bottom <= me.bottom;
    } else {
      return false;
    }
  }
  /**
   * Checks if this Rectangle intersects the passed Rectangle
   * @param {Core.helper.util.Rectangle} other The Rectangle to intersect with this.
   * @param {Boolean} [useBoolean] Specify `true` to return a boolean value instead of constructing a new Rectangle
   * @param {Boolean} [allowZeroDimensions] `true` to consider zero-width or zero-hight rectangles as intersecting if coordinates indicate the intersection
   * @return {Core.helper.util.Rectangle|Boolean} Returns the intersection Rectangle or `false` if there is no intersection.
   */

  intersect(other, useBoolean = false, allowZeroDimensions = false) {
    const me = this,
          y = Math.max(me.y, other.y),
          r = Math.min(me.right, other.right),
          b = Math.min(me.bottom, other.bottom),
          x = Math.max(me.x, other.x),
          intersect = allowZeroDimensions ? b >= y && r >= x : b > y && r > x;

    if (intersect) {
      return useBoolean ? true : new Rectangle(x, y, r - x, b - y);
    } else {
      return false;
    }
  }

  equals(other, round = false) {
    const processor = round ? x => Math.round(x) : x => x;
    return other.isRectangle && processor(other.x) === processor(this.x) && processor(other.y) === processor(this.y) && processor(other.width) === processor(this.width) && processor(other.height) === processor(this.height);
  }
  /**
   * Translates this Rectangle by the passed vector. Size is maintained.
   * @param {Number} x The X translation vector.
   * @param {Number} y The Y translation vector.
   * @returns This Rectangle;
   */

  translate(x, y) {
    this._x += x || 0;
    this._y += y || 0;
    return this;
  }
  /**
   * Moves this Rectangle to the passed `x`, `y` position. Size is maintained.
   * @param {Number} x The new X position.
   * @param {Number} y The new Y position.
   * @returns This Rectangle;
   */

  moveTo(x, y) {
    if (x != null) {
      this._x = x;
    }

    if (y != null) {
      this._y = y;
    }

    return this;
  }
  /**
   * Returns the vector which would translate this Rectangle (or Point) to the same position as the other Rectangle (or point)
   * @param {Core.helper.util.Rectangle|Core.helper.util.Point} other The Rectangle or Point to calculate the delta to.
   * @returns {Array} Returns a vector using format `[deltaX, deltaY]`
   * @internal
   */

  getDelta(other) {
    return [other.x - this.x, other.y - this.y];
  }
  /**
   * The center point of this rectangle.
   * @property {Core.helper.util.Point}
   */

  get center() {
    return new Rectangle.Point(this.x + this.width / 2, this.y + this.height / 2);
  }
  /**
   * Get/sets the X coordinate of the Rectangle. Note that this does *not* translate the
   * Rectangle. The requested {@link #property-width} will change.
   * @property {Number}
   */

  set x(x) {
    const xDelta = x - this._x;
    this._x = x;
    this._width -= xDelta;
  }

  get x() {
    return this._x;
  }
  /**
   * Alias for x. To match DOMRect.
   * @property {Number}
   */

  set left(x) {
    this.x = x;
  }

  get left() {
    return this.x;
  }
  /**
   * Alias for y. To match DOMRect.
   * @property {Number}
   */

  set top(y) {
    this.y = y;
  }

  get top() {
    return this.y;
  }
  /**
   * Get/sets the Y coordinate of the Rectangle. Note that this does *not* translate the
   * Rectangle. The requested {@link #property-height} will change.
   * @property {Number}
   */

  set y(y) {
    const yDelta = y - this._y;
    this._y = y;
    this._height -= yDelta;
  }

  get y() {
    return this._y;
  }
  /**
   * Get/sets the width of the Rectangle. Note that the requested {@link #property-right} will change.
   * @property {Number}
   */

  set width(width) {
    this._width = width;
  }

  get width() {
    return this._width;
  }
  /**
   * Get/sets the height of the Rectangle. Note that the requested {@link #property-bottom} will change.
   * @property {Number}
   */

  set height(height) {
    this._height = height;
  }

  get height() {
    return this._height;
  }
  /**
   * Get/sets the right edge of the Rectangle. Note that the requested {@link #property-width} will change.
   *
   * The right edge value is exclusive of the calculated rectangle width. So x=0 and right=10
   * means a width of 10.
   * @property {Number}
   */

  set right(right) {
    this._width = right - this._x;
  }

  get right() {
    return this._x + this._width;
  }
  /**
   * Get/sets the bottom edge of the Rectangle. Note that the requested {@link #property-height} will change.
   *
   * The bottom edge value is exclusive of the calculated rectangle height. So y=0 and bottom=10
   * means a height of 10.
   * @property {Number}
   */

  set bottom(bottom) {
    this._height = bottom - this._y;
  }

  get bottom() {
    return this._y + this._height;
  }

  get area() {
    return this.width * this.height;
  }

  set minWidth(minWidth) {
    const me = this;

    if (isNaN(minWidth)) {
      me._minWidth = null;
    } else {
      me._minWidth = Number(minWidth); // If this is being used as an alignment calculation rectangle, minWidth has a different meaning.
      // It does not mean that the width has to be at least this value. It means that under constraint,
      // it is willing to shrink down to that value before falling back to another align position.

      if (!me.isAlignRectangle) {
        me.width = Math.max(me.width, me._minWidth);
      }
    }
  }

  get minWidth() {
    return this._minWidth;
  }

  set minHeight(minHeight) {
    const me = this;

    if (isNaN(minHeight)) {
      me._minHeight = null;
    } else {
      me._minHeight = Number(minHeight); // If this is being used as an alignment calculation rectangle, minHeight has a different meaning.
      // It does not mean that the height has to be at least this value. It means that under constraint,
      // it is willing to shrink down to that value before falling back to another align position.

      if (!me.isAlignRectangle) {
        me.height = Math.max(me.height, me._minHeight);
      }
    }
  }

  get minHeight() {
    return this._minHeight;
  }
  /**
   * Modifies the bounds of this Rectangle by the specified deltas.
   * @param {Number} x How much to *add* to the x position.
   * @param {Number} y  How much to *add* to the y position.
   * @param {Number} width  How much to add to the width.
   * @param {Number} height  How much to add to the height.
   * @returns This Rectangle
   */

  adjust(x, y, width, height) {
    const me = this;
    me.x += x;
    me.y += y;
    me.width += width;
    me.height += height;
    return me;
  }
  /**
   * Modifies the bounds of this rectangle by expanding them by the specified amount in all directions.
   * The parameters are read the same way as CSS margin values.
   *
   * - If one value is passed, all edges are inflated by that value.
   * - If two values are passed they are top/bottom inflation and left/right inflation.
   * - If four values are passed, that's top, right, bottom, left.
   * @param {Number} amount How much to inflate, or the top value if more than one values passed.
   * @param {Number} right How much to inflate the right side, or both left and right is only two values passed.
   * @param {Number} bottom How much to inflate the bottom side.
   * @param {Number} left How much to inflate the left side.
   * @returns {Core.helper.util.Rectangle} This Rectangle
   * @internal
   */

  inflate(top, right = top, bottom = top, left = right) {
    if (Array.isArray(top)) {
      [top, right, bottom, left] = parseEdges(top);
    }

    return this.adjust(-left, -top, right, bottom);
  }
  /**
   * Modifies the bounds of this rectangle by reducing them by the specified amount in all directions.
   * The parameters are read the same way as CSS margin values.
   *
   * - If one value is passed, all edges are deflated by that value.
   * - If two values are passed they are top/bottom deflation and left/right deflation.
   * - If four values are passed, that's top, right, bottom, left.
   * @param {Number} amount How much to deflate, or the top value if more than one values passed.
   * @param {Number} right How much to deflate the right side, or both left and right is only two values passed.
   * @param {Number} bottom How much to deflate the bottom side.
   * @param {Number} left How much to deflate the left side.
   * @returns {Core.helper.util.Rectangle} This Rectangle
   * @internal
   */

  deflate(top, right = top, bottom = top, left = right) {
    if (Array.isArray(top)) {
      [top, right, bottom, left] = parseEdges(top);
    }

    return this.adjust(left, top, -right, -bottom);
  }
  /**
   * Attempts constrain this Rectangle into the passed Rectangle. If the `strict` parameter is `true`
   * then this method will return `false` if constraint could not be achieved.
   *
   * If this Rectangle has a `minHeight` or `minWidth` property, size will be adjusted while attempting to constrain.
   *
   * Right and bottom are adjusted first leaving the top and bottom sides to "win" in the case that this Rectangle overflows
   * the constrainTo Rectangle.
   * @param {Core.helper.util.Rectangle} constrainTo The Rectangle to constrain this Rectangle into if possible.
   * @param {Boolean} strict Pass `true` to return false, and leave this Rectangle unchanged if constraint
   * could not be achieved.
   * @returns This Rectangle. If `strict` is true, and constraining was not successful, `false`.
   */

  constrainTo(constrainTo, strict) {
    const me = this,
          originalHeight = me.height,
          originalY = me.y,
          minWidth = me.minWidth || me.width,
          minHeight = me.minHeight || me.height;

    if (me.height >= constrainTo.height) {
      // If we're strict, fail if we could *never* fit into available height.
      if (strict && minHeight > constrainTo.height) {
        return false;
      } // If we are >= constrain height, we will have to be at top edge of constrainTo

      me._y = constrainTo.y;
      me.height = constrainTo.height;
    }

    if (me.width >= constrainTo.width) {
      // If we're strict, fail if we could *never* fit into available width.
      if (strict && minWidth > constrainTo.width) {
        // Could not be constrained; undo any previous attempt with height
        me.y = originalY;
        me.height = originalHeight;
        return false;
      } // If we are >= constrain width, we will have to be at left edge of constrainTo

      me._x = constrainTo.x;
      me.width = constrainTo.width;
    }

    let overflow = me.bottom - constrainTo.bottom; // Overflowing the bottom side, translate upwards.

    if (overflow > 0) {
      me.translate(0, -overflow);
    }

    overflow = me.right - constrainTo.right; // Overflowing the right side, translate leftwards.

    if (overflow > 0) {
      me.translate(-overflow);
    }

    overflow = constrainTo.y - me.y; // Now, after possible translation upwards, we overflow the top, translate downwards.

    if (overflow > 0) {
      me.translate(0, overflow);
    }

    overflow = constrainTo.x - me.x; // Now, after possible translation leftwards, we overflow the left, translate rightwards.

    if (overflow > 0) {
      me.translate(overflow);
    }

    return me;
  }
  /**
   * Returns a cloned version of this Rectangle aligned to a target Rectangle, or element or {@link Core.widget.Widget}.
   * @param {Object} spec Alignment specification.
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} spec.target The Widget or element or Rectangle to align to.
   * @param {Number[]} [spec.anchorSize] The `[width, height]` of the anchor pointer when in `top` position. The
   * width is the baseline length, and the height is the height of the arrow. If passed, the anchor position
   * will be calculated to be at the centre of the overlap of the two aligned edges and returned in the `anchor`
   * property of the resulting Rectangle:
   *
   *     {
   *         edge: 'top',         // or 'right' or 'bottom' or 'left'
   *         x/y: offset          // dimension to translate and value to translate by.
   *     }
   *
   * @param {Object} [spec.anchorPosition] an `{ x: n, y: n }` anchor translation to be used *if the requested alignment
   * succeeds without violating constraints*. If a fallback alignment is used, the anchor will be centered in the
   * overlap of the aligned edges as usual.
   * @param {Boolean} [spec.overlap] True to allow this to overlap the target.
   * @param {String} spec.align The edge alignment specification string, specifying two points to bring together.
   * The form is `[trblc][n]-[trblc][n]. The `n` is a percentage offset
   * along that edge which defines the alignment point. This is not valid for alignment point `c`.
   * For example `t0-b0' would align this Rectangle's top left corner with the bottom left corner of the `target`.
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} [spec.constrainTo] The Widget or Element or Rectangle to constrain to.
   * If the requested alignment cannot be constrained (it will first shrink the resulting Rectangle according
   * to the `minWidth` and `minHeight` properties of this rectangle), then it will try aligning at other edges
   * (honouring the `axisLock` option), and pick the fallback alignment which results in the shortest translation.
   * @param {Boolean} [spec.axisLock] Specify as a truthy value to fall back to aligning against the opposite
   * edge first if the requested alignment cannot be constrained into the `constrainTo` option. If specified
   * as `'flexible'`, then fallback will continue searching for solutions on the remaining two sides.
   * @param {Boolean} [spec.matchSize] When aligning edge-to-edge, match the length of the aligned-to
   * edge of the target. This is only honored when `axisLock` is enabled and alignment succeeds on the requested axis.
   * If __not__ aligning edge-to-edge, `matchSize` matches both dimensions of the target.
   * @param {Number|Number[]} [spec.offset] The 'x' and 'y' offset values to create an extra margin round the target
   * to offset the aligned widget further from the target. May be configured as -ve to move the aligned widget
   * towards the target - for example producing the effect of the anchor pointer piercing the target.
   * @param {Number|Number[]} [spec.constrainPadding] The amount of pixels to pad from the `constrainTo` target,
   * either a single value, or an array of values in CSS edge order.
   * @returns {Core.helper.util.Rectangle} A new Rectangle aligned as requested if possible, but if the requested position violates
   * the `constrainTo` Rectangle, the shortest translation from the requested position which obeys constraints will be used.
   */

  alignTo(spec) {
    // The target and constrainTo may be passed as HtmlElements or Widgets.
    // If so, extract the Rectangles without mutating the incoming spec.
    let result = this.clone(),
        {
      target,
      constrainTo,
      constrainPadding,
      matchSize
    } = spec,
        calculatedAnchorPosition,
        zone,
        resultZone,
        constrainingToViewport;

    if (target && !target.isRectangle) {
      target = Rectangle.from(target.element ? target.element : target);
    }

    if (constrainTo) {
      if (!constrainTo.isRectangle) {
        // Viewport is denoted by requesting window or document.
        // document.body may overflow the viewport, so this must not be evaluated as the viewport.
        constrainingToViewport = constrainTo === globalThis || constrainTo === document; // When rectangle is constrained to some element on the page other than window/document, page scroll
        // should not be taken into account

        const ignorePageScroll = 'ignorePageScroll' in spec ? spec.ignorePageScroll : !constrainingToViewport;
        constrainTo = Rectangle.from(constrainTo.element ? constrainTo.element : constrainTo, null, ignorePageScroll);
      } // Shrink the constrainTo Rectangle to account for the constrainPadding

      if (constrainPadding) {
        // An array may be used to specify sides in the CSS standard order.
        // One value means all sides reduce by the same amount.
        constrainPadding = parseEdges(constrainPadding); // If we are aligning to an element which is closer to an edge than the
        // constrainPadding value for that edge, override the constrainPadding so that
        // the visual alignment is maintained.

        constrainPadding[0] = Math.min(constrainPadding[0], target.top);
        constrainPadding[1] = Math.min(constrainPadding[1], constrainTo.right - target.right);
        constrainPadding[2] = Math.min(constrainPadding[0], constrainTo.bottom - target.bottom);
        constrainPadding[3] = Math.min(constrainPadding[0], target.left); // Must clone a passed Rectangle so as not to mutate objects passed in.

        constrainTo = constrainTo.deflate.apply(constrainTo.clone(), constrainPadding);
      }
    }

    const me = this,
          targetOffsets = createOffsets(spec.offset),
          {
      align,
      axisLock,
      anchorSize,
      anchorPosition
    } = spec,
          alignSpec = parseAlign(align),
          position = spec.position || (target && target.$$name === 'Point' ? target : null),
          targetConstrainRect = constrainTo && constrainTo.clone(),
          constraintZones = [],
          zoneOrder = [{
      zone: zone = alignSpec.startZone,
      align
    }],
          matchDimension = matchSize && matchDimensions[alignSpec.startZone & 1],
          originalSize = me[matchDimension]; // Match the size of the edge we are aligning against

    if (matchDimension && axisLock) {
      result[matchDimension] = target[matchDimension];
    } // If we are not aligning to an edge, match both diensions.
    else if (!alignSpec.edgeAligned && matchSize) {
      result.width = target.width;
      result.height = target.height;
    } // Ensure we will fit before trying

    if (constrainTo) {
      result.constrainTo(constrainTo);
    } // If we are aligning edge-to-edge, then plan our fallback strategy when we are constrained.

    if (constrainTo && alignSpec.startZone != null) {
      // Create the list of zone numbers and alignments to try in the preferred order.
      //
      // In the case of axisLock, go through the zones by each axis.
      // So if they asked for t-b, which is zone 2,
      // the array will be [2, 0, 3, 1] (t-b, b-t, r-l, l-r)
      if (axisLock) {
        // First axis flip has to maintain the offset along that axis.
        // so align: l0-r0 has to flip to align: r0-l0. See submenu flipping when
        // constrained to the edge. It flips sides but maintains vertical position.
        zoneOrder.push({
          zone: zone = (zone + 2) % 4,
          align: flipAlign(alignSpec)
        }); // Only try the other axis is axisLock is 'flexible'

        if (axisLock === 'flexible') {
          zoneOrder.push({
            zone: zone = (alignSpec.startZone + 1) % 4,
            align: defaultAlignments[zone]
          });
          zoneOrder.push({
            zone: zone = (zone + 2) % 4,
            align: defaultAlignments[zone]
          });
        }
      } // Go through the zones in order from the requested start.
      // So if they asked for t-b, which is zone 2,
      // the array will be [2, 3, 0, 1] (t-b, r-l, b-t, l-r)
      else {
        for (let i = 1; i < 4; i++) {
          zoneOrder.push({
            zone: zone = (zone + 1) % 4,
            align: defaultAlignments[zone]
          });
        }
      }
    } // Allow them to pass anchorPosition: {x: 10} to indicate that after a fully successful,
    // unconstrained align, the anchor should be 10px from the start.

    if (anchorPosition) {
      const pos = alignSpec.startZone & 1 ? 'y' : 'x';
      calculatedAnchorPosition = {
        [pos]: anchorPosition[pos],
        edge: edgeNames[(alignSpec.startZone + 2) % 4]
      };
    } // Keep the target within reach. If it's way outside, pull it back so that it's only just outside);

    if (targetConstrainRect && target) {
      targetConstrainRect.adjust(-target.width, -target.height, target.width, target.height);
      target.constrainTo(targetConstrainRect);
    } // As part of fallback process when fitting within constraints, result may shrink to our minima

    result.minWidth = me.minWidth;
    result.minHeight = me.minHeight; // We're being commanded to try to align at a position

    if (position) {
      result.moveTo(position.x, position.y);

      if (constrainTo) {
        result.constrainTo(constrainTo);
      }
    } // We're aligning to a Target Rectangle within a ConstrainTo Rectangle, taking into account
    // a possible anchor pointer, or x/y offsets. Here's the situation:
    //
    //                             <-- ConstrainTo Rectangle -->
    //  +-----------------------------------+--------------------+-------------------------+
    //  |                                   |                    |                         |
    //  |                                   |                    |                         |
    //  |                                   |                    |                         |
    //  |-----------------------------------+--------------------+-------------------------+
    //  |                                   |          ▼         |                         |
    //  |                                   | +----------------+ |                         |
    //  |                                   | |                | |                         |
    //  |                                   | |                | |                         |
    //  |                                   |▶|     Target     |◀|                         |
    //  |                                   | |                | |                         |
    //  |                                   | |                | |                         |
    //  |                                   | +----------------+ |                         |
    //  |                                   |          ▲         |                         |
    //  +-----------------------------------+--------------------+-------------------------|
    //  |                                   |                    |                         |
    //  |                                   |                    |                         |
    //  |                                   |                    |                         |
    //  +-----------------------------------+--------------------+-------------------------+
    //
    // Which results in the four possible constraint zones above, which we index in standard CSS order.
    //
    // Top    = 0
    // Right  = 1
    // Bottom = 2
    // Left   = 3
    //
    // If the initially requested alignment is not within the constrainTo rectangle
    // then, calculate these four, and then loop through them, beginning at the requested one,
    // quitting when we find a position which does not violate constraints. This includes
    // shrinking the aligning Rectangle towards its minima to attempt a fit.
    //
    // The final fallback, if there is no position which does not violate constraints
    // is to position in whichever of the four rectangles has the largest area shrinking overflowing
    // dimensions down to minima if specified.
    //
    else {
      // Offsets: If we are using an anchor to move away from the target, use anchor height in both dimensions.
      // It's rotated so that "height" always has the same meaning. It's the height of the arrow.
      const centerAligned = alignSpec.myEdge === 'c' || alignSpec.targetEdge === 'c',
            offsets = anchorSize && !centerAligned ? [anchorSize[1] + targetOffsets[0], anchorSize[1] + targetOffsets[1]] : targetOffsets,
            targetPoint = target.getAlignmentPoint(alignSpec.targetAlignmentPoint, offsets),
            myPoint = result.getAlignmentPoint(alignSpec.myAlignmentPoint);
      result.translate(targetPoint[0] - myPoint[0], targetPoint[1] - myPoint[1]); // If an overlapping position was requested, then we are *not* trying out those four zones.
      // We just respect constraint, and that's it.

      let overlap = result.intersect(target, true); // If we are aligned over our target, we just obey that within any constraint.
      // No complex edge alignment attempts to fall back to.

      if (overlap) {
        if (constrainTo) {
          result.constrainTo(constrainTo);
        }

        resultZone = alignSpec.startZone;
        result.translate(...offsets);
      } // Aligned to outside of our target, and we need to be constrained
      else if (constrainTo && !constrainTo.contains(result)) {
        let requestedResult = result.clone(),
            solutions = [],
            zone,
            largestZone; // Any configured anchorPosition becomes invalid now that we're having to move the resulting zone
        // to some unpredictable new place where it fits. It will have to be calculated based upon where
        // we end up aligning.

        calculatedAnchorPosition = null; // Calculate the four constraint zones illustrated above.
        // Top

        constraintZones[0] = zone = constrainTo.clone();
        zone.bottom = target.y - offsets[1]; // Right

        constraintZones[1] = zone = constrainTo.clone();
        zone.x = target.right + offsets[0]; // Bottom

        constraintZones[2] = zone = constrainTo.clone();
        zone.y = target.bottom + offsets[1]; // Left

        constraintZones[3] = zone = constrainTo.clone();
        zone.right = target.x - offsets[0]; // Start from the preferred edge and see if we are able to constrain to within each rectangle

        for (let i = 0; i < zoneOrder.length; i++) {
          // Revert to incoming dimension for fallback out of axisLock
          if (matchDimension && i == 2) {
            result[matchDimension] = originalSize;
          }

          zone = constraintZones[resultZone = zoneOrder[i].zone]; // Perform unconstrained alignment at the calculated alignment for the zone

          result = result.alignTo({
            target: target,
            offsets: offsets,
            align: zoneOrder[i].align
          }); // If we are able to strictly constrain into this area, then it's one of the possible solutions.
          // We choose the solution which result in the shortest translation from the initial position.

          if (result.constrainTo(zone, true)) {
            solutions.push({
              result: result,
              zone: resultZone
            }); // If this successful constraint is at the requested alignment, or at a fallback
            // alignment which has used min size constraints, then that's the correct solution.
            // If there's no size compromising, we have to pick the shortest translation.

            if (!largestZone || result.width < me.width || result.height < me.height) {
              result.align = zoneOrder[i].align;
              break;
            }
          } // Cache the largest zone we find in case we need the final fallback.

          if (!largestZone || zone.area > largestZone.area) {
            const r = result.clone(); // And just move the result clone into the edge zone

            switch (resultZone) {
              // Top
              case 0:
                r.moveTo(null, zone.bottom - r.height);
                break;
              // Right

              case 1:
                r.moveTo(zone.left);
                break;
              // Bottom

              case 2:
                r.moveTo(null, zone.top);
                break;
              // Left

              case 3:
                r.moveTo(zone.right - r.width);
                break;
            }

            largestZone = {
              area: zone.area,
              result: r,
              zone: resultZone
            };
          }
        } // The loop found at least one solution

        if (solutions.length) {
          // Multiple fallbacks with no axisLock.
          // Use the solution which resulted in the shortest translation distance from the requested alignment.
          if (solutions.length > 1 && !axisLock) {
            solutions.sort((s1, s2) => {
              const s1TranslationDistance = Math.sqrt((requestedResult.x - s1.result.x) ** 2 + (requestedResult.y - s1.result.y) ** 2),
                    s2TranslationDistance = Math.sqrt((requestedResult.x - s2.result.x) ** 2 + (requestedResult.y - s2.result.y) ** 2);
              return s1TranslationDistance - s2TranslationDistance;
            });
          } // Initial success, or axisLock. Use first successful solution.

          result = solutions[0].result;
          resultZone = solutions[0].zone;
        } // No solutions found - use the largest rectangle.
        else {
          result = largestZone.result;
          resultZone = largestZone.zone; // When we are constraining to the viewport, we must still must be constrained,
          // even after we've given up making it align *and* constrain.

          if (constrainingToViewport) {
            result.constrainTo(constrainTo);
          }
        }
      } else {
        resultZone = alignSpec.startZone;
      }

      result.zone = resultZone;
      result.overlap = overlap = result.intersect(target, true); // If they included an anchor, calculate its position along its edge.
      // TODO: Handle the edge overlap being less than anchor width.

      if (anchorSize && !overlap) {
        // If we were passed an anchorPosition, and it has remained valid (meaning the requested
        // alignment succeeded with no constraint), then anchorPosition will be set. If not,
        // we have to calculate it based upon the aligned edge.
        if (!calculatedAnchorPosition) {
          let isLeftOrRight = resultZone & 1,
              start = isLeftOrRight ? 'y' : 'x',
              end = isLeftOrRight ? 'bottom' : 'right',
              startValue = Math.max(target[start], result[start]),
              endValue = Math.min(target[end], result[end]),
              anchorStart = startValue + (endValue - startValue) / 2 - anchorSize[0] / 2,
              anchorEnd = anchorStart + anchorSize[0];

          if (anchorEnd > result[end]) {
            anchorStart -= anchorEnd - result[end];
          }

          if (anchorStart < result[start]) {
            anchorStart += result[start] - anchorStart;
          } // Return an anchor property which will have an x or y property and an edge name onto which the
          // arrow should be aligned.

          calculatedAnchorPosition = {
            [start]: anchorStart - result[start],
            edge: edgeNames[(resultZone + 2) % 4]
          };
        }

        result.anchor = calculatedAnchorPosition;
      }
    }

    return result;
  }
  /**
   * Returns the `[x, y]` position of the specified anchor point of this Rectangle in <edge><offset> format.
   * for example passing "t50" will return the centre point of the top edge, passing "r0" will return the start
   * position of the right edge (the top right corner).
   *
   * Note that the offset defaults to 50, so "t" means the centre of the top edge.
   * @param {String} alignmentPoint The alignment point to calculate. Must match the RegExp `[trbl]\d*`
   * @param {Number[]} margins The `[x, y]` margins to add from the left/right, top/bottom edge.
   * @internal
   */

  getAlignmentPoint(alignmentPoint, margins = zeroOffsets) {
    const me = this,
          parts = alignPointRe.exec(alignmentPoint),
          edge = parts[1].toLowerCase(),
          edgeOffset = Math.min(Math.max(parseInt(parts[2] || 50), 0), 100) / 100;

    switch (edge) {
      case 't':
        return [me.x + me.width * edgeOffset, me.y - margins[1]];

      case 'r':
        return [me.right + margins[0], me.y + me.height * edgeOffset];

      case 'b':
        return [me.x + me.width * edgeOffset, me.bottom + margins[1]];

      case 'l':
        return [me.x - margins[0], me.y + me.height * edgeOffset];

      case 'c':
        {
          return [me.x + me.width / 2, me.y + me.height / 2];
        }
    }
  }
  /**
   * Highlights this Rectangle using the highlighting effect of {@link Core.helper.DomHelper}
   * on a transient element which encapsulates the region's area.
   */

  highlight() {
    const me = this,
          highlightElement = DomHelper.createElement({
      parent: document.body,
      style: `position:absolute;z-index:9999999;pointer-events:none;
                            left:${me.x}px;top:${me.y}px;width:${me.width}px;height:${me.height}px`
    });
    return DomHelper.highlight(highlightElement).then(() => highlightElement.remove());
  }
  /**
   * Visualizes this Rectangle by adding a DOM element which encapsulates the region's area into the provided parent element.
   * @param {DomConfig} config Element config object
   * @returns {Element} The highlight element
   * @internal
   */

  visualize(config, asDomConfig) {
    const me = this,
          domConfig = _objectSpread2({
      style: _objectSpread2({
        left: `${me.x}px`,
        top: `${me.y}px`,
        width: `${me.width}px`,
        height: `${me.height}px`
      }, config.class ? {} : {
        position: 'absolute',
        'z-index': 9999999
      })
    }, config);

    return asDomConfig ? domConfig : DomHelper.createElement(domConfig);
  }

  toString(delimiter = ',') {
    return [`${this.top}px`, `${this.right}px`, `${this.bottom}px`, `${this.left}px`].join(delimiter);
  }

}
Rectangle._$name = 'Rectangle';

/**
 * @module Core/helper/util/DomClassList
 */

const // Presence of '[' is likely an "[object Object]" or other bogus stringification
valueSymbol = Symbol('value'),
      lengthSymbol = Symbol('length');
/**
 * This class encapsulates a list of CSS classes which can be set as the `className`
 * on an `HTMLElement`.
 *
 * Properties names set on this class equate to *adding* a class if the property's value
 * is _truthy_, or removing a class if the value is _falsy_.
 *
 * ```javascript
 * const myClassList = new DomClassList('b-test-button');
 *
 * myClassList.add('test-class');
 * myClassList.important = 1;
 *
 * myHtmlElement.className = myClassList; // Sets it to "b-test-button test-class important"
 * ```
 */

class DomClassList {
  static change(cls, add, remove, as = 'string') {
    remove = DomClassList.normalize(remove, 'object');
    const after = DomClassList.normalize(cls, 'array').filter(c => !remove[c]);

    if (add) {
      add = DomClassList.normalize(add, 'array');

      for (let i = 0; i < add.length; ++i) {
        if (!after.includes(add[i])) {
          after.push(add[i]);
        }
      }
    }

    return DomClassList.normalize(after, as);
  }

  static from(classes, returnEmpty) {
    if (classes) {
      if (classes.isDomClassList) {
        var _returnEmpty;

        returnEmpty = (_returnEmpty = returnEmpty) !== null && _returnEmpty !== void 0 ? _returnEmpty : true;
      } else {
        var _returnEmpty2;

        returnEmpty = (_returnEmpty2 = returnEmpty) !== null && _returnEmpty2 !== void 0 ? _returnEmpty2 : Objects.isObject(classes) && !Objects.isEmpty(classes);
        classes = new DomClassList(classes);
      }

      if (!classes.value && !returnEmpty) {
        classes = null;
      }
    }

    return classes || (returnEmpty ? new DomClassList() : null);
  }
  /**
   * Converts a class name of any understood type to a desired form.
   * @param {String|String[]|Object|Set|Map|HTMLElement} cls
   * @param {String} as Pass `'object'` to return an object with the class names as its keys (all keys will have a
   * value of `true`), or pass `'array'` to return an array of class names, or pass `'string'` (the default) to
   * return a space-separated string of class names.
   * @returns {String|String[]|Object}
   * @internal
   */

  static normalize(cls, as = 'string') {
    cls = cls || ''; // promote null to '' to avoid typeof snag

    const type = typeof cls,
          asArray = as === 'array',
          asObject = as === 'object',
          asString = !asArray && !asObject;
    let isString = type === 'string',
        c,
        i,
        ret;

    if (type === 'object') {
      var _cls;

      if (cls.nodeType === Element.ELEMENT_NODE && typeof cls.getAttribute === 'function') {
        cls = cls.getAttribute('class') || ''; // cannot use className for SVG el's

        isString = true;
      } else if ((_cls = cls) !== null && _cls !== void 0 && _cls.isDomClassList) {
        cls = cls.values;
      } else if (cls instanceof DOMTokenList) {
        cls = Array.from(cls);
      } else if (cls instanceof Map) {
        cls = Array.from(cls.keys()).filter(k => cls.get(k));
      } else if (cls instanceof Set) {
        cls = Array.from(cls);
      } else if (!Array.isArray(cls)) {
        cls = Objects.getTruthyKeys(cls);
      }
    }

    if (isString) {
      // Pass through Set to ensure only unique class names
      cls = [...new Set(StringHelper.split(cls))];
    } // cls is now an array

    for (i = cls.length; i-- > 0;) {
      c = cls[i];

      if (!c.length) {
        cls.splice(i, 1);
      } else if (c.includes(' ')) {
        cls.splice(i, 1, ...StringHelper.split(c));
      }
    }

    if (asArray) {
      ret = cls;
    } else if (asString) {
      ret = cls.join(' ');
    } else {
      ret = Object.create(null);

      for (i = 0; i < cls.length; ++i) {
        ret[cls[i]] = true;
      }
    }

    return ret;
  }
  /**
   * Initializes a new DomClassList.
   * @param {...String|Object} classes The CSS classes as strings or objects.
   * @function constructor
   */

  constructor(...classes) {
    this.process(1, classes);
  } // To gain some speed in DomHelper.sync(), faster than instanceof etc

  get isDomClassList() {
    return true;
  }
  /**
   * Returns a clone of this DomClassList with all the same keys set.
   * @returns {Core.helper.util.DomClassList} A clone of this DomClassList.
   */

  clone() {
    return new DomClassList(this);
  }
  /**
   * Returns a Boolean value, indicating whether this ClassList has the specified CSS class name.
   * @param {String} className CSS class name to check
   * @return {Boolean} true if this ClassList contains the passed CSS class name, false otherwise
   */

  contains(className) {
    if (typeof className === 'string' && className) {
      return Boolean(this[className]);
    }

    return false;
  } // An instance of this class may be assigned directly to an element's className
  // it will be coerced to a string value using this method.

  toString() {
    // Adding space at the end if there is content to make concatenation code simpler in renderers.
    return this.length ? `${this.value} ` : '';
  }

  toJSON() {
    return this.toString();
  }
  /**
   * Analogous to string.trim, returns the string value of this `DomClassList` with no trailing space.
   * @returns {String} A concatenated string value of all the class names in this `DomClassList`
   * separated by spaces.
   */

  trim() {
    return this.value;
  }
  /**
   * Compares this DomClassList to another DomClassList (or class name string of space separated classes).
   * If the same class names (regardless of order) are present, the two are considered equal.
   *
   * So `new DomClassList('foo bar bletch').isEqual('bletch bar foo')` would return `true`
   * @param {Core.helper.util.DomClassList|String} other The `DomClassList` or string of classes to compare to.
   * @returns {Boolean} `true` if the two contain the same class names.
   */

  isEqual(other) {
    const otherClasses = DomClassList.normalize(other, 'array'),
          len = otherClasses.length;

    if (this.length === len) {
      for (let i = 0; i < len; i++) {
        if (!this[otherClasses[i]]) {
          return false;
        }
      }

      return true;
    }

    return false;
  }
  /**
   * Get/set string value.
   * Class names separated with space.
   * @property {String}
   */

  get value() {
    let value = this[valueSymbol],
        keys;

    if (value == null) {
      keys = Objects.getTruthyKeys(this);
      this[lengthSymbol] = keys.length;
      this[valueSymbol] = value = keys.join(' ');
    }

    return value;
  }

  set value(value) {
    const me = this,
          keys = Object.keys(me),
          len = keys.length;

    for (let i = 0; i < len; i++) {
      delete me[keys[i]];
    }

    if (value) {
      me.process(1, [value]);
    } else {
      // String value needs recalculating
      delete me[valueSymbol];
    }
  }
  /**
   * Returns string values as an array.
   * @readonly
   * @property {String[]}
   */

  get values() {
    return Objects.getTruthyKeys(this);
  }

  get length() {
    // Maintainer: We MUST access the value getter to force
    // the value to be calculated if it's currently dirty.
    return this.value ? this[lengthSymbol] : 0;
  }

  process(value, classes) {
    for (let cls, k, i = 0; i < classes.length; i++) {
      if (classes[i]) {
        cls = classes[i];

        if (cls.isDomClassList || Objects.isObject(cls)) {
          // preserve all keys, even falsy ones
          for (k in cls) {
            this[k] = value ? cls[k] : !cls[k];
          }
        } else {
          cls = DomClassList.normalize(classes[i], 'array');

          for (k = 0; k < cls.length; ++k) {
            this[cls[k]] = value;
          }
        }
      }
    } // String value needs recalculating

    delete this[valueSymbol];
    return this;
  }
  /**
   * Adds/removes class names according to the passed object's properties.
   *
   * Properties with truthy values are added.
   * Properties with falsy values are removed.
   * @param {Object} classList Object containing properties to set/clear
   */

  assign(classList) {
    for (const cls in classList) {
      if (!this[cls] !== !classList[cls]) {
        this[cls] = classList[cls]; // String value needs recalculating

        delete this[valueSymbol];
      }
    }

    return this;
  }
  /**
   * Adds/removes this objects classes to the passed `classList` or element.
   *
   * Properties with truthy values are added.
   * Properties with falsy values are removed.
   * @param {HTMLElement|DOMTokenList} element The element or the element's `classList` to be updated.
   */

  assignTo(element) {
    const classList = element.nodeType === Element.ELEMENT_NODE ? element.classList : element,
          temp = DomClassList.from(classList);
    temp.add(this);
    classList.value = temp.value;
  }
  /**
   * Add CSS class(es)
   * ```
   * myClassList.add('bold', 'small');
   * ```javascript
   * @param {String|String[]|Object} classes CSS classes to add
   */

  add(...classes) {
    return this.process(1, classes);
  }
  /**
   * Remove CSS class(es)
   * ```javascript
   * myClassList.remove('bold', 'small');
   * ```
   * @param {String} classes CSS classes to remove
   */

  remove(...classes) {
    return this.process(0, classes);
  }
  /**
   * Toggles the passed CSS class name.
   *
   * If the `force` parameter is passed, `true` means add the class name, `false` means remove it.
   *
   * ```javascript
   * myClassList.toggle('bold', isImportant);
   * ```
   * @param {String} className CSS class to toggle
   * @param {Boolean} [force] `true` to add the class, `false` to remove it.
   * @returns {Boolean} `true` if the operation changed the value.
   */

  toggle(className, flag = Boolean(!this[className])) {
    flag = Boolean(flag); // Only disturb the set classwes if we need to.

    if (Boolean(this[className]) !== flag) {
      this[className] = flag; // String value needs recalculating

      delete this[valueSymbol];
      return true;
    }
  }
  /**
   * Analogous to the `String#split` method, but with no delimiter
   * parameter. This method returns an array containing the individual
   * CSS class names set.
   * @returns {String[]} The individual class names in this `DomClassList`
   */

  split() {
    return Objects.getTruthyKeys(this);
  }

  forEach(fn) {
    return Objects.getTruthyKeys(this).forEach(fn);
  }

}
// the instance to help the JIT

DomClassList.prototype[valueSymbol] = null;
DomClassList._$name = 'DomClassList';

//import './util/Point.js';
// TODO import VersionHelper from './VersionHelper.js';
// https://app.assembla.com/spaces/bryntum/tickets/7903-rendering-fails
// HACK: this value is required to calculate width if it was configured relative to font size (em) but no element is set

const DEFAULT_FONT_SIZE = 14,
      t0t0 = {
  align: 't0-t0'
},
      // eslint-disable-next-line no-undef
ELEMENT_NODE = Node.ELEMENT_NODE,
      // eslint-disable-next-line no-undef
TEXT_NODE = Node.TEXT_NODE,
      {
  isObject: isObject$1
} = ObjectHelper,
      // Transform matrix parse Regex. CSS transform computed style looks like this:
// matrix(scaleX(), skewY(), skewX(), scaleY(), translateX(), translateY())
// or
// matrix3d(scaleX(), skewY(), 0, 0, skewX(), scaleY(), 0, 0, 0, 0, 1, 0, translateX(), translateY())
// This is more reliable than using the style literal which may include
// relative styles such as "translateX(-20em)", or not include the translation at all if it's from a CSS rule.
// Use a const so as to only compile RexExp once
// Extract repeating number regexp to simplify next expressions. Available values are: https://developer.mozilla.org/en-US/docs/Web/CSS/number
numberRe = /[+-]?\d*\.?\d+[eE]?-?\d*/g,
      // -2.4492935982947064e-16 should be supported
numberReSrc = numberRe.source,
      translateMatrix2dRe = new RegExp(`matrix\\((?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(?:${numberReSrc}),\\s?(${numberReSrc}),\\s?(${numberReSrc})`),
      translateMatrix3dRe = new RegExp(`matrix3d\\((?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(?:-?\\d*),\\s?(-?\\d*),\\s?(-?\\d*)`),
      translateMatrixRe = new RegExp(`(?:${translateMatrix2dRe.source})|(?:${translateMatrix3dRe.source})`),
      pxTtranslateXRe = new RegExp(`translate(3d|X)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`),
      pxTtranslateYRe = new RegExp(`translate(3d|Y)?\\((${numberReSrc})px(?:,\\s?(${numberReSrc})px)?`),
      whiteSpaceRe = /\s+/,
      semicolonRe = /\s*;\s*/,
      colonRe = /\s*:\s*/,
      digitsRe$1 = /^-?((\d+(\.\d*)?)|(\.?\d+))$/,
      elementPropKey = '$bryntum',
      // A blank value means the expando name is the same as the key in this object, otherwise the key in this object is
// the name of the domConfig property and the value is the name of the DOM element expando property.
elementCreateExpandos = {
  elementData: '',
  for: 'htmlFor',
  retainElement: ''
},
      // DomHelper#createElement properties which require special processing.
// All other configs such as id and type are applied directly to the element.
elementCreateProperties = {
  // these two are handled by being in elementCreateExpands:
  // elementData  : 1,
  // for          : 1,
  tag: 1,
  html: 1,
  text: 1,
  children: 1,
  tooltip: 1,
  style: 1,
  dataset: 1,
  parent: 1,
  nextSibling: 1,
  ns: 1,
  reference: 1,
  class: 1,
  className: 1,
  unmatched: 1,
  // Used by syncId approach
  onlyChildren: 1,
  // Used by sync to not touch the target element itself,
  listeners: 1,
  compareHtml: 1,
  // Sync
  syncOptions: 1,
  // Sync
  keepChildren: 1 // Sync

},
      styleIgnoreProperties = {
  length: 1,
  parentRule: 1,
  style: 1
},
      nativeFocusableTags = {
  BUTTON: 1,
  IFRAME: 1,
  EMBED: 1,
  INPUT: 1,
  OBJECT: 1,
  SELECT: 1,
  TEXTAREA: 1,
  BODY: 1
},
      win = globalThis,
      doc = document,
      emptyObject$a = Object.freeze({}),
      arraySlice$1 = Array.prototype.slice,
      immediatePromise$8 = Promise.resolve(),
      fontProps = ['font-size', 'font-size-adjust', 'font-style', 'font-weight', 'font-family', 'font-kerning', 'font-stretch', 'line-height', 'text-transform', 'text-decoration', 'letter-spacing', 'word-break'],
      isHiddenWidget = e => e._hidden,
      parentNode = el => el.parentNode || el.host,
      mergeChildren = (dest, src, options) => {
  if (options.key === 'children') {
    // Normally "children" is an array (for which we won't be here, due to isObject check in caller). To
    // maintain declarative order of children as an object, we need some special sauce:
    return ObjectHelper.mergeItems(dest, src, options);
  }

  return ObjectHelper.blend(dest, src, options);
},
      isVisible = e => {
  const style = e.ownerDocument.defaultView.getComputedStyle(e);
  return style.getPropertyValue('display') !== 'none' && style.getPropertyValue('visibility') !== 'hidden';
},
      getRootNode = doc.documentElement.getRootNode ? el => el.getRootNode() : el => {
  while (el.parentNode) el = el.parentNode;

  return el;
},
      // Both Shadow Root Element and Anchor <a> Element have `host` property.
// https://developer.mozilla.org/en-US/docs/Web/API/ShadowRoot/host
// https://developer.mozilla.org/en-US/docs/Web/API/HTMLHyperlinkElementUtils/host
// Need make sure `host` is an Element.
isShadowRoot = el => el.host && el.host instanceof Element,
      // Check whether the element has an offsetParent.
// Nodes such as SVG which do not expose such a property must have an ancestor which has an offsetParent.
hasLayout = el => el && (el === doc.body || ('offsetParent' in el ? el.offsetParent : hasLayout(el.parentNode))),
      // Check for node being in document.
// If part of shadow DOM, see if the root's host is in the DOM
isInDocument = el => el && (doc.body.contains(el) || (root = getRootNode(el)) && isShadowRoot(root) && isInDocument(root.host)),
      elementOrConfigToElement = elementOrConfig => {
  if (elementOrConfig instanceof Node) {
    return elementOrConfig;
  }

  if (typeof elementOrConfig === 'string') {
    return DH$1.createElementFromTemplate(elementOrConfig);
  }

  return DH$1.createElement(elementOrConfig);
},
      canonicalStyles = Object.create(null),
      canonicalizeStyle = (name, hasUnit) => {
  const entry = canonicalStyles[name] || [StringHelper.hyphenate(name), hasUnit];

  if (!canonicalStyles[name]) {
    canonicalStyles[entry[0]] = canonicalStyles[name] = entry;
  }

  return entry;
},
      getOffsetParent = node => node.ownerSVGElement ? node.ownerSVGElement.parentNode : node.offsetParent;

['top', 'right', 'bottom', 'left', 'width', 'height', 'maxWidth', 'maxHeight', 'minWidth', 'minHeight', 'borderSpacing', 'borderWidth', 'borderTopWidth', 'borderRightWidth', 'borderBottomWidth', 'borderLeftWidth', 'marginTop', 'marginRight', 'marginBottom', 'marginLeft', 'paddingTop', 'paddingRight', 'paddingBottom', 'paddingLeft', 'fontSize', 'letterSpacing', 'lineHeight', 'outlineWidth', 'textIndent', 'wordSpacing'].forEach(name => canonicalizeStyle(name, true)); // We only do the measurement once, if the value is null

let scrollBarWidth = null,
    idCounter = 0,
    themeInfo = null,
    root,
    templateElement,
    htmlParser,
    scrollBarMeasureElement;
/**
 * @module Core/helper/DomHelper
 */

/**
 * An object that describes a DOM element. Used for example by {@link #function-createElement-static createElement()}
 * and by {@link Core.helper.DomSync#function-sync-static DomSync.sync()}.
 *
 * ```javascript
 * DomHelper.createElement({
 *    class : {
 *        big   : true,
 *        small : false
 *    },
 *    children : [
 *        { tag : 'img', src : 'img.png' },
 *        { html : '<b style="color: red">Red text</b>' }
 *    ]
 * });
 * ```
 *
 * @typedef {Object} DomConfig
 * @property {String} [tag='div'] Tag name, for example 'span'
 * @property {HTMLElement} [parent] Parent element
 * @property {HTMLElement} [nextSibling] Element's next sibling in the parent element
 * @property {String|Object} [class] CSS classes, as a string or an object (truthy keys will be applied)
 * @property {String|Object} [className] Alias for `class`
 * @property {String|Object} [style] Style, as a string or an object (keys will be hyphenated)
 * @property {Object} [elementData] Data object stored as an expando on the resulting element
 * @property {Object} [dataset] Dataset applied to the resulting element
 * @property {DomConfig[]|DomConfig|String} [children] Child elements, as an array of DomConfigs or an object map thereof or a simple string value to create a text node
 * @property {String} [html] Html string, used as the resulting elements `innerHTML`. Mutually exclusive with the `children` property
 * @property {Object} [tooltip] Tooltip config applied to the resulting element
 * @property {String} [text] Text content, XSS safe when you want to display text in the element. Mutually exclusive with the `children` property
 * @property {String} [id] Element's `id`
 * @property {String} [href] Element's `href`
 * @property {String} [ns] Element's namespace
 * @property {String} [src] Element's `src`
 */

/**
 * Helps with dom querying and manipulation.
 *
 * ```javascript
 * DomHelper.createElement({
 *   tag: 'div',
 *   className: 'parent',
 *   style: 'background: red',
 *   children: [
 *      { tag: 'div', className: 'child' },
 *      { tag: 'div', className: 'child' }
 *   ]
 * });
 * ```
 */

class DomHelper {
  /**
   * Returns `true` if the passed element is focusable either programmatically or through pointer gestures.
   * @param {HTMLElement} element The element to test.
   * @return {Boolean} Returns `true` if the passed element is focusable
   */
  static isFocusable(element, skipAccessibilityCheck = false) {
    if (!skipAccessibilityCheck) {
      // If element is hidden or in a hidden Widget, it's not focusable.
      if (!DH$1.isVisible(element) || DH$1.Widget.fromElement(element, isHiddenWidget)) {
        return false;
      }
    }

    const nodeName = element.nodeName;
    /*
     * An element is focusable if:
     *   - It is natively focusable, or
     *   - It is an anchor or link with href attribute, or
     *   - It has a tabIndex, or
     *   - It is an editing host (contenteditable="true")
     */

    return nativeFocusableTags[nodeName] || (nodeName === 'A' || nodeName === 'LINK') && !!element.href || element.getAttribute('tabIndex') != null || element.contentEditable === 'true';
  }
  /**
   * Returns the rectangle of the element which is currently visible in the browser viewport, i.e. user can find it on
   * screen, or `false` if it is scrolled out of view.
   * @param {HTMLElement} target The element to test.
   * @param {Boolean} [whole=false] Whether to check that whole element is visible, not just part of it.
   * If this is passed as true, the result will be a boolean, `true` or `false`.
   * @privateparam {Core.widget.Widget} [caller] the Widget aligning to the target.
   * @return {Core.helper.util.Rectangle|Boolean} Returns the rectangle of the element which is currently visible in
   * the browser viewport, or `false` if it is out of view.
   */

  static isInView(target, whole = false, caller) {
    // If the target cannot yield a Rectangle, shortcut all processing.
    if (!hasLayout(target)) {
      return false;
    }

    const positioned = (caller === null || caller === void 0 ? void 0 : caller.positioned) && DomHelper.getStyleValue(caller.element, 'position') !== 'fixed',
          docRect = Rectangle.from(globalThis),
          method = whole ? 'contains' : 'intersect',
          {
      parentNode
    } = target,
          {
      parentElement
    } = parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ? target.getRootNode().host : target,
          peStyle = parentElement.ownerDocument.defaultView.getComputedStyle(parentElement),
          offsetParent = getOffsetParent(target),
          cOp = positioned && caller.element.offsetParent,
          cOpR = positioned && Rectangle.from(cOp),
          parentScroll = peStyle.overflowX !== 'visible' || peStyle.overflowY !== 'visible'; // If we get to the top, the visible rectangle is the entire document.

    docRect.height = doc.scrollingElement.scrollHeight; // If they asked to test the body, it's always in view

    if (target === doc.body) {
      return docRect;
    }

    let result = Rectangle.from(target, null, true);

    for (let viewport = parentScroll ? target.parentNode : offsetParent; result && viewport !== doc.documentElement; viewport = viewport.parentNode) {
      // Skip shadow root node.
      if (viewport.nodeType === Node.DOCUMENT_FRAGMENT_NODE && viewport.host) {
        viewport = viewport.host.parentNode;
      } // Viewport might be null in Salesforce, when we traverse the tree up and reach document fragment node
      // https://github.com/bryntum/support/issues/4127

      if (!viewport) {
        break;
      }

      const isTop = viewport === doc.body,
            style = viewport.ownerDocument.defaultView.getComputedStyle(viewport),
            viewportRect = isTop ? docRect : Rectangle.inner(viewport, null, true); // If this level allows overflow to show, don't clip. Obv, <body> can't show overflowing els.

      if (isTop || style.overflow !== 'visible') {
        result = viewportRect[method](result, false, true);
      }
    } // We must use the *viewport* coordinate system to ascertain viewability

    if (result && positioned) {
      result.translate(doc.scrollingElement.scrollLeft, doc.scrollingElement.scrollTop);
    } // Return any rectangle to its positioned coordinate system

    return positioned && result ? result.translate(-cOpR.x + cOp.scrollLeft, -cOpR.y + cOp.scrollTop) : result;
  }
  /**
   * Returns `true` if the passed element is deeply visible. Meaning it is not hidden using `display`
   * or `visibility` and no ancestor node is hidden.
   * @param {HTMLElement} element The element to test.
   * @returns {Boolean} `true` if deeply visible.
   */

  static isVisible(element) {
    const document = element.ownerDocument; // Use the parentNode function so that we can traverse upwards through shadow DOM
    // to correctly ascertain visibility of nodes in web components.

    for (; element; element = parentNode(element)) {
      // Visible if we've reached top of the owning document without finding a hidden Element.
      if (element === document) {
        return true;
      } // Must not evaluate a shadow DOM's root fragment.

      if (element.nodeType === element.ELEMENT_NODE && !isVisible(element)) {
        return false;
      }
    } // We get here if the node is detached.

    return false;
  }
  /**
   * Returns true if DOM Event instance is passed. It is handy to override to support Locker Service.
   * @param event
   * @internal
   * @returns {Boolean}
   */

  static isDOMEvent(event) {
    return event instanceof Event;
  }
  /**
   * Merges specified source DOM config objects into a `dest` object.
   * @param {Object} dest The destination DOM config object.
   * @param {...Object} sources The DOM config objects to merge into `dest`.
   * @returns {Object} The `dest` object.
   * @internal
   */

  static merge(dest, ...sources) {
    return ObjectHelper.blend(dest, sources, {
      merge: mergeChildren
    });
  }
  /**
   * Updates in-place a DOM config object whose `children` property may be an object instead of the typical array.
   * The keys of such objects become the `reference` property upon conversion.
   *
   * @param {DomConfig} domConfig
   * @param {Function} [namedChildren] A function to call for each named child element.
   * @privateparam {Boolean} [ignoreRefs] Not meant to be manually set, used when recursing.
   * @returns {Object} Returns the altered DOM config
   * @internal
   */

  static normalizeChildren(domConfig, namedChildren, ignoreRefs) {
    var _domConfig$syncOption;

    let children = domConfig === null || domConfig === void 0 ? void 0 : domConfig.children,
        child,
        i,
        name,
        kids,
        ref; // Allow redirecting/opting out of ref ownership in a hierarchy

    if (domConfig !== null && domConfig !== void 0 && (_domConfig$syncOption = domConfig.syncOptions) !== null && _domConfig$syncOption !== void 0 && _domConfig$syncOption.ignoreRefs) {
      ignoreRefs = true;
    }

    if (children && !(domConfig instanceof Node)) {
      if (Array.isArray(children)) {
        for (i = 0; i < children.length; ++i) {
          DH$1.normalizeChildren(children[i], namedChildren, ignoreRefs);
        }
      } else {
        kids = children;
        domConfig.children = children = [];

        for (name in kids) {
          var _child;

          child = kids[name];

          if ((_child = child) !== null && _child !== void 0 && _child.isWidget) {
            child = child.element;
          } // $ prefix indicates element is not a reference:

          ref = !name.startsWith('$') && !DH$1.isElement(child);
          ref && (namedChildren === null || namedChildren === void 0 ? void 0 : namedChildren(name,
          /* hoist = */
          !ignoreRefs));

          if (child) {
            if (!(child instanceof Node)) {
              if (child.reference === false) {
                delete child.reference;
              } else if (ref && typeof child !== 'string') {
                child.reference = name;
              }

              DH$1.normalizeChildren(child, namedChildren, ignoreRefs);
            }

            children.push(child);
          }
        }
      }
    }

    return domConfig;
  }

  static roundPx(px, devicePixelRatio = globalThis.devicePixelRatio || 1) {
    const multiplier = 1 / devicePixelRatio;
    return Math.round(px / multiplier) * multiplier;
  }
  /**
   * Returns true if element has opened shadow root
   * @param {HTMLElement} element Element to check
   * @returns {Boolean}
   */

  static isCustomElement(element) {
    return Boolean(element === null || element === void 0 ? void 0 : element.shadowRoot);
  }
  /**
   * Resolves element from point, checking shadow DOM if required
   * @param {Number} x
   * @param {Number} y
   * @returns {HTMLElement}
   */

  static elementFromPoint(x, y) {
    let el = document.elementFromPoint(x, y); // Try to check shadow dom if it exists

    if (DH$1.isCustomElement(el)) {
      el = el.shadowRoot.elementFromPoint(x, y) || el;
    }

    return el;
  }
  /**
   * Resolves child element from point __in the passed element's coordinate space__.
   * @param {HTMLElement} parent The element to find the occupying element in.
   * @param {Number|Core.helper.util.Point} x Either the `X` part of a point, or the point to find.
   * @param {Number} [y] The `Y` part of the point.
   * @returns {HTMLElement}
   * @internal
   */

  static childFromPoint(el, x, y,
  /* internal */
  parent = el) {
    const p = y == null ? x : new Rectangle(x, y, 0, 0);
    let result = null;
    Array.from(el.children).some(el => {
      if (Rectangle.from(el, parent).contains(p)) {
        // All rectangles must be relative to the topmost el, so that must be
        // passed down as the "parent" of all Rectangles.
        result = el.children.length && DH$1.childFromPoint(el, p, null, parent) || el;
        return true;
      }
    });
    return result;
  }
  /**
   * Converts a name/value pair of a style name and its value into the canonical (hyphenated) name of the style
   * property and a value with the `defaultUnit` suffix appended if no unit is already present in the `value`.
   *
   * For example:
   * ```javascript
   *  const [property, value] = DomHelper.unitize('marginLeft', 50);
   *  console.log(property, value);
   * ```
   *
   * ```
   *  > margin-left 50px
   * ```
   * @param {String} name
   * @param {String|Number} value
   * @param {String} [defaultUnit]
   * @returns {String[]}
   * @internal
   */

  static unitize(name, value, defaultUnit = 'px') {
    const [trueName, hasUnits] = canonicalizeStyle(name);

    if (value != null) {
      value = String(value);
      value = hasUnits && digitsRe$1.test(value) ? value + defaultUnit : value;
    }

    return [trueName, value];
  }
  /**
   * Returns active element checking shadow dom too
   * @readonly
   * @property {HTMLElement}
   */

  static get activeElement() {
    let el = document.activeElement;

    while (el.shadowRoot) {
      el = el.shadowRoot.activeElement;
    }

    return el;
  } // returns active element for DOM tree / shadow DOM tree to which element belongs

  static getActiveElement(element) {
    var _element, _element2;

    if ((_element = element) !== null && _element !== void 0 && _element.isWidget) {
      element = element.element;
    } // If no element passed, fallback to document

    let el = (((_element2 = element) === null || _element2 === void 0 ? void 0 : _element2.getRootNode()) || document).activeElement;

    while ((_el = el) !== null && _el !== void 0 && _el.shadowRoot) {
      var _el;

      el = el.shadowRoot.activeElement;
    }

    return el;
  } // Returns the visible root (either document.body or a web component shadow root)

  static getRootElement(element) {
    var _element$getRootNode;

    const root = (_element$getRootNode = element.getRootNode) === null || _element$getRootNode === void 0 ? void 0 : _element$getRootNode.call(element);
    return (root === null || root === void 0 ? void 0 : root.body) || root || element.ownerDocument.body;
  } // Returns the topmost HTMLElement inside the current context (either document.body or a direct child of a web component shadow root)

  static getOutermostElement(element) {
    var _element$getRootNode2, _element3;

    const root = (_element$getRootNode2 = (_element3 = element).getRootNode) === null || _element$getRootNode2 === void 0 ? void 0 : _element$getRootNode2.call(_element3);

    if (root !== null && root !== void 0 && root.body) {
      return root === null || root === void 0 ? void 0 : root.body;
    } // we are in a shadow root

    while (element.parentNode !== root) {
      element = element.parentNode;
    }

    return element;
  }

  static isValidFloatRootParent(target) {
    return target === document.body || target.constructor.name === 'ShadowRoot';
  }
  /**
   * Returns the `id` of the passed element. Generates a unique `id` if the element does not have one.
   * @param {HTMLElement} element The element to return the `id` of.
   */

  static getId(element) {
    return element.id || (element.id = 'b-element-' + ++idCounter);
  }
  /**
   * Returns common widget/node ancestor for from/to arguments
   * @param {Core.widget.Widget|HTMLElement} from
   * @param {Core.widget.Widget|HTMLElement} to
   * @returns {Core.widget.Widget|HTMLElement}
   * @internal
   */

  static getCommonAncestor(from, to) {
    if (from === to) {
      return from;
    }

    while (from && !((_from = (_from2 = from)[from.isWidget ? 'owns' : 'contains']) !== null && _from !== void 0 && _from.call(_from2, to) || from === to)) {
      var _from, _from2;

      from = from.owner || from.parentNode;
    }

    return from;
  } //region Internal

  /**
   * Internal convenience fn to allow specifying either an element or a CSS selector to retrieve one
   * @private
   * @param {String|HTMLElement} elementOrSelector element or selector to lookup in DOM
   * @returns {HTMLElement}
   */

  static getElement(elementOrSelector) {
    // also used for SVG elements, so need to use more basic class, that is also returned by querySelector
    if (elementOrSelector instanceof Element) {
      return elementOrSelector;
    }

    return doc.querySelector(elementOrSelector);
  }
  /**
   * Sets attributes passed as object to given element
   * @param {String|Element} elementOrSelector
   * @param {Object} attributes
   * @internal
   */

  static setAttributes(elementOrSelector, attributes) {
    const element = DH$1.getElement(elementOrSelector);

    if (element && attributes) {
      for (const key in attributes) {
        if (attributes[key] == null) {
          element.removeAttribute(key);
        } else {
          element.setAttribute(key, attributes[key]);
        }
      }
    }
  }
  /**
   * Sets a CSS [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length) style value.
   * @param {String|HTMLElement} element The element to set the style in, or, if just the result is required,
   * the style magnitude to return with units added. If a nullish value is passed, an empty string
   * is returned.
   * @param {String} [style] The name of a style property which specifies a [length](https://developer.mozilla.org/en-US/docs/Web/CSS/length)
   * @param {Number|String} [value] The magnitude. If a number is used, the value will be set in `px` units.
   * @returns {String} The style value string.
   */

  static setLength(element, style, value) {
    if (arguments.length === 1) {
      var _element4;

      value = typeof element === 'number' ? `${element}px` : (_element4 = element) !== null && _element4 !== void 0 ? _element4 : '';
    } else {
      var _value;

      element = DH$1.getElement(element);
      value = element.style[style] = typeof value === 'number' ? `${value}px` : (_value = value) !== null && _value !== void 0 ? _value : '';
    }

    return value;
  }
  /**
   * Returns string percentified and rounded value for setting element's height, width etc.
   * @param {String|Number} value percent value
   * @param {Number} digits number of decimal digits for rounding
   * @return {string} percentified value or empty string if value can not be parsed
   * @internal
   */

  static percentify(value, digits = 2) {
    const mult = Math.pow(10, digits);
    return value == null || value === '' || isNaN(value) ? '' : `${Math.round(value * mult) / mult}%`;
  } //endregion
  //region Children, going down...

  /**
   * Gets the first direct child of `element` that matches `selector`.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {HTMLElement}
   * @category Query children
   */

  static getChild(element, selector) {
    return element.querySelector(':scope>' + selector);
  }
  /**
   * Checks if `element` has any child that matches `selector`.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {Boolean} true if any child matches selector
   * @category Query children
   */

  static hasChild(element, selector) {
    return DH$1.getChild(element, selector) != null;
  }
  /**
   * Returns all child elements (not necessarily direct children) that matches `selector`.
   *
   * If `selector` starts with `'>'` or `'# '`, then all components of the `selector` must match inside of `element`.
   * The scope selector, `:scope` is prepended to the selector (and if `#` was used, it is removed).
   *
   * These are equivalent:
   *
   *      DomHelper.children(el, '# .foo .bar');
   *
   *      el.querySelectorAll(':scope .foo .bar');
   *
   * These are also equivalent:
   *
   *      DomHelper.children(el, '> .foo .bar');
   *
   *      el.querySelectorAll(':scope > .foo .bar');
   *
   * @param {HTMLElement} element The parent element
   * @param {String} selector The CSS selector
   * @returns {HTMLElement[]} Matched elements, somewhere below `element`
   * @category Query children
   */

  static children(element, selector) {
    // a '#' could be '#id' but '# ' (hash and space) is not a valid selector...
    if (selector[0] === '>' || selector.startsWith('# ')) {
      if (selector[0] === '#') {
        selector = selector.substr(2);
      }

      selector = ':scope ' + selector;
    }

    return Array.from(element.querySelectorAll(selector));
  } // Salesforce doesn't yet support childElementCount. So we relace all native usages with this wrapper and
  // override it for salesforce environment.
  // https://github.com/bryntum/support/issues/3008

  static getChildElementCount(element) {
    return element.childElementCount;
  }
  /**
   * Looks at the specified `element` and all of its children for the one that first matches `selector.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @returns {HTMLElement} Matched element, either element or an element below it
   * @category Query children
   */

  static down(element, selector) {
    if (!element) {
      return null;
    }

    if (element.matches && element.matches(selector)) {
      return element;
    }

    selector = ':scope ' + selector;
    return element.querySelector(selector);
  }
  /**
   * Checks if childElement is a descendant of parentElement (contained in it or a sub element)
   * @param {HTMLElement} parentElement Parent element
   * @param {HTMLElement} childElement Child element, at any level below parent (includes nested shadow roots)
   * @returns {Boolean}
   * @category Query children
   */

  static isDescendant(parentElement, childElement) {
    const parentRoot = DH$1.getRootElement(parentElement),
          childRoot = DH$1.getRootElement(childElement);

    if (parentRoot !== childRoot && childRoot.host) {
      return DH$1.isDescendant(parentRoot, childRoot.host);
    }

    return parentElement.contains(childElement);
  }
  /**
   * Returns the specified element of the given `event`. If the `event` is an `Element`, it is returned. Otherwise,
   * the `eventName` argument is used to retrieve the desired element property from `event` (this defaults to the
   * `'target'` property).
   * @param {Event|Element} event
   * @param {String} [elementName]
   * @returns {Element}
   */

  static getEventElement(event, elementName = 'target') {
    return !event || DH$1.isElement(event) ? event : event[elementName];
  }
  /**
   * Returns `true` if the provided value is _likely_ a DOM element. If the element can be assured to be from the
   * same document, `instanceof Element` is more reliable.
   * @param {*} value
   * @returns {Boolean}
   */

  static isElement(value) {
    return (value === null || value === void 0 ? void 0 : value.nodeType) === document.ELEMENT_NODE && DH$1.isNode(value);
  }
  /**
   * Returns `true` if the provided value is _likely_ a DOM node. If the node can be assured to be from the same
   * document, `instanceof Node` is more reliable.
   * @param {*} value
   * @returns {Boolean}
   */

  static isNode(value) {
    // cannot use instanceof across frames. Using it here won't help since we'd need the same logic if it were
    // false... meaning we'd have the same chances of a false-positive.
    return Boolean(value) && typeof value.nodeType === 'number' && !isObject$1(value);
  }
  /**
   * Iterates over each result returned from `element.querySelectorAll(selector)`. First turns it into an array to
   * work in IE. Can also be called with only two arguments, in which case the first argument is used as selector and
   * document is used as the element.
   * @param {HTMLElement} element Parent element
   * @param {String} selector CSS selector
   * @param {Function} fn Function called for each found element
   * @category Query children
   */

  static forEachSelector(element, selector, fn) {
    if (typeof element === 'string') {
      // Legacy internal API, no longer valid
      throw new Error('DomHelper.forEachSelector must provide a root element context (for shadow root scenario)');
    }

    DH$1.children(element, selector).forEach(fn);
  }
  /**
   * Iterates over the direct child elements of the specified element. First turns it into an array to
   * work in IE.
   * @param {HTMLElement} element Parent element
   * @param {Function} fn Function called for each child element
   * @category Query children
   */

  static forEachChild(element, fn) {
    Array.from(element.children).forEach(fn);
  }
  /**
   * Removes each element returned from `element.querySelectorAll(selector)`.
   * @param {HTMLElement} element
   * @param {String} selector
   * @category Query children
   */

  static removeEachSelector(element, selector) {
    DH$1.forEachSelector(element, selector, child => child.remove());
  }

  static removeClsGlobally(element, ...classes) {
    classes.forEach(cls => DH$1.forEachSelector(element, '.' + cls, child => child.classList.remove(cls)));
  } //endregion
  //region Parents, going up...

  static isOrphaned(element) {
    return !isInDocument(element);
  }
  /**
   * Looks at the specified element and all of its parents for the one that first matches selector.
   * @param {HTMLElement} element Element
   * @param {String} selector CSS selector
   * @returns {HTMLElement} Matched element, either the passed in element or an element above it
   * @category Query parents
   */

  static up(element, selector) {
    return element && element.closest(selector);
  }

  static getAncestor(element, possibleAncestorParents, outerElement = null) {
    let found = false,
        ancestor,
        parent = element;
    possibleAncestorParents = ArrayHelper.asArray(possibleAncestorParents);

    while (parent = parent.parentElement) {
      if (possibleAncestorParents.includes(parent)) {
        found = true;
        break;
      }

      if (outerElement && parent === outerElement) break;
      ancestor = parent;
    }

    if (!found) return null;
    return ancestor || element;
  }
  /**
   * Retrieves all parents to the specified element.
   * @param {HTMLElement} element Element
   * @returns {HTMLElement[]} All parent elements, bottom up
   * @category Query parents
   */

  static getParents(element) {
    const parents = [];

    while (element.parentElement) {
      parents.push(element.parentElement);
      element = element.parentElement;
    }

    return parents;
  } //endregion
  //region Creation

  /**
   * Converts the passed id to an id valid for usage as id on a DOM element.
   * @param {String} id
   * @returns {String}
   */

  static makeValidId(id, replaceValue = '') {
    return StringHelper.makeValidDomId(id, replaceValue);
  }
  /**
   * Creates an Element, accepts a {@link #typedef-DomConfig} object. Example usage:
   *
   * ```javascript
   * DomHelper.createElement({
   *   tag         : 'table', // defaults to 'div'
   *   className   : 'nacho',
   *   html        : 'I am a nacho',
   *   children    : [ { tag: 'tr', ... }, myDomElement ],
   *   parent      : myExistingElement // Or its id
   *   style       : 'font-weight: bold;color: red',
   *   dataset     : { index: 0, size: 10 },
   *   tooltip     : 'Yay!',
   *   ns          : 'http://www.w3.org/1999/xhtml'
   * });
   * ```
   *
   * @param {DomConfig} config Element config object
   * @param {Object} [options] An object specifying creation options. If this is a boolean value, it is
   * understood to be the `returnAll` option.
   * @param {Boolean} [options.ignoreRefs] Pass `true` to ignore element references.
   * @param {Boolean} [options.returnAll] Specify true to return all elements & child elements
   * created as an array.
   * @returns {HTMLElement|HTMLElement[]|Object} Single element or array of elements `returnAll` was set to true.
   * If any elements had a `reference` property, this will be an object containing a reference to
   * all those elements, keyed by the reference name.
   * @category Creation
   */

  static createElement(config = {}, options) {
    var _options, _options$callback, _config$dataset;

    let returnAll = options,
        element,
        i,
        ignoreChildRefs,
        ignoreRefOption,
        ignoreRefs,
        key,
        name,
        value,
        refOwner,
        refs,
        syncIdField;

    if (typeof returnAll === 'boolean') {
      throw new Error('Clean up');
    } else if (options) {
      ignoreRefs = options.ignoreRefs;
      refOwner = options.refOwner;
      refs = options.refs;
      returnAll = options.returnAll;
      syncIdField = options.syncIdField;

      if (ignoreRefs) {
        ignoreChildRefs = true;
        ignoreRefs = ignoreRefs !== 'children';
      }
    }

    if (typeof config.parent === 'string') {
      config.parent = document.getElementById(config.parent);
    } // nextSibling implies a parent

    const parent = config.parent || config.nextSibling && config.nextSibling.parentNode,
          {
      dataset,
      html,
      reference,
      syncOptions,
      text
    } = config;

    if (syncOptions) {
      syncIdField = syncOptions.syncIdField || syncIdField;
      ignoreRefOption = syncOptions.ignoreRefs;

      if (ignoreRefOption) {
        ignoreChildRefs = true;
        ignoreRefs = ignoreRefOption !== 'children';
        options = _objectSpread2(_objectSpread2({}, options), {}, {
          ignoreRefs: true
        });
      }
    }

    if (ignoreRefs) {
      refOwner = null;
    }

    if (config.ns) {
      element = doc.createElementNS(config.ns, config.tag || 'svg');
    } else {
      element = doc.createElement(config.tag || 'div');
    }

    if (text != null) {
      DH$1.setInnerText(element, text);
    } else if (html != null) {
      if (html instanceof DocumentFragment) {
        element.appendChild(html);
      } else {
        element.innerHTML = html;
      }
    }

    if (config.tooltip) {
      DH$1.Widget.attachTooltip(element, config.tooltip);
    }

    if (config.style) {
      DH$1.applyStyle(element, config.style);
    }

    if (dataset) {
      for (name in dataset) {
        value = dataset[name];

        if (value != null) {
          element.dataset[name] = value;
        }
      }
    }

    if (parent) {
      parent.insertBefore(element, config.nextSibling);
    }

    if (refOwner) {
      // Tag each element created by the refOwner's id to enable DomSync
      element.$refOwnerId = refOwner.id;
    }

    if (reference && !ignoreRefs) {
      // SalesForce platform does not allow custom attributes, but existing code
      // uses querySelector('[reference]'), so bypass it when we can:
      if (refOwner) {
        element.$reference = reference;
        refOwner.attachRef(reference, element, config);
      } else {
        // TODO fixup callers to do the above
        if (!refs) {
          options = Object.assign({}, options);
          options.refs = refs = {};
        }

        refs[reference] = element;
        element.setAttribute('data-reference', reference);
      }
    }

    const className = config.className || config.class,
          // matches DomSync
    keys = Object.keys(config);

    if (className) {
      element.setAttribute('class', DomClassList.normalize(className));
    }

    for (i = 0; i < keys.length; ++i) {
      name = keys[i];
      value = config[name]; // We have to use setAttribute() for custom attributes to work and this is inline with how DomSync
      // handles attributes. For "expando" properties, however, we have to simply assign them.

      if ((key = elementCreateExpandos[name]) != null) {
        element[key || name] = value;
      } else if (!elementCreateProperties[name] && name && value != null) {
        // if (config.ns) {
        //     element.setAttributeNS(config.ns, name, value);
        // }
        // else {
        //     element.setAttribute(name, value);
        // }
        element.setAttribute(name, value);
      }
    } // ARIA. In the absence of a defined role or the elememnt being hidden from ARIA,
    // omit unfocusable elements from the accessibility tree.

    if (!config['aria-hidden'] && !config.role && !DomHelper.isFocusable(element, true) && !element.htmlFor) {
      element.setAttribute('role', 'presentation');
    } // Mimic the way DomSync issues callbacks as elements are created (needed by TaskBoard to trigger custom
    // taskRenderer calls as elements get produced).

    (_options = options) === null || _options === void 0 ? void 0 : (_options$callback = _options.callback) === null || _options$callback === void 0 ? void 0 : _options$callback.call(_options, {
      action: 'newElement',
      domConfig: config,
      targetElement: element,
      syncId: refOwner ? reference : options.syncIdField && ((_config$dataset = config.dataset) === null || _config$dataset === void 0 ? void 0 : _config$dataset[options.syncIdField])
    }); // if returnAll is true, use array

    if (returnAll === true) {
      options.returnAll = returnAll = [element];
    } // if it already is an array, add to it (we are probably a child)
    else if (Array.isArray(returnAll)) {
      returnAll.push(element);
    }

    if (config.children) {
      if (syncIdField) {
        // Map syncId -> child element to avoid querying dom later on
        element.syncIdMap = {};
      }

      config.children.forEach(child => {
        // Skip null children, convenient to allow those for usage with Array.map()
        if (child) {
          // Append string children as text nodes
          if (typeof child === 'string') {
            const textNode = document.createTextNode(child);

            if (refOwner) {
              textNode.$refOwnerId = refOwner.id;
            }

            element.appendChild(textNode);
          } // Just append Elements directly.
          else if (isNaN(child.nodeType)) {
            var _config$syncOptions$i, _config$syncOptions, _child$dataset;

            child.parent = element;

            if (!child.ns && config.ns) {
              child.ns = config.ns;
            }

            const childElement = DH$1.createElement(child, _objectSpread2(_objectSpread2({}, options), {}, {
              ignoreRefs: (_config$syncOptions$i = (_config$syncOptions = config.syncOptions) === null || _config$syncOptions === void 0 ? void 0 : _config$syncOptions.ignoreRef) !== null && _config$syncOptions$i !== void 0 ? _config$syncOptions$i : ignoreChildRefs
            })),
                  syncId = (_child$dataset = child.dataset) === null || _child$dataset === void 0 ? void 0 : _child$dataset[syncIdField]; // syncId is used with DomHelper.sync to match elements. Populate a map here to make finding them faster

            if (syncId != null) {
              element.syncIdMap[syncId] = childElement;
            } // Do not want to alter the initial config

            delete child.parent;
          } else {
            element.appendChild(child);
          }
        }
      });
    } // Store used config, to be able to compare on sync to determine if changed without hitting dom

    element.lastDomConfig = config; // If references were used, return them in an object
    // If returnAll was specified, return the array
    // By default, return the root element

    return refs || returnAll || element;
  }
  /**
   * Create element(s) from a template (html string). Note that
   * `textNode`s are discarded unless the `raw` option is passed
   * as `true`.
   *
   * If the template has a single root element, then the single element will be returned
   * unless the `array` option is passed as `true`.
   *
   * If there are multiple elements, then an Array will be returned.
   *
   * @param {String} template The HTML string from which to create DOM content
   * @param {Object} [options] An object containing properties to modify how the DOM is created and returned.
   * @param {Boolean} [options.array] `true` to return an array even if there's only one resulting element.
   * @param {Boolean} [options.raw] Return all child nodes, including text nodes.
   * @param {Boolean} [options.fragment] Return a DocumentFragment.
   * @private
   */

  static createElementFromTemplate(template, options = emptyObject$a) {
    const {
      array,
      raw,
      fragment
    } = options;
    let result; // Use template by preference if it exists. It's faster on most supported platforms
    // https://jsperf.com/domparser-vs-template/

    if (DH$1.supportsTemplate) {
      (templateElement || (templateElement = doc.createElement('template'))).innerHTML = template;
      result = templateElement.content;

      if (fragment) {
        // The template is reused, so therefore is its fragment.
        // If we release the fragment to a caller, it must be a clone.
        return result.cloneNode(true);
      }
    } else {
      result = (htmlParser || (htmlParser = new DOMParser())).parseFromString(template, 'text/html').body; // We must return a DocumentFragment.
      // myElement.append(fragment) inserts the contents of the fragment, not the fragment itself.

      if (fragment) {
        const nodes = result.childNodes;
        result = document.createDocumentFragment();

        while (nodes.length) {
          result.appendChild(nodes[0]);
        }

        return result;
      }
    } // Raw means all child nodes are returned

    if (raw) {
      result = result.childNodes;
    } // Otherwise, only element nodes
    else {
      result = result.children;
    }

    return result.length === 1 && !array ? result[0] : arraySlice$1.call(result);
  }
  /**
   * Inserts an `element` at first position in `into`.
   * @param {HTMLElement} into Parent element
   * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
   * @returns {HTMLElement}
   * @category Creation
   */

  static insertFirst(into, element) {
    if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
      element = DH$1.createElement(element);
    }

    return into.insertBefore(element, into.firstElementChild);
  }
  /**
   * Inserts a `element` before `beforeElement` in `into`.
   * @param {HTMLElement} into Parent element
   * @param {HTMLElement} element Element to insert, or an element config passed on to createElement()
   * @param {HTMLElement} beforeElement Element before which passed element should be inserted
   * @returns {HTMLElement}
   * @category Creation
   */

  static insertBefore(into, element, beforeElement) {
    if (element && element.nodeType !== ELEMENT_NODE && element.tag) {
      element = DH$1.createElement(element);
    }

    return beforeElement ? into.insertBefore(element, beforeElement) : DH$1.insertFirst(into, element);
  }

  static insertAt(parentElement, newElement, index) {
    const siblings = Array.from(parentElement.children);

    if (index >= siblings.length) {
      return DH$1.append(parentElement, newElement);
    }

    const beforeElement = siblings[index];
    return DH$1.insertBefore(parentElement, newElement, beforeElement);
  }
  /**
   * Appends element to parentElement.
   * @param {HTMLElement} parentElement Parent element
   * @param {HTMLElement|Object|String} elementOrConfig Element to insert, or an element config passed on to createElement(), or an html string passed to createElementFromTemplate
   * @returns {HTMLElement}
   * @category Creation
   */

  static append(parentElement, elementOrConfig) {
    if (elementOrConfig.forEach) {
      // Ensure all elements of an Array are HTMLElements.
      // The other implementor of forEach is a NodeList which needs no conversion.
      if (Array.isArray(elementOrConfig)) {
        elementOrConfig = elementOrConfig.map(elementOrConfig => elementOrConfigToElement(elementOrConfig));
      }

      if (parentElement.append) {
        parentElement.append(...elementOrConfig);
      } else {
        const docFrag = document.createDocumentFragment();
        elementOrConfig.forEach(function (child) {
          docFrag.appendChild(child);
        });
        parentElement.appendChild(docFrag);
      }

      return elementOrConfig;
    } else {
      return parentElement.appendChild(elementOrConfigToElement(elementOrConfig));
    }
  } //endregion
  //region Get position

  /**
   * Returns the element's `transform translateX` value in pixels.
   * @param {HTMLElement} element
   * @returns {Number} X transform
   * @category Position, get
   */

  static getTranslateX(element) {
    const transformStyle = element.style.transform;
    let matches = pxTtranslateXRe.exec(transformStyle); // Use inline transform style if it contains "translate(npx, npx" or "translate3d(npx, npx" or "translateX(npx"

    if (matches) {
      return parseFloat(matches[2]);
    } else {
      // If the inline style is the matrix() form, then use that, otherwise, use computedStyle
      matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH$1.getStyleValue(element, 'transform'));
      return matches ? parseFloat(matches[1] || matches[3]) : 0;
    }
  }
  /**
   * Returns the element's `transform translateY` value in pixels.
   * @param {HTMLElement} element
   * @returns {Number} Y coordinate
   * @category Position, get
   */

  static getTranslateY(element) {
    const transformStyle = element.style.transform;
    let matches = pxTtranslateYRe.exec(transformStyle); // Use inline transform style if it contains "translate(npx, npx" or "translate3d(npx, npx" or "translateY(npx"

    if (matches) {
      // If it was translateY(npx), use first item in the parens.
      const y = parseFloat(matches[matches[1] === 'Y' ? 2 : 3]); // FF will strip `translate(x, 0)` -> `translate(x)`, so need to check for isNaN also

      return isNaN(y) ? 0 : y;
    } else {
      // If the inline style is the matrix() form, then use that, otherwise, use computedStyle
      matches = translateMatrixRe.exec(transformStyle) || translateMatrixRe.exec(DH$1.getStyleValue(element, 'transform'));
      return matches ? parseFloat(matches[2] || matches[4]) : 0;
    }
  }
  /**
   * Gets both X and Y coordinates as an array [x, y]
   * @param {HTMLElement} element
   * @returns {Number[]} [x, y]
   * @category Position, get
   */

  static getTranslateXY(element) {
    return [DH$1.getTranslateX(element), DH$1.getTranslateY(element)];
  }
  /**
   * Get elements X offset within a containing element
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number} X offset
   * @category Position, get
   */

  static getOffsetX(element, container = null) {
    return container ? element.getBoundingClientRect().left - container.getBoundingClientRect().left : element.offsetLeft;
  }
  /**
   * Get elements Y offset within a containing element
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number} Y offset
   * @category Position, get
   */

  static getOffsetY(element, container = null) {
    return container ? element.getBoundingClientRect().top - container.getBoundingClientRect().top : element.offsetTop;
  }
  /**
   * Gets elements X and Y offset within containing element as an array [x, y]
   * @param {HTMLElement} element
   * @param {HTMLElement} container
   * @returns {Number[]} [x, y]
   * @category Position, get
   */

  static getOffsetXY(element, container = null) {
    return [DH$1.getOffsetX(element, container), DH$1.getOffsetY(element, container)];
  }
  /**
   * Focus element without scrolling the element into view.
   * @param {HTMLElement} element
   */

  static focusWithoutScrolling(element) {
    function resetScroll(scrollHierarchy) {
      scrollHierarchy.forEach(({
        element,
        scrollLeft,
        scrollTop
      }) => {
        // Check first to avoid triggering unnecessary `scroll` events
        if (element.scrollLeft !== scrollLeft) {
          element.scrollLeft = scrollLeft;
        }

        if (element.scrollTop !== scrollTop) {
          element.scrollTop = scrollTop;
        }
      });
    } // Check browsers which do support focusOptions. Currently only Safari lags.
    // https://caniuse.com/mdn-api_htmlelement_focus_preventscroll_option

    const preventScrollSupported = !BrowserHelper.isSafari;

    if (preventScrollSupported) {
      element.focus({
        preventScroll: true
      });
    } else {
      // Examine every parentNode of the target and cache the scrollLeft and scrollTop,
      // and restore all values after the focus has taken place
      const parents = DH$1.getParents(element),
            scrollHierarchy = parents.map(parent => ({
        element: parent,
        scrollLeft: parent.scrollLeft,
        scrollTop: parent.scrollTop
      }));
      element.focus(); // Reset in async.

      setTimeout(() => resetScroll(scrollHierarchy), 0);
    }
  }
  /**
   * Get elements X position on page
   * @param {HTMLElement} element
   * @returns {Number}
   * @category Position, get
   */

  static getPageX(element) {
    return element.getBoundingClientRect().left + win.pageXOffset;
  }
  /**
   * Get elements Y position on page
   * @param {HTMLElement} element
   * @returns {Number}
   * @category Position, get
   */

  static getPageY(element) {
    return element.getBoundingClientRect().top + win.pageYOffset;
  }
  /**
   * Returns extremal (min/max) size (height/width) of the element in pixels
   * @param {HTMLElement} element
   * @param {String} style minWidth/minHeight/maxWidth/maxHeight
   * @returns {Number}
   * @internal
   */

  static getExtremalSizePX(element, style) {
    const prop = StringHelper.hyphenate(style),
          measure = prop.split('-')[1];
    let value = DH$1.getStyleValue(element, prop);

    if (/%/.test(value)) {
      // Element might be detached from DOM
      if (element.parentElement) {
        value = parseInt(DH$1.getStyleValue(element.parentElement, measure), 10);
      } else {
        value = NaN;
      }
    } else {
      value = parseInt(value, 10);
    }

    return value;
  } //endregion
  //region Set position

  /**
   * Set element's `scale`.
   * @param {HTMLElement} element
   * @param {Number} scaleX The value by which the element should be scaled in the X axis (0 to 1)
   * @param {Number} [scaleY] The value by which the element should be scaled in the Y axis (0 to 1).
   * Defaults to `scaleX`
   * @category Position, set
   * @internal
   */

  static setScale(element, scaleX, scaleY = scaleX) {
    const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/);

    if (t.length > 1) {
      if (t[0].startsWith('matrix3d')) {
        t[0] = `matrix3d(${scaleX}`;
        t[5] = scaleY;
      } else {
        t[0] = `matrix(${scaleX}`;
        t[3] = scaleY;
      }

      element.style.transform = t.join(',');
    } else {
      element.style.transform = `scale(${scaleX}, ${scaleY})`;
    }
  }
  /**
   * Set element's `X` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} x The value by which the element should be translated from its default position.
   * @category Position, set
   */

  static setTranslateX(element, x) {
    const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/); // Avoid blurry text on non-retina displays

    x = DH$1.roundPx(x);

    if (t.length > 1) {
      t[t[0].startsWith('matrix3d') ? 12 : 4] = x;
      element.style.transform = t.join(',');
    } else {
      element.style.transform = `translateX(${x}px)`;
    }
  }
  /**
   * Set element's `Y` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} y  The value by which the element should be translated from its default position.
   * @category Position, set
   */

  static setTranslateY(element, y) {
    const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/); // Avoid blurry text on non-retina displays

    y = DH$1.roundPx(y);

    if (t.length > 1) {
      t[t[0].startsWith('matrix3d') ? 13 : 5] = y;
      element.style.transform = t.join(',') + ')';
    } else {
      element.style.transform = `translateY(${y}px)`;
    }
  }
  /**
   * Set element's style `top`.
   * @param {HTMLElement} element
   * @param {Number|String} y The top position. If numeric, `'px'` is used as the unit.
   * @category Position, set
   */

  static setTop(element, y) {
    DH$1.setLength(element, 'top', y);
  }
  /**
   * Set element's style `left`.
   * @param {HTMLElement} element
   * @param {Number|String} x The top position. If numeric, `'px'` is used as the unit.
   * @category Position, set
   */

  static setLeft(element, x) {
    DH$1.setLength(element, 'left', x);
  }

  static setTopLeft(element, y, x) {
    DH$1.setLength(element, 'top', y);
    DH$1.setLength(element, 'left', x);
  }
  /**
   * Set elements `X` and `Y` translation in pixels.
   * @param {HTMLElement} element
   * @param {Number} [x] The `X translation.
   * @param {Number} [y] The `Y translation.
   * @category Position, set
   */

  static setTranslateXY(element, x, y) {
    if (x == null) {
      return DH$1.setTranslateY(element, y);
    }

    if (y == null) {
      return DH$1.setTranslateX(element, x);
    } // Avoid blurry text on non-retina displays

    x = DH$1.roundPx(x);
    y = DH$1.roundPx(y);
    const t = DH$1.getStyleValue(element, 'transform').split(/,\s*/),
          is3d = t[0].startsWith('matrix3d');

    if (t.length > 1) {
      t[is3d ? 12 : 4] = x;
      t[is3d ? 13 : 5] = y;
      element.style.transform = t.join(',') + ')';
    } else {
      element.style.transform = `translate(${x}px, ${y}px)`;
    }
  }
  /**
   * Increase `X` translation
   * @param {HTMLElement} element
   * @param {Number} x The number of pixels by which to increase the element's `X` translation.
   * @category Position, set
   */

  static addTranslateX(element, x) {
    DH$1.setTranslateX(element, DH$1.getTranslateX(element) + x);
  }
  /**
   * Increase `Y` position
   * @param {HTMLElement} element
   * @param {Number} y The number of pixels by which to increase the element's `Y` translation.
   * @category Position, set
   */

  static addTranslateY(element, y) {
    DH$1.setTranslateY(element, DH$1.getTranslateY(element) + y);
  }
  /**
   * Increase X position
   * @param {HTMLElement} element
   * @param {Number} x
   * @category Position, set
   */

  static addLeft(element, x) {
    DH$1.setLeft(element, DH$1.getOffsetX(element) + x);
  }
  /**
   * Increase Y position
   * @param {HTMLElement} element
   * @param {Number} y
   * @category Position, set
   */

  static addTop(element, y) {
    DH$1.setTop(element, DH$1.getOffsetY(element) + y);
  }
  /**
   * Align the passed element with the passed target according to the align spec.
   * @param {HTMLElement} element The element to align.
   * @param {HTMLElement|Core.helper.util.Rectangle} target The target element or rectangle to align to
   * @param {Object} [alignSpec] See {@link Core.helper.util.Rectangle#function-alignTo} Defaults to `{ align : 't0-t0' }`
   * @param {Boolean} [round] Round the calculated Rectangles (for example if dealing with scrolling which
   * is integer based).
   */

  static alignTo(element, target, alignSpec = t0t0, round) {
    target = target instanceof Rectangle ? target : Rectangle.from(target, true);
    const elXY = DH$1.getTranslateXY(element),
          elRect = Rectangle.from(element, true);

    if (round) {
      elRect.roundPx();
      target.roundPx();
    }

    const targetRect = elRect.alignTo(Object.assign(alignSpec, {
      target
    }));
    DH$1.setTranslateXY(element, elXY[0] + targetRect.x - elRect.x, elXY[1] + targetRect.y - elRect.y);
  } //endregion
  //region Styles & CSS

  /**
   * Returns a style value or values for the passed element.
   * @param {HTMLElement} element The element to read styles from
   * @param {String|String[]} propName The property or properties to read
   * @param {Boolean} [inline=false] Pass as `true` to read the element's inline style.
   * Note that this could return inaccurate results if CSS rules apply to this element.
   * @return {String|Object} The value or an object containing the values keyed by the requested property name.
   * @category CSS
   */

  static getStyleValue(element, propName, inline, pseudo) {
    const styles = inline ? element.style : element.ownerDocument.defaultView.getComputedStyle(element, pseudo);

    if (Array.isArray(propName)) {
      const result = {};

      for (const prop of propName) {
        result[prop] = styles.getPropertyValue(StringHelper.hyphenate(prop));
      }

      return result;
    } // Use the elements owning view to get the computed style.
    // Ensure the property name asked for is hyphenated.
    // getPropertyValue doesn't work with camelCase

    return styles.getPropertyValue(StringHelper.hyphenate(propName));
  }
  /**
   * Returns an object with the parse style values for the top, right, bottom, and left
   * components of the given edge style.
   *
   * The return value is an object with `top`, `right`, `bottom`, and `left` properties
   * for the respective components of the edge style, as well as `width` (the sum of
   * `left` and `right`) and `height` (the sum of `top` and `bottom`).
   *
   * @param {HTMLElement} element
   * @param {String} edgeStyle The element's desired edge style such as 'padding', 'margin',
   * or 'border'.
   * @param {String} [edges='trbl'] A string with one character codes for each edge. Only
   * those edges will be populated in the returned object. By default, all edges will be
   * populated.
   * @returns {Object}
   */

  static getEdgeSize(element, edgeStyle, edges) {
    const suffix = edgeStyle === 'border' ? '-width' : '',
          ret = {
      raw: {}
    };

    for (const edge of ['top', 'right', 'bottom', 'left']) {
      if (!edges || edges.includes(edge[0])) {
        // This produces px units even if the provided style is em or other (i.e.,
        // getComputedStyle normalizes this):
        ret[edge] = parseFloat(ret.raw[edge] = DH$1.getStyleValue(element, `${edgeStyle}-${edge}${suffix}`));
      }
    } // These may not even be requested (based on "edges") but conditional code here
    // would be wasted since the caller would still need to know not to use them...
    // Replace NaN with 0 to keep calculations correct if they only asked for one side.

    ret.width = (ret.left || 0) + (ret.right || 0);
    ret.height = (ret.top || 0) + (ret.bottom || 0);
    return ret;
  }
  /**
   * Splits a style string up into object form. For example `'font-weight:bold;font-size:150%'`
   * would convert to
   *
   * ```javascript
   * {
   *     font-weight : 'bold',
   *     font-size : '150%'
   * }
   * ```
   * @param {String} style A DOM style string
   * @returns {Object} the style declaration in object form.
   */

  static parseStyle(style) {
    if (typeof style === 'string') {
      const styles = style.split(semicolonRe);
      style = {};

      for (let i = 0, {
        length
      } = styles; i < length; i++) {
        const propVal = styles[i].split(colonRe);
        style[propVal[0]] = propVal[1];
      }
    }

    return style || {};
  }
  /**
   * Applies specified style to the passed element. Style can be an object or a string.
   * @param {HTMLElement} element Target element
   * @param {String|Object} style Style to apply, 'border: 1px solid black' or { border: '1px solid black' }
   * @param {Boolean} [overwrite] Specify `true` to replace style instead of applying changes
   * @category CSS
   */

  static applyStyle(element, style, overwrite = false) {
    if (typeof style === 'string') {
      if (overwrite) {
        // Only assign if either end has any styles, do not want to add empty `style` tag on element
        if (style.length || element.style.cssText.length) {
          element.style.cssText = style;
        }
      } else {
        // Add style so as not to delete configs in style such as width, height, flex etc.
        // If a style is already there, the newest, appended one will take precedence.
        element.style.cssText += style;
      }
    } else if (style) {
      if (overwrite) {
        element.style.cssText = ''; //element.removeAttribute('style');
      } // Has a sub-style block?

      if (style.style) {
        if (typeof style.style === 'string') {
          element.style.cssText = style.style;
        } else {
          style = Object.assign({}, style.style, style);
        }
      }

      let key, value; // Prototype chained objects may be passed, so use direct loop.

      for (key in style) {
        // Ignore readonly properties of the CSSStyleDeclaration object:
        // https://developer.mozilla.org/en-US/docs/Web/API/CSSStyleDeclaration
        // Also ignores sub-style blocks, which are applied above
        if (!styleIgnoreProperties[key]) {
          [key, value] = DH$1.unitize(key, style[key]); // Cannot use element.style[key], wont work with CSS vars

          element.style.setProperty(key, value);
        }
      }
    }
  }

  static getCSSText(style) {
    if (typeof style === 'string') {
      return style;
    }

    let cssText = '';

    for (const key in style) {
      // Ignore readonly properties of the CSSStyleDeclaration object:
      // https://developer.mozilla.org/en-US/docs/Web/API/CSSStyleDeclaration
      if (!styleIgnoreProperties[key]) {
        cssText += `${StringHelper.hyphenate(key)}:${style[key]};`;
      }
    }

    return cssText;
  }
  /**
   * Add multiple classes to elements classList.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @deprecated 5.0 Use {@link Element} classList.add method
   * @category CSS
   */

  static addClasses(element, classes) {
    VersionHelper.deprecate('Core', '6.0.0', 'DomHelper.addClasses should be replaced by native classList.add');
    element.classList.add(...classes);
  }
  /**
   * Remove multiple classes from elements classList.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @deprecated 5.0 Use {@link Element} classList.remove method
   * @category CSS
   */

  static removeClasses(element, classes) {
    VersionHelper.deprecate('Core', '6.0.0', 'DomHelper.removeClasses should be replaced by native classList.remove');
    element.classList.remove(...classes);
  }
  /**
   * Toggle multiple classes in elements classList. Helper for toggling multiple classes at once.
   * @param {HTMLElement} element
   * @param {String[]} classes
   * @param {Boolean} [force] Specify true to add classes, false to remove. Leave blank to toggle
   * @category CSS
   */

  static toggleClasses(element, classes, force = null) {
    classes = ArrayHelper.asArray(classes);

    if (force === true) {
      element.classList.add(...classes);
    } else if (force === false) {
      element.classList.remove(...classes);
    } else {
      classes.forEach(cls => element.classList.toggle(cls));
    }
  }
  /**
   * Adds a CSS class to an element during the specified duration
   * @param {HTMLElement} element Target element
   * @param {String} cls CSS class to add temporarily
   * @param {Number} duration Duration in ms, 0 means cls will not be applied
   * @param {Core.mixin.Delayable} delayable The delayable to tie the setTimeout call to
   * @typings delayable -> {typeof Delayable}
   * @category CSS
   */

  static addTemporaryClass(element, cls, duration, delayable = globalThis) {
    if (duration > 0) {
      element.classList.add(cls);
      delayable.setTimeout(() => element.classList.remove(cls), duration);
    }
  }
  /**
   * Reads computed style from the element and returns transition duration for a given property in milliseconds
   * @param {HTMLElement} element Target DOM element
   * @param {String} property Animated property name
   * @returns {Number} Duration in ms
   * @internal
   */

  static getPropertyTransitionDuration(element, property) {
    const style = globalThis.getComputedStyle(element),
          properties = style.transitionProperty.split(', '),
          durations = style.transitionDuration.split(', '),
          index = properties.indexOf(StringHelper.hyphenate(property));
    let result;

    if (index !== -1) {
      // get floating value of transition duration in seconds and convert into milliseconds
      result = parseFloat(durations[index]) * 1000;
    }

    return result;
  }
  /**
   * Reads computed style from the element and returns the animation duration for any
   * attached animation in milliseconds
   * @param {HTMLElement} element Target DOM element
   * @returns {Number} Duration in ms
   * @internal
   */

  static getAnimationDuration(element) {
    return parseFloat(DH$1.getStyleValue(element, 'animation-duration')) * 1000;
  } //endregion
  //region Effects

  /**
   * Highlights the passed element or Rectangle according to the theme's highlighting rules.
   * Usually an animated framing effect.
   *
   * The framing effect is achieved by adding the CSS class `b-fx-highlight` which references
   * a `keyframes` animation named `b-fx-highlight-animation`. You may override the animation
   * name referenced, or the animation itself in your own CSS.
   *
   * @param {HTMLElement|Core.helper.util.Rectangle} element The element or Rectangle to highlight.
   */

  static highlight(element, delayable = globalThis) {
    if (element instanceof Rectangle) {
      return element.highlight();
    }

    return new Promise(resolve => {
      delayable.setTimeout(() => {
        element.classList.add('b-fx-highlight');
        delayable.setTimeout(() => {
          element.classList.remove('b-fx-highlight');
          resolve();
        }, 1000);
      }, 0);
    });
  } //endregion
  //region Measuring / Scrollbar

  /**
   * Measures the scrollbar width using a hidden div. Caches result
   * @property {Number}
   * @readonly
   */

  static get scrollBarWidth() {
    // Ensure the measurement is only done once, when the value is null and body is available.
    // Leave measure element in place. It needs to be remeasured when the zoom level is changed
    // which is detected using a window resize listener, so *may* be called frequently.
    if (scrollBarWidth === null && doc.body) {
      const element = scrollBarMeasureElement || (scrollBarMeasureElement = DH$1.createElement({
        parent: doc.body,
        style: 'position:absolute;top:-9999em;height:100px;overflow-y:scroll'
      }));

      if (element.parentNode !== doc.body) {
        doc.body.appendChild(element);
      }

      scrollBarWidth = element.offsetWidth;
    }

    return scrollBarWidth;
  }

  static get scrollBarPadElement() {
    return {
      className: 'b-yscroll-pad',
      children: [{
        className: 'b-yscroll-pad-sizer'
      }]
    };
  }
  /**
   * Resets DomHelper.scrollBarWidth cache, triggering a new measurement next time it is read
   */

  static resetScrollBarWidth() {
    scrollBarWidth = null;
  }
  /**
   * Measures the text width using a hidden div
   * @param {String} text
   * @param {HTMLElement} sourceElement
   * @returns {Number} width
   * @category Measure
   */

  static measureText(text, sourceElement, useHTML = false, parentElement = undefined) {
    const offScreenDiv = DH$1.getMeasureElement(sourceElement, parentElement);
    offScreenDiv[useHTML ? 'innerHTML' : 'innerText'] = text;
    const result = offScreenDiv.clientWidth;
    offScreenDiv.className = '';
    return result;
  }
  /**
   * Measures a relative size, such as a size specified in `em` units for the passed element.
   * @param {String} size The CSS size value to measure.
   * @param {HTMLElement} sourceElement
   * @param {Boolean} [round] Pass true to return exact width, not rounded value
   * @returns {Number} size The size in pixels of the passed relative measurement.
   * @category Measure
   */

  static measureSize(size, sourceElement, round = true) {
    if (!size) {
      return 0;
    }

    if (typeof size === 'number') {
      return size;
    }

    if (!size.length) {
      return 0;
    }

    if (/^\d+(px)?$/.test(size)) {
      return parseInt(size);
    }

    if (sourceElement) {
      const offScreenDiv = DH$1.getMeasureElement(sourceElement);
      offScreenDiv.innerHTML = '';
      offScreenDiv.style.width = DH$1.setLength(size);
      const result = round ? offScreenDiv.offsetWidth : offScreenDiv.getBoundingClientRect().width;
      offScreenDiv.style.width = offScreenDiv.className = '';
      return result;
    }

    if (/^\d+em$/.test(size)) {
      return parseInt(size) * DEFAULT_FONT_SIZE;
    }

    return isNaN(size) ? 0 : parseInt(size);
  } // parentElement allows measurement to happen inside a specific element, allowing scoped css rules to match

  static getMeasureElement(sourceElement, parentElement = doc.body) {
    const sourceElementStyle = win.getComputedStyle(sourceElement),
          offScreenDiv = parentElement.offScreenDiv = parentElement.offScreenDiv || DH$1.createElement({
      parent: parentElement,
      style: 'position:fixed;top:-10000px;left:-10000px;visibility:hidden;contain:strict',
      className: 'b-measure-element',
      children: [{
        style: 'white-space:nowrap;display:inline-block;will-change:contents;width:auto;contain:none'
      }]
    }, {
      returnAll: true
    })[1];
    fontProps.forEach(prop => {
      if (offScreenDiv.style[prop] !== sourceElementStyle[prop]) {
        offScreenDiv.style[prop] = sourceElementStyle[prop];
      }
    });
    offScreenDiv.className = sourceElement.className; // In case the measure element was moved/removed, re-add it

    if (offScreenDiv.parentElement.parentElement !== parentElement) {
      parentElement.appendChild(offScreenDiv.parentElement);
    }

    return offScreenDiv;
  }
  /**
   * Strips the tags from a html string, returning text content.
   *
   * ```javascript
   * DomHelper.stripTags('<div class="custom"><b>Bold</b><i>Italic</i></div>'); // -> BoldItalic
   * ```
   *
   * @internal
   * @param {String} htmlString HTML string
   * @returns {String} Text content
   */

  static stripTags(htmlString) {
    const // we need to avoid any kind of evaluation of embedded XSS scripts or "web bugs" (img tags that issue
    // GET requests)
    parser = DH$1.$domParser || (DH$1.$domParser = new DOMParser()),
          doc = parser.parseFromString(htmlString, 'text/html');
    return doc.body.textContent;
  } //endregion
  //region Sync

  /**
   * Sync one source element attributes, children etc. to a target element. Source element can be specified as a html
   * string or an actual HTMLElement.
   *
   * NOTE: This function is superseded by {@link Core/helper/DomSync#function-sync-static DomSync.sync()}, which works
   * with DOM configs. For most usecases, use it instead.
   *
   * @param {String|HTMLElement} sourceElement Source "element" to copy from
   * @param {HTMLElement} targetElement Target element to apply to, can also be specified as part of the config object
   * @returns {HTMLElement} Returns the updated targetElement (which is also updated in place)
   */

  static sync(sourceElement, targetElement) {
    if (typeof sourceElement === 'string') {
      if (sourceElement === '') {
        targetElement.innerHTML = '';
        return;
      } else {
        sourceElement = DH$1.createElementFromTemplate(sourceElement);
      }
    }

    DH$1.performSync(sourceElement, targetElement);
    return targetElement;
  } // Internal helper used for recursive syncing

  static performSync(sourceElement, targetElement) {
    // Syncing identical elements is a no-op
    if (sourceElement.outerHTML !== targetElement.outerHTML) {
      DH$1.syncAttributes(sourceElement, targetElement);
      DH$1.syncContent(sourceElement, targetElement);
      DH$1.syncChildren(sourceElement, targetElement);
      return true;
    }

    return false;
  } // Attributes as map { attr : value, ... }, either from an html element or from a config

  static getSyncAttributes(element) {
    const attributes = {},
          // Attribute names, simplifies comparisons and calls to set/removeAttribute
    names = []; // Extract from element

    for (let i = 0; i < element.attributes.length; i++) {
      const attr = element.attributes[i];

      if (attr.specified) {
        const name = attr.name.toLowerCase();
        attributes[name] = attr.value;
        names.push(name);
      }
    }

    return {
      attributes,
      names
    };
  }
  /**
   * Syncs attributes from sourceElement to targetElement.
   * @private
   * @param {HTMLElement} sourceElement
   * @param {HTMLElement} targetElement
   */

  static syncAttributes(sourceElement, targetElement) {
    const // Extract attributes from elements (sourceElement might be a config)
    {
      attributes: sourceAttributes,
      names: sourceNames
    } = DH$1.getSyncAttributes(sourceElement),
          {
      attributes: targetAttributes,
      names: targetNames
    } = DH$1.getSyncAttributes(targetElement),
          // Used to ignore data-xx attributes when we will be setting entire dataset
    hasDataset = sourceNames.includes('dataset'),
          // Intersect arrays to determine what needs adding, removing and syncing
    toAdd = sourceNames.filter(attr => !targetNames.includes(attr)),
          toRemove = targetNames.filter(attr => !sourceNames.includes(attr) && (!hasDataset || !attr.startsWith('data-'))),
          toSync = sourceNames.filter(attr => targetNames.includes(attr));

    if (toAdd.length > 0) {
      for (let i = 0; i < toAdd.length; i++) {
        const attr = toAdd[i]; // Style requires special handling

        if (attr === 'style') {
          DH$1.applyStyle(targetElement, sourceAttributes.style, true);
        } // So does dataset
        else if (attr === 'dataset') {
          Object.assign(targetElement.dataset, sourceAttributes.dataset);
        } // Other attributes are set using setAttribute (since it calls toString() DomClassList works fine)
        else {
          targetElement.setAttribute(attr, sourceAttributes[attr]);
        }
      }
    }

    if (toRemove.length > 0) {
      for (let i = 0; i < toRemove.length; i++) {
        targetElement.removeAttribute(toRemove[i]);
      }
    }

    if (toSync.length > 0) {
      for (let i = 0; i < toSync.length; i++) {
        const attr = toSync[i]; // Set all attributes that has changed, with special handling for style

        if (attr === 'style') {
          // TODO: Check for changes?
          DH$1.applyStyle(targetElement, sourceAttributes.style, true);
        } // And dataset
        else if (attr === 'dataset') {
          // TODO: Any cost to assigning same values?
          Object.assign(targetElement.dataset, sourceAttributes.dataset);
        } // And class, which might be a DomClassList or an config for a DomClassList
        else if (attr === 'class' && (sourceAttributes.class.isDomClassList || typeof sourceAttributes.class === 'object')) {
          let classList;

          if (sourceAttributes.class.isDomClassList) {
            classList = sourceAttributes.class;
          } else {
            // TODO : Reuse a single DomClassList?
            classList = new DomClassList(sourceAttributes.class);
          }

          if (!classList.isEqual(targetAttributes.class)) {
            targetElement.setAttribute('class', classList);
          }
        } else if (targetAttributes[attr] !== sourceAttributes[attr]) {
          targetElement.setAttribute(attr, sourceAttributes[attr]);
        }
      }
    }
  }
  /**
   * Sync content (innerText) from sourceElement to targetElement
   * @private
   * @param {HTMLElement} sourceElement
   * @param {HTMLElement} targetElement
   */

  static syncContent(sourceElement, targetElement) {
    if (DH$1.getChildElementCount(sourceElement) === 0) {
      targetElement.innerText = sourceElement.innerText;
    }
  }

  static setInnerText(targetElement, text) {
    // setting firstChild.data is faster than innerText (and innerHTML),
    // but in some cases the inner node is lost and needs to be recreated
    const {
      firstChild
    } = targetElement;

    if ((firstChild === null || firstChild === void 0 ? void 0 : firstChild.nodeType) === Element.TEXT_NODE) {
      firstChild.data = text;
    } else {
      // textContent is supposed to be faster than innerText, since it does not trigger layout
      targetElement.textContent = text;
    }
  }
  /**
   * Sync traversing children
   * @private
   * @param {HTMLElement} sourceElement Source element
   * @param {HTMLElement} targetElement Target element
   */

  static syncChildren(sourceElement, targetElement) {
    const me = this,
          sourceNodes = arraySlice$1.call(sourceElement.childNodes),
          targetNodes = arraySlice$1.call(targetElement.childNodes);

    while (sourceNodes.length) {
      const sourceNode = sourceNodes.shift(),
            targetNode = targetNodes.shift(); // only textNodes and elements allowed (no comments)

      if (sourceNode && sourceNode.nodeType !== TEXT_NODE && sourceNode.nodeType !== ELEMENT_NODE) {
        throw new Error(`Source node type ${sourceNode.nodeType} not supported by DomHelper.sync()`);
      }

      if (targetNode && targetNode.nodeType !== TEXT_NODE && targetNode.nodeType !== ELEMENT_NODE) {
        throw new Error(`Target node type ${targetNode.nodeType} not supported by DomHelper.sync()`);
      }

      if (!targetNode) {
        // out of target nodes, add to target
        targetElement.appendChild(sourceNode);
      } else {
        // match node
        if (sourceNode.nodeType === targetNode.nodeType) {
          // same type of node, take action depending on which type
          if (sourceNode.nodeType === TEXT_NODE) {
            // text
            targetNode.data = sourceNode.data;
          } else {
            if (sourceNode.tagName === targetNode.tagName) {
              me.performSync(sourceNode, targetNode);
            } else {
              // new tag, remove targetNode and insert new element
              targetElement.insertBefore(sourceNode, targetNode);
              targetNode.remove();
            }
          }
        } // Trying to set text node as element, use it as innerText
        // (we get this in FF with store mutations and List)
        else if (sourceNode.nodeType === TEXT_NODE && targetNode.nodeType === ELEMENT_NODE) {
          targetElement.innerText = sourceNode.data.trim();
        } else {
          const logElement = sourceNode.parentElement || sourceNode;
          throw new Error(`Currently no support for transforming nodeType.\n${logElement.outerHTML}`);
        }
      }
    } // Out of source nodes, remove remaining target nodes

    targetNodes.forEach(targetNode => {
      targetNode.remove();
    });
  }
  /**
   * Replaces the passed element's `className` with the class names
   * passed in either Array or String format or Object.
   *
   * This method compares the existing class set with the incoming class set and
   * avoids mutating the element's class name set if possible.
   *
   * This can avoid browser style invalidations.
   * @param {HTMLElement} element The element whose class list to synchronize.
   * @param {String[]|String|Object} newClasses The incoming class names to set on the element.
   * @returns {Boolean} `true` if the DOM class list was changed.
   * @category CSS
   */

  static syncClassList(element, newClasses) {
    const {
      classList
    } = element,
          isString = typeof newClasses === 'string',
          newClsArray = isString ? newClasses.trim().split(whiteSpaceRe) : Array.isArray(newClasses) ? newClasses : ObjectHelper.getTruthyKeys(newClasses),
          classCount = newClsArray.length;
    let changed = classList.length !== classCount,
        i; // If the incoming and existing class lists are the same length
    // then check that each contains the same names. As soon as
    // we find a non-matching name, we know we have to update the
    // className.

    for (i = 0; !changed && i < classCount; i++) {
      changed = !classList.contains(newClsArray[i]);
    }

    if (changed) {
      element.className = isString ? newClasses : newClsArray.join(' ');
    }

    return changed;
  }
  /**
   * Applies the key state of the passed object or DomClassList to the passed element.
   *
   * Properties with a falsy value mean that property name is *removed* as a class name.
   *
   * Properties with a truthy value mean that property name is *added* as a class name.
   *
   * This is different from {@link #function-syncClassList-static}. That sets the `className` of the element to the
   * sum of all its truthy keys, regardless of what the pre-existing value of the `className` was, and ignoring falsy
   * keys.
   *
   * This _selectively_ updates the classes in the `className`. If there is a truthy key, the name is added. If there
   * is a falsy key, the name is removed.
   * @param {HTMLElement} element The element to apply the class list to .
   * @param {Object|Core.helper.util.DomClassList} classes The classes to add or remove.
   * @returns {Boolean} `true` if the DOM class list was changed.
   * @category CSS
   */

  static updateClassList(element, classes) {
    const {
      classList
    } = element;
    let cls,
        add,
        changed = false;

    for (cls in classes) {
      add = Boolean(classes[cls]);

      if (classList.contains(cls) !== add) {
        classList[add ? 'add' : 'remove'](cls);
        changed = true;
      }
    }

    return changed;
  }
  /**
   * Changes the theme to the passed theme name if possible.
   *
   * Theme names are case insensitive. The `href` used is all lower case.
   *
   * To use this method, the `<link rel="stylesheet">` _must_ use the default,
   * Bryntum-supplied CSS files where the `href` end with `<themeName>.css`, so that
   * it can be found in the document, and switched out for a new link with
   * the a modified `href`. The new `href` will use the same path, just
   * with the `themeName` portion substituted for the new name.
   *
   * If no `<link>` with that name pattern can be found, an error will be thrown.
   *
   * If you use this method, you  must ensure that the theme files are
   * all accessible on your server.
   *
   * Because this is an asynchronous operation, a `Promise` is returned.
   * The theme change event is passed to the success function. If the
   * theme was not changed, because the theme name passed is the current theme,
   * nothing is passed to the success function.
   *
   * The theme change event contains two properties:
   *
   *  - `prev` The previous Theme name.
   *  - `theme` The new Theme name.
   *
   * @param {String} newThemeName the name of the theme that should be applied
   * @privateparam {String} [defaultTheme] Optional, the name of the theme that should be used in case of fail
   * @returns {Promise} A promise who's success callback receives the theme change
   * event if the theme in fact changed. If the theme `href` could not be loaded,
   * the failure callback is called, passing the error event caught.
   */

  static setTheme(newThemeName, defaultTheme) {
    newThemeName = newThemeName.toLowerCase();
    const {
      head
    } = document,
          oldThemeName = DH$1.getThemeInfo(defaultTheme).name.toLowerCase();
    let oldThemeLinks = head.querySelectorAll('[data-bryntum-theme]:not([data-loading])'),
        loaded = 0;

    if (oldThemeName === newThemeName) {
      return immediatePromise$8;
    } // Remove any links currently loading

    DH$1.removeEachSelector(head, '#bryntum-theme[data-loading],link[data-bryntum-theme][data-loading]');
    const themeEvent = {
      theme: newThemeName,
      prev: oldThemeName
    };

    function replaceTheme(oldThemeLink, resolve, reject) {
      const newThemeLink = DomHelper.createElement({
        tag: 'link',
        rel: 'stylesheet',
        dataset: {
          loading: true,
          bryntumTheme: true
        },
        href: oldThemeLink.href.replace(oldThemeName, newThemeName),
        nextSibling: oldThemeLink
      });
      newThemeLink.addEventListener('load', () => {
        delete newThemeLink.dataset.loading;
        themeInfo = null; // Flip all products to the new theme at the same time

        if (++loaded === oldThemeLinks.length) {
          oldThemeLinks.forEach(link => link.remove());
          GlobalEvents.trigger('theme', themeEvent);
          resolve(themeEvent);
        }
      });
      newThemeLink.addEventListener('error', e => {
        delete newThemeLink.dataset.loading;
        reject(e);
      });
    }

    if (oldThemeLinks.length) {
      return new Promise((resolve, reject) => {
        oldThemeLinks.forEach((oldThemeLink, i) => {
          replaceTheme(oldThemeLink, resolve, reject, i === oldThemeLinks.length - 1);
        });
      });
    } else {
      const oldThemeLink = head.querySelector('#bryntum-theme:not([data-loading])') || head.querySelector(`[href*="${oldThemeName}.css"]:not([data-loading])`); // Theme link href ends with <themeName>.css also there could be a query - css?11111...

      if (!(oldThemeLink !== null && oldThemeLink !== void 0 && oldThemeLink.href.includes(`${oldThemeName}.css`))) {
        throw new Error(`Theme link for ${oldThemeName} not found`);
      }

      oldThemeLinks = [oldThemeLink];
      return new Promise((resolve, reject) => replaceTheme(oldThemeLink, resolve, reject));
    }
  }
  /**
   * A theme information object about the current theme.
   *
   * Currently this has only one property:
   *
   *   - `name` The current theme name.
   * @property {Object}
   * @readonly
   */

  static get themeInfo() {
    return DomHelper.getThemeInfo();
  }
  /**
   * A theme information object about the current theme.
   *
   * Currently this has only one property:
   *
   *   - `name` The current theme name.
   * @param {String} defaultTheme the name of the theme used as backup value in case of fail
   * @returns {Object} info, currently it contains only one property - 'name'.
   * @private
   */

  static getThemeInfo(defaultTheme) {
    if (!themeInfo) {
      const // The content it creates for 'b-theme-info' is described in corresponding theme in Core/resources/sass/themes
      // for example in Core/resources/sass/themes/material.scss
      // ```
      // .b-theme-info:before {
      //     content : '{"name":"Material"}';
      // }
      // ```
      testDiv = DH$1.createElement({
        parent: document.body,
        className: 'b-theme-info'
      }),
            // Theme desc object is in the :before pseudo element.
      themeData = DH$1.getStyleValue(testDiv, 'content', false, ':before');

      if (themeData) {
        // themeData could be invalid JSON string in case there is no content rule
        try {
          themeInfo = JSON.parse(themeData.replace(/^["']|["']$|\\/g, ''));
        } catch (e) {
          themeInfo = null;
        }
      } // CSS file has to be loaded to make the themeInfo available, so fallback to the default theme name

      themeInfo = themeInfo || (defaultTheme ? {
        name: defaultTheme
      } : null);
      testDiv.remove();
    }

    return themeInfo;
  } //endregion
  //region Transition

  static async transition({
    element: outerElement,
    selector = '[data-dom-transition]',
    duration,
    action,
    thisObj = this,
    addTransition = {},
    removeTransition = {}
  }) {
    const scrollers = new Set(),
          beforeElements = Array.from(outerElement.querySelectorAll(selector)),
          beforeMap = new Map(beforeElements.map(element => {
      let depth = 0,
          parent = element.parentElement;

      while (parent && parent !== outerElement) {
        depth++;
        parent = parent.parentElement;
      }

      element.$depth = depth; // Store scrolling elements and their current scroll pos, for restoring later

      if (element.scrollHeight > element.offsetHeight && getComputedStyle(element).overflow === 'auto') {
        element.$scrollTop = element.scrollTop;
        scrollers.add(element);
      } // Intersect our bounds with parents, to trim away overflow

      const {
        parentElement
      } = element,
            globalBounds = Rectangle.from(element, outerElement),
            localBounds = Rectangle.from(element, parentElement),
            style = getComputedStyle(parentElement),
            borderLeftWidth = parseFloat(style.borderLeftWidth);

      if (borderLeftWidth) {
        globalBounds.left -= borderLeftWidth;
        localBounds.left -= borderLeftWidth;
      }

      return [element.id, {
        element,
        globalBounds,
        localBounds,
        depth,
        parentElement
      }];
    }));
    action.call(thisObj);
    const afterElements = Array.from(outerElement.querySelectorAll(selector)),
          afterMap = new Map(afterElements.map(element => {
      const globalBounds = Rectangle.from(element, outerElement),
            localBounds = Rectangle.from(element, element.parentElement),
            style = globalThis.getComputedStyle(element.parentElement),
            borderLeftWidth = parseFloat(style.borderLeftWidth);

      if (borderLeftWidth) {
        globalBounds.left -= borderLeftWidth;
        localBounds.left -= borderLeftWidth;
      }

      return [element.id, {
        element,
        globalBounds,
        localBounds
      }];
    })),
          styleProps = ['position', 'top', 'left', 'width', 'height', 'padding', 'margin', 'zIndex', 'minWidth', 'minHeight', 'opacity', 'overflow']; // Convert to absolute layout, iterating elements remaining after action

    for (const [id, before] of beforeMap) {
      // We match before vs after on id and not actual element, allowing adding a new element with the same id to
      // transition from the old (which was removed or released). To match what will happen when DomSyncing with
      // multiple containing elements (columns in TaskBoard)
      const after = afterMap.get(id);

      if (after) {
        const {
          element
        } = after,
              {
          style,
          parentElement
        } = element,
              // Need to keep explicit zIndex to keep above other stuff
        zIndex = parseInt(DH$1.getStyleValue(element, 'zIndex')),
              {
          globalBounds,
          localBounds,
          depth,
          parentElement: beforeParent
        } = before,
              parentChanged = beforeParent !== parentElement; // Store initial state, in case element has a style prop we need to restore later

        ObjectHelper.copyProperties(element.$initial = {
          parentElement
        }, style, styleProps); // Prevent transition during the process, forced further down instead
        // element.remove();

        let bounds; // Action moved element to another parent, move it to the outer element to allow transitioning to the
        // new parent. Also use coordinates relative to that element

        if (parentChanged) {
          after.bounds = after.globalBounds;
          bounds = globalBounds;
          outerElement.appendChild(element);
        } // Keep element in current parent if it was not moved during the action call above.
        // Need to use coords relative to the parent
        else {
          after.bounds = after.localBounds;
          bounds = localBounds;
          beforeParent.appendChild(element);
        }

        let overflow = 'hidden'; // Looks weird with content sticking out if height is transitioned

        if (scrollers.has(element)) {
          element.$scrollPlaceholder = DH$1.createElement({
            parent: element,
            style: {
              height: element.scrollHeight
            }
          });
          overflow = 'auto';
        } // Move element back to where it started

        Object.assign(style, {
          position: 'absolute',
          top: `${bounds.top}px`,
          left: `${bounds.left}px`,
          width: `${bounds.width}px`,
          height: `${bounds.height}px`,
          minWidth: 0,
          minHeight: 0,
          padding: 0,
          margin: 0,
          zIndex: depth + (zIndex || 0),
          overflow
        });
        after.processed = true;
      } // Existed before but not after = removed
      else {
        const {
          element,
          localBounds: bounds,
          depth,
          parentElement
        } = before;
        element.$initial = {
          removed: true
        };
        Object.assign(element.style, {
          position: 'absolute',
          top: `${bounds.top}px`,
          left: `${bounds.left}px`,
          width: `${bounds.width}px`,
          height: `${bounds.height}px`,
          minWidth: 0,
          minHeight: 0,
          padding: 0,
          margin: 0,
          zIndex: depth,
          overflow: 'hidden' // Looks weird with content sticking out if height is transitioned

        });
        parentElement.appendChild(element); // Inject among non-removed elements to have it transition away

        afterMap.set(id, {
          element,
          bounds,
          removed: true,
          processed: true
        });
        afterElements.push(element);
      }
    } // Handle new elements

    for (const [, after] of afterMap) {
      if (!after.processed) {
        const {
          element
        } = after,
              {
          style,
          parentElement
        } = element,
              bounds = after.bounds = after.localBounds;
        element.classList.add('b-dom-transition-adding');
        ObjectHelper.copyProperties(element.$initial = {
          parentElement
        }, style, styleProps); // Props in `addTransition` will be transitioned

        Object.assign(style, {
          position: 'absolute',
          top: addTransition.top ? 0 : `${bounds.top}px`,
          left: addTransition.left ? 0 : `${bounds.left}px`,
          width: addTransition.width ? 0 : `${bounds.width}px`,
          height: addTransition.height ? 0 : `${bounds.height}px`,
          opacity: addTransition.opacity ? 0 : null,
          zIndex: parentElement.$depth + 1,
          overflow: 'hidden' // Looks weird with content sticking out if height is transitioned

        });
      }
    } // Restore scroll after modifying layout

    for (const element of scrollers) {
      element.scrollTop = element.$scrollTop;
    } // Enable transitions

    outerElement.classList.add('b-dom-transition'); // Trigger layout, to be able to transition below

    outerElement.firstElementChild.offsetWidth; // Transition to new layout

    for (const [, {
      element,
      bounds: afterBounds,
      removed
    }] of afterMap) {
      if (removed) {
        Object.assign(element.style, {
          top: removeTransition.top ? 0 : `${afterBounds.top}px`,
          left: removeTransition.left ? 0 : `${afterBounds.left}px`,
          width: removeTransition.width ? 0 : `${afterBounds.width}px`,
          height: removeTransition.height ? 0 : `${afterBounds.height}px`,
          opacity: removeTransition.opacity ? 0 : element.$initial.opacity
        });
      } else {
        Object.assign(element.style, {
          top: `${afterBounds.top}px`,
          left: `${afterBounds.left}px`,
          width: `${afterBounds.width}px`,
          height: `${afterBounds.height}px`,
          opacity: element.$initial.opacity
        });
      }
    } // Wait for transition to finish

    await AsyncHelper.sleep(duration);
    outerElement.classList.remove('b-dom-transition'); // Restore layout after transition

    for (const element of afterElements) {
      if (element.$initial) {
        if (element.$initial.removed) {
          element.remove();
        } else {
          ObjectHelper.copyProperties(element.style, element.$initial, styleProps);

          if (element.$scrollPlaceholder) {
            element.$scrollPlaceholder.remove();
            delete element.$scrollPlaceholder;
          }

          element.classList.remove('b-dom-transition-adding');
          element.$initial.parentElement.appendChild(element);
        }
      }
    } // Restore scroll positions last when all layout is restored

    for (const element of scrollers) {
      element.scrollTop = element.$scrollTop;
      delete element.$scrollTop;
    }
  } //endregion

  static async loadScript(url) {
    return new Promise((resolve, reject) => {
      const script = document.createElement('script');
      script.src = url;
      script.onload = resolve;
      script.onerror = reject;
      document.head.appendChild(script);
    });
  }

  static isNamedColor(color) {
    return color && !/^(#|hsl|rgb)/.test(color);
  }

}
const DH$1 = DomHelper;
let clearTouchTimer;

const clearTouchEvent = () => DH$1.isTouchEvent = false,
      setTouchEvent = () => {
  DH$1.isTouchEvent = true; // Jump round the click delay

  clearTimeout(clearTouchTimer);
  clearTouchTimer = setTimeout(clearTouchEvent, 400);
}; // Set event type flags so that mousedown and click handlers can know whether a touch gesture was used.
// This is used. This must stay until we have a unified DOM event system which handles both touch and mouse events.

doc.addEventListener('touchstart', setTouchEvent, true);
doc.addEventListener('touchend', setTouchEvent, true);
DH$1.canonicalStyles = canonicalStyles;
DH$1.supportsTemplate = 'content' in doc.createElement('template');
DH$1.elementPropKey = elementPropKey;
DH$1.numberRe = numberRe; //region Polyfills
// TODO: include babels polyfills instead of keeping own?

if (!('children' in Node.prototype)) {
  const elementFilter = node => node.nodeType === node.ELEMENT_NODE;

  Object.defineProperty(Node.prototype, 'children', {
    get: function () {
      return Array.prototype.filter.call(this.childNodes, elementFilter);
    }
  });
}

if (!Element.prototype.matches) {
  Element.prototype.matches = Element.prototype.matchesSelector || Element.prototype.mozMatchesSelector || Element.prototype.msMatchesSelector || Element.prototype.oMatchesSelector || Element.prototype.webkitMatchesSelector || function (s) {
    const matches = (this.document || this.ownerDocument).querySelectorAll(s);
    let i = matches.length; // eslint-disable-next-line no-empty

    while (--i >= 0 && matches.item(i) !== this) {}

    return i > -1;
  };
}

if (win.Element && !Element.prototype.closest) {
  Node.prototype.closest = Element.prototype.closest = function (s) {
    let el = this;
    if (!doc.documentElement.contains(el)) return null;

    do {
      if (el.matches(s)) return el;
      el = el.parentElement || el.parentNode;
    } while (el !== null && el.nodeType === el.ELEMENT_NODE);

    return null;
  };
} else {
  // It's crazy that closest is not already on the Node interface!
  // Note that some Node types (eg DocumentFragment) do not have a parentNode.
  Node.prototype.closest = function (selector) {
    var _this$parentNode;

    return (_this$parentNode = this.parentNode) === null || _this$parentNode === void 0 ? void 0 : _this$parentNode.closest(selector);
  };
} // from MDN (public domain): https://developer.mozilla.org/en-US/docs/Web/API/ChildNode/remove

(function (arr) {
  arr.forEach(function (item) {
    if (Object.prototype.hasOwnProperty.call(item, 'remove')) {
      return;
    }

    Object.defineProperty(item, 'remove', {
      configurable: true,
      enumerable: true,
      writable: true,
      value: function remove() {
        this.parentNode && this.parentNode.removeChild(this);
      }
    });
  });
})([Element.prototype, CharacterData.prototype, DocumentType.prototype]); //endregion
// CTRL/+ and  CTRL/- zoom gestures must invalidate the scrollbar width.
// Window resize is triggered by this operation on Blink (Chrome & Edge), Firefox and Safari.

globalThis.addEventListener('resize', () => scrollBarWidth = null);
DomHelper._$name = 'DomHelper';

/**
 * @module Core/GlobalEvents
 */

const isFloatingWidget = w => w.floating;

const GlobalEvents = new class extends Base$1.mixin(Events) {
  suspendFocusEvents() {
    focusEventsSuspended = true;
  }

  resumeFocusEvents() {
    focusEventsSuspended = false;
  }

  setupFocusListenersOnce(rootElement, EventHelper) {
    if (rootElement && !GlobalEvents.observedElements.has(rootElement)) {
      GlobalEvents.setupFocusListeners(rootElement, EventHelper);
      GlobalEvents.observedElements.add(rootElement);
    }
  } // This is imported by EventHelper and that makes the call to set up the listeners
  // `detach` argument is required to not setup more listeners than we need to. In case of salesforce we include floatroot
  // inside the webcomponent element and thus don't need default listeners on document. In regular webcomponents demo we
  // don't need to do it, because with multiple components on one page that would force us to make more complex lookups.

  setupFocusListeners(element = document, EventHelper, detach = false) {
    var _detacher;

    const listeners = {
      element,

      touchstart(touchstart) {
        if (!globaltouchStart && touchstart.changedTouches.length === 1) {
          globaltouchStart = touchstart.changedTouches[0];
        } else {
          globaltouchStart = null;
        }
      },

      // Just this one has to be passive: false so that we are allowed to preventDefault
      // if we are part of a contextmenu longpress emulation. Otherwise the gesture will
      // proceed to cause a mousedown event.
      touchend: {
        handler: event => {
          if (globaltouchStart) {
            // If the touchstart was used to synthesize a contextmenu event
            // stop the touch gesture processing right now.
            // Also prevent the conversion of the touch into  click.
            if (globaltouchStart.identifier === EventHelper.contextMenuTouchId) {
              event.stopImmediatePropagation();
              event.preventDefault();
            } else if (event.changedTouches.length === 1 && event.changedTouches[0].identifier === globaltouchStart.identifier) {
              GlobalEvents.trigger('globaltap', {
                event
              });
            }

            globaltouchStart = null;
          }
        },
        passive: false
      },
      mousedown: {
        handler: event => {
          lastInteractionType = 'mouse';

          if (!globaltouchStart) {
            GlobalEvents.trigger('globaltap', {
              event
            });
          }
        },
        passive: false
      },

      keydown() {
        lastInteractionType = 'key';
      },

      keypress() {
        lastInteractionType = 'key';
      },

      focusin(focusin) {
        const {
          Widget
        } = GlobalEvents; // https://app.assembla.com/spaces/bryntum/tickets/5503
        // Caused by the browser scrolling a focused element into view. The browser will do *whatever it takes*
        // to scroll a focused element so that as much of it is in view as possible. Its first point of scrolling will
        // be the float containing element. That must never scroll.
        // TODO: Remove when https://www.w3.org/TR/css-overflow-3/#valdef-overflow-clip is supported.

        Widget.resetFloatRootScroll();

        if (focusEventsSuspended) {
          return;
        }

        const fromElement = !focusin.relatedTarget ? null : focusin.relatedTarget instanceof HTMLElement ? focusin.relatedTarget : document.body,
              toElement = focusin.target || document.body,
              fromWidget = Widget.fromElement(fromElement),
              toWidget = Widget.fromElement(toElement),
              commonAncestor = DomHelper.getCommonAncestor(fromWidget, toWidget),
              // Flag if the fromElement is DOCUMENT_POSITION_FOLLOWING toElement
        backwards = !!(fromElement && toElement.compareDocumentPosition(fromElement) & 4),
              topVisibleModal = Widget.query(isTopVisibleModal);
        let currentFocus = null;

        if (toElement && toElement !== document.body) {
          currentFocus = DomHelper.getActiveElement(toElement);
        } else {
          currentFocus = DomHelper.getActiveElement(document);
        } // If there is a topmost modal, and the focus is moving to somewhere *not* a descendant of that
        // modal, and that somewhere is not in a floater that us *above* that modal (the compareDocumentPosition call),
        // then we enforce modality and sweep focus back into the modal.
        // By default the Container class will yield the first focusable descendant widget's focusEl as its
        // focusEl, so that will be out of the box behaviour for Popups.

        if (topVisibleModal) {
          if (!toWidget || !topVisibleModal.owns(toWidget) && !(topVisibleModal.element.compareDocumentPosition(toWidget.element) & 4 && toWidget.up(isFloatingWidget))) {
            return topVisibleModal.focus();
          }
        }

        let event = createWidgetEvent('focusout', fromElement, focusin.target, fromWidget, toWidget, backwards); // Bubble focusout event up the "from" side of the tree

        for (let target = fromWidget; target && target !== commonAncestor; target = target.owner) {
          if (!target.isDestroying && target.onFocusOut) {
            target.onFocusOut(event); // It is possible for focusout handlers to refocus themselves (editor's invalidAction='block'), so
            // check if the focus is still where it was when we started unless we are in a document
            // loss of focus situation (no target)

            if (focusin.target && currentFocus !== DomHelper.getActiveElement(focusin.target)) {
              // If the focus has moved, that movement would have kicked off a nested sequence of focusin/out
              // notifications, so everyone has already been notified... no more to do here.
              return;
            }
          }
        } // Focus is moving upwards to the ancestor widget.
        // Its focus method might delegate focus to a focusable descendant.

        if (commonAncestor && focusin.target === commonAncestor.element) {
          // If one of the handlers above has not moved focus onwards
          // and the common ancestor is a container which delegates
          // focus inwards to a descendant, then give it the opportunity to do that.
          if (!commonAncestor.isDestroying && DomHelper.getActiveElement(commonAncestor) === toElement && commonAncestor.focusElement && commonAncestor.focusElement !== commonAncestor.element) {
            // If focus is not inside, move focus inside
            if (!commonAncestor.element.contains(currentFocus) || commonAncestor.focusDescendant) {
              // Wait until out of the focusin handler to move focus on.
              commonAncestor.setTimeout(() => {
                var _commonAncestor$focus;

                return (_commonAncestor$focus = commonAncestor.focus) === null || _commonAncestor$focus === void 0 ? void 0 : _commonAncestor$focus.call(commonAncestor);
              }, 0);
            }
          }
        } // Focus is moving between two branches of a subtree.
        // Bubble focusin event up the "to" side of the tree
        else {
          event = createWidgetEvent('focusin', toElement, fromElement, fromWidget, toWidget, backwards);

          for (let target = toWidget; target && target !== commonAncestor; target = target.owner) {
            if (!target.isDestroying) {
              var _target$onFocusIn, _target;

              (_target$onFocusIn = (_target = target).onFocusIn) === null || _target$onFocusIn === void 0 ? void 0 : _target$onFocusIn.call(_target, event);
            }
          }
        } // Fire element focusmove event. Grid navigation will use  this when cells are focusable.

        const commonAncestorEl = DomHelper.getCommonAncestor((fromElement === null || fromElement === void 0 ? void 0 : fromElement.nodeType) === Element.ELEMENT_NODE ? fromElement : null, toElement) || toElement.parentNode;
        event = createWidgetEvent('focusmove', toElement, fromElement, fromWidget, toWidget, backwards, {
          bubbles: true
        });
        commonAncestorEl.dispatchEvent(event);
      },

      focusout(focusout) {
        if (focusEventsSuspended) {
          return;
        }

        if (!focusout.relatedTarget || !GlobalEvents.Widget.fromElement(focusout.relatedTarget)) {
          // When switching between tabs in Salesforce app `relatedTarget` of the focusout event might be not an instance of
          // HTMLElement.
          const target = focusout.relatedTarget && focusout.relatedTarget instanceof HTMLElement ? focusout.relatedTarget : null;
          listeners.focusin({
            target,
            relatedTarget: focusout.target
          });
        }
      },

      capture: true,
      passive: true
    }; // detach previous listeners

    detach && ((_detacher = detacher) === null || _detacher === void 0 ? void 0 : _detacher());
    detacher = EventHelper.on(listeners);
  }

  get lastInteractionType() {
    return lastInteractionType;
  }

}(),
      isTopVisibleModal = w => w.isVisible && w.isTopModal;

GlobalEvents.observedElements = new Set();
/**
 * Fired after the theme is changed
 * @event theme
 * @param {Core.GlobalEvents} source
 * @param {String} theme The new theme name
 */

let globaltouchStart,
    focusEventsSuspended = false,
    lastInteractionType,
    detacher;

function createWidgetEvent(eventName, target, relatedTarget, fromWidget, toWidget, backwards, options) {
  const result = new CustomEvent(eventName, options); // Workaround for Salesforce. They use strict mode and define non-configurable property `target`. We use this
  // CustomEvent as a synthetic one, feels fine to use non-standard handle for target.

  Object.defineProperty(result, '_target', {
    get() {
      return target;
    }

  });
  Object.defineProperty(result, 'relatedTarget', {
    get() {
      return relatedTarget;
    }

  });
  result.fromWidget = fromWidget;
  result.toWidget = toWidget;
  result.backwards = backwards;
  return result;
}

/**
 * @module Core/helper/DomSync
 */

const arraySlice = Array.prototype.slice,
      emptyArray$9 = Object.freeze([]),
      emptyObject$9 = Object.freeze({}),
      htmlRe = /[&<]/,
      // tests if setInnerText is equivalent to innerHTML
{
  getPrototypeOf
} = Object,
      {
  toString
} = Object.prototype,
      {
  isEqual,
  isObject
} = ObjectHelper,
      // Attributes used during creation that should not be compared
checkEqualityIgnore = {
  parent: 1,
  elementData: 1,
  ns: 1,
  syncOptions: 1
},
      makeCheckEqualityOptions = () => ({
  ignore: checkEqualityIgnore,
  refsFound: new Set()
}),
      isClass = {
  class: 1,
  className: 1,
  classname: 1
},
      simpleTypes = {
  bigint: 1,
  boolean: 1,
  function: 1,
  number: 1,
  // object
  string: 1,
  symbol: 1 // undefined

},
      // Attributes to ignore on sync
syncIgnoreAttributes = {
  tag: 1,
  html: 1,
  text: 1,
  children: 1,
  tooltip: 1,
  parent: 1,
  nextSibling: 1,
  ns: 1,
  reference: 1,
  elementData: 1,
  retainElement: 1,
  compareHtml: 1,
  syncOptions: 1,
  listeners: 1,
  isReleased: 1,
  null: 1,
  '': 1,
  keepChildren: 1
};

const addAndCacheCls = (cls, lastDomConfig) => {
  const propertyName = 'className' in lastDomConfig ? 'className' : 'class',
        propertyValue = lastDomConfig[propertyName];

  if (propertyValue) {
    if (typeof propertyValue === 'string') {
      const value = propertyValue.split(' ');

      if (!value.includes(cls)) {
        value.push(cls);
        lastDomConfig[propertyName] = value.join(' ');
      }
    } else if (Array.isArray(propertyValue)) {
      if (!propertyValue.includes(cls)) {
        propertyValue.push(cls);
      }
    } else if (propertyValue.isDomClassList) {
      propertyValue.add(cls);
    } else if (ObjectHelper.isObject(propertyValue)) {
      propertyValue[cls] = 1;
    }
  }
};

const removeAndUncacheCls = (cls, lastDomConfig) => {
  const propertyName = 'className' in lastDomConfig ? 'className' : 'class',
        propertyValue = lastDomConfig[propertyName];

  if (propertyValue) {
    if (typeof propertyValue === 'string') {
      const value = propertyValue.split(' ');

      if (value.includes(cls)) {
        value.splice(value.indexOf(cls), 1);
        lastDomConfig[propertyName] = value.join(' ');
      }
    } else if (Array.isArray(propertyValue)) {
      if (propertyValue.includes(cls)) {
        propertyValue.splice(propertyValue.indexOf(cls), 1);
      }
    } else if (propertyValue.isDomClassList) {
      propertyValue.remove(cls);
    } else if (ObjectHelper.isObject(propertyValue)) {
      delete propertyValue[cls];
    }
  }
};
/**
 * A utility class for syncing DOM config objects to DOM elements. Syncing compares the new config with the previously
 * used for that element, only applying the difference. Very much like a virtual DOM approach on a per element basis
 * (element + its children).
 *
 * Usage example:
 *
 * ```javascript
 * DomSync.sync({
 *     domConfig: {
 *         className : 'b-outer',
 *         children : [
 *             {
 *                 className : 'b-child',
 *                 html      : 'Child 1',
 *                 dataset   : {
 *                     custom : true
 *                 }
 *             },
 *             {
 *                 className : 'b-child',
 *                 html      : 'Child 2',
 *                 style     : {
 *                     fontWeight : 'bold',
 *                     color      : 'blue'
 *                 }
 *             }
 *         ]
 *     },
 *     targetElement : target
 * });
 * ```
 */

class DomSync {
  /**
   * Compares two DOM configs or properties of such objects for equality.
   * @param {Object} is The new value.
   * @param {Object} was The old value.
   * @param {Object} options An object with various options to control the comparison.
   * @param {Object} options.ignore An object containing names of attributes to ignore having `true` value.
   * @param {Map} options.equalityCache A map that can be used to record equality results for objects to avoid
   * recomputing the result for the same objects.
   * @param {Set} options.refsFound A Set that must be populated with the values of any `reference` properties found.
   * @param {Boolean|String} [ignoreRefs] Pass `true` to ignore `reference` properties on domConfigs. Pass `'children'`
   * to ignore `reference` properties only on child element configs.
   * @returns {Boolean}
   * @private
   */
  static checkEquality(is, was, options, ignoreRefs) {
    if (is === was) {
      return true;
    } // For purposes of DomSync, null and undefined are equivalent

    if (is == null) {
      return was == null;
    }

    if (!is || !was) {
      return false; // false since is !== was and is != null (we get here if was == null)
    }

    const typeA = typeof is,
          typeB = typeof was;

    if (typeA !== typeB || simpleTypes[typeA]) {
      // only test simpleTypes[typeA] since typeA === typeB
      return false;
    } // a and b are distinct objects or maybe arrays

    let cache = options.equalityCache || (options.equalityCache = new Map()),
        ignoreChildRefs = Boolean(ignoreRefs),
        equal,
        i,
        ignore,
        ignoreRefOpt,
        key,
        syncOptions,
        val; // We must cache results based on both sides of the comparison. If we only cache the result of "is" vs
    // any other value, we get failures in SchedulerWithAutoCommitStore.t.js

    cache = cache.get(is) || cache.set(is, new Map()).get(is);
    equal = cache.get(was);

    if (equal === undefined) {
      equal = true;

      if (getPrototypeOf(is) !== getPrototypeOf(was) || is instanceof Node) {
        // Two Nodes are not equal since they are !==
        equal = false;
      } else if (Array.isArray(is)) {
        // Since we have === prototypes, we know that "was" is also an array
        i = is.length;

        if (i !== was.length) {
          equal = false;
        } else {
          while (i-- > 0) {
            if (!DomSync.checkEquality(is[i], was[i], options, ignoreChildRefs)) {
              equal = false;
              break;
            }
          }
        }
      } else {
        var _syncOptions;

        syncOptions = is.syncOptions;
        ignoreRefOpt = (_syncOptions = syncOptions) === null || _syncOptions === void 0 ? void 0 : _syncOptions.ignoreRefs;

        if (ignoreRefOpt) {
          ignoreChildRefs = true;
          ignoreRefs = ignoreRefOpt !== 'children';
        }

        ignore = options.ignore || emptyObject$9; // We have 2 objects w/same prototype and that are not HTML nodes
        // https://jsbench.me/n2kgfre1r5/1 - profiles for-in-object loops vs for loop over keys array
        // fwiw, the smaller the object, the greater the benefit of for-in loop

        for (key in was) {
          if (!ignore[key] && !(key in is) && !(ignoreRefs && key === 'reference')) {
            equal = false;
            break;
          }
        }

        if (equal) {
          if (toString.call(was) === '[object Date]') {
            // Since we have === prototypes, we know that "was" is also a Date
            equal = is.getTime() === was.getTime();
          } else {
            for (key in is) {
              if (!ignore[key] && !(ignoreRefs && key === 'reference')) {
                if (!(key in was)) {
                  equal = false;
                  break;
                }

                val = is[key]; // Per Johan:
                //  Not sure we still use DocumentFragment. Used to be part of event rendering earlier,
                //  but I think I have refactored it away. Worth checking
                //  ...
                //  Not finding any usages
                //
                // DocumentFragment, compare separately supplied html
                // if (key === 'html' && typeof val !== 'string' && `compareHtml` in is) {
                //     if (is.compareHtml === was.compareHtml) {
                //         continue;
                //     }
                // }

                if (!DomSync.checkEquality(val, was[key], options, ignoreChildRefs)) {
                  equal = false;
                  break;
                }
              }
            }
          }
        }
      }

      if (!ignoreRefs && isObject(is) && is.reference) {
        var _options$refsFound;

        // We need to track object w/reference properties to know what reference elements may have
        // been removed since we may skip synchronizing them.
        (_options$refsFound = options.refsFound) === null || _options$refsFound === void 0 ? void 0 : _options$refsFound.add(is.reference);
      }

      cache.set(was, equal);
    }

    return equal;
  }
  /**
   * Sync a DOM config to a target element
   * @param {Object} options Options object
   * @param {DomConfig} options.domConfig A DOM config object
   * @param {HTMLElement} options.targetElement Target element to apply to
   * @param {Boolean} [options.strict=false] Specify `true` to limit synchronization to only the values set by
   * previous calls. Styles and classes placed directly on the DOM elements by other means will not be affected.
   * @param {String} [options.syncIdField] Field in dataset to use to match elements for re-usage
   * @param {String|String[]} [options.affected] The references affected by a partial sync.
   * @param {Function} [options.callback] A function that will be called on element re-usage, creation and similar
   * @param {Boolean} [options.configEquality] A function that will be called to compare an incoming config to
   * the last config applied to the `targetElement`. This function returns `true` if the passed values are equal and
   * `false` otherwise.
   * @returns {HTMLElement} Returns the updated target element (which is also updated in place)
   */

  static sync(options) {
    const optionsIn = options,
          {
      refOwner
    } = options,
          refsWas = refOwner === null || refOwner === void 0 ? void 0 : refOwner.byRef,
          checkEqualityOptions = makeCheckEqualityOptions();
    let affected = options.affected,
        i,
        ref,
        targetNode,
        lastDomConfig;

    if (typeof affected === 'string') {
      affected = [affected];
    } // NOTE: it is possible to reenter this method in at least the following way:
    //   - sync() causes a focus change by manipulating the activeElement.
    //   - a focus/blur/focusin/out event causes a widget config to initiate a recompose.
    //   - the event also triggers code that forces the recompose to flush (e.g., by using a reference el).

    options = _objectSpread2(_objectSpread2({}, options), {}, {
      checkEqualityOptions
    });

    if (refOwner) {
      // We always rebuild the byRef map on each call
      refOwner.byRef = {};

      if (affected) {
        // We need to preserve all previously rendered refs that are not going to be affected by this partial
        // update...
        for (ref in refsWas) {
          if (!affected.includes(ref)) {
            refOwner.byRef[ref] = refsWas[ref];
          }
        }
      }

      options.refsWas = refsWas;
    } // performSync() returns false if nothing was done because the configs were equal... we bend the rules on
    // modifying input objects so we can return this potentially important detail to our caller:

    optionsIn.changed = DomSync.performSync(options, options.targetElement);

    if (refOwner) {
      if (!affected) {
        affected = Object.keys(refsWas);
      }

      for (i = 0; i < affected.length; ++i) {
        ref = affected[i];
        targetNode = refsWas[ref];

        if (checkEqualityOptions.refsFound.has(ref) || targetNode.retainElement) {
          refOwner.byRef[ref] = targetNode;
        } else {
          lastDomConfig = targetNode.lastDomConfig;
          targetNode.remove();
          refOwner.detachRef(ref, targetNode, lastDomConfig);
        }
      }
    }

    return options.targetElement;
  }

  static performSync(options, targetElement) {
    const {
      domConfig,
      callback
    } = options,
          {
      lastDomConfig
    } = targetElement,
          configIsEqual = options.configEquality || DomSync.checkEquality;

    if (!configIsEqual(domConfig, lastDomConfig, options.checkEqualityOptions, options.ignoreRefs)) {
      if (domConfig) {
        // Sync without affecting the containing element?
        if (!domConfig.onlyChildren) {
          DomSync.syncAttributes(domConfig, targetElement, options);
          DomSync.syncContent(domConfig, targetElement);
        }

        if (!domConfig.keepChildren) {
          DomSync.syncChildren(options, targetElement);
        }
      } // Allow null to clear html
      else {
        targetElement.innerHTML = null;
        targetElement.syncIdMap = null;
      } // Cache the config on the target for future comparison

      targetElement.lastDomConfig = !(domConfig !== null && domConfig !== void 0 && domConfig.onlyChildren && lastDomConfig) ? domConfig : _objectSpread2(_objectSpread2({}, lastDomConfig), {}, {
        children: domConfig.children
      });
      return true;
    } else {
      // Sync took no action, notify the world
      callback === null || callback === void 0 ? void 0 : callback({
        action: 'none',
        domConfig,
        targetElement
      });
    }

    return false;
  } //region Attributes

  static syncDataset(domConfig, targetElement) {
    const {
      lastDomConfig
    } = targetElement,
          sameConfig = domConfig === lastDomConfig,
          source = Object.keys(domConfig.dataset),
          target = lastDomConfig && lastDomConfig.dataset && Object.keys(lastDomConfig.dataset),
          delta = ArrayHelper.delta(source, target);
    let attr, i, name, value; // New attributes in dataset

    for (i = 0; i < delta.onlyInA.length; i++) {
      attr = delta.onlyInA[i];
      value = domConfig.dataset[attr]; // Prevent data-property="null" or data-property="undefined"

      if (value != null) {
        targetElement.setAttribute(`data-${StringHelper.hyphenate(attr)}`, value);
      }
    } // Might have changed

    for (i = 0; i < delta.inBoth.length; i++) {
      attr = delta.inBoth[i];
      value = domConfig.dataset[attr]; // Intentional != since dataset is always string but want numbers to match
      // noinspection EqualityComparisonWithCoercionJS

      if (sameConfig || value != lastDomConfig.dataset[attr]) {
        name = `data-${StringHelper.hyphenate(attr)}`;

        if (value == null) {
          targetElement.removeAttribute(name);
        } else {
          targetElement.setAttribute(name, value);
        }
      }
    } // Removed

    for (i = 0; i < delta.onlyInB.length; i++) {
      targetElement.removeAttribute(`data-${StringHelper.hyphenate(delta.onlyInB[i])}`);
    }
  }
  /**
   * Adds CSS classes to the element and to the cache.
   * @param {Core.helper.util.DomClassList|String|String[]|Object} cls
   * @param {HTMLElement} targetElement A previously DomSynced element
   * @internal
   */

  static addCls(cls, targetElement) {
    const {
      lastDomConfig
    } = targetElement;
    cls = DomClassList.normalize(cls, 'array');
    cls.forEach(cls => {
      targetElement.classList.add(cls);
      addAndCacheCls(cls, lastDomConfig);
    });
  }
  /**
   * Adds CSS classes from the element and from the cache.
   * @param {Core.helper.util.DomClassList|String|String[]|Object} cls
   * @param {HTMLElement} targetElement A previously DomSynced element
   * @internal
   */

  static removeCls(cls, targetElement) {
    const {
      lastDomConfig
    } = targetElement;
    cls = DomClassList.normalize(cls, 'array');
    cls.forEach(cls => {
      targetElement.classList.remove(cls);
      removeAndUncacheCls(cls, lastDomConfig);
    });
  }

  static syncClassList(domConfig, targetElement, lastDomConfig) {
    let cls = domConfig.className || domConfig.class,
        c,
        currentClasses,
        i,
        k,
        keep,
        last;

    if (lastDomConfig) {
      // NOTE: The following reads the DOM to determine classes that may have been added by other means. This
      //  diff is only enabled when "strict" is used (see our callers)
      currentClasses = DomClassList.normalize(targetElement, 'array');
      cls = DomClassList.normalize(cls, 'object');
      last = DomClassList.normalize(lastDomConfig.className || lastDomConfig.class, 'object');
      keep = [];

      for (i = 0, k = currentClasses.length; i < k; ++i) {
        c = currentClasses[i]; // We want to keep classes not in cls if we didn't add them last time

        if (cls[c] || !(c in last)) {
          last[c] = 1;
          keep.push(c);
        }
      }

      for (c in cls) {
        if (!last[c]) {
          keep.push(c);
        }
      }

      cls = keep.join(' ');
    } else {
      cls = DomClassList.normalize(cls); // to string
    }

    targetElement.setAttribute('class', cls);
  } // Attributes as map { attr : value, ... }

  static getSyncAttributes(domConfig) {
    const attributes = {},
          // Attribute names, simplifies comparisons and calls to set/removeAttribute
    names = []; // On a first sync, there are no domConfig on the target element yet

    if (domConfig) {
      Object.keys(domConfig).forEach(attr => {
        if (!syncIgnoreAttributes[attr]) {
          const name = attr.toLowerCase();
          attributes[name] = domConfig[attr];
          names.push(name);
        }
      });
    }

    return {
      attributes,
      names
    };
  }

  static syncAttributes(domConfig, targetElement, options) {
    const {
      lastDomConfig
    } = targetElement,
          // If the same config has come through, due to configEquality, we must update all attrs.
    sameConfig = domConfig === lastDomConfig,
          sourceSyncAttrs = DomSync.getSyncAttributes(domConfig),
          // Extract attributes from elements (sourceElement might be a config)
    {
      attributes: sourceAttributes,
      names: sourceNames
    } = sourceSyncAttrs,
          {
      attributes: targetAttributes,
      names: targetNames
    } = sameConfig ? sourceSyncAttrs : DomSync.getSyncAttributes(lastDomConfig),
          // Intersect arrays to determine what needs adding, removing and syncing
    {
      onlyInA: toAdd,
      onlyInB: toRemove,
      inBoth: toSync
    } = sameConfig ? {
      onlyInA: emptyArray$9,
      onlyInB: emptyArray$9,
      inBoth: sourceNames
    } : ArrayHelper.delta(sourceNames, targetNames);
    let attr, i; // Add new attributes

    for (i = 0; i < toAdd.length; i++) {
      attr = toAdd[i];
      const sourceAttr = sourceAttributes[attr]; // Style requires special handling

      if (attr === 'style' && sourceAttr != null) {
        // TODO: Do diff style apply also instead of this replace
        DomHelper.applyStyle(targetElement, sourceAttr, true);
      } // So does dataset
      else if (attr === 'dataset') {
        DomSync.syncDataset(domConfig, targetElement);
      } // And class, which might be an object
      else if (isClass[attr]) {
        DomSync.syncClassList(domConfig, targetElement);
      } // Other attributes are set using setAttribute (since it calls toString() DomClassList works fine),
      // unless they are undefined in which case they are ignored to not get `href="undefined"` etc
      else if (sourceAttr != null) {
        targetElement.setAttribute(attr, sourceAttr);
      }
    } // Removed no longer used attributes

    for (i = 0; i < toRemove.length; i++) {
      targetElement.removeAttribute(toRemove[i]);
    } // TODO: toAdd and toSync are growing very alike, consider merging
    // Sync values for all other attributes

    for (i = 0; i < toSync.length; i++) {
      attr = toSync[i];
      const sourceAttr = sourceAttributes[attr],
            targetAttr = targetAttributes[attr]; // Attribute value null means remove attribute

      if (sourceAttr == null) {
        targetElement.removeAttribute(attr);
      } // Set all attributes that has changed, with special handling for style.
      else if (attr === 'style') {
        if (options.strict) {
          if (sameConfig) {
            DomSync.syncStyles(targetElement, sourceAttr);
          } else if (!isEqual(sourceAttr, targetAttr, true)) {
            DomSync.syncStyles(targetElement, sourceAttr, targetAttr);
          }
        } else if (sameConfig || !isEqual(sourceAttr, targetAttr, true)) {
          DomHelper.applyStyle(targetElement, sourceAttr, true);
        }
      } // And dataset
      else if (attr === 'dataset') {
        DomSync.syncDataset(domConfig, targetElement);
      } // And class, which might be an object
      else if (isClass[attr]) {
        DomSync.syncClassList(domConfig, targetElement, options.strict && targetElement.lastDomConfig);
      } else if (sameConfig || sourceAttr !== targetAttr) {
        targetElement.setAttribute(attr, sourceAttr);
      }
    }
  }

  static syncStyles(targetElement, sourceAttr, targetAttr) {
    let styles, key, value;

    if (!targetAttr) {
      styles = sourceAttr;
    } else {
      styles = {}; // Style could be a string so we parse it to object to iterate over it's properties correctly

      sourceAttr = DomHelper.parseStyle(sourceAttr);
      targetAttr = DomHelper.parseStyle(targetAttr);

      if (sourceAttr) {
        for (key in sourceAttr) {
          value = sourceAttr[key];

          if (targetAttr[key] !== value) {
            styles[key] = value;
          }
        }
      }

      for (key in targetAttr) {
        if (!(key in sourceAttr)) {
          styles[key] = '';
        }
      }
    }

    DomHelper.applyStyle(targetElement, styles);
  } //endregion
  //region Content

  static syncContent(domConfig, targetElement) {
    const {
      html,
      text
    } = domConfig,
          content = text !== null && text !== void 0 ? text : html; // elementData holds custom data that we want to attach to the element (not visible in dom)

    if (domConfig.elementData) {
      targetElement.elementData = domConfig.elementData;
    } // Apply html from config

    if (content instanceof DocumentFragment) {
      // If given a DocumentFragment, replace content with it
      if (targetElement.childNodes.length === 1 && DomHelper.getChildElementCount(targetElement) === 0 && content.childNodes.length === 1 && DomHelper.getChildElementCount(content) === 0) {
        // Syncing a textNode to a textNode? Use shortcut
        DomHelper.setInnerText(targetElement, content.firstChild.data);
      } else {
        targetElement.innerHTML = '';
        targetElement.appendChild(content);
      }
    } // If content is likely html we set innerHTML
    else if (html != null && htmlRe.test(html)) {
      targetElement.innerHTML = String(html); // convert numbers to strings
    } else if (content != null) {
      var _lastDomConfig$childr;

      const {
        lastDomConfig
      } = targetElement; // If element had children or actual html content in last sync we force innerHTML to get rid of them

      if (lastDomConfig !== null && lastDomConfig !== void 0 && (_lastDomConfig$childr = lastDomConfig.children) !== null && _lastDomConfig$childr !== void 0 && _lastDomConfig$childr.length || lastDomConfig !== null && lastDomConfig !== void 0 && lastDomConfig.html && htmlRe.test(targetElement.lastDomConfig.html)) {
        targetElement.innerHTML = content;
      } // Otherwise we try to update the text as cheaply as possible (node data if possible)
      else {
        DomHelper.setInnerText(targetElement, String(content));
      }
    }
  }

  static insertTextNode(text, targetElement, callback, refOwner, beforeElement = null) {
    const newNode = document.createTextNode(text);
    targetElement.insertBefore(newNode, beforeElement);

    if (refOwner) {
      newNode.$refOwnerId = refOwner.id;
    }

    callback === null || callback === void 0 ? void 0 : callback({
      action: 'newNode',
      domConfig: text,
      targetElement: newNode
    });
  }

  static insertElement(domConfig, targetElement, targetNode, refOwner, syncIdMap, syncId, options) {
    var _options$callback;

    // Create a new element
    const newElement = options.ns ? document.createElementNS(options.ns, domConfig.tag || 'svg') : document.createElement(domConfig.tag || 'div'); // Insert (or append if no targetNode)

    targetElement.insertBefore(newElement, targetNode); // Sync to it

    DomSync.performSync(options, newElement);

    if (syncId != null) {
      syncIdMap[syncId] = newElement;
    } // ARIA. In the absence of a defined role or the elememnt being hidden from ARIA,
    // omit unfocusable elements from the accessibility tree.

    if (!domConfig.role && !domConfig['aria-hidden'] && !DomHelper.isFocusable(newElement, true) && !newElement.htmlFor) {
      newElement.setAttribute('role', 'presentation');
    }

    if (refOwner) {
      newElement.$refOwnerId = refOwner.id;

      if (syncId) {
        newElement.$reference = syncId;
        refOwner.attachRef(syncId, newElement, domConfig);
      }
    }

    (_options$callback = options.callback) === null || _options$callback === void 0 ? void 0 : _options$callback.call(options, {
      action: 'newElement',
      domConfig,
      targetElement: newElement,
      syncId
    });
  } //endregion
  //region Children

  static syncChildren(options, targetElement) {
    var _domConfig$html, _syncOptions$ignoreRe;

    let {
      // eslint-disable-next-line prefer-const
      domConfig,
      syncIdField,
      callback,
      releaseThreshold,
      configEquality,
      ns,
      refOwner,
      refsWas,
      strict,
      checkEqualityOptions,
      ignoreRefs
    } = options,
        syncOptions = domConfig.syncOptions || {},
        // eslint-disable-line prefer-const
    cleanupNodes = null,
        index,
        nextNode,
        syncId; // Having specified html or text replaces all inner content, no point in syncing

    if ((_domConfig$html = domConfig.html) !== null && _domConfig$html !== void 0 ? _domConfig$html : domConfig.text) {
      return;
    } // This will affect the whole subtree since this goes into syncChildOptions (we convert 'children' to true
    // here because we are only processing the children):

    ignoreRefs = Boolean((_syncOptions$ignoreRe = syncOptions.ignoreRefs) !== null && _syncOptions$ignoreRe !== void 0 ? _syncOptions$ignoreRe : ignoreRefs);

    if (ignoreRefs) {
      refOwner = refsWas = null;
    }

    if ('strict' in syncOptions) {
      strict = syncOptions.strict;
    }

    const // Always repopulate the map, since elements might get used by other syncId below
    newSyncIdMap = refOwner ? refOwner.byRef : {},
          sourceConfigs = arraySlice.call(domConfig.children || []),
          targetNodes = arraySlice.call(targetElement.childNodes),
          syncIdMap = refsWas || targetElement.syncIdMap || {},
          releasedIdMap = targetElement.releasedIdMap || {},
          nextTarget = remove => {
      // Recursive calls to performSync can teleport elements around the DOM tree (when we are given
      // the DOM nodes in the domConfig), so be sure to skip over any elements that are no longer children
      // of our targetElement
      while (targetNodes.length && targetNodes[0].parentNode !== targetElement) {
        targetNodes.shift();
      }

      return (remove ? targetNodes.shift() : targetNodes[0]) || null;
    }; // Each level can optionally specify its own syncIdField, strict and callback, if left out parent levels will be used

    syncIdField = syncOptions.syncIdField || syncIdField;
    strict = syncOptions.strict || strict;
    callback = syncOptions.callback || callback;
    configEquality = syncOptions.configEquality || configEquality; // Make sure releaseThreshold 0 is respected...

    releaseThreshold = 'releaseThreshold' in syncOptions ? syncOptions.releaseThreshold : releaseThreshold;

    if (syncIdField) {
      targetElement.syncIdMap = newSyncIdMap;
    } // Settings to use in all syncs below

    const syncChildOptions = {
      checkEqualityOptions: checkEqualityOptions || makeCheckEqualityOptions(),
      ignoreRefs,
      refOwner,
      refsWas,
      strict,
      syncIdField,
      releaseThreshold,
      callback,
      configEquality
    };

    while (sourceConfigs.length) {
      const sourceConfig = sourceConfigs.shift();
      syncId = null; // Allowing null, convenient when using Array.map() to generate children

      if (!sourceConfig) {
        continue;
      }

      if (sourceConfig instanceof Node) {
        nextNode = nextTarget(); // Widgets may supply the element of another widget in their rendering... just insert it and move on

        if (sourceConfig !== nextNode) {
          targetElement.insertBefore(sourceConfig, nextNode);
        }

        index = targetNodes.indexOf(sourceConfig);

        if (index > -1) {
          targetNodes.splice(index, 1);
        }

        continue;
      }

      const isTextNode = typeof sourceConfig === 'string'; // Used in all syncs

      syncChildOptions.domConfig = sourceConfig;
      syncChildOptions.ns = sourceConfig.ns || ns;

      if (!isTextNode) {
        // If syncIdField was supplied, we should first try to reuse element with
        // matching "id"
        if (refOwner) {
          syncId = sourceConfig.reference;
        } else if (syncIdField && sourceConfig.dataset) {
          syncId = sourceConfig.dataset[syncIdField];
        } // We have an id to look for

        if (syncId != null && !sourceConfig.unmatched) {
          // Find any matching element, either in use or previously released
          const syncTargetElement = syncIdMap[syncId] || releasedIdMap[syncId];

          if (syncTargetElement) {
            const {
              lastDomConfig
            } = syncTargetElement; // Just relink if flagged with `retainElement` (for example during dragging)

            if (syncTargetElement.retainElement) ; // Otherwise sync with the matched element
            else if (DomSync.performSync(syncChildOptions, syncTargetElement)) {
              var _callback;

              // Sync took some action, notify the world
              (_callback = callback) === null || _callback === void 0 ? void 0 : _callback({
                action: 'reuseOwnElement',
                domConfig: sourceConfig,
                targetElement: syncTargetElement,
                lastDomConfig,
                syncId
              });
            } // Since it wont sync above when flagged to be retained, we need to apply the flag here

            if (sourceConfig.retainElement) {
              syncTargetElement.retainElement = true;
            } // Cache the element on the syncIdMap for faster retrieval later

            newSyncIdMap[syncId] = syncTargetElement; // Remove target from targetElements & release tracking, no-one else is allowed to sync with it

            ArrayHelper.remove(targetNodes, syncTargetElement);
            delete releasedIdMap[syncId];
            syncTargetElement.isReleased = false;
            nextNode = nextTarget();

            if (syncTargetElement.parentNode !== targetElement || strict && syncTargetElement.nextSibling !== nextNode) {
              targetElement.insertBefore(syncTargetElement, nextNode);
            }
          } else if (strict) {
            DomSync.insertElement(sourceConfig, targetElement, nextTarget(), refOwner, newSyncIdMap, syncId, syncChildOptions);
          } else {
            // No match, move to end of queue to not steal some one else's element
            sourceConfigs.push(sourceConfig); // Also flag as unmatched to know that when we reach this element again

            sourceConfig.unmatched = true;
          } // Node handled, carry on with next one

          continue;
        } // Avoid polluting the config object when done

        if (sourceConfig.unmatched) {
          delete sourceConfig.unmatched;
        }
      } // Skip over any retained elements

      let beforeNode = null,
          targetNode = null,
          cleanupNode;

      while (!targetNode && (cleanupNode = nextTarget(true))) {
        if (refOwner) {
          // When syncing for a refOwner, foreign elements are skipped.
          if (cleanupNode.$refOwnerId !== refOwner.id) {
            continue;
          }

          if (cleanupNode.$reference) {
            // In refOwner mode we always pass strict:true, so this won't happen... but if it did, the
            // idea is that ref els do not get cleaned up until the end of the sync process.
            if (!strict) {
              continue;
            } // Since we want to maintain DOM order, this ref el marks the spot where to insert. We also
            // don't want to put it into cleanupNodes (see above). We cannot reuse ref els.

            beforeNode = cleanupNode;
            break;
          } // The element is owned by this refOwner and not assigned a reference...
          // We can reuse it

          targetNode = cleanupNode;
        } else if (!cleanupNode.retainElement) {
          targetNode = cleanupNode;
        }

        if (!targetNode) {
          (cleanupNodes || (cleanupNodes = [])).push(cleanupNode);
        }
      }

      if (beforeNode || !targetNode) {
        if (isTextNode) {
          DomSync.insertTextNode(sourceConfig, targetElement, callback, refOwner, beforeNode);
        } else {
          // Will append if beforeNode === null
          DomSync.insertElement(sourceConfig, targetElement, beforeNode, refOwner, newSyncIdMap, syncId, syncChildOptions);
        }
      } // We have targets left
      else {
        // Matching element tag, sync it
        if (!isTextNode && targetNode.nodeType === Node.ELEMENT_NODE && (sourceConfig.tag || 'div').toLowerCase() === targetNode.tagName.toLowerCase()) {
          var _lastDomConfig$datase, _callback2;

          const {
            lastDomConfig
          } = targetNode,
                result = DomSync.performSync(syncChildOptions, targetNode); // Remove reused element from release tracking

          if (syncIdField && (lastDomConfig === null || lastDomConfig === void 0 ? void 0 : (_lastDomConfig$datase = lastDomConfig.dataset) === null || _lastDomConfig$datase === void 0 ? void 0 : _lastDomConfig$datase[syncIdField]) != null) {
            delete releasedIdMap[lastDomConfig.dataset[syncIdField]];
          }

          if (syncId != null) {
            newSyncIdMap[syncId] = targetNode;
          }

          targetNode.isReleased = false; // Only use callback if sync succeeded (anything changed)

          result && ((_callback2 = callback) === null || _callback2 === void 0 ? void 0 : _callback2({
            action: 'reuseElement',
            domConfig: sourceConfig,
            targetElement: targetNode,
            lastDomConfig,
            syncId
          }));
        } // Text node to text node, change text :)
        else if (isTextNode && targetNode.nodeType === Node.TEXT_NODE) {
          targetNode.data = sourceConfig; // Not using callback for updating text of node, have no usecase for it currently
        } // Not matching, replace it
        else {
          if (isTextNode) {
            DomSync.insertTextNode(sourceConfig, targetElement, callback, refOwner, targetNode);
          } else {
            // Will insert
            DomSync.insertElement(sourceConfig, targetElement, targetNode, refOwner, newSyncIdMap, syncId, syncChildOptions);
          }

          targetNode.remove();
        }
      }
    } // while (sourceConfigs.length)
    // Out of source nodes, remove remaining target nodes

    while (nextNode = nextTarget(true)) {
      // Any remaining nodes that belong to this refOwner need to be cleaned up. If
      // they have an assigned reference, however, they will be handled at the very
      // end of the sync process since those elements can move in the node hierarchy.
      if (!refOwner || nextNode.$refOwnerId === refOwner.id && !nextNode.$reference) {
        (cleanupNodes || (cleanupNodes = [])).push(nextNode);
      }
    }

    if (cleanupNodes) {
      DomSync.syncChildrenCleanup(targetElement, cleanupNodes, newSyncIdMap, callback, refOwner, releaseThreshold, syncIdField);
    }
  }

  static syncChildrenCleanup(targetElement, cleanupNodes, newSyncIdMap, callback, refOwner, releaseThreshold, syncIdField) {
    let releaseCount = 0,
        ref;

    for (const targetNode of cleanupNodes) {
      const {
        lastDomConfig
      } = targetNode; // Element might be retained, hands off (for example while dragging)

      if (!targetNode.retainElement) {
        // When using syncId to reuse elements, "release" left over elements instead of removing them, up to a
        // limit specified as releaseThreshold, above which elements are removed instead
        if (!refOwner && syncIdField && (releaseThreshold == null || releaseCount < releaseThreshold)) {
          // Prevent releasing already released element
          if (!targetNode.isReleased) {
            targetNode.className = 'b-released';
            targetNode.isReleased = true; // Store released element in syncIdMap, to facilitate reusing it for self later

            if (lastDomConfig !== null && lastDomConfig !== void 0 && lastDomConfig.dataset) {
              if (!targetElement.releasedIdMap) {
                targetElement.releasedIdMap = {};
              }

              targetElement.releasedIdMap[lastDomConfig.dataset[syncIdField]] = targetNode;
            }

            callback === null || callback === void 0 ? void 0 : callback({
              action: 'releaseElement',
              domConfig: lastDomConfig,
              lastDomConfig,
              targetElement: targetNode
            }); // Done after callback on purpose, to allow checking old className

            if (lastDomConfig) {
              // Make sure lastDomConfig differs even from the same domConfig applied again
              // Do not want to discard it completely since it is needed for diff when reused later
              lastDomConfig.isReleased = true; // To force reapply of classes on reuse

              if (lastDomConfig.className) {
                lastDomConfig.className = 'b-released';
              }

              if (lastDomConfig.class) {
                lastDomConfig.class = 'b-released';
              } // Same for style
              // (for elements positioned using style, when moved in a non DomSync way, aka EventDrag)

              if (lastDomConfig.style) {
                lastDomConfig.style = null;
              }
            }
          }

          releaseCount++;
        } // In normal sync mode, remove left overs
        else {
          targetNode.remove();

          if (refOwner) {
            ref = targetNode.$reference;

            if (ref) {
              refOwner.detachRef(ref, targetNode, lastDomConfig);
            }
          } // Remove from "release tracking"

          if (targetElement.releasedIdMap && syncIdField && lastDomConfig !== null && lastDomConfig !== void 0 && lastDomConfig.dataset) {
            delete targetElement.releasedIdMap[lastDomConfig.dataset[syncIdField]];
          }

          callback === null || callback === void 0 ? void 0 : callback({
            action: 'removeElement',
            domConfig: targetNode.lastDomConfig,
            lastDomConfig: targetNode.lastDomConfig,
            targetElement: targetNode
          });
        }
      } else if (syncIdField) {
        // Keep retained element in map
        if (lastDomConfig) {
          newSyncIdMap[targetNode.dataset[syncIdField]] = targetNode;
        }
      }
    }
  }
  /**
   * Remove a child element without syncing, for example when dragging an element to some other parent.
   * Removes it both from DOM and the parent elements syncMap
   * @param {HTMLElement} parentElement
   * @param {HTMLElement} childElement
   */

  static removeChild(parentElement, childElement) {
    if (parentElement.contains(childElement)) {
      const syncIdMap = parentElement.syncIdMap;

      if (syncIdMap) {
        const index = Object.values(syncIdMap).indexOf(childElement);

        if (index > -1) {
          delete syncIdMap[Object.keys(syncIdMap)[index]];
        }
      }

      parentElement.removeChild(childElement);
    }
  }
  /**
   * Adds a child element without syncing, making it properly available for later syncs. Useful for example
   * when dragging and dropping an element from some other parent.
   * @param {HTMLElement} parentElement
   * @param {HTMLElement} childElement
   * @param {String|Number} syncId
   */

  static addChild(parentElement, childElement, syncId) {
    parentElement.appendChild(childElement);

    if (!parentElement.syncIdMap) {
      parentElement.syncIdMap = {};
    }

    parentElement.syncIdMap[syncId] = childElement;
  }
  /**
   * Get a child element using a dot separated syncIdMap path.
   *
   * ```javascript
   * DomSync.getChild(eventWrap, 'event.percentBar');
   * ```
   *
   * @param {HTMLElement} element "root" element, under which the path starts
   * @param {String} path Dot '.' separated path of syncIdMap entries
   * @returns {HTMLElement} Child element or `null` if path did not match any element
   */

  static getChild(element, path) {
    const syncIds = String(path).split('.');

    for (const id of syncIds) {
      var _element, _element$syncIdMap;

      element = (_element = element) === null || _element === void 0 ? void 0 : (_element$syncIdMap = _element.syncIdMap) === null || _element$syncIdMap === void 0 ? void 0 : _element$syncIdMap[id];

      if (!element) {
        return null;
      }
    }

    return element;
  } //endregion

}
DomSync._$name = 'DomSync';

/**
 * @module Core/helper/util/Fullscreen
 */

/**
 * Encapsulates the functionality related to switching cross-browser to full screen view and back.
 */

class Fullscreen {
  static init() {
    const fnNames = ['fullscreenEnabled', 'requestFullscreen', 'exitFullscreen', 'fullscreenElement'],
          // turns fnNames into function calls to prefixed functions, fullscreenEnabled -> document.mozFullscreenEnabled
    prefixFn = prefix => fnNames.map(fn => {
      let result = prefix + StringHelper.capitalize(fn); // fullscreenEnabled in Firefox is called fullScreenEnabled

      if (prefix === 'moz') {
        result = result.replace('screen', 'Screen'); // #6555 - Crash when clicking full screen button twice
        // firefox doesn't support exitFullScreen method

        if ('mozCancelFullScreen' in document && fn === 'exitFullscreen') {
          result = 'mozCancelFullScreen';
        }
      }

      return result;
    });

    this.functions = 'fullscreenEnabled' in document && fnNames || 'webkitFullscreenEnabled' in document && prefixFn('webkit') || 'mozFullScreenEnabled' in document && prefixFn('moz') || 'msFullscreenEnabled' in document && prefixFn('ms') || [];

    const eventNames = ['fullscreenchange', 'fullscreenerror'],
          msEventNames = ['MSFullscreenChange', 'MSFullscreenError'],
          prefixEvt = prefix => eventNames.map(eventName => prefix + StringHelper.capitalize(eventName));

    this.events = 'fullscreenEnabled' in document && eventNames || 'webkitFullscreenEnabled' in document && prefixEvt('webkit') || 'mozFullscreenEnabled' in document && prefixEvt('moz') || 'msFullscreenEnabled' in document && msEventNames || [];
  }
  /**
   * True if the fullscreen mode is supported and enabled, false otherwise
   * @property {Boolean}
   */

  static get enabled() {
    return document[this.functions[0]];
  }
  /**
   * Request entering the fullscreen mode.
   * @param {HTMLElement} element Element to be displayed fullscreen
   * @returns {Promise} A Promise which is resolved with a value of undefined when the transition to full screen is complete.
   */

  static request(element) {
    return element[this.functions[1]]();
  }
  /**
   * Exit the previously entered fullscreen mode.
   * @returns {Promise} A Promise which is resolved once the user agent has finished exiting full-screen mode
   */

  static exit() {
    return document[this.functions[2]]();
  }
  /**
   * True if fullscreen mode is currently active, false otherwise
   * @property {Boolean}
   */

  static get isFullscreen() {
    return !!this.element;
  }

  static get element() {
    return document[this.functions[3]];
  }
  /**
   * Installs the passed listener to fullscreenchange event
   * @param {Function} fn The listener to install
   */

  static onFullscreenChange(fn) {
    document.addEventListener(this.events[0], fn);
  }
  /**
   * Uninstalls the passed listener from fullscreenchange event
   * @param {Function} fn
   */

  static unFullscreenChange(fn) {
    document.removeEventListener(this.events[0], fn);
  }

}
Fullscreen.init();
Fullscreen._$name = 'Fullscreen';

/* globals ResizeObserver: true */
let resizeFireTimer = null;

const resizedQueue = [],
      isAbsolutelyPositioned = n => {
  var _n$ownerDocument$defa;

  return n.nodeType === n.ELEMENT_NODE && ((_n$ownerDocument$defa = n.ownerDocument.defaultView) === null || _n$ownerDocument$defa === void 0 ? void 0 : _n$ownerDocument$defa.getComputedStyle(n).getPropertyValue('position')) === 'absolute';
};
/**
 * @module Core/helper/ResizeMonitor
 */

/**
 * Allows size monitoring of elements (or optionally a Window instance).
 *
 * ```
 * ResizeMonitor.addResizeListener(
 *   myElement,
 *   element => {
 *      console.log(element, ' changed size');
 *   }
 * );
 * ```
 *
 * @internal
 */

class ResizeMonitor {
  /**
   * Adds a resize listener to the passed element which is called when the element
   * is resized by layout.
   * @param {HTMLElement} element The element to listen for resizing.
   * @param {Function} handler The handling function. Will be passed the element.
   */
  static addResizeListener(element, handler) {
    const me = this;

    if (element === document || element === globalThis) {
      element = document.body;
    }

    if (element.nodeType === element.DOCUMENT_FRAGMENT_NODE) {
      element = element.host;
    }

    if (!element.classList.contains('b-resize-monitored')) {
      element.classList.add('b-resize-monitored');
      element._bResizemonitor = {
        handlers: []
      };
    } // If we're looking at the document, or body, use a window resize listener.

    if (element === document.body) {
      if (!me.hasWindowResizeListener) {
        // Throttle the reaction to window resize to only fire once every 100ms
        globalThis.addEventListener('resize', me.onWindowResize);
        me.hasWindowResizeListener = true;
      }
    } // Regular element - use ResizeObserver by preference
    else if (globalThis.ResizeObserver) {
      if (!me.resizeObserver) {
        me.resizeObserver = new ResizeObserver(me.onElementResize);
      }

      me.resizeObserver.observe(element);
    } // Polyfill ResizeObserver
    else {
      element.classList.add('b-no-resizeobserver');
      const [monitors, expand, shrink] = DomHelper.createElement({
        parent: element,
        className: 'b-resize-monitors',
        children: [{
          className: 'b-resize-monitor-expand'
        }, {
          className: 'b-resize-monitor-shrink'
        }]
      }, {
        returnAll: true
      });
      expand.scrollLeft = expand.scrollTop = shrink.scrollLeft = shrink.scrollTop = 1000000;
      expand.addEventListener('scroll', me.onSizeMonitorScroll, true);
      shrink.addEventListener('scroll', me.onSizeMonitorScroll, true); // Also need to fake a resize-scroll on DOM mutation

      (handler.targetMutationMonitor = new MutationObserver(m => {
        const changedNodes = [];

        for (const mr of m) {
          if (mr.type === 'childList') {
            changedNodes.push.apply(changedNodes, mr.removedNodes);
            changedNodes.push.apply(changedNodes, mr.addedNodes);
          }
        } // If the changed nodes were absolutely positioned, then they won't
        // cause a resize, so return

        if (changedNodes.length > 0 && changedNodes.every(isAbsolutelyPositioned)) {
          return;
        } // We only want the size monitor listener to trigger, so this event must NOT bubble
        // to any application or other framework listeners.

        expand.dispatchEvent(new CustomEvent('scroll', {
          bubbles: false
        }));
      })).observe(element, {
        childList: true,
        subtree: true
      }); // store reference for easier cleanup later

      handler.monitorElement = monitors;
    }

    element._bResizemonitor.handlers.push(handler);
  }
  /**
   * Removes a resize listener from the passed element.
   * @param {HTMLElement} element The element to listen for resizing.
   * @param {Function} handler The handling function to remove.
   */

  static removeResizeListener(element, handler) {
    if (element) {
      if (element === document || element === globalThis) {
        element = document.body;
      }

      const resizeMonitor = element._bResizemonitor;
      let listenerCount = 0;

      if (resizeMonitor && resizeMonitor.handlers) {
        ArrayHelper.remove(resizeMonitor.handlers, handler); // See if we should unobserve the element

        listenerCount = resizeMonitor.handlers.length;
      } // Down to no listeners.

      if (!listenerCount) {
        element.classList.remove('b-resize-monitored');

        if (this.resizeObserver) {
          this.resizeObserver.unobserve(element);
        } // Remove the polyfill resize listeners
        else {
          // remove any added elements
          if (handler.monitorElement) {
            handler.monitorElement.remove();
            handler.monitorElement = null;
          } // remove the DOM mutation observer

          if (handler.targetMutationMonitor) {
            handler.targetMutationMonitor.disconnect();
          }
        }
      }
    }
  }

  static onElementResize(entries) {
    for (const resizeObserverEntry of entries) {
      const resizedElement = resizeObserverEntry.target,
            resizeMonitor = resizedElement._bResizemonitor,
            newRect = resizeObserverEntry.contentRect || resizedElement.getBoundingClientRect();

      if (resizedElement.offsetParent) {
        if (!resizeMonitor.rectangle || newRect.width !== resizeMonitor.rectangle.width || newRect.height !== resizeMonitor.rectangle.height) {
          const oldRect = resizeMonitor.rectangle;
          resizeMonitor.rectangle = newRect;

          for (const resizeHandler of resizeMonitor.handlers) {
            resizeHandler(resizedElement, oldRect, newRect);
          }
        }
      }
    }
  }

  static onSizeMonitorScroll(e) {
    var _document$body;

    // If no body exists or the element has gone, ignore the event; the listener will be removed automatically.
    if ((_document$body = document.body) !== null && _document$body !== void 0 && _document$body.contains(e.target)) {
      e.stopImmediatePropagation();
      const monitorNode = e.target.parentNode,
            resizedElement = monitorNode.parentNode,
            resizeMonitor = resizedElement._bResizemonitor,
            newRect = resizedElement.getBoundingClientRect();

      if (!resizeMonitor.rectangle || newRect.width !== resizeMonitor.rectangle.width || newRect.height !== resizeMonitor.rectangle.height) {
        resizedQueue.push([resizedElement, resizeMonitor.rectangle, newRect]);
        resizeMonitor.rectangle = newRect;

        if (!resizeFireTimer) {
          resizeFireTimer = requestAnimationFrame(ResizeMonitor.fireResizeEvents);
        }
      }

      monitorNode.firstChild.scrollLeft = monitorNode.firstChild.scrollTop = monitorNode.childNodes[1].scrollTop = monitorNode.childNodes[1].scrollLeft = 1000000;
    }
  }

  static onWindowResize(e) {
    const resizedElement = document.body,
          resizeMonitor = resizedElement._bResizemonitor,
          oldRect = resizeMonitor.rectangle;
    resizeMonitor.rectangle = document.documentElement.getBoundingClientRect();

    for (const resizeHandler of resizeMonitor.handlers) {
      resizeHandler(resizedElement, oldRect, resizeMonitor.rectangle);
    }
  }

  static fireResizeEvents() {
    for (const resizedEntry of resizedQueue) {
      for (const resizeHandler of resizedEntry[0]._bResizemonitor.handlers) {
        // Checking offsetParent to avoid resizing of elements which are not visible or exist in DOM
        if (resizedEntry[0].offsetParent) {
          resizeHandler.apply(this, resizedEntry);
        }
      }
    }

    resizeFireTimer = null;
    resizedQueue.length = 0;
  }

}
ResizeMonitor._$name = 'ResizeMonitor';

/**
 * @module Core/helper/util/Point
 */

/**
 * Encapsulates an X,Y coordinate point.
 * @extends Core/helper/util/Rectangle
 */

class Point extends Rectangle {
  /**
   * Creates a new Point encapsulating the event's page position.
   * @param {Event} event
   * @return {Core.helper.util.Point}
   * @typings ignore
   */
  static from(event) {
    const touchPoints = event.changedTouches;
    return new Point(touchPoints ? touchPoints[0].screenX : event.screenX, touchPoints ? touchPoints[0].screenY : event.pageY);
  }
  /**
   * Constructs a Point
   * @param x The X coordinate
   * @param y The Y coordinate
   */

  constructor(x, y) {
    super(x, y, 0, 0);
  }
  /**
   * Coerces this Point to be within the passed Rectangle. Translates it into the bounds.
   * @param {Core.helper.util.Rectangle} into The Rectangle into which to coerce this Point.
   */

  constrain(into) {
    this.x = Math.min(Math.max(this.x, into.x), into.right - 1);
    this.y = Math.min(Math.max(this.y, into.y), into.bottom - 1);
    return this;
  }

  toArray() {
    return [this.x, this.y];
  }

} // The Rectangle class uses the Point class, but cannot import it.
// TODO: find a better way of getting a reference to the Point class in Rectangle.
// #8224 - Gantt angular demo doesn't work in production

Object.getPrototypeOf(Point).Point = Point;
Point._$name = 'Point';

/**
 * @module Core/helper/EventHelper
 */

const touchProperties = ['clientX', 'clientY', 'pageX', 'pageY', 'screenX', 'screenY'],
      isOption = {
  element: 1,
  thisObj: 1,
  once: 1,
  delegate: 1,
  delay: 1,
  capture: 1,
  passive: 1,
  throttled: 1,
  autoDetach: 1,
  expires: 1
},
      returnTrueProp = {
  get: () => true
},
      normalizedKeyNames = {
  Spacebar: 'Space',
  Del: 'Delete',
  Esc: 'Escape',
  Left: 'ArrowLeft',
  Up: 'ArrowUp',
  Right: 'ArrowRight',
  Down: 'ArrowDown'
},
      ignoreModifierKeys = {
  Meta: 1,
  Control: 1,
  Alt: 1
},
      // Allow an unsteady finger to jiggle by this much
longpressMoveThreshold = 5,
      specialKeyRe = /^(ctrl|shift|alt|meta)$/;

function getAccessorFromPrototype(target, property) {
  let descriptor, accessor;

  while ((target = Object.getPrototypeOf(target)) && !accessor) {
    var _descriptor;

    descriptor = Object.getOwnPropertyDescriptor(target, property);
    accessor = (_descriptor = descriptor) === null || _descriptor === void 0 ? void 0 : _descriptor.get;
  }

  return accessor;
}
/**
 * Utility methods for dealing with Events, normalizing Touch/Pointer/Mouse events.
 */

class EventHelper {
  /**
   * DOM event to trigger name mapping.
   * @private
   */
  static normalizeEvent(event) {
    return ObjectHelper.copyPropertiesIf(event, event.changedTouches[0] || event.touches[0], touchProperties);
  }
  /**
   * Returns the `[x, y]` coordinates of the event in the viewport coordinate system.
   * @param {Event} event The event
   * @return {Number[]} The coordinate.
   */

  static getXY(event) {
    if (event.touches) {
      event = event.touches[0];
    }

    return [event.clientX, event.clientY];
  }
  /**
   * Returns the pixel distance between two mouse/touch/pointer events.
   * @param {Event} event1 The first event.
   * @param {Event} event2 The second event.
   * @return {Number} The distance in pixels between the two events.
   */

  static getDistanceBetween(event1, event2) {
    const xy1 = EH.getXY(event1),
          xy2 = EH.getXY(event2); // No point in moving this to Point. We are dealing only with number values here.

    return Math.sqrt(Math.pow(xy1[0] - xy2[0], 2) + Math.pow(xy1[1] - xy2[1], 2));
  }
  /**
   * Returns a {@link Core.helper.util.Point} which encapsulates the `pageX/Y` position of the event.
   * May be used in {@link Core.helper.util.Rectangle} events.
   * @param {Event} event A browser mouse/touch/pointer event.
   * @return {Core.helper.util.Point} The page point.
   */

  static getPagePoint(event) {
    return new Rectangle.Point(event.pageX, event.pageY);
  }
  /**
   * Returns a {@link Core.helper.util.Point} which encapsulates the `clientX/Y` position of the event.
   * May be used in {@link Core.helper.util.Rectangle} events.
   * @param {Event} event A browser mouse/touch/pointer event.
   * @return {Core.helper.util.Point} The page point.
   */

  static getClientPoint(event) {
    return new Rectangle.Point(event.clientX, event.clientY);
  }
  /**
   * Add a listener or listeners to an element
   * @param {HTMLElement} element The element to add a listener/listeners to.
   * @param {String|Object} eventName Either a string, being the name of the event to listen for,
   * or an options object containing event names and options as keys. See the options parameter
   * for details, or the {@link #function-on-static} method for details.
   * @param {Function} [handler] If the second parameter is a string event name, this is the handler function.
   * @param {Object} [options] If the second parameter is a string event name, this is the options.
   * @param {HTMLElement} options.element The element to add the listener to.
   * @param {Object} options.thisObj The default `this` reference for all handlers added in this call.
   * @param {Boolean} [options.autoDetach=true] The listeners are automatically removed when the `thisObj` is destroyed.
   * @param {String} [options.delegate] A CSS selector string which only fires the handler when the event takes place in a matching element.
   * @param {Boolean} [options.once] Specify as `true` to have the listener(s) removed upon first invocation.
   * @param {Number} [options.delay] The number of milliseconds to delay the handler call after the event fires:
   * @param {Number|Object} [options.expires] The listener only waits for a specified time before
   * being removed. The value may be a number or an object containing an expiry handler.
   * @param {Number} [options.expires.delay] How long to wait for the event for.
   * @param {String|Function} [options.expires.alt] The function to call when the listener expires
   * **without having been triggered**.
   * @returns {Function} A detacher function which removes all the listeners when called.
   */

  static addListener(element, eventName, handler, options) {
    if (element.nodeType) {
      // All separate params, element, eventName and handler
      if (typeof eventName === 'string') {
        options = Object.assign({
          element,
          [eventName]: handler
        }, options);
      } // element, options
      else {
        options = Object.assign({
          element
        }, eventName);
      }
    } // Just an options object passed
    else {
      options = element;
    }

    return EH.on(options);
  }
  /**
   * Adds a listener or listeners to an element.
   * all property names other than the options listed below are taken to be event names,
   * and the values as handler specs.
   *
   * A handler spec is usually a function reference or the name of a function in the `thisObj`
   * option.
   *
   * But a handler spec may also be an options object containing a `handler` property which is
   * the function or function name, and local options, including `element` and `thisObj`
   * which override the top level options.
   *
   *  Usage example
   *
   * ```javascript
   * construct(config) {
   *     super.construct(config);
   *
   *     // Add auto detaching event handlers to this Widget's reference elements
   *     EventHelper.on({
   *         element : this.iconElement,
   *         click   : '_handleIconClick',
   *         thisObj : this,
   *         contextmenu : {
   *             element : document,
   *             handler : '_handleDocumentContextMenu'
   *         }
   *     });
   * }
   *```
   *
   * The `click` handler on the `iconElement` calls `this._handleIconClick`.
   *
   * The `contextmenu` handler is added to the `document` element, but the `thisObj`
   * is defaulted in from the top `options` and calls `this._handleDocumentContextMenu`.
   *
   * Note that on touch devices, `dblclick` and `contextmenu` events are synthesized.
   * Synthesized events contain a `browserEvent` property containing the final triggering
   * event of the gesture. For example a synthesized `dblclick` event would contain a
   * `browserEvent` property which is the last `touchend` event. A synthetic `contextmenu`
   * event will contain a `browserEvent` property which the longstanding `touchstart` event.
   *
   * @param {Object} options The full listener specification.
   * @param {HTMLElement} options.element The element to add the listener to.
   * @param {Object} options.thisObj The default `this` reference for all handlers added in this call.
   * @param {Boolean} [options.autoDetach=true] The listeners are automatically removed when the `thisObj` is destroyed.
   * @param {String} [options.delegate] A CSS selector string which only fires the handler when the event takes place in a matching element.
   * @param {Boolean} [options.once] Specify as `true` to have the listener(s) removed upon first invocation.
   * @param {Number} [options.delay] The number of milliseconds to delay the handler call after the event fires.
   * @param {Number} [options.throttled] For rapidly repeating events (Such as `wheel` or `scroll` or `mousemove`)
   * this is the number of milliseconds to delay subsequent handler calls after first invocation which happens immediately.
   * @param {Number|Object} [options.expires] The listener only waits for a specified time before
   * being removed. The value may be a number or an object containing an expiry handler.
   * @param {Number} [options.expires.delay] How long to wait for the event for.
   * @param {String|Function} [options.expires.alt] The function to call when the listener expires
   * **without having been triggered**.
   * @returns {Function} A detacher function which removes all the listeners when called.
   */

  static on(options) {
    const element = options.element,
          thisObj = options.thisObj,
          handlerDetails = [];

    for (const eventName in options) {
      // Only treat it as an event name if it's not a supported option
      if (!isOption[eventName]) {
        let handlerSpec = options[eventName];

        if (typeof handlerSpec !== 'object') {
          handlerSpec = {
            handler: handlerSpec
          };
        }

        const targetElement = handlerSpec.element || element; // If we need to convert taphold to an emulated contextmenu, add a wrapping function
        // in addition to the contextmenu listener. Platforms may support mouse *and* touch.

        if (BrowserHelper.isTouchDevice && !BrowserHelper.isAndroid) {
          if (eventName === 'contextmenu') {
            handlerDetails.push(EH.addElementListener(targetElement, 'touchstart', {
              handler: EH.createContextMenuWrapper(handlerSpec.handler, handlerSpec.thisObj || thisObj)
            }, options));
          }
        } // Keep track of the real handlers added.
        // addElementLister returns [ element, eventName, addedFunction, capture ]

        handlerDetails.push(EH.addElementListener(targetElement, eventName, handlerSpec, options));
      }
    }

    const detacher = () => {
      for (let handlerSpec, i = 0; i < handlerDetails.length; i++) {
        handlerSpec = handlerDetails[i];
        EH.removeEventListener(handlerSpec[0], handlerSpec[1], handlerSpec[2]);
      }

      handlerDetails.length = 0;
    }; // { autoDetach : true, thisObj : scheduler } means remove all listeners when the scheduler dies.

    if (thisObj && options.autoDetach !== false) {
      thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, detacher, thisObj);
    }

    return detacher;
  }
  /**
   * Used internally to add a single event handler to an element.
   * @param {HTMLElement} element The element to add the handler to.
   * @param {String} eventName The name of the event to add a handler for.
   * @param {Function|String|Object} handlerSpec Either a function to call, or
   * the name of a function to call in the `thisObj`, or an object containing
   * the handler local options.
   * @param {Function|String} [handlerSpec.handler] Either a function to call, or
   * the name of a function to call in the `thisObj`.
   * @param {HTMLElement} [handlerSpec.element] Optionally a local element for the listener.
   * @param {Object} [handlerSpec.thisObj] A local `this` specification for the handler.
   * @param {Object} defaults The `options` parameter from the {@link #function-addListener-static} call.
   * @private
   */

  static addElementListener(element, eventName, handlerSpec, defaults) {
    const handler = EH.createHandler(element, eventName, handlerSpec, defaults),
          {
      spec
    } = handler,
          expires = handlerSpec.expires || defaults.expires,
          options = spec.capture != null || spec.passive != null ? {
      capture: spec.capture,
      passive: spec.passive
    } : undefined;
    element.addEventListener(eventName, handler, options);

    if (expires) {
      // Extract expires : { delay : 100, alt : 'onExpireFn' }
      const thisObj = handlerSpec.thisObj || defaults.thisObj,
            delayable = thisObj !== null && thisObj !== void 0 && thisObj.isDelayable ? thisObj : globalThis,
            {
        alt
      } = expires,
            delay = alt ? expires.delay : expires,
            {
        spec
      } = handler; // expires is not applied with other options in createHandler(), store it here

      spec.expires = expires;
      spec.timerId = delayable[typeof delay === 'number' ? 'setTimeout' : 'requestAnimationFrame'](() => {
        spec.timerId = null;
        EH.removeEventListener(element, eventName, handler); // If we make it here and the handler has not been called, invoke the alt handler

        if (alt && !handler.called) {
          (typeof alt === 'string' ? thisObj[alt] : alt).call(thisObj);
        }
      }, delay, `listener-timer-${performance.now()}`);
    }

    return [element, eventName, handler, options];
  }

  static fixEvent(event, isRTL) {
    var _event$target, _event$target2, _event$target3, _event$relatedTarget;

    // When we listen to event on document and get event which bubbled from shadow dom, reading its target would
    // return shadow root element. We need actual element which started the event
    if ((_event$target = event.target) !== null && _event$target !== void 0 && _event$target.shadowRoot && event.composedPath) {
      const targetElement = event.composedPath()[0],
            originalTarget = event.target; // Can there be an event which actually originated from custom element, not its shadow dom?

      if (event.target !== targetElement) {
        Object.defineProperty(event, 'target', {
          get: () => targetElement,
          configurable: true
        }); // Save original target just in case

        Object.defineProperty(event, 'originalTarget', {
          get: () => originalTarget,
          configurable: true
        });
      }
    }

    if (event.fixed) {
      return event;
    } // Only perform all this once as the event bubbles

    Object.defineProperty(event, 'fixed', returnTrueProp);
    const {
      type
    } = event; // Normalize key names

    if (type.startsWith('key')) {
      const normalizedKeyName = normalizedKeyNames[event.key];

      if (normalizedKeyName) {
        Object.defineProperty(event, 'key', {
          get: () => normalizedKeyName
        });
      } // Polyfill the code property for SPACE because it is not set for synthetic events.

      if (event.key === ' ' && !event.code) {
        Object.defineProperty(event, 'code', {
          get: () => 'Space'
        });
      }
    } // Sync OSX's meta key with the ctrl key. This will only happen on Mac platform.
    // It's read-only, so define a local property to return true for ctrlKey.

    if (event.metaKey && !event.ctrlKey) {
      Object.defineProperty(event, 'ctrlKey', returnTrueProp);
    } // if (isRTL && (type.startsWith('mouse') || type.startsWith('pointer') || type === 'click')) {
    //     event.nativePageX = event.pageX;
    //
    //     if (!Object.getOwnPropertyDescriptor(event, 'pageX')) {
    //         Object.defineProperties(event, {
    //             pageX : {
    //                 get : () => {
    //                     return document.body.offsetWidth - event.nativePageX;
    //                 }
    //             }
    //         });
    //     }
    // }
    // Chrome 78 has a bug where moving out of the left edge of an element can report a mousemove
    // with that element as the target, but offsetX as -1 and moving out the right edge can report
    // that the element is the target but an offset of the offsetWidth. Patch the event until they fix it.
    // https://bugs.chromium.org/p/chromium/issues/detail?id=1010528
    // Tested in EventHelper.js

    if (BrowserHelper.isChrome && event.target && 'offsetX' in event) {
      // Wrap reading `offsetX/Y` until this property is actually accessed in the code to avoid forced reflow.
      // To avoid getting infinite loop, property descriptor is looked up in the prototype chain. Should be
      // better than forcing DOM reflow on every single event.
      if (!Object.getOwnPropertyDescriptor(event, 'offsetX')) {
        Object.defineProperty(event, 'offsetX', {
          get: () => {
            const accessor = getAccessorFromPrototype(event, 'offsetX'),
                  offsetX = accessor.call(event),
                  {
              target
            } = event,
                  {
              offsetWidth
            } = target,
                  x = Math.min(Math.max(offsetX, 0), offsetWidth - 1);

            if (offsetX < 0 || offsetX >= offsetWidth) {
              return x;
            }

            return offsetX;
          }
        });
      }

      if (!Object.getOwnPropertyDescriptor(event, 'offsetY')) {
        Object.defineProperty(event, 'offsetY', {
          get: () => {
            const accessor = getAccessorFromPrototype(event, 'offsetY'),
                  offsetY = accessor.call(event),
                  {
              target
            } = event,
                  {
              offsetHeight
            } = target,
                  y = Math.min(Math.max(offsetY, 0), offsetHeight - 1);

            if (offsetY < 0 || offsetY >= offsetHeight) {
              return y;
            }

            return offsetY;
          }
        });
      }
    } // Firefox has a bug where it can report that the target is the #document when mouse is over a pseudo element

    if (((_event$target2 = event.target) === null || _event$target2 === void 0 ? void 0 : _event$target2.nodeType) === Element.DOCUMENT_NODE && 'clientX' in event) {
      const targetElement = DomHelper.elementFromPoint(event.clientX, event.clientY);
      Object.defineProperty(event, 'target', {
        get: () => targetElement
      });
    } // Firefox has a bug where it can report a textNode as an event target/relatedTarget.
    // We standardize this to report the parentElement.

    if (((_event$target3 = event.target) === null || _event$target3 === void 0 ? void 0 : _event$target3.nodeType) === Element.TEXT_NODE) {
      const targetElement = event.target.parentElement;
      Object.defineProperty(event, 'target', {
        get: () => targetElement
      });
    }

    if (((_event$relatedTarget = event.relatedTarget) === null || _event$relatedTarget === void 0 ? void 0 : _event$relatedTarget.nodeType) === Element.TEXT_NODE) {
      const relatedTargetElement = event.target.parentElement;
      Object.defineProperty(event, 'relatedTarget', {
        get: () => relatedTargetElement
      });
    } // If it's a touch event, move the positional details
    // of touches[0] up to the event.

    if (type.startsWith('touch') && event.touches.length) {
      this.normalizeEvent(event);
    }

    return event;
  }

  static createHandler(element, eventName, handlerSpec, defaults) {
    const delay = handlerSpec.delay || defaults.delay,
          throttled = handlerSpec.throttled || defaults.throttled,
          block = handlerSpec.block || defaults.block,
          once = 'once' in handlerSpec ? handlerSpec.once : defaults.once,
          capture = 'capture' in handlerSpec ? handlerSpec.capture : defaults.capture,
          passive = 'passive' in handlerSpec ? handlerSpec.passive : defaults.passive,
          delegate = handlerSpec.delegate || defaults.delegate,
          wrappedFn = handlerSpec.handler,
          thisObj = handlerSpec.thisObj || defaults.thisObj,
          {
      rtlSource = thisObj
    } = thisObj || {}; //Capture initial conditions in case of destruction of thisObj.
    // Destruction completely wipes the object.
    // Innermost level of wrapping which calls the user's handler.
    // Normalize the event cross-browser, and attempt to normalize touch events.

    let handler = (event, ...args) => {
      // When playing a demo using DemoBot, only handle synthetic events
      if (EH.playingDemo && event.isTrusted) {
        return;
      } // If the thisObj is already destroyed, we cannot call the function.
      // If in dev mode, warn the developer with a JS error.

      if (thisObj !== null && thisObj !== void 0 && thisObj.isDestroyed) {
        // eslint-disable-next-line
        return;
      } // Fix up events to handle various browser inconsistencies

      event = EH.fixEvent(event, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl); // Flag for the expiry timer

      handler.called = true;
      (typeof wrappedFn === 'string' ? thisObj[wrappedFn] : wrappedFn).call(thisObj, event, ...args);
    }; // Allow events to be blocked for a certain time

    if (block) {
      const wrappedFn = handler;
      let lastCallTime, lastTarget;

      handler = (e, ...args) => {
        const now = performance.now();

        if (!lastCallTime || e.target !== lastTarget || now - lastCallTime > block) {
          lastTarget = e.target;
          lastCallTime = now;
          wrappedFn(e, ...args);
        }
      };
    } // Go through options, each creates a new handler by wrapping the previous handler to implement the options.
    // Right now, we have delay. Note that it may be zero, so test != null

    if (delay != null) {
      const wrappedFn = handler,
            delayable = thisObj !== null && thisObj !== void 0 && thisObj.setTimeout ? thisObj : globalThis;

      handler = (...args) => {
        delayable.setTimeout(() => {
          wrappedFn(...args);
        }, delay);
      };
    } // If they specified the throttled option, wrap the handler in a createdThrottled
    // version. Allow the called to specify an alt function to call when the event
    // fires before the buffer time has expired.

    if (throttled != null) {
      let alt,
          buffer = throttled;

      if (throttled.buffer) {
        alt = e => {
          return throttled.alt.call(EH, EH.fixEvent(e, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl));
        };

        buffer = throttled.buffer;
      }

      if (thisObj !== null && thisObj !== void 0 && thisObj.isDelayable) {
        handler = thisObj.throttle(handler, {
          delay: buffer,
          throttled: alt
        });
      } else {
        handler = FunctionHelper.createThrottled(handler, buffer, thisObj, null, alt);
      }
    } // This must always be added late to be processed before delay so that the handler is removed immediately.
    // Note that we cant use native once because of our support for `delegate`, it would remove the listener even
    // when delegate does not match

    if (once) {
      const wrappedFn = handler;

      handler = (...args) => {
        EH.removeEventListener(element, eventName, handler);
        wrappedFn(...args);
      };
    } // This must be added last to be called first, once and delay should not act on wrong targets when configured
    // with a delegate

    if (delegate) {
      const wrappedFn = handler;

      handler = (event, ...args) => {
        var _event$target$closest;

        event = EH.fixEvent(event, rtlSource === null || rtlSource === void 0 ? void 0 : rtlSource.rtl); // delegate: '.b-field-trigger' only fires when click is in a matching el.
        // currentTarget becomes the delegate.
        // Maintainer: In Edge event.target can be an empty object for transitionend events

        const delegatedTarget = ((_event$target$closest = event.target.closest) === null || _event$target$closest === void 0 ? void 0 : _event$target$closest.call) && event.target.closest(delegate);

        if (!delegatedTarget) {
          return;
        } // Allow this to be redefined as it bubbles through listeners up the parentNode axis
        // which might have their own delegate settings.

        Object.defineProperty(event, 'currentTarget', {
          get: () => delegatedTarget,
          configurable: true
        });
        wrappedFn(event, ...args);
      };
    } // Only autoDetach here if there's a local thisObj is in the handlerSpec for this one listener.
    // If it's in the defaults, then the "on" method will handle it.

    if (handlerSpec.thisObj && handlerSpec.autoDetach !== false) {
      thisObj.doDestroy = FunctionHelper.createInterceptor(thisObj.doDestroy, () => EH.removeEventListener(element, eventName, handler), thisObj);
    }

    handler.spec = {
      delay,
      throttled,
      block,
      once,
      thisObj,
      capture,
      passive,
      delegate
    };
    return handler;
  }

  static removeEventListener(element, eventName, handler) {
    const {
      expires,
      timerId,
      thisObj,
      capture
    } = handler.spec; // Cancel outstanding expires.alt() call when removing the listener

    if (expires !== null && expires !== void 0 && expires.alt && timerId) {
      const delayable = thisObj !== null && thisObj !== void 0 && thisObj.isDelayable ? thisObj : globalThis;
      delayable[typeof expires.delay === 'number' ? 'clearTimeout' : 'cancelAnimationFrame'](timerId);
    }

    element.removeEventListener(eventName, handler, capture);
  }

  static onTransitionEnd({
    element,
    property,
    handler,
    mode = 'transition',
    duration = DomHelper[`get${mode === 'transition' ? 'Property' : ''}${StringHelper.capitalize(mode)}Duration`](element, property),
    thisObj = globalThis,
    args = [],
    runOnDestroy,
    timerSource
  }) {
    let timerId;
    timerSource = timerSource || (thisObj.isDelayable ? thisObj : globalThis);

    const callbackArgs = [element, property, ...args],
          doCallback = () => {
      detacher();

      if (!thisObj.isDestroyed) {
        if (thisObj.callback) {
          thisObj.callback(handler, thisObj, callbackArgs);
        } else {
          handler.apply(thisObj, callbackArgs);
        }
      }
    },
          detacher = EH.on({
      element,

      [`${mode}end`]({
        propertyName,
        target
      }) {
        if (propertyName === property && target === element) {
          if (timerId) {
            timerSource.clearTimeout(timerId);
            timerId = null;
          }

          doCallback();
        }
      }

    }); // If the transition has not signalled its end within duration + 50 milliseconds
    // then give up on it. Remove the listener and call the handler.

    if (duration != null) {
      timerId = timerSource.setTimeout(doCallback, duration + 50, 'onTransitionEnd', runOnDestroy);
    }

    return detacher;
  }

  static async waitForTransitionEnd(config) {
    return new Promise(resolve => {
      config.handler = resolve;
      EventHelper.onTransitionEnd(config);
    });
  }
  /**
   * Private function to wrap the passed function. The returned wrapper function to be used as
   * a `touchstart` handler which will call the passed function passing a fabricated `contextmenu`
   * event if there's no `touchend` or `touchmove` after a default of 400ms.
   * @param {String|Function} handler The handler to call.
   * @param {Object} thisObj The owner of the function.
   * @private
   */

  static createContextMenuWrapper(handler, me) {
    return event => {
      // Only attempt conversion to contextmenu if it's a single touch start.
      if (event.touches.length === 1) {
        const tapholdStartTouch = event.touches[0],
              // Dispatch a synthetic "contextmenu" event from the touchpoint in <longPressTime> milliseconds.
        tapholdTimer = setTimeout(() => {
          // Remove the gesture cancelling listeners
          touchMoveRemover();
          const contextmenuEvent = new MouseEvent('contextmenu', tapholdStartTouch);
          Object.defineProperty(contextmenuEvent, 'target', {
            get() {
              return tapholdStartTouch.target;
            }

          });

          if (typeof handler === 'string') {
            handler = me[handler];
          }

          contextmenuEvent.browserEvent = event; // Call the wrapped handler passing the fabricated contextmenu event

          handler.call(me, contextmenuEvent);
          EH.contextMenuTouchId = tapholdStartTouch.identifier;
        }, EH.longPressTime),
              // This is what gets called if the user moves their touchpoint,
        // or releases the touch before <longPressTime>ms is up
        onMoveOrPointerUp = ({
          clientX,
          clientY,
          type
        }) => {
          let cancel = type === 'touchend' || type === 'pointerup'; // for move events, check if we only moved a little

          if (!cancel) {
            const deltaX = Math.abs(clientX - tapholdStartTouch.clientX),
                  deltaY = Math.abs(clientY - tapholdStartTouch.clientY);
            cancel = deltaX >= longpressMoveThreshold || deltaY >= longpressMoveThreshold;
          }

          if (cancel) {
            EH.contextMenuTouchId = null;
            touchMoveRemover();
            clearTimeout(tapholdTimer);
          }
        },
              // Touchmove or touchend before that timer fires cancels the timer and removes these listeners.
        touchMoveRemover = EH.on({
          element: document,
          touchmove: onMoveOrPointerUp,
          touchend: onMoveOrPointerUp,
          pointermove: onMoveOrPointerUp,
          pointerup: onMoveOrPointerUp,
          capture: true
        });
      }
    };
  }
  /**
   * Private function to wrap the passed function. The returned wrapper function to be used as
   * a `touchend` handler which will call the passed function passing a fabricated `dblclick`
   * event if there is a `click` within 300ms.
   * @param {String|Function} handler The handler to call.
   * @param {Object} thisObj The owner of the function.
   * @private
   */

  static createDblClickWrapper(element, handler, me) {
    let startId, secondListenerDetacher, tapholdTimer;
    return () => {
      if (!secondListenerDetacher) {
        secondListenerDetacher = EH.on({
          element,
          // We only get here if a touchstart arrives within 300ms of a click
          touchstart: secondStart => {
            startId = secondStart.changedTouches[0].identifier; // Prevent zoom

            secondStart.preventDefault();
          },
          touchend: secondClick => {
            if (secondClick.changedTouches[0].identifier === startId) {
              secondClick.preventDefault();
              clearTimeout(tapholdTimer);
              startId = secondListenerDetacher = null;
              const targetRect = Rectangle.from(secondClick.changedTouches[0].target, null, true),
                    offsetX = secondClick.changedTouches[0].pageX - targetRect.x,
                    offsetY = secondClick.changedTouches[0].pageY - targetRect.y,
                    dblclickEventConfig = Object.assign({
                browserEvent: secondClick
              }, secondClick),
                    dblclickEvent = new MouseEvent('dblclick', dblclickEventConfig);
              Object.defineProperty(dblclickEvent, 'target', {
                get() {
                  return secondClick.target;
                }

              });
              Object.defineProperty(dblclickEvent, 'offsetX', {
                get() {
                  return offsetX;
                }

              });
              Object.defineProperty(dblclickEvent, 'offsetY', {
                get() {
                  return offsetY;
                }

              });

              if (typeof handler === 'string') {
                handler = me[handler];
              } // Call the wrapped handler passing the fabricated dblclick event

              handler.call(me, dblclickEvent);
            }
          },
          once: true
        }); // Cancel the second listener is there's no second click within <dblClickTime> milliseconds.

        tapholdTimer = setTimeout(() => {
          secondListenerDetacher();
          startId = secondListenerDetacher = null;
        }, EH.dblClickTime);
      }
    };
  }
  /**
   * Handles various inputs to figure out the name of the special key of the event.
   *
   * ```javascript
   * EventHelper.toSpecialKey('ctrl') // 'ctrlKey'
   * EventHelper.toSpecialKey(true)   // 'ctrlKey', default
   * EventHelper.toSpecialKey(false)  // false
   * EventHelper.toSpecialKey('foo')  // false
   * ```
   *
   * @param {*} value User input value to process.
   * @param {String} defaultValue Default value to fall back to if `true` value is passed.
   * @returns {Boolean|String} Returns `false` if provided value cannot be converted to special key and special key
   * name otherwise.
   * @internal
   */

  static toSpecialKey(value, defaultValue = 'ctrlKey') {
    let result = false;

    if (value === true) {
      result = defaultValue;
    } else if (typeof value === 'string') {
      value = value.toLowerCase();

      if (value.match(specialKeyRe)) {
        result = `${value}Key`;
      }
    }

    return result;
  }

}

_defineProperty(EventHelper, "eventNameMap", {
  mousedown: 'MouseDown',
  mouseup: 'MouseUp',
  click: 'Click',
  dblclick: 'DblClick',
  contextmenu: 'ContextMenu',
  mouseover: 'MouseOver',
  mouseout: 'MouseOut'
});

const EH = EventHelper;
/**
 * The time in milliseconds for a `taphold` gesture to trigger a `contextmenu` event.
 * @member {Number} [longPressTime=500]
 * @readonly
 * @static
 */

EH.longPressTime = 500;
/**
 * The time in milliseconds within which a second touch tap event triggers a `dblclick` event.
 * @member {Number} [dblClickTime=300]
 * @readonly
 * @static
 */

EH.dblClickTime = 300; // Flag body if last user action used keyboard, used for focus styling etc.

EH.on({
  element: document,
  capture: true,

  mousedown({
    target
  }) {
    if (!DomHelper.isTouchEvent) {
      var _rootEl$classList;

      const rootEl = DomHelper.getRootElement(target);
      DomHelper.usingKeyboard = false;
      (_rootEl$classList = rootEl.classList) === null || _rootEl$classList === void 0 ? void 0 : _rootEl$classList.remove('b-using-keyboard');
      DomHelper.removeClsGlobally(rootEl, 'b-using-keyboard');
    }
  },

  touchmove({
    changedTouches
  }) {
    var _rootEl$classList2;

    const target = changedTouches[0].target,
          rootEl = DomHelper.getRootElement(target);
    DomHelper.usingKeyboard = false;
    (_rootEl$classList2 = rootEl.classList) === null || _rootEl$classList2 === void 0 ? void 0 : _rootEl$classList2.remove('b-using-keyboard');
    DomHelper.removeClsGlobally(rootEl, 'b-using-keyboard');
  },

  keydown({
    target,
    key
  }) {
    if (!ignoreModifierKeys[key]) {
      DomHelper.usingKeyboard = true;
      const rootElement = DomHelper.getRootElement(target); // if shadow root, add to outer children (shadow root itself lacks a classList :( )

      if (rootElement.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
        Array.from(rootElement.children).forEach(node => {
          if (node.matches('.b-outer')) {
            node.classList.add('b-using-keyboard');
          }
        });
      } else {
        // document.body
        rootElement.classList.add('b-using-keyboard');
      }
    }
  }

}); // When dragging on a touch device, we need to prevent scrolling from happening.
// Dragging only starts on a touchmove event, by which time it's too late to preventDefault
// on the touchstart event which started it.
// To do this we need a capturing, non-passive touchmove listener at the document level so we can preventDefault.
// This is in lieu of a functioning touch-action style on iOS Safari. When that's fixed, this will not be needed.

if (BrowserHelper.isTouchDevice) {
  EH.on({
    element: document,
    touchmove: event => {
      // If we're touching a b-dragging event, then stop any panning by preventing default.
      if (event.target.closest('.b-dragging')) {
        event.preventDefault();
      }
    },
    passive: false,
    capture: true
  });
}

EventHelper._$name = 'EventHelper';

const {
  defineProperty: defineProperty$5
} = Reflect;
let performance$1;

if (BrowserHelper.isBrowserEnv) {
  performance$1 = globalThis.performance;
} else {
  performance$1 = {
    now() {
      return new Date().getTime();
    }

  };
}
/**
 * @module Core/mixin/Delayable
 */
// Global timeout collections for tests

let globalDelays = null;

if (VersionHelper.isTestEnv) {
  const bryntum = globalThis.bryntum || (globalThis.bryntum = {});
  globalDelays = bryntum.globalDelays = {
    timeouts: new Map(),
    intervals: new Map(),
    animationFrames: new Map(),

    isEmpty(includeIntervals = false) {
      return globalDelays.timeouts.size + globalDelays.animationFrames.size + (includeIntervals ? globalDelays.intervals.size : 0) === 0;
    }

  };
}

const
/**
 * Creates and returns a function that will call the user-supplied `fn`.
 *
 * @param {Core.mixin.Delayable} me
 * @param {Function} fn The user function to call when the timer fires.
 * @param {Function} wrapFn The function the user will call to start the timer.
 * @param {Object} options The invoke options.
 * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
 * @param {Object} [options.thisObj] The `this` reference for `fn`.
 * @returns {Function}
 * @private
 */
makeInvoker = (me, fn, wrapFn, options) => {
  const named = typeof fn === 'string',
        appendArgs = options && options.appendArgs || [],
        // The invoker fn is intended to be wired directly to native setTimeout/requestAnimationFrame/etc and so
  // it does not receive any arguments worth passing on to the user's fn. Those come from the original call
  // to the wrapFn.
  invoker = () => {
    wrapFn.timerId = null;
    wrapFn.lastCallTime = performance$1.now(); // Grab args now and null the stored args out (to avoid leaks):

    const args = wrapFn.args;
    wrapFn.args = null;

    if (named) {
      me[fn](...args, ...appendArgs);
    } else {
      fn.call(me, ...args, ...appendArgs);
    }

    wrapFn.called = true;
    ++wrapFn.calls;
  };

  if (options) {
    me = options.thisObj || me;
  } // We put most everything as properties on the wrapFn so that it can all be inspected in the debugger (unlike
  // closure variables) and expected in tests.

  wrapFn.lastCallTime = -9e9; // performance.now() = 0 at start...

  wrapFn.calls = 0;
  wrapFn.invoker = invoker;
  invoker.wrapFn = wrapFn;
  return invoker;
},

/**
 * Decorates the supported `wrapFn` with additional methods and an `isPending` readonly
 * property. These decorations are available to user code to help manage the scheduling
 * behavior of the buffered function.
 *
 * @param {Core.mixin.Delayable} me
 * @param {Function} wrapFn The function the user will call to start the timer.
 * @param {String} cancelFn The name of the function that will cancel a timer.
 * @returns {Function} The `wrapFn` is returned.
 * @private
 */
decorateWrapFn = (me, wrapFn, cancelFn = 'clearTimeout') => {
  wrapFn.cancel = () => {
    if (wrapFn.isPending) {
      me[cancelFn](wrapFn.timerId); // avoid leaks and cleanup:

      wrapFn.args = wrapFn.timerId = null;
    }
  };

  wrapFn.flush = () => {
    if (wrapFn.isPending) {
      me[cancelFn](wrapFn.timerId);
      wrapFn.timerId = null; // we don't call cancel() since it also sets args=null

      wrapFn.invoker();
    }
  };

  wrapFn.now = (...args) => {
    wrapFn.cancel();
    wrapFn.args = args;
    wrapFn.invoker();
  };

  wrapFn.resume = all => {
    const n = wrapFn.suspended;
    wrapFn.suspended = all || n < 1 ? 0 : n - 1;
  };

  wrapFn.suspend = () => {
    ++wrapFn.suspended;
  };

  wrapFn.immediate = false;
  wrapFn.suspended = 0;
  wrapFn.timerId = null;
  defineProperty$5(wrapFn, 'isPending', {
    get() {
      return wrapFn.timerId !== null;
    }

  });
  return wrapFn;
};
/**
 * Tracks setTimeout, setInterval and requestAnimationFrame calls and clears them on destroy.
 *
 * @example
 * someClass.setTimeout(() => console.log('hi'), 200);
 * someClass.setInterval(() => console.log('annoy'), 100);
 * // can also use named timeouts for easier tracking
 * someClass.setTimeout(() => console.log('named'), 300, 'named');
 * someClass.clearTimeout('named');
 *
 * @mixin
 */

var Delayable = (Target => class Delayable extends (Target || Base$1) {
  static get $name() {
    return 'Delayable';
  }

  static get declarable() {
    return [
    /**
     * This class property returns an object that specifies methods to wrap with configurable timer behaviors.
     *
     * It is used like so:
     * ```
     *  class Foo extends Delayable() {
     *      static get delayable() {
     *          return {
     *              expensiveMethod : 500
     *          };
     *      }
     *
     *      expensiveMethod() {
     *          this.things();
     *          this.moreThings();
     *          this.evenMoreThings();
     *      }
     *  }
     * ```
     * With the above in place, consider:
     * ```
     *  let instance = new Foo();
     *
     *  instance.expensiveMethod();
     * ```
     * Instead of the above code immediately calling the `expensiveMethod()`, it will start a timer that will
     * invoke the method 500ms later. Because `expensiveMethod()` is an instance method, each instance of `Foo`
     * will have its own timer.
     *
     * NOTE: Only instance methods are currently supported (i.e., only non-`static` methods).
     *
     * #### Options
     * The value of each key configures how the method will be scheduled. If the value is a number, it is
     * promoted to a config object of `type='buffer'` as in the following:
     * ```
     *  class Foo extends Delayable() {
     *      static get delayable() {
     *          return {
     *              expensiveMethod : {
     *                  type  : 'buffer',
     *                  delay : 500
     *              }
     *          };
     *      }
     *  });
     * ```
     * The `type` property of the config object must be one of three values. Other options can be provided
     * depending on the `type`:
     *
     *  - `buffer`<br>
     *    Other options:
     *     - `delay` (Number) : The number of milliseconds to wait before calling the underlying method. A
     *       value of 0 is equivalent to setting `immediate: true`.
     *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the buffer).
     *  - `raf` (short for "request animation frame")<br>
     *    Other options:
     *     - `cancelOutstanding` (Boolean) : Set to `true` to cancel any pending animation frame requests and
     *       schedule a new one on each call.
     *     - `immediate` (Boolean) : Set to `true` to call immediately.
     *  - `throttle`<br>
     *    Other options:
     *     - `delay` (Number) : The number of milliseconds to wait after each execution before another
     *       execution takes place. A value of 0 is equivalent to setting `immediate: true`.
     *     - `immediate` (Boolean) : Set to `true` to call immediately (effectively disabling the throttle).
     *
     * While `immediate: true` can be specified at the class level, it is more typical to set it on the
     * instance's method as described below.
     *
     * #### Delayable Method API
     * Delayable methods have a consistent API to manage their scheduling. This API is added to the methods
     * themselves.
     *
     * For example:
     * ```
     *  let instance = new Foo();
     *
     *  instance.expensiveMethod();         // schedule a call in 500ms
     *  instance.expensiveMethod.isPending; // true
     *  instance.expensiveMethod.cancel();
     *  instance.expensiveMethod.flush();
     *  instance.expensiveMethod.now();
     *
     *  instance.expensiveMethod.delay = 10;
     *  instance.expensiveMethod();         // schedule a call in 10ms
     * ```
     *
     * ##### `isPending` (Boolean, readonly)
     * This boolean property will be `true` if a call has been scheduled, and false otherwise.
     *
     * ##### `cancel()`
     * Cancels a pending call if one has been scheduled. Otherwise this method does nothing.
     *
     * ##### `flush()`
     * Cancels the timer and causes the pending call to execute immediately. If there is no pending call, this
     * method does nothing.
     *
     * ##### `now()`
     * Cancels the timer (if one is pending) and executes the method immediately. If there is no pending call,
     * this method will still call the underlying method.
     *
     * @static
     * @member {Object} delayable
     * @internal
     */
    'delayable'];
  }

  doDestroy() {
    const me = this; // Normally one would expect this call at the end of this method... but in this case we need to run cleanup
    // of this stuff after config nullification since those can trigger delayable method calls.

    super.doDestroy();

    if (me.timeoutIds) {
      me.timeoutIds.forEach((fn, id) => {
        if (typeof fn === 'function') {
          fn();
        }

        clearTimeout(id);

        if (globalDelays) {
          globalDelays.timeouts.delete(id);
        }
      });
      me.timeoutIds = null;
    }

    if (me.timeoutMap) {
      me.timeoutMap.forEach((name, id) => clearTimeout(id));
      me.timeoutMap = null;
    }

    if (me.intervalIds) {
      me.intervalIds.forEach(id => {
        clearInterval(id);

        if (globalDelays) {
          globalDelays.intervals.delete(id);
        }
      });
      me.intervalIds = null;
    }

    if (me.animationFrameIds) {
      me.animationFrameIds.forEach(id => {
        cancelAnimationFrame(id);

        if (globalDelays) {
          globalDelays.animationFrames.delete(id);
        }
      });
      me.animationFrameIds = null;
    }
  }
  /**
   * Check if a named timeout is active
   * @param name
   * @internal
   */

  hasTimeout(name) {
    return !!(this.timeoutMap && this.timeoutMap.has(name));
  }
  /**
   * Same as native setTimeout, but will be cleared automatically on destroy. If a propertyName is supplied it will
   * be used to store the timeout id.
   * @param {Object} timeoutSpec An object containing the details about that function, and the time delay.
   * @param {Function|String} timeoutSpec.fn The function to call, or name of function in this object to call. Used as the `name` parameter if a string.
   * @param {Number} timeoutSpec.delay The milliseconds to delay the call by.
   * @param {Object[]} timeoutSpec.args The arguments to pass.
   * @param {String} [timeoutSpec.name] The name under which to register the timer. Defaults to `fn.name`.
   * @param {Boolean} [timeoutSpec.runOnDestroy] Pass `true` if this function should be executed if the Delayable instance is destroyed while function is scheduled.
   * @param {Boolean} [timeoutSpec.cancelOutstanding] Pass `true` to cancel any outstanding invocation of the passed function.
   * @returns {Number}
   * @internal
   */

  setTimeout({
    fn,
    delay,
    name,
    runOnDestroy,
    cancelOutstanding,
    args
  }) {
    if (arguments.length > 1 || typeof arguments[0] === 'function') {
      [fn, delay, name, runOnDestroy] = arguments;
    }

    if (typeof fn === 'string') {
      name = fn;
    } else if (!name) {
      name = fn.name;
    }

    if (cancelOutstanding) {
      this.clearTimeout(name);
    }

    const me = this,
          timeoutIds = me.timeoutIds || (me.timeoutIds = new Map()),
          timeoutMap = me.timeoutMap || (me.timeoutMap = new Map()),
          timeoutId = setTimeout(() => {
      if (typeof fn === 'string') {
        fn = me[name];
      } // Cleanup before invocation in case fn throws

      timeoutIds && timeoutIds.delete(timeoutId);
      timeoutMap && timeoutMap.delete(name);
      globalDelays && globalDelays.timeouts.delete(timeoutId);
      fn.apply(me, args);
    }, delay);
    timeoutIds.set(timeoutId, runOnDestroy ? fn : true);

    if (globalDelays) {
      globalDelays.timeouts.set(timeoutId, {
        fn,
        delay,
        name
      });
    }

    if (name) {
      timeoutMap.set(name, timeoutId);
    }

    return timeoutId;
  }
  /**
   * clearTimeout wrapper, either call with timeout id as normal clearTimeout or with timeout name (if you specified
   * a name to setTimeout())
   * property to null.
   * @param {Number|String} idOrName timeout id or name
   * @internal
   */

  clearTimeout(idOrName) {
    let id = idOrName;

    if (typeof id === 'string') {
      if (this.timeoutMap) {
        id = this.timeoutMap.get(idOrName);
        this.timeoutMap.delete(idOrName);
      } else {
        return;
      }
    }

    clearTimeout(id);
    this.timeoutIds && this.timeoutIds.delete(id);
    globalDelays && globalDelays.timeouts.delete(id);
  }
  /**
   * clearInterval wrapper
   * @param {Number} id
   * @internal
   */

  clearInterval(id) {
    clearInterval(id);
    this.intervalIds && this.intervalIds.delete(id);
    globalDelays && globalDelays.intervals.delete(id);
  }
  /**
   * Same as native setInterval, but will be cleared automatically on destroy
   * @param fn
   * @param delay
   * @returns {Number}
   * @internal
   */

  setInterval(fn, delay) {
    const intervalId = setInterval(fn, delay);
    (this.intervalIds || (this.intervalIds = new Set())).add(intervalId);
    globalDelays && globalDelays.intervals.set(intervalId, {
      fn,
      delay
    });
    return intervalId;
  }
  /**
   * Relays to native requestAnimationFrame and adds to tracking to have call automatically canceled on destroy.
   * @param {Function} fn
   * @param {Object[]} [extraArgs] The argument list to append to those passed to the function.
   * @param {Object} [thisObj] `this` reference for the function.
   * @returns {Number}
   * @internal
   */

  requestAnimationFrame(fn, extraArgs = [], thisObj = this) {
    const animationFrameIds = this.animationFrameIds || (this.animationFrameIds = new Set()),
          frameId = requestAnimationFrame(() => {
      globalDelays && globalDelays.animationFrames.delete(frameId); // [dongriffin 2022-01-19] It was observed that we can still be called even though we issued the
      // cancelAnimationFrame call. Since delete() returns true if our frameId was present and is now
      // removed, we can tell that we haven't been cancelled before we call our fn:

      animationFrameIds.delete(frameId) && fn.apply(thisObj, extraArgs);
    });
    animationFrameIds.add(frameId);
    globalDelays && globalDelays.animationFrames.set(frameId, {
      fn,
      extraArgs,
      thisObj
    });
    return frameId;
  }
  /**
   * Creates a function which will execute once, on the next animation frame. However many time it is
   * called in one event run, it will only be scheduled to run once.
   * @param {Function|String} fn The function to call, or name of function in this object to call.
   * @param {Object[]} [args] The argument list to append to those passed to the function.
   * @param {Object} [thisObj] `this` reference for the function.
   * @param {Boolean} [cancelOutstanding] Cancel any outstanding queued invocation upon call.
   * @internal
   */

  createOnFrame(fn, args = [], thisObj = this, cancelOutstanding) {
    let rafId;

    const result = (...callArgs) => {
      // Cancel if outstanding if requested
      if (rafId && cancelOutstanding) {
        this.cancelAnimationFrame(rafId);
        rafId = null;
      }

      if (!rafId) {
        rafId = this.requestAnimationFrame(() => {
          if (typeof fn === 'string') {
            fn = thisObj[fn];
          }

          rafId = null;
          callArgs.push(...args);
          fn.apply(thisObj, callArgs);
        });
      }
    };

    result.cancel = () => this.cancelAnimationFrame(rafId);

    return result;
  }
  /**
   * Relays to native cancelAnimationFrame and removes from tracking.
   * @param {Number} handle
   * @internal
   */

  cancelAnimationFrame(handle) {
    var _this$animationFrameI, _globalDelays;

    cancelAnimationFrame(handle);
    (_this$animationFrameI = this.animationFrameIds) === null || _this$animationFrameI === void 0 ? void 0 : _this$animationFrameI.delete(handle);
    (_globalDelays = globalDelays) === null || _globalDelays === void 0 ? void 0 : _globalDelays.animationFrames.delete(handle);
  }

  async nextAnimationFrame() {
    return new Promise(resolve => this.requestAnimationFrame(resolve));
  }
  /**
   * Wraps a function with another function that delays it specified amount of time, repeated calls to the wrapper
   * resets delay.
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Object|Number} options The delay in milliseconds or an options object.
   * @param {Number} options.delay The delay in milliseconds.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Object} [options.thisObj] The `this` reference for the function.
   * @returns {Function} Wrapped function to call.
   * @internal
   */

  buffer(fn, options) {
    let delay = options;

    if (options && typeof options !== 'number') {
      // if (config object)
      delay = options.delay;
    } else {
      options = null;
    }

    const bufferWrapFn = (...params) => {
      if (bufferWrapFn.suspended) {
        return;
      }

      const {
        delay
      } = bufferWrapFn;
      bufferWrapFn.cancel();
      bufferWrapFn.called = false;
      bufferWrapFn.args = params; // If delay=0, the buffer has been disabled so always call immediately.

      if (bufferWrapFn.immediate || !delay) {
        invoker();
      } else {
        bufferWrapFn.timerId = this.setTimeout(invoker, delay);
      }
    },
          invoker = makeInvoker(this, fn, bufferWrapFn, options);

    bufferWrapFn.delay = delay;
    return decorateWrapFn(this, bufferWrapFn);
  }
  /**
   * Returns a function that when called will schedule a call to `fn` on the next animation frame.
   *
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Boolean|Object} [options] An options object or the `cancelOutstanding` boolean property of it.
   * @param {Boolean} [options.cancelOutstanding] Pass `true` to cancel any pending animation frame requests and
   * schedule a new one on each call to the returned function.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Object} [options.thisObj] The `this` reference for the function.
   * @returns {Function}
   * @internal
   */

  raf(fn, options) {
    // NOTE: This method is only intended for use with `delayable`. It has a signature that is compatible
    // with `buffer()` and `throttle()`. The name is 'raf' to make the following aesthetically pleasing:
    //
    //  class Foo extends Delayable() {
    //      static get delayable() {
    //          return {
    //              bar : 'raf'
    //          };
    //      }
    //  }
    let cancelOutstanding = options;

    if (options && typeof options !== 'boolean') {
      // if (config object)
      cancelOutstanding = options.cancelOutstanding;
    } else {
      options = null;
    }

    const rafWrapFn = (...params) => {
      if (rafWrapFn.suspended) {
        return;
      } // Reschedule the frame on each call if requested

      if (rafWrapFn.cancelOutstanding) {
        rafWrapFn.cancel();
      }

      rafWrapFn.called = false;
      rafWrapFn.args = params;

      if (rafWrapFn.immediate) {
        invoker();
      } else if (!rafWrapFn.isPending) {
        rafWrapFn.timerId = this.requestAnimationFrame(invoker);
      }
    },
          invoker = makeInvoker(this, fn, rafWrapFn, options);

    rafWrapFn.cancelOutstanding = cancelOutstanding;
    return decorateWrapFn(this, rafWrapFn, 'cancelAnimationFrame');
  }
  /**
   * Create a "debounced" function which will call on the "leading edge" of a timer period.
   * When first invoked will call immediately, but invocations after that inside its buffer
   * period will be rejected, and *one* invocation will be made after the buffer period has expired.
   *
   * This is useful for responding immediately to a first mousemove, but from then on, only
   * calling the action function on a regular timer while the mouse continues to move.
   *
   * @param {Function|String} fn The function to call. If this is a string, it is looked up as a method on `this`
   * instance (or `options.thisObj` instead, if provided).
   * @param {Number|Object} options The milliseconds to wait after each execution before another execution takes place
   * or a object containing options.
   * @param {Object} [options.thisObj] `this` reference for the function.
   * @param {Array} [options.appendArgs] The argument list to append to those passed to the function.
   * @param {Function|String} [options.throttled] A function to call when the invocation is delayed due to buffer
   * time not having expired. If this is a string, it is looked up as a method on `this` instance (or `options.thisObj`
   * instead, if provided). When called, the same arguments are passed as would have been passed to `fn`, including
   * any `options.appendArgs`.
   * @internal
   */

  throttle(fn, options) {
    let delay = options,
        throttled;

    if (options && typeof options !== 'number') {
      // if (config object)
      delay = options.delay;
      throttled = options.throttled;
    } else {
      options = null;
    }

    const me = this,
          throttleWrapFn = (...args) => {
      if (throttleWrapFn.suspended) {
        return;
      }

      const {
        delay
      } = throttleWrapFn,
            elapsed = performance$1.now() - throttleWrapFn.lastCallTime;
      throttleWrapFn.args = args; // If it's been more then the delay period since we invoked, we can call it now.
      // Setting delay=0 effectively disables the throttle (which is the goal)

      if (throttleWrapFn.immediate || elapsed >= delay) {
        me.clearTimeout(throttleWrapFn.timerId);
        invoker();
      } else {
        // Kick off a timer for the requested period.
        if (!throttleWrapFn.isPending) {
          throttleWrapFn.timerId = me.setTimeout(invoker, delay - elapsed);
          throttleWrapFn.called = false;
        }

        if (throttled) {
          // Args have to be stored on the wrapFn for the invoker to pick them up:
          throttled.wrapFn.args = args;
          throttled();
        }
      }
    },
          invoker = makeInvoker(me, fn, throttleWrapFn, options);

    throttleWrapFn.delay = delay;

    if (throttled) {
      // Make an invoker for this callback to handle thisObj and typeof=string etc (pass a dud wrapFn):
      throttled = makeInvoker(me, throttled, () => {}, options);
    }

    return decorateWrapFn(me, throttleWrapFn);
  }

  static setupDelayable(cls) {
    cls.setupDelayableMethods(cls.delayable);
  }
  /**
   * This method initializes the `delayable` methods on this class.
   * @param {Object} delayable The `delayable` property.
   * @param {Function} [cls] This parameter will be used internally to process static methods.
   * @private
   */

  static setupDelayableMethods(delayable, cls = null) {
    const me = this,
          statics = delayable.static,
          target = cls || me.prototype;

    if (statics) {
      // TODO me.setupDelayableMethods(statics, me);
      delete delayable.static;
    }

    for (const name in delayable) {
      let options = delayable[name];
      const implName = name + 'Now',
            type = typeof options;

      if (!target[implName]) {
        // Only move foo() -> fooNow() if a base class hasn't done so already
        target[implName] = target[name];
      }

      if (type === 'number') {
        options = {
          type: 'buffer',
          delay: options
        };
      } else if (type === 'string') {
        options = {
          type: options
        };
      } // For instance methods, we place a getter on the prototype. When the method is first accessed from the
      // prototype, we create an instance-specific version by calling this.buffer()/throttle() (based on the type
      // desired) and set that as the instance-level property.

      defineProperty$5(target, name, {
        get() {
          const value = this[options.type]((...params) => {
            this[implName](...params);
          }, options);
          defineProperty$5(this, name, {
            value
          });
          return value;
        }

      });
    }
  } // This does not need a className on Widgets.
  // Each *Class* which doesn't need 'b-' + constructor.name.toLowerCase() automatically adding
  // to the Widget it's mixed in to should implement thus.

  get widgetClass() {}

});

/**
 * @module Core/helper/IdHelper
 */
// Id generation should be on a per page basis, not per module

const idCounts$1 = ObjectHelper.getPathDefault(globalThis, 'bryntum.idCounts', Object.create(null));
/**
 * IdHelper provides unique ID generation.
 *
 * This class is not intended for application use, it is used internally by the Bryntum infrastructure.
 * @internal
 */

class IdHelper {
  /**
   * Generate a new id, using IdHelpers internal counter and a prefix
   * @param {String} prefix Id prefix
   * @returns {String} Generated id
   */
  static generateId(prefix = 'generatedId') {
    // This produces "b-foo-1, b-foo-2, ..." for each prefix independently of the others. In other words, it makes
    // id's more stable since the counter is on a per-class basis.
    return prefix + (idCounts$1[prefix] = (idCounts$1[prefix] || 0) + 1);
  }

}
IdHelper._$name = 'IdHelper';

/**
 * @module Core/helper/util/Scroller
 */

const scrollLiterals = {
  true: 'auto',
  false: 'hidden',
  'hidden-scroll': 'auto',
  clip: BrowserHelper.supportsOverflowClip ? 'clip' : 'hidden'
},
      scrollerCls = 'b-widget-scroller',
      defaultScrollOptions$5 = {
  block: 'nearest'
},
      immediatePromise$7 = Promise.resolve(),
      scrollPromise = element => new Promise(resolve => EventHelper.on({
  element,
  scroll: resolve,
  once: true
})),
      xAxis$1 = {
  x: 1
},
      isScrollable = {
  auto: 1,
  scroll: 1
},
      allScroll = {
  overflowX: 'auto',
  overflowY: 'auto'
};
/**
 * Encapsulates scroll functionality for a Widget. All requests for scrolling and scrolling information
 * must go through a Widget's {@link Core.widget.Widget#config-scrollable} property.
 * @mixes Core/mixin/Events
 * @mixes Core/mixin/Delayable
 * @extends Core/Base
 */

class Scroller extends Delayable(Events(Base$1)) {
  static get configurable() {
    return {
      /**
       * The widget which is to scroll.
       * @config {Core.widget.Widget}
       */
      widget: null,

      /**
       * The element which is to scroll. Defaults to the
       * {@link Core.widget.Widget#property-overflowElement} of the configured
       * {@link #config-widget}
       * @config {HTMLElement}
       */
      element: {
        $config: {
          nullify: true
        },
        value: null
      },

      /**
       * How to handle overflowing in the `X` axis.
       * May be:
       * * `'auto'`
       * * `'visible'`
       * * `'hidden'`
       * * `'scroll'`
       * * `'hidden-scroll'` Meaning scrollable from the UI but with no scrollbar,
       * for example a grid header. Only on platforms which support this feature.
       * * `true` - meaning `'auto'`
       * * `false` - meaning `'hidden'`
       * * `clip` - Uses `clip` where supported. Where not supported it uses
       * `hidden` and rolls back any detected scrolls in this dimension.
       * @config {String|Boolean}
       */
      overflowX: null,

      /**
       * How to handle overflowing in the `Y` axis.
       * May be:
       * * `'auto'`
       * * `'visible'`
       * * `'hidden'`
       * * `'scroll'`
       * * `'hidden-scroll'` Meaning scrollable from the UI but with no scrollbar.
       * Only on platforms which support this feature.
       * * `true` - meaning `'auto'`
       * * `false` - meaning `'hidden'`
       * * `clip` - Uses `clip` where supported. Where not supported it uses
       * `hidden` and rolls back any detected scrolls in this dimension.
       * @config {String|Boolean}
       */
      overflowY: null,

      /**
       * If configured as `true`, the {@link #config-element} is not scrolled
       * but is translated using CSS transform when controlled by this class's API.
       * Scroll events are fired when the element is translated.
       * @default
       * @config {Boolean}
       */
      translate: null,
      x: 0,
      y: 0,
      rtlSource: null
    };
  }

  static get delayable() {
    return {
      onScrollEnd: {
        type: 'buffer',
        delay: 100
      }
    };
  }
  /**
   * Fired when scrolling happens on this Scroller's element. The event object is a native `scroll` event
   * with the described extra properties injected.
   * @event scroll
   * @param {Core.widget.Widget} widget The owning Widget which has been scrolled.
   * @param {Core.helper.util.Scroller} source This Scroller
   */

  /**
   * Fired when scrolling finished on this Scroller's element. The event object is the last native `scroll` event
   * fires by the element with the described extra properties injected.
   * @event scrollend
   * @param {Core.widget.Widget} widget The owning Widget which has been scrolled.
   * @param {Core.helper.util.Scroller} source This Scroller
   */

  /**
   * The `overflow-x` setting for the widget. `true` means `'auto'`.
   * @member {Boolean|String} overflowX
   */

  /**
   * The `overflow-y` setting for the widget. `true` means `'auto'`.
   * @member {Boolean|String} overflowY
   */

  get isRTL() {
    var _this$rtlSource;

    return Boolean((_this$rtlSource = this.rtlSource) === null || _this$rtlSource === void 0 ? void 0 : _this$rtlSource.rtl);
  }

  syncOverflowState() {
    const {
      element
    } = this,
          classList = new DomClassList(element.classList),
          x = this.hasOverflowX = element.scrollWidth > element.clientWidth,
          y = this.hasOverflowY = element.scrollHeight > element.clientHeight;
    classList.value = element.classList; // We use classes to indicate presence of overflow. This carries no rules by default.
    // Widget SASS may or may not attach rules or use these to select elements.

    const changed = classList.toggle('b-horizontal-overflow', x) || classList.toggle('b-vertical-overflow', y);

    if (changed) {
      DomHelper.syncClassList(element, classList);

      if (!this.isConfiguring) {
        /**
         * Fired when either the X or the Y axis changes from not showing a space-consuming scrollbar
         * to showing a space-consuming scrollbar or vice-versa.
         *
         * *_Does not fire on platforms which show overlayed scrollbars_*
         * @event overflowChange
         * @param {Boolean} x `true` if the X axis overflow, `false` otherwise.
         * @param {Boolean} y `true` if the Y axis overflow, `false` otherwise.
         * @internal
         */
        this.trigger('overflowChange', {
          x,
          y
        });
      }
    }
  }
  /**
   * Returns `true` if there is overflow in the specified axis.
   * @param {String} [axis='y'] The axis to check scrollbar for. Note that this is subtly different to asking
   * whether an axis is showing a space-consuming scrollbar, see {@link #function-hasScrollbar}.
   * @internal
   */

  hasOverflow(axis = 'y') {
    // If there are no space-consuming scrollbars, we will not be recording overflow
    // state on change of scrollbars (There will be no resize event when overflow state changes).
    // Also, work around Firefox ResizeObserver bug: https://bugzilla.mozilla.org/show_bug.cgi?id=1733042
    // We don't get to update our scroll status when content size causes scrollbar change
    // so FF has to measure each time we're asked about overflow.
    // if (!DomHelper.scrollBarWidth || BrowserHelper.isFirefox) {
    const dimension = axis === 'y' ? 'Height' : 'Width';
    return this[`scroll${dimension}`] > this[`client${dimension}`]; // }
    // else {
    //     return this[`hasOverflow${axis.toUpperCase()}`];
    // }
  }
  /**
   * Returns `true` if there is a *space-consuming* scrollbar controlling scroll in the specified axis.
   * @param {String} [axis='y'] The axis to check scrollbar for. Note that this is subtly different to asking
   * whether an axis *has any* overflow, see {@link #function-hasOverflow}.
   * @internal
   */

  hasScrollbar(axis = 'y') {
    const {
      element
    } = this;

    if (element && DomHelper.scrollBarWidth) {
      const vertical = axis === 'y',
            dimension = vertical ? 'Width' : 'Height',
            clientSize = element[`client${dimension}`],
            borderSize = parseInt(DomHelper.getStyleValue(element, `border${vertical ? 'Left' : 'Top'}Width`)) + parseInt(DomHelper.getStyleValue(element, `border${vertical ? 'Right' : 'Bottom'}Width`)),
            difference = element[`offset${dimension}`] - borderSize - clientSize; // If the difference between the content width and the client width is
      // scrollBarWidth, then we have a scrollbar

      return Math.abs(difference - DomHelper.scrollBarWidth) < 2;
    }
  }
  /**
   * Partners this Scroller with the passed scroller in order to sync the scrolling position in the passed axes
   * @param {Core.helper.util.Scroller} otherScroller
   * @param {String|Object} [axes='x'] `'x'` or `'y'` or `{x: true/false, y: true/false}` axes to sync
   */

  addPartner(otherScroller, axes = xAxis$1) {
    const me = this;

    if (typeof axes === 'string') {
      axes = {
        [axes]: 1
      };
    }

    if (!me.partners) {
      me.partners = {};
    }

    me.partners[otherScroller.id] = {
      scroller: otherScroller,
      axes
    }; // Initial sync of the other axis to match our current state

    if (axes.x) {
      otherScroller.x = me.x;
    }

    if (axes.y) {
      otherScroller.y = me.y;
    } // It's a mutual relationship - the other scroller partners with us.

    if (!otherScroller.isPartneredWith(me)) {
      otherScroller.addPartner(me, axes);
    }
  }

  eachPartner(fn) {
    const {
      partners
    } = this;

    if (partners) {
      Object.values(partners).forEach(fn);
    }
  }
  /**
   * Breaks the link between this Scroller and the passed Scroller set up by the
   * {@link #function-addPartner} method.
   * @param {Core.helper.util.Scroller} otherScroller The Scroller to unlink from.
   */

  removePartner(otherScroller) {
    if (this.isPartneredWith(otherScroller)) {
      delete this.partners[otherScroller.id];
      otherScroller.removePartner(this);
    }
  }

  isPartneredWith(otherScroller) {
    return Boolean(this.partners && this.partners[otherScroller.id]);
  }
  /**
   * Breaks the link between this Scroller and all other Scrollers set up by the
   * {@link #function-addPartner} method.
   * @internal
   */

  clearPartners() {
    if (this.partners) {
      Object.values(this.partners).forEach(otherScroller => otherScroller.scroller.removePartner(this));
    }
  }
  /**
   * Scrolls the passed element or {@link Core.helper.util.Rectangle} into view according to the passed options.
   * @param {HTMLElement|Core.helper.util.Rectangle} element The element or a Rectangle in document space to scroll into view.
   * @param {Object} [options] How to scroll.
   * @param {String} [options.block] How far to scroll the element: `start/end/center/nearest`.
   * @param {Number} [options.edgeOffset] edgeOffset A margin around the element or rectangle to bring into view.
   * @param {Object|Boolean|Number} [options.animate] Set to `true` to animate the scroll by 300ms,
   * or the number of milliseconds to animate over, or an animation config object.
   * @param {Number} [options.animate.duration] The number of milliseconds to animate over.
   * @param {String} [options.animate.easing] The name of an easing function.
   * @param {Boolean|Function} [options.highlight] Set to `true` to highlight the element when it is in view.
   * May be a function which is called passing the element, to provide customized highlighting.
   * @param {Boolean} [options.focus] Set to `true` to focus the element when it is in view.
   * @param {Boolean} [options.x] Pass as `false` to disable scrolling in the `X` axis.
   * @param {Boolean} [options.y] Pass as `false` to disable scrolling in the `Y` axis.
   * @returns {Promise} A promise which is resolved when the element has been scrolled into view.
   */

  async scrollIntoView(element, options = defaultScrollOptions$5) {
    const me = this,
          {
      isRectangle
    } = element,
          originalRect = isRectangle ? element : Rectangle.from(element),
          {
      xDelta,
      yDelta
    } = me.getDeltaTo(element, options),
          result = me.scrollBy(xDelta, yDelta, options);

    if (options.highlight || options.focus) {
      result.then(() => {
        if (isRectangle) {
          element = originalRect.translate(-xDelta, -yDelta);
        }

        if (options.highlight) {
          // Not coercible to a number means its a function or name of a function
          if (isNaN(options.highlight)) {
            (me.widget || me).callback(options.highlight, null, [element]);
          } // Otherwise, it's truthy or falsy
          else {
            DomHelper.highlight(element, me);
          }
        }

        if (options.focus) {
          DomHelper.focusWithoutScrolling(element);
        }
      });
    }

    return result;
  }
  /**
   * Scrolls the passed element into view according to the passed options.
   * @param {HTMLElement} element The element in document space to scroll into view.
   * @param {Object} [options] How to scroll.
   * @param {String} [options.block] How far to scroll the element: `start/end/center/nearest`.
   * @param {Number} [options.edgeOffset] edgeOffset A margin around the element or rectangle to bring into view.
   * @param {Object|Boolean|Number} [options.animate] Set to `true` to animate the scroll by 300ms,
   * or the number of milliseconds to animate over, or an animation config object.
   * @param {Number} [options.animate.duration] The number of milliseconds to animate over.
   * @param {String} [options.animate.easing] The name of an easing function.
   * @param {Boolean} [options.highlight] Set to `true` to highlight the element when it is in view.
   * @param {Boolean} [options.focus] Set to `true` to focus the element when it is in view.
   * @param {Boolean} [options.x] Pass as `false` to disable scrolling in the `X` axis.
   * @param {Boolean} [options.y] Pass as `false` to disable scrolling in the `Y` axis.
   * @returns {Promise} A promise which is resolved when the element has been scrolled into view.
   */

  static async scrollIntoView(element, options = defaultScrollOptions$5, rtl = false) {
    const target = Rectangle.from(element),
          animate = typeof options === 'object' ? options.animate : options,
          scrollable = Scroller._globalScroller || (Scroller._globalScroller = new Scroller()),
          deltas = [];
    scrollable.rtlSource = {
      rtl
    };
    let totalX = 0,
        totalY = 0,
        result; // Build up all the scroll deltas necessary to bring the requested element into view

    for (let ancestor = element.parentNode; ancestor.nodeType === Node.ELEMENT_NODE; ancestor = ancestor.parentNode) {
      if (ancestor === document.body && ancestor !== document.scrollingElement) {
        continue;
      } // The <html> element, althouhgh it scrolls is overflow:visioble by default.

      const style = ancestor === document.scrollingElement ? allScroll : ancestor.ownerDocument.defaultView.getComputedStyle(ancestor); // If the ancestor overflows and scrolls in a dimension we are being asked to scroll in
      // Accumulate a scroll command for the ancestor.

      if (options.y !== false && isScrollable[style.overflowY] && ancestor.scrollHeight > ancestor.clientHeight || options.x !== false && isScrollable[style.overflowX] && ancestor.scrollWidth > ancestor.clientWidth) {
        // Global Scrollable
        scrollable.element = ancestor; // In case same element used as last time and didn't make it to the updater.

        scrollable.positionDirty = true; // See if the target is outside of this ancestor

        const {
          xDelta,
          yDelta
        } = scrollable.getDeltaTo(target, options);

        if (xDelta || yDelta) {
          deltas.push({
            element: ancestor,
            x: ancestor.scrollLeft,
            y: ancestor.scrollTop,
            xDelta,
            yDelta
          });
          target.translate(-xDelta, -yDelta);
          totalX += xDelta;
          totalY += yDelta;
        }
      }
    } // If scrolling was found to be necessary

    if (deltas.length) {
      const absX = Math.abs(totalX),
            absY = Math.abs(totalY);
      let duration = animate && (typeof animate === 'number' ? animate : typeof animate.duration === 'number' ? animate.duration : 300); // Only go through animation if there is significant scrolling to do.

      if (duration && (absX > 10 || absY > 10)) {
        // For small distances, constrain duration
        if (Math.max(absX, absY) < 50) {
          duration = Math.min(duration, 500);
        }

        result = scrollable.scrollAnimation = FunctionHelper.animate(duration, progress => {
          const isEnd = progress === 1;

          for (const {
            element,
            x,
            y,
            xDelta,
            yDelta
          } of deltas) {
            scrollable.element = element;

            if (xDelta) {
              scrollable.x = Math[rtl ? 'min' : 'max'](x + (isEnd ? xDelta : Math.round(xDelta * progress)), 0);
            }

            if (yDelta) {
              scrollable.y = Math.max(y + (isEnd ? yDelta : Math.round(yDelta * progress)), 0);
            }
          }
        }, null, animate.easing);
        result.then(() => {
          scrollable.scrollAnimation = null;
        });
      } // No animation
      else {
        for (const {
          element,
          xDelta,
          yDelta
        } of deltas) {
          element.scrollTop += yDelta;
          element.scrollLeft += xDelta;
        }

        result = scrollPromise(deltas[deltas.length - 1].element);
      }
    } else {
      result = immediatePromise$7;
    } // Postprocess element after scroll.

    if (options.highlight || options.focus) {
      result.then(() => {
        if (options.highlight) {
          // Not coercible to a number means its a function or name of a function
          if (isNaN(options.highlight)) {
            scrollable.callback(options.highlight, null, [element]);
          } // Otherwise, it's truthy or falsy
          else {
            DomHelper.highlight(element, scrollable);
          }
        }

        if (options.focus) {
          element.focus();
        }
      });
    }

    return result;
  }
  /**
   * Scrolls by the passed deltas according to the passed options.
   * @param {Number} [xDelta=0] How far to scroll in the X axis.
   * @param {Number} [yDelta=0] How far to scroll in the Y axis.
   * @param {Object|Boolean} [options] How to scroll. May be passed as `true` to animate.
   * @param {Boolean} [options.silent] Set to `true` to suspend `scroll` events during scrolling.
   * @param {Object|Boolean|Number} [options.animate] Set to `true` to animate the scroll by 300ms,
   * or the number of milliseconds to animate over, or an animation config object.
   * @param {Number} [options.animate.duration] The number of milliseconds to animate over.
   * @param {String} [options.animate.easing] The name of an easing function.
   * @returns {Promise} A promise which is resolved when the scrolling has finished.
   */

  async scrollBy(xDelta = 0, yDelta = 0, options = defaultScrollOptions$5) {
    const me = this,
          animate = typeof options === 'object' ? options.animate : options,
          absX = Math.abs(xDelta),
          absY = Math.abs(yDelta);

    if (me.scrollAnimation) {
      me.scrollAnimation.cancel();
      me.scrollAnimation = null;
    } // Only set the flag if there is going to be scrolling done.
    // It is cleared by the scrollEnd handler, so there must be scrolling.

    if (xDelta || yDelta) {
      me.silent = options.silent;
    }

    let duration = animate && (typeof animate === 'number' ? animate : typeof animate.duration === 'number' ? animate.duration : 300); // Only go through animation if there is significant scrolling to do.

    if (duration && (absX > 10 || absY > 10)) {
      const {
        x,
        y
      } = me;
      let lastX = x,
          lastY = y; // For small distances, constrain duration

      if (Math.max(absX, absY) < 50) {
        duration = Math.min(duration, 500);
      }

      me.scrollAnimation = FunctionHelper.animate(duration, progress => {
        const isEnd = progress === 1;

        if (xDelta) {
          // If the user, or another process has changed the position since last time, abort.
          // Unless called with the force option to proceed regardless.
          if (me.x !== lastX && !options.force) {
            return me.scrollAnimation && me.scrollAnimation.cancel();
          }

          me.x = Math.max(x + (isEnd ? xDelta : Math.round(xDelta * progress)), 0);
        }

        if (yDelta) {
          // If the user, or another process has changed the position since last time, abort.
          // Unless called with the force option to proceed regardless.
          if (me.y !== lastY && !options.force) {
            return me.scrollAnimation && me.scrollAnimation.cancel();
          }

          me.y = Math.max(y + (isEnd ? yDelta : Math.round(yDelta * progress)), 0);
        } // Store actual position from DOM

        lastX = me.x;
        lastY = me.y;
      }, me, animate.easing);
      me.element.classList.add('b-scrolling');
      me.scrollAnimation.then(() => {
        if (!me.isDestroyed) {
          me.element.classList.remove('b-scrolling');
          me.scrollAnimation = null;
        }
      });
      return me.scrollAnimation;
    } else {
      if (xDelta | yDelta) {
        me.x += xDelta;
        me.y += yDelta;
        return scrollPromise(me.element);
      }

      return immediatePromise$7;
    }
  }
  /**
   * Scrolls to the passed position according to the passed options.
   * @param {Number} [toX=0] Where to scroll to in the X axis.
   * @param {Number} [toY=0] Where to scroll to in the Y axis.
   * @param {Object|Boolean} [options] How to scroll. May be passed as `true` to animate.
   * @param {Object|Boolean|Number} [options.animate] Set to `true` to animate the scroll by 300ms,
   * or the number of milliseconds to animate over, or an animation config object.
   * @param {Number} [options.animate.duration] The number of milliseconds to animate over.
   * @param {String} [options.animate.easing] The name of an easing function.
   * @returns {Promise} A promise which is resolved when the scrolling has finished.
   */

  async scrollTo(toX, toY, options) {
    const {
      x,
      y
    } = this,
          xDelta = toX == null ? 0 : toX - x,
          yDelta = toY == null ? 0 : toY - y;
    return this.scrollBy(xDelta, yDelta, options);
  }

  doDestroy() {
    const me = this;

    if (me._element) {
      me._element.removeEventListener('scroll', me.scrollHandler);

      me.wheelListenerRemover && me.wheelListenerRemover();
    }

    if (me.scrollAnimation) {
      me.scrollAnimation.cancel();
    }

    Object.values(me.partners || {}).forEach(({
      scroller
    }) => scroller.removePartner(me));
    super.doDestroy();
  }
  /**
   * Respond to style changes to monitor scroll *when this Scroller is in `translate: true` mode.*
   * @param {Object[]} mutations The ElementMutation records.
   * @private
   */

  onElMutation(mutations) {
    const me = this,
          [x, y] = DomHelper.getTranslateXY(me.element); // If the mutation was due to a change in the translateX/Y styles, this is
    // a scroll event, so inform observers and partners

    if (me._x !== -x || me.y !== -y) {
      const scrollEvent = new CustomEvent('scroll', {
        bubbles: true
      });
      Object.defineProperty(scrollEvent, 'target', {
        get: () => me.element
      });
      me.onScroll(scrollEvent);
    }
  }

  onElResize() {
    var _this$owner;

    if ((_this$owner = this.owner) !== null && _this$owner !== void 0 && _this$owner.isAnimating) {
      this.owner.on({
        animationEnd: 'onElResize',
        thisObj: this,
        once: true
      });
    } else {
      this.syncOverflowState();
    }
  }

  onScroll(e) {
    const me = this,
          {
      _x,
      _y,
      element
    } = me;
    let vetoed = 0; // Until overflow:clip is 100% supported just veto (and rollback) scrolls in clipped axes

    if (me.overflowX === 'clip' && element.scrollLeft !== _x) {
      element.scrollLeft = _x;
      ++vetoed;
    }

    if (me.overflowY === 'clip' && element.scrollTop !== _y) {
      element.scrollTop = _y;
      ++vetoed;
    }

    if (vetoed === 2) {
      return;
    }

    if (!me.widget || !me.widget.isDestroyed) {
      // Don't read the value until we have to. The x & y getters will check this flag
      me.positionDirty = true;

      if (!me.element.classList.contains('b-scrolling')) {
        me.element.classList.add('b-scrolling');
      }

      e.widget = me.widget; // If we have the scroll silent flag, do not fire the event.

      if (!me.silent) {
        me.trigger('scroll', e);
      } // Keep partners in sync

      me.syncPartners(); // If this scroll impulse was from a controlling partner, clear that now

      me.controllingPartner = null; // Buffered method will fire in 100ms, unless another scroll event comes round.
      // In which case execution will be pushed out by another 100ms.

      me.onScrollEnd(e);
    }
  }

  syncPartners(force) {
    const me = this; // Keep partners in sync

    if (me.partners) {
      Object.values(me.partners).forEach(({
        axes,
        scroller
      }) => {
        // Don't feed back to the one who's just told us to scroll here.
        // Unless we have assumed command. For example Scheduler timeline infinite scrolling
        // has reset the scroll position and the partner who thinks it's controlling
        // must stay in sync with that reset.
        if (scroller !== me.controllingPartner || force) {
          scroller.sync(me, axes);
        }
      });
    }
  }

  onScrollEnd(e) {
    const me = this;

    if (me.silent) {
      me.silent = false;
    }

    me.trigger('scrollEnd', e); // Controlling partner is required for scrollable not to change its partners on scroll. This method is buffered
    // and landing here essentially means that no scrolling has occurred during the onScrollEnd buffer
    // time. We can safely cleanup controlling partner here.
    // https://github.com/bryntum/support/issues/1095

    me.controllingPartner = null;
    me.element.classList.remove('b-scrolling');
  }
  /**
   * Returns the xDelta and yDelta values in an object from the current scroll position to the
   * passed element or Rectangle.
   * @param {HTMLElement|Core.helper.util.Rectangle} element The element or a Rectangle to calculate deltas for.
   * @param {Object} [options] How to scroll.
   * @param {String} [options.block] How far to scroll the element: `start/end/center/nearest`.
   * @param {Number} [options.edgeOffset] A margin around the element or rectangle to bring into view.
   * @param {Boolean} [options.x] Pass as `false` to disable scrolling in the `X` axis.
   * @param {Boolean} [options.y] Pass as `false` to disable scrolling in the `Y` axis.
   * @returns {Object} `{ xDelta, yDelta }`
   * @internal
   */

  getDeltaTo(element, options) {
    const me = this; // scroller may belong to a collapsed subgrid widget

    if (!me.viewport) {
      return {
        xDelta: 0,
        yDelta: 0
      };
    }

    const {
      x,
      y,
      scrollWidth,
      scrollHeight,
      isRTL
    } = me,
          elementRect = element instanceof Rectangle ? element : Rectangle.from(element),
          block = options.block || 'nearest',
          scrollerRect = me.viewport,
          edgeOffset = options.edgeOffset || 0,
          // Only include the offset round the target is the viewport is big enough to accommodate it.
    xOffset = scrollerRect.width >= elementRect.width + edgeOffset * 2 ? edgeOffset : 0,
          yOffset = scrollerRect.height >= elementRect.height + edgeOffset * 2 ? edgeOffset : 0,
          constrainTo = new Rectangle(isRTL ? scrollerRect.right - x - scrollWidth : scrollerRect.x - x, scrollerRect.y - y, scrollWidth, scrollHeight),
          elRect = elementRect.clone().adjust(-xOffset, -yOffset, xOffset, yOffset).constrainTo(constrainTo),
          targetRect = elRect.clone();
    let xDelta = 0,
        yDelta = 0;

    if (block === 'start') {
      targetRect.moveTo(scrollerRect.x, scrollerRect.y);
      xDelta = elRect.x - targetRect.x;
      yDelta = elRect.y - targetRect.y;
    } else if (block === 'end') {
      targetRect.translate(scrollerRect.right - targetRect.right, scrollerRect.bottom - targetRect.bottom);
      xDelta = elRect.x - targetRect.x;
      yDelta = elRect.y - targetRect.y;
    } else {
      // Calculate deltas unless the above has done that for non-fitting target
      if (block === 'center') {
        const center = scrollerRect.center;
        targetRect.moveTo(center.x - targetRect.width / 2, center.y - targetRect.height / 2);
        xDelta = xDelta || elRect.x - targetRect.x;
        yDelta = yDelta || elRect.y - targetRect.y;
      } // Use "nearest"
      else {
        // Can't fit width in, scroll what is possible into view so that start is visible.
        if (targetRect.width > scrollerRect.width) {
          xDelta = targetRect.x - scrollerRect.x;
        } // If it's *possible* to scroll to nearest x, calculate the delta
        else {
          if (targetRect.right > scrollerRect.right) {
            xDelta = targetRect.right - scrollerRect.right;
          } else if (targetRect.x < scrollerRect.x) {
            xDelta = targetRect.x - scrollerRect.x;
          }
        } // Can't fit height in, scroll what is possible into view so that start is visible.

        if (targetRect.height > scrollerRect.height) {
          yDelta = targetRect.y - scrollerRect.y;
        } // If it's *possible* to scroll to nearest y, calculate the delta
        else {
          if (targetRect.bottom > scrollerRect.bottom) {
            yDelta = targetRect.bottom - scrollerRect.bottom;
          } else if (targetRect.y < scrollerRect.y) {
            yDelta = targetRect.y - scrollerRect.y;
          }
        }
      }
    }

    xDelta = Math.round(xDelta);
    yDelta = Math.round(yDelta); // Do not allow deltas which would produce -ve scrolling or scrolling past the maxX/Y

    return {
      // When calculating how much delta is necessary to scroll the targetRect to the center
      // constrain that to what is *possible*. If what you are trying to scroll into the
      // center is hard against the right edge of the scroll range, then it cannot scroll
      // to the center, and the result must reflect that even though scroll is self limiting.
      // This is because highlighting the requested "element", if that element is in fact
      // a Rectangle, uses a temporary element placed at the requested region which
      // MUST match where the actual scroll has moved the requested region.
      xDelta: options.x === false ? 0 : Math[isRTL ? 'min' : 'max'](Math[isRTL ? 'max' : 'min'](xDelta, me.maxX - x), -x),
      yDelta: options.y === false ? 0 : Math.max(Math.min(yDelta, me.maxY - y), -y)
    };
  }
  /**
   * A {@link Core/helper/util/Rectangle} describing the bounds of the scrolling viewport.
   * @property {Core.helper.util.Rectangle}
   */

  get viewport() {
    return Rectangle.client(this.element);
  }

  updateWidget(widget) {
    this.rtlSource = widget;
  }

  updateElement(element, oldElement) {
    const me = this; // The global Scroller doesn't monitor its element.
    // It's only used for *commanding* scrolls.

    if (me === Scroller._globalScroller) {
      me._element = element;
      me.positionDirty = true;
      return;
    }

    const scrollHandler = me.scrollHandler || (me.scrollHandler = me.onScroll.bind(me)),
          resizeHandler = me.resizeHandler || (me.resizeHandler = me.onElResize.bind(me));

    if (oldElement) {
      if (me.translate) {
        var _me$mutationObserver;

        (_me$mutationObserver = me.mutationObserver) === null || _me$mutationObserver === void 0 ? void 0 : _me$mutationObserver.disconnect(oldElement);
      } else {
        oldElement.removeEventListener('scroll', scrollHandler);
        oldElement.classList.remove(scrollerCls);
        oldElement.style.overflowX = oldElement.style.overflowY = '';
      }

      ResizeMonitor.removeResizeListener(oldElement, resizeHandler);
    }

    if (element) {
      if (me.translate) {
        if (!me.mutationObserver) {
          me.mutationObserver = new MutationObserver(me.mutationHandler || (me.mutationHandler = me.onElMutation.bind(me)));
        }

        me._x = me._y = 0;

        if (document.contains(element)) {
          const [x, y] = DomHelper.getTranslateXY(element);
          me._x = -x;
          me._y = -y;
        }

        me.mutationObserver.observe(element, {
          attributes: true
        });
      } else {
        element.addEventListener('scroll', scrollHandler);
        element.classList.add(scrollerCls);
      }

      ResizeMonitor.addResizeListener(element, resizeHandler); // Ensure the overflow configs, which are unable to process themselves
      // in the absence of the element get applied to the newly arrived element.

      if (me.positionDirty) {
        me.updateOverflowX(me.overflowX);
        me.updateOverflowY(me.overflowY);
      } // Keep flags synced fro the start

      me.syncOverflowState(); // Apply initially configured scroll position if we have non-zero positions

      if (me.isConfiguring) {
        me._x && me.updateX(me._x);
        me._y && me.updateY(me._y);
      }
    }

    me.positionDirty = true;
  }
  /**
   * The horizontal scroll position of the widget.
   * @property {Number}
   */

  get x() {
    const me = this,
          {
      element
    } = me;

    if (element && me.positionDirty) {
      if (me.translate) {
        const [x, y] = DomHelper.getTranslateXY(element);
        me._x = -x;
        me._y = -y;
      } else {
        me._x = element.scrollLeft;
        me._y = element.scrollTop;
      }

      me.positionDirty = false;
    }

    return me._x;
  }

  changeX(x) {
    // Only process initial X if we were configured to start at non-zero
    if (!this.isConfiguring || x) {
      return x;
    }

    this._x = x;
  }

  updateX(x) {
    const {
      element,
      widget
    } = this; // When element is outside of DOM, this can have no effect

    if (element && !(widget !== null && widget !== void 0 && widget.isConfiguring)) {
      this.trigger('scrollStart', {
        x
      });

      if (this.translate) {
        DomHelper.setTranslateX(element, -x);
      } else {
        element.scrollLeft = x;
      }
    } // The scroll position will need to be read before we can return it.
    // Do not read it back now, we may not have our element, or if we do,
    // that would cause a forced synchronous layout.

    this.positionDirty = true;
  }

  sync(controllingPartner, axes) {
    const {
      x,
      y
    } = axes;

    if (x != null) {
      if (this.x !== controllingPartner.x) {
        // Only set controlling partner when scroll will actually change. This helps to increase stability of
        // state restoring API.
        this.controllingPartner = controllingPartner;
        this.x = controllingPartner.x;
      }
    }

    if (y != null) {
      if (this.y !== controllingPartner.y) {
        this.controllingPartner = controllingPartner;
        this.y = controllingPartner.y;
      }
    }
  }
  /**
   * The vertical scroll position of the widget.
   * @property {Number}
   */

  get y() {
    const me = this,
          {
      element
    } = me;

    if (element && me.positionDirty) {
      if (me.translate) {
        const [x, y] = DomHelper.getTranslateXY(element);
        me._x = -x;
        me._y = -y;
      } else {
        me._x = element.scrollLeft;
        me._y = element.scrollTop;
      }

      me.positionDirty = false;
    }

    return me._y;
  }

  changeY(y) {
    // Only process initial Y if we were configured to start at non-zero
    if (!this.isConfiguring || y) {
      return y;
    }

    this._y = y;
  }

  updateY(y) {
    const {
      element,
      widget
    } = this; // When element is outside of DOM, this can have no effect

    if (element && !(widget !== null && widget !== void 0 && widget.isConfiguring)) {
      this.trigger('scrollStart', {
        y
      });

      if (this.translate) {
        DomHelper.setTranslateY(element, -y);
      } else {
        element.scrollTop = y;
      }
    } // The scroll position will need to be read before we can return it.
    // Do not read it back now, we may not have our element, or if we do,
    // that would cause a forced synchronous layout.

    this.positionDirty = true;
  }
  /**
   * The maximum `X` scrollable position of the widget.
   * @property {Number}
   * @readonly
   */

  get maxX() {
    return (this.scrollWidth - this.clientWidth) * (this.isRTL ? -1 : 1);
  }
  /**
   * The maximum `Y` scrollable position of the widget.
   * @property {Number}
   * @readonly
   */

  get maxY() {
    return this.scrollHeight - this.clientHeight;
  }

  updateOverflowX(overflowX, oldOverflowX) {
    const {
      element,
      translate
    } = this,
          {
      style,
      classList
    } = element;

    if (oldOverflowX === 'hidden-scroll') {
      classList.remove('b-hide-scroll');
    } // Scroll, but without showing scrollbars.
    // For example a grid header. Only works on platforms which
    // support suppression of scrollbars through CSS.

    if (overflowX === 'hidden-scroll' && !translate) {
      const otherAxisScrollable = isScrollable[style.overflowY]; // Can't do one axis hidden-scroll, and the other scrollable because the b-hide-scroll
      // class hides "all" scrollbars, so we have to make this axis hidden and use a wheel
      // listener to scroll the content.

      if (otherAxisScrollable) {
        overflowX = 'hidden'; // Adds a wheel listener if we don't already have one.

        this.enableWheel();
      } else {
        classList.add('b-hide-scroll');
      }
    }

    if (!translate) {
      style.overflowX = scrollLiterals[overflowX] || overflowX;
    }

    this.positionDirty = !this.isConfiguring;
  }

  updateOverflowY(overflowY, oldOverflowY) {
    const {
      element,
      translate
    } = this,
          {
      style,
      classList
    } = element;

    if (oldOverflowY === 'hidden-scroll') {
      classList.remove('b-hide-scroll');
    } // Scroll, but without showing scrollbars.
    // For example a grid header.
    // On platforms which show space-consuming scrollbars we hide scrollbars
    // and add a 'wheel' listener.

    if (overflowY === 'hidden-scroll' && !translate) {
      const otherAxisScrollable = isScrollable[style.overflowX]; // Can't do one axis hidden-scroll, and the other scrollable because the b-hide-scroll
      // class hides "all" scrollbars, so we have to make this axis hidden and use a wheel
      // listener to scroll the content.

      if (otherAxisScrollable) {
        overflowY = 'hidden'; // Adds a wheel listener if we don't already have one.

        this.enableWheel();
      } else {
        classList.add('b-hide-scroll');
      }
    }

    if (!translate) {
      style.overflowY = scrollLiterals[overflowY] || overflowY;
    }

    this.positionDirty = !this.isConfiguring;
  }

  enableWheel() {
    if (!this.wheelListenerRemover) {
      this.wheelListenerRemover = EventHelper.on({
        element: this.element,
        wheel: 'onWheel',
        thisObj: this
      });
    }
  }

  onWheel(e) {
    if (Math.abs(e.deltaX) > Math.abs(e.deltaY) && this.overflowX === 'hidden-scroll') {
      this.x += e.deltaX;
    } else if (this.overflowY === 'hidden-scroll') {
      this.y += e.deltaY;
    }
  }
  /**
   * The horizontal scroll range of the widget.
   * @property {Number}
   * @readonly
   */

  get scrollWidth() {
    var _this$element$scrollW, _this$element;

    return (_this$element$scrollW = (_this$element = this.element) === null || _this$element === void 0 ? void 0 : _this$element.scrollWidth) !== null && _this$element$scrollW !== void 0 ? _this$element$scrollW : 0;
  }

  set scrollWidth(scrollWidth) {
    const me = this,
          {
      element
    } = me;
    let stretcher = me.widthStretcher; // "Unsetting" scrollWidth removes the stretcher

    if (stretcher && scrollWidth == null) {
      stretcher.remove();
      me.widthStretcher = null;
    } else if (scrollWidth) {
      if (!stretcher) {
        stretcher = me.widthStretcher = DomHelper.createElement({
          className: 'b-scroller-stretcher b-horizontal-stretcher',
          // Should survive its surroundings being DomSynced
          retainElement: true
        });
      }

      if (me.isRTL) {
        scrollWidth = me.clientWidth - scrollWidth + 1;
      }

      stretcher.style.transform = `translateX(${scrollWidth - 1}px)`;

      if (element && !element.contains(stretcher)) {
        element.insertBefore(stretcher, element.firstElementChild);
      }
    } // Propagate call to partners so they will establish own scroller stretcher

    if (me.propagate !== false) {
      me.eachPartner(({
        scroller
      }) => {
        // Raise a flag on partner to not propagate changes from it further
        scroller.propagate = false;
        scroller.scrollWidth = scrollWidth;
        delete scroller.propagate;
      });
    }

    me.positionDirty = true;
    me.syncOverflowState();
  }

  get scrollHeight() {
    var _this$element$scrollH, _this$element2;

    return (_this$element$scrollH = (_this$element2 = this.element) === null || _this$element2 === void 0 ? void 0 : _this$element2.scrollHeight) !== null && _this$element$scrollH !== void 0 ? _this$element$scrollH : 0;
  }
  /**
   * The vertical scroll range of the widget. May be set to larger than the actual data
   * height to enable virtual scrolling. This is how the grid extends its scroll range
   * while only rendering a small subset of the dataset.
   * @property {Number}
   */

  set scrollHeight(scrollHeight) {
    const me = this,
          stretcher = me.stretcher || (me.stretcher = DomHelper.createElement({
      className: 'b-scroller-stretcher'
    }));
    stretcher.style.transform = `translateY(${scrollHeight - 1}px)`;

    if (me.element && me.element.lastChild !== stretcher) {
      me.element.appendChild(stretcher);
    }

    me.positionDirty = true;
    me.syncOverflowState();
  }
  /**
   * The client width of the widget.
   * @property {Number}
   * @readonly
   */

  get clientWidth() {
    var _this$element3;

    return ((_this$element3 = this.element) === null || _this$element3 === void 0 ? void 0 : _this$element3.clientWidth) || 0;
  }
  /**
   * The client height of the widget.
   * @property {Number}
   * @readonly
   */

  get clientHeight() {
    var _this$element4;

    return ((_this$element4 = this.element) === null || _this$element4 === void 0 ? void 0 : _this$element4.clientHeight) || 0;
  }
  /**
   * The unique ID of this Scroller
   * @property {String}
   * @readonly
   */

  get id() {
    if (!this._id) {
      if (this.widget) {
        this._id = `${this.widget.id}-scroller`;
      } else {
        this._id = IdHelper.generateId('scroller-');
      }
    }

    return this._id;
  } //region Extract configs
  // This function is not meant to be called by any code other than Base#getCurrentConfig().

  preProcessCurrentConfigs(configs) {
    super.preProcessCurrentConfigs();
    delete configs.widget;
    delete configs.element;
  } //endregion

}
Scroller._$name = 'Scroller';

const localeName = 'En',
      localeDesc = 'English',
      locale$5 = {
  localeName,
  localeDesc,
  // Translations for common words and phrases which are used by all classes.
  Object: {
    Yes: 'Yes',
    No: 'No',
    Cancel: 'Cancel',
    Ok: 'OK'
  },
  //region Widgets
  Combo: {
    noResults: 'No results',
    recordNotCommitted: 'Record could not be addded',
    addNewValue: value => `Add ${value}`
  },
  FilePicker: {
    file: 'File'
  },
  Field: {
    // native input ValidityState statuses
    badInput: 'Invalid field value',
    patternMismatch: 'Value should match a specific pattern',
    rangeOverflow: value => `Value must be less than or equal to ${value.max}`,
    rangeUnderflow: value => `Value must be greater than or equal to ${value.min}`,
    stepMismatch: 'Value should fit the step',
    tooLong: 'Value should be shorter',
    tooShort: 'Value should be longer',
    typeMismatch: 'Value is required to be in a special format',
    valueMissing: 'This field is required',
    invalidValue: 'Invalid field value',
    minimumValueViolation: 'Minimum value violation',
    maximumValueViolation: 'Maximum value violation',
    fieldRequired: 'This field is required',
    validateFilter: 'Value must be selected from the list'
  },
  DateField: {
    invalidDate: 'Invalid date input'
  },
  DatePicker: {
    gotoPrevYear: 'Go to previous year',
    gotoPrevMonth: 'Go to previous month',
    gotoNextMonth: 'Go to next month',
    gotoNextYear: 'Go to next year'
  },
  NumberFormat: {
    locale: 'en-US',
    currency: 'USD'
  },
  DurationField: {
    invalidUnit: 'Invalid unit'
  },
  TimeField: {
    invalidTime: 'Invalid time input'
  },
  TimePicker: {
    hour: 'Hour',
    minute: 'Minute'
  },
  List: {
    loading: 'Loading...'
  },
  // needed here due to LoadMaskable
  GridBase: {
    loadMask: 'Loading...',
    syncMask: 'Saving changes, please wait...'
  },
  PagingToolbar: {
    firstPage: 'Go to first page',
    prevPage: 'Go to previous page',
    page: 'Page',
    nextPage: 'Go to next page',
    lastPage: 'Go to last page',
    reload: 'Reload current page',
    noRecords: 'No records to display',
    pageCountTemplate: data => `of ${data.lastPage}`,
    summaryTemplate: data => `Displaying records ${data.start} - ${data.end} of ${data.allCount}`
  },
  PanelCollapser: {
    Collapse: 'Collapse',
    Expand: 'Expand'
  },
  Popup: {
    close: 'Close Popup'
  },
  UndoRedo: {
    Undo: 'Undo',
    Redo: 'Redo',
    UndoLastAction: 'Undo last action',
    RedoLastAction: 'Redo last undone action'
  },
  //endregion
  //region Others
  DateHelper: {
    locale: 'en-US',
    weekStartDay: 0,
    // Non-working days which match weekends by default, but can be changed according to schedule needs
    nonWorkingDays: {
      0: true,
      6: true
    },
    // Days considered as weekends by the selected country, but could be working days in the schedule
    weekends: {
      0: true,
      6: true
    },
    unitNames: [{
      single: 'millisecond',
      plural: 'ms',
      abbrev: 'ms'
    }, {
      single: 'second',
      plural: 'seconds',
      abbrev: 's'
    }, {
      single: 'minute',
      plural: 'minutes',
      abbrev: 'min'
    }, {
      single: 'hour',
      plural: 'hours',
      abbrev: 'h'
    }, {
      single: 'day',
      plural: 'days',
      abbrev: 'd'
    }, {
      single: 'week',
      plural: 'weeks',
      abbrev: 'w'
    }, {
      single: 'month',
      plural: 'months',
      abbrev: 'mon'
    }, {
      single: 'quarter',
      plural: 'quarters',
      abbrev: 'q'
    }, {
      single: 'year',
      plural: 'years',
      abbrev: 'yr'
    }, {
      single: 'decade',
      plural: 'decades',
      abbrev: 'dec'
    }],
    // Used to build a RegExp for parsing time units.
    // The full names from above are added into the generated Regexp.
    // So you may type "2 w" or "2 wk" or "2 week" or "2 weeks" into a DurationField.
    // When generating its display value though, it uses the full localized names above.
    unitAbbreviations: [['mil'], ['s', 'sec'], ['m', 'min'], ['h', 'hr'], ['d'], ['w', 'wk'], ['mo', 'mon', 'mnt'], ['q', 'quar', 'qrt'], ['y', 'yr'], ['dec']],
    parsers: {
      L: 'MM/DD/YYYY',
      LT: 'HH:mm A'
    },
    ordinalSuffix: number => {
      const hasSpecialCase = ['11', '12', '13'].find(n => number.endsWith(n));
      let suffix = 'th';

      if (!hasSpecialCase) {
        const lastDigit = number[number.length - 1];
        suffix = {
          1: 'st',
          2: 'nd',
          3: 'rd'
        }[lastDigit] || 'th';
      }

      return number + suffix;
    }
  },
  //endregion
  //region Trial
  TrialButton: {
    downloadTrial: 'Download trial'
  },
  TrialPanel: {
    title: 'Please complete fields',
    name: 'Name',
    email: 'Email',
    company: 'Company',
    product: 'Product',
    cancel: 'Cancel',
    submit: 'Submit',
    downloadStarting: 'Download starting, please wait...'
  } //endregion

};

LocaleManagerSingleton.registerLocale(localeName, {
  desc: localeDesc,
  locale: locale$5
});

/**
 * @module Core/mixin/Factoryable
 */

const {
  defineProperty: defineProperty$4
} = Reflect,
      ownerSymbol = Symbol('owner'),
      typeSplitRe = /[\s,]+/;
/**
 * This mixin is applied to base classes of a type that will be dynamically created by type name aliases.
 * ```
 *  class Layout extends Base.mixin(Factoryable) {
 *      static get factoryable() {
 *          return {
 *              defaultType : 'default'
 *          };
 *      }
 *
 *      static get type() {
 *          return 'default';
 *      }
 *  }
 *
 *  class Fit extends Layout {
 *      static get type() {
 *          return 'fit';
 *      }
 *  }
 * ```
 *
 * Once a family of classes has been defined, instances are created using the `create()` method:
 * ```
 *  const layout = Layout.create(config);
 * ```
 *
 * In the above example, `config` can be a type name (such as "fit") or a config object with a `type` property that
 * holds the type name.
 *
 * Factories can also extend other factories. For example, one factory creates objects that are useful across a wide
 * range of consumers, and a second factory creates objects for a more specialized consumer. If that specialized
 * consumer can also consume objects from the first factory, then the second factory can specify this relationship:
 * ```
 *  class General extends Base.mixin(Factoryable) {
 *      ...
 *  }
 *
 *  class Specialized extends Base.mixin(Factoryable) {
 *      static get factoryable() {
 *          return {
 *              extends : General,
 *              ...
 *          };
 *      }
 *  }
 * ```
 * The `extends` factoryable option can be either a class that mixes in `Factoryable` or an array of such classes.
 * @mixin
 * @internal
 */

var Factoryable = (Target => class Factoryable extends (Target || Base$1) {
  static get $name() {
    return 'Factoryable';
  }

  static get declarable() {
    return [
    /**
     * This property getter returns options that control the factory process. This property getter must be
     * defined by the class that mixes in `Factoryable` in order to initialize the factory properly.
     * ```
     *  static get factoryable() {
     *      return {
     *          defaultType : 'default'
     *      };
     *  }
     * ```
     * If there are no special options to provide, this method can return nothing (`undefined`):
     * ```
     *  static get factoryable() {
     *      // initialize the factory with all default options
     *  }
     * ```
     * @static
     * @member {Object} factoryable
     * @property {Boolean} [factoryable.caseless=true] Specify `false` to use case-sensitive type names. The
     * default is to ignore case.
     * @property {String} [factoryable.defaultType=null] The default type to create when a config object has
     * no `typeKey` property.
     * @property {Function|Function[]} [factoryable.extends] One or more classes that mix in `Factoryable` to
     * use for resolving type names when a type name is not found in this factory.
     * @property {String} [factoryable.typeKey='type'] The name of the property in a config object that holds
     * the type name.
     * @internal
     */
    'factoryable',
    /**
     * One or more additional type name aliases for this class. This can be useful for renaming and maintaining
     * a previous type name.
     * ```
     *  class Fit extends Layout {
     *      static get type() {
     *          return 'fit';
     *      }
     *
     *      static get alias() {
     *          return 'fill';  // deprecated type name (now known as 'fit')
     *      }
     *  }
     * ```
     * @static
     * @member {String|String[]} alias
     * @internal
     */
    'alias',
    /**
     * The (canonical) type name for this class by which instances can be created using the static
     * {@link #function-create-static create()} method.
     * @static
     * @member {String} type
     */
    'type'];
  }
  /**
   * Registers a class (`cls`) associated with the given `type`.
   * @param {String|String[]} type A string, array of strings or a comma-separated string containing the type names
   * for the specified `cls` class.
   * @param {Function} cls The class (constructor function)
   * @param {Boolean} [replace] Pass `true` to overwrite existing registered types. Otherwise, this method will throw
   * an exception if the `type` is already registered with this factory.
   * @internal
   */

  static register(type, cls, replace = globalThis.__BRYNTUM_EXAMPLE) {
    // `replace` default value is set to true while we are run inside a bryntum demo where code editor changes might
    // lead to same widget class being registered over and over.
    const {
      factoryable
    } = this.initClass(),
          {
      caseless,
      registry
    } = factoryable,
          types = StringHelper.split(type, typeSplitRe); // if type is a string[] it will just be returned

    for (let lower, name, i = 0; i < types.length; ++i) {
      name = types[i];
      lower = caseless ? name.toLowerCase() : name;

      if (!replace && lower in registry) {
        throw new Error(`Type "${name}" already registered with ${factoryable.class.name} factory`);
      } // Ensure class being registered is initialized. (initClass returns the class)

      registry[name] = registry[lower] = cls.initClass();
    }
  }
  /**
   * Returns 'true` if the passed instance is of the passed type or of a derived class.
   * @param {Object} instance The object to test.
   * @param {String} type The type to test against
   */

  static isA(instance, type) {
    return this.isType(instance, type, true);
  }
  /**
   * Returns 'true` if the passed instance is of the passed type.
   * @param {Object} instance The object to test.
   * @param {String} type The type to test against
   * @param {Boolean} [deep] Pass `true` to return `true` if the class is a subclass of the passed type.
   */

  static isType(instance, type, deep) {
    const {
      factoryable
    } = this,
          {
      caseless,
      registry
    } = factoryable,
          typeCls = registry[caseless ? type.toLowerCase() : type]; // If the type to be tested against maps to a class, see if the instance is an instanceof that

    if (typeCls) {
      if (deep) {
        return instance instanceof typeCls;
      }

      return instance.constructor === typeCls;
    }

    return false;
  }

  static setupAlias(cls) {
    cls.register(cls.alias, cls);
  }

  static setupFactoryable(cls, meta) {
    const superClass = meta.super.class;
    let {
      factoryable
    } = cls;
    factoryable = _objectSpread2({
      caseless: true,
      defaultType: null,
      extends: superClass.factoryable ? [superClass] : null,
      typeKey: 'type'
    }, factoryable);
    factoryable.class = cls;
    factoryable.registry = Object.create(null);

    if (factoryable.extends && !Array.isArray(factoryable.extends)) {
      factoryable.extends = [factoryable.extends];
    } // Replace the class/static getter with a new one that returns the complete factoryable object:

    defineProperty$4(cls, 'factoryable', {
      get() {
        return factoryable;
      }

    });
  }

  static setupType(cls, meta) {
    const {
      type
    } = cls;
    cls.register(type, cls, meta.replaceType); // Copy the static type property onto the prototype as a readonly property:

    defineProperty$4(cls.prototype, 'type', {
      value: type
    });
  }
  /**
   * Creates an instance from this factory, given the type name or a config object.
   * @param {String|Object} config The type name string or config object.
   * @param {String|Function|Object} [options] Creation options (for details see {@link #function-reconfigure-static}).
   * @returns {Object}
   */

  static create(config, options) {
    return this.reconfigure(null, config, options);
  }
  /**
   * Reconfigures an optional existing instance based on the provided config and returns the correctly configured
   * instance. This will be the `existingInstance` if the `config` does not specify a different type.
   *
   * If `config` is `null` (or simply falsy), this method will destroy the `existingInstance` (if any) and return
   * `null`.
   *
   * If there is no `existingInstance`, the config must specify a type. That is, it must be a string (the type name)
   * or an object containing a `type` property, the `defaultType` must be provided or the factory itself must have
   * a `defaultType` specified (see {@link #property-factoryable-static}).
   *
   * When an `existingInstance` is provided and a type is specified, the instance will be reconfigured via `setConfig`
   * if it is of that type. Otherwise, the `existingInstance` is destroyed (if it is owned by the `options.owner`)
   * and a new instance of the correct type is created.
   *
   * @param {Object} existingInstance The instance to reconfigure. This can be `null`.
   *
   * @param {String|Object} config The type name string or config object.
   *
   * @param {String|Function|Object} [options] Additional options to control the reconfiguration process. If this
   * value is a string or a class constructor, it treated as `options.type`. If this value is an class instance, it
   * is used as the `options.owner`. If this is a function, it is treated as `options.setup`. NOTE: functions declared
   * using the `function` keyword are equivalent to class constructors. Use an arrow function or a class method to
   * avoid this when a `setup` function is intended.
   *
   * @param {String|Function} [options.type] The default type to use if the `config` object does not specify a type.
   *
   * @param {Object} [options.owner] The owner of any created instances. If the `existingInstance` is being replaced,
   * this value determines if the instance will be destroyed.
   *
   * @param {Object} [options.defaults] A config object of default values to use when creating a new instance.
   *
   * @param {Function|String} [options.setup] A function or the name of a method (on the `options.owner`) to call
   * prior to creating a new instance. It is passed the config object that will be used to create the instance. If a
   * truthy value is returned, that value is passed to the constructor instead of the provided config object.
   *
   * @param {Function|String} [options.transform] A function or the name of a method (on the `options.owner`) to call
   * with the raw config object prior to processing and the value it returns replaces the raw value. This function is
   * used to transform strings or arrays (for example) into proper config objects.
   *
   * @param {Function|String} [options.cleanup] A function or the name of a method (on the `options.owner`) to call
   * prior to destroying the `existingInstance`. The `existingInstance` is passed as the sole argument.
   *
   * @returns {Object} The reconfigured instance (either `existingInstance` or a new instance of the desired type)
   */

  static reconfigure(existingInstance, config, options) {
    const me = this,
          {
      factoryable
    } = me,
          {
      typeKey
    } = factoryable;
    let defaultType = options,
        cleanup,
        defaults,
        mergeType,
        owner,
        prepared,
        setup,
        t,
        transform,
        type; // Pick apart the options and set the vars accordingly

    if (options && !ObjectHelper.isClass(options)) {
      // if (options is not the defaultType)
      defaultType = null;
      t = typeof options;

      if (t === 'function') {
        setup = options;
      } else if (t === 'string') {
        defaultType = options;
      } else if (ObjectHelper.isObject(options)) {
        // TODO revert to
        //  ({ cleanup, defaults, owner, setup, type : defaultType } = options);
        //  after this issue is fixed https://github.com/bryntum/bryntum-suite/issues/1457
        cleanup = options.cleanup;
        defaults = options.defaults;
        owner = options.owner;
        setup = options.setup;
        defaultType = options.type;
        transform = options.transform;
      } else {
        owner = options;
      }
    }

    if (transform) {
      config = typeof transform === 'string' ? owner[transform](config) : transform(config);
    }

    type = config; // Figure out config... it's either a type (string), a config object or the actual instance.

    if (typeof type === 'string') {
      config = {};
    } else if (config) {
      if (config === true) {
        config = {};
      }

      if (!ObjectHelper.isObject(config)) {
        // If we are being given an instance (not a config object), discard or destroy the existingInstance
        if (owner && config !== existingInstance && (existingInstance === null || existingInstance === void 0 ? void 0 : existingInstance[ownerSymbol]) === owner) {
          var _cleanup;

          typeof cleanup === 'string' ? owner[cleanup](existingInstance) : (_cleanup = cleanup) === null || _cleanup === void 0 ? void 0 : _cleanup(existingInstance);
          existingInstance.destroy();
        }

        return config;
      }

      type = config[typeKey];
    }

    type = type && me.resolveType(type); // We've got our orders... make it so...

    if (existingInstance) {
      // We can have a type-less config object when reconfiguring an existing instance, but if we have a type,
      // the existingInstance must be of that type. If !config that means we are nulling out.
      if (config && (!type || existingInstance.constructor === type)) {
        if (typeKey in config) {
          config = ObjectHelper.assign({}, config);
          delete config[typeKey]; // so "type" won't be processed as a config property
        }

        existingInstance.setConfig(config);
        return existingInstance;
      }

      if (owner && existingInstance[ownerSymbol] === owner) {
        var _cleanup2;

        typeof cleanup === 'string' ? owner[cleanup](existingInstance) : (_cleanup2 = cleanup) === null || _cleanup2 === void 0 ? void 0 : _cleanup2(existingInstance);
        existingInstance.destroy();
      }
    }

    if (config) {
      if (defaults) {
        if (!(mergeType = type)) {
          if (!(mergeType = defaults[typeKey] || defaultType || factoryable.defaultType)) {
            throw new Error(`No default mergeType defined for ${factoryable.class.name} factory`);
          }

          mergeType = me.resolveType(mergeType);
        }

        if (mergeType) {
          // Allow the merge fn of each config to perform the task:
          config = mergeType.mergeConfigs(defaults, config);
        }
      }

      if (setup) {
        prepared = typeof setup === 'string' ? owner[setup](config, type, defaults) : setup(config, type, defaults);

        if (prepared === null) {
          return prepared;
        }

        config = prepared || config;
      }

      if (!type) {
        // One more check on config[typeKey] since the setup() function may have added it...
        if (!(type = config[typeKey] || defaultType || factoryable.defaultType)) {
          throw new Error(`No default type defined for ${factoryable.class.name} factory`);
        }

        type = me.resolveType(type);
      }

      if (defaults && !mergeType) {
        config = type.mergeConfigs(defaults, config);
      }

      if (typeKey in config) {
        config = ObjectHelper.assign({}, config);
        delete config[typeKey]; // so "type" won't be processed as a config property
      }

      config = new type(config);

      if (owner) {
        config[ownerSymbol] = owner;
      }
    }

    return config || null;
  }
  /**
   * This method returns the constructor of the class registered for the given type name.
   * @param {String} type The type name to look up.
   * @param {Boolean} [optional] Pass `true` to return `null` if `type` is not found instead of throwing an exception.
   * @returns {Function}
   * @private
   */

  static resolveType(type, optional) {
    if (typeof type !== 'string') {
      return type;
    }

    const {
      factoryable
    } = this,
          bases = factoryable.extends;
    let result = factoryable.registry[factoryable.caseless ? type.toLowerCase() : type],
        i;

    for (i = 0; !result && bases && i < bases.length; ++i) {
      // Pass optional=true to base factory so the error is our own should the lookup fail:
      result = bases[i].resolveType(type,
      /* optional = */
      true);
    }

    if (!result && !optional) {
      throw new Error(`Invalid type name "${type}" passed to ${factoryable.class.name} factory`);
    }

    return result;
  }

});

/**
 * @module Core/mixin/Identifiable
 */

const // Id generation should be on a per page basis, not per module
idCounts = ObjectHelper.getPathDefault(globalThis, 'bryntum.idCounts', Object.create(null)),
      idTypes = {
  string: 1,
  number: 1
};
/**
 * A mixin which provides identifier services such as auto-creation of `id`s and registration and
 * lookup of instances by `id`.
 *
 * @mixin
 * @internal
 */

var Identifiable = (Target => class Identifiable extends (Target || Base$1) {
  static get $name() {
    return 'Identifiable';
  }

  static get declarable() {
    return ['identifiable'];
  }

  static get configurable() {
    return {
      /**
       * The id of this object.  If not specified one will be generated. Also used for lookups through the
       * static `getById` of the class which mixes this in. An example being {@link Core.widget.Widget}.
       *
       * For a {@link Core.widget.Widget Widget}, this is assigned as the `id` of the DOM
       * {@link Core.widget.Widget#config-element element} and must be unique across all elements
       * in the page's `document`.
       * @config {String}
       * @category Common
       */
      id: ''
    };
  }

  static setupIdentifiable(cls, meta) {
    const {
      identifiable
    } = cls;
    identifiable.idMap = Object.create(null);
    Reflect.defineProperty(cls, 'identifiable', {
      get() {
        return identifiable;
      }

    });
  }

  doDestroy() {
    this.constructor.unregisterInstance(this);
    super.doDestroy();
  }

  changeId(id) {
    return (this.hasGeneratedId
    /* assignment */
    = !id) ? this.generateAutoId() : id;
  }

  updateId(id, oldId) {
    const me = this,
          C = me.constructor;
    oldId && C.unregisterInstance(me, oldId);

    if (!me.hasGeneratedId || C.identifiable.registerGeneratedId !== false) {
      C.registerInstance(me, id);
    }
  }
  /**
   * This method generates an id for this instance.
   * @returns {String}
   * @internal
   */

  generateAutoId() {
    return this.constructor.generateId(`b-${this.$$name.toLowerCase()}-`);
  }

  static get all() {
    // not documented here since type of array is not knowable... documented at mixin target class
    return Object.values(this.identifiable.idMap);
  }
  /**
   * Generate a new id, using an internal counter and a prefix.
   * @param {String} prefix Id prefix
   * @returns {String} Generated id
   */

  static generateId(prefix = 'generatedId') {
    // This produces "b-foo-1, b-foo-2, ..." for each prefix independently of the others. In other words, it makes
    // id's more stable since the counter is on a per-class basis.
    return prefix + (idCounts[prefix] = (idCounts[prefix] || 0) + 1);
  }

  static registerInstance(instance, instanceId = instance.id) {
    const {
      idMap
    } = this.identifiable; // Code editor sets `disableThrow` to not get conflicts when loading the same module again

    if (instanceId in idMap && !this.disableThrow) {
      throw new Error('Id ' + instanceId + ' already in use');
    }

    idMap[instanceId] = instance;
  }
  /**
   * Unregister Identifiable instance, normally done on destruction
   * @param {Object} instance Object to unregister
   * @param {String} id The id of the instance to unregister.
   */

  static unregisterInstance(instance, id = instance.id) {
    const {
      idMap
    } = this.identifiable; // ID may be passed, for example if the instance is destroyed and can no longer yield an id.

    if (idTypes[typeof instance]) {
      delete idMap[instance];
    } // Have to check for identity in case another instance by the same id has been created.
    // Allow that to be overridden. Stores have always just evicted the previous owner of their IDs
    else if (idMap[id] === instance) {
      delete idMap[id];
    }
  }

  static getById(id) {
    const idMap = this.identifiable.idMap;

    if (idMap) {
      return idMap[id];
    }
  }

  static get registeredInstances() {
    const idMap = this.identifiable.idMap;
    return idMap ? Object.values(idMap) : [];
  }

});

/**
 * @module Core/helper/util/Promissory
 */

/**
 * Encapsulates a Promise and provides `resolve()` and `reject()` methods.
 *
 * For example:
 * ```
 *  load() {
 *      this.loading = new Promissory();
 *      this.store.load();
 *
 *      return this.loading.promise;
 *  }
 *
 *  onStoreLoad(store, err) {
 *      if (err) {
 *          this.loading.resolve(this);
 *      }
 *      else {
 *          this.loading.reject(err);
 *      }
 *  }
 *
 * ```
 * @internal
 */
class Promissory {
  constructor(fn) {
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });

    if (fn) {
      fn(this);
    }
  }

}
Promissory._$name = 'Promissory';

const {
  defineProperty: defineProperty$3
} = Reflect;
/**
 * @module Core/widget/Renderable
 */

/**
 * This class provides element rendering and automatic synchronization based on config property changes.
 *```
 *  class Label extends Renderable {
 *      static get configurable() {
 *          return {
 *              text : {
 *                  value: null,
 *
 *                  $config : 'render'
 *              }
 *          };
 *      }
 *
 *      renderDom() {
 *          return {
 *              tag: 'label',
 *              children: [
 *                  this.text
 *              ]
 *          };
 *      }
 *  }
 *```
 * A class can opt to not specify its render configs. In this case, these will be determined automatically, but with
 * some additional cost. This may not be worth considering if only a few instances are ever likely.
 *
 * @private
 */

class Renderable extends Base$1.mixin(Localizable, Events, Delayable, Identifiable) {
  static get configurable() {
    return {
      /**
       * The top-level DOM element for this object. This element is produced from the {@link #function-renderDom}
       * method of the derived class. As configs used by `renderDom` change, a synchronization of the DOM is
       * scheduled by calling {@link #function-refreshDom}. This call is made automatically by this class, making
       * it rarely necessary to call `refreshDom` directly.
       * @config {HTMLElement}
       * @category DOM
       */
      element: null
    };
  }

  static get delayable() {
    return {
      refreshDom: 'raf'
    };
  }

  static get identifiable() {
    return {};
  }
  /**
   * Returns the Set of configs defined as `render: true`, that is, the configs that determine the rendering (via
   * `renderDom`). If no such configs are defined, this method returns `null`.
   *
   * @returns {Set}
   * @private
   */

  static get renderConfigs() {
    const me = this,
          meta = me.$meta;
    let renderConfigs = meta.renderConfigs,
        configs,
        name;

    if (renderConfigs === undefined) {
      // if (first time for this class)
      renderConfigs = null;
      configs = meta.configs;

      for (name in configs) {
        if (configs[name].render) {
          (renderConfigs || (renderConfigs = new Set())).add(name);
        }
      } // Put $renderConfigs on the prototype so that onConfigChange is as simple as possible:

      meta.renderConfigs = me.prototype.$renderConfigs = renderConfigs;
    }

    return renderConfigs;
  } //region Init

  construct(...args) {
    this.$iid = ++Renderable.$idSeed;
    this.byRef = {};
    super.construct(...args);

    if (this.onDraw !== Renderable.prototype.onDraw) {
      this.initDrawable();
    }
  }

  startConfigure(config) {
    this.element = this.renderContext.renderDom(); // calls changeElement()

    super.startConfigure(config);
  } //endregion
  //region Configs

  get element() {
    // NOTE: We can replace the getter of a config property
    // Asking for the primary el is a good sign that we need to sync the DOM:
    this.refreshDom.flush();
    return this._element;
  }
  /**
   * This is called when the `element` config is assigned (via the setter).
   * @param {HTMLElement} element The new element being assigned.
   * @param {HTMLElement|null} oldElement The old element (previously assigned) or `null`.
   * @returns {HTMLElement}
   * @private
   */

  changeElement(element, oldElement) {
    const me = this;

    if (oldElement) {
      oldElement.remove();
    }

    if (element) {
      element.id = me.id;
      element = DomHelper.createElement(element, {
        refOwner: me
      });
    }

    return element;
  }

  updateId(id, oldId) {
    const me = this;

    if (oldId) {
      const element = me.element;
      element.id = id;
      me.fixRefOwnerId(element, id, oldId);
    }
  } //endregion
  //region Misc

  /**
   * Returns the `classList` of this instance's `element`.
   * @property {DOMTokenList}
   */

  get classes() {
    return this.element.classList;
  }
  /**
   * This method fixes the element's `$refOwnerId` when this instance's `id` is changing.
   * @param {HTMLElement} el The element to fix.
   * @param {String} id The new id being assigned.
   * @param {String} oldId The old id (previously assigned).
   * @private
   */

  fixRefOwnerId(el, id, oldId) {
    if (el.$refOwnerId === oldId) {
      el.$refOwnerId = id;
      const ref = el.$reference;

      if (ref) {
        el.id = `${id}-${ref}`;
      }

      for (const c of el.childNodes) {
        this.fixRefOwnerId(c, id, oldId);
      }
    }
  }

  onConfigChange({
    name
  }) {
    // The $renderConfigs Set is either on our prototype (due to renderConfigs getter) or on our instance (due to
    // renderContext getter):
    if (!this.isConfiguring && this.$renderConfigs.has(name)) {
      this.refreshDom();
    }
  } //endregion
  //region Rendering

  /**
   * This method is called by `DomHelper.createElement` and `DomSync.sync` as new reference elements are created.
   * @param {String} name The name of the element, i.e., the value of its `reference` attribute.
   * @param {HTMLElement} el The element instance
   * @param {Object} domConfig The DOM config object.
   * @private
   */

  attachRef(name, el, domConfig) {
    const me = this,
          key = '_' + name,
          {
      listeners
    } = domConfig;

    if (!(key in me)) {
      defineProperty$3(me, name, {
        get() {
          // Asking for a ref el is a good sign that we need to sync the DOM:
          me.refreshDom.flush();
          return me[key];
        },

        set(el) {
          me[key] = el; // Key elements contain owner pointer (Not supported on IE SVG).
          // if (el && el.dataset) {
          //     el.dataset.ownerCmp = me.id;
          // }
        }

      });
    }

    el.id = `${me.id}-${name}`;
    me.byRef[name] = el;
    me[name] = el;

    if (listeners) {
      domConfig.listeners = {
        on: listeners,
        un: EventHelper.on(Object.assign({
          element: el,
          thisObj: me
        }, listeners))
      };
    }
  }
  /**
   * This method is called by `DomSync.sync` as reference elements are removed from the DOM.
   * @param {String} name The name of the element, i.e., the value of its `reference` attribute.
   * @param {HTMLElement} el The element instance
   * @param {Object} domConfig The DOM config object.
   * @private
   */

  detachRef(name, el, domConfig) {
    if (domConfig.listeners) {
      domConfig.listeners.un();
      domConfig.listeners = null;
    }

    this[name] = null;
    delete this.byRef[name];
  }
  /**
   * This method returns a {@link Core.helper.DomHelper#function-createElement-static} config object that describes
   * the desired elements for this instance.
   *
   * This method is called to produce the initial DOM structure and again as necessary to generate the DOM for the
   * current state. The DOM produced by these subsequent calls is then passed through
   * {@link Core.helper.DomSync#function-sync-static DomSync.sync()} to update the DOM.
   * @returns {Object}
   */

  renderDom() {
    // abstract method provided by derived class
    return {};
  }
  /**
   * This property is the object to use when calling the `renderDom` method. It may evaluate to this instance (if
   * the class declares its render configs) or a helper object to track config usage during the `renderDom` call.
   * @property {Object}
   * @private
   */

  get renderContext() {
    const me = this,
          meta = me.$meta,
          C = me.constructor;
    let renderConfigs = meta.renderConfigs || C.renderConfigs,
        context = me; // If the class author did not declare any configs as render:true, then we make a proxy-like object that can
    // detect getter calls to build that Set. Since the getters may not all trigger on any given rendering, we
    // cannot share this work across instances since they may take different control paths.

    if (!renderConfigs) {
      context = Object.create(me);
      renderConfigs = new Set();

      for (const name in meta.configs) {
        defineProperty$3(context, name, {
          get() {
            renderConfigs.add(name);
            return me[name];
          }

        });
      } // In order to be substitutable for the Renderable, we need the same method name... we just need to run
      // renderDom w/ our context as "this":

      context.renderDom = () => {
        return me.renderDom.call(context);
      };

      me.$renderConfigs = renderConfigs;
    } // Replace this getter with the actual context so we don't get called again:

    defineProperty$3(me, 'renderContext', {
      value: context
    });
    return context;
  }
  /**
   * This method synchronized the DOM produced by {@link #function-renderDom} with what was previously produced and
   * updates the elements accordingly.
   *
   * This method is buffered such that calls to it do not immediately execute. To perform the refresh immediately,
   * do this:
   *```
   *  instance.refreshDom.now();
   *```
   * To flush any potential updates to the DOM (and do nothing if there are none), do this:
   *```
   *  instance.refreshDom.flush();
   *```
   * To determine if there are updates to the DOM pending, do this:
   *```
   *  if (instance.refreshDom.isPending) {
   *      ...
   *  }
   *```
   */

  refreshDom() {
    DomSync.sync({
      targetElement: this.element,
      domConfig: this.renderContext.renderDom(),
      refOwner: this,
      // This limits the sync() to only removing the classes and styles added by previous renderings. This
      // allows dynamically added styles and classes to be preserved:
      strict: true
    });
  } //endregion

  onListen(eventName) {
    // Only add the b-drawable-el when someone first listens for the draw event since it may never happen...
    if (eventName === 'draw') {
      this.initDrawable();
    }
  }

  initDrawable() {
    const me = this;

    if (!me.intersector) {
      let el = Object.values(me.byRef).find(el => el.classList.contains('b-drawable'));

      if (!el) {
        el = me.element;
        el.classList.add('b-drawable');
      }

      me.intersector = new IntersectionObserver(entries => {
        if (!me.isDestroyed) {
          entries.forEach(ent => {
            if (ent.isIntersecting) {
              me.onDraw({
                count: ++me.drawCounter
              });
            }
          });
        }
      });
      me.intersector.observe(el);
    }
  }

  onDraw(ev) {
    this.trigger('draw', ev);
  }

}
Renderable.$idSeed = 0;
Object.assign(Renderable.prototype, {
  drawCounter: 0,
  hasGeneratedId: false,
  intersector: null
});
Renderable._$name = 'Renderable';

/**
 * @module Core/widget/Mask
 */

/**
 * Masks a target element (document.body if none is specified). Call static methods for ease of use or make instance
 * for reusability.
 *
 * @example
 * Mask.mask('hello');
 * Mask.unmask();
 *
 * @inlineexample Core/widget/Mask.js
 *
  * @example
 * // Using progress by calling static method
 * let mask = Mask.mask({
 *   text:'The task is in progress',
 *   progress: 0,
 *   maxProgress: 100
 * }, document.body);
 * let timer = setInterval(()=>{
 *   mask.progress += 5;
 *   if(mask.progress >= mask.maxProgress) {
 *     Mask.unmask();
 *     clearInterval(timer)
 *   }
 * },100)
 *
 * @example
 * // Using progress to mask a Bryntum component
 * scheduler.mask({
 *  text:'Loading in progress',
 *   progress: 0,
 *   maxProgress: 100
 * })
 * let timer = setInterval(()=>{
 *   scheduler.masked.progress += 5;
 *   if(scheduler.masked.progress >= scheduler.masked.maxProgress) {
 *     scheduler.unmask();
 *     clearInterval(timer)
 *   }
 * },100)
 *
 */

class Mask extends Renderable {
  //region Config
  static get $name() {
    return 'Mask';
  } // Factoryable type name

  static get type() {
    return 'mask';
  }

  static get configurable() {
    return {
      /**
       * Set this config to trigger an automatic close after the desired delay:
       * ```
       *  mask.autoClose = 2000;
       * ```
       * If the mask has an `owner`, its `onMaskAutoClosing` method is called when the close starts and its
       * `onMaskAutoClose` method is called when the close finishes.
       * @config {Number}
       * @private
       */
      autoClose: null,

      /**
       * The portion of the {@link #config-target} element to be covered by this mask. By default, the mask fully
       * covers the `target`. In some cases, however, it may be desired to only cover the `'body'` (for example,
       * in a grid).
       *
       * This config is set in conjunction with `owner` which implements the method `syncMaskCover`.
       *
       * @config {String}
       * @private
       */
      cover: null,

      /**
       * The icon to show next to the text. Defaults to showing a spinner
       * @config {String}
       * @default
       */
      icon: 'b-icon b-icon-spinner',
      errorDefaults: {
        icon: 'b-icon b-icon-warning',
        autoClose: 3000,
        showDelay: 0
      },

      /**
       * The maximum value of the progress indicator
       * @property {Number}
       */
      maxProgress: null,

      /**
       * Mode: bright, bright-blur, dark or dark-blur
       * @config {String}
       * @default
       */
      mode: 'dark',

      /**
       * Number expressing the progress
       * @property {Number}
       */
      progress: null,
      // The owner is involved in the following features:
      //
      // - The `autoClose` timer calls `onMaskAutoClose`.
      // - The `cover` config calls `syncMaskCover`.
      // - If the `target` is a string, that string names the property of the `owner` that holds the
      //   `HTMLElement` reference.

      /**
       * The owning widget of this mask. This is required if `target` is a string.
       *
       * @config {Object|Core.widget.Widget}
       */
      owner: null,

      /**
       * The element to be masked. If this config is a string, that string is the name of the property of the
       * `owner` that holds the `HTMLElement` that is the actual target of the mask.
       *
       * NOTE: In prior releases, this used to be specified as the `element` config, but that is now, as with
       * `Widget`, the primary element of the mask.
       *
       * @config {String|HTMLElement}
       */
      target: null,

      /**
       * The text (or HTML) to show in mask
       * @config {String}
       */
      text: null,
      // TODO - perhaps a better way to deal w/multiple reasons to mask something would be a mask FIFO behind
      //  the scenes so that only one mask is visible at a time. That would take a bit of tinkering in the
      //  hide/show mechanism but feels like a more natural way to go. In which case, this config would go away
      //  and multiple masks would simply cooperate. https://github.com/bryntum/support/issues/190
      // This property is used to block trial mask removal
      // https://github.com/bryntum/support/issues/2604
      type: null,

      /**
       * The number of milliseconds to delay before making the mask visible. If set, the mask will have an
       * initial `opacity` of 0 but will function in all other ways as a normal mask. Setting this delay can
       * reduce flicker in cases where load operations are typically short (for example, a second or less).
       *
       * @config {Number}
       */
      showDelay: null,
      useTransition: false
    };
  }

  static get delayable() {
    return {
      deferredClose: 0,
      delayedShow: 0
    };
  } //endregion
  //region Init

  construct(config) {
    if (config) {
      let el = config.element,
          cfg; // Upgrade config -> cfg
      // Treat config as readonly, cfg is lazily copied and writable

      if (el) {
        VersionHelper.deprecate('Core', '4.0.0', 'Mask "element" config has been renamed to "target"');
        config = cfg = Object.assign({}, config);
        delete cfg.element;
        cfg.target = el;
      }

      el = config.target;

      if (typeof el === 'string') {
        config = cfg = cfg || Object.assign({}, config);
        cfg.target = config.owner[el]; // must supply "owner" in this case
      }
    }

    super.construct(config);

    if (!this.target) {
      this.target = document.body;
    }

    this.show();
  }

  doDestroy() {
    const me = this,
          {
      element
    } = me; // Do not destroy a trial mask

    if (me.type === 'trial') {
      return false;
    }

    if (element) {
      me.element = null;

      if (me.mode.endsWith('blur')) {
        DomHelper.forEachChild(element, child => {
          child.classList.remove(`b-masked-${me.mode}`);
        });
      }

      me.target.classList.remove('b-masked');
      me.target[me.maskName] = null;
    }

    super.doDestroy();
  }

  get maskElement() {
    // TODO log on use of deprecated property?
    return this.element;
  }

  get maskName() {
    const type = this.type;
    return `mask${typeof type === 'string' ? type.trim() : ''}`;
  }

  set error(value) {
    this.setConfig(this.errorDefaults);
    this.text = value;
  }

  renderDom() {
    const me = this,
          {
      maxProgress
    } = me;
    return {
      class: {
        'b-mask': 1,
        'b-delayed-show': me.showDelay,
        'b-widget': 1,
        [`b-mask-${me.mode}`]: 1,
        'b-progress': maxProgress,
        'b-prevent-transitions': !me.useTransition
      },
      children: [{
        reference: 'maskContent',
        class: 'b-mask-content b-drawable',
        children: [maxProgress ? {
          reference: 'progressElement',
          class: 'b-mask-progress-bar',
          style: {
            width: `${Math.max(0, Math.min(100, Math.round(me.progress / maxProgress * 100)))}%`
          }
        } : null, {
          reference: 'maskText',
          class: 'b-mask-text',
          html: (me.icon ? `<i class="b-mask-icon ${me.icon}"></i>` : '') + me.text
        }]
      }]
    };
  } //endregion
  //region Static

  static mergeConfigs(...sources) {
    const ret = {};

    for (const src of sources) {
      if (typeof src === 'string') {
        ret.text = src;
      } else {
        ObjectHelper.assign(ret, src); // not Object.assign!
      }
    }

    return ret;
  }
  /**
   * Shows a mask with the specified message
   * @param {String|Object} text Message
   * @param {HTMLElement} target The element to mask
   * @returns {Core.widget.Mask}
   */

  static mask(text, target = document.body) {
    return Mask.new({
      target
    }, typeof text !== 'string' ? _objectSpread2({}, text) : {
      text
    });
  }
  /**
   * Unmask
   * @param {HTMLElement} element Element to unmask
   * @returns {Promise} A promise which is resolved when the mask is gone
   */

  static unmask(element = document.body) {
    var _element$mask;

    return (_element$mask = element.mask) === null || _element$mask === void 0 ? void 0 : _element$mask.close();
  } //endregion
  //region Config

  updateAutoClose(delay) {
    const me = this;
    me.deferredClose.cancel();

    if (delay) {
      me.deferredClose.delay = delay;
      me.deferredClose();
    }
  }

  updateCover() {
    var _this$owner, _this$owner$syncMaskC;

    (_this$owner = this.owner) === null || _this$owner === void 0 ? void 0 : (_this$owner$syncMaskC = _this$owner.syncMaskCover) === null || _this$owner$syncMaskC === void 0 ? void 0 : _this$owner$syncMaskC.call(_this$owner, this);
  }

  updateShowDelay(delay) {
    const {
      delayedShow
    } = this;
    delayedShow.delay = delay;

    if (!delay) {
      delayedShow.flush();
    }
  } //endregion
  //region Show & hide

  deferredClose() {
    var _owner$onMaskAutoClos2;

    const {
      owner
    } = this;
    this.close().then(() => {
      var _owner$onMaskAutoClos;

      owner === null || owner === void 0 ? void 0 : (_owner$onMaskAutoClos = owner.onMaskAutoClose) === null || _owner$onMaskAutoClos === void 0 ? void 0 : _owner$onMaskAutoClos.call(owner, this);
    });
    owner === null || owner === void 0 ? void 0 : (_owner$onMaskAutoClos2 = owner.onMaskAutoClosing) === null || _owner$onMaskAutoClos2 === void 0 ? void 0 : _owner$onMaskAutoClos2.call(owner, this);
  }

  delayedShow() {
    this.classes.remove('b-delayed-show');
  }
  /**
   * Show mask
   */

  show() {
    const me = this,
          {
      element,
      target,
      hiding,
      maskName
    } = me; // We don't do this because we may want to show but automatically close after a
    // brief delay. The order of applying those configs should not be an issue. In
    // other words, to stop the deferredClose, you must set autoClose to falsy.
    // me.deferredClose.cancel();

    if (hiding) {
      // Resolving seems much better than the only other options:
      //  1. Never settling
      //  2. Rejecting
      hiding.resolve(); // This will be nulled out as the promise resolves but that is a race condition
      // compared to the next hide() call.

      me.hiding = null;
      me.clearTimeout('hide');
    }

    if (me.showDelay) {
      element.classList.add('b-delayed-show');
      me.delayedShow();
    }

    element.classList.add('b-visible');
    element.classList.remove('b-hidden');
    target.classList.add('b-masked');

    if (!target[maskName]) {
      target[maskName] = me;
      target.appendChild(element);
    }

    me.shown = true;
    me.trigger('show'); // blur has to blur child elements

    if (me.mode.endsWith('blur')) {
      DomHelper.forEachChild(target, child => {
        if (child !== element) {
          child.classList.add(`b-masked-${me.mode}`);
        }
      });
    }
  }
  /**
   * Hide mask
   * @returns {Promise} A promise which is resolved when the mask is hidden, or immediately if already hidden
   */

  hide() {
    const me = this,
          {
      target,
      element
    } = me;
    let {
      hiding
    } = me;

    if (!hiding) {
      if (!me.shown) {
        return Promise.resolve();
      }

      me.hiding = hiding = new Promissory();
      me.shown = false;
      element.classList.remove('b-visible');
      element.classList.add('b-hidden');
      target.classList.remove('b-masked');

      if (me.mode.endsWith('blur')) {
        DomHelper.forEachChild(target, child => {
          if (child !== element) {
            child.classList.remove(`b-masked-${me.mode}`);
          }
        });
      }

      hiding.promise = hiding.promise.then(() => {
        if (me.hiding === hiding) {
          me.hiding = null;
        }
      }); // TODO: use AnimationHelper when available

      me.setTimeout(() => hiding.resolve(), 500, 'hide');
    }

    return hiding.promise;
  }
  /**
   * Close mask (removes it)
   * @returns {Promise} A promise which is resolved when the mask is closed
   */

  async close() {
    await this.hide();
    this.destroy();
  } //endregion

}
Mask._$name = 'Mask';

var RTL = (Target => {
  var _class;

  return _class = class RTL extends (Target || Base$1) {
    get widgetClass() {}

    updateRtl(rtl) {
      this.element.classList.toggle('b-rtl', rtl === true);
      this.element.classList.toggle('b-ltr', rtl === false);
    } // Render is only called on outer widgets, children read their setting from their owner unless explicitly set

    render(...args) {
      var _this$owner;

      super.render && super.render(...args); // TODO: Remove in 6.0

      if (BrowserHelper.isChrome && BrowserHelper.chromeVersion < 87 || BrowserHelper.isFirefox && BrowserHelper.firefoxVersion < 66 || BrowserHelper.isSafari && BrowserHelper.safariVersion < 14.1) {
        this.element.classList.add('b-legacy-inset');
      } // Detect if rtl (catches both attribute `dir="rtl"` and CSS `direction: rtl`, as well as if owner uses rtl)

      if (getComputedStyle(this.element).direction === 'rtl' || (_this$owner = this.owner) !== null && _this$owner !== void 0 && _this$owner.rtl) {
        this.rtl = true;
      }
    }

  }, _defineProperty(_class, "$name", 'RTL'), _defineProperty(_class, "configurable", {
    // Force rtl
    rtl: null
  }), _class;
});

/**
 * @module Core/widget/Widget
 */

const assignValueDefaults = Object.freeze({
  highlight: false,
  onlyName: false
}),
      floatRoots = [],
      highlightExternalChange = 'highlightExternalChange',
      isTransparent = /transparent|rgba\(0,\s*0,\s*0,\s*0\)/,
      textInputTypes = {
  INPUT: 1,
  TEXTAREA: 1
},
      addElementListeners = (me, element, domConfig, refName) => {
  var _listeners;

  let listeners = domConfig === null || domConfig === void 0 ? void 0 : domConfig.listeners;
  listeners = ((_listeners = listeners) === null || _listeners === void 0 ? void 0 : _listeners.on) || listeners;

  if (listeners) {
    const un = EventHelper.on(ObjectHelper.assign({
      element,
      thisObj: me
    }, listeners));

    if (refName) {
      // The domConfig for refs gets regenerated on each compose() so we cannot use them to store the un
      // functions.
      (me._refListeners || (me._refListeners = Object.create(null)))[refName] = un;
    } else {
      domConfig.listeners = {
        on: listeners,
        un
      };
    }
  }
},
      mergeAnim = (value, was) => {
  // The show/hideAnimation objects can only have one animation property, but it is fine to merge if they have
  // the same property.
  return value && was && was[ObjectHelper.keys(value)[0]] ? Config.merge(value, was) : value;
},
      // Need braces here. MUST NOT return false
widgetTriggerPaint = w => {
  w.isVisible && w.triggerPaint();
},
      negationPseudo = /^:not\((.+)\)$/,
      nonFlowedPositions = /absolute|fixed/i,
      isScaled = w => w.scale != null,
      {
  hasOwnProperty: hasOwnProperty$3
} = Object.prototype,
      {
  defineProperty: defineProperty$2
} = Reflect,
      parseDuration = d => parseFloat(d) * (d.endsWith('ms') ? 1 : 1000),
      immediatePromise$6 = Promise.resolve(),
      alignedClass = ['b-aligned-above', 'b-aligned-right', 'b-aligned-below', 'b-aligned-left'],
      returnFalseProp = {
  configurable: true,

  get() {
    return false;
  }

},
      localizeRE = /(?:L\{([^}.]+)\})/,
      localizeTooltip = (string, part) => 'L{Tooltip.' + part + '}',
      alignSpecRe = /^([trblc])(\d*)-([trblc])(\d*)$/i,
      mergeAlign = (oldValue, newValue) => {
  // Promote eg 'l-r' to { align : 'l-r' } so that align configs can merge.
  // But only if they are rectangle align strings. align:'left', align:'start' etc must not change.
  if (alignSpecRe.test(oldValue)) {
    oldValue = {
      align: oldValue
    };
  }

  if (alignSpecRe.test(newValue)) {
    newValue = {
      align: newValue
    };
  }

  return Config.merge(oldValue, newValue);
};
/**
 * Base class for other widgets. The Widget base class simply encapsulates an element, and may optionally contain some
 * specified {@link #config-html}.
 *
 * ## Rendering
 *
 * Subclasses should override the {@link #function-compose} method to return their encapsulating element and internal
 * DOM structure. The `compose()` method returns a {@link Core.helper.DomHelper#function-createElement-static} config
 * object that is* used to create the DOM structure, based on its {@link Core.Base#property-configurable-static}
 * properties:
 *
 * ```javascript
 *  class Button extends Widget {
 *      static get configurable() {
 *          return {
 *              cls  : null,
 *              text : null
 *          };
 *      }
 *
 *      compose() {
 *          const { cls, text } = this;  // collect all relevant configs properties (for auto-detection)
 *
 *          return {
 *              tag   : 'button',
 *              class : cls,
 *              text
 *          };
 *      }
 *  }
 * ```
 *
 * The config properties used by the `compose()` method are auto-detected when the method is first called for a class.
 * All relevant properties must be read, even if they end up not being used so that future changes to these properties
 * will mark the rendering as dirty.
 *
 * When a config property used by `compose()` is modified, the {@link #function-recompose} method is called. Since
 * `recompose()` is a {@link Core.mixin.Delayable#property-delayable-static delayable} method, calling it schedules a
 * delayed call to `compose()` and a DOM update. Accessing the Widget's primary `element` or any reference element
 * property will force the DOM update to occur immediately.
 *
 * ### Child Elements
 *
 * Unlike typical {@link Core.helper.DomHelper#function-createElement-static DOM config} objects, the object returned
 * by `compose()` can use an object to simplify naming:
 *
 * ```javascript
 *  class Button extends Widget {
 *      ...
 *
 *      compose() {
 *          const { cls, iconCls, text } = this;  // collect all relevant configs properties (for auto-detection)
 *
 *          return {
 *              tag   : 'button',
 *              class : cls,
 *
 *              children : {
 *                  iconElement : iconCls && {
 *                      class : {
 *                          'button-icon' : 1,
 *                          [iconCls]     : 1
 *                      }
 *                  },
 *
 *                  textElement : {
 *                      text
 *                  }
 *              }
 *          };
 *      }
 *  }
 * ```
 *
 * The keys of the `children` are [iterated](https://2ality.com/2015/10/property-traversal-order-es6.html) to convert
 * the values into the array required by {@link Core.helper.DomHelper#function-createElement-static}. The names of the
 * properties becomes the `reference` of the element.
 *
 * For example, the above is equivalent to the following:
 *
 * ```javascript
 *  class Button extends Widget {
 *      ...
 *
 *      compose() {
 *          const { cls, iconCls, text } = this;  // collect all relevant configs properties (for auto-detection)
 *
 *          return {
 *              tag   : 'button',
 *              class : cls,
 *
 *              children : [iconCls && {
 *                  reference : 'iconElement',
 *                  class : {
 *                      'button-icon' : 1,
 *                      [iconCls]     : 1
 *                  }
 *              }, {
 *                  reference : 'textElement',
 *                  text
 *              }]
 *          };
 *      }
 *  }
 * ```
 *
 * The object form of `children` is preferred for clarity but also because it facilitates inheritance.
 *
 * ### Inheritance
 *
 * When a derived class implements `compose()`, the object it returns is automatically merged with the object returned
 * by the base class.
 *
 * For example, the following class adds a new child element:
 *
 * ```javascript
 *  class MenuButton extends Button {
 *      ...
 *
 *      compose() {
 *          const { menuCls } = this;  // collect all relevant configs properties (for auto-detection)
 *
 *          return {
 *              children : {
 *                  menuElement : {
 *                      class : {
 *                          'button-menu' : 1,
 *                          [menuCls]     : 1
 *                      }
 *                  }
 *              }
 *          };
 *      }
 *  }
 * ```
 *
 * ### Listeners
 *
 * Reference elements may also define event `listeners` in the `compose()` method:
 *
 * ```javascript
 *  class Button extends Widget {
 *      compose() {
 *          const { cls, text } = this;
 *
 *          return {
 *              tag   : 'button',
 *              class : cls,
 *              text,
 *
 *              listeners : {
 *                  click : 'onClick'
 *              }
 *          };
 *      }
 *
 *      onClick(event) {
 *          // handle click event
 *      }
 *  }
 * ```
 *
 * ## Resolving properties
 *
 * Values for a Widgets properties can be resolved from the ownership hierarchy. For example a text field in a toolbar
 * can get its initial value from a property on the container owning the toolbar. This is achieved by prefixing the
 * desired property name with 'up.':
 *
 * ```javascript
 *  const grid = new Grid((
 *      tbar : [{
 *          type  : 'numberfield',
 *          // Fields value will be retrieved from the grids rowHeight property
 *          value : 'up.rowHeight'
 *      }]
 *  });
 * ```
 *
 * NOTE: this is for now a one way one time binding, the value will only be read initially and not kept up to date on
 * later changes.
 *
 * @mixes Core/mixin/Events
 * @mixes Core/localization/Localizable
 * @extends Core/Base
 * @classType widget
 */

class Widget extends Base$1.mixin(Localizable, Events, Delayable, Identifiable, Factoryable, RTL) {
  //region Config

  /**
   * Class name getter.
   * Used when original ES6 class name is minified or mangled during production build.
   * Should be overriden in each class which extends Widget or it descendants.
   *
   * ```javascript
   * class MyNewClass extends Widget {
   *     static get $name() {
   *        return 'MyNewClass';
   *     }
   * }
   * ```
   *
   * @static
   * @member {String} $name
   */
  static get $name() {
    return 'Widget';
  }
  /**
   * Widget name alias which you can use in the `items` of a Container widget.
   *
   * ```javascript
   * class MyWidget extends Widget {
   *     static get type() {
   *        return 'mywidget';
   *     }
   * }
   * ```
   *
   * ```javascript
   * const panel = new Panel({
   *    title : 'Cool widgets',
   *    items : [
   *       { type : 'mywidget', html : 'Lorem ipsum dolor sit amet...' }
   *    ]
   * });
   * ```
   *
   * @static
   * @member {String} type
   */

  static get type() {
    return 'widget';
  }

  static get configurable() {
    return {
      /**
       * Get this widget's encapsulating HTMLElement, which is created along with the widget but added to DOM at
       * render time.
       * @member {HTMLElement} element
       * @readonly
       * @category DOM
       */

      /**
       * A {@link Core.helper.DomHelper#function-createElement-static} config object or HTML string from which to
       * create the Widget's element.
       * @private
       * @config {Object|String}
       * @category DOM
       */
      element: true,

      /**
       * Set to false to not call onXXX method names (e.g. `onShow`, `onClick`), as an easy way to listen for events.
       *
       * ```javascript
       * const container = new Container({
       *     callOnFunctions : true
       *
       *     onHide() {
       *          // Do something when the 'hide' event is fired
       *     }
       * });
       * ```
       * @config {Boolean}
       * @default
       */
      callOnFunctions: true,

      /**
       * Get/set widgets id
       * @member {String} id
       * @category DOM
       */

      /**
       * Widget id, if not specified one will be generated. Also used for lookups through Widget.getById
       * @config {String}
       * @category DOM
       */
      id: '',

      /**
       * The HTML to display initially or a function returning the markup (called at widget construction time)
       * @config {String|Function}
       * @category DOM
       */
      html: null,

      /**
       * Set HTML content safely, without disturbing sibling elements which may have been
       * added to the {@link #property-contentElement} by plugins and features.
       * When specifying html, this widget's element will also have the {@link #config-htmlCls}
       * added to its classList, to allow targeted styling.
       * @member {String} content
       * @category DOM
       */

      /**
       * The HTML content that coexists with sibling elements which may have been added to the
       * {@link #property-contentElement} by plugins and features.
       * When specifying html, this widget's element will also have the {@link #config-htmlCls}
       * class added to its classList, to allow targeted styling.
       * @config {String} content
       * @category DOM
       */
      content: null,

      /**
       * Custom CSS classes to add to element.
       * May be specified as a space separated string, or as an object in which property names
       * with truthy values are used as the class names:
       * ```javascript
       *  cls : {
       *      'b-my-class'     : 1,
       *      [this.extraCls]  : 1,
       *      [this.activeCls] : this.isActive
       *  }
       *  ```
       *
       * @config {String|Object}
       * @category CSS
       */
      cls: {
        $config: {
          merge: 'classList'
        },
        value: null
      },

      /**
       * Custom CSS class name suffixes to apply to the elements rendered by this widget. This may be specified
       * as a space separated string, an array of strings, or as an object in which property names with truthy
       * values are used as the class names.
       *
       * For example, consider a `Panel` with a `ui` config like so:
       *
       * ```javascript
       *  new Panel({
       *      text : 'OK',
       *      ui   : 'light'
       *  });
       * ```
       * This will apply the CSS class `'b-panel-ui-light'` to the main element of the panel as well as its many
       * child elements. This allows simpler CSS selectors to match the child elements of this particular panel
       * UI:
       *
       * ```
       *  .b-panel-content.b-panel-ui-light {
       *      background-color : #eee;
       *  }
       * ```
       * Using the {@link #config-cls cls config} would make matching the content element more complex, and in
       * the presence of {@link Core.widget.Panel#config-strips docked items} and nested panels, impossible to
       * target accurately.
       *
       * @config {String|Object}
       * @category CSS
       */
      ui: {
        $config: {
          merge: 'classList'
        },
        value: null
      },

      /**
       * Determines how a {@link Core.widget.Panel#config-collapsed} panel will treat this widget if it resides
       * within the panel's header (for example, as one of its {@link Core.widget.Panel#config-strips} or
       * {@link Core.widget.Panel#config-tools}).
       *
       * Valid options are:
       *  - `null` : The widget will be moved to the overlay when collapsed (the default).
       *  - `false` : The widget will be unaffected when the panel is collapsed.
       *  - `'hide'` : The widget will be hidden when the panel is collapsed.
       *  - `'overlay'` : The widget will only appear in the collapsed panel's overlay header.
       *
       * @config {Boolean|String}
       * @private
       */
      collapsify: null,

      /**
       * Custom CSS classes to add to the {@link #property-contentElement}.
       * May be specified as a space separated string, or as an object in which property names
       * with truthy values are used as the class names:
       * ```javascript
       *  cls : {
       *      'b-my-class'     : 1,
       *      [this.extraCls]  : 1,
       *      [this.activeCls] : this.isActive
       *  }
       *  ```
       *
       * @config {String|Object}
       * @category CSS
       */
      contentElementCls: {
        $config: {
          merge: 'classList'
        },
        value: null
      },

      /**
       * Custom CSS classes to add to this widget's `element`. This property is typically used internally to
       * assign default CSS classes while allowing `cls` to alter these defaults. It is not recommended that
       * client code set this config but instead should set `cls`.
       *
       * For example, to remove a class defined by `defaultCls` using `cls`, declare the class name as a key with
       * a falsy value:
       *
       * ```javascript
       *  cls : {
       *      'default-class' : false
       *  }
       * ```
       * @config {String|Object|String[]}
       * @internal
       */
      defaultCls: {
        $config: {
          merge: 'classList'
        },
        value: null
      },

      /**
       * Controls the placement of this widget when it is added to a {@link Core.widget.Panel panel's}
       *  {@link Core.widget.Panel#config-strips strips collection}. Typical values for this config are `'top'`,
       * `'bottom'`, `'left'`, or `'right'`, which cause the widget to be placed on that side of the panel's
       * body. Such widgets are called "edge strips".
       *
       * Also accepts direction neutral horizontal values `'start'` and `'end'``.
       *
       * If this config is set to `'header'`, the widget is placed in the panel's header, following the title. If
       * this config is set to `'pre-header'`, the widget is placed before the title. Such widgets are called
       * "header strips".
       *
       * @config {String} dock
       * @category Layout
       */
      dock: null,
      parent: null,

      /**
       * The {@link Core.widget.Tab tab} created for this widget when it is placed in a
       * {@link Core.widget.TabPanel}.
       * @member {Core.widget.Tab} tab
       * @readonly
       * @category Misc
       */

      /**
       * A configuration for the {@link Core.widget.Tab tab} created for this widget when it is placed in a
       * {@link Core.widget.TabPanel}. For example, this config can be used to control the icon of the `tab` for
       * this widget:
       *
       * ```javascript
       *  items : [{
       *      type : 'panel',
       *      // other configs...
       *
       *      tab : {
       *          icon : 'b-fa-wrench'
       *      }
       *  }, ... ]
       * ```
       *
       * Another use for this config is to set the tab's {@link Core.widget.mixin.Rotatable#config-rotate} value
       * differently than the default managed by the `TabPanel`:
       *
       * ```javascript
       *  items : [{
       *      type : 'panel',
       *      // other configs...
       *
       *      tab : {
       *          rotate : false   // don't rotate even if tabBar is docked left or right
       *      }
       *  }, ... ]
       * ```
       *
       * Set this to `false` to prevent the creation of a `tab` for this widget. In this case, this widget must
       * be {@link #function-show shown} explicitly. The {@link Core.widget.TabPanel#config-activeTab} for the
       * tab panel will be -1 in this situation.
       *
       * ```javascript
       *  items : [{
       *      type : 'panel',
       *      tab  : false,    // no tab for this item
       *
       *      // other configs...
       *  }, ... ]
       * ```
       *
       * @config {Boolean|Object} tab
       * @category Misc
       */
      tab: null,

      /**
       * An object specifying attributes to assign to the root element of this widget
       * @internal
       * @config {Object}
       * @category Misc
       */
      elementAttributes: null,

      /**
       * The CSS class(es) to add when HTML content is being applied to this widget.
       * @config {String|Object}
       * @category CSS
       */
      htmlCls: {
        $config: {
          merge: 'classList'
        },
        value: {
          'b-html': 1
        }
      },

      /**
       * Custom style spec to add to element
       * @config {String}
       * @category CSS
       */
      style: null,

      /**
       * Get/set element's disabled state
       * @member {Boolean} disabled
       * @category Misc
       */

      /**
       * Disable or enable the widget. It is similar to {@link #config-readOnly} except a disabled widget
       * cannot be focused, uses a different rendition (usually greyish) and does not allow selecting its value.
       * @default false
       * @config {Boolean}
       * @category Misc
       */
      disabled: null,

      /**
       * Get/set element's readOnly state. This is only valid if the widget is an input
       * field, __or contains input fields at any depth__. Updating this property will trigger
       * a {@link #event-readOnly} event.
       *
       * All descendant input fields follow the widget's setting. If a descendant
       * widget has a readOnly config, that is set.
       * @member {Boolean} readOnly
       * @category Misc
       */

      /**
       * Whether this widget is read-only.  This is only valid if the widget is an input
       * field, __or contains input fields at any depth__.
       *
       * All descendant input fields follow the widget's setting. If a descendant
       * widget has a readOnly config, that is set.
       * @default false
       * @config {Boolean}
       * @category Misc
       */
      readOnly: {
        value: null,
        default: false,
        $config: null
      },

      /**
       * Element (or element id) to adopt as this Widget's encapsulating element. The widget's
       * content will be placed inside this element.
       *
       * If this widget has not been configured with an id, it will adopt the id of the element
       * in order to preserve CSS rules which may apply to the id.
       * @config {HTMLElement|String}
       * @default
       * @category DOM
       */
      adopt: null,

      /**
       * Element (or element id) to append this widgets element to
       * @config {HTMLElement|String}
       * @default
       * @category DOM
       */
      appendTo: null,

      /**
       * Element (or element id) to insert this widget before. If provided, {@link #config-appendTo} config is ignored.
       * @config {HTMLElement|String}
       * @category DOM
       */
      insertBefore: null,

      /**
       * Element (or element id) to append this widget element to, as a first child. If provided, {@link #config-appendTo} config is ignored.
       * @config {HTMLElement|String}
       * @category DOM
       */
      insertFirst: null,

      /**
       * Object to apply to elements dataset (each key will be used as a data-attribute on the element)
       * @config {Object}
       * @category DOM
       */
      dataset: null,

      /**
       * Tooltip for the widget, either as a string or as a Tooltip config object.
       *
       * By default, the Widget will use a single, shared instance to display its tooltip as configured,
       * reconfiguring it to the specification before showing it. Therefore, it may not be permanently
       * mutated by doing things such as adding fixed event listeners.
       *
       * To have this Widget *own* its own `Tooltip` instance, add the property `newInstance : true`
       * to the configuration. In this case, the tooltip's {@link #property-owner} will be this Widget.
       *
       * __Note that in the absence of a configured {@link #config-ariaDescription}, the tooltip's value
       * will be used to populate an `aria-describedBy` element within this Widget.__
       * @config {String|Object}
       * @category Misc
       */
      tooltip: {
        $config: ['lazy', 'nullify'],
        value: null
      },

      /**
       * Set to false to not show the tooltip when this widget is {@link #property-disabled}
       * @config {Boolean}
       * @default
       * @category Misc
       */
      showTooltipWhenDisabled: true,

      /**
       * Prevent tooltip from being displayed on touch devices. Useful for example for buttons that display a
       * menu on click etc, since the tooltip would be displayed at the same time.
       * @config {Boolean}
       * @default false
       * @category Misc
       */
      preventTooltipOnTouch: null,

      /**
       * Specify true to have widget monitoring its own resize.
       * @config {Boolean}
       * @default false
       * @category Misc
       */
      monitorResize: {
        $config: ['lazy', 'nullify'],
        value: null
      },

      /**
       * Set to `true` to apply the default mask to the widget. Alternatively, this can be the mask message or a
       * {@link Core.widget.Mask} config object.
       * @config {Boolean|String|Object|Core.widget.Mask}
       * @category Misc
       */
      masked: null,

      /**
       * This config object contains the defaults for the {@link Core.widget.Mask} created for the
       * {@link #config-masked} config. Any properties specified in the `masked` config will override these
       * values.
       * @config {Object|Core.widget.Mask}
       * @default
       * @category Misc
       */
      maskDefaults: {
        target: 'element'
      },
      cache: {},

      /**
       * Set to `true` to move the widget out of the document flow and position it
       * absolutely in browser viewport space.
       * @config {Boolean}
       * @default
       * @category Float & align
       */
      floating: null,

      /**
       * Set to `true` when a widget is rendered into another widget's  {@link #property-contentElement}, but must not
       * participate in the standard layout of that widget, and must be positioned relatively to that
       * widget's {@link #property-contentElement}.
       *
       * {@link Core.widget.Editor Editor}s are positioned widgets.
       * @config {Boolean}
       * @default
       * @category Float & align
       */
      positioned: null,

      /**
       * Only valid if this Widget is {@link #config-floating}
       * Set to `true` to be able to drag a widget freely on the page. Or set to an object with a ´handleSelector´
       * property which controls when a drag should start.
       *
       * ```javascript
       *
       * draggable : {
       *     handleSelector : ':not(button)'
       * }
       *
       * ```
       *
       * @config {Boolean|Object}
       * @default false
       * @category Float & align
       */
      draggable: null,

      /**
       * _Only valid if this Widget is {@link #config-floating}._
       *
       * How to align this element with its target when {@link #function-showBy} is called
       * passing a simple element as an align target.
       *
       * Either a full alignment config object as passed to {@link #function-showBy}, or for simple
       * cases, the edge alignment string to use.
       *
       * When using a simple string, the format is `'[trblc]n-[trblc]n'` and it specifies our edge and
       * the target edge plus optional offsets from 0 to 100 along the edges to align to. Also supports direction
       * independent edges horizontally, `s` for start and `e` for end (maps to `l` and `r` for LTR, `r` and `l`
       * for RTL).
       *
       * See the {@link #function-showBy} function for more details about using the object form.
       *
       * Once set, this is stored internally in object form.
       * @config {Object|String}
       * @category Float & align
       */
      align: {
        $config: {
          merge: mergeAlign
        },
        value: 't-b'
      },

      /**
       * Only valid if this Widget is {@link #config-floating}
       * Set to `true` to centre the Widget in browser viewport space.
       * @config {Boolean}
       * @default
       * @category Float & align
       */
      centered: null,

      /**
       * Only valid if this Widget is {@link #config-floating} and being shown through {@link #function-showBy}.
       * Element, Widget or Rectangle to which this Widget is constrained.
       * @config {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle}
       * @default document.body
       * @category Float & align
       */
      constrainTo: undefined,

      /**
       * Only valid if this Widget is {@link #config-floating} and being shown through {@link #function-showBy}.
       * `true` to show a connector arrow pointing to the align target.
       * @config {Boolean}
       * @default false
       * @category Float & align
       */
      anchor: null,

      /**
       * The owning Widget of this Widget. If this Widget is directly contained, this will be the containing Widget.
       * If there is a `forElement`, this config will be that element's encapsulating Widget.
       *
       * If this Widget is floating, this config must be specified by the developer.
       * @config {Core.widget.Widget}
       * @category Float & align
       */
      owner: null,

      /**
       * Defines what to do if document is scrolled while Widget is visible (only relevant when floating is set to true).
       * Valid values: ´null´: do nothing, ´hide´: hide the widget or ´realign´: realign to the target if possible.
       * @config {String}
       * @default
       * @category Float & align
       */
      scrollAction: null,

      /**
       * Only valid if this Widget is {@link #config-floating}. An object which defined which CSS style
       * property should be animated upon hide, and how it should be animated eg:
       *
       * ```javascript
       * {
       *    opacity: {
       *        to : 0,
       *        duration: '10s',
       *        delay: '0s'
       *    }
       * }
       * ```
       *
       * Set to `'false'` to disable animation.
       *
       * @config {Boolean|Object}
       * @category Float & align
       */
      hideAnimation: {
        $config: {
          merge: mergeAnim
        },
        value: null
      },

      /**
       * Only valid if this Widget is {@link #config-floating}. An object which defined which CSS style
       * property should be animated upon show, and how it should be animated eg:
       *
       * ```javascript
       * {
       *    opacity: {
       *        to : 1,
       *        duration: '10s',
       *        delay: '0s'
       *    }
       * }
       * ```
       *
       * Set to `'false'` to disable animation.
       *
       * @config {Boolean|Object}
       * @category Float & align
       */
      showAnimation: {
        $config: {
          merge: mergeAnim
        },
        value: null
      },

      /**
       * The x position for the widget.
       *
       * Only valid if this Widget is {@link #config-floating} and not aligned or anchored to an element.
       *
       * @config {Number}
       * @default
       * @category Float & align
       */
      x: null,

      /**
       * The y position for the widget.
       *
       * Only valid if this Widget is {@link #config-floating} and not aligned or anchored to an element.
       *
       * @config {Number}
       * @default
       * @category Float & align
       */
      y: null,

      /**
       * Accessor to the {@link Core.helper.util.Scroller} which can be used
       * to both set and read scroll information.
       * @member {Core.helper.util.Scroller} scrollable
       * @category Layout
       */

      /**
       * Specifies whether (and optionally in which axes) a Widget may scroll. `true` means this widget
       * may scroll in both axes. May be an object containing boolean `overflowX` and `overflowY` properties which are applied
       * to CSS style properties `overflowX` and `overflowY`. If they are boolean, they are translated
       * to CSS overflow properties thus:
       *
       * *`true` -> `'auto'`
       * *`false` -> `'hidden'`
       *
       * After initialization, this property yields a {@link Core.helper.util.Scroller} which may be used
       * to both set and read scroll information.
       *
       * A Widget uses its `get overflowElement` property to select which element is to be scrollable.
       * By default, in the base `Widget` class, this is the Widget's encapsulating element. Subclasses
       * may implement `get overflowElement` to scroll inner elements.
       * @config {Boolean|Object|Core.helper.util.Scroller}
       * @default false
       * @category Scrolling
       */
      scrollable: {
        $config: ['nullify'],
        value: null
      },

      /**
       * The class to instantiate to use as the {@link #config-scrollable}. Defaults to {@link Core.helper.util.Scroller}.
       * @internal
       * @config {Core.helper.util.Scroller}
       * @typings {typeof Scroller}
       * @category Scrolling
       */
      scrollerClass: Scroller,

      /**
       * The name of the property to set when a single value is to be applied to this Widget. Such as when used
       * in a grid WidgetColumn, this is the property to which the column's `field` is applied.
       * @config {String}
       * @default 'html'
       * @category Misc
       */
      defaultBindProperty: 'html',

      /**
       * Event that should be considered the default action of the widget. When that event is triggered the
       * widget is also expected to trigger an `action` event. Purpose is to allow reacting to most widgets in
       * a coherent way.
       * @private
       * @config {String}
       * @category Misc
       */
      defaultAction: null,

      /**
       * When set to `true`, this widget is considered as a whole when processing {@link Core.widget.Toolbar}
       * overflow. When `false`, this widget's child items are considered instead.
       * @config {Boolean}
       * @default true
       * @category Layout
       * @internal
       */
      overflowable: {
        value: null,
        default: true,
        $config: null
      },

      /**
       * Widget's width, used to set element style.width. Either specify a valid width string or a number, which
       * will get 'px' appended. We recommend using CSS as the primary way to control width, but in some cases
       * this config is convenient.
       * @config {String|Number}
       * @category Layout
       */
      width: null,

      /**
       * Widget's height, used to set element style.height. Either specify a valid height string or a number, which
       * will get 'px' appended. We recommend using CSS as the primary way to control height, but in some cases
       * this config is convenient.
       * @config {String|Number}
       * @category Layout
       */
      height: null,

      /**
       * The element's maxHeight. Can be either a String or a Number (which will have 'px' appended). Note that
       * like {@link #config-height}, _reading_ the value will return the numeric value in pixels.
       * @config {String|Number}
       * @category Layout
       */
      maxHeight: null,

      /**
       * The elements maxWidth. Can be either a String or a Number (which will have 'px' appended). Note that
       * like {@link #config-width}, _reading_ the value will return the numeric value in pixels.
       * @config {String|Number}
       * @category Layout
       */
      maxWidth: null,

      /**
       * The elements minWidth. Can be either a String or a Number (which will have 'px' appended). Note that
       * like {@link #config-width}, _reading_ the value will return the numeric value in pixels.
       * @config {String|Number}
       * @category Layout
       */
      minWidth: null,

      /**
       * The element's minHeight. Can be either a String or a Number (which will have 'px' appended). Note that
       * like {@link #config-height}, _reading_ the value will return the numeric value in pixels.
       * @config {String|Number}
       * @category Layout
       */
      minHeight: null,
      // not public, only used by us in docs
      scaleToFitWidth: null,
      allowGrowWidth: true,
      // only used if scaleToFitWidth is true

      /**
       * Get element's margin property. This may be configured as a single number or a `TRBL` format string.
       * numeric-only values are interpreted as pixels.
       * @member {Number|String} margin
       * @category Layout
       */

      /**
       * Widget's margin. This may be configured as a single number or a `TRBL` format string.
       * numeric-only values are interpreted as pixels.
       * @config {Number|String}
       * @category Layout
       */
      margin: null,

      /**
       * Get element's flex property. This may be configured as a single number or a format string:
       *
       *      <flex-grow> <flex-shrink> <flex-basis>
       *
       * Numeric-only values are interpreted as the `flex-grow` value.
       * @member {Number|String} flex
       * @category Layout
       */

      /**
       * When this widget is a child of a {@link Core.widget.Container}, it will by default be participating in a
       * flexbox layout. This config allows you to set this widget's
       * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/flex">flex</a> style.
       * This may be configured as a single number or a `<flex-grow> <flex-shrink> <flex-basis>` format string.
       * numeric-only values are interpreted as the `flex-grow` value.
       * @config {Number|String}
       * @category Layout
       */
      flex: null,

      /**
       * A widgets weight determines its position among siblings when added to a {@link Core.widget.Container}.
       * Higher weights go further down.
       * @config {Number}
       * @category Layout
       */
      weight: null,

      /**
       * Get/set this widget's `align-self` flexbox setting. This may be set to modify how this widget is aligned
       * within the cross axis of a flexbox layout container.
       * @member {String} alignSelf
       * @category Layout
       */

      /**
       * When this widget is a child of a {@link Core.widget.Container}, it will by default be participating in a
       * flexbox layout. This config allows you to set this widget's
       * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/align-self">align-self</a> style.
       * @config {String}
       * @category Layout
       */
      alignSelf: null,

      /**
       * Configure as `true` to have the component display a translucent ripple when its
       * {@link #property-focusElement}, or {@link #property-element} is tapped *if the
       * current theme supports ripples*. Out of the box, only the Material theme supports ripples.
       *
       * This may also be a config object containing the following properties:
       *
       *  - `delegate  ` Optional. A CSS selector to filter which child elements trigger ripples. By default
       * the ripple is clipped to the triggering element.
       *  - `color     ` Optional, default = `#000`. A CSS color name or specification.
       *  - `radius    ` Optional, default is 100. The ending radius of the ripple.
       * Note that it will be clipped by the target element by default.
       *  - `clip      ` A string which describes how to clip the ripple if it is not to be clipped to the default
       * element. Either the property of the widget to use as the clipping element, or a selector to
       * allow clipping to the closest matching ancestor to the target element.
       *
       * eg:
       *```
       *    columns  : [{}...],
       *    ripple   : {
       *        color : 'red',
       *        clip  : '.b-grid-row'
       *    },
       *    ...
       *```
       * @config {Boolean|Object}
       * @category Misc
       */
      ripple: null,

      /**
       * A title to display for the widget. Only in effect when inside a container that uses it (such as TabPanel)
       * @default
       * @config {String}
       * @category DOM
       */
      title: null,
      localizableProperties: ['title', 'ariaLabel', 'ariaDescription'],
      // Set this flag to require element to have a size to be considered visible
      requireSize: false,

      /**
       * An identifier by which this widget will be registered in the {@link Core.widget.Container#property-widgetMap}
       * of all ancestor containers.
       *
       * If omitted, this widget will be registered using its {@link #config-id}. In most cases `ref` is
       * preferable over `id` since `id` is required to be globally unique while `ref` is not.
       *
       * The `ref` value is also added to the elements dataset, to allow targeting it using CSS etc.
       * @default
       * @config {String}
       * @category Misc
       */
      ref: null,

      /**
       * Get/set the widget hidden state.
       *
       * Note: `hidden : false` does *not* mean that this widget is definitely visible.
       * To ascertain visibility, use the {@link #property-isVisible} property.
       * @member {Boolean} hidden
       * @category Visibility
       */

      /**
       * Configure with true to make widget initially hidden.
       * @default false
       * @config {Boolean}
       * @category Layout
       */
      hidden: null,

      /**
       * Text alignment: 'left', 'center' or 'right'. Also accepts direction neutral 'start' and 'end'.
       *
       * Applied by adding a `b-text-align-xx` class to the widgets element. Blank by default, which does not add
       * any alignment class.
       *
       * To be compliant with RTL, 'left' yields same result as 'start' and 'right' as 'end'.
       *
       * @config {String}
       * @category Layout
       */
      textAlign: null,
      // When adding our scroll listeners to hide/realign, we ignore events
      // happening too quickly as a result of the show/align action
      ignoreScrollDuration: 500,

      /**
       * The tag name of this Widget's root element
       * @config {String}
       * @default
       * @category DOM
       */
      tag: 'div',

      /**
       * Set this config to `false` to disable batching DOM updates on animation frames for this widget. This
       * has the effect of synchronously updating the DOM when configs affecting the rendered DOM are modified.
       * Depending on the situation, this could simplify code while increasing time spent updating the DOM.
       * @config {Boolean}
       * @default true
       * @internal
       */
      recomposeAsync: null,

      /**
       * If you are rendering this widget to a shadow root inside a web component, set this config to the shadowRoot
       * @config {ShadowRoot}
       * @default
       * @category Misc
       */
      rootElement: null,
      htmlMutationObserver: {
        $config: ['lazy', 'nullify'],
        value: {
          childList: true,
          subtree: true
        }
      },
      role: {
        $config: 'lazy',
        value: 'presentation'
      },

      /**
       * A localizable string (May contain `'L{}'` tokens which resolve in the locale file) to inject as
       * the `aria-label` attribute.
       *
       * This widget is passed as the `templateData` so that functions in the locale file can
       * interrogate the widget's state.
       * @config {String}
       * @category Accessibility
       */
      ariaLabel: {
        $config: 'lazy',
        value: null
      },

      /**
       * A localizable string (May contain `'L{}'` tokens which resolve in the locale file) to inject
       * into an element which will be linked using the `aria-describedby` attribute.
       *
       * This widget is passed as the `templateData` so that functions in the locale file can
       * interrogate the widget's state.
       * @config {String}
       * @category Accessibility
       */
      ariaDescription: {
        $config: 'lazy',
        value: null
      },
      ariaElement: 'element',
      ariaHasPopup: null
    };
  }

  static get prototypeProperties() {
    return {
      /**
       * true if no id was set, will use generated id instead (widget1, ...). Toggle automatically on creation
       * @member {Boolean} hasGeneratedId
       * @private
       * @category Misc
       */
      hasGeneratedId: false,

      /**
       * This readonly property is `true` for normal widgets in the {@link Core.widget.Container#config-items} of
       * a container. It is `false` for special widgets such as a {@link Core.widget.Panel#config-tbar}.
       * @member {Boolean} innerItem
       * @internal
       * @category Misc
       */
      innerItem: true
    };
  }

  static get declarable() {
    return [
    /**
     * This property declares the set of config properties that affect a Widget's rendering, i.e., the configs
     * used by the {@link #function-compose} method.
     *
     * For example:
     * ```javascript
     *  class Button extends Widget {
     *      static get renderConfigs() {
     *          return [ 'cls', 'iconCls', 'text' ];
     *      }
     *  }
     * ```
     *
     * Alternatively this can be an object:
     *
     * ```javascript
     *  class Button extends Widget {
     *      static get renderConfigs() {
     *          return {
     *              cls     : true,
     *              iconCls : true,
     *              text    : true
     *          };
     *      }
     *  }
     * ```
     * @member {Object|String[]} renderConfigs
     * @static
     * @category Configuration
     * @internal
     */
    'renderConfigs'];
  }
  /**
   * An object providing the `record` and `column` for a widget embedded inside a {@link Grid.column.WidgetColumn}
   *
   * ```javascript
   * columns : [
   *    {
   *        type   : 'widget',
   *        widgets: [{
   *            type     : 'button',
   *            icon     : 'b-fa b-fa-trash',
   *            onAction : ({ source : btn }) => btn.cellInfo.record.remove()
   *        }]
   *    }
   * ]
   * ```
   * @readonly
   * @member {Object} cellInfo
   * @property {Core.data.Model} cellInfo.record
   * @property {Grid.column.Column} cellInfo.column
   * @category Misc
   */

  static get delayable() {
    return {
      recompose: 'raf',
      doHideOrRealign: 'raf'
    };
  }

  static get factoryable() {
    return {
      defaultType: 'widget'
    };
  }

  static get identifiable() {
    return {};
  }
  /**
   * Returns an array containing all existing Widgets. The returned array is generated by this call and is not an
   * internal structure.
   * @property {Core.widget.Widget[]}
   * @readonly
   * @internal
   */

  static get all() {
    return super.all;
  }
  /**
   * Get/set the {@link #config-recomposeAsync} config for all widgets. Setting this value will set the config for
   * all existing widgets and will be the default value for newly created widgets. Set this value to `null` to disable
   * the default setting for new widgets while leaving existing widgets unaffected.
   * @property {Boolean}
   * @internal
   */

  static get recomposeAsync() {
    return Widget._recomposeAsync;
  }

  static set recomposeAsync(value) {
    Widget._recomposeAsync = value;

    if (value != null) {
      const {
        all
      } = Widget;

      for (let i = 0; i < all.length; ++i) {
        if (all[i].isComposable) {
          all[i].recomposeAsync = value;
        }
      }
    }
  }

  isType(type, deep) {
    return Widget.isType(this, type, deep);
  }

  static setupRenderConfigs(cls, meta) {
    let {
      renderConfigs
    } = cls,
        i;

    if (renderConfigs) {
      const obj = // Once a class declares renderConfigs, those are inherited and augmented:
      meta.getInherited('renderConfigs',
      /* create = */
      false) || // Otherwise, since renderConfigs are declared on this class, we need to create them:
      Object.create(null);

      if (Array.isArray(renderConfigs)) {
        // Ex: renderConfigs: ['cls', 'text']
        for (i = 0; i < renderConfigs.length; ++i) {
          obj[renderConfigs[i]] = true;
        }

        renderConfigs = obj;
      } else {
        // Ex: renderConfigs: { cls : true, text : true }
        renderConfigs = Object.assign(obj, renderConfigs);
      }
    } // else a class may declare renderConfigs=null to re-enable auto detection

    meta.renderConfigs = renderConfigs;
  }
  /**
   * Call once per class for custom widgets to have them register with the `Widget` class, allowing them to be created
   * by type.
   *
   * For example:
   * ```javascript
   * class MyWidget extends Widget {
   *   static get type() {
   *     return 'mywidget';
   *   }
   * }
   * MyWidget.initClass();
   * ```
   * @method initClass
   * @static
   * @category Lifecycle
   */
  //endregion
  //region Init & destroy

  construct(config = {}, ...args) {
    const me = this,
          {
      domSyncCallback
    } = me;
    me.configureAriaDescription = config.ariaDescription;
    me._isAnimatingCounter = 0; // Flag so we know when our dimensions have been constrained during alignment

    me.alignConstrained = 0;
    me.byRef = Object.create(null);
    me.callRealign = me.realign.bind(me);
    me.onTargetResize = me.onTargetResize.bind(me);
    me.onFullscreenChange = me.onFullscreenChange.bind(me);
    me.domSyncCallback = domSyncCallback.$nullFn ? null : domSyncCallback.bind(me);
    me._isUserAction = false; // Base class applies configs.

    super.construct(config, ...args);
    const {
      recomposeAsync
    } = Widget;

    if (recomposeAsync != null && me.recomposeAsync == null) {
      me.recomposeAsync = recomposeAsync;
    }

    me.finalizeInit();
  }

  startConfigure(config) {
    const me = this,
          // This will run the element change/update process if it was not kicked off by a derived class impl of
    // this method:
    {
      adopt,
      element
    } = me;

    if (adopt) {
      // Adopt the preexisting element as our element before configuration proceeds.
      me.adoptElement(element, adopt, config.id);
      me.updateElement(me._element, element);
    }

    super.startConfigure(config);
  }
  /**
   * Called by the Base constructor after all configs have been applied.
   * @internal
   * @category Lifecycle
   */

  finalizeInit() {
    const me = this;

    if (me.insertBefore || me.appendTo || me.insertFirst || me.adopt) {
      me.render();
    }
  }

  doDestroy() {
    const me = this,
          {
      preExistingElements,
      element,
      _refListeners
    } = me;

    if (_refListeners) {
      Object.values(_refListeners, un => un());
      me._refListeners = null;
    }

    if (element) {
      var _Widget$Tooltip, _me$dragEventDetacher, _me$dragOverEventDeta;

      const sharedTooltip = !me._tooltip && me._rootElement && ((_Widget$Tooltip = Widget.Tooltip) === null || _Widget$Tooltip === void 0 ? void 0 : _Widget$Tooltip.getSharedTooltip(me._rootElement, true)); // If we are current user of the shared tooltip, hide it

      if ((sharedTooltip === null || sharedTooltip === void 0 ? void 0 : sharedTooltip.owner) === me) {
        sharedTooltip.owner = null;
        sharedTooltip.hide();
      }

      me.onExitFullscreen(); // If we get destroyed very quickly after a call to show,
      // we must kill the timers which add the realign listeners.

      me.clearTimeout(me.scrollListenerTimeout);
      me.clearTimeout(me.resizeListenerTimeout); // Remove listeners which are only added during the visible phase.
      // In its own method because it's called on hide and destroy.

      me.removeTransientListeners();

      if (me.floating || me.positioned) {
        // Hide without animation, destruction is sync
        me.hide(false);
      } else {
        me.revertFocus();
      }

      ResizeMonitor.removeResizeListener(element.parentElement, me.onParentElementResize);
      ResizeMonitor.removeResizeListener(element, me.onElementResize); // Remove elements *which we own* on destroy,

      if (me.adopt) {
        for (let nodes = Array.from(element.childNodes), i = 0, {
          length
        } = nodes; i < length; i++) {
          const el = nodes[i]; // If it's not preexisting, and not the floatRoot, remove it

          if (!preExistingElements.includes(el) && el !== me.floatRoot) {
            el.remove();
          }
        }

        element.className = me.adoptedElementClassName;
        element.style.cssText = me.adoptedElementCssText;
      } else {
        element.remove();
      }

      (_me$dragEventDetacher = me.dragEventDetacher) === null || _me$dragEventDetacher === void 0 ? void 0 : _me$dragEventDetacher.call(me);
      (_me$dragOverEventDeta = me.dragOverEventDetacher) === null || _me$dragOverEventDeta === void 0 ? void 0 : _me$dragOverEventDeta.call(me);
      me.dragGhost.remove();
    }

    super.doDestroy();
  } //endregion
  //region Values

  get assignValueDefaults() {
    return assignValueDefaults;
  }

  get valueName() {
    return this.name || this.ref || this.id;
  }

  getValueName(onlyName) {
    onlyName = onlyName && typeof onlyName === 'object' ? onlyName.onlyName : onlyName;
    return onlyName ? this.name : this.valueName;
  }

  assignFieldValue(values, key, value) {
    const me = this,
          valueBindProperty = me.defaultBindProperty;

    if (valueBindProperty in me) {
      me[valueBindProperty] = value;
    }
  }

  assignValue(values, options) {
    options = options || assignValueDefaults;
    const me = this,
          hec = me[highlightExternalChange],
          key = me.getValueName(options);

    if (key && (!values || key in values)) {
      if (options.highlight === false) {
        // Don't want a field highlight on mass change
        me[highlightExternalChange] = false;
      } // Setting to null when value not matched clears field

      me.assignFieldValue(values, key, values ? values[key] : null);
      me[highlightExternalChange] = hec;
    }
  }

  gatherValue(values) {
    const me = this,
          valueBindProperty = me.defaultBindProperty;

    if (valueBindProperty in me) {
      values[me.name || me.ref || me.id] = me[valueBindProperty];
    }
  }

  gatherValues(values) {
    this.eachWidget(widget => widget.gatherValue(values), false);
  } //endregion
  //---------------------------------------------------------------------------------------------------------
  //region Render

  /**
   * Returns `true` if this class uses `compose()` to render itself.
   * @returns {Boolean}
   * @internal
   */

  get isComposable() {
    return !this.compose.$nullFn;
  }

  adoptElement(element, adopt, id) {
    const me = this,
          adoptElement = typeof adopt === 'string' ? document.getElementById(adopt) : adopt,
          previousHolder = Widget.fromElement(adoptElement); // If we are taking it over from a previous iteration, destroy the previous holder. This is not officially
    // supported, but CodeEditor relies on it working

    if (previousHolder && previousHolder.adopt && previousHolder !== me) {
      const previousHolderAdopt = typeof previousHolder.adopt === 'string' ? document.getElementById(previousHolder.adopt) : previousHolder.adopt;

      if (previousHolderAdopt === adoptElement) {
        previousHolder.destroy();
      }
    } // On destroy, leave these

    me.preExistingElements = Array.from(adoptElement.childNodes);
    me.adoptedElementClassName = adoptElement.className;
    me.adoptedElementCssText = adoptElement.style.cssText; // Adopt the host element's id if we don't have one so that we do not override
    // it and invalidate any ad-based CSS rules.

    if (adoptElement.id && !id) {
      me.id = element.id = adoptElement.id;
    }

    DomHelper.syncAttributes(element, adoptElement);

    for (let i = 0, {
      length
    } = element.childNodes; i < length; i++) {
      adoptElement.appendChild(element.childNodes[0]);
    }

    delete me._contentRange; // Silently update our element config (do not re-run change/update cycle):

    me._element = adoptElement;
    const domConfig = element.lastDomConfig,
          listeners = domConfig === null || domConfig === void 0 ? void 0 : domConfig.listeners;

    if (listeners && me.isComposable) {
      var _listeners$un;

      (_listeners$un = listeners.un) === null || _listeners$un === void 0 ? void 0 : _listeners$un.call(listeners);
      addElementListeners(me, adoptElement, domConfig);
    } // Maintain DomSync internal state from our original element:

    adoptElement.lastDomConfig = domConfig || adoptElement.lastDomConfig;
    adoptElement.$refOwnerId = me.id;

    if (!me.scaleToFitWidth) {
      me.getConfig('monitorResize');
    }
  }
  /**
   * Defines an element reference accessor on the class prototype. This accessor is used to flush any pending DOM
   * changes prior to accessing such elements.
   * @param {String} name
   * @param {String} key
   * @private
   */

  addRefAccessor(name, key) {
    const {
      prototype
    } = this.constructor;
    defineProperty$2(prototype, key, {
      writable: true,
      value: null
    });
    defineProperty$2(prototype, name, {
      get() {
        // Asking for a ref el is a good sign that we need to sync the DOM:
        this.recompose.flush();
        return this[key];
      },

      set(el) {
        this[key] = el;
      }

    });
  }
  /**
   * This method is called by `DomHelper.createElement` and `DomSync.sync` as new reference elements are created.
   * @param {String} name The name of the element, i.e., the value of its `reference` attribute.
   * @param {HTMLElement} el The element instance
   * @param {Object} [domConfig] The DOM config object.
   * @internal
   */

  attachRef(name, el, domConfig) {
    const me = this,
          key = '_' + name; // Key elements contain owner pointer if data is supported (Not on IE SVG).

    el.dataset && (el.dataset.ownerCmp = me.id);

    if (me.isComposable) {
      if (!(key in me)) {
        me.addRefAccessor(name, key);
      }

      addElementListeners(me, el, domConfig, name);
    }

    me.byRef[name] = el;
    me[name] = el;
  }
  /**
   * This method is called by `DomSync.sync` as reference elements are removed from the DOM.
   * @param {String} name The name of the element, i.e., the value of its `reference` attribute.
   * @param {HTMLElement} el The element instance
   * @param {Object} domConfig The DOM config object.
   * @internal
   */

  detachRef(name, el, domConfig) {
    const me = this,
          listeners = me._refListeners;

    if (listeners !== null && listeners !== void 0 && listeners[name]) {
      listeners[name]();
      delete listeners[name];
    }

    me[name] = null;
    delete me.byRef[name];
  }
  /**
   * This method is called following an update to the widget's rendered DOM.
   * @internal
   */

  afterRecompose() {// empty
  }
  /**
   * Returns a {@link Core.helper.DomHelper#function-createElement-static} config object that defines this widget's
   * DOM structure. This object should be determined using {@link Core.Base#property-configurable-static} properties
   * to ensure this method is called again if these properties are modified.
   *
   * For more information see {@link Core.widget.Widget class documentation}.
   * @returns {Object}
   */

  compose() {
    const classes = DomClassList.normalize(this.widgetClassList, 'object');
    return {
      class: classes
    };
  }
  /**
   * This method iterates the class hierarchy from Widget down to the class of this instance and calls any `compose`
   * methods implemented by derived classes.
   * @returns {Object}
   * @private
   */

  doCompose() {
    const me = this,
          {
      $meta: meta
    } = me,
          classes = meta.hierarchy;
    let {
      composers
    } = meta,
        domConfig = null,
        c,
        key,
        firstTime,
        i,
        proto,
        renderConfigs;
    me.recompose.suspend();

    if (!composers) {
      firstTime = true;
      meta.composers = composers = []; // Widget starts the process w/the widgetClassList

      for (i = classes.indexOf(Widget); i < classes.length; ++i) {
        proto = classes[i].prototype;

        if (hasOwnProperty$3.call(proto, 'compose')) {
          composers.push(proto);
        }
      } // See if the class (or a super class) has declared its `renderConfigs`. If so, we use them.

      if (!(renderConfigs = meta.getInherited('renderConfigs',
      /* create = */
      false))) {
        // If not, we create a Proxy or proxy-like object that can detect getter calls to determine them. While
        // technically the getters may not all trigger on any given rendering, we ignore this due to the high
        // cost of supporting such arbitrary compose() methods. By convention, compose() must read configs
        // consistently.
        renderConfigs = Object.create(null); // On the first composition for this class, watch the config properties that are read to auto populate
        // the renderConfigs:

        me.configObserver = {
          get(name) {
            renderConfigs[name] = true;
          }

        };
      }

      meta.$renderConfigs = renderConfigs;
    } // This loop always runs at least once due to Widget base class, so ret will be assigned here:

    for (i = 0; i < composers.length; ++i) {
      c = composers[i].compose.call(me, domConfig);
      domConfig = domConfig ? DomHelper.merge(domConfig, c) : c;
    }

    if (hasOwnProperty$3.call(me, 'compose') && (c = me.compose)) {
      c = c.call(me, domConfig);
      DomHelper.merge(domConfig, c);
    }

    firstTime && delete me.configObserver; // When converting a children:{} into an array, we take a moment to ensure we have an accessor defined for
    // the element. This is needed if the element is initially unrendered since we need the accessor to flush a
    // pending recompose through just in time via the reference element getter.

    return DomHelper.normalizeChildren(domConfig, (childName, hoist) => {
      // Only care about refs that should be hoisted up to us
      if (hoist) {
        key = '_' + childName;

        if (!(key in me)) {
          me.addRefAccessor(childName, key);
        }
      }
    });
  }

  get element() {
    // NOTE: We can replace the getter of a config property
    if (this.isComposable && !this.isDestroying) {
      // Asking for the primary el is a good sign that we need to sync the DOM:
      this.recompose.flush();
    }

    return this._element;
  }
  /**
   * Template method called during DOM updates. See {@link Core.helper.DomSync#function-sync-static DomSync.sync()}.
   * @param {Object} info Properties describing the sync action taken.
   * @internal
   */

  domSyncCallback(info) {// bound in construct. Override in subclass
  }

  changeElement(element) {
    const me = this,
          compose = me.isComposable;

    if (compose) {
      element = me.doCompose();
    }

    if (typeof element === 'string') {
      element = DomHelper.createElementFromTemplate(element);
    } else if (ObjectHelper.isObject(element)) {
      element = DomHelper.createElement(element, {
        refOwner: me,
        callback: me.domSyncCallback // mimic DomSync callbacks (needed by TaskBoard)

      });
      me.recompose.resume();
      compose && addElementListeners(me, element, element.lastDomConfig);
    } else if (element.nodeType !== 1) {
      element = DomHelper.createElementFromTemplate(me.template(me));
    }

    element.id = me.id;

    if (me.elementAttributes) {
      DomHelper.setAttributes(element, me.elementAttributes);
    }

    return element;
  }

  updateElement(element) {
    const me = this,
          {
      className
    } = element,
          {
      contentElement,
      contentElementCls,
      isComposable
    } = me,
          hasChildContent = contentElement !== element,
          namedElements = !isComposable && element.querySelectorAll('[data-reference]'),
          // Start with the hierarchy classes, eg ['b-combo b-pickerfield b-textfield b-widget']
    classes = isComposable ? [] : me.widgetClassList; // a dynamic array that we can safely modify
    // The ui classes need to put on the content element even if isComposable, but widgetClassList contains the
    // ui classes, so we don't need to do that if the main element is the contentElement

    let uiClasses = (hasChildContent || !isComposable) && me.uiClasses;
    className && classes.unshift(className);
    me._hidden && classes.push('b-hidden');
    me._readOnly && classes.push('b-readonly'); // Calling element.remove() when we have the focus can result in a DOMException (notably when a blur/focusout
    // handler reentrancy results in a remove):
    //  DOMException: Failed to execute 'remove' on 'Element': The node to be removed is no longer a child
    //  of this node. Perhaps it was moved in a 'blur' event handler?

    FunctionHelper.noThrow(element, 'remove'
    /*, () => { debugger; } /**/
    ); // delete "/*" to break on exception

    if (uiClasses) {
      if (contentElementCls !== null && contentElementCls !== void 0 && contentElementCls.value) {
        uiClasses = uiClasses.slice(); // clone cached array

        uiClasses.push(contentElementCls.value);
      }

      uiClasses = uiClasses.join(' ');
    } else {
      uiClasses = contentElementCls === null || contentElementCls === void 0 ? void 0 : contentElementCls.value;
    }

    if (uiClasses) {
      if (hasChildContent) {
        contentElement.className += ' ' + uiClasses;
      } else {
        classes.push(uiClasses);
      }
    } // The environmental classes only need to be added to outermost Widgets.
    // If we have a parent container, that will have them.

    if (!me.parent) {
      const {
        defaultCls
      } = me,
            {
        outerCls
      } = Widget;
      classes.push(...(defaultCls ? outerCls.filter(c => !(c in defaultCls) || defaultCls[c]) : outerCls));
    }

    element.className = classes.join(' ');

    if (namedElements) {
      for (let el, i = 0; i < namedElements.length; ++i) {
        el = namedElements[i];
        me.attachRef(el.getAttribute('data-reference'), el);
        el.removeAttribute('data-reference');
      }
    } // Mutually exclusive with scaleToFitWidth.
    // Observe container element before the cascade down to descendants.
    // Outer elements are expected to fire resize first.
    // It's a lazy config, so this is the time to flush it through to begin monitoring.

    if (!me.adopt && !me.scaleToFitWidth) {
      me.getConfig('monitorResize');
    } // Pull in lazy configs now we have the element.

    me.getConfig('role');
    me.getConfig('ariaLabel');
    me.getConfig('ariaDescription'); // Ensure our content mutation observer keeps us informed of changes by third parties
    // so that our config system can keep up to date.

    if (me._html) {
      me.getConfig('htmlMutationObserver');
    }
  }

  updateAriaDescription(ariaDescription) {
    const {
      ariaElement
    } = this,
          descElId = `${this.id}-aria-desc-el`;

    if (ariaDescription) {
      const ariaDescEl = this._ariaDescEl || (this._ariaDescEl = DomHelper.createElement({
        className: 'b-aria-desc-element',
        id: descElId,
        parent: ariaElement
      }));
      ariaDescEl.innerText = ariaDescription.match(localizeRE) ? this.L(ariaDescription, this) : ariaDescription;
      ariaElement.setAttribute('aria-describedBy', ariaDescEl.id);
    } else if (ariaElement.getAttribute('aria-describedby') === descElId) {
      ariaElement.removeAttribute('aria-describedBy');
    }
  }

  updateAriaLabel(ariaLabel) {
    DomHelper.setAttributes(this.ariaElement, {
      'aria-label': ariaLabel !== null && ariaLabel !== void 0 && ariaLabel.match(localizeRE) ? this.L(ariaLabel, this) : ariaLabel
    });
  }

  updateAriaHasPopup(ariaHasPopup) {
    DomHelper.setAttributes(this.ariaElement, {
      'aria-haspopup': ariaHasPopup
    });
  }

  updateRole(role) {
    if (role) {
      var _this$ariaElement;

      (_this$ariaElement = this.ariaElement) === null || _this$ariaElement === void 0 ? void 0 : _this$ariaElement.setAttribute('role', role);
    } else {
      var _this$ariaElement2;

      (_this$ariaElement2 = this.ariaElement) === null || _this$ariaElement2 === void 0 ? void 0 : _this$ariaElement2.removeAttribute('role');
    }
  }

  get ariaElement() {
    // Ensure element has been created.
    this.getConfig('element');
    const {
      _ariaElement
    } = this; // Note that we use ObjectHelper.getPath enabling expressions containing dots.
    // So that widget classes may use `ownedWidget.input` to reference elements inside owned widgets.

    return _ariaElement.nodeType === Node.ELEMENT_NODE ? _ariaElement : ObjectHelper.getPath(this, _ariaElement);
  }

  isCollapsified(collapsed, overlay) {
    const {
      collapsify
    } = this; // false => widget is unaffected when the panel is collapsed (always in main header)
    // 'hide' => widget is hidden when the panel is collapsed (but still in main header)

    let ret = !overlay;

    if (collapsify == null) {
      // widget is moved to the overlay when collapsed
      ret = overlay ? collapsed : !collapsed;
    } else if (collapsify === 'overlay') {
      // widget always appears in the overlay
      ret = overlay;
    }

    return Boolean(ret);
  }
  /**
   * Calling this {@link Core.mixin.Delayable#property-delayable-static} method marks this widget as dirty. The DOM
   * will be updated on the next animation frame:
   *
   * ```javascript
   *  widget.recompose();
   *
   *  console.log(widget.recompose.isPending);
   *  > true
   * ```
   *
   * A pending update can be flushed by calling `flush()` (this does nothing if no update is pending):
   *
   * ```javascript
   *  widget.recompose.flush();
   * ```
   *
   * This can be combined in one call to force a DOM update without first scheduling one:
   *
   * ```javascript
   *  widget.recompose.now();
   * ```
   */

  async recompose() {
    const me = this,
          options = {
      targetElement: me.element,
      domConfig: me.doCompose(),
      refOwner: me,
      callback: me.domSyncCallback,
      // This limits the sync() to only removing the classes and styles added by previous renderings. This
      // allows dynamically added styles and classes to be preserved:
      strict: true
    };

    if (me.transitionRecompose) {
      me.isTransitioningDom = true;
      await DomHelper.transition(ObjectHelper.assign({
        element: me.element,

        action() {
          DomSync.sync(options);
        }

      }, me.transitionRecompose));

      if (me.isDestroyed) {
        return;
      }

      me.isTransitioningDom = false;
      me.trigger('transitionedRecompose');
    } else {
      DomSync.sync(options);
    }

    if (options.changed) {
      me.afterRecompose();
    }

    me.recompose.resume();
  }

  render(parentElement = this.appendTo, triggerPaint = true) {
    const me = this,
          {
      element
    } = me;
    me.emptyCache();

    if (me.syncElement && me.currentElement) {
      DomHelper.sync(element, me.currentElement);
    } else {
      var _parentElement;

      let insertBefore = me.insertBefore;

      if (me.insertFirst) {
        parentElement = me.insertFirst;
        insertBefore = parentElement.firstChild;
      }

      if (typeof parentElement === 'string') {
        const id = parentElement;
        parentElement = document.getElementById(parentElement);

        if (!parentElement) {
          throw new Error(`No element found with id '${id}'`);
        }

        if (me.insertFirst) {
          insertBefore = parentElement.firstChild;
        }
      }

      if (insertBefore) {
        if (typeof insertBefore === 'string') {
          const id = insertBefore;
          insertBefore = document.getElementById(insertBefore);

          if (!insertBefore) {
            throw new Error(`No element found with id '${id}'`);
          }
        }

        if (!parentElement) {
          parentElement = insertBefore.parentElement;
        }
      }

      (_parentElement = parentElement) === null || _parentElement === void 0 ? void 0 : _parentElement.insertBefore(element, insertBefore);
      me.currentElement = element;
    } // The environmental classes only need to be added to a naked Widget.
    // If we are inside a Widget's element, that will have them.

    if (Widget.fromElement(element.parentElement)) {
      element.classList.remove(...Widget.outerCls);
    }

    super.render(parentElement, triggerPaint);
    me.rendered = true; // Now that we have our complete DOM, update our role if we have one.

    me.getConfig('role');

    if (triggerPaint) {
      me.getConfig('contentRange');
      me.triggerPaint();
    }

    me.setupFocusListeners();
  }
  /**
   * A function which, when passed an instance of this Widget, produces a valid HTML string which is compiled
   * to create the encapsulating element for this Widget, and its own internal DOM structure.
   *
   * Note that this just creates the DOM structure that *this* Widget owns. If it contains child widgets
   * (Such as for example a grid), this is not included. The template creates own structure.
   *
   * Certain elements within the generated element can be identified as special elements with a `reference="name"`
   * property. These will be extracted from the element upon creation and injected as the named property into
   * the Widget. For example, a {@link Core.widget.TextField} will have an `input` property which is its
   * `<input>` element.
   * @param {Core.widget.Widget} me The widget for which to produce the initial HTML structure.
   * @internal
   */

  template({
    tag,
    html,
    htmlCls,
    name
  }) {
    const content = html !== null && html !== void 0 && html.call ? html.call(this) : html;
    return `<${tag} class="${content ? htmlCls : ''}" ${name ? `data-name="${name}"` : ''}>${content || ''}</${tag}>`;
  }

  updateRecomposeAsync(async) {
    this.recompose.immediate = !async;
  } //endregion
  //---------------------------------------------------------------------------------------------------------

  onConfigChange({
    name
  }) {
    var _this$$meta$$renderCo;

    // The $renderConfigs object is either on our prototype (due to renderConfigs getter) or on our instance (due
    // to "get composer") unless we are not using compose(), in which case it will be null:
    if (this._element && !this.isDestroying && (_this$$meta$$renderCo = this.$meta.$renderConfigs) !== null && _this$$meta$$renderCo !== void 0 && _this$$meta$$renderCo[name]) {
      this.recompose();
    }
  } //region Extract config
  // These functions are not meant to be called by any code other than Base#getCurrentConfig()
  // Clean up configs

  preProcessCurrentConfigs(configs) {
    super.preProcessCurrentConfigs(configs); // Remove link to parent, is set when added

    delete configs.parent;
  } // Extract config's current value, special handling for style

  getConfigValue(name, options) {
    // Dont want the full CSSStyleDeclaration object
    if (name === 'style') {
      return this._style;
    }

    return super.getConfigValue(name, options);
  } // Extract current value of all initially used configs, special handling for widget type

  getCurrentConfig(options) {
    const result = super.getCurrentConfig(options); // Always include type, except for on outermost level

    if ((options === null || options === void 0 ? void 0 : options.depth) > 0) {
      result.type = this.type;
    }

    return result;
  } //endregion

  /**
   * Get widgets elements dataset or assign to it
   * @property {Object}
   * @category DOM
   */

  get dataset() {
    return this.element.dataset;
  }

  changeDataset(dataset) {
    Object.assign(this.dataset, dataset);
  }

  get dragGhost() {
    return this.constructor._dragGhost || (this.constructor._dragGhost = DomHelper.createElement({
      // Safari won't allow dragging an empty node
      html: '\xa0',
      style: 'position:absolute;top:-10000em;left:-10000em'
    }));
  }

  updateParent(parent) {
    const {
      _element: element
    } = this;

    if (element) {
      element.classList[parent ? 'remove' : 'add'](...Widget.outerCls);
    }
  }

  get constrainTo() {
    const ret = this._constrainTo;
    return ret === undefined ? globalThis : ret;
  }

  updateCentered(value) {
    const {
      element,
      _anchorElement
    } = this;

    if (value && !this.floating && !this.positioned) {
      throw new Error('`centered` is only relevant when a Widget is `floating` or `positioned`');
    }

    if (value) {
      element.classList.add('b-centered');
      element.style.transform = element.style.left = element.style.top = '';
      _anchorElement === null || _anchorElement === void 0 ? void 0 : _anchorElement.classList.add('b-hide-display');
      element.classList.remove('b-anchored');
    } else {
      element.classList.remove('b-centered');
    }
  }
  /**
   * The child element into which content should be placed. This means where {@link #config-html} should be put,
   * or, for {@link Core.widget.Container Container}s, where child items should be rendered.
   * @property {HTMLElement}
   * @readonly
   * @category DOM
   */

  get contentElement() {
    return this.element;
  }

  get contentRange() {
    const me = this,
          {
      contentElement
    } = me,
          contentRange = me._contentRange || (me._contentRange = new Range()); // Initialize the contentRange if it's collapsed.
    // It gets collapsed if the widget's element is removed from the DOM.

    if (contentRange.collapsed) {
      contentRange.setStart(contentElement, me.contentRangeStartOffset || 0);
      contentRange.setEnd(contentElement, me.contentRangeEndOffset || contentElement.childNodes.length);
    }

    return contentRange;
  }
  /**
   * This method fixes the element's `$refOwnerId` when this instance's `id` is changing.
   * @param {Node} el The element or DOM node to fix.
   * @param {String} id The new id being assigned.
   * @param {String} oldId The old id (previously assigned).
   * @private
   */

  fixRefOwnerId(el, id, oldId) {
    if (el.$refOwnerId === oldId) {
      el.$refOwnerId = id;

      for (let {
        childNodes
      } = el, i = childNodes.length; i-- > 0;) {
        this.fixRefOwnerId(childNodes[i], id, oldId);
      }
    }
  }

  get placement() {
    const me = this,
          {
      element
    } = me;
    let adjRect, placement, rect;

    if (element !== null && element !== void 0 && element.offsetParent && !nonFlowedPositions.test(DomHelper.getStyleValue(element, 'position'))) {
      const next = element.nextElementSibling,
            previous = element.previousElementSibling,
            last = !next && previous;
      placement = DomHelper.getStyleValue(element.parentElement, 'flex-direction'); // If used in a flex layout, determine orientation from flex-direction

      if (placement) {
        placement = placement.startsWith('row') ? 'h' : 'v';
      } else {
        var _ref;

        adjRect = (_ref = next || previous) === null || _ref === void 0 ? void 0 : _ref.getBoundingClientRect();
        rect = adjRect && element.getBoundingClientRect();
        placement = adjRect && Math.abs(adjRect.top - rect.top) < Math.abs(adjRect.left - rect.left) ? 'h' : 'v'; // if there is another item, check for more horz delta than vert and if so, call it a horz container
      }

      placement += placement === 'h' ? last ? 'r' : 'l' : last ? 'b' : 't';
    }

    return placement;
  }

  updateId(id, oldId) {
    super.updateId(id, oldId);

    if (oldId) {
      // NOTE this happens when we adopt an element w/an assigned id...
      const {
        byRef,
        element
      } = this;

      for (const ref in byRef) {
        byRef[ref].dataset && (byRef[ref].dataset.ownerCmp = id); // SVG elements have no dataset
      }

      element.id = id;
      this.fixRefOwnerId(element, id, oldId);
    }
  }
  /**
   * Get/set widgets elements style. The setter accepts a cssText string or a style config object, the getter always
   * returns a CSSStyleDeclaration
   * @property {CSSStyleDeclaration}
   * @accepts {String|Object|CSSStyleDeclaration}
   * @category DOM
   */

  get style() {
    const {
      element
    } = this;
    return (element === null || element === void 0 ? void 0 : element.ownerDocument.defaultView.getComputedStyle(element)) || this._style;
  }

  updateStyle(style) {
    this.element && DomHelper.applyStyle(this.element, style);
  }

  updateTitle(title) {
    if (this.titleElement) {
      this.titleElement.innerHTML = title;
    }
  } //region floating
  // Hook used by Tooltip to handle RTL

  beforeAlignTo(spec) {}
  /**
   * If this Widget is {@link #config-floating} or {@link #config-positioned}, and visible,
   * aligns the widget according to the passed specification. To stop aligning, call this method without arguments.
   * For details, see the {@link #function-showBy} method.
   * @param {Object} [spec] Alignment options. May be an object as processed by the {@link #function-showBy} method,
   * or an `HTMLElement` to align to using this Widget's {@link #config-align} configuration.
   * @category Float & align
   */

  alignTo(spec) {
    var _anchor;

    const me = this,
          {
      lastAlignSpec,
      element
    } = me,
          {
      offsetParent,
      style,
      classList
    } = element; // Hook used by Tooltip to handle RTL

    me.beforeAlignTo(spec);

    if (lastAlignSpec) {
      // Remove intersection observation from our previous align target element.
      lastAlignSpec.monitorIntersection && me.intersectionObserver.unobserve(lastAlignSpec.target); // Ensure marker class of previous alignment side is removed.

      if (isFinite(lastAlignSpec.zone)) {
        element.classList.remove(alignedClass[lastAlignSpec.zone]);
      }
    }

    if (!spec) {
      me.removeTransientListeners();
      me.anchor = me.lastAlignSpec = null;
      return;
    } // You can "alignTo" an element or a Widget or a Point, and allow our align config.
    // property to specify how.

    if (spec.nodeType === Element.ELEMENT_NODE || spec instanceof Widget || spec.$$name === 'Point') {
      spec = {
        target: spec
      };
    } // Release size constraints so we can align from scratch each time.

    me.releaseSizeConstraints();
    const {
      scrollable,
      constrainTo
    } = me,
          positioned = me.positioned && DomHelper.getStyleValue(element, 'position') !== 'fixed',
          scale = me.scale || 1,
          passedTarget = spec.target,
          target = passedTarget && (passedTarget.isRectangle ? passedTarget : passedTarget.element || passedTarget),
          myPosition = Rectangle.from(element, positioned ? offsetParent : null, true),
          {
      width: startWidth,
      height: startHeight
    } = myPosition,
          aligningToElement = target && target.nodeType === Element.ELEMENT_NODE;
    spec = spec.realignTarget ? spec : ObjectHelper.assign({
      aligningToElement,
      constrainTo,
      align: 'b-t',
      // we can end up w/o a value for this if an object replaces a string
      axisLock: me.axisLock,
      anchor: me.anchor
    }, me.align, spec); // As part of fallback process when fitting within constraints, this may shrink to minima specified
    // either on the align spec or the widget itself.

    const minWidth = spec.minWidth || me.minWidth,
          minHeight = spec.minHeight || me.minHeight; // Minima have a different meaning in an alignRectangle.
    // It means that the rectangle is willing to shrink down
    // to that size during constraint, *not* that it can never
    // be smaller than that size.

    myPosition.isAlignRectangle = true;
    minWidth && (myPosition.minWidth = minWidth * scale);
    minHeight && (myPosition.minHeight = minHeight * scale); // This is used by the realign call which may be called either when a global scroll is detected
    // or the constraining element is resized.

    me.lastAlignSpec = spec;

    if (aligningToElement && hasLayout(target.offsetParent)) {
      // Don't destroy the spec which was cached above with the element in it.
      spec = Object.setPrototypeOf({}, spec); // If we are being called from realign, there will be a realignTarget present which is
      // a viewport-based *visible* rectangle. Otherwise translate the element into a browser
      // viewport based Rectangle. Rectangle doesn't have the knowledge that we do to make this
      // decision. Floating alignment all takes place within browser viewport space, not document
      // space.

      spec.target = me.lastAlignSpec.targetRect = spec.realignTarget || (spec.allowTargetOut ? Rectangle.from(target, positioned ? offsetParent : null, !positioned) : DomHelper.isInView(target, false, me)); // This is the case where the target is scrolled or positioned out of view.

      if (!spec.target) {
        const result = me.hide(); // The hide method clears this flag.
        // Only this hide invocation must complete with the
        // targetOutOfView flag as true
        // Hiding *might* destroy if autoClose is set.

        if (!me.isDestroyed) {
          me.lastAlignSpec.targetOutOfView = true;
        }

        return result;
      } // Force the target to have an area so that intersect works.

      spec.target.height = Math.max(spec.target.height, 1);
      spec.target.width = Math.max(spec.target.width, 1); // This is the element which determines our position.
      // This is used in doHideOrRealign to see if a scroll event
      // will have affected our position.

      me.anchoredTo = target;
    } else {
      me.anchoredTo = null;
    }

    if (spec.anchor) {
      spec.anchorSize = me.anchorSize;

      if (!element.contains(me.anchorPathElement)) {
        element.appendChild(me.anchorElement);
      }
    } // If we're positioned, any constrainTo must be a Rectangle in our offsetParent's coordinate space

    if (positioned) {
      // We can't be seen outside our offsetParent, so that's the de-facto costrainTo
      // regardless of what is passed.
      if (offsetParent && DomHelper.getStyleValue(offsetParent, 'overflow') === 'hidden') {
        spec.constrainTo = Rectangle.from(offsetParent).moveTo(0, 0);
      } else if (constrainTo && !constrainTo.isRectangle) {
        const isViewport = constrainTo === document || constrainTo === globalThis;
        spec.constrainTo = Rectangle.from(spec.constrainTo, offsetParent);

        if (isViewport) {
          spec.constrainTo.translate(globalThis.pageXOffset, globalThis.pageYOffset);
        }
      }
    } // Flag to prevent infinite loop when setting html from a beforeAlign listener

    me.isAligning = true; // Allow outside world to modify the suggested position

    me.trigger('beforeAlign', spec);
    me.isAligning = false; // Handle direction neutral edges (s & e, asserted in PopupRTL.t.js)

    if (spec.align.includes('s') || spec.align.includes('e')) {
      if (me.rtl) {
        spec.align = spec.align.replace(/s/g, 'r').replace(/e/g, 'l');
      } else {
        spec.align = spec.align.replace(/s/g, 'l').replace(/e/g, 'r');
      }
    } // Calculate the best position WRT target rectangle, our rectangle, a constrainTo rectangle
    // and the rectangle of an anchor pointer.

    const result = myPosition.alignTo(spec); // May change if constraint changes our shape, and we have to go round again

    let {
      align,
      anchor,
      x,
      y,
      width,
      height,
      overlap
    } = result; // Which zone, T=0, R=1, B=2, L=3 the result is in

    me.lastAlignSpec.zone = result.zone; // If the alignment specified that we must constrain a dimension in order to
    // fit within our constrainTo, then obey that. If we own a Scroller, then
    // inform it that we do now need to scroll that dimension.
    // These conditions are released upon each alignment call because conditions may change.

    if (height != startHeight) {
      if (!('configuredHeight' in me)) {
        me.configuredHeight = style.height;
      }

      me.height = height / scale;
      me.alignConstrained = me.alignConstrained | 1;

      if (scrollable) {
        scrollable.overflowY = true;
      }
    }

    if (width != startWidth) {
      if (!('configuredWidth' in me)) {
        me.configuredWidth = style.width;
      }

      me.width = width / scale;
      me.alignConstrained = me.alignConstrained | 2;

      if (scrollable) {
        scrollable.overflowX = scrollable.clientWidth > scrollable.scrollWidth;
      }
    } // If either dimension has been constrained, we may have changed shape
    // due to text wrapping/overflowing, so we have to realign at the
    // successful align setting.

    if (align && me.alignConstrained) {
      spec.align = align;
      const newResult = Rectangle.from(element, positioned ? offsetParent : null, true).alignTo(spec);
      anchor = newResult.anchor;
      x = newResult.x;
      y = newResult.y;
      width = newResult.width;
      height = newResult.height;
    }

    me.setXY(x, y); // Class indicates which edge of the target this is aligned to: 0, 1, 2, or 3 (TRBL)

    if (!result.overlap && isFinite(result.zone)) {
      classList.add(alignedClass[result.zone]);
    } // If we asked it to also calculate our anchor position, position our anchor.
    // If we're not edge-to-edge aligned with our target, we cannot anchor.

    if ((_anchor = anchor) !== null && _anchor !== void 0 && _anchor.edge) {
      const {
        edge
      } = anchor,
            {
        anchorElement
      } = me,
            elRect = Rectangle.from(element),
            colorMatchPoint = []; // Make the anchor color match the color of the closest adjacent element

      if (edge === 'top' || edge === 'bottom') {
        colorMatchPoint[0] = anchor.x;
        colorMatchPoint[1] = edge === 'top' ? 1 : elRect.height - 1;
      } else {
        // No RTL handling needed here as long as `s` and `e` alignment is used
        colorMatchPoint[0] = edge === 'left' ? 1 : elRect.width - 1;
        colorMatchPoint[1] = anchor.y;
      }

      let colourSource = DomHelper.childFromPoint(element, ...colorMatchPoint); // 2nd check is relevant when stylesheet fails to load

      if (colourSource && colourSource !== document) {
        let fillColour = DomHelper.getStyleValue(colourSource, 'background-color');

        while (fillColour.match(isTransparent) && DomHelper.getStyleValue(colourSource, 'position') !== 'absolute') {
          colourSource = colourSource.parentNode; // Ensure stylesheet is loaded

          if (colourSource === document) {
            break;
          }

          fillColour = DomHelper.getStyleValue(colourSource, 'background-color');
        }

        if (fillColour.match(isTransparent)) {
          me.anchorPathElement.setAttribute('fill', me.defaultAnchorBackgroundColor);
        } else {
          me.anchorPathElement.setAttribute('fill', fillColour);
        }
      }

      anchorElement.classList.remove('b-hide-display');
      anchorElement.style.transform = '';
      anchorElement.className = `b-anchor b-anchor-${edge}`; // Anchor's position needs boosting if we are scaled down

      anchor.x && (anchor.x /= scale);
      anchor.y && (anchor.y /= scale);
      DomHelper.setTranslateXY(anchorElement, anchor.x, anchor.y);
      classList.add('b-anchored');
    } else if (me._anchorElement) {
      me.anchorElement.classList.add('b-hide-display');
      classList.remove('b-anchored');
    } // If we are to hide on scroll, we still need to know if the element we are
    // aligned to moves. If we have not been *explicitly* aligned to an element,
    // Use the element at our display position. For example, when a context menu
    // is shown on a grid header, then is the grid header is moved by a scroll
    // event, then we must hide.

    if (me.scrollAction === 'hide' && !aligningToElement) {
      // Our element is over the X, Y point now,
      // elementFromPoint must "see through" it.
      style.pointerEvents = 'none';
      const el = DomHelper.elementFromPoint(x, y); // If we own the element at the point, it means we are already visible
      // and have visible descendants, so we must not update the anchoredTo

      if (!me.owns(el)) {
        me.anchoredTo = el;
      }

      style.pointerEvents = '';
    } // If we're aligning to an element, then listen for scrolls so that we can remain aligned.
    // Scrolls can be instigated with no mousedown, so transient floating Widgets can be put
    // out of alignment by scrolls.

    if ((me.scrollAction === 'realign' && aligningToElement || me.scrollAction === 'hide') && !me.documentScrollListener) {
      // Firefox requires a longer timeout to not autohide as the result of a scroll event firing during Menu show
      me.clearTimeout(me.scrollListenerTimeout);
      me.scrollListenerTimeout = me.setTimeout(() => {
        var _target$getRootNode;

        const targetRoot = target === null || target === void 0 ? void 0 : (_target$getRootNode = target.getRootNode) === null || _target$getRootNode === void 0 ? void 0 : _target$getRootNode.call(target); // Realign if the main document detects a scroll.
        // On raf to avoid scroll syncing other elements causing multiple realigns (grids body and header etc)

        me.documentScrollListener = EventHelper.addListener(document, 'scroll', 'doHideOrRealign', {
          capture: true,
          thisObj: me
        }); // In case the align target is in a WC, also capture scrolls scoped with in its shadow root

        if (targetRoot && targetRoot !== document) {
          me.targetRootScrollListener = EventHelper.addListener(targetRoot, 'scroll', 'doHideOrRealign', {
            capture: true,
            thisObj: me
          });
        }
      }, me.scrollAction === 'hide' ? me.ignoreScrollDuration : 0);
    } // If alignment specified monitorResize add a resize listener to the target so we can stay aligned.

    if (aligningToElement) {
      if (spec.monitorResize && !me.targetResizeListener) {
        ResizeMonitor.addResizeListener(target, me.onTargetResize);
        me.targetResizeListener = true;
      } // If configured to monitor intersection, and we are not potentially obscuring
      // it ourselves, and it's not an SVG element, observe its intersection changes.
      // Bug with IntersectionObserver and SVG elements, so omit them:
      // https://bugs.chromium.org/p/chromium/issues/detail?id=1159196

      if (spec.monitorIntersection && !(overlap || target.contains(element) || target.ownerSVGElement)) {
        me.intersectionObserver.observe(target);
      }
    } // Don't try to listen for window resizes to try realigning on Android.
    // That just means the keyboard has been shown.

    if (!BrowserHelper.isAndroid) {
      if (!me.constrainListeners && !(constrainTo instanceof Rectangle)) {
        // Always observe for changes to window size since aligned things
        // will possibly be out of place after a window resize
        me.clearTimeout(me.resizeListenerTimeout);
        me.resizeListenerTimeout = me.setTimeout(() => {
          me.constrainListeners = true;
          ResizeMonitor.addResizeListener(constrainTo || globalThis, me.callRealign);
        }, me.ignoreScrollDuration);
      }
    }
  }

  get intersectionObserver() {
    return this._intersectionObserver || (this._intersectionObserver = new IntersectionObserver(this.onTargetIntersectionchange.bind(this), {
      root: BrowserHelper.isSafari ? this.rootElement : this.rootElement.ownerDocument
    }));
  }

  onTargetIntersectionchange(entries) {
    if (!this.isDestroyed) {
      // It may go through several states. Only interrogate the latest.
      const e = entries[entries.length - 1];

      if (!e.isIntersecting) {
        this.onAlignTargetOutOfView(e.target);
      }
    }
  }

  onTargetResize() {
    const {
      lastAlignSpec
    } = this;

    if (lastAlignSpec) {
      const {
        width: lastWidth,
        height: lastHeight
      } = lastAlignSpec.targetRect,
            {
        width,
        height
      } = lastAlignSpec.target.getBoundingClientRect(); // If the target's outer size has changed size since alignTo measured it, realign

      if (width !== lastWidth || height !== lastHeight) {
        this.realign();
      }
    }
  }
  /**
   * This method is called when the {@link #function-alignTo} target element loses intersection with the
   * visible viewport. That means it has been scrolled out of view, or becomes zero size, or hidden or
   * is removed from the DOM.
   *
   * The base class implementation hides by default.
   * @param {HTMLElement} target The alignTo target that is no longer in view.
   * @internal
   */

  onAlignTargetOutOfView(target) {
    this.hide();
    this.lastAlignSpec && (this.lastAlignSpec.targetOutOfView = true);
  }
  /**
   * Called when an element which affects the position of this Widget's
   * {@link #function-alignTo align target} scrolls so that this can realign.
   *
   * If the target has scrolled out of view, then this Widget is hidden.
   * @internal
   */

  realign() {
    const me = this,
          {
      lastAlignSpec
    } = me;

    if (me.isVisible && (me.floating || me.positioned) && lastAlignSpec) {
      if (lastAlignSpec.aligningToElement) {
        const insideTarget = lastAlignSpec.target.contains(this.element),
              realignTarget = DomHelper.isInView(lastAlignSpec.target, false, me); // If the target that we are realigning to is not in view, we hide, and set the
        // flag in the lastAlignSpec to explain why

        if (!lastAlignSpec.allowTargetOut && (!hasLayout(lastAlignSpec.target) || !realignTarget)) {
          me.hide(); // Hiding *might* destroy if autoClose is set.

          if (!me.isDestroyed) {
            me.lastAlignSpec.targetOutOfView = true;
          }

          return;
        } // We use a different align target when *re*aligning. It's the *visible* rectangle.
        // Unless we re inside the target, in which case the target itself is used.

        lastAlignSpec.realignTarget = insideTarget ? null : realignTarget;
      }

      DomHelper.addTemporaryClass(me.element, 'b-realigning', 300);
      me.alignTo(lastAlignSpec);
    }
  }
  /**
   * Returns the specified bounding rectangle of this widget.
   * @param {String} [which] By default, the rectangle returned is the bounding rectangle that contains the `element`
   * border. Pass any of these values to retrieve various rectangle:
   *  - `'border'` to get the {@link Core.helper.util.Rectangle#function-from-static border rectangle} (the default).
   *  - `'client'` to get the {@link Core.helper.util.Rectangle#function-client-static client rectangle}.
   *  - `'content'` to get the {@link Core.helper.util.Rectangle#function-content-static content rectangle}.
   *  - `'inner'` to get the {@link Core.helper.util.Rectangle#function-inner-static inner rectangle}.
   *  - `'outer'` to get the {@link Core.helper.util.Rectangle#function-outer-static outer rectangle}.
   * @param {HTMLElement|Core.widget.Widget} [relativeTo] Optionally, a parent element or widget in whose space to
   * calculate the Rectangle.
   * @param {Boolean} [ignorePageScroll=false] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle}
   * @internal
   */

  rectangle(which, relativeTo, ignorePageScroll) {
    return this.rectangleOf('element', which, relativeTo, ignorePageScroll);
  }
  /**
   * Returns the specified bounding rectangle of the specified child `element` of this widget.
   * @param {String} [element] The child element name.
   * @param {String} [which] By default, the rectangle returned is the bounding rectangle that contains the `element`
   * border. Pass any of these values to retrieve various rectangle:
   *  - `'border'` to get the {@link Core.helper.util.Rectangle#function-from-static border rectangle} (the default).
   *  - `'client'` to get the {@link Core.helper.util.Rectangle#function-client-static client rectangle}.
   *  - `'content'` to get the {@link Core.helper.util.Rectangle#function-content-static content rectangle}.
   *  - `'inner'` to get the {@link Core.helper.util.Rectangle#function-inner-static inner rectangle}.
   *  - `'outer'` to get the {@link Core.helper.util.Rectangle#function-outer-static outer rectangle}.
   * @param {HTMLElement|Core.widget.Widget} [relativeTo] Optionally, a parent element or widget in whose space to
   * calculate the Rectangle. If `element` is not `'element'`, then this defaults to the widget's primary element.
   * @param {Boolean} [ignorePageScroll=false] Use browser viewport based coordinates.
   * @returns {Core.helper.util.Rectangle}
   * @internal
   */

  rectangleOf(element, which, relativeTo, ignorePageScroll) {
    var _relativeTo;

    if (typeof which !== 'string') {
      ignorePageScroll = relativeTo;
      relativeTo = which;
      which = '';
    } else if (which === 'border') {
      which = '';
    } // which is locked in

    if (typeof relativeTo === 'boolean') {
      ignorePageScroll = relativeTo;
      relativeTo = undefined;
    }

    if (element !== 'element' && relativeTo === undefined) {
      relativeTo = this.element;
    }

    relativeTo = (_relativeTo = relativeTo) !== null && _relativeTo !== void 0 && _relativeTo.isWidget ? relativeTo.element : relativeTo;
    return Rectangle[which || 'from'](this[element], relativeTo, ignorePageScroll);
  }

  releaseSizeConstraints() {
    const me = this,
          scroller = me.scrollable; // Release constraints so we can align from scratch each time.

    if (me.alignConstrained & 1) {
      me.height = me.configuredHeight;

      if (scroller) {
        scroller.overflowY = scroller.config.overflowY;
      }
    }

    if (me.alignConstrained & 2) {
      me.width = me.configuredWidth;

      if (scroller) {
        scroller.overflowX = scroller.config.overflowX;
      }
    }

    me.alignConstrained = 0;
  }
  /**
   * Only valid for {@link #config-floating} Widgets. Moves to the front of the visual stacking order.
   * @category Float & align
   */

  toFront() {
    const {
      element
    } = this,
          parent = this.floating ? this.floatRoot : this.positioned ? element === null || element === void 0 ? void 0 : element.parentNode : null;

    if (element !== null && element !== void 0 && element.nextSibling) {
      const r = this._toFrontRange || (this._toFrontRange = document.createRange()); // Instead of moving this element (which may contain focus and therefore should not be moved),
      // Lasso all following nodes using a range and insert them before this.
      // If it contained focus, appending it triggers a focusOut event which will not be expected.

      r.setStartBefore(element.nextSibling);
      r.setEndAfter(parent.lastElementChild);
      parent.insertBefore(r.extractContents(), element);
    }
  } //endregion
  //region Getters/setters

  updateRef(ref) {
    this.element.dataset.ref = ref;
  }
  /**
   * The child element which scrolls if any. This means the element used by the {@link #config-scrollable}.
   * @property {HTMLElement}
   * @readonly
   * @category DOM
   */

  get overflowElement() {
    return this.contentElement;
  }

  get maxHeightElement() {
    return this.element;
  }

  changeAlign(align) {
    return typeof align === 'string' ? {
      align
    } : align;
  }

  changeScrollable(scrollable, oldScrollable) {
    if (typeof scrollable === 'boolean') {
      scrollable = {
        overflowX: scrollable,
        overflowY: scrollable
      };
    }

    if (scrollable) {
      scrollable.element = this.overflowElement;
      scrollable.widget = this;

      if (!scrollable.isScroller) {
        scrollable = oldScrollable ? oldScrollable.setConfig(scrollable) : new this.scrollerClass(scrollable);
      } // Keep overflow indicator classes in sync

      scrollable.syncOverflowState();
    } // Destroy the old scroller if the scroller is being nulled.
    else {
      oldScrollable === null || oldScrollable === void 0 ? void 0 : oldScrollable.destroy();
    }

    return scrollable;
  }
  /**
   * Get/set HTML to display. When specifying HTML, this widget's element will also have the
   * {@link #config-htmlCls} added to its classList, to allow targeted styling.
   * @property {String}
   * @category DOM
   */

  get html() {
    // Maintainer, we cannot use a ternary here, we need the this.initializingElement test to shortcut
    // to the true case to return the _html property to avoid infinite loops.
    if (this.initializingElement || !this.element) {
      return this.content || this._html;
    }

    return this.contentElement.innerHTML;
  }

  updateHtml(html) {
    const me = this,
          isClearing = html == null,
          {
      element,
      contentElement,
      htmlCls
    } = me;

    if (element) {
      // So that our contentElement MutationObserver doesn't react
      me.updatingHtml = true;
      const anchorEl = contentElement === element && me._anchorElement; // Flag class that we are an HTML carrying element

      if (htmlCls) {
        // Salesforce doesn't support passing array
        htmlCls.values.forEach(value => element.classList[isClearing ? 'remove' : 'add'](value));
      } // Setting innerHTML destroys the anchorElement in some browsers
      // so we must temporarily remove it to preserve it.
      // Only if the contentElement is the main element.

      if (anchorEl) {
        me.element.removeChild(anchorEl);
      }

      if (html && typeof html === 'object') {
        DomSync.sync({
          domConfig: html,
          targetElement: me.contentElement,
          onlyChildren: true
        });
      } else {
        me.contentElement.innerHTML = isClearing ? '' : html;
      } // Ensure our content mutation observer keeps us informed of changes by third parties
      // so that our config system can keep up to date.

      me.getConfig('htmlMutationObserver');

      if (anchorEl) {
        element.appendChild(anchorEl);
      }

      if (me.isComposable) {
        me.recompose();
      } else if (me.floating || me.positioned) {
        // Must realign because content change might change dimensions
        if (!me.isAligning) {
          me.realign();
        }
      }
    }
  }

  changeHtmlMutationObserver(htmlMutationObserver, was) {
    const me = this,
          {
      contentElement
    } = me; // Clean up old one

    was === null || was === void 0 ? void 0 : was.disconnect(contentElement); // Create MutationObserver

    if (htmlMutationObserver) {
      const result = new MutationObserver(() => {
        if (me.updatingHtml) {
          me.updatingHtml = false;
        } else {
          me._html = contentElement.innerHTML;
        }
      });
      result.observe(contentElement, htmlMutationObserver);
      return result;
    }
  }

  updateContent(html) {
    const me = this,
          isClearing = html == null,
          {
      element,
      htmlCls
    } = me;

    if (element) {
      const {
        contentRange
      } = me; // Flag class that we are an HTML carrying element

      if (htmlCls) {
        // Salesforce doesn't support passing array
        htmlCls.values.forEach(value => element.classList[isClearing ? 'remove' : 'add'](value));
      } // Only works if we are in the DOM

      if (isInDocument(element)) {
        // Replace the contents of our content range with the new content
        contentRange.deleteContents();

        if (!isClearing) {
          contentRange.insertNode(DomHelper.createElementFromTemplate(html, {
            fragment: true
          }));
        }
      } else {
        me.contentElement.innerHTML = html;
      } // Cache in case it gets collapsed

      me.contentRangeStartOffset = contentRange.startOffset;
      me.contentRangeEndOffset = contentRange.endOffset; // Must realign because content change might change dimensions

      if ((me.floating || me.positioned) && !me.isAligning) {
        me.realign();
      }
    }
  }

  onThemeChange() {
    var _this$anchorElement;

    // If we have a *visible* anchor element, then a theme change may
    // invalidate it's size or this.defaultAnchorBackgroundColor, so a
    // run through realign (and get anchorSize) will fix that.
    if ((_this$anchorElement = this.anchorElement) !== null && _this$anchorElement !== void 0 && _this$anchorElement.offsetParent) {
      this._anchorSize = null;
      this.realign();
    }
  }
  /**
   * Returns an `[x, y]` array containing the width and height of the anchor arrow used when
   * aligning this Widget to another Widget or element.
   *
   * The height is the height of the arrow when pointing upwards, the width is the width
   * of the baseline.
   * @property {Number[]}
   * @category Float & align
   */

  get anchorSize() {
    const me = this;
    let result = this._anchorSize;

    if (!result) {
      // TODO: Move the anchoring scheme to the Panel class when we have it.
      // These values will be in the SASS and styled into the SVG through the Panel's theme.
      const borderWidth = parseFloat(DomHelper.getStyleValue(me.element, 'border-top-width')),
            borderColour = DomHelper.getStyleValue(me.element, 'border-top-color'),
            anchorElement = me.anchorElement,
            {
        className
      } = anchorElement,
            svgEl = anchorElement.firstElementChild,
            pathElement = me.anchorPathElement = svgEl.lastElementChild,
            hidden = me._hidden; // In case we are measuring after the size has been invalidated (such as via theme change)
      // and the widget is shown and aligned left or right. We must measure it in top alignment
      // so as to get the dimensions the right way round.

      anchorElement.className = 'b-anchor b-anchor-top';
      let backgroundColour = DomHelper.getStyleValue(me.contentElement, 'background-color'); // If the background colour comes through from the outer element, use that.

      if (backgroundColour.match(isTransparent)) {
        backgroundColour = DomHelper.getStyleValue(me.element, 'background-color');
      }

      me.defaultAnchorBackgroundColor = backgroundColour;
      result = anchorElement.getBoundingClientRect();
      const [width, height] = result = me._anchorSize = [result.width, result.height]; // Restore orientation

      anchorElement.className = className;
      svgEl.setAttribute('height', height + borderWidth);
      svgEl.setAttribute('width', width);
      pathElement.setAttribute('d', `M0,${height}L${width / 2},0.5L${width},${height}`);

      if (borderWidth) {
        pathElement.setAttribute('stroke-width', borderWidth);
        pathElement.setAttribute('stroke', borderColour);
      }

      result[1] -= borderWidth;

      if (hidden) {
        me.element.classList.add('b-hidden');
      }

      if (!me.themeChangeListener) {
        me.themeChangeListener = GlobalEvents.on({
          theme: 'onThemeChange',
          thisObj: me
        });
      } // Reset to default in case it has been positioned by a coloured header

      me.anchorPathElement.setAttribute('fill', me.defaultAnchorBackgroundColor);
    }

    return result;
  }

  get anchorElement() {
    const me = this;

    if (!me._anchorElement) {
      const useFilter = me.floating,
            filterId = `${me.id}-shadow-filter`;
      me._anchorElement = DomHelper.createElement({
        parent: me.element,
        className: 'b-anchor b-anchor-top',
        children: [{
          tag: 'svg',
          ns: 'http://www.w3.org/2000/svg',
          version: '1.1',
          class: 'b-pointer-el',
          children: [useFilter ? {
            tag: 'defs',
            children: [{
              tag: 'filter',
              id: filterId,
              children: [{
                tag: 'feDropShadow',
                dx: 0,
                dy: -1,
                stdDeviation: 1,
                'flood-opacity': 0.2
              }]
            }]
          } : null, {
            tag: 'path',
            [useFilter ? 'filter' : '']: `url(#${filterId})`
          }]
        }]
      });
    }

    return me._anchorElement;
  }

  updateAnchor(anchor) {
    if (this._anchorElement) {
      this._anchorElement.classList[anchor ? 'remove' : 'add']('b-hide-display');
    }
  }

  updateDraggable(draggable) {
    const me = this,
          {
      element
    } = me;

    if (draggable) {
      me.dragEventDetacher = EventHelper.addListener({
        element,
        dragstart: 'onWidgetDragStart',
        dragend: 'onWidgetDragEnd',
        thisObj: me
      });
      me.dragDetacher = EventHelper.on({
        element,

        mousedown(event) {
          const {
            target
          } = event,
                closestWidget = Widget.fromElement(target); // Fix for FF draggable bug https://bugzilla.mozilla.org/show_bug.cgi?id=1189486

          if (!DomHelper.up(event.target, '.b-field-inner') && // Only allow drag to start when the action originates from the widget element itself,
          // or one of its toolbars. https://github.com/bryntum/support/issues/3214
          closestWidget === this || this.strips && Object.values(this.strips).includes(closestWidget)) {
            element.setAttribute('draggable', 'true');
          }
        },

        // Only needed for automatic listener removal on the destroy of the thisObj
        thisObj: me
      });
    } else {
      var _me$dragEventDetacher2, _me$dragOverEventDeta2, _me$dragDetacher;

      (_me$dragEventDetacher2 = me.dragEventDetacher) === null || _me$dragEventDetacher2 === void 0 ? void 0 : _me$dragEventDetacher2.call(me);
      (_me$dragOverEventDeta2 = me.dragOverEventDetacher) === null || _me$dragOverEventDeta2 === void 0 ? void 0 : _me$dragOverEventDeta2.call(me);
      (_me$dragDetacher = me.dragDetacher) === null || _me$dragDetacher === void 0 ? void 0 : _me$dragDetacher.call(me);
    }
  }

  onWidgetDragStart(e) {
    var _ref2;

    const me = this;

    if (!me.validateDragStartEvent(e)) {
      return;
    } // Opt out of auto-alignment on scroll or DOM mutation.

    me.removeTransientListeners();
    const {
      element,
      align,
      constrainTo
    } = me,
          positioned = me.positioned && DomHelper.getStyleValue(element, 'position') !== 'fixed',
          parentElement = positioned ? element.parentElement : me.rootElement,
          myRect = Rectangle.from(element, positioned ? parentElement : null),
          dragStartX = e.clientX,
          dragStartY = e.clientY,
          scrollingPageElement = document.scrollingElement || document.body,
          [widgetX, widgetY] = DomHelper.getOffsetXY(element, parentElement),
          constrainRect = (_ref2 = positioned ? Rectangle.content(parentElement).moveTo(0, 0) : constrainTo && (constrainTo !== null && constrainTo !== void 0 && constrainTo.isRectangle ? constrainTo : Rectangle.from(constrainTo))) === null || _ref2 === void 0 ? void 0 : _ref2.deflate(align.constrainPadding || 0),
          dragListeners = {
      element: parentElement,
      dragover: event => {
        // Centered adds positioning rules, it can't be centered during drag.
        element.classList.remove('b-centered'); // Shift our rectangle to the desired point.

        myRect.moveTo(widgetX + event.clientX - dragStartX - (positioned ? 0 : scrollingPageElement.scrollLeft), widgetY + event.clientY - dragStartY - (positioned ? 0 : scrollingPageElement.scrollTop)); // Constrain it if we are configured to be constrained

        if (constrainRect) {
          myRect.constrainTo(constrainRect);
        } // Position using direct DOM access, do not go though the setXY method which clears
        // any centered config. User dragging only moves this show of the widget. Upon next
        // neutral show (with no extra positioning info), a centered widget will show centered again.

        DomHelper.setTranslateXY(element, myRect.x, myRect.y);
      }
    }; // Stop viewport panning on drag on touch devices

    if (BrowserHelper.isTouchDevice) {
      dragListeners.touchmove = e => e.preventDefault();
    }

    me.floatRoot.appendChild(me.dragGhost);
    me.setDragImage(e); // Prevent special cursor from being shown

    e.dataTransfer.effectAllowed = 'none';
    me.dragOverEventDetacher = EventHelper.addListener(dragListeners); // Various app events (Such as resize or visible child count change) can
    // cause a request to realign, so opt out of anchoring and alignedness
    // until we are next hidden.

    me.alignTo();
  }
  /**
   * Validates a `dragstart` event with respect to the target element. Dragging is not normally
   * initiated when the target is interactive such as an input field or its label, or a button.
   * This may be overridden to provide custom drag start validation.
   * @param {DragEvent} e The `dragstart` event to validate.
   * @returns {Boolean} Return `true` if the drag is to be allowed.
   * @internal
   */

  validateDragStartEvent(e) {
    const me = this,
          {
      element
    } = me,
          actualTarget = DomHelper.elementFromPoint(e.clientX, e.clientY),
          // Can't be resolved from the event :/
    {
      handleSelector
    } = me.draggable;

    if (handleSelector) {
      var _negationPseudo$exec;

      const blacklist = (_negationPseudo$exec = negationPseudo.exec(handleSelector)) === null || _negationPseudo$exec === void 0 ? void 0 : _negationPseudo$exec[1]; // Extract the content of :not()
      // If the selector was :not(), then if we are a descendant of a matching element, it's a no-drag

      if (blacklist) {
        if (actualTarget.closest(`#${element.id} ${blacklist}`)) {
          e.preventDefault();
          return false;
        }
      } // If we are not the descendant of a matching element, it's a no-drag
      else if (!actualTarget.closest(`#${element.id} ${handleSelector}`)) {
        e.preventDefault();
        return false;
      }
    }

    return true;
  }

  setDragImage(e) {
    if (e.dataTransfer.setDragImage) {
      // Firefox requires this to be called before setDragImage
      e.dataTransfer.setData('application/node type', ''); // Override the default HTML5 drag ghost and just drag an empty node.
      // The large offset will cause it to be displayed offscreen on platforms
      // which will not hide drag images (iOS)

      e.dataTransfer.setDragImage(this.dragGhost, -9999, -9999);
    }
  }

  onWidgetDragEnd() {
    this.dragGhost.remove();
    this.dragOverEventDetacher();
    this.element.removeAttribute('draggable');
  }

  changeFloating(value) {
    // Coerce all to boolean so that we have a true/false value
    return Boolean(value);
  }

  changePositioned(value) {
    // Coerce all to boolean so that we have a true/false value
    return Boolean(value);
  }

  updatePositioned(positioned) {
    this.element.classList[positioned ? 'add' : 'remove']('b-positioned');
  }

  getXY() {
    return [DomHelper.getPageX(this.element), DomHelper.getPageY(this.element)];
  }
  /**
   * Moves this Widget to the x,y position. Both arguments can be omitted to just set one value.
   *
   * *For {@link #config-floating} Widgets, this is a position in the browser viewport.*
   * *For {@link #config-positioned} Widgets, this is a position in the element it was rendered into.*
   *
   * @param {Number} [x]
   * @param {Number} [y]
   * @category Float & align
   */

  setXY(x, y) {
    const me = this,
          {
      element
    } = me;

    if (me.floating || me.positioned) {
      if (x != null) {
        me._x = x;
        me.centered = false;
      }

      if (y != null) {
        me._y = y;
        me.centered = false;
      } // If we're position:fixed then it is positioned relative to either the viewport
      // or an ancestor which has a transform, perspective of filter property.
      // See https://developer.mozilla.org/en-US/docs/Web/CSS/position.
      // So translate it *relative* to its actual position/

      if (DomHelper.getStyleValue(element, 'position') === 'fixed') {
        const r = element.getBoundingClientRect(),
              [cx, cy] = DomHelper.getTranslateXY(element),
              xDelta = x - r.x,
              yDelta = y - r.y;
        DomHelper.setTranslateXY(element, cx + xDelta, cy + yDelta);
      } else {
        DomHelper.setTranslateXY(element, me._x || 0, me._y || 0);
      }
    }
  }
  /**
   * Moves this Widget to the desired x position.
   *
   * Only valid if this Widget is {@link #config-floating} and not aligned or anchored to an element.
   * @property {Number}
   * @category Float & align
   */

  get x() {
    return this.getXY()[0];
  }

  changeX(x) {
    this.setXY(x);
  }
  /**
   * Moves this Widget to the desired y position.
   *
   * Only valid if this Widget is {@link #config-floating} and not aligned or anchored to an element.
   * @property {Number}
   * @category Float & align
   */

  get y() {
    return this.getXY()[1];
  }

  changeY(y) {
    this.setXY(null, y);
  }
  /**
   * Get elements offsetWidth or sets its style.width, or specified width if element not created yet.
   * @property {Number}
   * @accepts {Number|String}
   * @category Layout
   */

  get width() {
    const me = this,
          element = me.element;

    if (me.monitorResize) {
      // If the width is invalid, read it now.
      if (me._width == null) {
        me._width = element.offsetWidth;
      } // Usually this will be set in onInternalResize

      return me._width;
    } // No monitoring, we have to measure;

    return element.offsetWidth;
  }

  changeWidth(width) {
    const me = this;
    DomHelper.setLength(me.element, 'width', width);
    me._lastWidth = width; // Invalidate the width, so it will be read from the DOM if a read is requested before the resize event

    me._width = null; // Setting width explicitly should reset flex, since it's not flexed anymore

    me._flex = null;
    me.element.style.flex = '';
  }
  /**
   * Get/set elements maxWidth. Getter returns max-width from elements style, which is always a string. Setter accepts
   * either a String or a Number (which will have 'px' appended). Note that like {@link #config-width},
   * _reading_ the value will return the numeric value in pixels.
   * @property {String}
   * @accepts {String|Number}
   * @category Layout
   */

  get maxWidth() {
    return DomHelper.measureSize(this.element.style.maxWidth, this.element);
  }

  updateMaxWidth(maxWidth) {
    DomHelper.setLength(this.element, 'maxWidth', maxWidth);
  }
  /**
   * Get/set elements minWidth. Getter returns min-width from elements style, which is always a string. Setter accepts
   * either a String or a Number (which will have 'px' appended). Note that like {@link #config-width},
   * _reading_ the value will return the numeric value in pixels.
   * @property {String}
   * @accepts {String|Number}
   * @category Layout
   */

  get minWidth() {
    return DomHelper.measureSize(this.element.style.minWidth, this.element);
  }

  updateMinWidth(minWidth) {
    DomHelper.setLength(this.element, 'minWidth', minWidth);
  }

  updateFlex(flex) {
    // Width must be processed first, because its changer clears flex because flex wins over width;
    // The assumption that the containing element's flex-direction is 'row'
    // seems dodgy.
    this.getConfig('width'); // Default grow to the same as flex and basis to 0.

    if (typeof flex === 'number' || !isNaN(flex)) {
      flex = `${flex} ${flex}`;
    }

    this.element.style.flex = flex;
    this.element.style.width = '';
  }

  updateAlignSelf(alignSelf) {
    this.element.style.alignSelf = alignSelf;
  }

  updateMargin(margin) {
    // Convert eg 1 to "1px 1px 1px 1px" or "0 8px" to "0px 8px 0px 8px"
    this.element.style.margin = this.parseTRBL(margin).join(' ');
  }

  updateTextAlign(align, oldAlign) {
    oldAlign && this.element.classList.remove(`b-text-align-${oldAlign}`);
    this.element.classList.add(`b-text-align-${align}`);
  }

  updatePlaceholder(placeholder) {
    if (this.input) {
      if (placeholder == null) {
        this.input.removeAttribute('placeholder');
      } else {
        this.input.placeholder = placeholder;
      }
    }
  }
  /**
   * Get element's offsetHeight or sets its style.height, or specified height if element no created yet.
   * @property {Number}
   * @accepts {Number|String}
   * @category Layout
   */

  get height() {
    const me = this,
          element = me.element;

    if (me.monitorResize) {
      // If the height is invalid, read it now.
      if (me._height == null) {
        me._height = element.offsetHeight;
      } // Usually this will be set in onInternalResize

      return me._height;
    } // No monitoring, we have to measure;

    return element.offsetHeight;
  }

  changeHeight(height) {
    DomHelper.setLength(this.element, 'height', height);
    this._lastHeight = height; // Invalidate the height, so it will be read from the DOM if a read is requested before the resize event

    this._height = null;
  }
  /**
   * Get/set element's maxHeight. Getter returns max-height from elements style, which is always a string. Setter
   * accepts either a String or a Number (which will have 'px' appended). Note that like {@link #config-height},
   * _reading_ the value will return the numeric value in pixels.
   * @property {String}
   * @accepts {String|Number}
   * @category Layout
   */

  get maxHeight() {
    return DomHelper.measureSize(this.maxHeightElement.style.maxHeight, this.element);
  }

  updateMaxHeight(maxHeight) {
    DomHelper.setLength(this.maxHeightElement, 'maxHeight', maxHeight);
  }
  /**
   * Get/set element's minHeight. Getter returns min-height from elements style, which is always a string. Setter
   * accepts either a String or a Number (which will have 'px' appended). Note that like {@link #config-height},
   * _reading_ the value will return the numeric value in pixels.
   * @property {String}
   * @accepts {String|Number}
   * @category Layout
   */

  get minHeight() {
    return DomHelper.measureSize(this.element.style.minHeight, this.element);
  }

  updateMinHeight(minHeight) {
    DomHelper.setLength(this.element, 'minHeight', minHeight);
  }

  updateDisabled(disabled = false) {
    const {
      element,
      focusElement,
      ariaElement
    } = this;
    this.trigger('beforeUpdateDisabled', {
      disabled
    });
    disabled && this.revertFocus();

    if (element) {
      element.classList[disabled ? 'add' : 'remove']('b-disabled');

      if (focusElement) {
        focusElement.disabled = disabled;
      }

      if (ariaElement) {
        ariaElement.setAttribute('aria-disabled', disabled);
      }
    }

    this.onDisabled(disabled);
  }
  /**
   * Called when disabled state is changed.
   * Override in subclass that needs special handling when being disabled.
   * @param {Boolean} disabled current state
   * @private
   */

  onDisabled(disabled) {}
  /**
   * Disable the widget
   */

  disable() {
    this.disabled = true;
  }
  /**
   * Enable the widget
   */

  enable() {
    this.disabled = false;
  }
  /**
   * Requests fullscreen display for this widget
   * @returns {Promise} A Promise which is resolved with a value of undefined when the transition to full screen is complete.
   */

  requestFullscreen() {
    const me = this,
          // If we are floating, target the float root as the fullscreen element
    result = Fullscreen.request(me.floating ? me.floatRoot : me.element);
    Fullscreen.onFullscreenChange(me.onFullscreenChange);

    if (!me.floating) {
      me.floatRoot._oldParent = me.floatRoot.parentElement;
      me.element.appendChild(me.floatRoot);
    }

    me.element.classList.add('b-fullscreen');
    return result;
  }
  /**
   * Exits fullscreen mode
   * @returns {Promise} A Promise which is resolved once the user agent has finished exiting full-screen mode
   */

  exitFullscreen() {
    return Fullscreen.exit();
  }

  onFullscreenChange() {
    if (!Fullscreen.isFullscreen) {
      this.onExitFullscreen();
    }
  }

  onExitFullscreen() {
    var _me$_rootElement;

    const me = this,
          floatRoot = (_me$_rootElement = me._rootElement) === null || _me$_rootElement === void 0 ? void 0 : _me$_rootElement.floatRoot;
    Fullscreen.unFullscreenChange(me.onFullscreenChange);
    me.element.classList.remove('b-fullscreen'); // When exiting fullscreen mode or when this widget is destroyed, move floatRoot back to its old parent

    if (me.element.contains(floatRoot) && floatRoot !== null && floatRoot !== void 0 && floatRoot._oldParent) {
      floatRoot._oldParent.appendChild(floatRoot);

      floatRoot._oldParent = null;
    }
  }
  /**
   * Get/set a tooltip on the widget. Accepts a string or tooltip config (specify true (or 'true') to use placeholder
   * as tooltip). When using a string it will configure the tooltip with `textContent: true` which enforces a default
   * max width.
   *
   * By default, this uses a singleton Tooltip instance which may be accessed from the
   * `{@link Core.widget.Widget Widget}` class under the name `Widget.tooltip`.
   * This is configured according to the config object on pointer over.
   *
   * To request a separate instance be created just for this widget, add `newInstance : true`
   * to the configuration.
   *
   * @property {String|Object}
   * @category Misc
   */

  get tooltip() {
    if (this._tooltip) {
      return this._tooltip;
    } else {
      var _Widget$Tooltip2;

      const tooltip = (_Widget$Tooltip2 = Widget.Tooltip) === null || _Widget$Tooltip2 === void 0 ? void 0 : _Widget$Tooltip2.getSharedTooltip(this.rootElement); // If the shared tooltip is currently in use by us, return it.
      // If it's not in use by us, we don't have a tooltip.

      if (tooltip && tooltip.activeTarget === this._element && tooltip.isVisible) {
        return tooltip;
      }
    }
  } //noinspection JSAnnotator

  changeTooltip(tooltip, oldTooltip) {
    const me = this,
          {
      element
    } = me;

    if (tooltip) {
      if (!(me.preventTooltipOnTouch && BrowserHelper.isTouchDevice)) {
        if (!tooltip.isTooltip && tooltip.constructor.name !== 'Object') {
          tooltip = {
            html: typeof tooltip === 'string' ? tooltip : me.placeholder,
            textContent: true
          };
        } // Tooltip text becomes ariaDescription unless we already have ariaDescription configured.
        // If it is localized using Ⳑ{key}, it will need to be converted to Ⳑ{Tooltip.key}
        // so that when we come to resolve it, localization looks in the right place.

        if (!me.configureAriaDescription) {
          var _tooltip$html;

          me.ariaDescription = (_tooltip$html = tooltip.html) !== null && _tooltip$html !== void 0 && _tooltip$html.match(localizeRE) ? tooltip.html.replace(localizeRE, localizeTooltip) : tooltip.html;
        } // We have to explicitly request a new instance to avoid spam Tooltip instances.
        // If there is an incoming oldTooltip, then we own a newInstance.
        // TODO verify .isTooltip check is needed

        if (oldTooltip !== null && oldTooltip !== void 0 && oldTooltip.isTooltip || tooltip.newInstance) {
          tooltip.type = 'tooltip';
          if (!tooltip.forElement) tooltip.forElement = element;
          if (!('showOnHover' in tooltip) && !tooltip.forSelector) tooltip.showOnHover = true;
          if (!('autoClose' in tooltip)) tooltip.autoClose = true;
          tooltip = Widget.reconfigure(oldTooltip, tooltip, me); // We need to update our ariaDescription when the tooltip changes

          me.detachListeners('tooltipValueListener');

          if (!me.configureAriaDescription) {
            tooltip.on({
              name: 'tooltipValueListener',
              innerHtmlUpdate: 'onTooltipValueChange',
              thisObj: me
            });
          }
        } // The default is that tooltip content and configs from tipConfig
        else {
          element.dataset.btip = true;
          me.tipConfig = tooltip; // We do not set our property if we are sharing the singleton

          return;
        }
      }
    } else {
      // If there is an incoming oldTooltip, then we own a newInstance.
      // Only destroy it if it's being set to null. Empty string
      // just means clear its content.
      if (oldTooltip) {
        if (tooltip == null && oldTooltip.isTooltip) {
          oldTooltip.destroy();
        } else {
          // We do not update the property if we are just clearing its content
          oldTooltip.html = null;
          return;
        }
      } // We are sharing, so just clear the btip
      else {
        delete element.dataset.btip;
      }
    }

    return tooltip;
  } // If our tooltip is dynamic, then we must update our aria-describedBy whenever it changes.

  onTooltipValueChange({
    value
  }) {
    this.ariaDescription = value;
  }

  get tooltipText() {
    const tooltip = this._tooltip;

    if (tooltip) {
      return tooltip.isTooltip ? tooltip.contentElement.innerText : typeof tooltip === 'string' ? tooltip : tooltip.html;
    } else if (this.tipConfig) {
      return this.tipConfig.html;
    }
  }
  /**
   * Determines visibility by checking if the Widget is hidden, or any ancestor is hidden and that it has an
   * element which is visible in the DOM
   * @property {Boolean}
   * @category Visibility
   * @readonly
   */

  get isVisible() {
    const me = this,
          {
      element
    } = me; // Added so that we only acquire owner once. `get owner()` *may* have to search DOM

    let owner; // If we are hidden, or destroying, or any ancestors are hidden, we're not visible

    return Boolean(element && !me._hidden && !me.isDestroying && isInDocument(element) && (!me.requireSize || hasLayout(element)) && (!(owner = me.containingWidget) || owner.isVisible));
  }
  /**
   * Focuses this widget if it has a focusable element.
   */

  focus() {
    if (this.isFocusable) {
      DomHelper.focusWithoutScrolling(this.focusElement);
    }
  }
  /**
   * Get this widget's primary focus holding element if this widget is itself focusable, or contains focusable widgets.
   * @property {HTMLElement}
   * @readonly
   * @category DOM
   */

  get focusElement() {// Override in widgets which are focusable.
  }

  get isFocusable() {
    // Not focusable if we are in a destroy sequence or are disabled or not visible.
    const focusElement = !this.isDestroying && this.isVisible && !this.disabled && this.focusElement; // We are only focusable if the focusEl is deeply visible, that means
    // it must have layout - an offsetParent. Body does not have offsetParent.

    return focusElement && (focusElement === document.body || focusElement.offsetParent);
  }
  /**
   * Shows this widget
   * @category Visibility
   * @async
   * @returns {Promise} A promise which is resolved when the widget is shown
   */

  async show() {
    const me = this,
          {
      showAnimation,
      element,
      floating
    } = me,
          {
      style
    } = element;
    let styleProp, animProps, trigger;

    if (trigger = !me.isVisible) {
      /**
       * Triggered before a widget is shown. Return `false` to prevent the action.
       * @preventable
       * @async
       * @event beforeShow
       * @param {Core.widget.Widget} source The widget being shown.
       */
      trigger = me.trigger('beforeShow');

      if (Objects.isPromise(trigger)) {
        trigger = await trigger;
      }
    }

    if (trigger !== false && (!me.internalBeforeShow || me.internalBeforeShow() !== false)) {
      return new Promise(resolve => {
        // Cancel any current hide/show animation
        me.cancelHideShowAnimation(); // Centered config value takes precedence over x and y configs.
        // This also ensures that widgets configured with centered: true
        // and draggable : true will show in the center on next show after
        // being dragged by the user which is the intuitive UX.

        me.updateCentered(me._centered);

        if (floating) {
          const floatRoot = me.floatRoot;

          if (!floatRoot.contains(element)) {
            // Replace this Widget's DOM into the container if it's already rendered
            if (me.rendered) {
              floatRoot.appendChild(me.element);
            } else {
              // Pass triggerPaint as false. The calls will not propagate
              // anyway since we are still hidden.
              me.render(floatRoot, false);
            }
          } // Because we are outside of any owner's element, we need to see if they're scaled so
          // that we match. See scaled examples with tooltips in API docs guides section.

          if (style.transform.includes('scale')) {
            me.scale = null;
            style.transform = style.transformOrigin = '';
          }

          const scaledAncestor = me.closest(isScaled);

          if (scaledAncestor) {
            const {
              scale
            } = scaledAncestor; // Our scale is the same while we are visible and owned by that scaled ancestor.
            // Now floating descendants will follow suit.

            me.scale = scale;
            style.transform = `scale(${scale})`;
            style.transformOrigin = `0 0`;
          }
        }

        me._hidden = false;
        element.classList.remove('b-hidden'); // The changer vetoes the config change and routes here, so we must call this.

        me.onConfigChange({
          name: 'hidden',
          value: false,
          was: true,
          config: me.$meta.configs.hidden
        });

        if (showAnimation) {
          styleProp = Object.keys(showAnimation)[0];
          animProps = showAnimation[styleProp];
          const currentAnimation = me.currentAnimation = {
            showing: true,
            styleProp,
            resolve
          };
          me.isAnimating = true; // afterHideShowAnimate will always be called even if the transition aborts

          me.currentAnimation.detacher = EventHelper.onTransitionEnd({
            element,
            property: styleProp,
            duration: parseDuration(animProps.duration) + 20,
            handler: () => me.afterHideShowAnimate(currentAnimation),
            thisObj: me
          }); // Setting transition style initial value before showing,
          // then reading the style to ensure transition will animate

          style[styleProp] = animProps.from;
          DomHelper.getStyleValue(element, styleProp);
          style.transition = `${styleProp} ${animProps.duration} ease ${animProps.delay}`;
          style[styleProp] = animProps.to;
        }

        me.afterShow(!showAnimation && resolve); // Note: popups can be dismissed in the process of focusin/out if an editor has invalidAction='block',
        // so we may be hidden at this point...
        // If we're not being called from showBy, do default aligning unless its centered

        if (!me.inShowBy && (floating || me.positioned) && me.forElement && !me.centered && !me.hidden) {
          me.alignTo(me.forElement);
        }
      });
    } else {
      return Promise.resolve();
    }
  }
  /**
   * Show aligned to another target element or {@link Core.widget.Widget} or {@link Core.helper.util.Rectangle}
   * @param {Object|HTMLElement} spec Alignment specification, or the element to align to using the configured
   * {@link #config-align}.
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} spec.target The Widget or Element or Rectangle
   * to align to.
   * @param {Boolean} [spec.anchor] True to show a pointer arrow connecting to the target. Defaults to false.
   * @param {Boolean} [spec.overlap] True to allow this to overlap the target.
   * @param {String} spec.align The alignment specification string, `[trblc]n-[trblc]n`. Defaults to this instance's
   * {@link #config-align} setting. Also supports direction independent edges horizontally, `s` for start and `e` for
   * end (maps to `l` and `r` for LTR, `r` and `l` for RTL).
   * @param {HTMLElement|Core.widget.Widget|Core.helper.util.Rectangle} [spec.constrainTo] The Widget or Element or
   * Rectangle to constrain to. If the requested alignment cannot be constrained (it will first shrink the resulting
   * Rectangle according to the `minWidth` and `minHeight` properties of this spec, or the Widget), then it will try
   * aligning at other edges (honouring the `axisLock` option), and pick the fallback alignment which results in the
   * shortest translation.
   * @param {Number|Number[]} [spec.constrainPadding] The amount of pixels to pad from the `constrainTo` target,
   * either a single value, or an array of values in CSS edge order.
   * @param {Number} [spec.minHeight] The minimum height this widget may be compressed to when constraining within the
   * `constrainTo` option.
   * @param {Number} [spec.minWidth] The minimum width this widget may be compressed to when constraining within the
   * `constrainTo` option.
   * @param {Boolean} [spec.axisLock] Specify as `true` to fall back to aligning against the opposite
   * edge if the requested alignment cannot be constrained into the `constrainTo` option. Specify as
   * `'flexible'` to allow continuation to try the other edges if a solution cannot be found on the originally
   * requested axis.
   * @param {Boolean} [spec.matchSize] When aligning edge-to-edge, match the length of the aligned-to
   * edge of the target. This is only honored when `axisLock` is enabled and alignment succeeds on the
   * requested axis.
   * If __not__ aligning edge-to-edge, `matchSize` matches both dimensions of the target.
   * Specify as `true` to have this widget's size along the aligned edge match the size of the target's edge.
   * For example, a combobox's dropdown should match the width of the combobox.
   * @param {Number|Number[]} [spec.offset] The offset to create an extra margin round the target
   * to offset the aligned widget further from the target. May be configured as -ve to move the aligned widget
   * towards the target - for example producing the effect of the anchor pointer piercing the target.
   * @param {Boolean} [spec.monitorResize] Configure as `true` to monitor the element being aligned to for
   * resizing while visible to correct alignment.
   * @category Float & align
   * @async
   * @returns {Promise} A promise which is resolved when the widget is shown
   */

  async showBy(spec) {
    const me = this;
    let result;

    if (me.isVisible) {
      DomHelper.addTemporaryClass(me.element, 'b-realigning', 300);
    } else {
      // Prevent show from rerouting here.
      me.inShowBy = true;
      result = me.show();
      me.inShowBy = false;
    }

    if (me.isVisible) {
      me.alignTo(spec);
    }

    return result || immediatePromise$6;
  }
  /**
   * Show this widget anchored to a coordinate
   * @param {Number|Number[]} x The x position (or an array with [x,y] values) to show by
   * @param {Number} [y] The y position to show by
   * @param {Object} [options] See {@link #function-showBy} for reference
   * @category Float & align
   * @async
   * @returns {Promise} A promise which is resolved when the widget is shown
   */

  async showByPoint(x, y, options) {
    const xy = arguments.length === 1 ? x : [x, y];
    return this.showBy(Object.assign({
      position: new Point(xy[0] + 1, xy[1] + 1),
      // Override any matchSize that we might have in our align config.
      // Otherwise we are going to be 1px wide/high
      matchSize: false,
      align: 't-b'
    }, options));
  }

  afterShow(resolveFn) {
    var _me$owner, _me$owner$onChildShow;

    const me = this;
    /**
     * Triggered after a widget is shown.
     * @event show
     * @param {Core.widget.Widget} source The widget
     */

    me.trigger('show'); // Cache our preferred anchoredness in case it's overridden by a drag.

    me._configuredAnchorState = me.anchor; // Keep any owning container informed about visibility state.
    // It may not be a Container. SubGrid class is still a Widget
    // which contains grid headers.

    (_me$owner = me.owner) === null || _me$owner === void 0 ? void 0 : (_me$owner$onChildShow = _me$owner.onChildShow) === null || _me$owner$onChildShow === void 0 ? void 0 : _me$owner$onChildShow.call(_me$owner, me);
    me.triggerPaint();
    resolveFn && resolveFn();
  }

  onChildHide(shown) {
    if (shown.floating) {
      this.ariaElement.removeAttribute('aria-owns');
    }
  }

  onChildShow(shown) {
    if (shown.floating) {
      this.ariaHasPopup = shown.role;
      this.ariaElement.setAttribute('aria-owns', shown.id);
    }
  }

  triggerPaint() {
    const me = this,
          {
      element
    } = me,
          firstPaint = !me.isPainted;

    if (me.isVisible) {
      if (firstPaint) {
        // Not for public use, only used in docs
        if (me.scaleToFitWidth && !me.monitorResize) {
          me.onParentElementResize = me.onParentElementResize.bind(me);
          ResizeMonitor.addResizeListener(element.parentElement, me.onParentElementResize);
          me.updateScale();
        } // Add a comment to state this is made by Bryntum

        if (!me.hideBryntumDomMessage && (me.isTaskBoardBase || me.isGridBase || me.isCalendar)) {
          element.insertBefore(new Comment('POWERED BY BRYNTUM (https://bryntum.com)'), element.firstChild);
        }
      } // Trigger paint only on immediate children.
      // Each one will call this recursively.
      // paint is triggered in a bottom up manner.

      me.eachWidget(widgetTriggerPaint, false);

      if (firstPaint && me.isVisible) {
        me.rootElement = DomHelper.getRootElement(element); // Make sure the shared tooltip is initialized

        me.getConfig('tooltip'); // Late setup of Ripple

        if (!Widget.Ripple && Widget.RippleClass) {
          Widget.Ripple = new Widget.RippleClass({
            rootElement: me.rootElement
          });
        }
      }
      /**
       * Triggered when a widget which had been in a non-visible state for any reason
       * achieves visibility.
       *
       * A non-visible state *might* mean the widget is hidden and has just been shown.
       *
       * But this event will also fire on widgets when a non-visible (unrendered, or hidden)
       * ancestor achieves visibility, for example a {@link Core.widget.Popup Popup} being shown.
       *
       * TLDR: __This event can fire multiple times__
       * @event paint
       * @param {Core.widget.Widget} source The widget being painted.
       * @param {Boolean} firstPaint `true` if this is the first paint.
       */

      me.isPainted = true;
      me.trigger('paint', {
        firstPaint
      });

      if (firstPaint) {
        // On first paint, we should announce our size immediately.
        // When the real event comes along, onElementResize will reject it because the size will be the same.
        if (me.monitorResize && !me.scaleToFitWidth) {
          ResizeMonitor.onElementResize([{
            target: element
          }]);
        }
      }
    }
  }

  cancelHideShowAnimation() {
    const me = this,
          {
      currentAnimation,
      element
    } = me;

    if (currentAnimation) {
      me.isAnimating = false; // If it is an animated hide that we are aborting, reverse the set of the hidden flag
      // If hide is animated, we only get genuinely hidden at animation end.

      if (element.classList.contains('b-hiding')) {
        element.classList.remove('b-hiding');
        me._hidden = false;
      }

      currentAnimation.detacher();
      currentAnimation.resolve();
      element.style.transition = element.style[currentAnimation.styleProp] = '';
      me.currentAnimation = null;
    }
  }

  afterHideShowAnimate(currentAnimation) {
    // We receive the currentAnimation as understood by the party starting the animation... if that is not the
    // current value of "this.currentAnimation" we can ignore this call.
    const me = this; // If menu is destroyed too soon in Edge, this method will be invoked for destroyed element. Since all of our
    // properties are cleared on destroy, this check will prevent undesired reactions:

    if (currentAnimation === me.currentAnimation) {
      // Ensure cancelHideShowAnimation doesn't think we're aborting before the end.
      me.element.classList.remove('b-hiding');
      me.cancelHideShowAnimation(); // Element must be fully hidden after the animation effect finishes

      if (me._hidden) {
        me.afterHideAnimation();
      }
    }
  }
  /**
   * Temporarily changes the {@link #property-isVisible} to yield `false` regardless of this
   * Widget's true visibility state. This can be useful for suspending operations which rely on
   * the {@link #property-isVisible} property.
   *
   * This increments a counter which {@link #function-resumeVisibility} decrements.
   * @internal
   */

  suspendVisibility() {
    this._visibilitySuspended = (this._visibilitySuspended || 0) + 1;
    Object.defineProperty(this, 'isVisible', returnFalseProp);
  }
  /**
   * Resumes visibility. If the suspension counter is returned to zero by this, then the
   * {@link #event-paint} event is triggered, causing a cascade of `paint` events on all
   * descendants. This can be prevented by passing `false` as the only parameter.
   * @param {Boolean} [triggerPaint=true] Trigger the {@link #event-paint} event.
   * @internal
   */

  resumeVisibility(triggerPaint = true) {
    if (! --this._visibilitySuspended) {
      delete this.isVisible;

      if (triggerPaint) {
        this.triggerPaint();
      }
    }
  }
  /**
   * Hide widget
   * @param {Boolean} animate Pass `true` (default) to animate the hide action
   * @category Visibility
   * @returns {Promise} A promise which is resolved when the widget has been hidden
   */

  hide(animate = true) {
    return new Promise(resolve => {
      const me = this,
            {
        element,
        lastAlignSpec
      } = me,
            {
        style
      } = element,
            hideAnimation = animate && me.hideAnimation; // If we get hidden very quickly after a call to show,
      // we must kill the timers which add the realign listeners.

      me.clearTimeout(me.scrollListenerTimeout);
      me.clearTimeout(me.resizeListenerTimeout);
      /**
       * Triggered before a widget is hidden. Return `false` to prevent the action.
       * @preventable
       * @event beforeHide
       * @param {Core.widget.Widget} source The widget being hidden.
       */
      // replaced check for isVisible with _hidden, need to hide a component not yet in view in EventEditor

      if (!me._hidden && me.trigger('beforeHide') !== false) {
        me._hidden = true; // The flag must be cleared on a normal hide.
        // It's set if we hide due to the target being scrolled out of view.

        if (lastAlignSpec) {
          lastAlignSpec.targetOutOfView = null;

          if (lastAlignSpec.monitorIntersection) {
            me.intersectionObserver.takeRecords();
            me.intersectionObserver.unobserve(lastAlignSpec.target);
          }
        } // The changer vetoes the config change and routes here, so we must call this.

        me.onConfigChange({
          name: 'hidden',
          value: true,
          was: false,
          config: me.$meta.configs.hidden
        });

        if (!element) {
          resolve();
          return;
        }

        if (element.contains(DomHelper.getActiveElement(element))) {
          me.revertFocus(true);
        } // Focus exit causes close if autoClose: true, and if closeAction: 'hide'
        // that might destroy us, so exit now if that happens.

        if (me.isDestroyed) {
          resolve();
          return;
        } // Cancel any current hide/show animation

        me.cancelHideShowAnimation();

        if (hideAnimation) {
          const styleProp = Object.keys(hideAnimation)[0],
                animProps = hideAnimation[styleProp]; // Make sure we are not already at the final value of the hide animation (i.e. calling hide() directly after show())

          if (Number(getComputedStyle(me.element)[styleProp]) !== animProps.to) {
            const currentAnimation = me.currentAnimation = {
              hiding: true,
              styleProp,
              resolve
            }; // Element must behave as though it were not there during
            // the animated hide. This means pointer-events:none

            element.classList.add('b-hiding');
            me.isAnimating = true; // afterHideShowAnimate will always be called even if the transition aborts

            me.currentAnimation.detacher = EventHelper.onTransitionEnd({
              element,
              property: styleProp,
              duration: parseDuration(animProps.duration) + 20,
              handler: () => me.afterHideShowAnimate(currentAnimation),
              thisObj: me
            }); // Setting transition style initial value before showing,
            // then reading the style to ensure transition will animate

            style[styleProp] = animProps.from;
            DomHelper.getStyleValue(element, styleProp);
            style.transition = `${styleProp} ${animProps.duration} ease ${animProps.delay}`;
            style[styleProp] = animProps.to;
          } else {
            element.classList.add('b-hidden');
          }
        } else {
          element.classList.add('b-hidden');
        } // only supply resolve function if not using animation

        me.afterHide(!hideAnimation && resolve, hideAnimation);
      }
    });
  }

  doHideOrRealign({
    target,
    isTrusted
  }) {
    const me = this,
          {
      anchoredTo,
      lastAlignSpec,
      element
    } = me,
          lastTarget = lastAlignSpec === null || lastAlignSpec === void 0 ? void 0 : lastAlignSpec.target,
          position = lastAlignSpec === null || lastAlignSpec === void 0 ? void 0 : lastAlignSpec.position,
          activeEl = DomHelper.getActiveElement(me); // If it's a synthesized scroll event (such as from our ResizeMonitor polyfill), ignore it
    // If we're scrolling because a focused textual input field which we contain is being shifted into view,
    // we must not reposition - we'll just move with the document content.
    // event.target might be missing with LockerService enabled

    if (!isTrusted || !target || target.nodeType === Element.DOCUMENT_NODE && element.contains(activeEl) && textInputTypes[activeEl] && globalThis.innerHeight < document.body.offsetHeight) {
      return;
    } // Scroll is inside this widget, or owned by this widget: no action

    if (target.nodeType === Node.ELEMENT_NODE && (element.contains(target) || me.owns(target))) {
      return;
    } // An element that is not document and does not contain the element we are anchored to was scrolled,
    // probably by scroll syncing header / body of Grid or similar. Ignore

    if ((lastTarget === null || lastTarget === void 0 ? void 0 : lastTarget.nodeType) === Node.ELEMENT_NODE && !target.contains(lastTarget) && target.nodeType !== Element.DOCUMENT_NODE) {
      return;
    } // Store current position if we are to hide on scroll below,
    // used to determine if realigning did actually move us and thus should hide

    const xy = me.scrollAction === 'hide' && me.getXY(); // Perform the realignment

    me.realign(); // Might destroy on hide in realign, so check for isDestroyed.

    if (!me.isDestroyed && me.scrollAction === 'hide') {
      const [newX, newY] = me.getXY(),
            moved = newX !== xy[0] || newY !== xy[1]; // If the scroll caused our position to become invalid, and we either don't know what element
      // we're anchored to (or not anchored to one at all), or the element we're anchored to has been
      // affected by the scroll, we must hide.
      // target might be missing with LockerService enabled

      if ((moved || (lastTarget === null || lastTarget === void 0 ? void 0 : lastTarget.$$name) === 'Point' || position) && (!anchoredTo || target && DomHelper.isDescendant(target, anchoredTo))) {
        me.hide();
      }
    }
  }

  afterHide(resolveFn = null, hideAnimation = this.hideAnimation) {
    var _me$owner2, _me$owner2$onChildHid;

    const me = this; // If a drag caused us to lose our anchor, restore it upon hide.

    me._anchor = me._configuredAnchorState; // Remove listeners which are only added during the visible phase.
    // In its own method because it's called on hide and destroy.

    me.removeTransientListeners(); // Postprocessing to be done after the hideAnimation finishes.
    // If there's no animation, we call it immediately.
    // We set the element to be hidden here, after any animation completes.
    // We also remove floating Widgets from the DOM when they are hidden.

    if (!hideAnimation) {
      me.afterHideAnimation();
    }
    /**
     * Triggered after a widget was hidden
     * @event hide
     * @param {Core.widget.Widget} source The widget
     */

    me.trigger('hide'); // Keep any owning container informed about visibility state.
    // It may not be a Container. SubGrid class is still a Widget
    // which contains grid headers.

    (_me$owner2 = me.owner) === null || _me$owner2 === void 0 ? void 0 : (_me$owner2$onChildHid = _me$owner2.onChildHide) === null || _me$owner2$onChildHid === void 0 ? void 0 : _me$owner2$onChildHid.call(_me$owner2, me);
    resolveFn && resolveFn(); // cannot do resolveFn?.() since resolveFn can be false
  }

  removeTransientListeners() {
    const me = this;
    me.clearTimeout(me.resizeListenerTimeout);
    me.clearTimeout(me.scrollListenerTimeout);

    if (me.documentScrollListener) {
      me.documentScrollListener();
      me.documentScrollListener = false;
    }

    if (me.rootScrollListener) {
      me.rootScrollListener();
      me.rootScrollListener = false;
    }

    if (me.targetResizeListener) {
      ResizeMonitor.removeResizeListener(me.lastAlignSpec.target, me.onTargetResize);
      me.targetResizeListener = false;
    }

    if (me.constrainListeners) {
      ResizeMonitor.removeResizeListener(me.lastAlignSpec.constrainTo || globalThis, me.callRealign);
      me.constrainMutationMonitor && me.constrainMutationMonitor.disconnect();
      me.constrainListeners = false;
    }
  }

  afterHideAnimation() {
    const me = this,
          {
      element
    } = me;

    if (me.floating && me.floatRoot.contains(element)) {
      element.remove();
    } else {
      element.classList.add('b-hidden');
    } // Reset anchor to its default colour after hide

    if (me.defaultAnchorBackgroundColor) {
      // Reset to default in case it has been positioned by a coloured header
      me.anchorPathElement.setAttribute('fill', me.defaultAnchorBackgroundColor);
    }
  }

  changeHidden(value) {
    const me = this;
    let ret;

    if (me.isConfiguring) {
      ret = Boolean(value);
      me.element.classList[value ? 'add' : 'remove']('b-hidden');
    } else {
      // These methods are async but set _hidden when they get past the before event, so don't set ret and
      // the setter won't set _hidden automatically.
      me.trigger('beforeChangeHidden', {
        hidden: value
      });

      if (value) {
        me.hide();
      } else {
        me.show();
      }
    }

    return ret;
  }
  /**
   * Get id assigned by user (not generated id)
   * @returns {String}
   * @readonly
   * @private
   * @category Misc
   */

  get assignedId() {
    return this.hasGeneratedId ? null : this.id;
  }
  /**
   * Get the owning Widget of this Widget. If this Widget is directly contained, then the containing
   * Widget is returned. If this Widget is floating, the configured `owner` property is returned.
   * If there is a `forElement`, that element's encapsulating Widget is returned.
   * @property {Core.widget.Widget}
   * @readonly
   * @category Widget hierarchy
   */

  get owner() {
    return this.parent || this._owner || this.containingWidget;
  }

  get containingWidget() {
    let result = this.parent;

    if (!result) {
      var _this$forElement, _this$element;

      const owningEl = ((_this$forElement = this.forElement) === null || _this$forElement === void 0 ? void 0 : _this$forElement.nodeType) === Element.ELEMENT_NODE ? this.forElement : (_this$element = this.element) === null || _this$element === void 0 ? void 0 : _this$element.parentNode;
      result = (owningEl === null || owningEl === void 0 ? void 0 : owningEl.closest('.b-widget')) && Widget.fromElement(owningEl);
    }

    return result;
  }
  /**
   * Get this Widget's previous sibling in the parent {@link Core.widget.Container Container}, or, if not
   * in a Container, the previous sibling widget in the same _parentElement_.
   * @property {Core.widget.Widget}
   * @readonly
   * @category Widget hierarchy
   */

  get previousSibling() {
    return this.getSibling(-1);
  }
  /**
   * Get this Widget's next sibling in the parent {@link Core.widget.Container Container}, or, if not
   * in a Container, the next sibling widget in the same _parentElement_.
   * @property {Core.widget.Widget}
   * @readonly
   * @category Widget hierarchy
   */

  get nextSibling() {
    return this.getSibling(1);
  }

  getSibling(increment) {
    const me = this,
          {
      parent
    } = me,
          siblings = parent ? parent.childItems : Array.from(me.element.parentElement.querySelectorAll('.b-widget'));
    return parent ? siblings[siblings.indexOf(me) + increment] : Widget.fromElement(siblings[siblings.indexOf(me.element) + increment]);
  }
  /**
   * Looks up the {@link #property-owner} axis to find an ancestor which matches the passed selector.
   * The selector may be a widget type identifier, such as `'grid'`, or a function which will return
   * `true` when passed the desired ancestor.
   * @param {String|Function} [selector] A Type identifier or selection function. If not provided, this method returns
   * the {@link #property-owner} of this widget
   * @param {Boolean} [deep] When using a string identifier, pass `true` if all superclasses should be included, i.e.,
   * if a `Grid` should match `'widget'`.
   * @param {Number|String|Core.widget.Widget} [limit] how many steps to step up before aborting the search, or a
   * selector to stop at or the topmost ancestor to consider.
   * @category Widget hierarchy
   */

  up(selector, deep, limit) {
    const {
      owner
    } = this;
    return selector ? owner === null || owner === void 0 ? void 0 : owner.closest(selector, deep, limit) : owner;
  }
  /**
   * Starts with this Widget, then Looks up the {@link #property-owner} axis to find an ancestor which matches the
   * passed selector. The selector may be a widget type identifier, such as `'grid'`, or a function which will return
   * `true` when passed the desired ancestor.
   * @param {String|Function} selector A Type identifier or selection function.
   * @param {Boolean} [deep] When using a string identifier, pass `true` if all superclasses should be included, i.e.,
   * if a `Grid` should match `'widget'`.
   * @param {Number|String|Core.widget.Widget} [limit] how many steps to step up before aborting the search, or a
   * selector to stop at or the topmost ancestor to consider.
   * @category Widget hierarchy
   */

  closest(selector, deep, limit) {
    const limitType = typeof limit,
          numericLimit = limitType === 'number',
          selectorLimit = limitType === 'string';

    for (let result = this, steps = 1; result; result = result.owner, steps++) {
      if (Widget.widgetMatches(result, selector, deep)) {
        return result;
      }

      if (numericLimit && steps >= limit) {
        return;
      } else if (selectorLimit && Widget.widgetMatches(result, limit, deep)) {
        return;
      } else if (result === limit) {
        return;
      }
    }
  }
  /**
   * Returns `true` if this Widget owns the passed Element, Event or Widget. This is based on the widget hierarchy,
   * not DOM containment. So an element in a `Combo`'s dropdown list will be owned by the `Combo`.
   * @param {HTMLElement|Event|Core.widget.Widget} target The element event or Widget to test for being
   * within the ownership tree of this Widget.
   * @category Widget hierarchy
   */

  owns(target) {
    if (target) {
      // Passed an event, grab its target
      if (target.eventPhase) {
        target = target.target;
      } // We were passed an HTMLElement

      if (target.nodeType === Element.ELEMENT_NODE) {
        if (this.element.contains(target)) {
          return true;
        }

        target = Widget.fromElement(target);
      }

      while (target) {
        if (target === this) {
          return true;
        }

        target = target.owner;
      }
    }

    return false;
  }
  /**
   * Iterate over all ancestors of this widget.
   *
   * *Note*: Due to this method aborting when the function returns `false`, beware of using short form arrow
   * functions. If the expression executed evaluates to `false`, iteration will terminate.
   * @param {Function} fn Function to execute for all ancestors. Terminate iteration by returning `false`.
   * @returns {Boolean} Returns `true` if iteration was not aborted by a step returning `false`
   * @category Widget hierarchy
   */

  eachAncestor(fn) {
    let ancestor = this.owner;

    while (ancestor) {
      if (fn(ancestor) === false) {
        return false;
      }

      ancestor = ancestor.owner;
    }

    return true;
  }

  changeMonitorResize(monitorResize, oldMonitorResize) {
    // They are mutually exclusive. scaleToFitWidth disables monitorResize
    const result = this.scaleToFitWidth ? false : Boolean(monitorResize); // null and undefined both mean false. Avoid going through the updater if no change.

    if (result !== Boolean(oldMonitorResize)) {
      return result;
    }
  }

  updateMonitorResize(monitorResize) {
    const me = this;

    if (!hasOwnProperty$3.call(me, 'onElementResize')) {
      me.onElementResize = me.onElementResize.bind(me);
    }

    ResizeMonitor[monitorResize ? 'addResizeListener' : 'removeResizeListener'](me.element, me.onElementResize);
  }

  changeReadOnly(readOnly) {
    readOnly = Boolean(readOnly); // It starts as undefined, so if false is passed, that's a no-change.

    if (Boolean(this._readOnly) !== readOnly) {
      return readOnly;
    }
  }

  updateReadOnly(readOnly) {
    var _this$element2;

    // Can be called from the element initialization because of the way Panel is set up.
    // tbar and bbar are instantiated, and their elements added to the gathered element
    // config object, but that can have consequences which can lead here.
    // eslint-disable-next-line no-unused-expressions
    (_this$element2 = this.element) === null || _this$element2 === void 0 ? void 0 : _this$element2.classList[readOnly ? 'add' : 'remove']('b-readonly'); // Do not update  children at configure time.
    // Container will sync its items.

    if (!this.isConfiguring) {
      // Implemented at this level because Widgets can own a descendant tree without being
      // a Container. For example Combos own a ChipView and a List. Buttons own a Menu etc.
      this.eachWidget(widget => {
        // Some fields may not want to automatically be readOnly (such as a nested filter field not affecting data)
        if (widget.ignoreParentReadOnly) {
          return;
        }

        if (!('_originalReadOnly' in widget)) {
          // Store initial readOnly/disabled value.
          // the config getter copies the properties in a loop
          // so execute once and cache the value.
          widget._originalReadOnly = widget.config.readOnly || false;
        } // Set if truthy, otherwise reset to initial value

        widget.readOnly = readOnly || widget._originalReadOnly;
      }, false);
      /**
       * Fired when a Widget's read only state is toggled
       * @event readOnly
       * @param {Boolean} readOnly Read only or not
       */

      this.trigger('readOnly', {
        readOnly
      });
    }
  }
  /**
   * Iterate over all widgets owned by this widget and any descendants.
   *
   * *Note*: Due to this method aborting when the function returns `false`, beware of using short form arrow
   * functions. If the expression executed evaluates to `false`, iteration will terminate.
   * @param {Function} fn A function to execute upon each descendant widget.
   * Iteration terminates if this function returns `false`.
   * @param {Core.widget.Widget} fn.widget The current descendant widget.
   * @param {Object} fn.control An object containing recursion control options.
   * @param {Boolean} fn.control.down A copy of the `deep` parameter. This can be adjusted by `fn` to decide which
   * widgets should be recursed. This value will always be the value of `deep` on entry and the value of `control.down`
   * upon return determines the recursion into the current widget.
   * @param {Boolean} [deep=true] Pass as `false` to only consider immediate child widgets.
   * @returns {Boolean} Returns `true` if iteration was not aborted by a step returning `false`
   * @category Widget hierarchy
   */

  eachWidget(fn, deep = true) {
    const widgets = this.childItems,
          length = (widgets === null || widgets === void 0 ? void 0 : widgets.length) || 0,
          control = {};

    for (let i = 0; i < length; i++) {
      const widget = widgets[i];
      control.down = deep; // Abort if a call returns false

      if (fn(widget, control) === false) {
        return false;
      }

      if (control.down && widget.eachWidget) {
        // Abort if a deep call returns false
        if (widget.eachWidget(fn, deep) === false) {
          return false;
        }
      }
    }

    return true;
  }
  /**
   * Returns an array of all descendant widgets which the passed
   * filter function returns `true` for.
   * @param {Function} filter A function which, when passed a widget,
   * returns `true` to include the widget in the results.
   * @returns {Core.widget.Widget[]} All matching descendant widgets.
   * @category Widget hierarchy
   */

  queryAll(filter) {
    const result = [];
    this.eachWidget(w => {
      if (filter(w)) {
        result.push(w);
      }
    });
    return result;
  }
  /**
   * Returns the first descendant widgets which the passed
   * filter function returns `true` for.
   * @param {Function} filter A function which, when passed a widget,
   * returns `true` to return the widget as the sole result.
   * @returns {Core.widget.Widget} The first matching descendant widget.
   * @category Widget hierarchy
   */

  query(filter) {
    let result = null;
    this.eachWidget(w => {
      if (filter(w)) {
        result = w;
        return false;
      }
    });
    return result;
  }
  /**
   * Get a widget by ref, starts on self and traverses up the owner hierarchy checking `widgetMap` at each level.
   * Not checking the top level widgetMap right away to have some acceptance for duplicate refs.
   * @param {String} ref ref to find
   * @returns {Core.widget.Widget}
   * @internal
   * @category Widget hierarchy
   */

  getWidgetByRef(ref) {
    var _this$widgetMap, _this$owner;

    if (ref instanceof Widget) {
      return ref;
    }

    return (this === null || this === void 0 ? void 0 : (_this$widgetMap = this.widgetMap) === null || _this$widgetMap === void 0 ? void 0 : _this$widgetMap[ref]) || (this === null || this === void 0 ? void 0 : (_this$owner = this.owner) === null || _this$owner === void 0 ? void 0 : _this$owner.getWidgetByRef(ref));
  }

  onFocusIn(e) {
    const me = this,
          {
      element
    } = me;
    me.containsFocus = true;
    me.focusInEvent = e;
    element.classList.add('b-contains-focus');
    me.updateAriaLabel(me.localizeProperty('ariaLabel'));
    me.updateAriaDescription(me.localizeProperty('ariaDescription'));

    if (element.contains(e._target) && me.onInternalKeyDown && !me.keyDownListenerRemover) {
      me.keyDownListenerRemover = EventHelper.on({
        element,
        keydown: 'onInternalKeyDown',
        thisObj: me
      });
    }
    /**
     * Fired when focus enters this Widget.
     * @event focusIn
     * @param {Core.widget.Widget} source - This Widget
     * @param {HTMLElement} fromElement The element which lost focus.
     * @param {HTMLElement} toElement The element which gained focus.
     * @param {Core.widget.Widget} fromWidget The widget which lost focus.
     * @param {Core.widget.Widget} toWidget The widget which gained focus.
     * @param {Boolean} backwards `true` if the `toElement` is before the `fromElement` in document order.
     */

    me.trigger('focusin', e);
  }

  onFocusOut(e) {
    const me = this;

    if (me.keyDownListenerRemover) {
      me.keyDownListenerRemover();
      me.keyDownListenerRemover = null;
    }

    if (!me.isDestroyed) {
      // Focus to nowhere, focus a close relation
      if (!e.relatedTarget) {
        me.revertFocus(!me.isVisible);
      }

      me.containsFocus = false;
      me.element.classList.remove('b-contains-focus');
      me.updateAriaLabel(me.localizeProperty('ariaLabel'));
      me.updateAriaDescription(me.localizeProperty('ariaDescription'));
      /**
       * Fired when focus exits this Widget's ownership tree. This is different from a `blur` event.
       * focus moving from within this Widget's ownership tree, even if there are floating widgets
       * will not trigger this event. This is when focus exits this widget completely.
       * @event focusOut
       * @param {Core.widget.Widget} source - This Widget
       * @param {HTMLElement} fromElement The element which lost focus.
       * @param {HTMLElement} toElement The element which gained focus.
       * @param {Core.widget.Widget} fromWidget The widget which lost focus.
       * @param {Core.widget.Widget} toWidget The widget which gained focus.
       * @param {Boolean} backwards `true` if the `toElement` is before the `fromElement` in document order.
       */

      me.trigger('focusout', e);
    }
  }
  /**
   * Returns a function that will set the focus (`document.activeElement`) to the most consistent element possible
   * based on the focus state at the time this method was called. Derived classes can implement `captureFocusItem()`
   * to refine this process to include logical items (e.g., a grid cell) that would be more stable than DOM element
   * references.
   *
   * If this widget does not contain the focus, the returned function will do nothing.
   * @returns {Function}
   * @internal
   */

  captureFocus() {
    const me = this,
          activeElementWas = DomHelper.getActiveElement(me),
          restore = me.contains(activeElementWas) && me.captureFocusItem(activeElementWas);
    return scrollIntoView => {
      if (restore && !me.isDestroying) {
        const activeElementNow = DomHelper.getActiveElement(me);

        if (activeElementNow !== activeElementWas) {
          restore(scrollIntoView);
        }
      }
    };
  }
  /**
   * This method is called by `captureFocus()` when this widget contains the focus and it returns a function that
   * restores the focus to the correct internal element. The returned function is only called if the current
   * `document.activeElement` is different from the passed `activeElement`.
   *
   * This method can be replaced by derived classes to capture stable identifiers for the currently focused, logical
   * item (for example, a cell of a grid).
   *
   * @param {HTMLElement} activeElement The current `document.activeElement`.
   * @returns {Function} Returns a function that accepts a boolean argument. Defaults to `true`, `false` attempts to
   * focus without scrolling.
   * @internal
   */

  captureFocusItem(activeElement) {
    return (scrollIntoView = true) => {
      if (this.contains(activeElement)) {
        scrollIntoView ? activeElement.focus() : DomHelper.focusWithoutScrolling(activeElement);
      }
    };
  }
  /**
   * Returns `true` if this widget is or contains the specified element or widget.
   * @param {HTMLElement|Core.widget.Widget} elementOrWidget The element or widget
   * @param {Boolean} [strict] Pass `true` to test for strict containment (if `elementOrWidget` is this widget, the
   * return value will be `false`).
   * @returns {Boolean}
   */

  contains(elementOrWidget, strict) {
    const {
      element
    } = this;

    if (elementOrWidget && element) {
      if (elementOrWidget.isWidget) {
        elementOrWidget = elementOrWidget.element;
      } // el.contains(el) === true

      return element.contains(elementOrWidget) && (!strict || element !== elementOrWidget);
    }
  }
  /**
   * If this Widget contains focus, focus is reverted to the source from which it entered if possible,
   * or to a close relative if not.
   * @param {Boolean} [force] Pass as `true` to move focus to the previously focused item, or the
   * closest possible relative even if this widget does not contain focus.
   */

  revertFocus(force) {
    var _me$focusInEvent, _target;

    const me = this,
          activeElement = DomHelper.getActiveElement(me);
    let target = (_me$focusInEvent = me.focusInEvent) === null || _me$focusInEvent === void 0 ? void 0 : _me$focusInEvent.relatedTarget;

    if (force || me.containsFocus && ((_target = target) === null || _target === void 0 ? void 0 : _target.nodeType) === Element.ELEMENT_NODE && me.element.contains(activeElement)) {
      if (!target || !DomHelper.isFocusable(target)) {
        target = me.getFocusRevertTarget();
      }

      if (target && DomHelper.isFocusable(target)) {
        DomHelper.focusWithoutScrolling(target);
      } else {
        // If we could not find a suitable target to receive focus, we still need to not be focused. Oddly,
        // one cannot do "document.body.focus()" but explicitly calling blur() has that effect. If we do not
        // do this, and we retain the focus, we can have issue w/closeAction=destroy which can cause the blur
        // in afterHideAnimation which then causes that element.remove() to throw DOM exceptions.
        activeElement === null || activeElement === void 0 ? void 0 : activeElement.blur();
      }
    }
  }
  /**
   * This method finds a close sibling (or parent, or parent's sibling etc recursively) to which focus
   * can be directed in the case of revertFocus not having a focusable element from our focusInEvent.
   *
   * This can happen when the "from" component is destroyed or hidden. We should endeavour to prevent
   * focus escaping to `document.body` for accessibility and ease of use, and keep focus close.
   * @internal
   */

  getFocusRevertTarget() {
    const me = this,
          {
      owner,
      focusInEvent
    } = me,
          searchDirection = focusInEvent ? focusInEvent.backwards ? 1 : -1 : -1;
    let target = focusInEvent && focusInEvent.relatedTarget;
    const toComponent = target && Widget.fromElement(target); // If the from element is now not focusable, for example an Editor which hid
    // itself on focus leave, then we have to find a sibling/parent/parent's sibling
    // to take focus. Anything is better than flipping to document.body.

    if (owner && !owner.isDestroyed && (!target || !DomHelper.isFocusable(target) || toComponent && !toComponent.isFocusable)) {
      target = null; // If this widget can have siblings, then find the closest
      // (in the direction focus arrived from) focusable sibling.

      if (owner.eachWidget) {
        const siblings = []; // Collect focusable siblings.
        // With this included so we can find ourselves.

        owner.eachWidget(w => {
          if (w === me || w.isFocusable) {
            siblings.push(w);
          }
        }, false);

        if (siblings.length > 1) {
          const myIndex = siblings.indexOf(me);
          target = siblings[myIndex + searchDirection] || siblings[myIndex - searchDirection];
        }
      } // No focusable siblings found to take focus, try the owner

      if (!target && owner.isFocusable) {
        target = owner;
      } // If non of the above found any related focusable widget,
      // Go through these steps for the owner.

      target = target ? target.focusElement : owner.getFocusRevertTarget();
    }

    return target;
  }
  /**
   * Returns a `DomClassList` computed from the `topMostBase` (e.g., `Widget` or `Panel`) with the given `suffix`
   * appended to each `widgetClass`.
   * @param {Function} topMostBase The top-most base class constructor at which to start gathering classes.
   * @param {String} [suffix] An optional suffix to apply to all widget classes.
   * @returns {Core.helper.util.DomClassList}
   * @internal
   * @category DOM
   */

  getStaticWidgetClasses(topMostBase, suffix) {
    const classList = new DomClassList(),
          hierarchy = this.$meta.hierarchy;
    let cls, i, name, widgetClass, widgetClassProperty;

    for (i = hierarchy.indexOf(topMostBase); i < hierarchy.length; ++i) {
      var _widgetClassProperty;

      cls = hierarchy[i];
      widgetClassProperty = Reflect.getOwnPropertyDescriptor(cls.prototype, 'widgetClass'); // If the Class has its own get widgetClass, call it upon this instance.

      if ((_widgetClassProperty = widgetClassProperty) !== null && _widgetClassProperty !== void 0 && _widgetClassProperty.get) {
        widgetClass = widgetClassProperty.get.call(this);
      } else {
        // All built in widgets should define $name to be safer from minification/obfuscation, but user
        // created might not so fall back to actual name. UMD files use a _$name property
        // which the Base $$name getter uses as a fallback.
        name = hasOwnProperty$3.call(cls, '$$name') || hasOwnProperty$3.call(cls, '$name') || hasOwnProperty$3.call(cls, '_$name') ? cls.$$name : cls.name; // Throw error in case of an obfuscated name or an autogenerated name.
        // These should never be released without a meaningful $name getter.

        if (name.length < 3 || name.includes('$')) {
          // class.$name comes from parent API class which has it
          console.warn(`Class "${name}" extending "${cls.$name}" should have "$name" static getter with no less than 3 chars.`);
        }

        widgetClass = `b-${name.toLowerCase()}`;
      }

      if (widgetClass) {
        classList.add(suffix ? widgetClass + suffix : widgetClass);
      }
    }

    return classList;
  }

  get rootUiClass() {
    return Widget;
  }
  /**
   * Returns the `DomClassList` for this widget's class. This object should not be mutated.
   * @returns {Core.helper.util.DomClassList}
   * @internal
   * @category DOM
   */

  get staticClassList() {
    const {
      $meta: meta
    } = this;
    let classList = meta.staticClassList;

    if (!classList) {
      // Compute the class part of the widgetList just once per class (cache it on the $meta object):
      meta.staticClassList = classList = this.getStaticWidgetClasses(Widget); // TODO this should probably be only for outer-most widgets

      BrowserHelper.isTouchDevice && classList.add('b-touch');
    }

    return classList;
  }
  /**
   * Returns the cross-product of the classes `staticClassList` with each `ui` as an array of strings.
   *
   * For example, a Combo with a `ui: 'foo bar'` would produce:
   *
   *      [
   *          'b-widget-foo', 'b-field-foo', 'b-textfield-foo', 'b-pickerfield-foo', 'b-combo-foo',
   *          'b-widget-bar', 'b-field-bar', 'b-textfield-bar', 'b-pickerfield-bar', 'b-combo-bar'
   *      ]
   *
   * @returns {String[]}
   * @internal
   * @category DOM
   */

  get uiClasses() {
    // our result is maintained by updateUi so ensure the ui config has been evaluated:
    this.getConfig('ui');
    return this._uiClasses;
  }
  /**
   * Returns the cross-product of the classes `staticClassList` with each `ui` as a `DomClassList` instance.
   *
   * For example, a Combo with a `ui: 'foo bar'` would produce:
   *
   * ```javascript
   *      new DomClassList({
   *          'b-field-ui-foo'       : 1,
   *          'b-textfield-ui-foo'   : 1,
   *          'b-pickerfield-ui-foo' : 1,
   *          'b-combo-ui-foo'       : 1,
   *
   *          'b-field-ui-bar'       : 1,
   *          'b-textfield-ui-bar'   : 1,
   *          'b-pickerfield-ui-bar' : 1,
   *          'b-combo-ui-bar'       : 1
   *      });
   * ```
   *
   * A Panel with a `ui: 'foo bar'` would produce:
   *
   * ```javascript
   *      new DomClassList({
   *          'b-panel-ui-foo' : 1,
   *          'b-panel-ui-bar' : 1
   *      });
   * ```
   * @returns {Core.helper.util.DomClassList}
   * @internal
   * @category DOM
   */

  get uiClassList() {
    // our result is maintained by updateUi so ensure the ui config has been evaluated:
    this.getConfig('ui');
    return this._uiClassList;
  }
  /**
   * Used by the Widget class internally to create CSS classes based on this Widget's
   * inheritance chain to allow styling from each level to apply.
   *
   * For example Combo would yield `"["b-widget", "b-field", "b-textfield", "b-pickerfield", "b-combo"]"`
   *
   * May be implemented in subclasses to add or remove classes from the super.widgetClassList
   * @returns {String[]} The css class list named using the class name.
   * @internal
   * @category DOM
   */

  get widgetClassList() {
    const me = this,
          {
      cls,
      defaultCls,
      uiClasses
    } = me;
    let {
      staticClassList
    } = me,
        classList;

    if (defaultCls || cls) {
      // clone the class-level classList before instance stuff goes on...
      staticClassList = staticClassList.clone();
      defaultCls && staticClassList.assign(defaultCls); // note: these can be falsy keys

      cls && staticClassList.assign(cls);
    }

    classList = staticClassList.values; // a new array of truthy keys...

    uiClasses && classList.push(...uiClasses);
    me.floating && classList.push('b-floating');

    if (me.collapsify === 'hide') {
      classList.push('b-collapsify-hide');
    }

    return classList;
  }

  changeCls(cls) {
    return DomClassList.from(cls);
  }

  changeContentElementCls(cls) {
    return DomClassList.from(cls);
  }

  changeHtmlCls(cls) {
    return DomClassList.from(cls);
  }

  changeDefaultCls(cls) {
    return DomClassList.from(cls,
    /* returnEmpty */
    true);
  }

  changeUi(ui) {
    return DomClassList.from(ui);
  }

  updateUi(ui) {
    var _uiClassList;

    let uiClassList = null,
        cls,
        suffix;

    if (ui) {
      const staticClassList = this.getStaticWidgetClasses(this.rootUiClass);

      for (suffix in ui) {
        if (ui[suffix]) {
          for (cls in staticClassList) {
            if (staticClassList[cls]) {
              (uiClassList || (uiClassList = new DomClassList()))[`${cls}-ui-${suffix}`] = 1;
            }
          }
        }
      }
    }

    this._uiClasses = (_uiClassList = uiClassList) === null || _uiClassList === void 0 ? void 0 : _uiClassList.values; // an array of each value

    this._uiClassList = uiClassList;
  } //endregion
  //region Cache

  /**
   * Gets dom elements in the view. Caches the results for faster future calls.
   * @param {String} query CSS selector
   * @param {Boolean} children true to fetch multiple elements
   * @param {HTMLElement} element Element to use as root for the query, defaults to the views outermost element
   * @returns {HTMLElement|HTMLElement[]|null} A single element or an array of elements (if parameter children is set to true)
   * @internal
   * @category DOM
   */

  fromCache(query, children = false, element = this.element) {
    if (!element) return null;
    const me = this;

    if (!me.cache[query]) {
      me.cache[query] = children ? DomHelper.children(element, query) : DomHelper.down(element, query);
    }

    return me.cache[query];
  }
  /**
   * Clear caches, forces all calls to fromCache to requery dom. Called on render/rerender.
   * @internal
   * @category DOM
   */

  emptyCache() {
    this.cache = {};
  } //endregion
  //region Mask

  changeMasked(mask, maskInstance) {
    if (mask === true || mask === '') {
      // empty string don't render well, so promote it to an &nbsp; as well
      mask = '\xA0'; // nbsp
    }

    if (maskInstance) {
      if (typeof mask === 'string') {
        maskInstance.text = mask;
        mask = maskInstance;
      } else if (mask) {
        maskInstance.setConfig(mask);
        mask = maskInstance;
      } else {
        maskInstance.destroy();
      }
    } else if (mask) {
      const Mask = Widget.resolveType('mask');
      mask = Mask.mergeConfigs(this.maskDefaults, mask);
      mask.owner = this;
      mask = Mask.mask(mask);
    }

    return mask || null;
  }

  onMaskAutoClose(mask) {
    if (mask.isDestroyed && mask === this.masked) {
      this.masked = null;
    }
  }
  /**
   * Mask the widget, showing the specified message
   * @param {String|Object} msg Mask message (or a {@link Core.widget.Mask} config object
   * @returns {Core.widget.Mask}
   */

  mask(msg) {
    this.masked = msg;
    return this.masked;
  }
  /**
   * Unmask the widget
   */

  unmask() {
    this.masked = null;
  } //endregion
  //region Monitor resize

  onInternalResize(element, width, height, oldWidth, oldHeight) {
    this._width = element.offsetWidth;
    this._height = element.offsetHeight;
  }

  onElementResize(resizedElement) {
    const me = this,
          {
      element
    } = me,
          oldWidth = me._width,
          oldHeight = me._height,
          newWidth = element.offsetWidth,
          newHeight = element.offsetHeight;

    if (me.floating) {
      me.onFloatingWidgetResize(...arguments);
    }

    if (!me.suspendResizeMonitor && (oldWidth !== newWidth || oldHeight !== newHeight)) {
      me.onInternalResize(element, newWidth, newHeight, oldWidth, oldHeight);
      /**
       * Fired when the encapsulating element of a Widget resizes *only when {@link #config-monitorResize} is `true`*.
       * @event resize
       * @param {Core.widget.Widget} source - This Widget
       * @param {Number} width The new width
       * @param {Number} height The new height
       * @param {Number} oldWidth The old width
       * @param {Number} oldHeight The old height
       */

      me.trigger('resize', {
        width: newWidth,
        height: newHeight,
        oldWidth,
        oldHeight
      });
    }
  }

  onFloatingWidgetResize(resizedElement, lastRect, myRect) {
    const me = this,
          {
      lastAlignSpec,
      constrainTo
    } = me; // If this Popup changes size while we are aligned and we are aligned to
    // a target (not a position), then we might need to realign.

    if (me.isVisible && lastAlignSpec && lastAlignSpec.target) {
      const heightChange = !lastRect || myRect.height !== lastRect.height,
            widthChange = !lastRect || myRect.width !== lastRect.width,
            failsConstraint = constrainTo && !Rectangle.from(constrainTo).contains(Rectangle.from(me.element, null, true)); // Only realign if:
      // the height has changed and we are not aligned below, or
      // the width has changed and we are not aligned to the right.

      if (heightChange && lastAlignSpec.zone !== 2 || widthChange && lastAlignSpec.zone !== 1 || failsConstraint) {
        // Must move to next AF because in Chrome, the resize monitor might fire
        // before the element is painted and the anchor color matching
        // scheme cannot work in that case.
        me.requestAnimationFrame(() => me.realign());
      }
    }
  }

  updateScale() {
    const me = this,
          element = me.element,
          parentElement = element.parentElement; // this could be placed elsewhere but want to keep it contained to not spam other code,
    // since this is a very specific use case in our docs

    if (!me.configuredWidth) {
      me.configuredWidth = me.width;
    } // TODO: handle autoHeight, but seems it assigns height to late with current setup
    // We are scaling to fit inside the width, so ensure that we are not the cause of a scrollbar
    // in our current, unscaled state by hiding while we measure the parent's offsetWidth which
    // we are going to scale to.

    element.style.display = 'none';
    const rect = Rectangle.client(parentElement),
          scale = rect.width / me.configuredWidth,
          adjustedScale = me.scale = me.allowGrowWidth ? Math.min(scale, 1) : scale;
    element.style.transform = `scale(${adjustedScale})`;
    element.style.transformOrigin = 'top left';
    element.style.display = '';

    if (me.allowGrowWidth && scale > 1) {
      // increase width
      me.width = me.configuredWidth * scale;
    }
  }

  onParentElementResize(event) {
    this.updateScale();
  } //endregion

  /**
   * Returns a `TRBL` array of values parse from the passed specification. This can be used to parse`
   * a value list for `margin` or `padding` or `border-width` etc - any CSS value which takes a `TRBL` value.
   * @param {Number|String|String[]} values The `TRBL` value
   * @param {String} [units=px] The units to add to values which are specified as numeric.
   * @internal
   */

  parseTRBL(values, units = 'px') {
    values = values || 0;

    if (typeof values === 'number') {
      return [`${values}${units}`, `${values}${units}`, `${values}${units}`, `${values}${units}`];
    }

    const parts = values.split(' '),
          len = parts.length;

    if (len === 1) {
      parts[1] = parts[2] = parts[3] = parts[0];
    } else if (len === 2) {
      parts[2] = parts[0];
      parts[3] = parts[1];
    } else if (len === 3) {
      parts[3] = parts[1];
    }

    return [isFinite(parts[0]) ? `${parts[0]}${units}` : parts[0], isFinite(parts[1]) ? `${parts[1]}${units}` : parts[2], isFinite(parts[2]) ? `${parts[2]}${units}` : parts[3], isFinite(parts[3]) ? `${parts[3]}${units}` : parts[4]];
  } // Returns root node for this widget, either a document or a shadowRoot

  get documentRoot() {
    var _this$owner2;

    return ((_this$owner2 = this.owner) === null || _this$owner2 === void 0 ? void 0 : _this$owner2.documentRoot) || this.element.getRootNode();
  } // Returns top most DOM element of the visible DOM tree for this widget element, either document.body or a shadowRoot

  get rootElement() {
    const me = this;

    if (!me._rootElement) {
      var _me$owner3;

      let root = ((_me$owner3 = me.owner) === null || _me$owner3 === void 0 ? void 0 : _me$owner3.rootElement) || me.forElement instanceof HTMLElement && DomHelper.getRootElement(me.forElement); // Check element, we may be in the DOM just not visible

      if (!root) {
        const elRoot = me.element instanceof HTMLElement && DomHelper.getRootElement(me.element); // Make sure we got a proper root

        if (elRoot === document.body || elRoot.nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
          root = elRoot;
        }
      }

      if (!root) {
        root = document.body;
      }

      me._rootElement = root;
    }

    return me._rootElement;
  }

  get floatRoot() {
    var _me$owner4;

    const me = this,
          rootElement = me.rootElement || ((_me$owner4 = me.owner) === null || _me$owner4 === void 0 ? void 0 : _me$owner4.rootElement);
    let {
      floatRoot
    } = rootElement;

    if (!floatRoot) {
      var _DomHelper$themeInfo;

      const {
        outerCls
      } = Widget,
            themeName = (_DomHelper$themeInfo = DomHelper.themeInfo) === null || _DomHelper$themeInfo === void 0 ? void 0 : _DomHelper$themeInfo.name;

      if (!DomHelper.isValidFloatRootParent(rootElement)) {
        throw new Error('Attaching float root to wrong root');
      } // When outside of our examples, the body element doesn't get the theme class.
      // The floatRoot must carry it for floating items to be themed.

      if (themeName) {
        outerCls.push(`b-theme-${themeName.toLowerCase()}`);
      }

      floatRoot = rootElement.floatRoot = DomHelper.createElement({
        className: `b-float-root ${outerCls.join(' ')}`,
        parent: rootElement
      });
      floatRoots.push(floatRoot); // Make float root immune to keyboard-caused size changes

      if (BrowserHelper.isAndroid) {
        floatRoot.style.height = `${screen.height}px`;
        EventHelper.on({
          element: globalThis,
          orientationchange: () => floatRoot.style.height = `${screen.height}px`
        });
      } // Resize float root upon keyboard-caused visual viewport size change
      // https://developer.mozilla.org/en-US/docs/Web/API/Visual_Viewport_API

      if (!BrowserHelper.isHoverableDevice && globalThis.visualViewport) {
        EventHelper.on({
          element: globalThis.visualViewport,
          resize: ({
            target: viewport
          }) => floatRoot.style.height = `${viewport.height}px`
        });
      } // Keep floatRoot up to date with the theme

      GlobalEvents.on({
        theme: ({
          theme,
          prev
        }) => {
          // TODO handle for all roots
          floatRoot.classList.add(`b-theme-${theme.toLowerCase()}`);
          floatRoot.classList.remove(`b-theme-${prev.toLowerCase()}`);
        }
      });
    } // Angular might shuffle elements around so we have to ensure floatRoot is a child of the right parent
    else if (!rootElement.contains(floatRoot)) {
      // Reattach floatRoot if it was detached
      rootElement.appendChild(floatRoot);
    }

    return floatRoot;
  }

  get floatRootMaxZIndex() {
    let max = 1;
    Array.from(this.floatRoot.children).forEach(child => {
      const zIndex = parseInt(getComputedStyle(child).zIndex || 0, 10);

      if (zIndex > max) {
        max = zIndex;
      }
    });
    return max;
  }

  static resetFloatRootScroll() {
    floatRoots.forEach(floatRoot => floatRoot.scrollTop = floatRoot.scrollLeft = 0);
  }

  static get floatRoots() {
    return floatRoots;
  }

  static removeFloatRoot(floatRoot) {
    floatRoots.splice(floatRoots.indexOf(floatRoot), 1);
  } // CSS classes describing outer-most Widgets to provide styling / behavioral CSS style rules

  static get outerCls() {
    const result = ['b-outer'];

    if (BrowserHelper.isTouchDevice) {
      result.push('b-touch-events');
    }

    if (DomHelper.scrollBarWidth) {
      result.push('b-visible-scrollbar');
    } else {
      result.push('b-overlay-scrollbar');
    }

    if (BrowserHelper.isChrome) {
      result.push('b-chrome');
    } else if (BrowserHelper.isSafari) {
      result.push('b-safari');
    } else if (BrowserHelper.isFirefox) {
      result.push('b-firefox');
    } // So that we don't get the polyfill styles applied if we have ResizeMonitor available.
    // The polyfill styles can break certain elements styling.

    if (!globalThis.ResizeObserver) {
      result.push('b-no-resizeobserver');
    }

    return result;
  }

  get isAnimating() {
    return this._isAnimatingCounter > 0;
  }

  set isAnimating(value) {
    const me = this;

    if (me._isAnimatingCounter === 0 && value) {
      me.element.classList.add('b-animating');
      me.trigger('animationStart');
    } else if (me._isAnimatingCounter === 1 && !value) {
      me.element.classList.remove('b-animating');
      me.trigger('animationEnd');
    }

    me._isAnimatingCounter += value ? 1 : -1;
    me._isAnimatingCounter = Math.max(0, me._isAnimatingCounter);
  } // Waits until all transitions are completed

  async waitForAnimations() {
    if (this.isAnimating) {
      await this.await('animationend', {
        checkLog: false
      });
    }
  }
  /**
   * Analogous to document.querySelector, finds the first Bryntum widget matching the passed
   * selector. Right now, only class name (lowercased) selector strings, or
   * a filter function which returns `true` for required object are allowed:
   * ```
   * bryntum.query('grid').destroy();
   * ```
   * @param {String|Function} selector A lowercased class name, or a filter function.
   * @param {Boolean} [deep] Specify `true` to search the prototype chain (requires supplying a string `selector`). For
   *   example 'widget' would then find a Grid
   * @return {Core.widget.Widget} The first matched widget if any.
   * @category Widget hierarchy
   */

  static query(selector, deep = false) {
    const {
      idMap
    } = Widget.identifiable;

    for (const id in idMap) {
      if (Widget.widgetMatches(idMap[id], selector, deep)) {
        return idMap[id];
      }
    }

    return null;
  }
  /**
   * Analogous to document.querySelectorAll, finds all Bryntum widgets matching the passed
   * selector. Right now, only registered widget `type` strings, or a filter function which
   * returns `true` for required object are allowed:
   * ```
   * let allFields = bryntum.queryAll('field', true);
   * ```
   * @param {String|Function} selector A lowercased class name, or a filter function.
   * @param {Boolean} [deep] Specify `true` to search the prototype chain (requires supplying a string `selector`). For
   *   example 'widget' would then find a Grid
   * @return {Core.widget.Widget[]} The first matched widgets if any - an empty array will be returned
   * if no matches are found.
   * @category Widget hierarchy
   */

  static queryAll(selector, deep = false) {
    const {
      idMap
    } = Widget.identifiable,
          result = [];

    for (const id in idMap) {
      if (Widget.widgetMatches(idMap[id], selector, deep)) {
        result.push(idMap[id]);
      }
    }

    return result;
  }
  /**
   * Returns the Widget which owns the passed element (or event).
   * @param {HTMLElement|Event} element The element or event to start from
   * @param {String|Function} [type] The type of Widget to scan upwards for. The lowercase
   * class name. Or a filter function which returns `true` for the required Widget.
   * @param {HTMLElement|Number} [limit] The number of components to traverse upwards to find a
   * match of the type parameter, or the element to stop at.
   * @return {Core.widget.Widget} The found Widget or null.
   * @category Misc
   */

  static fromElement(element, type, limit) {
    const typeOfType = typeof type; // Check if an event was passed

    if (element && !element.nodeType) {
      element = element.target;
    }

    if (typeOfType === 'number' || type && type.nodeType === Element.ELEMENT_NODE) {
      limit = type;
      type = null;
    }

    let target = element,
        depth = 0,
        topmost,
        cmpId,
        cmp;

    if (typeof limit !== 'number') {
      topmost = limit;
      limit = Number.MAX_VALUE;
    }

    if (typeOfType === 'string') {
      type = type.toLowerCase();
    }

    while (target && target.nodeType === Element.ELEMENT_NODE && depth < limit && target !== topmost) {
      cmpId = target.dataset && target.dataset.ownerCmp || target.id;

      if (cmpId) {
        cmp = Widget.getById(cmpId);

        if (cmp) {
          if (type) {
            if (typeOfType === 'function') {
              if (type(cmp)) {
                return cmp;
              }
            } else if (Widget.widgetMatches(cmp, type, true)) {
              return cmp;
            }
          } else {
            return cmp;
          }
        } // Increment depth on every *Widget* found

        depth++;
      }

      target = target.parentNode;
    }

    return null;
  } // NOTE: Not named `triggerChange` to not conflict with existing fn on Field

  /**
   * Triggers a 'change' event with the supplied params. After triggering it also calls `onFieldChange()` on each
   * ancestor the implements that function, supplying the same set of params.
   * @param {Object} params Event params, used both for triggering and notifying ancestors
   * @param {Boolean} [trigger] `false` to not trigger, only notifying ancestors
   * @internal
   */

  triggerFieldChange(params, trigger = true) {
    if (trigger) {
      this.trigger('change', params);
    }

    this.eachAncestor(ancestor => {
      if (ancestor.onFieldChange) {
        ancestor.onFieldChange(params);
      } // Stop going up when reaching an ancestor that isolates its fields

      if (ancestor.isolateFieldChange(this)) {
        return false;
      }
    });
  }
  /**
   * Returns `true` if the given `field`'s value change should be isolated (kept hidden by this widget). By default,
   * this method returns the value of {@link Core.widget.Container#config-isolateFields} for all fields.
   * @param {Core.widget.Field} field The field in question.
   * @internal
   */

  isolateFieldChange(field) {
    return this.isolateFields;
  } // Sets up the focus listeners, one set for every document root (shadow root or document)

  setupFocusListeners() {
    // Listen to focus events on shadow root to handle focus inside the shadow dom
    GlobalEvents.setupFocusListenersOnce(this._rootElement, EventHelper);
  }

  static widgetMatches(candidate, selector, deep) {
    if (selector === '*') {
      return true;
    }

    if (typeof selector === 'function') {
      return selector(candidate);
    }

    return Widget.isType(candidate, selector, deep);
  }
  /**
   * Attached a tooltip to the specified element.
   * @example
   * Widget.attachTooltip(element, {
   *   text: 'Useful information goes here'
   * });
   * @param {HTMLElement} element Element to attach tooltip for
   * @param {Object|String} configOrText Tooltip config or tooltip string, see example and source
   * @returns {HTMLElement}
   * @category Misc
   */

  static attachTooltip(element, configOrText) {
    if (typeof configOrText === 'string') configOrText = {
      html: configOrText
    }; // TODO: refactor this
    // eslint-disable-next-line no-new

    Widget.create(Object.assign({
      forElement: element
    }, configOrText), 'tooltip');
    return element;
  } //region RTL
  // Since we use flexbox docking flips correctly out of the box. start and end values can be mapped straight to
  // left and right, for both LTR and RTL

  changeDock(dock) {
    if (dock === 'start') {
      return 'left';
    }

    if (dock === 'end') {
      return 'right';
    }

    return dock;
  }

  updateRtl(rtl) {
    super.updateRtl(rtl); // Cascade the rtl setting to owned widgets

    this.eachWidget(item => item.rtl = rtl);
  } //endregion

}
const proto$1 = Widget.prototype;
['compose', 'domSyncCallback'].forEach(fn => proto$1[fn].$nullFn = true); // Register this widget type with its Factory

Widget.initClass();
Widget.register('mask', Mask); // These low level classes must not import Widget because that would cause circularity.
// Instead Widget injects a reference to itself into them.

DomHelper.Widget = Widget;
GlobalEvents.Widget = Widget; // We use the same map to track instances by ID

Mask.identifiable.idMap = Widget.identifiable.idMap; // Simplify querying widgets by exposing fns in bryntum ns

Object.assign(globalThis.bryntum || (globalThis.bryntum = {}), {
  get: Widget.getById.bind(Widget),
  query: Widget.query,
  queryAll: Widget.queryAll,
  fromElement: Widget.fromElement
});
Widget._$name = 'Widget';

/**
 * @module Core/widget/layout/Layout
 */

/**
  * A helper class used by {@link Core.widget.Container Container}s which renders child widgets to their
  * {@link Core.widget.Widget#property-contentElement}. It also adds the Container's
  * {@link Core.widget.Container#config-itemCls} class to child items.
  *
  * Subclasses may modify the way child widgets are rendered, or may offer APIs for manipulating the child widgets.
  *
  * The {@link Core.widget.layout.Card Card} layout class offers slide-in, slide-out animation of multiple
  * child widgets. {@link Core.widget.TabPanel} uses Card layout.
  */

class Layout extends Base$1.mixin(Events, Factoryable) {
  static get type() {
    return 'default';
  }

  static get configurable() {
    return {
      /**
       * The owning Widget.
       * @member {Core.widget.Widget} owner
       * @readonly
       */

      /**
       * @config {Core.widget.Widget} owner
       * @private
       */
      owner: null,

      /**
       * The CSS class which should be added to the owning {@link Core.widget.Container}'s.
       * {@link Core.widget.Widget#property-contentElement}.
       * @config {String}
       */
      containerCls: 'b-auto-container',

      /**
       * The CSS class which should be added to the encapsulating element of child items.
       * @config {String}
       */
      itemCls: null
    };
  }

  static get factoryable() {
    // establish this class as the Factoryable base
    return {
      defaultType: 'default'
    };
  }

  get contentElement() {
    var _this$owner;

    return (_this$owner = this.owner) === null || _this$owner === void 0 ? void 0 : _this$owner.contentElement;
  }

  onChildAdd(item) {}

  onChildRemove(item) {}

  renderChildren() {
    const me = this,
          {
      owner,
      containerCls,
      itemCls
    } = me,
          {
      contentElement,
      items
    } = owner,
          ownerItemCls = owner.itemCls,
          itemCount = items === null || items === void 0 ? void 0 : items.length;
    contentElement.classList.add('b-content-element');

    if (containerCls) {
      contentElement.classList.add(containerCls);
    } // Need to check that container has widgets, for example TabPanel can have no tabs

    if (itemCount) {
      owner.textContent = false;

      for (let i = 0; i < itemCount; i++) {
        const item = items[i],
              {
          element
        } = item;
        element.dataset.itemIndex = i;

        if (itemCls) {
          element.classList.add(itemCls);
        }

        if (ownerItemCls) {
          element.classList.add(ownerItemCls);
        } // If instantiated by the app developer, external to Container#createWidget
        // a widget will have the b-outer class. Remove that if it' contained.

        element.classList.remove('b-outer'); // Only trigger paint if the owner is itself painted, otherwise
        // the outermost Container will cascade the paint signal down.

        item.render(contentElement, Boolean(owner.isPainted));
      }
    }

    me.syncPendingConfigs();
    me.syncChildCount();
  }

  removeChild(child) {
    const me = this,
          {
      element
    } = child,
          {
      owner,
      itemCls
    } = me,
          {
      contentElement
    } = owner,
          ownerItemCls = owner.itemCls; // Chrome has turned very fussy recently.
    // If the parent does not contain the child to be removed, it throws.

    if (contentElement.contains(element)) {
      element.remove();
    }

    delete element.dataset.itemIndex;

    if (itemCls) {
      element.classList.remove(itemCls);
    }

    if (ownerItemCls) {
      element.classList.remove(ownerItemCls);
    }

    me.fixChildIndices();
    me.syncChildCount();
  }

  appendChild(child) {
    const {
      element
    } = child,
          {
      owner,
      itemCls
    } = this,
          {
      contentElement
    } = owner,
          ownerItemCls = owner.itemCls;
    element.dataset.itemIndex = owner.indexOfChild(child);
    owner.textContent = false;

    if (itemCls) {
      element.classList.add(itemCls);
    }

    if (ownerItemCls) {
      element.classList.add(ownerItemCls);
    }

    child.render(contentElement, Boolean(owner.isPainted));
    this.syncChildCount();
  }

  insertChild(toAdd, childIndex) {
    const me = this,
          {
      element
    } = toAdd,
          {
      owner,
      itemCls
    } = me,
          {
      contentElement
    } = owner,
          nextSibling = DomHelper.getChild(contentElement, `[data-item-index="${childIndex}"]`),
          ownerItemCls = owner.itemCls;
    owner.textContent = false;

    if (itemCls) {
      element.classList.add(itemCls);
    }

    if (ownerItemCls) {
      element.classList.add(ownerItemCls);
    }

    contentElement.insertBefore(element, nextSibling);
    toAdd.render(null, Boolean(owner.isPainted));
    me.fixChildIndices();
    me.syncChildCount();
  }

  fixChildIndices() {
    this.owner.items.forEach((child, index) => {
      child.element.dataset.itemIndex = index;
    });
  }

  syncChildCount() {
    var _owner$contentElement;

    const {
      owner
    } = this,
          {
      length
    } = owner.items; // Special CSS conditions may apply if there's only a single child.

    (_owner$contentElement = owner.contentElement) === null || _owner$contentElement === void 0 ? void 0 : _owner$contentElement.classList[length === 1 ? 'add' : 'remove']('b-single-child');
  }
  /**
   * Registers a layout `config` property that cannot be acted upon at this time but must wait for the `owner` to
   * fully render its elements (in particular the `contentElement`).
   * @param {String} config The name of the config to sync later.
   * @internal
   */

  syncConfigLater(config) {
    const pendingConfigs = this.pendingConfigs || (this.pendingConfigs = []);

    if (!pendingConfigs.includes(config)) {
      pendingConfigs.push(config);
    }
  }
  /**
   * Sets the specified `style` to the value of the config given its `name`.
   * @param {Object} options The name of the config with the value to apply to the given `style`.
   * @param {String} options.name The name of the config with the value to apply to the given `style`.
   * @param {String} options.style The style property to set on the `contentElement`.
   * @param {String[]} [options.classes] A list of config values that should be added as CSS classes.
   * @param {Object} [options.map] An mapping object to convert the config's value to the `style` value.
   * @internal
   */

  syncConfigStyle({
    name,
    style,
    classes,
    map
  }) {
    const me = this,
          baseCls = `b-box-${name}-`,
          // ex: 'b-box-justify-'
    {
      contentElement
    } = me,
          raw = me[name];
    let value = (map === null || map === void 0 ? void 0 : map[raw]) || raw;

    if (contentElement) {
      if (classes) {
        const {
          classList
        } = contentElement;
        classes.forEach(c => {
          if (c !== value) {
            classList.remove(baseCls + c);
          }
        });

        if (classes.includes(value)) {
          classList.add(baseCls + value);
          value = ''; // to remove the inline style
        }
      }

      contentElement.style[style] = value;
    } else {
      me.syncConfigLater(name);
    }
  }

  syncPendingConfigs() {
    const me = this,
          {
      pendingConfigs
    } = me;
    let name;

    if (pendingConfigs) {
      me.pendingConfigs = null;

      while (name
      /* assignment */
      = pendingConfigs.pop()) {
        me[me.$meta.configs[name].updater](me[name]);
      }
    }
  }

}
Layout.initClass();
Layout._$name = 'Layout';

let lastTouchTime = 0;

const hasRipple = w => w.ripple;

class Ripple extends Widget {
  static get defaultConfig() {
    return {
      old_element: {
        children: [{
          className: 'b-ripple-inner',
          reference: 'rippleElement'
        }]
      },
      element: {
        children: [{
          tag: 'svg',
          class: 'b-ripple-inner',
          reference: 'rippleElement',
          ns: 'http://www.w3.org/2000/svg',
          version: '1.1',
          viewBox: '0 0 100 100',
          children: [{
            reference: 'circleElement',
            tag: 'circle',
            cx: '0',
            cy: '0',
            r: 10
          }]
        }]
      },
      floating: true,
      hideAnimation: false,
      showAnimation: false,
      scrollAction: 'realign',
      color: 'rgba(0,0,0,.3)',
      startRadius: 10,
      radius: 100
    };
  }

  static get $name() {
    return 'Ripple';
  }

  afterConstruct() {
    super.afterConstruct();
    EventHelper.on({
      element: this.rootElement,
      mousedown: 'onRippleControllingEvent',
      thisObj: this,
      capture: true,
      once: true
    });
  }

  onRippleControllingEvent(event) {
    var _me$listenerDetacher;

    const me = this;
    me.show();
    const rippleAnimation = DomHelper.getStyleValue(me.circleElement, 'animationName');
    me.hide();
    (_me$listenerDetacher = me.listenerDetacher) === null || _me$listenerDetacher === void 0 ? void 0 : _me$listenerDetacher.call(me); // If our theme supports ripples, add our listeners

    if (rippleAnimation && rippleAnimation !== 'none') {
      me.listenerDetacher = EventHelper.on({
        // Trap all mousedowns and see if the encapsulating Component is configured to ripple
        mousedown: {
          element: me.rootElement,
          capture: true,
          handler: 'onMousedown'
        },
        touchstart: {
          element: me.rootElement,
          capture: true,
          handler: 'onTouchStart'
        },
        // Hide at the end of the ripple
        animationend: {
          element: me.circleElement,
          handler: 'onAnimationEnd'
        },
        thisObj: me
      }); // If this is the first mousedown, start listening to theme changes and trigger ripple manually

      if (event.type === 'mousedown') {
        me.onMousedown(event);
        GlobalEvents.on({
          theme: 'onRippleControllingEvent',
          thisObj: this
        });
      }
    }
  }

  onTouchStart(event) {
    lastTouchTime = performance.now();
    this.handleTriggerEvent(event);
  }

  onMousedown(event) {
    // We need to prevent a touchend->mousedown simulated mousedown from triggering a ripple.
    // https://developer.mozilla.org/en-US/docs/Web/API/Touch_events/Supporting_both_TouchEvent_and_MouseEvent
    if (performance.now() - lastTouchTime > 200) {
      this.handleTriggerEvent(event);
    }
  }

  handleTriggerEvent(event) {
    const targetWidget = Widget.fromElement(event.target, hasRipple);

    if (targetWidget) {
      const rippleCfg = targetWidget.ripple,
            target = rippleCfg.delegate ? event.target.closest(rippleCfg.delegate) : targetWidget.focusElement || targetWidget.element;

      if (target) {
        const ripple = ObjectHelper.assign({
          event,
          target,
          radius: this.radius
        }, rippleCfg); // The clip option is specified as a string property name or delegate

        if (typeof ripple.clip === 'string') {
          ripple.clip = targetWidget[ripple.clip] || event.target.closest(ripple.clip); // Not inside an instance of the clip delegate, then no ripple

          if (!ripple.clip) {
            return;
          }
        }

        this.ripple(ripple);
      }
    }
  }

  ripple({
    event,
    point = EventHelper.getClientPoint(event),
    target = event.target,
    clip = target,
    radius = this.radius,
    color = this.color
  }) {
    this.clip = clip;
    clip = Rectangle.from(clip, null, true);
    const me = this,
          centreDelta = clip.getDelta(point),
          rippleStyle = me.rippleElement.style,
          circleElement = me.circleElement,
          borderRadius = DomHelper.getStyleValue(target, 'border-radius');
    me.hide();
    rippleStyle.transform = `translateX(${centreDelta[0]}px) translateY(${centreDelta[1]}px)`;
    rippleStyle.height = rippleStyle.width = `${radius}px`;
    me.element.style.borderRadius = borderRadius;
    circleElement.setAttribute('r', radius);
    circleElement.setAttribute('fill', color); // Show aligned center to center with our clipping region.

    me.showBy({
      target: clip,
      align: 'c-c',
      matchSize: true
    });
  } // When fully expanded, it's all over.

  onAnimationEnd(event) {
    if (event.animationName === 'b-ripple-expand') {
      this.hide();
    }
  }

}
Widget.RippleClass = Ripple;
Ripple._$name = 'Ripple';

/**
 * @module Core/data/Duration
 */

/**
 * Class which represents a duration object. A duration consists of a `magnitude` and a `unit`.
 *
 * ```javascript
 * {
 *    unit      : String,
 *    magnitude : Number
 * }
 * ```
 *
 * Valid values are:
 * - "millisecond" - Milliseconds
 * - "second" - Seconds
 * - "minute" - Minutes
 * - "hour" - Hours
 * - "day" - Days
 * - "week" - Weeks
 * - "month" - Months
 * - "quarter" - Quarters
 * - "year"- Years
 */

class Duration {
  /**
   * Duration constructor.
   * @param {Number|String} magnitude Duration magnitude value or a duration + magnitude string ('2h', '4d')
   * @param {String} [unit] Duration unit value
   */
  constructor(magnitude, unit) {
    if (typeof magnitude === 'number') {
      this._magnitude = magnitude;
      this._unit = unit;
    } else {
      if (typeof magnitude === 'string') {
        magnitude = DateHelper.parseDuration(magnitude);
      }

      if (typeof magnitude === 'object') {
        Object.assign(this, magnitude);
      }
    }
  }
  /**
   * Get/Set numeric magnitude `value`.
   * @property {Number}
   */

  get magnitude() {
    return this._magnitude;
  }

  set magnitude(value) {
    this._magnitude = typeof value === 'number' && value;
  }
  /**
   * Get/Set duration unit to use with the current magnitude value.
   * Valid values are:
   * - "millisecond" - Milliseconds
   * - "second" - Seconds
   * - "minute" - Minutes
   * - "hour" - Hours
   * - "day" - Days
   * - "week" - Weeks
   * - "month" - Months
   * - "quarter" - Quarters
   * - "year"- Years
   *
   * @property {String}
   */

  get unit() {
    return this._unit;
  }

  set unit(value) {
    this._unit = DateHelper.parseTimeUnit(value);
  }

  get isValid() {
    return this._magnitude != null && this._unit;
  }
  /**
   * The `milliseconds` property is a read only property which returns the number of milliseconds in this Duration
   * @property {Number}
   * @readonly
   */

  get milliseconds() {
    // There's no smaller time unit in the Date class than milliseconds, so round any divided values
    return this.isValid ? Math.round(DateHelper.asMilliseconds(this._magnitude, this._unit)) : 0;
  }
  /**
   * Returns truthy value if this Duration equals the passed value.
   * @param {Core.data.Duration} value
   * @return {Boolean}
   */

  isEqual(value) {
    return Boolean(value && this.milliseconds === value.milliseconds);
  }

  toString(useAbbreviation) {
    const me = this,
          abbreviationFn = useAbbreviation ? 'getShortNameOfUnit' : 'getLocalizedNameOfUnit';
    return me.isValid ? `${me._magnitude} ${DateHelper[abbreviationFn](me._unit, me._magnitude !== 1)}` : '';
  }

  toJSON() {
    return this.toString();
  }

  valueOf() {
    return this.milliseconds;
  }

}
Duration._$name = 'Duration';

/**
 * @module Core/util/CollectionFilter
 */

const nestedValueReducer = (object, path) => object === null || object === void 0 ? void 0 : object[path];
/**
 * A class which encapsulates a single filter operation which may be applied to any object to decide whether to
 * include or exclude it from a set.
 *
 * A CollectionFilter generally has at least three main properties:
 *
 * * `property` - The name of a property in candidate objects from which to extract the value to test
 * * `value` - The value which  this filter uses to test against.
 * * `operator` - The comparison operator, eg: `'='` or `'>'` etc.
 *
 * Given these three essential values, further configurations may affect how the filter is applied:
 *
 * * `caseSensitive` - If configured as `false`, string comparisons are case insensitive.
 * * `convert` - A function which, when passed the extracted value from the candidate object, returns the value to test.
 *
 * A filter may also be configured with a single `filterBy` property. This function is just passed the raw
 * candidate object and must return `true` or `false`.
 *
 * A CollectionFilter may be configured to encapsulate a single filtering function by passing that function as the sole
 * parameter to the constructor:
 *
 *     new CollectionFilter(candidate => candidate.title.contains('search string'));
 *
 */

class CollectionFilter extends Base$1.mixin(Identifiable) {
  static get defaultConfig() {
    return {
      /**
       * The value against which to compare the {@link #config-property} of candidate objects.
       * @config {*}
       */
      value: null,

      /**
       * The operator to use when comparing a candidate object's {@link #config-property} with this CollectionFilter's {@link #config-value}.
       * May be: `'='`, `'!='`, `'>'`, `'>='`, `'<'`, `'<='`, `'*'`, `'startsWith'`, `'endsWith'`, `'isIncludedIn'`
       * @config {String}
       */
      operator: null,

      /**
       * May be used in place of the {@link #config-property}, {@link #config-value} and {@link #config-property} configs. A function which
       * accepts a candidate object and returns `true` or `false`
       * @config {Function}
       */
      filterBy: null,

      /**
       * A function which accepts a value extracted from a candidate object using the {@link #config-property} name, and
       * returns the value which the filter should use to compare against its {@link #config-value}.
       * @config {Function}
       */
      convert: null,

      /**
       * Configure as `false` to have string comparisons case insensitive.
       * @config {Boolean}
       */
      caseSensitive: true,

      /**
       * The `id` of this Filter for when used by a {@link Core.util.Collection} Collection.
       * By default the `id` is the {@link #config-property} value.
       * @config {String}
       */
      id: null,
      // Type is required to process the Date value in State API. Store doesn't always know about field type to
      // process filter value, when it applies it from the state, e.g. when you don't declare model field as `date`
      // type but provide a Date instance there. When DateColumn is used to shows this field, it could add date
      // filters to the store. When store is applying state it cannot just infer type, because model doesn't
      // declare it. Only column knows. So to properly process the Date instance for the filter State API would
      // have to process the field additionally, checking model field type and column type. So it is simpler to
      // make Filter to put this information. That way when filter is instantiated by the store, it can gracefully
      // handle value processing, converting date string to the Date instance.
      // Date is the only known value type so far which requires this processing.
      type: null
    };
  }

  static get configurable() {
    return {
      /**
       * The name of a property of candidate objects which yields the value to compare against this CollectionFilter's {@link #config-value}.
       * @member {String} property
       */

      /**
       * The name of a property of candidate objects which yields the value to compare against this CollectionFilter's {@link #config-value}.
       * @config {String}
       */
      property: null
    };
  }

  construct(config) {
    if (typeof config === 'function') {
      config = {
        filterBy: config
      };
    } // If Filter is created without a type (yet everything except applying state) create one

    if (!config.type) {
      if (DateHelper.isDate(config.value)) {
        config.type = 'date';
      } else if (config.value instanceof Duration) {
        config.type = 'duration';
      }
    } // If type already exist, it means we are applying state and should process value
    else {
      if (config.type === 'date') {
        config.value = new Date(config.value);
      } else if (config.type === 'duration') {
        config.value = new Duration(config.value);
      }
    }

    super.construct(config);
  }
  /**
   * When in a Collection (A Collection holds its Filters in a Collection), we need an id.
   * @property {String}
   * @private
   */

  get id() {
    return this._id || (this._id = this.property || CollectionFilter.generateId('b-filter'));
  }

  set id(id) {
    this._id = id;
  }

  onChange(propertyChanged) {
    const me = this; // Inform any owner (eg a Store), that it has to reassess its CollectionFilters

    if (!me.isConfiguring && me.owner && !me.owner.isConfiguring && me.owner.onFilterChanged) {
      me.owner.onFilterChanged(me, propertyChanged);
    }
  }

  get filterBy() {
    return this._filterBy || this.defaultFilterBy;
  }
  /**
   * May be used in place of the {@link #config-property}, {@link #config-value} and {@link #config-property} configs. A function which
   * accepts a candidate object and returns `true` or `false`
   * @type {Function}
   */

  set filterBy(filterBy) {
    this._filterBy = filterBy;
  }

  defaultFilterBy(candidate) {
    const me = this;
    let candidateValue; // check if is nested property

    if (me._propertyItems.length > 1) {
      // support nested props (https://github.com/bryntum/support/issues/1861)
      candidateValue = me._propertyItems.reduce(nestedValueReducer, candidate);
    } else {
      candidateValue = candidate[me.property];
    }

    return me[me.operator](me.convert(candidateValue));
  }

  updateProperty(property) {
    this._propertyItems = property.split('.'); // Signal to owner about filter change

    this.onChange('property');
  }
  /**
   * The value against which to compare the {@link #config-property} of candidate objects.
   * @type {*}
   */

  set value(value) {
    var _value$valueOf, _value;

    this._value = value;
    value = (_value$valueOf = (_value = value) === null || _value === void 0 ? void 0 : _value.valueOf()) !== null && _value$valueOf !== void 0 ? _value$valueOf : value; // Filter value is a processed value to be used by the comparators. Useful when value is object, like Duration field

    this._filterValue = !this.caseSensitive && typeof value === 'string' ? value.toLowerCase() : value; // Signal to owner about filter change

    this.onChange('value');
  }

  get value() {
    return this._value;
  }

  get filterValue() {
    return this._filterValue;
  }
  /**
   * The operator to use when comparing a candidate object's {@link #config-property} with this CollectionFilter's {@link #config-value}.
   * May be: `'='`, `'!='`, `'>'`, `'>='`, `'<'`, `'<='`, `'*'`, `'startsWith'`, `'endsWith'`, `'isIncludedIn'`
   * @type {String}
   */

  set operator(operator) {
    this._operator = operator; // Signal to owner about filter change

    this.onChange('operator');
  }

  get operator() {
    if (this._operator) {
      return this._operator;
    }

    if (Array.isArray(this.filterValue)) {
      return 'isIncludedIn';
    }

    return typeof this.filterValue === 'string' ? '*' : '=';
  }

  convert(value) {
    // This is a workaround for filterbar feature: it always converts input value to string. When date is typed,
    // it is converted into string, and Date.valueOf() would return number. So if we are matching date against string
    // type, we should not convert it.
    if (!(typeof this.filterValue === 'string' && value instanceof Date)) {
      var _value$valueOf2, _value2;

      // if value is a complex type, try to access `value` property to get primitive value
      value = (_value$valueOf2 = (_value2 = value) === null || _value2 === void 0 ? void 0 : _value2.valueOf()) !== null && _value$valueOf2 !== void 0 ? _value$valueOf2 : value;
    }

    value = !this.caseSensitive && typeof value === 'string' ? value.toLowerCase() : value;
    return value;
  }

  filter(candidate) {
    return this.filterBy(candidate);
  }

  startsWith(v) {
    return String(v).startsWith(this.filterValue);
  }

  endsWith(v) {
    return String(v).endsWith(this.filterValue);
  }

  isIncludedIn(v) {
    return this.filterValue.length === 0 || this.filterValue.includes(v);
  }

  '='(v) {
    return ObjectHelper.isEqual(v, this.filterValue);
  }

  '!='(v) {
    return !ObjectHelper.isEqual(v, this.filterValue);
  }

  '>'(v) {
    return ObjectHelper.isMoreThan(v, this.filterValue);
  }

  '>='(v) {
    return ObjectHelper.isMoreThan(v, this.filterValue) || ObjectHelper.isEqual(v, this.filterValue);
  }

  '<'(v) {
    return ObjectHelper.isLessThan(v, this.filterValue);
  }

  '<='(v) {
    return ObjectHelper.isLessThan(v, this.filterValue) || ObjectHelper.isEqual(v, this.filterValue);
  }

  '*'(v) {
    return ObjectHelper.isPartial(v, this.filterValue);
  } // Accepts an array or a Collection

  static generateFiltersFunction(filters) {
    if (!filters || !filters.length && !filters.count) {
      return FunctionHelper.returnTrue;
    }

    return function (candidate) {
      let match = true;

      for (const filter of filters) {
        // Skip disabled filters
        if (!filter.disabled) {
          match = filter.filter(candidate);
        }

        if (!match) {
          break;
        }
      }

      return match;
    };
  }

}
CollectionFilter._$name = 'CollectionFilter';

/**
 * @module Core/util/CollectionSorter
 */

/**
 * A class which encapsulates a single sorter operation which may be applied to any object to decide whether to
 * include or exclude it from a set.
 *
 * A CollectionSorter generally has two properties:
 *
 * * `property` - The name of a property in collection objects by which to sort
 * * `direction` - The sort direction, `'ASC'` or `'DESC'`.
 *
 * It may also be configured with just a {@link #config-sortFn} function which returns the desired comparison
 * result when passed two objects to compare. Note that this does *not* require or use the
 * {@link #config-property} config. Two collection items are passed for comparison.
 *
 * Further configurations may affect how the sorter is applied:
 *
 * * `convert` - A function which, when passed the {@link #config-property} value from
 * a collection object, returns the value to sort by.
 *
 * A CollectionSorter may be configured to encapsulate a {@link #config-sortFn} by passing that function as the sole
 * parameter to the constructor:
 *
 *     new CollectionSorter((lhs, rhs) => {
 *         lhs = lhs.customerDetails.companyName.toLowerCase();
 *         rhs = rhs.customerDetails.companyName.toLowerCase();
 *
 *         if (lhs < rhs) {
 *             return -1;
 *         }
 *         else if (lhs > rhs) {
 *             return 1;
 *         }
 *         else {
 *             return 0;
 *         }
 *     });
 *
 */

class CollectionSorter extends Base$1 {
  static get defaultConfig() {
    return {
      /**
       * The name of a property of collection objects which yields the value to sort by.
       * @config {String}
       */
      property: null,

      /**
       * The direction to sort in, `'ASC'` or `'DESC'`
       * @config {String}
       * @default
       */
      direction: 'ASC',

      /**
       * A function which takes the place of using {@link #config-property} and {@link #config-direction}.
       * The function is passed two objects from the collection to compare and must return the comparison result.
       * @config {Function}
       */
      sortFn: null,

      /**
       * When using {@link #config-property}, this may be specified as a function which takes the raw
       * property value and returns the value to actually sort by.
       * @config {Function}
       */
      convert: null,

      /**
       * The `id` of this Sorter for when used by a {@link Core.util.Collection} Collection.
       * By default the `id` is the {@link #config-property} value.
       * @config {String}
       */
      id: null,

      /**
       * Use `localeCompare()` when sorting, which lets the browser sort in a locale specific order. Set to `true`,
       * a locale string or a locale config to enable.
       *
       * Enabling this has big negative impact on sorting
       * performance. For more info on `localeCompare()`, see [MDN](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/localeCompare).
       *
       * ```javascript
       * collection.addSorter({ field: 'name', useLocaleSort : 'sv-SE' });
       * ```
       *
       * @config {Boolean|String|Object}
       * @default false
       */
      useLocaleSort: null
    };
  }

  construct(config) {
    if (typeof config === 'function') {
      config = {
        sortFn: config
      };
    }

    super.construct(config);
  }
  /**
   * When in a Collection (A Collection holds its Sorters in a Collection), we need an id.
   * @property {String}
   * @private
   */

  get id() {
    return this._id || (this._id = this.property || IdHelper.generateId('b-sorter'));
  }

  set id(id) {
    this._id = id;
  }

  set sortFn(sortFn) {
    this._sortFn = sortFn;
  }

  get sortFn() {
    if (this._sortFn) {
      return this._sortFn;
    }

    return this.defaultSortFn;
  }
  /**
   * Default sortFn used when no sortFn specified. Uses the {@link #config-property},
   * {@link #config-direction}, and {@link #config-convert}.
   * @private
  */

  defaultSortFn(lhs, rhs) {
    const me = this,
          {
      convert,
      property,
      useLocaleSort
    } = me,
          multiplier = me.direction.toLowerCase() === 'desc' ? -1 : 1;
    lhs = lhs[property];
    rhs = rhs[property];

    if (convert) {
      lhs = convert(lhs);
      rhs = convert(rhs);
    }

    if (useLocaleSort && lhs != null && rhs != null) {
      // Use systems locale
      if (useLocaleSort === true) {
        return String(lhs).localeCompare(rhs);
      } // Use specified locale

      if (typeof useLocaleSort === 'string') {
        return String(lhs).localeCompare(rhs, useLocaleSort);
      } // Use locale config

      if (typeof useLocaleSort === 'object') {
        return String(lhs).localeCompare(rhs, useLocaleSort.locale, useLocaleSort);
      }
    }

    return (lhs > rhs ? 1 : lhs < rhs ? -1 : 0) * multiplier;
  }

  static generateSortFunction(sorters, tieBreaker) {
    const items = sorters.isCollection ? sorters.values : sorters,
          n = items.length;
    return (lhs, rhs) => {
      let comp, i;

      for (i = 0; i < n; ++i) {
        comp = items[i].sortFn(lhs, rhs);

        if (comp) {
          return comp;
        }
      }

      return tieBreaker ? tieBreaker(lhs, rhs) : 0;
    };
  }

}
CollectionSorter._$name = 'CollectionSorter';

/**
 * @module Core/util/Collection
 */

const return0 = () => 0,
      reverseNumericSortFn = (a, b) => b - a,
      filteredIndicesProperty = Symbol('filteredIndicesProperty'),
      emptyArray$8 = Object.freeze([]),
      sortEvent = Object.freeze({
  action: 'sort',
  added: emptyArray$8,
  removed: emptyArray$8,
  replaced: emptyArray$8
}),
      filterEvent = Object.freeze({
  action: 'filter',
  added: emptyArray$8,
  removed: emptyArray$8,
  replaced: emptyArray$8
}),
      keyTypes = {
  string: 1,
  number: 1
}; // Adds a single item to a single index using the specified key

function addItemToIndex(item, index, key) {
  // Unique holds a single entry
  if (index.unique !== false) {
    index.set(key, item);
  } // Non-unique index holds a Set
  else {
    let set = index.get(key); // Add a set if this is the first entry

    if (!set) {
      set = new Set();
      index.set(key, set);
    } // Add entry to the set

    set.add(item);
  }
} // Removes a single item from a single index using the specified key

function removeItemFromIndex(item, index, key) {
  // Unique holds single entry, remove it
  if (index.unique !== false) {
    index.delete(key);
  } else if (index.has(key)) {
    // Remove from set
    index.get(key).delete(item); // Remove turned empty set

    if (!index.get(key).size) {
      index.delete(key);
    }
  }
} // Used to fully build the indices, normal and filtered (if used). Better to do full builds for performance reasons
// when assigning new datasets. For other CRUD operations, indices are kept up to date elsewhere

function doRebuildIndices(values, indices, keyProps, indexCount) {
  for (let i = 0; i < values.length; i++) {
    const item = values[i];

    for (let j = 0; j < indexCount; j++) {
      const keyProp = keyProps[j],
            key = item[keyProp],
            index = indices[keyProp];
      addItemToIndex(item, index, key);
    }
  }
}
/**
 * A class which encapsulates a {@link #function-get keyed},
 * {@link #function-addFilter filterable}, {@link #function-addSorter sortable}
 * collection of objects. Entries may not be atomic data types such as `string` or `number`.
 *
 * The entries are keyed by their `id` which is determined by interrogating the {@link #config-idProperty}.
 *
 * To filter a Collection, add a {@link Core.util.CollectionFilter CollectionFilter}
 * using the {@link #function-addFilter} method. A Filter config object may be specified here
 * which will be promoted to a CollectionFilter instance.
 *
 * To sort a Collection, add a {@link Core.util.CollectionSorter CollectionSorter}
 * using the {@link #function-addSorter} method. A Sorter config object may be specified here
 * which will be promoted to a CollectionSorter instance.
 */

class Collection extends Base$1.mixin(Events) {
  //region Config
  static get configurable() {
    return {
      /**
       * Specify the name of the property of added objects which provides the lookup key
       * @config {String}
       * @default
       */
      idProperty: 'id',

      /**
       * Specify the names or index configs of properties which are to be indexed for fast lookup.
       *
       * Index configs use the format `{ property : string, unique : boolean }`. Unique indices stores one index
       * per entry, non-unique stores a `Set`. If left out, `unique` defaults to `true`
       *
       * @config {String[]|Object[]}
       */
      extraKeys: null,

      /**
       * Automatically apply filters on item add.
       * @config {Boolean}
       * @default
       */
      autoFilter: true,

      /**
       * Automatically apply sorters on item add.
       * @config {Boolean}
       * @default
       */
      autoSort: true,

      /**
       * A {@link Core.util.CollectionSorter Sorter}, or Sorter config object, or
       * an array of these, to use to sort this Collection.
       * @config {Object[]}
       * @default
       */
      sorters: {
        $config: ['lazy'],
        value: []
      }
    };
  }

  get isCollection() {
    return true;
  } //endregion
  //region Init & destroy

  construct(config) {
    /**
     * A counter which is incremented whenever the Collection is mutated in a meaningful way.
     *
     * If a {@link #function-splice} call results in no net replacement, removal or addition,
     * then the `generation` will not be incremented.
     * @property {Number}
     * @readonly
     */
    this.generation = 0;
    this._values = [];
    super.construct(config);
  }

  doDestroy() {
    var _me$_sorters;

    super.doDestroy();
    const me = this;
    me._values.length = 0;

    if (me.isFiltered) {
      me._filteredValues.length = 0;
      me.filters.destroy();
    } // eslint-disable-next-line no-unused-expressions

    (_me$_sorters = me._sorters) === null || _me$_sorters === void 0 ? void 0 : _me$_sorters.destroy();
  } //endregion
  //region "CRUD"

  /**
   * Clears this collection.
   */

  clear() {
    const me = this,
          removed = me._values.slice();

    if (me.totalCount) {
      me._values.length = 0;

      if (me._filteredValues) {
        me._filteredValues.length = 0;
      }

      me._indicesInvalid = true; // Indicate to observers that data has changed.

      me.generation++;
      me.trigger('change', {
        action: 'clear',
        removed
      });
    }
  }
  /**
   * Compares the content of this Collection with the content of the passed Collection or
   * with the passed array. Order insensitive. This returns `true` if the two objects passed
   * contain the same set of items.
   * @param {Core.util.Collection|Array} other The Collection or array to compare with.
   * @param {Function} [map] Optionally a function to convert the items into a comparable object
   * to compare. For example `item => item.id` could be used to compare the ids of the
   * constituent items.
   * @returns {Boolean} `true` if the two objects passed have the same content.
   */

  equals(other, map) {
    if (other.isCollection) {
      other = other.values;
    }

    if (other.length === this.count) {
      let {
        values
      } = this;

      if (map) {
        other = other.map(map);
        values = values.map(map);
      }

      return ArrayHelper.delta(other, values).inBoth.length === this.count;
    }

    return false;
  }
  /**
   * Replaces the internal `values` array with the passed `values`, or `filteredValues` array with the passed `filteredValues`.
   * If `filteredValues` are not passed explicitly, but storage is filtered, decides internally `values` or `filteredValues` should
   * be replaced by passed `values`.
   *
   * Note that this takes ownership of the array, and the array must not be mutated by outside code.
   *
   * This is an internal utility method, not designed for use by application code.
   *
   * @param {Object} params Values and parameters to replace
   * @param {Object[]} params.values The new `values` array
   * @param {Object[]} [params.filteredValues] The new `filteredValues` array. Applicable only when storage is filtered.
   * @param {Boolean} [params.silent=false] If true, `change` event will not be fired
   * @param {Boolean} [params.isNewDataset=false] If true, `values` is a new dataset
   * @fires change
   * @internal
   */

  replaceValues({
    values,
    filteredValues,
    silent = false,
    isNewDataset = false
  }) {
    const me = this;
    let replacedValues, replacedFilteredValues; // The isNewDataset flag is passed by store#loadData to indicate that it's
    // a new data load, and that local filters can be applied.
    // Other use cases are for purely local updates of an existing dataset such as
    // refreshing the visible data with a values array containing group headers.

    if (me.isFiltered && !isNewDataset) {
      const filteredPassed = Boolean(filteredValues); // If `filteredValues` are missing, take `values` as a source of filtered values

      if (!filteredPassed) {
        filteredValues = values.slice();
        values = null;
      } // otherwise check if non-filtered values are passed together with filtered, and replace them too
      else if (values) {
        replacedValues = me._values;
        me._values = values.slice();
      }

      replacedFilteredValues = me._filteredValues;
      me._filteredValues = filteredValues.slice();
    } else {
      replacedValues = me._values;
      me._values = values.slice();
      filteredValues = null;

      if (me.isFiltered && isNewDataset && me.autoFilter) {
        me._filterFunction = null;
        me._filteredValues = me._values.filter(me.filterFunction);
      } else if (me._filteredValues) {
        me._filteredValues.length = 0;
      }
    }

    me._indicesInvalid = true; // Indicate to observers that data has changed.

    me.generation++;

    if (!silent) {
      me.trigger('change', {
        action: 'replaceValues',
        replacedValues,
        replacedFilteredValues,
        values,
        filteredValues
      });
    }
  }

  set values(values) {
    // Want a full rebuild for new dataset, less costly than doing it per item
    this.invalidateIndices();
    this.splice(0, this._values.length, values);
  }
  /**
   * The set of values of this Collection. If this Collection {@link #property-isFiltered},
   * this yields the filtered data set.
   *
   * Setting this property replaces the data set.
   * @property {Object[]}
   */

  get values() {
    return this.isFiltered ? this._filteredValues : this._values;
  }
  /**
   * Iterator that allows you to do `for (const item of collection)`
   */

  [Symbol.iterator]() {
    return this.values[Symbol.iterator]();
  }
  /**
   * Executes the passed function for each item in this Collection, passing in the item,
   * ths index, and the full item array.
   * @param {Function} fn The function to execute.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to include all items, bypassing filters.
   */

  forEach(fn, ignoreFilters = false) {
    (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).forEach(fn);
  }
  /**
   * Extracts ths content of this Collection into an array based upon the passed
   * value extraction function.
   * @param {Function} fn A function, which, when passed an item, returns a value to place into the resulting array.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to process an item even if it is filtered out.
   * @returns {Object[]} An array of values extracted from this Collection.
   */

  map(fn, ignoreFilters = false) {
    return (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).map(fn);
  }
  /**
   * Returns the first item in this Collection which elicits a *truthy* return value from the passed function.
   * @param {Function} fn A function, which, when passed an item, returns `true` to select it as the item to return.
   * @param {Boolean} [ignoreFilters=false] Pass `true` to include filtered out items.
   * @returns {Object} The matched item, or `undefined`.
   */

  find(fn, ignoreFilters = false) {
    return (this.isFiltered && !ignoreFilters ? this._filteredValues : this._values).find(fn);
  }

  get first() {
    return this.values[0];
  }

  get last() {
    return this.values[this.count - 1];
  }
  /**
   * The set of all values of this Collection regardless of filters applied.
   * @readonly
   * @property {Object[]}
   */

  get allValues() {
    return this._values;
  }
  /**
   * Adds items to this Collection. Multiple new items may be passed.
   *
   * By default, new items are appended to the existing values.
   *
   * Any {@link #property-sorters} {@link #property-sorters} present are re-run.
   *
   * Any {@link #property-filters} {@link #property-filters} present are re-run.
   *
   * *Note that if application functionality requires add and remove, the
   * {@link #function-splice} operation is preferred as it performs both
   * operations in an atomic manner*
   * @param  {...Object} items The item(s) to add.
   */

  add(...items) {
    if (items.length === 1) {
      this.splice(this._values.length, null, ...items);
    } else {
      this.splice(this._values.length, null, items);
    }
  }
  /**
   * Removes items from this Collection. Multiple items may be passed.
   *
   * Any {@link #property-sorters} {@link #property-sorters} present are re-run.
   *
   * Any {@link #property-filters} {@link #property-filters} present are re-run.
   *
   * *Note that if application functionality requires add and remove, the
   * {@link #function-splice} operation is preferred as it performs both
   * operations in an atomic manner*
   * @param  {...Object} items The item(s) to remove.
   */

  remove(...items) {
    if (items.length === 1) {
      this.splice(0, ...items);
    } else {
      this.splice(0, items);
    }
  }
  /**
   * Moves an individual item, or a block of items to another location.
   * @param {Object|Object[]} items The item/items to move.
   * @param {Object} [beforeItem] the item to insert the first item before. If omitted, the `item`
   * is moved to the end of the Collection.
   * @returns {Number} The new index of the `item`.
   */

  move(items, beforeItem) {
    items = ArrayHelper.asArray(items); // Handle the case of move(myItem, myItem). It's a no-op

    while (items.length && items[0] === beforeItem) {
      items.shift();
    }

    if (!items.length) {
      return;
    }

    const me = this,
          {
      _values
    } = me,
          itemIndex = me.indexOf(items[0], true); // move(record, followingrecord) is a no-op

    if (items.length === 1 && _values[itemIndex + 1] === beforeItem) {
      return;
    } // Silently remove the items that are to be inserted before the "beforeIt