"use strict";
/*
*  Copyright (C) 1998-2021 by Northwoods Software Corporation. All Rights Reserved.
*/

// This is the definitions of the predefined text editor used by TextEditingTool
// when you set or bind TextBlock.editable to true.
// You do not need to load this file in order to use in-place text editing.

// HTML + JavaScript text editor menu, made with HTMLInfo
// This is a re-implementation of the default text editor
// This file exposes one instance of HTMLInfo, window.TextEditor
// See also TextEditor.html
var textarea;
(function(window) {
  textarea = document.createElement('textarea');
  textarea.id = "textEditoTextArea";

  // textarea.style.opacity = 0.0;
  // textarea.style.overflow = "hidden";
  textarea.style.resize = "none";

  textarea.addEventListener('input', function(e) {
    var tool = TextEditor.tool;
    if (tool.textBlock === null) return;
    var tempText = tool.measureTemporaryTextBlock(this.value);

    var scale = this.textScale ;
    this.style.width =  tempText.measuredBounds.width * scale + 'px';
    this.rows = tempText.lineCount;

    board.model.commit(function (){
      tool.textBlock.text = textarea.value;
    },null);
    
    //..........................................................
    var textBlock = tool.textBlock;
    if (textBlock.part && textBlock.part.data && textBlock.panel && textBlock.part.data.category === "NOTE") {
      var auto = textBlock.part.data.textSize === "Auto";
      adjustAutoFont(textBlock, auto);
    }
    //..........................................................
    setTextAreaPoperties(tool.textBlock, tool.textBlock.diagram, tool);

  }, false);

  textarea.addEventListener('keydown', function(e) {
    var tool = TextEditor.tool;
    if (tool.textBlock === null) return;
    var key = e.key;
    if (key === "Enter") {
      if (tool.textBlock.isMultiline === false) e.preventDefault();
      tool.acceptText(go.TextEditingTool.Enter);
      return;
    } else if (key === "Tab") {
      tool.acceptText(go.TextEditingTool.Tab);
      e.preventDefault();
      return;
    } else if (key === "Escape") {
      tool.doCancel();
      if (tool.diagram !== null) tool.diagram.doFocus();
    }
  }, false);

  // handle focus:
  textarea.addEventListener('focus', function(e) {
    var tool = TextEditor.tool;
    if (!tool || tool.currentTextEditor === null) return;

    if (tool.state === go.TextEditingTool.StateActive) {
      tool.state = go.TextEditingTool.StateEditing;
    }

    if (tool.selectsTextOnActivate) {
      textarea.select();
      textarea.setSelectionRange(0, 9999);
    }
  }, false);

  // Disallow blur.
  // If the textEditingTool blurs and the text is not valid,
  // we do not want focus taken off the element just because a user clicked elsewhere.
  textarea.addEventListener('blur', function(e) {
    var tool = TextEditor.tool;
    if (!tool || tool.currentTextEditor === null || tool.state === go.TextEditingTool.StateNone) return;

    textarea.focus();

    if (tool.selectsTextOnActivate) {
      textarea.select();
      textarea.setSelectionRange(0, 9999);
    }
  }, false);


  var TextEditor = new go.HTMLInfo();

  TextEditor.valueFunction = function() { return textarea.value; }

  TextEditor.mainElement = textarea; // to reference it more easily

  TextEditor.tool = null; // Initialize

  // used to be in doActivate
  TextEditor.show = function(textBlock, diagram, tool) {
    if (!(textBlock instanceof go.TextBlock)) return;
    if (TextEditor.tool !== null) return; // Only one at a time.

    TextEditor.tool = tool;  // remember the TextEditingTool for use by listeners

    // This is called during validation, if validation failed:
    if (tool.state === go.TextEditingTool.StateInvalid) {
      textarea.style.border = '3px solid red';
      textarea.focus();
      return;
    }

    
    setTextAreaPoperties(textBlock, diagram, tool);

    // Show:
    diagram.div.appendChild(textarea);

    // After adding, focus:
    textarea.focus();
    if (tool.selectsTextOnActivate) {
      textarea.select();
      textarea.setSelectionRange(0, 9999);
    }
  };

  TextEditor.hide = function(diagram, tool) {
    diagram.div.removeChild(textarea);
    TextEditor.tool = null;  // forget reference to TextEditingTool
  }

  window.TextEditor = TextEditor;
})(window);


function setTextAreaPoperties (textBlock, diagram, tool) {

   // This part is called during initalization:

   textBlock.part.ensureBounds();

    var loc = textBlock.getDocumentPoint(go.Spot.Center);
    var pos = diagram.position;
    var sc = diagram.scale;

    var partScale = (textBlock.part && textBlock.part.data && textBlock.part.data.scale) ? textBlock.part.data.scale : 1;
    var textscale = textBlock.getDocumentScale() * sc;
    
    var extraScale = sc < 1 ? sc : 1; // orig
    var extraScale = 1; // for resolution
    partScale = partScale > 1 ? 1 : partScale; // orig
    // partScale = partScale * sc;

    if (textscale < tool.minimumEditorScale) textscale = tool.minimumEditorScale;
    // Add slightly more width/height to stop scrollbars and line wrapping on some browsers
    // +6 is firefox minimum, otherwise lines will be wrapped improperly
    var textwidth = ((textBlock.actualBounds.width * extraScale) * textscale * partScale) + 15;
    var textheight = ((textBlock.actualBounds.height * extraScale) * textscale * partScale) + 2;
    var left = (loc.x - pos.x) * sc ;
    var top = (loc.y - pos.y) * sc ;

    textarea.value = textBlock.text;
    // the only way you can mix font and fontSize is if the font inherits and the fontSize overrides
    // in the future maybe have textarea contained in its own div
    diagram.div.style['font'] = textBlock.font;

    var paddingsize = 0;
    textarea.style['background'] = 'transparent';
    textarea.style['color'] = textBlock.stroke != null ? textBlock.stroke : "black";
    textarea.style['position'] = 'absolute';
    textarea.style['zIndex'] = '100';
    textarea.style['font'] = 'inherit';
    textarea.style['fontSize'] = (textscale * 100 * extraScale * partScale) + '%';
    // textarea.style['fontSize'] = (100 ) + '%';
    textarea.style['lineHeight'] = 1.09;
    textarea.style['width'] = (textwidth) + 'px';
    textarea.style['height'] = (textheight) + 'px';
    textarea.style['left'] = ((left - (textwidth / 2) | 0) - paddingsize) + 'px';
    textarea.style['top'] = ((top - (textheight / 2) | 0) - paddingsize) + 'px';
    textarea.style['textAlign'] = textBlock.textAlign;
    textarea.style['margin'] = '0';
    textarea.style['padding'] = paddingsize + 'px';
    textarea.style['border'] = '0';
    textarea.style['outline'] = 'none';
    textarea.style['whiteSpace'] = 'pre-wrap';
    textarea.style['overflow'] = 'hidden'; // for proper IE wrap
    textarea.rows = textBlock.lineCount;
    textarea.textScale = textscale; // attach a value to the textarea, for convenience
    textarea.className = 'goTXareaQ';

}

// function adjustAutoFont(node, auto) {

//   var textBlock;
//   if (node.name && node.name === "textBlock") {
//     textBlock = node;
//   }
//   else {
//     textBlock = node.findObject("textBlock");
//   }

//   var finalFontSize;
//   var initialFontType = auto;
//   if (textBlock.panel) {

//     textBlock.part.ensureBounds();
//     var panelBound = textBlock.panel.actualBounds;

//     var textBlockMarginHorizontal = 30;
//     var textBlockMarginVertical = 20;

//     board.model.commit(function () {
//       tempTextForAutoFont.width = panelBound.width - textBlockMarginHorizontal;
//     }, null);

//     var helperTextBlock = tempTextForAutoFont.findObject("textBlock");

//     var textSize;
//     var font;
//     var actualBounds;

//     var fontFamily = textBlock.part.data.textFont;
//     if (!fontFamily) {
//       fontFamily = "sans-serif";
//     }

//     if (!auto) {

//       textSize = textBlock.part.data.textSize;
//       font = "" + textSize + "px "+fontFamily;

//       board.model.commit(function () {
//         helperTextBlock.font = font;
//         helperTextBlock.text = textBlock.text;
//       }, null);

//       tempTextForAutoFont.ensureBounds();

//       actualBounds = helperTextBlock.actualBounds;

//       if (actualBounds.height > (panelBound.height - textBlockMarginVertical)) {
//         auto = true;
//         textBlock.part.data.autoFont = textSize;
//         textBlock.part.data.textSize = "Auto";
//       }
//     }
//     else {
//       textSize = textBlock.part.data.autoFont;
//       font = "" + textSize + "px "+fontFamily;

//       board.skipsUndoManager = true;
//       board.model.commit(function () {
//         helperTextBlock.font = font;
//         helperTextBlock.text = textBlock.text;
//       }, null);
//       board.skipsUndoManager = false;
//       tempTextForAutoFont.ensureBounds();

//       actualBounds = helperTextBlock.actualBounds;
//     }

//     if (auto) {
//       var fontStep = 0;
//       var fit = true;
//       if (actualBounds.height < (panelBound.height - textBlockMarginVertical)) {
//         fontStep = 1;
//         fit = false;
//       }
//       else if (actualBounds.height > (panelBound.height - textBlockMarginVertical)) {
//         fontStep = -1;
//         fit = false;
//       }

//       finalFontSize = textSize;
//       while (!fit) {

//         textSize += fontStep;
//         var font = "" + textSize + "px "+fontFamily;

//         board.skipsUndoManager = false;
//         board.model.commit(function () {
//           helperTextBlock.font = font;
//           helperTextBlock.text = textBlock.text;
//         }, null);
//         board.skipsUndoManager = false;
        
//         tempTextForAutoFont.ensureBounds();

//         var tempBounds = helperTextBlock.actualBounds;

//         var textMetrics = helperTextBlock.metrics;
//         var expectedLineHeight = panelBound.height * (0.5 / textMetrics.arrSize.length);

//         if (fontStep > 0) {

//           if (tempBounds.height >= (panelBound.height - textBlockMarginVertical) || textMetrics.fontHeight > expectedLineHeight) {
//             fit = true;
//           }
//         }
//         else {

//           // $("#logDiv").html(":"+helperTextBlock.metrics.arrSize.length);
//           if (tempBounds.height <= (panelBound.height - textBlockMarginVertical)) {
//             fit = true;
//           }
//         }
//         if (!fit) {
//           finalFontSize = textSize;
//         }
//       }
//       if (finalFontSize != textSize) {
//         textBlock.part.data.autoFont = finalFontSize;
//         var font = "" + finalFontSize + "px "+fontFamily;

//         board.model.commit(function () {
//           textBlock.font = font;
//         }, null);

//         board.model.updateTargetBindings(textBlock.part.data);
//       }
//     }
//   }
//   return { changeMode: initialFontType != auto, size: finalFontSize, textBlock: textBlock };
//   //...................
// }
