"use strict";
/*
*  Copyright (C) 1998-2021 by Northwoods Software Corporation. All Rights Reserved.
*/

// A custom Tool for moving a label on a Node

/*
* This is an extension and not part of the main GoJS library.
* Note that the API for this class may change with any version, even point releases.
* If you intend to use an extension in production, you should copy the code to your own source directory.
* Extensions can be found in the GoJS kit under the extensions or extensionsTS folders.
* See the Extensions intro page (https://gojs.net/latest/intro/extensions.html) for more information.
*/


function NodeLabelDraggingTool() {
  go.Tool.call(this);
  this.name = "NodeLabelDragging";

  /** @type {GraphObject} */
  this.label = null;
  /** @type {Point} */
  this._offset = new go.Point();  // of the mouse relative to the center of the label object
  /** @type {Point} */
  this._originalAlignment = null;
  /** @type {Point} */
  this._originalCenter = null;
}
go.Diagram.inherit(NodeLabelDraggingTool, go.Tool);


NodeLabelDraggingTool.prototype.canStart = function() {
  if (!go.Tool.prototype.canStart.call(this)) return false;
  var diagram = this.diagram;
  if (diagram === null) return false;
  // require left button & that it has moved far enough away from the mouse down point, so it isn't a click
  var e = diagram.lastInput;
  if (!e.left) return false;
  if (!this.isBeyondDragSize()) return false;

  return this.findLabel() !== null;
}

NodeLabelDraggingTool.prototype.findLabel = function() {
  var diagram = this.diagram;
  var e = diagram.firstInput;
  var elt = diagram.findObjectAt(e.documentPoint, null, null);

  if (elt === null || !(elt.part instanceof go.Node)) return null;
  while (elt.panel !== null) {
    if (elt._isNodeLabel && elt.panel.type === go.Panel.Spot && elt.panel.findMainElement() !== elt) return elt;
    elt = elt.panel;
  }
  return null;
};

NodeLabelDraggingTool.prototype.doActivate = function() {
  this.startTransaction("Shifted Label");
  this.label = this.findLabel();
  if (this.label !== null) {
    // compute the offset of the mouse-down point relative to the center of the label
    this._offset = this.diagram.firstInput.documentPoint.copy().subtract(this.label.getDocumentPoint(go.Spot.Center));
    this._originalAlignment = this.label.alignment.copy();
    var main = this.label.panel.findMainElement();
    this._originalCenter = main.getDocumentPoint(go.Spot.Center);
  }
  go.Tool.prototype.doActivate.call(this);
}

NodeLabelDraggingTool.prototype.doDeactivate = function() {
  go.Tool.prototype.doDeactivate.call(this);
  this.stopTransaction();
}

NodeLabelDraggingTool.prototype.doStop = function() {
  this.label = null;
  go.Tool.prototype.doStop.call(this);
}

NodeLabelDraggingTool.prototype.doCancel = function() {
  if (this.label !== null) {
    this.label.alignment = this._originalAlignment;
  }
  go.Tool.prototype.doCancel.call(this);
}

NodeLabelDraggingTool.prototype.doMouseMove = function() {
  if (!this.isActive) return;
  this.updateAlignment();
}


NodeLabelDraggingTool.prototype.doMouseUp = function() {
  if (!this.isActive) return;
  this.updateAlignment();
  this.transactionResult = "Shifted Label";
  this.stopTool();
}

NodeLabelDraggingTool.prototype.updateAlignment = function() {
  if (this.label === null) return;
  var last = this.diagram.lastInput.documentPoint;
  var cntr = this._originalCenter;
  this.label.alignment = new go.Spot(0.5, 0.5, last.x - this._offset.x - cntr.x, last.y - this._offset.y - cntr.y);
}
