
var board = undefined;

class EraserTool extends go.Tool {
	
	constructor(init) {
		super();

		this.name = "EraserTool";
		this._archetypePartData = {}; // the data to copy for a new polyline Part
		this._isBackgroundOnly = true; // affects canStart()
		this._trailTimer = null;

		// this is the Shape that is shown during a drawing operation
		this._temporaryShape = go.GraphObject.make(go.Shape, { name: "SHAPE", fill: null, strokeWidth: 5.0, stroke: "lightgray" });
		// the Shape has to be inside a temporary Part that is used during the drawing operation
		go.GraphObject.make(go.Part, { layerName: "Tool" }, this._temporaryShape);

		if (init)
			Object.assign(this, init);
	}
	
	canStart() {
		if (!this.isEnabled) return false;
		var diagram = this.diagram;
		if (diagram === null || diagram.isReadOnly || diagram.isModelReadOnly) return false;
		if (!diagram.allowInsert) return false;
		// don't include the following check when this tool is running modally
		if (diagram.currentTool !== this && this.isBackgroundOnly) {
			// only operates in the background, not on some Part
			// var part = diagram.findPartAt(diagram.lastInput.documentPoint, true);
			// if (part !== null) return false;
		}
		return true;
	}
	
	doActivate() {

		super.doActivate();
		this.diagram.isMouseCaptured = true;
		// this.diagram.currentCursor = "crosshair";
		this._trailTimer = setInterval(this.trailTimerCall, 20);
	}
	
	doDeactivate() {
		super.doDeactivate();
		if (this.temporaryShape !== null) {
			this.diagram.remove(this.temporaryShape.part);
		}
		this.diagram.currentCursor = "";
		this.diagram.isMouseCaptured = false;

		if (this._trailTimer != null) {
			clearInterval(this._trailTimer);
			this._trailTimer = null;
		}
	}
	
	trailTimerCall () {

		if (!_.isEmpty(board)) {

			var eraserTool = board.toolManager.findTool("EraserTool");

			if (eraserTool != null && eraserTool.isEnabled && eraserTool.isActive) {

				var shape = eraserTool.temporaryShape;
				if (shape === null) return;

				if (shape.geometry === null) return;

				var geo = shape.geometry.copy();
				var fig = geo.figures.first();
				if (fig.segments.count > 0) {
					fig.segments.removeAt(0);
				}

				if (fig.segments.count > 0) {
					fig.startX = fig.segments.get(0).endX;
					fig.startY = fig.segments.get(0).endY;
				}
				shape.geometry = geo;
			}
		}
	}

	addPoint (p) {
		var shape = this.temporaryShape;
		if (shape === null) return;

		// for the temporary Shape, normalize the geometry to be in the viewport
		var viewpt = this.diagram.viewportBounds.position;
		var q = new go.Point(p.x - viewpt.x, p.y - viewpt.y);

		var part = shape.part;
		if (part.diagram === null) {
			var fig = new go.PathFigure(q.x, q.y, true);  // possibly filled, depending on Shape.fill
			var geo = new go.Geometry().add(fig);  // the Shape.geometry consists of a single PathFigure
			this.temporaryShape.geometry = geo;
			// position the Shape's Part, accounting for the strokeWidth
			part.position = new go.Point(viewpt.x - shape.strokeWidth / 2, viewpt.y - shape.strokeWidth / 2);
			this.diagram.add(part);
		}

		// only add a point if it isn't too close to the last one
		var segs = shape.geometry.figures.first().segments;
		var idx = segs.count - 1;
		if (idx >= 0) {
			var last = segs.elt(idx);
			if (Math.abs(q.x - last.endX) < 1.0 && Math.abs(q.y - last.endY) < 1.0) return;
		}

		var geo = shape.geometry.copy();
		var fig = geo.figures.first();
		var seg;

		// must copy whole Geometry in order to add a PathSegment
		seg = new go.PathSegment(go.PathSegment.Line, q.x, q.y);

		var seg = new go.PathSegment(go.PathSegment.Line, q.x, q.y);
		fig.add(seg);

		if (fig.segments.count == 1) {
			fig.startX = fig.segments.get(0).endX;
			fig.startY = fig.segments.get(0).endY;
		}

		shape.geometry = geo;
	}

	/**
	* Start drawing the line by starting to accumulate points in the {@link #temporaryShape}'s geometry.
	* @this {EraserTool}
	*/
	doMouseDown () {
		if (!this.isActive) {
			this.doActivate();
			// the first point
			this.addPoint(this.diagram.lastInput.documentPoint);
		}
	}

	/**
	* Keep accumulating points in the {@link #temporaryShape}'s geometry.
	* @this {EraserTool}
	*/
	doMouseMove () {
		if (this.isActive) {
			this.addPoint(this.diagram.lastInput.documentPoint);

			var e = this.diagram.lastInput;
			var curObj = this.diagram.findPartAt(e.documentPoint, false);

			if (curObj != null && curObj.data && curObj.data.category == "FreehandDrawing") {
				this.diagram.model.startTransaction(this.name);
				this.diagram.model.removeNodeData(curObj.data);
				this.diagram.model.commitTransaction(this.name);
			}
		}
	}

	/**
	* Finish drawing the line by adding a node data object holding the
	* geometry string and the node position that the node template can bind to.
	* This copies the {@link #archetypePartData} and adds it to the model.
	* @this {EraserTool}
	*/
	doMouseUp () {
		if (this.isActive) {
			var diagram = this.diagram;
			// the last point
			this.addPoint(diagram.lastInput.documentPoint);
		}
		this.stopTool();
	}
	
	get temporaryShape () {
		return this._temporaryShape;
	}
	
	set temporaryShape(val) {
		if (this._temporaryShape !== val && val !== null) {
			val.name = "SHAPE";
			var panel = this._temporaryShape.panel;
			panel.remove(this._temporaryShape);
			this._temporaryShape = val;
			panel.add(this._temporaryShape);
		}
	}
	
	get archetypePartData () {
		return this._archetypePartData;
	}
	set archetypePartData (val) {
		this._archetypePartData = val;
	}
	
	get isBackgroundOnly () {
		return this._isBackgroundOnly;
	}
	set isBackgroundOnly (val) {
		this._isBackgroundOnly = val;
	}
	
	get trailTimer () {
		return this._trailTimer;
	}
	set trailTimer (val) {
		this._trailTimer = val;
	}

}