"use strict";
/*
*  Copyright (C) 1998-2021 by Northwoods Software Corporation. All Rights Reserved.
*/

// A custom Tool for freehand drawing

/*
* This is an extension and not part of the main GoJS library.
* Note that the API for this class may change with any version, even point releases.
* If you intend to use an extension in production, you should copy the code to your own source directory.
* Extensions can be found in the GoJS kit under the extensions or extensionsTS folders.
* See the Extensions intro page (https://gojs.net/latest/intro/extensions.html) for more information.
*/

/**
* @constructor
* @extends Tool
* @class
* This tool allows the user to draw a shape using the mouse.
* It collects all of the points from a mouse-down, all mouse-moves, until a mouse-up,
* and puts all of those points in a {@link Geometry} used by a {@link Shape}.
* It then adds a node data object to the diagram's model.
* <p/>
* This tool may be installed as the first mouse down tool:
* <code>myDiagram.toolManager.mouseDownTools.insertAt(0, new EraserTool());</code>
* <p/>
* The Shape used during the drawing operation can be customized by setting {@link #temporaryShape}.
* The node data added to the model can be customized by setting {@link #archetypePartData}.
*/

var board = undefined;

function EraserTool() {
  go.Tool.call(this);
  this.name = "EraserTool";
  this._archetypePartData = {}; // the data to copy for a new polyline Part
  this._isBackgroundOnly = true; // affects canStart()
  this._trailTimer = null;

  // this is the Shape that is shown during a drawing operation
  this._temporaryShape = go.GraphObject.make(go.Shape, { name: "SHAPE", fill: null, strokeWidth: 5.0, stroke: "lightgray" });
  // the Shape has to be inside a temporary Part that is used during the drawing operation
  go.GraphObject.make(go.Part, { layerName: "Tool" }, this._temporaryShape);
}
go.Diagram.inherit(EraserTool, go.Tool);

/**
* Only start if the diagram is modifiable and allows insertions.
* OPTIONAL: if the user is starting in the diagram's background, not over an existing Part.
* @this {EraserTool}
*/
EraserTool.prototype.canStart = function() {
  if (!this.isEnabled) return false;
  var diagram = this.diagram;
  if (diagram === null || diagram.isReadOnly || diagram.isModelReadOnly) return false;
  if (!diagram.allowInsert) return false;
  // don't include the following check when this tool is running modally
  if (diagram.currentTool !== this && this.isBackgroundOnly) {
    // only operates in the background, not on some Part
    // var part = diagram.findPartAt(diagram.lastInput.documentPoint, true);
    // if (part !== null) return false;
  }
  return true;
};

/**
* Capture the mouse and use a "crosshair" cursor.
* @this {EraserTool}
*/
EraserTool.prototype.doActivate = function() {
  go.Tool.prototype.doActivate.call(this);
  this.diagram.isMouseCaptured = true;
  // this.diagram.currentCursor = "crosshair";
  this._trailTimer = setInterval(this.trailTimerCall, 20);
};

/**
* Cleanup.
* @this {EraserTool}
*/
EraserTool.prototype.doDeactivate = function() {
  go.Tool.prototype.doDeactivate.call(this);
  if (this.temporaryShape !== null) {
    this.diagram.remove(this.temporaryShape.part);
  }
  this.diagram.currentCursor = "";
  this.diagram.isMouseCaptured = false;

  if (this._trailTimer != null) {
    clearInterval(this._trailTimer);
    this._trailTimer = null;
  }
};

/**
* This adds a Point to the {@link #temporaryShape}'s geometry.
* <p/>
* If the Shape is not yet in the Diagram, its geometry is initialized and
* its parent Part is added to the Diagram.
* <p/>
* If the point is less than half a pixel away from the previous point, it is ignored.
* @this {EraserTool}
* @param {Point} p
*/

EraserTool.prototype.trailTimerCall = function() {

  if (!_.isEmpty(board)) {

    var eraserTool = board.toolManager.findTool("EraserTool");

    if (eraserTool != null && eraserTool.isEnabled && eraserTool.isActive) {

      var shape = eraserTool.temporaryShape;
      if (shape === null) return;

      if (shape.geometry === null) return;

      var geo = shape.geometry.copy();
      var fig = geo.figures.first();
      if (fig.segments.count > 0) {
        fig.segments.removeAt(0);
      }

      if (fig.segments.count > 0) {
        fig.startX = fig.segments.get(0).endX;
        fig.startY = fig.segments.get(0).endY;
      }
      shape.geometry = geo;
    }
  }
}

EraserTool.prototype.addPoint = function(p) {
  var shape = this.temporaryShape;
  if (shape === null) return;

  // for the temporary Shape, normalize the geometry to be in the viewport
  var viewpt = this.diagram.viewportBounds.position;
  var q = new go.Point(p.x-viewpt.x, p.y-viewpt.y);

  var part = shape.part;
  if (part.diagram === null) {
    var fig = new go.PathFigure(q.x, q.y, true);  // possibly filled, depending on Shape.fill
    var geo = new go.Geometry().add(fig);  // the Shape.geometry consists of a single PathFigure
    this.temporaryShape.geometry = geo;
    // position the Shape's Part, accounting for the strokeWidth
    part.position = new go.Point(viewpt.x - shape.strokeWidth/2, viewpt.y - shape.strokeWidth/2);
    this.diagram.add(part);
  }

  // only add a point if it isn't too close to the last one
  var segs = shape.geometry.figures.first().segments;
  var idx = segs.count-1;
  if (idx >= 0) {
    var last = segs.elt(idx);
    if (Math.abs(q.x - last.endX) < 1.0 && Math.abs(q.y - last.endY) < 1.0) return;
  }

  var geo = shape.geometry.copy();
  var fig = geo.figures.first();
  var seg;

  // must copy whole Geometry in order to add a PathSegment
  seg = new go.PathSegment(go.PathSegment.Line, q.x, q.y);

  var seg = new go.PathSegment(go.PathSegment.Line, q.x, q.y);
  fig.add(seg);

  if (fig.segments.count == 1) {
    fig.startX = fig.segments.get(0).endX;
    fig.startY = fig.segments.get(0).endY;
  }

  shape.geometry = geo;
};

/**
* Start drawing the line by starting to accumulate points in the {@link #temporaryShape}'s geometry.
* @this {EraserTool}
*/
EraserTool.prototype.doMouseDown = function() {
  if (!this.isActive) {
    this.doActivate();
    // the first point
    this.addPoint(this.diagram.lastInput.documentPoint);
  }
};

/**
* Keep accumulating points in the {@link #temporaryShape}'s geometry.
* @this {EraserTool}
*/
EraserTool.prototype.doMouseMove = function() {
  if (this.isActive) {
    this.addPoint(this.diagram.lastInput.documentPoint);

    var e = this.diagram.lastInput;
    var curObj = this.diagram.findPartAt(e.documentPoint, false);

    if (curObj != null && curObj.data && curObj.data.category == "FreehandDrawing") {
      this.diagram.model.startTransaction(this.name);
      this.diagram.model.removeNodeData(curObj.data);
      this.diagram.model.commitTransaction(this.name);
    }
  }
};

/**
* Finish drawing the line by adding a node data object holding the
* geometry string and the node position that the node template can bind to.
* This copies the {@link #archetypePartData} and adds it to the model.
* @this {EraserTool}
*/
EraserTool.prototype.doMouseUp = function() {
  if (this.isActive) {
    var diagram = this.diagram;
    // the last point
    this.addPoint(diagram.lastInput.documentPoint);
  }
  this.stopTool();
};

// Public properties

/**
* Gets or sets the Shape that is used to hold the line as it is being drawn.
* The default value is a simple Shape drawing an unfilled open thin black line.
* @name EraserTool#temporaryShape

* @return {Shape}
*/
Object.defineProperty(EraserTool.prototype, "temporaryShape", {
  get: function() { return this._temporaryShape; },
  set: function(val) {
    if (this._temporaryShape !== val && val !== null) {
      val.name = "SHAPE";
      var panel = this._temporaryShape.panel;
      panel.remove(this._temporaryShape);
      this._temporaryShape = val;
      panel.add(this._temporaryShape);
    }
  }
});

/**
* Gets or sets the node data object that is copied and added to the model
* when the freehand drawing operation completes.
* @name EraserTool#archetypePartData

* @return {Object}
*/
Object.defineProperty(EraserTool.prototype, "archetypePartData", {
  get: function() { return this._archetypePartData; },
  set: function(val) { this._archetypePartData = val; }
});

/**
* Gets or sets whether this tool can only run if the user starts in the diagram's background
* rather than on top of an existing Part.
* The default value is true.
* @name EraserTool#isBackgroundOnly

* @return {Object}
*/
Object.defineProperty(EraserTool.prototype, "isBackgroundOnly", {
  get: function() { return this._isBackgroundOnly; },
  set: function(val) { this._isBackgroundOnly = val; }
});

Object.defineProperty(EraserTool.prototype, "trailTimer", {
  get: function() { return this._trailTimer; },
  set: function(val) { this._trailTimer = val; }
});